/*******************************************************************************
 * Copyright (c) 2016 EmbSysRegView
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     ravenclaw78 - initial API and implementation
 *******************************************************************************/
package org.eclipse.cdt.embsysregview.internal.model;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.eclipse.cdt.embsysregview.core.SortingOrder;
import org.eclipse.cdt.embsysregview.internal.utils.Utils;

/**
 * 
 * */
public class TreeRegister extends TreeParent implements IEditable{
    public static final long BAD_VALUE = 0xdeadbeefL;

    private static final char FILLER_CHAR = '?';

    private long old_value;
    private long value = -1;
    private boolean retrievalActive = false;
    private final long resetValue;
    private final String type;
    private final long registerAddress;
    private final int size;
    private boolean uncertainFlag;
    private boolean readFlag;
    private boolean hasHistoryFlag;
    private List<TreeRegister> alterName;
    private final boolean isSPR;
    //
	private long maskReadOnly;

    public boolean isReadWrite() {
        return (getType().toUpperCase().equals("RW") || getType().toUpperCase().equals("RW1") //$NON-NLS-1$ //$NON-NLS-2$
                || getType().toUpperCase().equals("RW1C") || getType().toUpperCase().equals("RW1S") //$NON-NLS-1$ //$NON-NLS-2$
                || getType().toUpperCase().equals("RWH")); //$NON-NLS-1$
    }

    public boolean isReadOnly() {
        return (getType().toUpperCase().equals("RO") || getType().toUpperCase().equals("RC") //$NON-NLS-1$ //$NON-NLS-2$
                || getType().toUpperCase().equals("R")); //$NON-NLS-1$
    }

    public boolean isWriteOnly() {
        return (getType().toUpperCase().equals("WO") || getType().toUpperCase().equals("W") //$NON-NLS-1$ //$NON-NLS-2$
                || getType().toUpperCase().equals("W1C") || getType().toUpperCase().equals("W1S") //$NON-NLS-1$ //$NON-NLS-2$
                || getType().toUpperCase().equals("W1")); //$NON-NLS-1$
    }

    public boolean hasValueChanged() {
        if (!hasHistoryFlag) {
            return false;
        }
		return (old_value != value);
    }

    public boolean isRetrievalActive() {
        return retrievalActive;
    }

    public void setRetrievalActive(boolean retrievalActive) {
        this.retrievalActive = retrievalActive;        
    }

    public final long getRegisterAddress() {
        return registerAddress;
    }

    public final String getRegisterAddrString() {
        return String.valueOf(registerAddress);
    }

    public TreeRegister(String name, String description, long registerAddress, long resetValue, String type, int size, boolean isSPR) {
        super(name, description);
        this.registerAddress = registerAddress;
        this.resetValue = resetValue;
        this.type = type;
        this.size = size;
        this.readFlag = false;
        this.uncertainFlag = false;
        this.isSPR = isSPR;
        this.alterName = new ArrayList<>();
    }

    protected long getOldValue() {
        return old_value;
    }

    public long getValue() {
        if (isWriteOnly()) {
            return resetValue;
        }
        return value;
    }

    /**
     * 
     * */
    public String getPrintValue() {
        if (retrievalActive) {
            return Utils.longtoHexString(getValue(), getBitSize());
        }
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < getByteSize() * 2; i++) {
            sb.append(FILLER_CHAR);
        }
        return sb.toString();
    }

    public long getResetValue() {
        return resetValue;
    }

    public String getType() {
        return type;
    }

    /**
     * Sets the Register
     */
    public void setValue(long lvalue) {
        old_value = value;
        value = lvalue;
        uncertainFlag = false;
        if (readFlag) {
            hasHistoryFlag = true;
        }
        readFlag = true;        
    }

    public int getBitSize() {
        return size * 8;
    }

    public int getByteSize() {
        return size;
    }

    public void clearValue() {
        value = BAD_VALUE;
        uncertainFlag = false;
        readFlag = false;
    }

    /**
     * Return byte buffer representation with respect to endianess.
     */
    public byte[] getByteBuffer(boolean isBigEndian) {
        long l = value;
        byte[] result = new byte[getByteSize()];
        if (isBigEndian) {
            for (int i = getByteSize() - 1; i >= 0; i--) {
                result[i] = (byte) (l & 0xFF);
                l >>= 8;
            }
        } else {
            for (int i = 0; i < getByteSize(); i++) {
                result[i] = (byte) (l & 0xFF);
                l >>= 8;
            }
        }
        return result;
    }

    /**
     * @deprecated
     */
    public void setBytes(byte[] val) {
        long r = 0;
        for (int i = 0; i < val.length; i++) {
            r = r << 8;
            long tmp = val[i];
            r += (tmp & 0xFF);
        }
        setValue(r);
    }

    /** */
    public TreeRange getRange() {
        final TreeElement obj = getModelParent();
        if (obj instanceof TreeRange) {
            return (TreeRange) obj;
        }
        return null;

    }

    /**
     * Adds another facet to same register address.
     * 
     * @param reg
     *            not used now
     */
    public void addAlterName(final TreeRegister reg) {
        alterName.add(reg);
    }

    public List<TreeRegister> getAlterName() {
        return alterName;
    }

    /** */
    public void rollbackValue() {
        if (hasHistoryFlag) {
            value = old_value;
        }
        uncertainFlag = true;
    }

    /**  */
    public final boolean isUnsertain() {
        return uncertainFlag;
    }

    /** */
    public final boolean isAlreadyRead() {
        return readFlag;
    }

    /**
     * Check for editing rules. For R/W we can edit only then marked; for W/O
     * always yes for R/O always no; global flag disables all
     */
    public boolean isEditAllowed() {
        if (isModelAccessGranted()) {
            if (isReadWrite()) {
                return retrievalActive;
            } else if (isWriteOnly()) {
                return true;
            }
        }
        return false;
    }
    
    public boolean isSPR() {
    	return isSPR;
    }
    
    /** We cannot implement it as there is no information in XML*/
	public boolean checkWriteModifiers(final long value) {
		//1. calculate mask of acceptance on bitfields  
		//2. bitwize compare old | new | acceptance mask
		//3. check for ones in result
		long delta = getValue() ^ value;		
		return (delta & maskReadOnly) == 0;
	}
    
	/** */
	public void addBitFields(List<TreeField> bitFieldsList, SortingOrder.BitFieldSorter sorter) {
		if (sorter != SortingOrder.getDefaultSorter()) {
		  Collections.sort(bitFieldsList, sorter);
		}		
		for (final TreeField field : bitFieldsList) {
		   addChild(field);
		   modifyAccessMask(field);
		}
	}

	private void modifyAccessMask(final TreeField field) {
		String access = field.getAccess();
		if ("RO".equals(access)){		  		  
		  maskReadOnly = maskReadOnly | field.getBitMask();
		} 		
	}
	
    /** */
	public long getReadOnlyMask() {		
		return maskReadOnly;
	}

    public String getUniqueID() {
        return this.getName()+registerAddress;
    }
    
    public void sortBitFields() {
        TreeElement[] regChildren = getChildren();

        List<TreeField> regFields = Arrays.stream(regChildren)
                .filter(rc -> rc instanceof TreeField)
                .map(field -> (TreeField) field)
                .collect(Collectors.toList());

        Arrays.stream(regChildren)
                .filter(rc -> rc instanceof TreeField)
                .map(field -> (TreeField) field)
                .forEach(this::removeChild);

        addBitFields(regFields, SortingOrder.getFromPreference());
    }
}
