package org.eclipse.cdt.embsysregview.internal.model;

import java.util.SortedSet;
import java.util.TreeSet;

/**
 * Non-visual grouping entity. Represents continuous range of addresses with no
 * holes.
 */
public class TreeRange extends HiddenTreeParent {
    long startAddr;    
    SortedSet<Long> endAddrs;

    public TreeRange(String name, String description) {
        super(name, description);       
		endAddrs = new TreeSet<Long>() {
			/**
			 * 
			 */
			private static final long serialVersionUID = -3191919051825624865L;

			@Override 
			public boolean add(Long e) {
				if (isEmpty() || (last() < e)) {
					return super.add(e);
				} 
				return false;			 
			 }
		};
		
    }

    /**  */
    final public long getStartAddress() {
        return startAddr;
    }

    /**  */
    final public long getLastAddress() {
        return endAddrs.last();
    }

    /**  */
    @Override
    public boolean addModelChild(final TreeElement child) {
        boolean result = false;
        final TreeRegister reg = (TreeRegister) child;
        if (reg.isSPR()) {
        	throw new UnsupportedOperationException();
        }
        
        if (endAddrs.isEmpty()) {
            startAddr = reg.getRegisterAddress();            
        }
        
        final long newBoundary = reg.getRegisterAddress() + reg.getByteSize();
        if (endAddrs.add(newBoundary)) {        	
            result = super.addModelChild(reg);
        } else {
            addAlternative(reg);           
        }
        return result;
    }

    /** */
    public final String getRangeLowAddrString() {
        return Long.toString(startAddr);
    }

    /** */
    public int getRangeByteCount() {
        Long res = endAddrs.last() - startAddr;
        return res.intValue();
    }

    /** Adds another facet to same address. */
    protected void addAlternative(final TreeRegister node) {
    	boolean found = false;
        // find matching register by address in model
        for (final TreeElement obj : getModelChildren()) {
            if (obj instanceof TreeRegister) {
                final TreeRegister reg = (TreeRegister) obj;
                if (reg.getRegisterAddress() == node.getRegisterAddress() && 
                	(reg.getByteSize() == node.getByteSize())) {
                    //it is alias -> create round-table
                	node.addAlterName(reg);
                	for (TreeRegister altReg :reg.getAlterName()) {
                		node.addAlterName(altReg);
                	}                	
                	for (TreeRegister refReg :node.getAlterName()) {
                		refReg.addAlterName(node);
                	}                    
                    found = true;
                }
            }
        }
        //it is narrow-wide sub register
        if (!found) {
        	super.addModelChild(node);	
        }
    }

    @Override
    public String toString() {
        if (!endAddrs.isEmpty()) {
            return Long.toHexString(startAddr) + "-" + Long.toHexString(endAddrs.last());
        }
		return Long.toHexString(startAddr) + "- ???";
    }
}
