/*******************************************************************************
 * Copyright (c) 2016 EmbSysRegView
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     ravenclaw78 - initial API and implementation
 *******************************************************************************/
package org.eclipse.cdt.embsysregview.internal.model;

/**
 * Represents bit range (1 or more bits) within register.
 * */
public class TreeField extends TreeElement implements IEditable {
	
    private final byte bitOffset;
    private final byte bitLength;
    private boolean isLabelSwap;
    private Interpretations interpretations;
	private final String access;

    public TreeField(String name, String description, byte bitOffset, byte bitLength, Interpretations interpretations, String access, boolean isLabelSwap) {
        super(name, description);
        this.bitOffset = bitOffset;
        this.bitLength = bitLength;
        this.isLabelSwap = isLabelSwap;
        this.interpretations = interpretations;
        this.interpretations.setTreeField(this);
        this.access = access;
    }

    public boolean hasValueChanged() {
    	TreeRegister reg = (TreeRegister)getParent();
    	if (reg != null && reg.hasValueChanged()){
    	   return (getOldValue() != getValue());
    	} 
    	return false;
    }

    public byte getBitOffset() {
        return bitOffset;
    }

    public byte getBitLength() {
        return bitLength;
    }

    public Interpretations getInterpretations() {
        return interpretations;
    }

    public long getValue() {
        return stripValue(((TreeRegister)getParent()).getValue());
    }

    private long stripValue(final long value) {
        long tmp = value >> bitOffset; // drop the unnecessary bits "below" the
                                       // field
        tmp &= ~(0xFFFFFFFFL << bitLength); // drop the bits "above" the
        return tmp;
    }

    private long getOldValue() {
        return stripValue(((TreeRegister)getParent()).getOldValue());
    }

    public String getInterpretation() {
        return interpretations.getInterpretation(getValue());
    }

    /*
     * Check if Field has an actual interpretation for the current value
     */
    public boolean hasInterpretation() {
        return !this.getInterpretation().equals("");
    }

    /*
     * Check if Field has interpretations at all
     */
    public boolean hasInterpretations() {
        return interpretations.hasInterpretations();
    }

    /** */
    public String getLabelString() {
    	String format = String.format("%%%ds", getBitLength());
        return String.format(format, "").replace(' ', 'x');

    }
    /** */
    public String getFieldLabel() {
         if (bitLength == 1) {
             return String.format("%s (bit %d)", getName(), bitOffset); //$NON-NLS-1$
         } else if (!isLabelSwap) { 
             return String.format("%s (bits %d-%d)", getName(), (bitOffset + bitLength - 1) , bitOffset); //$NON-NLS-1$
         } else {
        	 return String.format("%s (bits %d-%d)", getName(), bitOffset, (bitOffset + bitLength - 1)); //$NON-NLS-1$
         }
    }
    
    /** */
	public final String getAccess() {
		return access;
	}
	
	public boolean isWriteOnly(){
		return "WO".equals(getAccess());
	}

    /** Return all ones in bitfield range*/
	public long getBitMask() {
		long mask = (0xFFFFFFFFL >> (32 - getBitLength())) << getBitOffset();
		//long mask2 = Long.rotateLeft(1, getBitLength() + 1) - 1;
		//mask2 = Long.rotateLeft(mask, getBitOffset());
		return mask;
	}

	@Override
	public boolean isEditAllowed() {		
		if (((TreeRegister)getParent()).isEditAllowed()) {
			return ! ("RO".equals(getAccess()));
		}
		return false;
	}

    public long getMaxAllowedValue() {
        return (long) Math.pow(2, getBitLength()) - 1;
    }
}
