package org.eclipse.cdt.embsysregview.internal.model;

import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.stream.Collectors;

import org.eclipse.cdt.embsysregview.Activator;
import org.eclipse.cdt.embsysregview.internal.RegisterXMLParser;
import org.eclipse.cdt.embsysregview.internal.utils.Utils;
import org.eclipse.cdt.embsysregview.views.EmbSysRegView.RegSet;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.model.MemoryByte;

/**
 * 
 * */
public class EmbSysRegModel {
    public final static int SIZE_ONEBYTE = 1;
    public final static int SIZE_TWOBYTES = 2;
    public final static int SIZE_FOURBYTES = 4;

    protected class EmbSysUIModel {
        private final TreeParent invisibleRoot;
        private Map<String, TreeRegister> liveRegisters;

        public EmbSysUIModel(TreeParent root) {
            liveRegisters = new HashMap<>();
            invisibleRoot = root;
        }

        public TreeParent getRoot() {
            return invisibleRoot;
        }

        public Map<String, TreeRegister> getLiveRegisters() {
            return liveRegisters;
        }
    }

    private IProject project;
    private Queue<TreeRegister> pendingToWrite;
    private Queue<TreeRegister> pendingToRead;
    private Map<IProject, EmbSysUIModel> mapProjectRegistry;
    private final RegSet regSet;

    /**
     * 
     * */
    public EmbSysRegModel(RegSet regSet) {
        mapProjectRegistry = new HashMap<>();
        // empty model tree for case when no open projects
        mapProjectRegistry.put(null, new EmbSysUIModel(new TreeParent("", "")));
        pendingToWrite = new LinkedList<>();
        pendingToRead = new LinkedList<>();
        this.regSet = regSet;
    }

    /**
     * 
     * */
    public void setRegisterLive(final TreeRegister treeRegister) {
        treeRegister.setRetrievalActive(true);
        // take care of your buddy
        for (TreeRegister altReg : treeRegister.getAlterName()) {
            altReg.setRetrievalActive(true);
        }
        mapProjectRegistry.get(project).getLiveRegisters().put(treeRegister.getUniqueID(), treeRegister);
        // here may be clone of register used
        pendingToRead.offer(treeRegister);
    }

    /**
     * 
     * */
    public void pushRegister(final TreeRegister register) {
        TreeRegister old = mapProjectRegistry.get(project).getLiveRegisters().get(register.getUniqueID());
        final long lvalue = register.getValue();
        if (old != null) {
            old.setValue(lvalue);
            for (TreeRegister altReg : old.getAlterName()) {
                if (old.getByteSize() == altReg.getByteSize()) {
                    altReg.setValue(lvalue);
                }
            }
        }
    }

    /**
     *  
     * */
    public Set<TreeRange> createRangeList() {
        pendingToRead.clear();
        return mapProjectRegistry.get(project).getLiveRegisters().values().stream().map(TreeRegister::getRange).distinct()
                .collect(Collectors.toSet());
    }

    /**  */
    public final TreeParent getRoot() {
    	if (mapProjectRegistry.containsKey(project)){
    		return mapProjectRegistry.get(project).getRoot();
    	}
		if (project != null) {
			return new TreeParent("", "");
		}
        return null;
    }

    /** */
    public boolean isEmpty() {
    	if (mapProjectRegistry.containsKey(project)){
            return mapProjectRegistry.get(project).getRoot() == null || !mapProjectRegistry.get(project).getRoot().hasChildren();
    	}
        return true;
    }

    /**
     * Initializes empty model tree
     * 
     * @return
     */
    public IStatus initialize() {
        pendingToRead.clear();
        project = null;
        return Status.OK_STATUS;
    }

    /**
     * Initializes model tree based on project's data * @return
     */
    public IStatus initialize(final IProject proj) {
        initialize();
        project = proj;
        try {
            if (!mapProjectRegistry.containsKey(project)) {
                mapProjectRegistry.put(project, new EmbSysUIModel(new RegisterXMLParser(project).loadXML(regSet)));
            } else {
                sortRegistersFieldsByBitFileldSorter();
            }
        } catch (CoreException e) {
            Activator.log(e.getStatus());
            return e.getStatus();
        }
        return Status.OK_STATUS;
    }

    private void sortRegistersFieldsByBitFileldSorter() {
        TreeParent parent = mapProjectRegistry.get(project).getRoot();
        if (parent != null) {
            Arrays.stream(parent.getChildren()).filter(child -> child instanceof TreeGroup)
                    .flatMap(group -> Arrays.stream(((TreeGroup) group).getChildren()))
                    .filter(child -> child instanceof TreeRegisterGroup)
                    .flatMap(regGroup -> Arrays.stream(((TreeRegisterGroup) regGroup).getChildren()))
                    .filter(child -> child instanceof TreeRegister)
                    .forEach(register -> ((TreeRegister) register).sortBitFields());
        }
    }

    /**  */
    public void setRegisterOffLine(final TreeRegister treeRegister) {
        treeRegister.setRetrievalActive(false);
        for (TreeRegister altReg : treeRegister.getAlterName()) {
            altReg.setRetrievalActive(false);
        }
        mapProjectRegistry.get(project).getLiveRegisters().remove(treeRegister.getUniqueID());
    }

    /**  */
    public final IProject getProject() {
        return project;
    }

    public void setRegisterToWrite(TreeRegister treeRegister) {
        pendingToWrite.offer(treeRegister);
    }

    /**
     * 
     * */
    public TreeRegister getPendingReg() {
        return pendingToWrite.poll();
    }

    /**
     * 
     * */
    public TreeRegister getPendingRead() {
        return pendingToRead.poll();
    }

    /**
     * Maps freshly read memory area to a number of registers in range.
     * Registers may be overlapped within range(special case!), but strictly contiguous.
     *  */
    public void applyRangeData(final TreeRange range, final MemoryByte[] bytes) {
        range.getModelChildren().stream()
                .filter(element -> element instanceof TreeRegister)
                .forEach(element -> applyRegisterData((TreeRegister) element, range, bytes));

    }

    private void applyRegisterData(TreeRegister register, TreeRange range, MemoryByte[] memoryBytes) {
        int registerByteSize = register.getByteSize();
        int registerOffset = (int) register.getRegisterAddress() - (int) range.getStartAddress();
        // look what the memory status at raw position
        long[] result = new long[] { -1 };
        if (Utils.convertToUnsigned(memoryBytes, registerOffset, registerByteSize, result)) {
            long regiserValue = result[0];
            register.setValue(regiserValue);
            register.getAlterName().stream().forEach(altRegister -> altRegister.setValue(regiserValue));
        } else {
            register.clearValue();
            register.getAlterName().stream().forEach(TreeRegister::clearValue);
        }
    }

    /** */
    public void setRootReadOnly(boolean flag) {
        TreeElement.setAccessGranted(!flag);
    }

    /** */
    public void cleanList(boolean sessionFlag) {
        if (sessionFlag && mapProjectRegistry.get(project) != null) {
            mapProjectRegistry.get(project).getLiveRegisters().forEach((key, value) -> {
                value.clearValue();
                for (final TreeRegister altReg : value.getAlterName()) {
                    altReg.clearValue();
                }
            });
        }
    }

    public Map<String, TreeRegister> getCherryPicked() {
        return mapProjectRegistry.get(project).getLiveRegisters();
    }

    public RegSet getRegSet() {
        return regSet;
    }

	public void removeEntry(IProject key) {
        if (mapProjectRegistry.containsKey(key)) {
            mapProjectRegistry.remove(key);
        }
    }

}
