/*******************************************************************************
 * Copyright (c) 2016 EmbSysRegView
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     ravenclaw78 - initial API and implementation
 *******************************************************************************/
package org.eclipse.cdt.embsysregview.internal;

import java.io.IOException;
import java.net.URL;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import org.eclipse.cdt.core.CProjectNature;
import org.eclipse.cdt.embsysregview.Activator;
import org.eclipse.cdt.embsysregview.Messages;
import org.eclipse.cdt.embsysregview.PropertiesKeys;
import org.eclipse.cdt.embsysregview.core.SortingOrder;
import org.eclipse.cdt.embsysregview.internal.model.Interpretations;
import org.eclipse.cdt.embsysregview.internal.model.TreeElement;
import org.eclipse.cdt.embsysregview.internal.model.TreeField;
import org.eclipse.cdt.embsysregview.internal.model.TreeGroup;
import org.eclipse.cdt.embsysregview.internal.model.TreeParent;
import org.eclipse.cdt.embsysregview.internal.model.TreeRegister;
import org.eclipse.cdt.embsysregview.internal.model.TreeRegisterGroup;
import org.eclipse.cdt.embsysregview.properties.PropertiesHolder;
import org.eclipse.cdt.embsysregview.properties.Property;
import org.eclipse.cdt.embsysregview.properties.SettingsUtils;
import org.eclipse.cdt.embsysregview.views.EmbSysRegView.RegSet;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.osgi.util.NLS;
import org.jdom2.Attribute;
import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.JDOMException;
import org.jdom2.input.SAXBuilder;
import org.jdom2.input.sax.XMLReaderJDOMFactory;
import org.jdom2.input.sax.XMLReaderXSDFactory;
import org.jdom2.input.sax.XMLReaders;
import org.osgi.framework.Bundle;

/**
 * 
 * */
public class RegisterXMLParser {
	private static final String HEX_PREFIX = "0X"; //$NON-NLS-1$
	/** added B to check for unsupported binary format */
	private static final String SCALE_SUFFIX = "KMGTB";// //$NON-NLS-1$
	private static final String separator = "/";
	private String store_board;
	private IProject project;

	@SuppressWarnings("serial")
	private static final Map<String, String> ACCESS_MAP = Collections.unmodifiableMap(new HashMap<String, String>() {
		{
			put("read-write", "RW");
			put("write-only", "WO");
			put("read-only", "RO");
			put("writeOnce", "W1");
			put("read-writeOnce", "RW1");
		}
	});

	public RegisterXMLParser(IProject project) {
		this.project = project;
	}

	// Keep for now if will need to implement possibility to configure
	// EmbSysRegView both in project settings and in global preferences,
	// inheritance of global preferences settings by newly created projects,
	// and possibility to choose what settings to use
	private Document openXML(boolean spr) throws CoreException {
		Map<String, String> registerDescriptionPaths = new RegisterXMLCollector().collectPaths();
		URL fileURL = null;

		for (String bundleName : registerDescriptionPaths.keySet()) {
			Bundle bundle = Platform.getBundle(bundleName);
			String directory = registerDescriptionPaths.get(bundleName);
			IPreferenceStore store = Activator.getDefault().getPreferenceStore();
			String store_architecture = store.getString(PropertiesKeys.ARCHITECTURE);
			String store_vendor = store.getString(PropertiesKeys.VENDOR);
			String store_chip = store.getString(PropertiesKeys.CHIP);
			store_board = store.getString(PropertiesKeys.BOARD);
			fileURL = bundle.getEntry(directory + "/" + store_architecture + "/" + store_vendor + "/" + store_chip //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					+ (spr ? ".spr" : "") + ".xml"); //$NON-NLS-1$
			if (fileURL != null)
				break;
		}

		SAXBuilder builder = new SAXBuilder(XMLReaders.NONVALIDATING);
		// builder.setValidation(true);
		// builder.setValidation(false);

		try {
			return builder.build(fileURL);
		} catch (JDOMException | IOException e) {
			IStatus status = new Status(IStatus.ERROR, Activator.PLUGIN_ID, "File reading problem", e);
			throw new CoreException(status);
		}
	}

	/** */
	private Document openXML(final IProject currentProject, boolean spr) throws CoreException {
		if (!currentProject.isNatureEnabled(CProjectNature.C_NATURE_ID)) {
			IStatus status = new Status(IStatus.INFO, Activator.PLUGIN_ID,
					String.format(Messages.EmbSysRegViewer_project, project.getName())
							+ Messages.EmbSysRegViewer_non_c_project);
			throw new CoreException(status);
		}

		URL fileURL = null;

		Set<Property> propertySet = SettingsUtils.initializePropertyValues(currentProject);
		if (propertySet.isEmpty()) {
			IStatus status = new Status(IStatus.INFO, Activator.PLUGIN_ID, String
					.format(Messages.RegisterXMLParser_Registers_definitions_not_found, currentProject.getName()));
			throw new CoreException(status);
		}
		Map<String, String> props = propertySet.stream()
				.collect(Collectors.toMap(Property::getName, Property::getValue));
		String store_architecture = props.get("architecture");
		String store_vendor = props.get("vendor");
		String store_chip = props.get("chip");
		store_board = props.get("board");

		if (store_board == null) {
			store_board = ""; //$NON-NLS-1$
		}

		if (store_architecture != null && store_vendor != null && store_chip != null) {
			fileURL = getFileURL(currentProject, spr);
		}
		// file with schema (CMSIS-SVD v1.1)
		final URL url = Activator.getDefault().getBundle().getEntry("check.xsd"); //$NON-NLS-1$
		if (url == null) {
			IStatus status = new Status(IStatus.ERROR, Activator.PLUGIN_ID,
					String.format(Messages.RegisterXMLParser_Bundle_not_found, "check.xsd", currentProject.getName()));
			throw new CoreException(status);
		}
		// input file may be null if not set in project properties or valid
		// path not constructed by properties
		if (fileURL == null) {
			IStatus status = new Status(IStatus.ERROR, Activator.PLUGIN_ID,
					String.format(Messages.RegisterXMLParser_No_data_file_for_project, currentProject.getName()));
			throw new CoreException(status);
		}

		try {
			// Create the XMLReaderJDOMFactory directly using the schema file
			// instead of internal 'Schema'
			XMLReaderJDOMFactory factory2 = new XMLReaderXSDFactory(url);
			SAXBuilder builder = new SAXBuilder(factory2);
			return builder.build(fileURL);
		} catch (JDOMException | IOException e) {
			IStatus status = new Status(IStatus.ERROR, Activator.PLUGIN_ID,
					String.format(Messages.EmbSysRegViewer_project, project.getName())
							+ String.format(Messages.RegisterXMLParser_File_reading_problem, fileURL.getFile()),
					e);
			throw new CoreException(status);
		}
	}

	/** */
	private static URL getFileURL(IProject project, boolean spr) {
		Map<String, String> registerDescriptionPaths = new RegisterXMLCollector().collectPaths();
		final String registerFileName = getRegisterFileName(project, spr);
		// scan all known bundles
		for (final String bundleName : registerDescriptionPaths.keySet()) {
			final Bundle bundle = Platform.getBundle(bundleName);
			final String directory = registerDescriptionPaths.get(bundleName);
			URL fileURL = bundle.getEntry(directory + separator + registerFileName);
			if (fileURL != null) {
				return fileURL;
			}
		}

		return null;
	}

	/** Creates name as /${architecture}/${vendor}/${chip}.xml */
	private static String getRegisterFileName(final IProject project, boolean spr) {
		Map<String, String> properties = PropertiesHolder.getInstance(project).getPropertySet().stream()
				.collect(Collectors.toMap(Property::getName, Property::getValue));
		return properties.get("architecture") + separator + properties.get("vendor") + separator
				+ properties.get(spr ? "core" : "chip") + (spr ? ".spr" : ".xml");
	}

	public TreeParent loadXML(final RegSet regSet) throws CoreException {
		return new TreeParent("", "") {
			{
				if (regSet.spr)
					for (TreeElement element : loadXML(true).getChildren())
						addChild(element);
				if (regSet.embsys)
					for (TreeElement element : loadXML(false).getChildren())
						addChild(element);
			}
		};
	}

	private TreeParent loadXML(boolean spr) throws CoreException {
		TreeParent tempRoot = new TreeParent("***", "root"); //$NON-NLS-1$ //$NON-NLS-2$

		Document doc = new Document();

		if (project == null) {
			doc = openXML(spr);
		} else {
			doc = openXML(project, spr);
		}

		if (doc != null) {
			final Element root = doc.getRootElement();
			try {
				if (root.getName() == "device") { //$NON-NLS-1$
					return parseSVD(root, tempRoot, spr);
				}
				return parseXML(root, tempRoot, spr);
			} catch (ParseException e) {
				throw new CoreException(new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Parse file problem", e));
			}
		}
		return tempRoot;
	}

	private static <E extends Exception> String attr(Element element, String name, Supplier<E> exception) throws E {
		Element attr = element.getChild(name);
		if (attr == null) {
			throw exception.get();
		} else {
			return attr.getValue();
		}
	}

	private static String attr(Element element, String name, String defaultValue) {
		Element attr = element.getChild(name);
		return attr == null ? "" : attr.getValue();
	}

	/*
	 * Tree Hierarchy (Base Class TreeElement): TreeParant TreeGroup
	 * TreeRegisterGroup TreeRegister TreeField
	 */

	/*
	 * Parse an SVD xml file
	 */
	private static TreeParent parseSVD(final Element root, final TreeParent tempRoot, final boolean spr)
			throws ParseException {
		TreeGroup oldTreeGroup = null;

		Element peripherals = root.getChild("peripherals"); //$NON-NLS-1$
		if (peripherals != null) {
			List<Element> grouplist = peripherals.getChildren("peripheral"); //$NON-NLS-1$

			for (Element group : grouplist) {

				// Mandatory attribute name
				String rgname = attr(group, "name", () -> new ParseException("peripheral requires name", 1));
				// Mandatory attribute baseAddress
				String baseAddress = attr(group, "baseAddress",
						() -> new ParseException("peripheral requires baseAddress", 1));
				long lbaseAddress = readScaledNonNegativeInteger(PropertiesKeys.BASE_ADDRESS_ATTRIBUTE_NAME,
						baseAddress);

				// Optional attribute description
				String gdescription = attr(group, "description", "").replaceAll("( )+", " ");

				// Handle derivedFrom
				Attribute attr_derivedFrom = group.getAttribute("derivedFrom"); //$NON-NLS-1$
				String derivedFrom;
				if (attr_derivedFrom != null) {
					derivedFrom = attr_derivedFrom.getValue();

					for (Element linkedgroup : grouplist) {
						// Mandatory attribute name
						String lname = attr(linkedgroup, "name",
								() -> new ParseException("derived peripheral requires name of super peripheral", 1));
						if (lname.equals(derivedFrom)) {
							group = linkedgroup;
							Element attr_lgdescription = linkedgroup.getChild("description"); //$NON-NLS-1$
							if (attr_lgdescription != null && gdescription.isEmpty()) {
								gdescription = attr_lgdescription.getValue().replaceAll("( )+", " "); //$NON-NLS-1$ //$NON-NLS-2$
								// TODO: add better description cleanup because
								// Vendors often don't provide
								// description Element for derived Register
								// groups
								String expression = ".*[0-9]$"; //$NON-NLS-1$
								if (gdescription.matches(expression) && gdescription
										.charAt(gdescription.length() - 1) == lname.charAt(lname.length() - 1)) {
									gdescription = gdescription.substring(0, gdescription.length() - 1)
											+ rgname.charAt(rgname.length() - 1);
								}
							}
							break;
						}
					}
				}

				// Group similar peripherals in TreeGroup
				TreeGroup obj_group = null;
				// Optional attribute groupName
				Element attr_gname = group.getChild("groupName"); //$NON-NLS-1$
				String gname;
				if (attr_gname != null) {
					gname = attr_gname.getValue();
					for (TreeElement te : tempRoot.getChildren()) {
						if (te.getName().equals(gname) && (te instanceof TreeGroup)) {
							obj_group = (TreeGroup) te;
							break;
						}
					}
					if (obj_group == null) {
						// TODO: add better description cleanup
						String expression = ".*[0-9]$"; //$NON-NLS-1$
						String tmpgdescription;
						if (gdescription.matches(expression) && gdescription.charAt(gdescription.length() - 1) == rgname
								.charAt(rgname.length() - 1)) {
							tmpgdescription = gdescription.substring(0, gdescription.length() - 1).trim();
						} else
							tmpgdescription = gdescription;

						obj_group = new TreeGroup(gname, tmpgdescription);
						tempRoot.addChild(obj_group);
					}
					oldTreeGroup = obj_group;
				} else {
					if (oldTreeGroup != null && oldTreeGroup.getName().equals(rgname))
						obj_group = oldTreeGroup;
					else {
						obj_group = new TreeGroup(rgname, gdescription);
						tempRoot.addChild(obj_group);
						oldTreeGroup = obj_group;
					}
				}

				TreeRegisterGroup obj_registergroup = new TreeRegisterGroup(rgname, gdescription);
				if (!spr) {
					obj_group.addChild(obj_registergroup);
				}

				Element registers = group.getChild("registers"); //$NON-NLS-1$

				if (registers != null) {
					processRegistersOrClusters(spr, lbaseAddress, obj_group, obj_registergroup, registers.getChildren(),
							""); //$NON-NLS-1$
					if (!spr) {
						obj_registergroup.extractRanges();
					}
				}
			}
		}
		return tempRoot;
	}

	private static void processRegistersOrClusters(final boolean spr, long lbaseAddress, TreeGroup obj_group,
			TreeRegisterGroup obj_registergroup, List<Element> children, String suffix) throws ParseException {
		for (Element element : children) {
			if ("register".equals(element.getName())) {
				processRegister(spr, lbaseAddress, obj_group, obj_registergroup, element, suffix);
			} else if ("cluster".equals(element.getName())) {
				processCluster(spr, lbaseAddress, obj_group, obj_registergroup, element, suffix);
			}
		}
	}

	private static void processCluster(boolean spr, long baseAddress, TreeGroup obj_group,
			TreeRegisterGroup obj_registergroup, Element cluster, String suffix) throws ParseException {

		String name = attr(cluster, "name", () -> new ParseException("cluster requires name", 1));
		String addressOffsetValue = attr(cluster, "addressOffset", //$NON-NLS-1$
				() -> new ParseException("cluster requires addressOffset", 1));

		long addressOffset;
		if (addressOffsetValue.startsWith("0x")) {//$NON-NLS-1$
			addressOffset = Long.parseLong(addressOffsetValue.substring(2), 16);
		} else {
			// handle missing 0x ... grrrr
			addressOffset = Long.parseLong(addressOffsetValue);
		}

		String dimValue = attr(cluster, "dim", () -> new ParseException("cluster requires dim", 1)); //$NON-NLS-1$

		int dim = Integer.parseInt(dimValue, 10);
		Element attr_dimIncrement = cluster.getChild("dimIncrement"); //$NON-NLS-1$
		int dimIncrement;
		// TODO: handle numbers with 0x
		if (attr_dimIncrement != null) {
			if (attr_dimIncrement.getValue().startsWith("0x")) { //$NON-NLS-1$
				dimIncrement = Integer.parseInt(attr_dimIncrement.getValue().substring(2), 16);
			} else {
				dimIncrement = Integer.parseInt(attr_dimIncrement.getValue(), 10);
			}
		} else {
			dimIncrement = 0;
		}
		Element dimIndex = cluster.getChild("dimIndex"); //$NON-NLS-1$
		List<String> indexes;
		if (dimIndex == null) {
			indexes = IntStream.range(0, dim).boxed().map(i -> String.valueOf(i)).collect(Collectors.toList());
		} else {
			DimIndexType dimIndexType = DimIndexType.parseDimIndexType(dimIndex.getValue());
			indexes = dimIndexType.getIndexes();
			if (indexes.size() != dim) {
				throw new ParseException(NLS.bind("Register {0} dimension {1} is not equal to index size {2}",
						new Object[] { name, dim, indexes.size() }), 1);
			}
		}
		for (String index : indexes) {
			processRegistersOrClusters(spr, baseAddress + addressOffset, obj_group, obj_registergroup,
					cluster.getChildren(), getSuffix(suffix, String.format(name, index)));
			baseAddress += dimIncrement;
		}

	}

	private static String getSuffix(String suffix, String name) {
		if (suffix.isEmpty()) {
			return name;
		}
		return suffix + " " + name; //$NON-NLS-1$
	}

	private static void processRegister(final boolean spr, long baseAddress, TreeGroup obj_group,
			TreeRegisterGroup obj_registergroup, Element register, String suffix) throws ParseException {
		// Mandatory attribute name
		String name = attr(register, "name", () -> new ParseException("register requires name", 1));
		// Optional attribute description
		String description = attr(register, "description", "");

		// Mandatory attribute address
		String addressOffsetValue = attr(register, "addressOffset", //$NON-NLS-1$
				() -> new ParseException("register requires addressOffset", 1));

		long addressOffset;
		if (addressOffsetValue.startsWith("0x")) {//$NON-NLS-1$
			addressOffset = Long.parseLong(addressOffsetValue.substring(2), 16);
		} else {
			// handle missing 0x ... grrrr
			addressOffset = Long.parseLong(addressOffsetValue);
		}

		// Optional attribute resetvalue
		Element resetValueElement = register.getChild("resetValue"); //$NON-NLS-1$
		long resetValue = 0x00000000;
		if (resetValueElement != null && !resetValueElement.getValue().equals("")) {//$NON-NLS-1$
			try {
				if (resetValueElement.getValue().startsWith("0x")) {//$NON-NLS-1$
					resetValue = Long.parseLong(resetValueElement.getValue().substring(2), 16);
				} else {
					resetValue = Long.parseLong(resetValueElement.getValue());
				}
			} catch (Exception e) {
				// just keep initialized value 0x00000000
				Activator.log(0, "Initialized by default", e);
			}
		}

		// Optional attribute access
		String access = attr(register, "access", "RO");

		// TODO: map access to RO/RW/...
		access = ACCESS_MAP.getOrDefault(access, access);

		// Optional attribute size (in byte)
		Element sizeElement = register.getChild("size"); //$NON-NLS-1$
		int size;
		if (sizeElement != null) {
			if (sizeElement.getValue().startsWith("0x")) {//$NON-NLS-1$
				size = Integer.parseInt(sizeElement.getValue().substring(2), 16) / 8;
			} else {
				size = Integer.parseInt(sizeElement.getValue()) / 8;
			}
		} else {
			size = 4;
		}

		TreeParent treeParent = spr ? obj_group : obj_registergroup;

		Element dimElement = register.getChild("dim"); //$NON-NLS-1$

		if (dimElement == null) {
			processRegister(treeParent, register, getRegisterName(name, suffix), description,
					baseAddress + addressOffset, resetValue, access, size, spr);
			return;
		}

		int rdim = Integer.parseInt(dimElement.getValue(), 10);
		Element dimIncrementElement = register.getChild("dimIncrement"); //$NON-NLS-1$

		int dimIncrement;
		// TODO: handle numbers with 0x
		if (dimIncrementElement != null) {
			if (dimIncrementElement.getValue().startsWith("0x")) { //$NON-NLS-1$
				dimIncrement = Integer.parseInt(dimIncrementElement.getValue().substring(2), 16);
			} else {
				dimIncrement = Integer.parseInt(dimIncrementElement.getValue(), 10);
			}
		} else {
			dimIncrement = 0;
		}
		Element dimIndex = register.getChild("dimIndex"); //$NON-NLS-1$
		List<String> indexes;
		if (dimIndex == null) {
			indexes = IntStream.range(0, rdim).boxed().map(i -> String.valueOf(i)).collect(Collectors.toList());
		} else {
			DimIndexType dimIndexType = DimIndexType.parseDimIndexType(dimIndex.getValue());

			indexes = dimIndexType.getIndexes();
			if (indexes.size() != rdim) {
				throw new ParseException(NLS.bind("Register {0} dimension {1} is not equal to index size {2}",
						new Object[] { name, rdim, indexes.size() }), 1);
			}
		}

		for (String index : indexes) {
			processRegister(treeParent, register, getRegisterName(String.format(name, index), suffix), description,
					baseAddress + addressOffset, resetValue, access, size, spr);
			baseAddress += dimIncrement;
		}

	}

	private static String getRegisterName(String name, String suffix) {
		if (suffix.isEmpty()) {
			return name;
		}
		return name + " (" + suffix + ")";
	}

	private static void processRegister(TreeParent treeParent, Element register, String rname, String rdescription,
			long address, long rresetvalue, String raccess, int rsize, final boolean spr) throws ParseException {

		TreeRegister obj_register = new TreeRegister(rname, rdescription, address, rresetvalue, raccess, rsize, spr);
		treeParent.addChild(obj_register);

		SortingOrder.BitFieldSorter sorter = SortingOrder.getFromPreference();
		processFields(register, obj_register, sorter);
	}

	private static void processFields(Element register, TreeRegister obj_register, SortingOrder.BitFieldSorter sorter)
			throws ParseException {
		Element fields_element = register.getChild("fields"); //$NON-NLS-1$
		if (fields_element != null) {
			List<TreeField> bitFieldsList = new ArrayList<TreeField>();

			final boolean isLittleEndianSorting = sorter.isLittleEndian();
			for (Element field : fields_element.getChildren("field")) {
				final TreeField obj_field = createBitField(field, isLittleEndianSorting);
				bitFieldsList.add(obj_field);
			}
			//
			obj_register.addBitFields(bitFieldsList, sorter);
		}
	}

	/**
	 * 
	 * */
	private static TreeField createBitField(Element field, final boolean isLittleEndianSorting) throws ParseException {
		// Mandatory attribute name
		final String fname = attr(field, "name", () -> new ParseException("field requires name", 1));

		// Optional attribute description
		final String fdescription = attr(field, "description", "");

		// Optional attribute access
		String fAccess = attr(field, "access", "");
		fAccess = ACCESS_MAP.getOrDefault(fAccess, "RW");

		// Mandatory attribute bitoffset
		Element attr_fbitoffset = field.getChild("bitOffset"); //$NON-NLS-1$
		byte fbitoffset = 0;
		byte fbitlength = 0;
		if (attr_fbitoffset != null) {
			fbitoffset = Byte.parseByte(attr_fbitoffset.getValue());

			// Mandatory attribute bitlength
			Element attr_fbitlength = field.getChild("bitWidth"); //$NON-NLS-1$

			if (attr_fbitlength != null)
				fbitlength = Byte.parseByte(attr_fbitlength.getValue());
			else
				throw new ParseException("field requires bitlength", 1); //$NON-NLS-1$
		} else {
			Element attr_fbitrange = field.getChild("bitRange"); //$NON-NLS-1$

			if (attr_fbitrange != null) {
				String range_string = attr_fbitrange.getValue();
				String range = range_string.substring(1, range_string.length() - 1);

				String[] rangeArray = range.split(":"); //$NON-NLS-1$
				// so 0:0 means
				// [endoffset:startoffset] ->
				// offset=startoffset
				// length=endoffset-startoffset+1 ->
				// offset 0 ...length 1
				fbitoffset = Byte.valueOf(rangeArray[1]);
				fbitlength = (byte) (Byte.valueOf(rangeArray[0]) - Byte.valueOf(rangeArray[1]) + 1);
			} else {
				Element element_lsb = field.getChild("lsb"); //$NON-NLS-1$
				Element element_msb = field.getChild("msb"); //$NON-NLS-1$
				if (element_lsb != null && element_msb != null) {
					fbitoffset = Byte.valueOf(element_lsb.getValue());
					fbitlength = (byte) (Byte.valueOf(element_msb.getValue()) - Byte.valueOf(element_lsb.getValue()));
				} else
					throw new ParseException("field requires some sort of start-end bit marker", 1); //$NON-NLS-1$
			}

		}

		Interpretations interpretations = new Interpretations();

		Element enumeratedValues = field.getChild("enumeratedValues"); //$NON-NLS-1$
		if (enumeratedValues != null) {
			// $NON-NLS-1$
			// Mandatory attribute value
			for (Element interpretation : enumeratedValues.getChildren("enumeratedValue")) {
				Element attr_key = interpretation.getChild("value"); //$NON-NLS-1$
				long key;
				if (attr_key != null) {
					key = extractKey(attr_key.getValue());
				} else {
					throw new ParseException("enumeratedValue requires value", 1); //$NON-NLS-1$
				}

				// Mandatory attribute name
				String name = attr(interpretation, "name",
						() -> new ParseException("enumeratedValue requires name", 1));

				// Mandatory/Optional attribute
				// description
				// Optional for SiliconLabortories
				// SVD Files .. they sometimes miss
				// descriptions
				String text = attr(interpretation, "description", name);
				// throw new
				// ParseException("enumeratedValue
				// requires description", 1);

				// filter non-ascii chars and more than 1 space
				text = text.replaceAll("(\\r|\\n|\\t| (?= ))", "");
				interpretations.addInterpretation(key, name + ": " + text); //$NON-NLS-1$
			}
		}

		return new TreeField(fname, fdescription, fbitoffset, fbitlength, interpretations, fAccess,
				isLittleEndianSorting);
	}

	/** Return key by parsing prefix '0x'--'#'--none */
	private static long extractKey(final String attrKey) {
		long key = -1;
		if (attrKey.startsWith("0x")) //$NON-NLS-1$
			key = Long.valueOf(attrKey.substring(2), 16);
		else if (attrKey.startsWith("#")) { //$NON-NLS-1$
			// Substitue x in #1xx
			// binary patterns
			String str = attrKey.substring(1);
			str = str.replace('x', '0');
			str = str.replace('X', '0');
			key = Long.valueOf(str, 2);
		} else {
			key = Long.valueOf(attrKey);
		}
		return key;
	}

	/**
	 * reads long from string with pattern
	 * value="[+]?(0x|0X|#)?[0-9a-fA-F]+[kmgtKMGT]?".
	 */
	private static long readScaledNonNegativeInteger(final String constructionName, String value)
			throws ParseException {
		Scale scale = readScale(value);
		if (scale == Scale.UNKNOWN) {
			final String formats = "Hex/Dec"; //$NON-NLS-1$
			String[] bindings = { value, constructionName, formats };
			String message = NLS.bind(Messages.unsupported_format_message, bindings);
			throw new ParseException(message, 1);
		}
		// remove scale symbol for scaled value
		if (scale != Scale.ONE) {
			value = value.substring(0, value.length() - 1);
		}

		long result = 0;
		//
		if (value.toUpperCase().startsWith(HEX_PREFIX)) {
			// hexadecimal
			result = Long.parseLong(value.substring(2), 16);
		} else if (value.charAt(0) == '#') {
			// binary with #
			result = Long.parseLong(value.substring(1), 2);
		} else {
			// decimal with no prefix
			result = Long.parseLong(value);
		}

		return result * scale.getMagnifier();
	}

	/** */
	private static Scale readScale(final String value) {
		if (value.length() == 0) {
			return Scale.UNKNOWN;
		}
		if (value.toUpperCase().endsWith(SCALE_SUFFIX)) {
			char ch = value.toUpperCase().charAt(value.length());
			if (ch == 'K') {
				return Scale.KILO;
			} else if (ch == 'M') {
				return Scale.MEGA;
			} else if (ch == 'G') {
				return Scale.GIGA;
			} else if (ch == 'T') {
				return Scale.TERA;
			} else if (ch == 'B') {
				return Scale.UNKNOWN;
			}

		}
		return Scale.ONE;
	}

	/*
	 * Parse an embsysregview xml file
	 */
	private TreeParent parseXML(Element root, TreeParent tempRoot, boolean spr) throws ParseException {

		List<Element> grouplist = root.getChildren("group"); //$NON-NLS-1$

		for (Element group : grouplist) {

			// Mandatory attribute name
			Attribute attr_gname = group.getAttribute("name"); //$NON-NLS-1$
			String gname;
			if (attr_gname != null) {
				gname = attr_gname.getValue();
			} else {
				throw new ParseException("group requires name", 1); //$NON-NLS-1$
			}

			// Optional attribute description
			Attribute attr_gdescription = group.getAttribute("description"); //$NON-NLS-1$
			String gdescription;
			if (attr_gdescription != null) {
				gdescription = attr_gdescription.getValue();
			} else {
				gdescription = ""; //$NON-NLS-1$
			}

			TreeGroup obj_group = new TreeGroup(gname, gdescription);
			tempRoot.addChild(obj_group);

			List<Element> registergrouplist = group.getChildren();
			for (Element registergroup : registergrouplist) {
				// Mandatory attribute name
				Attribute attr_rgname = registergroup.getAttribute("name"); //$NON-NLS-1$
				String rgname;
				if (attr_rgname != null)
					rgname = attr_rgname.getValue();
				else
					throw new ParseException("registergroup requires name", 1); //$NON-NLS-1$

				// Optional attribute description
				Attribute attr_rgdescription = registergroup.getAttribute("description"); //$NON-NLS-1$
				String rgdescription;
				if (attr_rgdescription != null)
					rgdescription = attr_rgdescription.getValue();
				else
					rgdescription = ""; //$NON-NLS-1$

				TreeRegisterGroup obj_registergroup = new TreeRegisterGroup(rgname, rgdescription);
				obj_group.addChild(obj_registergroup);

				// --Iterate registers in group
				List<Element> registerlist = registergroup.getChildren();
				for (Element register : registerlist) {
					// Mandatory attribute name
					String rname;
					Attribute attr_rname = register.getAttribute("name"); //$NON-NLS-1$
					if (attr_rname != null) {
						rname = attr_rname.getValue();
					} else {
						throw new ParseException("register requires name", 1); //$NON-NLS-1$
					}

					// Optional attribute description
					Attribute attr_rdescription = register.getAttribute("description"); //$NON-NLS-1$
					String rdescription;
					if (attr_rdescription != null)
						rdescription = attr_rdescription.getValue();
					else
						rdescription = ""; //$NON-NLS-1$

					// Mandatory attribute address
					Attribute attr_roffsetaddress = register.getAttribute("address"); //$NON-NLS-1$
					long raddress;
					if (attr_roffsetaddress != null)
						raddress = Long.parseLong(attr_roffsetaddress.getValue().substring(2), 16);
					else
						throw new ParseException("register requires address", 1); //$NON-NLS-1$

					// Optional attribute resetvalue
					Attribute attr_rresetvalue = register.getAttribute("resetvalue"); //$NON-NLS-1$
					long rresetvalue;
					if (attr_rresetvalue != null && attr_rresetvalue.getValue() != "") //$NON-NLS-1$
						rresetvalue = Long.parseLong(attr_rresetvalue.getValue().substring(2), 16);
					else
						rresetvalue = 0x00000000;

					// Optional attribute access
					Attribute attr_raccess = register.getAttribute("access"); //$NON-NLS-1$
					String raccess;
					if (attr_raccess != null)
						raccess = attr_raccess.getValue();
					else
						raccess = "RO"; //$NON-NLS-1$

					// Optional attribute size (in byte)
					Attribute attr_rsize = register.getAttribute("size"); //$NON-NLS-1$
					int rsize;
					if (attr_rsize != null)
						rsize = Integer.parseInt(attr_rsize.getValue());
					else
						rsize = 4;

					TreeRegister obj_register = new TreeRegister(rname, rdescription, raddress, rresetvalue, raccess,
							rsize, spr);
					obj_registergroup.addChild(obj_register);

					List<TreeField> bitFieldsList = new ArrayList<TreeField>();

					Element register_description = register.getChild("description"); //$NON-NLS-1$
					if (register_description != null) {
						rdescription = register_description.getText();
						obj_register.setDescription(rdescription);
					}

					SortingOrder.BitFieldSorter sorter = SortingOrder.getFromPreference();
					List<Element> fieldlist = register.getChildren("field"); //$NON-NLS-1$
					for (Element field : fieldlist) {
						// Optional attribute board_id
						Attribute attr_fboard_id = field.getAttribute("board_id"); //$NON-NLS-1$
						String fboard_id;
						if (attr_fboard_id != null)
							fboard_id = attr_fboard_id.getValue();
						else
							fboard_id = ""; //$NON-NLS-1$

						// only digg deeper if field is going to be
						// added

						if (fboard_id.equals("") || fboard_id.equals(store_board)) { //$NON-NLS-1$
							// Mandatory attribute name
							Attribute attr_fname = field.getAttribute("name"); //$NON-NLS-1$
							String fname;
							if (attr_fname != null)
								fname = attr_fname.getValue();
							else
								throw new ParseException("field requires name", 1); //$NON-NLS-1$

							// Optional attribute description
							Attribute attr_fdescription = field.getAttribute("description"); //$NON-NLS-1$
							String fdescription;
							if (attr_fdescription != null)
								fdescription = attr_fdescription.getValue();
							else
								fdescription = ""; //$NON-NLS-1$

							// Mandatory attribute bitoffset
							Attribute attr_fbitoffset = field.getAttribute("bitoffset"); //$NON-NLS-1$
							byte fbitoffset;
							if (attr_fbitoffset != null)
								fbitoffset = Byte.parseByte(attr_fbitoffset.getValue());
							else
								throw new ParseException("field requires bitoffset", 1); //$NON-NLS-1$

							// Mandatory attribute bitlength
							Attribute attr_fbitlength = field.getAttribute("bitlength"); //$NON-NLS-1$
							byte fbitlength;
							if (attr_fbitlength != null)
								fbitlength = Byte.parseByte(attr_fbitlength.getValue());
							else
								throw new ParseException("field requires bitlength", 1); //$NON-NLS-1$

							Element field_description = field.getChild("description"); //$NON-NLS-1$
							if (field_description != null)
								fdescription = field_description.getText();

							Interpretations interpretations = new Interpretations();
							List<Element> interpretationlist = field.getChildren("interpretation"); //$NON-NLS-1$
							for (Element interpretation : interpretationlist) {
								// Mandatory attribute key
								Attribute attr_key = interpretation.getAttribute("key"); //$NON-NLS-1$
								String skey;
								long key;
								if (attr_key != null) {
									skey = attr_key.getValue();
									if (skey.startsWith("0x")) //$NON-NLS-1$
										key = Long.parseLong(skey.substring(2), 16);
									else
										key = Long.parseLong(skey);
								} else
									throw new ParseException("interpretation requires key", 1); //$NON-NLS-1$

								// Mandatory attribute text
								Attribute attr_text = interpretation.getAttribute("text"); //$NON-NLS-1$
								String text;
								if (attr_text != null)
									text = attr_text.getValue();
								else
									throw new ParseException("interpretation requires text", 1); //$NON-NLS-1$

								interpretations.addInterpretation(key, text);
							}

							TreeField obj_field = new TreeField(fname, fdescription, fbitoffset, fbitlength,
									interpretations, "RW", false);
							bitFieldsList.add(obj_field);
						}

						obj_register.addBitFields(bitFieldsList, sorter);
						//
						obj_registergroup.extractRanges();
					}

				}

			}
		}

		return tempRoot;
	}

	/*
	 * public boolean isReadWrite() { return
	 * (getType().toUpperCase().equals("RW") ||
	 * getType().toUpperCase().equals("RW1") ||
	 * getType().toUpperCase().equals("RW1C") ||
	 * getType().toUpperCase().equals("RW1S") ||
	 * getType().toUpperCase().equals("RWH")); }
	 * 
	 * public boolean isReadOnly() { return
	 * (getType().toUpperCase().equals("RO") ||
	 * getType().toUpperCase().equals("RC") ||
	 * getType().toUpperCase().equals("R")); }
	 * 
	 * public boolean isWriteOnly() { return
	 * (getType().toUpperCase().equals("WO") ||
	 * getType().toUpperCase().equals("W") ||
	 * getType().toUpperCase().equals("W1C") ||
	 * getType().toUpperCase().equals("W1S") ||
	 * getType().toUpperCase().equals("W1")); }
	 */
}
