/*******************************************************************************
 * Copyright 2019 NXP
 *******************************************************************************/
package org.eclipse.cdt.embsysregview.internal;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.osgi.util.NLS;

public class DimIndexType {

	private static final String RANGE_SEPARATOR = "-"; //$NON-NLS-1$
	private static final String LIST_SEPARATOR = ","; //$NON-NLS-1$

	private static final ThreadLocal<Matcher> THREAD_LOCAL = ThreadLocal.withInitial(//
			() -> Pattern.compile("([0-9]+\\-[0-9]+)|([A-Z]\\-[A-Z])|([_0-9a-zA-Z]+(,\\s*[_0-9a-zA-Z]+)+)").matcher("") //$NON-NLS-1$ //$NON-NLS-2$
	);

	private final List<String> indexes;

	public DimIndexType(List<String> indexes) {
		this.indexes = Collections.unmodifiableList(indexes);
	}

	public List<String> getIndexes() {
		return indexes;
	}

	public static DimIndexType parseDimIndexType(String value) throws ParseException {
		if (value == null || value.trim().isEmpty()) {
			throw new ParseException(
					NLS.bind("{0} value should not be empty", DimIndexType.class.getSimpleName()),1);
		}
		Matcher matcher = THREAD_LOCAL.get();
		matcher.reset(value);
		if (!matcher.matches()) {
			throw new ParseException(
					NLS.bind("{0}  value should match {1}", DimIndexType.class.getSimpleName(),
							"[0-9]+\\-[0-9]+|[A-Z]-[A-Z]|[_0-9a-zA-Z]+(,\\s*[_0-9a-zA-Z]+)+"),1); //$NON-NLS-1$
		}
		List<String> indexes = new ArrayList<>();

		extractNumberRange(indexes, matcher.group(1));
		extractCharRange(indexes, matcher.group(2));
		extractList(indexes, matcher.group(3));
		return new DimIndexType(indexes);
	}

	private static void extractNumberRange(List<String> indexes, String value) throws ParseException {
		if (value == null) {
			return;
		}
		String[] range = value.split(RANGE_SEPARATOR);
		int start = Integer.parseInt(range[0]);
		int end = Integer.parseInt(range[1]);
		checkRange(start, end);
		// inclusive range
		for (int i = start; i <= end; i++) {
			indexes.add(String.valueOf(i));
		}
	}

	private static void extractCharRange(List<String> indexes, String value) throws ParseException {
		if (value == null) {
			return;
		}
		String[] range = value.split(RANGE_SEPARATOR);
		char start = range[0].charAt(0);
		char end = range[1].charAt(0);
		checkRange(start, end);
		// inclusive range
		for (char c = start; c <= end; c++) {
			indexes.add(String.valueOf(c));
		}
	}

	private static void extractList(List<String> indexes, String value) {
		if (value == null) {
			return;
		}
		String[] indices = value.split(LIST_SEPARATOR);
		for (int i = 0; i < indices.length; ++i) {
			indexes.add(indices[i].trim());
		}
	}

	private static void checkRange(int start, int end) throws ParseException {
		if (end < start) {
			throw new ParseException("range start should be less than range end",0);
		}
	}

	public String printDimIndexType() {
		return String.join(LIST_SEPARATOR, indexes);
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((indexes == null) ? 0 : indexes.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		DimIndexType other = (DimIndexType) obj;
		if (indexes == null) {
			if (other.indexes != null)
				return false;
		} else if (!indexes.equals(other.indexes))
			return false;
		return true;
	}
	
	@Override
	public String toString() {
		return printDimIndexType();
	}

}
