package org.eclipse.cdt.embsysregview.core;

import java.util.HashMap;
import java.util.Map;
import java.util.stream.Stream;

import org.eclipse.cdt.dsf.concurrent.ConfinedToDsfExecutor;
import org.eclipse.cdt.dsf.concurrent.DsfRunnable;
import org.eclipse.cdt.dsf.datamodel.IDMContext;
import org.eclipse.cdt.dsf.gdb.launching.GdbLaunch;
import org.eclipse.cdt.dsf.service.DsfSession;
import org.eclipse.cdt.embsysregview.Activator;
import org.eclipse.cdt.embsysregview.Messages;
import org.eclipse.cdt.embsysregview.internal.model.EmbSysRegModel;
import org.eclipse.cdt.embsysregview.internal.model.TreeElement;
import org.eclipse.cdt.embsysregview.internal.model.TreeField;
import org.eclipse.cdt.embsysregview.internal.model.TreeParent;
import org.eclipse.cdt.embsysregview.internal.model.TreeRange;
import org.eclipse.cdt.embsysregview.internal.model.TreeRegister;
import org.eclipse.cdt.embsysregview.internal.model.TreeRegisterGroup;
import org.eclipse.cdt.embsysregview.internal.ui.view.EmbSysRegViewer;
import org.eclipse.cdt.embsysregview.internal.utils.DSFAccessor;
import org.eclipse.cdt.embsysregview.internal.utils.DSFDebugModelListener;
import org.eclipse.cdt.embsysregview.internal.utils.DSFSessionState;
import org.eclipse.cdt.embsysregview.internal.utils.EmbSysRegViewDsfEventListener;
import org.eclipse.cdt.embsysregview.internal.utils.GUIUtils;
import org.eclipse.cdt.embsysregview.views.EmbSysRegView.RegSet;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationType;
import org.eclipse.debug.core.model.MemoryByte;
import org.eclipse.debug.ui.DebugUITools;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;

/**
 * Manage all the ways from model to presentation and vice versa.
 * */
public class EmbSysRegEngine implements DSFDebugModelListener, ISelectionChangedListener {

    /**
     * The data model drawn by this visualizer.
     */
    protected final EmbSysRegModel fDataModel;

    /** EmbSysReg viewer to automate. */
    protected final EmbSysRegViewer m_regViewer;

    /** DSF debug context session object. */
    private DSFSessionState m_sessionState;

    /** Event listener class for DSF events */
    protected final EmbSysRegViewDsfEventListener fEventListener;

    /** Hidden */
    private TreeViewer m_control;

    private Map<String, Boolean> disableAccessorReadRegisters = new HashMap<>();

    /** 
     * 
     * */
    class ViewContentFilter extends ViewerFilter {

        @Override
        public boolean select(Viewer viewer, Object parentElement, Object element) {
            return true;
        }

    }

    /**
     * This is the Content Provider that present the Static Model to the
     * TreeViewer
     */
    class ViewContentProvider implements // IStructuredContentProvider,
            ITreeContentProvider {

        public ViewContentProvider() {

            // Keep for now if will need to implement possibility to configure
            // EmbSysRegView both in project settings and in global preferences,
            // inheritance of global preferences settings by newly created
            // projects,
            // and possibility to choose what settings to use
            // initialize(null);
            //
            // Activator.getDefault().getPreferenceStore().addPropertyChangeListener(new
            // IPropertyChangeListener(){
            //
            //
            // @Override
            // public void propertyChange(PropertyChangeEvent event) {
            // // only rebuild tree on chip/board change
            // if(event.getProperty().equals(ARCHITECTURE) ||
            // event.getProperty().equals(VENDOR) ||
            // event.getProperty().equals(CHIP) ||
            // event.getProperty().equals(BOARD))
            // {
            // initialize(null);
            // viewer.setInput(invisibleRoot);
            // viewer.refresh();
            // updateInfoLabel(null);
            // }
            // }});
        }

        @Override
		public void inputChanged(Viewer v, Object oldInput, Object newInput){
			if (newInput == null) {
				fDataModel.initialize();
			} else if (newInput instanceof IProject) {
				fDataModel.initialize((IProject) newInput);
			}
			else{
				return;
			}
			updateRegViewerInfoLabel();		
		}

        private void updateRegViewerInfoLabel() {
        	Display.getDefault().asyncExec(new Runnable() {
				
				@Override
				public void run() {
					 m_regViewer.updateInfoLabel();
				}
			});
        }

        @Override
        public void dispose() {
        }

        @Override
        public Object[] getElements(Object parent) {
            return getChildren(parent);
        }

        @Override
        public Object getParent(Object child) {
            if (child instanceof TreeElement) {
                return ((TreeElement) child).getParent();
            }
            return null;
        }

        @Override
        public Object[] getChildren(Object parent) {
            if (parent instanceof TreeParent) {
                return ((TreeParent) parent).getChildren();
            }
            return new Object[0];
        }

        @Override
        public boolean hasChildren(Object parent) {
            if (parent instanceof TreeParent)
                return ((TreeParent) parent).hasChildren();
            return false;
        }
    }

    /**
     * 
     * */
    public EmbSysRegEngine(EmbSysRegViewer embSysRegViewer, RegSet regSet) {
        fEventListener = new EmbSysRegViewDsfEventListener(this);
        fDataModel = Activator.getInstance(regSet);
        m_regViewer = embSysRegViewer;
    }

    /**
     * 
     * */
    public TreeViewer createTreeViewer(Composite parent) {
        if (m_control == null) {
            m_control = new TreeViewer(parent, SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION);
            m_control.getControl().setLayoutData("height 100%,width 100%,hmin 0,wmin 0"); //$NON-NLS-1$
            m_control.getTree().setLinesVisible(true);
            m_control.getTree().setHeaderVisible(true);
            m_control.addSelectionChangedListener(this);
            m_control.setContentProvider(new ViewContentProvider());
            m_control.addFilter(new ViewContentFilter());                    
            m_regViewer.updateInfoLabel();
            setTreeViewModel(fDataModel.getRoot());
        }
        return m_control;
    }

    /**
     * Updates debug context being displayed by EmbSysRegTree. Returns true if
     * canvas context actually changes, false if not.
     */
    public synchronized boolean updateDebugContext() {
        final DsfSession session = getSessionContext();
        if (session != null) {
            return setSessionTrack(session);
        }
        // no DSF session
        return true;
    }

    /**  */
    private static DsfSession getSessionContext() {
    	IAdaptable dbgContext = DebugUITools.getDebugContext();
    	if (dbgContext != null) {
    	   final ILaunch launch = (ILaunch) dbgContext.getAdapter(ILaunch.class);    	
    	     if (launch instanceof GdbLaunch) {
    	       //GdbLaunch -> DsfSession
        	   return ((GdbLaunch)launch).getSession();    		         		         	
    	     }
    	}
    	return null;    	
    }

    /** */
    public final EmbSysRegModel getModel() {
        return fDataModel;
    }

    /**
     * Sets debug context being displayed by TreeView. Returns true if TreeView
     * context actually changes, false if not.
     */
    public boolean setSessionTrack(final DsfSession context) {
        boolean changed = false;
        final String sessionId = context.getId();
        if (m_sessionState != null && !m_sessionState.getSessionID().equals(sessionId)) {
            m_sessionState.removeServiceEventListener(fEventListener);
            m_sessionState.dispose();
            m_sessionState = null;
            changed = true;
        }

        if (m_sessionState == null && sessionId != null) {
            m_sessionState = new DSFSessionState(context);
            m_sessionState.addServiceEventListener(fEventListener);
            changed = true;
        }

        if (changed)
            update();

        return changed;
    }

    /** Updates EmbSys tree state. */
    private void update() {
        // Create new EmbSysModel and hand it to TreeView.
        if (m_sessionState == null) {
            // no state to display, we can immediately clear the table
            setTreeViewModel(null);
            fDataModel.setRootReadOnly(true);
            return;
        }
        fDataModel.setRootReadOnly(false);
    }

    /** */
    private void setTreeViewModel(TreeParent model) {
        m_control.setInput(model);
    }

    /**
     * 
     * */
    public void getEmbSysRegUpdate(final IDMContext context, final boolean onlyOne) {
        if (m_sessionState != null) {
            if (isDisableAccessorReadRegisters(context)) {
                return;
            }
            m_sessionState.execute(new DsfRunnable() {
                @Override
                public void run() {
                    // get model asynchronously, and update canvas
                    // in getEmbSysModelDone().
                    if (onlyOne) {
                        getSingleRegUpdate(context);
                    } else {
                        if (fDataModel.getRegSet().isSPR()) {
                            getCherryPickedRegUpdate(context);
                        } else {
                            getRangedRegUpdate(context);
                        }
                    }

                }
            });
        }
    }

    public void removeFlagDisableAcsessorReadRegisters() {
        disableAccessorReadRegisters.remove(getSessionTrackId());
    }

    private static boolean disableAcsessorReadRegisters(IDMContext context) {
		final String[] lcTypeBlackList = new String[]{"org.eclipse.cdt.launch.remoteApplicationLaunchType", "org.eclipse.cdt.launch.applicationLaunchType"};
		final String lcTypeS32Debugger = "com.nxp.s32ds.debug.ide.s32debugger.core.s32DebuggerLaunchConfigurationType";
		final String lcS32DebbugerCoreAtribute = "com.nxp.s32ds.debug.ide.core.core";
		final String[] disableAttributeValues = new String[] { "APEX", "APU", "ISP" };

		if (context instanceof IAdaptable) {
			ILaunch lanch = ((IAdaptable) context).getAdapter(ILaunch.class);
			if (lanch != null) {
				ILaunchConfiguration lc = lanch.getLaunchConfiguration();
				if (lc != null) {
					try {
						ILaunchConfigurationType lcType = lc.getType();
						if (Stream.of(lcTypeBlackList).anyMatch(p -> lcType.getIdentifier().equals(p))) {
							return true;
						}
						if (lcTypeS32Debugger.equals(lcType.getIdentifier())) {
							String coreAttribute = lcType.getAttribute(lcS32DebbugerCoreAtribute);
							if (coreAttribute != null && !coreAttribute.isEmpty()) {
                                return Stream.of(disableAttributeValues)
										.anyMatch(p -> lcType.getAttribute(lcS32DebbugerCoreAtribute).contains(p));
							}
						}
					} catch (CoreException e) {
						Activator.getDefault().getLog().log(e.getStatus());
					}
				}
			}
		}
		return false;
	}
    
    /**
     * Starts EmbSys register request. Calls getEmbSysDone() with the
     * constructed update.
     */
    @ConfinedToDsfExecutor("getSession().getExecutor()")
    protected void getSingleRegUpdate(IDMContext context) {
        final TreeRegister reg = fDataModel.getPendingRead();
        if (reg != null) {
            DSFAccessor.getEmbSysRegisterData(context, m_sessionState, reg, this, fDataModel);
        }

    }

    /**
     * Starts EmbSys ranged request. Calls getEmbSysDone() with the constructed
     * update.
     */
    @ConfinedToDsfExecutor("getSession().getExecutor()")
    protected void getRangedRegUpdate(IDMContext context) {
        fDataModel.createRangeList().stream()
                .forEach(range -> DSFAccessor.getEmbSysDeviceData(context, m_sessionState, range, this, fDataModel));
        m_regViewer.requestUpdate();
    }

    /** get TreeView */
    public final EmbSysRegViewer getEmbSysRegViewer() {
        return m_regViewer;
    }

    @Override
    public void getEmbSysRegisterDataDone(IStatus status, TreeRegister register, Object arg) {
        EmbSysRegModel model = (EmbSysRegModel) arg;
        assert model != null;

        if (status.isOK()) {
            model.pushRegister(register);
        } else {
            register.clearValue();
        }
        m_regViewer.requestUpdate();
    }

    /**
     * Invoked when EmbSysReg viewer control's selection changes.
     */
    @Override
    public void selectionChanged(SelectionChangedEvent arg0) {
        // nothing (not need yet)
    }

    /** */
    public final boolean isEmpty() {
        return fDataModel.isEmpty();
    }

    /**
     * 
     * */
    public void toggleOnLineState(final TreeRegister treeRegister) {
        if (treeRegister.isRetrievalActive()) {
            // unselect
            getModel().setRegisterOffLine(treeRegister);
            m_control.refresh(treeRegister);
            for (final TreeRegister reg: treeRegister.getAlterName()) {
            	m_control.refresh(reg);
            }
        } else if (!treeRegister.isWriteOnly()) {
            // select
            getModel().setRegisterLive(treeRegister);
            updateDebugContext();
            if (isSessionReady()) {
                getEmbSysRegUpdate(getDebugDmc(), true);
            }
        }
    }

    /** */
    private boolean isSessionReady() {
        return (m_sessionState != null);
    }

    /**  */
    public boolean setValue(final TreeRegister treeRegister, final long lvalue) {
        if (treeRegister.checkWriteModifiers(lvalue)) {
            updateDebugContext();
            if (isSessionReady()) {
                writeEmbSysReg(treeRegister, lvalue);
                return true;
            }
        }
        return false;
    }

    /**
     * 
     * */
    protected void writeEmbSysReg(final TreeRegister treeRegister, final long lvalue) {
        if (m_sessionState != null) {
        	final IDMContext context = getDebugDmc();
            fDataModel.setRegisterToWrite(treeRegister);
            final TreeRegister regHolder = new TreeRegister(treeRegister.getName(), null, treeRegister.getRegisterAddress(), 0,
                    treeRegister.getType(), treeRegister.getByteSize(), treeRegister.isSPR());
            regHolder.setValue(lvalue);
            m_sessionState.execute(new DsfRunnable() {
                @Override
                public void run() {
                    writeReg(context, regHolder);
                }
            });
        }
    }

    /**
     * Starts EmbSys write register request. Calls writeEmbSysDone() with the
     * constructed update.
     */
    @ConfinedToDsfExecutor("getSession().getExecutor()")
    protected void writeReg(IDMContext context, final TreeRegister regHolder) {
        // final TreeRegister reg = fDataModel.getPendingReg();
        if (regHolder != null) {
            DSFAccessor.writeEmbSysRegisterData(context, m_sessionState, regHolder, this, fDataModel);
        }
    }

    @Override
    public void writeEmbSysRegisterDataDone(final IStatus status, final TreeRegister regHolder, final Object arg) {
        EmbSysRegModel model = (EmbSysRegModel) arg;
        assert model != null;
        final TreeRegister reg = fDataModel.getPendingReg();
        if (reg != null) {
            if (status.isOK()) {
            	final long lvalue = regHolder.getValue();
                reg.setValue(lvalue);
                for (TreeRegister altReg: reg.getAlterName()) {
                	if (reg.getByteSize() == altReg.getByteSize()) {
                	  altReg.setValue(lvalue);
                	}
                }
            } else {
                reg.rollbackValue();
                for (TreeRegister altReg: reg.getAlterName()) {                	
                	  altReg.rollbackValue();                	
                }
            }
        }
        m_regViewer.requestUpdate();
    }

    /**
     * @param fvalue
     */
    public void updateRegisterByField(final TreeField registerField, final long fvalue) {
        final TreeRegister treeRegister = (TreeRegister) registerField.getParent();

        // calculate register value + modified field to write into register
        long rvalue = treeRegister.getValue();        
        long mask = registerField.getBitMask();
        rvalue = rvalue & (~mask); // clear field bits in register value
        long value = fvalue << registerField.getBitOffset(); // shift field value into its position
                                          // in the register
        value = value & mask; // just to be sure, cut everything but the field
        rvalue = rvalue | value; // blend the field value into the register
                                 // value
        setValue(treeRegister, rvalue);
    }

    @Override
    public void readEmbSysRangeDataDone(IStatus status, TreeRange range, MemoryByte[] bytes, Object arg) {
        if (status.isOK()) {
            EmbSysRegModel model = (EmbSysRegModel) arg;
            assert model != null;
            model.applyRangeData(range, bytes);
        }
    }

    public void restoreSelections() {
        // TODO Auto-generated method stub

    }

    public void toggleSelectedGroupOff(TreeRegisterGroup deviceEntry) {
        if (deviceEntry.isChecked()) {
            deviceEntry.setChecked(false);
            toggleSelectedGroupInternal(deviceEntry);
        }
    }

    public void toggleSelectedRegisterOff(TreeRegister register) {
        getModel().setRegisterOffLine(register);
    }

    /**
     * 
     * */
	public void toggleSelectGroup(TreeRegisterGroup deviceEntry) {
		deviceEntry.toggleChecked();
        toggleSelectedGroupInternal(deviceEntry);
    }


	/** Get debug context than called from UI part */
	private static IDMContext getDebugDmc() {
		Object context = DebugUITools.getDebugContext();		
		if (context instanceof IAdaptable) {
			return (IDMContext) ((IAdaptable) context).getAdapter(IDMContext.class);
		}
		return null;
	}

    /** */
	public String getSessionTrackId() {
		if (m_sessionState != null) {
			return m_sessionState.getSessionID();
		} 
		return null;
	}

    private void getCherryPickedRegUpdate(IDMContext context) {
        fDataModel.getCherryPicked().values().forEach(reg -> DSFAccessor.getEmbSysRegisterData(context, m_sessionState,
                reg, this, fDataModel));
    }

    private void toggleSelectedGroupInternal(TreeRegisterGroup deviceEntry) {
        deviceEntry.getModelChildren().stream()
                .filter(treeElementRange -> treeElementRange instanceof TreeRange)
                .flatMap(treeElementRange -> ((TreeRange) treeElementRange).getModelChildren().stream())
                .filter(treeElement -> treeElement instanceof TreeRegister)
                .map(treeElement -> (TreeRegister) treeElement)
                .filter(deviceRegister -> deviceRegister.isReadOnly() || deviceRegister.isReadWrite())
                .forEach(deviceRegister -> setRegisterState(deviceEntry, deviceRegister));

        if (deviceEntry.isChecked()) {
            updateDebugContext();
            if (isSessionReady()) {
                getEmbSysRegUpdate(getDebugDmc(), false);
            }
        } else {
            m_regViewer.requestUpdate();
        }
    }

    private void setRegisterState(TreeRegisterGroup deviceEntry, TreeRegister deviceRegister) {
        if (deviceEntry.isChecked() && !isDisableAccessorReadRegisters()) {
            getModel().setRegisterLive(deviceRegister);
        } else {
            getModel().setRegisterOffLine(deviceRegister);
        }
    }

    private boolean isDisableAccessorReadRegisters(IDMContext context) {
        if (getEmbSysRegViewer().isDisposed()) {
            return false;
        }
        String sessionTrackId = getSessionTrackId();
        if (disableAccessorReadRegisters == null) {
            return disableAcsessorReadRegisters(context);
        }
        Boolean isDisableAcsessorReadRegisters = disableAccessorReadRegisters.get(sessionTrackId);
        if (isDisableAcsessorReadRegisters == null) {
            isDisableAcsessorReadRegisters = disableAcsessorReadRegisters(context);
            disableAccessorReadRegisters.put(sessionTrackId, isDisableAcsessorReadRegisters);
            if (isDisableAcsessorReadRegisters) {
                MessageDialog.openInformation(GUIUtils.getShell(),
                        Messages.EmbSysRegEngine_Read_registers_restriction_rule_message_title,
                        Messages.EmbSysRegEngine_Read_registers_restriction_rule_message_text);
            }
        }
        return isDisableAcsessorReadRegisters;
    }

    protected boolean isDisableAccessorReadRegisters() {
        return isDisableAccessorReadRegisters(getDebugDmc());
    }

}
