/*******************************************************************************
 * Copyright (c) 2016 EmbSysRegView
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     ravenclaw78 - initial API and implementation
 *******************************************************************************/
package org.eclipse.cdt.embsysregview;

import org.eclipse.swt.graphics.Image;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.cdt.embsysregview.internal.IEmbSysImages;
import org.eclipse.cdt.embsysregview.internal.model.EmbSysRegModel;
import org.eclipse.cdt.embsysregview.views.EmbSysRegView.RegSet;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

/**
 * The activator class controls the plug-in life cycle
 */
public class Activator extends AbstractUIPlugin implements IEmbSysImages {

    // The plug-in ID
    public static final String PLUGIN_ID = "org.eclipse.cdt.embsysregview"; //$NON-NLS-1$

    // The shared instance
    private static Activator plugin;

    private IResourceChangeListener rcl = null;
    private static Map<RegSet, EmbSysRegModel> embSysRegModels = new HashMap<>();

    /**
     * The constructor
     */
    public Activator() {
        // By some distant memories this constructor needed for correct OSGI
        // functioning.
    }

    private static void setPlugin(final Activator activator) {
        plugin = activator;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.ui.plugin.AbstractUIPlugin#start(org.osgi.framework.
     * BundleContext)
     */
    @Override
    public void start(BundleContext context) throws Exception {
        super.start(context);
        setPlugin(this);
        IWorkspace workspace = ResourcesPlugin.getWorkspace();
        rcl = new IResourceChangeListener() {
            public void resourceChanged(IResourceChangeEvent event) {
                if (event == null || event.getDelta() == null) {
                    return;
                }
                int deltasLength = 0;
                IResourceDelta delta = event.getDelta();
                IResourceDelta[] resourceDeltas = delta.getAffectedChildren();
                if (resourceDeltas == null || (deltasLength = resourceDeltas.length) == 0) {
                    return;
                }
                for (int i = 0; i < deltasLength; i++) {
                    IResource target = resourceDeltas[i].getResource();
                    if (target instanceof IProject) {
                        IResourceDelta rd = resourceDeltas[i];
                        IProject prj = (IProject) target;
                        if (IResourceDelta.REMOVED == rd.getKind()) {
                            embSysRegModels.forEach((key, value) -> {
                                value.removeEntry(prj);
                            });
                        }
                    }
                }
            }
        };
        workspace.addResourceChangeListener(rcl);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.ui.plugin.AbstractUIPlugin#stop(org.osgi.framework.
     * BundleContext)
     */
    @Override
    public void stop(BundleContext context) throws Exception {
        setPlugin(null);
        super.stop(context);
        IWorkspace workspace = ResourcesPlugin.getWorkspace();
        workspace.removeResourceChangeListener(rcl);
    }

    /**
     * Returns the shared instance
     *
     * @return the shared instance
     */
    public static Activator getDefault() {
        return plugin;
    }

    public static void log(int severity, String message, Throwable exception) {
        Status status = new Status(severity, PLUGIN_ID, message, exception);
        plugin.getLog().log(status);
    }

    public static void log(int severity, String message) {
        Status status = new Status(severity, PLUGIN_ID, message);
        plugin.getLog().log(status);
    }

    public static void log(IStatus status) {
        plugin.getLog().log(status);
    }

    @Override
    public Image getImage(String key) {
        return getImageRegistry().get(key);
    }

    @Override
    public ImageDescriptor getImageDescriptor(String key) {
        return getImageRegistry().getDescriptor(key);
    }

    @Override
    protected void initializeImageRegistry(ImageRegistry reg) {
        reg.put(IMG_EMBSYS_SELECTED,
                AbstractUIPlugin.imageDescriptorFromPlugin(PLUGIN_ID, "icons/update_genericregister_obj.gif")); //$NON-NLS-1$
        reg.put(IMG_EMBSYS_UNSELECTED,
                AbstractUIPlugin.imageDescriptorFromPlugin(PLUGIN_ID, "icons/genericregister_obj.gif")); //$NON-NLS-1$
        reg.put(IMG_EMBSYS_SELECTED_FIELD,
                AbstractUIPlugin.imageDescriptorFromPlugin(PLUGIN_ID, "icons/selected_field.png")); //$NON-NLS-1$
        reg.put(IMG_EMBSYS_UNSELECTED_FIELD,
                AbstractUIPlugin.imageDescriptorFromPlugin(PLUGIN_ID, "icons/unselected_field.png")); //$NON-NLS-1$
        reg.put(IMG_EMBSYS_INFO, AbstractUIPlugin.imageDescriptorFromPlugin(PLUGIN_ID, "icons/info.png")); //$NON-NLS-1$
        reg.put(IMG_EMBSYS_INTERPRETATION,
                AbstractUIPlugin.imageDescriptorFromPlugin(PLUGIN_ID, "icons/interpretation.png")); //$NON-NLS-1$
        reg.put(IMG_EMBSYS_CONFIG_BUTTON, AbstractUIPlugin.imageDescriptorFromPlugin(PLUGIN_ID, "icons/config.png")); //$NON-NLS-1$
        reg.put(IMG_EMBSYS_REG_GROUP,
                AbstractUIPlugin.imageDescriptorFromPlugin(PLUGIN_ID, "icons/genericreggroup_obj.gif")); //$NON-NLS-1$
        reg.put(IMG_EMBSYS_REG_GROUP_SELECTED,
                AbstractUIPlugin.imageDescriptorFromPlugin(PLUGIN_ID, "icons/update_genericreggroup_obj.gif")); //$NON-NLS-1$
        reg.put(IMG_EMBSYS_TOOLBAR_FILTER, AbstractUIPlugin.imageDescriptorFromPlugin(PLUGIN_ID, "icons/vertical.gif")); //$NON-NLS-1$
        super.initializeImageRegistry(reg);
    }

    public static EmbSysRegModel getInstance(RegSet regSet) {
        if (!embSysRegModels.containsKey(regSet)) {
            embSysRegModels.put(regSet, new EmbSysRegModel(regSet));
        }
        return embSysRegModels.get(regSet);
    }
}
