/*
** ###################################################################
**     Processor:           S32V232
**     Reference manual:    S32V232RM Rev. 3 10/2017
**     Version:             rev. 2.0, 2018-03-21
**     Build:               b180627
**
**     Abstract:
**         Peripheral Access Layer for S32V232
**
**     Copyright 2016-2018 NXP
**     All rights reserved.
**
**     THIS SOFTWARE IS PROVIDED BY NXP "AS IS" AND ANY EXPRESSED OR
**     IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
**     OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
**     IN NO EVENT SHALL NXP OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
**     INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**     (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**     SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**     HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**     STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**     IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
**     THE POSSIBILITY OF SUCH DAMAGE.
**
**     http:                 www.nxp.com
**
**     Revisions:
**     - rev. 1.0 (2018-10-01) - Initial version, based on S32V232
** ###################################################################
*/

/*!
 * @file S32V232.h
 * @version 2.0
 * @date 2018-03-21
 * @brief Peripheral Access Layer for S32V232
 *
 * This file contains register definitions and macros for easy access to their
 * bit fields.
 *
 * This file assumes LITTLE endian system.
 */

/**
* @page misra_violations MISRA-C:2012 violations
*
* @section [global]
* Violates MISRA 2012 Advisory Rule 2.3, local typedef not referenced
* The SoC header defines typedef for all modules.
*
* @section [global]
* Violates MISRA 2012 Advisory Rule 2.5, local macro not referenced
* The SoC header defines macros for all modules and registers.
*
* @section [global]
* Violates MISRA 2012 Advisory Directive 4.9, Function-like macro
* These are generated macros used for accessing the bit-fields from registers.
*
* @section [global]
* Violates MISRA 2012 Required Rule 5.1, identifier clash
* The supported compilers use more than 31 significant characters for identifiers.
*
* @section [global]
* Violates MISRA 2012 Required Rule 5.2, identifier clash
* The supported compilers use more than 31 significant characters for identifiers.
*
* @section [global]
* Violates MISRA 2012 Required Rule 5.4, identifier clash
* The supported compilers use more than 31 significant characters for identifiers.
*
* @section [global]
* Violates MISRA 2012 Required Rule 5.5, identifier clash
* The supported compilers use more than 31 significant characters for identifiers.
*
* @section [global]
* Violates MISRA 2012 Required Rule 21.1, defined macro '__I' is reserved to the compiler
* This type qualifier is needed to ensure correct I/O access and addressing.
*/

/* ----------------------------------------------------------------------------
   -- MCU activation
   ---------------------------------------------------------------------------- */

/* Prevention from multiple including the same memory map */
#if !defined(S32V232_H_)  /* Check if memory map has not been already included */
#define S32V232_H_
#define MCU_S32V232

/* Check if another memory map has not been also included */
#if (defined(MCU_ACTIVE))
  #error S32V232 memory map: There is already included another memory map. Only one memory map can be included.
#endif /* (defined(MCU_ACTIVE)) */
#define MCU_ACTIVE

#include <stdint.h>

/** Memory map major version (memory maps with equal major version number are
 * compatible) */
#define MCU_MEM_MAP_VERSION 0x0200u
/** Memory map minor version */
#define MCU_MEM_MAP_VERSION_MINOR 0x0000u

/* ----------------------------------------------------------------------------
   -- Generic macros
   ---------------------------------------------------------------------------- */

/* IO definitions (access restrictions to peripheral registers) */
/**
*   IO Type Qualifiers are used
*   \li to specify the access to peripheral variables.
*   \li for automatic generation of peripheral register debug information.
*/
#ifndef __IO
#ifdef __cplusplus
  #define   __I     volatile             /*!< Defines 'read only' permissions                 */
#else
  #define   __I     volatile const       /*!< Defines 'read only' permissions                 */
#endif
#define     __O     volatile             /*!< Defines 'write only' permissions                */
#define     __IO    volatile             /*!< Defines 'read / write' permissions              */
#endif


/**
* @brief 32 bits memory read macro.
*/
#if !defined(REG_READ32)
  #define REG_READ32(address)               (*(volatile uint32_t*)(address))
#endif

/**
* @brief 32 bits memory write macro.
*/
#if !defined(REG_WRITE32)
  #define REG_WRITE32(address, value)       ((*(volatile uint32_t*)(address))= (uint32_t)(value))
#endif

/**
* @brief 32 bits bits setting macro.
*/
#if !defined(REG_BIT_SET32)
  #define REG_BIT_SET32(address, mask)      ((*(volatile uint32_t*)(address))|= (uint32_t)(mask))
#endif

/**
* @brief 32 bits bits clearing macro.
*/
#if !defined(REG_BIT_CLEAR32)
  #define REG_BIT_CLEAR32(address, mask)    ((*(volatile uint32_t*)(address))&= ((uint32_t)~((uint32_t)(mask))))
#endif

/**
* @brief 32 bit clear bits and set with new value
* @note It is user's responsability to make sure that value has only "mask" bits set - (value&~mask)==0
*/
#if !defined(REG_RMW32)
  #define REG_RMW32(address, mask, value)   (REG_WRITE32((address), ((REG_READ32(address)& ((uint32_t)~((uint32_t)(mask))))| ((uint32_t)(value)))))
#endif


/* ----------------------------------------------------------------------------
   -- Interrupt vector numbers for S32V232
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup Interrupt_vector_numbers_S32V232 Interrupt vector numbers for S32V232
 * @{
 */

/** Interrupt Number Definitions */
#define NUMBER_OF_INT_VECTORS 183u                /**< Number of interrupts in the Vector table */

/**
 * @brief Defines the Interrupt Numbers definitions
 *
 * This enumeration is used to configure the interrupts.
 *
 * Implements : IRQn_Type_Class
 */
typedef enum
{
  /* Auxiliary constants */
  NotAvail_IRQn                = -128,             /**< Not available device specific interrupt */

  /* Core interrupts */
  NonMaskableInt_IRQn          = -14,              /**< Non Maskable Interrupt */
  HardFault_IRQn               = -13,              /**< Cortex-M4 SV Hard Fault Interrupt */
  BusFault_IRQn                = -11,              /**< Cortex-M4 Bus Fault Interrupt */
  UsageFault_IRQn              = -10,              /**< Cortex-M4 Usage Fault Interrupt */
  SVCall_IRQn                  = -5,               /**< Cortex-M4 SV Call Interrupt */
  DebugMonitor_IRQn            = -4,               /**< Cortex-M4 Debug Monitor Interrupt */
  PendSV_IRQn                  = -2,               /**< Cortex-M4 Pend SV Interrupt */
  SysTick_IRQn                 = -1,               /**< Cortex-M4 System Tick Interrupt */

  /* CPU to CPU and Directed Interrupts */
  CPU_CPU0_IRQn                = 0,                /**< CPU to CPU int0 */
  CPU_CPU1_IRQn                = 1,                /**< CPU to CPU int1 */
  CPU_CPU2_IRQn                = 2,                /**< CPU to CPU int2 */
  CPU_CPU3_IRQn                = 3,                /**< CPU to CPU int3 */
  CPU_MCM_IRQn                 = 5,                /**< Directed Cortex-M4 (= MCM) */

  /* Shared peripheral interrupts */
  DMA0_15_IRQn                 = 8,                /**< DMA channel 0 - 15 transfer complete*/
  DMA16_31_IRQn                = 9,                /**< DMA channel 16 - 31 transfer complete*/
  DMA_Error_IRQn               = 10,               /**< DMA error interrupt channels 0-31*/
  SWT0_IRQn                    = 11,               /**< Software Watchdog Timer 0 */
  SWT1_IRQn                    = 12,               /**< Software Watchdog Timer 1 */
  SWT2_IRQn                    = 13,               /**< Software Watchdog Timer 2 */
  ERM_SB_IRQn                  = 16,               /**< Single bit ECC error */
  ERM_MB_IRQn                  = 17,               /**< Multi bit ECC error */
  CSE_IRQ_IRQn                 = 18,               /**< Command done */
  STM0_IRQn                    = 19,               /**< System Timer Module 0 */
  STM1_IRQn                    = 20,               /**< System Timer Module 1 */
  QSPI_IRQn                    = 24,               /**< QuadSPI IRQ */
  STCU_IRQn                    = 25,               /**< MBIST IRQ */
  USDHC_IRQn                   = 28,               /**< USDHC IRQ */
  PIT_RTI0_IRQn                = 31,               /**< 6 interrupts (one per channel) */
  PIT_RTI1_IRQn                = 32,               /**< 4 interrupts (one per channel) */
  FTM0_IRQn                    = 33,               /**< FTM0 single interrupt vector for all sources*/
  FTM1_IRQn                    = 34,               /**< FTM1 single interrupt vector for all sources*/
  ENET_Timer_IRQn              = 35,               /**< ENET Timer IRQ */
  ENET_TX_IRQn                 = 36,               /**< ENET TX IRQ */
  ENET_RX_IRQn                 = 37,               /**< ENET RX IRQ */
  ENET_Err_IRQn                = 38,               /**< ENET Error/Generic IRQ */
  SAR_ADC_IRQn                 = 39,               /**< EOC, ERR, WD */
  CAN0_IRQn                    = 41,               /**< FlexCAN0 FLEXCAN_ESR[ERR_INT], FLEXCAN_ESR_BOFF | FLEXCAN_Transmit_Warning | FLEXCAN_Receive_Warning | FLEXCAN_ERRSR[HANCEIF] | FLEXCAN_ERRSR[FANCEIF] | FLEXCAN_ERRSR[CEIF] or-ed together */
  CAN0_Buff_IRQn               = 42,               /**< FlexCAN0 FLEXCAN_BUFF_00_63 */
  CAN1_IRQn                    = 43,               /**< FlexCAN1 FLEXCAN_ESR[ERR_INT], FLEXCAN_ESR_BOFF | FLEXCAN_Transmit_Warning | FLEXCAN_Receive_Warning | FLEXCAN_ERRSR[HANCEIF] | FLEXCAN_ERRSR[FANCEIF] | FLEXCAN_ERRSR[CEIF] or-ed together */
  CAN1_Buff_IRQn               = 44,               /**< FlexCAN1 FLEXCAN_BUFF_00_63 */
  FR_xRNEIF_IRQn               = 49,               /**< Flexray LRNEIF | DRNEIF */
  FR_xRCEIF_IRQn               = 50,               /**< Flexray LRCEIF | DRCEIF */
  FR_FAFAIF_IRQn               = 51,               /**< Flexray FAFAIF */
  FR_FAFBIF_IRQn               = 52,               /**< Flexray FAFBIF */
  FR_WUPIF_IRQn                = 53,               /**< Flexray WUPIF */
  FR_PRIF_IRQn                 = 54,               /**< Flexray PRIF */
  FR_CHIF_IRQn                 = 55,               /**< Flexray CHIF */
  FR_TBIF_IRQn                 = 56,               /**< Flexray TBIF */
  FR_RBIF_IRQn                 = 57,               /**< Flexray RBIF */
  FR_MIF_IRQn                  = 58,               /**< Flexray MIF */
  Linflex_D0_IRQn              = 59,               /**< RX, TX and ERR or-ed together */
  Linflex_D1_IRQn              = 60,               /**< RX, TX and ERR or-ed together */
  SPI0_IRQn                    = 61,               /**< Serial peripheral Interface 0 */
  SPI1_IRQn                    = 62,               /**< Serial peripheral Interface 1 */
  SPI2_IRQn                    = 63,               /**< Serial peripheral Interface 2 */
  SPI3_IRQn                    = 64,               /**< Serial peripheral Interface 3 */
  CMU7_IRQn                    = 65,               /**< FIRC3 Clock Fail */
  I2C0_IRQn                    = 67,               /**< Inter-integrated circuit 0 */
  I2C1_IRQn                    = 68,               /**< Inter-integrated circuit 1 */
  I2C2_IRQn                    = 69,               /**< Inter-integrated circuit 2 */
  SIUL_IRQn                    = 70,               /**< Pin interrupts all or-ed together */
  MC_ME_IRQn                   = 71,               /**< Mode Transition Interrupt, Invalid Mode Interrupt, Invalid Mode Config */
  MC_RGM_IRQn                  = 72,               /**< Functional reset alternate event interrupt */
  PMC_IRQn                     = 73,               /**< IRQ */
  FCCU_ALARM_IRQn              = 74,               /**< FCCU Interrupt request (ALARM state) */
  FCCU_MISC_IRQn               = 75,               /**< FCCU Interrupt request (miscellaneous conditions) */
  SIPI1_IRQn                   = 76,               /**< Read interrupt channel 1 */
  SIPI2_IRQn                   = 77,               /**< Read interrupt channel 2 */
  SIPI3_IRQn                   = 78,               /**< Read interrupt channel 3 */
  SIPI4_IRQn                   = 79,               /**< Read interrupt channel 4 */
  SIPI_TRIG_IRQn               = 80,               /**< Trigger command */
  SIPI_ERR_IRQn                = 81,               /**< Err1, Err2 */
  LFAST_IRQn                   = 82,               /**< Tx interrupt, Tx exception, Rx interrupt, Rx exception, Rx ICLC: all or-ed together */
  FastDMA_TC_IRQn              = 83,               /**< FastDMA Transfer Complete */
  FastDMA_ERR_IRQn             = 84,               /**< FastDMA error | CRC error */
  HPSMI_ERR_IRQn               = 85,               /**< ECC error, MPU error, Exception interrupt */
  HPSMI_WTP_IRQn               = 87,               /**< Watchpoint hit */
  MEW0_IRQn                    = 88,               /**< Watchpoint hit */
  MEW1_IRQn                    = 89,               /**< Watchpoint hit */
  SSE_IRQn                     = 90,               /**< Combined interrupt (OUT2, OUT3) */
  MIPI_CSI_2_0_Camera_IRQn     = 91,               /**< Interrupt generation on error Interrupt at PHY , Protocol or Packet level */
  MIPI_CSI_2_1_Camera_IRQn     = 92,               /**< Interrupt generation on error Interrupt at PHY , Protocol or Packet level */
  MIPI_CSI_2_Bridge0_IRQn      = 93,               /**< Interrupt generation for the virtual channels and embedded data at start of frame */
  MIPI_CSI_2_Bridge1_IRQn      = 94,               /**< Interrupt generation for the virtual channels and embedded data at start of frame */
  DCU_2D_ACE_IRQn              = 95,               /**< IRQ */
  VIU0_IRQn                    = 96,               /**< Or-ed IRQ */
  DEC200_IRQn                  = 97,               /**< Flush done/error interrupt */
  ENC200_IRQn                  = 98,               /**< Flush done/error interrupt */
  TMU_IRQn                     = 99,               /**< Or-ed IRQ */
  VIU1_IRQn                    = 100,              /**< Or-ed IRQ */
  GC3000_IRQn                  = 101,              /**< One interrupt signal */
  APEX_CL0_IRQn                = 109,              /**< Global interrupt */
  APEX_CL1_IRQn                = 110,              /**< Global interrupt */
  CCI_400_IRQn                 = 111,              /**< nERRORIRQ, nEVNTCNTOVERFLOW[4:0] */
  VSEQ_PAR_IRQn                = 112,              /**< Code & Data RAM or CRAM parity error | PRAM parity error */
  VSEQ_MSG_READY_IRQn          = 113,              /**< Message-ready */
  VSEQ_MSG_ACK_IRQn            = 114,              /**< Message-ack */
  VSEQ_INT_ERR_IRQn            = 115,              /**< Seq_int_err */
  VSEQ_DMA_IRQn                = 116,              /**< Internal DMA done */
  VSEQ_IPUS_IRQn               = 117,              /**< IPUS Done */
  VSEQ_IPUV_IRQn               = 118,              /**< IPUV Done */
  VSEQ_MISC_IRQn               = 119,              /**< Other/Misc peripherals Done */
  VSEQ_TRIG0_IRQn              = 120,              /**< Event trigger 0 */
  VSEQ_TRIG1_IRQn              = 121,              /**< Event trigger 1 */
  VSEQ_TRIG2_IRQn              = 122,              /**< Event trigger 2 */
  VSEQ_TRIG3_IRQn              = 123,              /**< Event trigger 3 */
  VSEQ_TRIG4_IRQn              = 124,              /**< Event trigger 4 */
  VSEQ_TRIG5_IRQn              = 125,              /**< Event trigger 5 */
  VSEQ_TRIG6_IRQn              = 126,              /**< Event trigger 6 */
  VSEQ_TRIG7_IRQn              = 127,              /**< Event trigger 7 */
  JPEG_SOI_IRQn                = 128,              /**< SOI */
  JPEG_EOI_IRQn                = 129,              /**< EOI */
  JPEG_FIFO1_IRQn              = 130,              /**< st_fifo_watermark_1 */
  JPEG_FIFO2_IRQn              = 131,              /**< st_fifo_watermark_2 */
  JPEG_FIFO3_IRQn              = 132,              /**< st_fifo_watermark_3 */
  JPEG_FIFO4_IRQn              = 133,              /**< st_fifo_watermark_4 */
  JPEG_ERR_IRQn                = 134,              /**< Config_err */
  JPEG_ROW_IRQn                = 135,              /**< mcu_row_done */
  H264_GT0_IRQn                = 136,              /**< grt_than_thr_ch_0 */
  H264_EOF0_IRQn               = 137,              /**< EOF_0 */
  H264_EOF1_IRQn               = 138,              /**< EOF_1 */
  H264_EOF2_IRQn               = 139,              /**< EOF_2 */
  H264_EOF3_IRQn               = 140,              /**< EOF_3 */
  H264_FIFO1_IRQn              = 141,              /**< st_fifo_watermark_1 */
  H264_FIFO2_IRQn              = 142,              /**< st_fifo_watermark_2 */
  H264_FIFO3_IRQn              = 143,              /**< st_fifo_watermark_3 */
  H264_FIFO4_IRQn              = 144,              /**< st_fifo_watermark_4 */
  H264_ERR_IRQn                = 145,              /**< Err_irq or-ed with oos */
  H264_MBROW0_IRQn             = 146,              /**< MBROW_done_0 */
  H264_MBROW1_IRQn             = 147,              /**< MBROW_done_1 */
  H264_MBROW2_IRQn             = 148,              /**< MBROW_done_2 */
  H264_MBROW3_IRQn             = 149,              /**< MBROW_done_3 */
  H264_VF_IRQn                 = 150,              /**< video_frame_done */
  H264_NR_IRQn                 = 151,              /**< bs_new_row */
  H264_CB_ALARM_IRQn           = 152,              /**< cb_alarm_int */
  H264_LINE_IRQn               = 153,              /**< vi_line_int */
  H264_LCOUNT_ALARM_IRQn       = 154,              /**< vi_lcount_alarm_int */
  H264_OOS_IRQn                = 155,              /**< encoder_oos */
  H264_FETCH_16_IRQn           = 156,              /**< FETCH_16_DONE */
  H264_FETCH_16_FF_OVF_IRQn    = 157,              /**< FETCH_16_FF_OVF_INT */
  H264_ALL_IRQn                = 158,              /**< Or-ed irq */
  CA53_COMM_0_0_IRQn           = 159,              /**< nCOMMIRQ[0] Cluster 0 */
  CA53_COMM_0_1_IRQn           = 160,              /**< nCOMMIRQ[1] Cluster 0 */
  CA53_COMM_1_0_IRQn           = 161,              /**< nCOMMIRQ[0] Cluster 1 */
  CA53_COMM_1_1_IRQn           = 162,              /**< nCOMMIRQ[1] Cluster 1 */
  CA53_PMU_0_0_IRQn            = 163,              /**< nPMUIRQ[0] Cluster 0 */
  CA53_PMU_0_1_IRQn            = 164,              /**< nPMUIRQ[1] Cluster 0 */
  CA53_PMU_1_0_IRQn            = 165,              /**< nPMUIRQ[0] Cluster 1 */
  CA53_PMU_1_1_IRQn            = 166,              /**< nPMUIRQ[1] Cluster 1 */
  CA53_CTI_0_0_IRQn            = 167,              /**< CTIIRQ[0] Cluster 0 */
  CA53_CTI_0_1_IRQn            = 168,              /**< CTIIRQ[1] Cluster 0 */
  CA53_CTI_1_0_IRQn            = 169,              /**< CTIIRQ[0] Cluster 1 */
  CA53_CTI_1_1_IRQn            = 170,              /**< CTIIRQ[1] Cluster 1 */
  CA53_AXI_0_IRQn              = 171,              /**< nEXTERRIRQ Cluster 0 AXI write error */
  CA53_AXI_1_IRQn              = 172,              /**< nEXTERRIRQ Cluster 1 AXI write error */
  CA53_L2_0_IRQn               = 173,              /**< nINTERRIRQ Cluster 0 L2 RAM double-bit ECC error */
  CA53_L2_1_IRQn               = 174,              /**< nINTERRIRQ Cluster 1 L2 RAM double-bit ECC error */
  XOSC_IRQn                    = 175               /**< Counter expired */
} IRQn_Type;

/*!
 * @}
 */ /* end of group Interrupt_vector_numbers_S32V232 */


/* ----------------------------------------------------------------------------
   -- Device Peripheral Access Layer for S32V232
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup Peripheral_access_layer_S32V232 Device Peripheral Access Layer for S32V232
 * @{
 */

/* @brief This module covers memory mapped registers available on SoC */

/* ----------------------------------------------------------------------------
   -- ADC Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup ADC_Peripheral_Access_Layer ADC Peripheral Access Layer
 * @{
 */


/** ADC - Size of Registers Arrays */
#define ADC_CEOCFR_COUNT                         2u
#define ADC_CIMR_COUNT                           2u
#define ADC_DMAR_COUNT                           2u
#define ADC_PSR_COUNT                            2u
#define ADC_CTR_COUNT                            2u
#define ADC_NCMR_COUNT                           2u
#define ADC_JCMR_COUNT                           2u
#define ADC_CDR_COUNT                            40u
#define ADC_CWENR_COUNT                          2u
#define ADC_AWORR_COUNT                          2u

/** ADC - Register Layout Typedef */
typedef struct {
  __IO uint32_t MCR;                               /**< Main Configuration Register, offset: 0x0 */
  __IO uint32_t MSR;                               /**< Main Status Register, offset: 0x4 */
       uint8_t RESERVED_0[8];
  __IO uint32_t ISR;                               /**< Interrupt Status Register, offset: 0x10 */
  __IO uint32_t CEOCFR[ADC_CEOCFR_COUNT];          /**< Channel Pending Register..Channel Pending Register 1, array offset: 0x14, array step: 0x4 */
       uint8_t RESERVED_1[4];
  __IO uint32_t IMR;                               /**< Interrupt Mask Register, offset: 0x20 */
  __IO uint32_t CIMR[ADC_CIMR_COUNT];              /**< Channel Interrupt Mask Register 0..Channel Interrupt Mask Register 1, array offset: 0x24, array step: 0x4 */
       uint8_t RESERVED_2[4];
  __IO uint32_t WTISR;                             /**< Watchdog Threshold Interrupt Status Register, offset: 0x30 */
  __IO uint32_t WTIMR;                             /**< Watchdog Threshold Interrupt Mask Register, offset: 0x34 */
       uint8_t RESERVED_3[8];
  __IO uint32_t DMAE;                              /**< DMAE Register, offset: 0x40 */
  __IO uint32_t DMAR[ADC_DMAR_COUNT];              /**< DMA Register 0..DMA Register 1, array offset: 0x44, array step: 0x4 */
       uint8_t RESERVED_4[20];
  __IO uint32_t THRHLR0;                           /**< Analog Watchdog Threshold Register 0, offset: 0x60 */
  __IO uint32_t THRHLR1;                           /**< Analog Watchdog Threshold Register 1, offset: 0x64 */
  __IO uint32_t THRHLR2;                           /**< Analog Watchdog Threshold Register 2, offset: 0x68 */
  __IO uint32_t THRHLR3;                           /**< Analog Watchdog Threshold Register 2, offset: 0x6C */
       uint8_t RESERVED_5[16];
  __IO uint32_t PSCR;                              /**< Presampling Control Register, offset: 0x80 */
  __IO uint32_t PSR[ADC_PSR_COUNT];                /**< Presampling Register 0..Presampling Register 1, array offset: 0x84, array step: 0x4 */
       uint8_t RESERVED_6[8];
  __IO uint32_t CTR[ADC_CTR_COUNT];                /**< Conversion Timing Register 0..Conversion Timing Register 2, array offset: 0x94, array step: 0x4 */
       uint8_t RESERVED_7[8];
  __IO uint32_t NCMR[ADC_NCMR_COUNT];              /**< Normal Conversion Mask Register, array offset: 0xA4, array step: 0x4 */
       uint8_t RESERVED_8[8];
  __IO uint32_t JCMR[ADC_JCMR_COUNT];              /**< Injected Conversion Mask Register, array offset: 0xB4, array step: 0x4 */
       uint8_t RESERVED_9[4];
  __IO uint32_t USROFSGN;                          /**< User OFFSET and Gain Register, offset: 0xC0 */
       uint8_t RESERVED_10[4];
  __IO uint32_t PDEDR;                             /**< Power Down Exit Delay Register, offset: 0xC8 */
       uint8_t RESERVED_11[52];
  __I  uint32_t CDR[ADC_CDR_COUNT];                /**< Channel n Data Register, array offset: 0x100, array step: 0x4 */
       uint8_t RESERVED_12[224];
  __IO uint32_t THRHLR4;                           /**< Analog Watchdog Threshold Register 4, offset: 0x280 */
  __IO uint32_t THRHLR5;                           /**< Analog Watchdog Threshold Register 5, offset: 0x284 */
  __IO uint32_t THRHLR6;                           /**< Analog Watchdog Threshold Register 6, offset: 0x288 */
  __IO uint32_t THRHLR7;                           /**< Analog Watchdog Threshold Register 7, offset: 0x28C */
       uint8_t RESERVED_13[32];
  __IO uint32_t CWSELR0;                           /**< Channel Watchdog Select Register, offset: 0x2B0 */
       uint8_t RESERVED_14[12];
  __IO uint32_t CWSELR4;                           /**< Channel Watchdog Select Register, offset: 0x2C0 */
       uint8_t RESERVED_15[28];
  __IO uint32_t CWENR[ADC_CWENR_COUNT];            /**< Channel Watchdog Enable Register, array offset: 0x2E0, array step: 0x4 */
       uint8_t RESERVED_16[8];
  __IO uint32_t AWORR[ADC_AWORR_COUNT];            /**< Analog Watchdog Out of Range Register, array offset: 0x2F0, array step: 0x4 */
       uint8_t RESERVED_17[72];
  __IO uint32_t STCR1;                             /**< Self-Test Configuration Register 1, offset: 0x340 */
  __IO uint32_t STCR2;                             /**< Self-Test Configuration Register 2, offset: 0x344 */
  __IO uint32_t STCR3;                             /**< Self-Test Configuration Register 3, offset: 0x348 */
  __IO uint32_t STBRR;                             /**< Self-Test Baud Rate Register, offset: 0x34C */
  __IO uint32_t STSR1;                             /**< Self-Test Status Register 1, offset: 0x350 */
  __I  uint32_t STSR2;                             /**< Self-Test Status Register 2, offset: 0x354 */
  __I  uint32_t STSR3;                             /**< Self-Test Status Register 3, offset: 0x358 */
  __I  uint32_t STSR4;                             /**< Self-Test Status Register 4, offset: 0x35C */
       uint8_t RESERVED_18[16];
  __I  uint32_t STDR1;                             /**< Self-Test Data Register 1, offset: 0x370 */
  __I  uint32_t STDR2;                             /**< Self-Test Data Register 2, offset: 0x374 */
       uint8_t RESERVED_19[8];
  __IO uint32_t STAW0R;                            /**< Self-Test Analog Watchdog Register, offset: 0x380 */
  __IO uint32_t STAW1AR;                           /**< Self-Test Analog Watchdog Register, offset: 0x384 */
  __IO uint32_t STAW1BR;                           /**< Self-Test Analog Watchdog Register 1B, offset: 0x388 */
  __IO uint32_t STAW2R;                            /**< Self-Test Analog Watchdog Register 2, offset: 0x38C */
  __I  uint32_t STAW3R;                            /**< Self-Test Analog Watchdog Register 3, offset: 0x390 */
  __IO uint32_t STAW4R;                            /**< Self-Test Analog Watchdog Register 4, offset: 0x394 */
  __IO uint32_t STAW5R;                            /**< Self-Test Analog Watchdog Register 5, offset: 0x398 */
  __I  uint32_t CALSTAT;                           /**< Calibration Status register, offset: 0x39C */
} ADC_Type, *ADC_MemMapPtr;

 /** Number of instances of the ADC module. */
#define ADC_INSTANCE_COUNT                       (1u)


/* ADC - Peripheral instance base addresses */
/** Peripheral ADC base address */
#define ADC_BASE                                 (0x4004D000u)
/** Peripheral ADC base pointer */
#define ADC                                      ((ADC_Type *)ADC_BASE)
/** Array initializer of ADC peripheral base addresses */
#define ADC_BASE_ADDRS                           { ADC_BASE }
/** Array initializer of ADC peripheral base pointers */
#define ADC_BASE_PTRS                            { ADC }

/* ----------------------------------------------------------------------------
   -- ADC Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup ADC_Register_Masks ADC Register Masks
 * @{
 */

/* MCR Bit Fields */
#define ADC_MCR_PWDN_MASK                        0x1u
#define ADC_MCR_PWDN_SHIFT                       0u
#define ADC_MCR_PWDN_WIDTH                       1u
#define ADC_MCR_PWDN(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_MCR_PWDN_SHIFT))&ADC_MCR_PWDN_MASK)
#define ADC_MCR_ADCLKDIV_MASK                    0x10u
#define ADC_MCR_ADCLKDIV_SHIFT                   4u
#define ADC_MCR_ADCLKDIV_WIDTH                   1u
#define ADC_MCR_ADCLKDIV(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_MCR_ADCLKDIV_SHIFT))&ADC_MCR_ADCLKDIV_MASK)
#define ADC_MCR_ACKO_MASK                        0x20u
#define ADC_MCR_ACKO_SHIFT                       5u
#define ADC_MCR_ACKO_WIDTH                       1u
#define ADC_MCR_ACKO(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_MCR_ACKO_SHIFT))&ADC_MCR_ACKO_MASK)
#define ADC_MCR_ABORT_MASK                       0x40u
#define ADC_MCR_ABORT_SHIFT                      6u
#define ADC_MCR_ABORT_WIDTH                      1u
#define ADC_MCR_ABORT(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_MCR_ABORT_SHIFT))&ADC_MCR_ABORT_MASK)
#define ADC_MCR_ABORTCHAIN_MASK                  0x80u
#define ADC_MCR_ABORTCHAIN_SHIFT                 7u
#define ADC_MCR_ABORTCHAIN_WIDTH                 1u
#define ADC_MCR_ABORTCHAIN(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_MCR_ABORTCHAIN_SHIFT))&ADC_MCR_ABORTCHAIN_MASK)
#define ADC_MCR_ADCLKSE_MASK                     0x100u
#define ADC_MCR_ADCLKSE_SHIFT                    8u
#define ADC_MCR_ADCLKSE_WIDTH                    1u
#define ADC_MCR_ADCLKSE(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_MCR_ADCLKSE_SHIFT))&ADC_MCR_ADCLKSE_MASK)
#define ADC_MCR_TSAMP_MASK                       0x600u
#define ADC_MCR_TSAMP_SHIFT                      9u
#define ADC_MCR_TSAMP_WIDTH                      2u
#define ADC_MCR_TSAMP(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_MCR_TSAMP_SHIFT))&ADC_MCR_TSAMP_MASK)
#define ADC_MCR_NRSMPL_MASK                      0x1800u
#define ADC_MCR_NRSMPL_SHIFT                     11u
#define ADC_MCR_NRSMPL_WIDTH                     2u
#define ADC_MCR_NRSMPL(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_MCR_NRSMPL_SHIFT))&ADC_MCR_NRSMPL_MASK)
#define ADC_MCR_AVGEN_MASK                       0x2000u
#define ADC_MCR_AVGEN_SHIFT                      13u
#define ADC_MCR_AVGEN_WIDTH                      1u
#define ADC_MCR_AVGEN(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_MCR_AVGEN_SHIFT))&ADC_MCR_AVGEN_MASK)
#define ADC_MCR_CALSTART_MASK                    0x4000u
#define ADC_MCR_CALSTART_SHIFT                   14u
#define ADC_MCR_CALSTART_WIDTH                   1u
#define ADC_MCR_CALSTART(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_MCR_CALSTART_SHIFT))&ADC_MCR_CALSTART_MASK)
#define ADC_MCR_STCL_MASK                        0x8000u
#define ADC_MCR_STCL_SHIFT                       15u
#define ADC_MCR_STCL_WIDTH                       1u
#define ADC_MCR_STCL(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_MCR_STCL_SHIFT))&ADC_MCR_STCL_MASK)
#define ADC_MCR_JSTART_MASK                      0x100000u
#define ADC_MCR_JSTART_SHIFT                     20u
#define ADC_MCR_JSTART_WIDTH                     1u
#define ADC_MCR_JSTART(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_MCR_JSTART_SHIFT))&ADC_MCR_JSTART_MASK)
#define ADC_MCR_JEDGE_MASK                       0x200000u
#define ADC_MCR_JEDGE_SHIFT                      21u
#define ADC_MCR_JEDGE_WIDTH                      1u
#define ADC_MCR_JEDGE(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_MCR_JEDGE_SHIFT))&ADC_MCR_JEDGE_MASK)
#define ADC_MCR_JTRGEN_MASK                      0x400000u
#define ADC_MCR_JTRGEN_SHIFT                     22u
#define ADC_MCR_JTRGEN_WIDTH                     1u
#define ADC_MCR_JTRGEN(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_MCR_JTRGEN_SHIFT))&ADC_MCR_JTRGEN_MASK)
#define ADC_MCR_NSTART_MASK                      0x1000000u
#define ADC_MCR_NSTART_SHIFT                     24u
#define ADC_MCR_NSTART_WIDTH                     1u
#define ADC_MCR_NSTART(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_MCR_NSTART_SHIFT))&ADC_MCR_NSTART_MASK)
#define ADC_MCR_MODE_MASK                        0x20000000u
#define ADC_MCR_MODE_SHIFT                       29u
#define ADC_MCR_MODE_WIDTH                       1u
#define ADC_MCR_MODE(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_MCR_MODE_SHIFT))&ADC_MCR_MODE_MASK)
#define ADC_MCR_WLSIDE_MASK                      0x40000000u
#define ADC_MCR_WLSIDE_SHIFT                     30u
#define ADC_MCR_WLSIDE_WIDTH                     1u
#define ADC_MCR_WLSIDE(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_MCR_WLSIDE_SHIFT))&ADC_MCR_WLSIDE_MASK)
#define ADC_MCR_OWREN_MASK                       0x80000000u
#define ADC_MCR_OWREN_SHIFT                      31u
#define ADC_MCR_OWREN_WIDTH                      1u
#define ADC_MCR_OWREN(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_MCR_OWREN_SHIFT))&ADC_MCR_OWREN_MASK)
/* MSR Bit Fields */
#define ADC_MSR_ADCSTATUS_MASK                   0x7u
#define ADC_MSR_ADCSTATUS_SHIFT                  0u
#define ADC_MSR_ADCSTATUS_WIDTH                  3u
#define ADC_MSR_ADCSTATUS(x)                     (((uint32_t)(((uint32_t)(x))<<ADC_MSR_ADCSTATUS_SHIFT))&ADC_MSR_ADCSTATUS_MASK)
#define ADC_MSR_ACKO_MASK                        0x20u
#define ADC_MSR_ACKO_SHIFT                       5u
#define ADC_MSR_ACKO_WIDTH                       1u
#define ADC_MSR_ACKO(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_MSR_ACKO_SHIFT))&ADC_MSR_ACKO_MASK)
#define ADC_MSR_CHADDR_MASK                      0xFE00u
#define ADC_MSR_CHADDR_SHIFT                     9u
#define ADC_MSR_CHADDR_WIDTH                     7u
#define ADC_MSR_CHADDR(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_MSR_CHADDR_SHIFT))&ADC_MSR_CHADDR_MASK)
#define ADC_MSR_SELF_TEST_S_MASK                 0x40000u
#define ADC_MSR_SELF_TEST_S_SHIFT                18u
#define ADC_MSR_SELF_TEST_S_WIDTH                1u
#define ADC_MSR_SELF_TEST_S(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_MSR_SELF_TEST_S_SHIFT))&ADC_MSR_SELF_TEST_S_MASK)
#define ADC_MSR_JSTART_MASK                      0x100000u
#define ADC_MSR_JSTART_SHIFT                     20u
#define ADC_MSR_JSTART_WIDTH                     1u
#define ADC_MSR_JSTART(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_MSR_JSTART_SHIFT))&ADC_MSR_JSTART_MASK)
#define ADC_MSR_JABORT_MASK                      0x800000u
#define ADC_MSR_JABORT_SHIFT                     23u
#define ADC_MSR_JABORT_WIDTH                     1u
#define ADC_MSR_JABORT(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_MSR_JABORT_SHIFT))&ADC_MSR_JABORT_MASK)
#define ADC_MSR_NSTART_MASK                      0x1000000u
#define ADC_MSR_NSTART_SHIFT                     24u
#define ADC_MSR_NSTART_WIDTH                     1u
#define ADC_MSR_NSTART(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_MSR_NSTART_SHIFT))&ADC_MSR_NSTART_MASK)
#define ADC_MSR_CALBUSY_MASK                     0x20000000u
#define ADC_MSR_CALBUSY_SHIFT                    29u
#define ADC_MSR_CALBUSY_WIDTH                    1u
#define ADC_MSR_CALBUSY(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_MSR_CALBUSY_SHIFT))&ADC_MSR_CALBUSY_MASK)
#define ADC_MSR_CALFAIL_MASK                     0x40000000u
#define ADC_MSR_CALFAIL_SHIFT                    30u
#define ADC_MSR_CALFAIL_WIDTH                    1u
#define ADC_MSR_CALFAIL(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_MSR_CALFAIL_SHIFT))&ADC_MSR_CALFAIL_MASK)
#define ADC_MSR_CALIBRTD_MASK                    0x80000000u
#define ADC_MSR_CALIBRTD_SHIFT                   31u
#define ADC_MSR_CALIBRTD_WIDTH                   1u
#define ADC_MSR_CALIBRTD(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_MSR_CALIBRTD_SHIFT))&ADC_MSR_CALIBRTD_MASK)
/* ISR Bit Fields */
#define ADC_ISR_ECH_MASK                         0x1u
#define ADC_ISR_ECH_SHIFT                        0u
#define ADC_ISR_ECH_WIDTH                        1u
#define ADC_ISR_ECH(x)                           (((uint32_t)(((uint32_t)(x))<<ADC_ISR_ECH_SHIFT))&ADC_ISR_ECH_MASK)
#define ADC_ISR_EOC_MASK                         0x2u
#define ADC_ISR_EOC_SHIFT                        1u
#define ADC_ISR_EOC_WIDTH                        1u
#define ADC_ISR_EOC(x)                           (((uint32_t)(((uint32_t)(x))<<ADC_ISR_EOC_SHIFT))&ADC_ISR_EOC_MASK)
#define ADC_ISR_JECH_MASK                        0x4u
#define ADC_ISR_JECH_SHIFT                       2u
#define ADC_ISR_JECH_WIDTH                       1u
#define ADC_ISR_JECH(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_ISR_JECH_SHIFT))&ADC_ISR_JECH_MASK)
#define ADC_ISR_JEOC_MASK                        0x8u
#define ADC_ISR_JEOC_SHIFT                       3u
#define ADC_ISR_JEOC_WIDTH                       1u
#define ADC_ISR_JEOC(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_ISR_JEOC_SHIFT))&ADC_ISR_JEOC_MASK)
/* CEOCFR Bit Fields */
#define ADC_CEOCFR_EOC_CH0_MASK                  0x1u
#define ADC_CEOCFR_EOC_CH0_SHIFT                 0u
#define ADC_CEOCFR_EOC_CH0_WIDTH                 1u
#define ADC_CEOCFR_EOC_CH0(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOC_CH0_SHIFT))&ADC_CEOCFR_EOC_CH0_MASK)
#define ADC_CEOCFR_EOC_CH32_MASK                 0x1u
#define ADC_CEOCFR_EOC_CH32_SHIFT                0u
#define ADC_CEOCFR_EOC_CH32_WIDTH                1u
#define ADC_CEOCFR_EOC_CH32(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOC_CH32_SHIFT))&ADC_CEOCFR_EOC_CH32_MASK)
#define ADC_CEOCFR_EOC_CH1_MASK                  0x2u
#define ADC_CEOCFR_EOC_CH1_SHIFT                 1u
#define ADC_CEOCFR_EOC_CH1_WIDTH                 1u
#define ADC_CEOCFR_EOC_CH1(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOC_CH1_SHIFT))&ADC_CEOCFR_EOC_CH1_MASK)
#define ADC_CEOCFR_EOC_CH33_MASK                 0x2u
#define ADC_CEOCFR_EOC_CH33_SHIFT                1u
#define ADC_CEOCFR_EOC_CH33_WIDTH                1u
#define ADC_CEOCFR_EOC_CH33(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOC_CH33_SHIFT))&ADC_CEOCFR_EOC_CH33_MASK)
#define ADC_CEOCFR_EOC_CH34_MASK                 0x4u
#define ADC_CEOCFR_EOC_CH34_SHIFT                2u
#define ADC_CEOCFR_EOC_CH34_WIDTH                1u
#define ADC_CEOCFR_EOC_CH34(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOC_CH34_SHIFT))&ADC_CEOCFR_EOC_CH34_MASK)
#define ADC_CEOCFR_EOC_CH2_MASK                  0x4u
#define ADC_CEOCFR_EOC_CH2_SHIFT                 2u
#define ADC_CEOCFR_EOC_CH2_WIDTH                 1u
#define ADC_CEOCFR_EOC_CH2(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOC_CH2_SHIFT))&ADC_CEOCFR_EOC_CH2_MASK)
#define ADC_CEOCFR_EOC_CH35_MASK                 0x8u
#define ADC_CEOCFR_EOC_CH35_SHIFT                3u
#define ADC_CEOCFR_EOC_CH35_WIDTH                1u
#define ADC_CEOCFR_EOC_CH35(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOC_CH35_SHIFT))&ADC_CEOCFR_EOC_CH35_MASK)
#define ADC_CEOCFR_EOC_CH3_MASK                  0x8u
#define ADC_CEOCFR_EOC_CH3_SHIFT                 3u
#define ADC_CEOCFR_EOC_CH3_WIDTH                 1u
#define ADC_CEOCFR_EOC_CH3(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOC_CH3_SHIFT))&ADC_CEOCFR_EOC_CH3_MASK)
#define ADC_CEOCFR_EOC_CH36_MASK                 0x10u
#define ADC_CEOCFR_EOC_CH36_SHIFT                4u
#define ADC_CEOCFR_EOC_CH36_WIDTH                1u
#define ADC_CEOCFR_EOC_CH36(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOC_CH36_SHIFT))&ADC_CEOCFR_EOC_CH36_MASK)
#define ADC_CEOCFR_EOC_CH4_MASK                  0x10u
#define ADC_CEOCFR_EOC_CH4_SHIFT                 4u
#define ADC_CEOCFR_EOC_CH4_WIDTH                 1u
#define ADC_CEOCFR_EOC_CH4(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOC_CH4_SHIFT))&ADC_CEOCFR_EOC_CH4_MASK)
#define ADC_CEOCFR_EOC_CH5_MASK                  0x20u
#define ADC_CEOCFR_EOC_CH5_SHIFT                 5u
#define ADC_CEOCFR_EOC_CH5_WIDTH                 1u
#define ADC_CEOCFR_EOC_CH5(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOC_CH5_SHIFT))&ADC_CEOCFR_EOC_CH5_MASK)
#define ADC_CEOCFR_EOC_CH37_MASK                 0x20u
#define ADC_CEOCFR_EOC_CH37_SHIFT                5u
#define ADC_CEOCFR_EOC_CH37_WIDTH                1u
#define ADC_CEOCFR_EOC_CH37(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOC_CH37_SHIFT))&ADC_CEOCFR_EOC_CH37_MASK)
#define ADC_CEOCFR_EOC_CH38_MASK                 0x40u
#define ADC_CEOCFR_EOC_CH38_SHIFT                6u
#define ADC_CEOCFR_EOC_CH38_WIDTH                1u
#define ADC_CEOCFR_EOC_CH38(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOC_CH38_SHIFT))&ADC_CEOCFR_EOC_CH38_MASK)
#define ADC_CEOCFR_EOC_CH6_MASK                  0x40u
#define ADC_CEOCFR_EOC_CH6_SHIFT                 6u
#define ADC_CEOCFR_EOC_CH6_WIDTH                 1u
#define ADC_CEOCFR_EOC_CH6(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOC_CH6_SHIFT))&ADC_CEOCFR_EOC_CH6_MASK)
#define ADC_CEOCFR_EOC_CH7_MASK                  0x80u
#define ADC_CEOCFR_EOC_CH7_SHIFT                 7u
#define ADC_CEOCFR_EOC_CH7_WIDTH                 1u
#define ADC_CEOCFR_EOC_CH7(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOC_CH7_SHIFT))&ADC_CEOCFR_EOC_CH7_MASK)
#define ADC_CEOCFR_EOC_CH39_MASK                 0x80u
#define ADC_CEOCFR_EOC_CH39_SHIFT                7u
#define ADC_CEOCFR_EOC_CH39_WIDTH                1u
#define ADC_CEOCFR_EOC_CH39(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOC_CH39_SHIFT))&ADC_CEOCFR_EOC_CH39_MASK)
/* IMR Bit Fields */
#define ADC_IMR_MSKECH_MASK                      0x1u
#define ADC_IMR_MSKECH_SHIFT                     0u
#define ADC_IMR_MSKECH_WIDTH                     1u
#define ADC_IMR_MSKECH(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_IMR_MSKECH_SHIFT))&ADC_IMR_MSKECH_MASK)
#define ADC_IMR_MSKEOC_MASK                      0x2u
#define ADC_IMR_MSKEOC_SHIFT                     1u
#define ADC_IMR_MSKEOC_WIDTH                     1u
#define ADC_IMR_MSKEOC(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_IMR_MSKEOC_SHIFT))&ADC_IMR_MSKEOC_MASK)
#define ADC_IMR_MSKJECH_MASK                     0x4u
#define ADC_IMR_MSKJECH_SHIFT                    2u
#define ADC_IMR_MSKJECH_WIDTH                    1u
#define ADC_IMR_MSKJECH(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_IMR_MSKJECH_SHIFT))&ADC_IMR_MSKJECH_MASK)
#define ADC_IMR_MSKJEOC_MASK                     0x8u
#define ADC_IMR_MSKJEOC_SHIFT                    3u
#define ADC_IMR_MSKJEOC_WIDTH                    1u
#define ADC_IMR_MSKJEOC(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_IMR_MSKJEOC_SHIFT))&ADC_IMR_MSKJEOC_MASK)
/* CIMR Bit Fields */
#define ADC_CIMR_CIM0_MASK                       0x1u
#define ADC_CIMR_CIM0_SHIFT                      0u
#define ADC_CIMR_CIM0_WIDTH                      1u
#define ADC_CIMR_CIM0(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM0_SHIFT))&ADC_CIMR_CIM0_MASK)
#define ADC_CIMR_CIM32_MASK                      0x1u
#define ADC_CIMR_CIM32_SHIFT                     0u
#define ADC_CIMR_CIM32_WIDTH                     1u
#define ADC_CIMR_CIM32(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM32_SHIFT))&ADC_CIMR_CIM32_MASK)
#define ADC_CIMR_CIM1_MASK                       0x2u
#define ADC_CIMR_CIM1_SHIFT                      1u
#define ADC_CIMR_CIM1_WIDTH                      1u
#define ADC_CIMR_CIM1(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM1_SHIFT))&ADC_CIMR_CIM1_MASK)
#define ADC_CIMR_CIM33_MASK                      0x2u
#define ADC_CIMR_CIM33_SHIFT                     1u
#define ADC_CIMR_CIM33_WIDTH                     1u
#define ADC_CIMR_CIM33(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM33_SHIFT))&ADC_CIMR_CIM33_MASK)
#define ADC_CIMR_CIM34_MASK                      0x4u
#define ADC_CIMR_CIM34_SHIFT                     2u
#define ADC_CIMR_CIM34_WIDTH                     1u
#define ADC_CIMR_CIM34(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM34_SHIFT))&ADC_CIMR_CIM34_MASK)
#define ADC_CIMR_CIM2_MASK                       0x4u
#define ADC_CIMR_CIM2_SHIFT                      2u
#define ADC_CIMR_CIM2_WIDTH                      1u
#define ADC_CIMR_CIM2(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM2_SHIFT))&ADC_CIMR_CIM2_MASK)
#define ADC_CIMR_CIM35_MASK                      0x8u
#define ADC_CIMR_CIM35_SHIFT                     3u
#define ADC_CIMR_CIM35_WIDTH                     1u
#define ADC_CIMR_CIM35(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM35_SHIFT))&ADC_CIMR_CIM35_MASK)
#define ADC_CIMR_CIM3_MASK                       0x8u
#define ADC_CIMR_CIM3_SHIFT                      3u
#define ADC_CIMR_CIM3_WIDTH                      1u
#define ADC_CIMR_CIM3(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM3_SHIFT))&ADC_CIMR_CIM3_MASK)
#define ADC_CIMR_CIM36_MASK                      0x10u
#define ADC_CIMR_CIM36_SHIFT                     4u
#define ADC_CIMR_CIM36_WIDTH                     1u
#define ADC_CIMR_CIM36(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM36_SHIFT))&ADC_CIMR_CIM36_MASK)
#define ADC_CIMR_CIM4_MASK                       0x10u
#define ADC_CIMR_CIM4_SHIFT                      4u
#define ADC_CIMR_CIM4_WIDTH                      1u
#define ADC_CIMR_CIM4(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM4_SHIFT))&ADC_CIMR_CIM4_MASK)
#define ADC_CIMR_CIM5_MASK                       0x20u
#define ADC_CIMR_CIM5_SHIFT                      5u
#define ADC_CIMR_CIM5_WIDTH                      1u
#define ADC_CIMR_CIM5(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM5_SHIFT))&ADC_CIMR_CIM5_MASK)
#define ADC_CIMR_CIM37_MASK                      0x20u
#define ADC_CIMR_CIM37_SHIFT                     5u
#define ADC_CIMR_CIM37_WIDTH                     1u
#define ADC_CIMR_CIM37(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM37_SHIFT))&ADC_CIMR_CIM37_MASK)
#define ADC_CIMR_CIM38_MASK                      0x40u
#define ADC_CIMR_CIM38_SHIFT                     6u
#define ADC_CIMR_CIM38_WIDTH                     1u
#define ADC_CIMR_CIM38(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM38_SHIFT))&ADC_CIMR_CIM38_MASK)
#define ADC_CIMR_CIM6_MASK                       0x40u
#define ADC_CIMR_CIM6_SHIFT                      6u
#define ADC_CIMR_CIM6_WIDTH                      1u
#define ADC_CIMR_CIM6(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM6_SHIFT))&ADC_CIMR_CIM6_MASK)
#define ADC_CIMR_CIM7_MASK                       0x80u
#define ADC_CIMR_CIM7_SHIFT                      7u
#define ADC_CIMR_CIM7_WIDTH                      1u
#define ADC_CIMR_CIM7(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM7_SHIFT))&ADC_CIMR_CIM7_MASK)
#define ADC_CIMR_CIM39_MASK                      0x80u
#define ADC_CIMR_CIM39_SHIFT                     7u
#define ADC_CIMR_CIM39_WIDTH                     1u
#define ADC_CIMR_CIM39(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM39_SHIFT))&ADC_CIMR_CIM39_MASK)
/* WTISR Bit Fields */
#define ADC_WTISR_WDG0L_MASK                     0x1u
#define ADC_WTISR_WDG0L_SHIFT                    0u
#define ADC_WTISR_WDG0L_WIDTH                    1u
#define ADC_WTISR_WDG0L(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG0L_SHIFT))&ADC_WTISR_WDG0L_MASK)
#define ADC_WTISR_WDG0H_MASK                     0x2u
#define ADC_WTISR_WDG0H_SHIFT                    1u
#define ADC_WTISR_WDG0H_WIDTH                    1u
#define ADC_WTISR_WDG0H(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG0H_SHIFT))&ADC_WTISR_WDG0H_MASK)
#define ADC_WTISR_WDG1L_MASK                     0x4u
#define ADC_WTISR_WDG1L_SHIFT                    2u
#define ADC_WTISR_WDG1L_WIDTH                    1u
#define ADC_WTISR_WDG1L(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG1L_SHIFT))&ADC_WTISR_WDG1L_MASK)
#define ADC_WTISR_WDG1H_MASK                     0x8u
#define ADC_WTISR_WDG1H_SHIFT                    3u
#define ADC_WTISR_WDG1H_WIDTH                    1u
#define ADC_WTISR_WDG1H(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG1H_SHIFT))&ADC_WTISR_WDG1H_MASK)
#define ADC_WTISR_WDG2L_MASK                     0x10u
#define ADC_WTISR_WDG2L_SHIFT                    4u
#define ADC_WTISR_WDG2L_WIDTH                    1u
#define ADC_WTISR_WDG2L(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG2L_SHIFT))&ADC_WTISR_WDG2L_MASK)
#define ADC_WTISR_WDG2H_MASK                     0x20u
#define ADC_WTISR_WDG2H_SHIFT                    5u
#define ADC_WTISR_WDG2H_WIDTH                    1u
#define ADC_WTISR_WDG2H(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG2H_SHIFT))&ADC_WTISR_WDG2H_MASK)
#define ADC_WTISR_WDG3L_MASK                     0x40u
#define ADC_WTISR_WDG3L_SHIFT                    6u
#define ADC_WTISR_WDG3L_WIDTH                    1u
#define ADC_WTISR_WDG3L(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG3L_SHIFT))&ADC_WTISR_WDG3L_MASK)
#define ADC_WTISR_WDG3H_MASK                     0x80u
#define ADC_WTISR_WDG3H_SHIFT                    7u
#define ADC_WTISR_WDG3H_WIDTH                    1u
#define ADC_WTISR_WDG3H(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG3H_SHIFT))&ADC_WTISR_WDG3H_MASK)
#define ADC_WTISR_WDG4L_MASK                     0x100u
#define ADC_WTISR_WDG4L_SHIFT                    8u
#define ADC_WTISR_WDG4L_WIDTH                    1u
#define ADC_WTISR_WDG4L(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG4L_SHIFT))&ADC_WTISR_WDG4L_MASK)
#define ADC_WTISR_WDG4H_MASK                     0x200u
#define ADC_WTISR_WDG4H_SHIFT                    9u
#define ADC_WTISR_WDG4H_WIDTH                    1u
#define ADC_WTISR_WDG4H(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG4H_SHIFT))&ADC_WTISR_WDG4H_MASK)
#define ADC_WTISR_WDG5L_MASK                     0x400u
#define ADC_WTISR_WDG5L_SHIFT                    10u
#define ADC_WTISR_WDG5L_WIDTH                    1u
#define ADC_WTISR_WDG5L(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG5L_SHIFT))&ADC_WTISR_WDG5L_MASK)
#define ADC_WTISR_WDG5H_MASK                     0x800u
#define ADC_WTISR_WDG5H_SHIFT                    11u
#define ADC_WTISR_WDG5H_WIDTH                    1u
#define ADC_WTISR_WDG5H(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG5H_SHIFT))&ADC_WTISR_WDG5H_MASK)
#define ADC_WTISR_WDG6L_MASK                     0x1000u
#define ADC_WTISR_WDG6L_SHIFT                    12u
#define ADC_WTISR_WDG6L_WIDTH                    1u
#define ADC_WTISR_WDG6L(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG6L_SHIFT))&ADC_WTISR_WDG6L_MASK)
#define ADC_WTISR_WDG6H_MASK                     0x2000u
#define ADC_WTISR_WDG6H_SHIFT                    13u
#define ADC_WTISR_WDG6H_WIDTH                    1u
#define ADC_WTISR_WDG6H(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG6H_SHIFT))&ADC_WTISR_WDG6H_MASK)
#define ADC_WTISR_WDG7L_MASK                     0x4000u
#define ADC_WTISR_WDG7L_SHIFT                    14u
#define ADC_WTISR_WDG7L_WIDTH                    1u
#define ADC_WTISR_WDG7L(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG7L_SHIFT))&ADC_WTISR_WDG7L_MASK)
#define ADC_WTISR_WDG7H_MASK                     0x8000u
#define ADC_WTISR_WDG7H_SHIFT                    15u
#define ADC_WTISR_WDG7H_WIDTH                    1u
#define ADC_WTISR_WDG7H(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG7H_SHIFT))&ADC_WTISR_WDG7H_MASK)
/* WTIMR Bit Fields */
#define ADC_WTIMR_MSKWDG0L_MASK                  0x1u
#define ADC_WTIMR_MSKWDG0L_SHIFT                 0u
#define ADC_WTIMR_MSKWDG0L_WIDTH                 1u
#define ADC_WTIMR_MSKWDG0L(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG0L_SHIFT))&ADC_WTIMR_MSKWDG0L_MASK)
#define ADC_WTIMR_MSKWDG0H_MASK                  0x2u
#define ADC_WTIMR_MSKWDG0H_SHIFT                 1u
#define ADC_WTIMR_MSKWDG0H_WIDTH                 1u
#define ADC_WTIMR_MSKWDG0H(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG0H_SHIFT))&ADC_WTIMR_MSKWDG0H_MASK)
#define ADC_WTIMR_MSKWDG1L_MASK                  0x4u
#define ADC_WTIMR_MSKWDG1L_SHIFT                 2u
#define ADC_WTIMR_MSKWDG1L_WIDTH                 1u
#define ADC_WTIMR_MSKWDG1L(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG1L_SHIFT))&ADC_WTIMR_MSKWDG1L_MASK)
#define ADC_WTIMR_MSKWDG1H_MASK                  0x8u
#define ADC_WTIMR_MSKWDG1H_SHIFT                 3u
#define ADC_WTIMR_MSKWDG1H_WIDTH                 1u
#define ADC_WTIMR_MSKWDG1H(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG1H_SHIFT))&ADC_WTIMR_MSKWDG1H_MASK)
#define ADC_WTIMR_MSKWDG2L_MASK                  0x10u
#define ADC_WTIMR_MSKWDG2L_SHIFT                 4u
#define ADC_WTIMR_MSKWDG2L_WIDTH                 1u
#define ADC_WTIMR_MSKWDG2L(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG2L_SHIFT))&ADC_WTIMR_MSKWDG2L_MASK)
#define ADC_WTIMR_MSKWDG2H_MASK                  0x20u
#define ADC_WTIMR_MSKWDG2H_SHIFT                 5u
#define ADC_WTIMR_MSKWDG2H_WIDTH                 1u
#define ADC_WTIMR_MSKWDG2H(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG2H_SHIFT))&ADC_WTIMR_MSKWDG2H_MASK)
#define ADC_WTIMR_MSKWDG3L_MASK                  0x40u
#define ADC_WTIMR_MSKWDG3L_SHIFT                 6u
#define ADC_WTIMR_MSKWDG3L_WIDTH                 1u
#define ADC_WTIMR_MSKWDG3L(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG3L_SHIFT))&ADC_WTIMR_MSKWDG3L_MASK)
#define ADC_WTIMR_MSKWDG3H_MASK                  0x80u
#define ADC_WTIMR_MSKWDG3H_SHIFT                 7u
#define ADC_WTIMR_MSKWDG3H_WIDTH                 1u
#define ADC_WTIMR_MSKWDG3H(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG3H_SHIFT))&ADC_WTIMR_MSKWDG3H_MASK)
#define ADC_WTIMR_MSKWDG4L_MASK                  0x100u
#define ADC_WTIMR_MSKWDG4L_SHIFT                 8u
#define ADC_WTIMR_MSKWDG4L_WIDTH                 1u
#define ADC_WTIMR_MSKWDG4L(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG4L_SHIFT))&ADC_WTIMR_MSKWDG4L_MASK)
#define ADC_WTIMR_MSKWDG4H_MASK                  0x200u
#define ADC_WTIMR_MSKWDG4H_SHIFT                 9u
#define ADC_WTIMR_MSKWDG4H_WIDTH                 1u
#define ADC_WTIMR_MSKWDG4H(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG4H_SHIFT))&ADC_WTIMR_MSKWDG4H_MASK)
#define ADC_WTIMR_MSKWDG5L_MASK                  0x400u
#define ADC_WTIMR_MSKWDG5L_SHIFT                 10u
#define ADC_WTIMR_MSKWDG5L_WIDTH                 1u
#define ADC_WTIMR_MSKWDG5L(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG5L_SHIFT))&ADC_WTIMR_MSKWDG5L_MASK)
#define ADC_WTIMR_MSKWDG5H_MASK                  0x800u
#define ADC_WTIMR_MSKWDG5H_SHIFT                 11u
#define ADC_WTIMR_MSKWDG5H_WIDTH                 1u
#define ADC_WTIMR_MSKWDG5H(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG5H_SHIFT))&ADC_WTIMR_MSKWDG5H_MASK)
#define ADC_WTIMR_MSKWDG6L_MASK                  0x1000u
#define ADC_WTIMR_MSKWDG6L_SHIFT                 12u
#define ADC_WTIMR_MSKWDG6L_WIDTH                 1u
#define ADC_WTIMR_MSKWDG6L(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG6L_SHIFT))&ADC_WTIMR_MSKWDG6L_MASK)
#define ADC_WTIMR_MSKWDG6H_MASK                  0x2000u
#define ADC_WTIMR_MSKWDG6H_SHIFT                 13u
#define ADC_WTIMR_MSKWDG6H_WIDTH                 1u
#define ADC_WTIMR_MSKWDG6H(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG6H_SHIFT))&ADC_WTIMR_MSKWDG6H_MASK)
#define ADC_WTIMR_MSKWDG7L_MASK                  0x4000u
#define ADC_WTIMR_MSKWDG7L_SHIFT                 14u
#define ADC_WTIMR_MSKWDG7L_WIDTH                 1u
#define ADC_WTIMR_MSKWDG7L(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG7L_SHIFT))&ADC_WTIMR_MSKWDG7L_MASK)
#define ADC_WTIMR_MSKWDG7H_MASK                  0x8000u
#define ADC_WTIMR_MSKWDG7H_SHIFT                 15u
#define ADC_WTIMR_MSKWDG7H_WIDTH                 1u
#define ADC_WTIMR_MSKWDG7H(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG7H_SHIFT))&ADC_WTIMR_MSKWDG7H_MASK)
/* DMAE Bit Fields */
#define ADC_DMAE_DMAEN_MASK                      0x1u
#define ADC_DMAE_DMAEN_SHIFT                     0u
#define ADC_DMAE_DMAEN_WIDTH                     1u
#define ADC_DMAE_DMAEN(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAE_DMAEN_SHIFT))&ADC_DMAE_DMAEN_MASK)
#define ADC_DMAE_DCLR_MASK                       0x2u
#define ADC_DMAE_DCLR_SHIFT                      1u
#define ADC_DMAE_DCLR_WIDTH                      1u
#define ADC_DMAE_DCLR(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_DMAE_DCLR_SHIFT))&ADC_DMAE_DCLR_MASK)
/* DMAR Bit Fields */
#define ADC_DMAR_DMA0_MASK                       0x1u
#define ADC_DMAR_DMA0_SHIFT                      0u
#define ADC_DMAR_DMA0_WIDTH                      1u
#define ADC_DMAR_DMA0(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA0_SHIFT))&ADC_DMAR_DMA0_MASK)
#define ADC_DMAR_DMA32_MASK                      0x1u
#define ADC_DMAR_DMA32_SHIFT                     0u
#define ADC_DMAR_DMA32_WIDTH                     1u
#define ADC_DMAR_DMA32(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA32_SHIFT))&ADC_DMAR_DMA32_MASK)
#define ADC_DMAR_DMA1_MASK                       0x2u
#define ADC_DMAR_DMA1_SHIFT                      1u
#define ADC_DMAR_DMA1_WIDTH                      1u
#define ADC_DMAR_DMA1(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA1_SHIFT))&ADC_DMAR_DMA1_MASK)
#define ADC_DMAR_DMA33_MASK                      0x2u
#define ADC_DMAR_DMA33_SHIFT                     1u
#define ADC_DMAR_DMA33_WIDTH                     1u
#define ADC_DMAR_DMA33(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA33_SHIFT))&ADC_DMAR_DMA33_MASK)
#define ADC_DMAR_DMA34_MASK                      0x4u
#define ADC_DMAR_DMA34_SHIFT                     2u
#define ADC_DMAR_DMA34_WIDTH                     1u
#define ADC_DMAR_DMA34(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA34_SHIFT))&ADC_DMAR_DMA34_MASK)
#define ADC_DMAR_DMA2_MASK                       0x4u
#define ADC_DMAR_DMA2_SHIFT                      2u
#define ADC_DMAR_DMA2_WIDTH                      1u
#define ADC_DMAR_DMA2(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA2_SHIFT))&ADC_DMAR_DMA2_MASK)
#define ADC_DMAR_DMA35_MASK                      0x8u
#define ADC_DMAR_DMA35_SHIFT                     3u
#define ADC_DMAR_DMA35_WIDTH                     1u
#define ADC_DMAR_DMA35(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA35_SHIFT))&ADC_DMAR_DMA35_MASK)
#define ADC_DMAR_DMA3_MASK                       0x8u
#define ADC_DMAR_DMA3_SHIFT                      3u
#define ADC_DMAR_DMA3_WIDTH                      1u
#define ADC_DMAR_DMA3(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA3_SHIFT))&ADC_DMAR_DMA3_MASK)
#define ADC_DMAR_DMA36_MASK                      0x10u
#define ADC_DMAR_DMA36_SHIFT                     4u
#define ADC_DMAR_DMA36_WIDTH                     1u
#define ADC_DMAR_DMA36(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA36_SHIFT))&ADC_DMAR_DMA36_MASK)
#define ADC_DMAR_DMA4_MASK                       0x10u
#define ADC_DMAR_DMA4_SHIFT                      4u
#define ADC_DMAR_DMA4_WIDTH                      1u
#define ADC_DMAR_DMA4(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA4_SHIFT))&ADC_DMAR_DMA4_MASK)
#define ADC_DMAR_DMA5_MASK                       0x20u
#define ADC_DMAR_DMA5_SHIFT                      5u
#define ADC_DMAR_DMA5_WIDTH                      1u
#define ADC_DMAR_DMA5(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA5_SHIFT))&ADC_DMAR_DMA5_MASK)
#define ADC_DMAR_DMA37_MASK                      0x20u
#define ADC_DMAR_DMA37_SHIFT                     5u
#define ADC_DMAR_DMA37_WIDTH                     1u
#define ADC_DMAR_DMA37(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA37_SHIFT))&ADC_DMAR_DMA37_MASK)
#define ADC_DMAR_DMA38_MASK                      0x40u
#define ADC_DMAR_DMA38_SHIFT                     6u
#define ADC_DMAR_DMA38_WIDTH                     1u
#define ADC_DMAR_DMA38(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA38_SHIFT))&ADC_DMAR_DMA38_MASK)
#define ADC_DMAR_DMA6_MASK                       0x40u
#define ADC_DMAR_DMA6_SHIFT                      6u
#define ADC_DMAR_DMA6_WIDTH                      1u
#define ADC_DMAR_DMA6(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA6_SHIFT))&ADC_DMAR_DMA6_MASK)
#define ADC_DMAR_DMA7_MASK                       0x80u
#define ADC_DMAR_DMA7_SHIFT                      7u
#define ADC_DMAR_DMA7_WIDTH                      1u
#define ADC_DMAR_DMA7(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA7_SHIFT))&ADC_DMAR_DMA7_MASK)
#define ADC_DMAR_DMA39_MASK                      0x80u
#define ADC_DMAR_DMA39_SHIFT                     7u
#define ADC_DMAR_DMA39_WIDTH                     1u
#define ADC_DMAR_DMA39(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA39_SHIFT))&ADC_DMAR_DMA39_MASK)
/* THRHLR0 Bit Fields */
#define ADC_THRHLR0_THRL_MASK                    0xFFFu
#define ADC_THRHLR0_THRL_SHIFT                   0u
#define ADC_THRHLR0_THRL_WIDTH                   12u
#define ADC_THRHLR0_THRL(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR0_THRL_SHIFT))&ADC_THRHLR0_THRL_MASK)
#define ADC_THRHLR0_THRH_MASK                    0xFFF0000u
#define ADC_THRHLR0_THRH_SHIFT                   16u
#define ADC_THRHLR0_THRH_WIDTH                   12u
#define ADC_THRHLR0_THRH(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR0_THRH_SHIFT))&ADC_THRHLR0_THRH_MASK)
/* THRHLR1 Bit Fields */
#define ADC_THRHLR1_THRL_MASK                    0xFFFu
#define ADC_THRHLR1_THRL_SHIFT                   0u
#define ADC_THRHLR1_THRL_WIDTH                   12u
#define ADC_THRHLR1_THRL(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR1_THRL_SHIFT))&ADC_THRHLR1_THRL_MASK)
#define ADC_THRHLR1_THRH_MASK                    0xFFF0000u
#define ADC_THRHLR1_THRH_SHIFT                   16u
#define ADC_THRHLR1_THRH_WIDTH                   12u
#define ADC_THRHLR1_THRH(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR1_THRH_SHIFT))&ADC_THRHLR1_THRH_MASK)
/* THRHLR2 Bit Fields */
#define ADC_THRHLR2_THRL_MASK                    0xFFFu
#define ADC_THRHLR2_THRL_SHIFT                   0u
#define ADC_THRHLR2_THRL_WIDTH                   12u
#define ADC_THRHLR2_THRL(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR2_THRL_SHIFT))&ADC_THRHLR2_THRL_MASK)
#define ADC_THRHLR2_THRH_MASK                    0xFFF0000u
#define ADC_THRHLR2_THRH_SHIFT                   16u
#define ADC_THRHLR2_THRH_WIDTH                   12u
#define ADC_THRHLR2_THRH(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR2_THRH_SHIFT))&ADC_THRHLR2_THRH_MASK)
/* THRHLR3 Bit Fields */
#define ADC_THRHLR3_THRL_MASK                    0xFFFu
#define ADC_THRHLR3_THRL_SHIFT                   0u
#define ADC_THRHLR3_THRL_WIDTH                   12u
#define ADC_THRHLR3_THRL(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR3_THRL_SHIFT))&ADC_THRHLR3_THRL_MASK)
#define ADC_THRHLR3_THRH_MASK                    0xFFF0000u
#define ADC_THRHLR3_THRH_SHIFT                   16u
#define ADC_THRHLR3_THRH_WIDTH                   12u
#define ADC_THRHLR3_THRH(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR3_THRH_SHIFT))&ADC_THRHLR3_THRH_MASK)
/* PSCR Bit Fields */
#define ADC_PSCR_PRECONV_MASK                    0x1u
#define ADC_PSCR_PRECONV_SHIFT                   0u
#define ADC_PSCR_PRECONV_WIDTH                   1u
#define ADC_PSCR_PRECONV(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_PSCR_PRECONV_SHIFT))&ADC_PSCR_PRECONV_MASK)
#define ADC_PSCR_PREVAL0_MASK                    0x6u
#define ADC_PSCR_PREVAL0_SHIFT                   1u
#define ADC_PSCR_PREVAL0_WIDTH                   2u
#define ADC_PSCR_PREVAL0(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_PSCR_PREVAL0_SHIFT))&ADC_PSCR_PREVAL0_MASK)
#define ADC_PSCR_PREVAL1_MASK                    0x18u
#define ADC_PSCR_PREVAL1_SHIFT                   3u
#define ADC_PSCR_PREVAL1_WIDTH                   2u
#define ADC_PSCR_PREVAL1(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_PSCR_PREVAL1_SHIFT))&ADC_PSCR_PREVAL1_MASK)
/* PSR Bit Fields */
#define ADC_PSR_PRES0_MASK                       0x1u
#define ADC_PSR_PRES0_SHIFT                      0u
#define ADC_PSR_PRES0_WIDTH                      1u
#define ADC_PSR_PRES0(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES0_SHIFT))&ADC_PSR_PRES0_MASK)
#define ADC_PSR_PRES32_MASK                      0x1u
#define ADC_PSR_PRES32_SHIFT                     0u
#define ADC_PSR_PRES32_WIDTH                     1u
#define ADC_PSR_PRES32(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES32_SHIFT))&ADC_PSR_PRES32_MASK)
#define ADC_PSR_PRES1_MASK                       0x2u
#define ADC_PSR_PRES1_SHIFT                      1u
#define ADC_PSR_PRES1_WIDTH                      1u
#define ADC_PSR_PRES1(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES1_SHIFT))&ADC_PSR_PRES1_MASK)
#define ADC_PSR_PRES33_MASK                      0x2u
#define ADC_PSR_PRES33_SHIFT                     1u
#define ADC_PSR_PRES33_WIDTH                     1u
#define ADC_PSR_PRES33(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES33_SHIFT))&ADC_PSR_PRES33_MASK)
#define ADC_PSR_PRES34_MASK                      0x4u
#define ADC_PSR_PRES34_SHIFT                     2u
#define ADC_PSR_PRES34_WIDTH                     1u
#define ADC_PSR_PRES34(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES34_SHIFT))&ADC_PSR_PRES34_MASK)
#define ADC_PSR_PRES2_MASK                       0x4u
#define ADC_PSR_PRES2_SHIFT                      2u
#define ADC_PSR_PRES2_WIDTH                      1u
#define ADC_PSR_PRES2(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES2_SHIFT))&ADC_PSR_PRES2_MASK)
#define ADC_PSR_PRES35_MASK                      0x8u
#define ADC_PSR_PRES35_SHIFT                     3u
#define ADC_PSR_PRES35_WIDTH                     1u
#define ADC_PSR_PRES35(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES35_SHIFT))&ADC_PSR_PRES35_MASK)
#define ADC_PSR_PRES3_MASK                       0x8u
#define ADC_PSR_PRES3_SHIFT                      3u
#define ADC_PSR_PRES3_WIDTH                      1u
#define ADC_PSR_PRES3(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES3_SHIFT))&ADC_PSR_PRES3_MASK)
#define ADC_PSR_PRES36_MASK                      0x10u
#define ADC_PSR_PRES36_SHIFT                     4u
#define ADC_PSR_PRES36_WIDTH                     1u
#define ADC_PSR_PRES36(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES36_SHIFT))&ADC_PSR_PRES36_MASK)
#define ADC_PSR_PRES4_MASK                       0x10u
#define ADC_PSR_PRES4_SHIFT                      4u
#define ADC_PSR_PRES4_WIDTH                      1u
#define ADC_PSR_PRES4(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES4_SHIFT))&ADC_PSR_PRES4_MASK)
#define ADC_PSR_PRES5_MASK                       0x20u
#define ADC_PSR_PRES5_SHIFT                      5u
#define ADC_PSR_PRES5_WIDTH                      1u
#define ADC_PSR_PRES5(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES5_SHIFT))&ADC_PSR_PRES5_MASK)
#define ADC_PSR_PRES37_MASK                      0x20u
#define ADC_PSR_PRES37_SHIFT                     5u
#define ADC_PSR_PRES37_WIDTH                     1u
#define ADC_PSR_PRES37(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES37_SHIFT))&ADC_PSR_PRES37_MASK)
#define ADC_PSR_PRES38_MASK                      0x40u
#define ADC_PSR_PRES38_SHIFT                     6u
#define ADC_PSR_PRES38_WIDTH                     1u
#define ADC_PSR_PRES38(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES38_SHIFT))&ADC_PSR_PRES38_MASK)
#define ADC_PSR_PRES6_MASK                       0x40u
#define ADC_PSR_PRES6_SHIFT                      6u
#define ADC_PSR_PRES6_WIDTH                      1u
#define ADC_PSR_PRES6(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES6_SHIFT))&ADC_PSR_PRES6_MASK)
#define ADC_PSR_PRES7_MASK                       0x80u
#define ADC_PSR_PRES7_SHIFT                      7u
#define ADC_PSR_PRES7_WIDTH                      1u
#define ADC_PSR_PRES7(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES7_SHIFT))&ADC_PSR_PRES7_MASK)
#define ADC_PSR_PRES39_MASK                      0x80u
#define ADC_PSR_PRES39_SHIFT                     7u
#define ADC_PSR_PRES39_WIDTH                     1u
#define ADC_PSR_PRES39(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES39_SHIFT))&ADC_PSR_PRES39_MASK)
/* CTR Bit Fields */
#define ADC_CTR_INPSAMP_MASK                     0xFFu
#define ADC_CTR_INPSAMP_SHIFT                    0u
#define ADC_CTR_INPSAMP_WIDTH                    8u
#define ADC_CTR_INPSAMP(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_CTR_INPSAMP_SHIFT))&ADC_CTR_INPSAMP_MASK)
/* NCMR Bit Fields */
#define ADC_NCMR_CH0_MASK                        0x1u
#define ADC_NCMR_CH0_SHIFT                       0u
#define ADC_NCMR_CH0_WIDTH                       1u
#define ADC_NCMR_CH0(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH0_SHIFT))&ADC_NCMR_CH0_MASK)
#define ADC_NCMR_CH32_MASK                       0x1u
#define ADC_NCMR_CH32_SHIFT                      0u
#define ADC_NCMR_CH32_WIDTH                      1u
#define ADC_NCMR_CH32(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH32_SHIFT))&ADC_NCMR_CH32_MASK)
#define ADC_NCMR_CH1_MASK                        0x2u
#define ADC_NCMR_CH1_SHIFT                       1u
#define ADC_NCMR_CH1_WIDTH                       1u
#define ADC_NCMR_CH1(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH1_SHIFT))&ADC_NCMR_CH1_MASK)
#define ADC_NCMR_CH33_MASK                       0x2u
#define ADC_NCMR_CH33_SHIFT                      1u
#define ADC_NCMR_CH33_WIDTH                      1u
#define ADC_NCMR_CH33(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH33_SHIFT))&ADC_NCMR_CH33_MASK)
#define ADC_NCMR_CH34_MASK                       0x4u
#define ADC_NCMR_CH34_SHIFT                      2u
#define ADC_NCMR_CH34_WIDTH                      1u
#define ADC_NCMR_CH34(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH34_SHIFT))&ADC_NCMR_CH34_MASK)
#define ADC_NCMR_CH2_MASK                        0x4u
#define ADC_NCMR_CH2_SHIFT                       2u
#define ADC_NCMR_CH2_WIDTH                       1u
#define ADC_NCMR_CH2(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH2_SHIFT))&ADC_NCMR_CH2_MASK)
#define ADC_NCMR_CH35_MASK                       0x8u
#define ADC_NCMR_CH35_SHIFT                      3u
#define ADC_NCMR_CH35_WIDTH                      1u
#define ADC_NCMR_CH35(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH35_SHIFT))&ADC_NCMR_CH35_MASK)
#define ADC_NCMR_CH3_MASK                        0x8u
#define ADC_NCMR_CH3_SHIFT                       3u
#define ADC_NCMR_CH3_WIDTH                       1u
#define ADC_NCMR_CH3(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH3_SHIFT))&ADC_NCMR_CH3_MASK)
#define ADC_NCMR_CH36_MASK                       0x10u
#define ADC_NCMR_CH36_SHIFT                      4u
#define ADC_NCMR_CH36_WIDTH                      1u
#define ADC_NCMR_CH36(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH36_SHIFT))&ADC_NCMR_CH36_MASK)
#define ADC_NCMR_CH4_MASK                        0x10u
#define ADC_NCMR_CH4_SHIFT                       4u
#define ADC_NCMR_CH4_WIDTH                       1u
#define ADC_NCMR_CH4(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH4_SHIFT))&ADC_NCMR_CH4_MASK)
#define ADC_NCMR_CH5_MASK                        0x20u
#define ADC_NCMR_CH5_SHIFT                       5u
#define ADC_NCMR_CH5_WIDTH                       1u
#define ADC_NCMR_CH5(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH5_SHIFT))&ADC_NCMR_CH5_MASK)
#define ADC_NCMR_CH37_MASK                       0x20u
#define ADC_NCMR_CH37_SHIFT                      5u
#define ADC_NCMR_CH37_WIDTH                      1u
#define ADC_NCMR_CH37(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH37_SHIFT))&ADC_NCMR_CH37_MASK)
#define ADC_NCMR_CH38_MASK                       0x40u
#define ADC_NCMR_CH38_SHIFT                      6u
#define ADC_NCMR_CH38_WIDTH                      1u
#define ADC_NCMR_CH38(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH38_SHIFT))&ADC_NCMR_CH38_MASK)
#define ADC_NCMR_CH6_MASK                        0x40u
#define ADC_NCMR_CH6_SHIFT                       6u
#define ADC_NCMR_CH6_WIDTH                       1u
#define ADC_NCMR_CH6(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH6_SHIFT))&ADC_NCMR_CH6_MASK)
#define ADC_NCMR_CH7_MASK                        0x80u
#define ADC_NCMR_CH7_SHIFT                       7u
#define ADC_NCMR_CH7_WIDTH                       1u
#define ADC_NCMR_CH7(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH7_SHIFT))&ADC_NCMR_CH7_MASK)
#define ADC_NCMR_CH39_MASK                       0x80u
#define ADC_NCMR_CH39_SHIFT                      7u
#define ADC_NCMR_CH39_WIDTH                      1u
#define ADC_NCMR_CH39(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH39_SHIFT))&ADC_NCMR_CH39_MASK)
/* JCMR Bit Fields */
#define ADC_JCMR_CH0_MASK                        0x1u
#define ADC_JCMR_CH0_SHIFT                       0u
#define ADC_JCMR_CH0_WIDTH                       1u
#define ADC_JCMR_CH0(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH0_SHIFT))&ADC_JCMR_CH0_MASK)
#define ADC_JCMR_CH32_MASK                       0x1u
#define ADC_JCMR_CH32_SHIFT                      0u
#define ADC_JCMR_CH32_WIDTH                      1u
#define ADC_JCMR_CH32(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH32_SHIFT))&ADC_JCMR_CH32_MASK)
#define ADC_JCMR_CH1_MASK                        0x2u
#define ADC_JCMR_CH1_SHIFT                       1u
#define ADC_JCMR_CH1_WIDTH                       1u
#define ADC_JCMR_CH1(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH1_SHIFT))&ADC_JCMR_CH1_MASK)
#define ADC_JCMR_CH33_MASK                       0x2u
#define ADC_JCMR_CH33_SHIFT                      1u
#define ADC_JCMR_CH33_WIDTH                      1u
#define ADC_JCMR_CH33(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH33_SHIFT))&ADC_JCMR_CH33_MASK)
#define ADC_JCMR_CH34_MASK                       0x4u
#define ADC_JCMR_CH34_SHIFT                      2u
#define ADC_JCMR_CH34_WIDTH                      1u
#define ADC_JCMR_CH34(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH34_SHIFT))&ADC_JCMR_CH34_MASK)
#define ADC_JCMR_CH2_MASK                        0x4u
#define ADC_JCMR_CH2_SHIFT                       2u
#define ADC_JCMR_CH2_WIDTH                       1u
#define ADC_JCMR_CH2(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH2_SHIFT))&ADC_JCMR_CH2_MASK)
#define ADC_JCMR_CH35_MASK                       0x8u
#define ADC_JCMR_CH35_SHIFT                      3u
#define ADC_JCMR_CH35_WIDTH                      1u
#define ADC_JCMR_CH35(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH35_SHIFT))&ADC_JCMR_CH35_MASK)
#define ADC_JCMR_CH3_MASK                        0x8u
#define ADC_JCMR_CH3_SHIFT                       3u
#define ADC_JCMR_CH3_WIDTH                       1u
#define ADC_JCMR_CH3(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH3_SHIFT))&ADC_JCMR_CH3_MASK)
#define ADC_JCMR_CH36_MASK                       0x10u
#define ADC_JCMR_CH36_SHIFT                      4u
#define ADC_JCMR_CH36_WIDTH                      1u
#define ADC_JCMR_CH36(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH36_SHIFT))&ADC_JCMR_CH36_MASK)
#define ADC_JCMR_CH4_MASK                        0x10u
#define ADC_JCMR_CH4_SHIFT                       4u
#define ADC_JCMR_CH4_WIDTH                       1u
#define ADC_JCMR_CH4(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH4_SHIFT))&ADC_JCMR_CH4_MASK)
#define ADC_JCMR_CH5_MASK                        0x20u
#define ADC_JCMR_CH5_SHIFT                       5u
#define ADC_JCMR_CH5_WIDTH                       1u
#define ADC_JCMR_CH5(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH5_SHIFT))&ADC_JCMR_CH5_MASK)
#define ADC_JCMR_CH37_MASK                       0x20u
#define ADC_JCMR_CH37_SHIFT                      5u
#define ADC_JCMR_CH37_WIDTH                      1u
#define ADC_JCMR_CH37(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH37_SHIFT))&ADC_JCMR_CH37_MASK)
#define ADC_JCMR_CH38_MASK                       0x40u
#define ADC_JCMR_CH38_SHIFT                      6u
#define ADC_JCMR_CH38_WIDTH                      1u
#define ADC_JCMR_CH38(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH38_SHIFT))&ADC_JCMR_CH38_MASK)
#define ADC_JCMR_CH6_MASK                        0x40u
#define ADC_JCMR_CH6_SHIFT                       6u
#define ADC_JCMR_CH6_WIDTH                       1u
#define ADC_JCMR_CH6(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH6_SHIFT))&ADC_JCMR_CH6_MASK)
#define ADC_JCMR_CH7_MASK                        0x80u
#define ADC_JCMR_CH7_SHIFT                       7u
#define ADC_JCMR_CH7_WIDTH                       1u
#define ADC_JCMR_CH7(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH7_SHIFT))&ADC_JCMR_CH7_MASK)
#define ADC_JCMR_CH39_MASK                       0x80u
#define ADC_JCMR_CH39_SHIFT                      7u
#define ADC_JCMR_CH39_WIDTH                      1u
#define ADC_JCMR_CH39(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH39_SHIFT))&ADC_JCMR_CH39_MASK)
/* USROFSGN Bit Fields */
#define ADC_USROFSGN_OFFSUSER_MASK               0xFFu
#define ADC_USROFSGN_OFFSUSER_SHIFT              0u
#define ADC_USROFSGN_OFFSUSER_WIDTH              8u
#define ADC_USROFSGN_OFFSUSER(x)                 (((uint32_t)(((uint32_t)(x))<<ADC_USROFSGN_OFFSUSER_SHIFT))&ADC_USROFSGN_OFFSUSER_MASK)
#define ADC_USROFSGN_GAINUSER_MASK               0x3FF0000u
#define ADC_USROFSGN_GAINUSER_SHIFT              16u
#define ADC_USROFSGN_GAINUSER_WIDTH              10u
#define ADC_USROFSGN_GAINUSER(x)                 (((uint32_t)(((uint32_t)(x))<<ADC_USROFSGN_GAINUSER_SHIFT))&ADC_USROFSGN_GAINUSER_MASK)
/* PDEDR Bit Fields */
#define ADC_PDEDR_PDED_MASK                      0xFFu
#define ADC_PDEDR_PDED_SHIFT                     0u
#define ADC_PDEDR_PDED_WIDTH                     8u
#define ADC_PDEDR_PDED(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PDEDR_PDED_SHIFT))&ADC_PDEDR_PDED_MASK)
/* CDR Bit Fields */
#define ADC_CDR_CDATA_MASK                       0xFFFu
#define ADC_CDR_CDATA_SHIFT                      0u
#define ADC_CDR_CDATA_WIDTH                      12u
#define ADC_CDR_CDATA(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CDR_CDATA_SHIFT))&ADC_CDR_CDATA_MASK)
#define ADC_CDR_RESULT_MASK                      0x30000u
#define ADC_CDR_RESULT_SHIFT                     16u
#define ADC_CDR_RESULT_WIDTH                     2u
#define ADC_CDR_RESULT(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CDR_RESULT_SHIFT))&ADC_CDR_RESULT_MASK)
#define ADC_CDR_OVERW_MASK                       0x40000u
#define ADC_CDR_OVERW_SHIFT                      18u
#define ADC_CDR_OVERW_WIDTH                      1u
#define ADC_CDR_OVERW(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CDR_OVERW_SHIFT))&ADC_CDR_OVERW_MASK)
#define ADC_CDR_VALID_MASK                       0x80000u
#define ADC_CDR_VALID_SHIFT                      19u
#define ADC_CDR_VALID_WIDTH                      1u
#define ADC_CDR_VALID(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CDR_VALID_SHIFT))&ADC_CDR_VALID_MASK)
/* THRHLR4 Bit Fields */
#define ADC_THRHLR4_THRL_MASK                    0xFFFu
#define ADC_THRHLR4_THRL_SHIFT                   0u
#define ADC_THRHLR4_THRL_WIDTH                   12u
#define ADC_THRHLR4_THRL(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR4_THRL_SHIFT))&ADC_THRHLR4_THRL_MASK)
#define ADC_THRHLR4_THRH_MASK                    0xFFF0000u
#define ADC_THRHLR4_THRH_SHIFT                   16u
#define ADC_THRHLR4_THRH_WIDTH                   12u
#define ADC_THRHLR4_THRH(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR4_THRH_SHIFT))&ADC_THRHLR4_THRH_MASK)
/* THRHLR5 Bit Fields */
#define ADC_THRHLR5_THRL_MASK                    0xFFFu
#define ADC_THRHLR5_THRL_SHIFT                   0u
#define ADC_THRHLR5_THRL_WIDTH                   12u
#define ADC_THRHLR5_THRL(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR5_THRL_SHIFT))&ADC_THRHLR5_THRL_MASK)
#define ADC_THRHLR5_THRH_MASK                    0xFFF0000u
#define ADC_THRHLR5_THRH_SHIFT                   16u
#define ADC_THRHLR5_THRH_WIDTH                   12u
#define ADC_THRHLR5_THRH(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR5_THRH_SHIFT))&ADC_THRHLR5_THRH_MASK)
/* THRHLR6 Bit Fields */
#define ADC_THRHLR6_THRL_MASK                    0xFFFu
#define ADC_THRHLR6_THRL_SHIFT                   0u
#define ADC_THRHLR6_THRL_WIDTH                   12u
#define ADC_THRHLR6_THRL(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR6_THRL_SHIFT))&ADC_THRHLR6_THRL_MASK)
#define ADC_THRHLR6_THRH_MASK                    0xFFF0000u
#define ADC_THRHLR6_THRH_SHIFT                   16u
#define ADC_THRHLR6_THRH_WIDTH                   12u
#define ADC_THRHLR6_THRH(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR6_THRH_SHIFT))&ADC_THRHLR6_THRH_MASK)
/* THRHLR7 Bit Fields */
#define ADC_THRHLR7_THRL_MASK                    0xFFFu
#define ADC_THRHLR7_THRL_SHIFT                   0u
#define ADC_THRHLR7_THRL_WIDTH                   12u
#define ADC_THRHLR7_THRL(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR7_THRL_SHIFT))&ADC_THRHLR7_THRL_MASK)
#define ADC_THRHLR7_THRH_MASK                    0xFFF0000u
#define ADC_THRHLR7_THRH_SHIFT                   16u
#define ADC_THRHLR7_THRH_WIDTH                   12u
#define ADC_THRHLR7_THRH(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR7_THRH_SHIFT))&ADC_THRHLR7_THRH_MASK)
/* CWSELR0 Bit Fields */
#define ADC_CWSELR0_WSEL_CH0_MASK                0xFu
#define ADC_CWSELR0_WSEL_CH0_SHIFT               0u
#define ADC_CWSELR0_WSEL_CH0_WIDTH               4u
#define ADC_CWSELR0_WSEL_CH0(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR0_WSEL_CH0_SHIFT))&ADC_CWSELR0_WSEL_CH0_MASK)
#define ADC_CWSELR0_WSEL_CH1_MASK                0xF0u
#define ADC_CWSELR0_WSEL_CH1_SHIFT               4u
#define ADC_CWSELR0_WSEL_CH1_WIDTH               4u
#define ADC_CWSELR0_WSEL_CH1(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR0_WSEL_CH1_SHIFT))&ADC_CWSELR0_WSEL_CH1_MASK)
#define ADC_CWSELR0_WSEL_CH2_MASK                0xF00u
#define ADC_CWSELR0_WSEL_CH2_SHIFT               8u
#define ADC_CWSELR0_WSEL_CH2_WIDTH               4u
#define ADC_CWSELR0_WSEL_CH2(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR0_WSEL_CH2_SHIFT))&ADC_CWSELR0_WSEL_CH2_MASK)
#define ADC_CWSELR0_WSEL_CH3_MASK                0xF000u
#define ADC_CWSELR0_WSEL_CH3_SHIFT               12u
#define ADC_CWSELR0_WSEL_CH3_WIDTH               4u
#define ADC_CWSELR0_WSEL_CH3(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR0_WSEL_CH3_SHIFT))&ADC_CWSELR0_WSEL_CH3_MASK)
#define ADC_CWSELR0_WSEL_CH4_MASK                0xF0000u
#define ADC_CWSELR0_WSEL_CH4_SHIFT               16u
#define ADC_CWSELR0_WSEL_CH4_WIDTH               4u
#define ADC_CWSELR0_WSEL_CH4(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR0_WSEL_CH4_SHIFT))&ADC_CWSELR0_WSEL_CH4_MASK)
#define ADC_CWSELR0_WSEL_CH5_MASK                0xF00000u
#define ADC_CWSELR0_WSEL_CH5_SHIFT               20u
#define ADC_CWSELR0_WSEL_CH5_WIDTH               4u
#define ADC_CWSELR0_WSEL_CH5(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR0_WSEL_CH5_SHIFT))&ADC_CWSELR0_WSEL_CH5_MASK)
#define ADC_CWSELR0_WSEL_CH6_MASK                0xF000000u
#define ADC_CWSELR0_WSEL_CH6_SHIFT               24u
#define ADC_CWSELR0_WSEL_CH6_WIDTH               4u
#define ADC_CWSELR0_WSEL_CH6(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR0_WSEL_CH6_SHIFT))&ADC_CWSELR0_WSEL_CH6_MASK)
#define ADC_CWSELR0_WSEL_CH7_MASK                0xF0000000u
#define ADC_CWSELR0_WSEL_CH7_SHIFT               28u
#define ADC_CWSELR0_WSEL_CH7_WIDTH               4u
#define ADC_CWSELR0_WSEL_CH7(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR0_WSEL_CH7_SHIFT))&ADC_CWSELR0_WSEL_CH7_MASK)
/* CWSELR4 Bit Fields */
#define ADC_CWSELR4_WSEL_CH32_MASK               0xFu
#define ADC_CWSELR4_WSEL_CH32_SHIFT              0u
#define ADC_CWSELR4_WSEL_CH32_WIDTH              4u
#define ADC_CWSELR4_WSEL_CH32(x)                 (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR4_WSEL_CH32_SHIFT))&ADC_CWSELR4_WSEL_CH32_MASK)
#define ADC_CWSELR4_WSEL_CH33_MASK               0xF0u
#define ADC_CWSELR4_WSEL_CH33_SHIFT              4u
#define ADC_CWSELR4_WSEL_CH33_WIDTH              4u
#define ADC_CWSELR4_WSEL_CH33(x)                 (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR4_WSEL_CH33_SHIFT))&ADC_CWSELR4_WSEL_CH33_MASK)
#define ADC_CWSELR4_WSEL_CH34_MASK               0xF00u
#define ADC_CWSELR4_WSEL_CH34_SHIFT              8u
#define ADC_CWSELR4_WSEL_CH34_WIDTH              4u
#define ADC_CWSELR4_WSEL_CH34(x)                 (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR4_WSEL_CH34_SHIFT))&ADC_CWSELR4_WSEL_CH34_MASK)
#define ADC_CWSELR4_WSEL_CH35_MASK               0xF000u
#define ADC_CWSELR4_WSEL_CH35_SHIFT              12u
#define ADC_CWSELR4_WSEL_CH35_WIDTH              4u
#define ADC_CWSELR4_WSEL_CH35(x)                 (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR4_WSEL_CH35_SHIFT))&ADC_CWSELR4_WSEL_CH35_MASK)
#define ADC_CWSELR4_WSEL_CH36_MASK               0xF0000u
#define ADC_CWSELR4_WSEL_CH36_SHIFT              16u
#define ADC_CWSELR4_WSEL_CH36_WIDTH              4u
#define ADC_CWSELR4_WSEL_CH36(x)                 (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR4_WSEL_CH36_SHIFT))&ADC_CWSELR4_WSEL_CH36_MASK)
#define ADC_CWSELR4_WSEL_CH37_MASK               0xF00000u
#define ADC_CWSELR4_WSEL_CH37_SHIFT              20u
#define ADC_CWSELR4_WSEL_CH37_WIDTH              4u
#define ADC_CWSELR4_WSEL_CH37(x)                 (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR4_WSEL_CH37_SHIFT))&ADC_CWSELR4_WSEL_CH37_MASK)
#define ADC_CWSELR4_WSEL_CH38_MASK               0xF000000u
#define ADC_CWSELR4_WSEL_CH38_SHIFT              24u
#define ADC_CWSELR4_WSEL_CH38_WIDTH              4u
#define ADC_CWSELR4_WSEL_CH38(x)                 (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR4_WSEL_CH38_SHIFT))&ADC_CWSELR4_WSEL_CH38_MASK)
#define ADC_CWSELR4_WSEL_CH39_MASK               0xF0000000u
#define ADC_CWSELR4_WSEL_CH39_SHIFT              28u
#define ADC_CWSELR4_WSEL_CH39_WIDTH              4u
#define ADC_CWSELR4_WSEL_CH39(x)                 (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR4_WSEL_CH39_SHIFT))&ADC_CWSELR4_WSEL_CH39_MASK)
/* CWENR Bit Fields */
#define ADC_CWENR_CWEN0_MASK                     0x1u
#define ADC_CWENR_CWEN0_SHIFT                    0u
#define ADC_CWENR_CWEN0_WIDTH                    1u
#define ADC_CWENR_CWEN0(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN0_SHIFT))&ADC_CWENR_CWEN0_MASK)
#define ADC_CWENR_CWEN32_MASK                    0x1u
#define ADC_CWENR_CWEN32_SHIFT                   0u
#define ADC_CWENR_CWEN32_WIDTH                   1u
#define ADC_CWENR_CWEN32(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN32_SHIFT))&ADC_CWENR_CWEN32_MASK)
#define ADC_CWENR_CWEN1_MASK                     0x2u
#define ADC_CWENR_CWEN1_SHIFT                    1u
#define ADC_CWENR_CWEN1_WIDTH                    1u
#define ADC_CWENR_CWEN1(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN1_SHIFT))&ADC_CWENR_CWEN1_MASK)
#define ADC_CWENR_CWEN33_MASK                    0x2u
#define ADC_CWENR_CWEN33_SHIFT                   1u
#define ADC_CWENR_CWEN33_WIDTH                   1u
#define ADC_CWENR_CWEN33(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN33_SHIFT))&ADC_CWENR_CWEN33_MASK)
#define ADC_CWENR_CWEN34_MASK                    0x4u
#define ADC_CWENR_CWEN34_SHIFT                   2u
#define ADC_CWENR_CWEN34_WIDTH                   1u
#define ADC_CWENR_CWEN34(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN34_SHIFT))&ADC_CWENR_CWEN34_MASK)
#define ADC_CWENR_CWEN2_MASK                     0x4u
#define ADC_CWENR_CWEN2_SHIFT                    2u
#define ADC_CWENR_CWEN2_WIDTH                    1u
#define ADC_CWENR_CWEN2(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN2_SHIFT))&ADC_CWENR_CWEN2_MASK)
#define ADC_CWENR_CWEN35_MASK                    0x8u
#define ADC_CWENR_CWEN35_SHIFT                   3u
#define ADC_CWENR_CWEN35_WIDTH                   1u
#define ADC_CWENR_CWEN35(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN35_SHIFT))&ADC_CWENR_CWEN35_MASK)
#define ADC_CWENR_CWEN3_MASK                     0x8u
#define ADC_CWENR_CWEN3_SHIFT                    3u
#define ADC_CWENR_CWEN3_WIDTH                    1u
#define ADC_CWENR_CWEN3(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN3_SHIFT))&ADC_CWENR_CWEN3_MASK)
#define ADC_CWENR_CWEN36_MASK                    0x10u
#define ADC_CWENR_CWEN36_SHIFT                   4u
#define ADC_CWENR_CWEN36_WIDTH                   1u
#define ADC_CWENR_CWEN36(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN36_SHIFT))&ADC_CWENR_CWEN36_MASK)
#define ADC_CWENR_CWEN4_MASK                     0x10u
#define ADC_CWENR_CWEN4_SHIFT                    4u
#define ADC_CWENR_CWEN4_WIDTH                    1u
#define ADC_CWENR_CWEN4(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN4_SHIFT))&ADC_CWENR_CWEN4_MASK)
#define ADC_CWENR_CWEN5_MASK                     0x20u
#define ADC_CWENR_CWEN5_SHIFT                    5u
#define ADC_CWENR_CWEN5_WIDTH                    1u
#define ADC_CWENR_CWEN5(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN5_SHIFT))&ADC_CWENR_CWEN5_MASK)
#define ADC_CWENR_CWEN37_MASK                    0x20u
#define ADC_CWENR_CWEN37_SHIFT                   5u
#define ADC_CWENR_CWEN37_WIDTH                   1u
#define ADC_CWENR_CWEN37(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN37_SHIFT))&ADC_CWENR_CWEN37_MASK)
#define ADC_CWENR_CWEN38_MASK                    0x40u
#define ADC_CWENR_CWEN38_SHIFT                   6u
#define ADC_CWENR_CWEN38_WIDTH                   1u
#define ADC_CWENR_CWEN38(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN38_SHIFT))&ADC_CWENR_CWEN38_MASK)
#define ADC_CWENR_CWEN6_MASK                     0x40u
#define ADC_CWENR_CWEN6_SHIFT                    6u
#define ADC_CWENR_CWEN6_WIDTH                    1u
#define ADC_CWENR_CWEN6(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN6_SHIFT))&ADC_CWENR_CWEN6_MASK)
#define ADC_CWENR_CWEN7_MASK                     0x80u
#define ADC_CWENR_CWEN7_SHIFT                    7u
#define ADC_CWENR_CWEN7_WIDTH                    1u
#define ADC_CWENR_CWEN7(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN7_SHIFT))&ADC_CWENR_CWEN7_MASK)
#define ADC_CWENR_CWEN39_MASK                    0x80u
#define ADC_CWENR_CWEN39_SHIFT                   7u
#define ADC_CWENR_CWEN39_WIDTH                   1u
#define ADC_CWENR_CWEN39(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN39_SHIFT))&ADC_CWENR_CWEN39_MASK)
/* AWORR Bit Fields */
#define ADC_AWORR_AWOR_CH0_MASK                  0x1u
#define ADC_AWORR_AWOR_CH0_SHIFT                 0u
#define ADC_AWORR_AWOR_CH0_WIDTH                 1u
#define ADC_AWORR_AWOR_CH0(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH0_SHIFT))&ADC_AWORR_AWOR_CH0_MASK)
#define ADC_AWORR_AWOR_CH32_MASK                 0x1u
#define ADC_AWORR_AWOR_CH32_SHIFT                0u
#define ADC_AWORR_AWOR_CH32_WIDTH                1u
#define ADC_AWORR_AWOR_CH32(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH32_SHIFT))&ADC_AWORR_AWOR_CH32_MASK)
#define ADC_AWORR_AWOR_CH1_MASK                  0x2u
#define ADC_AWORR_AWOR_CH1_SHIFT                 1u
#define ADC_AWORR_AWOR_CH1_WIDTH                 1u
#define ADC_AWORR_AWOR_CH1(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH1_SHIFT))&ADC_AWORR_AWOR_CH1_MASK)
#define ADC_AWORR_AWOR_CH33_MASK                 0x2u
#define ADC_AWORR_AWOR_CH33_SHIFT                1u
#define ADC_AWORR_AWOR_CH33_WIDTH                1u
#define ADC_AWORR_AWOR_CH33(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH33_SHIFT))&ADC_AWORR_AWOR_CH33_MASK)
#define ADC_AWORR_AWOR_CH34_MASK                 0x4u
#define ADC_AWORR_AWOR_CH34_SHIFT                2u
#define ADC_AWORR_AWOR_CH34_WIDTH                1u
#define ADC_AWORR_AWOR_CH34(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH34_SHIFT))&ADC_AWORR_AWOR_CH34_MASK)
#define ADC_AWORR_AWOR_CH2_MASK                  0x4u
#define ADC_AWORR_AWOR_CH2_SHIFT                 2u
#define ADC_AWORR_AWOR_CH2_WIDTH                 1u
#define ADC_AWORR_AWOR_CH2(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH2_SHIFT))&ADC_AWORR_AWOR_CH2_MASK)
#define ADC_AWORR_AWOR_CH35_MASK                 0x8u
#define ADC_AWORR_AWOR_CH35_SHIFT                3u
#define ADC_AWORR_AWOR_CH35_WIDTH                1u
#define ADC_AWORR_AWOR_CH35(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH35_SHIFT))&ADC_AWORR_AWOR_CH35_MASK)
#define ADC_AWORR_AWOR_CH3_MASK                  0x8u
#define ADC_AWORR_AWOR_CH3_SHIFT                 3u
#define ADC_AWORR_AWOR_CH3_WIDTH                 1u
#define ADC_AWORR_AWOR_CH3(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH3_SHIFT))&ADC_AWORR_AWOR_CH3_MASK)
#define ADC_AWORR_AWOR_CH36_MASK                 0x10u
#define ADC_AWORR_AWOR_CH36_SHIFT                4u
#define ADC_AWORR_AWOR_CH36_WIDTH                1u
#define ADC_AWORR_AWOR_CH36(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH36_SHIFT))&ADC_AWORR_AWOR_CH36_MASK)
#define ADC_AWORR_AWOR_CH4_MASK                  0x10u
#define ADC_AWORR_AWOR_CH4_SHIFT                 4u
#define ADC_AWORR_AWOR_CH4_WIDTH                 1u
#define ADC_AWORR_AWOR_CH4(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH4_SHIFT))&ADC_AWORR_AWOR_CH4_MASK)
#define ADC_AWORR_AWOR_CH5_MASK                  0x20u
#define ADC_AWORR_AWOR_CH5_SHIFT                 5u
#define ADC_AWORR_AWOR_CH5_WIDTH                 1u
#define ADC_AWORR_AWOR_CH5(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH5_SHIFT))&ADC_AWORR_AWOR_CH5_MASK)
#define ADC_AWORR_AWOR_CH37_MASK                 0x20u
#define ADC_AWORR_AWOR_CH37_SHIFT                5u
#define ADC_AWORR_AWOR_CH37_WIDTH                1u
#define ADC_AWORR_AWOR_CH37(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH37_SHIFT))&ADC_AWORR_AWOR_CH37_MASK)
#define ADC_AWORR_AWOR_CH38_MASK                 0x40u
#define ADC_AWORR_AWOR_CH38_SHIFT                6u
#define ADC_AWORR_AWOR_CH38_WIDTH                1u
#define ADC_AWORR_AWOR_CH38(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH38_SHIFT))&ADC_AWORR_AWOR_CH38_MASK)
#define ADC_AWORR_AWOR_CH6_MASK                  0x40u
#define ADC_AWORR_AWOR_CH6_SHIFT                 6u
#define ADC_AWORR_AWOR_CH6_WIDTH                 1u
#define ADC_AWORR_AWOR_CH6(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH6_SHIFT))&ADC_AWORR_AWOR_CH6_MASK)
#define ADC_AWORR_AWOR_CH7_MASK                  0x80u
#define ADC_AWORR_AWOR_CH7_SHIFT                 7u
#define ADC_AWORR_AWOR_CH7_WIDTH                 1u
#define ADC_AWORR_AWOR_CH7(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH7_SHIFT))&ADC_AWORR_AWOR_CH7_MASK)
#define ADC_AWORR_AWOR_CH39_MASK                 0x80u
#define ADC_AWORR_AWOR_CH39_SHIFT                7u
#define ADC_AWORR_AWOR_CH39_WIDTH                1u
#define ADC_AWORR_AWOR_CH39(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH39_SHIFT))&ADC_AWORR_AWOR_CH39_MASK)
/* STCR1 Bit Fields */
#define ADC_STCR1_INPSAMP_S_MASK                 0xFF00u
#define ADC_STCR1_INPSAMP_S_SHIFT                8u
#define ADC_STCR1_INPSAMP_S_WIDTH                8u
#define ADC_STCR1_INPSAMP_S(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_STCR1_INPSAMP_S_SHIFT))&ADC_STCR1_INPSAMP_S_MASK)
#define ADC_STCR1_INPSAMP_C_MASK                 0xFF000000u
#define ADC_STCR1_INPSAMP_C_SHIFT                24u
#define ADC_STCR1_INPSAMP_C_WIDTH                8u
#define ADC_STCR1_INPSAMP_C(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_STCR1_INPSAMP_C_SHIFT))&ADC_STCR1_INPSAMP_C_MASK)
/* STCR2 Bit Fields */
#define ADC_STCR2_FMA_S_MASK                     0x1u
#define ADC_STCR2_FMA_S_SHIFT                    0u
#define ADC_STCR2_FMA_S_WIDTH                    1u
#define ADC_STCR2_FMA_S(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STCR2_FMA_S_SHIFT))&ADC_STCR2_FMA_S_MASK)
#define ADC_STCR2_FMA_C_MASK                     0x4u
#define ADC_STCR2_FMA_C_SHIFT                    2u
#define ADC_STCR2_FMA_C_WIDTH                    1u
#define ADC_STCR2_FMA_C(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STCR2_FMA_C_SHIFT))&ADC_STCR2_FMA_C_MASK)
#define ADC_STCR2_FMA_WDTERR_MASK                0x8u
#define ADC_STCR2_FMA_WDTERR_SHIFT               3u
#define ADC_STCR2_FMA_WDTERR_WIDTH               1u
#define ADC_STCR2_FMA_WDTERR(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_STCR2_FMA_WDTERR_SHIFT))&ADC_STCR2_FMA_WDTERR_MASK)
#define ADC_STCR2_FMA_WDSERR_MASK                0x10u
#define ADC_STCR2_FMA_WDSERR_SHIFT               4u
#define ADC_STCR2_FMA_WDSERR_WIDTH               1u
#define ADC_STCR2_FMA_WDSERR(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_STCR2_FMA_WDSERR_SHIFT))&ADC_STCR2_FMA_WDSERR_MASK)
#define ADC_STCR2_EN_MASK                        0x80u
#define ADC_STCR2_EN_SHIFT                       7u
#define ADC_STCR2_EN_WIDTH                       1u
#define ADC_STCR2_EN(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_STCR2_EN_SHIFT))&ADC_STCR2_EN_MASK)
#define ADC_STCR2_MSKERR_S0_MASK                 0x800u
#define ADC_STCR2_MSKERR_S0_SHIFT                11u
#define ADC_STCR2_MSKERR_S0_WIDTH                1u
#define ADC_STCR2_MSKERR_S0(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_STCR2_MSKERR_S0_SHIFT))&ADC_STCR2_MSKERR_S0_MASK)
#define ADC_STCR2_MSKERR_S1_MASK                 0x1000u
#define ADC_STCR2_MSKERR_S1_SHIFT                12u
#define ADC_STCR2_MSKERR_S1_WIDTH                1u
#define ADC_STCR2_MSKERR_S1(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_STCR2_MSKERR_S1_SHIFT))&ADC_STCR2_MSKERR_S1_MASK)
#define ADC_STCR2_MSKERR_S2_MASK                 0x2000u
#define ADC_STCR2_MSKERR_S2_SHIFT                13u
#define ADC_STCR2_MSKERR_S2_WIDTH                1u
#define ADC_STCR2_MSKERR_S2(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_STCR2_MSKERR_S2_SHIFT))&ADC_STCR2_MSKERR_S2_MASK)
#define ADC_STCR2_MSKERR_C_MASK                  0x8000u
#define ADC_STCR2_MSKERR_C_SHIFT                 15u
#define ADC_STCR2_MSKERR_C_WIDTH                 1u
#define ADC_STCR2_MSKERR_C(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_STCR2_MSKERR_C_SHIFT))&ADC_STCR2_MSKERR_C_MASK)
#define ADC_STCR2_MSKWDG_EOA_S_MASK              0x10000u
#define ADC_STCR2_MSKWDG_EOA_S_SHIFT             16u
#define ADC_STCR2_MSKWDG_EOA_S_WIDTH             1u
#define ADC_STCR2_MSKWDG_EOA_S(x)                (((uint32_t)(((uint32_t)(x))<<ADC_STCR2_MSKWDG_EOA_S_SHIFT))&ADC_STCR2_MSKWDG_EOA_S_MASK)
#define ADC_STCR2_MSKWDG_EOA_C_MASK              0x40000u
#define ADC_STCR2_MSKWDG_EOA_C_SHIFT             18u
#define ADC_STCR2_MSKWDG_EOA_C_WIDTH             1u
#define ADC_STCR2_MSKWDG_EOA_C(x)                (((uint32_t)(((uint32_t)(x))<<ADC_STCR2_MSKWDG_EOA_C_SHIFT))&ADC_STCR2_MSKWDG_EOA_C_MASK)
#define ADC_STCR2_MSKST_EOC_MASK                 0x800000u
#define ADC_STCR2_MSKST_EOC_SHIFT                23u
#define ADC_STCR2_MSKST_EOC_WIDTH                1u
#define ADC_STCR2_MSKST_EOC(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_STCR2_MSKST_EOC_SHIFT))&ADC_STCR2_MSKST_EOC_MASK)
#define ADC_STCR2_MSKWDTERR_MASK                 0x2000000u
#define ADC_STCR2_MSKWDTERR_SHIFT                25u
#define ADC_STCR2_MSKWDTERR_WIDTH                1u
#define ADC_STCR2_MSKWDTERR(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_STCR2_MSKWDTERR_SHIFT))&ADC_STCR2_MSKWDTERR_MASK)
#define ADC_STCR2_SERR_MASK                      0x4000000u
#define ADC_STCR2_SERR_SHIFT                     26u
#define ADC_STCR2_SERR_WIDTH                     1u
#define ADC_STCR2_SERR(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_STCR2_SERR_SHIFT))&ADC_STCR2_SERR_MASK)
#define ADC_STCR2_MSKWDSERR_MASK                 0x8000000u
#define ADC_STCR2_MSKWDSERR_SHIFT                27u
#define ADC_STCR2_MSKWDSERR_WIDTH                1u
#define ADC_STCR2_MSKWDSERR(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_STCR2_MSKWDSERR_SHIFT))&ADC_STCR2_MSKWDSERR_MASK)
/* STCR3 Bit Fields */
#define ADC_STCR3_MSTEP_MASK                     0x1Fu
#define ADC_STCR3_MSTEP_SHIFT                    0u
#define ADC_STCR3_MSTEP_WIDTH                    5u
#define ADC_STCR3_MSTEP(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STCR3_MSTEP_SHIFT))&ADC_STCR3_MSTEP_MASK)
#define ADC_STCR3_ALG_MASK                       0x300u
#define ADC_STCR3_ALG_SHIFT                      8u
#define ADC_STCR3_ALG_WIDTH                      2u
#define ADC_STCR3_ALG(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_STCR3_ALG_SHIFT))&ADC_STCR3_ALG_MASK)
/* STBRR Bit Fields */
#define ADC_STBRR_BR_MASK                        0xFFu
#define ADC_STBRR_BR_SHIFT                       0u
#define ADC_STBRR_BR_WIDTH                       8u
#define ADC_STBRR_BR(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_STBRR_BR_SHIFT))&ADC_STBRR_BR_MASK)
#define ADC_STBRR_WDT_MASK                       0x70000u
#define ADC_STBRR_WDT_SHIFT                      16u
#define ADC_STBRR_WDT_WIDTH                      3u
#define ADC_STBRR_WDT(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_STBRR_WDT_SHIFT))&ADC_STBRR_WDT_MASK)
/* STSR1 Bit Fields */
#define ADC_STSR1_STEP_C_MASK                    0x3E0u
#define ADC_STSR1_STEP_C_SHIFT                   5u
#define ADC_STSR1_STEP_C_WIDTH                   5u
#define ADC_STSR1_STEP_C(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_STSR1_STEP_C_SHIFT))&ADC_STSR1_STEP_C_MASK)
#define ADC_STSR1_ERR_S0_MASK                    0x800u
#define ADC_STSR1_ERR_S0_SHIFT                   11u
#define ADC_STSR1_ERR_S0_WIDTH                   1u
#define ADC_STSR1_ERR_S0(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_STSR1_ERR_S0_SHIFT))&ADC_STSR1_ERR_S0_MASK)
#define ADC_STSR1_ERR_S1_MASK                    0x1000u
#define ADC_STSR1_ERR_S1_SHIFT                   12u
#define ADC_STSR1_ERR_S1_WIDTH                   1u
#define ADC_STSR1_ERR_S1(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_STSR1_ERR_S1_SHIFT))&ADC_STSR1_ERR_S1_MASK)
#define ADC_STSR1_ERR_S2_MASK                    0x2000u
#define ADC_STSR1_ERR_S2_SHIFT                   13u
#define ADC_STSR1_ERR_S2_WIDTH                   1u
#define ADC_STSR1_ERR_S2(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_STSR1_ERR_S2_SHIFT))&ADC_STSR1_ERR_S2_MASK)
#define ADC_STSR1_ERR_C_MASK                     0x8000u
#define ADC_STSR1_ERR_C_SHIFT                    15u
#define ADC_STSR1_ERR_C_WIDTH                    1u
#define ADC_STSR1_ERR_C(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STSR1_ERR_C_SHIFT))&ADC_STSR1_ERR_C_MASK)
#define ADC_STSR1_WDG_EOA_S_MASK                 0x10000u
#define ADC_STSR1_WDG_EOA_S_SHIFT                16u
#define ADC_STSR1_WDG_EOA_S_WIDTH                1u
#define ADC_STSR1_WDG_EOA_S(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_STSR1_WDG_EOA_S_SHIFT))&ADC_STSR1_WDG_EOA_S_MASK)
#define ADC_STSR1_WDG_EOA_C_MASK                 0x40000u
#define ADC_STSR1_WDG_EOA_C_SHIFT                18u
#define ADC_STSR1_WDG_EOA_C_WIDTH                1u
#define ADC_STSR1_WDG_EOA_C(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_STSR1_WDG_EOA_C_SHIFT))&ADC_STSR1_WDG_EOA_C_MASK)
#define ADC_STSR1_ST_EOC_MASK                    0x800000u
#define ADC_STSR1_ST_EOC_SHIFT                   23u
#define ADC_STSR1_ST_EOC_WIDTH                   1u
#define ADC_STSR1_ST_EOC(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_STSR1_ST_EOC_SHIFT))&ADC_STSR1_ST_EOC_MASK)
#define ADC_STSR1_OVERWR_MASK                    0x1000000u
#define ADC_STSR1_OVERWR_SHIFT                   24u
#define ADC_STSR1_OVERWR_WIDTH                   1u
#define ADC_STSR1_OVERWR(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_STSR1_OVERWR_SHIFT))&ADC_STSR1_OVERWR_MASK)
#define ADC_STSR1_WDTERR_MASK                    0x2000000u
#define ADC_STSR1_WDTERR_SHIFT                   25u
#define ADC_STSR1_WDTERR_WIDTH                   1u
#define ADC_STSR1_WDTERR(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_STSR1_WDTERR_SHIFT))&ADC_STSR1_WDTERR_MASK)
#define ADC_STSR1_WDSERR_MASK                    0x8000000u
#define ADC_STSR1_WDSERR_SHIFT                   27u
#define ADC_STSR1_WDSERR_WIDTH                   1u
#define ADC_STSR1_WDSERR(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_STSR1_WDSERR_SHIFT))&ADC_STSR1_WDSERR_MASK)
/* STSR2 Bit Fields */
#define ADC_STSR2_DATA0_MASK                     0xFFFu
#define ADC_STSR2_DATA0_SHIFT                    0u
#define ADC_STSR2_DATA0_WIDTH                    12u
#define ADC_STSR2_DATA0(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STSR2_DATA0_SHIFT))&ADC_STSR2_DATA0_MASK)
#define ADC_STSR2_DATA1_MASK                     0xFFF0000u
#define ADC_STSR2_DATA1_SHIFT                    16u
#define ADC_STSR2_DATA1_WIDTH                    12u
#define ADC_STSR2_DATA1(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STSR2_DATA1_SHIFT))&ADC_STSR2_DATA1_MASK)
#define ADC_STSR2_OVFL_MASK                      0x80000000u
#define ADC_STSR2_OVFL_SHIFT                     31u
#define ADC_STSR2_OVFL_WIDTH                     1u
#define ADC_STSR2_OVFL(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_STSR2_OVFL_SHIFT))&ADC_STSR2_OVFL_MASK)
/* STSR3 Bit Fields */
#define ADC_STSR3_DATA0_MASK                     0xFFFu
#define ADC_STSR3_DATA0_SHIFT                    0u
#define ADC_STSR3_DATA0_WIDTH                    12u
#define ADC_STSR3_DATA0(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STSR3_DATA0_SHIFT))&ADC_STSR3_DATA0_MASK)
#define ADC_STSR3_DATA1_MASK                     0xFFF0000u
#define ADC_STSR3_DATA1_SHIFT                    16u
#define ADC_STSR3_DATA1_WIDTH                    12u
#define ADC_STSR3_DATA1(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STSR3_DATA1_SHIFT))&ADC_STSR3_DATA1_MASK)
/* STSR4 Bit Fields */
#define ADC_STSR4_DATA0_MASK                     0xFFFu
#define ADC_STSR4_DATA0_SHIFT                    0u
#define ADC_STSR4_DATA0_WIDTH                    12u
#define ADC_STSR4_DATA0(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STSR4_DATA0_SHIFT))&ADC_STSR4_DATA0_MASK)
#define ADC_STSR4_DATA1_MASK                     0xFFF0000u
#define ADC_STSR4_DATA1_SHIFT                    16u
#define ADC_STSR4_DATA1_WIDTH                    12u
#define ADC_STSR4_DATA1(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STSR4_DATA1_SHIFT))&ADC_STSR4_DATA1_MASK)
/* STDR1 Bit Fields */
#define ADC_STDR1_TCDATA_MASK                    0xFFFu
#define ADC_STDR1_TCDATA_SHIFT                   0u
#define ADC_STDR1_TCDATA_WIDTH                   12u
#define ADC_STDR1_TCDATA(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_STDR1_TCDATA_SHIFT))&ADC_STDR1_TCDATA_MASK)
#define ADC_STDR1_OWERWR_MASK                    0x40000u
#define ADC_STDR1_OWERWR_SHIFT                   18u
#define ADC_STDR1_OWERWR_WIDTH                   1u
#define ADC_STDR1_OWERWR(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_STDR1_OWERWR_SHIFT))&ADC_STDR1_OWERWR_MASK)
#define ADC_STDR1_VALID_MASK                     0x80000u
#define ADC_STDR1_VALID_SHIFT                    19u
#define ADC_STDR1_VALID_WIDTH                    1u
#define ADC_STDR1_VALID(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STDR1_VALID_SHIFT))&ADC_STDR1_VALID_MASK)
/* STDR2 Bit Fields */
#define ADC_STDR2_IDATA_MASK                     0xFFFu
#define ADC_STDR2_IDATA_SHIFT                    0u
#define ADC_STDR2_IDATA_WIDTH                    12u
#define ADC_STDR2_IDATA(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STDR2_IDATA_SHIFT))&ADC_STDR2_IDATA_MASK)
#define ADC_STDR2_OVERWR_MASK                    0x40000u
#define ADC_STDR2_OVERWR_SHIFT                   18u
#define ADC_STDR2_OVERWR_WIDTH                   1u
#define ADC_STDR2_OVERWR(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_STDR2_OVERWR_SHIFT))&ADC_STDR2_OVERWR_MASK)
#define ADC_STDR2_VALID_MASK                     0x80000u
#define ADC_STDR2_VALID_SHIFT                    19u
#define ADC_STDR2_VALID_WIDTH                    1u
#define ADC_STDR2_VALID(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STDR2_VALID_SHIFT))&ADC_STDR2_VALID_MASK)
#define ADC_STDR2_FDATA_MASK                     0xFFF00000u
#define ADC_STDR2_FDATA_SHIFT                    20u
#define ADC_STDR2_FDATA_WIDTH                    12u
#define ADC_STDR2_FDATA(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STDR2_FDATA_SHIFT))&ADC_STDR2_FDATA_MASK)
/* STAW0R Bit Fields */
#define ADC_STAW0R_THRL_MASK                     0xFFFu
#define ADC_STAW0R_THRL_SHIFT                    0u
#define ADC_STAW0R_THRL_WIDTH                    12u
#define ADC_STAW0R_THRL(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STAW0R_THRL_SHIFT))&ADC_STAW0R_THRL_MASK)
#define ADC_STAW0R_THRH_MASK                     0xFFF0000u
#define ADC_STAW0R_THRH_SHIFT                    16u
#define ADC_STAW0R_THRH_WIDTH                    12u
#define ADC_STAW0R_THRH(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STAW0R_THRH_SHIFT))&ADC_STAW0R_THRH_MASK)
#define ADC_STAW0R_WDTE_MASK                     0x40000000u
#define ADC_STAW0R_WDTE_SHIFT                    30u
#define ADC_STAW0R_WDTE_WIDTH                    1u
#define ADC_STAW0R_WDTE(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STAW0R_WDTE_SHIFT))&ADC_STAW0R_WDTE_MASK)
#define ADC_STAW0R_AWDE_MASK                     0x80000000u
#define ADC_STAW0R_AWDE_SHIFT                    31u
#define ADC_STAW0R_AWDE_WIDTH                    1u
#define ADC_STAW0R_AWDE(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STAW0R_AWDE_SHIFT))&ADC_STAW0R_AWDE_MASK)
/* STAW1AR Bit Fields */
#define ADC_STAW1AR_THRL_MASK                    0xFFFu
#define ADC_STAW1AR_THRL_SHIFT                   0u
#define ADC_STAW1AR_THRL_WIDTH                   12u
#define ADC_STAW1AR_THRL(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_STAW1AR_THRL_SHIFT))&ADC_STAW1AR_THRL_MASK)
#define ADC_STAW1AR_THRH_MASK                    0xFFF0000u
#define ADC_STAW1AR_THRH_SHIFT                   16u
#define ADC_STAW1AR_THRH_WIDTH                   12u
#define ADC_STAW1AR_THRH(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_STAW1AR_THRH_SHIFT))&ADC_STAW1AR_THRH_MASK)
#define ADC_STAW1AR_AWDE_MASK                    0x80000000u
#define ADC_STAW1AR_AWDE_SHIFT                   31u
#define ADC_STAW1AR_AWDE_WIDTH                   1u
#define ADC_STAW1AR_AWDE(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_STAW1AR_AWDE_SHIFT))&ADC_STAW1AR_AWDE_MASK)
/* STAW1BR Bit Fields */
#define ADC_STAW1BR_THRL_MASK                    0xFFFu
#define ADC_STAW1BR_THRL_SHIFT                   0u
#define ADC_STAW1BR_THRL_WIDTH                   12u
#define ADC_STAW1BR_THRL(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_STAW1BR_THRL_SHIFT))&ADC_STAW1BR_THRL_MASK)
#define ADC_STAW1BR_THRH_MASK                    0xFFF0000u
#define ADC_STAW1BR_THRH_SHIFT                   16u
#define ADC_STAW1BR_THRH_WIDTH                   12u
#define ADC_STAW1BR_THRH(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_STAW1BR_THRH_SHIFT))&ADC_STAW1BR_THRH_MASK)
/* STAW2R Bit Fields */
#define ADC_STAW2R_THRL_MASK                     0xFFFu
#define ADC_STAW2R_THRL_SHIFT                    0u
#define ADC_STAW2R_THRL_WIDTH                    12u
#define ADC_STAW2R_THRL(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STAW2R_THRL_SHIFT))&ADC_STAW2R_THRL_MASK)
#define ADC_STAW2R_AWDE_MASK                     0x80000000u
#define ADC_STAW2R_AWDE_SHIFT                    31u
#define ADC_STAW2R_AWDE_WIDTH                    1u
#define ADC_STAW2R_AWDE(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STAW2R_AWDE_SHIFT))&ADC_STAW2R_AWDE_MASK)
/* STAW4R Bit Fields */
#define ADC_STAW4R_THRL_MASK                     0xFFFu
#define ADC_STAW4R_THRL_SHIFT                    0u
#define ADC_STAW4R_THRL_WIDTH                    12u
#define ADC_STAW4R_THRL(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STAW4R_THRL_SHIFT))&ADC_STAW4R_THRL_MASK)
#define ADC_STAW4R_THRH_MASK                     0xFFF0000u
#define ADC_STAW4R_THRH_SHIFT                    16u
#define ADC_STAW4R_THRH_WIDTH                    12u
#define ADC_STAW4R_THRH(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STAW4R_THRH_SHIFT))&ADC_STAW4R_THRH_MASK)
#define ADC_STAW4R_WDTE_MASK                     0x40000000u
#define ADC_STAW4R_WDTE_SHIFT                    30u
#define ADC_STAW4R_WDTE_WIDTH                    1u
#define ADC_STAW4R_WDTE(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STAW4R_WDTE_SHIFT))&ADC_STAW4R_WDTE_MASK)
#define ADC_STAW4R_AWDE_MASK                     0x80000000u
#define ADC_STAW4R_AWDE_SHIFT                    31u
#define ADC_STAW4R_AWDE_WIDTH                    1u
#define ADC_STAW4R_AWDE(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STAW4R_AWDE_SHIFT))&ADC_STAW4R_AWDE_MASK)
/* STAW5R Bit Fields */
#define ADC_STAW5R_THRL_MASK                     0xFFFu
#define ADC_STAW5R_THRL_SHIFT                    0u
#define ADC_STAW5R_THRL_WIDTH                    12u
#define ADC_STAW5R_THRL(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STAW5R_THRL_SHIFT))&ADC_STAW5R_THRL_MASK)
#define ADC_STAW5R_THRH_MASK                     0xFFF0000u
#define ADC_STAW5R_THRH_SHIFT                    16u
#define ADC_STAW5R_THRH_WIDTH                    12u
#define ADC_STAW5R_THRH(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_STAW5R_THRH_SHIFT))&ADC_STAW5R_THRH_MASK)
/* CALSTAT Bit Fields */
#define ADC_CALSTAT_STAT_1_MASK                  0x1u
#define ADC_CALSTAT_STAT_1_SHIFT                 0u
#define ADC_CALSTAT_STAT_1_WIDTH                 1u
#define ADC_CALSTAT_STAT_1(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CALSTAT_STAT_1_SHIFT))&ADC_CALSTAT_STAT_1_MASK)
#define ADC_CALSTAT_STAT_2_MASK                  0x2u
#define ADC_CALSTAT_STAT_2_SHIFT                 1u
#define ADC_CALSTAT_STAT_2_WIDTH                 1u
#define ADC_CALSTAT_STAT_2(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CALSTAT_STAT_2_SHIFT))&ADC_CALSTAT_STAT_2_MASK)
#define ADC_CALSTAT_STAT_3_MASK                  0x4u
#define ADC_CALSTAT_STAT_3_SHIFT                 2u
#define ADC_CALSTAT_STAT_3_WIDTH                 1u
#define ADC_CALSTAT_STAT_3(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CALSTAT_STAT_3_SHIFT))&ADC_CALSTAT_STAT_3_MASK)
#define ADC_CALSTAT_STAT_4_MASK                  0x8u
#define ADC_CALSTAT_STAT_4_SHIFT                 3u
#define ADC_CALSTAT_STAT_4_WIDTH                 1u
#define ADC_CALSTAT_STAT_4(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CALSTAT_STAT_4_SHIFT))&ADC_CALSTAT_STAT_4_MASK)
#define ADC_CALSTAT_STAT_5_MASK                  0x10u
#define ADC_CALSTAT_STAT_5_SHIFT                 4u
#define ADC_CALSTAT_STAT_5_WIDTH                 1u
#define ADC_CALSTAT_STAT_5(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CALSTAT_STAT_5_SHIFT))&ADC_CALSTAT_STAT_5_MASK)
#define ADC_CALSTAT_STAT_6_MASK                  0x20u
#define ADC_CALSTAT_STAT_6_SHIFT                 5u
#define ADC_CALSTAT_STAT_6_WIDTH                 1u
#define ADC_CALSTAT_STAT_6(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CALSTAT_STAT_6_SHIFT))&ADC_CALSTAT_STAT_6_MASK)
#define ADC_CALSTAT_STAT_7_MASK                  0x40u
#define ADC_CALSTAT_STAT_7_SHIFT                 6u
#define ADC_CALSTAT_STAT_7_WIDTH                 1u
#define ADC_CALSTAT_STAT_7(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CALSTAT_STAT_7_SHIFT))&ADC_CALSTAT_STAT_7_MASK)
#define ADC_CALSTAT_STAT_8_MASK                  0x80u
#define ADC_CALSTAT_STAT_8_SHIFT                 7u
#define ADC_CALSTAT_STAT_8_WIDTH                 1u
#define ADC_CALSTAT_STAT_8(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CALSTAT_STAT_8_SHIFT))&ADC_CALSTAT_STAT_8_MASK)
#define ADC_CALSTAT_STAT_9_MASK                  0x100u
#define ADC_CALSTAT_STAT_9_SHIFT                 8u
#define ADC_CALSTAT_STAT_9_WIDTH                 1u
#define ADC_CALSTAT_STAT_9(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CALSTAT_STAT_9_SHIFT))&ADC_CALSTAT_STAT_9_MASK)
#define ADC_CALSTAT_STAT_10_MASK                 0x200u
#define ADC_CALSTAT_STAT_10_SHIFT                9u
#define ADC_CALSTAT_STAT_10_WIDTH                1u
#define ADC_CALSTAT_STAT_10(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CALSTAT_STAT_10_SHIFT))&ADC_CALSTAT_STAT_10_MASK)
#define ADC_CALSTAT_STAT_11_MASK                 0x400u
#define ADC_CALSTAT_STAT_11_SHIFT                10u
#define ADC_CALSTAT_STAT_11_WIDTH                1u
#define ADC_CALSTAT_STAT_11(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CALSTAT_STAT_11_SHIFT))&ADC_CALSTAT_STAT_11_MASK)
#define ADC_CALSTAT_STAT_12_MASK                 0x800u
#define ADC_CALSTAT_STAT_12_SHIFT                11u
#define ADC_CALSTAT_STAT_12_WIDTH                1u
#define ADC_CALSTAT_STAT_12(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CALSTAT_STAT_12_SHIFT))&ADC_CALSTAT_STAT_12_MASK)
#define ADC_CALSTAT_STAT_13_MASK                 0x1000u
#define ADC_CALSTAT_STAT_13_SHIFT                12u
#define ADC_CALSTAT_STAT_13_WIDTH                1u
#define ADC_CALSTAT_STAT_13(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CALSTAT_STAT_13_SHIFT))&ADC_CALSTAT_STAT_13_MASK)
#define ADC_CALSTAT_STAT_14_MASK                 0x2000u
#define ADC_CALSTAT_STAT_14_SHIFT                13u
#define ADC_CALSTAT_STAT_14_WIDTH                1u
#define ADC_CALSTAT_STAT_14(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CALSTAT_STAT_14_SHIFT))&ADC_CALSTAT_STAT_14_MASK)
#define ADC_CALSTAT_TEST_RESULT_MASK             0xFFFF0000u
#define ADC_CALSTAT_TEST_RESULT_SHIFT            16u
#define ADC_CALSTAT_TEST_RESULT_WIDTH            16u
#define ADC_CALSTAT_TEST_RESULT(x)               (((uint32_t)(((uint32_t)(x))<<ADC_CALSTAT_TEST_RESULT_SHIFT))&ADC_CALSTAT_TEST_RESULT_MASK)

/*!
 * @}
 */ /* end of group ADC_Register_Masks */


/*!
 * @}
 */ /* end of group ADC_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- AXBS Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup AXBS_Peripheral_Access_Layer AXBS Peripheral Access Layer
 * @{
 */


/** AXBS - Size of Registers Arrays */
#define AXBS_PORT_COUNT                          7u
#define AXBS_MGPCR_COUNT                         8u

/** AXBS - Register Layout Typedef */
typedef struct {
  struct {                                         /* offset: 0x0, array step: 0x100 */
    __IO uint32_t PRS;                               /**< Priority Registers Slave, array offset: 0x0, array step: 0x100 */
         uint8_t RESERVED_0[12];
    __IO uint32_t CRS;                               /**< Control Register, array offset: 0x10, array step: 0x100 */
         uint8_t RESERVED_1[236];
  } PORT[AXBS_PORT_COUNT];
       uint8_t RESERVED_0[256];
  struct {                                         /* offset: 0x800, array step: 0x100 */
    __IO uint32_t MGPCR;                             /**< Master General Purpose Control Register, array offset: 0x800, array step: 0x100 */
         uint8_t RESERVED_0[252];
  } MGPCR[AXBS_MGPCR_COUNT];
} AXBS_Type, *AXBS_MemMapPtr;

 /** Number of instances of the AXBS module. */
#define AXBS_INSTANCE_COUNT                      (1u)


/* AXBS - Peripheral instance base addresses */
/** Peripheral AXBS base address */
#define AXBS_BASE                                (0x40000000u)
/** Peripheral AXBS base pointer */
#define AXBS                                     ((AXBS_Type *)AXBS_BASE)
/** Array initializer of AXBS peripheral base addresses */
#define AXBS_BASE_ADDRS                          { AXBS_BASE }
/** Array initializer of AXBS peripheral base pointers */
#define AXBS_BASE_PTRS                           { AXBS }

/* ----------------------------------------------------------------------------
   -- AXBS Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup AXBS_Register_Masks AXBS Register Masks
 * @{
 */

/* PRS Bit Fields */
#define AXBS_PRS_M0_MASK                         0x7u
#define AXBS_PRS_M0_SHIFT                        0u
#define AXBS_PRS_M0_WIDTH                        3u
#define AXBS_PRS_M0(x)                           (((uint32_t)(((uint32_t)(x))<<AXBS_PRS_M0_SHIFT))&AXBS_PRS_M0_MASK)
#define AXBS_PRS_M1_MASK                         0x70u
#define AXBS_PRS_M1_SHIFT                        4u
#define AXBS_PRS_M1_WIDTH                        3u
#define AXBS_PRS_M1(x)                           (((uint32_t)(((uint32_t)(x))<<AXBS_PRS_M1_SHIFT))&AXBS_PRS_M1_MASK)
#define AXBS_PRS_M2_MASK                         0x700u
#define AXBS_PRS_M2_SHIFT                        8u
#define AXBS_PRS_M2_WIDTH                        3u
#define AXBS_PRS_M2(x)                           (((uint32_t)(((uint32_t)(x))<<AXBS_PRS_M2_SHIFT))&AXBS_PRS_M2_MASK)
#define AXBS_PRS_M3_MASK                         0x7000u
#define AXBS_PRS_M3_SHIFT                        12u
#define AXBS_PRS_M3_WIDTH                        3u
#define AXBS_PRS_M3(x)                           (((uint32_t)(((uint32_t)(x))<<AXBS_PRS_M3_SHIFT))&AXBS_PRS_M3_MASK)
#define AXBS_PRS_M4_MASK                         0x70000u
#define AXBS_PRS_M4_SHIFT                        16u
#define AXBS_PRS_M4_WIDTH                        3u
#define AXBS_PRS_M4(x)                           (((uint32_t)(((uint32_t)(x))<<AXBS_PRS_M4_SHIFT))&AXBS_PRS_M4_MASK)
#define AXBS_PRS_M5_MASK                         0x700000u
#define AXBS_PRS_M5_SHIFT                        20u
#define AXBS_PRS_M5_WIDTH                        3u
#define AXBS_PRS_M5(x)                           (((uint32_t)(((uint32_t)(x))<<AXBS_PRS_M5_SHIFT))&AXBS_PRS_M5_MASK)
#define AXBS_PRS_M6_MASK                         0x7000000u
#define AXBS_PRS_M6_SHIFT                        24u
#define AXBS_PRS_M6_WIDTH                        3u
#define AXBS_PRS_M6(x)                           (((uint32_t)(((uint32_t)(x))<<AXBS_PRS_M6_SHIFT))&AXBS_PRS_M6_MASK)
#define AXBS_PRS_M7_MASK                         0x70000000u
#define AXBS_PRS_M7_SHIFT                        28u
#define AXBS_PRS_M7_WIDTH                        3u
#define AXBS_PRS_M7(x)                           (((uint32_t)(((uint32_t)(x))<<AXBS_PRS_M7_SHIFT))&AXBS_PRS_M7_MASK)
/* CRS Bit Fields */
#define AXBS_CRS_PARK_MASK                       0x7u
#define AXBS_CRS_PARK_SHIFT                      0u
#define AXBS_CRS_PARK_WIDTH                      3u
#define AXBS_CRS_PARK(x)                         (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_PARK_SHIFT))&AXBS_CRS_PARK_MASK)
#define AXBS_CRS_PCTL_MASK                       0x30u
#define AXBS_CRS_PCTL_SHIFT                      4u
#define AXBS_CRS_PCTL_WIDTH                      2u
#define AXBS_CRS_PCTL(x)                         (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_PCTL_SHIFT))&AXBS_CRS_PCTL_MASK)
#define AXBS_CRS_ARB_MASK                        0x300u
#define AXBS_CRS_ARB_SHIFT                       8u
#define AXBS_CRS_ARB_WIDTH                       2u
#define AXBS_CRS_ARB(x)                          (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_ARB_SHIFT))&AXBS_CRS_ARB_MASK)
#define AXBS_CRS_HPE0_MASK                       0x10000u
#define AXBS_CRS_HPE0_SHIFT                      16u
#define AXBS_CRS_HPE0_WIDTH                      1u
#define AXBS_CRS_HPE0(x)                         (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_HPE0_SHIFT))&AXBS_CRS_HPE0_MASK)
#define AXBS_CRS_HPE1_MASK                       0x20000u
#define AXBS_CRS_HPE1_SHIFT                      17u
#define AXBS_CRS_HPE1_WIDTH                      1u
#define AXBS_CRS_HPE1(x)                         (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_HPE1_SHIFT))&AXBS_CRS_HPE1_MASK)
#define AXBS_CRS_HPE2_MASK                       0x40000u
#define AXBS_CRS_HPE2_SHIFT                      18u
#define AXBS_CRS_HPE2_WIDTH                      1u
#define AXBS_CRS_HPE2(x)                         (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_HPE2_SHIFT))&AXBS_CRS_HPE2_MASK)
#define AXBS_CRS_HPE3_MASK                       0x80000u
#define AXBS_CRS_HPE3_SHIFT                      19u
#define AXBS_CRS_HPE3_WIDTH                      1u
#define AXBS_CRS_HPE3(x)                         (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_HPE3_SHIFT))&AXBS_CRS_HPE3_MASK)
#define AXBS_CRS_HPE4_MASK                       0x100000u
#define AXBS_CRS_HPE4_SHIFT                      20u
#define AXBS_CRS_HPE4_WIDTH                      1u
#define AXBS_CRS_HPE4(x)                         (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_HPE4_SHIFT))&AXBS_CRS_HPE4_MASK)
#define AXBS_CRS_HPE5_MASK                       0x200000u
#define AXBS_CRS_HPE5_SHIFT                      21u
#define AXBS_CRS_HPE5_WIDTH                      1u
#define AXBS_CRS_HPE5(x)                         (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_HPE5_SHIFT))&AXBS_CRS_HPE5_MASK)
#define AXBS_CRS_HPE6_MASK                       0x400000u
#define AXBS_CRS_HPE6_SHIFT                      22u
#define AXBS_CRS_HPE6_WIDTH                      1u
#define AXBS_CRS_HPE6(x)                         (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_HPE6_SHIFT))&AXBS_CRS_HPE6_MASK)
#define AXBS_CRS_HPE7_MASK                       0x800000u
#define AXBS_CRS_HPE7_SHIFT                      23u
#define AXBS_CRS_HPE7_WIDTH                      1u
#define AXBS_CRS_HPE7(x)                         (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_HPE7_SHIFT))&AXBS_CRS_HPE7_MASK)
#define AXBS_CRS_HLP_MASK                        0x40000000u
#define AXBS_CRS_HLP_SHIFT                       30u
#define AXBS_CRS_HLP_WIDTH                       1u
#define AXBS_CRS_HLP(x)                          (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_HLP_SHIFT))&AXBS_CRS_HLP_MASK)
#define AXBS_CRS_RO_MASK                         0x80000000u
#define AXBS_CRS_RO_SHIFT                        31u
#define AXBS_CRS_RO_WIDTH                        1u
#define AXBS_CRS_RO(x)                           (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_RO_SHIFT))&AXBS_CRS_RO_MASK)
/* MGPCR Bit Fields */
#define AXBS_MGPCR_AULB_MASK                     0x7u
#define AXBS_MGPCR_AULB_SHIFT                    0u
#define AXBS_MGPCR_AULB_WIDTH                    3u
#define AXBS_MGPCR_AULB(x)                       (((uint32_t)(((uint32_t)(x))<<AXBS_MGPCR_AULB_SHIFT))&AXBS_MGPCR_AULB_MASK)

/*!
 * @}
 */ /* end of group AXBS_Register_Masks */


/*!
 * @}
 */ /* end of group AXBS_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- BCTRL Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup BCTRL_Peripheral_Access_Layer BCTRL Peripheral Access Layer
 * @{
 */


/** BCTRL - Size of Registers Arrays */

/** BCTRL - Register Layout Typedef */
typedef struct {
       uint8_t RESERVED_0[44];
  __I  uint32_t BISTALL;                           /**< BIST All register, offset: 0x2C */
       uint8_t RESERVED_1[464];
  __IO uint32_t BSEL;                              /**< BIST Select Register, offset: 0x200 */
       uint8_t RESERVED_2[252];
  __I  uint32_t BIST_STAT;                         /**< BIST Status register, offset: 0x300 */
} BCTRL_Type, *BCTRL_MemMapPtr;

 /** Number of instances of the BCTRL module. */
#define BCTRL_INSTANCE_COUNT                     (1u)


/* BCTRL - Peripheral instance base addresses */
/** Peripheral BCTRL base address */
#define BCTRL_BASE                               (0x400EA000u)
/** Peripheral BCTRL base pointer */
#define BCTRL                                    ((BCTRL_Type *)BCTRL_BASE)
/** Array initializer of BCTRL peripheral base addresses */
#define BCTRL_BASE_ADDRS                         { BCTRL_BASE }
/** Array initializer of BCTRL peripheral base pointers */
#define BCTRL_BASE_PTRS                          { BCTRL }

/* ----------------------------------------------------------------------------
   -- BCTRL Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup BCTRL_Register_Masks BCTRL Register Masks
 * @{
 */

/* BISTALL Bit Fields */
#define BCTRL_BISTALL_STAT_ALL_MASK              0x3u
#define BCTRL_BISTALL_STAT_ALL_SHIFT             0u
#define BCTRL_BISTALL_STAT_ALL_WIDTH             2u
#define BCTRL_BISTALL_STAT_ALL(x)                (((uint32_t)(((uint32_t)(x))<<BCTRL_BISTALL_STAT_ALL_SHIFT))&BCTRL_BISTALL_STAT_ALL_MASK)
#define BCTRL_BISTALL_BIRA_FAIL_MASK             0x4u
#define BCTRL_BISTALL_BIRA_FAIL_SHIFT            2u
#define BCTRL_BISTALL_BIRA_FAIL_WIDTH            1u
#define BCTRL_BISTALL_BIRA_FAIL(x)               (((uint32_t)(((uint32_t)(x))<<BCTRL_BISTALL_BIRA_FAIL_SHIFT))&BCTRL_BISTALL_BIRA_FAIL_MASK)
/* BSEL Bit Fields */
#define BCTRL_BSEL_BIST_CTRL_MASK                0xFFFFFFFFu
#define BCTRL_BSEL_BIST_CTRL_SHIFT               0u
#define BCTRL_BSEL_BIST_CTRL_WIDTH               32u
#define BCTRL_BSEL_BIST_CTRL(x)                  (((uint32_t)(((uint32_t)(x))<<BCTRL_BSEL_BIST_CTRL_SHIFT))&BCTRL_BSEL_BIST_CTRL_MASK)
/* BIST_STAT Bit Fields */
#define BCTRL_BIST_STAT_BIST0STAT_MASK           0xFu
#define BCTRL_BIST_STAT_BIST0STAT_SHIFT          0u
#define BCTRL_BIST_STAT_BIST0STAT_WIDTH          4u
#define BCTRL_BIST_STAT_BIST0STAT(x)             (((uint32_t)(((uint32_t)(x))<<BCTRL_BIST_STAT_BIST0STAT_SHIFT))&BCTRL_BIST_STAT_BIST0STAT_MASK)
#define BCTRL_BIST_STAT_BIST1STAT_MASK           0xF0u
#define BCTRL_BIST_STAT_BIST1STAT_SHIFT          4u
#define BCTRL_BIST_STAT_BIST1STAT_WIDTH          4u
#define BCTRL_BIST_STAT_BIST1STAT(x)             (((uint32_t)(((uint32_t)(x))<<BCTRL_BIST_STAT_BIST1STAT_SHIFT))&BCTRL_BIST_STAT_BIST1STAT_MASK)
#define BCTRL_BIST_STAT_BIST2STAT_MASK           0xF00u
#define BCTRL_BIST_STAT_BIST2STAT_SHIFT          8u
#define BCTRL_BIST_STAT_BIST2STAT_WIDTH          4u
#define BCTRL_BIST_STAT_BIST2STAT(x)             (((uint32_t)(((uint32_t)(x))<<BCTRL_BIST_STAT_BIST2STAT_SHIFT))&BCTRL_BIST_STAT_BIST2STAT_MASK)
#define BCTRL_BIST_STAT_BIST3STAT_MASK           0xF000u
#define BCTRL_BIST_STAT_BIST3STAT_SHIFT          12u
#define BCTRL_BIST_STAT_BIST3STAT_WIDTH          4u
#define BCTRL_BIST_STAT_BIST3STAT(x)             (((uint32_t)(((uint32_t)(x))<<BCTRL_BIST_STAT_BIST3STAT_SHIFT))&BCTRL_BIST_STAT_BIST3STAT_MASK)
#define BCTRL_BIST_STAT_BIST4STAT_MASK           0xF0000u
#define BCTRL_BIST_STAT_BIST4STAT_SHIFT          16u
#define BCTRL_BIST_STAT_BIST4STAT_WIDTH          4u
#define BCTRL_BIST_STAT_BIST4STAT(x)             (((uint32_t)(((uint32_t)(x))<<BCTRL_BIST_STAT_BIST4STAT_SHIFT))&BCTRL_BIST_STAT_BIST4STAT_MASK)
#define BCTRL_BIST_STAT_BIST5STAT_MASK           0xF00000u
#define BCTRL_BIST_STAT_BIST5STAT_SHIFT          20u
#define BCTRL_BIST_STAT_BIST5STAT_WIDTH          4u
#define BCTRL_BIST_STAT_BIST5STAT(x)             (((uint32_t)(((uint32_t)(x))<<BCTRL_BIST_STAT_BIST5STAT_SHIFT))&BCTRL_BIST_STAT_BIST5STAT_MASK)
#define BCTRL_BIST_STAT_BIST6STAT_MASK           0xF000000u
#define BCTRL_BIST_STAT_BIST6STAT_SHIFT          24u
#define BCTRL_BIST_STAT_BIST6STAT_WIDTH          4u
#define BCTRL_BIST_STAT_BIST6STAT(x)             (((uint32_t)(((uint32_t)(x))<<BCTRL_BIST_STAT_BIST6STAT_SHIFT))&BCTRL_BIST_STAT_BIST6STAT_MASK)
#define BCTRL_BIST_STAT_BIST7STAT_MASK           0xF0000000u
#define BCTRL_BIST_STAT_BIST7STAT_SHIFT          28u
#define BCTRL_BIST_STAT_BIST7STAT_WIDTH          4u
#define BCTRL_BIST_STAT_BIST7STAT(x)             (((uint32_t)(((uint32_t)(x))<<BCTRL_BIST_STAT_BIST7STAT_SHIFT))&BCTRL_BIST_STAT_BIST7STAT_MASK)

/*!
 * @}
 */ /* end of group BCTRL_Register_Masks */


/*!
 * @}
 */ /* end of group BCTRL_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- BIST Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup BIST_Peripheral_Access_Layer BIST Peripheral Access Layer
 * @{
 */


/** BIST - Size of Registers Arrays */
#define BIST_MSR_COUNT                           8u
#define BIST_BFPM_COUNT                          8u
#define BIST_DBG_COUNT                           16u

/** BIST - Register Layout Typedef */
typedef struct {
       uint8_t RESERVED_0[64];
  __IO uint32_t MSR[BIST_MSR_COUNT];               /**< Memory selection register, array offset: 0x40, array step: 0x4 */
       uint8_t RESERVED_1[68];
  __IO uint32_t BSTART;                            /**< BIST start register, offset: 0xA4 */
  __I  uint32_t BRST;                              /**< BIST reset register, offset: 0xA8 */
       uint8_t RESERVED_2[12];
  __I  uint32_t BSTAT;                             /**< BIST status register, offset: 0xB8 */
  __I  uint32_t BFPM[BIST_BFPM_COUNT];             /**< BIST Fail Per Memory register, array offset: 0xBC, array step: 0x4 */
       uint8_t RESERVED_3[32];
  __IO uint32_t ROM_SEL;                           /**< ROM Selection register, offset: 0xFC */
  __I  uint32_t ROM_STAT;                          /**< ROM General Status register, offset: 0x100 */
       uint8_t RESERVED_4[12];
  __I  uint32_t ADDR_DBG;                          /**< Address Debug register, offset: 0x110 */
  __I  uint32_t DBG[BIST_DBG_COUNT];               /**< Data Debug register, array offset: 0x114, array step: 0x4 */
} BIST_Type, *BIST_MemMapPtr;

 /** Number of instances of the BIST module. */
#define BIST_INSTANCE_COUNT                      (1u)


/* BIST - Peripheral instance base addresses */
/** Peripheral BIST base address */
#define BIST_BASE                                (0x400EC000u)
/** Peripheral BIST base pointer */
#define BIST                                     ((BIST_Type *)BIST_BASE)
/** Array initializer of BIST peripheral base addresses */
#define BIST_BASE_ADDRS                          { BIST_BASE }
/** Array initializer of BIST peripheral base pointers */
#define BIST_BASE_PTRS                           { BIST }

/* ----------------------------------------------------------------------------
   -- BIST Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup BIST_Register_Masks BIST Register Masks
 * @{
 */

/* MSR Bit Fields */
#define BIST_MSR_MSELn_MASK                      0xFFFFFFFFu
#define BIST_MSR_MSELn_SHIFT                     0u
#define BIST_MSR_MSELn_WIDTH                     32u
#define BIST_MSR_MSELn(x)                        (((uint32_t)(((uint32_t)(x))<<BIST_MSR_MSELn_SHIFT))&BIST_MSR_MSELn_MASK)
/* BSTART Bit Fields */
#define BIST_BSTART_BSTART_MASK                  0x3u
#define BIST_BSTART_BSTART_SHIFT                 0u
#define BIST_BSTART_BSTART_WIDTH                 2u
#define BIST_BSTART_BSTART(x)                    (((uint32_t)(((uint32_t)(x))<<BIST_BSTART_BSTART_SHIFT))&BIST_BSTART_BSTART_MASK)
/* BRST Bit Fields */
#define BIST_BRST_BRST_MASK                      0x1u
#define BIST_BRST_BRST_SHIFT                     0u
#define BIST_BRST_BRST_WIDTH                     1u
#define BIST_BRST_BRST(x)                        (((uint32_t)(((uint32_t)(x))<<BIST_BRST_BRST_SHIFT))&BIST_BRST_BRST_MASK)
#define BIST_BRST_SRST_MASK                      0x2u
#define BIST_BRST_SRST_SHIFT                     1u
#define BIST_BRST_SRST_WIDTH                     1u
#define BIST_BRST_SRST(x)                        (((uint32_t)(((uint32_t)(x))<<BIST_BRST_SRST_SHIFT))&BIST_BRST_SRST_MASK)
/* BSTAT Bit Fields */
#define BIST_BSTAT_BSTAT1_MASK                   0x3u
#define BIST_BSTAT_BSTAT1_SHIFT                  0u
#define BIST_BSTAT_BSTAT1_WIDTH                  2u
#define BIST_BSTAT_BSTAT1(x)                     (((uint32_t)(((uint32_t)(x))<<BIST_BSTAT_BSTAT1_SHIFT))&BIST_BSTAT_BSTAT1_MASK)
#define BIST_BSTAT_BSTAT3_MASK                   0x8u
#define BIST_BSTAT_BSTAT3_SHIFT                  3u
#define BIST_BSTAT_BSTAT3_WIDTH                  1u
#define BIST_BSTAT_BSTAT3(x)                     (((uint32_t)(((uint32_t)(x))<<BIST_BSTAT_BSTAT3_SHIFT))&BIST_BSTAT_BSTAT3_MASK)
/* BFPM Bit Fields */
#define BIST_BFPM_FPMn_MASK                      0xFFFFFFFFu
#define BIST_BFPM_FPMn_SHIFT                     0u
#define BIST_BFPM_FPMn_WIDTH                     32u
#define BIST_BFPM_FPMn(x)                        (((uint32_t)(((uint32_t)(x))<<BIST_BFPM_FPMn_SHIFT))&BIST_BFPM_FPMn_MASK)
/* ROM_SEL Bit Fields */
#define BIST_ROM_SEL_MISRL_MASK                  0x3u
#define BIST_ROM_SEL_MISRL_SHIFT                 0u
#define BIST_ROM_SEL_MISRL_WIDTH                 2u
#define BIST_ROM_SEL_MISRL(x)                    (((uint32_t)(((uint32_t)(x))<<BIST_ROM_SEL_MISRL_SHIFT))&BIST_ROM_SEL_MISRL_MASK)
#define BIST_ROM_SEL_ROMSEL0_MASK                0x4u
#define BIST_ROM_SEL_ROMSEL0_SHIFT               2u
#define BIST_ROM_SEL_ROMSEL0_WIDTH               1u
#define BIST_ROM_SEL_ROMSEL0(x)                  (((uint32_t)(((uint32_t)(x))<<BIST_ROM_SEL_ROMSEL0_SHIFT))&BIST_ROM_SEL_ROMSEL0_MASK)
#define BIST_ROM_SEL_ROMSEL1_MASK                0x8u
#define BIST_ROM_SEL_ROMSEL1_SHIFT               3u
#define BIST_ROM_SEL_ROMSEL1_WIDTH               1u
#define BIST_ROM_SEL_ROMSEL1(x)                  (((uint32_t)(((uint32_t)(x))<<BIST_ROM_SEL_ROMSEL1_SHIFT))&BIST_ROM_SEL_ROMSEL1_MASK)
#define BIST_ROM_SEL_ROMSEL2_MASK                0x10u
#define BIST_ROM_SEL_ROMSEL2_SHIFT               4u
#define BIST_ROM_SEL_ROMSEL2_WIDTH               1u
#define BIST_ROM_SEL_ROMSEL2(x)                  (((uint32_t)(((uint32_t)(x))<<BIST_ROM_SEL_ROMSEL2_SHIFT))&BIST_ROM_SEL_ROMSEL2_MASK)
#define BIST_ROM_SEL_ROMSEL3_MASK                0x20u
#define BIST_ROM_SEL_ROMSEL3_SHIFT               5u
#define BIST_ROM_SEL_ROMSEL3_WIDTH               1u
#define BIST_ROM_SEL_ROMSEL3(x)                  (((uint32_t)(((uint32_t)(x))<<BIST_ROM_SEL_ROMSEL3_SHIFT))&BIST_ROM_SEL_ROMSEL3_MASK)
#define BIST_ROM_SEL_ROMSEL4_MASK                0x40u
#define BIST_ROM_SEL_ROMSEL4_SHIFT               6u
#define BIST_ROM_SEL_ROMSEL4_WIDTH               1u
#define BIST_ROM_SEL_ROMSEL4(x)                  (((uint32_t)(((uint32_t)(x))<<BIST_ROM_SEL_ROMSEL4_SHIFT))&BIST_ROM_SEL_ROMSEL4_MASK)
#define BIST_ROM_SEL_ROMSEL5_MASK                0x80u
#define BIST_ROM_SEL_ROMSEL5_SHIFT               7u
#define BIST_ROM_SEL_ROMSEL5_WIDTH               1u
#define BIST_ROM_SEL_ROMSEL5(x)                  (((uint32_t)(((uint32_t)(x))<<BIST_ROM_SEL_ROMSEL5_SHIFT))&BIST_ROM_SEL_ROMSEL5_MASK)
#define BIST_ROM_SEL_ROMSEL6_MASK                0x100u
#define BIST_ROM_SEL_ROMSEL6_SHIFT               8u
#define BIST_ROM_SEL_ROMSEL6_WIDTH               1u
#define BIST_ROM_SEL_ROMSEL6(x)                  (((uint32_t)(((uint32_t)(x))<<BIST_ROM_SEL_ROMSEL6_SHIFT))&BIST_ROM_SEL_ROMSEL6_MASK)
#define BIST_ROM_SEL_ROMSEL7_MASK                0x200u
#define BIST_ROM_SEL_ROMSEL7_SHIFT               9u
#define BIST_ROM_SEL_ROMSEL7_WIDTH               1u
#define BIST_ROM_SEL_ROMSEL7(x)                  (((uint32_t)(((uint32_t)(x))<<BIST_ROM_SEL_ROMSEL7_SHIFT))&BIST_ROM_SEL_ROMSEL7_MASK)
#define BIST_ROM_SEL_ROMSEL8_MASK                0x400u
#define BIST_ROM_SEL_ROMSEL8_SHIFT               10u
#define BIST_ROM_SEL_ROMSEL8_WIDTH               1u
#define BIST_ROM_SEL_ROMSEL8(x)                  (((uint32_t)(((uint32_t)(x))<<BIST_ROM_SEL_ROMSEL8_SHIFT))&BIST_ROM_SEL_ROMSEL8_MASK)
#define BIST_ROM_SEL_ROMSEL9_MASK                0x800u
#define BIST_ROM_SEL_ROMSEL9_SHIFT               11u
#define BIST_ROM_SEL_ROMSEL9_WIDTH               1u
#define BIST_ROM_SEL_ROMSEL9(x)                  (((uint32_t)(((uint32_t)(x))<<BIST_ROM_SEL_ROMSEL9_SHIFT))&BIST_ROM_SEL_ROMSEL9_MASK)
#define BIST_ROM_SEL_ROMSEL10_MASK               0x1000u
#define BIST_ROM_SEL_ROMSEL10_SHIFT              12u
#define BIST_ROM_SEL_ROMSEL10_WIDTH              1u
#define BIST_ROM_SEL_ROMSEL10(x)                 (((uint32_t)(((uint32_t)(x))<<BIST_ROM_SEL_ROMSEL10_SHIFT))&BIST_ROM_SEL_ROMSEL10_MASK)
#define BIST_ROM_SEL_ROMSEL11_MASK               0x2000u
#define BIST_ROM_SEL_ROMSEL11_SHIFT              13u
#define BIST_ROM_SEL_ROMSEL11_WIDTH              1u
#define BIST_ROM_SEL_ROMSEL11(x)                 (((uint32_t)(((uint32_t)(x))<<BIST_ROM_SEL_ROMSEL11_SHIFT))&BIST_ROM_SEL_ROMSEL11_MASK)
#define BIST_ROM_SEL_ROMSEL12_MASK               0x4000u
#define BIST_ROM_SEL_ROMSEL12_SHIFT              14u
#define BIST_ROM_SEL_ROMSEL12_WIDTH              1u
#define BIST_ROM_SEL_ROMSEL12(x)                 (((uint32_t)(((uint32_t)(x))<<BIST_ROM_SEL_ROMSEL12_SHIFT))&BIST_ROM_SEL_ROMSEL12_MASK)
#define BIST_ROM_SEL_ROMSEL13_MASK               0x8000u
#define BIST_ROM_SEL_ROMSEL13_SHIFT              15u
#define BIST_ROM_SEL_ROMSEL13_WIDTH              1u
#define BIST_ROM_SEL_ROMSEL13(x)                 (((uint32_t)(((uint32_t)(x))<<BIST_ROM_SEL_ROMSEL13_SHIFT))&BIST_ROM_SEL_ROMSEL13_MASK)
#define BIST_ROM_SEL_ROMSEL14_MASK               0x10000u
#define BIST_ROM_SEL_ROMSEL14_SHIFT              16u
#define BIST_ROM_SEL_ROMSEL14_WIDTH              1u
#define BIST_ROM_SEL_ROMSEL14(x)                 (((uint32_t)(((uint32_t)(x))<<BIST_ROM_SEL_ROMSEL14_SHIFT))&BIST_ROM_SEL_ROMSEL14_MASK)
/* ROM_STAT Bit Fields */
#define BIST_ROM_STAT_BSTAT1_MASK                0x3u
#define BIST_ROM_STAT_BSTAT1_SHIFT               0u
#define BIST_ROM_STAT_BSTAT1_WIDTH               2u
#define BIST_ROM_STAT_BSTAT1(x)                  (((uint32_t)(((uint32_t)(x))<<BIST_ROM_STAT_BSTAT1_SHIFT))&BIST_ROM_STAT_BSTAT1_MASK)
#define BIST_ROM_STAT_BSTAT2_MASK                0x4u
#define BIST_ROM_STAT_BSTAT2_SHIFT               2u
#define BIST_ROM_STAT_BSTAT2_WIDTH               1u
#define BIST_ROM_STAT_BSTAT2(x)                  (((uint32_t)(((uint32_t)(x))<<BIST_ROM_STAT_BSTAT2_SHIFT))&BIST_ROM_STAT_BSTAT2_MASK)
#define BIST_ROM_STAT_BSTAT3_MASK                0x8u
#define BIST_ROM_STAT_BSTAT3_SHIFT               3u
#define BIST_ROM_STAT_BSTAT3_WIDTH               1u
#define BIST_ROM_STAT_BSTAT3(x)                  (((uint32_t)(((uint32_t)(x))<<BIST_ROM_STAT_BSTAT3_SHIFT))&BIST_ROM_STAT_BSTAT3_MASK)
#define BIST_ROM_STAT_RFPM0_MASK                 0x10u
#define BIST_ROM_STAT_RFPM0_SHIFT                4u
#define BIST_ROM_STAT_RFPM0_WIDTH                1u
#define BIST_ROM_STAT_RFPM0(x)                   (((uint32_t)(((uint32_t)(x))<<BIST_ROM_STAT_RFPM0_SHIFT))&BIST_ROM_STAT_RFPM0_MASK)
#define BIST_ROM_STAT_RFPM1_MASK                 0x20u
#define BIST_ROM_STAT_RFPM1_SHIFT                5u
#define BIST_ROM_STAT_RFPM1_WIDTH                1u
#define BIST_ROM_STAT_RFPM1(x)                   (((uint32_t)(((uint32_t)(x))<<BIST_ROM_STAT_RFPM1_SHIFT))&BIST_ROM_STAT_RFPM1_MASK)
#define BIST_ROM_STAT_RFPM2_MASK                 0x40u
#define BIST_ROM_STAT_RFPM2_SHIFT                6u
#define BIST_ROM_STAT_RFPM2_WIDTH                1u
#define BIST_ROM_STAT_RFPM2(x)                   (((uint32_t)(((uint32_t)(x))<<BIST_ROM_STAT_RFPM2_SHIFT))&BIST_ROM_STAT_RFPM2_MASK)
#define BIST_ROM_STAT_RFPM3_MASK                 0x80u
#define BIST_ROM_STAT_RFPM3_SHIFT                7u
#define BIST_ROM_STAT_RFPM3_WIDTH                1u
#define BIST_ROM_STAT_RFPM3(x)                   (((uint32_t)(((uint32_t)(x))<<BIST_ROM_STAT_RFPM3_SHIFT))&BIST_ROM_STAT_RFPM3_MASK)
#define BIST_ROM_STAT_RFPM4_MASK                 0x100u
#define BIST_ROM_STAT_RFPM4_SHIFT                8u
#define BIST_ROM_STAT_RFPM4_WIDTH                1u
#define BIST_ROM_STAT_RFPM4(x)                   (((uint32_t)(((uint32_t)(x))<<BIST_ROM_STAT_RFPM4_SHIFT))&BIST_ROM_STAT_RFPM4_MASK)
#define BIST_ROM_STAT_RFPM5_MASK                 0x200u
#define BIST_ROM_STAT_RFPM5_SHIFT                9u
#define BIST_ROM_STAT_RFPM5_WIDTH                1u
#define BIST_ROM_STAT_RFPM5(x)                   (((uint32_t)(((uint32_t)(x))<<BIST_ROM_STAT_RFPM5_SHIFT))&BIST_ROM_STAT_RFPM5_MASK)
#define BIST_ROM_STAT_RFPM6_MASK                 0x400u
#define BIST_ROM_STAT_RFPM6_SHIFT                10u
#define BIST_ROM_STAT_RFPM6_WIDTH                1u
#define BIST_ROM_STAT_RFPM6(x)                   (((uint32_t)(((uint32_t)(x))<<BIST_ROM_STAT_RFPM6_SHIFT))&BIST_ROM_STAT_RFPM6_MASK)
#define BIST_ROM_STAT_RFPM7_MASK                 0x800u
#define BIST_ROM_STAT_RFPM7_SHIFT                11u
#define BIST_ROM_STAT_RFPM7_WIDTH                1u
#define BIST_ROM_STAT_RFPM7(x)                   (((uint32_t)(((uint32_t)(x))<<BIST_ROM_STAT_RFPM7_SHIFT))&BIST_ROM_STAT_RFPM7_MASK)
#define BIST_ROM_STAT_RFPM8_MASK                 0x1000u
#define BIST_ROM_STAT_RFPM8_SHIFT                12u
#define BIST_ROM_STAT_RFPM8_WIDTH                1u
#define BIST_ROM_STAT_RFPM8(x)                   (((uint32_t)(((uint32_t)(x))<<BIST_ROM_STAT_RFPM8_SHIFT))&BIST_ROM_STAT_RFPM8_MASK)
#define BIST_ROM_STAT_RFPM9_MASK                 0x2000u
#define BIST_ROM_STAT_RFPM9_SHIFT                13u
#define BIST_ROM_STAT_RFPM9_WIDTH                1u
#define BIST_ROM_STAT_RFPM9(x)                   (((uint32_t)(((uint32_t)(x))<<BIST_ROM_STAT_RFPM9_SHIFT))&BIST_ROM_STAT_RFPM9_MASK)
#define BIST_ROM_STAT_RFPM10_MASK                0x4000u
#define BIST_ROM_STAT_RFPM10_SHIFT               14u
#define BIST_ROM_STAT_RFPM10_WIDTH               1u
#define BIST_ROM_STAT_RFPM10(x)                  (((uint32_t)(((uint32_t)(x))<<BIST_ROM_STAT_RFPM10_SHIFT))&BIST_ROM_STAT_RFPM10_MASK)
#define BIST_ROM_STAT_RFPM11_MASK                0x8000u
#define BIST_ROM_STAT_RFPM11_SHIFT               15u
#define BIST_ROM_STAT_RFPM11_WIDTH               1u
#define BIST_ROM_STAT_RFPM11(x)                  (((uint32_t)(((uint32_t)(x))<<BIST_ROM_STAT_RFPM11_SHIFT))&BIST_ROM_STAT_RFPM11_MASK)
#define BIST_ROM_STAT_RFPM12_MASK                0x10000u
#define BIST_ROM_STAT_RFPM12_SHIFT               16u
#define BIST_ROM_STAT_RFPM12_WIDTH               1u
#define BIST_ROM_STAT_RFPM12(x)                  (((uint32_t)(((uint32_t)(x))<<BIST_ROM_STAT_RFPM12_SHIFT))&BIST_ROM_STAT_RFPM12_MASK)
#define BIST_ROM_STAT_RFPM13_MASK                0x20000u
#define BIST_ROM_STAT_RFPM13_SHIFT               17u
#define BIST_ROM_STAT_RFPM13_WIDTH               1u
#define BIST_ROM_STAT_RFPM13(x)                  (((uint32_t)(((uint32_t)(x))<<BIST_ROM_STAT_RFPM13_SHIFT))&BIST_ROM_STAT_RFPM13_MASK)
#define BIST_ROM_STAT_RFPM14_MASK                0x40000u
#define BIST_ROM_STAT_RFPM14_SHIFT               18u
#define BIST_ROM_STAT_RFPM14_WIDTH               1u
#define BIST_ROM_STAT_RFPM14(x)                  (((uint32_t)(((uint32_t)(x))<<BIST_ROM_STAT_RFPM14_SHIFT))&BIST_ROM_STAT_RFPM14_MASK)
/* ADDR_DBG Bit Fields */
#define BIST_ADDR_DBG_ADDR_MASK                  0x7FFFFu
#define BIST_ADDR_DBG_ADDR_SHIFT                 0u
#define BIST_ADDR_DBG_ADDR_WIDTH                 19u
#define BIST_ADDR_DBG_ADDR(x)                    (((uint32_t)(((uint32_t)(x))<<BIST_ADDR_DBG_ADDR_SHIFT))&BIST_ADDR_DBG_ADDR_MASK)
/* DBG Bit Fields */
#define BIST_DBG_CDATAn_MASK                     0xFFFFFFFFu
#define BIST_DBG_CDATAn_SHIFT                    0u
#define BIST_DBG_CDATAn_WIDTH                    32u
#define BIST_DBG_CDATAn(x)                       (((uint32_t)(((uint32_t)(x))<<BIST_DBG_CDATAn_SHIFT))&BIST_DBG_CDATAn_MASK)

/*!
 * @}
 */ /* end of group BIST_Register_Masks */


/*!
 * @}
 */ /* end of group BIST_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- CAN Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup CAN_Peripheral_Access_Layer CAN Peripheral Access Layer
 * @{
 */


/** CAN - Size of Registers Arrays */
#define CAN_RAMn_COUNT                           256u
#define CAN_RXIMR_COUNT                          64u

/** CAN - Register Layout Typedef */
typedef struct {
  __IO uint32_t MCR;                               /**< Module Configuration Register, offset: 0x0 */
  __IO uint32_t CTRL1;                             /**< Control 1 register, offset: 0x4 */
  __IO uint32_t TIMER;                             /**< Free Running Timer, offset: 0x8 */
       uint8_t RESERVED_0[4];
  __IO uint32_t RXMGMASK;                          /**< Rx Mailboxes Global Mask Register, offset: 0x10 */
  __IO uint32_t RX14MASK;                          /**< Rx 14 Mask register, offset: 0x14 */
  __IO uint32_t RX15MASK;                          /**< Rx 15 Mask register, offset: 0x18 */
  __IO uint32_t ECR;                               /**< Error Counter, offset: 0x1C */
  __IO uint32_t ESR1;                              /**< Error and Status 1 register, offset: 0x20 */
  __IO uint32_t IMASK2;                            /**< Interrupt Masks 2 register, offset: 0x24 */
  __IO uint32_t IMASK1;                            /**< Interrupt Masks 1 register, offset: 0x28 */
  __IO uint32_t IFLAG2;                            /**< Interrupt Flags 2 register, offset: 0x2C */
  __IO uint32_t IFLAG1;                            /**< Interrupt Flags 1 register, offset: 0x30 */
  __IO uint32_t CTRL2;                             /**< Control 2 register, offset: 0x34 */
  __I  uint32_t ESR2;                              /**< Error and Status 2 register, offset: 0x38 */
       uint8_t RESERVED_1[8];
  __I  uint32_t CRCR;                              /**< CRC Register, offset: 0x44 */
  __IO uint32_t RXFGMASK;                          /**< Rx FIFO Global Mask register, offset: 0x48 */
  __I  uint32_t RXFIR;                             /**< Rx FIFO Information Register, offset: 0x4C */
  __IO uint32_t CBT;                               /**< CAN Bit Timing Register, offset: 0x50 */
       uint8_t RESERVED_2[44];
  __IO uint32_t RAMn[CAN_RAMn_COUNT];              /**< Embedded RAM, array offset: 0x80, array step: 0x4 */
       uint8_t RESERVED_3[1024];
  __IO uint32_t RXIMR[CAN_RXIMR_COUNT];            /**< Rx Individual Mask Registers, array offset: 0x880, array step: 0x4 */
       uint8_t RESERVED_4[352];
  __IO uint32_t MECR;                              /**< Memory Error Control Register, offset: 0xAE0 */
  __IO uint32_t ERRIAR;                            /**< Error Injection Address Register, offset: 0xAE4 */
  __IO uint32_t ERRIDPR;                           /**< Error Injection Data Pattern Register, offset: 0xAE8 */
  __IO uint32_t ERRIPPR;                           /**< Error Injection Parity Pattern Register, offset: 0xAEC */
  __I  uint32_t RERRAR;                            /**< Error Report Address Register, offset: 0xAF0 */
  __I  uint32_t RERRDR;                            /**< Error Report Data Register, offset: 0xAF4 */
  __I  uint32_t RERRSYNR;                          /**< Error Report Syndrome Register, offset: 0xAF8 */
  __IO uint32_t ERRSR;                             /**< Error Status Register, offset: 0xAFC */
       uint8_t RESERVED_5[256];
  __IO uint32_t FDCTRL;                            /**< CAN FD Control Register, offset: 0xC00 */
  __IO uint32_t FDCBT;                             /**< CAN FD Bit Timing Register, offset: 0xC04 */
  __I  uint32_t FDCRC;                             /**< CAN FD CRC Register, offset: 0xC08 */
} CAN_Type, *CAN_MemMapPtr;

 /** Number of instances of the CAN module. */
#define CAN_INSTANCE_COUNT                       (2u)


/* CAN - Peripheral instance base addresses */
/** Peripheral CAN_0 base address */
#define CAN_0_BASE                               (0x40055000u)
/** Peripheral CAN_0 base pointer */
#define CAN_0                                    ((CAN_Type *)CAN_0_BASE)
/** Peripheral CAN_1 base address */
#define CAN_1_BASE                               (0x400BE000u)
/** Peripheral CAN_1 base pointer */
#define CAN_1                                    ((CAN_Type *)CAN_1_BASE)
/** Array initializer of CAN peripheral base addresses */
#define CAN_BASE_ADDRS                           { CAN_0_BASE, CAN_1_BASE }
/** Array initializer of CAN peripheral base pointers */
#define CAN_BASE_PTRS                            { CAN_0, CAN_1 }

/** Number of interrupt vector arrays for the CAN module. */
#define CAN_IRQS_ARR_COUNT                       (4u)
/** Number of interrupt channels for the Rx_Warning type of CAN module. */
#define CAN_Rx_Warning_IRQS_CH_COUNT             (1u)
/** Number of interrupt channels for the Tx_Warning type of CAN module. */
#define CAN_Tx_Warning_IRQS_CH_COUNT             (1u)
/** Number of interrupt channels for the Wake_Up type of CAN module. */
#define CAN_Wake_Up_IRQS_CH_COUNT                (0u)
/** Number of interrupt channels for the Error type of CAN module. */
#define CAN_Error_IRQS_CH_COUNT                  (1u)
/** Number of interrupt channels for the Bus_Off type of CAN module. */
#define CAN_Bus_Off_IRQS_CH_COUNT                (1u)
/** Number of interrupt channels for the ORed_Message_buffer type of CAN module. */
#define CAN_ORed_Message_buffer_IRQS_CH_COUNT    (1u)
/** Interrupt vectors for the CAN peripheral type */
#define CAN_Rx_Warning_IRQS                      { CAN0_IRQn, CAN1_IRQn }
#define CAN_Tx_Warning_IRQS                      { CAN0_IRQn, CAN1_IRQn }
#define CAN_Wake_Up_IRQS                         { }
#define CAN_Error_IRQS                           { CAN0_IRQn, CAN1_IRQn }
#define CAN_Bus_Off_IRQS                         { CAN0_IRQn, CAN1_IRQn }
#define CAN_ORed_Message_buffer_IRQS             { { CAN0_Buff_IRQn }, \
                                                   { CAN1_Buff_IRQn } }


/* ----------------------------------------------------------------------------
   -- CAN Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup CAN_Register_Masks CAN Register Masks
 * @{
 */

/* MCR Bit Fields */
#define CAN_MCR_MAXMB_MASK                       0x7Fu
#define CAN_MCR_MAXMB_SHIFT                      0u
#define CAN_MCR_MAXMB_WIDTH                      7u
#define CAN_MCR_MAXMB(x)                         (((uint32_t)(((uint32_t)(x))<<CAN_MCR_MAXMB_SHIFT))&CAN_MCR_MAXMB_MASK)
#define CAN_MCR_IDAM_MASK                        0x300u
#define CAN_MCR_IDAM_SHIFT                       8u
#define CAN_MCR_IDAM_WIDTH                       2u
#define CAN_MCR_IDAM(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_MCR_IDAM_SHIFT))&CAN_MCR_IDAM_MASK)
#define CAN_MCR_FDEN_MASK                        0x800u
#define CAN_MCR_FDEN_SHIFT                       11u
#define CAN_MCR_FDEN_WIDTH                       1u
#define CAN_MCR_FDEN(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_MCR_FDEN_SHIFT))&CAN_MCR_FDEN_MASK)
#define CAN_MCR_AEN_MASK                         0x1000u
#define CAN_MCR_AEN_SHIFT                        12u
#define CAN_MCR_AEN_WIDTH                        1u
#define CAN_MCR_AEN(x)                           (((uint32_t)(((uint32_t)(x))<<CAN_MCR_AEN_SHIFT))&CAN_MCR_AEN_MASK)
#define CAN_MCR_LPRIOEN_MASK                     0x2000u
#define CAN_MCR_LPRIOEN_SHIFT                    13u
#define CAN_MCR_LPRIOEN_WIDTH                    1u
#define CAN_MCR_LPRIOEN(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_MCR_LPRIOEN_SHIFT))&CAN_MCR_LPRIOEN_MASK)
#define CAN_MCR_DMA_MASK                         0x8000u
#define CAN_MCR_DMA_SHIFT                        15u
#define CAN_MCR_DMA_WIDTH                        1u
#define CAN_MCR_DMA(x)                           (((uint32_t)(((uint32_t)(x))<<CAN_MCR_DMA_SHIFT))&CAN_MCR_DMA_MASK)
#define CAN_MCR_IRMQ_MASK                        0x10000u
#define CAN_MCR_IRMQ_SHIFT                       16u
#define CAN_MCR_IRMQ_WIDTH                       1u
#define CAN_MCR_IRMQ(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_MCR_IRMQ_SHIFT))&CAN_MCR_IRMQ_MASK)
#define CAN_MCR_SRXDIS_MASK                      0x20000u
#define CAN_MCR_SRXDIS_SHIFT                     17u
#define CAN_MCR_SRXDIS_WIDTH                     1u
#define CAN_MCR_SRXDIS(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_MCR_SRXDIS_SHIFT))&CAN_MCR_SRXDIS_MASK)
#define CAN_MCR_LPMACK_MASK                      0x100000u
#define CAN_MCR_LPMACK_SHIFT                     20u
#define CAN_MCR_LPMACK_WIDTH                     1u
#define CAN_MCR_LPMACK(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_MCR_LPMACK_SHIFT))&CAN_MCR_LPMACK_MASK)
#define CAN_MCR_WRNEN_MASK                       0x200000u
#define CAN_MCR_WRNEN_SHIFT                      21u
#define CAN_MCR_WRNEN_WIDTH                      1u
#define CAN_MCR_WRNEN(x)                         (((uint32_t)(((uint32_t)(x))<<CAN_MCR_WRNEN_SHIFT))&CAN_MCR_WRNEN_MASK)
#define CAN_MCR_FRZACK_MASK                      0x1000000u
#define CAN_MCR_FRZACK_SHIFT                     24u
#define CAN_MCR_FRZACK_WIDTH                     1u
#define CAN_MCR_FRZACK(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_MCR_FRZACK_SHIFT))&CAN_MCR_FRZACK_MASK)
#define CAN_MCR_SOFTRST_MASK                     0x2000000u
#define CAN_MCR_SOFTRST_SHIFT                    25u
#define CAN_MCR_SOFTRST_WIDTH                    1u
#define CAN_MCR_SOFTRST(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_MCR_SOFTRST_SHIFT))&CAN_MCR_SOFTRST_MASK)
#define CAN_MCR_NOTRDY_MASK                      0x8000000u
#define CAN_MCR_NOTRDY_SHIFT                     27u
#define CAN_MCR_NOTRDY_WIDTH                     1u
#define CAN_MCR_NOTRDY(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_MCR_NOTRDY_SHIFT))&CAN_MCR_NOTRDY_MASK)
#define CAN_MCR_HALT_MASK                        0x10000000u
#define CAN_MCR_HALT_SHIFT                       28u
#define CAN_MCR_HALT_WIDTH                       1u
#define CAN_MCR_HALT(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_MCR_HALT_SHIFT))&CAN_MCR_HALT_MASK)
#define CAN_MCR_RFEN_MASK                        0x20000000u
#define CAN_MCR_RFEN_SHIFT                       29u
#define CAN_MCR_RFEN_WIDTH                       1u
#define CAN_MCR_RFEN(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_MCR_RFEN_SHIFT))&CAN_MCR_RFEN_MASK)
#define CAN_MCR_FRZ_MASK                         0x40000000u
#define CAN_MCR_FRZ_SHIFT                        30u
#define CAN_MCR_FRZ_WIDTH                        1u
#define CAN_MCR_FRZ(x)                           (((uint32_t)(((uint32_t)(x))<<CAN_MCR_FRZ_SHIFT))&CAN_MCR_FRZ_MASK)
#define CAN_MCR_MDIS_MASK                        0x80000000u
#define CAN_MCR_MDIS_SHIFT                       31u
#define CAN_MCR_MDIS_WIDTH                       1u
#define CAN_MCR_MDIS(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_MCR_MDIS_SHIFT))&CAN_MCR_MDIS_MASK)
/* CTRL1 Bit Fields */
#define CAN_CTRL1_PROPSEG_MASK                   0x7u
#define CAN_CTRL1_PROPSEG_SHIFT                  0u
#define CAN_CTRL1_PROPSEG_WIDTH                  3u
#define CAN_CTRL1_PROPSEG(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_PROPSEG_SHIFT))&CAN_CTRL1_PROPSEG_MASK)
#define CAN_CTRL1_LOM_MASK                       0x8u
#define CAN_CTRL1_LOM_SHIFT                      3u
#define CAN_CTRL1_LOM_WIDTH                      1u
#define CAN_CTRL1_LOM(x)                         (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_LOM_SHIFT))&CAN_CTRL1_LOM_MASK)
#define CAN_CTRL1_LBUF_MASK                      0x10u
#define CAN_CTRL1_LBUF_SHIFT                     4u
#define CAN_CTRL1_LBUF_WIDTH                     1u
#define CAN_CTRL1_LBUF(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_LBUF_SHIFT))&CAN_CTRL1_LBUF_MASK)
#define CAN_CTRL1_TSYN_MASK                      0x20u
#define CAN_CTRL1_TSYN_SHIFT                     5u
#define CAN_CTRL1_TSYN_WIDTH                     1u
#define CAN_CTRL1_TSYN(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_TSYN_SHIFT))&CAN_CTRL1_TSYN_MASK)
#define CAN_CTRL1_BOFFREC_MASK                   0x40u
#define CAN_CTRL1_BOFFREC_SHIFT                  6u
#define CAN_CTRL1_BOFFREC_WIDTH                  1u
#define CAN_CTRL1_BOFFREC(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_BOFFREC_SHIFT))&CAN_CTRL1_BOFFREC_MASK)
#define CAN_CTRL1_SMP_MASK                       0x80u
#define CAN_CTRL1_SMP_SHIFT                      7u
#define CAN_CTRL1_SMP_WIDTH                      1u
#define CAN_CTRL1_SMP(x)                         (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_SMP_SHIFT))&CAN_CTRL1_SMP_MASK)
#define CAN_CTRL1_RWRNMSK_MASK                   0x400u
#define CAN_CTRL1_RWRNMSK_SHIFT                  10u
#define CAN_CTRL1_RWRNMSK_WIDTH                  1u
#define CAN_CTRL1_RWRNMSK(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_RWRNMSK_SHIFT))&CAN_CTRL1_RWRNMSK_MASK)
#define CAN_CTRL1_TWRNMSK_MASK                   0x800u
#define CAN_CTRL1_TWRNMSK_SHIFT                  11u
#define CAN_CTRL1_TWRNMSK_WIDTH                  1u
#define CAN_CTRL1_TWRNMSK(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_TWRNMSK_SHIFT))&CAN_CTRL1_TWRNMSK_MASK)
#define CAN_CTRL1_LPB_MASK                       0x1000u
#define CAN_CTRL1_LPB_SHIFT                      12u
#define CAN_CTRL1_LPB_WIDTH                      1u
#define CAN_CTRL1_LPB(x)                         (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_LPB_SHIFT))&CAN_CTRL1_LPB_MASK)
#define CAN_CTRL1_ERRMSK_MASK                    0x4000u
#define CAN_CTRL1_ERRMSK_SHIFT                   14u
#define CAN_CTRL1_ERRMSK_WIDTH                   1u
#define CAN_CTRL1_ERRMSK(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_ERRMSK_SHIFT))&CAN_CTRL1_ERRMSK_MASK)
#define CAN_CTRL1_BOFFMSK_MASK                   0x8000u
#define CAN_CTRL1_BOFFMSK_SHIFT                  15u
#define CAN_CTRL1_BOFFMSK_WIDTH                  1u
#define CAN_CTRL1_BOFFMSK(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_BOFFMSK_SHIFT))&CAN_CTRL1_BOFFMSK_MASK)
#define CAN_CTRL1_PSEG2_MASK                     0x70000u
#define CAN_CTRL1_PSEG2_SHIFT                    16u
#define CAN_CTRL1_PSEG2_WIDTH                    3u
#define CAN_CTRL1_PSEG2(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_PSEG2_SHIFT))&CAN_CTRL1_PSEG2_MASK)
#define CAN_CTRL1_PSEG1_MASK                     0x380000u
#define CAN_CTRL1_PSEG1_SHIFT                    19u
#define CAN_CTRL1_PSEG1_WIDTH                    3u
#define CAN_CTRL1_PSEG1(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_PSEG1_SHIFT))&CAN_CTRL1_PSEG1_MASK)
#define CAN_CTRL1_RJW_MASK                       0xC00000u
#define CAN_CTRL1_RJW_SHIFT                      22u
#define CAN_CTRL1_RJW_WIDTH                      2u
#define CAN_CTRL1_RJW(x)                         (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_RJW_SHIFT))&CAN_CTRL1_RJW_MASK)
#define CAN_CTRL1_PRESDIV_MASK                   0xFF000000u
#define CAN_CTRL1_PRESDIV_SHIFT                  24u
#define CAN_CTRL1_PRESDIV_WIDTH                  8u
#define CAN_CTRL1_PRESDIV(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_PRESDIV_SHIFT))&CAN_CTRL1_PRESDIV_MASK)
/* TIMER Bit Fields */
#define CAN_TIMER_TIMER_MASK                     0xFFFFu
#define CAN_TIMER_TIMER_SHIFT                    0u
#define CAN_TIMER_TIMER_WIDTH                    16u
#define CAN_TIMER_TIMER(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_TIMER_TIMER_SHIFT))&CAN_TIMER_TIMER_MASK)
/* RXMGMASK Bit Fields */
#define CAN_RXMGMASK_MG_MASK                     0xFFFFFFFFu
#define CAN_RXMGMASK_MG_SHIFT                    0u
#define CAN_RXMGMASK_MG_WIDTH                    32u
#define CAN_RXMGMASK_MG(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_RXMGMASK_MG_SHIFT))&CAN_RXMGMASK_MG_MASK)
/* RX14MASK Bit Fields */
#define CAN_RX14MASK_RX14M_MASK                  0xFFFFFFFFu
#define CAN_RX14MASK_RX14M_SHIFT                 0u
#define CAN_RX14MASK_RX14M_WIDTH                 32u
#define CAN_RX14MASK_RX14M(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_RX14MASK_RX14M_SHIFT))&CAN_RX14MASK_RX14M_MASK)
/* RX15MASK Bit Fields */
#define CAN_RX15MASK_RX15M_MASK                  0xFFFFFFFFu
#define CAN_RX15MASK_RX15M_SHIFT                 0u
#define CAN_RX15MASK_RX15M_WIDTH                 32u
#define CAN_RX15MASK_RX15M(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_RX15MASK_RX15M_SHIFT))&CAN_RX15MASK_RX15M_MASK)
/* ECR Bit Fields */
#define CAN_ECR_TXERRCNT_MASK                    0xFFu
#define CAN_ECR_TXERRCNT_SHIFT                   0u
#define CAN_ECR_TXERRCNT_WIDTH                   8u
#define CAN_ECR_TXERRCNT(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_ECR_TXERRCNT_SHIFT))&CAN_ECR_TXERRCNT_MASK)
#define CAN_ECR_RXERRCNT_MASK                    0xFF00u
#define CAN_ECR_RXERRCNT_SHIFT                   8u
#define CAN_ECR_RXERRCNT_WIDTH                   8u
#define CAN_ECR_RXERRCNT(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_ECR_RXERRCNT_SHIFT))&CAN_ECR_RXERRCNT_MASK)
#define CAN_ECR_TXERRCNT_FAST_MASK               0xFF0000u
#define CAN_ECR_TXERRCNT_FAST_SHIFT              16u
#define CAN_ECR_TXERRCNT_FAST_WIDTH              8u
#define CAN_ECR_TXERRCNT_FAST(x)                 (((uint32_t)(((uint32_t)(x))<<CAN_ECR_TXERRCNT_FAST_SHIFT))&CAN_ECR_TXERRCNT_FAST_MASK)
#define CAN_ECR_RXERRCNT_FAST_MASK               0xFF000000u
#define CAN_ECR_RXERRCNT_FAST_SHIFT              24u
#define CAN_ECR_RXERRCNT_FAST_WIDTH              8u
#define CAN_ECR_RXERRCNT_FAST(x)                 (((uint32_t)(((uint32_t)(x))<<CAN_ECR_RXERRCNT_FAST_SHIFT))&CAN_ECR_RXERRCNT_FAST_MASK)
/* ESR1 Bit Fields */
#define CAN_ESR1_ERRINT_MASK                     0x2u
#define CAN_ESR1_ERRINT_SHIFT                    1u
#define CAN_ESR1_ERRINT_WIDTH                    1u
#define CAN_ESR1_ERRINT(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_ERRINT_SHIFT))&CAN_ESR1_ERRINT_MASK)
#define CAN_ESR1_BOFFINT_MASK                    0x4u
#define CAN_ESR1_BOFFINT_SHIFT                   2u
#define CAN_ESR1_BOFFINT_WIDTH                   1u
#define CAN_ESR1_BOFFINT(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_BOFFINT_SHIFT))&CAN_ESR1_BOFFINT_MASK)
#define CAN_ESR1_RX_MASK                         0x8u
#define CAN_ESR1_RX_SHIFT                        3u
#define CAN_ESR1_RX_WIDTH                        1u
#define CAN_ESR1_RX(x)                           (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_RX_SHIFT))&CAN_ESR1_RX_MASK)
#define CAN_ESR1_FLTCONF_MASK                    0x30u
#define CAN_ESR1_FLTCONF_SHIFT                   4u
#define CAN_ESR1_FLTCONF_WIDTH                   2u
#define CAN_ESR1_FLTCONF(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_FLTCONF_SHIFT))&CAN_ESR1_FLTCONF_MASK)
#define CAN_ESR1_TX_MASK                         0x40u
#define CAN_ESR1_TX_SHIFT                        6u
#define CAN_ESR1_TX_WIDTH                        1u
#define CAN_ESR1_TX(x)                           (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_TX_SHIFT))&CAN_ESR1_TX_MASK)
#define CAN_ESR1_IDLE_MASK                       0x80u
#define CAN_ESR1_IDLE_SHIFT                      7u
#define CAN_ESR1_IDLE_WIDTH                      1u
#define CAN_ESR1_IDLE(x)                         (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_IDLE_SHIFT))&CAN_ESR1_IDLE_MASK)
#define CAN_ESR1_RXWRN_MASK                      0x100u
#define CAN_ESR1_RXWRN_SHIFT                     8u
#define CAN_ESR1_RXWRN_WIDTH                     1u
#define CAN_ESR1_RXWRN(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_RXWRN_SHIFT))&CAN_ESR1_RXWRN_MASK)
#define CAN_ESR1_TXWRN_MASK                      0x200u
#define CAN_ESR1_TXWRN_SHIFT                     9u
#define CAN_ESR1_TXWRN_WIDTH                     1u
#define CAN_ESR1_TXWRN(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_TXWRN_SHIFT))&CAN_ESR1_TXWRN_MASK)
#define CAN_ESR1_STFERR_MASK                     0x400u
#define CAN_ESR1_STFERR_SHIFT                    10u
#define CAN_ESR1_STFERR_WIDTH                    1u
#define CAN_ESR1_STFERR(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_STFERR_SHIFT))&CAN_ESR1_STFERR_MASK)
#define CAN_ESR1_FRMERR_MASK                     0x800u
#define CAN_ESR1_FRMERR_SHIFT                    11u
#define CAN_ESR1_FRMERR_WIDTH                    1u
#define CAN_ESR1_FRMERR(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_FRMERR_SHIFT))&CAN_ESR1_FRMERR_MASK)
#define CAN_ESR1_CRCERR_MASK                     0x1000u
#define CAN_ESR1_CRCERR_SHIFT                    12u
#define CAN_ESR1_CRCERR_WIDTH                    1u
#define CAN_ESR1_CRCERR(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_CRCERR_SHIFT))&CAN_ESR1_CRCERR_MASK)
#define CAN_ESR1_ACKERR_MASK                     0x2000u
#define CAN_ESR1_ACKERR_SHIFT                    13u
#define CAN_ESR1_ACKERR_WIDTH                    1u
#define CAN_ESR1_ACKERR(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_ACKERR_SHIFT))&CAN_ESR1_ACKERR_MASK)
#define CAN_ESR1_BIT0ERR_MASK                    0x4000u
#define CAN_ESR1_BIT0ERR_SHIFT                   14u
#define CAN_ESR1_BIT0ERR_WIDTH                   1u
#define CAN_ESR1_BIT0ERR(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_BIT0ERR_SHIFT))&CAN_ESR1_BIT0ERR_MASK)
#define CAN_ESR1_BIT1ERR_MASK                    0x8000u
#define CAN_ESR1_BIT1ERR_SHIFT                   15u
#define CAN_ESR1_BIT1ERR_WIDTH                   1u
#define CAN_ESR1_BIT1ERR(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_BIT1ERR_SHIFT))&CAN_ESR1_BIT1ERR_MASK)
#define CAN_ESR1_RWRNINT_MASK                    0x10000u
#define CAN_ESR1_RWRNINT_SHIFT                   16u
#define CAN_ESR1_RWRNINT_WIDTH                   1u
#define CAN_ESR1_RWRNINT(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_RWRNINT_SHIFT))&CAN_ESR1_RWRNINT_MASK)
#define CAN_ESR1_TWRNINT_MASK                    0x20000u
#define CAN_ESR1_TWRNINT_SHIFT                   17u
#define CAN_ESR1_TWRNINT_WIDTH                   1u
#define CAN_ESR1_TWRNINT(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_TWRNINT_SHIFT))&CAN_ESR1_TWRNINT_MASK)
#define CAN_ESR1_SYNCH_MASK                      0x40000u
#define CAN_ESR1_SYNCH_SHIFT                     18u
#define CAN_ESR1_SYNCH_WIDTH                     1u
#define CAN_ESR1_SYNCH(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_SYNCH_SHIFT))&CAN_ESR1_SYNCH_MASK)
#define CAN_ESR1_BOFFDONEINT_MASK                0x80000u
#define CAN_ESR1_BOFFDONEINT_SHIFT               19u
#define CAN_ESR1_BOFFDONEINT_WIDTH               1u
#define CAN_ESR1_BOFFDONEINT(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_BOFFDONEINT_SHIFT))&CAN_ESR1_BOFFDONEINT_MASK)
#define CAN_ESR1_ERRINT_FAST_MASK                0x100000u
#define CAN_ESR1_ERRINT_FAST_SHIFT               20u
#define CAN_ESR1_ERRINT_FAST_WIDTH               1u
#define CAN_ESR1_ERRINT_FAST(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_ERRINT_FAST_SHIFT))&CAN_ESR1_ERRINT_FAST_MASK)
#define CAN_ESR1_ERROVR_MASK                     0x200000u
#define CAN_ESR1_ERROVR_SHIFT                    21u
#define CAN_ESR1_ERROVR_WIDTH                    1u
#define CAN_ESR1_ERROVR(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_ERROVR_SHIFT))&CAN_ESR1_ERROVR_MASK)
#define CAN_ESR1_STFERR_FAST_MASK                0x4000000u
#define CAN_ESR1_STFERR_FAST_SHIFT               26u
#define CAN_ESR1_STFERR_FAST_WIDTH               1u
#define CAN_ESR1_STFERR_FAST(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_STFERR_FAST_SHIFT))&CAN_ESR1_STFERR_FAST_MASK)
#define CAN_ESR1_FRMERR_FAST_MASK                0x8000000u
#define CAN_ESR1_FRMERR_FAST_SHIFT               27u
#define CAN_ESR1_FRMERR_FAST_WIDTH               1u
#define CAN_ESR1_FRMERR_FAST(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_FRMERR_FAST_SHIFT))&CAN_ESR1_FRMERR_FAST_MASK)
#define CAN_ESR1_CRCERR_FAST_MASK                0x10000000u
#define CAN_ESR1_CRCERR_FAST_SHIFT               28u
#define CAN_ESR1_CRCERR_FAST_WIDTH               1u
#define CAN_ESR1_CRCERR_FAST(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_CRCERR_FAST_SHIFT))&CAN_ESR1_CRCERR_FAST_MASK)
#define CAN_ESR1_BIT0ERR_FAST_MASK               0x40000000u
#define CAN_ESR1_BIT0ERR_FAST_SHIFT              30u
#define CAN_ESR1_BIT0ERR_FAST_WIDTH              1u
#define CAN_ESR1_BIT0ERR_FAST(x)                 (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_BIT0ERR_FAST_SHIFT))&CAN_ESR1_BIT0ERR_FAST_MASK)
#define CAN_ESR1_BIT1ERR_FAST_MASK               0x80000000u
#define CAN_ESR1_BIT1ERR_FAST_SHIFT              31u
#define CAN_ESR1_BIT1ERR_FAST_WIDTH              1u
#define CAN_ESR1_BIT1ERR_FAST(x)                 (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_BIT1ERR_FAST_SHIFT))&CAN_ESR1_BIT1ERR_FAST_MASK)
/* IMASK2 Bit Fields */
#define CAN_IMASK2_BUF63TO32M_MASK               0xFFFFFFFFu
#define CAN_IMASK2_BUF63TO32M_SHIFT              0u
#define CAN_IMASK2_BUF63TO32M_WIDTH              32u
#define CAN_IMASK2_BUF63TO32M(x)                 (((uint32_t)(((uint32_t)(x))<<CAN_IMASK2_BUF63TO32M_SHIFT))&CAN_IMASK2_BUF63TO32M_MASK)
/* IMASK1 Bit Fields */
#define CAN_IMASK1_BUF31TO0M_MASK                0xFFFFFFFFu
#define CAN_IMASK1_BUF31TO0M_SHIFT               0u
#define CAN_IMASK1_BUF31TO0M_WIDTH               32u
#define CAN_IMASK1_BUF31TO0M(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_IMASK1_BUF31TO0M_SHIFT))&CAN_IMASK1_BUF31TO0M_MASK)
/* IFLAG2 Bit Fields */
#define CAN_IFLAG2_BUF63TO32I_MASK               0xFFFFFFFFu
#define CAN_IFLAG2_BUF63TO32I_SHIFT              0u
#define CAN_IFLAG2_BUF63TO32I_WIDTH              32u
#define CAN_IFLAG2_BUF63TO32I(x)                 (((uint32_t)(((uint32_t)(x))<<CAN_IFLAG2_BUF63TO32I_SHIFT))&CAN_IFLAG2_BUF63TO32I_MASK)
/* IFLAG1 Bit Fields */
#define CAN_IFLAG1_BUF0I_MASK                    0x1u
#define CAN_IFLAG1_BUF0I_SHIFT                   0u
#define CAN_IFLAG1_BUF0I_WIDTH                   1u
#define CAN_IFLAG1_BUF0I(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_IFLAG1_BUF0I_SHIFT))&CAN_IFLAG1_BUF0I_MASK)
#define CAN_IFLAG1_BUF4TO1I_MASK                 0x1Eu
#define CAN_IFLAG1_BUF4TO1I_SHIFT                1u
#define CAN_IFLAG1_BUF4TO1I_WIDTH                4u
#define CAN_IFLAG1_BUF4TO1I(x)                   (((uint32_t)(((uint32_t)(x))<<CAN_IFLAG1_BUF4TO1I_SHIFT))&CAN_IFLAG1_BUF4TO1I_MASK)
#define CAN_IFLAG1_BUF5I_MASK                    0x20u
#define CAN_IFLAG1_BUF5I_SHIFT                   5u
#define CAN_IFLAG1_BUF5I_WIDTH                   1u
#define CAN_IFLAG1_BUF5I(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_IFLAG1_BUF5I_SHIFT))&CAN_IFLAG1_BUF5I_MASK)
#define CAN_IFLAG1_BUF6I_MASK                    0x40u
#define CAN_IFLAG1_BUF6I_SHIFT                   6u
#define CAN_IFLAG1_BUF6I_WIDTH                   1u
#define CAN_IFLAG1_BUF6I(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_IFLAG1_BUF6I_SHIFT))&CAN_IFLAG1_BUF6I_MASK)
#define CAN_IFLAG1_BUF7I_MASK                    0x80u
#define CAN_IFLAG1_BUF7I_SHIFT                   7u
#define CAN_IFLAG1_BUF7I_WIDTH                   1u
#define CAN_IFLAG1_BUF7I(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_IFLAG1_BUF7I_SHIFT))&CAN_IFLAG1_BUF7I_MASK)
#define CAN_IFLAG1_BUF31TO8I_MASK                0xFFFFFF00u
#define CAN_IFLAG1_BUF31TO8I_SHIFT               8u
#define CAN_IFLAG1_BUF31TO8I_WIDTH               24u
#define CAN_IFLAG1_BUF31TO8I(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_IFLAG1_BUF31TO8I_SHIFT))&CAN_IFLAG1_BUF31TO8I_MASK)
/* CTRL2 Bit Fields */
#define CAN_CTRL2_EDFLTDIS_MASK                  0x800u
#define CAN_CTRL2_EDFLTDIS_SHIFT                 11u
#define CAN_CTRL2_EDFLTDIS_WIDTH                 1u
#define CAN_CTRL2_EDFLTDIS(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_EDFLTDIS_SHIFT))&CAN_CTRL2_EDFLTDIS_MASK)
#define CAN_CTRL2_ISOCANFDEN_MASK                0x1000u
#define CAN_CTRL2_ISOCANFDEN_SHIFT               12u
#define CAN_CTRL2_ISOCANFDEN_WIDTH               1u
#define CAN_CTRL2_ISOCANFDEN(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_ISOCANFDEN_SHIFT))&CAN_CTRL2_ISOCANFDEN_MASK)
#define CAN_CTRL2_PREXCEN_MASK                   0x4000u
#define CAN_CTRL2_PREXCEN_SHIFT                  14u
#define CAN_CTRL2_PREXCEN_WIDTH                  1u
#define CAN_CTRL2_PREXCEN(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_PREXCEN_SHIFT))&CAN_CTRL2_PREXCEN_MASK)
#define CAN_CTRL2_TIMER_SRC_MASK                 0x8000u
#define CAN_CTRL2_TIMER_SRC_SHIFT                15u
#define CAN_CTRL2_TIMER_SRC_WIDTH                1u
#define CAN_CTRL2_TIMER_SRC(x)                   (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_TIMER_SRC_SHIFT))&CAN_CTRL2_TIMER_SRC_MASK)
#define CAN_CTRL2_EACEN_MASK                     0x10000u
#define CAN_CTRL2_EACEN_SHIFT                    16u
#define CAN_CTRL2_EACEN_WIDTH                    1u
#define CAN_CTRL2_EACEN(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_EACEN_SHIFT))&CAN_CTRL2_EACEN_MASK)
#define CAN_CTRL2_RRS_MASK                       0x20000u
#define CAN_CTRL2_RRS_SHIFT                      17u
#define CAN_CTRL2_RRS_WIDTH                      1u
#define CAN_CTRL2_RRS(x)                         (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_RRS_SHIFT))&CAN_CTRL2_RRS_MASK)
#define CAN_CTRL2_MRP_MASK                       0x40000u
#define CAN_CTRL2_MRP_SHIFT                      18u
#define CAN_CTRL2_MRP_WIDTH                      1u
#define CAN_CTRL2_MRP(x)                         (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_MRP_SHIFT))&CAN_CTRL2_MRP_MASK)
#define CAN_CTRL2_TASD_MASK                      0xF80000u
#define CAN_CTRL2_TASD_SHIFT                     19u
#define CAN_CTRL2_TASD_WIDTH                     5u
#define CAN_CTRL2_TASD(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_TASD_SHIFT))&CAN_CTRL2_TASD_MASK)
#define CAN_CTRL2_RFFN_MASK                      0xF000000u
#define CAN_CTRL2_RFFN_SHIFT                     24u
#define CAN_CTRL2_RFFN_WIDTH                     4u
#define CAN_CTRL2_RFFN(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_RFFN_SHIFT))&CAN_CTRL2_RFFN_MASK)
#define CAN_CTRL2_WRMFRZ_MASK                    0x10000000u
#define CAN_CTRL2_WRMFRZ_SHIFT                   28u
#define CAN_CTRL2_WRMFRZ_WIDTH                   1u
#define CAN_CTRL2_WRMFRZ(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_WRMFRZ_SHIFT))&CAN_CTRL2_WRMFRZ_MASK)
#define CAN_CTRL2_ECRWRE_MASK                    0x20000000u
#define CAN_CTRL2_ECRWRE_SHIFT                   29u
#define CAN_CTRL2_ECRWRE_WIDTH                   1u
#define CAN_CTRL2_ECRWRE(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_ECRWRE_SHIFT))&CAN_CTRL2_ECRWRE_MASK)
#define CAN_CTRL2_BOFFDONEMSK_MASK               0x40000000u
#define CAN_CTRL2_BOFFDONEMSK_SHIFT              30u
#define CAN_CTRL2_BOFFDONEMSK_WIDTH              1u
#define CAN_CTRL2_BOFFDONEMSK(x)                 (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_BOFFDONEMSK_SHIFT))&CAN_CTRL2_BOFFDONEMSK_MASK)
#define CAN_CTRL2_ERRMSK_FAST_MASK               0x80000000u
#define CAN_CTRL2_ERRMSK_FAST_SHIFT              31u
#define CAN_CTRL2_ERRMSK_FAST_WIDTH              1u
#define CAN_CTRL2_ERRMSK_FAST(x)                 (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_ERRMSK_FAST_SHIFT))&CAN_CTRL2_ERRMSK_FAST_MASK)
/* ESR2 Bit Fields */
#define CAN_ESR2_IMB_MASK                        0x2000u
#define CAN_ESR2_IMB_SHIFT                       13u
#define CAN_ESR2_IMB_WIDTH                       1u
#define CAN_ESR2_IMB(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_ESR2_IMB_SHIFT))&CAN_ESR2_IMB_MASK)
#define CAN_ESR2_VPS_MASK                        0x4000u
#define CAN_ESR2_VPS_SHIFT                       14u
#define CAN_ESR2_VPS_WIDTH                       1u
#define CAN_ESR2_VPS(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_ESR2_VPS_SHIFT))&CAN_ESR2_VPS_MASK)
#define CAN_ESR2_LPTM_MASK                       0x7F0000u
#define CAN_ESR2_LPTM_SHIFT                      16u
#define CAN_ESR2_LPTM_WIDTH                      7u
#define CAN_ESR2_LPTM(x)                         (((uint32_t)(((uint32_t)(x))<<CAN_ESR2_LPTM_SHIFT))&CAN_ESR2_LPTM_MASK)
/* CRCR Bit Fields */
#define CAN_CRCR_TXCRC_MASK                      0x7FFFu
#define CAN_CRCR_TXCRC_SHIFT                     0u
#define CAN_CRCR_TXCRC_WIDTH                     15u
#define CAN_CRCR_TXCRC(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_CRCR_TXCRC_SHIFT))&CAN_CRCR_TXCRC_MASK)
#define CAN_CRCR_MBCRC_MASK                      0x7F0000u
#define CAN_CRCR_MBCRC_SHIFT                     16u
#define CAN_CRCR_MBCRC_WIDTH                     7u
#define CAN_CRCR_MBCRC(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_CRCR_MBCRC_SHIFT))&CAN_CRCR_MBCRC_MASK)
/* RXFGMASK Bit Fields */
#define CAN_RXFGMASK_FGM_MASK                    0xFFFFFFFFu
#define CAN_RXFGMASK_FGM_SHIFT                   0u
#define CAN_RXFGMASK_FGM_WIDTH                   32u
#define CAN_RXFGMASK_FGM(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_RXFGMASK_FGM_SHIFT))&CAN_RXFGMASK_FGM_MASK)
/* RXFIR Bit Fields */
#define CAN_RXFIR_IDHIT_MASK                     0x1FFu
#define CAN_RXFIR_IDHIT_SHIFT                    0u
#define CAN_RXFIR_IDHIT_WIDTH                    9u
#define CAN_RXFIR_IDHIT(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_RXFIR_IDHIT_SHIFT))&CAN_RXFIR_IDHIT_MASK)
/* CBT Bit Fields */
#define CAN_CBT_EPSEG2_MASK                      0x1Fu
#define CAN_CBT_EPSEG2_SHIFT                     0u
#define CAN_CBT_EPSEG2_WIDTH                     5u
#define CAN_CBT_EPSEG2(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_CBT_EPSEG2_SHIFT))&CAN_CBT_EPSEG2_MASK)
#define CAN_CBT_EPSEG1_MASK                      0x3E0u
#define CAN_CBT_EPSEG1_SHIFT                     5u
#define CAN_CBT_EPSEG1_WIDTH                     5u
#define CAN_CBT_EPSEG1(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_CBT_EPSEG1_SHIFT))&CAN_CBT_EPSEG1_MASK)
#define CAN_CBT_EPROPSEG_MASK                    0xFC00u
#define CAN_CBT_EPROPSEG_SHIFT                   10u
#define CAN_CBT_EPROPSEG_WIDTH                   6u
#define CAN_CBT_EPROPSEG(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_CBT_EPROPSEG_SHIFT))&CAN_CBT_EPROPSEG_MASK)
#define CAN_CBT_ERJW_MASK                        0x1F0000u
#define CAN_CBT_ERJW_SHIFT                       16u
#define CAN_CBT_ERJW_WIDTH                       5u
#define CAN_CBT_ERJW(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_CBT_ERJW_SHIFT))&CAN_CBT_ERJW_MASK)
#define CAN_CBT_EPRESDIV_MASK                    0x7FE00000u
#define CAN_CBT_EPRESDIV_SHIFT                   21u
#define CAN_CBT_EPRESDIV_WIDTH                   10u
#define CAN_CBT_EPRESDIV(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_CBT_EPRESDIV_SHIFT))&CAN_CBT_EPRESDIV_MASK)
#define CAN_CBT_BTF_MASK                         0x80000000u
#define CAN_CBT_BTF_SHIFT                        31u
#define CAN_CBT_BTF_WIDTH                        1u
#define CAN_CBT_BTF(x)                           (((uint32_t)(((uint32_t)(x))<<CAN_CBT_BTF_SHIFT))&CAN_CBT_BTF_MASK)
/* RAMn Bit Fields */
#define CAN_RAMn_DATA_BYTE_3_MASK                0xFFu
#define CAN_RAMn_DATA_BYTE_3_SHIFT               0u
#define CAN_RAMn_DATA_BYTE_3_WIDTH               8u
#define CAN_RAMn_DATA_BYTE_3(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_RAMn_DATA_BYTE_3_SHIFT))&CAN_RAMn_DATA_BYTE_3_MASK)
#define CAN_RAMn_DATA_BYTE_2_MASK                0xFF00u
#define CAN_RAMn_DATA_BYTE_2_SHIFT               8u
#define CAN_RAMn_DATA_BYTE_2_WIDTH               8u
#define CAN_RAMn_DATA_BYTE_2(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_RAMn_DATA_BYTE_2_SHIFT))&CAN_RAMn_DATA_BYTE_2_MASK)
#define CAN_RAMn_DATA_BYTE_1_MASK                0xFF0000u
#define CAN_RAMn_DATA_BYTE_1_SHIFT               16u
#define CAN_RAMn_DATA_BYTE_1_WIDTH               8u
#define CAN_RAMn_DATA_BYTE_1(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_RAMn_DATA_BYTE_1_SHIFT))&CAN_RAMn_DATA_BYTE_1_MASK)
#define CAN_RAMn_DATA_BYTE_0_MASK                0xFF000000u
#define CAN_RAMn_DATA_BYTE_0_SHIFT               24u
#define CAN_RAMn_DATA_BYTE_0_WIDTH               8u
#define CAN_RAMn_DATA_BYTE_0(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_RAMn_DATA_BYTE_0_SHIFT))&CAN_RAMn_DATA_BYTE_0_MASK)
/* RXIMR Bit Fields */
#define CAN_RXIMR_MI_MASK                        0xFFFFFFFFu
#define CAN_RXIMR_MI_SHIFT                       0u
#define CAN_RXIMR_MI_WIDTH                       32u
#define CAN_RXIMR_MI(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_RXIMR_MI_SHIFT))&CAN_RXIMR_MI_MASK)
/* MECR Bit Fields */
#define CAN_MECR_NCEFAFRZ_MASK                   0x80u
#define CAN_MECR_NCEFAFRZ_SHIFT                  7u
#define CAN_MECR_NCEFAFRZ_WIDTH                  1u
#define CAN_MECR_NCEFAFRZ(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_MECR_NCEFAFRZ_SHIFT))&CAN_MECR_NCEFAFRZ_MASK)
#define CAN_MECR_ECCDIS_MASK                     0x100u
#define CAN_MECR_ECCDIS_SHIFT                    8u
#define CAN_MECR_ECCDIS_WIDTH                    1u
#define CAN_MECR_ECCDIS(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_MECR_ECCDIS_SHIFT))&CAN_MECR_ECCDIS_MASK)
#define CAN_MECR_RERRDIS_MASK                    0x200u
#define CAN_MECR_RERRDIS_SHIFT                   9u
#define CAN_MECR_RERRDIS_WIDTH                   1u
#define CAN_MECR_RERRDIS(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_MECR_RERRDIS_SHIFT))&CAN_MECR_RERRDIS_MASK)
#define CAN_MECR_EXTERRIE_MASK                   0x2000u
#define CAN_MECR_EXTERRIE_SHIFT                  13u
#define CAN_MECR_EXTERRIE_WIDTH                  1u
#define CAN_MECR_EXTERRIE(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_MECR_EXTERRIE_SHIFT))&CAN_MECR_EXTERRIE_MASK)
#define CAN_MECR_FAERRIE_MASK                    0x4000u
#define CAN_MECR_FAERRIE_SHIFT                   14u
#define CAN_MECR_FAERRIE_WIDTH                   1u
#define CAN_MECR_FAERRIE(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_MECR_FAERRIE_SHIFT))&CAN_MECR_FAERRIE_MASK)
#define CAN_MECR_HAERRIE_MASK                    0x8000u
#define CAN_MECR_HAERRIE_SHIFT                   15u
#define CAN_MECR_HAERRIE_WIDTH                   1u
#define CAN_MECR_HAERRIE(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_MECR_HAERRIE_SHIFT))&CAN_MECR_HAERRIE_MASK)
#define CAN_MECR_CEI_MSK_MASK                    0x10000u
#define CAN_MECR_CEI_MSK_SHIFT                   16u
#define CAN_MECR_CEI_MSK_WIDTH                   1u
#define CAN_MECR_CEI_MSK(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_MECR_CEI_MSK_SHIFT))&CAN_MECR_CEI_MSK_MASK)
#define CAN_MECR_FANCEI_MSK_MASK                 0x40000u
#define CAN_MECR_FANCEI_MSK_SHIFT                18u
#define CAN_MECR_FANCEI_MSK_WIDTH                1u
#define CAN_MECR_FANCEI_MSK(x)                   (((uint32_t)(((uint32_t)(x))<<CAN_MECR_FANCEI_MSK_SHIFT))&CAN_MECR_FANCEI_MSK_MASK)
#define CAN_MECR_HANCEI_MSK_MASK                 0x80000u
#define CAN_MECR_HANCEI_MSK_SHIFT                19u
#define CAN_MECR_HANCEI_MSK_WIDTH                1u
#define CAN_MECR_HANCEI_MSK(x)                   (((uint32_t)(((uint32_t)(x))<<CAN_MECR_HANCEI_MSK_SHIFT))&CAN_MECR_HANCEI_MSK_MASK)
#define CAN_MECR_ECRWRDIS_MASK                   0x80000000u
#define CAN_MECR_ECRWRDIS_SHIFT                  31u
#define CAN_MECR_ECRWRDIS_WIDTH                  1u
#define CAN_MECR_ECRWRDIS(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_MECR_ECRWRDIS_SHIFT))&CAN_MECR_ECRWRDIS_MASK)
/* ERRIAR Bit Fields */
#define CAN_ERRIAR_INJADDR_L_MASK                0x3u
#define CAN_ERRIAR_INJADDR_L_SHIFT               0u
#define CAN_ERRIAR_INJADDR_L_WIDTH               2u
#define CAN_ERRIAR_INJADDR_L(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_ERRIAR_INJADDR_L_SHIFT))&CAN_ERRIAR_INJADDR_L_MASK)
#define CAN_ERRIAR_INJADDR_H_MASK                0x3FFCu
#define CAN_ERRIAR_INJADDR_H_SHIFT               2u
#define CAN_ERRIAR_INJADDR_H_WIDTH               12u
#define CAN_ERRIAR_INJADDR_H(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_ERRIAR_INJADDR_H_SHIFT))&CAN_ERRIAR_INJADDR_H_MASK)
/* ERRIDPR Bit Fields */
#define CAN_ERRIDPR_DFLIP_MASK                   0xFFFFFFFFu
#define CAN_ERRIDPR_DFLIP_SHIFT                  0u
#define CAN_ERRIDPR_DFLIP_WIDTH                  32u
#define CAN_ERRIDPR_DFLIP(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_ERRIDPR_DFLIP_SHIFT))&CAN_ERRIDPR_DFLIP_MASK)
/* ERRIPPR Bit Fields */
#define CAN_ERRIPPR_PFLIP0_MASK                  0x1Fu
#define CAN_ERRIPPR_PFLIP0_SHIFT                 0u
#define CAN_ERRIPPR_PFLIP0_WIDTH                 5u
#define CAN_ERRIPPR_PFLIP0(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_ERRIPPR_PFLIP0_SHIFT))&CAN_ERRIPPR_PFLIP0_MASK)
#define CAN_ERRIPPR_PFLIP1_MASK                  0x1F00u
#define CAN_ERRIPPR_PFLIP1_SHIFT                 8u
#define CAN_ERRIPPR_PFLIP1_WIDTH                 5u
#define CAN_ERRIPPR_PFLIP1(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_ERRIPPR_PFLIP1_SHIFT))&CAN_ERRIPPR_PFLIP1_MASK)
#define CAN_ERRIPPR_PFLIP2_MASK                  0x1F0000u
#define CAN_ERRIPPR_PFLIP2_SHIFT                 16u
#define CAN_ERRIPPR_PFLIP2_WIDTH                 5u
#define CAN_ERRIPPR_PFLIP2(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_ERRIPPR_PFLIP2_SHIFT))&CAN_ERRIPPR_PFLIP2_MASK)
#define CAN_ERRIPPR_PFLIP3_MASK                  0x1F000000u
#define CAN_ERRIPPR_PFLIP3_SHIFT                 24u
#define CAN_ERRIPPR_PFLIP3_WIDTH                 5u
#define CAN_ERRIPPR_PFLIP3(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_ERRIPPR_PFLIP3_SHIFT))&CAN_ERRIPPR_PFLIP3_MASK)
/* RERRAR Bit Fields */
#define CAN_RERRAR_ERRADDR_MASK                  0x3FFFu
#define CAN_RERRAR_ERRADDR_SHIFT                 0u
#define CAN_RERRAR_ERRADDR_WIDTH                 14u
#define CAN_RERRAR_ERRADDR(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_RERRAR_ERRADDR_SHIFT))&CAN_RERRAR_ERRADDR_MASK)
#define CAN_RERRAR_SAID_MASK                     0x70000u
#define CAN_RERRAR_SAID_SHIFT                    16u
#define CAN_RERRAR_SAID_WIDTH                    3u
#define CAN_RERRAR_SAID(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_RERRAR_SAID_SHIFT))&CAN_RERRAR_SAID_MASK)
#define CAN_RERRAR_NCE_MASK                      0x1000000u
#define CAN_RERRAR_NCE_SHIFT                     24u
#define CAN_RERRAR_NCE_WIDTH                     1u
#define CAN_RERRAR_NCE(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_RERRAR_NCE_SHIFT))&CAN_RERRAR_NCE_MASK)
/* RERRDR Bit Fields */
#define CAN_RERRDR_RDATA_MASK                    0xFFFFFFFFu
#define CAN_RERRDR_RDATA_SHIFT                   0u
#define CAN_RERRDR_RDATA_WIDTH                   32u
#define CAN_RERRDR_RDATA(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_RERRDR_RDATA_SHIFT))&CAN_RERRDR_RDATA_MASK)
/* RERRSYNR Bit Fields */
#define CAN_RERRSYNR_SYND0_MASK                  0x1Fu
#define CAN_RERRSYNR_SYND0_SHIFT                 0u
#define CAN_RERRSYNR_SYND0_WIDTH                 5u
#define CAN_RERRSYNR_SYND0(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_RERRSYNR_SYND0_SHIFT))&CAN_RERRSYNR_SYND0_MASK)
#define CAN_RERRSYNR_BE0_MASK                    0x80u
#define CAN_RERRSYNR_BE0_SHIFT                   7u
#define CAN_RERRSYNR_BE0_WIDTH                   1u
#define CAN_RERRSYNR_BE0(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_RERRSYNR_BE0_SHIFT))&CAN_RERRSYNR_BE0_MASK)
#define CAN_RERRSYNR_SYND1_MASK                  0x1F00u
#define CAN_RERRSYNR_SYND1_SHIFT                 8u
#define CAN_RERRSYNR_SYND1_WIDTH                 5u
#define CAN_RERRSYNR_SYND1(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_RERRSYNR_SYND1_SHIFT))&CAN_RERRSYNR_SYND1_MASK)
#define CAN_RERRSYNR_BE1_MASK                    0x8000u
#define CAN_RERRSYNR_BE1_SHIFT                   15u
#define CAN_RERRSYNR_BE1_WIDTH                   1u
#define CAN_RERRSYNR_BE1(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_RERRSYNR_BE1_SHIFT))&CAN_RERRSYNR_BE1_MASK)
#define CAN_RERRSYNR_SYND2_MASK                  0x1F0000u
#define CAN_RERRSYNR_SYND2_SHIFT                 16u
#define CAN_RERRSYNR_SYND2_WIDTH                 5u
#define CAN_RERRSYNR_SYND2(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_RERRSYNR_SYND2_SHIFT))&CAN_RERRSYNR_SYND2_MASK)
#define CAN_RERRSYNR_BE2_MASK                    0x800000u
#define CAN_RERRSYNR_BE2_SHIFT                   23u
#define CAN_RERRSYNR_BE2_WIDTH                   1u
#define CAN_RERRSYNR_BE2(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_RERRSYNR_BE2_SHIFT))&CAN_RERRSYNR_BE2_MASK)
#define CAN_RERRSYNR_SYND3_MASK                  0x1F000000u
#define CAN_RERRSYNR_SYND3_SHIFT                 24u
#define CAN_RERRSYNR_SYND3_WIDTH                 5u
#define CAN_RERRSYNR_SYND3(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_RERRSYNR_SYND3_SHIFT))&CAN_RERRSYNR_SYND3_MASK)
#define CAN_RERRSYNR_BE3_MASK                    0x80000000u
#define CAN_RERRSYNR_BE3_SHIFT                   31u
#define CAN_RERRSYNR_BE3_WIDTH                   1u
#define CAN_RERRSYNR_BE3(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_RERRSYNR_BE3_SHIFT))&CAN_RERRSYNR_BE3_MASK)
/* ERRSR Bit Fields */
#define CAN_ERRSR_CEIOF_MASK                     0x1u
#define CAN_ERRSR_CEIOF_SHIFT                    0u
#define CAN_ERRSR_CEIOF_WIDTH                    1u
#define CAN_ERRSR_CEIOF(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_ERRSR_CEIOF_SHIFT))&CAN_ERRSR_CEIOF_MASK)
#define CAN_ERRSR_FANCEIOF_MASK                  0x4u
#define CAN_ERRSR_FANCEIOF_SHIFT                 2u
#define CAN_ERRSR_FANCEIOF_WIDTH                 1u
#define CAN_ERRSR_FANCEIOF(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_ERRSR_FANCEIOF_SHIFT))&CAN_ERRSR_FANCEIOF_MASK)
#define CAN_ERRSR_HANCEIOF_MASK                  0x8u
#define CAN_ERRSR_HANCEIOF_SHIFT                 3u
#define CAN_ERRSR_HANCEIOF_WIDTH                 1u
#define CAN_ERRSR_HANCEIOF(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_ERRSR_HANCEIOF_SHIFT))&CAN_ERRSR_HANCEIOF_MASK)
#define CAN_ERRSR_CEIF_MASK                      0x10000u
#define CAN_ERRSR_CEIF_SHIFT                     16u
#define CAN_ERRSR_CEIF_WIDTH                     1u
#define CAN_ERRSR_CEIF(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_ERRSR_CEIF_SHIFT))&CAN_ERRSR_CEIF_MASK)
#define CAN_ERRSR_FANCEIF_MASK                   0x40000u
#define CAN_ERRSR_FANCEIF_SHIFT                  18u
#define CAN_ERRSR_FANCEIF_WIDTH                  1u
#define CAN_ERRSR_FANCEIF(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_ERRSR_FANCEIF_SHIFT))&CAN_ERRSR_FANCEIF_MASK)
#define CAN_ERRSR_HANCEIF_MASK                   0x80000u
#define CAN_ERRSR_HANCEIF_SHIFT                  19u
#define CAN_ERRSR_HANCEIF_WIDTH                  1u
#define CAN_ERRSR_HANCEIF(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_ERRSR_HANCEIF_SHIFT))&CAN_ERRSR_HANCEIF_MASK)
/* FDCTRL Bit Fields */
#define CAN_FDCTRL_TDCVAL_MASK                   0x3Fu
#define CAN_FDCTRL_TDCVAL_SHIFT                  0u
#define CAN_FDCTRL_TDCVAL_WIDTH                  6u
#define CAN_FDCTRL_TDCVAL(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_FDCTRL_TDCVAL_SHIFT))&CAN_FDCTRL_TDCVAL_MASK)
#define CAN_FDCTRL_TDCOFF_MASK                   0x1F00u
#define CAN_FDCTRL_TDCOFF_SHIFT                  8u
#define CAN_FDCTRL_TDCOFF_WIDTH                  5u
#define CAN_FDCTRL_TDCOFF(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_FDCTRL_TDCOFF_SHIFT))&CAN_FDCTRL_TDCOFF_MASK)
#define CAN_FDCTRL_TDCFAIL_MASK                  0x4000u
#define CAN_FDCTRL_TDCFAIL_SHIFT                 14u
#define CAN_FDCTRL_TDCFAIL_WIDTH                 1u
#define CAN_FDCTRL_TDCFAIL(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_FDCTRL_TDCFAIL_SHIFT))&CAN_FDCTRL_TDCFAIL_MASK)
#define CAN_FDCTRL_TDCEN_MASK                    0x8000u
#define CAN_FDCTRL_TDCEN_SHIFT                   15u
#define CAN_FDCTRL_TDCEN_WIDTH                   1u
#define CAN_FDCTRL_TDCEN(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_FDCTRL_TDCEN_SHIFT))&CAN_FDCTRL_TDCEN_MASK)
#define CAN_FDCTRL_MBDSR0_MASK                   0x30000u
#define CAN_FDCTRL_MBDSR0_SHIFT                  16u
#define CAN_FDCTRL_MBDSR0_WIDTH                  2u
#define CAN_FDCTRL_MBDSR0(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_FDCTRL_MBDSR0_SHIFT))&CAN_FDCTRL_MBDSR0_MASK)
#define CAN_FDCTRL_MBDSR1_MASK                   0x180000u
#define CAN_FDCTRL_MBDSR1_SHIFT                  19u
#define CAN_FDCTRL_MBDSR1_WIDTH                  2u
#define CAN_FDCTRL_MBDSR1(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_FDCTRL_MBDSR1_SHIFT))&CAN_FDCTRL_MBDSR1_MASK)
#define CAN_FDCTRL_FDRATE_MASK                   0x80000000u
#define CAN_FDCTRL_FDRATE_SHIFT                  31u
#define CAN_FDCTRL_FDRATE_WIDTH                  1u
#define CAN_FDCTRL_FDRATE(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_FDCTRL_FDRATE_SHIFT))&CAN_FDCTRL_FDRATE_MASK)
/* FDCBT Bit Fields */
#define CAN_FDCBT_FPSEG2_MASK                    0x7u
#define CAN_FDCBT_FPSEG2_SHIFT                   0u
#define CAN_FDCBT_FPSEG2_WIDTH                   3u
#define CAN_FDCBT_FPSEG2(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_FDCBT_FPSEG2_SHIFT))&CAN_FDCBT_FPSEG2_MASK)
#define CAN_FDCBT_FPSEG1_MASK                    0xE0u
#define CAN_FDCBT_FPSEG1_SHIFT                   5u
#define CAN_FDCBT_FPSEG1_WIDTH                   3u
#define CAN_FDCBT_FPSEG1(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_FDCBT_FPSEG1_SHIFT))&CAN_FDCBT_FPSEG1_MASK)
#define CAN_FDCBT_FPROPSEG_MASK                  0x7C00u
#define CAN_FDCBT_FPROPSEG_SHIFT                 10u
#define CAN_FDCBT_FPROPSEG_WIDTH                 5u
#define CAN_FDCBT_FPROPSEG(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_FDCBT_FPROPSEG_SHIFT))&CAN_FDCBT_FPROPSEG_MASK)
#define CAN_FDCBT_FRJW_MASK                      0x70000u
#define CAN_FDCBT_FRJW_SHIFT                     16u
#define CAN_FDCBT_FRJW_WIDTH                     3u
#define CAN_FDCBT_FRJW(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_FDCBT_FRJW_SHIFT))&CAN_FDCBT_FRJW_MASK)
#define CAN_FDCBT_FPRESDIV_MASK                  0x3FF00000u
#define CAN_FDCBT_FPRESDIV_SHIFT                 20u
#define CAN_FDCBT_FPRESDIV_WIDTH                 10u
#define CAN_FDCBT_FPRESDIV(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_FDCBT_FPRESDIV_SHIFT))&CAN_FDCBT_FPRESDIV_MASK)
/* FDCRC Bit Fields */
#define CAN_FDCRC_FD_TXCRC_MASK                  0x1FFFFFu
#define CAN_FDCRC_FD_TXCRC_SHIFT                 0u
#define CAN_FDCRC_FD_TXCRC_WIDTH                 21u
#define CAN_FDCRC_FD_TXCRC(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_FDCRC_FD_TXCRC_SHIFT))&CAN_FDCRC_FD_TXCRC_MASK)
#define CAN_FDCRC_FD_MBCRC_MASK                  0x7F000000u
#define CAN_FDCRC_FD_MBCRC_SHIFT                 24u
#define CAN_FDCRC_FD_MBCRC_WIDTH                 7u
#define CAN_FDCRC_FD_MBCRC(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_FDCRC_FD_MBCRC_SHIFT))&CAN_FDCRC_FD_MBCRC_MASK)

/*!
 * @}
 */ /* end of group CAN_Register_Masks */


/*!
 * @}
 */ /* end of group CAN_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- CMU Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup CMU_Peripheral_Access_Layer CMU Peripheral Access Layer
 * @{
 */


/** CMU - Size of Registers Arrays */

/** CMU - Register Layout Typedef */
typedef struct {
  __IO uint32_t CSR;                               /**< CMU Control Status Register, offset: 0x0 */
  __I  uint32_t FDR;                               /**< CMU Frequency Display Register, offset: 0x4 */
  __IO uint32_t HFREFR;                            /**< CMU High Frequency Reference Register CLKMN1, offset: 0x8 */
  __IO uint32_t LFREFR;                            /**< CMU Low Frequency Reference Register CLKMN1, offset: 0xC */
  __IO uint32_t ISR;                               /**< CMU Interrupt Status Register, offset: 0x10 */
       uint8_t RESERVED_0[4];
  __IO uint32_t MDR;                               /**< CMU Measurement Duration Register, offset: 0x18 */
} CMU_Type, *CMU_MemMapPtr;

 /** Number of instances of the CMU module. */
#define CMU_INSTANCE_COUNT                       (14u)


/* CMU - Peripheral instance base addresses */
/** Peripheral CMU_0 base address */
#define CMU_0_BASE                               (0x400C4000u)
/** Peripheral CMU_0 base pointer */
#define CMU_0                                    ((CMU_Type *)CMU_0_BASE)
/** Peripheral CMU_1 base address */
#define CMU_1_BASE                               (0x400C4020u)
/** Peripheral CMU_1 base pointer */
#define CMU_1                                    ((CMU_Type *)CMU_1_BASE)
/** Peripheral CMU_10 base address */
#define CMU_10_BASE                              (0x400C4140u)
/** Peripheral CMU_10 base pointer */
#define CMU_10                                   ((CMU_Type *)CMU_10_BASE)
/** Peripheral CMU_11 base address */
#define CMU_11_BASE                              (0x400C4160u)
/** Peripheral CMU_11 base pointer */
#define CMU_11                                   ((CMU_Type *)CMU_11_BASE)
/** Peripheral CMU_12 base address */
#define CMU_12_BASE                              (0x400C4180u)
/** Peripheral CMU_12 base pointer */
#define CMU_12                                   ((CMU_Type *)CMU_12_BASE)
/** Peripheral CMU_13 base address */
#define CMU_13_BASE                              (0x400C41A0u)
/** Peripheral CMU_13 base pointer */
#define CMU_13                                   ((CMU_Type *)CMU_13_BASE)
/** Peripheral CMU_2 base address */
#define CMU_2_BASE                               (0x400C4040u)
/** Peripheral CMU_2 base pointer */
#define CMU_2                                    ((CMU_Type *)CMU_2_BASE)
/** Peripheral CMU_3 base address */
#define CMU_3_BASE                               (0x400C4060u)
/** Peripheral CMU_3 base pointer */
#define CMU_3                                    ((CMU_Type *)CMU_3_BASE)
/** Peripheral CMU_4 base address */
#define CMU_4_BASE                               (0x400C4080u)
/** Peripheral CMU_4 base pointer */
#define CMU_4                                    ((CMU_Type *)CMU_4_BASE)
/** Peripheral CMU_5 base address */
#define CMU_5_BASE                               (0x400C40A0u)
/** Peripheral CMU_5 base pointer */
#define CMU_5                                    ((CMU_Type *)CMU_5_BASE)
/** Peripheral CMU_6 base address */
#define CMU_6_BASE                               (0x400C40C0u)
/** Peripheral CMU_6 base pointer */
#define CMU_6                                    ((CMU_Type *)CMU_6_BASE)
/** Peripheral CMU_7 base address */
#define CMU_7_BASE                               (0x400C40E0u)
/** Peripheral CMU_7 base pointer */
#define CMU_7                                    ((CMU_Type *)CMU_7_BASE)
/** Peripheral CMU_8 base address */
#define CMU_8_BASE                               (0x400C4100u)
/** Peripheral CMU_8 base pointer */
#define CMU_8                                    ((CMU_Type *)CMU_8_BASE)
/** Peripheral CMU_9 base address */
#define CMU_9_BASE                               (0x400C4120u)
/** Peripheral CMU_9 base pointer */
#define CMU_9                                    ((CMU_Type *)CMU_9_BASE)
/** Array initializer of CMU peripheral base addresses */
#define CMU_BASE_ADDRS                           { CMU_0_BASE, CMU_1_BASE, CMU_10_BASE, CMU_11_BASE, CMU_12_BASE, CMU_13_BASE, CMU_2_BASE, CMU_3_BASE, CMU_4_BASE, CMU_5_BASE, CMU_6_BASE, CMU_7_BASE, CMU_8_BASE, CMU_9_BASE }
/** Array initializer of CMU peripheral base pointers */
#define CMU_BASE_PTRS                            { CMU_0, CMU_1, CMU_10, CMU_11, CMU_12, CMU_13, CMU_2, CMU_3, CMU_4, CMU_5, CMU_6, CMU_7, CMU_8, CMU_9 }

/* ----------------------------------------------------------------------------
   -- CMU Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup CMU_Register_Masks CMU Register Masks
 * @{
 */

/* CSR Bit Fields */
#define CMU_CSR_CME_MASK                         0x1u
#define CMU_CSR_CME_SHIFT                        0u
#define CMU_CSR_CME_WIDTH                        1u
#define CMU_CSR_CME(x)                           (((uint32_t)(((uint32_t)(x))<<CMU_CSR_CME_SHIFT))&CMU_CSR_CME_MASK)
#define CMU_CSR_RCDIV_MASK                       0x6u
#define CMU_CSR_RCDIV_SHIFT                      1u
#define CMU_CSR_RCDIV_WIDTH                      2u
#define CMU_CSR_RCDIV(x)                         (((uint32_t)(((uint32_t)(x))<<CMU_CSR_RCDIV_SHIFT))&CMU_CSR_RCDIV_MASK)
#define CMU_CSR_CKSEL1_MASK                      0x300u
#define CMU_CSR_CKSEL1_SHIFT                     8u
#define CMU_CSR_CKSEL1_WIDTH                     2u
#define CMU_CSR_CKSEL1(x)                        (((uint32_t)(((uint32_t)(x))<<CMU_CSR_CKSEL1_SHIFT))&CMU_CSR_CKSEL1_MASK)
#define CMU_CSR_SFM_MASK                         0x800000u
#define CMU_CSR_SFM_SHIFT                        23u
#define CMU_CSR_SFM_WIDTH                        1u
#define CMU_CSR_SFM(x)                           (((uint32_t)(((uint32_t)(x))<<CMU_CSR_SFM_SHIFT))&CMU_CSR_SFM_MASK)
/* FDR Bit Fields */
#define CMU_FDR_FD_MASK                          0xFFFFFu
#define CMU_FDR_FD_SHIFT                         0u
#define CMU_FDR_FD_WIDTH                         20u
#define CMU_FDR_FD(x)                            (((uint32_t)(((uint32_t)(x))<<CMU_FDR_FD_SHIFT))&CMU_FDR_FD_MASK)
/* HFREFR Bit Fields */
#define CMU_HFREFR_HFREF_MASK                    0xFFFu
#define CMU_HFREFR_HFREF_SHIFT                   0u
#define CMU_HFREFR_HFREF_WIDTH                   12u
#define CMU_HFREFR_HFREF(x)                      (((uint32_t)(((uint32_t)(x))<<CMU_HFREFR_HFREF_SHIFT))&CMU_HFREFR_HFREF_MASK)
/* LFREFR Bit Fields */
#define CMU_LFREFR_LFREF_MASK                    0xFFFu
#define CMU_LFREFR_LFREF_SHIFT                   0u
#define CMU_LFREFR_LFREF_WIDTH                   12u
#define CMU_LFREFR_LFREF(x)                      (((uint32_t)(((uint32_t)(x))<<CMU_LFREFR_LFREF_SHIFT))&CMU_LFREFR_LFREF_MASK)
/* ISR Bit Fields */
#define CMU_ISR_OLRI_MASK                        0x1u
#define CMU_ISR_OLRI_SHIFT                       0u
#define CMU_ISR_OLRI_WIDTH                       1u
#define CMU_ISR_OLRI(x)                          (((uint32_t)(((uint32_t)(x))<<CMU_ISR_OLRI_SHIFT))&CMU_ISR_OLRI_MASK)
#define CMU_ISR_FLLI_MASK                        0x2u
#define CMU_ISR_FLLI_SHIFT                       1u
#define CMU_ISR_FLLI_WIDTH                       1u
#define CMU_ISR_FLLI(x)                          (((uint32_t)(((uint32_t)(x))<<CMU_ISR_FLLI_SHIFT))&CMU_ISR_FLLI_MASK)
#define CMU_ISR_FHHI_MASK                        0x4u
#define CMU_ISR_FHHI_SHIFT                       2u
#define CMU_ISR_FHHI_WIDTH                       1u
#define CMU_ISR_FHHI(x)                          (((uint32_t)(((uint32_t)(x))<<CMU_ISR_FHHI_SHIFT))&CMU_ISR_FHHI_MASK)
/* MDR Bit Fields */
#define CMU_MDR_MD_MASK                          0xFFFFFu
#define CMU_MDR_MD_SHIFT                         0u
#define CMU_MDR_MD_WIDTH                         20u
#define CMU_MDR_MD(x)                            (((uint32_t)(((uint32_t)(x))<<CMU_MDR_MD_SHIFT))&CMU_MDR_MD_MASK)

/*!
 * @}
 */ /* end of group CMU_Register_Masks */


/*!
 * @}
 */ /* end of group CMU_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- CRC Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup CRC_Peripheral_Access_Layer CRC Peripheral Access Layer
 * @{
 */


/** CRC - Size of Registers Arrays */

/** CRC - Register Layout Typedef */
typedef struct {
  __IO uint32_t CFG;                               /**< Configuration Register, offset: 0x0 */
  __IO uint32_t INP;                               /**< Input Register, offset: 0x4 */
  __IO uint32_t CSTAT;                             /**< Current Status Register, offset: 0x8 */
  __I  uint32_t OUTP;                              /**< Output Register, offset: 0xC */
} CRC_Type, *CRC_MemMapPtr;

 /** Number of instances of the CRC module. */
#define CRC_INSTANCE_COUNT                       (6u)


/* CRC - Peripheral instance base addresses */
/** Peripheral CRC_0 base address */
#define CRC_0_BASE                               (0x4005B000u)
/** Peripheral CRC_0 base pointer */
#define CRC_0                                    ((CRC_Type *)CRC_0_BASE)
/** Peripheral CRC_1 base address */
#define CRC_1_BASE                               (0x4005B010u)
/** Peripheral CRC_1 base pointer */
#define CRC_1                                    ((CRC_Type *)CRC_1_BASE)
/** Peripheral CRC_2 base address */
#define CRC_2_BASE                               (0x4005B020u)
/** Peripheral CRC_2 base pointer */
#define CRC_2                                    ((CRC_Type *)CRC_2_BASE)
/** Peripheral CRC_3 base address */
#define CRC_3_BASE                               (0x400CC000u)
/** Peripheral CRC_3 base pointer */
#define CRC_3                                    ((CRC_Type *)CRC_3_BASE)
/** Peripheral CRC_4 base address */
#define CRC_4_BASE                               (0x400CC010u)
/** Peripheral CRC_4 base pointer */
#define CRC_4                                    ((CRC_Type *)CRC_4_BASE)
/** Peripheral CRC_5 base address */
#define CRC_5_BASE                               (0x400CC020u)
/** Peripheral CRC_5 base pointer */
#define CRC_5                                    ((CRC_Type *)CRC_5_BASE)
/** Array initializer of CRC peripheral base addresses */
#define CRC_BASE_ADDRS                           { CRC_0_BASE, CRC_1_BASE, CRC_2_BASE, CRC_3_BASE, CRC_4_BASE, CRC_5_BASE }
/** Array initializer of CRC peripheral base pointers */
#define CRC_BASE_PTRS                            { CRC_0, CRC_1, CRC_2, CRC_3, CRC_4, CRC_5 }

/* ----------------------------------------------------------------------------
   -- CRC Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup CRC_Register_Masks CRC Register Masks
 * @{
 */

/* CFG Bit Fields */
#define CRC_CFG_INV_MASK                         0x1u
#define CRC_CFG_INV_SHIFT                        0u
#define CRC_CFG_INV_WIDTH                        1u
#define CRC_CFG_INV(x)                           (((uint32_t)(((uint32_t)(x))<<CRC_CFG_INV_SHIFT))&CRC_CFG_INV_MASK)
#define CRC_CFG_SWAP_MASK                        0x2u
#define CRC_CFG_SWAP_SHIFT                       1u
#define CRC_CFG_SWAP_WIDTH                       1u
#define CRC_CFG_SWAP(x)                          (((uint32_t)(((uint32_t)(x))<<CRC_CFG_SWAP_SHIFT))&CRC_CFG_SWAP_MASK)
#define CRC_CFG_POLYG_MASK                       0xCu
#define CRC_CFG_POLYG_SHIFT                      2u
#define CRC_CFG_POLYG_WIDTH                      2u
#define CRC_CFG_POLYG(x)                         (((uint32_t)(((uint32_t)(x))<<CRC_CFG_POLYG_SHIFT))&CRC_CFG_POLYG_MASK)
#define CRC_CFG_SWAP_BITWISE_MASK                0x10u
#define CRC_CFG_SWAP_BITWISE_SHIFT               4u
#define CRC_CFG_SWAP_BITWISE_WIDTH               1u
#define CRC_CFG_SWAP_BITWISE(x)                  (((uint32_t)(((uint32_t)(x))<<CRC_CFG_SWAP_BITWISE_SHIFT))&CRC_CFG_SWAP_BITWISE_MASK)
#define CRC_CFG_SWAP_BYTEWISE_MASK               0x20u
#define CRC_CFG_SWAP_BYTEWISE_SHIFT              5u
#define CRC_CFG_SWAP_BYTEWISE_WIDTH              1u
#define CRC_CFG_SWAP_BYTEWISE(x)                 (((uint32_t)(((uint32_t)(x))<<CRC_CFG_SWAP_BYTEWISE_SHIFT))&CRC_CFG_SWAP_BYTEWISE_MASK)
/* INP Bit Fields */
#define CRC_INP_INP_MASK                         0xFFFFFFFFu
#define CRC_INP_INP_SHIFT                        0u
#define CRC_INP_INP_WIDTH                        32u
#define CRC_INP_INP(x)                           (((uint32_t)(((uint32_t)(x))<<CRC_INP_INP_SHIFT))&CRC_INP_INP_MASK)
/* CSTAT Bit Fields */
#define CRC_CSTAT_CSTAT_MASK                     0xFFFFFFFFu
#define CRC_CSTAT_CSTAT_SHIFT                    0u
#define CRC_CSTAT_CSTAT_WIDTH                    32u
#define CRC_CSTAT_CSTAT(x)                       (((uint32_t)(((uint32_t)(x))<<CRC_CSTAT_CSTAT_SHIFT))&CRC_CSTAT_CSTAT_MASK)
/* OUTP Bit Fields */
#define CRC_OUTP_OUTP_MASK                       0xFFFFFFFFu
#define CRC_OUTP_OUTP_SHIFT                      0u
#define CRC_OUTP_OUTP_WIDTH                      32u
#define CRC_OUTP_OUTP(x)                         (((uint32_t)(((uint32_t)(x))<<CRC_OUTP_OUTP_SHIFT))&CRC_OUTP_OUTP_MASK)

/*!
 * @}
 */ /* end of group CRC_Register_Masks */


/*!
 * @}
 */ /* end of group CRC_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- CSE Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup CSE_Peripheral_Access_Layer CSE Peripheral Access Layer
 * @{
 */


/** CSE - Size of Registers Arrays */
#define CSE_P_COUNT                              5u
#define CSE_KIA_COUNT                            2u

/** CSE - Register Layout Typedef */
typedef struct {
  __IO uint32_t CR;                                /**< Control Register, offset: 0x0 */
  __I  uint32_t SR;                                /**< Status Register, offset: 0x4 */
  __IO uint32_t IR;                                /**< Interrupt Register, offset: 0x8 */
  __I  uint32_t ECR;                               /**< Error Code Register, offset: 0xC */
  __IO uint32_t TRNG;                              /**< TRNG Control Register, offset: 0x10 */
       uint8_t RESERVED_0[12];
  __IO uint32_t CMD;                               /**< Command Register, offset: 0x20 */
  __IO uint32_t P[CSE_P_COUNT];                    /**< Command Parameter Register, array offset: 0x24, array step: 0x4 */
       uint8_t RESERVED_1[24];
  __IO uint32_t KIA[CSE_KIA_COUNT];                /**< Key Image Address Register, array offset: 0x50, array step: 0x4 */
  __I  uint32_t OTP;                               /**< OTP Status Register, offset: 0x58 */
  __I  uint32_t PC;                                /**< Publish Count Register, offset: 0x5C */
       uint8_t RESERVED_2[16];
  __I  uint32_t SRA0;                              /**< Secure RAM Address Register, offset: 0x70 */
  __I  uint32_t SRS0;                              /**< Secure RAM Size Register, offset: 0x74 */
} CSE_Type, *CSE_MemMapPtr;

 /** Number of instances of the CSE module. */
#define CSE_INSTANCE_COUNT                       (1u)


/* CSE - Peripheral instance base addresses */
/** Peripheral CSE base address */
#define CSE_BASE                                 (0x40001000u)
/** Peripheral CSE base pointer */
#define CSE                                      ((CSE_Type *)CSE_BASE)
/** Array initializer of CSE peripheral base addresses */
#define CSE_BASE_ADDRS                           { CSE_BASE }
/** Array initializer of CSE peripheral base pointers */
#define CSE_BASE_PTRS                            { CSE }

/* ----------------------------------------------------------------------------
   -- CSE Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup CSE_Register_Masks CSE Register Masks
 * @{
 */

/* CR Bit Fields */
#define CSE_CR_CIE_MASK                          0x1u
#define CSE_CR_CIE_SHIFT                         0u
#define CSE_CR_CIE_WIDTH                         1u
#define CSE_CR_CIE(x)                            (((uint32_t)(((uint32_t)(x))<<CSE_CR_CIE_SHIFT))&CSE_CR_CIE_MASK)
#define CSE_CR_DRE_MASK                          0x2u
#define CSE_CR_DRE_SHIFT                         1u
#define CSE_CR_DRE_WIDTH                         1u
#define CSE_CR_DRE(x)                            (((uint32_t)(((uint32_t)(x))<<CSE_CR_DRE_SHIFT))&CSE_CR_DRE_MASK)
#define CSE_CR_SUS_MASK                          0x4u
#define CSE_CR_SUS_SHIFT                         2u
#define CSE_CR_SUS_WIDTH                         1u
#define CSE_CR_SUS(x)                            (((uint32_t)(((uint32_t)(x))<<CSE_CR_SUS_SHIFT))&CSE_CR_SUS_MASK)
#define CSE_CR_MDIS_MASK                         0x8u
#define CSE_CR_MDIS_SHIFT                        3u
#define CSE_CR_MDIS_WIDTH                        1u
#define CSE_CR_MDIS(x)                           (((uint32_t)(((uint32_t)(x))<<CSE_CR_MDIS_SHIFT))&CSE_CR_MDIS_MASK)
#define CSE_CR_SFE_MASK                          0x10u
#define CSE_CR_SFE_SHIFT                         4u
#define CSE_CR_SFE_WIDTH                         1u
#define CSE_CR_SFE(x)                            (((uint32_t)(((uint32_t)(x))<<CSE_CR_SFE_SHIFT))&CSE_CR_SFE_MASK)
#define CSE_CR_KBS_MASK                          0x20u
#define CSE_CR_KBS_SHIFT                         5u
#define CSE_CR_KBS_WIDTH                         1u
#define CSE_CR_KBS(x)                            (((uint32_t)(((uint32_t)(x))<<CSE_CR_KBS_SHIFT))&CSE_CR_KBS_MASK)
/* SR Bit Fields */
#define CSE_SR_BSY_MASK                          0x1u
#define CSE_SR_BSY_SHIFT                         0u
#define CSE_SR_BSY_WIDTH                         1u
#define CSE_SR_BSY(x)                            (((uint32_t)(((uint32_t)(x))<<CSE_SR_BSY_SHIFT))&CSE_SR_BSY_MASK)
#define CSE_SR_SB_MASK                           0x2u
#define CSE_SR_SB_SHIFT                          1u
#define CSE_SR_SB_WIDTH                          1u
#define CSE_SR_SB(x)                             (((uint32_t)(((uint32_t)(x))<<CSE_SR_SB_SHIFT))&CSE_SR_SB_MASK)
#define CSE_SR_BIN_MASK                          0x4u
#define CSE_SR_BIN_SHIFT                         2u
#define CSE_SR_BIN_WIDTH                         1u
#define CSE_SR_BIN(x)                            (((uint32_t)(((uint32_t)(x))<<CSE_SR_BIN_SHIFT))&CSE_SR_BIN_MASK)
#define CSE_SR_BFN_MASK                          0x8u
#define CSE_SR_BFN_SHIFT                         3u
#define CSE_SR_BFN_WIDTH                         1u
#define CSE_SR_BFN(x)                            (((uint32_t)(((uint32_t)(x))<<CSE_SR_BFN_SHIFT))&CSE_SR_BFN_MASK)
#define CSE_SR_BOK_MASK                          0x10u
#define CSE_SR_BOK_SHIFT                         4u
#define CSE_SR_BOK_WIDTH                         1u
#define CSE_SR_BOK(x)                            (((uint32_t)(((uint32_t)(x))<<CSE_SR_BOK_SHIFT))&CSE_SR_BOK_MASK)
#define CSE_SR_RIN_MASK                          0x20u
#define CSE_SR_RIN_SHIFT                         5u
#define CSE_SR_RIN_WIDTH                         1u
#define CSE_SR_RIN(x)                            (((uint32_t)(((uint32_t)(x))<<CSE_SR_RIN_SHIFT))&CSE_SR_RIN_MASK)
#define CSE_SR_EDB_MASK                          0x40u
#define CSE_SR_EDB_SHIFT                         6u
#define CSE_SR_EDB_WIDTH                         1u
#define CSE_SR_EDB(x)                            (((uint32_t)(((uint32_t)(x))<<CSE_SR_EDB_SHIFT))&CSE_SR_EDB_MASK)
#define CSE_SR_IDB_MASK                          0x80u
#define CSE_SR_IDB_SHIFT                         7u
#define CSE_SR_IDB_WIDTH                         1u
#define CSE_SR_IDB(x)                            (((uint32_t)(((uint32_t)(x))<<CSE_SR_IDB_SHIFT))&CSE_SR_IDB_MASK)
#define CSE_SR_EX_MASK                           0x100u
#define CSE_SR_EX_SHIFT                          8u
#define CSE_SR_EX_WIDTH                          1u
#define CSE_SR_EX(x)                             (((uint32_t)(((uint32_t)(x))<<CSE_SR_EX_SHIFT))&CSE_SR_EX_MASK)
#define CSE_SR_FLC_MASK                          0x200u
#define CSE_SR_FLC_SHIFT                         9u
#define CSE_SR_FLC_WIDTH                         1u
#define CSE_SR_FLC(x)                            (((uint32_t)(((uint32_t)(x))<<CSE_SR_FLC_SHIFT))&CSE_SR_FLC_MASK)
#define CSE_SR_LEC_MASK                          0x400u
#define CSE_SR_LEC_SHIFT                         10u
#define CSE_SR_LEC_WIDTH                         1u
#define CSE_SR_LEC(x)                            (((uint32_t)(((uint32_t)(x))<<CSE_SR_LEC_SHIFT))&CSE_SR_LEC_MASK)
/* IR Bit Fields */
#define CSE_IR_CIF_MASK                          0x1u
#define CSE_IR_CIF_SHIFT                         0u
#define CSE_IR_CIF_WIDTH                         1u
#define CSE_IR_CIF(x)                            (((uint32_t)(((uint32_t)(x))<<CSE_IR_CIF_SHIFT))&CSE_IR_CIF_MASK)
/* ECR Bit Fields */
#define CSE_ECR_EC_MASK                          0x3Fu
#define CSE_ECR_EC_SHIFT                         0u
#define CSE_ECR_EC_WIDTH                         6u
#define CSE_ECR_EC(x)                            (((uint32_t)(((uint32_t)(x))<<CSE_ECR_EC_SHIFT))&CSE_ECR_EC_MASK)
/* TRNG Bit Fields */
#define CSE_TRNG_TTM_MASK                        0x1u
#define CSE_TRNG_TTM_SHIFT                       0u
#define CSE_TRNG_TTM_WIDTH                       1u
#define CSE_TRNG_TTM(x)                          (((uint32_t)(((uint32_t)(x))<<CSE_TRNG_TTM_SHIFT))&CSE_TRNG_TTM_MASK)
#define CSE_TRNG_DLY_MASK                        0xFFFF0000u
#define CSE_TRNG_DLY_SHIFT                       16u
#define CSE_TRNG_DLY_WIDTH                       16u
#define CSE_TRNG_DLY(x)                          (((uint32_t)(((uint32_t)(x))<<CSE_TRNG_DLY_SHIFT))&CSE_TRNG_DLY_MASK)
/* CMD Bit Fields */
#define CSE_CMD_CMD_MASK                         0x1Fu
#define CSE_CMD_CMD_SHIFT                        0u
#define CSE_CMD_CMD_WIDTH                        5u
#define CSE_CMD_CMD(x)                           (((uint32_t)(((uint32_t)(x))<<CSE_CMD_CMD_SHIFT))&CSE_CMD_CMD_MASK)
/* P Bit Fields */
#define CSE_P_PARM_MASK                          0xFFFFFFFFu
#define CSE_P_PARM_SHIFT                         0u
#define CSE_P_PARM_WIDTH                         32u
#define CSE_P_PARM(x)                            (((uint32_t)(((uint32_t)(x))<<CSE_P_PARM_SHIFT))&CSE_P_PARM_MASK)
/* KIA Bit Fields */
#define CSE_KIA_KI_ADDR_MASK                     0xFFFFFFFFu
#define CSE_KIA_KI_ADDR_SHIFT                    0u
#define CSE_KIA_KI_ADDR_WIDTH                    32u
#define CSE_KIA_KI_ADDR(x)                       (((uint32_t)(((uint32_t)(x))<<CSE_KIA_KI_ADDR_SHIFT))&CSE_KIA_KI_ADDR_MASK)
/* OTP Bit Fields */
#define CSE_OTP_SEC_CNT_MASK                     0x3Fu
#define CSE_OTP_SEC_CNT_SHIFT                    0u
#define CSE_OTP_SEC_CNT_WIDTH                    6u
#define CSE_OTP_SEC_CNT(x)                       (((uint32_t)(((uint32_t)(x))<<CSE_OTP_SEC_CNT_SHIFT))&CSE_OTP_SEC_CNT_MASK)
/* PC Bit Fields */
#define CSE_PC_PUB_CNT_MASK                      0xFFFFFFFFu
#define CSE_PC_PUB_CNT_SHIFT                     0u
#define CSE_PC_PUB_CNT_WIDTH                     32u
#define CSE_PC_PUB_CNT(x)                        (((uint32_t)(((uint32_t)(x))<<CSE_PC_PUB_CNT_SHIFT))&CSE_PC_PUB_CNT_MASK)
/* SRA0 Bit Fields */
#define CSE_SRA0_SEC_RAM_ADDR_MASK               0xFFFFFFFFu
#define CSE_SRA0_SEC_RAM_ADDR_SHIFT              0u
#define CSE_SRA0_SEC_RAM_ADDR_WIDTH              32u
#define CSE_SRA0_SEC_RAM_ADDR(x)                 (((uint32_t)(((uint32_t)(x))<<CSE_SRA0_SEC_RAM_ADDR_SHIFT))&CSE_SRA0_SEC_RAM_ADDR_MASK)
/* SRS0 Bit Fields */
#define CSE_SRS0_SEC_RAM_SIZE_MASK               0xFFFFFFFFu
#define CSE_SRS0_SEC_RAM_SIZE_SHIFT              0u
#define CSE_SRS0_SEC_RAM_SIZE_WIDTH              32u
#define CSE_SRS0_SEC_RAM_SIZE(x)                 (((uint32_t)(((uint32_t)(x))<<CSE_SRS0_SEC_RAM_SIZE_SHIFT))&CSE_SRS0_SEC_RAM_SIZE_MASK)

/*!
 * @}
 */ /* end of group CSE_Register_Masks */


/*!
 * @}
 */ /* end of group CSE_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- DCU Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup DCU_Peripheral_Access_Layer DCU Peripheral Access Layer
 * @{
 */


/** DCU - Size of Registers Arrays */
#define DCU_LAYER_COUNT                          8u

/** DCU - Register Layout Typedef */
typedef struct {
  __IO uint32_t CTRLDESCCURSOR1;                   /**< Control Descriptor Cursor 1 Register, offset: 0x0 */
  __IO uint32_t CTRLDESCCURSOR2;                   /**< Control Descriptor Cursor 2 Register, offset: 0x4 */
  __IO uint32_t CTRLDESCCURSOR3;                   /**< Control Descriptor Cursor 3 Register, offset: 0x8 */
       uint8_t RESERVED_0[4];
  __IO uint32_t MODE;                              /**< Mode Register, offset: 0x10 */
  __IO uint32_t BGND;                              /**< Background Register, offset: 0x14 */
  __IO uint32_t DISP_SIZE;                         /**< Display Size Register, offset: 0x18 */
  __IO uint32_t HSYN_PARA;                         /**< Horizontal Sync Parameter Register, offset: 0x1C */
  __IO uint32_t VSYN_PARA;                         /**< Vertical Sync Parameter Register, offset: 0x20 */
  __IO uint32_t SYNPOL;                            /**< Synchronize Polarity Register, offset: 0x24 */
  __IO uint32_t THRESHOLD;                         /**< Threshold Register, offset: 0x28 */
  __IO uint32_t INT_STATUS;                        /**< Interrupt Status Register, offset: 0x2C */
  __IO uint32_t INT_MASK;                          /**< Interrupt Mask Register, offset: 0x30 */
  __IO uint32_t COLBAR_1;                          /**< COLBAR_1 Register, offset: 0x34 */
  __IO uint32_t COLBAR_2;                          /**< COLBAR_2 Register, offset: 0x38 */
  __IO uint32_t COLBAR_3;                          /**< COLBAR_3 Register, offset: 0x3C */
  __IO uint32_t COLBAR_4;                          /**< COLBAR_4 Register, offset: 0x40 */
  __IO uint32_t COLBAR_5;                          /**< COLBAR_5 Register, offset: 0x44 */
  __IO uint32_t COLBAR_6;                          /**< COLBAR_6 Register, offset: 0x48 */
  __IO uint32_t COLBAR_7;                          /**< COLBAR_7 Register, offset: 0x4C */
  __IO uint32_t COLBAR_8;                          /**< COLBAR_8 Register, offset: 0x50 */
  __IO uint32_t DIV_RATIO;                         /**< Divide Ratio Register, offset: 0x54 */
  __IO uint32_t SIGN_CALC_1;                       /**< Sign Calculation 1 Register, offset: 0x58 */
  __IO uint32_t SIGN_CALC_2;                       /**< Sign Calculation 2 Register, offset: 0x5C */
  __I  uint32_t CRC_VAL;                           /**< CRC Value Register, offset: 0x60 */
       uint8_t RESERVED_1[8];
  __IO uint32_t PARR_ERR_STATUS1;                  /**< Parameter Error Status 1 Register, offset: 0x6C */
       uint8_t RESERVED_2[12];
  __IO uint32_t PARR_ERR_STATUS3;                  /**< Parameter Error Status 3 Register, offset: 0x7C */
  __IO uint32_t MASK_PARR_ERR_STATUS1;             /**< Mask Parameter Error Status 1 Register, offset: 0x80 */
       uint8_t RESERVED_3[12];
  __IO uint32_t MASK_PARR_ERR_STATUS3;             /**< Mask Parameter Error Status 3 Register, offset: 0x90 */
  __IO uint32_t THRESHOLD_INP_BUF_1;               /**< Threshold Input 1 Register, offset: 0x94 */
       uint8_t RESERVED_4[8];
  __IO uint32_t LUMA_COMP;                         /**< LUMA Component Register, offset: 0xA0 */
  __IO uint32_t CHROMA_RED;                        /**< Red Chroma Components Register, offset: 0xA4 */
  __IO uint32_t CHROMA_GREEN;                      /**< Green Chroma Components Register, offset: 0xA8 */
  __IO uint32_t CHROMA_BLUE;                       /**< Blue Chroma Components Register, offset: 0xAC */
  __I  uint32_t CRC_POS;                           /**< CRC Position Register, offset: 0xB0 */
  __IO uint32_t LYR_INTPOL_EN;                     /**< Layer Interpolation Enable Register, offset: 0xB4 */
  __IO uint32_t LYR_LUMA_COMP;                     /**< Layer Luminance Component Register, offset: 0xB8 */
  __IO uint32_t LYR_CHRM_RED;                      /**< Layer Chroma Red Register, offset: 0xBC */
  __IO uint32_t LYR_CHRM_GRN;                      /**< Layer Chroma Green Register, offset: 0xC0 */
  __IO uint32_t LYR_CHRM_BLUE;                     /**< Layer Chroma Blue Register, offset: 0xC4 */
       uint8_t RESERVED_5[4];
  __IO uint32_t UPDATE_MODE;                       /**< Update Mode Register, offset: 0xCC */
  __I  uint32_t UNDERRUN;                          /**< Underrun Register, offset: 0xD0 */
       uint8_t RESERVED_6[12];
  __IO uint32_t FRM_CRC_CTRL;                      /**< Frame CRC Control, offset: 0xE0 */
  __I  uint32_t FRM_CRC_VAL;                       /**< Frame CRC Value, offset: 0xE4 */
  __IO uint32_t TX_ESCAL_LVL;                      /**< QoS Level, offset: 0xE8 */
       uint8_t RESERVED_7[20];
  __IO uint32_t GPR;                               /**< Global Protection Register, offset: 0x100 */
  __IO uint32_t SLR_L0;                            /**< Soft Lock Bit Layer 0 Register, offset: 0x104 */
  __IO uint32_t SLR_L1;                            /**< Soft Lock Bit Layer 1 Register, offset: 0x108 */
  __IO uint32_t SLR_DISP_SIZE;                     /**< Soft Lock Display Size Register, offset: 0x10C */
  __IO uint32_t SLR_HVSYNC_PARA;                   /**< Soft Lock Hsync/Vsync Parameter Register, offset: 0x110 */
  __IO uint32_t SLR_POL;                           /**< Soft Lock POL Register, offset: 0x114 */
  __IO uint32_t SLR_L0_TRANSP;                     /**< Soft Lock L0 Transparency Register, offset: 0x118 */
  __IO uint32_t SLR_L1_TRANSP;                     /**< Soft Lock L1 Transparency Register, offset: 0x11C */
       uint8_t RESERVED_8[224];
  struct {                                         /* offset: 0x200, array step: 0x40 */
    __IO uint32_t CTRLDESCL_1;                       /**< Control Descriptor Layer 1 Register, array offset: 0x200, array step: 0x40 */
    __IO uint32_t CTRLDESCL_2;                       /**< Control Descriptor Layer 2 Register, array offset: 0x204, array step: 0x40 */
    __IO uint32_t CTRLDESCL_3;                       /**< Control Descriptor Layer 3 Register, array offset: 0x208, array step: 0x40 */
    __IO uint32_t CTRLDESCL_4;                       /**< Control Descriptor Layer 4 Register, array offset: 0x20C, array step: 0x40 */
    __IO uint32_t CTRLDESCL_5;                       /**< Control Descriptor Layer 5 Register, array offset: 0x210, array step: 0x40 */
    __IO uint32_t CTRLDESCL_6;                       /**< Control Descriptor Layer 6 Register, array offset: 0x214, array step: 0x40 */
         uint8_t RESERVED_0[4];
    __IO uint32_t CTRLDESCL_8;                       /**< Control Descriptor Layer 8 Register, array offset: 0x21C, array step: 0x40 */
    __IO uint32_t CTRLDESCL_9;                       /**< Control Descriptor Layer 9 Register, array offset: 0x220, array step: 0x40 */
    __IO uint32_t CTRLDESCL_10;                      /**< Control Descriptor Layer 10 Register, array offset: 0x224, array step: 0x40 */
         uint8_t RESERVED_1[24];
  } LAYER[DCU_LAYER_COUNT];
} DCU_Type, *DCU_MemMapPtr;

 /** Number of instances of the DCU module. */
#define DCU_INSTANCE_COUNT                       (1u)


/* DCU - Peripheral instance base addresses */
/** Peripheral DCU base address */
#define DCU_BASE                                 (0x40028000u)
/** Peripheral DCU base pointer */
#define DCU                                      ((DCU_Type *)DCU_BASE)
/** Array initializer of DCU peripheral base addresses */
#define DCU_BASE_ADDRS                           { DCU_BASE }
/** Array initializer of DCU peripheral base pointers */
#define DCU_BASE_PTRS                            { DCU }

/* ----------------------------------------------------------------------------
   -- DCU Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup DCU_Register_Masks DCU Register Masks
 * @{
 */

/* CTRLDESCCURSOR1 Bit Fields */
#define DCU_CTRLDESCCURSOR1_WIDTH_MASK           0x7FFu
#define DCU_CTRLDESCCURSOR1_WIDTH_SHIFT          0u
#define DCU_CTRLDESCCURSOR1_WIDTH_WIDTH          11u
#define DCU_CTRLDESCCURSOR1_WIDTH(x)             (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCCURSOR1_WIDTH_SHIFT))&DCU_CTRLDESCCURSOR1_WIDTH_MASK)
#define DCU_CTRLDESCCURSOR1_HEIGHT_MASK          0x7FF0000u
#define DCU_CTRLDESCCURSOR1_HEIGHT_SHIFT         16u
#define DCU_CTRLDESCCURSOR1_HEIGHT_WIDTH         11u
#define DCU_CTRLDESCCURSOR1_HEIGHT(x)            (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCCURSOR1_HEIGHT_SHIFT))&DCU_CTRLDESCCURSOR1_HEIGHT_MASK)
/* CTRLDESCCURSOR2 Bit Fields */
#define DCU_CTRLDESCCURSOR2_POSX_MASK            0x7FFu
#define DCU_CTRLDESCCURSOR2_POSX_SHIFT           0u
#define DCU_CTRLDESCCURSOR2_POSX_WIDTH           11u
#define DCU_CTRLDESCCURSOR2_POSX(x)              (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCCURSOR2_POSX_SHIFT))&DCU_CTRLDESCCURSOR2_POSX_MASK)
#define DCU_CTRLDESCCURSOR2_POSY_MASK            0x7FF0000u
#define DCU_CTRLDESCCURSOR2_POSY_SHIFT           16u
#define DCU_CTRLDESCCURSOR2_POSY_WIDTH           11u
#define DCU_CTRLDESCCURSOR2_POSY(x)              (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCCURSOR2_POSY_SHIFT))&DCU_CTRLDESCCURSOR2_POSY_MASK)
/* CTRLDESCCURSOR3 Bit Fields */
#define DCU_CTRLDESCCURSOR3_DEFAULT_CURSOR_COLOR_MASK 0xFFFFFFu
#define DCU_CTRLDESCCURSOR3_DEFAULT_CURSOR_COLOR_SHIFT 0u
#define DCU_CTRLDESCCURSOR3_DEFAULT_CURSOR_COLOR_WIDTH 24u
#define DCU_CTRLDESCCURSOR3_DEFAULT_CURSOR_COLOR(x) (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCCURSOR3_DEFAULT_CURSOR_COLOR_SHIFT))&DCU_CTRLDESCCURSOR3_DEFAULT_CURSOR_COLOR_MASK)
#define DCU_CTRLDESCCURSOR3_CUR_EN_MASK          0x80000000u
#define DCU_CTRLDESCCURSOR3_CUR_EN_SHIFT         31u
#define DCU_CTRLDESCCURSOR3_CUR_EN_WIDTH         1u
#define DCU_CTRLDESCCURSOR3_CUR_EN(x)            (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCCURSOR3_CUR_EN_SHIFT))&DCU_CTRLDESCCURSOR3_CUR_EN_MASK)
/* MODE Bit Fields */
#define DCU_MODE_DCU_MODE_MASK                   0x3u
#define DCU_MODE_DCU_MODE_SHIFT                  0u
#define DCU_MODE_DCU_MODE_WIDTH                  2u
#define DCU_MODE_DCU_MODE(x)                     (((uint32_t)(((uint32_t)(x))<<DCU_MODE_DCU_MODE_SHIFT))&DCU_MODE_DCU_MODE_MASK)
#define DCU_MODE_EN_GAMMA_MASK                   0x4u
#define DCU_MODE_EN_GAMMA_SHIFT                  2u
#define DCU_MODE_EN_GAMMA_WIDTH                  1u
#define DCU_MODE_EN_GAMMA(x)                     (((uint32_t)(((uint32_t)(x))<<DCU_MODE_EN_GAMMA_SHIFT))&DCU_MODE_EN_GAMMA_MASK)
#define DCU_MODE_SIG_EN_MASK                     0x20u
#define DCU_MODE_SIG_EN_SHIFT                    5u
#define DCU_MODE_SIG_EN_WIDTH                    1u
#define DCU_MODE_SIG_EN(x)                       (((uint32_t)(((uint32_t)(x))<<DCU_MODE_SIG_EN_SHIFT))&DCU_MODE_SIG_EN_MASK)
#define DCU_MODE_TAG_EN_MASK                     0x40u
#define DCU_MODE_TAG_EN_SHIFT                    6u
#define DCU_MODE_TAG_EN_WIDTH                    1u
#define DCU_MODE_TAG_EN(x)                       (((uint32_t)(((uint32_t)(x))<<DCU_MODE_TAG_EN_SHIFT))&DCU_MODE_TAG_EN_MASK)
#define DCU_MODE_RASTER_EN_MASK                  0x4000u
#define DCU_MODE_RASTER_EN_SHIFT                 14u
#define DCU_MODE_RASTER_EN_WIDTH                 1u
#define DCU_MODE_RASTER_EN(x)                    (((uint32_t)(((uint32_t)(x))<<DCU_MODE_RASTER_EN_SHIFT))&DCU_MODE_RASTER_EN_MASK)
#define DCU_MODE_BLEND_ITER_MASK                 0x700000u
#define DCU_MODE_BLEND_ITER_SHIFT                20u
#define DCU_MODE_BLEND_ITER_WIDTH                3u
#define DCU_MODE_BLEND_ITER(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_MODE_BLEND_ITER_SHIFT))&DCU_MODE_BLEND_ITER_MASK)
#define DCU_MODE_ADDR_MASK                       0x3000000u
#define DCU_MODE_ADDR_SHIFT                      24u
#define DCU_MODE_ADDR_WIDTH                      2u
#define DCU_MODE_ADDR(x)                         (((uint32_t)(((uint32_t)(x))<<DCU_MODE_ADDR_SHIFT))&DCU_MODE_ADDR_MASK)
#define DCU_MODE_ADDG_MASK                       0xC000000u
#define DCU_MODE_ADDG_SHIFT                      26u
#define DCU_MODE_ADDG_WIDTH                      2u
#define DCU_MODE_ADDG(x)                         (((uint32_t)(((uint32_t)(x))<<DCU_MODE_ADDG_SHIFT))&DCU_MODE_ADDG_MASK)
#define DCU_MODE_ADDB_MASK                       0x30000000u
#define DCU_MODE_ADDB_SHIFT                      28u
#define DCU_MODE_ADDB_WIDTH                      2u
#define DCU_MODE_ADDB(x)                         (((uint32_t)(((uint32_t)(x))<<DCU_MODE_ADDB_SHIFT))&DCU_MODE_ADDB_MASK)
#define DCU_MODE_EN_DITHER_MASK                  0x40000000u
#define DCU_MODE_EN_DITHER_SHIFT                 30u
#define DCU_MODE_EN_DITHER_WIDTH                 1u
#define DCU_MODE_EN_DITHER(x)                    (((uint32_t)(((uint32_t)(x))<<DCU_MODE_EN_DITHER_SHIFT))&DCU_MODE_EN_DITHER_MASK)
#define DCU_MODE_DCU_SW_RESET_MASK               0x80000000u
#define DCU_MODE_DCU_SW_RESET_SHIFT              31u
#define DCU_MODE_DCU_SW_RESET_WIDTH              1u
#define DCU_MODE_DCU_SW_RESET(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_MODE_DCU_SW_RESET_SHIFT))&DCU_MODE_DCU_SW_RESET_MASK)
/* BGND Bit Fields */
#define DCU_BGND_BGND_B_MASK                     0xFFu
#define DCU_BGND_BGND_B_SHIFT                    0u
#define DCU_BGND_BGND_B_WIDTH                    8u
#define DCU_BGND_BGND_B(x)                       (((uint32_t)(((uint32_t)(x))<<DCU_BGND_BGND_B_SHIFT))&DCU_BGND_BGND_B_MASK)
#define DCU_BGND_BGND_G_MASK                     0xFF00u
#define DCU_BGND_BGND_G_SHIFT                    8u
#define DCU_BGND_BGND_G_WIDTH                    8u
#define DCU_BGND_BGND_G(x)                       (((uint32_t)(((uint32_t)(x))<<DCU_BGND_BGND_G_SHIFT))&DCU_BGND_BGND_G_MASK)
#define DCU_BGND_BGND_R_MASK                     0xFF0000u
#define DCU_BGND_BGND_R_SHIFT                    16u
#define DCU_BGND_BGND_R_WIDTH                    8u
#define DCU_BGND_BGND_R(x)                       (((uint32_t)(((uint32_t)(x))<<DCU_BGND_BGND_R_SHIFT))&DCU_BGND_BGND_R_MASK)
/* DISP_SIZE Bit Fields */
#define DCU_DISP_SIZE_DELTA_X_MASK               0xFFu
#define DCU_DISP_SIZE_DELTA_X_SHIFT              0u
#define DCU_DISP_SIZE_DELTA_X_WIDTH              8u
#define DCU_DISP_SIZE_DELTA_X(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_DISP_SIZE_DELTA_X_SHIFT))&DCU_DISP_SIZE_DELTA_X_MASK)
#define DCU_DISP_SIZE_DELTA_Y_MASK               0xFFF0000u
#define DCU_DISP_SIZE_DELTA_Y_SHIFT              16u
#define DCU_DISP_SIZE_DELTA_Y_WIDTH              12u
#define DCU_DISP_SIZE_DELTA_Y(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_DISP_SIZE_DELTA_Y_SHIFT))&DCU_DISP_SIZE_DELTA_Y_MASK)
/* HSYN_PARA Bit Fields */
#define DCU_HSYN_PARA_FP_H_MASK                  0x1FFu
#define DCU_HSYN_PARA_FP_H_SHIFT                 0u
#define DCU_HSYN_PARA_FP_H_WIDTH                 9u
#define DCU_HSYN_PARA_FP_H(x)                    (((uint32_t)(((uint32_t)(x))<<DCU_HSYN_PARA_FP_H_SHIFT))&DCU_HSYN_PARA_FP_H_MASK)
#define DCU_HSYN_PARA_PW_H_MASK                  0xFF800u
#define DCU_HSYN_PARA_PW_H_SHIFT                 11u
#define DCU_HSYN_PARA_PW_H_WIDTH                 9u
#define DCU_HSYN_PARA_PW_H(x)                    (((uint32_t)(((uint32_t)(x))<<DCU_HSYN_PARA_PW_H_SHIFT))&DCU_HSYN_PARA_PW_H_MASK)
#define DCU_HSYN_PARA_BP_H_MASK                  0x7FC00000u
#define DCU_HSYN_PARA_BP_H_SHIFT                 22u
#define DCU_HSYN_PARA_BP_H_WIDTH                 9u
#define DCU_HSYN_PARA_BP_H(x)                    (((uint32_t)(((uint32_t)(x))<<DCU_HSYN_PARA_BP_H_SHIFT))&DCU_HSYN_PARA_BP_H_MASK)
/* VSYN_PARA Bit Fields */
#define DCU_VSYN_PARA_FP_V_MASK                  0x1FFu
#define DCU_VSYN_PARA_FP_V_SHIFT                 0u
#define DCU_VSYN_PARA_FP_V_WIDTH                 9u
#define DCU_VSYN_PARA_FP_V(x)                    (((uint32_t)(((uint32_t)(x))<<DCU_VSYN_PARA_FP_V_SHIFT))&DCU_VSYN_PARA_FP_V_MASK)
#define DCU_VSYN_PARA_PW_V_MASK                  0xFF800u
#define DCU_VSYN_PARA_PW_V_SHIFT                 11u
#define DCU_VSYN_PARA_PW_V_WIDTH                 9u
#define DCU_VSYN_PARA_PW_V(x)                    (((uint32_t)(((uint32_t)(x))<<DCU_VSYN_PARA_PW_V_SHIFT))&DCU_VSYN_PARA_PW_V_MASK)
#define DCU_VSYN_PARA_BP_V_MASK                  0x7FC00000u
#define DCU_VSYN_PARA_BP_V_SHIFT                 22u
#define DCU_VSYN_PARA_BP_V_WIDTH                 9u
#define DCU_VSYN_PARA_BP_V(x)                    (((uint32_t)(((uint32_t)(x))<<DCU_VSYN_PARA_BP_V_SHIFT))&DCU_VSYN_PARA_BP_V_MASK)
/* SYNPOL Bit Fields */
#define DCU_SYNPOL_INV_HS_MASK                   0x1u
#define DCU_SYNPOL_INV_HS_SHIFT                  0u
#define DCU_SYNPOL_INV_HS_WIDTH                  1u
#define DCU_SYNPOL_INV_HS(x)                     (((uint32_t)(((uint32_t)(x))<<DCU_SYNPOL_INV_HS_SHIFT))&DCU_SYNPOL_INV_HS_MASK)
#define DCU_SYNPOL_INV_VS_MASK                   0x2u
#define DCU_SYNPOL_INV_VS_SHIFT                  1u
#define DCU_SYNPOL_INV_VS_WIDTH                  1u
#define DCU_SYNPOL_INV_VS(x)                     (((uint32_t)(((uint32_t)(x))<<DCU_SYNPOL_INV_VS_SHIFT))&DCU_SYNPOL_INV_VS_MASK)
#define DCU_SYNPOL_NEG_MASK                      0x20u
#define DCU_SYNPOL_NEG_SHIFT                     5u
#define DCU_SYNPOL_NEG_WIDTH                     1u
#define DCU_SYNPOL_NEG(x)                        (((uint32_t)(((uint32_t)(x))<<DCU_SYNPOL_NEG_SHIFT))&DCU_SYNPOL_NEG_MASK)
#define DCU_SYNPOL_INV_PXCK_MASK                 0x40u
#define DCU_SYNPOL_INV_PXCK_SHIFT                6u
#define DCU_SYNPOL_INV_PXCK_WIDTH                1u
#define DCU_SYNPOL_INV_PXCK(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SYNPOL_INV_PXCK_SHIFT))&DCU_SYNPOL_INV_PXCK_MASK)
#define DCU_SYNPOL_INV_DE_MASK                   0x800u
#define DCU_SYNPOL_INV_DE_SHIFT                  11u
#define DCU_SYNPOL_INV_DE_WIDTH                  1u
#define DCU_SYNPOL_INV_DE(x)                     (((uint32_t)(((uint32_t)(x))<<DCU_SYNPOL_INV_DE_SHIFT))&DCU_SYNPOL_INV_DE_MASK)
/* THRESHOLD Bit Fields */
#define DCU_THRESHOLD_OUT_BUF_LOW_MASK           0xFFu
#define DCU_THRESHOLD_OUT_BUF_LOW_SHIFT          0u
#define DCU_THRESHOLD_OUT_BUF_LOW_WIDTH          8u
#define DCU_THRESHOLD_OUT_BUF_LOW(x)             (((uint32_t)(((uint32_t)(x))<<DCU_THRESHOLD_OUT_BUF_LOW_SHIFT))&DCU_THRESHOLD_OUT_BUF_LOW_MASK)
#define DCU_THRESHOLD_OUT_BUF_HIGH_MASK          0xFF00u
#define DCU_THRESHOLD_OUT_BUF_HIGH_SHIFT         8u
#define DCU_THRESHOLD_OUT_BUF_HIGH_WIDTH         8u
#define DCU_THRESHOLD_OUT_BUF_HIGH(x)            (((uint32_t)(((uint32_t)(x))<<DCU_THRESHOLD_OUT_BUF_HIGH_SHIFT))&DCU_THRESHOLD_OUT_BUF_HIGH_MASK)
#define DCU_THRESHOLD_LS_BF_VS_MASK              0x7FF0000u
#define DCU_THRESHOLD_LS_BF_VS_SHIFT             16u
#define DCU_THRESHOLD_LS_BF_VS_WIDTH             11u
#define DCU_THRESHOLD_LS_BF_VS(x)                (((uint32_t)(((uint32_t)(x))<<DCU_THRESHOLD_LS_BF_VS_SHIFT))&DCU_THRESHOLD_LS_BF_VS_MASK)
/* INT_STATUS Bit Fields */
#define DCU_INT_STATUS_VSYNC_MASK                0x1u
#define DCU_INT_STATUS_VSYNC_SHIFT               0u
#define DCU_INT_STATUS_VSYNC_WIDTH               1u
#define DCU_INT_STATUS_VSYNC(x)                  (((uint32_t)(((uint32_t)(x))<<DCU_INT_STATUS_VSYNC_SHIFT))&DCU_INT_STATUS_VSYNC_MASK)
#define DCU_INT_STATUS_UNDRUN_MASK               0x2u
#define DCU_INT_STATUS_UNDRUN_SHIFT              1u
#define DCU_INT_STATUS_UNDRUN_WIDTH              1u
#define DCU_INT_STATUS_UNDRUN(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_INT_STATUS_UNDRUN_SHIFT))&DCU_INT_STATUS_UNDRUN_MASK)
#define DCU_INT_STATUS_LS_BF_VS_MASK             0x4u
#define DCU_INT_STATUS_LS_BF_VS_SHIFT            2u
#define DCU_INT_STATUS_LS_BF_VS_WIDTH            1u
#define DCU_INT_STATUS_LS_BF_VS(x)               (((uint32_t)(((uint32_t)(x))<<DCU_INT_STATUS_LS_BF_VS_SHIFT))&DCU_INT_STATUS_LS_BF_VS_MASK)
#define DCU_INT_STATUS_VS_BLANK_MASK             0x8u
#define DCU_INT_STATUS_VS_BLANK_SHIFT            3u
#define DCU_INT_STATUS_VS_BLANK_WIDTH            1u
#define DCU_INT_STATUS_VS_BLANK(x)               (((uint32_t)(((uint32_t)(x))<<DCU_INT_STATUS_VS_BLANK_SHIFT))&DCU_INT_STATUS_VS_BLANK_MASK)
#define DCU_INT_STATUS_CRC_READY_MASK            0x10u
#define DCU_INT_STATUS_CRC_READY_SHIFT           4u
#define DCU_INT_STATUS_CRC_READY_WIDTH           1u
#define DCU_INT_STATUS_CRC_READY(x)              (((uint32_t)(((uint32_t)(x))<<DCU_INT_STATUS_CRC_READY_SHIFT))&DCU_INT_STATUS_CRC_READY_MASK)
#define DCU_INT_STATUS_CRC_OVERFLOW_MASK         0x20u
#define DCU_INT_STATUS_CRC_OVERFLOW_SHIFT        5u
#define DCU_INT_STATUS_CRC_OVERFLOW_WIDTH        1u
#define DCU_INT_STATUS_CRC_OVERFLOW(x)           (((uint32_t)(((uint32_t)(x))<<DCU_INT_STATUS_CRC_OVERFLOW_SHIFT))&DCU_INT_STATUS_CRC_OVERFLOW_MASK)
#define DCU_INT_STATUS_P1_FIFO_LO_FLAG_MASK      0x40u
#define DCU_INT_STATUS_P1_FIFO_LO_FLAG_SHIFT     6u
#define DCU_INT_STATUS_P1_FIFO_LO_FLAG_WIDTH     1u
#define DCU_INT_STATUS_P1_FIFO_LO_FLAG(x)        (((uint32_t)(((uint32_t)(x))<<DCU_INT_STATUS_P1_FIFO_LO_FLAG_SHIFT))&DCU_INT_STATUS_P1_FIFO_LO_FLAG_MASK)
#define DCU_INT_STATUS_P1_FIFO_HI_FLAG_MASK      0x80u
#define DCU_INT_STATUS_P1_FIFO_HI_FLAG_SHIFT     7u
#define DCU_INT_STATUS_P1_FIFO_HI_FLAG_WIDTH     1u
#define DCU_INT_STATUS_P1_FIFO_HI_FLAG(x)        (((uint32_t)(((uint32_t)(x))<<DCU_INT_STATUS_P1_FIFO_HI_FLAG_SHIFT))&DCU_INT_STATUS_P1_FIFO_HI_FLAG_MASK)
#define DCU_INT_STATUS_P2_FIFO_LO_FLAG_MASK      0x100u
#define DCU_INT_STATUS_P2_FIFO_LO_FLAG_SHIFT     8u
#define DCU_INT_STATUS_P2_FIFO_LO_FLAG_WIDTH     1u
#define DCU_INT_STATUS_P2_FIFO_LO_FLAG(x)        (((uint32_t)(((uint32_t)(x))<<DCU_INT_STATUS_P2_FIFO_LO_FLAG_SHIFT))&DCU_INT_STATUS_P2_FIFO_LO_FLAG_MASK)
#define DCU_INT_STATUS_P2_FIFO_HI_FLAG_MASK      0x200u
#define DCU_INT_STATUS_P2_FIFO_HI_FLAG_SHIFT     9u
#define DCU_INT_STATUS_P2_FIFO_HI_FLAG_WIDTH     1u
#define DCU_INT_STATUS_P2_FIFO_HI_FLAG(x)        (((uint32_t)(((uint32_t)(x))<<DCU_INT_STATUS_P2_FIFO_HI_FLAG_SHIFT))&DCU_INT_STATUS_P2_FIFO_HI_FLAG_MASK)
#define DCU_INT_STATUS_PROG_END_MASK             0x400u
#define DCU_INT_STATUS_PROG_END_SHIFT            10u
#define DCU_INT_STATUS_PROG_END_WIDTH            1u
#define DCU_INT_STATUS_PROG_END(x)               (((uint32_t)(((uint32_t)(x))<<DCU_INT_STATUS_PROG_END_SHIFT))&DCU_INT_STATUS_PROG_END_MASK)
#define DCU_INT_STATUS_IPM_ERROR_MASK            0x800u
#define DCU_INT_STATUS_IPM_ERROR_SHIFT           11u
#define DCU_INT_STATUS_IPM_ERROR_WIDTH           1u
#define DCU_INT_STATUS_IPM_ERROR(x)              (((uint32_t)(((uint32_t)(x))<<DCU_INT_STATUS_IPM_ERROR_SHIFT))&DCU_INT_STATUS_IPM_ERROR_MASK)
#define DCU_INT_STATUS_LYR_TRANS_FINISH_MASK     0x1000u
#define DCU_INT_STATUS_LYR_TRANS_FINISH_SHIFT    12u
#define DCU_INT_STATUS_LYR_TRANS_FINISH_WIDTH    1u
#define DCU_INT_STATUS_LYR_TRANS_FINISH(x)       (((uint32_t)(((uint32_t)(x))<<DCU_INT_STATUS_LYR_TRANS_FINISH_SHIFT))&DCU_INT_STATUS_LYR_TRANS_FINISH_MASK)
#define DCU_INT_STATUS_DMA_TRANS_FINISH_MASK     0x4000u
#define DCU_INT_STATUS_DMA_TRANS_FINISH_SHIFT    14u
#define DCU_INT_STATUS_DMA_TRANS_FINISH_WIDTH    1u
#define DCU_INT_STATUS_DMA_TRANS_FINISH(x)       (((uint32_t)(((uint32_t)(x))<<DCU_INT_STATUS_DMA_TRANS_FINISH_SHIFT))&DCU_INT_STATUS_DMA_TRANS_FINISH_MASK)
#define DCU_INT_STATUS_P1_EMPTY_MASK             0x4000000u
#define DCU_INT_STATUS_P1_EMPTY_SHIFT            26u
#define DCU_INT_STATUS_P1_EMPTY_WIDTH            1u
#define DCU_INT_STATUS_P1_EMPTY(x)               (((uint32_t)(((uint32_t)(x))<<DCU_INT_STATUS_P1_EMPTY_SHIFT))&DCU_INT_STATUS_P1_EMPTY_MASK)
#define DCU_INT_STATUS_P2_EMPTY_MASK             0x8000000u
#define DCU_INT_STATUS_P2_EMPTY_SHIFT            27u
#define DCU_INT_STATUS_P2_EMPTY_WIDTH            1u
#define DCU_INT_STATUS_P2_EMPTY(x)               (((uint32_t)(((uint32_t)(x))<<DCU_INT_STATUS_P2_EMPTY_SHIFT))&DCU_INT_STATUS_P2_EMPTY_MASK)
/* INT_MASK Bit Fields */
#define DCU_INT_MASK_M_VSYNC_MASK                0x1u
#define DCU_INT_MASK_M_VSYNC_SHIFT               0u
#define DCU_INT_MASK_M_VSYNC_WIDTH               1u
#define DCU_INT_MASK_M_VSYNC(x)                  (((uint32_t)(((uint32_t)(x))<<DCU_INT_MASK_M_VSYNC_SHIFT))&DCU_INT_MASK_M_VSYNC_MASK)
#define DCU_INT_MASK_M_UNDRUN_MASK               0x2u
#define DCU_INT_MASK_M_UNDRUN_SHIFT              1u
#define DCU_INT_MASK_M_UNDRUN_WIDTH              1u
#define DCU_INT_MASK_M_UNDRUN(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_INT_MASK_M_UNDRUN_SHIFT))&DCU_INT_MASK_M_UNDRUN_MASK)
#define DCU_INT_MASK_M_LS_BF_VS_MASK             0x4u
#define DCU_INT_MASK_M_LS_BF_VS_SHIFT            2u
#define DCU_INT_MASK_M_LS_BF_VS_WIDTH            1u
#define DCU_INT_MASK_M_LS_BF_VS(x)               (((uint32_t)(((uint32_t)(x))<<DCU_INT_MASK_M_LS_BF_VS_SHIFT))&DCU_INT_MASK_M_LS_BF_VS_MASK)
#define DCU_INT_MASK_M_VS_BLANK_MASK             0x8u
#define DCU_INT_MASK_M_VS_BLANK_SHIFT            3u
#define DCU_INT_MASK_M_VS_BLANK_WIDTH            1u
#define DCU_INT_MASK_M_VS_BLANK(x)               (((uint32_t)(((uint32_t)(x))<<DCU_INT_MASK_M_VS_BLANK_SHIFT))&DCU_INT_MASK_M_VS_BLANK_MASK)
#define DCU_INT_MASK_M_CRC_READY_MASK            0x10u
#define DCU_INT_MASK_M_CRC_READY_SHIFT           4u
#define DCU_INT_MASK_M_CRC_READY_WIDTH           1u
#define DCU_INT_MASK_M_CRC_READY(x)              (((uint32_t)(((uint32_t)(x))<<DCU_INT_MASK_M_CRC_READY_SHIFT))&DCU_INT_MASK_M_CRC_READY_MASK)
#define DCU_INT_MASK_M_CRC_OVERFLOW_MASK         0x20u
#define DCU_INT_MASK_M_CRC_OVERFLOW_SHIFT        5u
#define DCU_INT_MASK_M_CRC_OVERFLOW_WIDTH        1u
#define DCU_INT_MASK_M_CRC_OVERFLOW(x)           (((uint32_t)(((uint32_t)(x))<<DCU_INT_MASK_M_CRC_OVERFLOW_SHIFT))&DCU_INT_MASK_M_CRC_OVERFLOW_MASK)
#define DCU_INT_MASK_M_P1_FIFO_LO_FLAG_MASK      0x40u
#define DCU_INT_MASK_M_P1_FIFO_LO_FLAG_SHIFT     6u
#define DCU_INT_MASK_M_P1_FIFO_LO_FLAG_WIDTH     1u
#define DCU_INT_MASK_M_P1_FIFO_LO_FLAG(x)        (((uint32_t)(((uint32_t)(x))<<DCU_INT_MASK_M_P1_FIFO_LO_FLAG_SHIFT))&DCU_INT_MASK_M_P1_FIFO_LO_FLAG_MASK)
#define DCU_INT_MASK_M_P1_FIFO_HI_FLAG_MASK      0x80u
#define DCU_INT_MASK_M_P1_FIFO_HI_FLAG_SHIFT     7u
#define DCU_INT_MASK_M_P1_FIFO_HI_FLAG_WIDTH     1u
#define DCU_INT_MASK_M_P1_FIFO_HI_FLAG(x)        (((uint32_t)(((uint32_t)(x))<<DCU_INT_MASK_M_P1_FIFO_HI_FLAG_SHIFT))&DCU_INT_MASK_M_P1_FIFO_HI_FLAG_MASK)
#define DCU_INT_MASK_M_P2_FIFO_LO_FLAG_MASK      0x100u
#define DCU_INT_MASK_M_P2_FIFO_LO_FLAG_SHIFT     8u
#define DCU_INT_MASK_M_P2_FIFO_LO_FLAG_WIDTH     1u
#define DCU_INT_MASK_M_P2_FIFO_LO_FLAG(x)        (((uint32_t)(((uint32_t)(x))<<DCU_INT_MASK_M_P2_FIFO_LO_FLAG_SHIFT))&DCU_INT_MASK_M_P2_FIFO_LO_FLAG_MASK)
#define DCU_INT_MASK_M_P2_FIFO_HI_FLAG_MASK      0x200u
#define DCU_INT_MASK_M_P2_FIFO_HI_FLAG_SHIFT     9u
#define DCU_INT_MASK_M_P2_FIFO_HI_FLAG_WIDTH     1u
#define DCU_INT_MASK_M_P2_FIFO_HI_FLAG(x)        (((uint32_t)(((uint32_t)(x))<<DCU_INT_MASK_M_P2_FIFO_HI_FLAG_SHIFT))&DCU_INT_MASK_M_P2_FIFO_HI_FLAG_MASK)
#define DCU_INT_MASK_M_PROG_END_MASK             0x400u
#define DCU_INT_MASK_M_PROG_END_SHIFT            10u
#define DCU_INT_MASK_M_PROG_END_WIDTH            1u
#define DCU_INT_MASK_M_PROG_END(x)               (((uint32_t)(((uint32_t)(x))<<DCU_INT_MASK_M_PROG_END_SHIFT))&DCU_INT_MASK_M_PROG_END_MASK)
#define DCU_INT_MASK_M_IPM_ERROR_MASK            0x800u
#define DCU_INT_MASK_M_IPM_ERROR_SHIFT           11u
#define DCU_INT_MASK_M_IPM_ERROR_WIDTH           1u
#define DCU_INT_MASK_M_IPM_ERROR(x)              (((uint32_t)(((uint32_t)(x))<<DCU_INT_MASK_M_IPM_ERROR_SHIFT))&DCU_INT_MASK_M_IPM_ERROR_MASK)
#define DCU_INT_MASK_M_LYR_TRANS_FINISH_MASK     0x1000u
#define DCU_INT_MASK_M_LYR_TRANS_FINISH_SHIFT    12u
#define DCU_INT_MASK_M_LYR_TRANS_FINISH_WIDTH    1u
#define DCU_INT_MASK_M_LYR_TRANS_FINISH(x)       (((uint32_t)(((uint32_t)(x))<<DCU_INT_MASK_M_LYR_TRANS_FINISH_SHIFT))&DCU_INT_MASK_M_LYR_TRANS_FINISH_MASK)
#define DCU_INT_MASK_M_DMA_TRANS_FINISH_MASK     0x4000u
#define DCU_INT_MASK_M_DMA_TRANS_FINISH_SHIFT    14u
#define DCU_INT_MASK_M_DMA_TRANS_FINISH_WIDTH    1u
#define DCU_INT_MASK_M_DMA_TRANS_FINISH(x)       (((uint32_t)(((uint32_t)(x))<<DCU_INT_MASK_M_DMA_TRANS_FINISH_SHIFT))&DCU_INT_MASK_M_DMA_TRANS_FINISH_MASK)
#define DCU_INT_MASK_M_P1_EMPTY_MASK             0x4000000u
#define DCU_INT_MASK_M_P1_EMPTY_SHIFT            26u
#define DCU_INT_MASK_M_P1_EMPTY_WIDTH            1u
#define DCU_INT_MASK_M_P1_EMPTY(x)               (((uint32_t)(((uint32_t)(x))<<DCU_INT_MASK_M_P1_EMPTY_SHIFT))&DCU_INT_MASK_M_P1_EMPTY_MASK)
#define DCU_INT_MASK_M_P2_EMPTY_MASK             0x8000000u
#define DCU_INT_MASK_M_P2_EMPTY_SHIFT            27u
#define DCU_INT_MASK_M_P2_EMPTY_WIDTH            1u
#define DCU_INT_MASK_M_P2_EMPTY(x)               (((uint32_t)(((uint32_t)(x))<<DCU_INT_MASK_M_P2_EMPTY_SHIFT))&DCU_INT_MASK_M_P2_EMPTY_MASK)
/* COLBAR_1 Bit Fields */
#define DCU_COLBAR_1_COLBAR_B_MASK               0xFFu
#define DCU_COLBAR_1_COLBAR_B_SHIFT              0u
#define DCU_COLBAR_1_COLBAR_B_WIDTH              8u
#define DCU_COLBAR_1_COLBAR_B(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_1_COLBAR_B_SHIFT))&DCU_COLBAR_1_COLBAR_B_MASK)
#define DCU_COLBAR_1_COLBAR_G_MASK               0xFF00u
#define DCU_COLBAR_1_COLBAR_G_SHIFT              8u
#define DCU_COLBAR_1_COLBAR_G_WIDTH              8u
#define DCU_COLBAR_1_COLBAR_G(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_1_COLBAR_G_SHIFT))&DCU_COLBAR_1_COLBAR_G_MASK)
#define DCU_COLBAR_1_COLBAR_R_MASK               0xFF0000u
#define DCU_COLBAR_1_COLBAR_R_SHIFT              16u
#define DCU_COLBAR_1_COLBAR_R_WIDTH              8u
#define DCU_COLBAR_1_COLBAR_R(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_1_COLBAR_R_SHIFT))&DCU_COLBAR_1_COLBAR_R_MASK)
/* COLBAR_2 Bit Fields */
#define DCU_COLBAR_2_COLBAR_B_MASK               0xFFu
#define DCU_COLBAR_2_COLBAR_B_SHIFT              0u
#define DCU_COLBAR_2_COLBAR_B_WIDTH              8u
#define DCU_COLBAR_2_COLBAR_B(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_2_COLBAR_B_SHIFT))&DCU_COLBAR_2_COLBAR_B_MASK)
#define DCU_COLBAR_2_COLBAR_G_MASK               0xFF00u
#define DCU_COLBAR_2_COLBAR_G_SHIFT              8u
#define DCU_COLBAR_2_COLBAR_G_WIDTH              8u
#define DCU_COLBAR_2_COLBAR_G(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_2_COLBAR_G_SHIFT))&DCU_COLBAR_2_COLBAR_G_MASK)
#define DCU_COLBAR_2_COLBAR_R_MASK               0xFF0000u
#define DCU_COLBAR_2_COLBAR_R_SHIFT              16u
#define DCU_COLBAR_2_COLBAR_R_WIDTH              8u
#define DCU_COLBAR_2_COLBAR_R(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_2_COLBAR_R_SHIFT))&DCU_COLBAR_2_COLBAR_R_MASK)
/* COLBAR_3 Bit Fields */
#define DCU_COLBAR_3_COLBAR_B_MASK               0xFFu
#define DCU_COLBAR_3_COLBAR_B_SHIFT              0u
#define DCU_COLBAR_3_COLBAR_B_WIDTH              8u
#define DCU_COLBAR_3_COLBAR_B(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_3_COLBAR_B_SHIFT))&DCU_COLBAR_3_COLBAR_B_MASK)
#define DCU_COLBAR_3_COLBAR_G_MASK               0xFF00u
#define DCU_COLBAR_3_COLBAR_G_SHIFT              8u
#define DCU_COLBAR_3_COLBAR_G_WIDTH              8u
#define DCU_COLBAR_3_COLBAR_G(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_3_COLBAR_G_SHIFT))&DCU_COLBAR_3_COLBAR_G_MASK)
#define DCU_COLBAR_3_COLBAR_R_MASK               0xFF0000u
#define DCU_COLBAR_3_COLBAR_R_SHIFT              16u
#define DCU_COLBAR_3_COLBAR_R_WIDTH              8u
#define DCU_COLBAR_3_COLBAR_R(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_3_COLBAR_R_SHIFT))&DCU_COLBAR_3_COLBAR_R_MASK)
/* COLBAR_4 Bit Fields */
#define DCU_COLBAR_4_COLBAR_B_MASK               0xFFu
#define DCU_COLBAR_4_COLBAR_B_SHIFT              0u
#define DCU_COLBAR_4_COLBAR_B_WIDTH              8u
#define DCU_COLBAR_4_COLBAR_B(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_4_COLBAR_B_SHIFT))&DCU_COLBAR_4_COLBAR_B_MASK)
#define DCU_COLBAR_4_COLBAR_G_MASK               0xFF00u
#define DCU_COLBAR_4_COLBAR_G_SHIFT              8u
#define DCU_COLBAR_4_COLBAR_G_WIDTH              8u
#define DCU_COLBAR_4_COLBAR_G(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_4_COLBAR_G_SHIFT))&DCU_COLBAR_4_COLBAR_G_MASK)
#define DCU_COLBAR_4_COLBAR_R_MASK               0xFF0000u
#define DCU_COLBAR_4_COLBAR_R_SHIFT              16u
#define DCU_COLBAR_4_COLBAR_R_WIDTH              8u
#define DCU_COLBAR_4_COLBAR_R(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_4_COLBAR_R_SHIFT))&DCU_COLBAR_4_COLBAR_R_MASK)
/* COLBAR_5 Bit Fields */
#define DCU_COLBAR_5_COLBAR_B_MASK               0xFFu
#define DCU_COLBAR_5_COLBAR_B_SHIFT              0u
#define DCU_COLBAR_5_COLBAR_B_WIDTH              8u
#define DCU_COLBAR_5_COLBAR_B(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_5_COLBAR_B_SHIFT))&DCU_COLBAR_5_COLBAR_B_MASK)
#define DCU_COLBAR_5_COLBAR_G_MASK               0xFF00u
#define DCU_COLBAR_5_COLBAR_G_SHIFT              8u
#define DCU_COLBAR_5_COLBAR_G_WIDTH              8u
#define DCU_COLBAR_5_COLBAR_G(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_5_COLBAR_G_SHIFT))&DCU_COLBAR_5_COLBAR_G_MASK)
#define DCU_COLBAR_5_COLBAR_R_MASK               0xFF0000u
#define DCU_COLBAR_5_COLBAR_R_SHIFT              16u
#define DCU_COLBAR_5_COLBAR_R_WIDTH              8u
#define DCU_COLBAR_5_COLBAR_R(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_5_COLBAR_R_SHIFT))&DCU_COLBAR_5_COLBAR_R_MASK)
/* COLBAR_6 Bit Fields */
#define DCU_COLBAR_6_COLBAR_B_MASK               0xFFu
#define DCU_COLBAR_6_COLBAR_B_SHIFT              0u
#define DCU_COLBAR_6_COLBAR_B_WIDTH              8u
#define DCU_COLBAR_6_COLBAR_B(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_6_COLBAR_B_SHIFT))&DCU_COLBAR_6_COLBAR_B_MASK)
#define DCU_COLBAR_6_COLBAR_G_MASK               0xFF00u
#define DCU_COLBAR_6_COLBAR_G_SHIFT              8u
#define DCU_COLBAR_6_COLBAR_G_WIDTH              8u
#define DCU_COLBAR_6_COLBAR_G(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_6_COLBAR_G_SHIFT))&DCU_COLBAR_6_COLBAR_G_MASK)
#define DCU_COLBAR_6_COLBAR_R_MASK               0xFF0000u
#define DCU_COLBAR_6_COLBAR_R_SHIFT              16u
#define DCU_COLBAR_6_COLBAR_R_WIDTH              8u
#define DCU_COLBAR_6_COLBAR_R(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_6_COLBAR_R_SHIFT))&DCU_COLBAR_6_COLBAR_R_MASK)
/* COLBAR_7 Bit Fields */
#define DCU_COLBAR_7_COLBAR_B_MASK               0xFFu
#define DCU_COLBAR_7_COLBAR_B_SHIFT              0u
#define DCU_COLBAR_7_COLBAR_B_WIDTH              8u
#define DCU_COLBAR_7_COLBAR_B(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_7_COLBAR_B_SHIFT))&DCU_COLBAR_7_COLBAR_B_MASK)
#define DCU_COLBAR_7_COLBAR_G_MASK               0xFF00u
#define DCU_COLBAR_7_COLBAR_G_SHIFT              8u
#define DCU_COLBAR_7_COLBAR_G_WIDTH              8u
#define DCU_COLBAR_7_COLBAR_G(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_7_COLBAR_G_SHIFT))&DCU_COLBAR_7_COLBAR_G_MASK)
#define DCU_COLBAR_7_COLBAR_R_MASK               0xFF0000u
#define DCU_COLBAR_7_COLBAR_R_SHIFT              16u
#define DCU_COLBAR_7_COLBAR_R_WIDTH              8u
#define DCU_COLBAR_7_COLBAR_R(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_7_COLBAR_R_SHIFT))&DCU_COLBAR_7_COLBAR_R_MASK)
/* COLBAR_8 Bit Fields */
#define DCU_COLBAR_8_COLBAR_B_MASK               0xFFu
#define DCU_COLBAR_8_COLBAR_B_SHIFT              0u
#define DCU_COLBAR_8_COLBAR_B_WIDTH              8u
#define DCU_COLBAR_8_COLBAR_B(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_8_COLBAR_B_SHIFT))&DCU_COLBAR_8_COLBAR_B_MASK)
#define DCU_COLBAR_8_COLBAR_G_MASK               0xFF00u
#define DCU_COLBAR_8_COLBAR_G_SHIFT              8u
#define DCU_COLBAR_8_COLBAR_G_WIDTH              8u
#define DCU_COLBAR_8_COLBAR_G(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_8_COLBAR_G_SHIFT))&DCU_COLBAR_8_COLBAR_G_MASK)
#define DCU_COLBAR_8_COLBAR_R_MASK               0xFF0000u
#define DCU_COLBAR_8_COLBAR_R_SHIFT              16u
#define DCU_COLBAR_8_COLBAR_R_WIDTH              8u
#define DCU_COLBAR_8_COLBAR_R(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_COLBAR_8_COLBAR_R_SHIFT))&DCU_COLBAR_8_COLBAR_R_MASK)
/* DIV_RATIO Bit Fields */
#define DCU_DIV_RATIO_DIV_RATIO_MASK             0xFFu
#define DCU_DIV_RATIO_DIV_RATIO_SHIFT            0u
#define DCU_DIV_RATIO_DIV_RATIO_WIDTH            8u
#define DCU_DIV_RATIO_DIV_RATIO(x)               (((uint32_t)(((uint32_t)(x))<<DCU_DIV_RATIO_DIV_RATIO_SHIFT))&DCU_DIV_RATIO_DIV_RATIO_MASK)
#define DCU_DIV_RATIO_DUAL_EDGE_EN_MASK          0x80000000u
#define DCU_DIV_RATIO_DUAL_EDGE_EN_SHIFT         31u
#define DCU_DIV_RATIO_DUAL_EDGE_EN_WIDTH         1u
#define DCU_DIV_RATIO_DUAL_EDGE_EN(x)            (((uint32_t)(((uint32_t)(x))<<DCU_DIV_RATIO_DUAL_EDGE_EN_SHIFT))&DCU_DIV_RATIO_DUAL_EDGE_EN_MASK)
/* SIGN_CALC_1 Bit Fields */
#define DCU_SIGN_CALC_1_SIG_HOR_SIZE_MASK        0xFFFu
#define DCU_SIGN_CALC_1_SIG_HOR_SIZE_SHIFT       0u
#define DCU_SIGN_CALC_1_SIG_HOR_SIZE_WIDTH       12u
#define DCU_SIGN_CALC_1_SIG_HOR_SIZE(x)          (((uint32_t)(((uint32_t)(x))<<DCU_SIGN_CALC_1_SIG_HOR_SIZE_SHIFT))&DCU_SIGN_CALC_1_SIG_HOR_SIZE_MASK)
#define DCU_SIGN_CALC_1_SIG_VER_SIZE_MASK        0xFFF0000u
#define DCU_SIGN_CALC_1_SIG_VER_SIZE_SHIFT       16u
#define DCU_SIGN_CALC_1_SIG_VER_SIZE_WIDTH       12u
#define DCU_SIGN_CALC_1_SIG_VER_SIZE(x)          (((uint32_t)(((uint32_t)(x))<<DCU_SIGN_CALC_1_SIG_VER_SIZE_SHIFT))&DCU_SIGN_CALC_1_SIG_VER_SIZE_MASK)
/* SIGN_CALC_2 Bit Fields */
#define DCU_SIGN_CALC_2_SIG_HOR_POS_MASK         0xFFFu
#define DCU_SIGN_CALC_2_SIG_HOR_POS_SHIFT        0u
#define DCU_SIGN_CALC_2_SIG_HOR_POS_WIDTH        12u
#define DCU_SIGN_CALC_2_SIG_HOR_POS(x)           (((uint32_t)(((uint32_t)(x))<<DCU_SIGN_CALC_2_SIG_HOR_POS_SHIFT))&DCU_SIGN_CALC_2_SIG_HOR_POS_MASK)
#define DCU_SIGN_CALC_2_SIG_VER_POS_MASK         0xFFF0000u
#define DCU_SIGN_CALC_2_SIG_VER_POS_SHIFT        16u
#define DCU_SIGN_CALC_2_SIG_VER_POS_WIDTH        12u
#define DCU_SIGN_CALC_2_SIG_VER_POS(x)           (((uint32_t)(((uint32_t)(x))<<DCU_SIGN_CALC_2_SIG_VER_POS_SHIFT))&DCU_SIGN_CALC_2_SIG_VER_POS_MASK)
/* CRC_VAL Bit Fields */
#define DCU_CRC_VAL_CRC_VAL_MASK                 0xFFFFFFFFu
#define DCU_CRC_VAL_CRC_VAL_SHIFT                0u
#define DCU_CRC_VAL_CRC_VAL_WIDTH                32u
#define DCU_CRC_VAL_CRC_VAL(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_CRC_VAL_CRC_VAL_SHIFT))&DCU_CRC_VAL_CRC_VAL_MASK)
/* PARR_ERR_STATUS1 Bit Fields */
#define DCU_PARR_ERR_STATUS1_L7_0_MASK           0xFFu
#define DCU_PARR_ERR_STATUS1_L7_0_SHIFT          0u
#define DCU_PARR_ERR_STATUS1_L7_0_WIDTH          8u
#define DCU_PARR_ERR_STATUS1_L7_0(x)             (((uint32_t)(((uint32_t)(x))<<DCU_PARR_ERR_STATUS1_L7_0_SHIFT))&DCU_PARR_ERR_STATUS1_L7_0_MASK)
/* PARR_ERR_STATUS3 Bit Fields */
#define DCU_PARR_ERR_STATUS3_DISP_ERR_MASK       0x1u
#define DCU_PARR_ERR_STATUS3_DISP_ERR_SHIFT      0u
#define DCU_PARR_ERR_STATUS3_DISP_ERR_WIDTH      1u
#define DCU_PARR_ERR_STATUS3_DISP_ERR(x)         (((uint32_t)(((uint32_t)(x))<<DCU_PARR_ERR_STATUS3_DISP_ERR_SHIFT))&DCU_PARR_ERR_STATUS3_DISP_ERR_MASK)
#define DCU_PARR_ERR_STATUS3_SIG_ERR_MASK        0x2u
#define DCU_PARR_ERR_STATUS3_SIG_ERR_SHIFT       1u
#define DCU_PARR_ERR_STATUS3_SIG_ERR_WIDTH       1u
#define DCU_PARR_ERR_STATUS3_SIG_ERR(x)          (((uint32_t)(((uint32_t)(x))<<DCU_PARR_ERR_STATUS3_SIG_ERR_SHIFT))&DCU_PARR_ERR_STATUS3_SIG_ERR_MASK)
#define DCU_PARR_ERR_STATUS3_HWC_ERR_MASK        0x4u
#define DCU_PARR_ERR_STATUS3_HWC_ERR_SHIFT       2u
#define DCU_PARR_ERR_STATUS3_HWC_ERR_WIDTH       1u
#define DCU_PARR_ERR_STATUS3_HWC_ERR(x)          (((uint32_t)(((uint32_t)(x))<<DCU_PARR_ERR_STATUS3_HWC_ERR_SHIFT))&DCU_PARR_ERR_STATUS3_HWC_ERR_MASK)
/* MASK_PARR_ERR_STATUS1 Bit Fields */
#define DCU_MASK_PARR_ERR_STATUS1_ML7_0_MASK     0xFFu
#define DCU_MASK_PARR_ERR_STATUS1_ML7_0_SHIFT    0u
#define DCU_MASK_PARR_ERR_STATUS1_ML7_0_WIDTH    8u
#define DCU_MASK_PARR_ERR_STATUS1_ML7_0(x)       (((uint32_t)(((uint32_t)(x))<<DCU_MASK_PARR_ERR_STATUS1_ML7_0_SHIFT))&DCU_MASK_PARR_ERR_STATUS1_ML7_0_MASK)
/* MASK_PARR_ERR_STATUS3 Bit Fields */
#define DCU_MASK_PARR_ERR_STATUS3_M_DISP_ERR_MASK 0x1u
#define DCU_MASK_PARR_ERR_STATUS3_M_DISP_ERR_SHIFT 0u
#define DCU_MASK_PARR_ERR_STATUS3_M_DISP_ERR_WIDTH 1u
#define DCU_MASK_PARR_ERR_STATUS3_M_DISP_ERR(x)  (((uint32_t)(((uint32_t)(x))<<DCU_MASK_PARR_ERR_STATUS3_M_DISP_ERR_SHIFT))&DCU_MASK_PARR_ERR_STATUS3_M_DISP_ERR_MASK)
#define DCU_MASK_PARR_ERR_STATUS3_M_SIG_ERR_MASK 0x2u
#define DCU_MASK_PARR_ERR_STATUS3_M_SIG_ERR_SHIFT 1u
#define DCU_MASK_PARR_ERR_STATUS3_M_SIG_ERR_WIDTH 1u
#define DCU_MASK_PARR_ERR_STATUS3_M_SIG_ERR(x)   (((uint32_t)(((uint32_t)(x))<<DCU_MASK_PARR_ERR_STATUS3_M_SIG_ERR_SHIFT))&DCU_MASK_PARR_ERR_STATUS3_M_SIG_ERR_MASK)
#define DCU_MASK_PARR_ERR_STATUS3_M_HWC_ERR_MASK 0x4u
#define DCU_MASK_PARR_ERR_STATUS3_M_HWC_ERR_SHIFT 2u
#define DCU_MASK_PARR_ERR_STATUS3_M_HWC_ERR_WIDTH 1u
#define DCU_MASK_PARR_ERR_STATUS3_M_HWC_ERR(x)   (((uint32_t)(((uint32_t)(x))<<DCU_MASK_PARR_ERR_STATUS3_M_HWC_ERR_SHIFT))&DCU_MASK_PARR_ERR_STATUS3_M_HWC_ERR_MASK)
/* THRESHOLD_INP_BUF_1 Bit Fields */
#define DCU_THRESHOLD_INP_BUF_1_INP_BUF_P1_LO_MASK 0x7Fu
#define DCU_THRESHOLD_INP_BUF_1_INP_BUF_P1_LO_SHIFT 0u
#define DCU_THRESHOLD_INP_BUF_1_INP_BUF_P1_LO_WIDTH 7u
#define DCU_THRESHOLD_INP_BUF_1_INP_BUF_P1_LO(x) (((uint32_t)(((uint32_t)(x))<<DCU_THRESHOLD_INP_BUF_1_INP_BUF_P1_LO_SHIFT))&DCU_THRESHOLD_INP_BUF_1_INP_BUF_P1_LO_MASK)
#define DCU_THRESHOLD_INP_BUF_1_INP_BUF_P1_HI_MASK 0x7F00u
#define DCU_THRESHOLD_INP_BUF_1_INP_BUF_P1_HI_SHIFT 8u
#define DCU_THRESHOLD_INP_BUF_1_INP_BUF_P1_HI_WIDTH 7u
#define DCU_THRESHOLD_INP_BUF_1_INP_BUF_P1_HI(x) (((uint32_t)(((uint32_t)(x))<<DCU_THRESHOLD_INP_BUF_1_INP_BUF_P1_HI_SHIFT))&DCU_THRESHOLD_INP_BUF_1_INP_BUF_P1_HI_MASK)
#define DCU_THRESHOLD_INP_BUF_1_INP_BUF_P2_LO_MASK 0x7F0000u
#define DCU_THRESHOLD_INP_BUF_1_INP_BUF_P2_LO_SHIFT 16u
#define DCU_THRESHOLD_INP_BUF_1_INP_BUF_P2_LO_WIDTH 7u
#define DCU_THRESHOLD_INP_BUF_1_INP_BUF_P2_LO(x) (((uint32_t)(((uint32_t)(x))<<DCU_THRESHOLD_INP_BUF_1_INP_BUF_P2_LO_SHIFT))&DCU_THRESHOLD_INP_BUF_1_INP_BUF_P2_LO_MASK)
#define DCU_THRESHOLD_INP_BUF_1_INP_BUF_P2_HI_MASK 0x7F000000u
#define DCU_THRESHOLD_INP_BUF_1_INP_BUF_P2_HI_SHIFT 24u
#define DCU_THRESHOLD_INP_BUF_1_INP_BUF_P2_HI_WIDTH 7u
#define DCU_THRESHOLD_INP_BUF_1_INP_BUF_P2_HI(x) (((uint32_t)(((uint32_t)(x))<<DCU_THRESHOLD_INP_BUF_1_INP_BUF_P2_HI_SHIFT))&DCU_THRESHOLD_INP_BUF_1_INP_BUF_P2_HI_MASK)
/* LUMA_COMP Bit Fields */
#define DCU_LUMA_COMP_Y_BLUE_MASK                0x3FFu
#define DCU_LUMA_COMP_Y_BLUE_SHIFT               0u
#define DCU_LUMA_COMP_Y_BLUE_WIDTH               10u
#define DCU_LUMA_COMP_Y_BLUE(x)                  (((uint32_t)(((uint32_t)(x))<<DCU_LUMA_COMP_Y_BLUE_SHIFT))&DCU_LUMA_COMP_Y_BLUE_MASK)
#define DCU_LUMA_COMP_Y_GREEN_MASK               0x1FF800u
#define DCU_LUMA_COMP_Y_GREEN_SHIFT              11u
#define DCU_LUMA_COMP_Y_GREEN_WIDTH              10u
#define DCU_LUMA_COMP_Y_GREEN(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_LUMA_COMP_Y_GREEN_SHIFT))&DCU_LUMA_COMP_Y_GREEN_MASK)
#define DCU_LUMA_COMP_Y_RED_MASK                 0xFFC00000u
#define DCU_LUMA_COMP_Y_RED_SHIFT                22u
#define DCU_LUMA_COMP_Y_RED_WIDTH                10u
#define DCU_LUMA_COMP_Y_RED(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_LUMA_COMP_Y_RED_SHIFT))&DCU_LUMA_COMP_Y_RED_MASK)
/* CHROMA_RED Bit Fields */
#define DCU_CHROMA_RED_CB_RED_MASK               0xFFFu
#define DCU_CHROMA_RED_CB_RED_SHIFT              0u
#define DCU_CHROMA_RED_CB_RED_WIDTH              12u
#define DCU_CHROMA_RED_CB_RED(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_CHROMA_RED_CB_RED_SHIFT))&DCU_CHROMA_RED_CB_RED_MASK)
#define DCU_CHROMA_RED_CR_RED_MASK               0x7FF0000u
#define DCU_CHROMA_RED_CR_RED_SHIFT              16u
#define DCU_CHROMA_RED_CR_RED_WIDTH              11u
#define DCU_CHROMA_RED_CR_RED(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_CHROMA_RED_CR_RED_SHIFT))&DCU_CHROMA_RED_CR_RED_MASK)
/* CHROMA_GREEN Bit Fields */
#define DCU_CHROMA_GREEN_CB_GREEN_MASK           0xFFFu
#define DCU_CHROMA_GREEN_CB_GREEN_SHIFT          0u
#define DCU_CHROMA_GREEN_CB_GREEN_WIDTH          12u
#define DCU_CHROMA_GREEN_CB_GREEN(x)             (((uint32_t)(((uint32_t)(x))<<DCU_CHROMA_GREEN_CB_GREEN_SHIFT))&DCU_CHROMA_GREEN_CB_GREEN_MASK)
#define DCU_CHROMA_GREEN_CR_GREEN_MASK           0x7FF0000u
#define DCU_CHROMA_GREEN_CR_GREEN_SHIFT          16u
#define DCU_CHROMA_GREEN_CR_GREEN_WIDTH          11u
#define DCU_CHROMA_GREEN_CR_GREEN(x)             (((uint32_t)(((uint32_t)(x))<<DCU_CHROMA_GREEN_CR_GREEN_SHIFT))&DCU_CHROMA_GREEN_CR_GREEN_MASK)
/* CHROMA_BLUE Bit Fields */
#define DCU_CHROMA_BLUE_CB_BLUE_MASK             0xFFFu
#define DCU_CHROMA_BLUE_CB_BLUE_SHIFT            0u
#define DCU_CHROMA_BLUE_CB_BLUE_WIDTH            12u
#define DCU_CHROMA_BLUE_CB_BLUE(x)               (((uint32_t)(((uint32_t)(x))<<DCU_CHROMA_BLUE_CB_BLUE_SHIFT))&DCU_CHROMA_BLUE_CB_BLUE_MASK)
#define DCU_CHROMA_BLUE_CR_BLUE_MASK             0x7FF0000u
#define DCU_CHROMA_BLUE_CR_BLUE_SHIFT            16u
#define DCU_CHROMA_BLUE_CR_BLUE_WIDTH            11u
#define DCU_CHROMA_BLUE_CR_BLUE(x)               (((uint32_t)(((uint32_t)(x))<<DCU_CHROMA_BLUE_CR_BLUE_SHIFT))&DCU_CHROMA_BLUE_CR_BLUE_MASK)
/* CRC_POS Bit Fields */
#define DCU_CRC_POS_CRC_POS_MASK                 0xFFFFFFFFu
#define DCU_CRC_POS_CRC_POS_SHIFT                0u
#define DCU_CRC_POS_CRC_POS_WIDTH                32u
#define DCU_CRC_POS_CRC_POS(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_CRC_POS_CRC_POS_SHIFT))&DCU_CRC_POS_CRC_POS_MASK)
/* LYR_INTPOL_EN Bit Fields */
#define DCU_LYR_INTPOL_EN_EN_MASK                0x80000000u
#define DCU_LYR_INTPOL_EN_EN_SHIFT               31u
#define DCU_LYR_INTPOL_EN_EN_WIDTH               1u
#define DCU_LYR_INTPOL_EN_EN(x)                  (((uint32_t)(((uint32_t)(x))<<DCU_LYR_INTPOL_EN_EN_SHIFT))&DCU_LYR_INTPOL_EN_EN_MASK)
/* LYR_LUMA_COMP Bit Fields */
#define DCU_LYR_LUMA_COMP_Y_BLUE_MASK            0x3FFu
#define DCU_LYR_LUMA_COMP_Y_BLUE_SHIFT           0u
#define DCU_LYR_LUMA_COMP_Y_BLUE_WIDTH           10u
#define DCU_LYR_LUMA_COMP_Y_BLUE(x)              (((uint32_t)(((uint32_t)(x))<<DCU_LYR_LUMA_COMP_Y_BLUE_SHIFT))&DCU_LYR_LUMA_COMP_Y_BLUE_MASK)
#define DCU_LYR_LUMA_COMP_Y_GREEN_MASK           0x1FF800u
#define DCU_LYR_LUMA_COMP_Y_GREEN_SHIFT          11u
#define DCU_LYR_LUMA_COMP_Y_GREEN_WIDTH          10u
#define DCU_LYR_LUMA_COMP_Y_GREEN(x)             (((uint32_t)(((uint32_t)(x))<<DCU_LYR_LUMA_COMP_Y_GREEN_SHIFT))&DCU_LYR_LUMA_COMP_Y_GREEN_MASK)
#define DCU_LYR_LUMA_COMP_Y_RED_MASK             0xFFC00000u
#define DCU_LYR_LUMA_COMP_Y_RED_SHIFT            22u
#define DCU_LYR_LUMA_COMP_Y_RED_WIDTH            10u
#define DCU_LYR_LUMA_COMP_Y_RED(x)               (((uint32_t)(((uint32_t)(x))<<DCU_LYR_LUMA_COMP_Y_RED_SHIFT))&DCU_LYR_LUMA_COMP_Y_RED_MASK)
/* LYR_CHRM_RED Bit Fields */
#define DCU_LYR_CHRM_RED_Cb_RED_MASK             0xFFFu
#define DCU_LYR_CHRM_RED_Cb_RED_SHIFT            0u
#define DCU_LYR_CHRM_RED_Cb_RED_WIDTH            12u
#define DCU_LYR_CHRM_RED_Cb_RED(x)               (((uint32_t)(((uint32_t)(x))<<DCU_LYR_CHRM_RED_Cb_RED_SHIFT))&DCU_LYR_CHRM_RED_Cb_RED_MASK)
#define DCU_LYR_CHRM_RED_Cr_RED_MASK             0x7FF0000u
#define DCU_LYR_CHRM_RED_Cr_RED_SHIFT            16u
#define DCU_LYR_CHRM_RED_Cr_RED_WIDTH            11u
#define DCU_LYR_CHRM_RED_Cr_RED(x)               (((uint32_t)(((uint32_t)(x))<<DCU_LYR_CHRM_RED_Cr_RED_SHIFT))&DCU_LYR_CHRM_RED_Cr_RED_MASK)
/* LYR_CHRM_GRN Bit Fields */
#define DCU_LYR_CHRM_GRN_Cb_GREEN_MASK           0xFFFu
#define DCU_LYR_CHRM_GRN_Cb_GREEN_SHIFT          0u
#define DCU_LYR_CHRM_GRN_Cb_GREEN_WIDTH          12u
#define DCU_LYR_CHRM_GRN_Cb_GREEN(x)             (((uint32_t)(((uint32_t)(x))<<DCU_LYR_CHRM_GRN_Cb_GREEN_SHIFT))&DCU_LYR_CHRM_GRN_Cb_GREEN_MASK)
#define DCU_LYR_CHRM_GRN_Cr_GREEN_MASK           0x7FF0000u
#define DCU_LYR_CHRM_GRN_Cr_GREEN_SHIFT          16u
#define DCU_LYR_CHRM_GRN_Cr_GREEN_WIDTH          11u
#define DCU_LYR_CHRM_GRN_Cr_GREEN(x)             (((uint32_t)(((uint32_t)(x))<<DCU_LYR_CHRM_GRN_Cr_GREEN_SHIFT))&DCU_LYR_CHRM_GRN_Cr_GREEN_MASK)
/* LYR_CHRM_BLUE Bit Fields */
#define DCU_LYR_CHRM_BLUE_Cb_BLUE_MASK           0xFFFu
#define DCU_LYR_CHRM_BLUE_Cb_BLUE_SHIFT          0u
#define DCU_LYR_CHRM_BLUE_Cb_BLUE_WIDTH          12u
#define DCU_LYR_CHRM_BLUE_Cb_BLUE(x)             (((uint32_t)(((uint32_t)(x))<<DCU_LYR_CHRM_BLUE_Cb_BLUE_SHIFT))&DCU_LYR_CHRM_BLUE_Cb_BLUE_MASK)
#define DCU_LYR_CHRM_BLUE_Cr_BLUE_MASK           0x7FF0000u
#define DCU_LYR_CHRM_BLUE_Cr_BLUE_SHIFT          16u
#define DCU_LYR_CHRM_BLUE_Cr_BLUE_WIDTH          11u
#define DCU_LYR_CHRM_BLUE_Cr_BLUE(x)             (((uint32_t)(((uint32_t)(x))<<DCU_LYR_CHRM_BLUE_Cr_BLUE_SHIFT))&DCU_LYR_CHRM_BLUE_Cr_BLUE_MASK)
/* UPDATE_MODE Bit Fields */
#define DCU_UPDATE_MODE_TRIG_PROG_END_MASK       0xFFu
#define DCU_UPDATE_MODE_TRIG_PROG_END_SHIFT      0u
#define DCU_UPDATE_MODE_TRIG_PROG_END_WIDTH      8u
#define DCU_UPDATE_MODE_TRIG_PROG_END(x)         (((uint32_t)(((uint32_t)(x))<<DCU_UPDATE_MODE_TRIG_PROG_END_SHIFT))&DCU_UPDATE_MODE_TRIG_PROG_END_MASK)
#define DCU_UPDATE_MODE_READREG_MASK             0x40000000u
#define DCU_UPDATE_MODE_READREG_SHIFT            30u
#define DCU_UPDATE_MODE_READREG_WIDTH            1u
#define DCU_UPDATE_MODE_READREG(x)               (((uint32_t)(((uint32_t)(x))<<DCU_UPDATE_MODE_READREG_SHIFT))&DCU_UPDATE_MODE_READREG_MASK)
#define DCU_UPDATE_MODE_MODE_MASK                0x80000000u
#define DCU_UPDATE_MODE_MODE_SHIFT               31u
#define DCU_UPDATE_MODE_MODE_WIDTH               1u
#define DCU_UPDATE_MODE_MODE(x)                  (((uint32_t)(((uint32_t)(x))<<DCU_UPDATE_MODE_MODE_SHIFT))&DCU_UPDATE_MODE_MODE_MASK)
/* UNDERRUN Bit Fields */
#define DCU_UNDERRUN_PIXEL_MASK                  0x7FFu
#define DCU_UNDERRUN_PIXEL_SHIFT                 0u
#define DCU_UNDERRUN_PIXEL_WIDTH                 11u
#define DCU_UNDERRUN_PIXEL(x)                    (((uint32_t)(((uint32_t)(x))<<DCU_UNDERRUN_PIXEL_SHIFT))&DCU_UNDERRUN_PIXEL_MASK)
#define DCU_UNDERRUN_LINE_MASK                   0x7FF0000u
#define DCU_UNDERRUN_LINE_SHIFT                  16u
#define DCU_UNDERRUN_LINE_WIDTH                  11u
#define DCU_UNDERRUN_LINE(x)                     (((uint32_t)(((uint32_t)(x))<<DCU_UNDERRUN_LINE_SHIFT))&DCU_UNDERRUN_LINE_MASK)
/* FRM_CRC_CTRL Bit Fields */
#define DCU_FRM_CRC_CTRL_FRM_CRC_EN_MASK         0x1u
#define DCU_FRM_CRC_CTRL_FRM_CRC_EN_SHIFT        0u
#define DCU_FRM_CRC_CTRL_FRM_CRC_EN_WIDTH        1u
#define DCU_FRM_CRC_CTRL_FRM_CRC_EN(x)           (((uint32_t)(((uint32_t)(x))<<DCU_FRM_CRC_CTRL_FRM_CRC_EN_SHIFT))&DCU_FRM_CRC_CTRL_FRM_CRC_EN_MASK)
/* FRM_CRC_VAL Bit Fields */
#define DCU_FRM_CRC_VAL_FRM_CRC_VAL_MASK         0xFFFFFFFFu
#define DCU_FRM_CRC_VAL_FRM_CRC_VAL_SHIFT        0u
#define DCU_FRM_CRC_VAL_FRM_CRC_VAL_WIDTH        32u
#define DCU_FRM_CRC_VAL_FRM_CRC_VAL(x)           (((uint32_t)(((uint32_t)(x))<<DCU_FRM_CRC_VAL_FRM_CRC_VAL_SHIFT))&DCU_FRM_CRC_VAL_FRM_CRC_VAL_MASK)
/* TX_ESCAL_LVL Bit Fields */
#define DCU_TX_ESCAL_LVL_TX_ESCAL_LVL_MASK       0xFu
#define DCU_TX_ESCAL_LVL_TX_ESCAL_LVL_SHIFT      0u
#define DCU_TX_ESCAL_LVL_TX_ESCAL_LVL_WIDTH      4u
#define DCU_TX_ESCAL_LVL_TX_ESCAL_LVL(x)         (((uint32_t)(((uint32_t)(x))<<DCU_TX_ESCAL_LVL_TX_ESCAL_LVL_SHIFT))&DCU_TX_ESCAL_LVL_TX_ESCAL_LVL_MASK)
/* GPR Bit Fields */
#define DCU_GPR_HLB_MASK                         0x80000000u
#define DCU_GPR_HLB_SHIFT                        31u
#define DCU_GPR_HLB_WIDTH                        1u
#define DCU_GPR_HLB(x)                           (((uint32_t)(((uint32_t)(x))<<DCU_GPR_HLB_SHIFT))&DCU_GPR_HLB_MASK)
/* SLR_L0 Bit Fields */
#define DCU_SLR_L0_SLB_L0_11_MASK                0x800u
#define DCU_SLR_L0_SLB_L0_11_SHIFT               11u
#define DCU_SLR_L0_SLB_L0_11_WIDTH               1u
#define DCU_SLR_L0_SLB_L0_11(x)                  (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_SLB_L0_11_SHIFT))&DCU_SLR_L0_SLB_L0_11_MASK)
#define DCU_SLR_L0_WEN_L0_11_MASK                0x8000u
#define DCU_SLR_L0_WEN_L0_11_SHIFT               15u
#define DCU_SLR_L0_WEN_L0_11_WIDTH               1u
#define DCU_SLR_L0_WEN_L0_11(x)                  (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_WEN_L0_11_SHIFT))&DCU_SLR_L0_WEN_L0_11_MASK)
#define DCU_SLR_L0_SLB_L0_10_MASK                0x10000u
#define DCU_SLR_L0_SLB_L0_10_SHIFT               16u
#define DCU_SLR_L0_SLB_L0_10_WIDTH               1u
#define DCU_SLR_L0_SLB_L0_10(x)                  (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_SLB_L0_10_SHIFT))&DCU_SLR_L0_SLB_L0_10_MASK)
#define DCU_SLR_L0_SLB_L0_7_MASK                 0x20000u
#define DCU_SLR_L0_SLB_L0_7_SHIFT                17u
#define DCU_SLR_L0_SLB_L0_7_WIDTH                1u
#define DCU_SLR_L0_SLB_L0_7(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_SLB_L0_7_SHIFT))&DCU_SLR_L0_SLB_L0_7_MASK)
#define DCU_SLR_L0_SLB_L0_6_MASK                 0x40000u
#define DCU_SLR_L0_SLB_L0_6_SHIFT                18u
#define DCU_SLR_L0_SLB_L0_6_WIDTH                1u
#define DCU_SLR_L0_SLB_L0_6(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_SLB_L0_6_SHIFT))&DCU_SLR_L0_SLB_L0_6_MASK)
#define DCU_SLR_L0_SLB_L0_5_MASK                 0x80000u
#define DCU_SLR_L0_SLB_L0_5_SHIFT                19u
#define DCU_SLR_L0_SLB_L0_5_WIDTH                1u
#define DCU_SLR_L0_SLB_L0_5(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_SLB_L0_5_SHIFT))&DCU_SLR_L0_SLB_L0_5_MASK)
#define DCU_SLR_L0_WEN_L0_10_MASK                0x100000u
#define DCU_SLR_L0_WEN_L0_10_SHIFT               20u
#define DCU_SLR_L0_WEN_L0_10_WIDTH               1u
#define DCU_SLR_L0_WEN_L0_10(x)                  (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_WEN_L0_10_SHIFT))&DCU_SLR_L0_WEN_L0_10_MASK)
#define DCU_SLR_L0_WEN_LO_7_MASK                 0x200000u
#define DCU_SLR_L0_WEN_LO_7_SHIFT                21u
#define DCU_SLR_L0_WEN_LO_7_WIDTH                1u
#define DCU_SLR_L0_WEN_LO_7(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_WEN_LO_7_SHIFT))&DCU_SLR_L0_WEN_LO_7_MASK)
#define DCU_SLR_L0_WEN_LO_6_MASK                 0x400000u
#define DCU_SLR_L0_WEN_LO_6_SHIFT                22u
#define DCU_SLR_L0_WEN_LO_6_WIDTH                1u
#define DCU_SLR_L0_WEN_LO_6(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_WEN_LO_6_SHIFT))&DCU_SLR_L0_WEN_LO_6_MASK)
#define DCU_SLR_L0_WEN_LO_5_MASK                 0x800000u
#define DCU_SLR_L0_WEN_LO_5_SHIFT                23u
#define DCU_SLR_L0_WEN_LO_5_WIDTH                1u
#define DCU_SLR_L0_WEN_LO_5(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_WEN_LO_5_SHIFT))&DCU_SLR_L0_WEN_LO_5_MASK)
#define DCU_SLR_L0_SLB_L0_4_MASK                 0x1000000u
#define DCU_SLR_L0_SLB_L0_4_SHIFT                24u
#define DCU_SLR_L0_SLB_L0_4_WIDTH                1u
#define DCU_SLR_L0_SLB_L0_4(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_SLB_L0_4_SHIFT))&DCU_SLR_L0_SLB_L0_4_MASK)
#define DCU_SLR_L0_SLB_L0_3_MASK                 0x2000000u
#define DCU_SLR_L0_SLB_L0_3_SHIFT                25u
#define DCU_SLR_L0_SLB_L0_3_WIDTH                1u
#define DCU_SLR_L0_SLB_L0_3(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_SLB_L0_3_SHIFT))&DCU_SLR_L0_SLB_L0_3_MASK)
#define DCU_SLR_L0_SLB_L0_2_MASK                 0x4000000u
#define DCU_SLR_L0_SLB_L0_2_SHIFT                26u
#define DCU_SLR_L0_SLB_L0_2_WIDTH                1u
#define DCU_SLR_L0_SLB_L0_2(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_SLB_L0_2_SHIFT))&DCU_SLR_L0_SLB_L0_2_MASK)
#define DCU_SLR_L0_SLB_L0_1_MASK                 0x8000000u
#define DCU_SLR_L0_SLB_L0_1_SHIFT                27u
#define DCU_SLR_L0_SLB_L0_1_WIDTH                1u
#define DCU_SLR_L0_SLB_L0_1(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_SLB_L0_1_SHIFT))&DCU_SLR_L0_SLB_L0_1_MASK)
#define DCU_SLR_L0_WEN_LO_4_MASK                 0x10000000u
#define DCU_SLR_L0_WEN_LO_4_SHIFT                28u
#define DCU_SLR_L0_WEN_LO_4_WIDTH                1u
#define DCU_SLR_L0_WEN_LO_4(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_WEN_LO_4_SHIFT))&DCU_SLR_L0_WEN_LO_4_MASK)
#define DCU_SLR_L0_WEN_LO_3_MASK                 0x20000000u
#define DCU_SLR_L0_WEN_LO_3_SHIFT                29u
#define DCU_SLR_L0_WEN_LO_3_WIDTH                1u
#define DCU_SLR_L0_WEN_LO_3(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_WEN_LO_3_SHIFT))&DCU_SLR_L0_WEN_LO_3_MASK)
#define DCU_SLR_L0_WEN_LO_2_MASK                 0x40000000u
#define DCU_SLR_L0_WEN_LO_2_SHIFT                30u
#define DCU_SLR_L0_WEN_LO_2_WIDTH                1u
#define DCU_SLR_L0_WEN_LO_2(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_WEN_LO_2_SHIFT))&DCU_SLR_L0_WEN_LO_2_MASK)
#define DCU_SLR_L0_WEN_LO_1_MASK                 0x80000000u
#define DCU_SLR_L0_WEN_LO_1_SHIFT                31u
#define DCU_SLR_L0_WEN_LO_1_WIDTH                1u
#define DCU_SLR_L0_WEN_LO_1(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_WEN_LO_1_SHIFT))&DCU_SLR_L0_WEN_LO_1_MASK)
/* SLR_L1 Bit Fields */
#define DCU_SLR_L1_SLB_L1_11_MASK                0x800u
#define DCU_SLR_L1_SLB_L1_11_SHIFT               11u
#define DCU_SLR_L1_SLB_L1_11_WIDTH               1u
#define DCU_SLR_L1_SLB_L1_11(x)                  (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_SLB_L1_11_SHIFT))&DCU_SLR_L1_SLB_L1_11_MASK)
#define DCU_SLR_L1_WEN_L1_11_MASK                0x8000u
#define DCU_SLR_L1_WEN_L1_11_SHIFT               15u
#define DCU_SLR_L1_WEN_L1_11_WIDTH               1u
#define DCU_SLR_L1_WEN_L1_11(x)                  (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_WEN_L1_11_SHIFT))&DCU_SLR_L1_WEN_L1_11_MASK)
#define DCU_SLR_L1_SLB_L1_10_MASK                0x10000u
#define DCU_SLR_L1_SLB_L1_10_SHIFT               16u
#define DCU_SLR_L1_SLB_L1_10_WIDTH               1u
#define DCU_SLR_L1_SLB_L1_10(x)                  (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_SLB_L1_10_SHIFT))&DCU_SLR_L1_SLB_L1_10_MASK)
#define DCU_SLR_L1_SLB_L1_7_MASK                 0x20000u
#define DCU_SLR_L1_SLB_L1_7_SHIFT                17u
#define DCU_SLR_L1_SLB_L1_7_WIDTH                1u
#define DCU_SLR_L1_SLB_L1_7(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_SLB_L1_7_SHIFT))&DCU_SLR_L1_SLB_L1_7_MASK)
#define DCU_SLR_L1_SLB_L1_6_MASK                 0x40000u
#define DCU_SLR_L1_SLB_L1_6_SHIFT                18u
#define DCU_SLR_L1_SLB_L1_6_WIDTH                1u
#define DCU_SLR_L1_SLB_L1_6(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_SLB_L1_6_SHIFT))&DCU_SLR_L1_SLB_L1_6_MASK)
#define DCU_SLR_L1_SLB_L1_5_MASK                 0x80000u
#define DCU_SLR_L1_SLB_L1_5_SHIFT                19u
#define DCU_SLR_L1_SLB_L1_5_WIDTH                1u
#define DCU_SLR_L1_SLB_L1_5(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_SLB_L1_5_SHIFT))&DCU_SLR_L1_SLB_L1_5_MASK)
#define DCU_SLR_L1_WEN_L1_10_MASK                0x100000u
#define DCU_SLR_L1_WEN_L1_10_SHIFT               20u
#define DCU_SLR_L1_WEN_L1_10_WIDTH               1u
#define DCU_SLR_L1_WEN_L1_10(x)                  (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_WEN_L1_10_SHIFT))&DCU_SLR_L1_WEN_L1_10_MASK)
#define DCU_SLR_L1_WEN_L1_7_MASK                 0x200000u
#define DCU_SLR_L1_WEN_L1_7_SHIFT                21u
#define DCU_SLR_L1_WEN_L1_7_WIDTH                1u
#define DCU_SLR_L1_WEN_L1_7(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_WEN_L1_7_SHIFT))&DCU_SLR_L1_WEN_L1_7_MASK)
#define DCU_SLR_L1_WEN_L1_6_MASK                 0x400000u
#define DCU_SLR_L1_WEN_L1_6_SHIFT                22u
#define DCU_SLR_L1_WEN_L1_6_WIDTH                1u
#define DCU_SLR_L1_WEN_L1_6(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_WEN_L1_6_SHIFT))&DCU_SLR_L1_WEN_L1_6_MASK)
#define DCU_SLR_L1_WEN_L1_5_MASK                 0x800000u
#define DCU_SLR_L1_WEN_L1_5_SHIFT                23u
#define DCU_SLR_L1_WEN_L1_5_WIDTH                1u
#define DCU_SLR_L1_WEN_L1_5(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_WEN_L1_5_SHIFT))&DCU_SLR_L1_WEN_L1_5_MASK)
#define DCU_SLR_L1_SLB_L1_4_MASK                 0x1000000u
#define DCU_SLR_L1_SLB_L1_4_SHIFT                24u
#define DCU_SLR_L1_SLB_L1_4_WIDTH                1u
#define DCU_SLR_L1_SLB_L1_4(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_SLB_L1_4_SHIFT))&DCU_SLR_L1_SLB_L1_4_MASK)
#define DCU_SLR_L1_SLB_L1_3_MASK                 0x2000000u
#define DCU_SLR_L1_SLB_L1_3_SHIFT                25u
#define DCU_SLR_L1_SLB_L1_3_WIDTH                1u
#define DCU_SLR_L1_SLB_L1_3(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_SLB_L1_3_SHIFT))&DCU_SLR_L1_SLB_L1_3_MASK)
#define DCU_SLR_L1_SLB_L1_2_MASK                 0x4000000u
#define DCU_SLR_L1_SLB_L1_2_SHIFT                26u
#define DCU_SLR_L1_SLB_L1_2_WIDTH                1u
#define DCU_SLR_L1_SLB_L1_2(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_SLB_L1_2_SHIFT))&DCU_SLR_L1_SLB_L1_2_MASK)
#define DCU_SLR_L1_SLB_L1_1_MASK                 0x8000000u
#define DCU_SLR_L1_SLB_L1_1_SHIFT                27u
#define DCU_SLR_L1_SLB_L1_1_WIDTH                1u
#define DCU_SLR_L1_SLB_L1_1(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_SLB_L1_1_SHIFT))&DCU_SLR_L1_SLB_L1_1_MASK)
#define DCU_SLR_L1_WEN_L1_4_MASK                 0x10000000u
#define DCU_SLR_L1_WEN_L1_4_SHIFT                28u
#define DCU_SLR_L1_WEN_L1_4_WIDTH                1u
#define DCU_SLR_L1_WEN_L1_4(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_WEN_L1_4_SHIFT))&DCU_SLR_L1_WEN_L1_4_MASK)
#define DCU_SLR_L1_WEN_L1_3_MASK                 0x20000000u
#define DCU_SLR_L1_WEN_L1_3_SHIFT                29u
#define DCU_SLR_L1_WEN_L1_3_WIDTH                1u
#define DCU_SLR_L1_WEN_L1_3(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_WEN_L1_3_SHIFT))&DCU_SLR_L1_WEN_L1_3_MASK)
#define DCU_SLR_L1_WEN_L1_2_MASK                 0x40000000u
#define DCU_SLR_L1_WEN_L1_2_SHIFT                30u
#define DCU_SLR_L1_WEN_L1_2_WIDTH                1u
#define DCU_SLR_L1_WEN_L1_2(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_WEN_L1_2_SHIFT))&DCU_SLR_L1_WEN_L1_2_MASK)
#define DCU_SLR_L1_WEN_L1_1_MASK                 0x80000000u
#define DCU_SLR_L1_WEN_L1_1_SHIFT                31u
#define DCU_SLR_L1_WEN_L1_1_WIDTH                1u
#define DCU_SLR_L1_WEN_L1_1(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_WEN_L1_1_SHIFT))&DCU_SLR_L1_WEN_L1_1_MASK)
/* SLR_DISP_SIZE Bit Fields */
#define DCU_SLR_DISP_SIZE_SLB_DISP_MASK          0x8000000u
#define DCU_SLR_DISP_SIZE_SLB_DISP_SHIFT         27u
#define DCU_SLR_DISP_SIZE_SLB_DISP_WIDTH         1u
#define DCU_SLR_DISP_SIZE_SLB_DISP(x)            (((uint32_t)(((uint32_t)(x))<<DCU_SLR_DISP_SIZE_SLB_DISP_SHIFT))&DCU_SLR_DISP_SIZE_SLB_DISP_MASK)
#define DCU_SLR_DISP_SIZE_WEN_DISP_MASK          0x80000000u
#define DCU_SLR_DISP_SIZE_WEN_DISP_SHIFT         31u
#define DCU_SLR_DISP_SIZE_WEN_DISP_WIDTH         1u
#define DCU_SLR_DISP_SIZE_WEN_DISP(x)            (((uint32_t)(((uint32_t)(x))<<DCU_SLR_DISP_SIZE_WEN_DISP_SHIFT))&DCU_SLR_DISP_SIZE_WEN_DISP_MASK)
/* SLR_HVSYNC_PARA Bit Fields */
#define DCU_SLR_HVSYNC_PARA_SLB_VSYNC_MASK       0x4000000u
#define DCU_SLR_HVSYNC_PARA_SLB_VSYNC_SHIFT      26u
#define DCU_SLR_HVSYNC_PARA_SLB_VSYNC_WIDTH      1u
#define DCU_SLR_HVSYNC_PARA_SLB_VSYNC(x)         (((uint32_t)(((uint32_t)(x))<<DCU_SLR_HVSYNC_PARA_SLB_VSYNC_SHIFT))&DCU_SLR_HVSYNC_PARA_SLB_VSYNC_MASK)
#define DCU_SLR_HVSYNC_PARA_SLB_HSYNC_MASK       0x8000000u
#define DCU_SLR_HVSYNC_PARA_SLB_HSYNC_SHIFT      27u
#define DCU_SLR_HVSYNC_PARA_SLB_HSYNC_WIDTH      1u
#define DCU_SLR_HVSYNC_PARA_SLB_HSYNC(x)         (((uint32_t)(((uint32_t)(x))<<DCU_SLR_HVSYNC_PARA_SLB_HSYNC_SHIFT))&DCU_SLR_HVSYNC_PARA_SLB_HSYNC_MASK)
#define DCU_SLR_HVSYNC_PARA_WEN_VSYNC_MASK       0x40000000u
#define DCU_SLR_HVSYNC_PARA_WEN_VSYNC_SHIFT      30u
#define DCU_SLR_HVSYNC_PARA_WEN_VSYNC_WIDTH      1u
#define DCU_SLR_HVSYNC_PARA_WEN_VSYNC(x)         (((uint32_t)(((uint32_t)(x))<<DCU_SLR_HVSYNC_PARA_WEN_VSYNC_SHIFT))&DCU_SLR_HVSYNC_PARA_WEN_VSYNC_MASK)
#define DCU_SLR_HVSYNC_PARA_WEN_HSYNC_MASK       0x80000000u
#define DCU_SLR_HVSYNC_PARA_WEN_HSYNC_SHIFT      31u
#define DCU_SLR_HVSYNC_PARA_WEN_HSYNC_WIDTH      1u
#define DCU_SLR_HVSYNC_PARA_WEN_HSYNC(x)         (((uint32_t)(((uint32_t)(x))<<DCU_SLR_HVSYNC_PARA_WEN_HSYNC_SHIFT))&DCU_SLR_HVSYNC_PARA_WEN_HSYNC_MASK)
/* SLR_POL Bit Fields */
#define DCU_SLR_POL_SLB_POL_MASK                 0x8000000u
#define DCU_SLR_POL_SLB_POL_SHIFT                27u
#define DCU_SLR_POL_SLB_POL_WIDTH                1u
#define DCU_SLR_POL_SLB_POL(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_POL_SLB_POL_SHIFT))&DCU_SLR_POL_SLB_POL_MASK)
#define DCU_SLR_POL_WEN_POL_MASK                 0x80000000u
#define DCU_SLR_POL_WEN_POL_SHIFT                31u
#define DCU_SLR_POL_WEN_POL_WIDTH                1u
#define DCU_SLR_POL_WEN_POL(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_SLR_POL_WEN_POL_SHIFT))&DCU_SLR_POL_WEN_POL_MASK)
/* SLR_L0_TRANSP Bit Fields */
#define DCU_SLR_L0_TRANSP_SLB_L0_BCOLOR_MASK     0x4000000u
#define DCU_SLR_L0_TRANSP_SLB_L0_BCOLOR_SHIFT    26u
#define DCU_SLR_L0_TRANSP_SLB_L0_BCOLOR_WIDTH    1u
#define DCU_SLR_L0_TRANSP_SLB_L0_BCOLOR(x)       (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_TRANSP_SLB_L0_BCOLOR_SHIFT))&DCU_SLR_L0_TRANSP_SLB_L0_BCOLOR_MASK)
#define DCU_SLR_L0_TRANSP_SLB_L0_FCOLOR_MASK     0x8000000u
#define DCU_SLR_L0_TRANSP_SLB_L0_FCOLOR_SHIFT    27u
#define DCU_SLR_L0_TRANSP_SLB_L0_FCOLOR_WIDTH    1u
#define DCU_SLR_L0_TRANSP_SLB_L0_FCOLOR(x)       (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_TRANSP_SLB_L0_FCOLOR_SHIFT))&DCU_SLR_L0_TRANSP_SLB_L0_FCOLOR_MASK)
#define DCU_SLR_L0_TRANSP_WEN_L0_BCOLOR_MASK     0x40000000u
#define DCU_SLR_L0_TRANSP_WEN_L0_BCOLOR_SHIFT    30u
#define DCU_SLR_L0_TRANSP_WEN_L0_BCOLOR_WIDTH    1u
#define DCU_SLR_L0_TRANSP_WEN_L0_BCOLOR(x)       (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_TRANSP_WEN_L0_BCOLOR_SHIFT))&DCU_SLR_L0_TRANSP_WEN_L0_BCOLOR_MASK)
#define DCU_SLR_L0_TRANSP_WEN_L0_FCOLOR_MASK     0x80000000u
#define DCU_SLR_L0_TRANSP_WEN_L0_FCOLOR_SHIFT    31u
#define DCU_SLR_L0_TRANSP_WEN_L0_FCOLOR_WIDTH    1u
#define DCU_SLR_L0_TRANSP_WEN_L0_FCOLOR(x)       (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L0_TRANSP_WEN_L0_FCOLOR_SHIFT))&DCU_SLR_L0_TRANSP_WEN_L0_FCOLOR_MASK)
/* SLR_L1_TRANSP Bit Fields */
#define DCU_SLR_L1_TRANSP_SLB_L1_BCOLOR_MASK     0x4000000u
#define DCU_SLR_L1_TRANSP_SLB_L1_BCOLOR_SHIFT    26u
#define DCU_SLR_L1_TRANSP_SLB_L1_BCOLOR_WIDTH    1u
#define DCU_SLR_L1_TRANSP_SLB_L1_BCOLOR(x)       (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_TRANSP_SLB_L1_BCOLOR_SHIFT))&DCU_SLR_L1_TRANSP_SLB_L1_BCOLOR_MASK)
#define DCU_SLR_L1_TRANSP_SLB_L1_FCOLOR_MASK     0x8000000u
#define DCU_SLR_L1_TRANSP_SLB_L1_FCOLOR_SHIFT    27u
#define DCU_SLR_L1_TRANSP_SLB_L1_FCOLOR_WIDTH    1u
#define DCU_SLR_L1_TRANSP_SLB_L1_FCOLOR(x)       (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_TRANSP_SLB_L1_FCOLOR_SHIFT))&DCU_SLR_L1_TRANSP_SLB_L1_FCOLOR_MASK)
#define DCU_SLR_L1_TRANSP_WEN_L1_BCOLOR_MASK     0x40000000u
#define DCU_SLR_L1_TRANSP_WEN_L1_BCOLOR_SHIFT    30u
#define DCU_SLR_L1_TRANSP_WEN_L1_BCOLOR_WIDTH    1u
#define DCU_SLR_L1_TRANSP_WEN_L1_BCOLOR(x)       (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_TRANSP_WEN_L1_BCOLOR_SHIFT))&DCU_SLR_L1_TRANSP_WEN_L1_BCOLOR_MASK)
#define DCU_SLR_L1_TRANSP_WEN_L1_FCOLOR_MASK     0x80000000u
#define DCU_SLR_L1_TRANSP_WEN_L1_FCOLOR_SHIFT    31u
#define DCU_SLR_L1_TRANSP_WEN_L1_FCOLOR_WIDTH    1u
#define DCU_SLR_L1_TRANSP_WEN_L1_FCOLOR(x)       (((uint32_t)(((uint32_t)(x))<<DCU_SLR_L1_TRANSP_WEN_L1_FCOLOR_SHIFT))&DCU_SLR_L1_TRANSP_WEN_L1_FCOLOR_MASK)
/* CTRLDESCL_1 Bit Fields */
#define DCU_CTRLDESCL_1_WIDTH_MASK               0xFFFu
#define DCU_CTRLDESCL_1_WIDTH_SHIFT              0u
#define DCU_CTRLDESCL_1_WIDTH_WIDTH              12u
#define DCU_CTRLDESCL_1_WIDTH(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_1_WIDTH_SHIFT))&DCU_CTRLDESCL_1_WIDTH_MASK)
#define DCU_CTRLDESCL_1_HEIGHT_MASK              0xFFF0000u
#define DCU_CTRLDESCL_1_HEIGHT_SHIFT             16u
#define DCU_CTRLDESCL_1_HEIGHT_WIDTH             12u
#define DCU_CTRLDESCL_1_HEIGHT(x)                (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_1_HEIGHT_SHIFT))&DCU_CTRLDESCL_1_HEIGHT_MASK)
/* CTRLDESCL_2 Bit Fields */
#define DCU_CTRLDESCL_2_POSX_MASK                0x1FFFu
#define DCU_CTRLDESCL_2_POSX_SHIFT               0u
#define DCU_CTRLDESCL_2_POSX_WIDTH               13u
#define DCU_CTRLDESCL_2_POSX(x)                  (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_2_POSX_SHIFT))&DCU_CTRLDESCL_2_POSX_MASK)
#define DCU_CTRLDESCL_2_POSY_MASK                0x1FFF0000u
#define DCU_CTRLDESCL_2_POSY_SHIFT               16u
#define DCU_CTRLDESCL_2_POSY_WIDTH               13u
#define DCU_CTRLDESCL_2_POSY(x)                  (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_2_POSY_SHIFT))&DCU_CTRLDESCL_2_POSY_MASK)
/* CTRLDESCL_3 Bit Fields */
#define DCU_CTRLDESCL_3_ADDR_MASK                0xFFFFFFFFu
#define DCU_CTRLDESCL_3_ADDR_SHIFT               0u
#define DCU_CTRLDESCL_3_ADDR_WIDTH               32u
#define DCU_CTRLDESCL_3_ADDR(x)                  (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_3_ADDR_SHIFT))&DCU_CTRLDESCL_3_ADDR_MASK)
/* CTRLDESCL_4 Bit Fields */
#define DCU_CTRLDESCL_4_AB_MASK                  0x3u
#define DCU_CTRLDESCL_4_AB_SHIFT                 0u
#define DCU_CTRLDESCL_4_AB_WIDTH                 2u
#define DCU_CTRLDESCL_4_AB(x)                    (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_4_AB_SHIFT))&DCU_CTRLDESCL_4_AB_MASK)
#define DCU_CTRLDESCL_4_BB_MASK                  0x4u
#define DCU_CTRLDESCL_4_BB_SHIFT                 2u
#define DCU_CTRLDESCL_4_BB_WIDTH                 1u
#define DCU_CTRLDESCL_4_BB(x)                    (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_4_BB_SHIFT))&DCU_CTRLDESCL_4_BB_MASK)
#define DCU_CTRLDESCL_4_LUOFFS_MASK              0x7FF0u
#define DCU_CTRLDESCL_4_LUOFFS_SHIFT             4u
#define DCU_CTRLDESCL_4_LUOFFS_WIDTH             11u
#define DCU_CTRLDESCL_4_LUOFFS(x)                (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_4_LUOFFS_SHIFT))&DCU_CTRLDESCL_4_LUOFFS_MASK)
#define DCU_CTRLDESCL_4_BPP_MASK                 0xF0000u
#define DCU_CTRLDESCL_4_BPP_SHIFT                16u
#define DCU_CTRLDESCL_4_BPP_WIDTH                4u
#define DCU_CTRLDESCL_4_BPP(x)                   (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_4_BPP_SHIFT))&DCU_CTRLDESCL_4_BPP_MASK)
#define DCU_CTRLDESCL_4_TRANS_MASK               0xFF00000u
#define DCU_CTRLDESCL_4_TRANS_SHIFT              20u
#define DCU_CTRLDESCL_4_TRANS_WIDTH              8u
#define DCU_CTRLDESCL_4_TRANS(x)                 (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_4_TRANS_SHIFT))&DCU_CTRLDESCL_4_TRANS_MASK)
#define DCU_CTRLDESCL_4_SAFETY_EN_MASK           0x10000000u
#define DCU_CTRLDESCL_4_SAFETY_EN_SHIFT          28u
#define DCU_CTRLDESCL_4_SAFETY_EN_WIDTH          1u
#define DCU_CTRLDESCL_4_SAFETY_EN(x)             (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_4_SAFETY_EN_SHIFT))&DCU_CTRLDESCL_4_SAFETY_EN_MASK)
#define DCU_CTRLDESCL_4_EN_MASK                  0x80000000u
#define DCU_CTRLDESCL_4_EN_SHIFT                 31u
#define DCU_CTRLDESCL_4_EN_WIDTH                 1u
#define DCU_CTRLDESCL_4_EN(x)                    (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_4_EN_SHIFT))&DCU_CTRLDESCL_4_EN_MASK)
/* CTRLDESCL_5 Bit Fields */
#define DCU_CTRLDESCL_5_CKMAX_B_MASK             0xFFu
#define DCU_CTRLDESCL_5_CKMAX_B_SHIFT            0u
#define DCU_CTRLDESCL_5_CKMAX_B_WIDTH            8u
#define DCU_CTRLDESCL_5_CKMAX_B(x)               (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_5_CKMAX_B_SHIFT))&DCU_CTRLDESCL_5_CKMAX_B_MASK)
#define DCU_CTRLDESCL_5_CKMAX_G_MASK             0xFF00u
#define DCU_CTRLDESCL_5_CKMAX_G_SHIFT            8u
#define DCU_CTRLDESCL_5_CKMAX_G_WIDTH            8u
#define DCU_CTRLDESCL_5_CKMAX_G(x)               (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_5_CKMAX_G_SHIFT))&DCU_CTRLDESCL_5_CKMAX_G_MASK)
#define DCU_CTRLDESCL_5_CKMAX_R_MASK             0xFF0000u
#define DCU_CTRLDESCL_5_CKMAX_R_SHIFT            16u
#define DCU_CTRLDESCL_5_CKMAX_R_WIDTH            8u
#define DCU_CTRLDESCL_5_CKMAX_R(x)               (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_5_CKMAX_R_SHIFT))&DCU_CTRLDESCL_5_CKMAX_R_MASK)
/* CTRLDESCL_6 Bit Fields */
#define DCU_CTRLDESCL_6_CKMIN_B_MASK             0xFFu
#define DCU_CTRLDESCL_6_CKMIN_B_SHIFT            0u
#define DCU_CTRLDESCL_6_CKMIN_B_WIDTH            8u
#define DCU_CTRLDESCL_6_CKMIN_B(x)               (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_6_CKMIN_B_SHIFT))&DCU_CTRLDESCL_6_CKMIN_B_MASK)
#define DCU_CTRLDESCL_6_CKMIN_G_MASK             0xFF00u
#define DCU_CTRLDESCL_6_CKMIN_G_SHIFT            8u
#define DCU_CTRLDESCL_6_CKMIN_G_WIDTH            8u
#define DCU_CTRLDESCL_6_CKMIN_G(x)               (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_6_CKMIN_G_SHIFT))&DCU_CTRLDESCL_6_CKMIN_G_MASK)
#define DCU_CTRLDESCL_6_CKMIN_R_MASK             0xFF0000u
#define DCU_CTRLDESCL_6_CKMIN_R_SHIFT            16u
#define DCU_CTRLDESCL_6_CKMIN_R_WIDTH            8u
#define DCU_CTRLDESCL_6_CKMIN_R(x)               (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_6_CKMIN_R_SHIFT))&DCU_CTRLDESCL_6_CKMIN_R_MASK)
/* CTRLDESCL_8 Bit Fields */
#define DCU_CTRLDESCL_8_FGn_FCOLOR_MASK          0xFFFFFFu
#define DCU_CTRLDESCL_8_FGn_FCOLOR_SHIFT         0u
#define DCU_CTRLDESCL_8_FGn_FCOLOR_WIDTH         24u
#define DCU_CTRLDESCL_8_FGn_FCOLOR(x)            (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_8_FGn_FCOLOR_SHIFT))&DCU_CTRLDESCL_8_FGn_FCOLOR_MASK)
/* CTRLDESCL_9 Bit Fields */
#define DCU_CTRLDESCL_9_FGn_BCOLOR_MASK          0xFFFFFFu
#define DCU_CTRLDESCL_9_FGn_BCOLOR_SHIFT         0u
#define DCU_CTRLDESCL_9_FGn_BCOLOR_WIDTH         24u
#define DCU_CTRLDESCL_9_FGn_BCOLOR(x)            (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_9_FGn_BCOLOR_SHIFT))&DCU_CTRLDESCL_9_FGn_BCOLOR_MASK)
/* CTRLDESCL_10 Bit Fields */
#define DCU_CTRLDESCL_10_PRE_SKIP_MASK           0xFFFu
#define DCU_CTRLDESCL_10_PRE_SKIP_SHIFT          0u
#define DCU_CTRLDESCL_10_PRE_SKIP_WIDTH          12u
#define DCU_CTRLDESCL_10_PRE_SKIP(x)             (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_10_PRE_SKIP_SHIFT))&DCU_CTRLDESCL_10_PRE_SKIP_MASK)
#define DCU_CTRLDESCL_10_POST_SKIP_MASK          0xFFF0000u
#define DCU_CTRLDESCL_10_POST_SKIP_SHIFT         16u
#define DCU_CTRLDESCL_10_POST_SKIP_WIDTH         12u
#define DCU_CTRLDESCL_10_POST_SKIP(x)            (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_10_POST_SKIP_SHIFT))&DCU_CTRLDESCL_10_POST_SKIP_MASK)
#define DCU_CTRLDESCL_10_EN_GPUTILE_MASK         0x80000000u
#define DCU_CTRLDESCL_10_EN_GPUTILE_SHIFT        31u
#define DCU_CTRLDESCL_10_EN_GPUTILE_WIDTH        1u
#define DCU_CTRLDESCL_10_EN_GPUTILE(x)           (((uint32_t)(((uint32_t)(x))<<DCU_CTRLDESCL_10_EN_GPUTILE_SHIFT))&DCU_CTRLDESCL_10_EN_GPUTILE_MASK)

/*!
 * @}
 */ /* end of group DCU_Register_Masks */


/*!
 * @}
 */ /* end of group DCU_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- DEC200 Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup DEC200_Peripheral_Access_Layer DEC200 Peripheral Access Layer
 * @{
 */


/** DEC200 - Size of Registers Arrays */
#define DEC200_gcregDECReadConfig_COUNT          8u
#define DEC200_gcregDECWriteConfig_COUNT         8u
#define DEC200_gcregDECReadBufferBase_COUNT      8u
#define DEC200_gcregDECReadCacheBase_COUNT       8u
#define DEC200_gcregDECWriteBufferBase_COUNT     8u
#define DEC200_gcregDECWriteCacheBase_COUNT      8u

/** DEC200 - Register Layout Typedef */
typedef struct {
       uint8_t RESERVED_0[2048];
  __IO uint32_t gcregDECReadConfig[DEC200_gcregDECReadConfig_COUNT]; /**< , array offset: 0x800, array step: 0x4 */
       uint8_t RESERVED_1[32];
  __IO uint32_t gcregDECWriteConfig[DEC200_gcregDECWriteConfig_COUNT]; /**< , array offset: 0x840, array step: 0x4 */
       uint8_t RESERVED_2[32];
  __IO uint32_t gcregDECReadBufferBase[DEC200_gcregDECReadBufferBase_COUNT]; /**< , array offset: 0x880, array step: 0x4 */
       uint8_t RESERVED_3[32];
  __IO uint32_t gcregDECReadCacheBase[DEC200_gcregDECReadCacheBase_COUNT]; /**< , array offset: 0x8C0, array step: 0x4 */
       uint8_t RESERVED_4[32];
  __IO uint32_t gcregDECWriteBufferBase[DEC200_gcregDECWriteBufferBase_COUNT]; /**< , array offset: 0x900, array step: 0x4 */
       uint8_t RESERVED_5[32];
  __IO uint32_t gcregDECWriteCacheBase[DEC200_gcregDECWriteCacheBase_COUNT]; /**< , array offset: 0x940, array step: 0x4 */
       uint8_t RESERVED_6[32];
  __IO uint32_t gcregDECControl;                   /**< , offset: 0x980 */
  __I  uint32_t gcregDECIntrAcknowledge;           /**< , offset: 0x984 */
  __IO uint32_t gcregDECIntrEnbl;                  /**< , offset: 0x988 */
       uint8_t RESERVED_7[12];
  __I  uint32_t gcDECTotalReadsIn;                 /**< , offset: 0x998 */
  __I  uint32_t gcDECTotalWritesIn;                /**< , offset: 0x99C */
  __I  uint32_t gcDECTotalReadBurstsIn;            /**< , offset: 0x9A0 */
  __I  uint32_t gcDECTotalWriteBurstsIn;           /**< , offset: 0x9A4 */
  __I  uint32_t gcDECTotalReadsReqIn;              /**< , offset: 0x9A8 */
  __I  uint32_t gcDECTotalWritesReqIn;             /**< , offset: 0x9AC */
  __I  uint32_t gcDECTotalReadLastsIn;             /**< , offset: 0x9B0 */
  __I  uint32_t gcDECTotalWriteLastsIn;            /**< , offset: 0x9B4 */
  __I  uint32_t gcDECTotalReadsOUT;                /**< , offset: 0x9B8 */
  __I  uint32_t gcDECTotalWritesOUT;               /**< , offset: 0x9BC */
  __I  uint32_t gcDECTotalReadBurstsOUT;           /**< , offset: 0x9C0 */
  __I  uint32_t gcDECTotalWriteBurstsOUT;          /**< , offset: 0x9C4 */
  __I  uint32_t gcDECTotalReadsReqOUT;             /**< , offset: 0x9C8 */
  __I  uint32_t gcDECTotalWritesReqOUT;            /**< , offset: 0x9CC */
  __I  uint32_t gcDECTotalReadLastsOUT;            /**< , offset: 0x9D0 */
  __I  uint32_t gcDECTotalWriteLastsOUT;           /**< , offset: 0x9D4 */
} DEC200_Type, *DEC200_MemMapPtr;

 /** Number of instances of the DEC200 module. */
#define DEC200_INSTANCE_COUNT                    (1u)


/* DEC200 - Peripheral instance base addresses */
/** Peripheral DEC200 base address */
#define DEC200_BASE                              (0x40026000u)
/** Peripheral DEC200 base pointer */
#define DEC200                                   ((DEC200_Type *)DEC200_BASE)
/** Array initializer of DEC200 peripheral base addresses */
#define DEC200_BASE_ADDRS                        { DEC200_BASE }
/** Array initializer of DEC200 peripheral base pointers */
#define DEC200_BASE_PTRS                         { DEC200 }

/* ----------------------------------------------------------------------------
   -- DEC200 Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup DEC200_Register_Masks DEC200 Register Masks
 * @{
 */

/* gcregDECReadConfig Bit Fields */
#define DEC200_gcregDECReadConfig_COMPRESSION_ENABLE_MASK 0x1u
#define DEC200_gcregDECReadConfig_COMPRESSION_ENABLE_SHIFT 0u
#define DEC200_gcregDECReadConfig_COMPRESSION_ENABLE_WIDTH 1u
#define DEC200_gcregDECReadConfig_COMPRESSION_ENABLE(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECReadConfig_COMPRESSION_ENABLE_SHIFT))&DEC200_gcregDECReadConfig_COMPRESSION_ENABLE_MASK)
#define DEC200_gcregDECReadConfig_COMPRESSION_SIZE_MASK 0x6u
#define DEC200_gcregDECReadConfig_COMPRESSION_SIZE_SHIFT 1u
#define DEC200_gcregDECReadConfig_COMPRESSION_SIZE_WIDTH 2u
#define DEC200_gcregDECReadConfig_COMPRESSION_SIZE(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECReadConfig_COMPRESSION_SIZE_SHIFT))&DEC200_gcregDECReadConfig_COMPRESSION_SIZE_MASK)
#define DEC200_gcregDECReadConfig_COMPRESSION_FORMAT_MASK 0x78u
#define DEC200_gcregDECReadConfig_COMPRESSION_FORMAT_SHIFT 3u
#define DEC200_gcregDECReadConfig_COMPRESSION_FORMAT_WIDTH 4u
#define DEC200_gcregDECReadConfig_COMPRESSION_FORMAT(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECReadConfig_COMPRESSION_FORMAT_SHIFT))&DEC200_gcregDECReadConfig_COMPRESSION_FORMAT_MASK)
#define DEC200_gcregDECReadConfig_SWIZZLE_MASK   0x300000u
#define DEC200_gcregDECReadConfig_SWIZZLE_SHIFT  20u
#define DEC200_gcregDECReadConfig_SWIZZLE_WIDTH  2u
#define DEC200_gcregDECReadConfig_SWIZZLE(x)     (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECReadConfig_SWIZZLE_SHIFT))&DEC200_gcregDECReadConfig_SWIZZLE_MASK)
/* gcregDECWriteConfig Bit Fields */
#define DEC200_gcregDECWriteConfig_COMPRESSION_ENABLE_MASK 0x1u
#define DEC200_gcregDECWriteConfig_COMPRESSION_ENABLE_SHIFT 0u
#define DEC200_gcregDECWriteConfig_COMPRESSION_ENABLE_WIDTH 1u
#define DEC200_gcregDECWriteConfig_COMPRESSION_ENABLE(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECWriteConfig_COMPRESSION_ENABLE_SHIFT))&DEC200_gcregDECWriteConfig_COMPRESSION_ENABLE_MASK)
#define DEC200_gcregDECWriteConfig_COMPRESSION_SIZE_MASK 0x6u
#define DEC200_gcregDECWriteConfig_COMPRESSION_SIZE_SHIFT 1u
#define DEC200_gcregDECWriteConfig_COMPRESSION_SIZE_WIDTH 2u
#define DEC200_gcregDECWriteConfig_COMPRESSION_SIZE(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECWriteConfig_COMPRESSION_SIZE_SHIFT))&DEC200_gcregDECWriteConfig_COMPRESSION_SIZE_MASK)
#define DEC200_gcregDECWriteConfig_COMPRESSION_FORMAT_MASK 0x78u
#define DEC200_gcregDECWriteConfig_COMPRESSION_FORMAT_SHIFT 3u
#define DEC200_gcregDECWriteConfig_COMPRESSION_FORMAT_WIDTH 4u
#define DEC200_gcregDECWriteConfig_COMPRESSION_FORMAT(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECWriteConfig_COMPRESSION_FORMAT_SHIFT))&DEC200_gcregDECWriteConfig_COMPRESSION_FORMAT_MASK)
#define DEC200_gcregDECWriteConfig_SWIZZLE_MASK  0x300000u
#define DEC200_gcregDECWriteConfig_SWIZZLE_SHIFT 20u
#define DEC200_gcregDECWriteConfig_SWIZZLE_WIDTH 2u
#define DEC200_gcregDECWriteConfig_SWIZZLE(x)    (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECWriteConfig_SWIZZLE_SHIFT))&DEC200_gcregDECWriteConfig_SWIZZLE_MASK)
/* gcregDECReadBufferBase Bit Fields */
#define DEC200_gcregDECReadBufferBase_ADDRESS_MASK 0xFFFFFFFFu
#define DEC200_gcregDECReadBufferBase_ADDRESS_SHIFT 0u
#define DEC200_gcregDECReadBufferBase_ADDRESS_WIDTH 32u
#define DEC200_gcregDECReadBufferBase_ADDRESS(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECReadBufferBase_ADDRESS_SHIFT))&DEC200_gcregDECReadBufferBase_ADDRESS_MASK)
/* gcregDECReadCacheBase Bit Fields */
#define DEC200_gcregDECReadCacheBase_ADDRESS_MASK 0xFFFFFFFFu
#define DEC200_gcregDECReadCacheBase_ADDRESS_SHIFT 0u
#define DEC200_gcregDECReadCacheBase_ADDRESS_WIDTH 32u
#define DEC200_gcregDECReadCacheBase_ADDRESS(x)  (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECReadCacheBase_ADDRESS_SHIFT))&DEC200_gcregDECReadCacheBase_ADDRESS_MASK)
/* gcregDECWriteBufferBase Bit Fields */
#define DEC200_gcregDECWriteBufferBase_ADDRESS_MASK 0xFFFFFFFFu
#define DEC200_gcregDECWriteBufferBase_ADDRESS_SHIFT 0u
#define DEC200_gcregDECWriteBufferBase_ADDRESS_WIDTH 32u
#define DEC200_gcregDECWriteBufferBase_ADDRESS(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECWriteBufferBase_ADDRESS_SHIFT))&DEC200_gcregDECWriteBufferBase_ADDRESS_MASK)
/* gcregDECWriteCacheBase Bit Fields */
#define DEC200_gcregDECWriteCacheBase_ADDRESS_MASK 0xFFFFFFFFu
#define DEC200_gcregDECWriteCacheBase_ADDRESS_SHIFT 0u
#define DEC200_gcregDECWriteCacheBase_ADDRESS_WIDTH 32u
#define DEC200_gcregDECWriteCacheBase_ADDRESS(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECWriteCacheBase_ADDRESS_SHIFT))&DEC200_gcregDECWriteCacheBase_ADDRESS_MASK)
/* gcregDECControl Bit Fields */
#define DEC200_gcregDECControl_FLUSH_MASK        0x1u
#define DEC200_gcregDECControl_FLUSH_SHIFT       0u
#define DEC200_gcregDECControl_FLUSH_WIDTH       1u
#define DEC200_gcregDECControl_FLUSH(x)          (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECControl_FLUSH_SHIFT))&DEC200_gcregDECControl_FLUSH_MASK)
#define DEC200_gcregDECControl_DISABLE_COMPRESSION_MASK 0x2u
#define DEC200_gcregDECControl_DISABLE_COMPRESSION_SHIFT 1u
#define DEC200_gcregDECControl_DISABLE_COMPRESSION_WIDTH 1u
#define DEC200_gcregDECControl_DISABLE_COMPRESSION(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECControl_DISABLE_COMPRESSION_SHIFT))&DEC200_gcregDECControl_DISABLE_COMPRESSION_MASK)
#define DEC200_gcregDECControl_DISABLE_RAM_CLOCK_GATING_MASK 0x4u
#define DEC200_gcregDECControl_DISABLE_RAM_CLOCK_GATING_SHIFT 2u
#define DEC200_gcregDECControl_DISABLE_RAM_CLOCK_GATING_WIDTH 1u
#define DEC200_gcregDECControl_DISABLE_RAM_CLOCK_GATING(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECControl_DISABLE_RAM_CLOCK_GATING_SHIFT))&DEC200_gcregDECControl_DISABLE_RAM_CLOCK_GATING_MASK)
#define DEC200_gcregDECControl_DISABLE_DEBUG_REGISTERS_MASK 0x8u
#define DEC200_gcregDECControl_DISABLE_DEBUG_REGISTERS_SHIFT 3u
#define DEC200_gcregDECControl_DISABLE_DEBUG_REGISTERS_WIDTH 1u
#define DEC200_gcregDECControl_DISABLE_DEBUG_REGISTERS(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECControl_DISABLE_DEBUG_REGISTERS_SHIFT))&DEC200_gcregDECControl_DISABLE_DEBUG_REGISTERS_MASK)
#define DEC200_gcregDECControl_SOFT_RESET_MASK   0x10u
#define DEC200_gcregDECControl_SOFT_RESET_SHIFT  4u
#define DEC200_gcregDECControl_SOFT_RESET_WIDTH  1u
#define DEC200_gcregDECControl_SOFT_RESET(x)     (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECControl_SOFT_RESET_SHIFT))&DEC200_gcregDECControl_SOFT_RESET_MASK)
#define DEC200_gcregDECControl_ENABLE_WRITE_SYNC_MASK 0x40u
#define DEC200_gcregDECControl_ENABLE_WRITE_SYNC_SHIFT 6u
#define DEC200_gcregDECControl_ENABLE_WRITE_SYNC_WIDTH 1u
#define DEC200_gcregDECControl_ENABLE_WRITE_SYNC(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECControl_ENABLE_WRITE_SYNC_SHIFT))&DEC200_gcregDECControl_ENABLE_WRITE_SYNC_MASK)
#define DEC200_gcregDECControl_ENABLE96_BYTE_YUV_COMP_MASK 0x80u
#define DEC200_gcregDECControl_ENABLE96_BYTE_YUV_COMP_SHIFT 7u
#define DEC200_gcregDECControl_ENABLE96_BYTE_YUV_COMP_WIDTH 1u
#define DEC200_gcregDECControl_ENABLE96_BYTE_YUV_COMP(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECControl_ENABLE96_BYTE_YUV_COMP_SHIFT))&DEC200_gcregDECControl_ENABLE96_BYTE_YUV_COMP_MASK)
#define DEC200_gcregDECControl_TILE_STATUS_READ_ID_MASK 0xF00u
#define DEC200_gcregDECControl_TILE_STATUS_READ_ID_SHIFT 8u
#define DEC200_gcregDECControl_TILE_STATUS_READ_ID_WIDTH 4u
#define DEC200_gcregDECControl_TILE_STATUS_READ_ID(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECControl_TILE_STATUS_READ_ID_SHIFT))&DEC200_gcregDECControl_TILE_STATUS_READ_ID_MASK)
#define DEC200_gcregDECControl_TILE_STATUS_WRITE_ID_MASK 0xF000u
#define DEC200_gcregDECControl_TILE_STATUS_WRITE_ID_SHIFT 12u
#define DEC200_gcregDECControl_TILE_STATUS_WRITE_ID_WIDTH 4u
#define DEC200_gcregDECControl_TILE_STATUS_WRITE_ID(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECControl_TILE_STATUS_WRITE_ID_SHIFT))&DEC200_gcregDECControl_TILE_STATUS_WRITE_ID_MASK)
#define DEC200_gcregDECControl_DISABLE_HW_FLUSH_MASK 0x10000u
#define DEC200_gcregDECControl_DISABLE_HW_FLUSH_SHIFT 16u
#define DEC200_gcregDECControl_DISABLE_HW_FLUSH_WIDTH 1u
#define DEC200_gcregDECControl_DISABLE_HW_FLUSH(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECControl_DISABLE_HW_FLUSH_SHIFT))&DEC200_gcregDECControl_DISABLE_HW_FLUSH_MASK)
#define DEC200_gcregDECControl_CLK_DIS_MASK      0x20000u
#define DEC200_gcregDECControl_CLK_DIS_SHIFT     17u
#define DEC200_gcregDECControl_CLK_DIS_WIDTH     1u
#define DEC200_gcregDECControl_CLK_DIS(x)        (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECControl_CLK_DIS_SHIFT))&DEC200_gcregDECControl_CLK_DIS_MASK)
#define DEC200_gcregDECControl_SW_FLUSH_ID_MASK  0x3C0000u
#define DEC200_gcregDECControl_SW_FLUSH_ID_SHIFT 18u
#define DEC200_gcregDECControl_SW_FLUSH_ID_WIDTH 4u
#define DEC200_gcregDECControl_SW_FLUSH_ID(x)    (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECControl_SW_FLUSH_ID_SHIFT))&DEC200_gcregDECControl_SW_FLUSH_ID_MASK)
/* gcregDECIntrAcknowledge Bit Fields */
#define DEC200_gcregDECIntrAcknowledge_AXI_ERR_INT_MASK 0x40000000u
#define DEC200_gcregDECIntrAcknowledge_AXI_ERR_INT_SHIFT 30u
#define DEC200_gcregDECIntrAcknowledge_AXI_ERR_INT_WIDTH 1u
#define DEC200_gcregDECIntrAcknowledge_AXI_ERR_INT(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECIntrAcknowledge_AXI_ERR_INT_SHIFT))&DEC200_gcregDECIntrAcknowledge_AXI_ERR_INT_MASK)
#define DEC200_gcregDECIntrAcknowledge_FLUSH_DN_INT_MASK 0x80000000u
#define DEC200_gcregDECIntrAcknowledge_FLUSH_DN_INT_SHIFT 31u
#define DEC200_gcregDECIntrAcknowledge_FLUSH_DN_INT_WIDTH 1u
#define DEC200_gcregDECIntrAcknowledge_FLUSH_DN_INT(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECIntrAcknowledge_FLUSH_DN_INT_SHIFT))&DEC200_gcregDECIntrAcknowledge_FLUSH_DN_INT_MASK)
/* gcregDECIntrEnbl Bit Fields */
#define DEC200_gcregDECIntrEnbl_AXI_ERR_INT_ENBL_MASK 0x40000000u
#define DEC200_gcregDECIntrEnbl_AXI_ERR_INT_ENBL_SHIFT 30u
#define DEC200_gcregDECIntrEnbl_AXI_ERR_INT_ENBL_WIDTH 1u
#define DEC200_gcregDECIntrEnbl_AXI_ERR_INT_ENBL(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECIntrEnbl_AXI_ERR_INT_ENBL_SHIFT))&DEC200_gcregDECIntrEnbl_AXI_ERR_INT_ENBL_MASK)
#define DEC200_gcregDECIntrEnbl_FLUSH_DN_INT_ENBL_MASK 0x80000000u
#define DEC200_gcregDECIntrEnbl_FLUSH_DN_INT_ENBL_SHIFT 31u
#define DEC200_gcregDECIntrEnbl_FLUSH_DN_INT_ENBL_WIDTH 1u
#define DEC200_gcregDECIntrEnbl_FLUSH_DN_INT_ENBL(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcregDECIntrEnbl_FLUSH_DN_INT_ENBL_SHIFT))&DEC200_gcregDECIntrEnbl_FLUSH_DN_INT_ENBL_MASK)
/* gcDECTotalReadsIn Bit Fields */
#define DEC200_gcDECTotalReadsIn_COUNT_MASK      0xFFFFFFFFu
#define DEC200_gcDECTotalReadsIn_COUNT_SHIFT     0u
#define DEC200_gcDECTotalReadsIn_COUNT_WIDTH     32u
#define DEC200_gcDECTotalReadsIn_COUNT(x)        (((uint32_t)(((uint32_t)(x))<<DEC200_gcDECTotalReadsIn_COUNT_SHIFT))&DEC200_gcDECTotalReadsIn_COUNT_MASK)
/* gcDECTotalWritesIn Bit Fields */
#define DEC200_gcDECTotalWritesIn_COUNT_MASK     0xFFFFFFFFu
#define DEC200_gcDECTotalWritesIn_COUNT_SHIFT    0u
#define DEC200_gcDECTotalWritesIn_COUNT_WIDTH    32u
#define DEC200_gcDECTotalWritesIn_COUNT(x)       (((uint32_t)(((uint32_t)(x))<<DEC200_gcDECTotalWritesIn_COUNT_SHIFT))&DEC200_gcDECTotalWritesIn_COUNT_MASK)
/* gcDECTotalReadBurstsIn Bit Fields */
#define DEC200_gcDECTotalReadBurstsIn_COUNT_MASK 0xFFFFFFFFu
#define DEC200_gcDECTotalReadBurstsIn_COUNT_SHIFT 0u
#define DEC200_gcDECTotalReadBurstsIn_COUNT_WIDTH 32u
#define DEC200_gcDECTotalReadBurstsIn_COUNT(x)   (((uint32_t)(((uint32_t)(x))<<DEC200_gcDECTotalReadBurstsIn_COUNT_SHIFT))&DEC200_gcDECTotalReadBurstsIn_COUNT_MASK)
/* gcDECTotalWriteBurstsIn Bit Fields */
#define DEC200_gcDECTotalWriteBurstsIn_COUNT_MASK 0xFFFFFFFFu
#define DEC200_gcDECTotalWriteBurstsIn_COUNT_SHIFT 0u
#define DEC200_gcDECTotalWriteBurstsIn_COUNT_WIDTH 32u
#define DEC200_gcDECTotalWriteBurstsIn_COUNT(x)  (((uint32_t)(((uint32_t)(x))<<DEC200_gcDECTotalWriteBurstsIn_COUNT_SHIFT))&DEC200_gcDECTotalWriteBurstsIn_COUNT_MASK)
/* gcDECTotalReadsReqIn Bit Fields */
#define DEC200_gcDECTotalReadsReqIn_COUNT_MASK   0xFFFFFFFFu
#define DEC200_gcDECTotalReadsReqIn_COUNT_SHIFT  0u
#define DEC200_gcDECTotalReadsReqIn_COUNT_WIDTH  32u
#define DEC200_gcDECTotalReadsReqIn_COUNT(x)     (((uint32_t)(((uint32_t)(x))<<DEC200_gcDECTotalReadsReqIn_COUNT_SHIFT))&DEC200_gcDECTotalReadsReqIn_COUNT_MASK)
/* gcDECTotalWritesReqIn Bit Fields */
#define DEC200_gcDECTotalWritesReqIn_COUNT_MASK  0xFFFFFFFFu
#define DEC200_gcDECTotalWritesReqIn_COUNT_SHIFT 0u
#define DEC200_gcDECTotalWritesReqIn_COUNT_WIDTH 32u
#define DEC200_gcDECTotalWritesReqIn_COUNT(x)    (((uint32_t)(((uint32_t)(x))<<DEC200_gcDECTotalWritesReqIn_COUNT_SHIFT))&DEC200_gcDECTotalWritesReqIn_COUNT_MASK)
/* gcDECTotalReadLastsIn Bit Fields */
#define DEC200_gcDECTotalReadLastsIn_COUNT_MASK  0xFFFFFFFFu
#define DEC200_gcDECTotalReadLastsIn_COUNT_SHIFT 0u
#define DEC200_gcDECTotalReadLastsIn_COUNT_WIDTH 32u
#define DEC200_gcDECTotalReadLastsIn_COUNT(x)    (((uint32_t)(((uint32_t)(x))<<DEC200_gcDECTotalReadLastsIn_COUNT_SHIFT))&DEC200_gcDECTotalReadLastsIn_COUNT_MASK)
/* gcDECTotalWriteLastsIn Bit Fields */
#define DEC200_gcDECTotalWriteLastsIn_COUNT_MASK 0xFFFFFFFFu
#define DEC200_gcDECTotalWriteLastsIn_COUNT_SHIFT 0u
#define DEC200_gcDECTotalWriteLastsIn_COUNT_WIDTH 32u
#define DEC200_gcDECTotalWriteLastsIn_COUNT(x)   (((uint32_t)(((uint32_t)(x))<<DEC200_gcDECTotalWriteLastsIn_COUNT_SHIFT))&DEC200_gcDECTotalWriteLastsIn_COUNT_MASK)
/* gcDECTotalReadsOUT Bit Fields */
#define DEC200_gcDECTotalReadsOUT_COUNT_MASK     0xFFFFFFFFu
#define DEC200_gcDECTotalReadsOUT_COUNT_SHIFT    0u
#define DEC200_gcDECTotalReadsOUT_COUNT_WIDTH    32u
#define DEC200_gcDECTotalReadsOUT_COUNT(x)       (((uint32_t)(((uint32_t)(x))<<DEC200_gcDECTotalReadsOUT_COUNT_SHIFT))&DEC200_gcDECTotalReadsOUT_COUNT_MASK)
/* gcDECTotalWritesOUT Bit Fields */
#define DEC200_gcDECTotalWritesOUT_COUNT_MASK    0xFFFFFFFFu
#define DEC200_gcDECTotalWritesOUT_COUNT_SHIFT   0u
#define DEC200_gcDECTotalWritesOUT_COUNT_WIDTH   32u
#define DEC200_gcDECTotalWritesOUT_COUNT(x)      (((uint32_t)(((uint32_t)(x))<<DEC200_gcDECTotalWritesOUT_COUNT_SHIFT))&DEC200_gcDECTotalWritesOUT_COUNT_MASK)
/* gcDECTotalReadBurstsOUT Bit Fields */
#define DEC200_gcDECTotalReadBurstsOUT_COUNT_MASK 0xFFFFFFFFu
#define DEC200_gcDECTotalReadBurstsOUT_COUNT_SHIFT 0u
#define DEC200_gcDECTotalReadBurstsOUT_COUNT_WIDTH 32u
#define DEC200_gcDECTotalReadBurstsOUT_COUNT(x)  (((uint32_t)(((uint32_t)(x))<<DEC200_gcDECTotalReadBurstsOUT_COUNT_SHIFT))&DEC200_gcDECTotalReadBurstsOUT_COUNT_MASK)
/* gcDECTotalWriteBurstsOUT Bit Fields */
#define DEC200_gcDECTotalWriteBurstsOUT_COUNT_MASK 0xFFFFFFFFu
#define DEC200_gcDECTotalWriteBurstsOUT_COUNT_SHIFT 0u
#define DEC200_gcDECTotalWriteBurstsOUT_COUNT_WIDTH 32u
#define DEC200_gcDECTotalWriteBurstsOUT_COUNT(x) (((uint32_t)(((uint32_t)(x))<<DEC200_gcDECTotalWriteBurstsOUT_COUNT_SHIFT))&DEC200_gcDECTotalWriteBurstsOUT_COUNT_MASK)
/* gcDECTotalReadsReqOUT Bit Fields */
#define DEC200_gcDECTotalReadsReqOUT_COUNT_MASK  0xFFFFFFFFu
#define DEC200_gcDECTotalReadsReqOUT_COUNT_SHIFT 0u
#define DEC200_gcDECTotalReadsReqOUT_COUNT_WIDTH 32u
#define DEC200_gcDECTotalReadsReqOUT_COUNT(x)    (((uint32_t)(((uint32_t)(x))<<DEC200_gcDECTotalReadsReqOUT_COUNT_SHIFT))&DEC200_gcDECTotalReadsReqOUT_COUNT_MASK)
/* gcDECTotalWritesReqOUT Bit Fields */
#define DEC200_gcDECTotalWritesReqOUT_COUNT_MASK 0xFFFFFFFFu
#define DEC200_gcDECTotalWritesReqOUT_COUNT_SHIFT 0u
#define DEC200_gcDECTotalWritesReqOUT_COUNT_WIDTH 32u
#define DEC200_gcDECTotalWritesReqOUT_COUNT(x)   (((uint32_t)(((uint32_t)(x))<<DEC200_gcDECTotalWritesReqOUT_COUNT_SHIFT))&DEC200_gcDECTotalWritesReqOUT_COUNT_MASK)
/* gcDECTotalReadLastsOUT Bit Fields */
#define DEC200_gcDECTotalReadLastsOUT_COUNT_MASK 0xFFFFFFFFu
#define DEC200_gcDECTotalReadLastsOUT_COUNT_SHIFT 0u
#define DEC200_gcDECTotalReadLastsOUT_COUNT_WIDTH 32u
#define DEC200_gcDECTotalReadLastsOUT_COUNT(x)   (((uint32_t)(((uint32_t)(x))<<DEC200_gcDECTotalReadLastsOUT_COUNT_SHIFT))&DEC200_gcDECTotalReadLastsOUT_COUNT_MASK)
/* gcDECTotalWriteLastsOUT Bit Fields */
#define DEC200_gcDECTotalWriteLastsOUT_COUNT_MASK 0xFFFFFFFFu
#define DEC200_gcDECTotalWriteLastsOUT_COUNT_SHIFT 0u
#define DEC200_gcDECTotalWriteLastsOUT_COUNT_WIDTH 32u
#define DEC200_gcDECTotalWriteLastsOUT_COUNT(x)  (((uint32_t)(((uint32_t)(x))<<DEC200_gcDECTotalWriteLastsOUT_COUNT_SHIFT))&DEC200_gcDECTotalWriteLastsOUT_COUNT_MASK)

/*!
 * @}
 */ /* end of group DEC200_Register_Masks */


/*!
 * @}
 */ /* end of group DEC200_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- DFS Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup DFS_Peripheral_Access_Layer DFS Peripheral Access Layer
 * @{
 */


/** DFS - Size of Registers Arrays */
#define DFS_DVPORT_COUNT                         4u

/** DFS - Register Layout Typedef */
typedef struct {
  __IO uint32_t DLLPRG1;                           /**< DFS DLL Program Register 1, offset: 0x0 */
       uint8_t RESERVED_0[4];
  __IO uint32_t CLKOUTEN;                          /**< DFS Clockout Enable Register, offset: 0x8 */
  __IO uint32_t PORTSR;                            /**< DFS Port Status Register, offset: 0xC */
  __IO uint32_t PORTLOLSR;                         /**< DFS Port Loss of Lock Status Register, offset: 0x10 */
  __IO uint32_t PORTRESET;                         /**< DFS Port Reset register, offset: 0x14 */
  __IO uint32_t CTRL;                              /**< DFS Control Register, offset: 0x18 */
  __IO uint32_t DVPORT[DFS_DVPORT_COUNT];          /**< DFS Divide Register Portn, array offset: 0x1C, array step: 0x4 */
} DFS_Type, *DFS_MemMapPtr;

 /** Number of instances of the DFS module. */
#define DFS_INSTANCE_COUNT                       (3u)


/* DFS - Peripheral instance base addresses */
/** Peripheral DFS_0 base address */
#define DFS_0_BASE                               (0x4003C040u)
/** Peripheral DFS_0 base pointer */
#define DFS_0                                    ((DFS_Type *)DFS_0_BASE)
/** Peripheral DFS_1 base address */
#define DFS_1_BASE                               (0x4003C140u)
/** Peripheral DFS_1 base pointer */
#define DFS_1                                    ((DFS_Type *)DFS_1_BASE)
/** Peripheral DFS_2 base address */
#define DFS_2_BASE                               (0x4003C1C0u)
/** Peripheral DFS_2 base pointer */
#define DFS_2                                    ((DFS_Type *)DFS_2_BASE)
/** Array initializer of DFS peripheral base addresses */
#define DFS_BASE_ADDRS                           { DFS_0_BASE, DFS_1_BASE, DFS_2_BASE }
/** Array initializer of DFS peripheral base pointers */
#define DFS_BASE_PTRS                            { DFS_0, DFS_1, DFS_2 }

/* ----------------------------------------------------------------------------
   -- DFS Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup DFS_Register_Masks DFS Register Masks
 * @{
 */

/* DLLPRG1 Bit Fields */
#define DFS_DLLPRG1_V2IGC_MASK                   0x7u
#define DFS_DLLPRG1_V2IGC_SHIFT                  0u
#define DFS_DLLPRG1_V2IGC_WIDTH                  3u
#define DFS_DLLPRG1_V2IGC(x)                     (((uint32_t)(((uint32_t)(x))<<DFS_DLLPRG1_V2IGC_SHIFT))&DFS_DLLPRG1_V2IGC_MASK)
#define DFS_DLLPRG1_LCKWT_MASK                   0x30u
#define DFS_DLLPRG1_LCKWT_SHIFT                  4u
#define DFS_DLLPRG1_LCKWT_WIDTH                  2u
#define DFS_DLLPRG1_LCKWT(x)                     (((uint32_t)(((uint32_t)(x))<<DFS_DLLPRG1_LCKWT_SHIFT))&DFS_DLLPRG1_LCKWT_MASK)
#define DFS_DLLPRG1_DACIN_MASK                   0x1C0u
#define DFS_DLLPRG1_DACIN_SHIFT                  6u
#define DFS_DLLPRG1_DACIN_WIDTH                  3u
#define DFS_DLLPRG1_DACIN(x)                     (((uint32_t)(((uint32_t)(x))<<DFS_DLLPRG1_DACIN_SHIFT))&DFS_DLLPRG1_DACIN_MASK)
#define DFS_DLLPRG1_CALBYPEN_MASK                0x200u
#define DFS_DLLPRG1_CALBYPEN_SHIFT               9u
#define DFS_DLLPRG1_CALBYPEN_WIDTH               1u
#define DFS_DLLPRG1_CALBYPEN(x)                  (((uint32_t)(((uint32_t)(x))<<DFS_DLLPRG1_CALBYPEN_SHIFT))&DFS_DLLPRG1_CALBYPEN_MASK)
#define DFS_DLLPRG1_VSETTLCTRL_MASK              0xC00u
#define DFS_DLLPRG1_VSETTLCTRL_SHIFT             10u
#define DFS_DLLPRG1_VSETTLCTRL_WIDTH             2u
#define DFS_DLLPRG1_VSETTLCTRL(x)                (((uint32_t)(((uint32_t)(x))<<DFS_DLLPRG1_VSETTLCTRL_SHIFT))&DFS_DLLPRG1_VSETTLCTRL_MASK)
#define DFS_DLLPRG1_CPICTRL_MASK                 0x7000u
#define DFS_DLLPRG1_CPICTRL_SHIFT                12u
#define DFS_DLLPRG1_CPICTRL_WIDTH                3u
#define DFS_DLLPRG1_CPICTRL(x)                   (((uint32_t)(((uint32_t)(x))<<DFS_DLLPRG1_CPICTRL_SHIFT))&DFS_DLLPRG1_CPICTRL_MASK)
/* CLKOUTEN Bit Fields */
#define DFS_CLKOUTEN_CLKOUTEN_MASK               0xFu
#define DFS_CLKOUTEN_CLKOUTEN_SHIFT              0u
#define DFS_CLKOUTEN_CLKOUTEN_WIDTH              4u
#define DFS_CLKOUTEN_CLKOUTEN(x)                 (((uint32_t)(((uint32_t)(x))<<DFS_CLKOUTEN_CLKOUTEN_SHIFT))&DFS_CLKOUTEN_CLKOUTEN_MASK)
/* PORTSR Bit Fields */
#define DFS_PORTSR_PORTSTAT_MASK                 0xFu
#define DFS_PORTSR_PORTSTAT_SHIFT                0u
#define DFS_PORTSR_PORTSTAT_WIDTH                4u
#define DFS_PORTSR_PORTSTAT(x)                   (((uint32_t)(((uint32_t)(x))<<DFS_PORTSR_PORTSTAT_SHIFT))&DFS_PORTSR_PORTSTAT_MASK)
/* PORTLOLSR Bit Fields */
#define DFS_PORTLOLSR_LOLF_MASK                  0xFu
#define DFS_PORTLOLSR_LOLF_SHIFT                 0u
#define DFS_PORTLOLSR_LOLF_WIDTH                 4u
#define DFS_PORTLOLSR_LOLF(x)                    (((uint32_t)(((uint32_t)(x))<<DFS_PORTLOLSR_LOLF_SHIFT))&DFS_PORTLOLSR_LOLF_MASK)
/* PORTRESET Bit Fields */
#define DFS_PORTRESET_PORTRESET_MASK             0xFu
#define DFS_PORTRESET_PORTRESET_SHIFT            0u
#define DFS_PORTRESET_PORTRESET_WIDTH            4u
#define DFS_PORTRESET_PORTRESET(x)               (((uint32_t)(((uint32_t)(x))<<DFS_PORTRESET_PORTRESET_SHIFT))&DFS_PORTRESET_PORTRESET_MASK)
/* CTRL Bit Fields */
#define DFS_CTRL_LOLIE_MASK                      0x1u
#define DFS_CTRL_LOLIE_SHIFT                     0u
#define DFS_CTRL_LOLIE_WIDTH                     1u
#define DFS_CTRL_LOLIE(x)                        (((uint32_t)(((uint32_t)(x))<<DFS_CTRL_LOLIE_SHIFT))&DFS_CTRL_LOLIE_MASK)
#define DFS_CTRL_DLL_RESET_MASK                  0x2u
#define DFS_CTRL_DLL_RESET_SHIFT                 1u
#define DFS_CTRL_DLL_RESET_WIDTH                 1u
#define DFS_CTRL_DLL_RESET(x)                    (((uint32_t)(((uint32_t)(x))<<DFS_CTRL_DLL_RESET_SHIFT))&DFS_CTRL_DLL_RESET_MASK)
/* DVPORT Bit Fields */
#define DFS_DVPORT_MFN_MASK                      0xFFu
#define DFS_DVPORT_MFN_SHIFT                     0u
#define DFS_DVPORT_MFN_WIDTH                     8u
#define DFS_DVPORT_MFN(x)                        (((uint32_t)(((uint32_t)(x))<<DFS_DVPORT_MFN_SHIFT))&DFS_DVPORT_MFN_MASK)
#define DFS_DVPORT_MFI_MASK                      0xFF00u
#define DFS_DVPORT_MFI_SHIFT                     8u
#define DFS_DVPORT_MFI_WIDTH                     8u
#define DFS_DVPORT_MFI(x)                        (((uint32_t)(((uint32_t)(x))<<DFS_DVPORT_MFI_SHIFT))&DFS_DVPORT_MFI_MASK)

/*!
 * @}
 */ /* end of group DFS_Register_Masks */


/*!
 * @}
 */ /* end of group DFS_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- DMA Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup DMA_Peripheral_Access_Layer DMA Peripheral Access Layer
 * @{
 */


/** DMA - Size of Registers Arrays */
#define DMA_GPOR_COUNT                           2u
#define DMA_DCHPRI_COUNT                         32u
#define DMA_DCHMID_COUNT                         32u
#define DMA_TCD_COUNT                            32u

/** DMA - Register Layout Typedef */
typedef struct {
  __IO uint32_t CR;                                /**< Control Register, offset: 0x0 */
  __I  uint32_t ES;                                /**< Error Status Register, offset: 0x4 */
       uint8_t RESERVED_0[4];
  __IO uint32_t ERQ;                               /**< Enable Request Register, offset: 0xC */
       uint8_t RESERVED_1[4];
  __IO uint32_t EEI;                               /**< Enable Error Interrupt Register, offset: 0x14 */
  __O  uint8_t CEEI;                               /**< Clear Enable Error Interrupt Register, offset: 0x18 */
  __O  uint8_t SEEI;                               /**< Set Enable Error Interrupt Register, offset: 0x19 */
  __O  uint8_t CERQ;                               /**< Clear Enable Request Register, offset: 0x1A */
  __O  uint8_t SERQ;                               /**< Set Enable Request Register, offset: 0x1B */
  __O  uint8_t CDNE;                               /**< Clear DONE Status Bit Register, offset: 0x1C */
  __O  uint8_t SSRT;                               /**< Set START Bit Register, offset: 0x1D */
  __O  uint8_t CERR;                               /**< Clear Error Register, offset: 0x1E */
  __O  uint8_t CINT;                               /**< Clear Interrupt Request Register, offset: 0x1F */
       uint8_t RESERVED_2[4];
  __IO uint32_t INT;                               /**< Interrupt Request Register, offset: 0x24 */
       uint8_t RESERVED_3[4];
  __IO uint32_t ERR;                               /**< Error Register, offset: 0x2C */
       uint8_t RESERVED_4[4];
  __I  uint32_t HRS;                               /**< Hardware Request Status Register, offset: 0x34 */
  __IO uint32_t GPOR[DMA_GPOR_COUNT];              /**< General-Purpose Output Register, array offset: 0x38, array step: 0x4 */
       uint8_t RESERVED_5[192];
  __IO uint8_t DCHPRI[DMA_DCHPRI_COUNT];           /**< Channel n Priority Register, array offset: 0x100, array step: 0x1 */
       uint8_t RESERVED_6[32];
  __IO uint8_t DCHMID[DMA_DCHMID_COUNT];           /**< Channel n Master ID Register, array offset: 0x140, array step: 0x1 */
       uint8_t RESERVED_7[3744];
  struct {                                         /* offset: 0x1000, array step: 0x20 */
    __IO uint32_t SADDR;                             /**< TCD Source Address, array offset: 0x1000, array step: 0x20 */
    __IO uint16_t SOFF;                              /**< TCD Signed Source Address Offset, array offset: 0x1004, array step: 0x20 */
    __IO uint16_t ATTR;                              /**< TCD Transfer Attributes, array offset: 0x1006, array step: 0x20 */
    union {                                          /* offset: 0x1008, array step: 0x20 */
      __IO uint32_t MLNO;                              /**< TCD Minor Byte Count (Minor Loop Mapping Disabled), array offset: 0x1008, array step: 0x20 */
      __IO uint32_t MLOFFNO;                           /**< TCD Signed Minor Loop Offset (Minor Loop Mapping Enabled and Offset Disabled), array offset: 0x1008, array step: 0x20 */
      __IO uint32_t MLOFFYES;                          /**< TCD Signed Minor Loop Offset (Minor Loop Mapping and Offset Enabled), array offset: 0x1008, array step: 0x20 */
    } NBYTES;
    __IO uint32_t SLAST;                             /**< TCD Last Source Address Adjustment, array offset: 0x100C, array step: 0x20 */
    __IO uint32_t DADDR;                             /**< TCD Destination Address, array offset: 0x1010, array step: 0x20 */
    __IO uint16_t DOFF;                              /**< TCD Signed Destination Address Offset, array offset: 0x1014, array step: 0x20 */
    union {                                          /* offset: 0x1016, array step: 0x20 */
      __IO uint16_t ELINKNO;                           /**< TCD Current Minor Loop Link, Major Loop Count (Channel Linking Disabled), array offset: 0x1016, array step: 0x20 */
      __IO uint16_t ELINKYES;                          /**< TCD Current Minor Loop Link, Major Loop Count (Channel Linking Enabled), array offset: 0x1016, array step: 0x20 */
    } CITER;
    __IO uint32_t DLASTSGA;                          /**< TCD Last Destination Address Adjustment/Scatter Gather Address, array offset: 0x1018, array step: 0x20 */
    __IO uint16_t CSR;                               /**< TCD Control and Status, array offset: 0x101C, array step: 0x20 */
    union {                                          /* offset: 0x101E, array step: 0x20 */
      __IO uint16_t ELINKNO;                           /**< TCD Beginning Minor Loop Link, Major Loop Count (Channel Linking Disabled), array offset: 0x101E, array step: 0x20 */
      __IO uint16_t ELINKYES;                          /**< TCD Beginning Minor Loop Link, Major Loop Count (Channel Linking Enabled), array offset: 0x101E, array step: 0x20 */
    } BITER;
  } TCD[DMA_TCD_COUNT];
} DMA_Type, *DMA_MemMapPtr;

 /** Number of instances of the DMA module. */
#define DMA_INSTANCE_COUNT                       (1u)


/* DMA - Peripheral instance base addresses */
/** Peripheral DMA base address */
#define DMA_BASE                                 (0x40002000u)
/** Peripheral DMA base pointer */
#define DMA                                      ((DMA_Type *)DMA_BASE)
/** Array initializer of DMA peripheral base addresses */
#define DMA_BASE_ADDRS                           { DMA_BASE }
/** Array initializer of DMA peripheral base pointers */
#define DMA_BASE_PTRS                            { DMA }

/** Interrupt vectors for the DMA peripheral type */
#define DMA_CHN_IRQS                             { DMA0_15_IRQn, DMA16_31_IRQn}
#define DMA_ERROR_IRQS                           { DMA_Error_IRQn }

/* ----------------------------------------------------------------------------
   -- DMA Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup DMA_Register_Masks DMA Register Masks
 * @{
 */

/* CR Bit Fields */
#define DMA_CR_EDBG_MASK                         0x2u
#define DMA_CR_EDBG_SHIFT                        1u
#define DMA_CR_EDBG_WIDTH                        1u
#define DMA_CR_EDBG(x)                           (((uint32_t)(((uint32_t)(x))<<DMA_CR_EDBG_SHIFT))&DMA_CR_EDBG_MASK)
#define DMA_CR_ERCA_MASK                         0x4u
#define DMA_CR_ERCA_SHIFT                        2u
#define DMA_CR_ERCA_WIDTH                        1u
#define DMA_CR_ERCA(x)                           (((uint32_t)(((uint32_t)(x))<<DMA_CR_ERCA_SHIFT))&DMA_CR_ERCA_MASK)
#define DMA_CR_ERGA_MASK                         0x8u
#define DMA_CR_ERGA_SHIFT                        3u
#define DMA_CR_ERGA_WIDTH                        1u
#define DMA_CR_ERGA(x)                           (((uint32_t)(((uint32_t)(x))<<DMA_CR_ERGA_SHIFT))&DMA_CR_ERGA_MASK)
#define DMA_CR_HOE_MASK                          0x10u
#define DMA_CR_HOE_SHIFT                         4u
#define DMA_CR_HOE_WIDTH                         1u
#define DMA_CR_HOE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_CR_HOE_SHIFT))&DMA_CR_HOE_MASK)
#define DMA_CR_HALT_MASK                         0x20u
#define DMA_CR_HALT_SHIFT                        5u
#define DMA_CR_HALT_WIDTH                        1u
#define DMA_CR_HALT(x)                           (((uint32_t)(((uint32_t)(x))<<DMA_CR_HALT_SHIFT))&DMA_CR_HALT_MASK)
#define DMA_CR_CLM_MASK                          0x40u
#define DMA_CR_CLM_SHIFT                         6u
#define DMA_CR_CLM_WIDTH                         1u
#define DMA_CR_CLM(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_CR_CLM_SHIFT))&DMA_CR_CLM_MASK)
#define DMA_CR_EMLM_MASK                         0x80u
#define DMA_CR_EMLM_SHIFT                        7u
#define DMA_CR_EMLM_WIDTH                        1u
#define DMA_CR_EMLM(x)                           (((uint32_t)(((uint32_t)(x))<<DMA_CR_EMLM_SHIFT))&DMA_CR_EMLM_MASK)
#define DMA_CR_GRP0PRI_MASK                      0x100u
#define DMA_CR_GRP0PRI_SHIFT                     8u
#define DMA_CR_GRP0PRI_WIDTH                     1u
#define DMA_CR_GRP0PRI(x)                        (((uint32_t)(((uint32_t)(x))<<DMA_CR_GRP0PRI_SHIFT))&DMA_CR_GRP0PRI_MASK)
#define DMA_CR_GRP1PRI_MASK                      0x400u
#define DMA_CR_GRP1PRI_SHIFT                     10u
#define DMA_CR_GRP1PRI_WIDTH                     1u
#define DMA_CR_GRP1PRI(x)                        (((uint32_t)(((uint32_t)(x))<<DMA_CR_GRP1PRI_SHIFT))&DMA_CR_GRP1PRI_MASK)
#define DMA_CR_ECX_MASK                          0x10000u
#define DMA_CR_ECX_SHIFT                         16u
#define DMA_CR_ECX_WIDTH                         1u
#define DMA_CR_ECX(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_CR_ECX_SHIFT))&DMA_CR_ECX_MASK)
#define DMA_CR_CX_MASK                           0x20000u
#define DMA_CR_CX_SHIFT                          17u
#define DMA_CR_CX_WIDTH                          1u
#define DMA_CR_CX(x)                             (((uint32_t)(((uint32_t)(x))<<DMA_CR_CX_SHIFT))&DMA_CR_CX_MASK)
/* ES Bit Fields */
#define DMA_ES_DBE_MASK                          0x1u
#define DMA_ES_DBE_SHIFT                         0u
#define DMA_ES_DBE_WIDTH                         1u
#define DMA_ES_DBE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_DBE_SHIFT))&DMA_ES_DBE_MASK)
#define DMA_ES_SBE_MASK                          0x2u
#define DMA_ES_SBE_SHIFT                         1u
#define DMA_ES_SBE_WIDTH                         1u
#define DMA_ES_SBE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_SBE_SHIFT))&DMA_ES_SBE_MASK)
#define DMA_ES_SGE_MASK                          0x4u
#define DMA_ES_SGE_SHIFT                         2u
#define DMA_ES_SGE_WIDTH                         1u
#define DMA_ES_SGE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_SGE_SHIFT))&DMA_ES_SGE_MASK)
#define DMA_ES_NCE_MASK                          0x8u
#define DMA_ES_NCE_SHIFT                         3u
#define DMA_ES_NCE_WIDTH                         1u
#define DMA_ES_NCE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_NCE_SHIFT))&DMA_ES_NCE_MASK)
#define DMA_ES_DOE_MASK                          0x10u
#define DMA_ES_DOE_SHIFT                         4u
#define DMA_ES_DOE_WIDTH                         1u
#define DMA_ES_DOE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_DOE_SHIFT))&DMA_ES_DOE_MASK)
#define DMA_ES_DAE_MASK                          0x20u
#define DMA_ES_DAE_SHIFT                         5u
#define DMA_ES_DAE_WIDTH                         1u
#define DMA_ES_DAE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_DAE_SHIFT))&DMA_ES_DAE_MASK)
#define DMA_ES_SOE_MASK                          0x40u
#define DMA_ES_SOE_SHIFT                         6u
#define DMA_ES_SOE_WIDTH                         1u
#define DMA_ES_SOE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_SOE_SHIFT))&DMA_ES_SOE_MASK)
#define DMA_ES_SAE_MASK                          0x80u
#define DMA_ES_SAE_SHIFT                         7u
#define DMA_ES_SAE_WIDTH                         1u
#define DMA_ES_SAE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_SAE_SHIFT))&DMA_ES_SAE_MASK)
#define DMA_ES_ERRCHN_MASK                       0x1F00u
#define DMA_ES_ERRCHN_SHIFT                      8u
#define DMA_ES_ERRCHN_WIDTH                      5u
#define DMA_ES_ERRCHN(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ES_ERRCHN_SHIFT))&DMA_ES_ERRCHN_MASK)
#define DMA_ES_CPE_MASK                          0x4000u
#define DMA_ES_CPE_SHIFT                         14u
#define DMA_ES_CPE_WIDTH                         1u
#define DMA_ES_CPE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_CPE_SHIFT))&DMA_ES_CPE_MASK)
#define DMA_ES_GPE_MASK                          0x8000u
#define DMA_ES_GPE_SHIFT                         15u
#define DMA_ES_GPE_WIDTH                         1u
#define DMA_ES_GPE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_GPE_SHIFT))&DMA_ES_GPE_MASK)
#define DMA_ES_ECX_MASK                          0x10000u
#define DMA_ES_ECX_SHIFT                         16u
#define DMA_ES_ECX_WIDTH                         1u
#define DMA_ES_ECX(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_ECX_SHIFT))&DMA_ES_ECX_MASK)
#define DMA_ES_UCE_MASK                          0x20000u
#define DMA_ES_UCE_SHIFT                         17u
#define DMA_ES_UCE_WIDTH                         1u
#define DMA_ES_UCE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_UCE_SHIFT))&DMA_ES_UCE_MASK)
#define DMA_ES_VLD_MASK                          0x80000000u
#define DMA_ES_VLD_SHIFT                         31u
#define DMA_ES_VLD_WIDTH                         1u
#define DMA_ES_VLD(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_VLD_SHIFT))&DMA_ES_VLD_MASK)
/* ERQ Bit Fields */
#define DMA_ERQ_ERQ0_MASK                        0x1u
#define DMA_ERQ_ERQ0_SHIFT                       0u
#define DMA_ERQ_ERQ0_WIDTH                       1u
#define DMA_ERQ_ERQ0(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ0_SHIFT))&DMA_ERQ_ERQ0_MASK)
#define DMA_ERQ_ERQ1_MASK                        0x2u
#define DMA_ERQ_ERQ1_SHIFT                       1u
#define DMA_ERQ_ERQ1_WIDTH                       1u
#define DMA_ERQ_ERQ1(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ1_SHIFT))&DMA_ERQ_ERQ1_MASK)
#define DMA_ERQ_ERQ2_MASK                        0x4u
#define DMA_ERQ_ERQ2_SHIFT                       2u
#define DMA_ERQ_ERQ2_WIDTH                       1u
#define DMA_ERQ_ERQ2(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ2_SHIFT))&DMA_ERQ_ERQ2_MASK)
#define DMA_ERQ_ERQ3_MASK                        0x8u
#define DMA_ERQ_ERQ3_SHIFT                       3u
#define DMA_ERQ_ERQ3_WIDTH                       1u
#define DMA_ERQ_ERQ3(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ3_SHIFT))&DMA_ERQ_ERQ3_MASK)
#define DMA_ERQ_ERQ4_MASK                        0x10u
#define DMA_ERQ_ERQ4_SHIFT                       4u
#define DMA_ERQ_ERQ4_WIDTH                       1u
#define DMA_ERQ_ERQ4(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ4_SHIFT))&DMA_ERQ_ERQ4_MASK)
#define DMA_ERQ_ERQ5_MASK                        0x20u
#define DMA_ERQ_ERQ5_SHIFT                       5u
#define DMA_ERQ_ERQ5_WIDTH                       1u
#define DMA_ERQ_ERQ5(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ5_SHIFT))&DMA_ERQ_ERQ5_MASK)
#define DMA_ERQ_ERQ6_MASK                        0x40u
#define DMA_ERQ_ERQ6_SHIFT                       6u
#define DMA_ERQ_ERQ6_WIDTH                       1u
#define DMA_ERQ_ERQ6(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ6_SHIFT))&DMA_ERQ_ERQ6_MASK)
#define DMA_ERQ_ERQ7_MASK                        0x80u
#define DMA_ERQ_ERQ7_SHIFT                       7u
#define DMA_ERQ_ERQ7_WIDTH                       1u
#define DMA_ERQ_ERQ7(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ7_SHIFT))&DMA_ERQ_ERQ7_MASK)
#define DMA_ERQ_ERQ8_MASK                        0x100u
#define DMA_ERQ_ERQ8_SHIFT                       8u
#define DMA_ERQ_ERQ8_WIDTH                       1u
#define DMA_ERQ_ERQ8(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ8_SHIFT))&DMA_ERQ_ERQ8_MASK)
#define DMA_ERQ_ERQ9_MASK                        0x200u
#define DMA_ERQ_ERQ9_SHIFT                       9u
#define DMA_ERQ_ERQ9_WIDTH                       1u
#define DMA_ERQ_ERQ9(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ9_SHIFT))&DMA_ERQ_ERQ9_MASK)
#define DMA_ERQ_ERQ10_MASK                       0x400u
#define DMA_ERQ_ERQ10_SHIFT                      10u
#define DMA_ERQ_ERQ10_WIDTH                      1u
#define DMA_ERQ_ERQ10(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ10_SHIFT))&DMA_ERQ_ERQ10_MASK)
#define DMA_ERQ_ERQ11_MASK                       0x800u
#define DMA_ERQ_ERQ11_SHIFT                      11u
#define DMA_ERQ_ERQ11_WIDTH                      1u
#define DMA_ERQ_ERQ11(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ11_SHIFT))&DMA_ERQ_ERQ11_MASK)
#define DMA_ERQ_ERQ12_MASK                       0x1000u
#define DMA_ERQ_ERQ12_SHIFT                      12u
#define DMA_ERQ_ERQ12_WIDTH                      1u
#define DMA_ERQ_ERQ12(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ12_SHIFT))&DMA_ERQ_ERQ12_MASK)
#define DMA_ERQ_ERQ13_MASK                       0x2000u
#define DMA_ERQ_ERQ13_SHIFT                      13u
#define DMA_ERQ_ERQ13_WIDTH                      1u
#define DMA_ERQ_ERQ13(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ13_SHIFT))&DMA_ERQ_ERQ13_MASK)
#define DMA_ERQ_ERQ14_MASK                       0x4000u
#define DMA_ERQ_ERQ14_SHIFT                      14u
#define DMA_ERQ_ERQ14_WIDTH                      1u
#define DMA_ERQ_ERQ14(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ14_SHIFT))&DMA_ERQ_ERQ14_MASK)
#define DMA_ERQ_ERQ15_MASK                       0x8000u
#define DMA_ERQ_ERQ15_SHIFT                      15u
#define DMA_ERQ_ERQ15_WIDTH                      1u
#define DMA_ERQ_ERQ15(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ15_SHIFT))&DMA_ERQ_ERQ15_MASK)
#define DMA_ERQ_ERQ16_MASK                       0x10000u
#define DMA_ERQ_ERQ16_SHIFT                      16u
#define DMA_ERQ_ERQ16_WIDTH                      1u
#define DMA_ERQ_ERQ16(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ16_SHIFT))&DMA_ERQ_ERQ16_MASK)
#define DMA_ERQ_ERQ17_MASK                       0x20000u
#define DMA_ERQ_ERQ17_SHIFT                      17u
#define DMA_ERQ_ERQ17_WIDTH                      1u
#define DMA_ERQ_ERQ17(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ17_SHIFT))&DMA_ERQ_ERQ17_MASK)
#define DMA_ERQ_ERQ18_MASK                       0x40000u
#define DMA_ERQ_ERQ18_SHIFT                      18u
#define DMA_ERQ_ERQ18_WIDTH                      1u
#define DMA_ERQ_ERQ18(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ18_SHIFT))&DMA_ERQ_ERQ18_MASK)
#define DMA_ERQ_ERQ19_MASK                       0x80000u
#define DMA_ERQ_ERQ19_SHIFT                      19u
#define DMA_ERQ_ERQ19_WIDTH                      1u
#define DMA_ERQ_ERQ19(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ19_SHIFT))&DMA_ERQ_ERQ19_MASK)
#define DMA_ERQ_ERQ20_MASK                       0x100000u
#define DMA_ERQ_ERQ20_SHIFT                      20u
#define DMA_ERQ_ERQ20_WIDTH                      1u
#define DMA_ERQ_ERQ20(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ20_SHIFT))&DMA_ERQ_ERQ20_MASK)
#define DMA_ERQ_ERQ21_MASK                       0x200000u
#define DMA_ERQ_ERQ21_SHIFT                      21u
#define DMA_ERQ_ERQ21_WIDTH                      1u
#define DMA_ERQ_ERQ21(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ21_SHIFT))&DMA_ERQ_ERQ21_MASK)
#define DMA_ERQ_ERQ22_MASK                       0x400000u
#define DMA_ERQ_ERQ22_SHIFT                      22u
#define DMA_ERQ_ERQ22_WIDTH                      1u
#define DMA_ERQ_ERQ22(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ22_SHIFT))&DMA_ERQ_ERQ22_MASK)
#define DMA_ERQ_ERQ23_MASK                       0x800000u
#define DMA_ERQ_ERQ23_SHIFT                      23u
#define DMA_ERQ_ERQ23_WIDTH                      1u
#define DMA_ERQ_ERQ23(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ23_SHIFT))&DMA_ERQ_ERQ23_MASK)
#define DMA_ERQ_ERQ24_MASK                       0x1000000u
#define DMA_ERQ_ERQ24_SHIFT                      24u
#define DMA_ERQ_ERQ24_WIDTH                      1u
#define DMA_ERQ_ERQ24(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ24_SHIFT))&DMA_ERQ_ERQ24_MASK)
#define DMA_ERQ_ERQ25_MASK                       0x2000000u
#define DMA_ERQ_ERQ25_SHIFT                      25u
#define DMA_ERQ_ERQ25_WIDTH                      1u
#define DMA_ERQ_ERQ25(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ25_SHIFT))&DMA_ERQ_ERQ25_MASK)
#define DMA_ERQ_ERQ26_MASK                       0x4000000u
#define DMA_ERQ_ERQ26_SHIFT                      26u
#define DMA_ERQ_ERQ26_WIDTH                      1u
#define DMA_ERQ_ERQ26(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ26_SHIFT))&DMA_ERQ_ERQ26_MASK)
#define DMA_ERQ_ERQ27_MASK                       0x8000000u
#define DMA_ERQ_ERQ27_SHIFT                      27u
#define DMA_ERQ_ERQ27_WIDTH                      1u
#define DMA_ERQ_ERQ27(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ27_SHIFT))&DMA_ERQ_ERQ27_MASK)
#define DMA_ERQ_ERQ28_MASK                       0x10000000u
#define DMA_ERQ_ERQ28_SHIFT                      28u
#define DMA_ERQ_ERQ28_WIDTH                      1u
#define DMA_ERQ_ERQ28(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ28_SHIFT))&DMA_ERQ_ERQ28_MASK)
#define DMA_ERQ_ERQ29_MASK                       0x20000000u
#define DMA_ERQ_ERQ29_SHIFT                      29u
#define DMA_ERQ_ERQ29_WIDTH                      1u
#define DMA_ERQ_ERQ29(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ29_SHIFT))&DMA_ERQ_ERQ29_MASK)
#define DMA_ERQ_ERQ30_MASK                       0x40000000u
#define DMA_ERQ_ERQ30_SHIFT                      30u
#define DMA_ERQ_ERQ30_WIDTH                      1u
#define DMA_ERQ_ERQ30(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ30_SHIFT))&DMA_ERQ_ERQ30_MASK)
#define DMA_ERQ_ERQ31_MASK                       0x80000000u
#define DMA_ERQ_ERQ31_SHIFT                      31u
#define DMA_ERQ_ERQ31_WIDTH                      1u
#define DMA_ERQ_ERQ31(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ31_SHIFT))&DMA_ERQ_ERQ31_MASK)
/* EEI Bit Fields */
#define DMA_EEI_EEI0_MASK                        0x1u
#define DMA_EEI_EEI0_SHIFT                       0u
#define DMA_EEI_EEI0_WIDTH                       1u
#define DMA_EEI_EEI0(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI0_SHIFT))&DMA_EEI_EEI0_MASK)
#define DMA_EEI_EEI1_MASK                        0x2u
#define DMA_EEI_EEI1_SHIFT                       1u
#define DMA_EEI_EEI1_WIDTH                       1u
#define DMA_EEI_EEI1(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI1_SHIFT))&DMA_EEI_EEI1_MASK)
#define DMA_EEI_EEI2_MASK                        0x4u
#define DMA_EEI_EEI2_SHIFT                       2u
#define DMA_EEI_EEI2_WIDTH                       1u
#define DMA_EEI_EEI2(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI2_SHIFT))&DMA_EEI_EEI2_MASK)
#define DMA_EEI_EEI3_MASK                        0x8u
#define DMA_EEI_EEI3_SHIFT                       3u
#define DMA_EEI_EEI3_WIDTH                       1u
#define DMA_EEI_EEI3(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI3_SHIFT))&DMA_EEI_EEI3_MASK)
#define DMA_EEI_EEI4_MASK                        0x10u
#define DMA_EEI_EEI4_SHIFT                       4u
#define DMA_EEI_EEI4_WIDTH                       1u
#define DMA_EEI_EEI4(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI4_SHIFT))&DMA_EEI_EEI4_MASK)
#define DMA_EEI_EEI5_MASK                        0x20u
#define DMA_EEI_EEI5_SHIFT                       5u
#define DMA_EEI_EEI5_WIDTH                       1u
#define DMA_EEI_EEI5(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI5_SHIFT))&DMA_EEI_EEI5_MASK)
#define DMA_EEI_EEI6_MASK                        0x40u
#define DMA_EEI_EEI6_SHIFT                       6u
#define DMA_EEI_EEI6_WIDTH                       1u
#define DMA_EEI_EEI6(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI6_SHIFT))&DMA_EEI_EEI6_MASK)
#define DMA_EEI_EEI7_MASK                        0x80u
#define DMA_EEI_EEI7_SHIFT                       7u
#define DMA_EEI_EEI7_WIDTH                       1u
#define DMA_EEI_EEI7(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI7_SHIFT))&DMA_EEI_EEI7_MASK)
#define DMA_EEI_EEI8_MASK                        0x100u
#define DMA_EEI_EEI8_SHIFT                       8u
#define DMA_EEI_EEI8_WIDTH                       1u
#define DMA_EEI_EEI8(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI8_SHIFT))&DMA_EEI_EEI8_MASK)
#define DMA_EEI_EEI9_MASK                        0x200u
#define DMA_EEI_EEI9_SHIFT                       9u
#define DMA_EEI_EEI9_WIDTH                       1u
#define DMA_EEI_EEI9(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI9_SHIFT))&DMA_EEI_EEI9_MASK)
#define DMA_EEI_EEI10_MASK                       0x400u
#define DMA_EEI_EEI10_SHIFT                      10u
#define DMA_EEI_EEI10_WIDTH                      1u
#define DMA_EEI_EEI10(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI10_SHIFT))&DMA_EEI_EEI10_MASK)
#define DMA_EEI_EEI11_MASK                       0x800u
#define DMA_EEI_EEI11_SHIFT                      11u
#define DMA_EEI_EEI11_WIDTH                      1u
#define DMA_EEI_EEI11(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI11_SHIFT))&DMA_EEI_EEI11_MASK)
#define DMA_EEI_EEI12_MASK                       0x1000u
#define DMA_EEI_EEI12_SHIFT                      12u
#define DMA_EEI_EEI12_WIDTH                      1u
#define DMA_EEI_EEI12(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI12_SHIFT))&DMA_EEI_EEI12_MASK)
#define DMA_EEI_EEI13_MASK                       0x2000u
#define DMA_EEI_EEI13_SHIFT                      13u
#define DMA_EEI_EEI13_WIDTH                      1u
#define DMA_EEI_EEI13(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI13_SHIFT))&DMA_EEI_EEI13_MASK)
#define DMA_EEI_EEI14_MASK                       0x4000u
#define DMA_EEI_EEI14_SHIFT                      14u
#define DMA_EEI_EEI14_WIDTH                      1u
#define DMA_EEI_EEI14(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI14_SHIFT))&DMA_EEI_EEI14_MASK)
#define DMA_EEI_EEI15_MASK                       0x8000u
#define DMA_EEI_EEI15_SHIFT                      15u
#define DMA_EEI_EEI15_WIDTH                      1u
#define DMA_EEI_EEI15(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI15_SHIFT))&DMA_EEI_EEI15_MASK)
#define DMA_EEI_EEI16_MASK                       0x10000u
#define DMA_EEI_EEI16_SHIFT                      16u
#define DMA_EEI_EEI16_WIDTH                      1u
#define DMA_EEI_EEI16(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI16_SHIFT))&DMA_EEI_EEI16_MASK)
#define DMA_EEI_EEI17_MASK                       0x20000u
#define DMA_EEI_EEI17_SHIFT                      17u
#define DMA_EEI_EEI17_WIDTH                      1u
#define DMA_EEI_EEI17(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI17_SHIFT))&DMA_EEI_EEI17_MASK)
#define DMA_EEI_EEI18_MASK                       0x40000u
#define DMA_EEI_EEI18_SHIFT                      18u
#define DMA_EEI_EEI18_WIDTH                      1u
#define DMA_EEI_EEI18(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI18_SHIFT))&DMA_EEI_EEI18_MASK)
#define DMA_EEI_EEI19_MASK                       0x80000u
#define DMA_EEI_EEI19_SHIFT                      19u
#define DMA_EEI_EEI19_WIDTH                      1u
#define DMA_EEI_EEI19(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI19_SHIFT))&DMA_EEI_EEI19_MASK)
#define DMA_EEI_EEI20_MASK                       0x100000u
#define DMA_EEI_EEI20_SHIFT                      20u
#define DMA_EEI_EEI20_WIDTH                      1u
#define DMA_EEI_EEI20(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI20_SHIFT))&DMA_EEI_EEI20_MASK)
#define DMA_EEI_EEI21_MASK                       0x200000u
#define DMA_EEI_EEI21_SHIFT                      21u
#define DMA_EEI_EEI21_WIDTH                      1u
#define DMA_EEI_EEI21(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI21_SHIFT))&DMA_EEI_EEI21_MASK)
#define DMA_EEI_EEI22_MASK                       0x400000u
#define DMA_EEI_EEI22_SHIFT                      22u
#define DMA_EEI_EEI22_WIDTH                      1u
#define DMA_EEI_EEI22(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI22_SHIFT))&DMA_EEI_EEI22_MASK)
#define DMA_EEI_EEI23_MASK                       0x800000u
#define DMA_EEI_EEI23_SHIFT                      23u
#define DMA_EEI_EEI23_WIDTH                      1u
#define DMA_EEI_EEI23(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI23_SHIFT))&DMA_EEI_EEI23_MASK)
#define DMA_EEI_EEI24_MASK                       0x1000000u
#define DMA_EEI_EEI24_SHIFT                      24u
#define DMA_EEI_EEI24_WIDTH                      1u
#define DMA_EEI_EEI24(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI24_SHIFT))&DMA_EEI_EEI24_MASK)
#define DMA_EEI_EEI25_MASK                       0x2000000u
#define DMA_EEI_EEI25_SHIFT                      25u
#define DMA_EEI_EEI25_WIDTH                      1u
#define DMA_EEI_EEI25(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI25_SHIFT))&DMA_EEI_EEI25_MASK)
#define DMA_EEI_EEI26_MASK                       0x4000000u
#define DMA_EEI_EEI26_SHIFT                      26u
#define DMA_EEI_EEI26_WIDTH                      1u
#define DMA_EEI_EEI26(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI26_SHIFT))&DMA_EEI_EEI26_MASK)
#define DMA_EEI_EEI27_MASK                       0x8000000u
#define DMA_EEI_EEI27_SHIFT                      27u
#define DMA_EEI_EEI27_WIDTH                      1u
#define DMA_EEI_EEI27(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI27_SHIFT))&DMA_EEI_EEI27_MASK)
#define DMA_EEI_EEI28_MASK                       0x10000000u
#define DMA_EEI_EEI28_SHIFT                      28u
#define DMA_EEI_EEI28_WIDTH                      1u
#define DMA_EEI_EEI28(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI28_SHIFT))&DMA_EEI_EEI28_MASK)
#define DMA_EEI_EEI29_MASK                       0x20000000u
#define DMA_EEI_EEI29_SHIFT                      29u
#define DMA_EEI_EEI29_WIDTH                      1u
#define DMA_EEI_EEI29(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI29_SHIFT))&DMA_EEI_EEI29_MASK)
#define DMA_EEI_EEI30_MASK                       0x40000000u
#define DMA_EEI_EEI30_SHIFT                      30u
#define DMA_EEI_EEI30_WIDTH                      1u
#define DMA_EEI_EEI30(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI30_SHIFT))&DMA_EEI_EEI30_MASK)
#define DMA_EEI_EEI31_MASK                       0x80000000u
#define DMA_EEI_EEI31_SHIFT                      31u
#define DMA_EEI_EEI31_WIDTH                      1u
#define DMA_EEI_EEI31(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI31_SHIFT))&DMA_EEI_EEI31_MASK)
/* CEEI Bit Fields */
#define DMA_CEEI_CEEI_MASK                       0x1Fu
#define DMA_CEEI_CEEI_SHIFT                      0u
#define DMA_CEEI_CEEI_WIDTH                      5u
#define DMA_CEEI_CEEI(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_CEEI_CEEI_SHIFT))&DMA_CEEI_CEEI_MASK)
#define DMA_CEEI_CAEE_MASK                       0x40u
#define DMA_CEEI_CAEE_SHIFT                      6u
#define DMA_CEEI_CAEE_WIDTH                      1u
#define DMA_CEEI_CAEE(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_CEEI_CAEE_SHIFT))&DMA_CEEI_CAEE_MASK)
#define DMA_CEEI_NOP_MASK                        0x80u
#define DMA_CEEI_NOP_SHIFT                       7u
#define DMA_CEEI_NOP_WIDTH                       1u
#define DMA_CEEI_NOP(x)                          (((uint8_t)(((uint8_t)(x))<<DMA_CEEI_NOP_SHIFT))&DMA_CEEI_NOP_MASK)
/* SEEI Bit Fields */
#define DMA_SEEI_SEEI_MASK                       0x1Fu
#define DMA_SEEI_SEEI_SHIFT                      0u
#define DMA_SEEI_SEEI_WIDTH                      5u
#define DMA_SEEI_SEEI(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_SEEI_SEEI_SHIFT))&DMA_SEEI_SEEI_MASK)
#define DMA_SEEI_SAEE_MASK                       0x40u
#define DMA_SEEI_SAEE_SHIFT                      6u
#define DMA_SEEI_SAEE_WIDTH                      1u
#define DMA_SEEI_SAEE(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_SEEI_SAEE_SHIFT))&DMA_SEEI_SAEE_MASK)
#define DMA_SEEI_NOP_MASK                        0x80u
#define DMA_SEEI_NOP_SHIFT                       7u
#define DMA_SEEI_NOP_WIDTH                       1u
#define DMA_SEEI_NOP(x)                          (((uint8_t)(((uint8_t)(x))<<DMA_SEEI_NOP_SHIFT))&DMA_SEEI_NOP_MASK)
/* CERQ Bit Fields */
#define DMA_CERQ_CERQ_MASK                       0x1Fu
#define DMA_CERQ_CERQ_SHIFT                      0u
#define DMA_CERQ_CERQ_WIDTH                      5u
#define DMA_CERQ_CERQ(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_CERQ_CERQ_SHIFT))&DMA_CERQ_CERQ_MASK)
#define DMA_CERQ_CAER_MASK                       0x40u
#define DMA_CERQ_CAER_SHIFT                      6u
#define DMA_CERQ_CAER_WIDTH                      1u
#define DMA_CERQ_CAER(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_CERQ_CAER_SHIFT))&DMA_CERQ_CAER_MASK)
#define DMA_CERQ_NOP_MASK                        0x80u
#define DMA_CERQ_NOP_SHIFT                       7u
#define DMA_CERQ_NOP_WIDTH                       1u
#define DMA_CERQ_NOP(x)                          (((uint8_t)(((uint8_t)(x))<<DMA_CERQ_NOP_SHIFT))&DMA_CERQ_NOP_MASK)
/* SERQ Bit Fields */
#define DMA_SERQ_SERQ_MASK                       0x1Fu
#define DMA_SERQ_SERQ_SHIFT                      0u
#define DMA_SERQ_SERQ_WIDTH                      5u
#define DMA_SERQ_SERQ(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_SERQ_SERQ_SHIFT))&DMA_SERQ_SERQ_MASK)
#define DMA_SERQ_SAER_MASK                       0x40u
#define DMA_SERQ_SAER_SHIFT                      6u
#define DMA_SERQ_SAER_WIDTH                      1u
#define DMA_SERQ_SAER(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_SERQ_SAER_SHIFT))&DMA_SERQ_SAER_MASK)
#define DMA_SERQ_NOP_MASK                        0x80u
#define DMA_SERQ_NOP_SHIFT                       7u
#define DMA_SERQ_NOP_WIDTH                       1u
#define DMA_SERQ_NOP(x)                          (((uint8_t)(((uint8_t)(x))<<DMA_SERQ_NOP_SHIFT))&DMA_SERQ_NOP_MASK)
/* CDNE Bit Fields */
#define DMA_CDNE_CDNE_MASK                       0x1Fu
#define DMA_CDNE_CDNE_SHIFT                      0u
#define DMA_CDNE_CDNE_WIDTH                      5u
#define DMA_CDNE_CDNE(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_CDNE_CDNE_SHIFT))&DMA_CDNE_CDNE_MASK)
#define DMA_CDNE_CADN_MASK                       0x40u
#define DMA_CDNE_CADN_SHIFT                      6u
#define DMA_CDNE_CADN_WIDTH                      1u
#define DMA_CDNE_CADN(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_CDNE_CADN_SHIFT))&DMA_CDNE_CADN_MASK)
#define DMA_CDNE_NOP_MASK                        0x80u
#define DMA_CDNE_NOP_SHIFT                       7u
#define DMA_CDNE_NOP_WIDTH                       1u
#define DMA_CDNE_NOP(x)                          (((uint8_t)(((uint8_t)(x))<<DMA_CDNE_NOP_SHIFT))&DMA_CDNE_NOP_MASK)
/* SSRT Bit Fields */
#define DMA_SSRT_SSRT_MASK                       0x1Fu
#define DMA_SSRT_SSRT_SHIFT                      0u
#define DMA_SSRT_SSRT_WIDTH                      5u
#define DMA_SSRT_SSRT(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_SSRT_SSRT_SHIFT))&DMA_SSRT_SSRT_MASK)
#define DMA_SSRT_SAST_MASK                       0x40u
#define DMA_SSRT_SAST_SHIFT                      6u
#define DMA_SSRT_SAST_WIDTH                      1u
#define DMA_SSRT_SAST(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_SSRT_SAST_SHIFT))&DMA_SSRT_SAST_MASK)
#define DMA_SSRT_NOP_MASK                        0x80u
#define DMA_SSRT_NOP_SHIFT                       7u
#define DMA_SSRT_NOP_WIDTH                       1u
#define DMA_SSRT_NOP(x)                          (((uint8_t)(((uint8_t)(x))<<DMA_SSRT_NOP_SHIFT))&DMA_SSRT_NOP_MASK)
/* CERR Bit Fields */
#define DMA_CERR_CERR_MASK                       0x1Fu
#define DMA_CERR_CERR_SHIFT                      0u
#define DMA_CERR_CERR_WIDTH                      5u
#define DMA_CERR_CERR(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_CERR_CERR_SHIFT))&DMA_CERR_CERR_MASK)
#define DMA_CERR_CAEI_MASK                       0x40u
#define DMA_CERR_CAEI_SHIFT                      6u
#define DMA_CERR_CAEI_WIDTH                      1u
#define DMA_CERR_CAEI(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_CERR_CAEI_SHIFT))&DMA_CERR_CAEI_MASK)
#define DMA_CERR_NOP_MASK                        0x80u
#define DMA_CERR_NOP_SHIFT                       7u
#define DMA_CERR_NOP_WIDTH                       1u
#define DMA_CERR_NOP(x)                          (((uint8_t)(((uint8_t)(x))<<DMA_CERR_NOP_SHIFT))&DMA_CERR_NOP_MASK)
/* CINT Bit Fields */
#define DMA_CINT_CINT_MASK                       0x1Fu
#define DMA_CINT_CINT_SHIFT                      0u
#define DMA_CINT_CINT_WIDTH                      5u
#define DMA_CINT_CINT(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_CINT_CINT_SHIFT))&DMA_CINT_CINT_MASK)
#define DMA_CINT_CAIR_MASK                       0x40u
#define DMA_CINT_CAIR_SHIFT                      6u
#define DMA_CINT_CAIR_WIDTH                      1u
#define DMA_CINT_CAIR(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_CINT_CAIR_SHIFT))&DMA_CINT_CAIR_MASK)
#define DMA_CINT_NOP_MASK                        0x80u
#define DMA_CINT_NOP_SHIFT                       7u
#define DMA_CINT_NOP_WIDTH                       1u
#define DMA_CINT_NOP(x)                          (((uint8_t)(((uint8_t)(x))<<DMA_CINT_NOP_SHIFT))&DMA_CINT_NOP_MASK)
/* INT Bit Fields */
#define DMA_INT_INT0_MASK                        0x1u
#define DMA_INT_INT0_SHIFT                       0u
#define DMA_INT_INT0_WIDTH                       1u
#define DMA_INT_INT0(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT0_SHIFT))&DMA_INT_INT0_MASK)
#define DMA_INT_INT1_MASK                        0x2u
#define DMA_INT_INT1_SHIFT                       1u
#define DMA_INT_INT1_WIDTH                       1u
#define DMA_INT_INT1(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT1_SHIFT))&DMA_INT_INT1_MASK)
#define DMA_INT_INT2_MASK                        0x4u
#define DMA_INT_INT2_SHIFT                       2u
#define DMA_INT_INT2_WIDTH                       1u
#define DMA_INT_INT2(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT2_SHIFT))&DMA_INT_INT2_MASK)
#define DMA_INT_INT3_MASK                        0x8u
#define DMA_INT_INT3_SHIFT                       3u
#define DMA_INT_INT3_WIDTH                       1u
#define DMA_INT_INT3(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT3_SHIFT))&DMA_INT_INT3_MASK)
#define DMA_INT_INT4_MASK                        0x10u
#define DMA_INT_INT4_SHIFT                       4u
#define DMA_INT_INT4_WIDTH                       1u
#define DMA_INT_INT4(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT4_SHIFT))&DMA_INT_INT4_MASK)
#define DMA_INT_INT5_MASK                        0x20u
#define DMA_INT_INT5_SHIFT                       5u
#define DMA_INT_INT5_WIDTH                       1u
#define DMA_INT_INT5(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT5_SHIFT))&DMA_INT_INT5_MASK)
#define DMA_INT_INT6_MASK                        0x40u
#define DMA_INT_INT6_SHIFT                       6u
#define DMA_INT_INT6_WIDTH                       1u
#define DMA_INT_INT6(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT6_SHIFT))&DMA_INT_INT6_MASK)
#define DMA_INT_INT7_MASK                        0x80u
#define DMA_INT_INT7_SHIFT                       7u
#define DMA_INT_INT7_WIDTH                       1u
#define DMA_INT_INT7(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT7_SHIFT))&DMA_INT_INT7_MASK)
#define DMA_INT_INT8_MASK                        0x100u
#define DMA_INT_INT8_SHIFT                       8u
#define DMA_INT_INT8_WIDTH                       1u
#define DMA_INT_INT8(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT8_SHIFT))&DMA_INT_INT8_MASK)
#define DMA_INT_INT9_MASK                        0x200u
#define DMA_INT_INT9_SHIFT                       9u
#define DMA_INT_INT9_WIDTH                       1u
#define DMA_INT_INT9(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT9_SHIFT))&DMA_INT_INT9_MASK)
#define DMA_INT_INT10_MASK                       0x400u
#define DMA_INT_INT10_SHIFT                      10u
#define DMA_INT_INT10_WIDTH                      1u
#define DMA_INT_INT10(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT10_SHIFT))&DMA_INT_INT10_MASK)
#define DMA_INT_INT11_MASK                       0x800u
#define DMA_INT_INT11_SHIFT                      11u
#define DMA_INT_INT11_WIDTH                      1u
#define DMA_INT_INT11(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT11_SHIFT))&DMA_INT_INT11_MASK)
#define DMA_INT_INT12_MASK                       0x1000u
#define DMA_INT_INT12_SHIFT                      12u
#define DMA_INT_INT12_WIDTH                      1u
#define DMA_INT_INT12(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT12_SHIFT))&DMA_INT_INT12_MASK)
#define DMA_INT_INT13_MASK                       0x2000u
#define DMA_INT_INT13_SHIFT                      13u
#define DMA_INT_INT13_WIDTH                      1u
#define DMA_INT_INT13(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT13_SHIFT))&DMA_INT_INT13_MASK)
#define DMA_INT_INT14_MASK                       0x4000u
#define DMA_INT_INT14_SHIFT                      14u
#define DMA_INT_INT14_WIDTH                      1u
#define DMA_INT_INT14(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT14_SHIFT))&DMA_INT_INT14_MASK)
#define DMA_INT_INT15_MASK                       0x8000u
#define DMA_INT_INT15_SHIFT                      15u
#define DMA_INT_INT15_WIDTH                      1u
#define DMA_INT_INT15(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT15_SHIFT))&DMA_INT_INT15_MASK)
#define DMA_INT_INT16_MASK                       0x10000u
#define DMA_INT_INT16_SHIFT                      16u
#define DMA_INT_INT16_WIDTH                      1u
#define DMA_INT_INT16(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT16_SHIFT))&DMA_INT_INT16_MASK)
#define DMA_INT_INT17_MASK                       0x20000u
#define DMA_INT_INT17_SHIFT                      17u
#define DMA_INT_INT17_WIDTH                      1u
#define DMA_INT_INT17(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT17_SHIFT))&DMA_INT_INT17_MASK)
#define DMA_INT_INT18_MASK                       0x40000u
#define DMA_INT_INT18_SHIFT                      18u
#define DMA_INT_INT18_WIDTH                      1u
#define DMA_INT_INT18(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT18_SHIFT))&DMA_INT_INT18_MASK)
#define DMA_INT_INT19_MASK                       0x80000u
#define DMA_INT_INT19_SHIFT                      19u
#define DMA_INT_INT19_WIDTH                      1u
#define DMA_INT_INT19(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT19_SHIFT))&DMA_INT_INT19_MASK)
#define DMA_INT_INT20_MASK                       0x100000u
#define DMA_INT_INT20_SHIFT                      20u
#define DMA_INT_INT20_WIDTH                      1u
#define DMA_INT_INT20(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT20_SHIFT))&DMA_INT_INT20_MASK)
#define DMA_INT_INT21_MASK                       0x200000u
#define DMA_INT_INT21_SHIFT                      21u
#define DMA_INT_INT21_WIDTH                      1u
#define DMA_INT_INT21(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT21_SHIFT))&DMA_INT_INT21_MASK)
#define DMA_INT_INT22_MASK                       0x400000u
#define DMA_INT_INT22_SHIFT                      22u
#define DMA_INT_INT22_WIDTH                      1u
#define DMA_INT_INT22(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT22_SHIFT))&DMA_INT_INT22_MASK)
#define DMA_INT_INT23_MASK                       0x800000u
#define DMA_INT_INT23_SHIFT                      23u
#define DMA_INT_INT23_WIDTH                      1u
#define DMA_INT_INT23(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT23_SHIFT))&DMA_INT_INT23_MASK)
#define DMA_INT_INT24_MASK                       0x1000000u
#define DMA_INT_INT24_SHIFT                      24u
#define DMA_INT_INT24_WIDTH                      1u
#define DMA_INT_INT24(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT24_SHIFT))&DMA_INT_INT24_MASK)
#define DMA_INT_INT25_MASK                       0x2000000u
#define DMA_INT_INT25_SHIFT                      25u
#define DMA_INT_INT25_WIDTH                      1u
#define DMA_INT_INT25(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT25_SHIFT))&DMA_INT_INT25_MASK)
#define DMA_INT_INT26_MASK                       0x4000000u
#define DMA_INT_INT26_SHIFT                      26u
#define DMA_INT_INT26_WIDTH                      1u
#define DMA_INT_INT26(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT26_SHIFT))&DMA_INT_INT26_MASK)
#define DMA_INT_INT27_MASK                       0x8000000u
#define DMA_INT_INT27_SHIFT                      27u
#define DMA_INT_INT27_WIDTH                      1u
#define DMA_INT_INT27(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT27_SHIFT))&DMA_INT_INT27_MASK)
#define DMA_INT_INT28_MASK                       0x10000000u
#define DMA_INT_INT28_SHIFT                      28u
#define DMA_INT_INT28_WIDTH                      1u
#define DMA_INT_INT28(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT28_SHIFT))&DMA_INT_INT28_MASK)
#define DMA_INT_INT29_MASK                       0x20000000u
#define DMA_INT_INT29_SHIFT                      29u
#define DMA_INT_INT29_WIDTH                      1u
#define DMA_INT_INT29(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT29_SHIFT))&DMA_INT_INT29_MASK)
#define DMA_INT_INT30_MASK                       0x40000000u
#define DMA_INT_INT30_SHIFT                      30u
#define DMA_INT_INT30_WIDTH                      1u
#define DMA_INT_INT30(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT30_SHIFT))&DMA_INT_INT30_MASK)
#define DMA_INT_INT31_MASK                       0x80000000u
#define DMA_INT_INT31_SHIFT                      31u
#define DMA_INT_INT31_WIDTH                      1u
#define DMA_INT_INT31(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT31_SHIFT))&DMA_INT_INT31_MASK)
/* ERR Bit Fields */
#define DMA_ERR_ERR0_MASK                        0x1u
#define DMA_ERR_ERR0_SHIFT                       0u
#define DMA_ERR_ERR0_WIDTH                       1u
#define DMA_ERR_ERR0(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR0_SHIFT))&DMA_ERR_ERR0_MASK)
#define DMA_ERR_ERR1_MASK                        0x2u
#define DMA_ERR_ERR1_SHIFT                       1u
#define DMA_ERR_ERR1_WIDTH                       1u
#define DMA_ERR_ERR1(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR1_SHIFT))&DMA_ERR_ERR1_MASK)
#define DMA_ERR_ERR2_MASK                        0x4u
#define DMA_ERR_ERR2_SHIFT                       2u
#define DMA_ERR_ERR2_WIDTH                       1u
#define DMA_ERR_ERR2(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR2_SHIFT))&DMA_ERR_ERR2_MASK)
#define DMA_ERR_ERR3_MASK                        0x8u
#define DMA_ERR_ERR3_SHIFT                       3u
#define DMA_ERR_ERR3_WIDTH                       1u
#define DMA_ERR_ERR3(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR3_SHIFT))&DMA_ERR_ERR3_MASK)
#define DMA_ERR_ERR4_MASK                        0x10u
#define DMA_ERR_ERR4_SHIFT                       4u
#define DMA_ERR_ERR4_WIDTH                       1u
#define DMA_ERR_ERR4(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR4_SHIFT))&DMA_ERR_ERR4_MASK)
#define DMA_ERR_ERR5_MASK                        0x20u
#define DMA_ERR_ERR5_SHIFT                       5u
#define DMA_ERR_ERR5_WIDTH                       1u
#define DMA_ERR_ERR5(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR5_SHIFT))&DMA_ERR_ERR5_MASK)
#define DMA_ERR_ERR6_MASK                        0x40u
#define DMA_ERR_ERR6_SHIFT                       6u
#define DMA_ERR_ERR6_WIDTH                       1u
#define DMA_ERR_ERR6(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR6_SHIFT))&DMA_ERR_ERR6_MASK)
#define DMA_ERR_ERR7_MASK                        0x80u
#define DMA_ERR_ERR7_SHIFT                       7u
#define DMA_ERR_ERR7_WIDTH                       1u
#define DMA_ERR_ERR7(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR7_SHIFT))&DMA_ERR_ERR7_MASK)
#define DMA_ERR_ERR8_MASK                        0x100u
#define DMA_ERR_ERR8_SHIFT                       8u
#define DMA_ERR_ERR8_WIDTH                       1u
#define DMA_ERR_ERR8(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR8_SHIFT))&DMA_ERR_ERR8_MASK)
#define DMA_ERR_ERR9_MASK                        0x200u
#define DMA_ERR_ERR9_SHIFT                       9u
#define DMA_ERR_ERR9_WIDTH                       1u
#define DMA_ERR_ERR9(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR9_SHIFT))&DMA_ERR_ERR9_MASK)
#define DMA_ERR_ERR10_MASK                       0x400u
#define DMA_ERR_ERR10_SHIFT                      10u
#define DMA_ERR_ERR10_WIDTH                      1u
#define DMA_ERR_ERR10(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR10_SHIFT))&DMA_ERR_ERR10_MASK)
#define DMA_ERR_ERR11_MASK                       0x800u
#define DMA_ERR_ERR11_SHIFT                      11u
#define DMA_ERR_ERR11_WIDTH                      1u
#define DMA_ERR_ERR11(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR11_SHIFT))&DMA_ERR_ERR11_MASK)
#define DMA_ERR_ERR12_MASK                       0x1000u
#define DMA_ERR_ERR12_SHIFT                      12u
#define DMA_ERR_ERR12_WIDTH                      1u
#define DMA_ERR_ERR12(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR12_SHIFT))&DMA_ERR_ERR12_MASK)
#define DMA_ERR_ERR13_MASK                       0x2000u
#define DMA_ERR_ERR13_SHIFT                      13u
#define DMA_ERR_ERR13_WIDTH                      1u
#define DMA_ERR_ERR13(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR13_SHIFT))&DMA_ERR_ERR13_MASK)
#define DMA_ERR_ERR14_MASK                       0x4000u
#define DMA_ERR_ERR14_SHIFT                      14u
#define DMA_ERR_ERR14_WIDTH                      1u
#define DMA_ERR_ERR14(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR14_SHIFT))&DMA_ERR_ERR14_MASK)
#define DMA_ERR_ERR15_MASK                       0x8000u
#define DMA_ERR_ERR15_SHIFT                      15u
#define DMA_ERR_ERR15_WIDTH                      1u
#define DMA_ERR_ERR15(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR15_SHIFT))&DMA_ERR_ERR15_MASK)
#define DMA_ERR_ERR16_MASK                       0x10000u
#define DMA_ERR_ERR16_SHIFT                      16u
#define DMA_ERR_ERR16_WIDTH                      1u
#define DMA_ERR_ERR16(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR16_SHIFT))&DMA_ERR_ERR16_MASK)
#define DMA_ERR_ERR17_MASK                       0x20000u
#define DMA_ERR_ERR17_SHIFT                      17u
#define DMA_ERR_ERR17_WIDTH                      1u
#define DMA_ERR_ERR17(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR17_SHIFT))&DMA_ERR_ERR17_MASK)
#define DMA_ERR_ERR18_MASK                       0x40000u
#define DMA_ERR_ERR18_SHIFT                      18u
#define DMA_ERR_ERR18_WIDTH                      1u
#define DMA_ERR_ERR18(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR18_SHIFT))&DMA_ERR_ERR18_MASK)
#define DMA_ERR_ERR19_MASK                       0x80000u
#define DMA_ERR_ERR19_SHIFT                      19u
#define DMA_ERR_ERR19_WIDTH                      1u
#define DMA_ERR_ERR19(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR19_SHIFT))&DMA_ERR_ERR19_MASK)
#define DMA_ERR_ERR20_MASK                       0x100000u
#define DMA_ERR_ERR20_SHIFT                      20u
#define DMA_ERR_ERR20_WIDTH                      1u
#define DMA_ERR_ERR20(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR20_SHIFT))&DMA_ERR_ERR20_MASK)
#define DMA_ERR_ERR21_MASK                       0x200000u
#define DMA_ERR_ERR21_SHIFT                      21u
#define DMA_ERR_ERR21_WIDTH                      1u
#define DMA_ERR_ERR21(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR21_SHIFT))&DMA_ERR_ERR21_MASK)
#define DMA_ERR_ERR22_MASK                       0x400000u
#define DMA_ERR_ERR22_SHIFT                      22u
#define DMA_ERR_ERR22_WIDTH                      1u
#define DMA_ERR_ERR22(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR22_SHIFT))&DMA_ERR_ERR22_MASK)
#define DMA_ERR_ERR23_MASK                       0x800000u
#define DMA_ERR_ERR23_SHIFT                      23u
#define DMA_ERR_ERR23_WIDTH                      1u
#define DMA_ERR_ERR23(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR23_SHIFT))&DMA_ERR_ERR23_MASK)
#define DMA_ERR_ERR24_MASK                       0x1000000u
#define DMA_ERR_ERR24_SHIFT                      24u
#define DMA_ERR_ERR24_WIDTH                      1u
#define DMA_ERR_ERR24(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR24_SHIFT))&DMA_ERR_ERR24_MASK)
#define DMA_ERR_ERR25_MASK                       0x2000000u
#define DMA_ERR_ERR25_SHIFT                      25u
#define DMA_ERR_ERR25_WIDTH                      1u
#define DMA_ERR_ERR25(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR25_SHIFT))&DMA_ERR_ERR25_MASK)
#define DMA_ERR_ERR26_MASK                       0x4000000u
#define DMA_ERR_ERR26_SHIFT                      26u
#define DMA_ERR_ERR26_WIDTH                      1u
#define DMA_ERR_ERR26(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR26_SHIFT))&DMA_ERR_ERR26_MASK)
#define DMA_ERR_ERR27_MASK                       0x8000000u
#define DMA_ERR_ERR27_SHIFT                      27u
#define DMA_ERR_ERR27_WIDTH                      1u
#define DMA_ERR_ERR27(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR27_SHIFT))&DMA_ERR_ERR27_MASK)
#define DMA_ERR_ERR28_MASK                       0x10000000u
#define DMA_ERR_ERR28_SHIFT                      28u
#define DMA_ERR_ERR28_WIDTH                      1u
#define DMA_ERR_ERR28(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR28_SHIFT))&DMA_ERR_ERR28_MASK)
#define DMA_ERR_ERR29_MASK                       0x20000000u
#define DMA_ERR_ERR29_SHIFT                      29u
#define DMA_ERR_ERR29_WIDTH                      1u
#define DMA_ERR_ERR29(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR29_SHIFT))&DMA_ERR_ERR29_MASK)
#define DMA_ERR_ERR30_MASK                       0x40000000u
#define DMA_ERR_ERR30_SHIFT                      30u
#define DMA_ERR_ERR30_WIDTH                      1u
#define DMA_ERR_ERR30(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR30_SHIFT))&DMA_ERR_ERR30_MASK)
#define DMA_ERR_ERR31_MASK                       0x80000000u
#define DMA_ERR_ERR31_SHIFT                      31u
#define DMA_ERR_ERR31_WIDTH                      1u
#define DMA_ERR_ERR31(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR31_SHIFT))&DMA_ERR_ERR31_MASK)
/* HRS Bit Fields */
#define DMA_HRS_HRS0_MASK                        0x1u
#define DMA_HRS_HRS0_SHIFT                       0u
#define DMA_HRS_HRS0_WIDTH                       1u
#define DMA_HRS_HRS0(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS0_SHIFT))&DMA_HRS_HRS0_MASK)
#define DMA_HRS_HRS1_MASK                        0x2u
#define DMA_HRS_HRS1_SHIFT                       1u
#define DMA_HRS_HRS1_WIDTH                       1u
#define DMA_HRS_HRS1(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS1_SHIFT))&DMA_HRS_HRS1_MASK)
#define DMA_HRS_HRS2_MASK                        0x4u
#define DMA_HRS_HRS2_SHIFT                       2u
#define DMA_HRS_HRS2_WIDTH                       1u
#define DMA_HRS_HRS2(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS2_SHIFT))&DMA_HRS_HRS2_MASK)
#define DMA_HRS_HRS3_MASK                        0x8u
#define DMA_HRS_HRS3_SHIFT                       3u
#define DMA_HRS_HRS3_WIDTH                       1u
#define DMA_HRS_HRS3(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS3_SHIFT))&DMA_HRS_HRS3_MASK)
#define DMA_HRS_HRS4_MASK                        0x10u
#define DMA_HRS_HRS4_SHIFT                       4u
#define DMA_HRS_HRS4_WIDTH                       1u
#define DMA_HRS_HRS4(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS4_SHIFT))&DMA_HRS_HRS4_MASK)
#define DMA_HRS_HRS5_MASK                        0x20u
#define DMA_HRS_HRS5_SHIFT                       5u
#define DMA_HRS_HRS5_WIDTH                       1u
#define DMA_HRS_HRS5(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS5_SHIFT))&DMA_HRS_HRS5_MASK)
#define DMA_HRS_HRS6_MASK                        0x40u
#define DMA_HRS_HRS6_SHIFT                       6u
#define DMA_HRS_HRS6_WIDTH                       1u
#define DMA_HRS_HRS6(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS6_SHIFT))&DMA_HRS_HRS6_MASK)
#define DMA_HRS_HRS7_MASK                        0x80u
#define DMA_HRS_HRS7_SHIFT                       7u
#define DMA_HRS_HRS7_WIDTH                       1u
#define DMA_HRS_HRS7(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS7_SHIFT))&DMA_HRS_HRS7_MASK)
#define DMA_HRS_HRS8_MASK                        0x100u
#define DMA_HRS_HRS8_SHIFT                       8u
#define DMA_HRS_HRS8_WIDTH                       1u
#define DMA_HRS_HRS8(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS8_SHIFT))&DMA_HRS_HRS8_MASK)
#define DMA_HRS_HRS9_MASK                        0x200u
#define DMA_HRS_HRS9_SHIFT                       9u
#define DMA_HRS_HRS9_WIDTH                       1u
#define DMA_HRS_HRS9(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS9_SHIFT))&DMA_HRS_HRS9_MASK)
#define DMA_HRS_HRS10_MASK                       0x400u
#define DMA_HRS_HRS10_SHIFT                      10u
#define DMA_HRS_HRS10_WIDTH                      1u
#define DMA_HRS_HRS10(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS10_SHIFT))&DMA_HRS_HRS10_MASK)
#define DMA_HRS_HRS11_MASK                       0x800u
#define DMA_HRS_HRS11_SHIFT                      11u
#define DMA_HRS_HRS11_WIDTH                      1u
#define DMA_HRS_HRS11(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS11_SHIFT))&DMA_HRS_HRS11_MASK)
#define DMA_HRS_HRS12_MASK                       0x1000u
#define DMA_HRS_HRS12_SHIFT                      12u
#define DMA_HRS_HRS12_WIDTH                      1u
#define DMA_HRS_HRS12(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS12_SHIFT))&DMA_HRS_HRS12_MASK)
#define DMA_HRS_HRS13_MASK                       0x2000u
#define DMA_HRS_HRS13_SHIFT                      13u
#define DMA_HRS_HRS13_WIDTH                      1u
#define DMA_HRS_HRS13(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS13_SHIFT))&DMA_HRS_HRS13_MASK)
#define DMA_HRS_HRS14_MASK                       0x4000u
#define DMA_HRS_HRS14_SHIFT                      14u
#define DMA_HRS_HRS14_WIDTH                      1u
#define DMA_HRS_HRS14(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS14_SHIFT))&DMA_HRS_HRS14_MASK)
#define DMA_HRS_HRS15_MASK                       0x8000u
#define DMA_HRS_HRS15_SHIFT                      15u
#define DMA_HRS_HRS15_WIDTH                      1u
#define DMA_HRS_HRS15(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS15_SHIFT))&DMA_HRS_HRS15_MASK)
#define DMA_HRS_HRS16_MASK                       0x10000u
#define DMA_HRS_HRS16_SHIFT                      16u
#define DMA_HRS_HRS16_WIDTH                      1u
#define DMA_HRS_HRS16(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS16_SHIFT))&DMA_HRS_HRS16_MASK)
#define DMA_HRS_HRS17_MASK                       0x20000u
#define DMA_HRS_HRS17_SHIFT                      17u
#define DMA_HRS_HRS17_WIDTH                      1u
#define DMA_HRS_HRS17(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS17_SHIFT))&DMA_HRS_HRS17_MASK)
#define DMA_HRS_HRS18_MASK                       0x40000u
#define DMA_HRS_HRS18_SHIFT                      18u
#define DMA_HRS_HRS18_WIDTH                      1u
#define DMA_HRS_HRS18(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS18_SHIFT))&DMA_HRS_HRS18_MASK)
#define DMA_HRS_HRS19_MASK                       0x80000u
#define DMA_HRS_HRS19_SHIFT                      19u
#define DMA_HRS_HRS19_WIDTH                      1u
#define DMA_HRS_HRS19(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS19_SHIFT))&DMA_HRS_HRS19_MASK)
#define DMA_HRS_HRS20_MASK                       0x100000u
#define DMA_HRS_HRS20_SHIFT                      20u
#define DMA_HRS_HRS20_WIDTH                      1u
#define DMA_HRS_HRS20(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS20_SHIFT))&DMA_HRS_HRS20_MASK)
#define DMA_HRS_HRS21_MASK                       0x200000u
#define DMA_HRS_HRS21_SHIFT                      21u
#define DMA_HRS_HRS21_WIDTH                      1u
#define DMA_HRS_HRS21(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS21_SHIFT))&DMA_HRS_HRS21_MASK)
#define DMA_HRS_HRS22_MASK                       0x400000u
#define DMA_HRS_HRS22_SHIFT                      22u
#define DMA_HRS_HRS22_WIDTH                      1u
#define DMA_HRS_HRS22(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS22_SHIFT))&DMA_HRS_HRS22_MASK)
#define DMA_HRS_HRS23_MASK                       0x800000u
#define DMA_HRS_HRS23_SHIFT                      23u
#define DMA_HRS_HRS23_WIDTH                      1u
#define DMA_HRS_HRS23(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS23_SHIFT))&DMA_HRS_HRS23_MASK)
#define DMA_HRS_HRS24_MASK                       0x1000000u
#define DMA_HRS_HRS24_SHIFT                      24u
#define DMA_HRS_HRS24_WIDTH                      1u
#define DMA_HRS_HRS24(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS24_SHIFT))&DMA_HRS_HRS24_MASK)
#define DMA_HRS_HRS25_MASK                       0x2000000u
#define DMA_HRS_HRS25_SHIFT                      25u
#define DMA_HRS_HRS25_WIDTH                      1u
#define DMA_HRS_HRS25(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS25_SHIFT))&DMA_HRS_HRS25_MASK)
#define DMA_HRS_HRS26_MASK                       0x4000000u
#define DMA_HRS_HRS26_SHIFT                      26u
#define DMA_HRS_HRS26_WIDTH                      1u
#define DMA_HRS_HRS26(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS26_SHIFT))&DMA_HRS_HRS26_MASK)
#define DMA_HRS_HRS27_MASK                       0x8000000u
#define DMA_HRS_HRS27_SHIFT                      27u
#define DMA_HRS_HRS27_WIDTH                      1u
#define DMA_HRS_HRS27(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS27_SHIFT))&DMA_HRS_HRS27_MASK)
#define DMA_HRS_HRS28_MASK                       0x10000000u
#define DMA_HRS_HRS28_SHIFT                      28u
#define DMA_HRS_HRS28_WIDTH                      1u
#define DMA_HRS_HRS28(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS28_SHIFT))&DMA_HRS_HRS28_MASK)
#define DMA_HRS_HRS29_MASK                       0x20000000u
#define DMA_HRS_HRS29_SHIFT                      29u
#define DMA_HRS_HRS29_WIDTH                      1u
#define DMA_HRS_HRS29(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS29_SHIFT))&DMA_HRS_HRS29_MASK)
#define DMA_HRS_HRS30_MASK                       0x40000000u
#define DMA_HRS_HRS30_SHIFT                      30u
#define DMA_HRS_HRS30_WIDTH                      1u
#define DMA_HRS_HRS30(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS30_SHIFT))&DMA_HRS_HRS30_MASK)
#define DMA_HRS_HRS31_MASK                       0x80000000u
#define DMA_HRS_HRS31_SHIFT                      31u
#define DMA_HRS_HRS31_WIDTH                      1u
#define DMA_HRS_HRS31(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS31_SHIFT))&DMA_HRS_HRS31_MASK)
/* GPOR Bit Fields */
#define DMA_GPOR_GPOR_MASK                       0xFFFFFFFFu
#define DMA_GPOR_GPOR_SHIFT                      0u
#define DMA_GPOR_GPOR_WIDTH                      32u
#define DMA_GPOR_GPOR(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_GPOR_GPOR_SHIFT))&DMA_GPOR_GPOR_MASK)
/* DCHPRI Bit Fields */
#define DMA_DCHPRI_CHPRI_MASK                    0xFu
#define DMA_DCHPRI_CHPRI_SHIFT                   0u
#define DMA_DCHPRI_CHPRI_WIDTH                   4u
#define DMA_DCHPRI_CHPRI(x)                      (((uint8_t)(((uint8_t)(x))<<DMA_DCHPRI_CHPRI_SHIFT))&DMA_DCHPRI_CHPRI_MASK)
#define DMA_DCHPRI_GRPPRI_MASK                   0x30u
#define DMA_DCHPRI_GRPPRI_SHIFT                  4u
#define DMA_DCHPRI_GRPPRI_WIDTH                  2u
#define DMA_DCHPRI_GRPPRI(x)                     (((uint8_t)(((uint8_t)(x))<<DMA_DCHPRI_GRPPRI_SHIFT))&DMA_DCHPRI_GRPPRI_MASK)
#define DMA_DCHPRI_DPA_MASK                      0x40u
#define DMA_DCHPRI_DPA_SHIFT                     6u
#define DMA_DCHPRI_DPA_WIDTH                     1u
#define DMA_DCHPRI_DPA(x)                        (((uint8_t)(((uint8_t)(x))<<DMA_DCHPRI_DPA_SHIFT))&DMA_DCHPRI_DPA_MASK)
#define DMA_DCHPRI_ECP_MASK                      0x80u
#define DMA_DCHPRI_ECP_SHIFT                     7u
#define DMA_DCHPRI_ECP_WIDTH                     1u
#define DMA_DCHPRI_ECP(x)                        (((uint8_t)(((uint8_t)(x))<<DMA_DCHPRI_ECP_SHIFT))&DMA_DCHPRI_ECP_MASK)
/* DCHMID Bit Fields */
#define DMA_DCHMID_MID_MASK                      0xFu
#define DMA_DCHMID_MID_SHIFT                     0u
#define DMA_DCHMID_MID_WIDTH                     4u
#define DMA_DCHMID_MID(x)                        (((uint8_t)(((uint8_t)(x))<<DMA_DCHMID_MID_SHIFT))&DMA_DCHMID_MID_MASK)
#define DMA_DCHMID_PAL_MASK                      0x40u
#define DMA_DCHMID_PAL_SHIFT                     6u
#define DMA_DCHMID_PAL_WIDTH                     1u
#define DMA_DCHMID_PAL(x)                        (((uint8_t)(((uint8_t)(x))<<DMA_DCHMID_PAL_SHIFT))&DMA_DCHMID_PAL_MASK)
#define DMA_DCHMID_EMI_MASK                      0x80u
#define DMA_DCHMID_EMI_SHIFT                     7u
#define DMA_DCHMID_EMI_WIDTH                     1u
#define DMA_DCHMID_EMI(x)                        (((uint8_t)(((uint8_t)(x))<<DMA_DCHMID_EMI_SHIFT))&DMA_DCHMID_EMI_MASK)
/* TCD_SADDR Bit Fields */
#define DMA_TCD_SADDR_SADDR_MASK                 0xFFFFFFFFu
#define DMA_TCD_SADDR_SADDR_SHIFT                0u
#define DMA_TCD_SADDR_SADDR_WIDTH                32u
#define DMA_TCD_SADDR_SADDR(x)                   (((uint32_t)(((uint32_t)(x))<<DMA_TCD_SADDR_SADDR_SHIFT))&DMA_TCD_SADDR_SADDR_MASK)
/* TCD_SOFF Bit Fields */
#define DMA_TCD_SOFF_SOFF_MASK                   0xFFFFu
#define DMA_TCD_SOFF_SOFF_SHIFT                  0u
#define DMA_TCD_SOFF_SOFF_WIDTH                  16u
#define DMA_TCD_SOFF_SOFF(x)                     (((uint16_t)(((uint16_t)(x))<<DMA_TCD_SOFF_SOFF_SHIFT))&DMA_TCD_SOFF_SOFF_MASK)
/* TCD_ATTR Bit Fields */
#define DMA_TCD_ATTR_DSIZE_MASK                  0x7u
#define DMA_TCD_ATTR_DSIZE_SHIFT                 0u
#define DMA_TCD_ATTR_DSIZE_WIDTH                 3u
#define DMA_TCD_ATTR_DSIZE(x)                    (((uint16_t)(((uint16_t)(x))<<DMA_TCD_ATTR_DSIZE_SHIFT))&DMA_TCD_ATTR_DSIZE_MASK)
#define DMA_TCD_ATTR_DMOD_MASK                   0xF8u
#define DMA_TCD_ATTR_DMOD_SHIFT                  3u
#define DMA_TCD_ATTR_DMOD_WIDTH                  5u
#define DMA_TCD_ATTR_DMOD(x)                     (((uint16_t)(((uint16_t)(x))<<DMA_TCD_ATTR_DMOD_SHIFT))&DMA_TCD_ATTR_DMOD_MASK)
#define DMA_TCD_ATTR_SSIZE_MASK                  0x700u
#define DMA_TCD_ATTR_SSIZE_SHIFT                 8u
#define DMA_TCD_ATTR_SSIZE_WIDTH                 3u
#define DMA_TCD_ATTR_SSIZE(x)                    (((uint16_t)(((uint16_t)(x))<<DMA_TCD_ATTR_SSIZE_SHIFT))&DMA_TCD_ATTR_SSIZE_MASK)
#define DMA_TCD_ATTR_SMOD_MASK                   0xF800u
#define DMA_TCD_ATTR_SMOD_SHIFT                  11u
#define DMA_TCD_ATTR_SMOD_WIDTH                  5u
#define DMA_TCD_ATTR_SMOD(x)                     (((uint16_t)(((uint16_t)(x))<<DMA_TCD_ATTR_SMOD_SHIFT))&DMA_TCD_ATTR_SMOD_MASK)
/* TCD_NBYTES_MLNO Bit Fields */
#define DMA_TCD_NBYTES_MLNO_NBYTES_MASK          0xFFFFFFFFu
#define DMA_TCD_NBYTES_MLNO_NBYTES_SHIFT         0u
#define DMA_TCD_NBYTES_MLNO_NBYTES_WIDTH         32u
#define DMA_TCD_NBYTES_MLNO_NBYTES(x)            (((uint32_t)(((uint32_t)(x))<<DMA_TCD_NBYTES_MLNO_NBYTES_SHIFT))&DMA_TCD_NBYTES_MLNO_NBYTES_MASK)
/* TCD_NBYTES_MLOFFNO Bit Fields */
#define DMA_TCD_NBYTES_MLOFFNO_NBYTES_MASK       0x3FFFFFFFu
#define DMA_TCD_NBYTES_MLOFFNO_NBYTES_SHIFT      0u
#define DMA_TCD_NBYTES_MLOFFNO_NBYTES_WIDTH      30u
#define DMA_TCD_NBYTES_MLOFFNO_NBYTES(x)         (((uint32_t)(((uint32_t)(x))<<DMA_TCD_NBYTES_MLOFFNO_NBYTES_SHIFT))&DMA_TCD_NBYTES_MLOFFNO_NBYTES_MASK)
#define DMA_TCD_NBYTES_MLOFFNO_DMLOE_MASK        0x40000000u
#define DMA_TCD_NBYTES_MLOFFNO_DMLOE_SHIFT       30u
#define DMA_TCD_NBYTES_MLOFFNO_DMLOE_WIDTH       1u
#define DMA_TCD_NBYTES_MLOFFNO_DMLOE(x)          (((uint32_t)(((uint32_t)(x))<<DMA_TCD_NBYTES_MLOFFNO_DMLOE_SHIFT))&DMA_TCD_NBYTES_MLOFFNO_DMLOE_MASK)
#define DMA_TCD_NBYTES_MLOFFNO_SMLOE_MASK        0x80000000u
#define DMA_TCD_NBYTES_MLOFFNO_SMLOE_SHIFT       31u
#define DMA_TCD_NBYTES_MLOFFNO_SMLOE_WIDTH       1u
#define DMA_TCD_NBYTES_MLOFFNO_SMLOE(x)          (((uint32_t)(((uint32_t)(x))<<DMA_TCD_NBYTES_MLOFFNO_SMLOE_SHIFT))&DMA_TCD_NBYTES_MLOFFNO_SMLOE_MASK)
/* TCD_NBYTES_MLOFFYES Bit Fields */
#define DMA_TCD_NBYTES_MLOFFYES_NBYTES_MASK      0x3FFu
#define DMA_TCD_NBYTES_MLOFFYES_NBYTES_SHIFT     0u
#define DMA_TCD_NBYTES_MLOFFYES_NBYTES_WIDTH     10u
#define DMA_TCD_NBYTES_MLOFFYES_NBYTES(x)        (((uint32_t)(((uint32_t)(x))<<DMA_TCD_NBYTES_MLOFFYES_NBYTES_SHIFT))&DMA_TCD_NBYTES_MLOFFYES_NBYTES_MASK)
#define DMA_TCD_NBYTES_MLOFFYES_MLOFF_MASK       0x3FFFFC00u
#define DMA_TCD_NBYTES_MLOFFYES_MLOFF_SHIFT      10u
#define DMA_TCD_NBYTES_MLOFFYES_MLOFF_WIDTH      20u
#define DMA_TCD_NBYTES_MLOFFYES_MLOFF(x)         (((uint32_t)(((uint32_t)(x))<<DMA_TCD_NBYTES_MLOFFYES_MLOFF_SHIFT))&DMA_TCD_NBYTES_MLOFFYES_MLOFF_MASK)
#define DMA_TCD_NBYTES_MLOFFYES_DMLOE_MASK       0x40000000u
#define DMA_TCD_NBYTES_MLOFFYES_DMLOE_SHIFT      30u
#define DMA_TCD_NBYTES_MLOFFYES_DMLOE_WIDTH      1u
#define DMA_TCD_NBYTES_MLOFFYES_DMLOE(x)         (((uint32_t)(((uint32_t)(x))<<DMA_TCD_NBYTES_MLOFFYES_DMLOE_SHIFT))&DMA_TCD_NBYTES_MLOFFYES_DMLOE_MASK)
#define DMA_TCD_NBYTES_MLOFFYES_SMLOE_MASK       0x80000000u
#define DMA_TCD_NBYTES_MLOFFYES_SMLOE_SHIFT      31u
#define DMA_TCD_NBYTES_MLOFFYES_SMLOE_WIDTH      1u
#define DMA_TCD_NBYTES_MLOFFYES_SMLOE(x)         (((uint32_t)(((uint32_t)(x))<<DMA_TCD_NBYTES_MLOFFYES_SMLOE_SHIFT))&DMA_TCD_NBYTES_MLOFFYES_SMLOE_MASK)
/* TCD_SLAST Bit Fields */
#define DMA_TCD_SLAST_SLAST_MASK                 0xFFFFFFFFu
#define DMA_TCD_SLAST_SLAST_SHIFT                0u
#define DMA_TCD_SLAST_SLAST_WIDTH                32u
#define DMA_TCD_SLAST_SLAST(x)                   (((uint32_t)(((uint32_t)(x))<<DMA_TCD_SLAST_SLAST_SHIFT))&DMA_TCD_SLAST_SLAST_MASK)
/* TCD_DADDR Bit Fields */
#define DMA_TCD_DADDR_DADDR_MASK                 0xFFFFFFFFu
#define DMA_TCD_DADDR_DADDR_SHIFT                0u
#define DMA_TCD_DADDR_DADDR_WIDTH                32u
#define DMA_TCD_DADDR_DADDR(x)                   (((uint32_t)(((uint32_t)(x))<<DMA_TCD_DADDR_DADDR_SHIFT))&DMA_TCD_DADDR_DADDR_MASK)
/* TCD_DOFF Bit Fields */
#define DMA_TCD_DOFF_DOFF_MASK                   0xFFFFu
#define DMA_TCD_DOFF_DOFF_SHIFT                  0u
#define DMA_TCD_DOFF_DOFF_WIDTH                  16u
#define DMA_TCD_DOFF_DOFF(x)                     (((uint16_t)(((uint16_t)(x))<<DMA_TCD_DOFF_DOFF_SHIFT))&DMA_TCD_DOFF_DOFF_MASK)
/* TCD_CITER_ELINKNO Bit Fields */
#define DMA_TCD_CITER_ELINKNO_CITER_MASK         0x7FFFu
#define DMA_TCD_CITER_ELINKNO_CITER_SHIFT        0u
#define DMA_TCD_CITER_ELINKNO_CITER_WIDTH        15u
#define DMA_TCD_CITER_ELINKNO_CITER(x)           (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CITER_ELINKNO_CITER_SHIFT))&DMA_TCD_CITER_ELINKNO_CITER_MASK)
#define DMA_TCD_CITER_ELINKNO_ELINK_MASK         0x8000u
#define DMA_TCD_CITER_ELINKNO_ELINK_SHIFT        15u
#define DMA_TCD_CITER_ELINKNO_ELINK_WIDTH        1u
#define DMA_TCD_CITER_ELINKNO_ELINK(x)           (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CITER_ELINKNO_ELINK_SHIFT))&DMA_TCD_CITER_ELINKNO_ELINK_MASK)
/* TCD_CITER_ELINKYES Bit Fields */
#define DMA_TCD_CITER_ELINKYES_CITER_LE_MASK     0x1FFu
#define DMA_TCD_CITER_ELINKYES_CITER_LE_SHIFT    0u
#define DMA_TCD_CITER_ELINKYES_CITER_LE_WIDTH    9u
#define DMA_TCD_CITER_ELINKYES_CITER_LE(x)       (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CITER_ELINKYES_CITER_LE_SHIFT))&DMA_TCD_CITER_ELINKYES_CITER_LE_MASK)
#define DMA_TCD_CITER_ELINKYES_LINKCH_MASK       0x3E00u
#define DMA_TCD_CITER_ELINKYES_LINKCH_SHIFT      9u
#define DMA_TCD_CITER_ELINKYES_LINKCH_WIDTH      5u
#define DMA_TCD_CITER_ELINKYES_LINKCH(x)         (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CITER_ELINKYES_LINKCH_SHIFT))&DMA_TCD_CITER_ELINKYES_LINKCH_MASK)
#define DMA_TCD_CITER_ELINKYES_ELINK_MASK        0x8000u
#define DMA_TCD_CITER_ELINKYES_ELINK_SHIFT       15u
#define DMA_TCD_CITER_ELINKYES_ELINK_WIDTH       1u
#define DMA_TCD_CITER_ELINKYES_ELINK(x)          (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CITER_ELINKYES_ELINK_SHIFT))&DMA_TCD_CITER_ELINKYES_ELINK_MASK)
/* TCD_DLASTSGA Bit Fields */
#define DMA_TCD_DLASTSGA_DLASTSGA_MASK           0xFFFFFFFFu
#define DMA_TCD_DLASTSGA_DLASTSGA_SHIFT          0u
#define DMA_TCD_DLASTSGA_DLASTSGA_WIDTH          32u
#define DMA_TCD_DLASTSGA_DLASTSGA(x)             (((uint32_t)(((uint32_t)(x))<<DMA_TCD_DLASTSGA_DLASTSGA_SHIFT))&DMA_TCD_DLASTSGA_DLASTSGA_MASK)
/* TCD_CSR Bit Fields */
#define DMA_TCD_CSR_START_MASK                   0x1u
#define DMA_TCD_CSR_START_SHIFT                  0u
#define DMA_TCD_CSR_START_WIDTH                  1u
#define DMA_TCD_CSR_START(x)                     (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CSR_START_SHIFT))&DMA_TCD_CSR_START_MASK)
#define DMA_TCD_CSR_INTMAJOR_MASK                0x2u
#define DMA_TCD_CSR_INTMAJOR_SHIFT               1u
#define DMA_TCD_CSR_INTMAJOR_WIDTH               1u
#define DMA_TCD_CSR_INTMAJOR(x)                  (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CSR_INTMAJOR_SHIFT))&DMA_TCD_CSR_INTMAJOR_MASK)
#define DMA_TCD_CSR_INTHALF_MASK                 0x4u
#define DMA_TCD_CSR_INTHALF_SHIFT                2u
#define DMA_TCD_CSR_INTHALF_WIDTH                1u
#define DMA_TCD_CSR_INTHALF(x)                   (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CSR_INTHALF_SHIFT))&DMA_TCD_CSR_INTHALF_MASK)
#define DMA_TCD_CSR_DREQ_MASK                    0x8u
#define DMA_TCD_CSR_DREQ_SHIFT                   3u
#define DMA_TCD_CSR_DREQ_WIDTH                   1u
#define DMA_TCD_CSR_DREQ(x)                      (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CSR_DREQ_SHIFT))&DMA_TCD_CSR_DREQ_MASK)
#define DMA_TCD_CSR_ESG_MASK                     0x10u
#define DMA_TCD_CSR_ESG_SHIFT                    4u
#define DMA_TCD_CSR_ESG_WIDTH                    1u
#define DMA_TCD_CSR_ESG(x)                       (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CSR_ESG_SHIFT))&DMA_TCD_CSR_ESG_MASK)
#define DMA_TCD_CSR_MAJORELINK_MASK              0x20u
#define DMA_TCD_CSR_MAJORELINK_SHIFT             5u
#define DMA_TCD_CSR_MAJORELINK_WIDTH             1u
#define DMA_TCD_CSR_MAJORELINK(x)                (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CSR_MAJORELINK_SHIFT))&DMA_TCD_CSR_MAJORELINK_MASK)
#define DMA_TCD_CSR_ACTIVE_MASK                  0x40u
#define DMA_TCD_CSR_ACTIVE_SHIFT                 6u
#define DMA_TCD_CSR_ACTIVE_WIDTH                 1u
#define DMA_TCD_CSR_ACTIVE(x)                    (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CSR_ACTIVE_SHIFT))&DMA_TCD_CSR_ACTIVE_MASK)
#define DMA_TCD_CSR_DONE_MASK                    0x80u
#define DMA_TCD_CSR_DONE_SHIFT                   7u
#define DMA_TCD_CSR_DONE_WIDTH                   1u
#define DMA_TCD_CSR_DONE(x)                      (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CSR_DONE_SHIFT))&DMA_TCD_CSR_DONE_MASK)
#define DMA_TCD_CSR_MAJORLINKCH_MASK             0x1F00u
#define DMA_TCD_CSR_MAJORLINKCH_SHIFT            8u
#define DMA_TCD_CSR_MAJORLINKCH_WIDTH            5u
#define DMA_TCD_CSR_MAJORLINKCH(x)               (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CSR_MAJORLINKCH_SHIFT))&DMA_TCD_CSR_MAJORLINKCH_MASK)
#define DMA_TCD_CSR_BWC_MASK                     0xC000u
#define DMA_TCD_CSR_BWC_SHIFT                    14u
#define DMA_TCD_CSR_BWC_WIDTH                    2u
#define DMA_TCD_CSR_BWC(x)                       (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CSR_BWC_SHIFT))&DMA_TCD_CSR_BWC_MASK)
/* TCD_BITER_ELINKNO Bit Fields */
#define DMA_TCD_BITER_ELINKNO_BITER_MASK         0x7FFFu
#define DMA_TCD_BITER_ELINKNO_BITER_SHIFT        0u
#define DMA_TCD_BITER_ELINKNO_BITER_WIDTH        15u
#define DMA_TCD_BITER_ELINKNO_BITER(x)           (((uint16_t)(((uint16_t)(x))<<DMA_TCD_BITER_ELINKNO_BITER_SHIFT))&DMA_TCD_BITER_ELINKNO_BITER_MASK)
#define DMA_TCD_BITER_ELINKNO_ELINK_MASK         0x8000u
#define DMA_TCD_BITER_ELINKNO_ELINK_SHIFT        15u
#define DMA_TCD_BITER_ELINKNO_ELINK_WIDTH        1u
#define DMA_TCD_BITER_ELINKNO_ELINK(x)           (((uint16_t)(((uint16_t)(x))<<DMA_TCD_BITER_ELINKNO_ELINK_SHIFT))&DMA_TCD_BITER_ELINKNO_ELINK_MASK)
/* TCD_BITER_ELINKYES Bit Fields */
#define DMA_TCD_BITER_ELINKYES_BITER_MASK        0x1FFu
#define DMA_TCD_BITER_ELINKYES_BITER_SHIFT       0u
#define DMA_TCD_BITER_ELINKYES_BITER_WIDTH       9u
#define DMA_TCD_BITER_ELINKYES_BITER(x)          (((uint16_t)(((uint16_t)(x))<<DMA_TCD_BITER_ELINKYES_BITER_SHIFT))&DMA_TCD_BITER_ELINKYES_BITER_MASK)
#define DMA_TCD_BITER_ELINKYES_LINKCH_MASK       0x3E00u
#define DMA_TCD_BITER_ELINKYES_LINKCH_SHIFT      9u
#define DMA_TCD_BITER_ELINKYES_LINKCH_WIDTH      5u
#define DMA_TCD_BITER_ELINKYES_LINKCH(x)         (((uint16_t)(((uint16_t)(x))<<DMA_TCD_BITER_ELINKYES_LINKCH_SHIFT))&DMA_TCD_BITER_ELINKYES_LINKCH_MASK)
#define DMA_TCD_BITER_ELINKYES_ELINK_MASK        0x8000u
#define DMA_TCD_BITER_ELINKYES_ELINK_SHIFT       15u
#define DMA_TCD_BITER_ELINKYES_ELINK_WIDTH       1u
#define DMA_TCD_BITER_ELINKYES_ELINK(x)          (((uint16_t)(((uint16_t)(x))<<DMA_TCD_BITER_ELINKYES_ELINK_SHIFT))&DMA_TCD_BITER_ELINKYES_ELINK_MASK)

/*!
 * @}
 */ /* end of group DMA_Register_Masks */


/*!
 * @}
 */ /* end of group DMA_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- DMAMUX Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup DMAMUX_Peripheral_Access_Layer DMAMUX Peripheral Access Layer
 * @{
 */


/** DMAMUX - Size of Registers Arrays */
#define DMAMUX_CHCFG_COUNT                       16u

/** DMAMUX - Register Layout Typedef */
typedef struct {
  __IO uint8_t CHCFG[DMAMUX_CHCFG_COUNT];          /**< Channel Configuration register, array offset: 0x0, array step: 0x1 */
} DMAMUX_Type, *DMAMUX_MemMapPtr;

 /** Number of instances of the DMAMUX module. */
#define DMAMUX_INSTANCE_COUNT                    (2u)


/* DMAMUX - Peripheral instance base addresses */
/** Peripheral DMAMUX_0 base address */
#define DMAMUX_0_BASE                            (0x40031000u)
/** Peripheral DMAMUX_0 base pointer */
#define DMAMUX_0                                 ((DMAMUX_Type *)DMAMUX_0_BASE)
/** Peripheral DMAMUX_1 base address */
#define DMAMUX_1_BASE                            (0x400A1000u)
/** Peripheral DMAMUX_1 base pointer */
#define DMAMUX_1                                 ((DMAMUX_Type *)DMAMUX_1_BASE)
/** Array initializer of DMAMUX peripheral base addresses */
#define DMAMUX_BASE_ADDRS                        { DMAMUX_0_BASE, DMAMUX_1_BASE }
/** Array initializer of DMAMUX peripheral base pointers */
#define DMAMUX_BASE_PTRS                         { DMAMUX_0, DMAMUX_1 }

/* ----------------------------------------------------------------------------
   -- DMAMUX Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup DMAMUX_Register_Masks DMAMUX Register Masks
 * @{
 */

/* CHCFG Bit Fields */
#define DMAMUX_CHCFG_SOURCE_MASK                 0x3Fu
#define DMAMUX_CHCFG_SOURCE_SHIFT                0u
#define DMAMUX_CHCFG_SOURCE_WIDTH                6u
#define DMAMUX_CHCFG_SOURCE(x)                   (((uint8_t)(((uint8_t)(x))<<DMAMUX_CHCFG_SOURCE_SHIFT))&DMAMUX_CHCFG_SOURCE_MASK)
#define DMAMUX_CHCFG_TRIG_MASK                   0x40u
#define DMAMUX_CHCFG_TRIG_SHIFT                  6u
#define DMAMUX_CHCFG_TRIG_WIDTH                  1u
#define DMAMUX_CHCFG_TRIG(x)                     (((uint8_t)(((uint8_t)(x))<<DMAMUX_CHCFG_TRIG_SHIFT))&DMAMUX_CHCFG_TRIG_MASK)
#define DMAMUX_CHCFG_ENBL_MASK                   0x80u
#define DMAMUX_CHCFG_ENBL_SHIFT                  7u
#define DMAMUX_CHCFG_ENBL_WIDTH                  1u
#define DMAMUX_CHCFG_ENBL(x)                     (((uint8_t)(((uint8_t)(x))<<DMAMUX_CHCFG_ENBL_SHIFT))&DMAMUX_CHCFG_ENBL_MASK)

/*!
 * @}
 */ /* end of group DMAMUX_Register_Masks */


/*!
 * @}
 */ /* end of group DMAMUX_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- EIM Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup EIM_Peripheral_Access_Layer EIM Peripheral Access Layer
 * @{
 */


/** EIM - Size of Registers Arrays */
#define EIM_EICHDn_COUNT                         13u

/** EIM - Register Layout Typedef */
typedef struct {
  __IO uint32_t EIMCR;                             /**< Error Injection Module Configuration Register, offset: 0x0 */
  __IO uint32_t EICHEN;                            /**< Error Injection Channel Enable register, offset: 0x4 */
       uint8_t RESERVED_0[248];
  struct {                                         /* offset: 0x100, array step: 0x100 */
    __IO uint32_t WORD0;                             /**< Error Injection Channel Descriptor, Word0, array offset: 0x100, array step: 0x100 */
    __IO uint32_t WORD1;                             /**< Error Injection Channel Descriptor, Word1, array offset: 0x104, array step: 0x100 */
    __IO uint32_t WORD2;                             /**< Error Injection Channel Descriptor, Word2, array offset: 0x108, array step: 0x100 */
         uint8_t RESERVED_0[244];
  } EICHDn[EIM_EICHDn_COUNT];
} EIM_Type, *EIM_MemMapPtr;

 /** Number of instances of the EIM module. */
#define EIM_INSTANCE_COUNT                       (1u)


/* EIM - Peripheral instance base addresses */
/** Peripheral EIM base address */
#define EIM_BASE                                 (0x40088000u)
/** Peripheral EIM base pointer */
#define EIM                                      ((EIM_Type *)EIM_BASE)
/** Array initializer of EIM peripheral base addresses */
#define EIM_BASE_ADDRS                           { EIM_BASE }
/** Array initializer of EIM peripheral base pointers */
#define EIM_BASE_PTRS                            { EIM }

/* ----------------------------------------------------------------------------
   -- EIM Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup EIM_Register_Masks EIM Register Masks
 * @{
 */

/* EIMCR Bit Fields */
#define EIM_EIMCR_GEIEN_MASK                     0x1u
#define EIM_EIMCR_GEIEN_SHIFT                    0u
#define EIM_EIMCR_GEIEN_WIDTH                    1u
#define EIM_EIMCR_GEIEN(x)                       (((uint32_t)(((uint32_t)(x))<<EIM_EIMCR_GEIEN_SHIFT))&EIM_EIMCR_GEIEN_MASK)
/* EICHEN Bit Fields */
#define EIM_EICHEN_EICH12EN_MASK                 0x80000u
#define EIM_EICHEN_EICH12EN_SHIFT                19u
#define EIM_EICHEN_EICH12EN_WIDTH                1u
#define EIM_EICHEN_EICH12EN(x)                   (((uint32_t)(((uint32_t)(x))<<EIM_EICHEN_EICH12EN_SHIFT))&EIM_EICHEN_EICH12EN_MASK)
#define EIM_EICHEN_EICH11EN_MASK                 0x100000u
#define EIM_EICHEN_EICH11EN_SHIFT                20u
#define EIM_EICHEN_EICH11EN_WIDTH                1u
#define EIM_EICHEN_EICH11EN(x)                   (((uint32_t)(((uint32_t)(x))<<EIM_EICHEN_EICH11EN_SHIFT))&EIM_EICHEN_EICH11EN_MASK)
#define EIM_EICHEN_EICH10EN_MASK                 0x200000u
#define EIM_EICHEN_EICH10EN_SHIFT                21u
#define EIM_EICHEN_EICH10EN_WIDTH                1u
#define EIM_EICHEN_EICH10EN(x)                   (((uint32_t)(((uint32_t)(x))<<EIM_EICHEN_EICH10EN_SHIFT))&EIM_EICHEN_EICH10EN_MASK)
#define EIM_EICHEN_EICH9EN_MASK                  0x400000u
#define EIM_EICHEN_EICH9EN_SHIFT                 22u
#define EIM_EICHEN_EICH9EN_WIDTH                 1u
#define EIM_EICHEN_EICH9EN(x)                    (((uint32_t)(((uint32_t)(x))<<EIM_EICHEN_EICH9EN_SHIFT))&EIM_EICHEN_EICH9EN_MASK)
#define EIM_EICHEN_EICH8EN_MASK                  0x800000u
#define EIM_EICHEN_EICH8EN_SHIFT                 23u
#define EIM_EICHEN_EICH8EN_WIDTH                 1u
#define EIM_EICHEN_EICH8EN(x)                    (((uint32_t)(((uint32_t)(x))<<EIM_EICHEN_EICH8EN_SHIFT))&EIM_EICHEN_EICH8EN_MASK)
#define EIM_EICHEN_EICH7EN_MASK                  0x1000000u
#define EIM_EICHEN_EICH7EN_SHIFT                 24u
#define EIM_EICHEN_EICH7EN_WIDTH                 1u
#define EIM_EICHEN_EICH7EN(x)                    (((uint32_t)(((uint32_t)(x))<<EIM_EICHEN_EICH7EN_SHIFT))&EIM_EICHEN_EICH7EN_MASK)
#define EIM_EICHEN_EICH6EN_MASK                  0x2000000u
#define EIM_EICHEN_EICH6EN_SHIFT                 25u
#define EIM_EICHEN_EICH6EN_WIDTH                 1u
#define EIM_EICHEN_EICH6EN(x)                    (((uint32_t)(((uint32_t)(x))<<EIM_EICHEN_EICH6EN_SHIFT))&EIM_EICHEN_EICH6EN_MASK)
#define EIM_EICHEN_EICH5EN_MASK                  0x4000000u
#define EIM_EICHEN_EICH5EN_SHIFT                 26u
#define EIM_EICHEN_EICH5EN_WIDTH                 1u
#define EIM_EICHEN_EICH5EN(x)                    (((uint32_t)(((uint32_t)(x))<<EIM_EICHEN_EICH5EN_SHIFT))&EIM_EICHEN_EICH5EN_MASK)
#define EIM_EICHEN_EICH4EN_MASK                  0x8000000u
#define EIM_EICHEN_EICH4EN_SHIFT                 27u
#define EIM_EICHEN_EICH4EN_WIDTH                 1u
#define EIM_EICHEN_EICH4EN(x)                    (((uint32_t)(((uint32_t)(x))<<EIM_EICHEN_EICH4EN_SHIFT))&EIM_EICHEN_EICH4EN_MASK)
#define EIM_EICHEN_EICH3EN_MASK                  0x10000000u
#define EIM_EICHEN_EICH3EN_SHIFT                 28u
#define EIM_EICHEN_EICH3EN_WIDTH                 1u
#define EIM_EICHEN_EICH3EN(x)                    (((uint32_t)(((uint32_t)(x))<<EIM_EICHEN_EICH3EN_SHIFT))&EIM_EICHEN_EICH3EN_MASK)
#define EIM_EICHEN_EICH2EN_MASK                  0x20000000u
#define EIM_EICHEN_EICH2EN_SHIFT                 29u
#define EIM_EICHEN_EICH2EN_WIDTH                 1u
#define EIM_EICHEN_EICH2EN(x)                    (((uint32_t)(((uint32_t)(x))<<EIM_EICHEN_EICH2EN_SHIFT))&EIM_EICHEN_EICH2EN_MASK)
#define EIM_EICHEN_EICH1EN_MASK                  0x40000000u
#define EIM_EICHEN_EICH1EN_SHIFT                 30u
#define EIM_EICHEN_EICH1EN_WIDTH                 1u
#define EIM_EICHEN_EICH1EN(x)                    (((uint32_t)(((uint32_t)(x))<<EIM_EICHEN_EICH1EN_SHIFT))&EIM_EICHEN_EICH1EN_MASK)
#define EIM_EICHEN_EICH0EN_MASK                  0x80000000u
#define EIM_EICHEN_EICH0EN_SHIFT                 31u
#define EIM_EICHEN_EICH0EN_WIDTH                 1u
#define EIM_EICHEN_EICH0EN(x)                    (((uint32_t)(((uint32_t)(x))<<EIM_EICHEN_EICH0EN_SHIFT))&EIM_EICHEN_EICH0EN_MASK)
/* EICHDn_WORD0 Bit Fields */
#define EIM_EICHDn_WORD0_CHKBIT_MASK_MASK        0xFFFFF000u
#define EIM_EICHDn_WORD0_CHKBIT_MASK_SHIFT       12u
#define EIM_EICHDn_WORD0_CHKBIT_MASK_WIDTH       20u
#define EIM_EICHDn_WORD0_CHKBIT_MASK(x)          (((uint32_t)(((uint32_t)(x))<<EIM_EICHDn_WORD0_CHKBIT_MASK_SHIFT))&EIM_EICHDn_WORD0_CHKBIT_MASK_MASK)
/* EICHDn_WORD1 Bit Fields */
#define EIM_EICHDn_WORD1_B0_3DATA_MASK_MASK      0xFFFFFFFFu
#define EIM_EICHDn_WORD1_B0_3DATA_MASK_SHIFT     0u
#define EIM_EICHDn_WORD1_B0_3DATA_MASK_WIDTH     32u
#define EIM_EICHDn_WORD1_B0_3DATA_MASK(x)        (((uint32_t)(((uint32_t)(x))<<EIM_EICHDn_WORD1_B0_3DATA_MASK_SHIFT))&EIM_EICHDn_WORD1_B0_3DATA_MASK_MASK)
/* EICHDn_WORD2 Bit Fields */
#define EIM_EICHDn_WORD2_B4_7DATA_MASK_MASK      0xFFFFFFFFu
#define EIM_EICHDn_WORD2_B4_7DATA_MASK_SHIFT     0u
#define EIM_EICHDn_WORD2_B4_7DATA_MASK_WIDTH     32u
#define EIM_EICHDn_WORD2_B4_7DATA_MASK(x)        (((uint32_t)(((uint32_t)(x))<<EIM_EICHDn_WORD2_B4_7DATA_MASK_SHIFT))&EIM_EICHDn_WORD2_B4_7DATA_MASK_MASK)

/*!
 * @}
 */ /* end of group EIM_Register_Masks */


/*!
 * @}
 */ /* end of group EIM_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- ENET Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup ENET_Peripheral_Access_Layer ENET Peripheral Access Layer
 * @{
 */


/** ENET - Size of Registers Arrays */
#define ENET_TXIC_COUNT                          3u
#define ENET_RXIC_COUNT                          3u
#define ENET_RCMR_COUNT                          2u
#define ENET_DMACFG_COUNT                        2u
#define ENET_CHANNEL_COUNT                       4u

/** ENET - Register Layout Typedef */
typedef struct {
       uint8_t RESERVED_0[4];
  __IO uint32_t EIR;                               /**< Interrupt Event Register, offset: 0x4 */
  __IO uint32_t EIMR;                              /**< Interrupt Mask Register, offset: 0x8 */
       uint8_t RESERVED_1[4];
  __IO uint32_t RDAR;                              /**< Receive Descriptor Active Register - Ring 0, offset: 0x10 */
  __IO uint32_t TDAR;                              /**< Transmit Descriptor Active Register - Ring 0, offset: 0x14 */
       uint8_t RESERVED_2[12];
  __IO uint32_t ECR;                               /**< Ethernet Control Register, offset: 0x24 */
       uint8_t RESERVED_3[24];
  __IO uint32_t MMFR;                              /**< MII Management Frame Register, offset: 0x40 */
  __IO uint32_t MSCR;                              /**< MII Speed Control Register, offset: 0x44 */
       uint8_t RESERVED_4[28];
  __IO uint32_t MIBC;                              /**< MIB Control Register, offset: 0x64 */
       uint8_t RESERVED_5[28];
  __IO uint32_t RCR;                               /**< Receive Control Register, offset: 0x84 */
       uint8_t RESERVED_6[60];
  __IO uint32_t TCR;                               /**< Transmit Control Register, offset: 0xC4 */
       uint8_t RESERVED_7[28];
  __IO uint32_t PALR;                              /**< Physical Address Lower Register, offset: 0xE4 */
  __IO uint32_t PAUR;                              /**< Physical Address Upper Register, offset: 0xE8 */
  __IO uint32_t OPD;                               /**< Opcode/Pause Duration Register, offset: 0xEC */
  __IO uint32_t TXIC[ENET_TXIC_COUNT];             /**< Transmit Interrupt Coalescing Register, array offset: 0xF0, array step: 0x4 */
       uint8_t RESERVED_8[4];
  __IO uint32_t RXIC[ENET_RXIC_COUNT];             /**< Receive Interrupt Coalescing Register, array offset: 0x100, array step: 0x4 */
       uint8_t RESERVED_9[12];
  __IO uint32_t IAUR;                              /**< Descriptor Individual Upper Address Register, offset: 0x118 */
  __IO uint32_t IALR;                              /**< Descriptor Individual Lower Address Register, offset: 0x11C */
  __IO uint32_t GAUR;                              /**< Descriptor Group Upper Address Register, offset: 0x120 */
  __IO uint32_t GALR;                              /**< Descriptor Group Lower Address Register, offset: 0x124 */
       uint8_t RESERVED_10[28];
  __IO uint32_t TFWR;                              /**< Transmit FIFO Watermark Register, offset: 0x144 */
       uint8_t RESERVED_11[24];
  __IO uint32_t RDSR1;                             /**< Receive Descriptor Ring 1 Start Register, offset: 0x160 */
  __IO uint32_t TDSR1;                             /**< Transmit Buffer Descriptor Ring 1 Start Register, offset: 0x164 */
  __IO uint32_t MRBR1;                             /**< Maximum Receive Buffer Size Register - Ring 1, offset: 0x168 */
  __IO uint32_t RDSR2;                             /**< Receive Descriptor Ring 2 Start Register, offset: 0x16C */
  __IO uint32_t TDSR2;                             /**< Transmit Buffer Descriptor Ring 2 Start Register, offset: 0x170 */
  __IO uint32_t MRBR2;                             /**< Maximum Receive Buffer Size Register - Ring 2, offset: 0x174 */
       uint8_t RESERVED_12[8];
  __IO uint32_t RDSR;                              /**< Receive Descriptor Ring 0 Start Register, offset: 0x180 */
  __IO uint32_t TDSR;                              /**< Transmit Buffer Descriptor Ring 0 Start Register, offset: 0x184 */
  __IO uint32_t MRBR;                              /**< Maximum Receive Buffer Size Register - Ring 0, offset: 0x188 */
       uint8_t RESERVED_13[4];
  __IO uint32_t RSFL;                              /**< Receive FIFO Section Full Threshold, offset: 0x190 */
  __IO uint32_t RSEM;                              /**< Receive FIFO Section Empty Threshold, offset: 0x194 */
  __IO uint32_t RAEM;                              /**< Receive FIFO Almost Empty Threshold, offset: 0x198 */
  __IO uint32_t RAFL;                              /**< Receive FIFO Almost Full Threshold, offset: 0x19C */
  __IO uint32_t TSEM;                              /**< Transmit FIFO Section Empty Threshold, offset: 0x1A0 */
  __IO uint32_t TAEM;                              /**< Transmit FIFO Almost Empty Threshold, offset: 0x1A4 */
  __IO uint32_t TAFL;                              /**< Transmit FIFO Almost Full Threshold, offset: 0x1A8 */
  __IO uint32_t TIPG;                              /**< Transmit Inter-Packet Gap, offset: 0x1AC */
  __IO uint32_t FTRL;                              /**< Frame Truncation Length, offset: 0x1B0 */
       uint8_t RESERVED_14[12];
  __IO uint32_t TACC;                              /**< Transmit Accelerator Function Configuration, offset: 0x1C0 */
  __IO uint32_t RACC;                              /**< Receive Accelerator Function Configuration, offset: 0x1C4 */
  __IO uint32_t RCMR[ENET_RCMR_COUNT];             /**< Receive Classification Match Register for Class n, array offset: 0x1C8, array step: 0x4 */
       uint8_t RESERVED_15[8];
  __IO uint32_t DMACFG[ENET_DMACFG_COUNT];         /**< DMA Class Based Configuration, array offset: 0x1D8, array step: 0x4 */
  __IO uint32_t RDAR1;                             /**< Receive Descriptor Active Register - Ring 1, offset: 0x1E0 */
  __IO uint32_t TDAR1;                             /**< Transmit Descriptor Active Register - Ring 1, offset: 0x1E4 */
  __IO uint32_t RDAR2;                             /**< Receive Descriptor Active Register - Ring 2, offset: 0x1E8 */
  __IO uint32_t TDAR2;                             /**< Transmit Descriptor Active Register - Ring 2, offset: 0x1EC */
  __IO uint32_t QOS;                               /**< QOS Scheme, offset: 0x1F0 */
       uint8_t RESERVED_16[12];
  __I  uint32_t RMON_T_DROP;                       /**< Reserved Statistic Register, offset: 0x200 */
  __I  uint32_t RMON_T_PACKETS;                    /**< Tx Packet Count Statistic Register, offset: 0x204 */
  __I  uint32_t RMON_T_BC_PKT;                     /**< Tx Broadcast Packets Statistic Register, offset: 0x208 */
  __I  uint32_t RMON_T_MC_PKT;                     /**< Tx Multicast Packets Statistic Register, offset: 0x20C */
  __I  uint32_t RMON_T_CRC_ALIGN;                  /**< Tx Packets with CRC/Align Error Statistic Register, offset: 0x210 */
  __I  uint32_t RMON_T_UNDERSIZE;                  /**< Tx Packets Less Than Bytes and Good CRC Statistic Register, offset: 0x214 */
  __I  uint32_t RMON_T_OVERSIZE;                   /**< Tx Packets GT MAX_FL bytes and Good CRC Statistic Register, offset: 0x218 */
  __I  uint32_t RMON_T_FRAG;                       /**< Tx Packets Less Than 64 Bytes and Bad CRC Statistic Register, offset: 0x21C */
  __I  uint32_t RMON_T_JAB;                        /**< Tx Packets Greater Than MAX_FL bytes and Bad CRC Statistic Register, offset: 0x220 */
  __I  uint32_t RMON_T_COL;                        /**< Tx Collision Count Statistic Register, offset: 0x224 */
  __I  uint32_t RMON_T_P64;                        /**< Tx 64-Byte Packets Statistic Register, offset: 0x228 */
  __I  uint32_t RMON_T_P65TO127;                   /**< Tx 65- to 127-byte Packets Statistic Register, offset: 0x22C */
  __I  uint32_t RMON_T_P128TO255;                  /**< Tx 128- to 255-byte Packets Statistic Register, offset: 0x230 */
  __I  uint32_t RMON_T_P256TO511;                  /**< Tx 256- to 511-byte Packets Statistic Register, offset: 0x234 */
  __I  uint32_t RMON_T_P512TO1023;                 /**< Tx 512- to 1023-byte Packets Statistic Register, offset: 0x238 */
  __I  uint32_t RMON_T_P1024TO2047;                /**< Tx 1024- to 2047-byte Packets Statistic Register, offset: 0x23C */
  __I  uint32_t RMON_T_P_GTE2048;                  /**< Tx Packets Greater Than 2048 Bytes Statistic Register, offset: 0x240 */
  __I  uint32_t RMON_T_OCTETS;                     /**< Tx Octets Statistic Register, offset: 0x244 */
  __I  uint32_t IEEE_T_DROP;                       /**< Reserved Statistic Register, offset: 0x248 */
  __I  uint32_t IEEE_T_FRAME_OK;                   /**< Frames Transmitted OK Statistic Register, offset: 0x24C */
  __I  uint32_t IEEE_T_1COL;                       /**< Frames Transmitted with Single Collision Statistic Register, offset: 0x250 */
  __I  uint32_t IEEE_T_MCOL;                       /**< Frames Transmitted with Multiple Collisions Statistic Register, offset: 0x254 */
  __I  uint32_t IEEE_T_DEF;                        /**< Frames Transmitted after Deferral Delay Statistic Register, offset: 0x258 */
  __I  uint32_t IEEE_T_LCOL;                       /**< Frames Transmitted with Late Collision Statistic Register, offset: 0x25C */
  __I  uint32_t IEEE_T_EXCOL;                      /**< Frames Transmitted with Excessive Collisions Statistic Register, offset: 0x260 */
  __I  uint32_t IEEE_T_MACERR;                     /**< Frames Transmitted with Tx FIFO Underrun Statistic Register, offset: 0x264 */
  __I  uint32_t IEEE_T_CSERR;                      /**< Frames Transmitted with Carrier Sense Error Statistic Register, offset: 0x268 */
  __I  uint32_t IEEE_T_SQE;                        /**< Reserved Statistic Register, offset: 0x26C */
  __I  uint32_t IEEE_T_FDXFC;                      /**< Flow Control Pause Frames Transmitted Statistic Register, offset: 0x270 */
  __I  uint32_t IEEE_T_OCTETS_OK;                  /**< Octet Count for Frames Transmitted w/o Error Statistic Register, offset: 0x274 */
       uint8_t RESERVED_17[12];
  __I  uint32_t RMON_R_PACKETS;                    /**< Rx Packet Count Statistic Register, offset: 0x284 */
  __I  uint32_t RMON_R_BC_PKT;                     /**< Rx Broadcast Packets Statistic Register, offset: 0x288 */
  __I  uint32_t RMON_R_MC_PKT;                     /**< Rx Multicast Packets Statistic Register, offset: 0x28C */
  __I  uint32_t RMON_R_CRC_ALIGN;                  /**< Rx Packets with CRC/Align Error Statistic Register, offset: 0x290 */
  __I  uint32_t RMON_R_UNDERSIZE;                  /**< Rx Packets with Less Than 64 Bytes and Good CRC Statistic Register, offset: 0x294 */
  __I  uint32_t RMON_R_OVERSIZE;                   /**< Rx Packets Greater Than MAX_FL and Good CRC Statistic Register, offset: 0x298 */
  __I  uint32_t RMON_R_FRAG;                       /**< Rx Packets Less Than 64 Bytes and Bad CRC Statistic Register, offset: 0x29C */
  __I  uint32_t RMON_R_JAB;                        /**< Rx Packets Greater Than MAX_FL Bytes and Bad CRC Statistic Register, offset: 0x2A0 */
  __I  uint32_t RMON_R_RESVD_0;                    /**< Reserved Statistic Register, offset: 0x2A4 */
  __I  uint32_t RMON_R_P64;                        /**< Rx 64-Byte Packets Statistic Register, offset: 0x2A8 */
  __I  uint32_t RMON_R_P65TO127;                   /**< Rx 65- to 127-Byte Packets Statistic Register, offset: 0x2AC */
  __I  uint32_t RMON_R_P128TO255;                  /**< Rx 128- to 255-Byte Packets Statistic Register, offset: 0x2B0 */
  __I  uint32_t RMON_R_P256TO511;                  /**< Rx 256- to 511-Byte Packets Statistic Register, offset: 0x2B4 */
  __I  uint32_t RMON_R_P512TO1023;                 /**< Rx 512- to 1023-Byte Packets Statistic Register, offset: 0x2B8 */
  __I  uint32_t RMON_R_P1024TO2047;                /**< Rx 1024- to 2047-Byte Packets Statistic Register, offset: 0x2BC */
  __I  uint32_t RMON_R_P_GTE2048;                  /**< Rx Packets Greater than 2048 Bytes Statistic Register, offset: 0x2C0 */
  __I  uint32_t RMON_R_OCTETS;                     /**< Rx Octets Statistic Register, offset: 0x2C4 */
  __I  uint32_t IEEE_R_DROP;                       /**< Frames not Counted Correctly Statistic Register, offset: 0x2C8 */
  __I  uint32_t IEEE_R_FRAME_OK;                   /**< Frames Received OK Statistic Register, offset: 0x2CC */
  __I  uint32_t IEEE_R_CRC;                        /**< Frames Received with CRC Error Statistic Register, offset: 0x2D0 */
  __I  uint32_t IEEE_R_ALIGN;                      /**< Frames Received with Alignment Error Statistic Register, offset: 0x2D4 */
  __I  uint32_t IEEE_R_MACERR;                     /**< Receive FIFO Overflow Count Statistic Register, offset: 0x2D8 */
  __I  uint32_t IEEE_R_FDXFC;                      /**< Flow Control Pause Frames Received Statistic Register, offset: 0x2DC */
  __I  uint32_t IEEE_R_OCTETS_OK;                  /**< Octet Count for Frames Received without Error Statistic Register, offset: 0x2E0 */
       uint8_t RESERVED_18[284];
  __IO uint32_t ATCR;                              /**< Adjustable Timer Control Register, offset: 0x400 */
  __IO uint32_t ATVR;                              /**< Timer Value Register, offset: 0x404 */
  __IO uint32_t ATOFF;                             /**< Timer Offset Register, offset: 0x408 */
  __IO uint32_t ATPER;                             /**< Timer Period Register, offset: 0x40C */
  __IO uint32_t ATCOR;                             /**< Timer Correction Register, offset: 0x410 */
  __IO uint32_t ATINC;                             /**< Time-Stamping Clock Period Register, offset: 0x414 */
  __I  uint32_t ATSTMP;                            /**< Timestamp of Last Transmitted Frame, offset: 0x418 */
       uint8_t RESERVED_19[356];
  __IO uint32_t MDATA;                             /**< Pattern Match Data Register, offset: 0x580 */
  __IO uint32_t MMASK;                             /**< Match Entry Mask Register, offset: 0x584 */
  __IO uint32_t MCONFIG;                           /**< Match Entry Rules Configuration Register, offset: 0x588 */
  __IO uint32_t MENTRYRW;                          /**< Match Entry Read/Write Command Register, offset: 0x58C */
  __IO uint32_t RXPCTL;                            /**< Receive Parser Control Register, offset: 0x590 */
  __IO uint32_t MAXFRMOFF;                         /**< Maximum Frame Offset, offset: 0x594 */
  __I  uint32_t RXPARST;                           /**< Receive Parser Status, offset: 0x598 */
       uint8_t RESERVED_20[4];
  __I  uint32_t PARSDSCD;                          /**< Parser Discard Count, offset: 0x5A0 */
  __I  uint32_t PRSACPT0;                          /**< Parser Accept Count 0, offset: 0x5A4 */
  __I  uint32_t PRSRJCT0;                          /**< Parser Reject Count 0, offset: 0x5A8 */
  __I  uint32_t PRSACPT1;                          /**< Parser Accept Count 1, offset: 0x5AC */
  __I  uint32_t PRSRJCT1;                          /**< Parser Reject Count 1, offset: 0x5B0 */
  __I  uint32_t PRSACPT2;                          /**< Parser Accept Count 2, offset: 0x5B4 */
  __I  uint32_t PRSRJCT2;                          /**< Parser Reject Count 2, offset: 0x5B8 */
       uint8_t RESERVED_21[72];
  __IO uint32_t TGSR;                              /**< Timer Global Status Register, offset: 0x604 */
  struct {                                         /* offset: 0x608, array step: 0x8 */
    __IO uint32_t TCSR;                              /**< Timer Control Status Register, array offset: 0x608, array step: 0x8 */
    __IO uint32_t TCCR;                              /**< Timer Compare Capture Register, array offset: 0x60C, array step: 0x8 */
  } CHANNEL[ENET_CHANNEL_COUNT];
} ENET_Type, *ENET_MemMapPtr;

 /** Number of instances of the ENET module. */
#define ENET_INSTANCE_COUNT                      (1u)


/* ENET - Peripheral instance base addresses */
/** Peripheral ENET base address */
#define ENET_BASE                                (0x40032000u)
/** Peripheral ENET base pointer */
#define ENET                                     ((ENET_Type *)ENET_BASE)
/** Array initializer of ENET peripheral base addresses */
#define ENET_BASE_ADDRS                          { ENET_BASE }
/** Array initializer of ENET peripheral base pointers */
#define ENET_BASE_PTRS                           { ENET }

/** Interrupt vectors for the ENET peripheral type */
#define ENET_TIMER_IRQS                          { ENET_Timer_IRQn }
#define ENET_TX_IRQS                             { ENET_TX_IRQn }
#define ENET_RX_IRQS                             { ENET_RX_IRQn }
#define ENET_ERR_IRQS                            { ENET_Err_IRQn }

/* ----------------------------------------------------------------------------
   -- ENET Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup ENET_Register_Masks ENET Register Masks
 * @{
 */

/* EIR Bit Fields */
#define ENET_EIR_RXB1_MASK                       0x1u
#define ENET_EIR_RXB1_SHIFT                      0u
#define ENET_EIR_RXB1_WIDTH                      1u
#define ENET_EIR_RXB1(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIR_RXB1_SHIFT))&ENET_EIR_RXB1_MASK)
#define ENET_EIR_RXF1_MASK                       0x2u
#define ENET_EIR_RXF1_SHIFT                      1u
#define ENET_EIR_RXF1_WIDTH                      1u
#define ENET_EIR_RXF1(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIR_RXF1_SHIFT))&ENET_EIR_RXF1_MASK)
#define ENET_EIR_TXB1_MASK                       0x4u
#define ENET_EIR_TXB1_SHIFT                      2u
#define ENET_EIR_TXB1_WIDTH                      1u
#define ENET_EIR_TXB1(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIR_TXB1_SHIFT))&ENET_EIR_TXB1_MASK)
#define ENET_EIR_TXF1_MASK                       0x8u
#define ENET_EIR_TXF1_SHIFT                      3u
#define ENET_EIR_TXF1_WIDTH                      1u
#define ENET_EIR_TXF1(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIR_TXF1_SHIFT))&ENET_EIR_TXF1_MASK)
#define ENET_EIR_RXB2_MASK                       0x10u
#define ENET_EIR_RXB2_SHIFT                      4u
#define ENET_EIR_RXB2_WIDTH                      1u
#define ENET_EIR_RXB2(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIR_RXB2_SHIFT))&ENET_EIR_RXB2_MASK)
#define ENET_EIR_RXF2_MASK                       0x20u
#define ENET_EIR_RXF2_SHIFT                      5u
#define ENET_EIR_RXF2_WIDTH                      1u
#define ENET_EIR_RXF2(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIR_RXF2_SHIFT))&ENET_EIR_RXF2_MASK)
#define ENET_EIR_TXB2_MASK                       0x40u
#define ENET_EIR_TXB2_SHIFT                      6u
#define ENET_EIR_TXB2_WIDTH                      1u
#define ENET_EIR_TXB2(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIR_TXB2_SHIFT))&ENET_EIR_TXB2_MASK)
#define ENET_EIR_TXF2_MASK                       0x80u
#define ENET_EIR_TXF2_SHIFT                      7u
#define ENET_EIR_TXF2_WIDTH                      1u
#define ENET_EIR_TXF2(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIR_TXF2_SHIFT))&ENET_EIR_TXF2_MASK)
#define ENET_EIR_PARSRF_MASK                     0x200u
#define ENET_EIR_PARSRF_SHIFT                    9u
#define ENET_EIR_PARSRF_WIDTH                    1u
#define ENET_EIR_PARSRF(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_EIR_PARSRF_SHIFT))&ENET_EIR_PARSRF_MASK)
#define ENET_EIR_PARSERR_MASK                    0x400u
#define ENET_EIR_PARSERR_SHIFT                   10u
#define ENET_EIR_PARSERR_WIDTH                   1u
#define ENET_EIR_PARSERR(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_EIR_PARSERR_SHIFT))&ENET_EIR_PARSERR_MASK)
#define ENET_EIR_RXFLUSH_0_MASK                  0x1000u
#define ENET_EIR_RXFLUSH_0_SHIFT                 12u
#define ENET_EIR_RXFLUSH_0_WIDTH                 1u
#define ENET_EIR_RXFLUSH_0(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_EIR_RXFLUSH_0_SHIFT))&ENET_EIR_RXFLUSH_0_MASK)
#define ENET_EIR_RXFLUSH_1_MASK                  0x2000u
#define ENET_EIR_RXFLUSH_1_SHIFT                 13u
#define ENET_EIR_RXFLUSH_1_WIDTH                 1u
#define ENET_EIR_RXFLUSH_1(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_EIR_RXFLUSH_1_SHIFT))&ENET_EIR_RXFLUSH_1_MASK)
#define ENET_EIR_RXFLUSH_2_MASK                  0x4000u
#define ENET_EIR_RXFLUSH_2_SHIFT                 14u
#define ENET_EIR_RXFLUSH_2_WIDTH                 1u
#define ENET_EIR_RXFLUSH_2(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_EIR_RXFLUSH_2_SHIFT))&ENET_EIR_RXFLUSH_2_MASK)
#define ENET_EIR_TS_TIMER_MASK                   0x8000u
#define ENET_EIR_TS_TIMER_SHIFT                  15u
#define ENET_EIR_TS_TIMER_WIDTH                  1u
#define ENET_EIR_TS_TIMER(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_EIR_TS_TIMER_SHIFT))&ENET_EIR_TS_TIMER_MASK)
#define ENET_EIR_TS_AVAIL_MASK                   0x10000u
#define ENET_EIR_TS_AVAIL_SHIFT                  16u
#define ENET_EIR_TS_AVAIL_WIDTH                  1u
#define ENET_EIR_TS_AVAIL(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_EIR_TS_AVAIL_SHIFT))&ENET_EIR_TS_AVAIL_MASK)
#define ENET_EIR_WAKEUP_MASK                     0x20000u
#define ENET_EIR_WAKEUP_SHIFT                    17u
#define ENET_EIR_WAKEUP_WIDTH                    1u
#define ENET_EIR_WAKEUP(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_EIR_WAKEUP_SHIFT))&ENET_EIR_WAKEUP_MASK)
#define ENET_EIR_PLR_MASK                        0x40000u
#define ENET_EIR_PLR_SHIFT                       18u
#define ENET_EIR_PLR_WIDTH                       1u
#define ENET_EIR_PLR(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_EIR_PLR_SHIFT))&ENET_EIR_PLR_MASK)
#define ENET_EIR_UN_MASK                         0x80000u
#define ENET_EIR_UN_SHIFT                        19u
#define ENET_EIR_UN_WIDTH                        1u
#define ENET_EIR_UN(x)                           (((uint32_t)(((uint32_t)(x))<<ENET_EIR_UN_SHIFT))&ENET_EIR_UN_MASK)
#define ENET_EIR_RL_MASK                         0x100000u
#define ENET_EIR_RL_SHIFT                        20u
#define ENET_EIR_RL_WIDTH                        1u
#define ENET_EIR_RL(x)                           (((uint32_t)(((uint32_t)(x))<<ENET_EIR_RL_SHIFT))&ENET_EIR_RL_MASK)
#define ENET_EIR_LC_MASK                         0x200000u
#define ENET_EIR_LC_SHIFT                        21u
#define ENET_EIR_LC_WIDTH                        1u
#define ENET_EIR_LC(x)                           (((uint32_t)(((uint32_t)(x))<<ENET_EIR_LC_SHIFT))&ENET_EIR_LC_MASK)
#define ENET_EIR_EBERR_MASK                      0x400000u
#define ENET_EIR_EBERR_SHIFT                     22u
#define ENET_EIR_EBERR_WIDTH                     1u
#define ENET_EIR_EBERR(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIR_EBERR_SHIFT))&ENET_EIR_EBERR_MASK)
#define ENET_EIR_MII_MASK                        0x800000u
#define ENET_EIR_MII_SHIFT                       23u
#define ENET_EIR_MII_WIDTH                       1u
#define ENET_EIR_MII(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_EIR_MII_SHIFT))&ENET_EIR_MII_MASK)
#define ENET_EIR_RXB_MASK                        0x1000000u
#define ENET_EIR_RXB_SHIFT                       24u
#define ENET_EIR_RXB_WIDTH                       1u
#define ENET_EIR_RXB(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_EIR_RXB_SHIFT))&ENET_EIR_RXB_MASK)
#define ENET_EIR_RXF_MASK                        0x2000000u
#define ENET_EIR_RXF_SHIFT                       25u
#define ENET_EIR_RXF_WIDTH                       1u
#define ENET_EIR_RXF(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_EIR_RXF_SHIFT))&ENET_EIR_RXF_MASK)
#define ENET_EIR_TXB_MASK                        0x4000000u
#define ENET_EIR_TXB_SHIFT                       26u
#define ENET_EIR_TXB_WIDTH                       1u
#define ENET_EIR_TXB(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_EIR_TXB_SHIFT))&ENET_EIR_TXB_MASK)
#define ENET_EIR_TXF_MASK                        0x8000000u
#define ENET_EIR_TXF_SHIFT                       27u
#define ENET_EIR_TXF_WIDTH                       1u
#define ENET_EIR_TXF(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_EIR_TXF_SHIFT))&ENET_EIR_TXF_MASK)
#define ENET_EIR_GRA_MASK                        0x10000000u
#define ENET_EIR_GRA_SHIFT                       28u
#define ENET_EIR_GRA_WIDTH                       1u
#define ENET_EIR_GRA(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_EIR_GRA_SHIFT))&ENET_EIR_GRA_MASK)
#define ENET_EIR_BABT_MASK                       0x20000000u
#define ENET_EIR_BABT_SHIFT                      29u
#define ENET_EIR_BABT_WIDTH                      1u
#define ENET_EIR_BABT(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIR_BABT_SHIFT))&ENET_EIR_BABT_MASK)
#define ENET_EIR_BABR_MASK                       0x40000000u
#define ENET_EIR_BABR_SHIFT                      30u
#define ENET_EIR_BABR_WIDTH                      1u
#define ENET_EIR_BABR(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIR_BABR_SHIFT))&ENET_EIR_BABR_MASK)
/* EIMR Bit Fields */
#define ENET_EIMR_RXB1_MASK                      0x1u
#define ENET_EIMR_RXB1_SHIFT                     0u
#define ENET_EIMR_RXB1_WIDTH                     1u
#define ENET_EIMR_RXB1(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_RXB1_SHIFT))&ENET_EIMR_RXB1_MASK)
#define ENET_EIMR_RXF1_MASK                      0x2u
#define ENET_EIMR_RXF1_SHIFT                     1u
#define ENET_EIMR_RXF1_WIDTH                     1u
#define ENET_EIMR_RXF1(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_RXF1_SHIFT))&ENET_EIMR_RXF1_MASK)
#define ENET_EIMR_TXB1_MASK                      0x4u
#define ENET_EIMR_TXB1_SHIFT                     2u
#define ENET_EIMR_TXB1_WIDTH                     1u
#define ENET_EIMR_TXB1(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_TXB1_SHIFT))&ENET_EIMR_TXB1_MASK)
#define ENET_EIMR_TXF1_MASK                      0x8u
#define ENET_EIMR_TXF1_SHIFT                     3u
#define ENET_EIMR_TXF1_WIDTH                     1u
#define ENET_EIMR_TXF1(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_TXF1_SHIFT))&ENET_EIMR_TXF1_MASK)
#define ENET_EIMR_RXB2_MASK                      0x10u
#define ENET_EIMR_RXB2_SHIFT                     4u
#define ENET_EIMR_RXB2_WIDTH                     1u
#define ENET_EIMR_RXB2(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_RXB2_SHIFT))&ENET_EIMR_RXB2_MASK)
#define ENET_EIMR_RXF2_MASK                      0x20u
#define ENET_EIMR_RXF2_SHIFT                     5u
#define ENET_EIMR_RXF2_WIDTH                     1u
#define ENET_EIMR_RXF2(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_RXF2_SHIFT))&ENET_EIMR_RXF2_MASK)
#define ENET_EIMR_TXB2_MASK                      0x40u
#define ENET_EIMR_TXB2_SHIFT                     6u
#define ENET_EIMR_TXB2_WIDTH                     1u
#define ENET_EIMR_TXB2(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_TXB2_SHIFT))&ENET_EIMR_TXB2_MASK)
#define ENET_EIMR_TXF2_MASK                      0x80u
#define ENET_EIMR_TXF2_SHIFT                     7u
#define ENET_EIMR_TXF2_WIDTH                     1u
#define ENET_EIMR_TXF2(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_TXF2_SHIFT))&ENET_EIMR_TXF2_MASK)
#define ENET_EIMR_PARSRF_MASK                    0x200u
#define ENET_EIMR_PARSRF_SHIFT                   9u
#define ENET_EIMR_PARSRF_WIDTH                   1u
#define ENET_EIMR_PARSRF(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_PARSRF_SHIFT))&ENET_EIMR_PARSRF_MASK)
#define ENET_EIMR_PARSERR_MASK                   0x400u
#define ENET_EIMR_PARSERR_SHIFT                  10u
#define ENET_EIMR_PARSERR_WIDTH                  1u
#define ENET_EIMR_PARSERR(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_PARSERR_SHIFT))&ENET_EIMR_PARSERR_MASK)
#define ENET_EIMR_RXFLUSH_0_MASK                 0x1000u
#define ENET_EIMR_RXFLUSH_0_SHIFT                12u
#define ENET_EIMR_RXFLUSH_0_WIDTH                1u
#define ENET_EIMR_RXFLUSH_0(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_RXFLUSH_0_SHIFT))&ENET_EIMR_RXFLUSH_0_MASK)
#define ENET_EIMR_RXFLUSH_1_MASK                 0x2000u
#define ENET_EIMR_RXFLUSH_1_SHIFT                13u
#define ENET_EIMR_RXFLUSH_1_WIDTH                1u
#define ENET_EIMR_RXFLUSH_1(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_RXFLUSH_1_SHIFT))&ENET_EIMR_RXFLUSH_1_MASK)
#define ENET_EIMR_RXFLUSH_2_MASK                 0x4000u
#define ENET_EIMR_RXFLUSH_2_SHIFT                14u
#define ENET_EIMR_RXFLUSH_2_WIDTH                1u
#define ENET_EIMR_RXFLUSH_2(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_RXFLUSH_2_SHIFT))&ENET_EIMR_RXFLUSH_2_MASK)
#define ENET_EIMR_TS_TIMER_MASK                  0x8000u
#define ENET_EIMR_TS_TIMER_SHIFT                 15u
#define ENET_EIMR_TS_TIMER_WIDTH                 1u
#define ENET_EIMR_TS_TIMER(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_TS_TIMER_SHIFT))&ENET_EIMR_TS_TIMER_MASK)
#define ENET_EIMR_TS_AVAIL_MASK                  0x10000u
#define ENET_EIMR_TS_AVAIL_SHIFT                 16u
#define ENET_EIMR_TS_AVAIL_WIDTH                 1u
#define ENET_EIMR_TS_AVAIL(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_TS_AVAIL_SHIFT))&ENET_EIMR_TS_AVAIL_MASK)
#define ENET_EIMR_WAKEUP_MASK                    0x20000u
#define ENET_EIMR_WAKEUP_SHIFT                   17u
#define ENET_EIMR_WAKEUP_WIDTH                   1u
#define ENET_EIMR_WAKEUP(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_WAKEUP_SHIFT))&ENET_EIMR_WAKEUP_MASK)
#define ENET_EIMR_PLR_MASK                       0x40000u
#define ENET_EIMR_PLR_SHIFT                      18u
#define ENET_EIMR_PLR_WIDTH                      1u
#define ENET_EIMR_PLR(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_PLR_SHIFT))&ENET_EIMR_PLR_MASK)
#define ENET_EIMR_UN_MASK                        0x80000u
#define ENET_EIMR_UN_SHIFT                       19u
#define ENET_EIMR_UN_WIDTH                       1u
#define ENET_EIMR_UN(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_UN_SHIFT))&ENET_EIMR_UN_MASK)
#define ENET_EIMR_RL_MASK                        0x100000u
#define ENET_EIMR_RL_SHIFT                       20u
#define ENET_EIMR_RL_WIDTH                       1u
#define ENET_EIMR_RL(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_RL_SHIFT))&ENET_EIMR_RL_MASK)
#define ENET_EIMR_LC_MASK                        0x200000u
#define ENET_EIMR_LC_SHIFT                       21u
#define ENET_EIMR_LC_WIDTH                       1u
#define ENET_EIMR_LC(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_LC_SHIFT))&ENET_EIMR_LC_MASK)
#define ENET_EIMR_EBERR_MASK                     0x400000u
#define ENET_EIMR_EBERR_SHIFT                    22u
#define ENET_EIMR_EBERR_WIDTH                    1u
#define ENET_EIMR_EBERR(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_EBERR_SHIFT))&ENET_EIMR_EBERR_MASK)
#define ENET_EIMR_MII_MASK                       0x800000u
#define ENET_EIMR_MII_SHIFT                      23u
#define ENET_EIMR_MII_WIDTH                      1u
#define ENET_EIMR_MII(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_MII_SHIFT))&ENET_EIMR_MII_MASK)
#define ENET_EIMR_RXB_MASK                       0x1000000u
#define ENET_EIMR_RXB_SHIFT                      24u
#define ENET_EIMR_RXB_WIDTH                      1u
#define ENET_EIMR_RXB(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_RXB_SHIFT))&ENET_EIMR_RXB_MASK)
#define ENET_EIMR_RXF_MASK                       0x2000000u
#define ENET_EIMR_RXF_SHIFT                      25u
#define ENET_EIMR_RXF_WIDTH                      1u
#define ENET_EIMR_RXF(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_RXF_SHIFT))&ENET_EIMR_RXF_MASK)
#define ENET_EIMR_TXB_MASK                       0x4000000u
#define ENET_EIMR_TXB_SHIFT                      26u
#define ENET_EIMR_TXB_WIDTH                      1u
#define ENET_EIMR_TXB(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_TXB_SHIFT))&ENET_EIMR_TXB_MASK)
#define ENET_EIMR_TXF_MASK                       0x8000000u
#define ENET_EIMR_TXF_SHIFT                      27u
#define ENET_EIMR_TXF_WIDTH                      1u
#define ENET_EIMR_TXF(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_TXF_SHIFT))&ENET_EIMR_TXF_MASK)
#define ENET_EIMR_GRA_MASK                       0x10000000u
#define ENET_EIMR_GRA_SHIFT                      28u
#define ENET_EIMR_GRA_WIDTH                      1u
#define ENET_EIMR_GRA(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_GRA_SHIFT))&ENET_EIMR_GRA_MASK)
#define ENET_EIMR_BABT_MASK                      0x20000000u
#define ENET_EIMR_BABT_SHIFT                     29u
#define ENET_EIMR_BABT_WIDTH                     1u
#define ENET_EIMR_BABT(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_BABT_SHIFT))&ENET_EIMR_BABT_MASK)
#define ENET_EIMR_BABR_MASK                      0x40000000u
#define ENET_EIMR_BABR_SHIFT                     30u
#define ENET_EIMR_BABR_WIDTH                     1u
#define ENET_EIMR_BABR(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_BABR_SHIFT))&ENET_EIMR_BABR_MASK)
/* RDAR Bit Fields */
#define ENET_RDAR_RDAR_MASK                      0x1000000u
#define ENET_RDAR_RDAR_SHIFT                     24u
#define ENET_RDAR_RDAR_WIDTH                     1u
#define ENET_RDAR_RDAR(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_RDAR_RDAR_SHIFT))&ENET_RDAR_RDAR_MASK)
/* TDAR Bit Fields */
#define ENET_TDAR_TDAR_MASK                      0x1000000u
#define ENET_TDAR_TDAR_SHIFT                     24u
#define ENET_TDAR_TDAR_WIDTH                     1u
#define ENET_TDAR_TDAR(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_TDAR_TDAR_SHIFT))&ENET_TDAR_TDAR_MASK)
/* ECR Bit Fields */
#define ENET_ECR_RESET_MASK                      0x1u
#define ENET_ECR_RESET_SHIFT                     0u
#define ENET_ECR_RESET_WIDTH                     1u
#define ENET_ECR_RESET(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_ECR_RESET_SHIFT))&ENET_ECR_RESET_MASK)
#define ENET_ECR_ETHEREN_MASK                    0x2u
#define ENET_ECR_ETHEREN_SHIFT                   1u
#define ENET_ECR_ETHEREN_WIDTH                   1u
#define ENET_ECR_ETHEREN(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_ECR_ETHEREN_SHIFT))&ENET_ECR_ETHEREN_MASK)
#define ENET_ECR_MAGICEN_MASK                    0x4u
#define ENET_ECR_MAGICEN_SHIFT                   2u
#define ENET_ECR_MAGICEN_WIDTH                   1u
#define ENET_ECR_MAGICEN(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_ECR_MAGICEN_SHIFT))&ENET_ECR_MAGICEN_MASK)
#define ENET_ECR_SLEEP_MASK                      0x8u
#define ENET_ECR_SLEEP_SHIFT                     3u
#define ENET_ECR_SLEEP_WIDTH                     1u
#define ENET_ECR_SLEEP(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_ECR_SLEEP_SHIFT))&ENET_ECR_SLEEP_MASK)
#define ENET_ECR_EN1588_MASK                     0x10u
#define ENET_ECR_EN1588_SHIFT                    4u
#define ENET_ECR_EN1588_WIDTH                    1u
#define ENET_ECR_EN1588(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_ECR_EN1588_SHIFT))&ENET_ECR_EN1588_MASK)
#define ENET_ECR_SPEED_MASK                      0x20u
#define ENET_ECR_SPEED_SHIFT                     5u
#define ENET_ECR_SPEED_WIDTH                     1u
#define ENET_ECR_SPEED(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_ECR_SPEED_SHIFT))&ENET_ECR_SPEED_MASK)
#define ENET_ECR_DBGEN_MASK                      0x40u
#define ENET_ECR_DBGEN_SHIFT                     6u
#define ENET_ECR_DBGEN_WIDTH                     1u
#define ENET_ECR_DBGEN(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_ECR_DBGEN_SHIFT))&ENET_ECR_DBGEN_MASK)
#define ENET_ECR_DBSWP_MASK                      0x100u
#define ENET_ECR_DBSWP_SHIFT                     8u
#define ENET_ECR_DBSWP_WIDTH                     1u
#define ENET_ECR_DBSWP(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_ECR_DBSWP_SHIFT))&ENET_ECR_DBSWP_MASK)
#define ENET_ECR_SVLANEN_MASK                    0x200u
#define ENET_ECR_SVLANEN_SHIFT                   9u
#define ENET_ECR_SVLANEN_WIDTH                   1u
#define ENET_ECR_SVLANEN(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_ECR_SVLANEN_SHIFT))&ENET_ECR_SVLANEN_MASK)
#define ENET_ECR_VLANUSE2ND_MASK                 0x400u
#define ENET_ECR_VLANUSE2ND_SHIFT                10u
#define ENET_ECR_VLANUSE2ND_WIDTH                1u
#define ENET_ECR_VLANUSE2ND(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_ECR_VLANUSE2ND_SHIFT))&ENET_ECR_VLANUSE2ND_MASK)
#define ENET_ECR_SVLANDBL_MASK                   0x800u
#define ENET_ECR_SVLANDBL_SHIFT                  11u
#define ENET_ECR_SVLANDBL_WIDTH                  1u
#define ENET_ECR_SVLANDBL(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_ECR_SVLANDBL_SHIFT))&ENET_ECR_SVLANDBL_MASK)
/* MMFR Bit Fields */
#define ENET_MMFR_DATA_MASK                      0xFFFFu
#define ENET_MMFR_DATA_SHIFT                     0u
#define ENET_MMFR_DATA_WIDTH                     16u
#define ENET_MMFR_DATA(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_MMFR_DATA_SHIFT))&ENET_MMFR_DATA_MASK)
#define ENET_MMFR_TA_MASK                        0x30000u
#define ENET_MMFR_TA_SHIFT                       16u
#define ENET_MMFR_TA_WIDTH                       2u
#define ENET_MMFR_TA(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_MMFR_TA_SHIFT))&ENET_MMFR_TA_MASK)
#define ENET_MMFR_RA_MASK                        0x7C0000u
#define ENET_MMFR_RA_SHIFT                       18u
#define ENET_MMFR_RA_WIDTH                       5u
#define ENET_MMFR_RA(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_MMFR_RA_SHIFT))&ENET_MMFR_RA_MASK)
#define ENET_MMFR_PA_MASK                        0xF800000u
#define ENET_MMFR_PA_SHIFT                       23u
#define ENET_MMFR_PA_WIDTH                       5u
#define ENET_MMFR_PA(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_MMFR_PA_SHIFT))&ENET_MMFR_PA_MASK)
#define ENET_MMFR_OP_MASK                        0x30000000u
#define ENET_MMFR_OP_SHIFT                       28u
#define ENET_MMFR_OP_WIDTH                       2u
#define ENET_MMFR_OP(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_MMFR_OP_SHIFT))&ENET_MMFR_OP_MASK)
#define ENET_MMFR_ST_MASK                        0xC0000000u
#define ENET_MMFR_ST_SHIFT                       30u
#define ENET_MMFR_ST_WIDTH                       2u
#define ENET_MMFR_ST(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_MMFR_ST_SHIFT))&ENET_MMFR_ST_MASK)
/* MSCR Bit Fields */
#define ENET_MSCR_MII_SPEED_MASK                 0x7Eu
#define ENET_MSCR_MII_SPEED_SHIFT                1u
#define ENET_MSCR_MII_SPEED_WIDTH                6u
#define ENET_MSCR_MII_SPEED(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_MSCR_MII_SPEED_SHIFT))&ENET_MSCR_MII_SPEED_MASK)
#define ENET_MSCR_DIS_PRE_MASK                   0x80u
#define ENET_MSCR_DIS_PRE_SHIFT                  7u
#define ENET_MSCR_DIS_PRE_WIDTH                  1u
#define ENET_MSCR_DIS_PRE(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_MSCR_DIS_PRE_SHIFT))&ENET_MSCR_DIS_PRE_MASK)
#define ENET_MSCR_HOLDTIME_MASK                  0x700u
#define ENET_MSCR_HOLDTIME_SHIFT                 8u
#define ENET_MSCR_HOLDTIME_WIDTH                 3u
#define ENET_MSCR_HOLDTIME(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_MSCR_HOLDTIME_SHIFT))&ENET_MSCR_HOLDTIME_MASK)
/* MIBC Bit Fields */
#define ENET_MIBC_MIB_CLEAR_MASK                 0x20000000u
#define ENET_MIBC_MIB_CLEAR_SHIFT                29u
#define ENET_MIBC_MIB_CLEAR_WIDTH                1u
#define ENET_MIBC_MIB_CLEAR(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_MIBC_MIB_CLEAR_SHIFT))&ENET_MIBC_MIB_CLEAR_MASK)
#define ENET_MIBC_MIB_IDLE_MASK                  0x40000000u
#define ENET_MIBC_MIB_IDLE_SHIFT                 30u
#define ENET_MIBC_MIB_IDLE_WIDTH                 1u
#define ENET_MIBC_MIB_IDLE(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_MIBC_MIB_IDLE_SHIFT))&ENET_MIBC_MIB_IDLE_MASK)
#define ENET_MIBC_MIB_DIS_MASK                   0x80000000u
#define ENET_MIBC_MIB_DIS_SHIFT                  31u
#define ENET_MIBC_MIB_DIS_WIDTH                  1u
#define ENET_MIBC_MIB_DIS(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_MIBC_MIB_DIS_SHIFT))&ENET_MIBC_MIB_DIS_MASK)
/* RCR Bit Fields */
#define ENET_RCR_LOOP_MASK                       0x1u
#define ENET_RCR_LOOP_SHIFT                      0u
#define ENET_RCR_LOOP_WIDTH                      1u
#define ENET_RCR_LOOP(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_RCR_LOOP_SHIFT))&ENET_RCR_LOOP_MASK)
#define ENET_RCR_DRT_MASK                        0x2u
#define ENET_RCR_DRT_SHIFT                       1u
#define ENET_RCR_DRT_WIDTH                       1u
#define ENET_RCR_DRT(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_RCR_DRT_SHIFT))&ENET_RCR_DRT_MASK)
#define ENET_RCR_MII_MODE_MASK                   0x4u
#define ENET_RCR_MII_MODE_SHIFT                  2u
#define ENET_RCR_MII_MODE_WIDTH                  1u
#define ENET_RCR_MII_MODE(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_RCR_MII_MODE_SHIFT))&ENET_RCR_MII_MODE_MASK)
#define ENET_RCR_PROM_MASK                       0x8u
#define ENET_RCR_PROM_SHIFT                      3u
#define ENET_RCR_PROM_WIDTH                      1u
#define ENET_RCR_PROM(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_RCR_PROM_SHIFT))&ENET_RCR_PROM_MASK)
#define ENET_RCR_BC_REJ_MASK                     0x10u
#define ENET_RCR_BC_REJ_SHIFT                    4u
#define ENET_RCR_BC_REJ_WIDTH                    1u
#define ENET_RCR_BC_REJ(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_RCR_BC_REJ_SHIFT))&ENET_RCR_BC_REJ_MASK)
#define ENET_RCR_FCE_MASK                        0x20u
#define ENET_RCR_FCE_SHIFT                       5u
#define ENET_RCR_FCE_WIDTH                       1u
#define ENET_RCR_FCE(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_RCR_FCE_SHIFT))&ENET_RCR_FCE_MASK)
#define ENET_RCR_RGMII_EN_MASK                   0x40u
#define ENET_RCR_RGMII_EN_SHIFT                  6u
#define ENET_RCR_RGMII_EN_WIDTH                  1u
#define ENET_RCR_RGMII_EN(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_RCR_RGMII_EN_SHIFT))&ENET_RCR_RGMII_EN_MASK)
#define ENET_RCR_RMII_MODE_MASK                  0x100u
#define ENET_RCR_RMII_MODE_SHIFT                 8u
#define ENET_RCR_RMII_MODE_WIDTH                 1u
#define ENET_RCR_RMII_MODE(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_RCR_RMII_MODE_SHIFT))&ENET_RCR_RMII_MODE_MASK)
#define ENET_RCR_RMII_10T_MASK                   0x200u
#define ENET_RCR_RMII_10T_SHIFT                  9u
#define ENET_RCR_RMII_10T_WIDTH                  1u
#define ENET_RCR_RMII_10T(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_RCR_RMII_10T_SHIFT))&ENET_RCR_RMII_10T_MASK)
#define ENET_RCR_PADEN_MASK                      0x1000u
#define ENET_RCR_PADEN_SHIFT                     12u
#define ENET_RCR_PADEN_WIDTH                     1u
#define ENET_RCR_PADEN(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_RCR_PADEN_SHIFT))&ENET_RCR_PADEN_MASK)
#define ENET_RCR_PAUFWD_MASK                     0x2000u
#define ENET_RCR_PAUFWD_SHIFT                    13u
#define ENET_RCR_PAUFWD_WIDTH                    1u
#define ENET_RCR_PAUFWD(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_RCR_PAUFWD_SHIFT))&ENET_RCR_PAUFWD_MASK)
#define ENET_RCR_CRCFWD_MASK                     0x4000u
#define ENET_RCR_CRCFWD_SHIFT                    14u
#define ENET_RCR_CRCFWD_WIDTH                    1u
#define ENET_RCR_CRCFWD(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_RCR_CRCFWD_SHIFT))&ENET_RCR_CRCFWD_MASK)
#define ENET_RCR_CFEN_MASK                       0x8000u
#define ENET_RCR_CFEN_SHIFT                      15u
#define ENET_RCR_CFEN_WIDTH                      1u
#define ENET_RCR_CFEN(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_RCR_CFEN_SHIFT))&ENET_RCR_CFEN_MASK)
#define ENET_RCR_MAX_FL_MASK                     0x3FFF0000u
#define ENET_RCR_MAX_FL_SHIFT                    16u
#define ENET_RCR_MAX_FL_WIDTH                    14u
#define ENET_RCR_MAX_FL(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_RCR_MAX_FL_SHIFT))&ENET_RCR_MAX_FL_MASK)
#define ENET_RCR_NLC_MASK                        0x40000000u
#define ENET_RCR_NLC_SHIFT                       30u
#define ENET_RCR_NLC_WIDTH                       1u
#define ENET_RCR_NLC(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_RCR_NLC_SHIFT))&ENET_RCR_NLC_MASK)
#define ENET_RCR_GRS_MASK                        0x80000000u
#define ENET_RCR_GRS_SHIFT                       31u
#define ENET_RCR_GRS_WIDTH                       1u
#define ENET_RCR_GRS(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_RCR_GRS_SHIFT))&ENET_RCR_GRS_MASK)
/* TCR Bit Fields */
#define ENET_TCR_GTS_MASK                        0x1u
#define ENET_TCR_GTS_SHIFT                       0u
#define ENET_TCR_GTS_WIDTH                       1u
#define ENET_TCR_GTS(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_TCR_GTS_SHIFT))&ENET_TCR_GTS_MASK)
#define ENET_TCR_FDEN_MASK                       0x4u
#define ENET_TCR_FDEN_SHIFT                      2u
#define ENET_TCR_FDEN_WIDTH                      1u
#define ENET_TCR_FDEN(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_TCR_FDEN_SHIFT))&ENET_TCR_FDEN_MASK)
#define ENET_TCR_TFC_PAUSE_MASK                  0x8u
#define ENET_TCR_TFC_PAUSE_SHIFT                 3u
#define ENET_TCR_TFC_PAUSE_WIDTH                 1u
#define ENET_TCR_TFC_PAUSE(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_TCR_TFC_PAUSE_SHIFT))&ENET_TCR_TFC_PAUSE_MASK)
#define ENET_TCR_RFC_PAUSE_MASK                  0x10u
#define ENET_TCR_RFC_PAUSE_SHIFT                 4u
#define ENET_TCR_RFC_PAUSE_WIDTH                 1u
#define ENET_TCR_RFC_PAUSE(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_TCR_RFC_PAUSE_SHIFT))&ENET_TCR_RFC_PAUSE_MASK)
#define ENET_TCR_ADDSEL_MASK                     0xE0u
#define ENET_TCR_ADDSEL_SHIFT                    5u
#define ENET_TCR_ADDSEL_WIDTH                    3u
#define ENET_TCR_ADDSEL(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_TCR_ADDSEL_SHIFT))&ENET_TCR_ADDSEL_MASK)
#define ENET_TCR_ADDINS_MASK                     0x100u
#define ENET_TCR_ADDINS_SHIFT                    8u
#define ENET_TCR_ADDINS_WIDTH                    1u
#define ENET_TCR_ADDINS(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_TCR_ADDINS_SHIFT))&ENET_TCR_ADDINS_MASK)
#define ENET_TCR_CRCFWD_MASK                     0x200u
#define ENET_TCR_CRCFWD_SHIFT                    9u
#define ENET_TCR_CRCFWD_WIDTH                    1u
#define ENET_TCR_CRCFWD(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_TCR_CRCFWD_SHIFT))&ENET_TCR_CRCFWD_MASK)
/* PALR Bit Fields */
#define ENET_PALR_PADDR1_MASK                    0xFFFFFFFFu
#define ENET_PALR_PADDR1_SHIFT                   0u
#define ENET_PALR_PADDR1_WIDTH                   32u
#define ENET_PALR_PADDR1(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_PALR_PADDR1_SHIFT))&ENET_PALR_PADDR1_MASK)
/* PAUR Bit Fields */
#define ENET_PAUR_TYPE_MASK                      0xFFFFu
#define ENET_PAUR_TYPE_SHIFT                     0u
#define ENET_PAUR_TYPE_WIDTH                     16u
#define ENET_PAUR_TYPE(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_PAUR_TYPE_SHIFT))&ENET_PAUR_TYPE_MASK)
#define ENET_PAUR_PADDR2_MASK                    0xFFFF0000u
#define ENET_PAUR_PADDR2_SHIFT                   16u
#define ENET_PAUR_PADDR2_WIDTH                   16u
#define ENET_PAUR_PADDR2(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_PAUR_PADDR2_SHIFT))&ENET_PAUR_PADDR2_MASK)
/* OPD Bit Fields */
#define ENET_OPD_PAUSE_DUR_MASK                  0xFFFFu
#define ENET_OPD_PAUSE_DUR_SHIFT                 0u
#define ENET_OPD_PAUSE_DUR_WIDTH                 16u
#define ENET_OPD_PAUSE_DUR(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_OPD_PAUSE_DUR_SHIFT))&ENET_OPD_PAUSE_DUR_MASK)
#define ENET_OPD_OPCODE_MASK                     0xFFFF0000u
#define ENET_OPD_OPCODE_SHIFT                    16u
#define ENET_OPD_OPCODE_WIDTH                    16u
#define ENET_OPD_OPCODE(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_OPD_OPCODE_SHIFT))&ENET_OPD_OPCODE_MASK)
/* TXIC Bit Fields */
#define ENET_TXIC_ICTT_MASK                      0xFFFFu
#define ENET_TXIC_ICTT_SHIFT                     0u
#define ENET_TXIC_ICTT_WIDTH                     16u
#define ENET_TXIC_ICTT(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_TXIC_ICTT_SHIFT))&ENET_TXIC_ICTT_MASK)
#define ENET_TXIC_ICFT_MASK                      0xFF00000u
#define ENET_TXIC_ICFT_SHIFT                     20u
#define ENET_TXIC_ICFT_WIDTH                     8u
#define ENET_TXIC_ICFT(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_TXIC_ICFT_SHIFT))&ENET_TXIC_ICFT_MASK)
#define ENET_TXIC_ICCS_MASK                      0x40000000u
#define ENET_TXIC_ICCS_SHIFT                     30u
#define ENET_TXIC_ICCS_WIDTH                     1u
#define ENET_TXIC_ICCS(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_TXIC_ICCS_SHIFT))&ENET_TXIC_ICCS_MASK)
#define ENET_TXIC_ICEN_MASK                      0x80000000u
#define ENET_TXIC_ICEN_SHIFT                     31u
#define ENET_TXIC_ICEN_WIDTH                     1u
#define ENET_TXIC_ICEN(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_TXIC_ICEN_SHIFT))&ENET_TXIC_ICEN_MASK)
/* RXIC Bit Fields */
#define ENET_RXIC_ICTT_MASK                      0xFFFFu
#define ENET_RXIC_ICTT_SHIFT                     0u
#define ENET_RXIC_ICTT_WIDTH                     16u
#define ENET_RXIC_ICTT(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_RXIC_ICTT_SHIFT))&ENET_RXIC_ICTT_MASK)
#define ENET_RXIC_ICFT_MASK                      0xFF00000u
#define ENET_RXIC_ICFT_SHIFT                     20u
#define ENET_RXIC_ICFT_WIDTH                     8u
#define ENET_RXIC_ICFT(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_RXIC_ICFT_SHIFT))&ENET_RXIC_ICFT_MASK)
#define ENET_RXIC_ICCS_MASK                      0x40000000u
#define ENET_RXIC_ICCS_SHIFT                     30u
#define ENET_RXIC_ICCS_WIDTH                     1u
#define ENET_RXIC_ICCS(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_RXIC_ICCS_SHIFT))&ENET_RXIC_ICCS_MASK)
#define ENET_RXIC_ICEN_MASK                      0x80000000u
#define ENET_RXIC_ICEN_SHIFT                     31u
#define ENET_RXIC_ICEN_WIDTH                     1u
#define ENET_RXIC_ICEN(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_RXIC_ICEN_SHIFT))&ENET_RXIC_ICEN_MASK)
/* IAUR Bit Fields */
#define ENET_IAUR_IADDR1_MASK                    0xFFFFFFFFu
#define ENET_IAUR_IADDR1_SHIFT                   0u
#define ENET_IAUR_IADDR1_WIDTH                   32u
#define ENET_IAUR_IADDR1(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_IAUR_IADDR1_SHIFT))&ENET_IAUR_IADDR1_MASK)
/* IALR Bit Fields */
#define ENET_IALR_IADDR2_MASK                    0xFFFFFFFFu
#define ENET_IALR_IADDR2_SHIFT                   0u
#define ENET_IALR_IADDR2_WIDTH                   32u
#define ENET_IALR_IADDR2(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_IALR_IADDR2_SHIFT))&ENET_IALR_IADDR2_MASK)
/* GAUR Bit Fields */
#define ENET_GAUR_GADDR1_MASK                    0xFFFFFFFFu
#define ENET_GAUR_GADDR1_SHIFT                   0u
#define ENET_GAUR_GADDR1_WIDTH                   32u
#define ENET_GAUR_GADDR1(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_GAUR_GADDR1_SHIFT))&ENET_GAUR_GADDR1_MASK)
/* GALR Bit Fields */
#define ENET_GALR_GADDR2_MASK                    0xFFFFFFFFu
#define ENET_GALR_GADDR2_SHIFT                   0u
#define ENET_GALR_GADDR2_WIDTH                   32u
#define ENET_GALR_GADDR2(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_GALR_GADDR2_SHIFT))&ENET_GALR_GADDR2_MASK)
/* TFWR Bit Fields */
#define ENET_TFWR_TFWR_MASK                      0x3Fu
#define ENET_TFWR_TFWR_SHIFT                     0u
#define ENET_TFWR_TFWR_WIDTH                     6u
#define ENET_TFWR_TFWR(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_TFWR_TFWR_SHIFT))&ENET_TFWR_TFWR_MASK)
#define ENET_TFWR_STRFWD_MASK                    0x100u
#define ENET_TFWR_STRFWD_SHIFT                   8u
#define ENET_TFWR_STRFWD_WIDTH                   1u
#define ENET_TFWR_STRFWD(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_TFWR_STRFWD_SHIFT))&ENET_TFWR_STRFWD_MASK)
/* RDSR1 Bit Fields */
#define ENET_RDSR1_R_DES_START_MASK              0xFFFFFFF8u
#define ENET_RDSR1_R_DES_START_SHIFT             3u
#define ENET_RDSR1_R_DES_START_WIDTH             29u
#define ENET_RDSR1_R_DES_START(x)                (((uint32_t)(((uint32_t)(x))<<ENET_RDSR1_R_DES_START_SHIFT))&ENET_RDSR1_R_DES_START_MASK)
/* TDSR1 Bit Fields */
#define ENET_TDSR1_X_DES_START_MASK              0xFFFFFFF8u
#define ENET_TDSR1_X_DES_START_SHIFT             3u
#define ENET_TDSR1_X_DES_START_WIDTH             29u
#define ENET_TDSR1_X_DES_START(x)                (((uint32_t)(((uint32_t)(x))<<ENET_TDSR1_X_DES_START_SHIFT))&ENET_TDSR1_X_DES_START_MASK)
/* MRBR1 Bit Fields */
#define ENET_MRBR1_R_BUF_SIZE_MASK               0x3FF0u
#define ENET_MRBR1_R_BUF_SIZE_SHIFT              4u
#define ENET_MRBR1_R_BUF_SIZE_WIDTH              10u
#define ENET_MRBR1_R_BUF_SIZE(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_MRBR1_R_BUF_SIZE_SHIFT))&ENET_MRBR1_R_BUF_SIZE_MASK)
/* RDSR2 Bit Fields */
#define ENET_RDSR2_R_DES_START_MASK              0xFFFFFFF8u
#define ENET_RDSR2_R_DES_START_SHIFT             3u
#define ENET_RDSR2_R_DES_START_WIDTH             29u
#define ENET_RDSR2_R_DES_START(x)                (((uint32_t)(((uint32_t)(x))<<ENET_RDSR2_R_DES_START_SHIFT))&ENET_RDSR2_R_DES_START_MASK)
/* TDSR2 Bit Fields */
#define ENET_TDSR2_X_DES_START_MASK              0xFFFFFFF8u
#define ENET_TDSR2_X_DES_START_SHIFT             3u
#define ENET_TDSR2_X_DES_START_WIDTH             29u
#define ENET_TDSR2_X_DES_START(x)                (((uint32_t)(((uint32_t)(x))<<ENET_TDSR2_X_DES_START_SHIFT))&ENET_TDSR2_X_DES_START_MASK)
/* MRBR2 Bit Fields */
#define ENET_MRBR2_R_BUF_SIZE_MASK               0x3FF0u
#define ENET_MRBR2_R_BUF_SIZE_SHIFT              4u
#define ENET_MRBR2_R_BUF_SIZE_WIDTH              10u
#define ENET_MRBR2_R_BUF_SIZE(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_MRBR2_R_BUF_SIZE_SHIFT))&ENET_MRBR2_R_BUF_SIZE_MASK)
/* RDSR Bit Fields */
#define ENET_RDSR_R_DES_START_MASK               0xFFFFFFF8u
#define ENET_RDSR_R_DES_START_SHIFT              3u
#define ENET_RDSR_R_DES_START_WIDTH              29u
#define ENET_RDSR_R_DES_START(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_RDSR_R_DES_START_SHIFT))&ENET_RDSR_R_DES_START_MASK)
/* TDSR Bit Fields */
#define ENET_TDSR_X_DES_START_MASK               0xFFFFFFF8u
#define ENET_TDSR_X_DES_START_SHIFT              3u
#define ENET_TDSR_X_DES_START_WIDTH              29u
#define ENET_TDSR_X_DES_START(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_TDSR_X_DES_START_SHIFT))&ENET_TDSR_X_DES_START_MASK)
/* MRBR Bit Fields */
#define ENET_MRBR_R_BUF_SIZE_MASK                0x3FF0u
#define ENET_MRBR_R_BUF_SIZE_SHIFT               4u
#define ENET_MRBR_R_BUF_SIZE_WIDTH               10u
#define ENET_MRBR_R_BUF_SIZE(x)                  (((uint32_t)(((uint32_t)(x))<<ENET_MRBR_R_BUF_SIZE_SHIFT))&ENET_MRBR_R_BUF_SIZE_MASK)
/* RSFL Bit Fields */
#define ENET_RSFL_RX_SECTION_FULL_MASK           0x3FFu
#define ENET_RSFL_RX_SECTION_FULL_SHIFT          0u
#define ENET_RSFL_RX_SECTION_FULL_WIDTH          10u
#define ENET_RSFL_RX_SECTION_FULL(x)             (((uint32_t)(((uint32_t)(x))<<ENET_RSFL_RX_SECTION_FULL_SHIFT))&ENET_RSFL_RX_SECTION_FULL_MASK)
/* RSEM Bit Fields */
#define ENET_RSEM_RX_SECTION_EMPTY_MASK          0x3FFu
#define ENET_RSEM_RX_SECTION_EMPTY_SHIFT         0u
#define ENET_RSEM_RX_SECTION_EMPTY_WIDTH         10u
#define ENET_RSEM_RX_SECTION_EMPTY(x)            (((uint32_t)(((uint32_t)(x))<<ENET_RSEM_RX_SECTION_EMPTY_SHIFT))&ENET_RSEM_RX_SECTION_EMPTY_MASK)
#define ENET_RSEM_STAT_SECTION_EMPTY_MASK        0x1F0000u
#define ENET_RSEM_STAT_SECTION_EMPTY_SHIFT       16u
#define ENET_RSEM_STAT_SECTION_EMPTY_WIDTH       5u
#define ENET_RSEM_STAT_SECTION_EMPTY(x)          (((uint32_t)(((uint32_t)(x))<<ENET_RSEM_STAT_SECTION_EMPTY_SHIFT))&ENET_RSEM_STAT_SECTION_EMPTY_MASK)
/* RAEM Bit Fields */
#define ENET_RAEM_RX_ALMOST_EMPTY_MASK           0x3FFu
#define ENET_RAEM_RX_ALMOST_EMPTY_SHIFT          0u
#define ENET_RAEM_RX_ALMOST_EMPTY_WIDTH          10u
#define ENET_RAEM_RX_ALMOST_EMPTY(x)             (((uint32_t)(((uint32_t)(x))<<ENET_RAEM_RX_ALMOST_EMPTY_SHIFT))&ENET_RAEM_RX_ALMOST_EMPTY_MASK)
/* RAFL Bit Fields */
#define ENET_RAFL_RX_ALMOST_FULL_MASK            0x3FFu
#define ENET_RAFL_RX_ALMOST_FULL_SHIFT           0u
#define ENET_RAFL_RX_ALMOST_FULL_WIDTH           10u
#define ENET_RAFL_RX_ALMOST_FULL(x)              (((uint32_t)(((uint32_t)(x))<<ENET_RAFL_RX_ALMOST_FULL_SHIFT))&ENET_RAFL_RX_ALMOST_FULL_MASK)
/* TSEM Bit Fields */
#define ENET_TSEM_TX_SECTION_EMPTY_MASK          0x3FFu
#define ENET_TSEM_TX_SECTION_EMPTY_SHIFT         0u
#define ENET_TSEM_TX_SECTION_EMPTY_WIDTH         10u
#define ENET_TSEM_TX_SECTION_EMPTY(x)            (((uint32_t)(((uint32_t)(x))<<ENET_TSEM_TX_SECTION_EMPTY_SHIFT))&ENET_TSEM_TX_SECTION_EMPTY_MASK)
/* TAEM Bit Fields */
#define ENET_TAEM_TX_ALMOST_EMPTY_MASK           0x3FFu
#define ENET_TAEM_TX_ALMOST_EMPTY_SHIFT          0u
#define ENET_TAEM_TX_ALMOST_EMPTY_WIDTH          10u
#define ENET_TAEM_TX_ALMOST_EMPTY(x)             (((uint32_t)(((uint32_t)(x))<<ENET_TAEM_TX_ALMOST_EMPTY_SHIFT))&ENET_TAEM_TX_ALMOST_EMPTY_MASK)
/* TAFL Bit Fields */
#define ENET_TAFL_TX_ALMOST_FULL_MASK            0x3FFu
#define ENET_TAFL_TX_ALMOST_FULL_SHIFT           0u
#define ENET_TAFL_TX_ALMOST_FULL_WIDTH           10u
#define ENET_TAFL_TX_ALMOST_FULL(x)              (((uint32_t)(((uint32_t)(x))<<ENET_TAFL_TX_ALMOST_FULL_SHIFT))&ENET_TAFL_TX_ALMOST_FULL_MASK)
/* TIPG Bit Fields */
#define ENET_TIPG_IPG_MASK                       0x1Fu
#define ENET_TIPG_IPG_SHIFT                      0u
#define ENET_TIPG_IPG_WIDTH                      5u
#define ENET_TIPG_IPG(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_TIPG_IPG_SHIFT))&ENET_TIPG_IPG_MASK)
/* FTRL Bit Fields */
#define ENET_FTRL_TRUNC_FL_MASK                  0x3FFFu
#define ENET_FTRL_TRUNC_FL_SHIFT                 0u
#define ENET_FTRL_TRUNC_FL_WIDTH                 14u
#define ENET_FTRL_TRUNC_FL(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_FTRL_TRUNC_FL_SHIFT))&ENET_FTRL_TRUNC_FL_MASK)
/* TACC Bit Fields */
#define ENET_TACC_SHIFT16_MASK                   0x1u
#define ENET_TACC_SHIFT16_SHIFT                  0u
#define ENET_TACC_SHIFT16_WIDTH                  1u
#define ENET_TACC_SHIFT16(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_TACC_SHIFT16_SHIFT))&ENET_TACC_SHIFT16_MASK)
#define ENET_TACC_IPCHK_MASK                     0x8u
#define ENET_TACC_IPCHK_SHIFT                    3u
#define ENET_TACC_IPCHK_WIDTH                    1u
#define ENET_TACC_IPCHK(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_TACC_IPCHK_SHIFT))&ENET_TACC_IPCHK_MASK)
#define ENET_TACC_PROCHK_MASK                    0x10u
#define ENET_TACC_PROCHK_SHIFT                   4u
#define ENET_TACC_PROCHK_WIDTH                   1u
#define ENET_TACC_PROCHK(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_TACC_PROCHK_SHIFT))&ENET_TACC_PROCHK_MASK)
/* RACC Bit Fields */
#define ENET_RACC_PADREM_MASK                    0x1u
#define ENET_RACC_PADREM_SHIFT                   0u
#define ENET_RACC_PADREM_WIDTH                   1u
#define ENET_RACC_PADREM(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_RACC_PADREM_SHIFT))&ENET_RACC_PADREM_MASK)
#define ENET_RACC_IPDIS_MASK                     0x2u
#define ENET_RACC_IPDIS_SHIFT                    1u
#define ENET_RACC_IPDIS_WIDTH                    1u
#define ENET_RACC_IPDIS(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_RACC_IPDIS_SHIFT))&ENET_RACC_IPDIS_MASK)
#define ENET_RACC_PRODIS_MASK                    0x4u
#define ENET_RACC_PRODIS_SHIFT                   2u
#define ENET_RACC_PRODIS_WIDTH                   1u
#define ENET_RACC_PRODIS(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_RACC_PRODIS_SHIFT))&ENET_RACC_PRODIS_MASK)
#define ENET_RACC_LINEDIS_MASK                   0x40u
#define ENET_RACC_LINEDIS_SHIFT                  6u
#define ENET_RACC_LINEDIS_WIDTH                  1u
#define ENET_RACC_LINEDIS(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_RACC_LINEDIS_SHIFT))&ENET_RACC_LINEDIS_MASK)
#define ENET_RACC_SHIFT16_MASK                   0x80u
#define ENET_RACC_SHIFT16_SHIFT                  7u
#define ENET_RACC_SHIFT16_WIDTH                  1u
#define ENET_RACC_SHIFT16(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_RACC_SHIFT16_SHIFT))&ENET_RACC_SHIFT16_MASK)
/* RCMR Bit Fields */
#define ENET_RCMR_CMP0_MASK                      0x7u
#define ENET_RCMR_CMP0_SHIFT                     0u
#define ENET_RCMR_CMP0_WIDTH                     3u
#define ENET_RCMR_CMP0(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_RCMR_CMP0_SHIFT))&ENET_RCMR_CMP0_MASK)
#define ENET_RCMR_CMP1_MASK                      0x70u
#define ENET_RCMR_CMP1_SHIFT                     4u
#define ENET_RCMR_CMP1_WIDTH                     3u
#define ENET_RCMR_CMP1(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_RCMR_CMP1_SHIFT))&ENET_RCMR_CMP1_MASK)
#define ENET_RCMR_CMP2_MASK                      0x700u
#define ENET_RCMR_CMP2_SHIFT                     8u
#define ENET_RCMR_CMP2_WIDTH                     3u
#define ENET_RCMR_CMP2(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_RCMR_CMP2_SHIFT))&ENET_RCMR_CMP2_MASK)
#define ENET_RCMR_CMP3_MASK                      0x7000u
#define ENET_RCMR_CMP3_SHIFT                     12u
#define ENET_RCMR_CMP3_WIDTH                     3u
#define ENET_RCMR_CMP3(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_RCMR_CMP3_SHIFT))&ENET_RCMR_CMP3_MASK)
#define ENET_RCMR_MATCHEN_MASK                   0x10000u
#define ENET_RCMR_MATCHEN_SHIFT                  16u
#define ENET_RCMR_MATCHEN_WIDTH                  1u
#define ENET_RCMR_MATCHEN(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_RCMR_MATCHEN_SHIFT))&ENET_RCMR_MATCHEN_MASK)
/* DMACFG Bit Fields */
#define ENET_DMACFG_IDLE_SLOPE_MASK              0xFFFFu
#define ENET_DMACFG_IDLE_SLOPE_SHIFT             0u
#define ENET_DMACFG_IDLE_SLOPE_WIDTH             16u
#define ENET_DMACFG_IDLE_SLOPE(x)                (((uint32_t)(((uint32_t)(x))<<ENET_DMACFG_IDLE_SLOPE_SHIFT))&ENET_DMACFG_IDLE_SLOPE_MASK)
#define ENET_DMACFG_DMA_CLASS_EN_MASK            0x10000u
#define ENET_DMACFG_DMA_CLASS_EN_SHIFT           16u
#define ENET_DMACFG_DMA_CLASS_EN_WIDTH           1u
#define ENET_DMACFG_DMA_CLASS_EN(x)              (((uint32_t)(((uint32_t)(x))<<ENET_DMACFG_DMA_CLASS_EN_SHIFT))&ENET_DMACFG_DMA_CLASS_EN_MASK)
#define ENET_DMACFG_CALC_NOIPG_MASK              0x20000u
#define ENET_DMACFG_CALC_NOIPG_SHIFT             17u
#define ENET_DMACFG_CALC_NOIPG_WIDTH             1u
#define ENET_DMACFG_CALC_NOIPG(x)                (((uint32_t)(((uint32_t)(x))<<ENET_DMACFG_CALC_NOIPG_SHIFT))&ENET_DMACFG_CALC_NOIPG_MASK)
/* RDAR1 Bit Fields */
#define ENET_RDAR1_RDAR_MASK                     0x1000000u
#define ENET_RDAR1_RDAR_SHIFT                    24u
#define ENET_RDAR1_RDAR_WIDTH                    1u
#define ENET_RDAR1_RDAR(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_RDAR1_RDAR_SHIFT))&ENET_RDAR1_RDAR_MASK)
/* TDAR1 Bit Fields */
#define ENET_TDAR1_TDAR_MASK                     0x1000000u
#define ENET_TDAR1_TDAR_SHIFT                    24u
#define ENET_TDAR1_TDAR_WIDTH                    1u
#define ENET_TDAR1_TDAR(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_TDAR1_TDAR_SHIFT))&ENET_TDAR1_TDAR_MASK)
/* RDAR2 Bit Fields */
#define ENET_RDAR2_RDAR_MASK                     0x1000000u
#define ENET_RDAR2_RDAR_SHIFT                    24u
#define ENET_RDAR2_RDAR_WIDTH                    1u
#define ENET_RDAR2_RDAR(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_RDAR2_RDAR_SHIFT))&ENET_RDAR2_RDAR_MASK)
/* TDAR2 Bit Fields */
#define ENET_TDAR2_TDAR_MASK                     0x1000000u
#define ENET_TDAR2_TDAR_SHIFT                    24u
#define ENET_TDAR2_TDAR_WIDTH                    1u
#define ENET_TDAR2_TDAR(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_TDAR2_TDAR_SHIFT))&ENET_TDAR2_TDAR_MASK)
/* QOS Bit Fields */
#define ENET_QOS_TX_SCHEME_MASK                  0x7u
#define ENET_QOS_TX_SCHEME_SHIFT                 0u
#define ENET_QOS_TX_SCHEME_WIDTH                 3u
#define ENET_QOS_TX_SCHEME(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_QOS_TX_SCHEME_SHIFT))&ENET_QOS_TX_SCHEME_MASK)
#define ENET_QOS_RX_FLUSH0_MASK                  0x8u
#define ENET_QOS_RX_FLUSH0_SHIFT                 3u
#define ENET_QOS_RX_FLUSH0_WIDTH                 1u
#define ENET_QOS_RX_FLUSH0(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_QOS_RX_FLUSH0_SHIFT))&ENET_QOS_RX_FLUSH0_MASK)
#define ENET_QOS_RX_FLUSH1_MASK                  0x10u
#define ENET_QOS_RX_FLUSH1_SHIFT                 4u
#define ENET_QOS_RX_FLUSH1_WIDTH                 1u
#define ENET_QOS_RX_FLUSH1(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_QOS_RX_FLUSH1_SHIFT))&ENET_QOS_RX_FLUSH1_MASK)
#define ENET_QOS_RX_FLUSH2_MASK                  0x20u
#define ENET_QOS_RX_FLUSH2_SHIFT                 5u
#define ENET_QOS_RX_FLUSH2_WIDTH                 1u
#define ENET_QOS_RX_FLUSH2(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_QOS_RX_FLUSH2_SHIFT))&ENET_QOS_RX_FLUSH2_MASK)
/* RMON_T_PACKETS Bit Fields */
#define ENET_RMON_T_PACKETS_TXPKTS_MASK          0xFFFFu
#define ENET_RMON_T_PACKETS_TXPKTS_SHIFT         0u
#define ENET_RMON_T_PACKETS_TXPKTS_WIDTH         16u
#define ENET_RMON_T_PACKETS_TXPKTS(x)            (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_PACKETS_TXPKTS_SHIFT))&ENET_RMON_T_PACKETS_TXPKTS_MASK)
/* RMON_T_BC_PKT Bit Fields */
#define ENET_RMON_T_BC_PKT_TXPKTS_MASK           0xFFFFu
#define ENET_RMON_T_BC_PKT_TXPKTS_SHIFT          0u
#define ENET_RMON_T_BC_PKT_TXPKTS_WIDTH          16u
#define ENET_RMON_T_BC_PKT_TXPKTS(x)             (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_BC_PKT_TXPKTS_SHIFT))&ENET_RMON_T_BC_PKT_TXPKTS_MASK)
/* RMON_T_MC_PKT Bit Fields */
#define ENET_RMON_T_MC_PKT_TXPKTS_MASK           0xFFFFu
#define ENET_RMON_T_MC_PKT_TXPKTS_SHIFT          0u
#define ENET_RMON_T_MC_PKT_TXPKTS_WIDTH          16u
#define ENET_RMON_T_MC_PKT_TXPKTS(x)             (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_MC_PKT_TXPKTS_SHIFT))&ENET_RMON_T_MC_PKT_TXPKTS_MASK)
/* RMON_T_CRC_ALIGN Bit Fields */
#define ENET_RMON_T_CRC_ALIGN_TXPKTS_MASK        0xFFFFu
#define ENET_RMON_T_CRC_ALIGN_TXPKTS_SHIFT       0u
#define ENET_RMON_T_CRC_ALIGN_TXPKTS_WIDTH       16u
#define ENET_RMON_T_CRC_ALIGN_TXPKTS(x)          (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_CRC_ALIGN_TXPKTS_SHIFT))&ENET_RMON_T_CRC_ALIGN_TXPKTS_MASK)
/* RMON_T_UNDERSIZE Bit Fields */
#define ENET_RMON_T_UNDERSIZE_TXPKTS_MASK        0xFFFFu
#define ENET_RMON_T_UNDERSIZE_TXPKTS_SHIFT       0u
#define ENET_RMON_T_UNDERSIZE_TXPKTS_WIDTH       16u
#define ENET_RMON_T_UNDERSIZE_TXPKTS(x)          (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_UNDERSIZE_TXPKTS_SHIFT))&ENET_RMON_T_UNDERSIZE_TXPKTS_MASK)
/* RMON_T_OVERSIZE Bit Fields */
#define ENET_RMON_T_OVERSIZE_TXPKTS_MASK         0xFFFFu
#define ENET_RMON_T_OVERSIZE_TXPKTS_SHIFT        0u
#define ENET_RMON_T_OVERSIZE_TXPKTS_WIDTH        16u
#define ENET_RMON_T_OVERSIZE_TXPKTS(x)           (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_OVERSIZE_TXPKTS_SHIFT))&ENET_RMON_T_OVERSIZE_TXPKTS_MASK)
/* RMON_T_FRAG Bit Fields */
#define ENET_RMON_T_FRAG_TXPKTS_MASK             0xFFFFu
#define ENET_RMON_T_FRAG_TXPKTS_SHIFT            0u
#define ENET_RMON_T_FRAG_TXPKTS_WIDTH            16u
#define ENET_RMON_T_FRAG_TXPKTS(x)               (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_FRAG_TXPKTS_SHIFT))&ENET_RMON_T_FRAG_TXPKTS_MASK)
/* RMON_T_JAB Bit Fields */
#define ENET_RMON_T_JAB_TXPKTS_MASK              0xFFFFu
#define ENET_RMON_T_JAB_TXPKTS_SHIFT             0u
#define ENET_RMON_T_JAB_TXPKTS_WIDTH             16u
#define ENET_RMON_T_JAB_TXPKTS(x)                (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_JAB_TXPKTS_SHIFT))&ENET_RMON_T_JAB_TXPKTS_MASK)
/* RMON_T_COL Bit Fields */
#define ENET_RMON_T_COL_TXPKTS_MASK              0xFFFFu
#define ENET_RMON_T_COL_TXPKTS_SHIFT             0u
#define ENET_RMON_T_COL_TXPKTS_WIDTH             16u
#define ENET_RMON_T_COL_TXPKTS(x)                (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_COL_TXPKTS_SHIFT))&ENET_RMON_T_COL_TXPKTS_MASK)
/* RMON_T_P64 Bit Fields */
#define ENET_RMON_T_P64_TXPKTS_MASK              0xFFFFu
#define ENET_RMON_T_P64_TXPKTS_SHIFT             0u
#define ENET_RMON_T_P64_TXPKTS_WIDTH             16u
#define ENET_RMON_T_P64_TXPKTS(x)                (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_P64_TXPKTS_SHIFT))&ENET_RMON_T_P64_TXPKTS_MASK)
/* RMON_T_P65TO127 Bit Fields */
#define ENET_RMON_T_P65TO127_TXPKTS_MASK         0xFFFFu
#define ENET_RMON_T_P65TO127_TXPKTS_SHIFT        0u
#define ENET_RMON_T_P65TO127_TXPKTS_WIDTH        16u
#define ENET_RMON_T_P65TO127_TXPKTS(x)           (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_P65TO127_TXPKTS_SHIFT))&ENET_RMON_T_P65TO127_TXPKTS_MASK)
/* RMON_T_P128TO255 Bit Fields */
#define ENET_RMON_T_P128TO255_TXPKTS_MASK        0xFFFFu
#define ENET_RMON_T_P128TO255_TXPKTS_SHIFT       0u
#define ENET_RMON_T_P128TO255_TXPKTS_WIDTH       16u
#define ENET_RMON_T_P128TO255_TXPKTS(x)          (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_P128TO255_TXPKTS_SHIFT))&ENET_RMON_T_P128TO255_TXPKTS_MASK)
/* RMON_T_P256TO511 Bit Fields */
#define ENET_RMON_T_P256TO511_TXPKTS_MASK        0xFFFFu
#define ENET_RMON_T_P256TO511_TXPKTS_SHIFT       0u
#define ENET_RMON_T_P256TO511_TXPKTS_WIDTH       16u
#define ENET_RMON_T_P256TO511_TXPKTS(x)          (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_P256TO511_TXPKTS_SHIFT))&ENET_RMON_T_P256TO511_TXPKTS_MASK)
/* RMON_T_P512TO1023 Bit Fields */
#define ENET_RMON_T_P512TO1023_TXPKTS_MASK       0xFFFFu
#define ENET_RMON_T_P512TO1023_TXPKTS_SHIFT      0u
#define ENET_RMON_T_P512TO1023_TXPKTS_WIDTH      16u
#define ENET_RMON_T_P512TO1023_TXPKTS(x)         (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_P512TO1023_TXPKTS_SHIFT))&ENET_RMON_T_P512TO1023_TXPKTS_MASK)
/* RMON_T_P1024TO2047 Bit Fields */
#define ENET_RMON_T_P1024TO2047_TXPKTS_MASK      0xFFFFu
#define ENET_RMON_T_P1024TO2047_TXPKTS_SHIFT     0u
#define ENET_RMON_T_P1024TO2047_TXPKTS_WIDTH     16u
#define ENET_RMON_T_P1024TO2047_TXPKTS(x)        (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_P1024TO2047_TXPKTS_SHIFT))&ENET_RMON_T_P1024TO2047_TXPKTS_MASK)
/* RMON_T_P_GTE2048 Bit Fields */
#define ENET_RMON_T_P_GTE2048_TXPKTS_MASK        0xFFFFu
#define ENET_RMON_T_P_GTE2048_TXPKTS_SHIFT       0u
#define ENET_RMON_T_P_GTE2048_TXPKTS_WIDTH       16u
#define ENET_RMON_T_P_GTE2048_TXPKTS(x)          (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_P_GTE2048_TXPKTS_SHIFT))&ENET_RMON_T_P_GTE2048_TXPKTS_MASK)
/* RMON_T_OCTETS Bit Fields */
#define ENET_RMON_T_OCTETS_TXOCTS_MASK           0xFFFFFFFFu
#define ENET_RMON_T_OCTETS_TXOCTS_SHIFT          0u
#define ENET_RMON_T_OCTETS_TXOCTS_WIDTH          32u
#define ENET_RMON_T_OCTETS_TXOCTS(x)             (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_OCTETS_TXOCTS_SHIFT))&ENET_RMON_T_OCTETS_TXOCTS_MASK)
/* IEEE_T_FRAME_OK Bit Fields */
#define ENET_IEEE_T_FRAME_OK_COUNT_MASK          0xFFFFu
#define ENET_IEEE_T_FRAME_OK_COUNT_SHIFT         0u
#define ENET_IEEE_T_FRAME_OK_COUNT_WIDTH         16u
#define ENET_IEEE_T_FRAME_OK_COUNT(x)            (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_FRAME_OK_COUNT_SHIFT))&ENET_IEEE_T_FRAME_OK_COUNT_MASK)
/* IEEE_T_1COL Bit Fields */
#define ENET_IEEE_T_1COL_COUNT_MASK              0xFFFFu
#define ENET_IEEE_T_1COL_COUNT_SHIFT             0u
#define ENET_IEEE_T_1COL_COUNT_WIDTH             16u
#define ENET_IEEE_T_1COL_COUNT(x)                (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_1COL_COUNT_SHIFT))&ENET_IEEE_T_1COL_COUNT_MASK)
/* IEEE_T_MCOL Bit Fields */
#define ENET_IEEE_T_MCOL_COUNT_MASK              0xFFFFu
#define ENET_IEEE_T_MCOL_COUNT_SHIFT             0u
#define ENET_IEEE_T_MCOL_COUNT_WIDTH             16u
#define ENET_IEEE_T_MCOL_COUNT(x)                (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_MCOL_COUNT_SHIFT))&ENET_IEEE_T_MCOL_COUNT_MASK)
/* IEEE_T_DEF Bit Fields */
#define ENET_IEEE_T_DEF_COUNT_MASK               0xFFFFu
#define ENET_IEEE_T_DEF_COUNT_SHIFT              0u
#define ENET_IEEE_T_DEF_COUNT_WIDTH              16u
#define ENET_IEEE_T_DEF_COUNT(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_DEF_COUNT_SHIFT))&ENET_IEEE_T_DEF_COUNT_MASK)
/* IEEE_T_LCOL Bit Fields */
#define ENET_IEEE_T_LCOL_COUNT_MASK              0xFFFFu
#define ENET_IEEE_T_LCOL_COUNT_SHIFT             0u
#define ENET_IEEE_T_LCOL_COUNT_WIDTH             16u
#define ENET_IEEE_T_LCOL_COUNT(x)                (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_LCOL_COUNT_SHIFT))&ENET_IEEE_T_LCOL_COUNT_MASK)
/* IEEE_T_EXCOL Bit Fields */
#define ENET_IEEE_T_EXCOL_COUNT_MASK             0xFFFFu
#define ENET_IEEE_T_EXCOL_COUNT_SHIFT            0u
#define ENET_IEEE_T_EXCOL_COUNT_WIDTH            16u
#define ENET_IEEE_T_EXCOL_COUNT(x)               (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_EXCOL_COUNT_SHIFT))&ENET_IEEE_T_EXCOL_COUNT_MASK)
/* IEEE_T_MACERR Bit Fields */
#define ENET_IEEE_T_MACERR_COUNT_MASK            0xFFFFu
#define ENET_IEEE_T_MACERR_COUNT_SHIFT           0u
#define ENET_IEEE_T_MACERR_COUNT_WIDTH           16u
#define ENET_IEEE_T_MACERR_COUNT(x)              (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_MACERR_COUNT_SHIFT))&ENET_IEEE_T_MACERR_COUNT_MASK)
/* IEEE_T_CSERR Bit Fields */
#define ENET_IEEE_T_CSERR_COUNT_MASK             0xFFFFu
#define ENET_IEEE_T_CSERR_COUNT_SHIFT            0u
#define ENET_IEEE_T_CSERR_COUNT_WIDTH            16u
#define ENET_IEEE_T_CSERR_COUNT(x)               (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_CSERR_COUNT_SHIFT))&ENET_IEEE_T_CSERR_COUNT_MASK)
/* IEEE_T_SQE Bit Fields */
#define ENET_IEEE_T_SQE_COUNT_MASK               0xFFFFu
#define ENET_IEEE_T_SQE_COUNT_SHIFT              0u
#define ENET_IEEE_T_SQE_COUNT_WIDTH              16u
#define ENET_IEEE_T_SQE_COUNT(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_SQE_COUNT_SHIFT))&ENET_IEEE_T_SQE_COUNT_MASK)
/* IEEE_T_FDXFC Bit Fields */
#define ENET_IEEE_T_FDXFC_COUNT_MASK             0xFFFFu
#define ENET_IEEE_T_FDXFC_COUNT_SHIFT            0u
#define ENET_IEEE_T_FDXFC_COUNT_WIDTH            16u
#define ENET_IEEE_T_FDXFC_COUNT(x)               (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_FDXFC_COUNT_SHIFT))&ENET_IEEE_T_FDXFC_COUNT_MASK)
/* IEEE_T_OCTETS_OK Bit Fields */
#define ENET_IEEE_T_OCTETS_OK_COUNT_MASK         0xFFFFFFFFu
#define ENET_IEEE_T_OCTETS_OK_COUNT_SHIFT        0u
#define ENET_IEEE_T_OCTETS_OK_COUNT_WIDTH        32u
#define ENET_IEEE_T_OCTETS_OK_COUNT(x)           (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_OCTETS_OK_COUNT_SHIFT))&ENET_IEEE_T_OCTETS_OK_COUNT_MASK)
/* RMON_R_PACKETS Bit Fields */
#define ENET_RMON_R_PACKETS_COUNT_MASK           0xFFFFu
#define ENET_RMON_R_PACKETS_COUNT_SHIFT          0u
#define ENET_RMON_R_PACKETS_COUNT_WIDTH          16u
#define ENET_RMON_R_PACKETS_COUNT(x)             (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_PACKETS_COUNT_SHIFT))&ENET_RMON_R_PACKETS_COUNT_MASK)
/* RMON_R_BC_PKT Bit Fields */
#define ENET_RMON_R_BC_PKT_COUNT_MASK            0xFFFFu
#define ENET_RMON_R_BC_PKT_COUNT_SHIFT           0u
#define ENET_RMON_R_BC_PKT_COUNT_WIDTH           16u
#define ENET_RMON_R_BC_PKT_COUNT(x)              (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_BC_PKT_COUNT_SHIFT))&ENET_RMON_R_BC_PKT_COUNT_MASK)
/* RMON_R_MC_PKT Bit Fields */
#define ENET_RMON_R_MC_PKT_COUNT_MASK            0xFFFFu
#define ENET_RMON_R_MC_PKT_COUNT_SHIFT           0u
#define ENET_RMON_R_MC_PKT_COUNT_WIDTH           16u
#define ENET_RMON_R_MC_PKT_COUNT(x)              (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_MC_PKT_COUNT_SHIFT))&ENET_RMON_R_MC_PKT_COUNT_MASK)
/* RMON_R_CRC_ALIGN Bit Fields */
#define ENET_RMON_R_CRC_ALIGN_COUNT_MASK         0xFFFFu
#define ENET_RMON_R_CRC_ALIGN_COUNT_SHIFT        0u
#define ENET_RMON_R_CRC_ALIGN_COUNT_WIDTH        16u
#define ENET_RMON_R_CRC_ALIGN_COUNT(x)           (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_CRC_ALIGN_COUNT_SHIFT))&ENET_RMON_R_CRC_ALIGN_COUNT_MASK)
/* RMON_R_UNDERSIZE Bit Fields */
#define ENET_RMON_R_UNDERSIZE_COUNT_MASK         0xFFFFu
#define ENET_RMON_R_UNDERSIZE_COUNT_SHIFT        0u
#define ENET_RMON_R_UNDERSIZE_COUNT_WIDTH        16u
#define ENET_RMON_R_UNDERSIZE_COUNT(x)           (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_UNDERSIZE_COUNT_SHIFT))&ENET_RMON_R_UNDERSIZE_COUNT_MASK)
/* RMON_R_OVERSIZE Bit Fields */
#define ENET_RMON_R_OVERSIZE_COUNT_MASK          0xFFFFu
#define ENET_RMON_R_OVERSIZE_COUNT_SHIFT         0u
#define ENET_RMON_R_OVERSIZE_COUNT_WIDTH         16u
#define ENET_RMON_R_OVERSIZE_COUNT(x)            (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_OVERSIZE_COUNT_SHIFT))&ENET_RMON_R_OVERSIZE_COUNT_MASK)
/* RMON_R_FRAG Bit Fields */
#define ENET_RMON_R_FRAG_COUNT_MASK              0xFFFFu
#define ENET_RMON_R_FRAG_COUNT_SHIFT             0u
#define ENET_RMON_R_FRAG_COUNT_WIDTH             16u
#define ENET_RMON_R_FRAG_COUNT(x)                (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_FRAG_COUNT_SHIFT))&ENET_RMON_R_FRAG_COUNT_MASK)
/* RMON_R_JAB Bit Fields */
#define ENET_RMON_R_JAB_COUNT_MASK               0xFFFFu
#define ENET_RMON_R_JAB_COUNT_SHIFT              0u
#define ENET_RMON_R_JAB_COUNT_WIDTH              16u
#define ENET_RMON_R_JAB_COUNT(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_JAB_COUNT_SHIFT))&ENET_RMON_R_JAB_COUNT_MASK)
/* RMON_R_P64 Bit Fields */
#define ENET_RMON_R_P64_COUNT_MASK               0xFFFFu
#define ENET_RMON_R_P64_COUNT_SHIFT              0u
#define ENET_RMON_R_P64_COUNT_WIDTH              16u
#define ENET_RMON_R_P64_COUNT(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_P64_COUNT_SHIFT))&ENET_RMON_R_P64_COUNT_MASK)
/* RMON_R_P65TO127 Bit Fields */
#define ENET_RMON_R_P65TO127_COUNT_MASK          0xFFFFu
#define ENET_RMON_R_P65TO127_COUNT_SHIFT         0u
#define ENET_RMON_R_P65TO127_COUNT_WIDTH         16u
#define ENET_RMON_R_P65TO127_COUNT(x)            (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_P65TO127_COUNT_SHIFT))&ENET_RMON_R_P65TO127_COUNT_MASK)
/* RMON_R_P128TO255 Bit Fields */
#define ENET_RMON_R_P128TO255_COUNT_MASK         0xFFFFu
#define ENET_RMON_R_P128TO255_COUNT_SHIFT        0u
#define ENET_RMON_R_P128TO255_COUNT_WIDTH        16u
#define ENET_RMON_R_P128TO255_COUNT(x)           (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_P128TO255_COUNT_SHIFT))&ENET_RMON_R_P128TO255_COUNT_MASK)
/* RMON_R_P256TO511 Bit Fields */
#define ENET_RMON_R_P256TO511_COUNT_MASK         0xFFFFu
#define ENET_RMON_R_P256TO511_COUNT_SHIFT        0u
#define ENET_RMON_R_P256TO511_COUNT_WIDTH        16u
#define ENET_RMON_R_P256TO511_COUNT(x)           (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_P256TO511_COUNT_SHIFT))&ENET_RMON_R_P256TO511_COUNT_MASK)
/* RMON_R_P512TO1023 Bit Fields */
#define ENET_RMON_R_P512TO1023_COUNT_MASK        0xFFFFu
#define ENET_RMON_R_P512TO1023_COUNT_SHIFT       0u
#define ENET_RMON_R_P512TO1023_COUNT_WIDTH       16u
#define ENET_RMON_R_P512TO1023_COUNT(x)          (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_P512TO1023_COUNT_SHIFT))&ENET_RMON_R_P512TO1023_COUNT_MASK)
/* RMON_R_P1024TO2047 Bit Fields */
#define ENET_RMON_R_P1024TO2047_COUNT_MASK       0xFFFFu
#define ENET_RMON_R_P1024TO2047_COUNT_SHIFT      0u
#define ENET_RMON_R_P1024TO2047_COUNT_WIDTH      16u
#define ENET_RMON_R_P1024TO2047_COUNT(x)         (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_P1024TO2047_COUNT_SHIFT))&ENET_RMON_R_P1024TO2047_COUNT_MASK)
/* RMON_R_P_GTE2048 Bit Fields */
#define ENET_RMON_R_P_GTE2048_COUNT_MASK         0xFFFFu
#define ENET_RMON_R_P_GTE2048_COUNT_SHIFT        0u
#define ENET_RMON_R_P_GTE2048_COUNT_WIDTH        16u
#define ENET_RMON_R_P_GTE2048_COUNT(x)           (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_P_GTE2048_COUNT_SHIFT))&ENET_RMON_R_P_GTE2048_COUNT_MASK)
/* RMON_R_OCTETS Bit Fields */
#define ENET_RMON_R_OCTETS_COUNT_MASK            0xFFFFFFFFu
#define ENET_RMON_R_OCTETS_COUNT_SHIFT           0u
#define ENET_RMON_R_OCTETS_COUNT_WIDTH           32u
#define ENET_RMON_R_OCTETS_COUNT(x)              (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_OCTETS_COUNT_SHIFT))&ENET_RMON_R_OCTETS_COUNT_MASK)
/* IEEE_R_DROP Bit Fields */
#define ENET_IEEE_R_DROP_COUNT_MASK              0xFFFFu
#define ENET_IEEE_R_DROP_COUNT_SHIFT             0u
#define ENET_IEEE_R_DROP_COUNT_WIDTH             16u
#define ENET_IEEE_R_DROP_COUNT(x)                (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_R_DROP_COUNT_SHIFT))&ENET_IEEE_R_DROP_COUNT_MASK)
/* IEEE_R_FRAME_OK Bit Fields */
#define ENET_IEEE_R_FRAME_OK_COUNT_MASK          0xFFFFu
#define ENET_IEEE_R_FRAME_OK_COUNT_SHIFT         0u
#define ENET_IEEE_R_FRAME_OK_COUNT_WIDTH         16u
#define ENET_IEEE_R_FRAME_OK_COUNT(x)            (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_R_FRAME_OK_COUNT_SHIFT))&ENET_IEEE_R_FRAME_OK_COUNT_MASK)
/* IEEE_R_CRC Bit Fields */
#define ENET_IEEE_R_CRC_COUNT_MASK               0xFFFFu
#define ENET_IEEE_R_CRC_COUNT_SHIFT              0u
#define ENET_IEEE_R_CRC_COUNT_WIDTH              16u
#define ENET_IEEE_R_CRC_COUNT(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_R_CRC_COUNT_SHIFT))&ENET_IEEE_R_CRC_COUNT_MASK)
/* IEEE_R_ALIGN Bit Fields */
#define ENET_IEEE_R_ALIGN_COUNT_MASK             0xFFFFu
#define ENET_IEEE_R_ALIGN_COUNT_SHIFT            0u
#define ENET_IEEE_R_ALIGN_COUNT_WIDTH            16u
#define ENET_IEEE_R_ALIGN_COUNT(x)               (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_R_ALIGN_COUNT_SHIFT))&ENET_IEEE_R_ALIGN_COUNT_MASK)
/* IEEE_R_MACERR Bit Fields */
#define ENET_IEEE_R_MACERR_COUNT_MASK            0xFFFFu
#define ENET_IEEE_R_MACERR_COUNT_SHIFT           0u
#define ENET_IEEE_R_MACERR_COUNT_WIDTH           16u
#define ENET_IEEE_R_MACERR_COUNT(x)              (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_R_MACERR_COUNT_SHIFT))&ENET_IEEE_R_MACERR_COUNT_MASK)
/* IEEE_R_FDXFC Bit Fields */
#define ENET_IEEE_R_FDXFC_COUNT_MASK             0xFFFFu
#define ENET_IEEE_R_FDXFC_COUNT_SHIFT            0u
#define ENET_IEEE_R_FDXFC_COUNT_WIDTH            16u
#define ENET_IEEE_R_FDXFC_COUNT(x)               (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_R_FDXFC_COUNT_SHIFT))&ENET_IEEE_R_FDXFC_COUNT_MASK)
/* IEEE_R_OCTETS_OK Bit Fields */
#define ENET_IEEE_R_OCTETS_OK_COUNT_MASK         0xFFFFFFFFu
#define ENET_IEEE_R_OCTETS_OK_COUNT_SHIFT        0u
#define ENET_IEEE_R_OCTETS_OK_COUNT_WIDTH        32u
#define ENET_IEEE_R_OCTETS_OK_COUNT(x)           (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_R_OCTETS_OK_COUNT_SHIFT))&ENET_IEEE_R_OCTETS_OK_COUNT_MASK)
/* ATCR Bit Fields */
#define ENET_ATCR_EN_MASK                        0x1u
#define ENET_ATCR_EN_SHIFT                       0u
#define ENET_ATCR_EN_WIDTH                       1u
#define ENET_ATCR_EN(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_ATCR_EN_SHIFT))&ENET_ATCR_EN_MASK)
#define ENET_ATCR_OFFEN_MASK                     0x4u
#define ENET_ATCR_OFFEN_SHIFT                    2u
#define ENET_ATCR_OFFEN_WIDTH                    1u
#define ENET_ATCR_OFFEN(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_ATCR_OFFEN_SHIFT))&ENET_ATCR_OFFEN_MASK)
#define ENET_ATCR_OFFRST_MASK                    0x8u
#define ENET_ATCR_OFFRST_SHIFT                   3u
#define ENET_ATCR_OFFRST_WIDTH                   1u
#define ENET_ATCR_OFFRST(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_ATCR_OFFRST_SHIFT))&ENET_ATCR_OFFRST_MASK)
#define ENET_ATCR_PEREN_MASK                     0x10u
#define ENET_ATCR_PEREN_SHIFT                    4u
#define ENET_ATCR_PEREN_WIDTH                    1u
#define ENET_ATCR_PEREN(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_ATCR_PEREN_SHIFT))&ENET_ATCR_PEREN_MASK)
#define ENET_ATCR_PINPER_MASK                    0x80u
#define ENET_ATCR_PINPER_SHIFT                   7u
#define ENET_ATCR_PINPER_WIDTH                   1u
#define ENET_ATCR_PINPER(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_ATCR_PINPER_SHIFT))&ENET_ATCR_PINPER_MASK)
#define ENET_ATCR_RESTART_MASK                   0x200u
#define ENET_ATCR_RESTART_SHIFT                  9u
#define ENET_ATCR_RESTART_WIDTH                  1u
#define ENET_ATCR_RESTART(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_ATCR_RESTART_SHIFT))&ENET_ATCR_RESTART_MASK)
#define ENET_ATCR_CAPTURE_MASK                   0x800u
#define ENET_ATCR_CAPTURE_SHIFT                  11u
#define ENET_ATCR_CAPTURE_WIDTH                  1u
#define ENET_ATCR_CAPTURE(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_ATCR_CAPTURE_SHIFT))&ENET_ATCR_CAPTURE_MASK)
#define ENET_ATCR_SLAVE_MASK                     0x2000u
#define ENET_ATCR_SLAVE_SHIFT                    13u
#define ENET_ATCR_SLAVE_WIDTH                    1u
#define ENET_ATCR_SLAVE(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_ATCR_SLAVE_SHIFT))&ENET_ATCR_SLAVE_MASK)
/* ATVR Bit Fields */
#define ENET_ATVR_ATIME_MASK                     0xFFFFFFFFu
#define ENET_ATVR_ATIME_SHIFT                    0u
#define ENET_ATVR_ATIME_WIDTH                    32u
#define ENET_ATVR_ATIME(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_ATVR_ATIME_SHIFT))&ENET_ATVR_ATIME_MASK)
/* ATOFF Bit Fields */
#define ENET_ATOFF_OFFSET_MASK                   0xFFFFFFFFu
#define ENET_ATOFF_OFFSET_SHIFT                  0u
#define ENET_ATOFF_OFFSET_WIDTH                  32u
#define ENET_ATOFF_OFFSET(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_ATOFF_OFFSET_SHIFT))&ENET_ATOFF_OFFSET_MASK)
/* ATPER Bit Fields */
#define ENET_ATPER_PERIOD_MASK                   0xFFFFFFFFu
#define ENET_ATPER_PERIOD_SHIFT                  0u
#define ENET_ATPER_PERIOD_WIDTH                  32u
#define ENET_ATPER_PERIOD(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_ATPER_PERIOD_SHIFT))&ENET_ATPER_PERIOD_MASK)
/* ATCOR Bit Fields */
#define ENET_ATCOR_COR_MASK                      0x7FFFFFFFu
#define ENET_ATCOR_COR_SHIFT                     0u
#define ENET_ATCOR_COR_WIDTH                     31u
#define ENET_ATCOR_COR(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_ATCOR_COR_SHIFT))&ENET_ATCOR_COR_MASK)
/* ATINC Bit Fields */
#define ENET_ATINC_INC_MASK                      0x7Fu
#define ENET_ATINC_INC_SHIFT                     0u
#define ENET_ATINC_INC_WIDTH                     7u
#define ENET_ATINC_INC(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_ATINC_INC_SHIFT))&ENET_ATINC_INC_MASK)
#define ENET_ATINC_INC_CORR_MASK                 0x7F00u
#define ENET_ATINC_INC_CORR_SHIFT                8u
#define ENET_ATINC_INC_CORR_WIDTH                7u
#define ENET_ATINC_INC_CORR(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_ATINC_INC_CORR_SHIFT))&ENET_ATINC_INC_CORR_MASK)
/* ATSTMP Bit Fields */
#define ENET_ATSTMP_TIMESTAMP_MASK               0xFFFFFFFFu
#define ENET_ATSTMP_TIMESTAMP_SHIFT              0u
#define ENET_ATSTMP_TIMESTAMP_WIDTH              32u
#define ENET_ATSTMP_TIMESTAMP(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_ATSTMP_TIMESTAMP_SHIFT))&ENET_ATSTMP_TIMESTAMP_MASK)
/* MDATA Bit Fields */
#define ENET_MDATA_MATCHDATA_MASK                0xFFFFFFFFu
#define ENET_MDATA_MATCHDATA_SHIFT               0u
#define ENET_MDATA_MATCHDATA_WIDTH               32u
#define ENET_MDATA_MATCHDATA(x)                  (((uint32_t)(((uint32_t)(x))<<ENET_MDATA_MATCHDATA_SHIFT))&ENET_MDATA_MATCHDATA_MASK)
/* MMASK Bit Fields */
#define ENET_MMASK_MATCHMASK_MASK                0xFFFFFFFFu
#define ENET_MMASK_MATCHMASK_SHIFT               0u
#define ENET_MMASK_MATCHMASK_WIDTH               32u
#define ENET_MMASK_MATCHMASK(x)                  (((uint32_t)(((uint32_t)(x))<<ENET_MMASK_MATCHMASK_SHIFT))&ENET_MMASK_MATCHMASK_MASK)
/* MCONFIG Bit Fields */
#define ENET_MCONFIG_FRMOFF_MASK                 0xFCu
#define ENET_MCONFIG_FRMOFF_SHIFT                2u
#define ENET_MCONFIG_FRMOFF_WIDTH                6u
#define ENET_MCONFIG_FRMOFF(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_MCONFIG_FRMOFF_SHIFT))&ENET_MCONFIG_FRMOFF_MASK)
#define ENET_MCONFIG_OK_INDEX_MASK               0xFF0000u
#define ENET_MCONFIG_OK_INDEX_SHIFT              16u
#define ENET_MCONFIG_OK_INDEX_WIDTH              8u
#define ENET_MCONFIG_OK_INDEX(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_MCONFIG_OK_INDEX_SHIFT))&ENET_MCONFIG_OK_INDEX_MASK)
#define ENET_MCONFIG_IM_MASK                     0x20000000u
#define ENET_MCONFIG_IM_SHIFT                    29u
#define ENET_MCONFIG_IM_WIDTH                    1u
#define ENET_MCONFIG_IM(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_MCONFIG_IM_SHIFT))&ENET_MCONFIG_IM_MASK)
#define ENET_MCONFIG_RF_MASK                     0x40000000u
#define ENET_MCONFIG_RF_SHIFT                    30u
#define ENET_MCONFIG_RF_WIDTH                    1u
#define ENET_MCONFIG_RF(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_MCONFIG_RF_SHIFT))&ENET_MCONFIG_RF_MASK)
#define ENET_MCONFIG_AF_MASK                     0x80000000u
#define ENET_MCONFIG_AF_SHIFT                    31u
#define ENET_MCONFIG_AF_WIDTH                    1u
#define ENET_MCONFIG_AF(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_MCONFIG_AF_SHIFT))&ENET_MCONFIG_AF_MASK)
/* MENTRYRW Bit Fields */
#define ENET_MENTRYRW_ENTRYADD_MASK              0xFFu
#define ENET_MENTRYRW_ENTRYADD_SHIFT             0u
#define ENET_MENTRYRW_ENTRYADD_WIDTH             8u
#define ENET_MENTRYRW_ENTRYADD(x)                (((uint32_t)(((uint32_t)(x))<<ENET_MENTRYRW_ENTRYADD_SHIFT))&ENET_MENTRYRW_ENTRYADD_MASK)
#define ENET_MENTRYRW_WR_MASK                    0x100u
#define ENET_MENTRYRW_WR_SHIFT                   8u
#define ENET_MENTRYRW_WR_WIDTH                   1u
#define ENET_MENTRYRW_WR(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_MENTRYRW_WR_SHIFT))&ENET_MENTRYRW_WR_MASK)
#define ENET_MENTRYRW_RD_MASK                    0x200u
#define ENET_MENTRYRW_RD_SHIFT                   9u
#define ENET_MENTRYRW_RD_WIDTH                   1u
#define ENET_MENTRYRW_RD(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_MENTRYRW_RD_SHIFT))&ENET_MENTRYRW_RD_MASK)
/* RXPCTL Bit Fields */
#define ENET_RXPCTL_ENPARSER_MASK                0x1u
#define ENET_RXPCTL_ENPARSER_SHIFT               0u
#define ENET_RXPCTL_ENPARSER_WIDTH               1u
#define ENET_RXPCTL_ENPARSER(x)                  (((uint32_t)(((uint32_t)(x))<<ENET_RXPCTL_ENPARSER_SHIFT))&ENET_RXPCTL_ENPARSER_MASK)
#define ENET_RXPCTL_INVBYTORD_MASK               0x2u
#define ENET_RXPCTL_INVBYTORD_SHIFT              1u
#define ENET_RXPCTL_INVBYTORD_WIDTH              1u
#define ENET_RXPCTL_INVBYTORD(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_RXPCTL_INVBYTORD_SHIFT))&ENET_RXPCTL_INVBYTORD_MASK)
#define ENET_RXPCTL_PRSRSCLR_MASK                0x10u
#define ENET_RXPCTL_PRSRSCLR_SHIFT               4u
#define ENET_RXPCTL_PRSRSCLR_WIDTH               1u
#define ENET_RXPCTL_PRSRSCLR(x)                  (((uint32_t)(((uint32_t)(x))<<ENET_RXPCTL_PRSRSCLR_SHIFT))&ENET_RXPCTL_PRSRSCLR_MASK)
#define ENET_RXPCTL_MAXINDEX_MASK                0xFF00u
#define ENET_RXPCTL_MAXINDEX_SHIFT               8u
#define ENET_RXPCTL_MAXINDEX_WIDTH               8u
#define ENET_RXPCTL_MAXINDEX(x)                  (((uint32_t)(((uint32_t)(x))<<ENET_RXPCTL_MAXINDEX_SHIFT))&ENET_RXPCTL_MAXINDEX_MASK)
#define ENET_RXPCTL_ENDERRQ_MASK                 0xFF0000u
#define ENET_RXPCTL_ENDERRQ_SHIFT                16u
#define ENET_RXPCTL_ENDERRQ_WIDTH                8u
#define ENET_RXPCTL_ENDERRQ(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_RXPCTL_ENDERRQ_SHIFT))&ENET_RXPCTL_ENDERRQ_MASK)
#define ENET_RXPCTL_ACPTEERR_MASK                0x1000000u
#define ENET_RXPCTL_ACPTEERR_SHIFT               24u
#define ENET_RXPCTL_ACPTEERR_WIDTH               1u
#define ENET_RXPCTL_ACPTEERR(x)                  (((uint32_t)(((uint32_t)(x))<<ENET_RXPCTL_ACPTEERR_SHIFT))&ENET_RXPCTL_ACPTEERR_MASK)
/* MAXFRMOFF Bit Fields */
#define ENET_MAXFRMOFF_MXFRMOFF_MASK             0x3Fu
#define ENET_MAXFRMOFF_MXFRMOFF_SHIFT            0u
#define ENET_MAXFRMOFF_MXFRMOFF_WIDTH            6u
#define ENET_MAXFRMOFF_MXFRMOFF(x)               (((uint32_t)(((uint32_t)(x))<<ENET_MAXFRMOFF_MXFRMOFF_SHIFT))&ENET_MAXFRMOFF_MXFRMOFF_MASK)
/* RXPARST Bit Fields */
#define ENET_RXPARST_MXINDERR_MASK               0x1u
#define ENET_RXPARST_MXINDERR_SHIFT              0u
#define ENET_RXPARST_MXINDERR_WIDTH              1u
#define ENET_RXPARST_MXINDERR(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_RXPARST_MXINDERR_SHIFT))&ENET_RXPARST_MXINDERR_MASK)
#define ENET_RXPARST_TBLDPTERR_MASK              0x2u
#define ENET_RXPARST_TBLDPTERR_SHIFT             1u
#define ENET_RXPARST_TBLDPTERR_WIDTH             1u
#define ENET_RXPARST_TBLDPTERR(x)                (((uint32_t)(((uint32_t)(x))<<ENET_RXPARST_TBLDPTERR_SHIFT))&ENET_RXPARST_TBLDPTERR_MASK)
#define ENET_RXPARST_NOMTCERR_MASK               0x4u
#define ENET_RXPARST_NOMTCERR_SHIFT              2u
#define ENET_RXPARST_NOMTCERR_WIDTH              1u
#define ENET_RXPARST_NOMTCERR(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_RXPARST_NOMTCERR_SHIFT))&ENET_RXPARST_NOMTCERR_MASK)
#define ENET_RXPARST_FMOFFERR_MASK               0x8u
#define ENET_RXPARST_FMOFFERR_SHIFT              3u
#define ENET_RXPARST_FMOFFERR_WIDTH              1u
#define ENET_RXPARST_FMOFFERR(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_RXPARST_FMOFFERR_SHIFT))&ENET_RXPARST_FMOFFERR_MASK)
#define ENET_RXPARST_PRSENDERR_MASK              0x10u
#define ENET_RXPARST_PRSENDERR_SHIFT             4u
#define ENET_RXPARST_PRSENDERR_WIDTH             1u
#define ENET_RXPARST_PRSENDERR(x)                (((uint32_t)(((uint32_t)(x))<<ENET_RXPARST_PRSENDERR_SHIFT))&ENET_RXPARST_PRSENDERR_MASK)
#define ENET_RXPARST_INVMAXIDX_MASK              0x20u
#define ENET_RXPARST_INVMAXIDX_SHIFT             5u
#define ENET_RXPARST_INVMAXIDX_WIDTH             1u
#define ENET_RXPARST_INVMAXIDX(x)                (((uint32_t)(((uint32_t)(x))<<ENET_RXPARST_INVMAXIDX_SHIFT))&ENET_RXPARST_INVMAXIDX_MASK)
#define ENET_RXPARST_RXPRSDN_MASK                0x100u
#define ENET_RXPARST_RXPRSDN_SHIFT               8u
#define ENET_RXPARST_RXPRSDN_WIDTH               1u
#define ENET_RXPARST_RXPRSDN(x)                  (((uint32_t)(((uint32_t)(x))<<ENET_RXPARST_RXPRSDN_SHIFT))&ENET_RXPARST_RXPRSDN_MASK)
/* PARSDSCD Bit Fields */
#define ENET_PARSDSCD_COUNT_MASK                 0xFFFFFFFFu
#define ENET_PARSDSCD_COUNT_SHIFT                0u
#define ENET_PARSDSCD_COUNT_WIDTH                32u
#define ENET_PARSDSCD_COUNT(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_PARSDSCD_COUNT_SHIFT))&ENET_PARSDSCD_COUNT_MASK)
/* PRSACPT0 Bit Fields */
#define ENET_PRSACPT0_COUNT_MASK                 0xFFFFFFFFu
#define ENET_PRSACPT0_COUNT_SHIFT                0u
#define ENET_PRSACPT0_COUNT_WIDTH                32u
#define ENET_PRSACPT0_COUNT(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_PRSACPT0_COUNT_SHIFT))&ENET_PRSACPT0_COUNT_MASK)
/* PRSRJCT0 Bit Fields */
#define ENET_PRSRJCT0_COUNT_MASK                 0xFFFFFFFFu
#define ENET_PRSRJCT0_COUNT_SHIFT                0u
#define ENET_PRSRJCT0_COUNT_WIDTH                32u
#define ENET_PRSRJCT0_COUNT(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_PRSRJCT0_COUNT_SHIFT))&ENET_PRSRJCT0_COUNT_MASK)
/* PRSACPT1 Bit Fields */
#define ENET_PRSACPT1_COUNT_MASK                 0xFFFFFFFFu
#define ENET_PRSACPT1_COUNT_SHIFT                0u
#define ENET_PRSACPT1_COUNT_WIDTH                32u
#define ENET_PRSACPT1_COUNT(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_PRSACPT1_COUNT_SHIFT))&ENET_PRSACPT1_COUNT_MASK)
/* PRSRJCT1 Bit Fields */
#define ENET_PRSRJCT1_COUNT_MASK                 0xFFFFFFFFu
#define ENET_PRSRJCT1_COUNT_SHIFT                0u
#define ENET_PRSRJCT1_COUNT_WIDTH                32u
#define ENET_PRSRJCT1_COUNT(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_PRSRJCT1_COUNT_SHIFT))&ENET_PRSRJCT1_COUNT_MASK)
/* PRSACPT2 Bit Fields */
#define ENET_PRSACPT2_COUNT_MASK                 0xFFFFFFFFu
#define ENET_PRSACPT2_COUNT_SHIFT                0u
#define ENET_PRSACPT2_COUNT_WIDTH                32u
#define ENET_PRSACPT2_COUNT(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_PRSACPT2_COUNT_SHIFT))&ENET_PRSACPT2_COUNT_MASK)
/* PRSRJCT2 Bit Fields */
#define ENET_PRSRJCT2_COUNT_MASK                 0xFFFFFFFFu
#define ENET_PRSRJCT2_COUNT_SHIFT                0u
#define ENET_PRSRJCT2_COUNT_WIDTH                32u
#define ENET_PRSRJCT2_COUNT(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_PRSRJCT2_COUNT_SHIFT))&ENET_PRSRJCT2_COUNT_MASK)
/* TGSR Bit Fields */
#define ENET_TGSR_TF0_MASK                       0x1u
#define ENET_TGSR_TF0_SHIFT                      0u
#define ENET_TGSR_TF0_WIDTH                      1u
#define ENET_TGSR_TF0(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_TGSR_TF0_SHIFT))&ENET_TGSR_TF0_MASK)
#define ENET_TGSR_TF1_MASK                       0x2u
#define ENET_TGSR_TF1_SHIFT                      1u
#define ENET_TGSR_TF1_WIDTH                      1u
#define ENET_TGSR_TF1(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_TGSR_TF1_SHIFT))&ENET_TGSR_TF1_MASK)
#define ENET_TGSR_TF2_MASK                       0x4u
#define ENET_TGSR_TF2_SHIFT                      2u
#define ENET_TGSR_TF2_WIDTH                      1u
#define ENET_TGSR_TF2(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_TGSR_TF2_SHIFT))&ENET_TGSR_TF2_MASK)
#define ENET_TGSR_TF3_MASK                       0x8u
#define ENET_TGSR_TF3_SHIFT                      3u
#define ENET_TGSR_TF3_WIDTH                      1u
#define ENET_TGSR_TF3(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_TGSR_TF3_SHIFT))&ENET_TGSR_TF3_MASK)
/* TCSR Bit Fields */
#define ENET_TCSR_TDRE_MASK                      0x1u
#define ENET_TCSR_TDRE_SHIFT                     0u
#define ENET_TCSR_TDRE_WIDTH                     1u
#define ENET_TCSR_TDRE(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_TCSR_TDRE_SHIFT))&ENET_TCSR_TDRE_MASK)
#define ENET_TCSR_TMODE_MASK                     0x3Cu
#define ENET_TCSR_TMODE_SHIFT                    2u
#define ENET_TCSR_TMODE_WIDTH                    4u
#define ENET_TCSR_TMODE(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_TCSR_TMODE_SHIFT))&ENET_TCSR_TMODE_MASK)
#define ENET_TCSR_TIE_MASK                       0x40u
#define ENET_TCSR_TIE_SHIFT                      6u
#define ENET_TCSR_TIE_WIDTH                      1u
#define ENET_TCSR_TIE(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_TCSR_TIE_SHIFT))&ENET_TCSR_TIE_MASK)
#define ENET_TCSR_TF_MASK                        0x80u
#define ENET_TCSR_TF_SHIFT                       7u
#define ENET_TCSR_TF_WIDTH                       1u
#define ENET_TCSR_TF(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_TCSR_TF_SHIFT))&ENET_TCSR_TF_MASK)
/* TCCR Bit Fields */
#define ENET_TCCR_TCC_MASK                       0xFFFFFFFFu
#define ENET_TCCR_TCC_SHIFT                      0u
#define ENET_TCCR_TCC_WIDTH                      32u
#define ENET_TCCR_TCC(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_TCCR_TCC_SHIFT))&ENET_TCCR_TCC_MASK)

/*!
 * @}
 */ /* end of group ENET_Register_Masks */


/*!
 * @}
 */ /* end of group ENET_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- ERM Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup ERM_Peripheral_Access_Layer ERM Peripheral Access Layer
 * @{
 */


/** ERM - Size of Registers Arrays */
#define ERM_EARn_COUNT                           7u

/** ERM - Register Layout Typedef */
typedef struct {
  __IO uint32_t CR0;                               /**< ERM Configuration Register, offset: 0x0 */
       uint8_t RESERVED_0[12];
  __IO uint32_t SR0;                               /**< ERM Status Register, offset: 0x10 */
       uint8_t RESERVED_1[236];
  struct {                                         /* offset: 0x100, array step: 0x10 */
    __I  uint32_t EAR;                               /**< ERM Memory n Error Address Register, array offset: 0x100, array step: 0x10 */
    __I  uint32_t SYN;                               /**< ERM Memory n Syndrome Register, array offset: 0x104, array step: 0x10 */
         uint8_t RESERVED_0[8];
  } EARn[ERM_EARn_COUNT];
} ERM_Type, *ERM_MemMapPtr;

 /** Number of instances of the ERM module. */
#define ERM_INSTANCE_COUNT                       (1u)


/* ERM - Peripheral instance base addresses */
/** Peripheral ERM base address */
#define ERM_BASE                                 (0x40080000u)
/** Peripheral ERM base pointer */
#define ERM                                      ((ERM_Type *)ERM_BASE)
/** Array initializer of ERM peripheral base addresses */
#define ERM_BASE_ADDRS                           { ERM_BASE }
/** Array initializer of ERM peripheral base pointers */
#define ERM_BASE_PTRS                            { ERM }

/* ----------------------------------------------------------------------------
   -- ERM Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup ERM_Register_Masks ERM Register Masks
 * @{
 */

/* CR0 Bit Fields */
#define ERM_CR0_ENCIE6_MASK                      0x40u
#define ERM_CR0_ENCIE6_SHIFT                     6u
#define ERM_CR0_ENCIE6_WIDTH                     1u
#define ERM_CR0_ENCIE6(x)                        (((uint32_t)(((uint32_t)(x))<<ERM_CR0_ENCIE6_SHIFT))&ERM_CR0_ENCIE6_MASK)
#define ERM_CR0_ESCIE6_MASK                      0x80u
#define ERM_CR0_ESCIE6_SHIFT                     7u
#define ERM_CR0_ESCIE6_WIDTH                     1u
#define ERM_CR0_ESCIE6(x)                        (((uint32_t)(((uint32_t)(x))<<ERM_CR0_ESCIE6_SHIFT))&ERM_CR0_ESCIE6_MASK)
#define ERM_CR0_ENCIE5_MASK                      0x400u
#define ERM_CR0_ENCIE5_SHIFT                     10u
#define ERM_CR0_ENCIE5_WIDTH                     1u
#define ERM_CR0_ENCIE5(x)                        (((uint32_t)(((uint32_t)(x))<<ERM_CR0_ENCIE5_SHIFT))&ERM_CR0_ENCIE5_MASK)
#define ERM_CR0_ESCIE5_MASK                      0x800u
#define ERM_CR0_ESCIE5_SHIFT                     11u
#define ERM_CR0_ESCIE5_WIDTH                     1u
#define ERM_CR0_ESCIE5(x)                        (((uint32_t)(((uint32_t)(x))<<ERM_CR0_ESCIE5_SHIFT))&ERM_CR0_ESCIE5_MASK)
#define ERM_CR0_ENCIE4_MASK                      0x4000u
#define ERM_CR0_ENCIE4_SHIFT                     14u
#define ERM_CR0_ENCIE4_WIDTH                     1u
#define ERM_CR0_ENCIE4(x)                        (((uint32_t)(((uint32_t)(x))<<ERM_CR0_ENCIE4_SHIFT))&ERM_CR0_ENCIE4_MASK)
#define ERM_CR0_ESCIE4_MASK                      0x8000u
#define ERM_CR0_ESCIE4_SHIFT                     15u
#define ERM_CR0_ESCIE4_WIDTH                     1u
#define ERM_CR0_ESCIE4(x)                        (((uint32_t)(((uint32_t)(x))<<ERM_CR0_ESCIE4_SHIFT))&ERM_CR0_ESCIE4_MASK)
#define ERM_CR0_ENCIE3_MASK                      0x40000u
#define ERM_CR0_ENCIE3_SHIFT                     18u
#define ERM_CR0_ENCIE3_WIDTH                     1u
#define ERM_CR0_ENCIE3(x)                        (((uint32_t)(((uint32_t)(x))<<ERM_CR0_ENCIE3_SHIFT))&ERM_CR0_ENCIE3_MASK)
#define ERM_CR0_ESCIE3_MASK                      0x80000u
#define ERM_CR0_ESCIE3_SHIFT                     19u
#define ERM_CR0_ESCIE3_WIDTH                     1u
#define ERM_CR0_ESCIE3(x)                        (((uint32_t)(((uint32_t)(x))<<ERM_CR0_ESCIE3_SHIFT))&ERM_CR0_ESCIE3_MASK)
#define ERM_CR0_ENCIE2_MASK                      0x400000u
#define ERM_CR0_ENCIE2_SHIFT                     22u
#define ERM_CR0_ENCIE2_WIDTH                     1u
#define ERM_CR0_ENCIE2(x)                        (((uint32_t)(((uint32_t)(x))<<ERM_CR0_ENCIE2_SHIFT))&ERM_CR0_ENCIE2_MASK)
#define ERM_CR0_ESCIE2_MASK                      0x800000u
#define ERM_CR0_ESCIE2_SHIFT                     23u
#define ERM_CR0_ESCIE2_WIDTH                     1u
#define ERM_CR0_ESCIE2(x)                        (((uint32_t)(((uint32_t)(x))<<ERM_CR0_ESCIE2_SHIFT))&ERM_CR0_ESCIE2_MASK)
#define ERM_CR0_ENCIE1_MASK                      0x4000000u
#define ERM_CR0_ENCIE1_SHIFT                     26u
#define ERM_CR0_ENCIE1_WIDTH                     1u
#define ERM_CR0_ENCIE1(x)                        (((uint32_t)(((uint32_t)(x))<<ERM_CR0_ENCIE1_SHIFT))&ERM_CR0_ENCIE1_MASK)
#define ERM_CR0_ESCIE1_MASK                      0x8000000u
#define ERM_CR0_ESCIE1_SHIFT                     27u
#define ERM_CR0_ESCIE1_WIDTH                     1u
#define ERM_CR0_ESCIE1(x)                        (((uint32_t)(((uint32_t)(x))<<ERM_CR0_ESCIE1_SHIFT))&ERM_CR0_ESCIE1_MASK)
#define ERM_CR0_ENCIE0_MASK                      0x40000000u
#define ERM_CR0_ENCIE0_SHIFT                     30u
#define ERM_CR0_ENCIE0_WIDTH                     1u
#define ERM_CR0_ENCIE0(x)                        (((uint32_t)(((uint32_t)(x))<<ERM_CR0_ENCIE0_SHIFT))&ERM_CR0_ENCIE0_MASK)
#define ERM_CR0_ESCIE0_MASK                      0x80000000u
#define ERM_CR0_ESCIE0_SHIFT                     31u
#define ERM_CR0_ESCIE0_WIDTH                     1u
#define ERM_CR0_ESCIE0(x)                        (((uint32_t)(((uint32_t)(x))<<ERM_CR0_ESCIE0_SHIFT))&ERM_CR0_ESCIE0_MASK)
/* SR0 Bit Fields */
#define ERM_SR0_NCE6_MASK                        0x40u
#define ERM_SR0_NCE6_SHIFT                       6u
#define ERM_SR0_NCE6_WIDTH                       1u
#define ERM_SR0_NCE6(x)                          (((uint32_t)(((uint32_t)(x))<<ERM_SR0_NCE6_SHIFT))&ERM_SR0_NCE6_MASK)
#define ERM_SR0_SBC6_MASK                        0x80u
#define ERM_SR0_SBC6_SHIFT                       7u
#define ERM_SR0_SBC6_WIDTH                       1u
#define ERM_SR0_SBC6(x)                          (((uint32_t)(((uint32_t)(x))<<ERM_SR0_SBC6_SHIFT))&ERM_SR0_SBC6_MASK)
#define ERM_SR0_NCE5_MASK                        0x400u
#define ERM_SR0_NCE5_SHIFT                       10u
#define ERM_SR0_NCE5_WIDTH                       1u
#define ERM_SR0_NCE5(x)                          (((uint32_t)(((uint32_t)(x))<<ERM_SR0_NCE5_SHIFT))&ERM_SR0_NCE5_MASK)
#define ERM_SR0_SBC5_MASK                        0x800u
#define ERM_SR0_SBC5_SHIFT                       11u
#define ERM_SR0_SBC5_WIDTH                       1u
#define ERM_SR0_SBC5(x)                          (((uint32_t)(((uint32_t)(x))<<ERM_SR0_SBC5_SHIFT))&ERM_SR0_SBC5_MASK)
#define ERM_SR0_NCE4_MASK                        0x4000u
#define ERM_SR0_NCE4_SHIFT                       14u
#define ERM_SR0_NCE4_WIDTH                       1u
#define ERM_SR0_NCE4(x)                          (((uint32_t)(((uint32_t)(x))<<ERM_SR0_NCE4_SHIFT))&ERM_SR0_NCE4_MASK)
#define ERM_SR0_SBC4_MASK                        0x8000u
#define ERM_SR0_SBC4_SHIFT                       15u
#define ERM_SR0_SBC4_WIDTH                       1u
#define ERM_SR0_SBC4(x)                          (((uint32_t)(((uint32_t)(x))<<ERM_SR0_SBC4_SHIFT))&ERM_SR0_SBC4_MASK)
#define ERM_SR0_NCE3_MASK                        0x40000u
#define ERM_SR0_NCE3_SHIFT                       18u
#define ERM_SR0_NCE3_WIDTH                       1u
#define ERM_SR0_NCE3(x)                          (((uint32_t)(((uint32_t)(x))<<ERM_SR0_NCE3_SHIFT))&ERM_SR0_NCE3_MASK)
#define ERM_SR0_SBC3_MASK                        0x80000u
#define ERM_SR0_SBC3_SHIFT                       19u
#define ERM_SR0_SBC3_WIDTH                       1u
#define ERM_SR0_SBC3(x)                          (((uint32_t)(((uint32_t)(x))<<ERM_SR0_SBC3_SHIFT))&ERM_SR0_SBC3_MASK)
#define ERM_SR0_NCE2_MASK                        0x400000u
#define ERM_SR0_NCE2_SHIFT                       22u
#define ERM_SR0_NCE2_WIDTH                       1u
#define ERM_SR0_NCE2(x)                          (((uint32_t)(((uint32_t)(x))<<ERM_SR0_NCE2_SHIFT))&ERM_SR0_NCE2_MASK)
#define ERM_SR0_SBC2_MASK                        0x800000u
#define ERM_SR0_SBC2_SHIFT                       23u
#define ERM_SR0_SBC2_WIDTH                       1u
#define ERM_SR0_SBC2(x)                          (((uint32_t)(((uint32_t)(x))<<ERM_SR0_SBC2_SHIFT))&ERM_SR0_SBC2_MASK)
#define ERM_SR0_NCE1_MASK                        0x4000000u
#define ERM_SR0_NCE1_SHIFT                       26u
#define ERM_SR0_NCE1_WIDTH                       1u
#define ERM_SR0_NCE1(x)                          (((uint32_t)(((uint32_t)(x))<<ERM_SR0_NCE1_SHIFT))&ERM_SR0_NCE1_MASK)
#define ERM_SR0_SBC1_MASK                        0x8000000u
#define ERM_SR0_SBC1_SHIFT                       27u
#define ERM_SR0_SBC1_WIDTH                       1u
#define ERM_SR0_SBC1(x)                          (((uint32_t)(((uint32_t)(x))<<ERM_SR0_SBC1_SHIFT))&ERM_SR0_SBC1_MASK)
#define ERM_SR0_NCE0_MASK                        0x40000000u
#define ERM_SR0_NCE0_SHIFT                       30u
#define ERM_SR0_NCE0_WIDTH                       1u
#define ERM_SR0_NCE0(x)                          (((uint32_t)(((uint32_t)(x))<<ERM_SR0_NCE0_SHIFT))&ERM_SR0_NCE0_MASK)
#define ERM_SR0_SBC0_MASK                        0x80000000u
#define ERM_SR0_SBC0_SHIFT                       31u
#define ERM_SR0_SBC0_WIDTH                       1u
#define ERM_SR0_SBC0(x)                          (((uint32_t)(((uint32_t)(x))<<ERM_SR0_SBC0_SHIFT))&ERM_SR0_SBC0_MASK)
/* EAR Bit Fields */
#define ERM_EAR_EAR_MASK                         0xFFFFFFFFu
#define ERM_EAR_EAR_SHIFT                        0u
#define ERM_EAR_EAR_WIDTH                        32u
#define ERM_EAR_EAR(x)                           (((uint32_t)(((uint32_t)(x))<<ERM_EAR_EAR_SHIFT))&ERM_EAR_EAR_MASK)
/* SYN Bit Fields */
#define ERM_SYN_SYNDROME_MASK                    0xFF000000u
#define ERM_SYN_SYNDROME_SHIFT                   24u
#define ERM_SYN_SYNDROME_WIDTH                   8u
#define ERM_SYN_SYNDROME(x)                      (((uint32_t)(((uint32_t)(x))<<ERM_SYN_SYNDROME_SHIFT))&ERM_SYN_SYNDROME_MASK)

/*!
 * @}
 */ /* end of group ERM_Register_Masks */


/*!
 * @}
 */ /* end of group ERM_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- FCCU Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup FCCU_Peripheral_Access_Layer FCCU Peripheral Access Layer
 * @{
 */


/** FCCU - Size of Registers Arrays */
#define FCCU_NCF_CFG_COUNT                       4u
#define FCCU_NCFS_CFG_COUNT                      8u
#define FCCU_NCF_S_COUNT                         4u
#define FCCU_NCF_E_COUNT                         4u
#define FCCU_NCF_TOE_COUNT                       4u
#define FCCU_IRQ_ALARM_EN_COUNT                  4u
#define FCCU_NMI_EN_COUNT                        4u
#define FCCU_EOUT_SIG_EN_COUNT                   4u

/** FCCU - Register Layout Typedef */
typedef struct {
  __IO uint32_t CTRL;                              /**< Control, offset: 0x0 */
  __O  uint32_t CTRLK;                             /**< Control Key, offset: 0x4 */
  __IO uint32_t CFG;                               /**< Configuration, offset: 0x8 */
       uint8_t RESERVED_0[16];
  __IO uint32_t NCF_CFG[FCCU_NCF_CFG_COUNT];       /**< Noncritical Fault Configuration, array offset: 0x1C, array step: 0x4 */
       uint8_t RESERVED_1[32];
  __IO uint32_t NCFS_CFG[FCCU_NCFS_CFG_COUNT];     /**< Noncritical Fault State Configuration, array offset: 0x4C, array step: 0x4 */
       uint8_t RESERVED_2[20];
  __IO uint32_t NCF_S[FCCU_NCF_S_COUNT];           /**< Noncritical Fault Status, array offset: 0x80, array step: 0x4 */
  __O  uint32_t NCFK;                              /**< Noncritical Fault Key, offset: 0x90 */
  __IO uint32_t NCF_E[FCCU_NCF_E_COUNT];           /**< Noncritical Fault Enable, array offset: 0x94, array step: 0x4 */
  __IO uint32_t NCF_TOE[FCCU_NCF_TOE_COUNT];       /**< Noncritical Fault Timeout Enable, array offset: 0xA4, array step: 0x4 */
  __IO uint32_t NCF_TO;                            /**< Noncritical Fault Timeout, offset: 0xB4 */
  __IO uint32_t CFG_TO;                            /**< Configuration-State Timer Interval, offset: 0xB8 */
  __IO uint32_t EINOUT;                            /**< IO Control, offset: 0xBC */
  __I  uint32_t STAT;                              /**< Status, offset: 0xC0 */
  __I  uint32_t N2AF_STATUS;                       /**< NA Freeze Status, offset: 0xC4 */
  __I  uint32_t A2FF_STATUS;                       /**< AF Freeze Status, offset: 0xC8 */
  __I  uint32_t N2FF_STATUS;                       /**< NF Freeze Status, offset: 0xCC */
  __I  uint32_t F2A_STATUS;                        /**< FA Freeze Status, offset: 0xD0 */
       uint8_t RESERVED_3[8];
  __IO uint32_t NCFF;                              /**< Noncritical Fault Fake, offset: 0xDC */
  __IO uint32_t IRQ_STAT;                          /**< IRQ Status, offset: 0xE0 */
  __IO uint32_t IRQ_EN;                            /**< IRQ Enable, offset: 0xE4 */
  __I  uint32_t XTMR;                              /**< X Timer, offset: 0xE8 */
  __I  uint32_t MCS;                               /**< Mode Controller Status, offset: 0xEC */
  __IO uint32_t TRANS_LOCK;                        /**< Transient Configuration Lock, offset: 0xF0 */
  __IO uint32_t PERMNT_LOCK;                       /**< Permanent Configuration Lock, offset: 0xF4 */
  __IO uint32_t DELTA_T;                           /**< Delta T, offset: 0xF8 */
  __IO uint32_t IRQ_ALARM_EN[FCCU_IRQ_ALARM_EN_COUNT]; /**< IRQ Alarm Enable, array offset: 0xFC, array step: 0x4 */
  __IO uint32_t NMI_EN[FCCU_NMI_EN_COUNT];         /**< NMI Enable, array offset: 0x10C, array step: 0x4 */
  __IO uint32_t EOUT_SIG_EN[FCCU_EOUT_SIG_EN_COUNT]; /**< Noncritical Fault-State EOUT Signaling Enable, array offset: 0x11C, array step: 0x4 */
} FCCU_Type, *FCCU_MemMapPtr;

 /** Number of instances of the FCCU module. */
#define FCCU_INSTANCE_COUNT                      (1u)


/* FCCU - Peripheral instance base addresses */
/** Peripheral FCCU base address */
#define FCCU_BASE                                (0x400B0000u)
/** Peripheral FCCU base pointer */
#define FCCU                                     ((FCCU_Type *)FCCU_BASE)
/** Array initializer of FCCU peripheral base addresses */
#define FCCU_BASE_ADDRS                          { FCCU_BASE }
/** Array initializer of FCCU peripheral base pointers */
#define FCCU_BASE_PTRS                           { FCCU }

/* ----------------------------------------------------------------------------
   -- FCCU Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup FCCU_Register_Masks FCCU Register Masks
 * @{
 */

/* CTRL Bit Fields */
#define FCCU_CTRL_OPR_MASK                       0x1Fu
#define FCCU_CTRL_OPR_SHIFT                      0u
#define FCCU_CTRL_OPR_WIDTH                      5u
#define FCCU_CTRL_OPR(x)                         (((uint32_t)(((uint32_t)(x))<<FCCU_CTRL_OPR_SHIFT))&FCCU_CTRL_OPR_MASK)
#define FCCU_CTRL_OPS_MASK                       0xC0u
#define FCCU_CTRL_OPS_SHIFT                      6u
#define FCCU_CTRL_OPS_WIDTH                      2u
#define FCCU_CTRL_OPS(x)                         (((uint32_t)(((uint32_t)(x))<<FCCU_CTRL_OPS_SHIFT))&FCCU_CTRL_OPS_MASK)
#define FCCU_CTRL_DEBUG_MASK                     0x200u
#define FCCU_CTRL_DEBUG_SHIFT                    9u
#define FCCU_CTRL_DEBUG_WIDTH                    1u
#define FCCU_CTRL_DEBUG(x)                       (((uint32_t)(((uint32_t)(x))<<FCCU_CTRL_DEBUG_SHIFT))&FCCU_CTRL_DEBUG_MASK)
#define FCCU_CTRL_FILTER_WIDTH_MASK              0x60000000u
#define FCCU_CTRL_FILTER_WIDTH_SHIFT             29u
#define FCCU_CTRL_FILTER_WIDTH_WIDTH             2u
#define FCCU_CTRL_FILTER_WIDTH(x)                (((uint32_t)(((uint32_t)(x))<<FCCU_CTRL_FILTER_WIDTH_SHIFT))&FCCU_CTRL_FILTER_WIDTH_MASK)
#define FCCU_CTRL_FILTER_BYPASS_MASK             0x80000000u
#define FCCU_CTRL_FILTER_BYPASS_SHIFT            31u
#define FCCU_CTRL_FILTER_BYPASS_WIDTH            1u
#define FCCU_CTRL_FILTER_BYPASS(x)               (((uint32_t)(((uint32_t)(x))<<FCCU_CTRL_FILTER_BYPASS_SHIFT))&FCCU_CTRL_FILTER_BYPASS_MASK)
/* CTRLK Bit Fields */
#define FCCU_CTRLK_CTRLK_MASK                    0xFFFFFFFFu
#define FCCU_CTRLK_CTRLK_SHIFT                   0u
#define FCCU_CTRLK_CTRLK_WIDTH                   32u
#define FCCU_CTRLK_CTRLK(x)                      (((uint32_t)(((uint32_t)(x))<<FCCU_CTRLK_CTRLK_SHIFT))&FCCU_CTRLK_CTRLK_MASK)
/* CFG Bit Fields */
#define FCCU_CFG_FOP_MASK                        0x3Fu
#define FCCU_CFG_FOP_SHIFT                       0u
#define FCCU_CFG_FOP_WIDTH                       6u
#define FCCU_CFG_FOP(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_CFG_FOP_SHIFT))&FCCU_CFG_FOP_MASK)
#define FCCU_CFG_FOM_MASK                        0x1C0u
#define FCCU_CFG_FOM_SHIFT                       6u
#define FCCU_CFG_FOM_WIDTH                       3u
#define FCCU_CFG_FOM(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_CFG_FOM_SHIFT))&FCCU_CFG_FOM_MASK)
#define FCCU_CFG_PS_MASK                         0x200u
#define FCCU_CFG_PS_SHIFT                        9u
#define FCCU_CFG_PS_WIDTH                        1u
#define FCCU_CFG_PS(x)                           (((uint32_t)(((uint32_t)(x))<<FCCU_CFG_PS_SHIFT))&FCCU_CFG_PS_MASK)
#define FCCU_CFG_SM_MASK                         0x400u
#define FCCU_CFG_SM_SHIFT                        10u
#define FCCU_CFG_SM_WIDTH                        1u
#define FCCU_CFG_SM(x)                           (((uint32_t)(((uint32_t)(x))<<FCCU_CFG_SM_SHIFT))&FCCU_CFG_SM_MASK)
#define FCCU_CFG_OD_MASK                         0x1000u
#define FCCU_CFG_OD_SHIFT                        12u
#define FCCU_CFG_OD_WIDTH                        1u
#define FCCU_CFG_OD(x)                           (((uint32_t)(((uint32_t)(x))<<FCCU_CFG_OD_SHIFT))&FCCU_CFG_OD_MASK)
#define FCCU_CFG_FOPE_MASK                       0x8000u
#define FCCU_CFG_FOPE_SHIFT                      15u
#define FCCU_CFG_FOPE_WIDTH                      1u
#define FCCU_CFG_FOPE(x)                         (((uint32_t)(((uint32_t)(x))<<FCCU_CFG_FOPE_SHIFT))&FCCU_CFG_FOPE_MASK)
#define FCCU_CFG_FCCU_SET_CLEAR_MASK             0xC00000u
#define FCCU_CFG_FCCU_SET_CLEAR_SHIFT            22u
#define FCCU_CFG_FCCU_SET_CLEAR_WIDTH            2u
#define FCCU_CFG_FCCU_SET_CLEAR(x)               (((uint32_t)(((uint32_t)(x))<<FCCU_CFG_FCCU_SET_CLEAR_SHIFT))&FCCU_CFG_FCCU_SET_CLEAR_MASK)
#define FCCU_CFG_FCCU_SET_AFTER_RESET_MASK       0x1000000u
#define FCCU_CFG_FCCU_SET_AFTER_RESET_SHIFT      24u
#define FCCU_CFG_FCCU_SET_AFTER_RESET_WIDTH      1u
#define FCCU_CFG_FCCU_SET_AFTER_RESET(x)         (((uint32_t)(((uint32_t)(x))<<FCCU_CFG_FCCU_SET_AFTER_RESET_SHIFT))&FCCU_CFG_FCCU_SET_AFTER_RESET_MASK)
/* NCF_CFG Bit Fields */
#define FCCU_NCF_CFG_NCFCx_MASK                  0xFFFFFFFFu
#define FCCU_NCF_CFG_NCFCx_SHIFT                 0u
#define FCCU_NCF_CFG_NCFCx_WIDTH                 32u
#define FCCU_NCF_CFG_NCFCx(x)                    (((uint32_t)(((uint32_t)(x))<<FCCU_NCF_CFG_NCFCx_SHIFT))&FCCU_NCF_CFG_NCFCx_MASK)
/* NCFS_CFG Bit Fields */
#define FCCU_NCFS_CFG_NCFSCx_MASK                0xFFFFFFFFu
#define FCCU_NCFS_CFG_NCFSCx_SHIFT               0u
#define FCCU_NCFS_CFG_NCFSCx_WIDTH               32u
#define FCCU_NCFS_CFG_NCFSCx(x)                  (((uint32_t)(((uint32_t)(x))<<FCCU_NCFS_CFG_NCFSCx_SHIFT))&FCCU_NCFS_CFG_NCFSCx_MASK)
/* NCF_S Bit Fields */
#define FCCU_NCF_S_NCFSx_MASK                    0xFFFFFFFFu
#define FCCU_NCF_S_NCFSx_SHIFT                   0u
#define FCCU_NCF_S_NCFSx_WIDTH                   32u
#define FCCU_NCF_S_NCFSx(x)                      (((uint32_t)(((uint32_t)(x))<<FCCU_NCF_S_NCFSx_SHIFT))&FCCU_NCF_S_NCFSx_MASK)
/* NCFK Bit Fields */
#define FCCU_NCFK_NCFK_MASK                      0xFFFFFFFFu
#define FCCU_NCFK_NCFK_SHIFT                     0u
#define FCCU_NCFK_NCFK_WIDTH                     32u
#define FCCU_NCFK_NCFK(x)                        (((uint32_t)(((uint32_t)(x))<<FCCU_NCFK_NCFK_SHIFT))&FCCU_NCFK_NCFK_MASK)
/* NCF_E Bit Fields */
#define FCCU_NCF_E_NCFEx_MASK                    0xFFFFFFFFu
#define FCCU_NCF_E_NCFEx_SHIFT                   0u
#define FCCU_NCF_E_NCFEx_WIDTH                   32u
#define FCCU_NCF_E_NCFEx(x)                      (((uint32_t)(((uint32_t)(x))<<FCCU_NCF_E_NCFEx_SHIFT))&FCCU_NCF_E_NCFEx_MASK)
/* NCF_TOE Bit Fields */
#define FCCU_NCF_TOE_NCFTOEx_MASK                0xFFFFFFFFu
#define FCCU_NCF_TOE_NCFTOEx_SHIFT               0u
#define FCCU_NCF_TOE_NCFTOEx_WIDTH               32u
#define FCCU_NCF_TOE_NCFTOEx(x)                  (((uint32_t)(((uint32_t)(x))<<FCCU_NCF_TOE_NCFTOEx_SHIFT))&FCCU_NCF_TOE_NCFTOEx_MASK)
/* NCF_TO Bit Fields */
#define FCCU_NCF_TO_TO_MASK                      0xFFFFFFFFu
#define FCCU_NCF_TO_TO_SHIFT                     0u
#define FCCU_NCF_TO_TO_WIDTH                     32u
#define FCCU_NCF_TO_TO(x)                        (((uint32_t)(((uint32_t)(x))<<FCCU_NCF_TO_TO_SHIFT))&FCCU_NCF_TO_TO_MASK)
/* CFG_TO Bit Fields */
#define FCCU_CFG_TO_TO_MASK                      0x7u
#define FCCU_CFG_TO_TO_SHIFT                     0u
#define FCCU_CFG_TO_TO_WIDTH                     3u
#define FCCU_CFG_TO_TO(x)                        (((uint32_t)(((uint32_t)(x))<<FCCU_CFG_TO_TO_SHIFT))&FCCU_CFG_TO_TO_MASK)
/* EINOUT Bit Fields */
#define FCCU_EINOUT_EOUT0_MASK                   0x1u
#define FCCU_EINOUT_EOUT0_SHIFT                  0u
#define FCCU_EINOUT_EOUT0_WIDTH                  1u
#define FCCU_EINOUT_EOUT0(x)                     (((uint32_t)(((uint32_t)(x))<<FCCU_EINOUT_EOUT0_SHIFT))&FCCU_EINOUT_EOUT0_MASK)
#define FCCU_EINOUT_EOUT1_MASK                   0x2u
#define FCCU_EINOUT_EOUT1_SHIFT                  1u
#define FCCU_EINOUT_EOUT1_WIDTH                  1u
#define FCCU_EINOUT_EOUT1(x)                     (((uint32_t)(((uint32_t)(x))<<FCCU_EINOUT_EOUT1_SHIFT))&FCCU_EINOUT_EOUT1_MASK)
#define FCCU_EINOUT_EIN0_MASK                    0x10u
#define FCCU_EINOUT_EIN0_SHIFT                   4u
#define FCCU_EINOUT_EIN0_WIDTH                   1u
#define FCCU_EINOUT_EIN0(x)                      (((uint32_t)(((uint32_t)(x))<<FCCU_EINOUT_EIN0_SHIFT))&FCCU_EINOUT_EIN0_MASK)
#define FCCU_EINOUT_EIN1_MASK                    0x20u
#define FCCU_EINOUT_EIN1_SHIFT                   5u
#define FCCU_EINOUT_EIN1_WIDTH                   1u
#define FCCU_EINOUT_EIN1(x)                      (((uint32_t)(((uint32_t)(x))<<FCCU_EINOUT_EIN1_SHIFT))&FCCU_EINOUT_EIN1_MASK)
/* STAT Bit Fields */
#define FCCU_STAT_STATUS_MASK                    0x7u
#define FCCU_STAT_STATUS_SHIFT                   0u
#define FCCU_STAT_STATUS_WIDTH                   3u
#define FCCU_STAT_STATUS(x)                      (((uint32_t)(((uint32_t)(x))<<FCCU_STAT_STATUS_SHIFT))&FCCU_STAT_STATUS_MASK)
#define FCCU_STAT_ESTAT_MASK                     0x8u
#define FCCU_STAT_ESTAT_SHIFT                    3u
#define FCCU_STAT_ESTAT_WIDTH                    1u
#define FCCU_STAT_ESTAT(x)                       (((uint32_t)(((uint32_t)(x))<<FCCU_STAT_ESTAT_SHIFT))&FCCU_STAT_ESTAT_MASK)
#define FCCU_STAT_PhysicErrorPin_MASK            0x30u
#define FCCU_STAT_PhysicErrorPin_SHIFT           4u
#define FCCU_STAT_PhysicErrorPin_WIDTH           2u
#define FCCU_STAT_PhysicErrorPin(x)              (((uint32_t)(((uint32_t)(x))<<FCCU_STAT_PhysicErrorPin_SHIFT))&FCCU_STAT_PhysicErrorPin_MASK)
/* N2AF_STATUS Bit Fields */
#define FCCU_N2AF_STATUS_NAFS_MASK               0xFFu
#define FCCU_N2AF_STATUS_NAFS_SHIFT              0u
#define FCCU_N2AF_STATUS_NAFS_WIDTH              8u
#define FCCU_N2AF_STATUS_NAFS(x)                 (((uint32_t)(((uint32_t)(x))<<FCCU_N2AF_STATUS_NAFS_SHIFT))&FCCU_N2AF_STATUS_NAFS_MASK)
/* A2FF_STATUS Bit Fields */
#define FCCU_A2FF_STATUS_AFFS_MASK               0xFFu
#define FCCU_A2FF_STATUS_AFFS_SHIFT              0u
#define FCCU_A2FF_STATUS_AFFS_WIDTH              8u
#define FCCU_A2FF_STATUS_AFFS(x)                 (((uint32_t)(((uint32_t)(x))<<FCCU_A2FF_STATUS_AFFS_SHIFT))&FCCU_A2FF_STATUS_AFFS_MASK)
#define FCCU_A2FF_STATUS_AF_SRC_MASK             0x300u
#define FCCU_A2FF_STATUS_AF_SRC_SHIFT            8u
#define FCCU_A2FF_STATUS_AF_SRC_WIDTH            2u
#define FCCU_A2FF_STATUS_AF_SRC(x)               (((uint32_t)(((uint32_t)(x))<<FCCU_A2FF_STATUS_AF_SRC_SHIFT))&FCCU_A2FF_STATUS_AF_SRC_MASK)
/* N2FF_STATUS Bit Fields */
#define FCCU_N2FF_STATUS_NFFS_MASK               0xFFu
#define FCCU_N2FF_STATUS_NFFS_SHIFT              0u
#define FCCU_N2FF_STATUS_NFFS_WIDTH              8u
#define FCCU_N2FF_STATUS_NFFS(x)                 (((uint32_t)(((uint32_t)(x))<<FCCU_N2FF_STATUS_NFFS_SHIFT))&FCCU_N2FF_STATUS_NFFS_MASK)
#define FCCU_N2FF_STATUS_NF_SRC_MASK             0x300u
#define FCCU_N2FF_STATUS_NF_SRC_SHIFT            8u
#define FCCU_N2FF_STATUS_NF_SRC_WIDTH            2u
#define FCCU_N2FF_STATUS_NF_SRC(x)               (((uint32_t)(((uint32_t)(x))<<FCCU_N2FF_STATUS_NF_SRC_SHIFT))&FCCU_N2FF_STATUS_NF_SRC_MASK)
/* F2A_STATUS Bit Fields */
#define FCCU_F2A_STATUS_FAFS_MASK                0x1FFu
#define FCCU_F2A_STATUS_FAFS_SHIFT               0u
#define FCCU_F2A_STATUS_FAFS_WIDTH               9u
#define FCCU_F2A_STATUS_FAFS(x)                  (((uint32_t)(((uint32_t)(x))<<FCCU_F2A_STATUS_FAFS_SHIFT))&FCCU_F2A_STATUS_FAFS_MASK)
/* NCFF Bit Fields */
#define FCCU_NCFF_FNCFC_MASK                     0x7Fu
#define FCCU_NCFF_FNCFC_SHIFT                    0u
#define FCCU_NCFF_FNCFC_WIDTH                    7u
#define FCCU_NCFF_FNCFC(x)                       (((uint32_t)(((uint32_t)(x))<<FCCU_NCFF_FNCFC_SHIFT))&FCCU_NCFF_FNCFC_MASK)
/* IRQ_STAT Bit Fields */
#define FCCU_IRQ_STAT_CFG_TO_STAT_MASK           0x1u
#define FCCU_IRQ_STAT_CFG_TO_STAT_SHIFT          0u
#define FCCU_IRQ_STAT_CFG_TO_STAT_WIDTH          1u
#define FCCU_IRQ_STAT_CFG_TO_STAT(x)             (((uint32_t)(((uint32_t)(x))<<FCCU_IRQ_STAT_CFG_TO_STAT_SHIFT))&FCCU_IRQ_STAT_CFG_TO_STAT_MASK)
#define FCCU_IRQ_STAT_ALRM_STAT_MASK             0x2u
#define FCCU_IRQ_STAT_ALRM_STAT_SHIFT            1u
#define FCCU_IRQ_STAT_ALRM_STAT_WIDTH            1u
#define FCCU_IRQ_STAT_ALRM_STAT(x)               (((uint32_t)(((uint32_t)(x))<<FCCU_IRQ_STAT_ALRM_STAT_SHIFT))&FCCU_IRQ_STAT_ALRM_STAT_MASK)
#define FCCU_IRQ_STAT_NMI_STAT_MASK              0x4u
#define FCCU_IRQ_STAT_NMI_STAT_SHIFT             2u
#define FCCU_IRQ_STAT_NMI_STAT_WIDTH             1u
#define FCCU_IRQ_STAT_NMI_STAT(x)                (((uint32_t)(((uint32_t)(x))<<FCCU_IRQ_STAT_NMI_STAT_SHIFT))&FCCU_IRQ_STAT_NMI_STAT_MASK)
/* IRQ_EN Bit Fields */
#define FCCU_IRQ_EN_CFG_TO_IEN_MASK              0x1u
#define FCCU_IRQ_EN_CFG_TO_IEN_SHIFT             0u
#define FCCU_IRQ_EN_CFG_TO_IEN_WIDTH             1u
#define FCCU_IRQ_EN_CFG_TO_IEN(x)                (((uint32_t)(((uint32_t)(x))<<FCCU_IRQ_EN_CFG_TO_IEN_SHIFT))&FCCU_IRQ_EN_CFG_TO_IEN_MASK)
/* XTMR Bit Fields */
#define FCCU_XTMR_XTMR_MASK                      0xFFFFFFFFu
#define FCCU_XTMR_XTMR_SHIFT                     0u
#define FCCU_XTMR_XTMR_WIDTH                     32u
#define FCCU_XTMR_XTMR(x)                        (((uint32_t)(((uint32_t)(x))<<FCCU_XTMR_XTMR_SHIFT))&FCCU_XTMR_XTMR_MASK)
/* MCS Bit Fields */
#define FCCU_MCS_MCS0_MASK                       0xFu
#define FCCU_MCS_MCS0_SHIFT                      0u
#define FCCU_MCS_MCS0_WIDTH                      4u
#define FCCU_MCS_MCS0(x)                         (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_MCS0_SHIFT))&FCCU_MCS_MCS0_MASK)
#define FCCU_MCS_FS0_MASK                        0x40u
#define FCCU_MCS_FS0_SHIFT                       6u
#define FCCU_MCS_FS0_WIDTH                       1u
#define FCCU_MCS_FS0(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_FS0_SHIFT))&FCCU_MCS_FS0_MASK)
#define FCCU_MCS_VL0_MASK                        0x80u
#define FCCU_MCS_VL0_SHIFT                       7u
#define FCCU_MCS_VL0_WIDTH                       1u
#define FCCU_MCS_VL0(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_VL0_SHIFT))&FCCU_MCS_VL0_MASK)
#define FCCU_MCS_MCS1_MASK                       0xF00u
#define FCCU_MCS_MCS1_SHIFT                      8u
#define FCCU_MCS_MCS1_WIDTH                      4u
#define FCCU_MCS_MCS1(x)                         (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_MCS1_SHIFT))&FCCU_MCS_MCS1_MASK)
#define FCCU_MCS_FS1_MASK                        0x4000u
#define FCCU_MCS_FS1_SHIFT                       14u
#define FCCU_MCS_FS1_WIDTH                       1u
#define FCCU_MCS_FS1(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_FS1_SHIFT))&FCCU_MCS_FS1_MASK)
#define FCCU_MCS_VL1_MASK                        0x8000u
#define FCCU_MCS_VL1_SHIFT                       15u
#define FCCU_MCS_VL1_WIDTH                       1u
#define FCCU_MCS_VL1(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_VL1_SHIFT))&FCCU_MCS_VL1_MASK)
#define FCCU_MCS_MCS2_MASK                       0xF0000u
#define FCCU_MCS_MCS2_SHIFT                      16u
#define FCCU_MCS_MCS2_WIDTH                      4u
#define FCCU_MCS_MCS2(x)                         (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_MCS2_SHIFT))&FCCU_MCS_MCS2_MASK)
#define FCCU_MCS_FS2_MASK                        0x400000u
#define FCCU_MCS_FS2_SHIFT                       22u
#define FCCU_MCS_FS2_WIDTH                       1u
#define FCCU_MCS_FS2(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_FS2_SHIFT))&FCCU_MCS_FS2_MASK)
#define FCCU_MCS_VL2_MASK                        0x800000u
#define FCCU_MCS_VL2_SHIFT                       23u
#define FCCU_MCS_VL2_WIDTH                       1u
#define FCCU_MCS_VL2(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_VL2_SHIFT))&FCCU_MCS_VL2_MASK)
#define FCCU_MCS_MCS3_MASK                       0xF000000u
#define FCCU_MCS_MCS3_SHIFT                      24u
#define FCCU_MCS_MCS3_WIDTH                      4u
#define FCCU_MCS_MCS3(x)                         (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_MCS3_SHIFT))&FCCU_MCS_MCS3_MASK)
#define FCCU_MCS_FS3_MASK                        0x40000000u
#define FCCU_MCS_FS3_SHIFT                       30u
#define FCCU_MCS_FS3_WIDTH                       1u
#define FCCU_MCS_FS3(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_FS3_SHIFT))&FCCU_MCS_FS3_MASK)
#define FCCU_MCS_VL3_MASK                        0x80000000u
#define FCCU_MCS_VL3_SHIFT                       31u
#define FCCU_MCS_VL3_WIDTH                       1u
#define FCCU_MCS_VL3(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_VL3_SHIFT))&FCCU_MCS_VL3_MASK)
/* TRANS_LOCK Bit Fields */
#define FCCU_TRANS_LOCK_TRANSKEY_MASK            0x1FFu
#define FCCU_TRANS_LOCK_TRANSKEY_SHIFT           0u
#define FCCU_TRANS_LOCK_TRANSKEY_WIDTH           9u
#define FCCU_TRANS_LOCK_TRANSKEY(x)              (((uint32_t)(((uint32_t)(x))<<FCCU_TRANS_LOCK_TRANSKEY_SHIFT))&FCCU_TRANS_LOCK_TRANSKEY_MASK)
/* PERMNT_LOCK Bit Fields */
#define FCCU_PERMNT_LOCK_PERMNTKEY_MASK          0x1FFu
#define FCCU_PERMNT_LOCK_PERMNTKEY_SHIFT         0u
#define FCCU_PERMNT_LOCK_PERMNTKEY_WIDTH         9u
#define FCCU_PERMNT_LOCK_PERMNTKEY(x)            (((uint32_t)(((uint32_t)(x))<<FCCU_PERMNT_LOCK_PERMNTKEY_SHIFT))&FCCU_PERMNT_LOCK_PERMNTKEY_MASK)
/* DELTA_T Bit Fields */
#define FCCU_DELTA_T_DELTA_T_MASK                0x3FFFu
#define FCCU_DELTA_T_DELTA_T_SHIFT               0u
#define FCCU_DELTA_T_DELTA_T_WIDTH               14u
#define FCCU_DELTA_T_DELTA_T(x)                  (((uint32_t)(((uint32_t)(x))<<FCCU_DELTA_T_DELTA_T_SHIFT))&FCCU_DELTA_T_DELTA_T_MASK)
/* IRQ_ALARM_EN Bit Fields */
#define FCCU_IRQ_ALARM_EN_IRQENx_MASK            0xFFFFFFFFu
#define FCCU_IRQ_ALARM_EN_IRQENx_SHIFT           0u
#define FCCU_IRQ_ALARM_EN_IRQENx_WIDTH           32u
#define FCCU_IRQ_ALARM_EN_IRQENx(x)              (((uint32_t)(((uint32_t)(x))<<FCCU_IRQ_ALARM_EN_IRQENx_SHIFT))&FCCU_IRQ_ALARM_EN_IRQENx_MASK)
/* NMI_EN Bit Fields */
#define FCCU_NMI_EN_NMIENx_MASK                  0xFFFFFFFFu
#define FCCU_NMI_EN_NMIENx_SHIFT                 0u
#define FCCU_NMI_EN_NMIENx_WIDTH                 32u
#define FCCU_NMI_EN_NMIENx(x)                    (((uint32_t)(((uint32_t)(x))<<FCCU_NMI_EN_NMIENx_SHIFT))&FCCU_NMI_EN_NMIENx_MASK)
/* EOUT_SIG_EN Bit Fields */
#define FCCU_EOUT_SIG_EN_EOUTENx_MASK            0xFFFFFFFFu
#define FCCU_EOUT_SIG_EN_EOUTENx_SHIFT           0u
#define FCCU_EOUT_SIG_EN_EOUTENx_WIDTH           32u
#define FCCU_EOUT_SIG_EN_EOUTENx(x)              (((uint32_t)(((uint32_t)(x))<<FCCU_EOUT_SIG_EN_EOUTENx_SHIFT))&FCCU_EOUT_SIG_EN_EOUTENx_MASK)

/*!
 * @}
 */ /* end of group FCCU_Register_Masks */


/*!
 * @}
 */ /* end of group FCCU_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- FDMA Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup FDMA_Peripheral_Access_Layer FDMA Peripheral Access Layer
 * @{
 */


/** FDMA - Size of Registers Arrays */

/** FDMA - Register Layout Typedef */
typedef struct {
  __IO uint32_t XFR_REC_LIST_PTR;                  /**< Transfer Records List Pointer register, offset: 0x0 */
  __IO uint32_t XFR_REC_CNT;                       /**< Total entries in Transfer Record List, offset: 0x4 */
  __IO uint32_t XFR_REC_NUM;                       /**< Transfer Record number for current Line transfer, offset: 0x8 */
  __IO uint32_t XFR_LINE_NUM;                      /**< DDR and SRAM Line numbers for current transfer, offset: 0xC */
  __IO uint32_t LINE_INCR;                         /**< Line increment value for SRAM and DDR, offset: 0x10 */
  __IO uint32_t IRQ_EN;                            /**< Interrupt enable register, offset: 0x14 */
  __IO uint32_t XFR_STAT;                          /**< Status register, offset: 0x18 */
  __I  uint32_t CALC_CRC_VAL;                      /**< Calculated CRC value, offset: 0x1C */
  __I  uint32_t CURR_DDR_PTR;                      /**< Current DDR address, offset: 0x20 */
  __I  uint32_t CURR_SRAM_PTR;                     /**< Current SRAM address, offset: 0x24 */
  __I  uint32_t XFR_REC_NUM_DONE;                  /**< Last completed Transfer Record Number, offset: 0x28 */
  __I  uint32_t ERR_XFR_REC_NUM;                   /**< Transfer Record Number of an Erroneous Transfer, offset: 0x2C */
  __I  uint32_t NEXT_LINE;                         /**< SRAM and DDR next Line number, offset: 0x30 */
  __IO uint32_t CTRL;                              /**< Control register, offset: 0x34 */
} FDMA_Type, *FDMA_MemMapPtr;

 /** Number of instances of the FDMA module. */
#define FDMA_INSTANCE_COUNT                      (1u)


/* FDMA - Peripheral instance base addresses */
/** Peripheral FDMA base address */
#define FDMA_BASE                                (0x7C040000u)
/** Peripheral FDMA base pointer */
#define FDMA                                     ((FDMA_Type *)FDMA_BASE)
/** Array initializer of FDMA peripheral base addresses */
#define FDMA_BASE_ADDRS                          { FDMA_BASE }
/** Array initializer of FDMA peripheral base pointers */
#define FDMA_BASE_PTRS                           { FDMA }

/* ----------------------------------------------------------------------------
   -- FDMA Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup FDMA_Register_Masks FDMA Register Masks
 * @{
 */

/* XFR_REC_LIST_PTR Bit Fields */
#define FDMA_XFR_REC_LIST_PTR_XFR_REC_LIST_PTR_MASK 0xFFFFFFFFu
#define FDMA_XFR_REC_LIST_PTR_XFR_REC_LIST_PTR_SHIFT 0u
#define FDMA_XFR_REC_LIST_PTR_XFR_REC_LIST_PTR_WIDTH 32u
#define FDMA_XFR_REC_LIST_PTR_XFR_REC_LIST_PTR(x) (((uint32_t)(((uint32_t)(x))<<FDMA_XFR_REC_LIST_PTR_XFR_REC_LIST_PTR_SHIFT))&FDMA_XFR_REC_LIST_PTR_XFR_REC_LIST_PTR_MASK)
/* XFR_REC_CNT Bit Fields */
#define FDMA_XFR_REC_CNT_XFR_REC_CNT_MASK        0x7Fu
#define FDMA_XFR_REC_CNT_XFR_REC_CNT_SHIFT       0u
#define FDMA_XFR_REC_CNT_XFR_REC_CNT_WIDTH       7u
#define FDMA_XFR_REC_CNT_XFR_REC_CNT(x)          (((uint32_t)(((uint32_t)(x))<<FDMA_XFR_REC_CNT_XFR_REC_CNT_SHIFT))&FDMA_XFR_REC_CNT_XFR_REC_CNT_MASK)
/* XFR_REC_NUM Bit Fields */
#define FDMA_XFR_REC_NUM_XFR_REC_NUM_MASK        0x7Fu
#define FDMA_XFR_REC_NUM_XFR_REC_NUM_SHIFT       0u
#define FDMA_XFR_REC_NUM_XFR_REC_NUM_WIDTH       7u
#define FDMA_XFR_REC_NUM_XFR_REC_NUM(x)          (((uint32_t)(((uint32_t)(x))<<FDMA_XFR_REC_NUM_XFR_REC_NUM_SHIFT))&FDMA_XFR_REC_NUM_XFR_REC_NUM_MASK)
/* XFR_LINE_NUM Bit Fields */
#define FDMA_XFR_LINE_NUM_DDR_IMG_LINE_NUM_MASK  0xFFFFu
#define FDMA_XFR_LINE_NUM_DDR_IMG_LINE_NUM_SHIFT 0u
#define FDMA_XFR_LINE_NUM_DDR_IMG_LINE_NUM_WIDTH 16u
#define FDMA_XFR_LINE_NUM_DDR_IMG_LINE_NUM(x)    (((uint32_t)(((uint32_t)(x))<<FDMA_XFR_LINE_NUM_DDR_IMG_LINE_NUM_SHIFT))&FDMA_XFR_LINE_NUM_DDR_IMG_LINE_NUM_MASK)
#define FDMA_XFR_LINE_NUM_SRAM_LINE_NUM_MASK     0x3FFF0000u
#define FDMA_XFR_LINE_NUM_SRAM_LINE_NUM_SHIFT    16u
#define FDMA_XFR_LINE_NUM_SRAM_LINE_NUM_WIDTH    14u
#define FDMA_XFR_LINE_NUM_SRAM_LINE_NUM(x)       (((uint32_t)(((uint32_t)(x))<<FDMA_XFR_LINE_NUM_SRAM_LINE_NUM_SHIFT))&FDMA_XFR_LINE_NUM_SRAM_LINE_NUM_MASK)
/* LINE_INCR Bit Fields */
#define FDMA_LINE_INCR_DDR_LINE_INCR_MASK        0xFFu
#define FDMA_LINE_INCR_DDR_LINE_INCR_SHIFT       0u
#define FDMA_LINE_INCR_DDR_LINE_INCR_WIDTH       8u
#define FDMA_LINE_INCR_DDR_LINE_INCR(x)          (((uint32_t)(((uint32_t)(x))<<FDMA_LINE_INCR_DDR_LINE_INCR_SHIFT))&FDMA_LINE_INCR_DDR_LINE_INCR_MASK)
#define FDMA_LINE_INCR_SRAM_LINE_INCR_MASK       0xFF0000u
#define FDMA_LINE_INCR_SRAM_LINE_INCR_SHIFT      16u
#define FDMA_LINE_INCR_SRAM_LINE_INCR_WIDTH      8u
#define FDMA_LINE_INCR_SRAM_LINE_INCR(x)         (((uint32_t)(((uint32_t)(x))<<FDMA_LINE_INCR_SRAM_LINE_INCR_SHIFT))&FDMA_LINE_INCR_SRAM_LINE_INCR_MASK)
/* IRQ_EN Bit Fields */
#define FDMA_IRQ_EN_FDMA_DONE_IRQ_EN_MASK        0x1u
#define FDMA_IRQ_EN_FDMA_DONE_IRQ_EN_SHIFT       0u
#define FDMA_IRQ_EN_FDMA_DONE_IRQ_EN_WIDTH       1u
#define FDMA_IRQ_EN_FDMA_DONE_IRQ_EN(x)          (((uint32_t)(((uint32_t)(x))<<FDMA_IRQ_EN_FDMA_DONE_IRQ_EN_SHIFT))&FDMA_IRQ_EN_FDMA_DONE_IRQ_EN_MASK)
#define FDMA_IRQ_EN_FDMA_ERR_IRQ_EN_MASK         0x2u
#define FDMA_IRQ_EN_FDMA_ERR_IRQ_EN_SHIFT        1u
#define FDMA_IRQ_EN_FDMA_ERR_IRQ_EN_WIDTH        1u
#define FDMA_IRQ_EN_FDMA_ERR_IRQ_EN(x)           (((uint32_t)(((uint32_t)(x))<<FDMA_IRQ_EN_FDMA_ERR_IRQ_EN_SHIFT))&FDMA_IRQ_EN_FDMA_ERR_IRQ_EN_MASK)
#define FDMA_IRQ_EN_FDMA_CRC_ERR_IRQ_EN_MASK     0x4u
#define FDMA_IRQ_EN_FDMA_CRC_ERR_IRQ_EN_SHIFT    2u
#define FDMA_IRQ_EN_FDMA_CRC_ERR_IRQ_EN_WIDTH    1u
#define FDMA_IRQ_EN_FDMA_CRC_ERR_IRQ_EN(x)       (((uint32_t)(((uint32_t)(x))<<FDMA_IRQ_EN_FDMA_CRC_ERR_IRQ_EN_SHIFT))&FDMA_IRQ_EN_FDMA_CRC_ERR_IRQ_EN_MASK)
#define FDMA_IRQ_EN_EDMA_TRIG_EN_MASK            0x8u
#define FDMA_IRQ_EN_EDMA_TRIG_EN_SHIFT           3u
#define FDMA_IRQ_EN_EDMA_TRIG_EN_WIDTH           1u
#define FDMA_IRQ_EN_EDMA_TRIG_EN(x)              (((uint32_t)(((uint32_t)(x))<<FDMA_IRQ_EN_EDMA_TRIG_EN_SHIFT))&FDMA_IRQ_EN_EDMA_TRIG_EN_MASK)
/* XFR_STAT Bit Fields */
#define FDMA_XFR_STAT_XFR_CMD_QUEUE_EMPTY_MASK   0x1u
#define FDMA_XFR_STAT_XFR_CMD_QUEUE_EMPTY_SHIFT  0u
#define FDMA_XFR_STAT_XFR_CMD_QUEUE_EMPTY_WIDTH  1u
#define FDMA_XFR_STAT_XFR_CMD_QUEUE_EMPTY(x)     (((uint32_t)(((uint32_t)(x))<<FDMA_XFR_STAT_XFR_CMD_QUEUE_EMPTY_SHIFT))&FDMA_XFR_STAT_XFR_CMD_QUEUE_EMPTY_MASK)
#define FDMA_XFR_STAT_XFR_CMD_QUEUE_FULL_MASK    0x2u
#define FDMA_XFR_STAT_XFR_CMD_QUEUE_FULL_SHIFT   1u
#define FDMA_XFR_STAT_XFR_CMD_QUEUE_FULL_WIDTH   1u
#define FDMA_XFR_STAT_XFR_CMD_QUEUE_FULL(x)      (((uint32_t)(((uint32_t)(x))<<FDMA_XFR_STAT_XFR_CMD_QUEUE_FULL_SHIFT))&FDMA_XFR_STAT_XFR_CMD_QUEUE_FULL_MASK)
#define FDMA_XFR_STAT_CRC_ERR_MASK               0x4u
#define FDMA_XFR_STAT_CRC_ERR_SHIFT              2u
#define FDMA_XFR_STAT_CRC_ERR_WIDTH              1u
#define FDMA_XFR_STAT_CRC_ERR(x)                 (((uint32_t)(((uint32_t)(x))<<FDMA_XFR_STAT_CRC_ERR_SHIFT))&FDMA_XFR_STAT_CRC_ERR_MASK)
#define FDMA_XFR_STAT_CMD_CFG_ERR_MASK           0x8u
#define FDMA_XFR_STAT_CMD_CFG_ERR_SHIFT          3u
#define FDMA_XFR_STAT_CMD_CFG_ERR_WIDTH          1u
#define FDMA_XFR_STAT_CMD_CFG_ERR(x)             (((uint32_t)(((uint32_t)(x))<<FDMA_XFR_STAT_CMD_CFG_ERR_SHIFT))&FDMA_XFR_STAT_CMD_CFG_ERR_MASK)
#define FDMA_XFR_STAT_TR_RD_ERR_MASK             0x10u
#define FDMA_XFR_STAT_TR_RD_ERR_SHIFT            4u
#define FDMA_XFR_STAT_TR_RD_ERR_WIDTH            1u
#define FDMA_XFR_STAT_TR_RD_ERR(x)               (((uint32_t)(((uint32_t)(x))<<FDMA_XFR_STAT_TR_RD_ERR_SHIFT))&FDMA_XFR_STAT_TR_RD_ERR_MASK)
#define FDMA_XFR_STAT_CRC_RD_ERR_MASK            0x20u
#define FDMA_XFR_STAT_CRC_RD_ERR_SHIFT           5u
#define FDMA_XFR_STAT_CRC_RD_ERR_WIDTH           1u
#define FDMA_XFR_STAT_CRC_RD_ERR(x)              (((uint32_t)(((uint32_t)(x))<<FDMA_XFR_STAT_CRC_RD_ERR_SHIFT))&FDMA_XFR_STAT_CRC_RD_ERR_MASK)
#define FDMA_XFR_STAT_XFR_CFG_ERR_MASK           0x40u
#define FDMA_XFR_STAT_XFR_CFG_ERR_SHIFT          6u
#define FDMA_XFR_STAT_XFR_CFG_ERR_WIDTH          1u
#define FDMA_XFR_STAT_XFR_CFG_ERR(x)             (((uint32_t)(((uint32_t)(x))<<FDMA_XFR_STAT_XFR_CFG_ERR_SHIFT))&FDMA_XFR_STAT_XFR_CFG_ERR_MASK)
#define FDMA_XFR_STAT_XFR_ERR_DDR_MASK           0x80u
#define FDMA_XFR_STAT_XFR_ERR_DDR_SHIFT          7u
#define FDMA_XFR_STAT_XFR_ERR_DDR_WIDTH          1u
#define FDMA_XFR_STAT_XFR_ERR_DDR(x)             (((uint32_t)(((uint32_t)(x))<<FDMA_XFR_STAT_XFR_ERR_DDR_SHIFT))&FDMA_XFR_STAT_XFR_ERR_DDR_MASK)
#define FDMA_XFR_STAT_XFR_ERR_SRAM_MASK          0x100u
#define FDMA_XFR_STAT_XFR_ERR_SRAM_SHIFT         8u
#define FDMA_XFR_STAT_XFR_ERR_SRAM_WIDTH         1u
#define FDMA_XFR_STAT_XFR_ERR_SRAM(x)            (((uint32_t)(((uint32_t)(x))<<FDMA_XFR_STAT_XFR_ERR_SRAM_SHIFT))&FDMA_XFR_STAT_XFR_ERR_SRAM_MASK)
#define FDMA_XFR_STAT_CRC_WR_ERR_MASK            0x200u
#define FDMA_XFR_STAT_CRC_WR_ERR_SHIFT           9u
#define FDMA_XFR_STAT_CRC_WR_ERR_WIDTH           1u
#define FDMA_XFR_STAT_CRC_WR_ERR(x)              (((uint32_t)(((uint32_t)(x))<<FDMA_XFR_STAT_CRC_WR_ERR_SHIFT))&FDMA_XFR_STAT_CRC_WR_ERR_MASK)
#define FDMA_XFR_STAT_TR_WBACK_ERR_MASK          0x400u
#define FDMA_XFR_STAT_TR_WBACK_ERR_SHIFT         10u
#define FDMA_XFR_STAT_TR_WBACK_ERR_WIDTH         1u
#define FDMA_XFR_STAT_TR_WBACK_ERR(x)            (((uint32_t)(((uint32_t)(x))<<FDMA_XFR_STAT_TR_WBACK_ERR_SHIFT))&FDMA_XFR_STAT_TR_WBACK_ERR_MASK)
#define FDMA_XFR_STAT_XFR_DONE_MASK              0x10000u
#define FDMA_XFR_STAT_XFR_DONE_SHIFT             16u
#define FDMA_XFR_STAT_XFR_DONE_WIDTH             1u
#define FDMA_XFR_STAT_XFR_DONE(x)                (((uint32_t)(((uint32_t)(x))<<FDMA_XFR_STAT_XFR_DONE_SHIFT))&FDMA_XFR_STAT_XFR_DONE_MASK)
#define FDMA_XFR_STAT_DONE_CNT_MASK              0xFF000000u
#define FDMA_XFR_STAT_DONE_CNT_SHIFT             24u
#define FDMA_XFR_STAT_DONE_CNT_WIDTH             8u
#define FDMA_XFR_STAT_DONE_CNT(x)                (((uint32_t)(((uint32_t)(x))<<FDMA_XFR_STAT_DONE_CNT_SHIFT))&FDMA_XFR_STAT_DONE_CNT_MASK)
/* CALC_CRC_VAL Bit Fields */
#define FDMA_CALC_CRC_VAL_CALC_CRC_VAL_MASK      0xFFFFFFFFu
#define FDMA_CALC_CRC_VAL_CALC_CRC_VAL_SHIFT     0u
#define FDMA_CALC_CRC_VAL_CALC_CRC_VAL_WIDTH     32u
#define FDMA_CALC_CRC_VAL_CALC_CRC_VAL(x)        (((uint32_t)(((uint32_t)(x))<<FDMA_CALC_CRC_VAL_CALC_CRC_VAL_SHIFT))&FDMA_CALC_CRC_VAL_CALC_CRC_VAL_MASK)
/* CURR_DDR_PTR Bit Fields */
#define FDMA_CURR_DDR_PTR_CURR_DDR_PTR_MASK      0xFFFFFFFFu
#define FDMA_CURR_DDR_PTR_CURR_DDR_PTR_SHIFT     0u
#define FDMA_CURR_DDR_PTR_CURR_DDR_PTR_WIDTH     32u
#define FDMA_CURR_DDR_PTR_CURR_DDR_PTR(x)        (((uint32_t)(((uint32_t)(x))<<FDMA_CURR_DDR_PTR_CURR_DDR_PTR_SHIFT))&FDMA_CURR_DDR_PTR_CURR_DDR_PTR_MASK)
/* CURR_SRAM_PTR Bit Fields */
#define FDMA_CURR_SRAM_PTR_CURR_SRAM_PTR_MASK    0xFFFFFFFFu
#define FDMA_CURR_SRAM_PTR_CURR_SRAM_PTR_SHIFT   0u
#define FDMA_CURR_SRAM_PTR_CURR_SRAM_PTR_WIDTH   32u
#define FDMA_CURR_SRAM_PTR_CURR_SRAM_PTR(x)      (((uint32_t)(((uint32_t)(x))<<FDMA_CURR_SRAM_PTR_CURR_SRAM_PTR_SHIFT))&FDMA_CURR_SRAM_PTR_CURR_SRAM_PTR_MASK)
/* XFR_REC_NUM_DONE Bit Fields */
#define FDMA_XFR_REC_NUM_DONE_XFR_REC_NUM_DONE_MASK 0x7Fu
#define FDMA_XFR_REC_NUM_DONE_XFR_REC_NUM_DONE_SHIFT 0u
#define FDMA_XFR_REC_NUM_DONE_XFR_REC_NUM_DONE_WIDTH 7u
#define FDMA_XFR_REC_NUM_DONE_XFR_REC_NUM_DONE(x) (((uint32_t)(((uint32_t)(x))<<FDMA_XFR_REC_NUM_DONE_XFR_REC_NUM_DONE_SHIFT))&FDMA_XFR_REC_NUM_DONE_XFR_REC_NUM_DONE_MASK)
/* ERR_XFR_REC_NUM Bit Fields */
#define FDMA_ERR_XFR_REC_NUM_ERR_XFR_REC_NUM_MASK 0x7Fu
#define FDMA_ERR_XFR_REC_NUM_ERR_XFR_REC_NUM_SHIFT 0u
#define FDMA_ERR_XFR_REC_NUM_ERR_XFR_REC_NUM_WIDTH 7u
#define FDMA_ERR_XFR_REC_NUM_ERR_XFR_REC_NUM(x)  (((uint32_t)(((uint32_t)(x))<<FDMA_ERR_XFR_REC_NUM_ERR_XFR_REC_NUM_SHIFT))&FDMA_ERR_XFR_REC_NUM_ERR_XFR_REC_NUM_MASK)
/* NEXT_LINE Bit Fields */
#define FDMA_NEXT_LINE_DDR_NEXT_LINE_NUM_MASK    0xFFFFu
#define FDMA_NEXT_LINE_DDR_NEXT_LINE_NUM_SHIFT   0u
#define FDMA_NEXT_LINE_DDR_NEXT_LINE_NUM_WIDTH   16u
#define FDMA_NEXT_LINE_DDR_NEXT_LINE_NUM(x)      (((uint32_t)(((uint32_t)(x))<<FDMA_NEXT_LINE_DDR_NEXT_LINE_NUM_SHIFT))&FDMA_NEXT_LINE_DDR_NEXT_LINE_NUM_MASK)
#define FDMA_NEXT_LINE_SRAM_NEXT_LINE_NUM_MASK   0x3FFF0000u
#define FDMA_NEXT_LINE_SRAM_NEXT_LINE_NUM_SHIFT  16u
#define FDMA_NEXT_LINE_SRAM_NEXT_LINE_NUM_WIDTH  14u
#define FDMA_NEXT_LINE_SRAM_NEXT_LINE_NUM(x)     (((uint32_t)(((uint32_t)(x))<<FDMA_NEXT_LINE_SRAM_NEXT_LINE_NUM_SHIFT))&FDMA_NEXT_LINE_SRAM_NEXT_LINE_NUM_MASK)
/* CTRL Bit Fields */
#define FDMA_CTRL_SOFT_RST_MASK                  0x1u
#define FDMA_CTRL_SOFT_RST_SHIFT                 0u
#define FDMA_CTRL_SOFT_RST_WIDTH                 1u
#define FDMA_CTRL_SOFT_RST(x)                    (((uint32_t)(((uint32_t)(x))<<FDMA_CTRL_SOFT_RST_SHIFT))&FDMA_CTRL_SOFT_RST_MASK)

/*!
 * @}
 */ /* end of group FDMA_Register_Masks */


/*!
 * @}
 */ /* end of group FDMA_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- FR Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup FR_Peripheral_Access_Layer FR Peripheral Access Layer
 * @{
 */


/** FR - Size of Registers Arrays */
#define FR_NMVR_COUNT                            6u
#define FR_SSR_COUNT                             8u
#define FR_SSCR_COUNT                            4u
#define FR_MB_COUNT                              128u
#define FR_MBDOR_COUNT                           132u
#define FR_LEETR_COUNT                           6u

/** FR - Register Layout Typedef */
typedef struct {
  __I  uint16_t MVR;                               /**< Module Version Register, offset: 0x0 */
  __IO uint16_t MCR;                               /**< Module Configuration Register, offset: 0x2 */
  __IO uint16_t SYMBADHR;                          /**< System Memory Base Address High Register, offset: 0x4 */
  __IO uint16_t SYMBADLR;                          /**< System Memory Base Address Low Register, offset: 0x6 */
  __IO uint16_t STBSCR;                            /**< Strobe Signal Control Register, offset: 0x8 */
       uint8_t RESERVED_0[2];
  __IO uint16_t MBDSR;                             /**< Message Buffer Data Size Register, offset: 0xC */
  __IO uint16_t MBSSUTR;                           /**< Message Buffer Segment Size and Utilization Register, offset: 0xE */
  __IO uint16_t PEDRAR;                            /**< PE DRAM Access Register, offset: 0x10 */
  __IO uint16_t PEDRDR;                            /**< PE DRAM Data Register, offset: 0x12 */
  __IO uint16_t POCR;                              /**< Protocol Operation Control Register, offset: 0x14 */
  __IO uint16_t GIFER;                             /**< Global Interrupt Flag and Enable Register, offset: 0x16 */
  __IO uint16_t PIFR0;                             /**< Protocol Interrupt Flag Register 0, offset: 0x18 */
  __IO uint16_t PIFR1;                             /**< Protocol Interrupt Flag Register 1, offset: 0x1A */
  __IO uint16_t PIER0;                             /**< Protocol Interrupt Enable Register 0, offset: 0x1C */
  __IO uint16_t PIER1;                             /**< Protocol Interrupt Enable Register 1, offset: 0x1E */
  __IO uint16_t CHIERFR;                           /**< CHI Error Flag Register, offset: 0x20 */
  __IO uint16_t MBIVEC;                            /**< Message Buffer Interrupt Vector Register, offset: 0x22 */
  __I  uint16_t CASERCR;                           /**< Channel A Status Error Counter Register, offset: 0x24 */
  __I  uint16_t CBSERCR;                           /**< Channel B Status Error Counter Register, offset: 0x26 */
  __I  uint16_t PSR0;                              /**< Protocol Status Register 0, offset: 0x28 */
  __IO uint16_t PSR1;                              /**< Protocol Status Register 1, offset: 0x2A */
  __I  uint16_t PSR2;                              /**< Protocol Status Register 2, offset: 0x2C */
  __IO uint16_t PSR3;                              /**< Protocol Status Register 3, offset: 0x2E */
  __I  uint16_t MTCTR;                             /**< Macrotick Counter Register, offset: 0x30 */
  __I  uint16_t CYCTR;                             /**< Cycle Counter Register, offset: 0x32 */
  __I  uint16_t SLTCTAR;                           /**< Slot Counter Channel A Register, offset: 0x34 */
  __I  uint16_t SLTCTBR;                           /**< Slot Counter Channel B Register, offset: 0x36 */
  __I  uint16_t RTCORVR;                           /**< Rate Correction Value Register, offset: 0x38 */
  __I  uint16_t OFCORVR;                           /**< Offset Correction Value Register, offset: 0x3A */
  __I  uint16_t CIFR;                              /**< Combined Interrupt Flag Register, offset: 0x3C */
  __IO uint16_t SYMATOR;                           /**< System Memory Access Time-Out Register, offset: 0x3E */
  __I  uint16_t SFCNTR;                            /**< Sync Frame Counter Register, offset: 0x40 */
  __IO uint16_t SFTOR;                             /**< Sync Frame Table Offset Register, offset: 0x42 */
  __IO uint16_t SFTCCSR;                           /**< Sync Frame Table Configuration, Control, Status Register, offset: 0x44 */
  __IO uint16_t SFIDRFR;                           /**< Sync Frame ID Rejection Filter Register, offset: 0x46 */
  __IO uint16_t SFIDAFVR;                          /**< Sync Frame ID Acceptance Filter Value Register, offset: 0x48 */
  __IO uint16_t SFIDAFMR;                          /**< Sync Frame ID Acceptance Filter Mask Register, offset: 0x4A */
  __I  uint16_t NMVR[FR_NMVR_COUNT];               /**< Network Management Vector Register, array offset: 0x4C, array step: 0x2 */
  __IO uint16_t NMVLR;                             /**< Network Management Vector Length Register, offset: 0x58 */
  __IO uint16_t TICCR;                             /**< Timer Configuration and Control Register, offset: 0x5A */
  __IO uint16_t TI1CYSR;                           /**< Timer 1 Cycle Set Register, offset: 0x5C */
  __IO uint16_t TI1MTOR;                           /**< Timer 1 Macrotick Offset Register, offset: 0x5E */
  union {                                          /* offset: 0x60 */
    __IO uint16_t ABS;                               /**< Timer 2 Configuration Register 0 (Absolute Timer Configuration), offset: 0x60 */
    __IO uint16_t REL;                               /**< Timer 2 Configuration Register 0 (Relative Timer Configuration), offset: 0x60 */
  } TI2CR0;
  union {                                          /* offset: 0x62 */
    __IO uint16_t ABS;                               /**< Timer 2 Configuration Register 1 (Absolute Timer Configuration), offset: 0x62 */
    __IO uint16_t REL;                               /**< Timer 2 Configuration Register 1 (Relative Timer Configuration), offset: 0x62 */
  } TI2CR1;
  __IO uint16_t SSSR;                              /**< Slot Status Selection Register, offset: 0x64 */
  __IO uint16_t SSCCR;                             /**< Slot Status Counter Condition Register, offset: 0x66 */
  __I  uint16_t SSR[FR_SSR_COUNT];                 /**< Slot Status Register, array offset: 0x68, array step: 0x2 */
  __I  uint16_t SSCR[FR_SSCR_COUNT];               /**< Slot Status Counter Register, array offset: 0x78, array step: 0x2 */
  __IO uint16_t MTSACFR;                           /**< MTS A Configuration Register, offset: 0x80 */
  __IO uint16_t MTSBCFR;                           /**< MTS B Configuration Register, offset: 0x82 */
  __IO uint16_t RSBIR;                             /**< Receive Shadow Buffer Index Register, offset: 0x84 */
  __IO uint16_t RFWMSR;                            /**< Receive FIFO Watermark and Selection Register, offset: 0x86 */
  __IO uint16_t RFSIR;                             /**< Receive FIFO Start Index Register, offset: 0x88 */
  __IO uint16_t RFDSR;                             /**< Receive FIFO Depth and Size Register, offset: 0x8A */
  __I  uint16_t RFARIR;                            /**< Receive FIFO A Read Index Register, offset: 0x8C */
  __I  uint16_t RFBRIR;                            /**< Receive FIFO B Read Index Register, offset: 0x8E */
  __IO uint16_t RFMIDAFVR;                         /**< Receive FIFO Message ID Acceptance Filter Value Register, offset: 0x90 */
  __IO uint16_t RFMIDAFMR;                         /**< Receive FIFO Message ID Acceptance Filter Mask Register, offset: 0x92 */
  __IO uint16_t RFFIDRFVR;                         /**< Receive FIFO Frame ID Rejection Filter Value Register, offset: 0x94 */
  __IO uint16_t RFFIDRFMR;                         /**< Receive FIFO Frame ID Rejection Filter Mask Register, offset: 0x96 */
  __IO uint16_t RFRFCFR;                           /**< Receive FIFO Range Filter Configuration Register, offset: 0x98 */
  __IO uint16_t RFRFCTR;                           /**< Receive FIFO Range Filter Control Register, offset: 0x9A */
  __I  uint16_t LDTXSLAR;                          /**< Last Dynamic Transmit Slot Channel A Register, offset: 0x9C */
  __I  uint16_t LDTXSLBR;                          /**< Last Dynamic Transmit Slot Channel B Register, offset: 0x9E */
  __IO uint16_t PCR0;                              /**< Protocol Configuration Register 0, offset: 0xA0 */
  __IO uint16_t PCR1;                              /**< Protocol Configuration Register 1, offset: 0xA2 */
  __IO uint16_t PCR2;                              /**< Protocol Configuration Register 2, offset: 0xA4 */
  __IO uint16_t PCR3;                              /**< Protocol Configuration Register 3, offset: 0xA6 */
  __IO uint16_t PCR4;                              /**< Protocol Configuration Register 4, offset: 0xA8 */
  __IO uint16_t PCR5;                              /**< Protocol Configuration Register 5, offset: 0xAA */
  __IO uint16_t PCR6;                              /**< Protocol Configuration Register 6, offset: 0xAC */
  __IO uint16_t PCR7;                              /**< Protocol Configuration Register 7, offset: 0xAE */
  __IO uint16_t PCR8;                              /**< Protocol Configuration Register 8, offset: 0xB0 */
  __IO uint16_t PCR9;                              /**< Protocol Configuration Register 9, offset: 0xB2 */
  __IO uint16_t PCR10;                             /**< Protocol Configuration Register 10, offset: 0xB4 */
  __IO uint16_t PCR11;                             /**< Protocol Configuration Register 11, offset: 0xB6 */
  __IO uint16_t PCR12;                             /**< Protocol Configuration Register 12, offset: 0xB8 */
  __IO uint16_t PCR13;                             /**< Protocol Configuration Register 13, offset: 0xBA */
  __IO uint16_t PCR14;                             /**< Protocol Configuration Register 14, offset: 0xBC */
  __IO uint16_t PCR15;                             /**< Protocol Configuration Register 15, offset: 0xBE */
  __IO uint16_t PCR16;                             /**< Protocol Configuration Register 16, offset: 0xC0 */
  __IO uint16_t PCR17;                             /**< Protocol Configuration Register 17, offset: 0xC2 */
  __IO uint16_t PCR18;                             /**< Protocol Configuration Register 18, offset: 0xC4 */
  __IO uint16_t PCR19;                             /**< Protocol Configuration Register 19, offset: 0xC6 */
  __IO uint16_t PCR20;                             /**< Protocol Configuration Register 20, offset: 0xC8 */
  __IO uint16_t PCR21;                             /**< Protocol Configuration Register 21, offset: 0xCA */
  __IO uint16_t PCR22;                             /**< Protocol Configuration Register 22, offset: 0xCC */
  __IO uint16_t PCR23;                             /**< Protocol Configuration Register 23, offset: 0xCE */
  __IO uint16_t PCR24;                             /**< Protocol Configuration Register 24, offset: 0xD0 */
  __IO uint16_t PCR25;                             /**< Protocol Configuration Register 25, offset: 0xD2 */
  __IO uint16_t PCR26;                             /**< Protocol Configuration Register 26, offset: 0xD4 */
  __IO uint16_t PCR27;                             /**< Protocol Configuration Register 27, offset: 0xD6 */
  __IO uint16_t PCR28;                             /**< Protocol Configuration Register 28, offset: 0xD8 */
  __IO uint16_t PCR29;                             /**< Protocol Configuration Register 29, offset: 0xDA */
  __IO uint16_t PCR30;                             /**< Protocol Configuration Register 30, offset: 0xDC */
  __IO uint16_t STPWHR;                            /**< StopWatch Count High Register, offset: 0xDE */
  __IO uint16_t STPWLR;                            /**< Stop Watch Count Low Register, offset: 0xE0 */
  __IO uint16_t PEOER;                             /**< Protocol Event Output Enable and StopWatch Control Register, offset: 0xE2 */
       uint8_t RESERVED_1[2];
  __IO uint16_t RFSDOR;                            /**< Receive FIFO Start Data Offset Register, offset: 0xE6 */
  __IO uint16_t RFSYMBADHR;                        /**< Receive FIFO System Memory Base Address High Register, offset: 0xE8 */
  __IO uint16_t RFSYMBADLR;                        /**< Receive FIFO System Memory Base Address Low Register, offset: 0xEA */
  __IO uint16_t RFPTR;                             /**< Receive FIFO Periodic Timer Register, offset: 0xEC */
  __IO uint16_t RFFLPCR;                           /**< Receive FIFO Fill Level and POP Count Register, offset: 0xEE */
  __IO uint16_t EEIFER;                            /**< ECC Error Interrupt Flag and Enable Register, offset: 0xF0 */
  __IO uint16_t EERICR;                            /**< ECC Error Report and Injection Control Register, offset: 0xF2 */
  __I  uint16_t EERAR;                             /**< ECC Error Report Address Register, offset: 0xF4 */
  __I  uint16_t EERDR;                             /**< ECC Error Report Data Register, offset: 0xF6 */
  __I  uint16_t EERCR;                             /**< ECC Error Report Code Register, offset: 0xF8 */
  __IO uint16_t EEIAR;                             /**< ECC Error Injection Address Register, offset: 0xFA */
  __IO uint16_t EEIDR;                             /**< ECC Error Injection Data Register, offset: 0xFC */
  __IO uint16_t EEICR;                             /**< ECC Error Injection Code Register, offset: 0xFE */
       uint8_t RESERVED_2[1792];
  struct {                                         /* offset: 0x800, array step: 0x8 */
    __IO uint16_t CCSR;                              /**< Message Buffer Configuration, Control, Status Register, array offset: 0x800, array step: 0x8 */
    __IO uint16_t CCFR;                              /**< Message Buffer Cycle Counter Filter Register, array offset: 0x802, array step: 0x8 */
    __IO uint16_t FIDR;                              /**< Message Buffer Frame ID Register, array offset: 0x804, array step: 0x8 */
    __IO uint16_t IDXR;                              /**< Message Buffer Index Register, array offset: 0x806, array step: 0x8 */
  } MB[FR_MB_COUNT];
       uint8_t RESERVED_3[1024];
  __IO uint16_t MBDOR[FR_MBDOR_COUNT];             /**< Message Buffer Data Field Offset Register, array offset: 0x1000, array step: 0x2 */
  __IO uint16_t LEETR[FR_LEETR_COUNT];             /**< LRAM ECC Error Test Register, array offset: 0x1108, array step: 0x2 */
} FR_Type, *FR_MemMapPtr;

 /** Number of instances of the FR module. */
#define FR_INSTANCE_COUNT                        (1u)


/* FR - Peripheral instance base addresses */
/** Peripheral FR base address */
#define FR_BASE                                  (0x40034000u)
/** Peripheral FR base pointer */
#define FR                                       ((FR_Type *)FR_BASE)
/** Array initializer of FR peripheral base addresses */
#define FR_BASE_ADDRS                            { FR_BASE }
/** Array initializer of FR peripheral base pointers */
#define FR_BASE_PTRS                             { FR }

/* ----------------------------------------------------------------------------
   -- FR Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup FR_Register_Masks FR Register Masks
 * @{
 */

/* MVR Bit Fields */
#define FR_MVR_PEVER_MASK                        0xFFu
#define FR_MVR_PEVER_SHIFT                       0u
#define FR_MVR_PEVER_WIDTH                       8u
#define FR_MVR_PEVER(x)                          (((uint16_t)(((uint16_t)(x))<<FR_MVR_PEVER_SHIFT))&FR_MVR_PEVER_MASK)
#define FR_MVR_CHIVER_MASK                       0xFF00u
#define FR_MVR_CHIVER_SHIFT                      8u
#define FR_MVR_CHIVER_WIDTH                      8u
#define FR_MVR_CHIVER(x)                         (((uint16_t)(((uint16_t)(x))<<FR_MVR_CHIVER_SHIFT))&FR_MVR_CHIVER_MASK)
/* MCR Bit Fields */
#define FR_MCR_BITRATE_MASK                      0xEu
#define FR_MCR_BITRATE_SHIFT                     1u
#define FR_MCR_BITRATE_WIDTH                     3u
#define FR_MCR_BITRATE(x)                        (((uint16_t)(((uint16_t)(x))<<FR_MCR_BITRATE_SHIFT))&FR_MCR_BITRATE_MASK)
#define FR_MCR_CLKSEL_MASK                       0x10u
#define FR_MCR_CLKSEL_SHIFT                      4u
#define FR_MCR_CLKSEL_WIDTH                      1u
#define FR_MCR_CLKSEL(x)                         (((uint16_t)(((uint16_t)(x))<<FR_MCR_CLKSEL_SHIFT))&FR_MCR_CLKSEL_MASK)
#define FR_MCR_FAM_MASK                          0x40u
#define FR_MCR_FAM_SHIFT                         6u
#define FR_MCR_FAM_WIDTH                         1u
#define FR_MCR_FAM(x)                            (((uint16_t)(((uint16_t)(x))<<FR_MCR_FAM_SHIFT))&FR_MCR_FAM_MASK)
#define FR_MCR_FUM_MASK                          0x80u
#define FR_MCR_FUM_SHIFT                         7u
#define FR_MCR_FUM_WIDTH                         1u
#define FR_MCR_FUM(x)                            (((uint16_t)(((uint16_t)(x))<<FR_MCR_FUM_SHIFT))&FR_MCR_FUM_MASK)
#define FR_MCR_ECCE_MASK                         0x200u
#define FR_MCR_ECCE_SHIFT                        9u
#define FR_MCR_ECCE_WIDTH                        1u
#define FR_MCR_ECCE(x)                           (((uint16_t)(((uint16_t)(x))<<FR_MCR_ECCE_SHIFT))&FR_MCR_ECCE_MASK)
#define FR_MCR_SFFE_MASK                         0x400u
#define FR_MCR_SFFE_SHIFT                        10u
#define FR_MCR_SFFE_WIDTH                        1u
#define FR_MCR_SFFE(x)                           (((uint16_t)(((uint16_t)(x))<<FR_MCR_SFFE_SHIFT))&FR_MCR_SFFE_MASK)
#define FR_MCR_CHA_MASK                          0x800u
#define FR_MCR_CHA_SHIFT                         11u
#define FR_MCR_CHA_WIDTH                         1u
#define FR_MCR_CHA(x)                            (((uint16_t)(((uint16_t)(x))<<FR_MCR_CHA_SHIFT))&FR_MCR_CHA_MASK)
#define FR_MCR_CHB_MASK                          0x1000u
#define FR_MCR_CHB_SHIFT                         12u
#define FR_MCR_CHB_WIDTH                         1u
#define FR_MCR_CHB(x)                            (((uint16_t)(((uint16_t)(x))<<FR_MCR_CHB_SHIFT))&FR_MCR_CHB_MASK)
#define FR_MCR_SCM_MASK                          0x2000u
#define FR_MCR_SCM_SHIFT                         13u
#define FR_MCR_SCM_WIDTH                         1u
#define FR_MCR_SCM(x)                            (((uint16_t)(((uint16_t)(x))<<FR_MCR_SCM_SHIFT))&FR_MCR_SCM_MASK)
#define FR_MCR_SBFF_MASK                         0x4000u
#define FR_MCR_SBFF_SHIFT                        14u
#define FR_MCR_SBFF_WIDTH                        1u
#define FR_MCR_SBFF(x)                           (((uint16_t)(((uint16_t)(x))<<FR_MCR_SBFF_SHIFT))&FR_MCR_SBFF_MASK)
#define FR_MCR_MEN_MASK                          0x8000u
#define FR_MCR_MEN_SHIFT                         15u
#define FR_MCR_MEN_WIDTH                         1u
#define FR_MCR_MEN(x)                            (((uint16_t)(((uint16_t)(x))<<FR_MCR_MEN_SHIFT))&FR_MCR_MEN_MASK)
/* SYMBADHR Bit Fields */
#define FR_SYMBADHR_SMBA_MASK                    0xFFFFu
#define FR_SYMBADHR_SMBA_SHIFT                   0u
#define FR_SYMBADHR_SMBA_WIDTH                   16u
#define FR_SYMBADHR_SMBA(x)                      (((uint16_t)(((uint16_t)(x))<<FR_SYMBADHR_SMBA_SHIFT))&FR_SYMBADHR_SMBA_MASK)
/* SYMBADLR Bit Fields */
#define FR_SYMBADLR_SMBA_MASK                    0xFFF0u
#define FR_SYMBADLR_SMBA_SHIFT                   4u
#define FR_SYMBADLR_SMBA_WIDTH                   12u
#define FR_SYMBADLR_SMBA(x)                      (((uint16_t)(((uint16_t)(x))<<FR_SYMBADLR_SMBA_SHIFT))&FR_SYMBADLR_SMBA_MASK)
/* STBSCR Bit Fields */
#define FR_STBSCR_STBPSEL_MASK                   0x3u
#define FR_STBSCR_STBPSEL_SHIFT                  0u
#define FR_STBSCR_STBPSEL_WIDTH                  2u
#define FR_STBSCR_STBPSEL(x)                     (((uint16_t)(((uint16_t)(x))<<FR_STBSCR_STBPSEL_SHIFT))&FR_STBSCR_STBPSEL_MASK)
#define FR_STBSCR_ENB_MASK                       0x10u
#define FR_STBSCR_ENB_SHIFT                      4u
#define FR_STBSCR_ENB_WIDTH                      1u
#define FR_STBSCR_ENB(x)                         (((uint16_t)(((uint16_t)(x))<<FR_STBSCR_ENB_SHIFT))&FR_STBSCR_ENB_MASK)
#define FR_STBSCR_SEL_MASK                       0xF00u
#define FR_STBSCR_SEL_SHIFT                      8u
#define FR_STBSCR_SEL_WIDTH                      4u
#define FR_STBSCR_SEL(x)                         (((uint16_t)(((uint16_t)(x))<<FR_STBSCR_SEL_SHIFT))&FR_STBSCR_SEL_MASK)
#define FR_STBSCR_WMD_MASK                       0x8000u
#define FR_STBSCR_WMD_SHIFT                      15u
#define FR_STBSCR_WMD_WIDTH                      1u
#define FR_STBSCR_WMD(x)                         (((uint16_t)(((uint16_t)(x))<<FR_STBSCR_WMD_SHIFT))&FR_STBSCR_WMD_MASK)
/* MBDSR Bit Fields */
#define FR_MBDSR_MBSEG1DS_MASK                   0x7Fu
#define FR_MBDSR_MBSEG1DS_SHIFT                  0u
#define FR_MBDSR_MBSEG1DS_WIDTH                  7u
#define FR_MBDSR_MBSEG1DS(x)                     (((uint16_t)(((uint16_t)(x))<<FR_MBDSR_MBSEG1DS_SHIFT))&FR_MBDSR_MBSEG1DS_MASK)
#define FR_MBDSR_MBSEG2DS_MASK                   0x7F00u
#define FR_MBDSR_MBSEG2DS_SHIFT                  8u
#define FR_MBDSR_MBSEG2DS_WIDTH                  7u
#define FR_MBDSR_MBSEG2DS(x)                     (((uint16_t)(((uint16_t)(x))<<FR_MBDSR_MBSEG2DS_SHIFT))&FR_MBDSR_MBSEG2DS_MASK)
/* MBSSUTR Bit Fields */
#define FR_MBSSUTR_LAST_MB_UTIL_MASK             0x7Fu
#define FR_MBSSUTR_LAST_MB_UTIL_SHIFT            0u
#define FR_MBSSUTR_LAST_MB_UTIL_WIDTH            7u
#define FR_MBSSUTR_LAST_MB_UTIL(x)               (((uint16_t)(((uint16_t)(x))<<FR_MBSSUTR_LAST_MB_UTIL_SHIFT))&FR_MBSSUTR_LAST_MB_UTIL_MASK)
#define FR_MBSSUTR_LAST_MB_SEG1_MASK             0x7F00u
#define FR_MBSSUTR_LAST_MB_SEG1_SHIFT            8u
#define FR_MBSSUTR_LAST_MB_SEG1_WIDTH            7u
#define FR_MBSSUTR_LAST_MB_SEG1(x)               (((uint16_t)(((uint16_t)(x))<<FR_MBSSUTR_LAST_MB_SEG1_SHIFT))&FR_MBSSUTR_LAST_MB_SEG1_MASK)
/* PEDRAR Bit Fields */
#define FR_PEDRAR_DAD_MASK                       0x1u
#define FR_PEDRAR_DAD_SHIFT                      0u
#define FR_PEDRAR_DAD_WIDTH                      1u
#define FR_PEDRAR_DAD(x)                         (((uint16_t)(((uint16_t)(x))<<FR_PEDRAR_DAD_SHIFT))&FR_PEDRAR_DAD_MASK)
#define FR_PEDRAR_ADDR_MASK                      0xFFEu
#define FR_PEDRAR_ADDR_SHIFT                     1u
#define FR_PEDRAR_ADDR_WIDTH                     11u
#define FR_PEDRAR_ADDR(x)                        (((uint16_t)(((uint16_t)(x))<<FR_PEDRAR_ADDR_SHIFT))&FR_PEDRAR_ADDR_MASK)
#define FR_PEDRAR_INST_MASK                      0xF000u
#define FR_PEDRAR_INST_SHIFT                     12u
#define FR_PEDRAR_INST_WIDTH                     4u
#define FR_PEDRAR_INST(x)                        (((uint16_t)(((uint16_t)(x))<<FR_PEDRAR_INST_SHIFT))&FR_PEDRAR_INST_MASK)
/* PEDRDR Bit Fields */
#define FR_PEDRDR_DATA_MASK                      0xFFFFu
#define FR_PEDRDR_DATA_SHIFT                     0u
#define FR_PEDRDR_DATA_WIDTH                     16u
#define FR_PEDRDR_DATA(x)                        (((uint16_t)(((uint16_t)(x))<<FR_PEDRDR_DATA_SHIFT))&FR_PEDRDR_DATA_MASK)
/* POCR Bit Fields */
#define FR_POCR_POCCMD_MASK                      0xFu
#define FR_POCR_POCCMD_SHIFT                     0u
#define FR_POCR_POCCMD_WIDTH                     4u
#define FR_POCR_POCCMD(x)                        (((uint16_t)(((uint16_t)(x))<<FR_POCR_POCCMD_SHIFT))&FR_POCR_POCCMD_MASK)
#define FR_POCR_BSY_WMC_MASK                     0x80u
#define FR_POCR_BSY_WMC_SHIFT                    7u
#define FR_POCR_BSY_WMC_WIDTH                    1u
#define FR_POCR_BSY_WMC(x)                       (((uint16_t)(((uint16_t)(x))<<FR_POCR_BSY_WMC_SHIFT))&FR_POCR_BSY_WMC_MASK)
#define FR_POCR_ERC_AP_MASK                      0x300u
#define FR_POCR_ERC_AP_SHIFT                     8u
#define FR_POCR_ERC_AP_WIDTH                     2u
#define FR_POCR_ERC_AP(x)                        (((uint16_t)(((uint16_t)(x))<<FR_POCR_ERC_AP_SHIFT))&FR_POCR_ERC_AP_MASK)
#define FR_POCR_EOC_AP_MASK                      0xC00u
#define FR_POCR_EOC_AP_SHIFT                     10u
#define FR_POCR_EOC_AP_WIDTH                     2u
#define FR_POCR_EOC_AP(x)                        (((uint16_t)(((uint16_t)(x))<<FR_POCR_EOC_AP_SHIFT))&FR_POCR_EOC_AP_MASK)
#define FR_POCR_WME_MASK                         0x8000u
#define FR_POCR_WME_SHIFT                        15u
#define FR_POCR_WME_WIDTH                        1u
#define FR_POCR_WME(x)                           (((uint16_t)(((uint16_t)(x))<<FR_POCR_WME_SHIFT))&FR_POCR_WME_MASK)
/* GIFER Bit Fields */
#define FR_GIFER_TBIE_MASK                       0x1u
#define FR_GIFER_TBIE_SHIFT                      0u
#define FR_GIFER_TBIE_WIDTH                      1u
#define FR_GIFER_TBIE(x)                         (((uint16_t)(((uint16_t)(x))<<FR_GIFER_TBIE_SHIFT))&FR_GIFER_TBIE_MASK)
#define FR_GIFER_RBIE_MASK                       0x2u
#define FR_GIFER_RBIE_SHIFT                      1u
#define FR_GIFER_RBIE_WIDTH                      1u
#define FR_GIFER_RBIE(x)                         (((uint16_t)(((uint16_t)(x))<<FR_GIFER_RBIE_SHIFT))&FR_GIFER_RBIE_MASK)
#define FR_GIFER_FAFAIE_MASK                     0x4u
#define FR_GIFER_FAFAIE_SHIFT                    2u
#define FR_GIFER_FAFAIE_WIDTH                    1u
#define FR_GIFER_FAFAIE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_GIFER_FAFAIE_SHIFT))&FR_GIFER_FAFAIE_MASK)
#define FR_GIFER_FAFBIE_MASK                     0x8u
#define FR_GIFER_FAFBIE_SHIFT                    3u
#define FR_GIFER_FAFBIE_WIDTH                    1u
#define FR_GIFER_FAFBIE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_GIFER_FAFBIE_SHIFT))&FR_GIFER_FAFBIE_MASK)
#define FR_GIFER_WUPIE_MASK                      0x10u
#define FR_GIFER_WUPIE_SHIFT                     4u
#define FR_GIFER_WUPIE_WIDTH                     1u
#define FR_GIFER_WUPIE(x)                        (((uint16_t)(((uint16_t)(x))<<FR_GIFER_WUPIE_SHIFT))&FR_GIFER_WUPIE_MASK)
#define FR_GIFER_CHIE_MASK                       0x20u
#define FR_GIFER_CHIE_SHIFT                      5u
#define FR_GIFER_CHIE_WIDTH                      1u
#define FR_GIFER_CHIE(x)                         (((uint16_t)(((uint16_t)(x))<<FR_GIFER_CHIE_SHIFT))&FR_GIFER_CHIE_MASK)
#define FR_GIFER_PRIE_MASK                       0x40u
#define FR_GIFER_PRIE_SHIFT                      6u
#define FR_GIFER_PRIE_WIDTH                      1u
#define FR_GIFER_PRIE(x)                         (((uint16_t)(((uint16_t)(x))<<FR_GIFER_PRIE_SHIFT))&FR_GIFER_PRIE_MASK)
#define FR_GIFER_MIE_MASK                        0x80u
#define FR_GIFER_MIE_SHIFT                       7u
#define FR_GIFER_MIE_WIDTH                       1u
#define FR_GIFER_MIE(x)                          (((uint16_t)(((uint16_t)(x))<<FR_GIFER_MIE_SHIFT))&FR_GIFER_MIE_MASK)
#define FR_GIFER_TBIF_MASK                       0x100u
#define FR_GIFER_TBIF_SHIFT                      8u
#define FR_GIFER_TBIF_WIDTH                      1u
#define FR_GIFER_TBIF(x)                         (((uint16_t)(((uint16_t)(x))<<FR_GIFER_TBIF_SHIFT))&FR_GIFER_TBIF_MASK)
#define FR_GIFER_RBIF_MASK                       0x200u
#define FR_GIFER_RBIF_SHIFT                      9u
#define FR_GIFER_RBIF_WIDTH                      1u
#define FR_GIFER_RBIF(x)                         (((uint16_t)(((uint16_t)(x))<<FR_GIFER_RBIF_SHIFT))&FR_GIFER_RBIF_MASK)
#define FR_GIFER_FAFAIF_MASK                     0x400u
#define FR_GIFER_FAFAIF_SHIFT                    10u
#define FR_GIFER_FAFAIF_WIDTH                    1u
#define FR_GIFER_FAFAIF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_GIFER_FAFAIF_SHIFT))&FR_GIFER_FAFAIF_MASK)
#define FR_GIFER_FAFBIF_MASK                     0x800u
#define FR_GIFER_FAFBIF_SHIFT                    11u
#define FR_GIFER_FAFBIF_WIDTH                    1u
#define FR_GIFER_FAFBIF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_GIFER_FAFBIF_SHIFT))&FR_GIFER_FAFBIF_MASK)
#define FR_GIFER_WUPIF_MASK                      0x1000u
#define FR_GIFER_WUPIF_SHIFT                     12u
#define FR_GIFER_WUPIF_WIDTH                     1u
#define FR_GIFER_WUPIF(x)                        (((uint16_t)(((uint16_t)(x))<<FR_GIFER_WUPIF_SHIFT))&FR_GIFER_WUPIF_MASK)
#define FR_GIFER_CHIF_MASK                       0x2000u
#define FR_GIFER_CHIF_SHIFT                      13u
#define FR_GIFER_CHIF_WIDTH                      1u
#define FR_GIFER_CHIF(x)                         (((uint16_t)(((uint16_t)(x))<<FR_GIFER_CHIF_SHIFT))&FR_GIFER_CHIF_MASK)
#define FR_GIFER_PRIF_MASK                       0x4000u
#define FR_GIFER_PRIF_SHIFT                      14u
#define FR_GIFER_PRIF_WIDTH                      1u
#define FR_GIFER_PRIF(x)                         (((uint16_t)(((uint16_t)(x))<<FR_GIFER_PRIF_SHIFT))&FR_GIFER_PRIF_MASK)
#define FR_GIFER_MIF_MASK                        0x8000u
#define FR_GIFER_MIF_SHIFT                       15u
#define FR_GIFER_MIF_WIDTH                       1u
#define FR_GIFER_MIF(x)                          (((uint16_t)(((uint16_t)(x))<<FR_GIFER_MIF_SHIFT))&FR_GIFER_MIF_MASK)
/* PIFR0 Bit Fields */
#define FR_PIFR0_CYS_IF_MASK                     0x1u
#define FR_PIFR0_CYS_IF_SHIFT                    0u
#define FR_PIFR0_CYS_IF_WIDTH                    1u
#define FR_PIFR0_CYS_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_CYS_IF_SHIFT))&FR_PIFR0_CYS_IF_MASK)
#define FR_PIFR0_TI1_IF_MASK                     0x2u
#define FR_PIFR0_TI1_IF_SHIFT                    1u
#define FR_PIFR0_TI1_IF_WIDTH                    1u
#define FR_PIFR0_TI1_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_TI1_IF_SHIFT))&FR_PIFR0_TI1_IF_MASK)
#define FR_PIFR0_TI2_IF_MASK                     0x4u
#define FR_PIFR0_TI2_IF_SHIFT                    2u
#define FR_PIFR0_TI2_IF_WIDTH                    1u
#define FR_PIFR0_TI2_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_TI2_IF_SHIFT))&FR_PIFR0_TI2_IF_MASK)
#define FR_PIFR0_TBVA_IF_MASK                    0x8u
#define FR_PIFR0_TBVA_IF_SHIFT                   3u
#define FR_PIFR0_TBVA_IF_WIDTH                   1u
#define FR_PIFR0_TBVA_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_TBVA_IF_SHIFT))&FR_PIFR0_TBVA_IF_MASK)
#define FR_PIFR0_TBVB_IF_MASK                    0x10u
#define FR_PIFR0_TBVB_IF_SHIFT                   4u
#define FR_PIFR0_TBVB_IF_WIDTH                   1u
#define FR_PIFR0_TBVB_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_TBVB_IF_SHIFT))&FR_PIFR0_TBVB_IF_MASK)
#define FR_PIFR0_LTXA_IF_MASK                    0x20u
#define FR_PIFR0_LTXA_IF_SHIFT                   5u
#define FR_PIFR0_LTXA_IF_WIDTH                   1u
#define FR_PIFR0_LTXA_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_LTXA_IF_SHIFT))&FR_PIFR0_LTXA_IF_MASK)
#define FR_PIFR0_LTXB_IF_MASK                    0x40u
#define FR_PIFR0_LTXB_IF_SHIFT                   6u
#define FR_PIFR0_LTXB_IF_WIDTH                   1u
#define FR_PIFR0_LTXB_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_LTXB_IF_SHIFT))&FR_PIFR0_LTXB_IF_MASK)
#define FR_PIFR0_MTX_IF_MASK                     0x80u
#define FR_PIFR0_MTX_IF_SHIFT                    7u
#define FR_PIFR0_MTX_IF_WIDTH                    1u
#define FR_PIFR0_MTX_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_MTX_IF_SHIFT))&FR_PIFR0_MTX_IF_MASK)
#define FR_PIFR0_MXS_IF_MASK                     0x100u
#define FR_PIFR0_MXS_IF_SHIFT                    8u
#define FR_PIFR0_MXS_IF_WIDTH                    1u
#define FR_PIFR0_MXS_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_MXS_IF_SHIFT))&FR_PIFR0_MXS_IF_MASK)
#define FR_PIFR0_CCL_IF_MASK                     0x200u
#define FR_PIFR0_CCL_IF_SHIFT                    9u
#define FR_PIFR0_CCL_IF_WIDTH                    1u
#define FR_PIFR0_CCL_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_CCL_IF_SHIFT))&FR_PIFR0_CCL_IF_MASK)
#define FR_PIFR0_MOC_IF_MASK                     0x400u
#define FR_PIFR0_MOC_IF_SHIFT                    10u
#define FR_PIFR0_MOC_IF_WIDTH                    1u
#define FR_PIFR0_MOC_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_MOC_IF_SHIFT))&FR_PIFR0_MOC_IF_MASK)
#define FR_PIFR0_MRC_IF_MASK                     0x800u
#define FR_PIFR0_MRC_IF_SHIFT                    11u
#define FR_PIFR0_MRC_IF_WIDTH                    1u
#define FR_PIFR0_MRC_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_MRC_IF_SHIFT))&FR_PIFR0_MRC_IF_MASK)
#define FR_PIFR0_CSA_IF_MASK                     0x1000u
#define FR_PIFR0_CSA_IF_SHIFT                    12u
#define FR_PIFR0_CSA_IF_WIDTH                    1u
#define FR_PIFR0_CSA_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_CSA_IF_SHIFT))&FR_PIFR0_CSA_IF_MASK)
#define FR_PIFR0_ILCF_IF_MASK                    0x2000u
#define FR_PIFR0_ILCF_IF_SHIFT                   13u
#define FR_PIFR0_ILCF_IF_WIDTH                   1u
#define FR_PIFR0_ILCF_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_ILCF_IF_SHIFT))&FR_PIFR0_ILCF_IF_MASK)
#define FR_PIFR0_INTL_IF_MASK                    0x4000u
#define FR_PIFR0_INTL_IF_SHIFT                   14u
#define FR_PIFR0_INTL_IF_WIDTH                   1u
#define FR_PIFR0_INTL_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_INTL_IF_SHIFT))&FR_PIFR0_INTL_IF_MASK)
#define FR_PIFR0_FATL_IF_MASK                    0x8000u
#define FR_PIFR0_FATL_IF_SHIFT                   15u
#define FR_PIFR0_FATL_IF_WIDTH                   1u
#define FR_PIFR0_FATL_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_FATL_IF_SHIFT))&FR_PIFR0_FATL_IF_MASK)
/* PIFR1 Bit Fields */
#define FR_PIFR1_ODT_IF_MASK                     0x10u
#define FR_PIFR1_ODT_IF_SHIFT                    4u
#define FR_PIFR1_ODT_IF_WIDTH                    1u
#define FR_PIFR1_ODT_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR1_ODT_IF_SHIFT))&FR_PIFR1_ODT_IF_MASK)
#define FR_PIFR1_EVT_IF_MASK                     0x20u
#define FR_PIFR1_EVT_IF_SHIFT                    5u
#define FR_PIFR1_EVT_IF_WIDTH                    1u
#define FR_PIFR1_EVT_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR1_EVT_IF_SHIFT))&FR_PIFR1_EVT_IF_MASK)
#define FR_PIFR1_SSI0_IF_MASK                    0x100u
#define FR_PIFR1_SSI0_IF_SHIFT                   8u
#define FR_PIFR1_SSI0_IF_WIDTH                   1u
#define FR_PIFR1_SSI0_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR1_SSI0_IF_SHIFT))&FR_PIFR1_SSI0_IF_MASK)
#define FR_PIFR1_SSI1_IF_MASK                    0x200u
#define FR_PIFR1_SSI1_IF_SHIFT                   9u
#define FR_PIFR1_SSI1_IF_WIDTH                   1u
#define FR_PIFR1_SSI1_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR1_SSI1_IF_SHIFT))&FR_PIFR1_SSI1_IF_MASK)
#define FR_PIFR1_SSI2_IF_MASK                    0x400u
#define FR_PIFR1_SSI2_IF_SHIFT                   10u
#define FR_PIFR1_SSI2_IF_WIDTH                   1u
#define FR_PIFR1_SSI2_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR1_SSI2_IF_SHIFT))&FR_PIFR1_SSI2_IF_MASK)
#define FR_PIFR1_SSI3_IF_MASK                    0x800u
#define FR_PIFR1_SSI3_IF_SHIFT                   11u
#define FR_PIFR1_SSI3_IF_WIDTH                   1u
#define FR_PIFR1_SSI3_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR1_SSI3_IF_SHIFT))&FR_PIFR1_SSI3_IF_MASK)
#define FR_PIFR1_PSC_IF_MASK                     0x1000u
#define FR_PIFR1_PSC_IF_SHIFT                    12u
#define FR_PIFR1_PSC_IF_WIDTH                    1u
#define FR_PIFR1_PSC_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR1_PSC_IF_SHIFT))&FR_PIFR1_PSC_IF_MASK)
#define FR_PIFR1_PECF_IF_MASK                    0x2000u
#define FR_PIFR1_PECF_IF_SHIFT                   13u
#define FR_PIFR1_PECF_IF_WIDTH                   1u
#define FR_PIFR1_PECF_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR1_PECF_IF_SHIFT))&FR_PIFR1_PECF_IF_MASK)
#define FR_PIFR1_IPC_IF_MASK                     0x4000u
#define FR_PIFR1_IPC_IF_SHIFT                    14u
#define FR_PIFR1_IPC_IF_WIDTH                    1u
#define FR_PIFR1_IPC_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR1_IPC_IF_SHIFT))&FR_PIFR1_IPC_IF_MASK)
#define FR_PIFR1_EMC_IF_MASK                     0x8000u
#define FR_PIFR1_EMC_IF_SHIFT                    15u
#define FR_PIFR1_EMC_IF_WIDTH                    1u
#define FR_PIFR1_EMC_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR1_EMC_IF_SHIFT))&FR_PIFR1_EMC_IF_MASK)
/* PIER0 Bit Fields */
#define FR_PIER0_CYS_IE_MASK                     0x1u
#define FR_PIER0_CYS_IE_SHIFT                    0u
#define FR_PIER0_CYS_IE_WIDTH                    1u
#define FR_PIER0_CYS_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER0_CYS_IE_SHIFT))&FR_PIER0_CYS_IE_MASK)
#define FR_PIER0_TI1_IE_MASK                     0x2u
#define FR_PIER0_TI1_IE_SHIFT                    1u
#define FR_PIER0_TI1_IE_WIDTH                    1u
#define FR_PIER0_TI1_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER0_TI1_IE_SHIFT))&FR_PIER0_TI1_IE_MASK)
#define FR_PIER0_TI2_IE_MASK                     0x4u
#define FR_PIER0_TI2_IE_SHIFT                    2u
#define FR_PIER0_TI2_IE_WIDTH                    1u
#define FR_PIER0_TI2_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER0_TI2_IE_SHIFT))&FR_PIER0_TI2_IE_MASK)
#define FR_PIER0_TBVA_IE_MASK                    0x8u
#define FR_PIER0_TBVA_IE_SHIFT                   3u
#define FR_PIER0_TBVA_IE_WIDTH                   1u
#define FR_PIER0_TBVA_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER0_TBVA_IE_SHIFT))&FR_PIER0_TBVA_IE_MASK)
#define FR_PIER0_TBVB_IE_MASK                    0x10u
#define FR_PIER0_TBVB_IE_SHIFT                   4u
#define FR_PIER0_TBVB_IE_WIDTH                   1u
#define FR_PIER0_TBVB_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER0_TBVB_IE_SHIFT))&FR_PIER0_TBVB_IE_MASK)
#define FR_PIER0_LTXA_IE_MASK                    0x20u
#define FR_PIER0_LTXA_IE_SHIFT                   5u
#define FR_PIER0_LTXA_IE_WIDTH                   1u
#define FR_PIER0_LTXA_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER0_LTXA_IE_SHIFT))&FR_PIER0_LTXA_IE_MASK)
#define FR_PIER0_LTXB_IE_MASK                    0x40u
#define FR_PIER0_LTXB_IE_SHIFT                   6u
#define FR_PIER0_LTXB_IE_WIDTH                   1u
#define FR_PIER0_LTXB_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER0_LTXB_IE_SHIFT))&FR_PIER0_LTXB_IE_MASK)
#define FR_PIER0_MTX_IE_MASK                     0x80u
#define FR_PIER0_MTX_IE_SHIFT                    7u
#define FR_PIER0_MTX_IE_WIDTH                    1u
#define FR_PIER0_MTX_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER0_MTX_IE_SHIFT))&FR_PIER0_MTX_IE_MASK)
#define FR_PIER0_MXS_IE_MASK                     0x100u
#define FR_PIER0_MXS_IE_SHIFT                    8u
#define FR_PIER0_MXS_IE_WIDTH                    1u
#define FR_PIER0_MXS_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER0_MXS_IE_SHIFT))&FR_PIER0_MXS_IE_MASK)
#define FR_PIER0_CCL_IE_MASK                     0x200u
#define FR_PIER0_CCL_IE_SHIFT                    9u
#define FR_PIER0_CCL_IE_WIDTH                    1u
#define FR_PIER0_CCL_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER0_CCL_IE_SHIFT))&FR_PIER0_CCL_IE_MASK)
#define FR_PIER0_MOC_IE_MASK                     0x400u
#define FR_PIER0_MOC_IE_SHIFT                    10u
#define FR_PIER0_MOC_IE_WIDTH                    1u
#define FR_PIER0_MOC_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER0_MOC_IE_SHIFT))&FR_PIER0_MOC_IE_MASK)
#define FR_PIER0_MRC_IE_MASK                     0x800u
#define FR_PIER0_MRC_IE_SHIFT                    11u
#define FR_PIER0_MRC_IE_WIDTH                    1u
#define FR_PIER0_MRC_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER0_MRC_IE_SHIFT))&FR_PIER0_MRC_IE_MASK)
#define FR_PIER0_CSA_IE_MASK                     0x1000u
#define FR_PIER0_CSA_IE_SHIFT                    12u
#define FR_PIER0_CSA_IE_WIDTH                    1u
#define FR_PIER0_CSA_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER0_CSA_IE_SHIFT))&FR_PIER0_CSA_IE_MASK)
#define FR_PIER0_ILCF_IE_MASK                    0x2000u
#define FR_PIER0_ILCF_IE_SHIFT                   13u
#define FR_PIER0_ILCF_IE_WIDTH                   1u
#define FR_PIER0_ILCF_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER0_ILCF_IE_SHIFT))&FR_PIER0_ILCF_IE_MASK)
#define FR_PIER0_INTL_IE_MASK                    0x4000u
#define FR_PIER0_INTL_IE_SHIFT                   14u
#define FR_PIER0_INTL_IE_WIDTH                   1u
#define FR_PIER0_INTL_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER0_INTL_IE_SHIFT))&FR_PIER0_INTL_IE_MASK)
#define FR_PIER0_FATL_IE_MASK                    0x8000u
#define FR_PIER0_FATL_IE_SHIFT                   15u
#define FR_PIER0_FATL_IE_WIDTH                   1u
#define FR_PIER0_FATL_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER0_FATL_IE_SHIFT))&FR_PIER0_FATL_IE_MASK)
/* PIER1 Bit Fields */
#define FR_PIER1_ODT_IE_MASK                     0x10u
#define FR_PIER1_ODT_IE_SHIFT                    4u
#define FR_PIER1_ODT_IE_WIDTH                    1u
#define FR_PIER1_ODT_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER1_ODT_IE_SHIFT))&FR_PIER1_ODT_IE_MASK)
#define FR_PIER1_EVT_IE_MASK                     0x20u
#define FR_PIER1_EVT_IE_SHIFT                    5u
#define FR_PIER1_EVT_IE_WIDTH                    1u
#define FR_PIER1_EVT_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER1_EVT_IE_SHIFT))&FR_PIER1_EVT_IE_MASK)
#define FR_PIER1_SSI0_IE_MASK                    0x100u
#define FR_PIER1_SSI0_IE_SHIFT                   8u
#define FR_PIER1_SSI0_IE_WIDTH                   1u
#define FR_PIER1_SSI0_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER1_SSI0_IE_SHIFT))&FR_PIER1_SSI0_IE_MASK)
#define FR_PIER1_SSI1_IE_MASK                    0x200u
#define FR_PIER1_SSI1_IE_SHIFT                   9u
#define FR_PIER1_SSI1_IE_WIDTH                   1u
#define FR_PIER1_SSI1_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER1_SSI1_IE_SHIFT))&FR_PIER1_SSI1_IE_MASK)
#define FR_PIER1_SSI2_IE_MASK                    0x400u
#define FR_PIER1_SSI2_IE_SHIFT                   10u
#define FR_PIER1_SSI2_IE_WIDTH                   1u
#define FR_PIER1_SSI2_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER1_SSI2_IE_SHIFT))&FR_PIER1_SSI2_IE_MASK)
#define FR_PIER1_SSI3_IE_MASK                    0x800u
#define FR_PIER1_SSI3_IE_SHIFT                   11u
#define FR_PIER1_SSI3_IE_WIDTH                   1u
#define FR_PIER1_SSI3_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER1_SSI3_IE_SHIFT))&FR_PIER1_SSI3_IE_MASK)
#define FR_PIER1_PSC_IE_MASK                     0x1000u
#define FR_PIER1_PSC_IE_SHIFT                    12u
#define FR_PIER1_PSC_IE_WIDTH                    1u
#define FR_PIER1_PSC_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER1_PSC_IE_SHIFT))&FR_PIER1_PSC_IE_MASK)
#define FR_PIER1_PECF_IE_MASK                    0x2000u
#define FR_PIER1_PECF_IE_SHIFT                   13u
#define FR_PIER1_PECF_IE_WIDTH                   1u
#define FR_PIER1_PECF_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER1_PECF_IE_SHIFT))&FR_PIER1_PECF_IE_MASK)
#define FR_PIER1_IPC_IE_MASK                     0x4000u
#define FR_PIER1_IPC_IE_SHIFT                    14u
#define FR_PIER1_IPC_IE_WIDTH                    1u
#define FR_PIER1_IPC_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER1_IPC_IE_SHIFT))&FR_PIER1_IPC_IE_MASK)
#define FR_PIER1_EMC_IE_MASK                     0x8000u
#define FR_PIER1_EMC_IE_SHIFT                    15u
#define FR_PIER1_EMC_IE_WIDTH                    1u
#define FR_PIER1_EMC_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER1_EMC_IE_SHIFT))&FR_PIER1_EMC_IE_MASK)
/* CHIERFR Bit Fields */
#define FR_CHIERFR_ILSA_EF_MASK                  0x1u
#define FR_CHIERFR_ILSA_EF_SHIFT                 0u
#define FR_CHIERFR_ILSA_EF_WIDTH                 1u
#define FR_CHIERFR_ILSA_EF(x)                    (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_ILSA_EF_SHIFT))&FR_CHIERFR_ILSA_EF_MASK)
#define FR_CHIERFR_NMF_EF_MASK                   0x2u
#define FR_CHIERFR_NMF_EF_SHIFT                  1u
#define FR_CHIERFR_NMF_EF_WIDTH                  1u
#define FR_CHIERFR_NMF_EF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_NMF_EF_SHIFT))&FR_CHIERFR_NMF_EF_MASK)
#define FR_CHIERFR_NML_EF_MASK                   0x4u
#define FR_CHIERFR_NML_EF_SHIFT                  2u
#define FR_CHIERFR_NML_EF_WIDTH                  1u
#define FR_CHIERFR_NML_EF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_NML_EF_SHIFT))&FR_CHIERFR_NML_EF_MASK)
#define FR_CHIERFR_SPL_EF_MASK                   0x8u
#define FR_CHIERFR_SPL_EF_SHIFT                  3u
#define FR_CHIERFR_SPL_EF_WIDTH                  1u
#define FR_CHIERFR_SPL_EF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_SPL_EF_SHIFT))&FR_CHIERFR_SPL_EF_MASK)
#define FR_CHIERFR_DPL_EF_MASK                   0x10u
#define FR_CHIERFR_DPL_EF_SHIFT                  4u
#define FR_CHIERFR_DPL_EF_WIDTH                  1u
#define FR_CHIERFR_DPL_EF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_DPL_EF_SHIFT))&FR_CHIERFR_DPL_EF_MASK)
#define FR_CHIERFR_FID_EF_MASK                   0x20u
#define FR_CHIERFR_FID_EF_SHIFT                  5u
#define FR_CHIERFR_FID_EF_WIDTH                  1u
#define FR_CHIERFR_FID_EF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_FID_EF_SHIFT))&FR_CHIERFR_FID_EF_MASK)
#define FR_CHIERFR_SBCF_EF_MASK                  0x40u
#define FR_CHIERFR_SBCF_EF_SHIFT                 6u
#define FR_CHIERFR_SBCF_EF_WIDTH                 1u
#define FR_CHIERFR_SBCF_EF(x)                    (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_SBCF_EF_SHIFT))&FR_CHIERFR_SBCF_EF_MASK)
#define FR_CHIERFR_LCK_EF_MASK                   0x100u
#define FR_CHIERFR_LCK_EF_SHIFT                  8u
#define FR_CHIERFR_LCK_EF_WIDTH                  1u
#define FR_CHIERFR_LCK_EF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_LCK_EF_SHIFT))&FR_CHIERFR_LCK_EF_MASK)
#define FR_CHIERFR_MBU_EF_MASK                   0x200u
#define FR_CHIERFR_MBU_EF_SHIFT                  9u
#define FR_CHIERFR_MBU_EF_WIDTH                  1u
#define FR_CHIERFR_MBU_EF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_MBU_EF_SHIFT))&FR_CHIERFR_MBU_EF_MASK)
#define FR_CHIERFR_MBS_EF_MASK                   0x400u
#define FR_CHIERFR_MBS_EF_SHIFT                  10u
#define FR_CHIERFR_MBS_EF_WIDTH                  1u
#define FR_CHIERFR_MBS_EF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_MBS_EF_SHIFT))&FR_CHIERFR_MBS_EF_MASK)
#define FR_CHIERFR_FOVA_EF_MASK                  0x800u
#define FR_CHIERFR_FOVA_EF_SHIFT                 11u
#define FR_CHIERFR_FOVA_EF_WIDTH                 1u
#define FR_CHIERFR_FOVA_EF(x)                    (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_FOVA_EF_SHIFT))&FR_CHIERFR_FOVA_EF_MASK)
#define FR_CHIERFR_FOVB_EF_MASK                  0x1000u
#define FR_CHIERFR_FOVB_EF_SHIFT                 12u
#define FR_CHIERFR_FOVB_EF_WIDTH                 1u
#define FR_CHIERFR_FOVB_EF(x)                    (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_FOVB_EF_SHIFT))&FR_CHIERFR_FOVB_EF_MASK)
#define FR_CHIERFR_PCMI_EF_MASK                  0x2000u
#define FR_CHIERFR_PCMI_EF_SHIFT                 13u
#define FR_CHIERFR_PCMI_EF_WIDTH                 1u
#define FR_CHIERFR_PCMI_EF(x)                    (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_PCMI_EF_SHIFT))&FR_CHIERFR_PCMI_EF_MASK)
#define FR_CHIERFR_FRLA_EF_MASK                  0x4000u
#define FR_CHIERFR_FRLA_EF_SHIFT                 14u
#define FR_CHIERFR_FRLA_EF_WIDTH                 1u
#define FR_CHIERFR_FRLA_EF(x)                    (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_FRLA_EF_SHIFT))&FR_CHIERFR_FRLA_EF_MASK)
#define FR_CHIERFR_FRLB_EF_MASK                  0x8000u
#define FR_CHIERFR_FRLB_EF_SHIFT                 15u
#define FR_CHIERFR_FRLB_EF_WIDTH                 1u
#define FR_CHIERFR_FRLB_EF(x)                    (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_FRLB_EF_SHIFT))&FR_CHIERFR_FRLB_EF_MASK)
/* MBIVEC Bit Fields */
#define FR_MBIVEC_RBIVEC_MASK                    0x7Fu
#define FR_MBIVEC_RBIVEC_SHIFT                   0u
#define FR_MBIVEC_RBIVEC_WIDTH                   7u
#define FR_MBIVEC_RBIVEC(x)                      (((uint16_t)(((uint16_t)(x))<<FR_MBIVEC_RBIVEC_SHIFT))&FR_MBIVEC_RBIVEC_MASK)
#define FR_MBIVEC_TBIVEC_MASK                    0x7F00u
#define FR_MBIVEC_TBIVEC_SHIFT                   8u
#define FR_MBIVEC_TBIVEC_WIDTH                   7u
#define FR_MBIVEC_TBIVEC(x)                      (((uint16_t)(((uint16_t)(x))<<FR_MBIVEC_TBIVEC_SHIFT))&FR_MBIVEC_TBIVEC_MASK)
/* CASERCR Bit Fields */
#define FR_CASERCR_CHAERSCNT_MASK                0xFFFFu
#define FR_CASERCR_CHAERSCNT_SHIFT               0u
#define FR_CASERCR_CHAERSCNT_WIDTH               16u
#define FR_CASERCR_CHAERSCNT(x)                  (((uint16_t)(((uint16_t)(x))<<FR_CASERCR_CHAERSCNT_SHIFT))&FR_CASERCR_CHAERSCNT_MASK)
/* CBSERCR Bit Fields */
#define FR_CBSERCR_CHBERSCNT_MASK                0xFFFFu
#define FR_CBSERCR_CHBERSCNT_SHIFT               0u
#define FR_CBSERCR_CHBERSCNT_WIDTH               16u
#define FR_CBSERCR_CHBERSCNT(x)                  (((uint16_t)(((uint16_t)(x))<<FR_CBSERCR_CHBERSCNT_SHIFT))&FR_CBSERCR_CHBERSCNT_MASK)
/* PSR0 Bit Fields */
#define FR_PSR0_WAKEUPSTATUS_MASK                0x7u
#define FR_PSR0_WAKEUPSTATUS_SHIFT               0u
#define FR_PSR0_WAKEUPSTATUS_WIDTH               3u
#define FR_PSR0_WAKEUPSTATUS(x)                  (((uint16_t)(((uint16_t)(x))<<FR_PSR0_WAKEUPSTATUS_SHIFT))&FR_PSR0_WAKEUPSTATUS_MASK)
#define FR_PSR0_STARTUPSTATE_MASK                0xF0u
#define FR_PSR0_STARTUPSTATE_SHIFT               4u
#define FR_PSR0_STARTUPSTATE_WIDTH               4u
#define FR_PSR0_STARTUPSTATE(x)                  (((uint16_t)(((uint16_t)(x))<<FR_PSR0_STARTUPSTATE_SHIFT))&FR_PSR0_STARTUPSTATE_MASK)
#define FR_PSR0_PROTSTATE_MASK                   0x700u
#define FR_PSR0_PROTSTATE_SHIFT                  8u
#define FR_PSR0_PROTSTATE_WIDTH                  3u
#define FR_PSR0_PROTSTATE(x)                     (((uint16_t)(((uint16_t)(x))<<FR_PSR0_PROTSTATE_SHIFT))&FR_PSR0_PROTSTATE_MASK)
#define FR_PSR0_SLOTMODE_MASK                    0x3000u
#define FR_PSR0_SLOTMODE_SHIFT                   12u
#define FR_PSR0_SLOTMODE_WIDTH                   2u
#define FR_PSR0_SLOTMODE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PSR0_SLOTMODE_SHIFT))&FR_PSR0_SLOTMODE_MASK)
#define FR_PSR0_ERRMODE_MASK                     0xC000u
#define FR_PSR0_ERRMODE_SHIFT                    14u
#define FR_PSR0_ERRMODE_WIDTH                    2u
#define FR_PSR0_ERRMODE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PSR0_ERRMODE_SHIFT))&FR_PSR0_ERRMODE_MASK)
/* PSR1 Bit Fields */
#define FR_PSR1_APTAC_MASK                       0x1Fu
#define FR_PSR1_APTAC_SHIFT                      0u
#define FR_PSR1_APTAC_WIDTH                      5u
#define FR_PSR1_APTAC(x)                         (((uint16_t)(((uint16_t)(x))<<FR_PSR1_APTAC_SHIFT))&FR_PSR1_APTAC_MASK)
#define FR_PSR1_FRZ_MASK                         0x20u
#define FR_PSR1_FRZ_SHIFT                        5u
#define FR_PSR1_FRZ_WIDTH                        1u
#define FR_PSR1_FRZ(x)                           (((uint16_t)(((uint16_t)(x))<<FR_PSR1_FRZ_SHIFT))&FR_PSR1_FRZ_MASK)
#define FR_PSR1_HHR_MASK                         0x40u
#define FR_PSR1_HHR_SHIFT                        6u
#define FR_PSR1_HHR_WIDTH                        1u
#define FR_PSR1_HHR(x)                           (((uint16_t)(((uint16_t)(x))<<FR_PSR1_HHR_SHIFT))&FR_PSR1_HHR_MASK)
#define FR_PSR1_CPN_MASK                         0x80u
#define FR_PSR1_CPN_SHIFT                        7u
#define FR_PSR1_CPN_WIDTH                        1u
#define FR_PSR1_CPN(x)                           (((uint16_t)(((uint16_t)(x))<<FR_PSR1_CPN_SHIFT))&FR_PSR1_CPN_MASK)
#define FR_PSR1_REMCSAT_MASK                     0x1F00u
#define FR_PSR1_REMCSAT_SHIFT                    8u
#define FR_PSR1_REMCSAT_WIDTH                    5u
#define FR_PSR1_REMCSAT(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PSR1_REMCSAT_SHIFT))&FR_PSR1_REMCSAT_MASK)
#define FR_PSR1_CSP_MASK                         0x4000u
#define FR_PSR1_CSP_SHIFT                        14u
#define FR_PSR1_CSP_WIDTH                        1u
#define FR_PSR1_CSP(x)                           (((uint16_t)(((uint16_t)(x))<<FR_PSR1_CSP_SHIFT))&FR_PSR1_CSP_MASK)
#define FR_PSR1_CSAA_MASK                        0x8000u
#define FR_PSR1_CSAA_SHIFT                       15u
#define FR_PSR1_CSAA_WIDTH                       1u
#define FR_PSR1_CSAA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR1_CSAA_SHIFT))&FR_PSR1_CSAA_MASK)
/* PSR2 Bit Fields */
#define FR_PSR2_CKCORFCNT_MASK                   0xFu
#define FR_PSR2_CKCORFCNT_SHIFT                  0u
#define FR_PSR2_CKCORFCNT_WIDTH                  4u
#define FR_PSR2_CKCORFCNT(x)                     (((uint16_t)(((uint16_t)(x))<<FR_PSR2_CKCORFCNT_SHIFT))&FR_PSR2_CKCORFCNT_MASK)
#define FR_PSR2_MTA_MASK                         0x10u
#define FR_PSR2_MTA_SHIFT                        4u
#define FR_PSR2_MTA_WIDTH                        1u
#define FR_PSR2_MTA(x)                           (((uint16_t)(((uint16_t)(x))<<FR_PSR2_MTA_SHIFT))&FR_PSR2_MTA_MASK)
#define FR_PSR2_SSEA_MASK                        0x20u
#define FR_PSR2_SSEA_SHIFT                       5u
#define FR_PSR2_SSEA_WIDTH                       1u
#define FR_PSR2_SSEA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR2_SSEA_SHIFT))&FR_PSR2_SSEA_MASK)
#define FR_PSR2_SBVA_MASK                        0x40u
#define FR_PSR2_SBVA_SHIFT                       6u
#define FR_PSR2_SBVA_WIDTH                       1u
#define FR_PSR2_SBVA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR2_SBVA_SHIFT))&FR_PSR2_SBVA_MASK)
#define FR_PSR2_STCA_MASK                        0x80u
#define FR_PSR2_STCA_SHIFT                       7u
#define FR_PSR2_STCA_WIDTH                       1u
#define FR_PSR2_STCA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR2_STCA_SHIFT))&FR_PSR2_STCA_MASK)
#define FR_PSR2_NSEA_MASK                        0x100u
#define FR_PSR2_NSEA_SHIFT                       8u
#define FR_PSR2_NSEA_WIDTH                       1u
#define FR_PSR2_NSEA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR2_NSEA_SHIFT))&FR_PSR2_NSEA_MASK)
#define FR_PSR2_NBVA_MASK                        0x200u
#define FR_PSR2_NBVA_SHIFT                       9u
#define FR_PSR2_NBVA_WIDTH                       1u
#define FR_PSR2_NBVA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR2_NBVA_SHIFT))&FR_PSR2_NBVA_MASK)
#define FR_PSR2_MTB_MASK                         0x400u
#define FR_PSR2_MTB_SHIFT                        10u
#define FR_PSR2_MTB_WIDTH                        1u
#define FR_PSR2_MTB(x)                           (((uint16_t)(((uint16_t)(x))<<FR_PSR2_MTB_SHIFT))&FR_PSR2_MTB_MASK)
#define FR_PSR2_SSEB_MASK                        0x800u
#define FR_PSR2_SSEB_SHIFT                       11u
#define FR_PSR2_SSEB_WIDTH                       1u
#define FR_PSR2_SSEB(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR2_SSEB_SHIFT))&FR_PSR2_SSEB_MASK)
#define FR_PSR2_SBVB_MASK                        0x1000u
#define FR_PSR2_SBVB_SHIFT                       12u
#define FR_PSR2_SBVB_WIDTH                       1u
#define FR_PSR2_SBVB(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR2_SBVB_SHIFT))&FR_PSR2_SBVB_MASK)
#define FR_PSR2_STCB_MASK                        0x2000u
#define FR_PSR2_STCB_SHIFT                       13u
#define FR_PSR2_STCB_WIDTH                       1u
#define FR_PSR2_STCB(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR2_STCB_SHIFT))&FR_PSR2_STCB_MASK)
#define FR_PSR2_NSEB_MASK                        0x4000u
#define FR_PSR2_NSEB_SHIFT                       14u
#define FR_PSR2_NSEB_WIDTH                       1u
#define FR_PSR2_NSEB(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR2_NSEB_SHIFT))&FR_PSR2_NSEB_MASK)
#define FR_PSR2_NBVB_MASK                        0x8000u
#define FR_PSR2_NBVB_SHIFT                       15u
#define FR_PSR2_NBVB_WIDTH                       1u
#define FR_PSR2_NBVB(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR2_NBVB_SHIFT))&FR_PSR2_NBVB_MASK)
/* PSR3 Bit Fields */
#define FR_PSR3_AVFA_MASK                        0x1u
#define FR_PSR3_AVFA_SHIFT                       0u
#define FR_PSR3_AVFA_WIDTH                       1u
#define FR_PSR3_AVFA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR3_AVFA_SHIFT))&FR_PSR3_AVFA_MASK)
#define FR_PSR3_ASEA_MASK                        0x2u
#define FR_PSR3_ASEA_SHIFT                       1u
#define FR_PSR3_ASEA_WIDTH                       1u
#define FR_PSR3_ASEA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR3_ASEA_SHIFT))&FR_PSR3_ASEA_MASK)
#define FR_PSR3_ACEA_MASK                        0x4u
#define FR_PSR3_ACEA_SHIFT                       2u
#define FR_PSR3_ACEA_WIDTH                       1u
#define FR_PSR3_ACEA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR3_ACEA_SHIFT))&FR_PSR3_ACEA_MASK)
#define FR_PSR3_AACA_MASK                        0x8u
#define FR_PSR3_AACA_SHIFT                       3u
#define FR_PSR3_AACA_WIDTH                       1u
#define FR_PSR3_AACA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR3_AACA_SHIFT))&FR_PSR3_AACA_MASK)
#define FR_PSR3_ABVA_MASK                        0x10u
#define FR_PSR3_ABVA_SHIFT                       4u
#define FR_PSR3_ABVA_WIDTH                       1u
#define FR_PSR3_ABVA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR3_ABVA_SHIFT))&FR_PSR3_ABVA_MASK)
#define FR_PSR3_WUA_MASK                         0x20u
#define FR_PSR3_WUA_SHIFT                        5u
#define FR_PSR3_WUA_WIDTH                        1u
#define FR_PSR3_WUA(x)                           (((uint16_t)(((uint16_t)(x))<<FR_PSR3_WUA_SHIFT))&FR_PSR3_WUA_MASK)
#define FR_PSR3_AVFB_MASK                        0x100u
#define FR_PSR3_AVFB_SHIFT                       8u
#define FR_PSR3_AVFB_WIDTH                       1u
#define FR_PSR3_AVFB(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR3_AVFB_SHIFT))&FR_PSR3_AVFB_MASK)
#define FR_PSR3_ASEB_MASK                        0x200u
#define FR_PSR3_ASEB_SHIFT                       9u
#define FR_PSR3_ASEB_WIDTH                       1u
#define FR_PSR3_ASEB(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR3_ASEB_SHIFT))&FR_PSR3_ASEB_MASK)
#define FR_PSR3_ACEB_MASK                        0x400u
#define FR_PSR3_ACEB_SHIFT                       10u
#define FR_PSR3_ACEB_WIDTH                       1u
#define FR_PSR3_ACEB(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR3_ACEB_SHIFT))&FR_PSR3_ACEB_MASK)
#define FR_PSR3_AACB_MASK                        0x800u
#define FR_PSR3_AACB_SHIFT                       11u
#define FR_PSR3_AACB_WIDTH                       1u
#define FR_PSR3_AACB(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR3_AACB_SHIFT))&FR_PSR3_AACB_MASK)
#define FR_PSR3_ABVB_MASK                        0x1000u
#define FR_PSR3_ABVB_SHIFT                       12u
#define FR_PSR3_ABVB_WIDTH                       1u
#define FR_PSR3_ABVB(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR3_ABVB_SHIFT))&FR_PSR3_ABVB_MASK)
#define FR_PSR3_WUB_MASK                         0x2000u
#define FR_PSR3_WUB_SHIFT                        13u
#define FR_PSR3_WUB_WIDTH                        1u
#define FR_PSR3_WUB(x)                           (((uint16_t)(((uint16_t)(x))<<FR_PSR3_WUB_SHIFT))&FR_PSR3_WUB_MASK)
/* MTCTR Bit Fields */
#define FR_MTCTR_MTCT_MASK                       0x3FFFu
#define FR_MTCTR_MTCT_SHIFT                      0u
#define FR_MTCTR_MTCT_WIDTH                      14u
#define FR_MTCTR_MTCT(x)                         (((uint16_t)(((uint16_t)(x))<<FR_MTCTR_MTCT_SHIFT))&FR_MTCTR_MTCT_MASK)
/* CYCTR Bit Fields */
#define FR_CYCTR_CYCCNT_MASK                     0x3Fu
#define FR_CYCTR_CYCCNT_SHIFT                    0u
#define FR_CYCTR_CYCCNT_WIDTH                    6u
#define FR_CYCTR_CYCCNT(x)                       (((uint16_t)(((uint16_t)(x))<<FR_CYCTR_CYCCNT_SHIFT))&FR_CYCTR_CYCCNT_MASK)
/* SLTCTAR Bit Fields */
#define FR_SLTCTAR_SLOTCNTA_MASK                 0x7FFu
#define FR_SLTCTAR_SLOTCNTA_SHIFT                0u
#define FR_SLTCTAR_SLOTCNTA_WIDTH                11u
#define FR_SLTCTAR_SLOTCNTA(x)                   (((uint16_t)(((uint16_t)(x))<<FR_SLTCTAR_SLOTCNTA_SHIFT))&FR_SLTCTAR_SLOTCNTA_MASK)
/* SLTCTBR Bit Fields */
#define FR_SLTCTBR_SLOTCNTB_MASK                 0x7FFu
#define FR_SLTCTBR_SLOTCNTB_SHIFT                0u
#define FR_SLTCTBR_SLOTCNTB_WIDTH                11u
#define FR_SLTCTBR_SLOTCNTB(x)                   (((uint16_t)(((uint16_t)(x))<<FR_SLTCTBR_SLOTCNTB_SHIFT))&FR_SLTCTBR_SLOTCNTB_MASK)
/* RTCORVR Bit Fields */
#define FR_RTCORVR_RATECORR_MASK                 0xFFFFu
#define FR_RTCORVR_RATECORR_SHIFT                0u
#define FR_RTCORVR_RATECORR_WIDTH                16u
#define FR_RTCORVR_RATECORR(x)                   (((uint16_t)(((uint16_t)(x))<<FR_RTCORVR_RATECORR_SHIFT))&FR_RTCORVR_RATECORR_MASK)
/* OFCORVR Bit Fields */
#define FR_OFCORVR_OFFSETCORR_MASK               0xFFFFu
#define FR_OFCORVR_OFFSETCORR_SHIFT              0u
#define FR_OFCORVR_OFFSETCORR_WIDTH              16u
#define FR_OFCORVR_OFFSETCORR(x)                 (((uint16_t)(((uint16_t)(x))<<FR_OFCORVR_OFFSETCORR_SHIFT))&FR_OFCORVR_OFFSETCORR_MASK)
/* CIFR Bit Fields */
#define FR_CIFR_TBIF_MASK                        0x1u
#define FR_CIFR_TBIF_SHIFT                       0u
#define FR_CIFR_TBIF_WIDTH                       1u
#define FR_CIFR_TBIF(x)                          (((uint16_t)(((uint16_t)(x))<<FR_CIFR_TBIF_SHIFT))&FR_CIFR_TBIF_MASK)
#define FR_CIFR_RBIF_MASK                        0x2u
#define FR_CIFR_RBIF_SHIFT                       1u
#define FR_CIFR_RBIF_WIDTH                       1u
#define FR_CIFR_RBIF(x)                          (((uint16_t)(((uint16_t)(x))<<FR_CIFR_RBIF_SHIFT))&FR_CIFR_RBIF_MASK)
#define FR_CIFR_FAFAIF_MASK                      0x4u
#define FR_CIFR_FAFAIF_SHIFT                     2u
#define FR_CIFR_FAFAIF_WIDTH                     1u
#define FR_CIFR_FAFAIF(x)                        (((uint16_t)(((uint16_t)(x))<<FR_CIFR_FAFAIF_SHIFT))&FR_CIFR_FAFAIF_MASK)
#define FR_CIFR_FAFBIF_MASK                      0x8u
#define FR_CIFR_FAFBIF_SHIFT                     3u
#define FR_CIFR_FAFBIF_WIDTH                     1u
#define FR_CIFR_FAFBIF(x)                        (((uint16_t)(((uint16_t)(x))<<FR_CIFR_FAFBIF_SHIFT))&FR_CIFR_FAFBIF_MASK)
#define FR_CIFR_WUPIF_MASK                       0x10u
#define FR_CIFR_WUPIF_SHIFT                      4u
#define FR_CIFR_WUPIF_WIDTH                      1u
#define FR_CIFR_WUPIF(x)                         (((uint16_t)(((uint16_t)(x))<<FR_CIFR_WUPIF_SHIFT))&FR_CIFR_WUPIF_MASK)
#define FR_CIFR_CHIF_MASK                        0x20u
#define FR_CIFR_CHIF_SHIFT                       5u
#define FR_CIFR_CHIF_WIDTH                       1u
#define FR_CIFR_CHIF(x)                          (((uint16_t)(((uint16_t)(x))<<FR_CIFR_CHIF_SHIFT))&FR_CIFR_CHIF_MASK)
#define FR_CIFR_PRIF_MASK                        0x40u
#define FR_CIFR_PRIF_SHIFT                       6u
#define FR_CIFR_PRIF_WIDTH                       1u
#define FR_CIFR_PRIF(x)                          (((uint16_t)(((uint16_t)(x))<<FR_CIFR_PRIF_SHIFT))&FR_CIFR_PRIF_MASK)
#define FR_CIFR_MIF_MASK                         0x80u
#define FR_CIFR_MIF_SHIFT                        7u
#define FR_CIFR_MIF_WIDTH                        1u
#define FR_CIFR_MIF(x)                           (((uint16_t)(((uint16_t)(x))<<FR_CIFR_MIF_SHIFT))&FR_CIFR_MIF_MASK)
/* SYMATOR Bit Fields */
#define FR_SYMATOR_TIMEOUT_MASK                  0xFFu
#define FR_SYMATOR_TIMEOUT_SHIFT                 0u
#define FR_SYMATOR_TIMEOUT_WIDTH                 8u
#define FR_SYMATOR_TIMEOUT(x)                    (((uint16_t)(((uint16_t)(x))<<FR_SYMATOR_TIMEOUT_SHIFT))&FR_SYMATOR_TIMEOUT_MASK)
/* SFCNTR Bit Fields */
#define FR_SFCNTR_SFODA_MASK                     0xFu
#define FR_SFCNTR_SFODA_SHIFT                    0u
#define FR_SFCNTR_SFODA_WIDTH                    4u
#define FR_SFCNTR_SFODA(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SFCNTR_SFODA_SHIFT))&FR_SFCNTR_SFODA_MASK)
#define FR_SFCNTR_SFODB_MASK                     0xF0u
#define FR_SFCNTR_SFODB_SHIFT                    4u
#define FR_SFCNTR_SFODB_WIDTH                    4u
#define FR_SFCNTR_SFODB(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SFCNTR_SFODB_SHIFT))&FR_SFCNTR_SFODB_MASK)
#define FR_SFCNTR_SFEVA_MASK                     0xF00u
#define FR_SFCNTR_SFEVA_SHIFT                    8u
#define FR_SFCNTR_SFEVA_WIDTH                    4u
#define FR_SFCNTR_SFEVA(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SFCNTR_SFEVA_SHIFT))&FR_SFCNTR_SFEVA_MASK)
#define FR_SFCNTR_SFEVB_MASK                     0xF000u
#define FR_SFCNTR_SFEVB_SHIFT                    12u
#define FR_SFCNTR_SFEVB_WIDTH                    4u
#define FR_SFCNTR_SFEVB(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SFCNTR_SFEVB_SHIFT))&FR_SFCNTR_SFEVB_MASK)
/* SFTOR Bit Fields */
#define FR_SFTOR_SFT_OFFSET_MASK                 0xFFFEu
#define FR_SFTOR_SFT_OFFSET_SHIFT                1u
#define FR_SFTOR_SFT_OFFSET_WIDTH                15u
#define FR_SFTOR_SFT_OFFSET(x)                   (((uint16_t)(((uint16_t)(x))<<FR_SFTOR_SFT_OFFSET_SHIFT))&FR_SFTOR_SFT_OFFSET_MASK)
/* SFTCCSR Bit Fields */
#define FR_SFTCCSR_SIDEN_MASK                    0x1u
#define FR_SFTCCSR_SIDEN_SHIFT                   0u
#define FR_SFTCCSR_SIDEN_WIDTH                   1u
#define FR_SFTCCSR_SIDEN(x)                      (((uint16_t)(((uint16_t)(x))<<FR_SFTCCSR_SIDEN_SHIFT))&FR_SFTCCSR_SIDEN_MASK)
#define FR_SFTCCSR_SDVEN_MASK                    0x2u
#define FR_SFTCCSR_SDVEN_SHIFT                   1u
#define FR_SFTCCSR_SDVEN_WIDTH                   1u
#define FR_SFTCCSR_SDVEN(x)                      (((uint16_t)(((uint16_t)(x))<<FR_SFTCCSR_SDVEN_SHIFT))&FR_SFTCCSR_SDVEN_MASK)
#define FR_SFTCCSR_OPT_MASK                      0x4u
#define FR_SFTCCSR_OPT_SHIFT                     2u
#define FR_SFTCCSR_OPT_WIDTH                     1u
#define FR_SFTCCSR_OPT(x)                        (((uint16_t)(((uint16_t)(x))<<FR_SFTCCSR_OPT_SHIFT))&FR_SFTCCSR_OPT_MASK)
#define FR_SFTCCSR_OVAL_MASK                     0x10u
#define FR_SFTCCSR_OVAL_SHIFT                    4u
#define FR_SFTCCSR_OVAL_WIDTH                    1u
#define FR_SFTCCSR_OVAL(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SFTCCSR_OVAL_SHIFT))&FR_SFTCCSR_OVAL_MASK)
#define FR_SFTCCSR_EVAL_MASK                     0x20u
#define FR_SFTCCSR_EVAL_SHIFT                    5u
#define FR_SFTCCSR_EVAL_WIDTH                    1u
#define FR_SFTCCSR_EVAL(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SFTCCSR_EVAL_SHIFT))&FR_SFTCCSR_EVAL_MASK)
#define FR_SFTCCSR_OLKS_MASK                     0x40u
#define FR_SFTCCSR_OLKS_SHIFT                    6u
#define FR_SFTCCSR_OLKS_WIDTH                    1u
#define FR_SFTCCSR_OLKS(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SFTCCSR_OLKS_SHIFT))&FR_SFTCCSR_OLKS_MASK)
#define FR_SFTCCSR_ELKS_MASK                     0x80u
#define FR_SFTCCSR_ELKS_SHIFT                    7u
#define FR_SFTCCSR_ELKS_WIDTH                    1u
#define FR_SFTCCSR_ELKS(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SFTCCSR_ELKS_SHIFT))&FR_SFTCCSR_ELKS_MASK)
#define FR_SFTCCSR_CYCNUM_MASK                   0x3F00u
#define FR_SFTCCSR_CYCNUM_SHIFT                  8u
#define FR_SFTCCSR_CYCNUM_WIDTH                  6u
#define FR_SFTCCSR_CYCNUM(x)                     (((uint16_t)(((uint16_t)(x))<<FR_SFTCCSR_CYCNUM_SHIFT))&FR_SFTCCSR_CYCNUM_MASK)
#define FR_SFTCCSR_OLKT_MASK                     0x4000u
#define FR_SFTCCSR_OLKT_SHIFT                    14u
#define FR_SFTCCSR_OLKT_WIDTH                    1u
#define FR_SFTCCSR_OLKT(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SFTCCSR_OLKT_SHIFT))&FR_SFTCCSR_OLKT_MASK)
#define FR_SFTCCSR_ELKT_MASK                     0x8000u
#define FR_SFTCCSR_ELKT_SHIFT                    15u
#define FR_SFTCCSR_ELKT_WIDTH                    1u
#define FR_SFTCCSR_ELKT(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SFTCCSR_ELKT_SHIFT))&FR_SFTCCSR_ELKT_MASK)
/* SFIDRFR Bit Fields */
#define FR_SFIDRFR_SYNFRID_MASK                  0x3FFu
#define FR_SFIDRFR_SYNFRID_SHIFT                 0u
#define FR_SFIDRFR_SYNFRID_WIDTH                 10u
#define FR_SFIDRFR_SYNFRID(x)                    (((uint16_t)(((uint16_t)(x))<<FR_SFIDRFR_SYNFRID_SHIFT))&FR_SFIDRFR_SYNFRID_MASK)
/* SFIDAFVR Bit Fields */
#define FR_SFIDAFVR_FVAL_MASK                    0x3FFu
#define FR_SFIDAFVR_FVAL_SHIFT                   0u
#define FR_SFIDAFVR_FVAL_WIDTH                   10u
#define FR_SFIDAFVR_FVAL(x)                      (((uint16_t)(((uint16_t)(x))<<FR_SFIDAFVR_FVAL_SHIFT))&FR_SFIDAFVR_FVAL_MASK)
/* SFIDAFMR Bit Fields */
#define FR_SFIDAFMR_FMSK_MASK                    0x3FFu
#define FR_SFIDAFMR_FMSK_SHIFT                   0u
#define FR_SFIDAFMR_FMSK_WIDTH                   10u
#define FR_SFIDAFMR_FMSK(x)                      (((uint16_t)(((uint16_t)(x))<<FR_SFIDAFMR_FMSK_SHIFT))&FR_SFIDAFMR_FMSK_MASK)
/* NMVR Bit Fields */
#define FR_NMVR_NMVP_MASK                        0xFFFFu
#define FR_NMVR_NMVP_SHIFT                       0u
#define FR_NMVR_NMVP_WIDTH                       16u
#define FR_NMVR_NMVP(x)                          (((uint16_t)(((uint16_t)(x))<<FR_NMVR_NMVP_SHIFT))&FR_NMVR_NMVP_MASK)
/* NMVLR Bit Fields */
#define FR_NMVLR_NMVL_MASK                       0xFu
#define FR_NMVLR_NMVL_SHIFT                      0u
#define FR_NMVLR_NMVL_WIDTH                      4u
#define FR_NMVLR_NMVL(x)                         (((uint16_t)(((uint16_t)(x))<<FR_NMVLR_NMVL_SHIFT))&FR_NMVLR_NMVL_MASK)
/* TICCR Bit Fields */
#define FR_TICCR_T1ST_MASK                       0x1u
#define FR_TICCR_T1ST_SHIFT                      0u
#define FR_TICCR_T1ST_WIDTH                      1u
#define FR_TICCR_T1ST(x)                         (((uint16_t)(((uint16_t)(x))<<FR_TICCR_T1ST_SHIFT))&FR_TICCR_T1ST_MASK)
#define FR_TICCR_T1TR_MASK                       0x2u
#define FR_TICCR_T1TR_SHIFT                      1u
#define FR_TICCR_T1TR_WIDTH                      1u
#define FR_TICCR_T1TR(x)                         (((uint16_t)(((uint16_t)(x))<<FR_TICCR_T1TR_SHIFT))&FR_TICCR_T1TR_MASK)
#define FR_TICCR_T1SP_MASK                       0x4u
#define FR_TICCR_T1SP_SHIFT                      2u
#define FR_TICCR_T1SP_WIDTH                      1u
#define FR_TICCR_T1SP(x)                         (((uint16_t)(((uint16_t)(x))<<FR_TICCR_T1SP_SHIFT))&FR_TICCR_T1SP_MASK)
#define FR_TICCR_T1_REP_MASK                     0x10u
#define FR_TICCR_T1_REP_SHIFT                    4u
#define FR_TICCR_T1_REP_WIDTH                    1u
#define FR_TICCR_T1_REP(x)                       (((uint16_t)(((uint16_t)(x))<<FR_TICCR_T1_REP_SHIFT))&FR_TICCR_T1_REP_MASK)
#define FR_TICCR_T2ST_MASK                       0x100u
#define FR_TICCR_T2ST_SHIFT                      8u
#define FR_TICCR_T2ST_WIDTH                      1u
#define FR_TICCR_T2ST(x)                         (((uint16_t)(((uint16_t)(x))<<FR_TICCR_T2ST_SHIFT))&FR_TICCR_T2ST_MASK)
#define FR_TICCR_T2TR_MASK                       0x200u
#define FR_TICCR_T2TR_SHIFT                      9u
#define FR_TICCR_T2TR_WIDTH                      1u
#define FR_TICCR_T2TR(x)                         (((uint16_t)(((uint16_t)(x))<<FR_TICCR_T2TR_SHIFT))&FR_TICCR_T2TR_MASK)
#define FR_TICCR_T2SP_MASK                       0x400u
#define FR_TICCR_T2SP_SHIFT                      10u
#define FR_TICCR_T2SP_WIDTH                      1u
#define FR_TICCR_T2SP(x)                         (((uint16_t)(((uint16_t)(x))<<FR_TICCR_T2SP_SHIFT))&FR_TICCR_T2SP_MASK)
#define FR_TICCR_T2_REP_MASK                     0x1000u
#define FR_TICCR_T2_REP_SHIFT                    12u
#define FR_TICCR_T2_REP_WIDTH                    1u
#define FR_TICCR_T2_REP(x)                       (((uint16_t)(((uint16_t)(x))<<FR_TICCR_T2_REP_SHIFT))&FR_TICCR_T2_REP_MASK)
#define FR_TICCR_T2_CFG_MASK                     0x2000u
#define FR_TICCR_T2_CFG_SHIFT                    13u
#define FR_TICCR_T2_CFG_WIDTH                    1u
#define FR_TICCR_T2_CFG(x)                       (((uint16_t)(((uint16_t)(x))<<FR_TICCR_T2_CFG_SHIFT))&FR_TICCR_T2_CFG_MASK)
/* TI1CYSR Bit Fields */
#define FR_TI1CYSR_T1_CYC_MSK_MASK               0x3Fu
#define FR_TI1CYSR_T1_CYC_MSK_SHIFT              0u
#define FR_TI1CYSR_T1_CYC_MSK_WIDTH              6u
#define FR_TI1CYSR_T1_CYC_MSK(x)                 (((uint16_t)(((uint16_t)(x))<<FR_TI1CYSR_T1_CYC_MSK_SHIFT))&FR_TI1CYSR_T1_CYC_MSK_MASK)
#define FR_TI1CYSR_T1_CYC_VAL_MASK               0x3F00u
#define FR_TI1CYSR_T1_CYC_VAL_SHIFT              8u
#define FR_TI1CYSR_T1_CYC_VAL_WIDTH              6u
#define FR_TI1CYSR_T1_CYC_VAL(x)                 (((uint16_t)(((uint16_t)(x))<<FR_TI1CYSR_T1_CYC_VAL_SHIFT))&FR_TI1CYSR_T1_CYC_VAL_MASK)
/* TI1MTOR Bit Fields */
#define FR_TI1MTOR_T1_MTOFFSET_MASK              0x3FFFu
#define FR_TI1MTOR_T1_MTOFFSET_SHIFT             0u
#define FR_TI1MTOR_T1_MTOFFSET_WIDTH             14u
#define FR_TI1MTOR_T1_MTOFFSET(x)                (((uint16_t)(((uint16_t)(x))<<FR_TI1MTOR_T1_MTOFFSET_SHIFT))&FR_TI1MTOR_T1_MTOFFSET_MASK)
/* TI2CR0_ABS Bit Fields */
#define FR_TI2CR0_ABS_T2CYCMSK_MASK              0x3Fu
#define FR_TI2CR0_ABS_T2CYCMSK_SHIFT             0u
#define FR_TI2CR0_ABS_T2CYCMSK_WIDTH             6u
#define FR_TI2CR0_ABS_T2CYCMSK(x)                (((uint16_t)(((uint16_t)(x))<<FR_TI2CR0_ABS_T2CYCMSK_SHIFT))&FR_TI2CR0_ABS_T2CYCMSK_MASK)
#define FR_TI2CR0_ABS_T2CYCVAL_MASK              0x3F00u
#define FR_TI2CR0_ABS_T2CYCVAL_SHIFT             8u
#define FR_TI2CR0_ABS_T2CYCVAL_WIDTH             6u
#define FR_TI2CR0_ABS_T2CYCVAL(x)                (((uint16_t)(((uint16_t)(x))<<FR_TI2CR0_ABS_T2CYCVAL_SHIFT))&FR_TI2CR0_ABS_T2CYCVAL_MASK)
/* TI2CR0_REL Bit Fields */
#define FR_TI2CR0_REL_T2MTCNT_MASK               0xFFFFu
#define FR_TI2CR0_REL_T2MTCNT_SHIFT              0u
#define FR_TI2CR0_REL_T2MTCNT_WIDTH              16u
#define FR_TI2CR0_REL_T2MTCNT(x)                 (((uint16_t)(((uint16_t)(x))<<FR_TI2CR0_REL_T2MTCNT_SHIFT))&FR_TI2CR0_REL_T2MTCNT_MASK)
/* TI2CR1_ABS Bit Fields */
#define FR_TI2CR1_ABS_T2MOFF_MASK                0x3FFFu
#define FR_TI2CR1_ABS_T2MOFF_SHIFT               0u
#define FR_TI2CR1_ABS_T2MOFF_WIDTH               14u
#define FR_TI2CR1_ABS_T2MOFF(x)                  (((uint16_t)(((uint16_t)(x))<<FR_TI2CR1_ABS_T2MOFF_SHIFT))&FR_TI2CR1_ABS_T2MOFF_MASK)
/* TI2CR1_REL Bit Fields */
#define FR_TI2CR1_REL_T2MTCNT_MASK               0xFFFFu
#define FR_TI2CR1_REL_T2MTCNT_SHIFT              0u
#define FR_TI2CR1_REL_T2MTCNT_WIDTH              16u
#define FR_TI2CR1_REL_T2MTCNT(x)                 (((uint16_t)(((uint16_t)(x))<<FR_TI2CR1_REL_T2MTCNT_SHIFT))&FR_TI2CR1_REL_T2MTCNT_MASK)
/* SSSR Bit Fields */
#define FR_SSSR_SLOTNUMBER_MASK                  0x7FFu
#define FR_SSSR_SLOTNUMBER_SHIFT                 0u
#define FR_SSSR_SLOTNUMBER_WIDTH                 11u
#define FR_SSSR_SLOTNUMBER(x)                    (((uint16_t)(((uint16_t)(x))<<FR_SSSR_SLOTNUMBER_SHIFT))&FR_SSSR_SLOTNUMBER_MASK)
#define FR_SSSR_SEL_MASK                         0x3000u
#define FR_SSSR_SEL_SHIFT                        12u
#define FR_SSSR_SEL_WIDTH                        2u
#define FR_SSSR_SEL(x)                           (((uint16_t)(((uint16_t)(x))<<FR_SSSR_SEL_SHIFT))&FR_SSSR_SEL_MASK)
#define FR_SSSR_WMD_MASK                         0x8000u
#define FR_SSSR_WMD_SHIFT                        15u
#define FR_SSSR_WMD_WIDTH                        1u
#define FR_SSSR_WMD(x)                           (((uint16_t)(((uint16_t)(x))<<FR_SSSR_WMD_SHIFT))&FR_SSSR_WMD_MASK)
/* SSCCR Bit Fields */
#define FR_SSCCR_STATUSMASK_MASK                 0xFu
#define FR_SSCCR_STATUSMASK_SHIFT                0u
#define FR_SSCCR_STATUSMASK_WIDTH                4u
#define FR_SSCCR_STATUSMASK(x)                   (((uint16_t)(((uint16_t)(x))<<FR_SSCCR_STATUSMASK_SHIFT))&FR_SSCCR_STATUSMASK_MASK)
#define FR_SSCCR_SUF_MASK                        0x10u
#define FR_SSCCR_SUF_SHIFT                       4u
#define FR_SSCCR_SUF_WIDTH                       1u
#define FR_SSCCR_SUF(x)                          (((uint16_t)(((uint16_t)(x))<<FR_SSCCR_SUF_SHIFT))&FR_SSCCR_SUF_MASK)
#define FR_SSCCR_NUF_MASK                        0x20u
#define FR_SSCCR_NUF_SHIFT                       5u
#define FR_SSCCR_NUF_WIDTH                       1u
#define FR_SSCCR_NUF(x)                          (((uint16_t)(((uint16_t)(x))<<FR_SSCCR_NUF_SHIFT))&FR_SSCCR_NUF_MASK)
#define FR_SSCCR_SYF_MASK                        0x40u
#define FR_SSCCR_SYF_SHIFT                       6u
#define FR_SSCCR_SYF_WIDTH                       1u
#define FR_SSCCR_SYF(x)                          (((uint16_t)(((uint16_t)(x))<<FR_SSCCR_SYF_SHIFT))&FR_SSCCR_SYF_MASK)
#define FR_SSCCR_VFR_MASK                        0x80u
#define FR_SSCCR_VFR_SHIFT                       7u
#define FR_SSCCR_VFR_WIDTH                       1u
#define FR_SSCCR_VFR(x)                          (((uint16_t)(((uint16_t)(x))<<FR_SSCCR_VFR_SHIFT))&FR_SSCCR_VFR_MASK)
#define FR_SSCCR_MCY_MASK                        0x100u
#define FR_SSCCR_MCY_SHIFT                       8u
#define FR_SSCCR_MCY_WIDTH                       1u
#define FR_SSCCR_MCY(x)                          (((uint16_t)(((uint16_t)(x))<<FR_SSCCR_MCY_SHIFT))&FR_SSCCR_MCY_MASK)
#define FR_SSCCR_CNTCFG_MASK                     0x600u
#define FR_SSCCR_CNTCFG_SHIFT                    9u
#define FR_SSCCR_CNTCFG_WIDTH                    2u
#define FR_SSCCR_CNTCFG(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SSCCR_CNTCFG_SHIFT))&FR_SSCCR_CNTCFG_MASK)
#define FR_SSCCR_SEL_MASK                        0x3000u
#define FR_SSCCR_SEL_SHIFT                       12u
#define FR_SSCCR_SEL_WIDTH                       2u
#define FR_SSCCR_SEL(x)                          (((uint16_t)(((uint16_t)(x))<<FR_SSCCR_SEL_SHIFT))&FR_SSCCR_SEL_MASK)
#define FR_SSCCR_WMD_MASK                        0x8000u
#define FR_SSCCR_WMD_SHIFT                       15u
#define FR_SSCCR_WMD_WIDTH                       1u
#define FR_SSCCR_WMD(x)                          (((uint16_t)(((uint16_t)(x))<<FR_SSCCR_WMD_SHIFT))&FR_SSCCR_WMD_MASK)
/* SSR Bit Fields */
#define FR_SSR_TCA_MASK                          0x1u
#define FR_SSR_TCA_SHIFT                         0u
#define FR_SSR_TCA_WIDTH                         1u
#define FR_SSR_TCA(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_TCA_SHIFT))&FR_SSR_TCA_MASK)
#define FR_SSR_BVA_MASK                          0x2u
#define FR_SSR_BVA_SHIFT                         1u
#define FR_SSR_BVA_WIDTH                         1u
#define FR_SSR_BVA(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_BVA_SHIFT))&FR_SSR_BVA_MASK)
#define FR_SSR_CEA_MASK                          0x4u
#define FR_SSR_CEA_SHIFT                         2u
#define FR_SSR_CEA_WIDTH                         1u
#define FR_SSR_CEA(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_CEA_SHIFT))&FR_SSR_CEA_MASK)
#define FR_SSR_SEA_MASK                          0x8u
#define FR_SSR_SEA_SHIFT                         3u
#define FR_SSR_SEA_WIDTH                         1u
#define FR_SSR_SEA(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_SEA_SHIFT))&FR_SSR_SEA_MASK)
#define FR_SSR_SUA_MASK                          0x10u
#define FR_SSR_SUA_SHIFT                         4u
#define FR_SSR_SUA_WIDTH                         1u
#define FR_SSR_SUA(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_SUA_SHIFT))&FR_SSR_SUA_MASK)
#define FR_SSR_NFA_MASK                          0x20u
#define FR_SSR_NFA_SHIFT                         5u
#define FR_SSR_NFA_WIDTH                         1u
#define FR_SSR_NFA(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_NFA_SHIFT))&FR_SSR_NFA_MASK)
#define FR_SSR_SYA_MASK                          0x40u
#define FR_SSR_SYA_SHIFT                         6u
#define FR_SSR_SYA_WIDTH                         1u
#define FR_SSR_SYA(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_SYA_SHIFT))&FR_SSR_SYA_MASK)
#define FR_SSR_VFA_MASK                          0x80u
#define FR_SSR_VFA_SHIFT                         7u
#define FR_SSR_VFA_WIDTH                         1u
#define FR_SSR_VFA(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_VFA_SHIFT))&FR_SSR_VFA_MASK)
#define FR_SSR_TCB_MASK                          0x100u
#define FR_SSR_TCB_SHIFT                         8u
#define FR_SSR_TCB_WIDTH                         1u
#define FR_SSR_TCB(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_TCB_SHIFT))&FR_SSR_TCB_MASK)
#define FR_SSR_BVB_MASK                          0x200u
#define FR_SSR_BVB_SHIFT                         9u
#define FR_SSR_BVB_WIDTH                         1u
#define FR_SSR_BVB(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_BVB_SHIFT))&FR_SSR_BVB_MASK)
#define FR_SSR_CEB_MASK                          0x400u
#define FR_SSR_CEB_SHIFT                         10u
#define FR_SSR_CEB_WIDTH                         1u
#define FR_SSR_CEB(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_CEB_SHIFT))&FR_SSR_CEB_MASK)
#define FR_SSR_SEB_MASK                          0x800u
#define FR_SSR_SEB_SHIFT                         11u
#define FR_SSR_SEB_WIDTH                         1u
#define FR_SSR_SEB(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_SEB_SHIFT))&FR_SSR_SEB_MASK)
#define FR_SSR_SUB_MASK                          0x1000u
#define FR_SSR_SUB_SHIFT                         12u
#define FR_SSR_SUB_WIDTH                         1u
#define FR_SSR_SUB(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_SUB_SHIFT))&FR_SSR_SUB_MASK)
#define FR_SSR_NFB_MASK                          0x2000u
#define FR_SSR_NFB_SHIFT                         13u
#define FR_SSR_NFB_WIDTH                         1u
#define FR_SSR_NFB(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_NFB_SHIFT))&FR_SSR_NFB_MASK)
#define FR_SSR_SYB_MASK                          0x4000u
#define FR_SSR_SYB_SHIFT                         14u
#define FR_SSR_SYB_WIDTH                         1u
#define FR_SSR_SYB(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_SYB_SHIFT))&FR_SSR_SYB_MASK)
#define FR_SSR_VFB_MASK                          0x8000u
#define FR_SSR_VFB_SHIFT                         15u
#define FR_SSR_VFB_WIDTH                         1u
#define FR_SSR_VFB(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_VFB_SHIFT))&FR_SSR_VFB_MASK)
/* SSCR Bit Fields */
#define FR_SSCR_SLOTSTATUSCNT_MASK               0xFFFFu
#define FR_SSCR_SLOTSTATUSCNT_SHIFT              0u
#define FR_SSCR_SLOTSTATUSCNT_WIDTH              16u
#define FR_SSCR_SLOTSTATUSCNT(x)                 (((uint16_t)(((uint16_t)(x))<<FR_SSCR_SLOTSTATUSCNT_SHIFT))&FR_SSCR_SLOTSTATUSCNT_MASK)
/* MTSACFR Bit Fields */
#define FR_MTSACFR_CYCCNTVAL_MASK                0x3Fu
#define FR_MTSACFR_CYCCNTVAL_SHIFT               0u
#define FR_MTSACFR_CYCCNTVAL_WIDTH               6u
#define FR_MTSACFR_CYCCNTVAL(x)                  (((uint16_t)(((uint16_t)(x))<<FR_MTSACFR_CYCCNTVAL_SHIFT))&FR_MTSACFR_CYCCNTVAL_MASK)
#define FR_MTSACFR_CYCCNTMSK_MASK                0x3F00u
#define FR_MTSACFR_CYCCNTMSK_SHIFT               8u
#define FR_MTSACFR_CYCCNTMSK_WIDTH               6u
#define FR_MTSACFR_CYCCNTMSK(x)                  (((uint16_t)(((uint16_t)(x))<<FR_MTSACFR_CYCCNTMSK_SHIFT))&FR_MTSACFR_CYCCNTMSK_MASK)
#define FR_MTSACFR_MTE_MASK                      0x8000u
#define FR_MTSACFR_MTE_SHIFT                     15u
#define FR_MTSACFR_MTE_WIDTH                     1u
#define FR_MTSACFR_MTE(x)                        (((uint16_t)(((uint16_t)(x))<<FR_MTSACFR_MTE_SHIFT))&FR_MTSACFR_MTE_MASK)
/* MTSBCFR Bit Fields */
#define FR_MTSBCFR_CYCCNTVAL_MASK                0x3Fu
#define FR_MTSBCFR_CYCCNTVAL_SHIFT               0u
#define FR_MTSBCFR_CYCCNTVAL_WIDTH               6u
#define FR_MTSBCFR_CYCCNTVAL(x)                  (((uint16_t)(((uint16_t)(x))<<FR_MTSBCFR_CYCCNTVAL_SHIFT))&FR_MTSBCFR_CYCCNTVAL_MASK)
#define FR_MTSBCFR_CYCCNTMSK_MASK                0x3F00u
#define FR_MTSBCFR_CYCCNTMSK_SHIFT               8u
#define FR_MTSBCFR_CYCCNTMSK_WIDTH               6u
#define FR_MTSBCFR_CYCCNTMSK(x)                  (((uint16_t)(((uint16_t)(x))<<FR_MTSBCFR_CYCCNTMSK_SHIFT))&FR_MTSBCFR_CYCCNTMSK_MASK)
#define FR_MTSBCFR_MTE_MASK                      0x8000u
#define FR_MTSBCFR_MTE_SHIFT                     15u
#define FR_MTSBCFR_MTE_WIDTH                     1u
#define FR_MTSBCFR_MTE(x)                        (((uint16_t)(((uint16_t)(x))<<FR_MTSBCFR_MTE_SHIFT))&FR_MTSBCFR_MTE_MASK)
/* RSBIR Bit Fields */
#define FR_RSBIR_RSBIDX_MASK                     0xFFu
#define FR_RSBIR_RSBIDX_SHIFT                    0u
#define FR_RSBIR_RSBIDX_WIDTH                    8u
#define FR_RSBIR_RSBIDX(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RSBIR_RSBIDX_SHIFT))&FR_RSBIR_RSBIDX_MASK)
#define FR_RSBIR_SEL_MASK                        0x3000u
#define FR_RSBIR_SEL_SHIFT                       12u
#define FR_RSBIR_SEL_WIDTH                       2u
#define FR_RSBIR_SEL(x)                          (((uint16_t)(((uint16_t)(x))<<FR_RSBIR_SEL_SHIFT))&FR_RSBIR_SEL_MASK)
#define FR_RSBIR_WMD_MASK                        0x8000u
#define FR_RSBIR_WMD_SHIFT                       15u
#define FR_RSBIR_WMD_WIDTH                       1u
#define FR_RSBIR_WMD(x)                          (((uint16_t)(((uint16_t)(x))<<FR_RSBIR_WMD_SHIFT))&FR_RSBIR_WMD_MASK)
/* RFWMSR Bit Fields */
#define FR_RFWMSR_SEL_MASK                       0x1u
#define FR_RFWMSR_SEL_SHIFT                      0u
#define FR_RFWMSR_SEL_WIDTH                      1u
#define FR_RFWMSR_SEL(x)                         (((uint16_t)(((uint16_t)(x))<<FR_RFWMSR_SEL_SHIFT))&FR_RFWMSR_SEL_MASK)
#define FR_RFWMSR_WM_MASK                        0xFF00u
#define FR_RFWMSR_WM_SHIFT                       8u
#define FR_RFWMSR_WM_WIDTH                       8u
#define FR_RFWMSR_WM(x)                          (((uint16_t)(((uint16_t)(x))<<FR_RFWMSR_WM_SHIFT))&FR_RFWMSR_WM_MASK)
/* RFSIR Bit Fields */
#define FR_RFSIR_SIDX_MASK                       0x3FFu
#define FR_RFSIR_SIDX_SHIFT                      0u
#define FR_RFSIR_SIDX_WIDTH                      10u
#define FR_RFSIR_SIDX(x)                         (((uint16_t)(((uint16_t)(x))<<FR_RFSIR_SIDX_SHIFT))&FR_RFSIR_SIDX_MASK)
/* RFDSR Bit Fields */
#define FR_RFDSR_ENTRY_SIZE_MASK                 0x7Fu
#define FR_RFDSR_ENTRY_SIZE_SHIFT                0u
#define FR_RFDSR_ENTRY_SIZE_WIDTH                7u
#define FR_RFDSR_ENTRY_SIZE(x)                   (((uint16_t)(((uint16_t)(x))<<FR_RFDSR_ENTRY_SIZE_SHIFT))&FR_RFDSR_ENTRY_SIZE_MASK)
#define FR_RFDSR_FIFO_DEPTH_MASK                 0xFF00u
#define FR_RFDSR_FIFO_DEPTH_SHIFT                8u
#define FR_RFDSR_FIFO_DEPTH_WIDTH                8u
#define FR_RFDSR_FIFO_DEPTH(x)                   (((uint16_t)(((uint16_t)(x))<<FR_RFDSR_FIFO_DEPTH_SHIFT))&FR_RFDSR_FIFO_DEPTH_MASK)
/* RFARIR Bit Fields */
#define FR_RFARIR_RDIDX_MASK                     0x3FFu
#define FR_RFARIR_RDIDX_SHIFT                    0u
#define FR_RFARIR_RDIDX_WIDTH                    10u
#define FR_RFARIR_RDIDX(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RFARIR_RDIDX_SHIFT))&FR_RFARIR_RDIDX_MASK)
/* RFBRIR Bit Fields */
#define FR_RFBRIR_RDIDX_MASK                     0x3FFu
#define FR_RFBRIR_RDIDX_SHIFT                    0u
#define FR_RFBRIR_RDIDX_WIDTH                    10u
#define FR_RFBRIR_RDIDX(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RFBRIR_RDIDX_SHIFT))&FR_RFBRIR_RDIDX_MASK)
/* RFMIDAFVR Bit Fields */
#define FR_RFMIDAFVR_MIDAFVAL_MASK               0xFFFFu
#define FR_RFMIDAFVR_MIDAFVAL_SHIFT              0u
#define FR_RFMIDAFVR_MIDAFVAL_WIDTH              16u
#define FR_RFMIDAFVR_MIDAFVAL(x)                 (((uint16_t)(((uint16_t)(x))<<FR_RFMIDAFVR_MIDAFVAL_SHIFT))&FR_RFMIDAFVR_MIDAFVAL_MASK)
/* RFMIDAFMR Bit Fields */
#define FR_RFMIDAFMR_MIDAFMSK_MASK               0xFFFFu
#define FR_RFMIDAFMR_MIDAFMSK_SHIFT              0u
#define FR_RFMIDAFMR_MIDAFMSK_WIDTH              16u
#define FR_RFMIDAFMR_MIDAFMSK(x)                 (((uint16_t)(((uint16_t)(x))<<FR_RFMIDAFMR_MIDAFMSK_SHIFT))&FR_RFMIDAFMR_MIDAFMSK_MASK)
/* RFFIDRFVR Bit Fields */
#define FR_RFFIDRFVR_FIDRFVAL_MASK               0x7FFu
#define FR_RFFIDRFVR_FIDRFVAL_SHIFT              0u
#define FR_RFFIDRFVR_FIDRFVAL_WIDTH              11u
#define FR_RFFIDRFVR_FIDRFVAL(x)                 (((uint16_t)(((uint16_t)(x))<<FR_RFFIDRFVR_FIDRFVAL_SHIFT))&FR_RFFIDRFVR_FIDRFVAL_MASK)
/* RFFIDRFMR Bit Fields */
#define FR_RFFIDRFMR_FIDRFMSK_MASK               0x7FFu
#define FR_RFFIDRFMR_FIDRFMSK_SHIFT              0u
#define FR_RFFIDRFMR_FIDRFMSK_WIDTH              11u
#define FR_RFFIDRFMR_FIDRFMSK(x)                 (((uint16_t)(((uint16_t)(x))<<FR_RFFIDRFMR_FIDRFMSK_SHIFT))&FR_RFFIDRFMR_FIDRFMSK_MASK)
/* RFRFCFR Bit Fields */
#define FR_RFRFCFR_SID_MASK                      0x7FFu
#define FR_RFRFCFR_SID_SHIFT                     0u
#define FR_RFRFCFR_SID_WIDTH                     11u
#define FR_RFRFCFR_SID(x)                        (((uint16_t)(((uint16_t)(x))<<FR_RFRFCFR_SID_SHIFT))&FR_RFRFCFR_SID_MASK)
#define FR_RFRFCFR_SEL_MASK                      0x3000u
#define FR_RFRFCFR_SEL_SHIFT                     12u
#define FR_RFRFCFR_SEL_WIDTH                     2u
#define FR_RFRFCFR_SEL(x)                        (((uint16_t)(((uint16_t)(x))<<FR_RFRFCFR_SEL_SHIFT))&FR_RFRFCFR_SEL_MASK)
#define FR_RFRFCFR_IBD_MASK                      0x4000u
#define FR_RFRFCFR_IBD_SHIFT                     14u
#define FR_RFRFCFR_IBD_WIDTH                     1u
#define FR_RFRFCFR_IBD(x)                        (((uint16_t)(((uint16_t)(x))<<FR_RFRFCFR_IBD_SHIFT))&FR_RFRFCFR_IBD_MASK)
#define FR_RFRFCFR_WMD_MASK                      0x8000u
#define FR_RFRFCFR_WMD_SHIFT                     15u
#define FR_RFRFCFR_WMD_WIDTH                     1u
#define FR_RFRFCFR_WMD(x)                        (((uint16_t)(((uint16_t)(x))<<FR_RFRFCFR_WMD_SHIFT))&FR_RFRFCFR_WMD_MASK)
/* RFRFCTR Bit Fields */
#define FR_RFRFCTR_F0EN_MASK                     0x1u
#define FR_RFRFCTR_F0EN_SHIFT                    0u
#define FR_RFRFCTR_F0EN_WIDTH                    1u
#define FR_RFRFCTR_F0EN(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RFRFCTR_F0EN_SHIFT))&FR_RFRFCTR_F0EN_MASK)
#define FR_RFRFCTR_F1EN_MASK                     0x2u
#define FR_RFRFCTR_F1EN_SHIFT                    1u
#define FR_RFRFCTR_F1EN_WIDTH                    1u
#define FR_RFRFCTR_F1EN(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RFRFCTR_F1EN_SHIFT))&FR_RFRFCTR_F1EN_MASK)
#define FR_RFRFCTR_F2EN_MASK                     0x4u
#define FR_RFRFCTR_F2EN_SHIFT                    2u
#define FR_RFRFCTR_F2EN_WIDTH                    1u
#define FR_RFRFCTR_F2EN(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RFRFCTR_F2EN_SHIFT))&FR_RFRFCTR_F2EN_MASK)
#define FR_RFRFCTR_F3EN_MASK                     0x8u
#define FR_RFRFCTR_F3EN_SHIFT                    3u
#define FR_RFRFCTR_F3EN_WIDTH                    1u
#define FR_RFRFCTR_F3EN(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RFRFCTR_F3EN_SHIFT))&FR_RFRFCTR_F3EN_MASK)
#define FR_RFRFCTR_F0MD_MASK                     0x100u
#define FR_RFRFCTR_F0MD_SHIFT                    8u
#define FR_RFRFCTR_F0MD_WIDTH                    1u
#define FR_RFRFCTR_F0MD(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RFRFCTR_F0MD_SHIFT))&FR_RFRFCTR_F0MD_MASK)
#define FR_RFRFCTR_F1MD_MASK                     0x200u
#define FR_RFRFCTR_F1MD_SHIFT                    9u
#define FR_RFRFCTR_F1MD_WIDTH                    1u
#define FR_RFRFCTR_F1MD(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RFRFCTR_F1MD_SHIFT))&FR_RFRFCTR_F1MD_MASK)
#define FR_RFRFCTR_F2MD_MASK                     0x400u
#define FR_RFRFCTR_F2MD_SHIFT                    10u
#define FR_RFRFCTR_F2MD_WIDTH                    1u
#define FR_RFRFCTR_F2MD(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RFRFCTR_F2MD_SHIFT))&FR_RFRFCTR_F2MD_MASK)
#define FR_RFRFCTR_F3MD_MASK                     0x800u
#define FR_RFRFCTR_F3MD_SHIFT                    11u
#define FR_RFRFCTR_F3MD_WIDTH                    1u
#define FR_RFRFCTR_F3MD(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RFRFCTR_F3MD_SHIFT))&FR_RFRFCTR_F3MD_MASK)
/* LDTXSLAR Bit Fields */
#define FR_LDTXSLAR_LDYNTXSLOTA_MASK             0x7FFu
#define FR_LDTXSLAR_LDYNTXSLOTA_SHIFT            0u
#define FR_LDTXSLAR_LDYNTXSLOTA_WIDTH            11u
#define FR_LDTXSLAR_LDYNTXSLOTA(x)               (((uint16_t)(((uint16_t)(x))<<FR_LDTXSLAR_LDYNTXSLOTA_SHIFT))&FR_LDTXSLAR_LDYNTXSLOTA_MASK)
/* LDTXSLBR Bit Fields */
#define FR_LDTXSLBR_LDYNTXSLOTB_MASK             0x7FFu
#define FR_LDTXSLBR_LDYNTXSLOTB_SHIFT            0u
#define FR_LDTXSLBR_LDYNTXSLOTB_WIDTH            11u
#define FR_LDTXSLBR_LDYNTXSLOTB(x)               (((uint16_t)(((uint16_t)(x))<<FR_LDTXSLBR_LDYNTXSLOTB_SHIFT))&FR_LDTXSLBR_LDYNTXSLOTB_MASK)
/* PCR0 Bit Fields */
#define FR_PCR0_static_slot_length_MASK          0x3FFu
#define FR_PCR0_static_slot_length_SHIFT         0u
#define FR_PCR0_static_slot_length_WIDTH         10u
#define FR_PCR0_static_slot_length(x)            (((uint16_t)(((uint16_t)(x))<<FR_PCR0_static_slot_length_SHIFT))&FR_PCR0_static_slot_length_MASK)
#define FR_PCR0_action_point_offset_MASK         0xFC00u
#define FR_PCR0_action_point_offset_SHIFT        10u
#define FR_PCR0_action_point_offset_WIDTH        6u
#define FR_PCR0_action_point_offset(x)           (((uint16_t)(((uint16_t)(x))<<FR_PCR0_action_point_offset_SHIFT))&FR_PCR0_action_point_offset_MASK)
/* PCR1 Bit Fields */
#define FR_PCR1_macro_after_first_static_slot_MASK 0x3FFFu
#define FR_PCR1_macro_after_first_static_slot_SHIFT 0u
#define FR_PCR1_macro_after_first_static_slot_WIDTH 14u
#define FR_PCR1_macro_after_first_static_slot(x) (((uint16_t)(((uint16_t)(x))<<FR_PCR1_macro_after_first_static_slot_SHIFT))&FR_PCR1_macro_after_first_static_slot_MASK)
/* PCR2 Bit Fields */
#define FR_PCR2_number_of_static_slots_MASK      0x3FFu
#define FR_PCR2_number_of_static_slots_SHIFT     0u
#define FR_PCR2_number_of_static_slots_WIDTH     10u
#define FR_PCR2_number_of_static_slots(x)        (((uint16_t)(((uint16_t)(x))<<FR_PCR2_number_of_static_slots_SHIFT))&FR_PCR2_number_of_static_slots_MASK)
#define FR_PCR2_minislot_after_action_point_MASK 0xFC00u
#define FR_PCR2_minislot_after_action_point_SHIFT 10u
#define FR_PCR2_minislot_after_action_point_WIDTH 6u
#define FR_PCR2_minislot_after_action_point(x)   (((uint16_t)(((uint16_t)(x))<<FR_PCR2_minislot_after_action_point_SHIFT))&FR_PCR2_minislot_after_action_point_MASK)
/* PCR3 Bit Fields */
#define FR_PCR3_coldstart_attempts_MASK          0x1Fu
#define FR_PCR3_coldstart_attempts_SHIFT         0u
#define FR_PCR3_coldstart_attempts_WIDTH         5u
#define FR_PCR3_coldstart_attempts(x)            (((uint16_t)(((uint16_t)(x))<<FR_PCR3_coldstart_attempts_SHIFT))&FR_PCR3_coldstart_attempts_MASK)
#define FR_PCR3_minislot_action_point_offset_MASK 0x3E0u
#define FR_PCR3_minislot_action_point_offset_SHIFT 5u
#define FR_PCR3_minislot_action_point_offset_WIDTH 5u
#define FR_PCR3_minislot_action_point_offset(x)  (((uint16_t)(((uint16_t)(x))<<FR_PCR3_minislot_action_point_offset_SHIFT))&FR_PCR3_minislot_action_point_offset_MASK)
#define FR_PCR3_wakeup_symbol_rx_low_MASK        0xFC00u
#define FR_PCR3_wakeup_symbol_rx_low_SHIFT       10u
#define FR_PCR3_wakeup_symbol_rx_low_WIDTH       6u
#define FR_PCR3_wakeup_symbol_rx_low(x)          (((uint16_t)(((uint16_t)(x))<<FR_PCR3_wakeup_symbol_rx_low_SHIFT))&FR_PCR3_wakeup_symbol_rx_low_MASK)
/* PCR4 Bit Fields */
#define FR_PCR4_wakeup_symbol_rx_window_MASK     0x1FFu
#define FR_PCR4_wakeup_symbol_rx_window_SHIFT    0u
#define FR_PCR4_wakeup_symbol_rx_window_WIDTH    9u
#define FR_PCR4_wakeup_symbol_rx_window(x)       (((uint16_t)(((uint16_t)(x))<<FR_PCR4_wakeup_symbol_rx_window_SHIFT))&FR_PCR4_wakeup_symbol_rx_window_MASK)
#define FR_PCR4_cas_rx_low_max_MASK              0xFE00u
#define FR_PCR4_cas_rx_low_max_SHIFT             9u
#define FR_PCR4_cas_rx_low_max_WIDTH             7u
#define FR_PCR4_cas_rx_low_max(x)                (((uint16_t)(((uint16_t)(x))<<FR_PCR4_cas_rx_low_max_SHIFT))&FR_PCR4_cas_rx_low_max_MASK)
/* PCR5 Bit Fields */
#define FR_PCR5_wakeup_symbol_rx_idle_MASK       0x3Fu
#define FR_PCR5_wakeup_symbol_rx_idle_SHIFT      0u
#define FR_PCR5_wakeup_symbol_rx_idle_WIDTH      6u
#define FR_PCR5_wakeup_symbol_rx_idle(x)         (((uint16_t)(((uint16_t)(x))<<FR_PCR5_wakeup_symbol_rx_idle_SHIFT))&FR_PCR5_wakeup_symbol_rx_idle_MASK)
#define FR_PCR5_wakeup_symbol_tx_low_MASK        0xFC0u
#define FR_PCR5_wakeup_symbol_tx_low_SHIFT       6u
#define FR_PCR5_wakeup_symbol_tx_low_WIDTH       6u
#define FR_PCR5_wakeup_symbol_tx_low(x)          (((uint16_t)(((uint16_t)(x))<<FR_PCR5_wakeup_symbol_tx_low_SHIFT))&FR_PCR5_wakeup_symbol_tx_low_MASK)
#define FR_PCR5_tss_transmitter_MASK             0xF000u
#define FR_PCR5_tss_transmitter_SHIFT            12u
#define FR_PCR5_tss_transmitter_WIDTH            4u
#define FR_PCR5_tss_transmitter(x)               (((uint16_t)(((uint16_t)(x))<<FR_PCR5_tss_transmitter_SHIFT))&FR_PCR5_tss_transmitter_MASK)
/* PCR6 Bit Fields */
#define FR_PCR6_macro_initial_offset_a_MASK      0x7Fu
#define FR_PCR6_macro_initial_offset_a_SHIFT     0u
#define FR_PCR6_macro_initial_offset_a_WIDTH     7u
#define FR_PCR6_macro_initial_offset_a(x)        (((uint16_t)(((uint16_t)(x))<<FR_PCR6_macro_initial_offset_a_SHIFT))&FR_PCR6_macro_initial_offset_a_MASK)
#define FR_PCR6_symbol_window_after_action_point_MASK 0x7F80u
#define FR_PCR6_symbol_window_after_action_point_SHIFT 7u
#define FR_PCR6_symbol_window_after_action_point_WIDTH 8u
#define FR_PCR6_symbol_window_after_action_point(x) (((uint16_t)(((uint16_t)(x))<<FR_PCR6_symbol_window_after_action_point_SHIFT))&FR_PCR6_symbol_window_after_action_point_MASK)
/* PCR7 Bit Fields */
#define FR_PCR7_micro_per_macro_nom_half_MASK    0x7Fu
#define FR_PCR7_micro_per_macro_nom_half_SHIFT   0u
#define FR_PCR7_micro_per_macro_nom_half_WIDTH   7u
#define FR_PCR7_micro_per_macro_nom_half(x)      (((uint16_t)(((uint16_t)(x))<<FR_PCR7_micro_per_macro_nom_half_SHIFT))&FR_PCR7_micro_per_macro_nom_half_MASK)
#define FR_PCR7_decoding_correction_b_MASK       0xFF80u
#define FR_PCR7_decoding_correction_b_SHIFT      7u
#define FR_PCR7_decoding_correction_b_WIDTH      9u
#define FR_PCR7_decoding_correction_b(x)         (((uint16_t)(((uint16_t)(x))<<FR_PCR7_decoding_correction_b_SHIFT))&FR_PCR7_decoding_correction_b_MASK)
/* PCR8 Bit Fields */
#define FR_PCR8_wakeup_symbol_tx_idle_MASK       0xFFu
#define FR_PCR8_wakeup_symbol_tx_idle_SHIFT      0u
#define FR_PCR8_wakeup_symbol_tx_idle_WIDTH      8u
#define FR_PCR8_wakeup_symbol_tx_idle(x)         (((uint16_t)(((uint16_t)(x))<<FR_PCR8_wakeup_symbol_tx_idle_SHIFT))&FR_PCR8_wakeup_symbol_tx_idle_MASK)
#define FR_PCR8_max_without_clock_correction_passive_MASK 0xF00u
#define FR_PCR8_max_without_clock_correction_passive_SHIFT 8u
#define FR_PCR8_max_without_clock_correction_passive_WIDTH 4u
#define FR_PCR8_max_without_clock_correction_passive(x) (((uint16_t)(((uint16_t)(x))<<FR_PCR8_max_without_clock_correction_passive_SHIFT))&FR_PCR8_max_without_clock_correction_passive_MASK)
#define FR_PCR8_max_without_clock_correction_fatal_MASK 0xF000u
#define FR_PCR8_max_without_clock_correction_fatal_SHIFT 12u
#define FR_PCR8_max_without_clock_correction_fatal_WIDTH 4u
#define FR_PCR8_max_without_clock_correction_fatal(x) (((uint16_t)(((uint16_t)(x))<<FR_PCR8_max_without_clock_correction_fatal_SHIFT))&FR_PCR8_max_without_clock_correction_fatal_MASK)
/* PCR9 Bit Fields */
#define FR_PCR9_offset_correction_out_MASK       0x3FFFu
#define FR_PCR9_offset_correction_out_SHIFT      0u
#define FR_PCR9_offset_correction_out_WIDTH      14u
#define FR_PCR9_offset_correction_out(x)         (((uint16_t)(((uint16_t)(x))<<FR_PCR9_offset_correction_out_SHIFT))&FR_PCR9_offset_correction_out_MASK)
#define FR_PCR9_symbol_window_exists_MASK        0x4000u
#define FR_PCR9_symbol_window_exists_SHIFT       14u
#define FR_PCR9_symbol_window_exists_WIDTH       1u
#define FR_PCR9_symbol_window_exists(x)          (((uint16_t)(((uint16_t)(x))<<FR_PCR9_symbol_window_exists_SHIFT))&FR_PCR9_symbol_window_exists_MASK)
#define FR_PCR9_minislot_exists_MASK             0x8000u
#define FR_PCR9_minislot_exists_SHIFT            15u
#define FR_PCR9_minislot_exists_WIDTH            1u
#define FR_PCR9_minislot_exists(x)               (((uint16_t)(((uint16_t)(x))<<FR_PCR9_minislot_exists_SHIFT))&FR_PCR9_minislot_exists_MASK)
/* PCR10 Bit Fields */
#define FR_PCR10_macro_per_cycle_MASK            0x3FFFu
#define FR_PCR10_macro_per_cycle_SHIFT           0u
#define FR_PCR10_macro_per_cycle_WIDTH           14u
#define FR_PCR10_macro_per_cycle(x)              (((uint16_t)(((uint16_t)(x))<<FR_PCR10_macro_per_cycle_SHIFT))&FR_PCR10_macro_per_cycle_MASK)
#define FR_PCR10_wakeup_channel_MASK             0x4000u
#define FR_PCR10_wakeup_channel_SHIFT            14u
#define FR_PCR10_wakeup_channel_WIDTH            1u
#define FR_PCR10_wakeup_channel(x)               (((uint16_t)(((uint16_t)(x))<<FR_PCR10_wakeup_channel_SHIFT))&FR_PCR10_wakeup_channel_MASK)
#define FR_PCR10_single_slot_enabled_MASK        0x8000u
#define FR_PCR10_single_slot_enabled_SHIFT       15u
#define FR_PCR10_single_slot_enabled_WIDTH       1u
#define FR_PCR10_single_slot_enabled(x)          (((uint16_t)(((uint16_t)(x))<<FR_PCR10_single_slot_enabled_SHIFT))&FR_PCR10_single_slot_enabled_MASK)
/* PCR11 Bit Fields */
#define FR_PCR11_offset_correction_start_MASK    0x3FFFu
#define FR_PCR11_offset_correction_start_SHIFT   0u
#define FR_PCR11_offset_correction_start_WIDTH   14u
#define FR_PCR11_offset_correction_start(x)      (((uint16_t)(((uint16_t)(x))<<FR_PCR11_offset_correction_start_SHIFT))&FR_PCR11_offset_correction_start_MASK)
#define FR_PCR11_key_slot_used_for_sync_MASK     0x4000u
#define FR_PCR11_key_slot_used_for_sync_SHIFT    14u
#define FR_PCR11_key_slot_used_for_sync_WIDTH    1u
#define FR_PCR11_key_slot_used_for_sync(x)       (((uint16_t)(((uint16_t)(x))<<FR_PCR11_key_slot_used_for_sync_SHIFT))&FR_PCR11_key_slot_used_for_sync_MASK)
#define FR_PCR11_key_slot_used_for_startup_MASK  0x8000u
#define FR_PCR11_key_slot_used_for_startup_SHIFT 15u
#define FR_PCR11_key_slot_used_for_startup_WIDTH 1u
#define FR_PCR11_key_slot_used_for_startup(x)    (((uint16_t)(((uint16_t)(x))<<FR_PCR11_key_slot_used_for_startup_SHIFT))&FR_PCR11_key_slot_used_for_startup_MASK)
/* PCR12 Bit Fields */
#define FR_PCR12_key_slot_header_crc_MASK        0x7FFu
#define FR_PCR12_key_slot_header_crc_SHIFT       0u
#define FR_PCR12_key_slot_header_crc_WIDTH       11u
#define FR_PCR12_key_slot_header_crc(x)          (((uint16_t)(((uint16_t)(x))<<FR_PCR12_key_slot_header_crc_SHIFT))&FR_PCR12_key_slot_header_crc_MASK)
#define FR_PCR12_allow_passive_to_active_MASK    0xF800u
#define FR_PCR12_allow_passive_to_active_SHIFT   11u
#define FR_PCR12_allow_passive_to_active_WIDTH   5u
#define FR_PCR12_allow_passive_to_active(x)      (((uint16_t)(((uint16_t)(x))<<FR_PCR12_allow_passive_to_active_SHIFT))&FR_PCR12_allow_passive_to_active_MASK)
/* PCR13 Bit Fields */
#define FR_PCR13_static_slot_after_action_point_MASK 0x3FFu
#define FR_PCR13_static_slot_after_action_point_SHIFT 0u
#define FR_PCR13_static_slot_after_action_point_WIDTH 10u
#define FR_PCR13_static_slot_after_action_point(x) (((uint16_t)(((uint16_t)(x))<<FR_PCR13_static_slot_after_action_point_SHIFT))&FR_PCR13_static_slot_after_action_point_MASK)
#define FR_PCR13_first_minislot_action_point_offset_MASK 0xFC00u
#define FR_PCR13_first_minislot_action_point_offset_SHIFT 10u
#define FR_PCR13_first_minislot_action_point_offset_WIDTH 6u
#define FR_PCR13_first_minislot_action_point_offset(x) (((uint16_t)(((uint16_t)(x))<<FR_PCR13_first_minislot_action_point_offset_SHIFT))&FR_PCR13_first_minislot_action_point_offset_MASK)
/* PCR14 Bit Fields */
#define FR_PCR14_listen_timeout_MASK             0x1Fu
#define FR_PCR14_listen_timeout_SHIFT            0u
#define FR_PCR14_listen_timeout_WIDTH            5u
#define FR_PCR14_listen_timeout(x)               (((uint16_t)(((uint16_t)(x))<<FR_PCR14_listen_timeout_SHIFT))&FR_PCR14_listen_timeout_MASK)
#define FR_PCR14_rate_correction_out_MASK        0xFFE0u
#define FR_PCR14_rate_correction_out_SHIFT       5u
#define FR_PCR14_rate_correction_out_WIDTH       11u
#define FR_PCR14_rate_correction_out(x)          (((uint16_t)(((uint16_t)(x))<<FR_PCR14_rate_correction_out_SHIFT))&FR_PCR14_rate_correction_out_MASK)
/* PCR15 Bit Fields */
#define FR_PCR15_listen_timeout_MASK             0xFFFFu
#define FR_PCR15_listen_timeout_SHIFT            0u
#define FR_PCR15_listen_timeout_WIDTH            16u
#define FR_PCR15_listen_timeout(x)               (((uint16_t)(((uint16_t)(x))<<FR_PCR15_listen_timeout_SHIFT))&FR_PCR15_listen_timeout_MASK)
/* PCR16 Bit Fields */
#define FR_PCR16_noise_listen_timeout_MASK       0x1FFu
#define FR_PCR16_noise_listen_timeout_SHIFT      0u
#define FR_PCR16_noise_listen_timeout_WIDTH      9u
#define FR_PCR16_noise_listen_timeout(x)         (((uint16_t)(((uint16_t)(x))<<FR_PCR16_noise_listen_timeout_SHIFT))&FR_PCR16_noise_listen_timeout_MASK)
#define FR_PCR16_macro_initial_offset_b_MASK     0xFE00u
#define FR_PCR16_macro_initial_offset_b_SHIFT    9u
#define FR_PCR16_macro_initial_offset_b_WIDTH    7u
#define FR_PCR16_macro_initial_offset_b(x)       (((uint16_t)(((uint16_t)(x))<<FR_PCR16_macro_initial_offset_b_SHIFT))&FR_PCR16_macro_initial_offset_b_MASK)
/* PCR17 Bit Fields */
#define FR_PCR17_noise_listen_timeout_MASK       0xFFFFu
#define FR_PCR17_noise_listen_timeout_SHIFT      0u
#define FR_PCR17_noise_listen_timeout_WIDTH      16u
#define FR_PCR17_noise_listen_timeout(x)         (((uint16_t)(((uint16_t)(x))<<FR_PCR17_noise_listen_timeout_SHIFT))&FR_PCR17_noise_listen_timeout_MASK)
/* PCR18 Bit Fields */
#define FR_PCR18_key_slot_id_MASK                0x3FFu
#define FR_PCR18_key_slot_id_SHIFT               0u
#define FR_PCR18_key_slot_id_WIDTH               10u
#define FR_PCR18_key_slot_id(x)                  (((uint16_t)(((uint16_t)(x))<<FR_PCR18_key_slot_id_SHIFT))&FR_PCR18_key_slot_id_MASK)
#define FR_PCR18_wakeup_pattern_MASK             0xFC00u
#define FR_PCR18_wakeup_pattern_SHIFT            10u
#define FR_PCR18_wakeup_pattern_WIDTH            6u
#define FR_PCR18_wakeup_pattern(x)               (((uint16_t)(((uint16_t)(x))<<FR_PCR18_wakeup_pattern_SHIFT))&FR_PCR18_wakeup_pattern_MASK)
/* PCR19 Bit Fields */
#define FR_PCR19_payload_length_static_MASK      0x7Fu
#define FR_PCR19_payload_length_static_SHIFT     0u
#define FR_PCR19_payload_length_static_WIDTH     7u
#define FR_PCR19_payload_length_static(x)        (((uint16_t)(((uint16_t)(x))<<FR_PCR19_payload_length_static_SHIFT))&FR_PCR19_payload_length_static_MASK)
#define FR_PCR19_decoding_correction_a_MASK      0xFF80u
#define FR_PCR19_decoding_correction_a_SHIFT     7u
#define FR_PCR19_decoding_correction_a_WIDTH     9u
#define FR_PCR19_decoding_correction_a(x)        (((uint16_t)(((uint16_t)(x))<<FR_PCR19_decoding_correction_a_SHIFT))&FR_PCR19_decoding_correction_a_MASK)
/* PCR20 Bit Fields */
#define FR_PCR20_micro_initial_offset_a_MASK     0xFFu
#define FR_PCR20_micro_initial_offset_a_SHIFT    0u
#define FR_PCR20_micro_initial_offset_a_WIDTH    8u
#define FR_PCR20_micro_initial_offset_a(x)       (((uint16_t)(((uint16_t)(x))<<FR_PCR20_micro_initial_offset_a_SHIFT))&FR_PCR20_micro_initial_offset_a_MASK)
#define FR_PCR20_micro_initial_offset_b_MASK     0xFF00u
#define FR_PCR20_micro_initial_offset_b_SHIFT    8u
#define FR_PCR20_micro_initial_offset_b_WIDTH    8u
#define FR_PCR20_micro_initial_offset_b(x)       (((uint16_t)(((uint16_t)(x))<<FR_PCR20_micro_initial_offset_b_SHIFT))&FR_PCR20_micro_initial_offset_b_MASK)
/* PCR21 Bit Fields */
#define FR_PCR21_latest_tx_MASK                  0x1FFFu
#define FR_PCR21_latest_tx_SHIFT                 0u
#define FR_PCR21_latest_tx_WIDTH                 13u
#define FR_PCR21_latest_tx(x)                    (((uint16_t)(((uint16_t)(x))<<FR_PCR21_latest_tx_SHIFT))&FR_PCR21_latest_tx_MASK)
#define FR_PCR21_extern_rate_correction_MASK     0xE000u
#define FR_PCR21_extern_rate_correction_SHIFT    13u
#define FR_PCR21_extern_rate_correction_WIDTH    3u
#define FR_PCR21_extern_rate_correction(x)       (((uint16_t)(((uint16_t)(x))<<FR_PCR21_extern_rate_correction_SHIFT))&FR_PCR21_extern_rate_correction_MASK)
/* PCR22 Bit Fields */
#define FR_PCR22_micro_per_cycle_MASK            0xFu
#define FR_PCR22_micro_per_cycle_SHIFT           0u
#define FR_PCR22_micro_per_cycle_WIDTH           4u
#define FR_PCR22_micro_per_cycle(x)              (((uint16_t)(((uint16_t)(x))<<FR_PCR22_micro_per_cycle_SHIFT))&FR_PCR22_micro_per_cycle_MASK)
#define FR_PCR22_comp_accepted_startup_range_a_MASK 0x7FF0u
#define FR_PCR22_comp_accepted_startup_range_a_SHIFT 4u
#define FR_PCR22_comp_accepted_startup_range_a_WIDTH 11u
#define FR_PCR22_comp_accepted_startup_range_a(x) (((uint16_t)(((uint16_t)(x))<<FR_PCR22_comp_accepted_startup_range_a_SHIFT))&FR_PCR22_comp_accepted_startup_range_a_MASK)
/* PCR23 Bit Fields */
#define FR_PCR23_micro_per_cycle_MASK            0xFFFFu
#define FR_PCR23_micro_per_cycle_SHIFT           0u
#define FR_PCR23_micro_per_cycle_WIDTH           16u
#define FR_PCR23_micro_per_cycle(x)              (((uint16_t)(((uint16_t)(x))<<FR_PCR23_micro_per_cycle_SHIFT))&FR_PCR23_micro_per_cycle_MASK)
/* PCR24 Bit Fields */
#define FR_PCR24_micro_per_cycle_min_MASK        0xFu
#define FR_PCR24_micro_per_cycle_min_SHIFT       0u
#define FR_PCR24_micro_per_cycle_min_WIDTH       4u
#define FR_PCR24_micro_per_cycle_min(x)          (((uint16_t)(((uint16_t)(x))<<FR_PCR24_micro_per_cycle_min_SHIFT))&FR_PCR24_micro_per_cycle_min_MASK)
#define FR_PCR24_max_payload_length_dynamic_MASK 0x7F0u
#define FR_PCR24_max_payload_length_dynamic_SHIFT 4u
#define FR_PCR24_max_payload_length_dynamic_WIDTH 7u
#define FR_PCR24_max_payload_length_dynamic(x)   (((uint16_t)(((uint16_t)(x))<<FR_PCR24_max_payload_length_dynamic_SHIFT))&FR_PCR24_max_payload_length_dynamic_MASK)
#define FR_PCR24_cluster_drift_damping_MASK      0xF800u
#define FR_PCR24_cluster_drift_damping_SHIFT     11u
#define FR_PCR24_cluster_drift_damping_WIDTH     5u
#define FR_PCR24_cluster_drift_damping(x)        (((uint16_t)(((uint16_t)(x))<<FR_PCR24_cluster_drift_damping_SHIFT))&FR_PCR24_cluster_drift_damping_MASK)
/* PCR25 Bit Fields */
#define FR_PCR25_micro_per_cycle_min_MASK        0xFFFFu
#define FR_PCR25_micro_per_cycle_min_SHIFT       0u
#define FR_PCR25_micro_per_cycle_min_WIDTH       16u
#define FR_PCR25_micro_per_cycle_min(x)          (((uint16_t)(((uint16_t)(x))<<FR_PCR25_micro_per_cycle_min_SHIFT))&FR_PCR25_micro_per_cycle_min_MASK)
/* PCR26 Bit Fields */
#define FR_PCR26_micro_per_cycle_max_MASK        0xFu
#define FR_PCR26_micro_per_cycle_max_SHIFT       0u
#define FR_PCR26_micro_per_cycle_max_WIDTH       4u
#define FR_PCR26_micro_per_cycle_max(x)          (((uint16_t)(((uint16_t)(x))<<FR_PCR26_micro_per_cycle_max_SHIFT))&FR_PCR26_micro_per_cycle_max_MASK)
#define FR_PCR26_comp_accepted_startup_range_b_MASK 0x7FF0u
#define FR_PCR26_comp_accepted_startup_range_b_SHIFT 4u
#define FR_PCR26_comp_accepted_startup_range_b_WIDTH 11u
#define FR_PCR26_comp_accepted_startup_range_b(x) (((uint16_t)(((uint16_t)(x))<<FR_PCR26_comp_accepted_startup_range_b_SHIFT))&FR_PCR26_comp_accepted_startup_range_b_MASK)
#define FR_PCR26_allow_halt_due_to_clock_MASK    0x8000u
#define FR_PCR26_allow_halt_due_to_clock_SHIFT   15u
#define FR_PCR26_allow_halt_due_to_clock_WIDTH   1u
#define FR_PCR26_allow_halt_due_to_clock(x)      (((uint16_t)(((uint16_t)(x))<<FR_PCR26_allow_halt_due_to_clock_SHIFT))&FR_PCR26_allow_halt_due_to_clock_MASK)
/* PCR27 Bit Fields */
#define FR_PCR27_micro_per_cycle_max_MASK        0xFFFFu
#define FR_PCR27_micro_per_cycle_max_SHIFT       0u
#define FR_PCR27_micro_per_cycle_max_WIDTH       16u
#define FR_PCR27_micro_per_cycle_max(x)          (((uint16_t)(((uint16_t)(x))<<FR_PCR27_micro_per_cycle_max_SHIFT))&FR_PCR27_micro_per_cycle_max_MASK)
/* PCR28 Bit Fields */
#define FR_PCR28_macro_after_offset_correction_MASK 0x3FFFu
#define FR_PCR28_macro_after_offset_correction_SHIFT 0u
#define FR_PCR28_macro_after_offset_correction_WIDTH 14u
#define FR_PCR28_macro_after_offset_correction(x) (((uint16_t)(((uint16_t)(x))<<FR_PCR28_macro_after_offset_correction_SHIFT))&FR_PCR28_macro_after_offset_correction_MASK)
#define FR_PCR28_dynamic_slot_idle_phase_MASK    0xC000u
#define FR_PCR28_dynamic_slot_idle_phase_SHIFT   14u
#define FR_PCR28_dynamic_slot_idle_phase_WIDTH   2u
#define FR_PCR28_dynamic_slot_idle_phase(x)      (((uint16_t)(((uint16_t)(x))<<FR_PCR28_dynamic_slot_idle_phase_SHIFT))&FR_PCR28_dynamic_slot_idle_phase_MASK)
/* PCR29 Bit Fields */
#define FR_PCR29_minislots_max_MASK              0x1FFFu
#define FR_PCR29_minislots_max_SHIFT             0u
#define FR_PCR29_minislots_max_WIDTH             13u
#define FR_PCR29_minislots_max(x)                (((uint16_t)(((uint16_t)(x))<<FR_PCR29_minislots_max_SHIFT))&FR_PCR29_minislots_max_MASK)
#define FR_PCR29_extern_offset_correction_MASK   0xE000u
#define FR_PCR29_extern_offset_correction_SHIFT  13u
#define FR_PCR29_extern_offset_correction_WIDTH  3u
#define FR_PCR29_extern_offset_correction(x)     (((uint16_t)(((uint16_t)(x))<<FR_PCR29_extern_offset_correction_SHIFT))&FR_PCR29_extern_offset_correction_MASK)
/* PCR30 Bit Fields */
#define FR_PCR30_sync_node_max_MASK              0xFu
#define FR_PCR30_sync_node_max_SHIFT             0u
#define FR_PCR30_sync_node_max_WIDTH             4u
#define FR_PCR30_sync_node_max(x)                (((uint16_t)(((uint16_t)(x))<<FR_PCR30_sync_node_max_SHIFT))&FR_PCR30_sync_node_max_MASK)
/* STPWHR Bit Fields */
#define FR_STPWHR_STPW_S_H_MASK                  0xFFFFu
#define FR_STPWHR_STPW_S_H_SHIFT                 0u
#define FR_STPWHR_STPW_S_H_WIDTH                 16u
#define FR_STPWHR_STPW_S_H(x)                    (((uint16_t)(((uint16_t)(x))<<FR_STPWHR_STPW_S_H_SHIFT))&FR_STPWHR_STPW_S_H_MASK)
/* STPWLR Bit Fields */
#define FR_STPWLR_STPW_S_L_MASK                  0xFFFFu
#define FR_STPWLR_STPW_S_L_SHIFT                 0u
#define FR_STPWLR_STPW_S_L_WIDTH                 16u
#define FR_STPWLR_STPW_S_L(x)                    (((uint16_t)(((uint16_t)(x))<<FR_STPWLR_STPW_S_L_SHIFT))&FR_STPWLR_STPW_S_L_MASK)
/* PEOER Bit Fields */
#define FR_PEOER_CYC_EE_MASK                     0x1u
#define FR_PEOER_CYC_EE_SHIFT                    0u
#define FR_PEOER_CYC_EE_WIDTH                    1u
#define FR_PEOER_CYC_EE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PEOER_CYC_EE_SHIFT))&FR_PEOER_CYC_EE_MASK)
#define FR_PEOER_TIM1_EE_MASK                    0x2u
#define FR_PEOER_TIM1_EE_SHIFT                   1u
#define FR_PEOER_TIM1_EE_WIDTH                   1u
#define FR_PEOER_TIM1_EE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PEOER_TIM1_EE_SHIFT))&FR_PEOER_TIM1_EE_MASK)
#define FR_PEOER_TIM2_EE_MASK                    0x4u
#define FR_PEOER_TIM2_EE_SHIFT                   2u
#define FR_PEOER_TIM2_EE_WIDTH                   1u
#define FR_PEOER_TIM2_EE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PEOER_TIM2_EE_SHIFT))&FR_PEOER_TIM2_EE_MASK)
#define FR_PEOER_STPW_EN_MASK                    0x100u
#define FR_PEOER_STPW_EN_SHIFT                   8u
#define FR_PEOER_STPW_EN_WIDTH                   1u
#define FR_PEOER_STPW_EN(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PEOER_STPW_EN_SHIFT))&FR_PEOER_STPW_EN_MASK)
/* RFSDOR Bit Fields */
#define FR_RFSDOR_SDO_MASK                       0xFFFFu
#define FR_RFSDOR_SDO_SHIFT                      0u
#define FR_RFSDOR_SDO_WIDTH                      16u
#define FR_RFSDOR_SDO(x)                         (((uint16_t)(((uint16_t)(x))<<FR_RFSDOR_SDO_SHIFT))&FR_RFSDOR_SDO_MASK)
/* RFSYMBADHR Bit Fields */
#define FR_RFSYMBADHR_SMBA_MASK                  0xFFFFu
#define FR_RFSYMBADHR_SMBA_SHIFT                 0u
#define FR_RFSYMBADHR_SMBA_WIDTH                 16u
#define FR_RFSYMBADHR_SMBA(x)                    (((uint16_t)(((uint16_t)(x))<<FR_RFSYMBADHR_SMBA_SHIFT))&FR_RFSYMBADHR_SMBA_MASK)
/* RFSYMBADLR Bit Fields */
#define FR_RFSYMBADLR_SMBA_MASK                  0xFFF0u
#define FR_RFSYMBADLR_SMBA_SHIFT                 4u
#define FR_RFSYMBADLR_SMBA_WIDTH                 12u
#define FR_RFSYMBADLR_SMBA(x)                    (((uint16_t)(((uint16_t)(x))<<FR_RFSYMBADLR_SMBA_SHIFT))&FR_RFSYMBADLR_SMBA_MASK)
/* RFPTR Bit Fields */
#define FR_RFPTR_PTD_MASK                        0x3FFFu
#define FR_RFPTR_PTD_SHIFT                       0u
#define FR_RFPTR_PTD_WIDTH                       14u
#define FR_RFPTR_PTD(x)                          (((uint16_t)(((uint16_t)(x))<<FR_RFPTR_PTD_SHIFT))&FR_RFPTR_PTD_MASK)
/* RFFLPCR Bit Fields */
#define FR_RFFLPCR_FLA_or_PCA_MASK               0xFFu
#define FR_RFFLPCR_FLA_or_PCA_SHIFT              0u
#define FR_RFFLPCR_FLA_or_PCA_WIDTH              8u
#define FR_RFFLPCR_FLA_or_PCA(x)                 (((uint16_t)(((uint16_t)(x))<<FR_RFFLPCR_FLA_or_PCA_SHIFT))&FR_RFFLPCR_FLA_or_PCA_MASK)
#define FR_RFFLPCR_FLB_or_PCB_MASK               0xFF00u
#define FR_RFFLPCR_FLB_or_PCB_SHIFT              8u
#define FR_RFFLPCR_FLB_or_PCB_WIDTH              8u
#define FR_RFFLPCR_FLB_or_PCB(x)                 (((uint16_t)(((uint16_t)(x))<<FR_RFFLPCR_FLB_or_PCB_SHIFT))&FR_RFFLPCR_FLB_or_PCB_MASK)
/* EEIFER Bit Fields */
#define FR_EEIFER_DRCE_IE_MASK                   0x1u
#define FR_EEIFER_DRCE_IE_SHIFT                  0u
#define FR_EEIFER_DRCE_IE_WIDTH                  1u
#define FR_EEIFER_DRCE_IE(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_DRCE_IE_SHIFT))&FR_EEIFER_DRCE_IE_MASK)
#define FR_EEIFER_DRNE_IE_MASK                   0x2u
#define FR_EEIFER_DRNE_IE_SHIFT                  1u
#define FR_EEIFER_DRNE_IE_WIDTH                  1u
#define FR_EEIFER_DRNE_IE(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_DRNE_IE_SHIFT))&FR_EEIFER_DRNE_IE_MASK)
#define FR_EEIFER_LRCE_IE_MASK                   0x4u
#define FR_EEIFER_LRCE_IE_SHIFT                  2u
#define FR_EEIFER_LRCE_IE_WIDTH                  1u
#define FR_EEIFER_LRCE_IE(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_LRCE_IE_SHIFT))&FR_EEIFER_LRCE_IE_MASK)
#define FR_EEIFER_LRNE_IE_MASK                   0x8u
#define FR_EEIFER_LRNE_IE_SHIFT                  3u
#define FR_EEIFER_LRNE_IE_WIDTH                  1u
#define FR_EEIFER_LRNE_IE(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_LRNE_IE_SHIFT))&FR_EEIFER_LRNE_IE_MASK)
#define FR_EEIFER_DRCE_IF_MASK                   0x100u
#define FR_EEIFER_DRCE_IF_SHIFT                  8u
#define FR_EEIFER_DRCE_IF_WIDTH                  1u
#define FR_EEIFER_DRCE_IF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_DRCE_IF_SHIFT))&FR_EEIFER_DRCE_IF_MASK)
#define FR_EEIFER_DRNE_IF_MASK                   0x200u
#define FR_EEIFER_DRNE_IF_SHIFT                  9u
#define FR_EEIFER_DRNE_IF_WIDTH                  1u
#define FR_EEIFER_DRNE_IF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_DRNE_IF_SHIFT))&FR_EEIFER_DRNE_IF_MASK)
#define FR_EEIFER_LRCE_IF_MASK                   0x400u
#define FR_EEIFER_LRCE_IF_SHIFT                  10u
#define FR_EEIFER_LRCE_IF_WIDTH                  1u
#define FR_EEIFER_LRCE_IF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_LRCE_IF_SHIFT))&FR_EEIFER_LRCE_IF_MASK)
#define FR_EEIFER_LRNE_IF_MASK                   0x800u
#define FR_EEIFER_LRNE_IF_SHIFT                  11u
#define FR_EEIFER_LRNE_IF_WIDTH                  1u
#define FR_EEIFER_LRNE_IF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_LRNE_IF_SHIFT))&FR_EEIFER_LRNE_IF_MASK)
#define FR_EEIFER_DRCE_OF_MASK                   0x1000u
#define FR_EEIFER_DRCE_OF_SHIFT                  12u
#define FR_EEIFER_DRCE_OF_WIDTH                  1u
#define FR_EEIFER_DRCE_OF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_DRCE_OF_SHIFT))&FR_EEIFER_DRCE_OF_MASK)
#define FR_EEIFER_DRNE_OF_MASK                   0x2000u
#define FR_EEIFER_DRNE_OF_SHIFT                  13u
#define FR_EEIFER_DRNE_OF_WIDTH                  1u
#define FR_EEIFER_DRNE_OF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_DRNE_OF_SHIFT))&FR_EEIFER_DRNE_OF_MASK)
#define FR_EEIFER_LRCE_OF_MASK                   0x4000u
#define FR_EEIFER_LRCE_OF_SHIFT                  14u
#define FR_EEIFER_LRCE_OF_WIDTH                  1u
#define FR_EEIFER_LRCE_OF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_LRCE_OF_SHIFT))&FR_EEIFER_LRCE_OF_MASK)
#define FR_EEIFER_LRNE_OF_MASK                   0x8000u
#define FR_EEIFER_LRNE_OF_SHIFT                  15u
#define FR_EEIFER_LRNE_OF_WIDTH                  1u
#define FR_EEIFER_LRNE_OF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_LRNE_OF_SHIFT))&FR_EEIFER_LRNE_OF_MASK)
/* EERICR Bit Fields */
#define FR_EERICR_EIE_MASK                       0x1u
#define FR_EERICR_EIE_SHIFT                      0u
#define FR_EERICR_EIE_WIDTH                      1u
#define FR_EERICR_EIE(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EERICR_EIE_SHIFT))&FR_EERICR_EIE_MASK)
#define FR_EERICR_EIM_MASK                       0x2u
#define FR_EERICR_EIM_SHIFT                      1u
#define FR_EERICR_EIM_WIDTH                      1u
#define FR_EERICR_EIM(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EERICR_EIM_SHIFT))&FR_EERICR_EIM_MASK)
#define FR_EERICR_ERM_MASK                       0x10u
#define FR_EERICR_ERM_SHIFT                      4u
#define FR_EERICR_ERM_WIDTH                      1u
#define FR_EERICR_ERM(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EERICR_ERM_SHIFT))&FR_EERICR_ERM_MASK)
#define FR_EERICR_ERS_MASK                       0x300u
#define FR_EERICR_ERS_SHIFT                      8u
#define FR_EERICR_ERS_WIDTH                      2u
#define FR_EERICR_ERS(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EERICR_ERS_SHIFT))&FR_EERICR_ERS_MASK)
#define FR_EERICR_BSY_MASK                       0x8000u
#define FR_EERICR_BSY_SHIFT                      15u
#define FR_EERICR_BSY_WIDTH                      1u
#define FR_EERICR_BSY(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EERICR_BSY_SHIFT))&FR_EERICR_BSY_MASK)
/* EERAR Bit Fields */
#define FR_EERAR_ADDR_MASK                       0xFFFu
#define FR_EERAR_ADDR_SHIFT                      0u
#define FR_EERAR_ADDR_WIDTH                      12u
#define FR_EERAR_ADDR(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EERAR_ADDR_SHIFT))&FR_EERAR_ADDR_MASK)
#define FR_EERAR_BANK_MASK                       0x7000u
#define FR_EERAR_BANK_SHIFT                      12u
#define FR_EERAR_BANK_WIDTH                      3u
#define FR_EERAR_BANK(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EERAR_BANK_SHIFT))&FR_EERAR_BANK_MASK)
#define FR_EERAR_MID_MASK                        0x8000u
#define FR_EERAR_MID_SHIFT                       15u
#define FR_EERAR_MID_WIDTH                       1u
#define FR_EERAR_MID(x)                          (((uint16_t)(((uint16_t)(x))<<FR_EERAR_MID_SHIFT))&FR_EERAR_MID_MASK)
/* EERDR Bit Fields */
#define FR_EERDR_DATA_MASK                       0xFFFFu
#define FR_EERDR_DATA_SHIFT                      0u
#define FR_EERDR_DATA_WIDTH                      16u
#define FR_EERDR_DATA(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EERDR_DATA_SHIFT))&FR_EERDR_DATA_MASK)
/* EERCR Bit Fields */
#define FR_EERCR_CODE_MASK                       0x1Fu
#define FR_EERCR_CODE_SHIFT                      0u
#define FR_EERCR_CODE_WIDTH                      5u
#define FR_EERCR_CODE(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EERCR_CODE_SHIFT))&FR_EERCR_CODE_MASK)
/* EEIAR Bit Fields */
#define FR_EEIAR_ADDR_MASK                       0xFFFu
#define FR_EEIAR_ADDR_SHIFT                      0u
#define FR_EEIAR_ADDR_WIDTH                      12u
#define FR_EEIAR_ADDR(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EEIAR_ADDR_SHIFT))&FR_EEIAR_ADDR_MASK)
#define FR_EEIAR_BANK_MASK                       0x7000u
#define FR_EEIAR_BANK_SHIFT                      12u
#define FR_EEIAR_BANK_WIDTH                      3u
#define FR_EEIAR_BANK(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EEIAR_BANK_SHIFT))&FR_EEIAR_BANK_MASK)
#define FR_EEIAR_MID_MASK                        0x8000u
#define FR_EEIAR_MID_SHIFT                       15u
#define FR_EEIAR_MID_WIDTH                       1u
#define FR_EEIAR_MID(x)                          (((uint16_t)(((uint16_t)(x))<<FR_EEIAR_MID_SHIFT))&FR_EEIAR_MID_MASK)
/* EEIDR Bit Fields */
#define FR_EEIDR_DATA_MASK                       0xFFFFu
#define FR_EEIDR_DATA_SHIFT                      0u
#define FR_EEIDR_DATA_WIDTH                      16u
#define FR_EEIDR_DATA(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EEIDR_DATA_SHIFT))&FR_EEIDR_DATA_MASK)
/* EEICR Bit Fields */
#define FR_EEICR_CODE_MASK                       0x1Fu
#define FR_EEICR_CODE_SHIFT                      0u
#define FR_EEICR_CODE_WIDTH                      5u
#define FR_EEICR_CODE(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EEICR_CODE_SHIFT))&FR_EEICR_CODE_MASK)
/* CCSR Bit Fields */
#define FR_CCSR_MBIF_MASK                        0x1u
#define FR_CCSR_MBIF_SHIFT                       0u
#define FR_CCSR_MBIF_WIDTH                       1u
#define FR_CCSR_MBIF(x)                          (((uint16_t)(((uint16_t)(x))<<FR_CCSR_MBIF_SHIFT))&FR_CCSR_MBIF_MASK)
#define FR_CCSR_LCKS_MASK                        0x2u
#define FR_CCSR_LCKS_SHIFT                       1u
#define FR_CCSR_LCKS_WIDTH                       1u
#define FR_CCSR_LCKS(x)                          (((uint16_t)(((uint16_t)(x))<<FR_CCSR_LCKS_SHIFT))&FR_CCSR_LCKS_MASK)
#define FR_CCSR_EDS_MASK                         0x4u
#define FR_CCSR_EDS_SHIFT                        2u
#define FR_CCSR_EDS_WIDTH                        1u
#define FR_CCSR_EDS(x)                           (((uint16_t)(((uint16_t)(x))<<FR_CCSR_EDS_SHIFT))&FR_CCSR_EDS_MASK)
#define FR_CCSR_DVAL_MASK                        0x8u
#define FR_CCSR_DVAL_SHIFT                       3u
#define FR_CCSR_DVAL_WIDTH                       1u
#define FR_CCSR_DVAL(x)                          (((uint16_t)(((uint16_t)(x))<<FR_CCSR_DVAL_SHIFT))&FR_CCSR_DVAL_MASK)
#define FR_CCSR_DUP_MASK                         0x10u
#define FR_CCSR_DUP_SHIFT                        4u
#define FR_CCSR_DUP_WIDTH                        1u
#define FR_CCSR_DUP(x)                           (((uint16_t)(((uint16_t)(x))<<FR_CCSR_DUP_SHIFT))&FR_CCSR_DUP_MASK)
#define FR_CCSR_MBIE_MASK                        0x100u
#define FR_CCSR_MBIE_SHIFT                       8u
#define FR_CCSR_MBIE_WIDTH                       1u
#define FR_CCSR_MBIE(x)                          (((uint16_t)(((uint16_t)(x))<<FR_CCSR_MBIE_SHIFT))&FR_CCSR_MBIE_MASK)
#define FR_CCSR_LCKT_MASK                        0x200u
#define FR_CCSR_LCKT_SHIFT                       9u
#define FR_CCSR_LCKT_WIDTH                       1u
#define FR_CCSR_LCKT(x)                          (((uint16_t)(((uint16_t)(x))<<FR_CCSR_LCKT_SHIFT))&FR_CCSR_LCKT_MASK)
#define FR_CCSR_EDT_MASK                         0x400u
#define FR_CCSR_EDT_SHIFT                        10u
#define FR_CCSR_EDT_WIDTH                        1u
#define FR_CCSR_EDT(x)                           (((uint16_t)(((uint16_t)(x))<<FR_CCSR_EDT_SHIFT))&FR_CCSR_EDT_MASK)
#define FR_CCSR_CMT_MASK                         0x800u
#define FR_CCSR_CMT_SHIFT                        11u
#define FR_CCSR_CMT_WIDTH                        1u
#define FR_CCSR_CMT(x)                           (((uint16_t)(((uint16_t)(x))<<FR_CCSR_CMT_SHIFT))&FR_CCSR_CMT_MASK)
#define FR_CCSR_MTD_MASK                         0x1000u
#define FR_CCSR_MTD_SHIFT                        12u
#define FR_CCSR_MTD_WIDTH                        1u
#define FR_CCSR_MTD(x)                           (((uint16_t)(((uint16_t)(x))<<FR_CCSR_MTD_SHIFT))&FR_CCSR_MTD_MASK)
/* CCFR Bit Fields */
#define FR_CCFR_CCFVAL_MASK                      0x3Fu
#define FR_CCFR_CCFVAL_SHIFT                     0u
#define FR_CCFR_CCFVAL_WIDTH                     6u
#define FR_CCFR_CCFVAL(x)                        (((uint16_t)(((uint16_t)(x))<<FR_CCFR_CCFVAL_SHIFT))&FR_CCFR_CCFVAL_MASK)
#define FR_CCFR_CCFMSK_MASK                      0xFC0u
#define FR_CCFR_CCFMSK_SHIFT                     6u
#define FR_CCFR_CCFMSK_WIDTH                     6u
#define FR_CCFR_CCFMSK(x)                        (((uint16_t)(((uint16_t)(x))<<FR_CCFR_CCFMSK_SHIFT))&FR_CCFR_CCFMSK_MASK)
#define FR_CCFR_CCFE_MASK                        0x1000u
#define FR_CCFR_CCFE_SHIFT                       12u
#define FR_CCFR_CCFE_WIDTH                       1u
#define FR_CCFR_CCFE(x)                          (((uint16_t)(((uint16_t)(x))<<FR_CCFR_CCFE_SHIFT))&FR_CCFR_CCFE_MASK)
#define FR_CCFR_CHB_MASK                         0x2000u
#define FR_CCFR_CHB_SHIFT                        13u
#define FR_CCFR_CHB_WIDTH                        1u
#define FR_CCFR_CHB(x)                           (((uint16_t)(((uint16_t)(x))<<FR_CCFR_CHB_SHIFT))&FR_CCFR_CHB_MASK)
#define FR_CCFR_CHA_MASK                         0x4000u
#define FR_CCFR_CHA_SHIFT                        14u
#define FR_CCFR_CHA_WIDTH                        1u
#define FR_CCFR_CHA(x)                           (((uint16_t)(((uint16_t)(x))<<FR_CCFR_CHA_SHIFT))&FR_CCFR_CHA_MASK)
#define FR_CCFR_MTM_MASK                         0x8000u
#define FR_CCFR_MTM_SHIFT                        15u
#define FR_CCFR_MTM_WIDTH                        1u
#define FR_CCFR_MTM(x)                           (((uint16_t)(((uint16_t)(x))<<FR_CCFR_MTM_SHIFT))&FR_CCFR_MTM_MASK)
/* FIDR Bit Fields */
#define FR_FIDR_FID_MASK                         0x7FFu
#define FR_FIDR_FID_SHIFT                        0u
#define FR_FIDR_FID_WIDTH                        11u
#define FR_FIDR_FID(x)                           (((uint16_t)(((uint16_t)(x))<<FR_FIDR_FID_SHIFT))&FR_FIDR_FID_MASK)
/* IDXR Bit Fields */
#define FR_IDXR_MBIDX_MASK                       0xFFu
#define FR_IDXR_MBIDX_SHIFT                      0u
#define FR_IDXR_MBIDX_WIDTH                      8u
#define FR_IDXR_MBIDX(x)                         (((uint16_t)(((uint16_t)(x))<<FR_IDXR_MBIDX_SHIFT))&FR_IDXR_MBIDX_MASK)
/* MBDOR Bit Fields */
#define FR_MBDOR_MBDO_MASK                       0xFFFFu
#define FR_MBDOR_MBDO_SHIFT                      0u
#define FR_MBDOR_MBDO_WIDTH                      16u
#define FR_MBDOR_MBDO(x)                         (((uint16_t)(((uint16_t)(x))<<FR_MBDOR_MBDO_SHIFT))&FR_MBDOR_MBDO_MASK)
/* LEETR Bit Fields */
#define FR_LEETR_LEETD_MASK                      0xFFFFu
#define FR_LEETR_LEETD_SHIFT                     0u
#define FR_LEETR_LEETD_WIDTH                     16u
#define FR_LEETR_LEETD(x)                        (((uint16_t)(((uint16_t)(x))<<FR_LEETR_LEETD_SHIFT))&FR_LEETR_LEETD_MASK)

/*!
 * @}
 */ /* end of group FR_Register_Masks */


/*!
 * @}
 */ /* end of group FR_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- FTM Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup FTM_Peripheral_Access_Layer FTM Peripheral Access Layer
 * @{
 */


/** FTM - Size of Registers Arrays */
#define FTM_CONTROLS_COUNT                       8u

/** FTM - Register Layout Typedef */
typedef struct {
  __IO uint32_t SC;                                /**< Status And Control, offset: 0x0 */
  __IO uint32_t CNT;                               /**< Counter, offset: 0x4 */
  __IO uint32_t MOD;                               /**< Modulo, offset: 0x8 */
  struct {                                         /* offset: 0xC, array step: 0x8 */
    __IO uint32_t CnSC;                              /**< Channel (n) Status And Control, array offset: 0xC, array step: 0x8 */
    __IO uint32_t CnV;                               /**< Channel (n) Value, array offset: 0x10, array step: 0x8 */
  } CONTROLS[FTM_CONTROLS_COUNT];
  __IO uint32_t CNTIN;                             /**< Counter Initial Value, offset: 0x4C */
  __IO uint32_t STATUS;                            /**< Capture And Compare Status, offset: 0x50 */
  __IO uint32_t MODE;                              /**< Features Mode Selection, offset: 0x54 */
  __IO uint32_t SYNC;                              /**< Synchronization, offset: 0x58 */
  __IO uint32_t OUTINIT;                           /**< Initial State For Channels Output, offset: 0x5C */
  __IO uint32_t OUTMASK;                           /**< Output Mask, offset: 0x60 */
  __IO uint32_t COMBINE;                           /**< Function For Linked Channels, offset: 0x64 */
  __IO uint32_t DEADTIME;                          /**< Deadtime Insertion Control, offset: 0x68 */
  __IO uint32_t EXTTRIG;                           /**< FTM External Trigger, offset: 0x6C */
  __IO uint32_t POL;                               /**< Channels Polarity, offset: 0x70 */
  __IO uint32_t FMS;                               /**< Fault Mode Status, offset: 0x74 */
  __IO uint32_t FILTER;                            /**< Input Capture Filter Control, offset: 0x78 */
       uint8_t RESERVED_0[4];
  __IO uint32_t QDCTRL;                            /**< Quadrature Decoder Control And Status, offset: 0x80 */
  __IO uint32_t CONF;                              /**< Configuration, offset: 0x84 */
       uint8_t RESERVED_1[4];
  __IO uint32_t SYNCONF;                           /**< Synchronization Configuration, offset: 0x8C */
  __IO uint32_t INVCTRL;                           /**< FTM Inverting Control, offset: 0x90 */
  __IO uint32_t SWOCTRL;                           /**< FTM Software Output Control, offset: 0x94 */
  __IO uint32_t PWMLOAD;                           /**< FTM PWM Load, offset: 0x98 */
} FTM_Type, *FTM_MemMapPtr;

 /** Number of instances of the FTM module. */
#define FTM_INSTANCE_COUNT                       (2u)


/* FTM - Peripheral instance base addresses */
/** Peripheral FTM_0 base address */
#define FTM_0_BASE                               (0x4004F000u)
/** Peripheral FTM_0 base pointer */
#define FTM_0                                    ((FTM_Type *)FTM_0_BASE)
/** Peripheral FTM_1 base address */
#define FTM_1_BASE                               (0x400B6000u)
/** Peripheral FTM_1 base pointer */
#define FTM_1                                    ((FTM_Type *)FTM_1_BASE)
/** Array initializer of FTM peripheral base addresses */
#define FTM_BASE_ADDRS                           { FTM_0_BASE, FTM_1_BASE }
/** Array initializer of FTM peripheral base pointers */
#define FTM_BASE_PTRS                            { FTM_0, FTM_1 }
 /** Number of interrupt vector arrays for the FTM module. */
#define FTM_IRQS_ARR_COUNT                       (2u)
 /** Number of interrupt channels for the FTM module. */
#define FTM_IRQS_CH_COUNT                        (6u)
/** Interrupt vectors for the FTM peripheral type */
#define FTM_IRQS                                 { { FTM0_IRQn, FTM0_IRQn, FTM0_IRQn, FTM0_IRQn, FTM0_IRQn, FTM0_IRQn }, \
                                                   { FTM1_IRQn, FTM1_IRQn, FTM1_IRQn, FTM1_IRQn, FTM1_IRQn, FTM1_IRQn } }
#define FTM_Overflow_IRQS                        { FTM0_IRQn, FTM1_IRQn }
/* ----------------------------------------------------------------------------
   -- FTM Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup FTM_Register_Masks FTM Register Masks
 * @{
 */

/* SC Bit Fields */
#define FTM_SC_PS_MASK                           0x7u
#define FTM_SC_PS_SHIFT                          0u
#define FTM_SC_PS_WIDTH                          3u
#define FTM_SC_PS(x)                             (((uint32_t)(((uint32_t)(x))<<FTM_SC_PS_SHIFT))&FTM_SC_PS_MASK)
#define FTM_SC_CLKS_MASK                         0x18u
#define FTM_SC_CLKS_SHIFT                        3u
#define FTM_SC_CLKS_WIDTH                        2u
#define FTM_SC_CLKS(x)                           (((uint32_t)(((uint32_t)(x))<<FTM_SC_CLKS_SHIFT))&FTM_SC_CLKS_MASK)
#define FTM_SC_CPWMS_MASK                        0x20u
#define FTM_SC_CPWMS_SHIFT                       5u
#define FTM_SC_CPWMS_WIDTH                       1u
#define FTM_SC_CPWMS(x)                          (((uint32_t)(((uint32_t)(x))<<FTM_SC_CPWMS_SHIFT))&FTM_SC_CPWMS_MASK)
#define FTM_SC_TOIE_MASK                         0x40u
#define FTM_SC_TOIE_SHIFT                        6u
#define FTM_SC_TOIE_WIDTH                        1u
#define FTM_SC_TOIE(x)                           (((uint32_t)(((uint32_t)(x))<<FTM_SC_TOIE_SHIFT))&FTM_SC_TOIE_MASK)
#define FTM_SC_TOF_MASK                          0x80u
#define FTM_SC_TOF_SHIFT                         7u
#define FTM_SC_TOF_WIDTH                         1u
#define FTM_SC_TOF(x)                            (((uint32_t)(((uint32_t)(x))<<FTM_SC_TOF_SHIFT))&FTM_SC_TOF_MASK)
/* CNT Bit Fields */
#define FTM_CNT_COUNT_MASK                       0xFFFFu
#define FTM_CNT_COUNT_SHIFT                      0u
#define FTM_CNT_COUNT_WIDTH                      16u
#define FTM_CNT_COUNT(x)                         (((uint32_t)(((uint32_t)(x))<<FTM_CNT_COUNT_SHIFT))&FTM_CNT_COUNT_MASK)
/* MOD Bit Fields */
#define FTM_MOD_MOD_MASK                         0xFFFFu
#define FTM_MOD_MOD_SHIFT                        0u
#define FTM_MOD_MOD_WIDTH                        16u
#define FTM_MOD_MOD(x)                           (((uint32_t)(((uint32_t)(x))<<FTM_MOD_MOD_SHIFT))&FTM_MOD_MOD_MASK)
/* CnSC Bit Fields */
#define FTM_CnSC_DMA_MASK                        0x1u
#define FTM_CnSC_DMA_SHIFT                       0u
#define FTM_CnSC_DMA_WIDTH                       1u
#define FTM_CnSC_DMA(x)                          (((uint32_t)(((uint32_t)(x))<<FTM_CnSC_DMA_SHIFT))&FTM_CnSC_DMA_MASK)
#define FTM_CnSC_ICRST_MASK                      0x2u
#define FTM_CnSC_ICRST_SHIFT                     1u
#define FTM_CnSC_ICRST_WIDTH                     1u
#define FTM_CnSC_ICRST(x)                        (((uint32_t)(((uint32_t)(x))<<FTM_CnSC_ICRST_SHIFT))&FTM_CnSC_ICRST_MASK)
#define FTM_CnSC_ELSA_MASK                       0x4u
#define FTM_CnSC_ELSA_SHIFT                      2u
#define FTM_CnSC_ELSA_WIDTH                      1u
#define FTM_CnSC_ELSA(x)                         (((uint32_t)(((uint32_t)(x))<<FTM_CnSC_ELSA_SHIFT))&FTM_CnSC_ELSA_MASK)
#define FTM_CnSC_ELSB_MASK                       0x8u
#define FTM_CnSC_ELSB_SHIFT                      3u
#define FTM_CnSC_ELSB_WIDTH                      1u
#define FTM_CnSC_ELSB(x)                         (((uint32_t)(((uint32_t)(x))<<FTM_CnSC_ELSB_SHIFT))&FTM_CnSC_ELSB_MASK)
#define FTM_CnSC_MSA_MASK                        0x10u
#define FTM_CnSC_MSA_SHIFT                       4u
#define FTM_CnSC_MSA_WIDTH                       1u
#define FTM_CnSC_MSA(x)                          (((uint32_t)(((uint32_t)(x))<<FTM_CnSC_MSA_SHIFT))&FTM_CnSC_MSA_MASK)
#define FTM_CnSC_MSB_MASK                        0x20u
#define FTM_CnSC_MSB_SHIFT                       5u
#define FTM_CnSC_MSB_WIDTH                       1u
#define FTM_CnSC_MSB(x)                          (((uint32_t)(((uint32_t)(x))<<FTM_CnSC_MSB_SHIFT))&FTM_CnSC_MSB_MASK)
#define FTM_CnSC_CHIE_MASK                       0x40u
#define FTM_CnSC_CHIE_SHIFT                      6u
#define FTM_CnSC_CHIE_WIDTH                      1u
#define FTM_CnSC_CHIE(x)                         (((uint32_t)(((uint32_t)(x))<<FTM_CnSC_CHIE_SHIFT))&FTM_CnSC_CHIE_MASK)
#define FTM_CnSC_CHF_MASK                        0x80u
#define FTM_CnSC_CHF_SHIFT                       7u
#define FTM_CnSC_CHF_WIDTH                       1u
#define FTM_CnSC_CHF(x)                          (((uint32_t)(((uint32_t)(x))<<FTM_CnSC_CHF_SHIFT))&FTM_CnSC_CHF_MASK)
/* CnV Bit Fields */
#define FTM_CnV_VAL_MASK                         0xFFFFu
#define FTM_CnV_VAL_SHIFT                        0u
#define FTM_CnV_VAL_WIDTH                        16u
#define FTM_CnV_VAL(x)                           (((uint32_t)(((uint32_t)(x))<<FTM_CnV_VAL_SHIFT))&FTM_CnV_VAL_MASK)
/* CNTIN Bit Fields */
#define FTM_CNTIN_INIT_MASK                      0xFFFFu
#define FTM_CNTIN_INIT_SHIFT                     0u
#define FTM_CNTIN_INIT_WIDTH                     16u
#define FTM_CNTIN_INIT(x)                        (((uint32_t)(((uint32_t)(x))<<FTM_CNTIN_INIT_SHIFT))&FTM_CNTIN_INIT_MASK)
/* STATUS Bit Fields */
#define FTM_STATUS_CH0F_MASK                     0x1u
#define FTM_STATUS_CH0F_SHIFT                    0u
#define FTM_STATUS_CH0F_WIDTH                    1u
#define FTM_STATUS_CH0F(x)                       (((uint32_t)(((uint32_t)(x))<<FTM_STATUS_CH0F_SHIFT))&FTM_STATUS_CH0F_MASK)
#define FTM_STATUS_CH1F_MASK                     0x2u
#define FTM_STATUS_CH1F_SHIFT                    1u
#define FTM_STATUS_CH1F_WIDTH                    1u
#define FTM_STATUS_CH1F(x)                       (((uint32_t)(((uint32_t)(x))<<FTM_STATUS_CH1F_SHIFT))&FTM_STATUS_CH1F_MASK)
#define FTM_STATUS_CH2F_MASK                     0x4u
#define FTM_STATUS_CH2F_SHIFT                    2u
#define FTM_STATUS_CH2F_WIDTH                    1u
#define FTM_STATUS_CH2F(x)                       (((uint32_t)(((uint32_t)(x))<<FTM_STATUS_CH2F_SHIFT))&FTM_STATUS_CH2F_MASK)
#define FTM_STATUS_CH3F_MASK                     0x8u
#define FTM_STATUS_CH3F_SHIFT                    3u
#define FTM_STATUS_CH3F_WIDTH                    1u
#define FTM_STATUS_CH3F(x)                       (((uint32_t)(((uint32_t)(x))<<FTM_STATUS_CH3F_SHIFT))&FTM_STATUS_CH3F_MASK)
#define FTM_STATUS_CH4F_MASK                     0x10u
#define FTM_STATUS_CH4F_SHIFT                    4u
#define FTM_STATUS_CH4F_WIDTH                    1u
#define FTM_STATUS_CH4F(x)                       (((uint32_t)(((uint32_t)(x))<<FTM_STATUS_CH4F_SHIFT))&FTM_STATUS_CH4F_MASK)
#define FTM_STATUS_CH5F_MASK                     0x20u
#define FTM_STATUS_CH5F_SHIFT                    5u
#define FTM_STATUS_CH5F_WIDTH                    1u
#define FTM_STATUS_CH5F(x)                       (((uint32_t)(((uint32_t)(x))<<FTM_STATUS_CH5F_SHIFT))&FTM_STATUS_CH5F_MASK)
#define FTM_STATUS_CH6F_MASK                     0x40u
#define FTM_STATUS_CH6F_SHIFT                    6u
#define FTM_STATUS_CH6F_WIDTH                    1u
#define FTM_STATUS_CH6F(x)                       (((uint32_t)(((uint32_t)(x))<<FTM_STATUS_CH6F_SHIFT))&FTM_STATUS_CH6F_MASK)
#define FTM_STATUS_CH7F_MASK                     0x80u
#define FTM_STATUS_CH7F_SHIFT                    7u
#define FTM_STATUS_CH7F_WIDTH                    1u
#define FTM_STATUS_CH7F(x)                       (((uint32_t)(((uint32_t)(x))<<FTM_STATUS_CH7F_SHIFT))&FTM_STATUS_CH7F_MASK)
/* MODE Bit Fields */
#define FTM_MODE_FTMEN_MASK                      0x1u
#define FTM_MODE_FTMEN_SHIFT                     0u
#define FTM_MODE_FTMEN_WIDTH                     1u
#define FTM_MODE_FTMEN(x)                        (((uint32_t)(((uint32_t)(x))<<FTM_MODE_FTMEN_SHIFT))&FTM_MODE_FTMEN_MASK)
#define FTM_MODE_INIT_MASK                       0x2u
#define FTM_MODE_INIT_SHIFT                      1u
#define FTM_MODE_INIT_WIDTH                      1u
#define FTM_MODE_INIT(x)                         (((uint32_t)(((uint32_t)(x))<<FTM_MODE_INIT_SHIFT))&FTM_MODE_INIT_MASK)
#define FTM_MODE_WPDIS_MASK                      0x4u
#define FTM_MODE_WPDIS_SHIFT                     2u
#define FTM_MODE_WPDIS_WIDTH                     1u
#define FTM_MODE_WPDIS(x)                        (((uint32_t)(((uint32_t)(x))<<FTM_MODE_WPDIS_SHIFT))&FTM_MODE_WPDIS_MASK)
#define FTM_MODE_PWMSYNC_MASK                    0x8u
#define FTM_MODE_PWMSYNC_SHIFT                   3u
#define FTM_MODE_PWMSYNC_WIDTH                   1u
#define FTM_MODE_PWMSYNC(x)                      (((uint32_t)(((uint32_t)(x))<<FTM_MODE_PWMSYNC_SHIFT))&FTM_MODE_PWMSYNC_MASK)
#define FTM_MODE_CAPTEST_MASK                    0x10u
#define FTM_MODE_CAPTEST_SHIFT                   4u
#define FTM_MODE_CAPTEST_WIDTH                   1u
#define FTM_MODE_CAPTEST(x)                      (((uint32_t)(((uint32_t)(x))<<FTM_MODE_CAPTEST_SHIFT))&FTM_MODE_CAPTEST_MASK)
/* SYNC Bit Fields */
#define FTM_SYNC_CNTMIN_MASK                     0x1u
#define FTM_SYNC_CNTMIN_SHIFT                    0u
#define FTM_SYNC_CNTMIN_WIDTH                    1u
#define FTM_SYNC_CNTMIN(x)                       (((uint32_t)(((uint32_t)(x))<<FTM_SYNC_CNTMIN_SHIFT))&FTM_SYNC_CNTMIN_MASK)
#define FTM_SYNC_CNTMAX_MASK                     0x2u
#define FTM_SYNC_CNTMAX_SHIFT                    1u
#define FTM_SYNC_CNTMAX_WIDTH                    1u
#define FTM_SYNC_CNTMAX(x)                       (((uint32_t)(((uint32_t)(x))<<FTM_SYNC_CNTMAX_SHIFT))&FTM_SYNC_CNTMAX_MASK)
#define FTM_SYNC_REINIT_MASK                     0x4u
#define FTM_SYNC_REINIT_SHIFT                    2u
#define FTM_SYNC_REINIT_WIDTH                    1u
#define FTM_SYNC_REINIT(x)                       (((uint32_t)(((uint32_t)(x))<<FTM_SYNC_REINIT_SHIFT))&FTM_SYNC_REINIT_MASK)
#define FTM_SYNC_SYNCHOM_MASK                    0x8u
#define FTM_SYNC_SYNCHOM_SHIFT                   3u
#define FTM_SYNC_SYNCHOM_WIDTH                   1u
#define FTM_SYNC_SYNCHOM(x)                      (((uint32_t)(((uint32_t)(x))<<FTM_SYNC_SYNCHOM_SHIFT))&FTM_SYNC_SYNCHOM_MASK)
#define FTM_SYNC_TRIG0_MASK                      0x10u
#define FTM_SYNC_TRIG0_SHIFT                     4u
#define FTM_SYNC_TRIG0_WIDTH                     1u
#define FTM_SYNC_TRIG0(x)                        (((uint32_t)(((uint32_t)(x))<<FTM_SYNC_TRIG0_SHIFT))&FTM_SYNC_TRIG0_MASK)
#define FTM_SYNC_TRIG1_MASK                      0x20u
#define FTM_SYNC_TRIG1_SHIFT                     5u
#define FTM_SYNC_TRIG1_WIDTH                     1u
#define FTM_SYNC_TRIG1(x)                        (((uint32_t)(((uint32_t)(x))<<FTM_SYNC_TRIG1_SHIFT))&FTM_SYNC_TRIG1_MASK)
#define FTM_SYNC_TRIG2_MASK                      0x40u
#define FTM_SYNC_TRIG2_SHIFT                     6u
#define FTM_SYNC_TRIG2_WIDTH                     1u
#define FTM_SYNC_TRIG2(x)                        (((uint32_t)(((uint32_t)(x))<<FTM_SYNC_TRIG2_SHIFT))&FTM_SYNC_TRIG2_MASK)
#define FTM_SYNC_SWSYNC_MASK                     0x80u
#define FTM_SYNC_SWSYNC_SHIFT                    7u
#define FTM_SYNC_SWSYNC_WIDTH                    1u
#define FTM_SYNC_SWSYNC(x)                       (((uint32_t)(((uint32_t)(x))<<FTM_SYNC_SWSYNC_SHIFT))&FTM_SYNC_SWSYNC_MASK)
/* OUTINIT Bit Fields */
#define FTM_OUTINIT_CH0OI_MASK                   0x1u
#define FTM_OUTINIT_CH0OI_SHIFT                  0u
#define FTM_OUTINIT_CH0OI_WIDTH                  1u
#define FTM_OUTINIT_CH0OI(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_OUTINIT_CH0OI_SHIFT))&FTM_OUTINIT_CH0OI_MASK)
#define FTM_OUTINIT_CH1OI_MASK                   0x2u
#define FTM_OUTINIT_CH1OI_SHIFT                  1u
#define FTM_OUTINIT_CH1OI_WIDTH                  1u
#define FTM_OUTINIT_CH1OI(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_OUTINIT_CH1OI_SHIFT))&FTM_OUTINIT_CH1OI_MASK)
#define FTM_OUTINIT_CH2OI_MASK                   0x4u
#define FTM_OUTINIT_CH2OI_SHIFT                  2u
#define FTM_OUTINIT_CH2OI_WIDTH                  1u
#define FTM_OUTINIT_CH2OI(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_OUTINIT_CH2OI_SHIFT))&FTM_OUTINIT_CH2OI_MASK)
#define FTM_OUTINIT_CH3OI_MASK                   0x8u
#define FTM_OUTINIT_CH3OI_SHIFT                  3u
#define FTM_OUTINIT_CH3OI_WIDTH                  1u
#define FTM_OUTINIT_CH3OI(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_OUTINIT_CH3OI_SHIFT))&FTM_OUTINIT_CH3OI_MASK)
#define FTM_OUTINIT_CH4OI_MASK                   0x10u
#define FTM_OUTINIT_CH4OI_SHIFT                  4u
#define FTM_OUTINIT_CH4OI_WIDTH                  1u
#define FTM_OUTINIT_CH4OI(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_OUTINIT_CH4OI_SHIFT))&FTM_OUTINIT_CH4OI_MASK)
#define FTM_OUTINIT_CH5OI_MASK                   0x20u
#define FTM_OUTINIT_CH5OI_SHIFT                  5u
#define FTM_OUTINIT_CH5OI_WIDTH                  1u
#define FTM_OUTINIT_CH5OI(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_OUTINIT_CH5OI_SHIFT))&FTM_OUTINIT_CH5OI_MASK)
#define FTM_OUTINIT_CH6OI_MASK                   0x40u
#define FTM_OUTINIT_CH6OI_SHIFT                  6u
#define FTM_OUTINIT_CH6OI_WIDTH                  1u
#define FTM_OUTINIT_CH6OI(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_OUTINIT_CH6OI_SHIFT))&FTM_OUTINIT_CH6OI_MASK)
#define FTM_OUTINIT_CH7OI_MASK                   0x80u
#define FTM_OUTINIT_CH7OI_SHIFT                  7u
#define FTM_OUTINIT_CH7OI_WIDTH                  1u
#define FTM_OUTINIT_CH7OI(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_OUTINIT_CH7OI_SHIFT))&FTM_OUTINIT_CH7OI_MASK)
/* OUTMASK Bit Fields */
#define FTM_OUTMASK_CH0OM_MASK                   0x1u
#define FTM_OUTMASK_CH0OM_SHIFT                  0u
#define FTM_OUTMASK_CH0OM_WIDTH                  1u
#define FTM_OUTMASK_CH0OM(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_OUTMASK_CH0OM_SHIFT))&FTM_OUTMASK_CH0OM_MASK)
#define FTM_OUTMASK_CH1OM_MASK                   0x2u
#define FTM_OUTMASK_CH1OM_SHIFT                  1u
#define FTM_OUTMASK_CH1OM_WIDTH                  1u
#define FTM_OUTMASK_CH1OM(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_OUTMASK_CH1OM_SHIFT))&FTM_OUTMASK_CH1OM_MASK)
#define FTM_OUTMASK_CH2OM_MASK                   0x4u
#define FTM_OUTMASK_CH2OM_SHIFT                  2u
#define FTM_OUTMASK_CH2OM_WIDTH                  1u
#define FTM_OUTMASK_CH2OM(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_OUTMASK_CH2OM_SHIFT))&FTM_OUTMASK_CH2OM_MASK)
#define FTM_OUTMASK_CH3OM_MASK                   0x8u
#define FTM_OUTMASK_CH3OM_SHIFT                  3u
#define FTM_OUTMASK_CH3OM_WIDTH                  1u
#define FTM_OUTMASK_CH3OM(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_OUTMASK_CH3OM_SHIFT))&FTM_OUTMASK_CH3OM_MASK)
#define FTM_OUTMASK_CH4OM_MASK                   0x10u
#define FTM_OUTMASK_CH4OM_SHIFT                  4u
#define FTM_OUTMASK_CH4OM_WIDTH                  1u
#define FTM_OUTMASK_CH4OM(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_OUTMASK_CH4OM_SHIFT))&FTM_OUTMASK_CH4OM_MASK)
#define FTM_OUTMASK_CH5OM_MASK                   0x20u
#define FTM_OUTMASK_CH5OM_SHIFT                  5u
#define FTM_OUTMASK_CH5OM_WIDTH                  1u
#define FTM_OUTMASK_CH5OM(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_OUTMASK_CH5OM_SHIFT))&FTM_OUTMASK_CH5OM_MASK)
#define FTM_OUTMASK_CH6OM_MASK                   0x40u
#define FTM_OUTMASK_CH6OM_SHIFT                  6u
#define FTM_OUTMASK_CH6OM_WIDTH                  1u
#define FTM_OUTMASK_CH6OM(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_OUTMASK_CH6OM_SHIFT))&FTM_OUTMASK_CH6OM_MASK)
#define FTM_OUTMASK_CH7OM_MASK                   0x80u
#define FTM_OUTMASK_CH7OM_SHIFT                  7u
#define FTM_OUTMASK_CH7OM_WIDTH                  1u
#define FTM_OUTMASK_CH7OM(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_OUTMASK_CH7OM_SHIFT))&FTM_OUTMASK_CH7OM_MASK)
/* COMBINE Bit Fields */
#define FTM_COMBINE_COMBINE0_MASK                0x1u
#define FTM_COMBINE_COMBINE0_SHIFT               0u
#define FTM_COMBINE_COMBINE0_WIDTH               1u
#define FTM_COMBINE_COMBINE0(x)                  (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_COMBINE0_SHIFT))&FTM_COMBINE_COMBINE0_MASK)
#define FTM_COMBINE_COMP0_MASK                   0x2u
#define FTM_COMBINE_COMP0_SHIFT                  1u
#define FTM_COMBINE_COMP0_WIDTH                  1u
#define FTM_COMBINE_COMP0(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_COMP0_SHIFT))&FTM_COMBINE_COMP0_MASK)
#define FTM_COMBINE_DECAPEN0_MASK                0x4u
#define FTM_COMBINE_DECAPEN0_SHIFT               2u
#define FTM_COMBINE_DECAPEN0_WIDTH               1u
#define FTM_COMBINE_DECAPEN0(x)                  (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_DECAPEN0_SHIFT))&FTM_COMBINE_DECAPEN0_MASK)
#define FTM_COMBINE_DECAP0_MASK                  0x8u
#define FTM_COMBINE_DECAP0_SHIFT                 3u
#define FTM_COMBINE_DECAP0_WIDTH                 1u
#define FTM_COMBINE_DECAP0(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_DECAP0_SHIFT))&FTM_COMBINE_DECAP0_MASK)
#define FTM_COMBINE_DTEN0_MASK                   0x10u
#define FTM_COMBINE_DTEN0_SHIFT                  4u
#define FTM_COMBINE_DTEN0_WIDTH                  1u
#define FTM_COMBINE_DTEN0(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_DTEN0_SHIFT))&FTM_COMBINE_DTEN0_MASK)
#define FTM_COMBINE_SYNCEN0_MASK                 0x20u
#define FTM_COMBINE_SYNCEN0_SHIFT                5u
#define FTM_COMBINE_SYNCEN0_WIDTH                1u
#define FTM_COMBINE_SYNCEN0(x)                   (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_SYNCEN0_SHIFT))&FTM_COMBINE_SYNCEN0_MASK)
#define FTM_COMBINE_COMBINE1_MASK                0x100u
#define FTM_COMBINE_COMBINE1_SHIFT               8u
#define FTM_COMBINE_COMBINE1_WIDTH               1u
#define FTM_COMBINE_COMBINE1(x)                  (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_COMBINE1_SHIFT))&FTM_COMBINE_COMBINE1_MASK)
#define FTM_COMBINE_COMP1_MASK                   0x200u
#define FTM_COMBINE_COMP1_SHIFT                  9u
#define FTM_COMBINE_COMP1_WIDTH                  1u
#define FTM_COMBINE_COMP1(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_COMP1_SHIFT))&FTM_COMBINE_COMP1_MASK)
#define FTM_COMBINE_DECAPEN1_MASK                0x400u
#define FTM_COMBINE_DECAPEN1_SHIFT               10u
#define FTM_COMBINE_DECAPEN1_WIDTH               1u
#define FTM_COMBINE_DECAPEN1(x)                  (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_DECAPEN1_SHIFT))&FTM_COMBINE_DECAPEN1_MASK)
#define FTM_COMBINE_DECAP1_MASK                  0x800u
#define FTM_COMBINE_DECAP1_SHIFT                 11u
#define FTM_COMBINE_DECAP1_WIDTH                 1u
#define FTM_COMBINE_DECAP1(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_DECAP1_SHIFT))&FTM_COMBINE_DECAP1_MASK)
#define FTM_COMBINE_DTEN1_MASK                   0x1000u
#define FTM_COMBINE_DTEN1_SHIFT                  12u
#define FTM_COMBINE_DTEN1_WIDTH                  1u
#define FTM_COMBINE_DTEN1(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_DTEN1_SHIFT))&FTM_COMBINE_DTEN1_MASK)
#define FTM_COMBINE_SYNCEN1_MASK                 0x2000u
#define FTM_COMBINE_SYNCEN1_SHIFT                13u
#define FTM_COMBINE_SYNCEN1_WIDTH                1u
#define FTM_COMBINE_SYNCEN1(x)                   (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_SYNCEN1_SHIFT))&FTM_COMBINE_SYNCEN1_MASK)
#define FTM_COMBINE_COMBINE2_MASK                0x10000u
#define FTM_COMBINE_COMBINE2_SHIFT               16u
#define FTM_COMBINE_COMBINE2_WIDTH               1u
#define FTM_COMBINE_COMBINE2(x)                  (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_COMBINE2_SHIFT))&FTM_COMBINE_COMBINE2_MASK)
#define FTM_COMBINE_COMP2_MASK                   0x20000u
#define FTM_COMBINE_COMP2_SHIFT                  17u
#define FTM_COMBINE_COMP2_WIDTH                  1u
#define FTM_COMBINE_COMP2(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_COMP2_SHIFT))&FTM_COMBINE_COMP2_MASK)
#define FTM_COMBINE_DECAPEN2_MASK                0x40000u
#define FTM_COMBINE_DECAPEN2_SHIFT               18u
#define FTM_COMBINE_DECAPEN2_WIDTH               1u
#define FTM_COMBINE_DECAPEN2(x)                  (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_DECAPEN2_SHIFT))&FTM_COMBINE_DECAPEN2_MASK)
#define FTM_COMBINE_DECAP2_MASK                  0x80000u
#define FTM_COMBINE_DECAP2_SHIFT                 19u
#define FTM_COMBINE_DECAP2_WIDTH                 1u
#define FTM_COMBINE_DECAP2(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_DECAP2_SHIFT))&FTM_COMBINE_DECAP2_MASK)
#define FTM_COMBINE_DTEN2_MASK                   0x100000u
#define FTM_COMBINE_DTEN2_SHIFT                  20u
#define FTM_COMBINE_DTEN2_WIDTH                  1u
#define FTM_COMBINE_DTEN2(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_DTEN2_SHIFT))&FTM_COMBINE_DTEN2_MASK)
#define FTM_COMBINE_SYNCEN2_MASK                 0x200000u
#define FTM_COMBINE_SYNCEN2_SHIFT                21u
#define FTM_COMBINE_SYNCEN2_WIDTH                1u
#define FTM_COMBINE_SYNCEN2(x)                   (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_SYNCEN2_SHIFT))&FTM_COMBINE_SYNCEN2_MASK)
#define FTM_COMBINE_COMBINE3_MASK                0x1000000u
#define FTM_COMBINE_COMBINE3_SHIFT               24u
#define FTM_COMBINE_COMBINE3_WIDTH               1u
#define FTM_COMBINE_COMBINE3(x)                  (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_COMBINE3_SHIFT))&FTM_COMBINE_COMBINE3_MASK)
#define FTM_COMBINE_COMP3_MASK                   0x2000000u
#define FTM_COMBINE_COMP3_SHIFT                  25u
#define FTM_COMBINE_COMP3_WIDTH                  1u
#define FTM_COMBINE_COMP3(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_COMP3_SHIFT))&FTM_COMBINE_COMP3_MASK)
#define FTM_COMBINE_DECAPEN3_MASK                0x4000000u
#define FTM_COMBINE_DECAPEN3_SHIFT               26u
#define FTM_COMBINE_DECAPEN3_WIDTH               1u
#define FTM_COMBINE_DECAPEN3(x)                  (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_DECAPEN3_SHIFT))&FTM_COMBINE_DECAPEN3_MASK)
#define FTM_COMBINE_DECAP3_MASK                  0x8000000u
#define FTM_COMBINE_DECAP3_SHIFT                 27u
#define FTM_COMBINE_DECAP3_WIDTH                 1u
#define FTM_COMBINE_DECAP3(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_DECAP3_SHIFT))&FTM_COMBINE_DECAP3_MASK)
#define FTM_COMBINE_DTEN3_MASK                   0x10000000u
#define FTM_COMBINE_DTEN3_SHIFT                  28u
#define FTM_COMBINE_DTEN3_WIDTH                  1u
#define FTM_COMBINE_DTEN3(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_DTEN3_SHIFT))&FTM_COMBINE_DTEN3_MASK)
#define FTM_COMBINE_SYNCEN3_MASK                 0x20000000u
#define FTM_COMBINE_SYNCEN3_SHIFT                29u
#define FTM_COMBINE_SYNCEN3_WIDTH                1u
#define FTM_COMBINE_SYNCEN3(x)                   (((uint32_t)(((uint32_t)(x))<<FTM_COMBINE_SYNCEN3_SHIFT))&FTM_COMBINE_SYNCEN3_MASK)
/* DEADTIME Bit Fields */
#define FTM_DEADTIME_DTVAL_MASK                  0x3Fu
#define FTM_DEADTIME_DTVAL_SHIFT                 0u
#define FTM_DEADTIME_DTVAL_WIDTH                 6u
#define FTM_DEADTIME_DTVAL(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_DEADTIME_DTVAL_SHIFT))&FTM_DEADTIME_DTVAL_MASK)
#define FTM_DEADTIME_DTPS_MASK                   0xC0u
#define FTM_DEADTIME_DTPS_SHIFT                  6u
#define FTM_DEADTIME_DTPS_WIDTH                  2u
#define FTM_DEADTIME_DTPS(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_DEADTIME_DTPS_SHIFT))&FTM_DEADTIME_DTPS_MASK)
/* EXTTRIG Bit Fields */
#define FTM_EXTTRIG_CH2TRIG_MASK                 0x1u
#define FTM_EXTTRIG_CH2TRIG_SHIFT                0u
#define FTM_EXTTRIG_CH2TRIG_WIDTH                1u
#define FTM_EXTTRIG_CH2TRIG(x)                   (((uint32_t)(((uint32_t)(x))<<FTM_EXTTRIG_CH2TRIG_SHIFT))&FTM_EXTTRIG_CH2TRIG_MASK)
#define FTM_EXTTRIG_CH3TRIG_MASK                 0x2u
#define FTM_EXTTRIG_CH3TRIG_SHIFT                1u
#define FTM_EXTTRIG_CH3TRIG_WIDTH                1u
#define FTM_EXTTRIG_CH3TRIG(x)                   (((uint32_t)(((uint32_t)(x))<<FTM_EXTTRIG_CH3TRIG_SHIFT))&FTM_EXTTRIG_CH3TRIG_MASK)
#define FTM_EXTTRIG_CH4TRIG_MASK                 0x4u
#define FTM_EXTTRIG_CH4TRIG_SHIFT                2u
#define FTM_EXTTRIG_CH4TRIG_WIDTH                1u
#define FTM_EXTTRIG_CH4TRIG(x)                   (((uint32_t)(((uint32_t)(x))<<FTM_EXTTRIG_CH4TRIG_SHIFT))&FTM_EXTTRIG_CH4TRIG_MASK)
#define FTM_EXTTRIG_CH5TRIG_MASK                 0x8u
#define FTM_EXTTRIG_CH5TRIG_SHIFT                3u
#define FTM_EXTTRIG_CH5TRIG_WIDTH                1u
#define FTM_EXTTRIG_CH5TRIG(x)                   (((uint32_t)(((uint32_t)(x))<<FTM_EXTTRIG_CH5TRIG_SHIFT))&FTM_EXTTRIG_CH5TRIG_MASK)
#define FTM_EXTTRIG_CH0TRIG_MASK                 0x10u
#define FTM_EXTTRIG_CH0TRIG_SHIFT                4u
#define FTM_EXTTRIG_CH0TRIG_WIDTH                1u
#define FTM_EXTTRIG_CH0TRIG(x)                   (((uint32_t)(((uint32_t)(x))<<FTM_EXTTRIG_CH0TRIG_SHIFT))&FTM_EXTTRIG_CH0TRIG_MASK)
#define FTM_EXTTRIG_CH1TRIG_MASK                 0x20u
#define FTM_EXTTRIG_CH1TRIG_SHIFT                5u
#define FTM_EXTTRIG_CH1TRIG_WIDTH                1u
#define FTM_EXTTRIG_CH1TRIG(x)                   (((uint32_t)(((uint32_t)(x))<<FTM_EXTTRIG_CH1TRIG_SHIFT))&FTM_EXTTRIG_CH1TRIG_MASK)
#define FTM_EXTTRIG_INITTRIGEN_MASK              0x40u
#define FTM_EXTTRIG_INITTRIGEN_SHIFT             6u
#define FTM_EXTTRIG_INITTRIGEN_WIDTH             1u
#define FTM_EXTTRIG_INITTRIGEN(x)                (((uint32_t)(((uint32_t)(x))<<FTM_EXTTRIG_INITTRIGEN_SHIFT))&FTM_EXTTRIG_INITTRIGEN_MASK)
#define FTM_EXTTRIG_TRIGF_MASK                   0x80u
#define FTM_EXTTRIG_TRIGF_SHIFT                  7u
#define FTM_EXTTRIG_TRIGF_WIDTH                  1u
#define FTM_EXTTRIG_TRIGF(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_EXTTRIG_TRIGF_SHIFT))&FTM_EXTTRIG_TRIGF_MASK)
/* POL Bit Fields */
#define FTM_POL_POL0_MASK                        0x1u
#define FTM_POL_POL0_SHIFT                       0u
#define FTM_POL_POL0_WIDTH                       1u
#define FTM_POL_POL0(x)                          (((uint32_t)(((uint32_t)(x))<<FTM_POL_POL0_SHIFT))&FTM_POL_POL0_MASK)
#define FTM_POL_POL1_MASK                        0x2u
#define FTM_POL_POL1_SHIFT                       1u
#define FTM_POL_POL1_WIDTH                       1u
#define FTM_POL_POL1(x)                          (((uint32_t)(((uint32_t)(x))<<FTM_POL_POL1_SHIFT))&FTM_POL_POL1_MASK)
#define FTM_POL_POL2_MASK                        0x4u
#define FTM_POL_POL2_SHIFT                       2u
#define FTM_POL_POL2_WIDTH                       1u
#define FTM_POL_POL2(x)                          (((uint32_t)(((uint32_t)(x))<<FTM_POL_POL2_SHIFT))&FTM_POL_POL2_MASK)
#define FTM_POL_POL3_MASK                        0x8u
#define FTM_POL_POL3_SHIFT                       3u
#define FTM_POL_POL3_WIDTH                       1u
#define FTM_POL_POL3(x)                          (((uint32_t)(((uint32_t)(x))<<FTM_POL_POL3_SHIFT))&FTM_POL_POL3_MASK)
#define FTM_POL_POL4_MASK                        0x10u
#define FTM_POL_POL4_SHIFT                       4u
#define FTM_POL_POL4_WIDTH                       1u
#define FTM_POL_POL4(x)                          (((uint32_t)(((uint32_t)(x))<<FTM_POL_POL4_SHIFT))&FTM_POL_POL4_MASK)
#define FTM_POL_POL5_MASK                        0x20u
#define FTM_POL_POL5_SHIFT                       5u
#define FTM_POL_POL5_WIDTH                       1u
#define FTM_POL_POL5(x)                          (((uint32_t)(((uint32_t)(x))<<FTM_POL_POL5_SHIFT))&FTM_POL_POL5_MASK)
#define FTM_POL_POL6_MASK                        0x40u
#define FTM_POL_POL6_SHIFT                       6u
#define FTM_POL_POL6_WIDTH                       1u
#define FTM_POL_POL6(x)                          (((uint32_t)(((uint32_t)(x))<<FTM_POL_POL6_SHIFT))&FTM_POL_POL6_MASK)
#define FTM_POL_POL7_MASK                        0x80u
#define FTM_POL_POL7_SHIFT                       7u
#define FTM_POL_POL7_WIDTH                       1u
#define FTM_POL_POL7(x)                          (((uint32_t)(((uint32_t)(x))<<FTM_POL_POL7_SHIFT))&FTM_POL_POL7_MASK)
/* FMS Bit Fields */
#define FTM_FMS_WPEN_MASK                        0x40u
#define FTM_FMS_WPEN_SHIFT                       6u
#define FTM_FMS_WPEN_WIDTH                       1u
#define FTM_FMS_WPEN(x)                          (((uint32_t)(((uint32_t)(x))<<FTM_FMS_WPEN_SHIFT))&FTM_FMS_WPEN_MASK)
/* FILTER Bit Fields */
#define FTM_FILTER_CH0FVAL_MASK                  0xFu
#define FTM_FILTER_CH0FVAL_SHIFT                 0u
#define FTM_FILTER_CH0FVAL_WIDTH                 4u
#define FTM_FILTER_CH0FVAL(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_FILTER_CH0FVAL_SHIFT))&FTM_FILTER_CH0FVAL_MASK)
#define FTM_FILTER_CH1FVAL_MASK                  0xF0u
#define FTM_FILTER_CH1FVAL_SHIFT                 4u
#define FTM_FILTER_CH1FVAL_WIDTH                 4u
#define FTM_FILTER_CH1FVAL(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_FILTER_CH1FVAL_SHIFT))&FTM_FILTER_CH1FVAL_MASK)
#define FTM_FILTER_CH2FVAL_MASK                  0xF00u
#define FTM_FILTER_CH2FVAL_SHIFT                 8u
#define FTM_FILTER_CH2FVAL_WIDTH                 4u
#define FTM_FILTER_CH2FVAL(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_FILTER_CH2FVAL_SHIFT))&FTM_FILTER_CH2FVAL_MASK)
#define FTM_FILTER_CH3FVAL_MASK                  0xF000u
#define FTM_FILTER_CH3FVAL_SHIFT                 12u
#define FTM_FILTER_CH3FVAL_WIDTH                 4u
#define FTM_FILTER_CH3FVAL(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_FILTER_CH3FVAL_SHIFT))&FTM_FILTER_CH3FVAL_MASK)
/* QDCTRL Bit Fields */
#define FTM_QDCTRL_QUADEN_MASK                   0x1u
#define FTM_QDCTRL_QUADEN_SHIFT                  0u
#define FTM_QDCTRL_QUADEN_WIDTH                  1u
#define FTM_QDCTRL_QUADEN(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_QDCTRL_QUADEN_SHIFT))&FTM_QDCTRL_QUADEN_MASK)
#define FTM_QDCTRL_TOFDIR_MASK                   0x2u
#define FTM_QDCTRL_TOFDIR_SHIFT                  1u
#define FTM_QDCTRL_TOFDIR_WIDTH                  1u
#define FTM_QDCTRL_TOFDIR(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_QDCTRL_TOFDIR_SHIFT))&FTM_QDCTRL_TOFDIR_MASK)
#define FTM_QDCTRL_QUADIR_MASK                   0x4u
#define FTM_QDCTRL_QUADIR_SHIFT                  2u
#define FTM_QDCTRL_QUADIR_WIDTH                  1u
#define FTM_QDCTRL_QUADIR(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_QDCTRL_QUADIR_SHIFT))&FTM_QDCTRL_QUADIR_MASK)
#define FTM_QDCTRL_QUADMODE_MASK                 0x8u
#define FTM_QDCTRL_QUADMODE_SHIFT                3u
#define FTM_QDCTRL_QUADMODE_WIDTH                1u
#define FTM_QDCTRL_QUADMODE(x)                   (((uint32_t)(((uint32_t)(x))<<FTM_QDCTRL_QUADMODE_SHIFT))&FTM_QDCTRL_QUADMODE_MASK)
#define FTM_QDCTRL_PHBPOL_MASK                   0x10u
#define FTM_QDCTRL_PHBPOL_SHIFT                  4u
#define FTM_QDCTRL_PHBPOL_WIDTH                  1u
#define FTM_QDCTRL_PHBPOL(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_QDCTRL_PHBPOL_SHIFT))&FTM_QDCTRL_PHBPOL_MASK)
#define FTM_QDCTRL_PHAPOL_MASK                   0x20u
#define FTM_QDCTRL_PHAPOL_SHIFT                  5u
#define FTM_QDCTRL_PHAPOL_WIDTH                  1u
#define FTM_QDCTRL_PHAPOL(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_QDCTRL_PHAPOL_SHIFT))&FTM_QDCTRL_PHAPOL_MASK)
#define FTM_QDCTRL_PHBFLTREN_MASK                0x40u
#define FTM_QDCTRL_PHBFLTREN_SHIFT               6u
#define FTM_QDCTRL_PHBFLTREN_WIDTH               1u
#define FTM_QDCTRL_PHBFLTREN(x)                  (((uint32_t)(((uint32_t)(x))<<FTM_QDCTRL_PHBFLTREN_SHIFT))&FTM_QDCTRL_PHBFLTREN_MASK)
#define FTM_QDCTRL_PHAFLTREN_MASK                0x80u
#define FTM_QDCTRL_PHAFLTREN_SHIFT               7u
#define FTM_QDCTRL_PHAFLTREN_WIDTH               1u
#define FTM_QDCTRL_PHAFLTREN(x)                  (((uint32_t)(((uint32_t)(x))<<FTM_QDCTRL_PHAFLTREN_SHIFT))&FTM_QDCTRL_PHAFLTREN_MASK)
/* CONF Bit Fields */
#define FTM_CONF_NUMTOF_MASK                     0x1Fu
#define FTM_CONF_NUMTOF_SHIFT                    0u
#define FTM_CONF_NUMTOF_WIDTH                    5u
#define FTM_CONF_NUMTOF(x)                       (((uint32_t)(((uint32_t)(x))<<FTM_CONF_NUMTOF_SHIFT))&FTM_CONF_NUMTOF_MASK)
#define FTM_CONF_BDMMODE_MASK                    0xC0u
#define FTM_CONF_BDMMODE_SHIFT                   6u
#define FTM_CONF_BDMMODE_WIDTH                   2u
#define FTM_CONF_BDMMODE(x)                      (((uint32_t)(((uint32_t)(x))<<FTM_CONF_BDMMODE_SHIFT))&FTM_CONF_BDMMODE_MASK)
#define FTM_CONF_GTBEEN_MASK                     0x200u
#define FTM_CONF_GTBEEN_SHIFT                    9u
#define FTM_CONF_GTBEEN_WIDTH                    1u
#define FTM_CONF_GTBEEN(x)                       (((uint32_t)(((uint32_t)(x))<<FTM_CONF_GTBEEN_SHIFT))&FTM_CONF_GTBEEN_MASK)
#define FTM_CONF_GTBEOUT_MASK                    0x400u
#define FTM_CONF_GTBEOUT_SHIFT                   10u
#define FTM_CONF_GTBEOUT_WIDTH                   1u
#define FTM_CONF_GTBEOUT(x)                      (((uint32_t)(((uint32_t)(x))<<FTM_CONF_GTBEOUT_SHIFT))&FTM_CONF_GTBEOUT_MASK)
/* SYNCONF Bit Fields */
#define FTM_SYNCONF_HWTRIGMODE_MASK              0x1u
#define FTM_SYNCONF_HWTRIGMODE_SHIFT             0u
#define FTM_SYNCONF_HWTRIGMODE_WIDTH             1u
#define FTM_SYNCONF_HWTRIGMODE(x)                (((uint32_t)(((uint32_t)(x))<<FTM_SYNCONF_HWTRIGMODE_SHIFT))&FTM_SYNCONF_HWTRIGMODE_MASK)
#define FTM_SYNCONF_CNTINC_MASK                  0x4u
#define FTM_SYNCONF_CNTINC_SHIFT                 2u
#define FTM_SYNCONF_CNTINC_WIDTH                 1u
#define FTM_SYNCONF_CNTINC(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_SYNCONF_CNTINC_SHIFT))&FTM_SYNCONF_CNTINC_MASK)
#define FTM_SYNCONF_INVC_MASK                    0x10u
#define FTM_SYNCONF_INVC_SHIFT                   4u
#define FTM_SYNCONF_INVC_WIDTH                   1u
#define FTM_SYNCONF_INVC(x)                      (((uint32_t)(((uint32_t)(x))<<FTM_SYNCONF_INVC_SHIFT))&FTM_SYNCONF_INVC_MASK)
#define FTM_SYNCONF_SWOC_MASK                    0x20u
#define FTM_SYNCONF_SWOC_SHIFT                   5u
#define FTM_SYNCONF_SWOC_WIDTH                   1u
#define FTM_SYNCONF_SWOC(x)                      (((uint32_t)(((uint32_t)(x))<<FTM_SYNCONF_SWOC_SHIFT))&FTM_SYNCONF_SWOC_MASK)
#define FTM_SYNCONF_SYNCMODE_MASK                0x80u
#define FTM_SYNCONF_SYNCMODE_SHIFT               7u
#define FTM_SYNCONF_SYNCMODE_WIDTH               1u
#define FTM_SYNCONF_SYNCMODE(x)                  (((uint32_t)(((uint32_t)(x))<<FTM_SYNCONF_SYNCMODE_SHIFT))&FTM_SYNCONF_SYNCMODE_MASK)
#define FTM_SYNCONF_SWRSTCNT_MASK                0x100u
#define FTM_SYNCONF_SWRSTCNT_SHIFT               8u
#define FTM_SYNCONF_SWRSTCNT_WIDTH               1u
#define FTM_SYNCONF_SWRSTCNT(x)                  (((uint32_t)(((uint32_t)(x))<<FTM_SYNCONF_SWRSTCNT_SHIFT))&FTM_SYNCONF_SWRSTCNT_MASK)
#define FTM_SYNCONF_SWWRBUF_MASK                 0x200u
#define FTM_SYNCONF_SWWRBUF_SHIFT                9u
#define FTM_SYNCONF_SWWRBUF_WIDTH                1u
#define FTM_SYNCONF_SWWRBUF(x)                   (((uint32_t)(((uint32_t)(x))<<FTM_SYNCONF_SWWRBUF_SHIFT))&FTM_SYNCONF_SWWRBUF_MASK)
#define FTM_SYNCONF_SWOM_MASK                    0x400u
#define FTM_SYNCONF_SWOM_SHIFT                   10u
#define FTM_SYNCONF_SWOM_WIDTH                   1u
#define FTM_SYNCONF_SWOM(x)                      (((uint32_t)(((uint32_t)(x))<<FTM_SYNCONF_SWOM_SHIFT))&FTM_SYNCONF_SWOM_MASK)
#define FTM_SYNCONF_SWINVC_MASK                  0x800u
#define FTM_SYNCONF_SWINVC_SHIFT                 11u
#define FTM_SYNCONF_SWINVC_WIDTH                 1u
#define FTM_SYNCONF_SWINVC(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_SYNCONF_SWINVC_SHIFT))&FTM_SYNCONF_SWINVC_MASK)
#define FTM_SYNCONF_SWSOC_MASK                   0x1000u
#define FTM_SYNCONF_SWSOC_SHIFT                  12u
#define FTM_SYNCONF_SWSOC_WIDTH                  1u
#define FTM_SYNCONF_SWSOC(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_SYNCONF_SWSOC_SHIFT))&FTM_SYNCONF_SWSOC_MASK)
#define FTM_SYNCONF_HWRSTCNT_MASK                0x10000u
#define FTM_SYNCONF_HWRSTCNT_SHIFT               16u
#define FTM_SYNCONF_HWRSTCNT_WIDTH               1u
#define FTM_SYNCONF_HWRSTCNT(x)                  (((uint32_t)(((uint32_t)(x))<<FTM_SYNCONF_HWRSTCNT_SHIFT))&FTM_SYNCONF_HWRSTCNT_MASK)
#define FTM_SYNCONF_HWWRBUF_MASK                 0x20000u
#define FTM_SYNCONF_HWWRBUF_SHIFT                17u
#define FTM_SYNCONF_HWWRBUF_WIDTH                1u
#define FTM_SYNCONF_HWWRBUF(x)                   (((uint32_t)(((uint32_t)(x))<<FTM_SYNCONF_HWWRBUF_SHIFT))&FTM_SYNCONF_HWWRBUF_MASK)
#define FTM_SYNCONF_HWOM_MASK                    0x40000u
#define FTM_SYNCONF_HWOM_SHIFT                   18u
#define FTM_SYNCONF_HWOM_WIDTH                   1u
#define FTM_SYNCONF_HWOM(x)                      (((uint32_t)(((uint32_t)(x))<<FTM_SYNCONF_HWOM_SHIFT))&FTM_SYNCONF_HWOM_MASK)
#define FTM_SYNCONF_HWINVC_MASK                  0x80000u
#define FTM_SYNCONF_HWINVC_SHIFT                 19u
#define FTM_SYNCONF_HWINVC_WIDTH                 1u
#define FTM_SYNCONF_HWINVC(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_SYNCONF_HWINVC_SHIFT))&FTM_SYNCONF_HWINVC_MASK)
#define FTM_SYNCONF_HWSOC_MASK                   0x100000u
#define FTM_SYNCONF_HWSOC_SHIFT                  20u
#define FTM_SYNCONF_HWSOC_WIDTH                  1u
#define FTM_SYNCONF_HWSOC(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_SYNCONF_HWSOC_SHIFT))&FTM_SYNCONF_HWSOC_MASK)
/* INVCTRL Bit Fields */
#define FTM_INVCTRL_INV0EN_MASK                  0x1u
#define FTM_INVCTRL_INV0EN_SHIFT                 0u
#define FTM_INVCTRL_INV0EN_WIDTH                 1u
#define FTM_INVCTRL_INV0EN(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_INVCTRL_INV0EN_SHIFT))&FTM_INVCTRL_INV0EN_MASK)
#define FTM_INVCTRL_INV1EN_MASK                  0x2u
#define FTM_INVCTRL_INV1EN_SHIFT                 1u
#define FTM_INVCTRL_INV1EN_WIDTH                 1u
#define FTM_INVCTRL_INV1EN(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_INVCTRL_INV1EN_SHIFT))&FTM_INVCTRL_INV1EN_MASK)
#define FTM_INVCTRL_INV2EN_MASK                  0x4u
#define FTM_INVCTRL_INV2EN_SHIFT                 2u
#define FTM_INVCTRL_INV2EN_WIDTH                 1u
#define FTM_INVCTRL_INV2EN(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_INVCTRL_INV2EN_SHIFT))&FTM_INVCTRL_INV2EN_MASK)
#define FTM_INVCTRL_INV3EN_MASK                  0x8u
#define FTM_INVCTRL_INV3EN_SHIFT                 3u
#define FTM_INVCTRL_INV3EN_WIDTH                 1u
#define FTM_INVCTRL_INV3EN(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_INVCTRL_INV3EN_SHIFT))&FTM_INVCTRL_INV3EN_MASK)
/* SWOCTRL Bit Fields */
#define FTM_SWOCTRL_CH0OC_MASK                   0x1u
#define FTM_SWOCTRL_CH0OC_SHIFT                  0u
#define FTM_SWOCTRL_CH0OC_WIDTH                  1u
#define FTM_SWOCTRL_CH0OC(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_SWOCTRL_CH0OC_SHIFT))&FTM_SWOCTRL_CH0OC_MASK)
#define FTM_SWOCTRL_CH1OC_MASK                   0x2u
#define FTM_SWOCTRL_CH1OC_SHIFT                  1u
#define FTM_SWOCTRL_CH1OC_WIDTH                  1u
#define FTM_SWOCTRL_CH1OC(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_SWOCTRL_CH1OC_SHIFT))&FTM_SWOCTRL_CH1OC_MASK)
#define FTM_SWOCTRL_CH2OC_MASK                   0x4u
#define FTM_SWOCTRL_CH2OC_SHIFT                  2u
#define FTM_SWOCTRL_CH2OC_WIDTH                  1u
#define FTM_SWOCTRL_CH2OC(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_SWOCTRL_CH2OC_SHIFT))&FTM_SWOCTRL_CH2OC_MASK)
#define FTM_SWOCTRL_CH3OC_MASK                   0x8u
#define FTM_SWOCTRL_CH3OC_SHIFT                  3u
#define FTM_SWOCTRL_CH3OC_WIDTH                  1u
#define FTM_SWOCTRL_CH3OC(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_SWOCTRL_CH3OC_SHIFT))&FTM_SWOCTRL_CH3OC_MASK)
#define FTM_SWOCTRL_CH4OC_MASK                   0x10u
#define FTM_SWOCTRL_CH4OC_SHIFT                  4u
#define FTM_SWOCTRL_CH4OC_WIDTH                  1u
#define FTM_SWOCTRL_CH4OC(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_SWOCTRL_CH4OC_SHIFT))&FTM_SWOCTRL_CH4OC_MASK)
#define FTM_SWOCTRL_CH5OC_MASK                   0x20u
#define FTM_SWOCTRL_CH5OC_SHIFT                  5u
#define FTM_SWOCTRL_CH5OC_WIDTH                  1u
#define FTM_SWOCTRL_CH5OC(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_SWOCTRL_CH5OC_SHIFT))&FTM_SWOCTRL_CH5OC_MASK)
#define FTM_SWOCTRL_CH6OC_MASK                   0x40u
#define FTM_SWOCTRL_CH6OC_SHIFT                  6u
#define FTM_SWOCTRL_CH6OC_WIDTH                  1u
#define FTM_SWOCTRL_CH6OC(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_SWOCTRL_CH6OC_SHIFT))&FTM_SWOCTRL_CH6OC_MASK)
#define FTM_SWOCTRL_CH7OC_MASK                   0x80u
#define FTM_SWOCTRL_CH7OC_SHIFT                  7u
#define FTM_SWOCTRL_CH7OC_WIDTH                  1u
#define FTM_SWOCTRL_CH7OC(x)                     (((uint32_t)(((uint32_t)(x))<<FTM_SWOCTRL_CH7OC_SHIFT))&FTM_SWOCTRL_CH7OC_MASK)
#define FTM_SWOCTRL_CH0OCV_MASK                  0x100u
#define FTM_SWOCTRL_CH0OCV_SHIFT                 8u
#define FTM_SWOCTRL_CH0OCV_WIDTH                 1u
#define FTM_SWOCTRL_CH0OCV(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_SWOCTRL_CH0OCV_SHIFT))&FTM_SWOCTRL_CH0OCV_MASK)
#define FTM_SWOCTRL_CH1OCV_MASK                  0x200u
#define FTM_SWOCTRL_CH1OCV_SHIFT                 9u
#define FTM_SWOCTRL_CH1OCV_WIDTH                 1u
#define FTM_SWOCTRL_CH1OCV(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_SWOCTRL_CH1OCV_SHIFT))&FTM_SWOCTRL_CH1OCV_MASK)
#define FTM_SWOCTRL_CH2OCV_MASK                  0x400u
#define FTM_SWOCTRL_CH2OCV_SHIFT                 10u
#define FTM_SWOCTRL_CH2OCV_WIDTH                 1u
#define FTM_SWOCTRL_CH2OCV(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_SWOCTRL_CH2OCV_SHIFT))&FTM_SWOCTRL_CH2OCV_MASK)
#define FTM_SWOCTRL_CH3OCV_MASK                  0x800u
#define FTM_SWOCTRL_CH3OCV_SHIFT                 11u
#define FTM_SWOCTRL_CH3OCV_WIDTH                 1u
#define FTM_SWOCTRL_CH3OCV(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_SWOCTRL_CH3OCV_SHIFT))&FTM_SWOCTRL_CH3OCV_MASK)
#define FTM_SWOCTRL_CH4OCV_MASK                  0x1000u
#define FTM_SWOCTRL_CH4OCV_SHIFT                 12u
#define FTM_SWOCTRL_CH4OCV_WIDTH                 1u
#define FTM_SWOCTRL_CH4OCV(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_SWOCTRL_CH4OCV_SHIFT))&FTM_SWOCTRL_CH4OCV_MASK)
#define FTM_SWOCTRL_CH5OCV_MASK                  0x2000u
#define FTM_SWOCTRL_CH5OCV_SHIFT                 13u
#define FTM_SWOCTRL_CH5OCV_WIDTH                 1u
#define FTM_SWOCTRL_CH5OCV(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_SWOCTRL_CH5OCV_SHIFT))&FTM_SWOCTRL_CH5OCV_MASK)
#define FTM_SWOCTRL_CH6OCV_MASK                  0x4000u
#define FTM_SWOCTRL_CH6OCV_SHIFT                 14u
#define FTM_SWOCTRL_CH6OCV_WIDTH                 1u
#define FTM_SWOCTRL_CH6OCV(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_SWOCTRL_CH6OCV_SHIFT))&FTM_SWOCTRL_CH6OCV_MASK)
#define FTM_SWOCTRL_CH7OCV_MASK                  0x8000u
#define FTM_SWOCTRL_CH7OCV_SHIFT                 15u
#define FTM_SWOCTRL_CH7OCV_WIDTH                 1u
#define FTM_SWOCTRL_CH7OCV(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_SWOCTRL_CH7OCV_SHIFT))&FTM_SWOCTRL_CH7OCV_MASK)
/* PWMLOAD Bit Fields */
#define FTM_PWMLOAD_CH0SEL_MASK                  0x1u
#define FTM_PWMLOAD_CH0SEL_SHIFT                 0u
#define FTM_PWMLOAD_CH0SEL_WIDTH                 1u
#define FTM_PWMLOAD_CH0SEL(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_PWMLOAD_CH0SEL_SHIFT))&FTM_PWMLOAD_CH0SEL_MASK)
#define FTM_PWMLOAD_CH1SEL_MASK                  0x2u
#define FTM_PWMLOAD_CH1SEL_SHIFT                 1u
#define FTM_PWMLOAD_CH1SEL_WIDTH                 1u
#define FTM_PWMLOAD_CH1SEL(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_PWMLOAD_CH1SEL_SHIFT))&FTM_PWMLOAD_CH1SEL_MASK)
#define FTM_PWMLOAD_CH2SEL_MASK                  0x4u
#define FTM_PWMLOAD_CH2SEL_SHIFT                 2u
#define FTM_PWMLOAD_CH2SEL_WIDTH                 1u
#define FTM_PWMLOAD_CH2SEL(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_PWMLOAD_CH2SEL_SHIFT))&FTM_PWMLOAD_CH2SEL_MASK)
#define FTM_PWMLOAD_CH3SEL_MASK                  0x8u
#define FTM_PWMLOAD_CH3SEL_SHIFT                 3u
#define FTM_PWMLOAD_CH3SEL_WIDTH                 1u
#define FTM_PWMLOAD_CH3SEL(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_PWMLOAD_CH3SEL_SHIFT))&FTM_PWMLOAD_CH3SEL_MASK)
#define FTM_PWMLOAD_CH4SEL_MASK                  0x10u
#define FTM_PWMLOAD_CH4SEL_SHIFT                 4u
#define FTM_PWMLOAD_CH4SEL_WIDTH                 1u
#define FTM_PWMLOAD_CH4SEL(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_PWMLOAD_CH4SEL_SHIFT))&FTM_PWMLOAD_CH4SEL_MASK)
#define FTM_PWMLOAD_CH5SEL_MASK                  0x20u
#define FTM_PWMLOAD_CH5SEL_SHIFT                 5u
#define FTM_PWMLOAD_CH5SEL_WIDTH                 1u
#define FTM_PWMLOAD_CH5SEL(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_PWMLOAD_CH5SEL_SHIFT))&FTM_PWMLOAD_CH5SEL_MASK)
#define FTM_PWMLOAD_CH6SEL_MASK                  0x40u
#define FTM_PWMLOAD_CH6SEL_SHIFT                 6u
#define FTM_PWMLOAD_CH6SEL_WIDTH                 1u
#define FTM_PWMLOAD_CH6SEL(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_PWMLOAD_CH6SEL_SHIFT))&FTM_PWMLOAD_CH6SEL_MASK)
#define FTM_PWMLOAD_CH7SEL_MASK                  0x80u
#define FTM_PWMLOAD_CH7SEL_SHIFT                 7u
#define FTM_PWMLOAD_CH7SEL_WIDTH                 1u
#define FTM_PWMLOAD_CH7SEL(x)                    (((uint32_t)(((uint32_t)(x))<<FTM_PWMLOAD_CH7SEL_SHIFT))&FTM_PWMLOAD_CH7SEL_MASK)
#define FTM_PWMLOAD_LDOK_MASK                    0x200u
#define FTM_PWMLOAD_LDOK_SHIFT                   9u
#define FTM_PWMLOAD_LDOK_WIDTH                   1u
#define FTM_PWMLOAD_LDOK(x)                      (((uint32_t)(((uint32_t)(x))<<FTM_PWMLOAD_LDOK_SHIFT))&FTM_PWMLOAD_LDOK_MASK)

/*!
 * @}
 */ /* end of group FTM_Register_Masks */


/*!
 * @}
 */ /* end of group FTM_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- FXOSC Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup FXOSC_Peripheral_Access_Layer FXOSC Peripheral Access Layer
 * @{
 */


/** FXOSC - Size of Registers Arrays */

/** FXOSC - Register Layout Typedef */
typedef struct {
  __IO uint32_t CTL;                               /**< FXOSC Control Register, offset: 0x0 */
} FXOSC_Type, *FXOSC_MemMapPtr;

 /** Number of instances of the FXOSC module. */
#define FXOSC_INSTANCE_COUNT                     (1u)


/* FXOSC - Peripheral instance base addresses */
/** Peripheral FXOSC base address */
#define FXOSC_BASE                               (0x4003C280u)
/** Peripheral FXOSC base pointer */
#define FXOSC                                    ((FXOSC_Type *)FXOSC_BASE)
/** Array initializer of FXOSC peripheral base addresses */
#define FXOSC_BASE_ADDRS                         { FXOSC_BASE }
/** Array initializer of FXOSC peripheral base pointers */
#define FXOSC_BASE_PTRS                          { FXOSC }

/* ----------------------------------------------------------------------------
   -- FXOSC Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup FXOSC_Register_Masks FXOSC Register Masks
 * @{
 */

/* CTL Bit Fields */
#define FXOSC_CTL_GM_SEL_MASK                    0x70u
#define FXOSC_CTL_GM_SEL_SHIFT                   4u
#define FXOSC_CTL_GM_SEL_WIDTH                   3u
#define FXOSC_CTL_GM_SEL(x)                      (((uint32_t)(((uint32_t)(x))<<FXOSC_CTL_GM_SEL_SHIFT))&FXOSC_CTL_GM_SEL_MASK)
#define FXOSC_CTL_I_OSC_MASK                     0x80u
#define FXOSC_CTL_I_OSC_SHIFT                    7u
#define FXOSC_CTL_I_OSC_WIDTH                    1u
#define FXOSC_CTL_I_OSC(x)                       (((uint32_t)(((uint32_t)(x))<<FXOSC_CTL_I_OSC_SHIFT))&FXOSC_CTL_I_OSC_MASK)
#define FXOSC_CTL_M_OSC_MASK                     0x8000u
#define FXOSC_CTL_M_OSC_SHIFT                    15u
#define FXOSC_CTL_M_OSC_WIDTH                    1u
#define FXOSC_CTL_M_OSC(x)                       (((uint32_t)(((uint32_t)(x))<<FXOSC_CTL_M_OSC_SHIFT))&FXOSC_CTL_M_OSC_MASK)
#define FXOSC_CTL_EOCV_MASK                      0xFF0000u
#define FXOSC_CTL_EOCV_SHIFT                     16u
#define FXOSC_CTL_EOCV_WIDTH                     8u
#define FXOSC_CTL_EOCV(x)                        (((uint32_t)(((uint32_t)(x))<<FXOSC_CTL_EOCV_SHIFT))&FXOSC_CTL_EOCV_MASK)
#define FXOSC_CTL_MISC_IN1_MASK                  0x1000000u
#define FXOSC_CTL_MISC_IN1_SHIFT                 24u
#define FXOSC_CTL_MISC_IN1_WIDTH                 1u
#define FXOSC_CTL_MISC_IN1(x)                    (((uint32_t)(((uint32_t)(x))<<FXOSC_CTL_MISC_IN1_SHIFT))&FXOSC_CTL_MISC_IN1_MASK)
#define FXOSC_CTL_OSCBYP_MASK                    0x80000000u
#define FXOSC_CTL_OSCBYP_SHIFT                   31u
#define FXOSC_CTL_OSCBYP_WIDTH                   1u
#define FXOSC_CTL_OSCBYP(x)                      (((uint32_t)(((uint32_t)(x))<<FXOSC_CTL_OSCBYP_SHIFT))&FXOSC_CTL_OSCBYP_MASK)

/*!
 * @}
 */ /* end of group FXOSC_Register_Masks */


/*!
 * @}
 */ /* end of group FXOSC_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- H264_DEC Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup H264_DEC_Peripheral_Access_Layer H264_DEC Peripheral Access Layer
 * @{
 */


/** H264_DEC - Size of Registers Arrays */
#define H264_DEC_STR_VCR_COUNT                   4u
#define H264_DEC_STR_0_PKT_STATUS_COUNT          16u
#define H264_DEC_STR_1_PKT_STATUS_COUNT          16u
#define H264_DEC_STR_2_PKT_STATUS_COUNT          16u
#define H264_DEC_STR_3_PKT_STATUS_COUNT          16u
#define H264_DEC_VO_STR_COUNT                    4u
#define H264_DEC_TESTLINE_STR_COUNT              4u
#define H264_DEC_ELLVC_REF_ADDR_CH_COUNT         4u
#define H264_DEC_ELLVC_DBG_FRAME_CYC_CNT_THR_COUNT 3u

/** H264_DEC - Register Layout Typedef */
typedef struct {
  __IO uint32_t MCR;                               /**< MODULE CONFIGURATION REGISTER, offset: 0x0 */
  __IO uint32_t TIMEOUT_CFG;                       /**< TIMEOUT CONFIGURATION REGISTER, offset: 0x4 */
       uint8_t RESERVED_0[8];
  __IO uint32_t STR_VCR[H264_DEC_STR_VCR_COUNT];   /**< STREAM VIDEO CONFIGURATION REGISTER, array offset: 0x10, array step: 0x4 */
       uint8_t RESERVED_1[16];
  __IO uint32_t STR_PKT_ADDR;                      /**< STREAM PACKET ADDRESS REGISTER, offset: 0x30 */
  __IO uint32_t STR_TRIG_PKT_CFG;                  /**< STREAM TRIGGER AND PACKET CONFIGURATION REGISTER, offset: 0x34 */
  __IO uint32_t STR_PKT_FIFO_WMRK;                 /**< STREAM PACKET FIFO WATERMARK REGISTER, offset: 0x38 */
       uint8_t RESERVED_2[4];
  __I  uint32_t STR_01_PKT_FIFO_STAT;              /**< STREAM 0 and 1 PACKET FIFO STATUS REGISTER, offset: 0x40 */
  __I  uint32_t STR_23_PKT_FIFO_STAT;              /**< STREAM 2 and 3 PACKET FIFO STATUS REGISTER, offset: 0x44 */
       uint8_t RESERVED_3[8];
  __I  uint32_t STR_0_PKT_STATUS[H264_DEC_STR_0_PKT_STATUS_COUNT]; /**< STREAM 0 PACKET STATUS REGISTER, array offset: 0x50, array step: 0x4 */
       uint8_t RESERVED_4[64];
  __I  uint32_t STR_1_PKT_STATUS[H264_DEC_STR_1_PKT_STATUS_COUNT]; /**< STREAM 1 PACKET STATUS REGISTER, array offset: 0xD0, array step: 0x4 */
       uint8_t RESERVED_5[64];
  __I  uint32_t STR_2_PKT_STATUS[H264_DEC_STR_2_PKT_STATUS_COUNT]; /**< STREAM 2 PACKET STATUS REGISTER, array offset: 0x150, array step: 0x4 */
       uint8_t RESERVED_6[64];
  __I  uint32_t STR_3_PKT_STATUS[H264_DEC_STR_3_PKT_STATUS_COUNT]; /**< STREAM 3 PACKET STATUS REGISTER, array offset: 0x1D0, array step: 0x4 */
       uint8_t RESERVED_7[240];
  struct {                                         /* offset: 0x300, array step: 0x10 */
    __IO uint32_t Y_ADDR;                            /**< VIDEO OUTPUT STREAM LUMA ADDRESS REGISTER, array offset: 0x300, array step: 0x10 */
    __IO uint32_t CB_ADDR;                           /**< VIDEO OUTPUT STREAM Cb ADDRESS REGISTER, array offset: 0x304, array step: 0x10 */
    __IO uint32_t CR_ADDR;                           /**< VIDEO OUTPUT STREAM Cr ADDRESS REGISTER, array offset: 0x308, array step: 0x10 */
    __IO uint32_t NRLINES;                           /**< VIDEO OUTPUT STREAM NUMBER OF LINES REGISTER, array offset: 0x30C, array step: 0x10 */
  } VO_STR[H264_DEC_VO_STR_COUNT];
       uint8_t RESERVED_8[32];
  __IO uint32_t RATE_FLOW_CNTRL;                   /**< RATE FLOW CONTROL REGISTER, offset: 0x360 */
       uint8_t RESERVED_9[28];
  __I  uint32_t LINE_CNT_STAT;                     /**< LINE COUNT STATUS REGISTER, offset: 0x380 */
  __IO uint32_t STAT;                              /**< STATUS REGISTER, offset: 0x384 */
       uint8_t RESERVED_10[88];
  __IO uint32_t ISR;                               /**< INTERRUPT STATUS REGISTER, offset: 0x3E0 */
  __IO uint32_t IER;                               /**< INTERRUPT ENABLE REGISTER, offset: 0x3E4 */
       uint8_t RESERVED_11[24];
  __IO uint32_t TESTLINE_CFG;                      /**< TESTLINE CONFIGURATION REGISTER, offset: 0x400 */
  __IO uint32_t TESTLINE_PXL_LOC;                  /**< TESTLINE PIXEL LOCATION REGISTER, offset: 0x404 */
       uint8_t RESERVED_12[8];
  struct {                                         /* offset: 0x410, array step: 0x10 */
    __I  uint32_t LUMA_VAL;                          /**< TESTLINE STREAM LUMA PIXEL VALUE, array offset: 0x410, array step: 0x10 */
    __I  uint32_t CB_VAL;                            /**< TESTLINE STREAM Cb PIXEL VALUE REGISTER, array offset: 0x414, array step: 0x10 */
    __I  uint32_t CR_VAL;                            /**< TESTLINE STREAM Cr PIXEL VALUE REGISTER, array offset: 0x418, array step: 0x10 */
         uint8_t RESERVED_0[4];
  } TESTLINE_STR[H264_DEC_TESTLINE_STR_COUNT];
       uint8_t RESERVED_13[176];
  __IO uint32_t ELLVC_CFG;                         /**< ELLVC CONFIGURATION REGISTER, offset: 0x500 */
       uint8_t RESERVED_14[12];
  __IO uint32_t ELLVC_REF_ADDR_CH[H264_DEC_ELLVC_REF_ADDR_CH_COUNT]; /**< ELLVC REFERENCE ADDRESS CHANNEL REGISTER, array offset: 0x510, array step: 0x4 */
       uint8_t RESERVED_15[16];
  __IO uint32_t ELLVC_DBG_FRAME_CYC_CNT_THR[H264_DEC_ELLVC_DBG_FRAME_CYC_CNT_THR_COUNT]; /**< ELLVC DEBUG FRAME CYCLE COUNTER THRESHOLD REGISTER, array offset: 0x530, array step: 0x4 */
       uint8_t RESERVED_16[36];
  __I  uint32_t ELLVC_STATE;                       /**< ELLVC STATE REGISTER, offset: 0x560 */
  __I  uint32_t ELLVC_PIC_BIT;                     /**< ELLVC PIC BIT REGISTER, offset: 0x564 */
  __I  uint32_t ELLVC_PIC_WIDTH_IN_MBS;            /**< ELLVC PIC WIDTH IN MBS REGISTER, offset: 0x568 */
  __I  uint32_t ELLVC_PIC_HEIGHT_IN_MBS;           /**< ELLVC PIC HEIGHT IN MBS REGISTER, offset: 0x56C */
  __I  uint32_t ELLVC_PIC_CROP_LEFT_CH_01;         /**< ELLVC PIC CROP LEFT CHANNEL 0, 1 REGISTER, offset: 0x570 */
  __I  uint32_t ELLVC_PIC_CROP_LEFT_CH_23;         /**< ELLVC PIC CROP LEFT CHANNEL 2, 3 REGISTER, offset: 0x574 */
  __I  uint32_t ELLVC_PIC_CROP_RGHT_CH_01;         /**< ELLVC PIC CROP RIGHT CHANNEL 0, 1 REGISTER, offset: 0x578 */
  __I  uint32_t ELLVC_PIC_CROP_RGHT_CH_23;         /**< ELLVC PIC CROP RIGHT CHANNEL 2, 3 REGISTER, offset: 0x57C */
  __I  uint32_t ELLVC_PIC_CROP_TOP_CH_01;          /**< ELLVC PIC CROP TOP CHANNEL 0, 1 REGISTER, offset: 0x580 */
  __I  uint32_t ELLVC_PIC_CROP_TOP_CH_23;          /**< ELLVC PIC CROP TOP CHANNEL 2, 3 REGISTER, offset: 0x584 */
  __I  uint32_t ELLVC_PIC_CROP_BTTM_CH_01;         /**< ELLVC PIC CROP BOTTOM CHANNEL 0, 1 REGISTER, offset: 0x588 */
  __I  uint32_t ELLVC_PIC_CROP_BTTM_CH_23;         /**< ELLVC PIC CROP BOTTOM CHANNEL 2, 3 REGISTER, offset: 0x58C */
  __I  uint32_t ELLVC_PIC_PARAM_VLD;               /**< ELLVC PIC PARAM VALID REGISTER, offset: 0x590 */
  __I  uint32_t ELLVC_POC_VALUE;                   /**< ELLVC PICTURE ORDER COUNT VALUE REGISTER, offset: 0x594 */
  __I  uint32_t ELLVC_STAT;                        /**< ELLVC STATUS REGISTER, offset: 0x598 */
} H264_DEC_Type, *H264_DEC_MemMapPtr;

 /** Number of instances of the H264_DEC module. */
#define H264_DEC_INSTANCE_COUNT                  (1u)


/* H264_DEC - Peripheral instance base addresses */
/** Peripheral H264_DEC base address */
#define H264_DEC_BASE                            (0x400D8000u)
/** Peripheral H264_DEC base pointer */
#define H264_DEC                                 ((H264_DEC_Type *)H264_DEC_BASE)
/** Array initializer of H264_DEC peripheral base addresses */
#define H264_DEC_BASE_ADDRS                      { H264_DEC_BASE }
/** Array initializer of H264_DEC peripheral base pointers */
#define H264_DEC_BASE_PTRS                       { H264_DEC }

/* ----------------------------------------------------------------------------
   -- H264_DEC Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup H264_DEC_Register_Masks H264_DEC Register Masks
 * @{
 */

/* MCR Bit Fields */
#define H264_DEC_MCR_SW_RESET_MASK               0x1u
#define H264_DEC_MCR_SW_RESET_SHIFT              0u
#define H264_DEC_MCR_SW_RESET_WIDTH              1u
#define H264_DEC_MCR_SW_RESET(x)                 (((uint32_t)(((uint32_t)(x))<<H264_DEC_MCR_SW_RESET_SHIFT))&H264_DEC_MCR_SW_RESET_MASK)
#define H264_DEC_MCR_VO_BUF_0_RSTRT_MASK         0x2u
#define H264_DEC_MCR_VO_BUF_0_RSTRT_SHIFT        1u
#define H264_DEC_MCR_VO_BUF_0_RSTRT_WIDTH        1u
#define H264_DEC_MCR_VO_BUF_0_RSTRT(x)           (((uint32_t)(((uint32_t)(x))<<H264_DEC_MCR_VO_BUF_0_RSTRT_SHIFT))&H264_DEC_MCR_VO_BUF_0_RSTRT_MASK)
#define H264_DEC_MCR_VO_BUF_1_RSTRT_MASK         0x4u
#define H264_DEC_MCR_VO_BUF_1_RSTRT_SHIFT        2u
#define H264_DEC_MCR_VO_BUF_1_RSTRT_WIDTH        1u
#define H264_DEC_MCR_VO_BUF_1_RSTRT(x)           (((uint32_t)(((uint32_t)(x))<<H264_DEC_MCR_VO_BUF_1_RSTRT_SHIFT))&H264_DEC_MCR_VO_BUF_1_RSTRT_MASK)
#define H264_DEC_MCR_VO_BUF_2_RSTRT_MASK         0x8u
#define H264_DEC_MCR_VO_BUF_2_RSTRT_SHIFT        3u
#define H264_DEC_MCR_VO_BUF_2_RSTRT_WIDTH        1u
#define H264_DEC_MCR_VO_BUF_2_RSTRT(x)           (((uint32_t)(((uint32_t)(x))<<H264_DEC_MCR_VO_BUF_2_RSTRT_SHIFT))&H264_DEC_MCR_VO_BUF_2_RSTRT_MASK)
#define H264_DEC_MCR_VO_BUF_3_RSTRT_MASK         0x10u
#define H264_DEC_MCR_VO_BUF_3_RSTRT_SHIFT        4u
#define H264_DEC_MCR_VO_BUF_3_RSTRT_WIDTH        1u
#define H264_DEC_MCR_VO_BUF_3_RSTRT(x)           (((uint32_t)(((uint32_t)(x))<<H264_DEC_MCR_VO_BUF_3_RSTRT_SHIFT))&H264_DEC_MCR_VO_BUF_3_RSTRT_MASK)
#define H264_DEC_MCR_VO_BIT_WIDTH_STR_0_MASK     0x100u
#define H264_DEC_MCR_VO_BIT_WIDTH_STR_0_SHIFT    8u
#define H264_DEC_MCR_VO_BIT_WIDTH_STR_0_WIDTH    1u
#define H264_DEC_MCR_VO_BIT_WIDTH_STR_0(x)       (((uint32_t)(((uint32_t)(x))<<H264_DEC_MCR_VO_BIT_WIDTH_STR_0_SHIFT))&H264_DEC_MCR_VO_BIT_WIDTH_STR_0_MASK)
#define H264_DEC_MCR_VO_BIT_WIDTH_STR_1_MASK     0x200u
#define H264_DEC_MCR_VO_BIT_WIDTH_STR_1_SHIFT    9u
#define H264_DEC_MCR_VO_BIT_WIDTH_STR_1_WIDTH    1u
#define H264_DEC_MCR_VO_BIT_WIDTH_STR_1(x)       (((uint32_t)(((uint32_t)(x))<<H264_DEC_MCR_VO_BIT_WIDTH_STR_1_SHIFT))&H264_DEC_MCR_VO_BIT_WIDTH_STR_1_MASK)
#define H264_DEC_MCR_VO_BIT_WIDTH_STR_2_MASK     0x400u
#define H264_DEC_MCR_VO_BIT_WIDTH_STR_2_SHIFT    10u
#define H264_DEC_MCR_VO_BIT_WIDTH_STR_2_WIDTH    1u
#define H264_DEC_MCR_VO_BIT_WIDTH_STR_2(x)       (((uint32_t)(((uint32_t)(x))<<H264_DEC_MCR_VO_BIT_WIDTH_STR_2_SHIFT))&H264_DEC_MCR_VO_BIT_WIDTH_STR_2_MASK)
#define H264_DEC_MCR_VO_BIT_WIDTH_STR_3_MASK     0x800u
#define H264_DEC_MCR_VO_BIT_WIDTH_STR_3_SHIFT    11u
#define H264_DEC_MCR_VO_BIT_WIDTH_STR_3_WIDTH    1u
#define H264_DEC_MCR_VO_BIT_WIDTH_STR_3(x)       (((uint32_t)(((uint32_t)(x))<<H264_DEC_MCR_VO_BIT_WIDTH_STR_3_SHIFT))&H264_DEC_MCR_VO_BIT_WIDTH_STR_3_MASK)
#define H264_DEC_MCR_DATA_FLOW_MODE_MASK         0x30000u
#define H264_DEC_MCR_DATA_FLOW_MODE_SHIFT        16u
#define H264_DEC_MCR_DATA_FLOW_MODE_WIDTH        2u
#define H264_DEC_MCR_DATA_FLOW_MODE(x)           (((uint32_t)(((uint32_t)(x))<<H264_DEC_MCR_DATA_FLOW_MODE_SHIFT))&H264_DEC_MCR_DATA_FLOW_MODE_MASK)
#define H264_DEC_MCR_VO_COLOR_FMT_STR_0_MASK     0x100000u
#define H264_DEC_MCR_VO_COLOR_FMT_STR_0_SHIFT    20u
#define H264_DEC_MCR_VO_COLOR_FMT_STR_0_WIDTH    1u
#define H264_DEC_MCR_VO_COLOR_FMT_STR_0(x)       (((uint32_t)(((uint32_t)(x))<<H264_DEC_MCR_VO_COLOR_FMT_STR_0_SHIFT))&H264_DEC_MCR_VO_COLOR_FMT_STR_0_MASK)
#define H264_DEC_MCR_VO_COLOR_FMT_STR_1_MASK     0x200000u
#define H264_DEC_MCR_VO_COLOR_FMT_STR_1_SHIFT    21u
#define H264_DEC_MCR_VO_COLOR_FMT_STR_1_WIDTH    1u
#define H264_DEC_MCR_VO_COLOR_FMT_STR_1(x)       (((uint32_t)(((uint32_t)(x))<<H264_DEC_MCR_VO_COLOR_FMT_STR_1_SHIFT))&H264_DEC_MCR_VO_COLOR_FMT_STR_1_MASK)
#define H264_DEC_MCR_VO_COLOR_FMT_STR_2_MASK     0x400000u
#define H264_DEC_MCR_VO_COLOR_FMT_STR_2_SHIFT    22u
#define H264_DEC_MCR_VO_COLOR_FMT_STR_2_WIDTH    1u
#define H264_DEC_MCR_VO_COLOR_FMT_STR_2(x)       (((uint32_t)(((uint32_t)(x))<<H264_DEC_MCR_VO_COLOR_FMT_STR_2_SHIFT))&H264_DEC_MCR_VO_COLOR_FMT_STR_2_MASK)
#define H264_DEC_MCR_VO_COLOR_FMT_STR_3_MASK     0x800000u
#define H264_DEC_MCR_VO_COLOR_FMT_STR_3_SHIFT    23u
#define H264_DEC_MCR_VO_COLOR_FMT_STR_3_WIDTH    1u
#define H264_DEC_MCR_VO_COLOR_FMT_STR_3(x)       (((uint32_t)(((uint32_t)(x))<<H264_DEC_MCR_VO_COLOR_FMT_STR_3_SHIFT))&H264_DEC_MCR_VO_COLOR_FMT_STR_3_MASK)
#define H264_DEC_MCR_DATA_MODE_CHR_MASK          0x1000000u
#define H264_DEC_MCR_DATA_MODE_CHR_SHIFT         24u
#define H264_DEC_MCR_DATA_MODE_CHR_WIDTH         1u
#define H264_DEC_MCR_DATA_MODE_CHR(x)            (((uint32_t)(((uint32_t)(x))<<H264_DEC_MCR_DATA_MODE_CHR_SHIFT))&H264_DEC_MCR_DATA_MODE_CHR_MASK)
#define H264_DEC_MCR_CLR_IPS_FIFO_0_MASK         0x10000000u
#define H264_DEC_MCR_CLR_IPS_FIFO_0_SHIFT        28u
#define H264_DEC_MCR_CLR_IPS_FIFO_0_WIDTH        1u
#define H264_DEC_MCR_CLR_IPS_FIFO_0(x)           (((uint32_t)(((uint32_t)(x))<<H264_DEC_MCR_CLR_IPS_FIFO_0_SHIFT))&H264_DEC_MCR_CLR_IPS_FIFO_0_MASK)
#define H264_DEC_MCR_CLR_IPS_FIFO_1_MASK         0x20000000u
#define H264_DEC_MCR_CLR_IPS_FIFO_1_SHIFT        29u
#define H264_DEC_MCR_CLR_IPS_FIFO_1_WIDTH        1u
#define H264_DEC_MCR_CLR_IPS_FIFO_1(x)           (((uint32_t)(((uint32_t)(x))<<H264_DEC_MCR_CLR_IPS_FIFO_1_SHIFT))&H264_DEC_MCR_CLR_IPS_FIFO_1_MASK)
#define H264_DEC_MCR_CLR_IPS_FIFO_2_MASK         0x40000000u
#define H264_DEC_MCR_CLR_IPS_FIFO_2_SHIFT        30u
#define H264_DEC_MCR_CLR_IPS_FIFO_2_WIDTH        1u
#define H264_DEC_MCR_CLR_IPS_FIFO_2(x)           (((uint32_t)(((uint32_t)(x))<<H264_DEC_MCR_CLR_IPS_FIFO_2_SHIFT))&H264_DEC_MCR_CLR_IPS_FIFO_2_MASK)
#define H264_DEC_MCR_CLR_IPS_FIFO_3_MASK         0x80000000u
#define H264_DEC_MCR_CLR_IPS_FIFO_3_SHIFT        31u
#define H264_DEC_MCR_CLR_IPS_FIFO_3_WIDTH        1u
#define H264_DEC_MCR_CLR_IPS_FIFO_3(x)           (((uint32_t)(((uint32_t)(x))<<H264_DEC_MCR_CLR_IPS_FIFO_3_SHIFT))&H264_DEC_MCR_CLR_IPS_FIFO_3_MASK)
/* TIMEOUT_CFG Bit Fields */
#define H264_DEC_TIMEOUT_CFG_VAL_MASK            0xFFFFu
#define H264_DEC_TIMEOUT_CFG_VAL_SHIFT           0u
#define H264_DEC_TIMEOUT_CFG_VAL_WIDTH           16u
#define H264_DEC_TIMEOUT_CFG_VAL(x)              (((uint32_t)(((uint32_t)(x))<<H264_DEC_TIMEOUT_CFG_VAL_SHIFT))&H264_DEC_TIMEOUT_CFG_VAL_MASK)
#define H264_DEC_TIMEOUT_CFG_EN_STR_0_MASK       0x10000000u
#define H264_DEC_TIMEOUT_CFG_EN_STR_0_SHIFT      28u
#define H264_DEC_TIMEOUT_CFG_EN_STR_0_WIDTH      1u
#define H264_DEC_TIMEOUT_CFG_EN_STR_0(x)         (((uint32_t)(((uint32_t)(x))<<H264_DEC_TIMEOUT_CFG_EN_STR_0_SHIFT))&H264_DEC_TIMEOUT_CFG_EN_STR_0_MASK)
#define H264_DEC_TIMEOUT_CFG_EN_STR_1_MASK       0x20000000u
#define H264_DEC_TIMEOUT_CFG_EN_STR_1_SHIFT      29u
#define H264_DEC_TIMEOUT_CFG_EN_STR_1_WIDTH      1u
#define H264_DEC_TIMEOUT_CFG_EN_STR_1(x)         (((uint32_t)(((uint32_t)(x))<<H264_DEC_TIMEOUT_CFG_EN_STR_1_SHIFT))&H264_DEC_TIMEOUT_CFG_EN_STR_1_MASK)
#define H264_DEC_TIMEOUT_CFG_EN_STR_2_MASK       0x40000000u
#define H264_DEC_TIMEOUT_CFG_EN_STR_2_SHIFT      30u
#define H264_DEC_TIMEOUT_CFG_EN_STR_2_WIDTH      1u
#define H264_DEC_TIMEOUT_CFG_EN_STR_2(x)         (((uint32_t)(((uint32_t)(x))<<H264_DEC_TIMEOUT_CFG_EN_STR_2_SHIFT))&H264_DEC_TIMEOUT_CFG_EN_STR_2_MASK)
#define H264_DEC_TIMEOUT_CFG_EN_STR_3_MASK       0x80000000u
#define H264_DEC_TIMEOUT_CFG_EN_STR_3_SHIFT      31u
#define H264_DEC_TIMEOUT_CFG_EN_STR_3_WIDTH      1u
#define H264_DEC_TIMEOUT_CFG_EN_STR_3(x)         (((uint32_t)(((uint32_t)(x))<<H264_DEC_TIMEOUT_CFG_EN_STR_3_SHIFT))&H264_DEC_TIMEOUT_CFG_EN_STR_3_MASK)
/* STR_VCR Bit Fields */
#define H264_DEC_STR_VCR_FR_WIDTH_MASK           0xFF0u
#define H264_DEC_STR_VCR_FR_WIDTH_SHIFT          4u
#define H264_DEC_STR_VCR_FR_WIDTH_WIDTH          8u
#define H264_DEC_STR_VCR_FR_WIDTH(x)             (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_VCR_FR_WIDTH_SHIFT))&H264_DEC_STR_VCR_FR_WIDTH_MASK)
#define H264_DEC_STR_VCR_FR_HEIGHT_MASK          0xFF00000u
#define H264_DEC_STR_VCR_FR_HEIGHT_SHIFT         20u
#define H264_DEC_STR_VCR_FR_HEIGHT_WIDTH         8u
#define H264_DEC_STR_VCR_FR_HEIGHT(x)            (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_VCR_FR_HEIGHT_SHIFT))&H264_DEC_STR_VCR_FR_HEIGHT_MASK)
/* STR_PKT_ADDR Bit Fields */
#define H264_DEC_STR_PKT_ADDR_ADDR_MASK          0x3FFFFFu
#define H264_DEC_STR_PKT_ADDR_ADDR_SHIFT         0u
#define H264_DEC_STR_PKT_ADDR_ADDR_WIDTH         22u
#define H264_DEC_STR_PKT_ADDR_ADDR(x)            (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_PKT_ADDR_ADDR_SHIFT))&H264_DEC_STR_PKT_ADDR_ADDR_MASK)
/* STR_TRIG_PKT_CFG Bit Fields */
#define H264_DEC_STR_TRIG_PKT_CFG_STR_PKT_LTH_MASK 0x7FFFu
#define H264_DEC_STR_TRIG_PKT_CFG_STR_PKT_LTH_SHIFT 0u
#define H264_DEC_STR_TRIG_PKT_CFG_STR_PKT_LTH_WIDTH 15u
#define H264_DEC_STR_TRIG_PKT_CFG_STR_PKT_LTH(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_TRIG_PKT_CFG_STR_PKT_LTH_SHIFT))&H264_DEC_STR_TRIG_PKT_CFG_STR_PKT_LTH_MASK)
#define H264_DEC_STR_TRIG_PKT_CFG_STR_NUM_MASK   0xC0000000u
#define H264_DEC_STR_TRIG_PKT_CFG_STR_NUM_SHIFT  30u
#define H264_DEC_STR_TRIG_PKT_CFG_STR_NUM_WIDTH  2u
#define H264_DEC_STR_TRIG_PKT_CFG_STR_NUM(x)     (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_TRIG_PKT_CFG_STR_NUM_SHIFT))&H264_DEC_STR_TRIG_PKT_CFG_STR_NUM_MASK)
/* STR_PKT_FIFO_WMRK Bit Fields */
#define H264_DEC_STR_PKT_FIFO_WMRK_STR_0_MASK    0xFFu
#define H264_DEC_STR_PKT_FIFO_WMRK_STR_0_SHIFT   0u
#define H264_DEC_STR_PKT_FIFO_WMRK_STR_0_WIDTH   8u
#define H264_DEC_STR_PKT_FIFO_WMRK_STR_0(x)      (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_PKT_FIFO_WMRK_STR_0_SHIFT))&H264_DEC_STR_PKT_FIFO_WMRK_STR_0_MASK)
#define H264_DEC_STR_PKT_FIFO_WMRK_STR_1_MASK    0xFF00u
#define H264_DEC_STR_PKT_FIFO_WMRK_STR_1_SHIFT   8u
#define H264_DEC_STR_PKT_FIFO_WMRK_STR_1_WIDTH   8u
#define H264_DEC_STR_PKT_FIFO_WMRK_STR_1(x)      (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_PKT_FIFO_WMRK_STR_1_SHIFT))&H264_DEC_STR_PKT_FIFO_WMRK_STR_1_MASK)
#define H264_DEC_STR_PKT_FIFO_WMRK_STR_2_MASK    0xFF0000u
#define H264_DEC_STR_PKT_FIFO_WMRK_STR_2_SHIFT   16u
#define H264_DEC_STR_PKT_FIFO_WMRK_STR_2_WIDTH   8u
#define H264_DEC_STR_PKT_FIFO_WMRK_STR_2(x)      (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_PKT_FIFO_WMRK_STR_2_SHIFT))&H264_DEC_STR_PKT_FIFO_WMRK_STR_2_MASK)
#define H264_DEC_STR_PKT_FIFO_WMRK_STR_3_MASK    0xFF000000u
#define H264_DEC_STR_PKT_FIFO_WMRK_STR_3_SHIFT   24u
#define H264_DEC_STR_PKT_FIFO_WMRK_STR_3_WIDTH   8u
#define H264_DEC_STR_PKT_FIFO_WMRK_STR_3(x)      (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_PKT_FIFO_WMRK_STR_3_SHIFT))&H264_DEC_STR_PKT_FIFO_WMRK_STR_3_MASK)
/* STR_01_PKT_FIFO_STAT Bit Fields */
#define H264_DEC_STR_01_PKT_FIFO_STAT_STR_0_LVL_MASK 0xFFu
#define H264_DEC_STR_01_PKT_FIFO_STAT_STR_0_LVL_SHIFT 0u
#define H264_DEC_STR_01_PKT_FIFO_STAT_STR_0_LVL_WIDTH 8u
#define H264_DEC_STR_01_PKT_FIFO_STAT_STR_0_LVL(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_01_PKT_FIFO_STAT_STR_0_LVL_SHIFT))&H264_DEC_STR_01_PKT_FIFO_STAT_STR_0_LVL_MASK)
#define H264_DEC_STR_01_PKT_FIFO_STAT_STR_1_LVL_MASK 0xFF00u
#define H264_DEC_STR_01_PKT_FIFO_STAT_STR_1_LVL_SHIFT 8u
#define H264_DEC_STR_01_PKT_FIFO_STAT_STR_1_LVL_WIDTH 8u
#define H264_DEC_STR_01_PKT_FIFO_STAT_STR_1_LVL(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_01_PKT_FIFO_STAT_STR_1_LVL_SHIFT))&H264_DEC_STR_01_PKT_FIFO_STAT_STR_1_LVL_MASK)
#define H264_DEC_STR_01_PKT_FIFO_STAT_STR_0_ACTIV_LOC_MASK 0xFF0000u
#define H264_DEC_STR_01_PKT_FIFO_STAT_STR_0_ACTIV_LOC_SHIFT 16u
#define H264_DEC_STR_01_PKT_FIFO_STAT_STR_0_ACTIV_LOC_WIDTH 8u
#define H264_DEC_STR_01_PKT_FIFO_STAT_STR_0_ACTIV_LOC(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_01_PKT_FIFO_STAT_STR_0_ACTIV_LOC_SHIFT))&H264_DEC_STR_01_PKT_FIFO_STAT_STR_0_ACTIV_LOC_MASK)
#define H264_DEC_STR_01_PKT_FIFO_STAT_STR_1_ACTIV_LOC_MASK 0xFF000000u
#define H264_DEC_STR_01_PKT_FIFO_STAT_STR_1_ACTIV_LOC_SHIFT 24u
#define H264_DEC_STR_01_PKT_FIFO_STAT_STR_1_ACTIV_LOC_WIDTH 8u
#define H264_DEC_STR_01_PKT_FIFO_STAT_STR_1_ACTIV_LOC(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_01_PKT_FIFO_STAT_STR_1_ACTIV_LOC_SHIFT))&H264_DEC_STR_01_PKT_FIFO_STAT_STR_1_ACTIV_LOC_MASK)
/* STR_23_PKT_FIFO_STAT Bit Fields */
#define H264_DEC_STR_23_PKT_FIFO_STAT_STR_2_LVL_MASK 0xFFu
#define H264_DEC_STR_23_PKT_FIFO_STAT_STR_2_LVL_SHIFT 0u
#define H264_DEC_STR_23_PKT_FIFO_STAT_STR_2_LVL_WIDTH 8u
#define H264_DEC_STR_23_PKT_FIFO_STAT_STR_2_LVL(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_23_PKT_FIFO_STAT_STR_2_LVL_SHIFT))&H264_DEC_STR_23_PKT_FIFO_STAT_STR_2_LVL_MASK)
#define H264_DEC_STR_23_PKT_FIFO_STAT_STR_3_LVL_MASK 0xFF00u
#define H264_DEC_STR_23_PKT_FIFO_STAT_STR_3_LVL_SHIFT 8u
#define H264_DEC_STR_23_PKT_FIFO_STAT_STR_3_LVL_WIDTH 8u
#define H264_DEC_STR_23_PKT_FIFO_STAT_STR_3_LVL(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_23_PKT_FIFO_STAT_STR_3_LVL_SHIFT))&H264_DEC_STR_23_PKT_FIFO_STAT_STR_3_LVL_MASK)
#define H264_DEC_STR_23_PKT_FIFO_STAT_STR_2_ACTIV_LOC_MASK 0xFF0000u
#define H264_DEC_STR_23_PKT_FIFO_STAT_STR_2_ACTIV_LOC_SHIFT 16u
#define H264_DEC_STR_23_PKT_FIFO_STAT_STR_2_ACTIV_LOC_WIDTH 8u
#define H264_DEC_STR_23_PKT_FIFO_STAT_STR_2_ACTIV_LOC(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_23_PKT_FIFO_STAT_STR_2_ACTIV_LOC_SHIFT))&H264_DEC_STR_23_PKT_FIFO_STAT_STR_2_ACTIV_LOC_MASK)
#define H264_DEC_STR_23_PKT_FIFO_STAT_STR_3_ACTIV_LOC_MASK 0xFF000000u
#define H264_DEC_STR_23_PKT_FIFO_STAT_STR_3_ACTIV_LOC_SHIFT 24u
#define H264_DEC_STR_23_PKT_FIFO_STAT_STR_3_ACTIV_LOC_WIDTH 8u
#define H264_DEC_STR_23_PKT_FIFO_STAT_STR_3_ACTIV_LOC(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_23_PKT_FIFO_STAT_STR_3_ACTIV_LOC_SHIFT))&H264_DEC_STR_23_PKT_FIFO_STAT_STR_3_ACTIV_LOC_MASK)
/* STR_0_PKT_STATUS Bit Fields */
#define H264_DEC_STR_0_PKT_STATUS_ADDR_OFFSET_MASK 0xFFFFu
#define H264_DEC_STR_0_PKT_STATUS_ADDR_OFFSET_SHIFT 0u
#define H264_DEC_STR_0_PKT_STATUS_ADDR_OFFSET_WIDTH 16u
#define H264_DEC_STR_0_PKT_STATUS_ADDR_OFFSET(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_0_PKT_STATUS_ADDR_OFFSET_SHIFT))&H264_DEC_STR_0_PKT_STATUS_ADDR_OFFSET_MASK)
/* STR_1_PKT_STATUS Bit Fields */
#define H264_DEC_STR_1_PKT_STATUS_ADDR_OFFSET_MASK 0xFFFFu
#define H264_DEC_STR_1_PKT_STATUS_ADDR_OFFSET_SHIFT 0u
#define H264_DEC_STR_1_PKT_STATUS_ADDR_OFFSET_WIDTH 16u
#define H264_DEC_STR_1_PKT_STATUS_ADDR_OFFSET(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_1_PKT_STATUS_ADDR_OFFSET_SHIFT))&H264_DEC_STR_1_PKT_STATUS_ADDR_OFFSET_MASK)
/* STR_2_PKT_STATUS Bit Fields */
#define H264_DEC_STR_2_PKT_STATUS_ADDR_OFFSET_MASK 0xFFFFu
#define H264_DEC_STR_2_PKT_STATUS_ADDR_OFFSET_SHIFT 0u
#define H264_DEC_STR_2_PKT_STATUS_ADDR_OFFSET_WIDTH 16u
#define H264_DEC_STR_2_PKT_STATUS_ADDR_OFFSET(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_2_PKT_STATUS_ADDR_OFFSET_SHIFT))&H264_DEC_STR_2_PKT_STATUS_ADDR_OFFSET_MASK)
/* STR_3_PKT_STATUS Bit Fields */
#define H264_DEC_STR_3_PKT_STATUS_ADDR_OFFSET_MASK 0xFFFFu
#define H264_DEC_STR_3_PKT_STATUS_ADDR_OFFSET_SHIFT 0u
#define H264_DEC_STR_3_PKT_STATUS_ADDR_OFFSET_WIDTH 16u
#define H264_DEC_STR_3_PKT_STATUS_ADDR_OFFSET(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_STR_3_PKT_STATUS_ADDR_OFFSET_SHIFT))&H264_DEC_STR_3_PKT_STATUS_ADDR_OFFSET_MASK)
/* VO_STR_Y_ADDR Bit Fields */
#define H264_DEC_VO_STR_Y_ADDR_ADDR_MASK         0x3FFFF8u
#define H264_DEC_VO_STR_Y_ADDR_ADDR_SHIFT        3u
#define H264_DEC_VO_STR_Y_ADDR_ADDR_WIDTH        19u
#define H264_DEC_VO_STR_Y_ADDR_ADDR(x)           (((uint32_t)(((uint32_t)(x))<<H264_DEC_VO_STR_Y_ADDR_ADDR_SHIFT))&H264_DEC_VO_STR_Y_ADDR_ADDR_MASK)
/* VO_STR_CB_ADDR Bit Fields */
#define H264_DEC_VO_STR_CB_ADDR_ADDR_MASK        0x3FFFF8u
#define H264_DEC_VO_STR_CB_ADDR_ADDR_SHIFT       3u
#define H264_DEC_VO_STR_CB_ADDR_ADDR_WIDTH       19u
#define H264_DEC_VO_STR_CB_ADDR_ADDR(x)          (((uint32_t)(((uint32_t)(x))<<H264_DEC_VO_STR_CB_ADDR_ADDR_SHIFT))&H264_DEC_VO_STR_CB_ADDR_ADDR_MASK)
/* VO_STR_CR_ADDR Bit Fields */
#define H264_DEC_VO_STR_CR_ADDR_ADDR_MASK        0x3FFFF8u
#define H264_DEC_VO_STR_CR_ADDR_ADDR_SHIFT       3u
#define H264_DEC_VO_STR_CR_ADDR_ADDR_WIDTH       19u
#define H264_DEC_VO_STR_CR_ADDR_ADDR(x)          (((uint32_t)(((uint32_t)(x))<<H264_DEC_VO_STR_CR_ADDR_ADDR_SHIFT))&H264_DEC_VO_STR_CR_ADDR_ADDR_MASK)
/* VO_STR_NRLINES Bit Fields */
#define H264_DEC_VO_STR_NRLINES_Y_LINES_MASK     0xFFFFu
#define H264_DEC_VO_STR_NRLINES_Y_LINES_SHIFT    0u
#define H264_DEC_VO_STR_NRLINES_Y_LINES_WIDTH    16u
#define H264_DEC_VO_STR_NRLINES_Y_LINES(x)       (((uint32_t)(((uint32_t)(x))<<H264_DEC_VO_STR_NRLINES_Y_LINES_SHIFT))&H264_DEC_VO_STR_NRLINES_Y_LINES_MASK)
/* RATE_FLOW_CNTRL Bit Fields */
#define H264_DEC_RATE_FLOW_CNTRL_BANK_STRIDE_LTH_MASK 0x7u
#define H264_DEC_RATE_FLOW_CNTRL_BANK_STRIDE_LTH_SHIFT 0u
#define H264_DEC_RATE_FLOW_CNTRL_BANK_STRIDE_LTH_WIDTH 3u
#define H264_DEC_RATE_FLOW_CNTRL_BANK_STRIDE_LTH(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_RATE_FLOW_CNTRL_BANK_STRIDE_LTH_SHIFT))&H264_DEC_RATE_FLOW_CNTRL_BANK_STRIDE_LTH_MASK)
#define H264_DEC_RATE_FLOW_CNTRL_EN_MASK         0x80000000u
#define H264_DEC_RATE_FLOW_CNTRL_EN_SHIFT        31u
#define H264_DEC_RATE_FLOW_CNTRL_EN_WIDTH        1u
#define H264_DEC_RATE_FLOW_CNTRL_EN(x)           (((uint32_t)(((uint32_t)(x))<<H264_DEC_RATE_FLOW_CNTRL_EN_SHIFT))&H264_DEC_RATE_FLOW_CNTRL_EN_MASK)
/* LINE_CNT_STAT Bit Fields */
#define H264_DEC_LINE_CNT_STAT_STR_0_MASK        0xFFu
#define H264_DEC_LINE_CNT_STAT_STR_0_SHIFT       0u
#define H264_DEC_LINE_CNT_STAT_STR_0_WIDTH       8u
#define H264_DEC_LINE_CNT_STAT_STR_0(x)          (((uint32_t)(((uint32_t)(x))<<H264_DEC_LINE_CNT_STAT_STR_0_SHIFT))&H264_DEC_LINE_CNT_STAT_STR_0_MASK)
#define H264_DEC_LINE_CNT_STAT_STR_1_MASK        0xFF00u
#define H264_DEC_LINE_CNT_STAT_STR_1_SHIFT       8u
#define H264_DEC_LINE_CNT_STAT_STR_1_WIDTH       8u
#define H264_DEC_LINE_CNT_STAT_STR_1(x)          (((uint32_t)(((uint32_t)(x))<<H264_DEC_LINE_CNT_STAT_STR_1_SHIFT))&H264_DEC_LINE_CNT_STAT_STR_1_MASK)
#define H264_DEC_LINE_CNT_STAT_STR_2_MASK        0xFF0000u
#define H264_DEC_LINE_CNT_STAT_STR_2_SHIFT       16u
#define H264_DEC_LINE_CNT_STAT_STR_2_WIDTH       8u
#define H264_DEC_LINE_CNT_STAT_STR_2(x)          (((uint32_t)(((uint32_t)(x))<<H264_DEC_LINE_CNT_STAT_STR_2_SHIFT))&H264_DEC_LINE_CNT_STAT_STR_2_MASK)
#define H264_DEC_LINE_CNT_STAT_STR_3_MASK        0xFF000000u
#define H264_DEC_LINE_CNT_STAT_STR_3_SHIFT       24u
#define H264_DEC_LINE_CNT_STAT_STR_3_WIDTH       8u
#define H264_DEC_LINE_CNT_STAT_STR_3(x)          (((uint32_t)(((uint32_t)(x))<<H264_DEC_LINE_CNT_STAT_STR_3_SHIFT))&H264_DEC_LINE_CNT_STAT_STR_3_MASK)
/* STAT Bit Fields */
#define H264_DEC_STAT_STR_0_TIMED_OUT_MASK       0x1u
#define H264_DEC_STAT_STR_0_TIMED_OUT_SHIFT      0u
#define H264_DEC_STAT_STR_0_TIMED_OUT_WIDTH      1u
#define H264_DEC_STAT_STR_0_TIMED_OUT(x)         (((uint32_t)(((uint32_t)(x))<<H264_DEC_STAT_STR_0_TIMED_OUT_SHIFT))&H264_DEC_STAT_STR_0_TIMED_OUT_MASK)
#define H264_DEC_STAT_STR_1_TIMED_OUT_MASK       0x2u
#define H264_DEC_STAT_STR_1_TIMED_OUT_SHIFT      1u
#define H264_DEC_STAT_STR_1_TIMED_OUT_WIDTH      1u
#define H264_DEC_STAT_STR_1_TIMED_OUT(x)         (((uint32_t)(((uint32_t)(x))<<H264_DEC_STAT_STR_1_TIMED_OUT_SHIFT))&H264_DEC_STAT_STR_1_TIMED_OUT_MASK)
#define H264_DEC_STAT_STR_2_TIMED_OUT_MASK       0x4u
#define H264_DEC_STAT_STR_2_TIMED_OUT_SHIFT      2u
#define H264_DEC_STAT_STR_2_TIMED_OUT_WIDTH      1u
#define H264_DEC_STAT_STR_2_TIMED_OUT(x)         (((uint32_t)(((uint32_t)(x))<<H264_DEC_STAT_STR_2_TIMED_OUT_SHIFT))&H264_DEC_STAT_STR_2_TIMED_OUT_MASK)
#define H264_DEC_STAT_STR_3_TIMED_OUT_MASK       0x8u
#define H264_DEC_STAT_STR_3_TIMED_OUT_SHIFT      3u
#define H264_DEC_STAT_STR_3_TIMED_OUT_WIDTH      1u
#define H264_DEC_STAT_STR_3_TIMED_OUT(x)         (((uint32_t)(((uint32_t)(x))<<H264_DEC_STAT_STR_3_TIMED_OUT_SHIFT))&H264_DEC_STAT_STR_3_TIMED_OUT_MASK)
/* ISR Bit Fields */
#define H264_DEC_ISR_EOF_0_INT_MASK              0x1u
#define H264_DEC_ISR_EOF_0_INT_SHIFT             0u
#define H264_DEC_ISR_EOF_0_INT_WIDTH             1u
#define H264_DEC_ISR_EOF_0_INT(x)                (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_EOF_0_INT_SHIFT))&H264_DEC_ISR_EOF_0_INT_MASK)
#define H264_DEC_ISR_EOF_1_INT_MASK              0x2u
#define H264_DEC_ISR_EOF_1_INT_SHIFT             1u
#define H264_DEC_ISR_EOF_1_INT_WIDTH             1u
#define H264_DEC_ISR_EOF_1_INT(x)                (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_EOF_1_INT_SHIFT))&H264_DEC_ISR_EOF_1_INT_MASK)
#define H264_DEC_ISR_EOF_2_INT_MASK              0x4u
#define H264_DEC_ISR_EOF_2_INT_SHIFT             2u
#define H264_DEC_ISR_EOF_2_INT_WIDTH             1u
#define H264_DEC_ISR_EOF_2_INT(x)                (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_EOF_2_INT_SHIFT))&H264_DEC_ISR_EOF_2_INT_MASK)
#define H264_DEC_ISR_EOF_3_INT_MASK              0x8u
#define H264_DEC_ISR_EOF_3_INT_SHIFT             3u
#define H264_DEC_ISR_EOF_3_INT_WIDTH             1u
#define H264_DEC_ISR_EOF_3_INT(x)                (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_EOF_3_INT_SHIFT))&H264_DEC_ISR_EOF_3_INT_MASK)
#define H264_DEC_ISR_MBROW_DONE_0_INT_MASK       0x10u
#define H264_DEC_ISR_MBROW_DONE_0_INT_SHIFT      4u
#define H264_DEC_ISR_MBROW_DONE_0_INT_WIDTH      1u
#define H264_DEC_ISR_MBROW_DONE_0_INT(x)         (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_MBROW_DONE_0_INT_SHIFT))&H264_DEC_ISR_MBROW_DONE_0_INT_MASK)
#define H264_DEC_ISR_MBROW_DONE_1_INT_MASK       0x20u
#define H264_DEC_ISR_MBROW_DONE_1_INT_SHIFT      5u
#define H264_DEC_ISR_MBROW_DONE_1_INT_WIDTH      1u
#define H264_DEC_ISR_MBROW_DONE_1_INT(x)         (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_MBROW_DONE_1_INT_SHIFT))&H264_DEC_ISR_MBROW_DONE_1_INT_MASK)
#define H264_DEC_ISR_MBROW_DONE_2_INT_MASK       0x40u
#define H264_DEC_ISR_MBROW_DONE_2_INT_SHIFT      6u
#define H264_DEC_ISR_MBROW_DONE_2_INT_WIDTH      1u
#define H264_DEC_ISR_MBROW_DONE_2_INT(x)         (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_MBROW_DONE_2_INT_SHIFT))&H264_DEC_ISR_MBROW_DONE_2_INT_MASK)
#define H264_DEC_ISR_MBROW_DONE_3_INT_MASK       0x80u
#define H264_DEC_ISR_MBROW_DONE_3_INT_SHIFT      7u
#define H264_DEC_ISR_MBROW_DONE_3_INT_WIDTH      1u
#define H264_DEC_ISR_MBROW_DONE_3_INT(x)         (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_MBROW_DONE_3_INT_SHIFT))&H264_DEC_ISR_MBROW_DONE_3_INT_MASK)
#define H264_DEC_ISR_STR_0_PKT_FIFO_WE_INT_MASK  0x100u
#define H264_DEC_ISR_STR_0_PKT_FIFO_WE_INT_SHIFT 8u
#define H264_DEC_ISR_STR_0_PKT_FIFO_WE_INT_WIDTH 1u
#define H264_DEC_ISR_STR_0_PKT_FIFO_WE_INT(x)    (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_STR_0_PKT_FIFO_WE_INT_SHIFT))&H264_DEC_ISR_STR_0_PKT_FIFO_WE_INT_MASK)
#define H264_DEC_ISR_STR_1_PKT_FIFO_WE_INT_MASK  0x200u
#define H264_DEC_ISR_STR_1_PKT_FIFO_WE_INT_SHIFT 9u
#define H264_DEC_ISR_STR_1_PKT_FIFO_WE_INT_WIDTH 1u
#define H264_DEC_ISR_STR_1_PKT_FIFO_WE_INT(x)    (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_STR_1_PKT_FIFO_WE_INT_SHIFT))&H264_DEC_ISR_STR_1_PKT_FIFO_WE_INT_MASK)
#define H264_DEC_ISR_STR_2_PKT_FIFO_WE_INT_MASK  0x400u
#define H264_DEC_ISR_STR_2_PKT_FIFO_WE_INT_SHIFT 10u
#define H264_DEC_ISR_STR_2_PKT_FIFO_WE_INT_WIDTH 1u
#define H264_DEC_ISR_STR_2_PKT_FIFO_WE_INT(x)    (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_STR_2_PKT_FIFO_WE_INT_SHIFT))&H264_DEC_ISR_STR_2_PKT_FIFO_WE_INT_MASK)
#define H264_DEC_ISR_STR_3_PKT_FIFO_WE_INT_MASK  0x800u
#define H264_DEC_ISR_STR_3_PKT_FIFO_WE_INT_SHIFT 11u
#define H264_DEC_ISR_STR_3_PKT_FIFO_WE_INT_WIDTH 1u
#define H264_DEC_ISR_STR_3_PKT_FIFO_WE_INT(x)    (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_STR_3_PKT_FIFO_WE_INT_SHIFT))&H264_DEC_ISR_STR_3_PKT_FIFO_WE_INT_MASK)
#define H264_DEC_ISR_DEC_OOS_STR_0_INT_MASK      0x1000u
#define H264_DEC_ISR_DEC_OOS_STR_0_INT_SHIFT     12u
#define H264_DEC_ISR_DEC_OOS_STR_0_INT_WIDTH     1u
#define H264_DEC_ISR_DEC_OOS_STR_0_INT(x)        (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_DEC_OOS_STR_0_INT_SHIFT))&H264_DEC_ISR_DEC_OOS_STR_0_INT_MASK)
#define H264_DEC_ISR_DEC_OOS_STR_1_INT_MASK      0x2000u
#define H264_DEC_ISR_DEC_OOS_STR_1_INT_SHIFT     13u
#define H264_DEC_ISR_DEC_OOS_STR_1_INT_WIDTH     1u
#define H264_DEC_ISR_DEC_OOS_STR_1_INT(x)        (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_DEC_OOS_STR_1_INT_SHIFT))&H264_DEC_ISR_DEC_OOS_STR_1_INT_MASK)
#define H264_DEC_ISR_DEC_OOS_STR_2_INT_MASK      0x4000u
#define H264_DEC_ISR_DEC_OOS_STR_2_INT_SHIFT     14u
#define H264_DEC_ISR_DEC_OOS_STR_2_INT_WIDTH     1u
#define H264_DEC_ISR_DEC_OOS_STR_2_INT(x)        (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_DEC_OOS_STR_2_INT_SHIFT))&H264_DEC_ISR_DEC_OOS_STR_2_INT_MASK)
#define H264_DEC_ISR_DEC_OOS_STR_3_INT_MASK      0x8000u
#define H264_DEC_ISR_DEC_OOS_STR_3_INT_SHIFT     15u
#define H264_DEC_ISR_DEC_OOS_STR_3_INT_WIDTH     1u
#define H264_DEC_ISR_DEC_OOS_STR_3_INT(x)        (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_DEC_OOS_STR_3_INT_SHIFT))&H264_DEC_ISR_DEC_OOS_STR_3_INT_MASK)
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_0_INT_MASK 0x10000u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_0_INT_SHIFT 16u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_0_INT_WIDTH 1u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_0_INT(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_0_INT_SHIFT))&H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_0_INT_MASK)
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_1_INT_MASK 0x20000u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_1_INT_SHIFT 17u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_1_INT_WIDTH 1u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_1_INT(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_1_INT_SHIFT))&H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_1_INT_MASK)
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_2_INT_MASK 0x40000u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_2_INT_SHIFT 18u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_2_INT_WIDTH 1u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_2_INT(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_2_INT_SHIFT))&H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_2_INT_MASK)
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_3_INT_MASK 0x80000u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_3_INT_SHIFT 19u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_3_INT_WIDTH 1u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_3_INT(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_3_INT_SHIFT))&H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_0_CH_3_INT_MASK)
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_0_INT_MASK 0x100000u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_0_INT_SHIFT 20u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_0_INT_WIDTH 1u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_0_INT(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_0_INT_SHIFT))&H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_0_INT_MASK)
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_1_INT_MASK 0x200000u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_1_INT_SHIFT 21u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_1_INT_WIDTH 1u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_1_INT(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_1_INT_SHIFT))&H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_1_INT_MASK)
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_2_INT_MASK 0x400000u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_2_INT_SHIFT 22u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_2_INT_WIDTH 1u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_2_INT(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_2_INT_SHIFT))&H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_2_INT_MASK)
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_3_INT_MASK 0x800000u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_3_INT_SHIFT 23u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_3_INT_WIDTH 1u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_3_INT(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_3_INT_SHIFT))&H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_1_CH_3_INT_MASK)
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_0_INT_MASK 0x1000000u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_0_INT_SHIFT 24u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_0_INT_WIDTH 1u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_0_INT(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_0_INT_SHIFT))&H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_0_INT_MASK)
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_1_INT_MASK 0x2000000u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_1_INT_SHIFT 25u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_1_INT_WIDTH 1u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_1_INT(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_1_INT_SHIFT))&H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_1_INT_MASK)
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_2_INT_MASK 0x4000000u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_2_INT_SHIFT 26u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_2_INT_WIDTH 1u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_2_INT(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_2_INT_SHIFT))&H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_2_INT_MASK)
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_3_INT_MASK 0x8000000u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_3_INT_SHIFT 27u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_3_INT_WIDTH 1u
#define H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_3_INT(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_3_INT_SHIFT))&H264_DEC_ISR_FRM_CYC_CNT_GRT_THR_2_CH_3_INT_MASK)
#define H264_DEC_ISR_STR_TIMED_OUT_INT_MASK      0x40000000u
#define H264_DEC_ISR_STR_TIMED_OUT_INT_SHIFT     30u
#define H264_DEC_ISR_STR_TIMED_OUT_INT_WIDTH     1u
#define H264_DEC_ISR_STR_TIMED_OUT_INT(x)        (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_STR_TIMED_OUT_INT_SHIFT))&H264_DEC_ISR_STR_TIMED_OUT_INT_MASK)
#define H264_DEC_ISR_ERR_REQ_INT_MASK            0x80000000u
#define H264_DEC_ISR_ERR_REQ_INT_SHIFT           31u
#define H264_DEC_ISR_ERR_REQ_INT_WIDTH           1u
#define H264_DEC_ISR_ERR_REQ_INT(x)              (((uint32_t)(((uint32_t)(x))<<H264_DEC_ISR_ERR_REQ_INT_SHIFT))&H264_DEC_ISR_ERR_REQ_INT_MASK)
/* IER Bit Fields */
#define H264_DEC_IER_EOF_0_EN_MASK               0x1u
#define H264_DEC_IER_EOF_0_EN_SHIFT              0u
#define H264_DEC_IER_EOF_0_EN_WIDTH              1u
#define H264_DEC_IER_EOF_0_EN(x)                 (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_EOF_0_EN_SHIFT))&H264_DEC_IER_EOF_0_EN_MASK)
#define H264_DEC_IER_EOF_1_EN_MASK               0x2u
#define H264_DEC_IER_EOF_1_EN_SHIFT              1u
#define H264_DEC_IER_EOF_1_EN_WIDTH              1u
#define H264_DEC_IER_EOF_1_EN(x)                 (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_EOF_1_EN_SHIFT))&H264_DEC_IER_EOF_1_EN_MASK)
#define H264_DEC_IER_EOF_2_EN_MASK               0x4u
#define H264_DEC_IER_EOF_2_EN_SHIFT              2u
#define H264_DEC_IER_EOF_2_EN_WIDTH              1u
#define H264_DEC_IER_EOF_2_EN(x)                 (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_EOF_2_EN_SHIFT))&H264_DEC_IER_EOF_2_EN_MASK)
#define H264_DEC_IER_EOF_3_EN_MASK               0x8u
#define H264_DEC_IER_EOF_3_EN_SHIFT              3u
#define H264_DEC_IER_EOF_3_EN_WIDTH              1u
#define H264_DEC_IER_EOF_3_EN(x)                 (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_EOF_3_EN_SHIFT))&H264_DEC_IER_EOF_3_EN_MASK)
#define H264_DEC_IER_MBROW_DONE_0_EN_MASK        0x10u
#define H264_DEC_IER_MBROW_DONE_0_EN_SHIFT       4u
#define H264_DEC_IER_MBROW_DONE_0_EN_WIDTH       1u
#define H264_DEC_IER_MBROW_DONE_0_EN(x)          (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_MBROW_DONE_0_EN_SHIFT))&H264_DEC_IER_MBROW_DONE_0_EN_MASK)
#define H264_DEC_IER_MBROW_DONE_1_EN_MASK        0x20u
#define H264_DEC_IER_MBROW_DONE_1_EN_SHIFT       5u
#define H264_DEC_IER_MBROW_DONE_1_EN_WIDTH       1u
#define H264_DEC_IER_MBROW_DONE_1_EN(x)          (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_MBROW_DONE_1_EN_SHIFT))&H264_DEC_IER_MBROW_DONE_1_EN_MASK)
#define H264_DEC_IER_MBROW_DONE_2_EN_MASK        0x40u
#define H264_DEC_IER_MBROW_DONE_2_EN_SHIFT       6u
#define H264_DEC_IER_MBROW_DONE_2_EN_WIDTH       1u
#define H264_DEC_IER_MBROW_DONE_2_EN(x)          (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_MBROW_DONE_2_EN_SHIFT))&H264_DEC_IER_MBROW_DONE_2_EN_MASK)
#define H264_DEC_IER_MBROW_DONE_3_EN_MASK        0x80u
#define H264_DEC_IER_MBROW_DONE_3_EN_SHIFT       7u
#define H264_DEC_IER_MBROW_DONE_3_EN_WIDTH       1u
#define H264_DEC_IER_MBROW_DONE_3_EN(x)          (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_MBROW_DONE_3_EN_SHIFT))&H264_DEC_IER_MBROW_DONE_3_EN_MASK)
#define H264_DEC_IER_STR_0_PKT_FIFO_WE_EN_MASK   0x100u
#define H264_DEC_IER_STR_0_PKT_FIFO_WE_EN_SHIFT  8u
#define H264_DEC_IER_STR_0_PKT_FIFO_WE_EN_WIDTH  1u
#define H264_DEC_IER_STR_0_PKT_FIFO_WE_EN(x)     (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_STR_0_PKT_FIFO_WE_EN_SHIFT))&H264_DEC_IER_STR_0_PKT_FIFO_WE_EN_MASK)
#define H264_DEC_IER_STR_1_PKT_FIFO_WE_EN_MASK   0x200u
#define H264_DEC_IER_STR_1_PKT_FIFO_WE_EN_SHIFT  9u
#define H264_DEC_IER_STR_1_PKT_FIFO_WE_EN_WIDTH  1u
#define H264_DEC_IER_STR_1_PKT_FIFO_WE_EN(x)     (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_STR_1_PKT_FIFO_WE_EN_SHIFT))&H264_DEC_IER_STR_1_PKT_FIFO_WE_EN_MASK)
#define H264_DEC_IER_STR_2_PKT_FIFO_WE_EN_MASK   0x400u
#define H264_DEC_IER_STR_2_PKT_FIFO_WE_EN_SHIFT  10u
#define H264_DEC_IER_STR_2_PKT_FIFO_WE_EN_WIDTH  1u
#define H264_DEC_IER_STR_2_PKT_FIFO_WE_EN(x)     (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_STR_2_PKT_FIFO_WE_EN_SHIFT))&H264_DEC_IER_STR_2_PKT_FIFO_WE_EN_MASK)
#define H264_DEC_IER_STR_3_PKT_FIFO_WE_EN_MASK   0x800u
#define H264_DEC_IER_STR_3_PKT_FIFO_WE_EN_SHIFT  11u
#define H264_DEC_IER_STR_3_PKT_FIFO_WE_EN_WIDTH  1u
#define H264_DEC_IER_STR_3_PKT_FIFO_WE_EN(x)     (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_STR_3_PKT_FIFO_WE_EN_SHIFT))&H264_DEC_IER_STR_3_PKT_FIFO_WE_EN_MASK)
#define H264_DEC_IER_DEC_OOS_STR_0_EN_MASK       0x1000u
#define H264_DEC_IER_DEC_OOS_STR_0_EN_SHIFT      12u
#define H264_DEC_IER_DEC_OOS_STR_0_EN_WIDTH      1u
#define H264_DEC_IER_DEC_OOS_STR_0_EN(x)         (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_DEC_OOS_STR_0_EN_SHIFT))&H264_DEC_IER_DEC_OOS_STR_0_EN_MASK)
#define H264_DEC_IER_DEC_OOS_STR_1_EN_MASK       0x2000u
#define H264_DEC_IER_DEC_OOS_STR_1_EN_SHIFT      13u
#define H264_DEC_IER_DEC_OOS_STR_1_EN_WIDTH      1u
#define H264_DEC_IER_DEC_OOS_STR_1_EN(x)         (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_DEC_OOS_STR_1_EN_SHIFT))&H264_DEC_IER_DEC_OOS_STR_1_EN_MASK)
#define H264_DEC_IER_DEC_OOS_STR_2_EN_MASK       0x4000u
#define H264_DEC_IER_DEC_OOS_STR_2_EN_SHIFT      14u
#define H264_DEC_IER_DEC_OOS_STR_2_EN_WIDTH      1u
#define H264_DEC_IER_DEC_OOS_STR_2_EN(x)         (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_DEC_OOS_STR_2_EN_SHIFT))&H264_DEC_IER_DEC_OOS_STR_2_EN_MASK)
#define H264_DEC_IER_DEC_OOS_STR_3_EN_MASK       0x8000u
#define H264_DEC_IER_DEC_OOS_STR_3_EN_SHIFT      15u
#define H264_DEC_IER_DEC_OOS_STR_3_EN_WIDTH      1u
#define H264_DEC_IER_DEC_OOS_STR_3_EN(x)         (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_DEC_OOS_STR_3_EN_SHIFT))&H264_DEC_IER_DEC_OOS_STR_3_EN_MASK)
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_0_EN_MASK 0x10000u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_0_EN_SHIFT 16u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_0_EN_WIDTH 1u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_0_EN(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_0_EN_SHIFT))&H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_0_EN_MASK)
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_1_EN_MASK 0x20000u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_1_EN_SHIFT 17u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_1_EN_WIDTH 1u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_1_EN(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_1_EN_SHIFT))&H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_1_EN_MASK)
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_2_EN_MASK 0x40000u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_2_EN_SHIFT 18u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_2_EN_WIDTH 1u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_2_EN(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_2_EN_SHIFT))&H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_2_EN_MASK)
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_3_EN_MASK 0x80000u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_3_EN_SHIFT 19u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_3_EN_WIDTH 1u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_3_EN(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_3_EN_SHIFT))&H264_DEC_IER_FRM_CYC_CNT_GRT_THR_0_CH_3_EN_MASK)
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_0_EN_MASK 0x100000u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_0_EN_SHIFT 20u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_0_EN_WIDTH 1u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_0_EN(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_0_EN_SHIFT))&H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_0_EN_MASK)
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_1_EN_MASK 0x200000u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_1_EN_SHIFT 21u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_1_EN_WIDTH 1u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_1_EN(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_1_EN_SHIFT))&H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_1_EN_MASK)
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_2_EN_MASK 0x400000u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_2_EN_SHIFT 22u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_2_EN_WIDTH 1u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_2_EN(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_2_EN_SHIFT))&H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_2_EN_MASK)
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_3_EN_MASK 0x800000u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_3_EN_SHIFT 23u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_3_EN_WIDTH 1u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_3_EN(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_3_EN_SHIFT))&H264_DEC_IER_FRM_CYC_CNT_GRT_THR_1_CH_3_EN_MASK)
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_0_EN_MASK 0x1000000u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_0_EN_SHIFT 24u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_0_EN_WIDTH 1u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_0_EN(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_0_EN_SHIFT))&H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_0_EN_MASK)
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_1_EN_MASK 0x2000000u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_1_EN_SHIFT 25u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_1_EN_WIDTH 1u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_1_EN(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_1_EN_SHIFT))&H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_1_EN_MASK)
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_2_EN_MASK 0x4000000u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_2_EN_SHIFT 26u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_2_EN_WIDTH 1u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_2_EN(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_2_EN_SHIFT))&H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_2_EN_MASK)
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_3_EN_MASK 0x8000000u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_3_EN_SHIFT 27u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_3_EN_WIDTH 1u
#define H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_3_EN(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_3_EN_SHIFT))&H264_DEC_IER_FRM_CYC_CNT_GRT_THR_2_CH_3_EN_MASK)
#define H264_DEC_IER_STR_TIMED_OUT_EN_MASK       0x40000000u
#define H264_DEC_IER_STR_TIMED_OUT_EN_SHIFT      30u
#define H264_DEC_IER_STR_TIMED_OUT_EN_WIDTH      1u
#define H264_DEC_IER_STR_TIMED_OUT_EN(x)         (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_STR_TIMED_OUT_EN_SHIFT))&H264_DEC_IER_STR_TIMED_OUT_EN_MASK)
#define H264_DEC_IER_ERR_REQ_EN_MASK             0x80000000u
#define H264_DEC_IER_ERR_REQ_EN_SHIFT            31u
#define H264_DEC_IER_ERR_REQ_EN_WIDTH            1u
#define H264_DEC_IER_ERR_REQ_EN(x)               (((uint32_t)(((uint32_t)(x))<<H264_DEC_IER_ERR_REQ_EN_SHIFT))&H264_DEC_IER_ERR_REQ_EN_MASK)
/* TESTLINE_CFG Bit Fields */
#define H264_DEC_TESTLINE_CFG_STRT_LOC_MASK      0xFFFu
#define H264_DEC_TESTLINE_CFG_STRT_LOC_SHIFT     0u
#define H264_DEC_TESTLINE_CFG_STRT_LOC_WIDTH     12u
#define H264_DEC_TESTLINE_CFG_STRT_LOC(x)        (((uint32_t)(((uint32_t)(x))<<H264_DEC_TESTLINE_CFG_STRT_LOC_SHIFT))&H264_DEC_TESTLINE_CFG_STRT_LOC_MASK)
#define H264_DEC_TESTLINE_CFG_EN_MASK            0x80000000u
#define H264_DEC_TESTLINE_CFG_EN_SHIFT           31u
#define H264_DEC_TESTLINE_CFG_EN_WIDTH           1u
#define H264_DEC_TESTLINE_CFG_EN(x)              (((uint32_t)(((uint32_t)(x))<<H264_DEC_TESTLINE_CFG_EN_SHIFT))&H264_DEC_TESTLINE_CFG_EN_MASK)
/* TESTLINE_PXL_LOC Bit Fields */
#define H264_DEC_TESTLINE_PXL_LOC_LOC_1_MASK     0xFF0u
#define H264_DEC_TESTLINE_PXL_LOC_LOC_1_SHIFT    4u
#define H264_DEC_TESTLINE_PXL_LOC_LOC_1_WIDTH    8u
#define H264_DEC_TESTLINE_PXL_LOC_LOC_1(x)       (((uint32_t)(((uint32_t)(x))<<H264_DEC_TESTLINE_PXL_LOC_LOC_1_SHIFT))&H264_DEC_TESTLINE_PXL_LOC_LOC_1_MASK)
#define H264_DEC_TESTLINE_PXL_LOC_LOC_2_MASK     0xFF00000u
#define H264_DEC_TESTLINE_PXL_LOC_LOC_2_SHIFT    20u
#define H264_DEC_TESTLINE_PXL_LOC_LOC_2_WIDTH    8u
#define H264_DEC_TESTLINE_PXL_LOC_LOC_2(x)       (((uint32_t)(((uint32_t)(x))<<H264_DEC_TESTLINE_PXL_LOC_LOC_2_SHIFT))&H264_DEC_TESTLINE_PXL_LOC_LOC_2_MASK)
/* TESTLINE_STR_LUMA_VAL Bit Fields */
#define H264_DEC_TESTLINE_STR_LUMA_VAL_VAL_1_MASK 0xFFFu
#define H264_DEC_TESTLINE_STR_LUMA_VAL_VAL_1_SHIFT 0u
#define H264_DEC_TESTLINE_STR_LUMA_VAL_VAL_1_WIDTH 12u
#define H264_DEC_TESTLINE_STR_LUMA_VAL_VAL_1(x)  (((uint32_t)(((uint32_t)(x))<<H264_DEC_TESTLINE_STR_LUMA_VAL_VAL_1_SHIFT))&H264_DEC_TESTLINE_STR_LUMA_VAL_VAL_1_MASK)
#define H264_DEC_TESTLINE_STR_LUMA_VAL_VAL_2_MASK 0xFFF0000u
#define H264_DEC_TESTLINE_STR_LUMA_VAL_VAL_2_SHIFT 16u
#define H264_DEC_TESTLINE_STR_LUMA_VAL_VAL_2_WIDTH 12u
#define H264_DEC_TESTLINE_STR_LUMA_VAL_VAL_2(x)  (((uint32_t)(((uint32_t)(x))<<H264_DEC_TESTLINE_STR_LUMA_VAL_VAL_2_SHIFT))&H264_DEC_TESTLINE_STR_LUMA_VAL_VAL_2_MASK)
/* TESTLINE_STR_CB_VAL Bit Fields */
#define H264_DEC_TESTLINE_STR_CB_VAL_VAL_1_MASK  0xFFFu
#define H264_DEC_TESTLINE_STR_CB_VAL_VAL_1_SHIFT 0u
#define H264_DEC_TESTLINE_STR_CB_VAL_VAL_1_WIDTH 12u
#define H264_DEC_TESTLINE_STR_CB_VAL_VAL_1(x)    (((uint32_t)(((uint32_t)(x))<<H264_DEC_TESTLINE_STR_CB_VAL_VAL_1_SHIFT))&H264_DEC_TESTLINE_STR_CB_VAL_VAL_1_MASK)
#define H264_DEC_TESTLINE_STR_CB_VAL_VAL_2_MASK  0xFFF0000u
#define H264_DEC_TESTLINE_STR_CB_VAL_VAL_2_SHIFT 16u
#define H264_DEC_TESTLINE_STR_CB_VAL_VAL_2_WIDTH 12u
#define H264_DEC_TESTLINE_STR_CB_VAL_VAL_2(x)    (((uint32_t)(((uint32_t)(x))<<H264_DEC_TESTLINE_STR_CB_VAL_VAL_2_SHIFT))&H264_DEC_TESTLINE_STR_CB_VAL_VAL_2_MASK)
/* TESTLINE_STR_CR_VAL Bit Fields */
#define H264_DEC_TESTLINE_STR_CR_VAL_VAL_1_MASK  0xFFFu
#define H264_DEC_TESTLINE_STR_CR_VAL_VAL_1_SHIFT 0u
#define H264_DEC_TESTLINE_STR_CR_VAL_VAL_1_WIDTH 12u
#define H264_DEC_TESTLINE_STR_CR_VAL_VAL_1(x)    (((uint32_t)(((uint32_t)(x))<<H264_DEC_TESTLINE_STR_CR_VAL_VAL_1_SHIFT))&H264_DEC_TESTLINE_STR_CR_VAL_VAL_1_MASK)
#define H264_DEC_TESTLINE_STR_CR_VAL_VAL_2_MASK  0xFFF0000u
#define H264_DEC_TESTLINE_STR_CR_VAL_VAL_2_SHIFT 16u
#define H264_DEC_TESTLINE_STR_CR_VAL_VAL_2_WIDTH 12u
#define H264_DEC_TESTLINE_STR_CR_VAL_VAL_2(x)    (((uint32_t)(((uint32_t)(x))<<H264_DEC_TESTLINE_STR_CR_VAL_VAL_2_SHIFT))&H264_DEC_TESTLINE_STR_CR_VAL_VAL_2_MASK)
/* ELLVC_CFG Bit Fields */
#define H264_DEC_ELLVC_CFG_DBF_FORCE_OFF_MASK    0x1u
#define H264_DEC_ELLVC_CFG_DBF_FORCE_OFF_SHIFT   0u
#define H264_DEC_ELLVC_CFG_DBF_FORCE_OFF_WIDTH   1u
#define H264_DEC_ELLVC_CFG_DBF_FORCE_OFF(x)      (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_CFG_DBF_FORCE_OFF_SHIFT))&H264_DEC_ELLVC_CFG_DBF_FORCE_OFF_MASK)
#define H264_DEC_ELLVC_CFG_USE_SRAM_DDR_MASK     0x100u
#define H264_DEC_ELLVC_CFG_USE_SRAM_DDR_SHIFT    8u
#define H264_DEC_ELLVC_CFG_USE_SRAM_DDR_WIDTH    1u
#define H264_DEC_ELLVC_CFG_USE_SRAM_DDR(x)       (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_CFG_USE_SRAM_DDR_SHIFT))&H264_DEC_ELLVC_CFG_USE_SRAM_DDR_MASK)
#define H264_DEC_ELLVC_CFG_STOP_CH_0_MASK        0x1000000u
#define H264_DEC_ELLVC_CFG_STOP_CH_0_SHIFT       24u
#define H264_DEC_ELLVC_CFG_STOP_CH_0_WIDTH       1u
#define H264_DEC_ELLVC_CFG_STOP_CH_0(x)          (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_CFG_STOP_CH_0_SHIFT))&H264_DEC_ELLVC_CFG_STOP_CH_0_MASK)
#define H264_DEC_ELLVC_CFG_STOP_CH_1_MASK        0x2000000u
#define H264_DEC_ELLVC_CFG_STOP_CH_1_SHIFT       25u
#define H264_DEC_ELLVC_CFG_STOP_CH_1_WIDTH       1u
#define H264_DEC_ELLVC_CFG_STOP_CH_1(x)          (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_CFG_STOP_CH_1_SHIFT))&H264_DEC_ELLVC_CFG_STOP_CH_1_MASK)
#define H264_DEC_ELLVC_CFG_STOP_CH_2_MASK        0x4000000u
#define H264_DEC_ELLVC_CFG_STOP_CH_2_SHIFT       26u
#define H264_DEC_ELLVC_CFG_STOP_CH_2_WIDTH       1u
#define H264_DEC_ELLVC_CFG_STOP_CH_2(x)          (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_CFG_STOP_CH_2_SHIFT))&H264_DEC_ELLVC_CFG_STOP_CH_2_MASK)
#define H264_DEC_ELLVC_CFG_STOP_CH_3_MASK        0x8000000u
#define H264_DEC_ELLVC_CFG_STOP_CH_3_SHIFT       27u
#define H264_DEC_ELLVC_CFG_STOP_CH_3_WIDTH       1u
#define H264_DEC_ELLVC_CFG_STOP_CH_3(x)          (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_CFG_STOP_CH_3_SHIFT))&H264_DEC_ELLVC_CFG_STOP_CH_3_MASK)
/* ELLVC_REF_ADDR_CH Bit Fields */
#define H264_DEC_ELLVC_REF_ADDR_CH_ADDR_MASK     0xFFFFFFFFu
#define H264_DEC_ELLVC_REF_ADDR_CH_ADDR_SHIFT    0u
#define H264_DEC_ELLVC_REF_ADDR_CH_ADDR_WIDTH    32u
#define H264_DEC_ELLVC_REF_ADDR_CH_ADDR(x)       (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_REF_ADDR_CH_ADDR_SHIFT))&H264_DEC_ELLVC_REF_ADDR_CH_ADDR_MASK)
/* ELLVC_DBG_FRAME_CYC_CNT_THR Bit Fields */
#define H264_DEC_ELLVC_DBG_FRAME_CYC_CNT_THR_VAL_MASK 0xFFFFFFFFu
#define H264_DEC_ELLVC_DBG_FRAME_CYC_CNT_THR_VAL_SHIFT 0u
#define H264_DEC_ELLVC_DBG_FRAME_CYC_CNT_THR_VAL_WIDTH 32u
#define H264_DEC_ELLVC_DBG_FRAME_CYC_CNT_THR_VAL(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_DBG_FRAME_CYC_CNT_THR_VAL_SHIFT))&H264_DEC_ELLVC_DBG_FRAME_CYC_CNT_THR_VAL_MASK)
/* ELLVC_STATE Bit Fields */
#define H264_DEC_ELLVC_STATE_CH_0_MASK           0x1Fu
#define H264_DEC_ELLVC_STATE_CH_0_SHIFT          0u
#define H264_DEC_ELLVC_STATE_CH_0_WIDTH          5u
#define H264_DEC_ELLVC_STATE_CH_0(x)             (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_STATE_CH_0_SHIFT))&H264_DEC_ELLVC_STATE_CH_0_MASK)
#define H264_DEC_ELLVC_STATE_CH_1_MASK           0x1F00u
#define H264_DEC_ELLVC_STATE_CH_1_SHIFT          8u
#define H264_DEC_ELLVC_STATE_CH_1_WIDTH          5u
#define H264_DEC_ELLVC_STATE_CH_1(x)             (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_STATE_CH_1_SHIFT))&H264_DEC_ELLVC_STATE_CH_1_MASK)
#define H264_DEC_ELLVC_STATE_CH_2_MASK           0x1F0000u
#define H264_DEC_ELLVC_STATE_CH_2_SHIFT          16u
#define H264_DEC_ELLVC_STATE_CH_2_WIDTH          5u
#define H264_DEC_ELLVC_STATE_CH_2(x)             (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_STATE_CH_2_SHIFT))&H264_DEC_ELLVC_STATE_CH_2_MASK)
#define H264_DEC_ELLVC_STATE_CH_3_MASK           0x1F000000u
#define H264_DEC_ELLVC_STATE_CH_3_SHIFT          24u
#define H264_DEC_ELLVC_STATE_CH_3_WIDTH          5u
#define H264_DEC_ELLVC_STATE_CH_3(x)             (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_STATE_CH_3_SHIFT))&H264_DEC_ELLVC_STATE_CH_3_MASK)
/* ELLVC_PIC_BIT Bit Fields */
#define H264_DEC_ELLVC_PIC_BIT_CH_0_MASK         0x3u
#define H264_DEC_ELLVC_PIC_BIT_CH_0_SHIFT        0u
#define H264_DEC_ELLVC_PIC_BIT_CH_0_WIDTH        2u
#define H264_DEC_ELLVC_PIC_BIT_CH_0(x)           (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_BIT_CH_0_SHIFT))&H264_DEC_ELLVC_PIC_BIT_CH_0_MASK)
#define H264_DEC_ELLVC_PIC_BIT_CH_1_MASK         0x300u
#define H264_DEC_ELLVC_PIC_BIT_CH_1_SHIFT        8u
#define H264_DEC_ELLVC_PIC_BIT_CH_1_WIDTH        2u
#define H264_DEC_ELLVC_PIC_BIT_CH_1(x)           (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_BIT_CH_1_SHIFT))&H264_DEC_ELLVC_PIC_BIT_CH_1_MASK)
#define H264_DEC_ELLVC_PIC_BIT_CH_2_MASK         0x30000u
#define H264_DEC_ELLVC_PIC_BIT_CH_2_SHIFT        16u
#define H264_DEC_ELLVC_PIC_BIT_CH_2_WIDTH        2u
#define H264_DEC_ELLVC_PIC_BIT_CH_2(x)           (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_BIT_CH_2_SHIFT))&H264_DEC_ELLVC_PIC_BIT_CH_2_MASK)
#define H264_DEC_ELLVC_PIC_BIT_CH_3_MASK         0x3000000u
#define H264_DEC_ELLVC_PIC_BIT_CH_3_SHIFT        24u
#define H264_DEC_ELLVC_PIC_BIT_CH_3_WIDTH        2u
#define H264_DEC_ELLVC_PIC_BIT_CH_3(x)           (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_BIT_CH_3_SHIFT))&H264_DEC_ELLVC_PIC_BIT_CH_3_MASK)
/* ELLVC_PIC_WIDTH_IN_MBS Bit Fields */
#define H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_0_MASK 0xFFu
#define H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_0_SHIFT 0u
#define H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_0_WIDTH 8u
#define H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_0(x)  (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_0_SHIFT))&H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_0_MASK)
#define H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_1_MASK 0xFF00u
#define H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_1_SHIFT 8u
#define H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_1_WIDTH 8u
#define H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_1(x)  (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_1_SHIFT))&H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_1_MASK)
#define H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_2_MASK 0xFF0000u
#define H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_2_SHIFT 16u
#define H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_2_WIDTH 8u
#define H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_2(x)  (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_2_SHIFT))&H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_2_MASK)
#define H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_3_MASK 0xFF000000u
#define H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_3_SHIFT 24u
#define H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_3_WIDTH 8u
#define H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_3(x)  (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_3_SHIFT))&H264_DEC_ELLVC_PIC_WIDTH_IN_MBS_CH_3_MASK)
/* ELLVC_PIC_HEIGHT_IN_MBS Bit Fields */
#define H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_0_MASK 0xFFu
#define H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_0_SHIFT 0u
#define H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_0_WIDTH 8u
#define H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_0(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_0_SHIFT))&H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_0_MASK)
#define H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_1_MASK 0xFF00u
#define H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_1_SHIFT 8u
#define H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_1_WIDTH 8u
#define H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_1(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_1_SHIFT))&H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_1_MASK)
#define H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_2_MASK 0xFF0000u
#define H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_2_SHIFT 16u
#define H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_2_WIDTH 8u
#define H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_2(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_2_SHIFT))&H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_2_MASK)
#define H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_3_MASK 0xFF000000u
#define H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_3_SHIFT 24u
#define H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_3_WIDTH 8u
#define H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_3(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_3_SHIFT))&H264_DEC_ELLVC_PIC_HEIGHT_IN_MBS_CH_3_MASK)
/* ELLVC_PIC_CROP_LEFT_CH_01 Bit Fields */
#define H264_DEC_ELLVC_PIC_CROP_LEFT_CH_01_CH_0_MASK 0xFFFu
#define H264_DEC_ELLVC_PIC_CROP_LEFT_CH_01_CH_0_SHIFT 0u
#define H264_DEC_ELLVC_PIC_CROP_LEFT_CH_01_CH_0_WIDTH 12u
#define H264_DEC_ELLVC_PIC_CROP_LEFT_CH_01_CH_0(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_CROP_LEFT_CH_01_CH_0_SHIFT))&H264_DEC_ELLVC_PIC_CROP_LEFT_CH_01_CH_0_MASK)
#define H264_DEC_ELLVC_PIC_CROP_LEFT_CH_01_CH_1_MASK 0xFFF0000u
#define H264_DEC_ELLVC_PIC_CROP_LEFT_CH_01_CH_1_SHIFT 16u
#define H264_DEC_ELLVC_PIC_CROP_LEFT_CH_01_CH_1_WIDTH 12u
#define H264_DEC_ELLVC_PIC_CROP_LEFT_CH_01_CH_1(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_CROP_LEFT_CH_01_CH_1_SHIFT))&H264_DEC_ELLVC_PIC_CROP_LEFT_CH_01_CH_1_MASK)
/* ELLVC_PIC_CROP_LEFT_CH_23 Bit Fields */
#define H264_DEC_ELLVC_PIC_CROP_LEFT_CH_23_CH_2_MASK 0xFFFu
#define H264_DEC_ELLVC_PIC_CROP_LEFT_CH_23_CH_2_SHIFT 0u
#define H264_DEC_ELLVC_PIC_CROP_LEFT_CH_23_CH_2_WIDTH 12u
#define H264_DEC_ELLVC_PIC_CROP_LEFT_CH_23_CH_2(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_CROP_LEFT_CH_23_CH_2_SHIFT))&H264_DEC_ELLVC_PIC_CROP_LEFT_CH_23_CH_2_MASK)
#define H264_DEC_ELLVC_PIC_CROP_LEFT_CH_23_CH_3_MASK 0xFFF0000u
#define H264_DEC_ELLVC_PIC_CROP_LEFT_CH_23_CH_3_SHIFT 16u
#define H264_DEC_ELLVC_PIC_CROP_LEFT_CH_23_CH_3_WIDTH 12u
#define H264_DEC_ELLVC_PIC_CROP_LEFT_CH_23_CH_3(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_CROP_LEFT_CH_23_CH_3_SHIFT))&H264_DEC_ELLVC_PIC_CROP_LEFT_CH_23_CH_3_MASK)
/* ELLVC_PIC_CROP_RGHT_CH_01 Bit Fields */
#define H264_DEC_ELLVC_PIC_CROP_RGHT_CH_01_CH_0_MASK 0xFFFu
#define H264_DEC_ELLVC_PIC_CROP_RGHT_CH_01_CH_0_SHIFT 0u
#define H264_DEC_ELLVC_PIC_CROP_RGHT_CH_01_CH_0_WIDTH 12u
#define H264_DEC_ELLVC_PIC_CROP_RGHT_CH_01_CH_0(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_CROP_RGHT_CH_01_CH_0_SHIFT))&H264_DEC_ELLVC_PIC_CROP_RGHT_CH_01_CH_0_MASK)
#define H264_DEC_ELLVC_PIC_CROP_RGHT_CH_01_CH_1_MASK 0xFFF0000u
#define H264_DEC_ELLVC_PIC_CROP_RGHT_CH_01_CH_1_SHIFT 16u
#define H264_DEC_ELLVC_PIC_CROP_RGHT_CH_01_CH_1_WIDTH 12u
#define H264_DEC_ELLVC_PIC_CROP_RGHT_CH_01_CH_1(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_CROP_RGHT_CH_01_CH_1_SHIFT))&H264_DEC_ELLVC_PIC_CROP_RGHT_CH_01_CH_1_MASK)
/* ELLVC_PIC_CROP_RGHT_CH_23 Bit Fields */
#define H264_DEC_ELLVC_PIC_CROP_RGHT_CH_23_CH_2_MASK 0xFFFu
#define H264_DEC_ELLVC_PIC_CROP_RGHT_CH_23_CH_2_SHIFT 0u
#define H264_DEC_ELLVC_PIC_CROP_RGHT_CH_23_CH_2_WIDTH 12u
#define H264_DEC_ELLVC_PIC_CROP_RGHT_CH_23_CH_2(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_CROP_RGHT_CH_23_CH_2_SHIFT))&H264_DEC_ELLVC_PIC_CROP_RGHT_CH_23_CH_2_MASK)
#define H264_DEC_ELLVC_PIC_CROP_RGHT_CH_23_CH_3_MASK 0xFFF0000u
#define H264_DEC_ELLVC_PIC_CROP_RGHT_CH_23_CH_3_SHIFT 16u
#define H264_DEC_ELLVC_PIC_CROP_RGHT_CH_23_CH_3_WIDTH 12u
#define H264_DEC_ELLVC_PIC_CROP_RGHT_CH_23_CH_3(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_CROP_RGHT_CH_23_CH_3_SHIFT))&H264_DEC_ELLVC_PIC_CROP_RGHT_CH_23_CH_3_MASK)
/* ELLVC_PIC_CROP_TOP_CH_01 Bit Fields */
#define H264_DEC_ELLVC_PIC_CROP_TOP_CH_01_CH_0_MASK 0xFFFu
#define H264_DEC_ELLVC_PIC_CROP_TOP_CH_01_CH_0_SHIFT 0u
#define H264_DEC_ELLVC_PIC_CROP_TOP_CH_01_CH_0_WIDTH 12u
#define H264_DEC_ELLVC_PIC_CROP_TOP_CH_01_CH_0(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_CROP_TOP_CH_01_CH_0_SHIFT))&H264_DEC_ELLVC_PIC_CROP_TOP_CH_01_CH_0_MASK)
#define H264_DEC_ELLVC_PIC_CROP_TOP_CH_01_CH_1_MASK 0xFFF0000u
#define H264_DEC_ELLVC_PIC_CROP_TOP_CH_01_CH_1_SHIFT 16u
#define H264_DEC_ELLVC_PIC_CROP_TOP_CH_01_CH_1_WIDTH 12u
#define H264_DEC_ELLVC_PIC_CROP_TOP_CH_01_CH_1(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_CROP_TOP_CH_01_CH_1_SHIFT))&H264_DEC_ELLVC_PIC_CROP_TOP_CH_01_CH_1_MASK)
/* ELLVC_PIC_CROP_TOP_CH_23 Bit Fields */
#define H264_DEC_ELLVC_PIC_CROP_TOP_CH_23_CH_2_MASK 0xFFFu
#define H264_DEC_ELLVC_PIC_CROP_TOP_CH_23_CH_2_SHIFT 0u
#define H264_DEC_ELLVC_PIC_CROP_TOP_CH_23_CH_2_WIDTH 12u
#define H264_DEC_ELLVC_PIC_CROP_TOP_CH_23_CH_2(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_CROP_TOP_CH_23_CH_2_SHIFT))&H264_DEC_ELLVC_PIC_CROP_TOP_CH_23_CH_2_MASK)
#define H264_DEC_ELLVC_PIC_CROP_TOP_CH_23_CH_3_MASK 0xFFF0000u
#define H264_DEC_ELLVC_PIC_CROP_TOP_CH_23_CH_3_SHIFT 16u
#define H264_DEC_ELLVC_PIC_CROP_TOP_CH_23_CH_3_WIDTH 12u
#define H264_DEC_ELLVC_PIC_CROP_TOP_CH_23_CH_3(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_CROP_TOP_CH_23_CH_3_SHIFT))&H264_DEC_ELLVC_PIC_CROP_TOP_CH_23_CH_3_MASK)
/* ELLVC_PIC_CROP_BTTM_CH_01 Bit Fields */
#define H264_DEC_ELLVC_PIC_CROP_BTTM_CH_01_CH_0_MASK 0xFFFu
#define H264_DEC_ELLVC_PIC_CROP_BTTM_CH_01_CH_0_SHIFT 0u
#define H264_DEC_ELLVC_PIC_CROP_BTTM_CH_01_CH_0_WIDTH 12u
#define H264_DEC_ELLVC_PIC_CROP_BTTM_CH_01_CH_0(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_CROP_BTTM_CH_01_CH_0_SHIFT))&H264_DEC_ELLVC_PIC_CROP_BTTM_CH_01_CH_0_MASK)
#define H264_DEC_ELLVC_PIC_CROP_BTTM_CH_01_CH_1_MASK 0xFFF0000u
#define H264_DEC_ELLVC_PIC_CROP_BTTM_CH_01_CH_1_SHIFT 16u
#define H264_DEC_ELLVC_PIC_CROP_BTTM_CH_01_CH_1_WIDTH 12u
#define H264_DEC_ELLVC_PIC_CROP_BTTM_CH_01_CH_1(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_CROP_BTTM_CH_01_CH_1_SHIFT))&H264_DEC_ELLVC_PIC_CROP_BTTM_CH_01_CH_1_MASK)
/* ELLVC_PIC_CROP_BTTM_CH_23 Bit Fields */
#define H264_DEC_ELLVC_PIC_CROP_BTTM_CH_23_CH_2_MASK 0xFFFu
#define H264_DEC_ELLVC_PIC_CROP_BTTM_CH_23_CH_2_SHIFT 0u
#define H264_DEC_ELLVC_PIC_CROP_BTTM_CH_23_CH_2_WIDTH 12u
#define H264_DEC_ELLVC_PIC_CROP_BTTM_CH_23_CH_2(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_CROP_BTTM_CH_23_CH_2_SHIFT))&H264_DEC_ELLVC_PIC_CROP_BTTM_CH_23_CH_2_MASK)
#define H264_DEC_ELLVC_PIC_CROP_BTTM_CH_23_CH_3_MASK 0xFFF0000u
#define H264_DEC_ELLVC_PIC_CROP_BTTM_CH_23_CH_3_SHIFT 16u
#define H264_DEC_ELLVC_PIC_CROP_BTTM_CH_23_CH_3_WIDTH 12u
#define H264_DEC_ELLVC_PIC_CROP_BTTM_CH_23_CH_3(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_CROP_BTTM_CH_23_CH_3_SHIFT))&H264_DEC_ELLVC_PIC_CROP_BTTM_CH_23_CH_3_MASK)
/* ELLVC_PIC_PARAM_VLD Bit Fields */
#define H264_DEC_ELLVC_PIC_PARAM_VLD_CH_0_MASK   0x1u
#define H264_DEC_ELLVC_PIC_PARAM_VLD_CH_0_SHIFT  0u
#define H264_DEC_ELLVC_PIC_PARAM_VLD_CH_0_WIDTH  1u
#define H264_DEC_ELLVC_PIC_PARAM_VLD_CH_0(x)     (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_PARAM_VLD_CH_0_SHIFT))&H264_DEC_ELLVC_PIC_PARAM_VLD_CH_0_MASK)
#define H264_DEC_ELLVC_PIC_PARAM_VLD_CH_1_MASK   0x100u
#define H264_DEC_ELLVC_PIC_PARAM_VLD_CH_1_SHIFT  8u
#define H264_DEC_ELLVC_PIC_PARAM_VLD_CH_1_WIDTH  1u
#define H264_DEC_ELLVC_PIC_PARAM_VLD_CH_1(x)     (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_PARAM_VLD_CH_1_SHIFT))&H264_DEC_ELLVC_PIC_PARAM_VLD_CH_1_MASK)
#define H264_DEC_ELLVC_PIC_PARAM_VLD_CH_2_MASK   0x10000u
#define H264_DEC_ELLVC_PIC_PARAM_VLD_CH_2_SHIFT  16u
#define H264_DEC_ELLVC_PIC_PARAM_VLD_CH_2_WIDTH  1u
#define H264_DEC_ELLVC_PIC_PARAM_VLD_CH_2(x)     (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_PARAM_VLD_CH_2_SHIFT))&H264_DEC_ELLVC_PIC_PARAM_VLD_CH_2_MASK)
#define H264_DEC_ELLVC_PIC_PARAM_VLD_CH_3_MASK   0x1000000u
#define H264_DEC_ELLVC_PIC_PARAM_VLD_CH_3_SHIFT  24u
#define H264_DEC_ELLVC_PIC_PARAM_VLD_CH_3_WIDTH  1u
#define H264_DEC_ELLVC_PIC_PARAM_VLD_CH_3(x)     (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_PIC_PARAM_VLD_CH_3_SHIFT))&H264_DEC_ELLVC_PIC_PARAM_VLD_CH_3_MASK)
/* ELLVC_POC_VALUE Bit Fields */
#define H264_DEC_ELLVC_POC_VALUE_CH_0_MASK       0xFFFFFFFFu
#define H264_DEC_ELLVC_POC_VALUE_CH_0_SHIFT      0u
#define H264_DEC_ELLVC_POC_VALUE_CH_0_WIDTH      32u
#define H264_DEC_ELLVC_POC_VALUE_CH_0(x)         (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_POC_VALUE_CH_0_SHIFT))&H264_DEC_ELLVC_POC_VALUE_CH_0_MASK)
/* ELLVC_STAT Bit Fields */
#define H264_DEC_ELLVC_STAT_MAX_NUM_REF_FRM_CH_0_MASK 0x1Fu
#define H264_DEC_ELLVC_STAT_MAX_NUM_REF_FRM_CH_0_SHIFT 0u
#define H264_DEC_ELLVC_STAT_MAX_NUM_REF_FRM_CH_0_WIDTH 5u
#define H264_DEC_ELLVC_STAT_MAX_NUM_REF_FRM_CH_0(x) (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_STAT_MAX_NUM_REF_FRM_CH_0_SHIFT))&H264_DEC_ELLVC_STAT_MAX_NUM_REF_FRM_CH_0_MASK)
#define H264_DEC_ELLVC_STAT_NEW_POC_CYC_CH_0_MASK 0x20000u
#define H264_DEC_ELLVC_STAT_NEW_POC_CYC_CH_0_SHIFT 17u
#define H264_DEC_ELLVC_STAT_NEW_POC_CYC_CH_0_WIDTH 1u
#define H264_DEC_ELLVC_STAT_NEW_POC_CYC_CH_0(x)  (((uint32_t)(((uint32_t)(x))<<H264_DEC_ELLVC_STAT_NEW_POC_CYC_CH_0_SHIFT))&H264_DEC_ELLVC_STAT_NEW_POC_CYC_CH_0_MASK)

/*!
 * @}
 */ /* end of group H264_DEC_Register_Masks */


/*!
 * @}
 */ /* end of group H264_DEC_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- H264_ENC Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup H264_ENC_Peripheral_Access_Layer H264_ENC Peripheral Access Layer
 * @{
 */


/** H264_ENC - Size of Registers Arrays */
#define H264_ENC_ULLVC_QP_INC_THLD_COUNT         3u
#define H264_ENC_ULLVC_QP_DEC_THLD_COUNT         3u

/** H264_ENC - Register Layout Typedef */
typedef struct {
  __IO uint32_t MCR;                               /**< Module Configuration Register, offset: 0x0 */
       uint8_t RESERVED_0[12];
  __IO uint32_t VCR;                               /**< Video Configuration Register, offset: 0x10 */
  __IO uint32_t VI_ALARM;                          /**< Video Input Alarm Register, offset: 0x14 */
  __IO uint32_t FETCH_MBRW;                        /**< Fetch Macroblock Row Register, offset: 0x18 */
       uint8_t RESERVED_1[20];
  __IO uint32_t VI_CBUF_Y_ADDR;                    /**< Video Input Circular Buffer Luma Start Address Register, offset: 0x30 */
  __IO uint32_t VI_CBUF_CB_ADDR;                   /**< Video Input Circular Buffer Cb Start Address Register, offset: 0x34 */
  __IO uint32_t VI_CBUF_CR_ADDR;                   /**< Video Input Circular Buffer Cr Start Address Register, offset: 0x38 */
  __IO uint32_t VI_NRLINES;                        /**< Video Input Number of Lines Register, offset: 0x3C */
  __IO uint32_t RATE_FLOW_CTRL;                    /**< Rate Flow Control Register, offset: 0x40 */
       uint8_t RESERVED_2[60];
  __IO uint32_t OUT_CBUF_START_ADDR;               /**< Output Circular Buffer Start Address Register, offset: 0x80 */
  __IO uint32_t OUT_CBUF_END_ADDR;                 /**< Output Circular Buffer End Address Register, offset: 0x84 */
  __IO uint32_t OUT_CBUF_ALARM_ADDR;               /**< Ouput Circular Buffer Alarm Address Register, offset: 0x88 */
       uint8_t RESERVED_3[4];
  __I  uint32_t OUT_CBUF_CURR_ADDR;                /**< Output Circular Buffer Current Address Register, offset: 0x90 */
  __I  uint32_t OUT_CBUF_VEND_ADDR;                /**< Output Circular Buffer Vend Address Register, offset: 0x94 */
       uint8_t RESERVED_4[108];
  __I  uint32_t LINE_CNTR_STAT;                    /**< Line Counter Status Register, offset: 0x104 */
       uint8_t RESERVED_5[20];
  __IO uint32_t ISR;                               /**< Interrupt Status Register, offset: 0x11C */
  __IO uint32_t IER;                               /**< Interrupt Enable Register, offset: 0x120 */
       uint8_t RESERVED_6[28];
  __IO uint32_t TESTLINE_CFG;                      /**< Testline Configuration Register, offset: 0x140 */
  __IO uint32_t TESTLINE_STRT_LOC;                 /**< Testline Start Location Register, offset: 0x144 */
  __IO uint32_t TESTLINE_LUMA_VAL;                 /**< Testline Luma Value Register, offset: 0x148 */
  __IO uint32_t TESTLINE_CB_VAL;                   /**< Testline Chroma Cb Value Register, offset: 0x14C */
  __IO uint32_t TESTLINE_CR_VAL;                   /**< Testline Chroma Cr Value Register, offset: 0x150 */
       uint8_t RESERVED_7[172];
  __IO uint32_t ULLVC_FRAME_RATE;                  /**< ULLVC Frame Rate Register, offset: 0x200 */
  __IO uint32_t ULLVC_QP_INIT;                     /**< ULLVC Quantization Parameter Initial Register, offset: 0x204 */
  __IO uint32_t ULLVC_QP_RANGE;                    /**< ULLVC Quantization Parameter Range Register, offset: 0x208 */
  __IO uint32_t ULLVC_BITS_PER_MB_ROW;             /**< ULLVC Bits Per Macroblock Row Register, offset: 0x20C */
  __IO uint32_t ULLVC_QP_FALLBACK_LIMIT;           /**< ULLVC Fallback QP Limit Register, offset: 0x210 */
  __IO uint32_t ULLVC_QP_INC;                      /**< ULLVC Increment QP Register, offset: 0x214 */
  __IO uint32_t ULLVC_QP_INC_THLD[H264_ENC_ULLVC_QP_INC_THLD_COUNT]; /**< ULLVC Increment Threshold Register, array offset: 0x218, array step: 0x4 */
  __IO uint32_t ULLVC_QP_DEC;                      /**< ULLVC Decrement QP Register, offset: 0x224 */
  __IO uint32_t ULLVC_QP_DEC_THLD[H264_ENC_ULLVC_QP_DEC_THLD_COUNT]; /**< ULLVC Decrement Threshold Register, array offset: 0x228, array step: 0x4 */
  __IO uint32_t ULLVC_WAIT_FRAMES;                 /**< ULLVC Wait Frames Register, offset: 0x234 */
  __IO uint32_t ULLVC_DISABLE_DBF;                 /**< ULLVC Disable DBF Register, offset: 0x238 */
       uint8_t RESERVED_8[72];
  __I  uint32_t ULLVC_BITRATE_STREAM;              /**< ULLVC Bitrate Stream Register, offset: 0x284 */
} H264_ENC_Type, *H264_ENC_MemMapPtr;

 /** Number of instances of the H264_ENC module. */
#define H264_ENC_INSTANCE_COUNT                  (1u)


/* H264_ENC - Peripheral instance base addresses */
/** Peripheral H264_ENC base address */
#define H264_ENC_BASE                            (0x400DC000u)
/** Peripheral H264_ENC base pointer */
#define H264_ENC                                 ((H264_ENC_Type *)H264_ENC_BASE)
/** Array initializer of H264_ENC peripheral base addresses */
#define H264_ENC_BASE_ADDRS                      { H264_ENC_BASE }
/** Array initializer of H264_ENC peripheral base pointers */
#define H264_ENC_BASE_PTRS                       { H264_ENC }

/* ----------------------------------------------------------------------------
   -- H264_ENC Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup H264_ENC_Register_Masks H264_ENC Register Masks
 * @{
 */

/* MCR Bit Fields */
#define H264_ENC_MCR_SW_RESET_MASK               0x1u
#define H264_ENC_MCR_SW_RESET_SHIFT              0u
#define H264_ENC_MCR_SW_RESET_WIDTH              1u
#define H264_ENC_MCR_SW_RESET(x)                 (((uint32_t)(((uint32_t)(x))<<H264_ENC_MCR_SW_RESET_SHIFT))&H264_ENC_MCR_SW_RESET_MASK)
#define H264_ENC_MCR_VI_BUF_RESTART_MASK         0x10u
#define H264_ENC_MCR_VI_BUF_RESTART_SHIFT        4u
#define H264_ENC_MCR_VI_BUF_RESTART_WIDTH        1u
#define H264_ENC_MCR_VI_BUF_RESTART(x)           (((uint32_t)(((uint32_t)(x))<<H264_ENC_MCR_VI_BUF_RESTART_SHIFT))&H264_ENC_MCR_VI_BUF_RESTART_MASK)
#define H264_ENC_MCR_OUT_BUF_RESTART_MASK        0x20u
#define H264_ENC_MCR_OUT_BUF_RESTART_SHIFT       5u
#define H264_ENC_MCR_OUT_BUF_RESTART_WIDTH       1u
#define H264_ENC_MCR_OUT_BUF_RESTART(x)          (((uint32_t)(((uint32_t)(x))<<H264_ENC_MCR_OUT_BUF_RESTART_SHIFT))&H264_ENC_MCR_OUT_BUF_RESTART_MASK)
#define H264_ENC_MCR_VI_BIT_WIDTH_MASK           0x300u
#define H264_ENC_MCR_VI_BIT_WIDTH_SHIFT          8u
#define H264_ENC_MCR_VI_BIT_WIDTH_WIDTH          2u
#define H264_ENC_MCR_VI_BIT_WIDTH(x)             (((uint32_t)(((uint32_t)(x))<<H264_ENC_MCR_VI_BIT_WIDTH_SHIFT))&H264_ENC_MCR_VI_BIT_WIDTH_MASK)
#define H264_ENC_MCR_VI_COLOR_FMT_MASK           0x1000u
#define H264_ENC_MCR_VI_COLOR_FMT_SHIFT          12u
#define H264_ENC_MCR_VI_COLOR_FMT_WIDTH          1u
#define H264_ENC_MCR_VI_COLOR_FMT(x)             (((uint32_t)(((uint32_t)(x))<<H264_ENC_MCR_VI_COLOR_FMT_SHIFT))&H264_ENC_MCR_VI_COLOR_FMT_MASK)
#define H264_ENC_MCR_FETCH_TRIG_MASK             0x10000u
#define H264_ENC_MCR_FETCH_TRIG_SHIFT            16u
#define H264_ENC_MCR_FETCH_TRIG_WIDTH            1u
#define H264_ENC_MCR_FETCH_TRIG(x)               (((uint32_t)(((uint32_t)(x))<<H264_ENC_MCR_FETCH_TRIG_SHIFT))&H264_ENC_MCR_FETCH_TRIG_MASK)
#define H264_ENC_MCR_DATA_MODE_CHR_MASK          0x1000000u
#define H264_ENC_MCR_DATA_MODE_CHR_SHIFT         24u
#define H264_ENC_MCR_DATA_MODE_CHR_WIDTH         1u
#define H264_ENC_MCR_DATA_MODE_CHR(x)            (((uint32_t)(((uint32_t)(x))<<H264_ENC_MCR_DATA_MODE_CHR_SHIFT))&H264_ENC_MCR_DATA_MODE_CHR_MASK)
/* VCR Bit Fields */
#define H264_ENC_VCR_FR_WIDTH_MASK               0xFFEu
#define H264_ENC_VCR_FR_WIDTH_SHIFT              1u
#define H264_ENC_VCR_FR_WIDTH_WIDTH              11u
#define H264_ENC_VCR_FR_WIDTH(x)                 (((uint32_t)(((uint32_t)(x))<<H264_ENC_VCR_FR_WIDTH_SHIFT))&H264_ENC_VCR_FR_WIDTH_MASK)
#define H264_ENC_VCR_FR_HEIGHT_MASK              0xFFE0000u
#define H264_ENC_VCR_FR_HEIGHT_SHIFT             17u
#define H264_ENC_VCR_FR_HEIGHT_WIDTH             11u
#define H264_ENC_VCR_FR_HEIGHT(x)                (((uint32_t)(((uint32_t)(x))<<H264_ENC_VCR_FR_HEIGHT_SHIFT))&H264_ENC_VCR_FR_HEIGHT_MASK)
/* VI_ALARM Bit Fields */
#define H264_ENC_VI_ALARM_MBRW_THLD_MASK         0xFFu
#define H264_ENC_VI_ALARM_MBRW_THLD_SHIFT        0u
#define H264_ENC_VI_ALARM_MBRW_THLD_WIDTH        8u
#define H264_ENC_VI_ALARM_MBRW_THLD(x)           (((uint32_t)(((uint32_t)(x))<<H264_ENC_VI_ALARM_MBRW_THLD_SHIFT))&H264_ENC_VI_ALARM_MBRW_THLD_MASK)
#define H264_ENC_VI_ALARM_MBRW_ALARM_THLD_MASK   0xFF0000u
#define H264_ENC_VI_ALARM_MBRW_ALARM_THLD_SHIFT  16u
#define H264_ENC_VI_ALARM_MBRW_ALARM_THLD_WIDTH  8u
#define H264_ENC_VI_ALARM_MBRW_ALARM_THLD(x)     (((uint32_t)(((uint32_t)(x))<<H264_ENC_VI_ALARM_MBRW_ALARM_THLD_SHIFT))&H264_ENC_VI_ALARM_MBRW_ALARM_THLD_MASK)
/* FETCH_MBRW Bit Fields */
#define H264_ENC_FETCH_MBRW_FETCH_MASK           0x1u
#define H264_ENC_FETCH_MBRW_FETCH_SHIFT          0u
#define H264_ENC_FETCH_MBRW_FETCH_WIDTH          1u
#define H264_ENC_FETCH_MBRW_FETCH(x)             (((uint32_t)(((uint32_t)(x))<<H264_ENC_FETCH_MBRW_FETCH_SHIFT))&H264_ENC_FETCH_MBRW_FETCH_MASK)
/* VI_CBUF_Y_ADDR Bit Fields */
#define H264_ENC_VI_CBUF_Y_ADDR_ADDR_MASK        0x3FFFF8u
#define H264_ENC_VI_CBUF_Y_ADDR_ADDR_SHIFT       3u
#define H264_ENC_VI_CBUF_Y_ADDR_ADDR_WIDTH       19u
#define H264_ENC_VI_CBUF_Y_ADDR_ADDR(x)          (((uint32_t)(((uint32_t)(x))<<H264_ENC_VI_CBUF_Y_ADDR_ADDR_SHIFT))&H264_ENC_VI_CBUF_Y_ADDR_ADDR_MASK)
/* VI_CBUF_CB_ADDR Bit Fields */
#define H264_ENC_VI_CBUF_CB_ADDR_ADDR_MASK       0x3FFFF8u
#define H264_ENC_VI_CBUF_CB_ADDR_ADDR_SHIFT      3u
#define H264_ENC_VI_CBUF_CB_ADDR_ADDR_WIDTH      19u
#define H264_ENC_VI_CBUF_CB_ADDR_ADDR(x)         (((uint32_t)(((uint32_t)(x))<<H264_ENC_VI_CBUF_CB_ADDR_ADDR_SHIFT))&H264_ENC_VI_CBUF_CB_ADDR_ADDR_MASK)
/* VI_CBUF_CR_ADDR Bit Fields */
#define H264_ENC_VI_CBUF_CR_ADDR_ADDR_MASK       0x3FFFF8u
#define H264_ENC_VI_CBUF_CR_ADDR_ADDR_SHIFT      3u
#define H264_ENC_VI_CBUF_CR_ADDR_ADDR_WIDTH      19u
#define H264_ENC_VI_CBUF_CR_ADDR_ADDR(x)         (((uint32_t)(((uint32_t)(x))<<H264_ENC_VI_CBUF_CR_ADDR_ADDR_SHIFT))&H264_ENC_VI_CBUF_CR_ADDR_ADDR_MASK)
/* VI_NRLINES Bit Fields */
#define H264_ENC_VI_NRLINES_Y_LINES_MASK         0xFEu
#define H264_ENC_VI_NRLINES_Y_LINES_SHIFT        1u
#define H264_ENC_VI_NRLINES_Y_LINES_WIDTH        7u
#define H264_ENC_VI_NRLINES_Y_LINES(x)           (((uint32_t)(((uint32_t)(x))<<H264_ENC_VI_NRLINES_Y_LINES_SHIFT))&H264_ENC_VI_NRLINES_Y_LINES_MASK)
/* RATE_FLOW_CTRL Bit Fields */
#define H264_ENC_RATE_FLOW_CTRL_BANK_STRIDE_LTH_MASK 0x7u
#define H264_ENC_RATE_FLOW_CTRL_BANK_STRIDE_LTH_SHIFT 0u
#define H264_ENC_RATE_FLOW_CTRL_BANK_STRIDE_LTH_WIDTH 3u
#define H264_ENC_RATE_FLOW_CTRL_BANK_STRIDE_LTH(x) (((uint32_t)(((uint32_t)(x))<<H264_ENC_RATE_FLOW_CTRL_BANK_STRIDE_LTH_SHIFT))&H264_ENC_RATE_FLOW_CTRL_BANK_STRIDE_LTH_MASK)
#define H264_ENC_RATE_FLOW_CTRL_EN_MASK          0x80000000u
#define H264_ENC_RATE_FLOW_CTRL_EN_SHIFT         31u
#define H264_ENC_RATE_FLOW_CTRL_EN_WIDTH         1u
#define H264_ENC_RATE_FLOW_CTRL_EN(x)            (((uint32_t)(((uint32_t)(x))<<H264_ENC_RATE_FLOW_CTRL_EN_SHIFT))&H264_ENC_RATE_FLOW_CTRL_EN_MASK)
/* OUT_CBUF_START_ADDR Bit Fields */
#define H264_ENC_OUT_CBUF_START_ADDR_ADDR_MASK   0x3FFFF8u
#define H264_ENC_OUT_CBUF_START_ADDR_ADDR_SHIFT  3u
#define H264_ENC_OUT_CBUF_START_ADDR_ADDR_WIDTH  19u
#define H264_ENC_OUT_CBUF_START_ADDR_ADDR(x)     (((uint32_t)(((uint32_t)(x))<<H264_ENC_OUT_CBUF_START_ADDR_ADDR_SHIFT))&H264_ENC_OUT_CBUF_START_ADDR_ADDR_MASK)
/* OUT_CBUF_END_ADDR Bit Fields */
#define H264_ENC_OUT_CBUF_END_ADDR_ADDR_MASK     0x3FFFF8u
#define H264_ENC_OUT_CBUF_END_ADDR_ADDR_SHIFT    3u
#define H264_ENC_OUT_CBUF_END_ADDR_ADDR_WIDTH    19u
#define H264_ENC_OUT_CBUF_END_ADDR_ADDR(x)       (((uint32_t)(((uint32_t)(x))<<H264_ENC_OUT_CBUF_END_ADDR_ADDR_SHIFT))&H264_ENC_OUT_CBUF_END_ADDR_ADDR_MASK)
/* OUT_CBUF_ALARM_ADDR Bit Fields */
#define H264_ENC_OUT_CBUF_ALARM_ADDR_ADDR_MASK   0x3FFFF8u
#define H264_ENC_OUT_CBUF_ALARM_ADDR_ADDR_SHIFT  3u
#define H264_ENC_OUT_CBUF_ALARM_ADDR_ADDR_WIDTH  19u
#define H264_ENC_OUT_CBUF_ALARM_ADDR_ADDR(x)     (((uint32_t)(((uint32_t)(x))<<H264_ENC_OUT_CBUF_ALARM_ADDR_ADDR_SHIFT))&H264_ENC_OUT_CBUF_ALARM_ADDR_ADDR_MASK)
#define H264_ENC_OUT_CBUF_ALARM_ADDR_WRAP_MASK   0x400000u
#define H264_ENC_OUT_CBUF_ALARM_ADDR_WRAP_SHIFT  22u
#define H264_ENC_OUT_CBUF_ALARM_ADDR_WRAP_WIDTH  1u
#define H264_ENC_OUT_CBUF_ALARM_ADDR_WRAP(x)     (((uint32_t)(((uint32_t)(x))<<H264_ENC_OUT_CBUF_ALARM_ADDR_WRAP_SHIFT))&H264_ENC_OUT_CBUF_ALARM_ADDR_WRAP_MASK)
/* OUT_CBUF_CURR_ADDR Bit Fields */
#define H264_ENC_OUT_CBUF_CURR_ADDR_ADDR_MASK    0x3FFFF8u
#define H264_ENC_OUT_CBUF_CURR_ADDR_ADDR_SHIFT   3u
#define H264_ENC_OUT_CBUF_CURR_ADDR_ADDR_WIDTH   19u
#define H264_ENC_OUT_CBUF_CURR_ADDR_ADDR(x)      (((uint32_t)(((uint32_t)(x))<<H264_ENC_OUT_CBUF_CURR_ADDR_ADDR_SHIFT))&H264_ENC_OUT_CBUF_CURR_ADDR_ADDR_MASK)
#define H264_ENC_OUT_CBUF_CURR_ADDR_WRAP_MASK    0x400000u
#define H264_ENC_OUT_CBUF_CURR_ADDR_WRAP_SHIFT   22u
#define H264_ENC_OUT_CBUF_CURR_ADDR_WRAP_WIDTH   1u
#define H264_ENC_OUT_CBUF_CURR_ADDR_WRAP(x)      (((uint32_t)(((uint32_t)(x))<<H264_ENC_OUT_CBUF_CURR_ADDR_WRAP_SHIFT))&H264_ENC_OUT_CBUF_CURR_ADDR_WRAP_MASK)
/* OUT_CBUF_VEND_ADDR Bit Fields */
#define H264_ENC_OUT_CBUF_VEND_ADDR_ADDR_MASK    0x3FFFF8u
#define H264_ENC_OUT_CBUF_VEND_ADDR_ADDR_SHIFT   3u
#define H264_ENC_OUT_CBUF_VEND_ADDR_ADDR_WIDTH   19u
#define H264_ENC_OUT_CBUF_VEND_ADDR_ADDR(x)      (((uint32_t)(((uint32_t)(x))<<H264_ENC_OUT_CBUF_VEND_ADDR_ADDR_SHIFT))&H264_ENC_OUT_CBUF_VEND_ADDR_ADDR_MASK)
/* LINE_CNTR_STAT Bit Fields */
#define H264_ENC_LINE_CNTR_STAT_VAL_MASK         0xFFFu
#define H264_ENC_LINE_CNTR_STAT_VAL_SHIFT        0u
#define H264_ENC_LINE_CNTR_STAT_VAL_WIDTH        12u
#define H264_ENC_LINE_CNTR_STAT_VAL(x)           (((uint32_t)(((uint32_t)(x))<<H264_ENC_LINE_CNTR_STAT_VAL_SHIFT))&H264_ENC_LINE_CNTR_STAT_VAL_MASK)
/* ISR Bit Fields */
#define H264_ENC_ISR_VI_FRAME_DONE_INT_MASK      0x1u
#define H264_ENC_ISR_VI_FRAME_DONE_INT_SHIFT     0u
#define H264_ENC_ISR_VI_FRAME_DONE_INT_WIDTH     1u
#define H264_ENC_ISR_VI_FRAME_DONE_INT(x)        (((uint32_t)(((uint32_t)(x))<<H264_ENC_ISR_VI_FRAME_DONE_INT_SHIFT))&H264_ENC_ISR_VI_FRAME_DONE_INT_MASK)
#define H264_ENC_ISR_BS_NEW_ROW_INT_MASK         0x2u
#define H264_ENC_ISR_BS_NEW_ROW_INT_SHIFT        1u
#define H264_ENC_ISR_BS_NEW_ROW_INT_WIDTH        1u
#define H264_ENC_ISR_BS_NEW_ROW_INT(x)           (((uint32_t)(((uint32_t)(x))<<H264_ENC_ISR_BS_NEW_ROW_INT_SHIFT))&H264_ENC_ISR_BS_NEW_ROW_INT_MASK)
#define H264_ENC_ISR_CB_ALARM_INT_MASK           0x4u
#define H264_ENC_ISR_CB_ALARM_INT_SHIFT          2u
#define H264_ENC_ISR_CB_ALARM_INT_WIDTH          1u
#define H264_ENC_ISR_CB_ALARM_INT(x)             (((uint32_t)(((uint32_t)(x))<<H264_ENC_ISR_CB_ALARM_INT_SHIFT))&H264_ENC_ISR_CB_ALARM_INT_MASK)
#define H264_ENC_ISR_VI_LINE_INT_MASK            0x8u
#define H264_ENC_ISR_VI_LINE_INT_SHIFT           3u
#define H264_ENC_ISR_VI_LINE_INT_WIDTH           1u
#define H264_ENC_ISR_VI_LINE_INT(x)              (((uint32_t)(((uint32_t)(x))<<H264_ENC_ISR_VI_LINE_INT_SHIFT))&H264_ENC_ISR_VI_LINE_INT_MASK)
#define H264_ENC_ISR_VI_LCOUNT_ALARM_INT_MASK    0x10u
#define H264_ENC_ISR_VI_LCOUNT_ALARM_INT_SHIFT   4u
#define H264_ENC_ISR_VI_LCOUNT_ALARM_INT_WIDTH   1u
#define H264_ENC_ISR_VI_LCOUNT_ALARM_INT(x)      (((uint32_t)(((uint32_t)(x))<<H264_ENC_ISR_VI_LCOUNT_ALARM_INT_SHIFT))&H264_ENC_ISR_VI_LCOUNT_ALARM_INT_MASK)
#define H264_ENC_ISR_ENC_OOS_INT_MASK            0x20u
#define H264_ENC_ISR_ENC_OOS_INT_SHIFT           5u
#define H264_ENC_ISR_ENC_OOS_INT_WIDTH           1u
#define H264_ENC_ISR_ENC_OOS_INT(x)              (((uint32_t)(((uint32_t)(x))<<H264_ENC_ISR_ENC_OOS_INT_SHIFT))&H264_ENC_ISR_ENC_OOS_INT_MASK)
#define H264_ENC_ISR_MBRW_DONE_INT_MASK          0x40u
#define H264_ENC_ISR_MBRW_DONE_INT_SHIFT         6u
#define H264_ENC_ISR_MBRW_DONE_INT_WIDTH         1u
#define H264_ENC_ISR_MBRW_DONE_INT(x)            (((uint32_t)(((uint32_t)(x))<<H264_ENC_ISR_MBRW_DONE_INT_SHIFT))&H264_ENC_ISR_MBRW_DONE_INT_MASK)
#define H264_ENC_ISR_MBRW_FF_OVF_INT_MASK        0x80u
#define H264_ENC_ISR_MBRW_FF_OVF_INT_SHIFT       7u
#define H264_ENC_ISR_MBRW_FF_OVF_INT_WIDTH       1u
#define H264_ENC_ISR_MBRW_FF_OVF_INT(x)          (((uint32_t)(((uint32_t)(x))<<H264_ENC_ISR_MBRW_FF_OVF_INT_SHIFT))&H264_ENC_ISR_MBRW_FF_OVF_INT_MASK)
#define H264_ENC_ISR_BS_FRAME_DONE_INT_MASK      0x100u
#define H264_ENC_ISR_BS_FRAME_DONE_INT_SHIFT     8u
#define H264_ENC_ISR_BS_FRAME_DONE_INT_WIDTH     1u
#define H264_ENC_ISR_BS_FRAME_DONE_INT(x)        (((uint32_t)(((uint32_t)(x))<<H264_ENC_ISR_BS_FRAME_DONE_INT_SHIFT))&H264_ENC_ISR_BS_FRAME_DONE_INT_MASK)
#define H264_ENC_ISR_OUT_CBUF_OVF_INT_MASK       0x200u
#define H264_ENC_ISR_OUT_CBUF_OVF_INT_SHIFT      9u
#define H264_ENC_ISR_OUT_CBUF_OVF_INT_WIDTH      1u
#define H264_ENC_ISR_OUT_CBUF_OVF_INT(x)         (((uint32_t)(((uint32_t)(x))<<H264_ENC_ISR_OUT_CBUF_OVF_INT_SHIFT))&H264_ENC_ISR_OUT_CBUF_OVF_INT_MASK)
#define H264_ENC_ISR_BR_TOO_HIGH_INT_MASK        0x400u
#define H264_ENC_ISR_BR_TOO_HIGH_INT_SHIFT       10u
#define H264_ENC_ISR_BR_TOO_HIGH_INT_WIDTH       1u
#define H264_ENC_ISR_BR_TOO_HIGH_INT(x)          (((uint32_t)(((uint32_t)(x))<<H264_ENC_ISR_BR_TOO_HIGH_INT_SHIFT))&H264_ENC_ISR_BR_TOO_HIGH_INT_MASK)
#define H264_ENC_ISR_BS_FIFO_ALMOST_FULL_INT_MASK 0x800u
#define H264_ENC_ISR_BS_FIFO_ALMOST_FULL_INT_SHIFT 11u
#define H264_ENC_ISR_BS_FIFO_ALMOST_FULL_INT_WIDTH 1u
#define H264_ENC_ISR_BS_FIFO_ALMOST_FULL_INT(x)  (((uint32_t)(((uint32_t)(x))<<H264_ENC_ISR_BS_FIFO_ALMOST_FULL_INT_SHIFT))&H264_ENC_ISR_BS_FIFO_ALMOST_FULL_INT_MASK)
#define H264_ENC_ISR_BS_FIFO_FULL_INT_MASK       0x1000u
#define H264_ENC_ISR_BS_FIFO_FULL_INT_SHIFT      12u
#define H264_ENC_ISR_BS_FIFO_FULL_INT_WIDTH      1u
#define H264_ENC_ISR_BS_FIFO_FULL_INT(x)         (((uint32_t)(((uint32_t)(x))<<H264_ENC_ISR_BS_FIFO_FULL_INT_SHIFT))&H264_ENC_ISR_BS_FIFO_FULL_INT_MASK)
/* IER Bit Fields */
#define H264_ENC_IER_VI_FRAME_DONE_INT_EN_MASK   0x1u
#define H264_ENC_IER_VI_FRAME_DONE_INT_EN_SHIFT  0u
#define H264_ENC_IER_VI_FRAME_DONE_INT_EN_WIDTH  1u
#define H264_ENC_IER_VI_FRAME_DONE_INT_EN(x)     (((uint32_t)(((uint32_t)(x))<<H264_ENC_IER_VI_FRAME_DONE_INT_EN_SHIFT))&H264_ENC_IER_VI_FRAME_DONE_INT_EN_MASK)
#define H264_ENC_IER_BS_NEW_ROW_INT_EN_MASK      0x2u
#define H264_ENC_IER_BS_NEW_ROW_INT_EN_SHIFT     1u
#define H264_ENC_IER_BS_NEW_ROW_INT_EN_WIDTH     1u
#define H264_ENC_IER_BS_NEW_ROW_INT_EN(x)        (((uint32_t)(((uint32_t)(x))<<H264_ENC_IER_BS_NEW_ROW_INT_EN_SHIFT))&H264_ENC_IER_BS_NEW_ROW_INT_EN_MASK)
#define H264_ENC_IER_CB_ALARM_INT_EN_MASK        0x4u
#define H264_ENC_IER_CB_ALARM_INT_EN_SHIFT       2u
#define H264_ENC_IER_CB_ALARM_INT_EN_WIDTH       1u
#define H264_ENC_IER_CB_ALARM_INT_EN(x)          (((uint32_t)(((uint32_t)(x))<<H264_ENC_IER_CB_ALARM_INT_EN_SHIFT))&H264_ENC_IER_CB_ALARM_INT_EN_MASK)
#define H264_ENC_IER_VI_LINE_INT_EN_MASK         0x8u
#define H264_ENC_IER_VI_LINE_INT_EN_SHIFT        3u
#define H264_ENC_IER_VI_LINE_INT_EN_WIDTH        1u
#define H264_ENC_IER_VI_LINE_INT_EN(x)           (((uint32_t)(((uint32_t)(x))<<H264_ENC_IER_VI_LINE_INT_EN_SHIFT))&H264_ENC_IER_VI_LINE_INT_EN_MASK)
#define H264_ENC_IER_VI_LCOUNT_ALARM_INT_EN_MASK 0x10u
#define H264_ENC_IER_VI_LCOUNT_ALARM_INT_EN_SHIFT 4u
#define H264_ENC_IER_VI_LCOUNT_ALARM_INT_EN_WIDTH 1u
#define H264_ENC_IER_VI_LCOUNT_ALARM_INT_EN(x)   (((uint32_t)(((uint32_t)(x))<<H264_ENC_IER_VI_LCOUNT_ALARM_INT_EN_SHIFT))&H264_ENC_IER_VI_LCOUNT_ALARM_INT_EN_MASK)
#define H264_ENC_IER_ENC_OOS_INT_EN_MASK         0x20u
#define H264_ENC_IER_ENC_OOS_INT_EN_SHIFT        5u
#define H264_ENC_IER_ENC_OOS_INT_EN_WIDTH        1u
#define H264_ENC_IER_ENC_OOS_INT_EN(x)           (((uint32_t)(((uint32_t)(x))<<H264_ENC_IER_ENC_OOS_INT_EN_SHIFT))&H264_ENC_IER_ENC_OOS_INT_EN_MASK)
#define H264_ENC_IER_MBRW_DONE_INT_EN_MASK       0x40u
#define H264_ENC_IER_MBRW_DONE_INT_EN_SHIFT      6u
#define H264_ENC_IER_MBRW_DONE_INT_EN_WIDTH      1u
#define H264_ENC_IER_MBRW_DONE_INT_EN(x)         (((uint32_t)(((uint32_t)(x))<<H264_ENC_IER_MBRW_DONE_INT_EN_SHIFT))&H264_ENC_IER_MBRW_DONE_INT_EN_MASK)
#define H264_ENC_IER_MBRW_FF_OVF_EN_MASK         0x80u
#define H264_ENC_IER_MBRW_FF_OVF_EN_SHIFT        7u
#define H264_ENC_IER_MBRW_FF_OVF_EN_WIDTH        1u
#define H264_ENC_IER_MBRW_FF_OVF_EN(x)           (((uint32_t)(((uint32_t)(x))<<H264_ENC_IER_MBRW_FF_OVF_EN_SHIFT))&H264_ENC_IER_MBRW_FF_OVF_EN_MASK)
#define H264_ENC_IER_BS_FRAME_DONE_INT_EN_MASK   0x100u
#define H264_ENC_IER_BS_FRAME_DONE_INT_EN_SHIFT  8u
#define H264_ENC_IER_BS_FRAME_DONE_INT_EN_WIDTH  1u
#define H264_ENC_IER_BS_FRAME_DONE_INT_EN(x)     (((uint32_t)(((uint32_t)(x))<<H264_ENC_IER_BS_FRAME_DONE_INT_EN_SHIFT))&H264_ENC_IER_BS_FRAME_DONE_INT_EN_MASK)
#define H264_ENC_IER_OUT_CBUF_OVF_INT_EN_MASK    0x200u
#define H264_ENC_IER_OUT_CBUF_OVF_INT_EN_SHIFT   9u
#define H264_ENC_IER_OUT_CBUF_OVF_INT_EN_WIDTH   1u
#define H264_ENC_IER_OUT_CBUF_OVF_INT_EN(x)      (((uint32_t)(((uint32_t)(x))<<H264_ENC_IER_OUT_CBUF_OVF_INT_EN_SHIFT))&H264_ENC_IER_OUT_CBUF_OVF_INT_EN_MASK)
#define H264_ENC_IER_BR_TOO_HIGH_INT_EN_MASK     0x400u
#define H264_ENC_IER_BR_TOO_HIGH_INT_EN_SHIFT    10u
#define H264_ENC_IER_BR_TOO_HIGH_INT_EN_WIDTH    1u
#define H264_ENC_IER_BR_TOO_HIGH_INT_EN(x)       (((uint32_t)(((uint32_t)(x))<<H264_ENC_IER_BR_TOO_HIGH_INT_EN_SHIFT))&H264_ENC_IER_BR_TOO_HIGH_INT_EN_MASK)
#define H264_ENC_IER_BS_FIFO_ALMOST_FULL_INT_EN_MASK 0x800u
#define H264_ENC_IER_BS_FIFO_ALMOST_FULL_INT_EN_SHIFT 11u
#define H264_ENC_IER_BS_FIFO_ALMOST_FULL_INT_EN_WIDTH 1u
#define H264_ENC_IER_BS_FIFO_ALMOST_FULL_INT_EN(x) (((uint32_t)(((uint32_t)(x))<<H264_ENC_IER_BS_FIFO_ALMOST_FULL_INT_EN_SHIFT))&H264_ENC_IER_BS_FIFO_ALMOST_FULL_INT_EN_MASK)
#define H264_ENC_IER_BS_FIFO_FULL_INT_EN_MASK    0x1000u
#define H264_ENC_IER_BS_FIFO_FULL_INT_EN_SHIFT   12u
#define H264_ENC_IER_BS_FIFO_FULL_INT_EN_WIDTH   1u
#define H264_ENC_IER_BS_FIFO_FULL_INT_EN(x)      (((uint32_t)(((uint32_t)(x))<<H264_ENC_IER_BS_FIFO_FULL_INT_EN_SHIFT))&H264_ENC_IER_BS_FIFO_FULL_INT_EN_MASK)
/* TESTLINE_CFG Bit Fields */
#define H264_ENC_TESTLINE_CFG_NR_W_MASK          0x3Fu
#define H264_ENC_TESTLINE_CFG_NR_W_SHIFT         0u
#define H264_ENC_TESTLINE_CFG_NR_W_WIDTH         6u
#define H264_ENC_TESTLINE_CFG_NR_W(x)            (((uint32_t)(((uint32_t)(x))<<H264_ENC_TESTLINE_CFG_NR_W_SHIFT))&H264_ENC_TESTLINE_CFG_NR_W_MASK)
#define H264_ENC_TESTLINE_CFG_EN_MASK            0x80000000u
#define H264_ENC_TESTLINE_CFG_EN_SHIFT           31u
#define H264_ENC_TESTLINE_CFG_EN_WIDTH           1u
#define H264_ENC_TESTLINE_CFG_EN(x)              (((uint32_t)(((uint32_t)(x))<<H264_ENC_TESTLINE_CFG_EN_SHIFT))&H264_ENC_TESTLINE_CFG_EN_MASK)
/* TESTLINE_STRT_LOC Bit Fields */
#define H264_ENC_TESTLINE_STRT_LOC_STRT_LOC_MASK 0xFFFu
#define H264_ENC_TESTLINE_STRT_LOC_STRT_LOC_SHIFT 0u
#define H264_ENC_TESTLINE_STRT_LOC_STRT_LOC_WIDTH 12u
#define H264_ENC_TESTLINE_STRT_LOC_STRT_LOC(x)   (((uint32_t)(((uint32_t)(x))<<H264_ENC_TESTLINE_STRT_LOC_STRT_LOC_SHIFT))&H264_ENC_TESTLINE_STRT_LOC_STRT_LOC_MASK)
/* TESTLINE_LUMA_VAL Bit Fields */
#define H264_ENC_TESTLINE_LUMA_VAL_VAL_1_MASK    0xFFFu
#define H264_ENC_TESTLINE_LUMA_VAL_VAL_1_SHIFT   0u
#define H264_ENC_TESTLINE_LUMA_VAL_VAL_1_WIDTH   12u
#define H264_ENC_TESTLINE_LUMA_VAL_VAL_1(x)      (((uint32_t)(((uint32_t)(x))<<H264_ENC_TESTLINE_LUMA_VAL_VAL_1_SHIFT))&H264_ENC_TESTLINE_LUMA_VAL_VAL_1_MASK)
#define H264_ENC_TESTLINE_LUMA_VAL_VAL_2_MASK    0xFFF0000u
#define H264_ENC_TESTLINE_LUMA_VAL_VAL_2_SHIFT   16u
#define H264_ENC_TESTLINE_LUMA_VAL_VAL_2_WIDTH   12u
#define H264_ENC_TESTLINE_LUMA_VAL_VAL_2(x)      (((uint32_t)(((uint32_t)(x))<<H264_ENC_TESTLINE_LUMA_VAL_VAL_2_SHIFT))&H264_ENC_TESTLINE_LUMA_VAL_VAL_2_MASK)
/* TESTLINE_CB_VAL Bit Fields */
#define H264_ENC_TESTLINE_CB_VAL_VAL_1_MASK      0xFFFu
#define H264_ENC_TESTLINE_CB_VAL_VAL_1_SHIFT     0u
#define H264_ENC_TESTLINE_CB_VAL_VAL_1_WIDTH     12u
#define H264_ENC_TESTLINE_CB_VAL_VAL_1(x)        (((uint32_t)(((uint32_t)(x))<<H264_ENC_TESTLINE_CB_VAL_VAL_1_SHIFT))&H264_ENC_TESTLINE_CB_VAL_VAL_1_MASK)
#define H264_ENC_TESTLINE_CB_VAL_VAL_2_MASK      0xFFF0000u
#define H264_ENC_TESTLINE_CB_VAL_VAL_2_SHIFT     16u
#define H264_ENC_TESTLINE_CB_VAL_VAL_2_WIDTH     12u
#define H264_ENC_TESTLINE_CB_VAL_VAL_2(x)        (((uint32_t)(((uint32_t)(x))<<H264_ENC_TESTLINE_CB_VAL_VAL_2_SHIFT))&H264_ENC_TESTLINE_CB_VAL_VAL_2_MASK)
/* TESTLINE_CR_VAL Bit Fields */
#define H264_ENC_TESTLINE_CR_VAL_VAL_1_MASK      0xFFFu
#define H264_ENC_TESTLINE_CR_VAL_VAL_1_SHIFT     0u
#define H264_ENC_TESTLINE_CR_VAL_VAL_1_WIDTH     12u
#define H264_ENC_TESTLINE_CR_VAL_VAL_1(x)        (((uint32_t)(((uint32_t)(x))<<H264_ENC_TESTLINE_CR_VAL_VAL_1_SHIFT))&H264_ENC_TESTLINE_CR_VAL_VAL_1_MASK)
#define H264_ENC_TESTLINE_CR_VAL_VAL_2_MASK      0xFFF0000u
#define H264_ENC_TESTLINE_CR_VAL_VAL_2_SHIFT     16u
#define H264_ENC_TESTLINE_CR_VAL_VAL_2_WIDTH     12u
#define H264_ENC_TESTLINE_CR_VAL_VAL_2(x)        (((uint32_t)(((uint32_t)(x))<<H264_ENC_TESTLINE_CR_VAL_VAL_2_SHIFT))&H264_ENC_TESTLINE_CR_VAL_VAL_2_MASK)
/* ULLVC_FRAME_RATE Bit Fields */
#define H264_ENC_ULLVC_FRAME_RATE_VAL_MASK       0xFFu
#define H264_ENC_ULLVC_FRAME_RATE_VAL_SHIFT      0u
#define H264_ENC_ULLVC_FRAME_RATE_VAL_WIDTH      8u
#define H264_ENC_ULLVC_FRAME_RATE_VAL(x)         (((uint32_t)(((uint32_t)(x))<<H264_ENC_ULLVC_FRAME_RATE_VAL_SHIFT))&H264_ENC_ULLVC_FRAME_RATE_VAL_MASK)
/* ULLVC_QP_INIT Bit Fields */
#define H264_ENC_ULLVC_QP_INIT_INITIAL_MASK      0x7Fu
#define H264_ENC_ULLVC_QP_INIT_INITIAL_SHIFT     0u
#define H264_ENC_ULLVC_QP_INIT_INITIAL_WIDTH     7u
#define H264_ENC_ULLVC_QP_INIT_INITIAL(x)        (((uint32_t)(((uint32_t)(x))<<H264_ENC_ULLVC_QP_INIT_INITIAL_SHIFT))&H264_ENC_ULLVC_QP_INIT_INITIAL_MASK)
/* ULLVC_QP_RANGE Bit Fields */
#define H264_ENC_ULLVC_QP_RANGE_MIN_MASK         0x7Fu
#define H264_ENC_ULLVC_QP_RANGE_MIN_SHIFT        0u
#define H264_ENC_ULLVC_QP_RANGE_MIN_WIDTH        7u
#define H264_ENC_ULLVC_QP_RANGE_MIN(x)           (((uint32_t)(((uint32_t)(x))<<H264_ENC_ULLVC_QP_RANGE_MIN_SHIFT))&H264_ENC_ULLVC_QP_RANGE_MIN_MASK)
#define H264_ENC_ULLVC_QP_RANGE_MAX_MASK         0x7F0000u
#define H264_ENC_ULLVC_QP_RANGE_MAX_SHIFT        16u
#define H264_ENC_ULLVC_QP_RANGE_MAX_WIDTH        7u
#define H264_ENC_ULLVC_QP_RANGE_MAX(x)           (((uint32_t)(((uint32_t)(x))<<H264_ENC_ULLVC_QP_RANGE_MAX_SHIFT))&H264_ENC_ULLVC_QP_RANGE_MAX_MASK)
/* ULLVC_BITS_PER_MB_ROW Bit Fields */
#define H264_ENC_ULLVC_BITS_PER_MB_ROW_BITRATE_MASK 0x7FFFFFFFu
#define H264_ENC_ULLVC_BITS_PER_MB_ROW_BITRATE_SHIFT 0u
#define H264_ENC_ULLVC_BITS_PER_MB_ROW_BITRATE_WIDTH 31u
#define H264_ENC_ULLVC_BITS_PER_MB_ROW_BITRATE(x) (((uint32_t)(((uint32_t)(x))<<H264_ENC_ULLVC_BITS_PER_MB_ROW_BITRATE_SHIFT))&H264_ENC_ULLVC_BITS_PER_MB_ROW_BITRATE_MASK)
/* ULLVC_QP_FALLBACK_LIMIT Bit Fields */
#define H264_ENC_ULLVC_QP_FALLBACK_LIMIT_LIMIT_MASK 0xFFu
#define H264_ENC_ULLVC_QP_FALLBACK_LIMIT_LIMIT_SHIFT 0u
#define H264_ENC_ULLVC_QP_FALLBACK_LIMIT_LIMIT_WIDTH 8u
#define H264_ENC_ULLVC_QP_FALLBACK_LIMIT_LIMIT(x) (((uint32_t)(((uint32_t)(x))<<H264_ENC_ULLVC_QP_FALLBACK_LIMIT_LIMIT_SHIFT))&H264_ENC_ULLVC_QP_FALLBACK_LIMIT_LIMIT_MASK)
/* ULLVC_QP_INC Bit Fields */
#define H264_ENC_ULLVC_QP_INC_OFFSET_MASK        0x7Fu
#define H264_ENC_ULLVC_QP_INC_OFFSET_SHIFT       0u
#define H264_ENC_ULLVC_QP_INC_OFFSET_WIDTH       7u
#define H264_ENC_ULLVC_QP_INC_OFFSET(x)          (((uint32_t)(((uint32_t)(x))<<H264_ENC_ULLVC_QP_INC_OFFSET_SHIFT))&H264_ENC_ULLVC_QP_INC_OFFSET_MASK)
#define H264_ENC_ULLVC_QP_INC_DIFF_0_MASK        0x7F00u
#define H264_ENC_ULLVC_QP_INC_DIFF_0_SHIFT       8u
#define H264_ENC_ULLVC_QP_INC_DIFF_0_WIDTH       7u
#define H264_ENC_ULLVC_QP_INC_DIFF_0(x)          (((uint32_t)(((uint32_t)(x))<<H264_ENC_ULLVC_QP_INC_DIFF_0_SHIFT))&H264_ENC_ULLVC_QP_INC_DIFF_0_MASK)
#define H264_ENC_ULLVC_QP_INC_DIFF_1_MASK        0x7F0000u
#define H264_ENC_ULLVC_QP_INC_DIFF_1_SHIFT       16u
#define H264_ENC_ULLVC_QP_INC_DIFF_1_WIDTH       7u
#define H264_ENC_ULLVC_QP_INC_DIFF_1(x)          (((uint32_t)(((uint32_t)(x))<<H264_ENC_ULLVC_QP_INC_DIFF_1_SHIFT))&H264_ENC_ULLVC_QP_INC_DIFF_1_MASK)
#define H264_ENC_ULLVC_QP_INC_DIFF_2_MASK        0x7F000000u
#define H264_ENC_ULLVC_QP_INC_DIFF_2_SHIFT       24u
#define H264_ENC_ULLVC_QP_INC_DIFF_2_WIDTH       7u
#define H264_ENC_ULLVC_QP_INC_DIFF_2(x)          (((uint32_t)(((uint32_t)(x))<<H264_ENC_ULLVC_QP_INC_DIFF_2_SHIFT))&H264_ENC_ULLVC_QP_INC_DIFF_2_MASK)
/* ULLVC_QP_INC_THLD Bit Fields */
#define H264_ENC_ULLVC_QP_INC_THLD_THLD_MASK     0x7FFFFFFFu
#define H264_ENC_ULLVC_QP_INC_THLD_THLD_SHIFT    0u
#define H264_ENC_ULLVC_QP_INC_THLD_THLD_WIDTH    31u
#define H264_ENC_ULLVC_QP_INC_THLD_THLD(x)       (((uint32_t)(((uint32_t)(x))<<H264_ENC_ULLVC_QP_INC_THLD_THLD_SHIFT))&H264_ENC_ULLVC_QP_INC_THLD_THLD_MASK)
/* ULLVC_QP_DEC Bit Fields */
#define H264_ENC_ULLVC_QP_DEC_OFFSET_MASK        0x7Fu
#define H264_ENC_ULLVC_QP_DEC_OFFSET_SHIFT       0u
#define H264_ENC_ULLVC_QP_DEC_OFFSET_WIDTH       7u
#define H264_ENC_ULLVC_QP_DEC_OFFSET(x)          (((uint32_t)(((uint32_t)(x))<<H264_ENC_ULLVC_QP_DEC_OFFSET_SHIFT))&H264_ENC_ULLVC_QP_DEC_OFFSET_MASK)
#define H264_ENC_ULLVC_QP_DEC_DIFF_0_MASK        0x7F00u
#define H264_ENC_ULLVC_QP_DEC_DIFF_0_SHIFT       8u
#define H264_ENC_ULLVC_QP_DEC_DIFF_0_WIDTH       7u
#define H264_ENC_ULLVC_QP_DEC_DIFF_0(x)          (((uint32_t)(((uint32_t)(x))<<H264_ENC_ULLVC_QP_DEC_DIFF_0_SHIFT))&H264_ENC_ULLVC_QP_DEC_DIFF_0_MASK)
#define H264_ENC_ULLVC_QP_DEC_DIFF_1_MASK        0x7F0000u
#define H264_ENC_ULLVC_QP_DEC_DIFF_1_SHIFT       16u
#define H264_ENC_ULLVC_QP_DEC_DIFF_1_WIDTH       7u
#define H264_ENC_ULLVC_QP_DEC_DIFF_1(x)          (((uint32_t)(((uint32_t)(x))<<H264_ENC_ULLVC_QP_DEC_DIFF_1_SHIFT))&H264_ENC_ULLVC_QP_DEC_DIFF_1_MASK)
#define H264_ENC_ULLVC_QP_DEC_DIFF_2_MASK        0x7F000000u
#define H264_ENC_ULLVC_QP_DEC_DIFF_2_SHIFT       24u
#define H264_ENC_ULLVC_QP_DEC_DIFF_2_WIDTH       7u
#define H264_ENC_ULLVC_QP_DEC_DIFF_2(x)          (((uint32_t)(((uint32_t)(x))<<H264_ENC_ULLVC_QP_DEC_DIFF_2_SHIFT))&H264_ENC_ULLVC_QP_DEC_DIFF_2_MASK)
/* ULLVC_QP_DEC_THLD Bit Fields */
#define H264_ENC_ULLVC_QP_DEC_THLD_THLD_MASK     0x7FFFFFFFu
#define H264_ENC_ULLVC_QP_DEC_THLD_THLD_SHIFT    0u
#define H264_ENC_ULLVC_QP_DEC_THLD_THLD_WIDTH    31u
#define H264_ENC_ULLVC_QP_DEC_THLD_THLD(x)       (((uint32_t)(((uint32_t)(x))<<H264_ENC_ULLVC_QP_DEC_THLD_THLD_SHIFT))&H264_ENC_ULLVC_QP_DEC_THLD_THLD_MASK)
/* ULLVC_WAIT_FRAMES Bit Fields */
#define H264_ENC_ULLVC_WAIT_FRAMES_WAIT_FRAMES_MASK 0x3Fu
#define H264_ENC_ULLVC_WAIT_FRAMES_WAIT_FRAMES_SHIFT 0u
#define H264_ENC_ULLVC_WAIT_FRAMES_WAIT_FRAMES_WIDTH 6u
#define H264_ENC_ULLVC_WAIT_FRAMES_WAIT_FRAMES(x) (((uint32_t)(((uint32_t)(x))<<H264_ENC_ULLVC_WAIT_FRAMES_WAIT_FRAMES_SHIFT))&H264_ENC_ULLVC_WAIT_FRAMES_WAIT_FRAMES_MASK)
/* ULLVC_DISABLE_DBF Bit Fields */
#define H264_ENC_ULLVC_DISABLE_DBF_DISABLE_MASK  0x1u
#define H264_ENC_ULLVC_DISABLE_DBF_DISABLE_SHIFT 0u
#define H264_ENC_ULLVC_DISABLE_DBF_DISABLE_WIDTH 1u
#define H264_ENC_ULLVC_DISABLE_DBF_DISABLE(x)    (((uint32_t)(((uint32_t)(x))<<H264_ENC_ULLVC_DISABLE_DBF_DISABLE_SHIFT))&H264_ENC_ULLVC_DISABLE_DBF_DISABLE_MASK)
/* ULLVC_BITRATE_STREAM Bit Fields */
#define H264_ENC_ULLVC_BITRATE_STREAM_VAL_MASK   0xFFFFFFFFu
#define H264_ENC_ULLVC_BITRATE_STREAM_VAL_SHIFT  0u
#define H264_ENC_ULLVC_BITRATE_STREAM_VAL_WIDTH  32u
#define H264_ENC_ULLVC_BITRATE_STREAM_VAL(x)     (((uint32_t)(((uint32_t)(x))<<H264_ENC_ULLVC_BITRATE_STREAM_VAL_SHIFT))&H264_ENC_ULLVC_BITRATE_STREAM_VAL_MASK)

/*!
 * @}
 */ /* end of group H264_ENC_Register_Masks */


/*!
 * @}
 */ /* end of group H264_ENC_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- HPSMI Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup HPSMI_Peripheral_Access_Layer HPSMI Peripheral Access Layer
 * @{
 */


/** HPSMI - Size of Registers Arrays */

/** HPSMI - Register Layout Typedef */
typedef struct {
  __IO uint32_t GBL_CTRL;                          /**< HPSMI Global Control Register, offset: 0x0 */
       uint8_t RESERVED_0[12];
  __IO uint32_t MPU_SEL0;                          /**< HPSMI MPU Address Select 0 Register, offset: 0x10 */
  __IO uint32_t MPU_SEL1;                          /**< HPSMI MPU Address Select 1 Register, offset: 0x14 */
  __IO uint32_t MPU_SEL2;                          /**< HPSMI MPU Address Select 2 Register, offset: 0x18 */
  __IO uint32_t MPU_SEL3;                          /**< HPSMI MPU Address Select 3 Register, offset: 0x1C */
  __IO uint32_t PRIO_CHANGE;                       /**< HPSMI Stream DMA Master Priority Change Register, offset: 0x20 */
  __I  uint32_t PROFILE_LSB2AXICYCCNTR_0;          /**< HPSMI LSB2AXI Master 0 Cycle counter Register, offset: 0x24 */
  __I  uint32_t PROFILE_LSB2AXICYCCNTR_1;          /**< HPSMI LSB2AXI Master 1 Cycle Counter Value Register, offset: 0x28 */
  __IO uint32_t PROFILE_CYCCNTR_OVF;               /**< HPSMI Profile Cycle Counter Overflow Register, offset: 0x2C */
  __IO uint32_t PROFILE_STOP;                      /**< HPSMI AXI Profile Stop Register, offset: 0x30 */
       uint8_t RESERVED_1[28];
  __IO uint32_t INTR_EN0;                          /**< HPSMI Interrupt Enable 0 Register, offset: 0x50 */
  __IO uint32_t INTR_EN1;                          /**< HPSMI Interrupt Enable 1 Register, offset: 0x54 */
  __IO uint32_t PROFILE_START;                     /**< HPSMI AXI Profile Start Register, offset: 0x58 */
  __IO uint32_t PROFILE_RESET;                     /**< HPSMI AXI Profile Reset Register, offset: 0x5C */
  __IO uint32_t PROFILE_FREEZE;                    /**< HPSMI AXI Profile Freeze Register, offset: 0x60 */
  __IO uint32_t PROFILE_DECVAL;                    /**< HPSMI AXI Profile Decrement Value Register, offset: 0x64 */
  __IO uint32_t PROFILE_DECVAL_CNTRNUM;            /**< HPSMI AXI Profile Decrement Counter Information Register, offset: 0x68 */
  __I  uint32_t PROFILE_AXIRDCNTRVAL_0;            /**< HPSMI AXI Master 0 Read Port Profile Counter Value Register, offset: 0x6C */
  __I  uint32_t PROFILE_AXIWRCNTRVAL_0;            /**< HPSMI AXI Master 0 Write Port Profile Counter Value Register, offset: 0x70 */
  __I  uint32_t PROFILE_AXIRDCNTRVAL_1;            /**< HPSMI AXI Master 1 Read Port Profile Counter Value Register, offset: 0x74 */
  __I  uint32_t PROFILE_AXIWRCNTRVAL_1;            /**< HPSMI AXI Master 1 Write Port Profile Counter Value Register, offset: 0x78 */
  __I  uint32_t PROFILE_AXIRDCNTRVAL_2;            /**< HPSMI AXI Master 2 Read Port Profile Counter Value Register, offset: 0x7C */
  __I  uint32_t PROFILE_AXIWRCNTRVAL_2;            /**< HPSMI AXI Master 2 Write Port Profile Counter Value Register, offset: 0x80 */
  __I  uint32_t PROFILE_AXIRDCNTRVAL_3;            /**< HPSMI AXI Master 3 Read Port Profile Counter Value Register, offset: 0x84 */
  __I  uint32_t PROFILE_AXIWRCNTRVAL_3;            /**< HPSMI AXI Master 3 Write Port Profile Counter Value Register, offset: 0x88 */
  __I  uint32_t PROFILE_AXIRDCNTRVAL_4;            /**< HPSMI AXI Master 4 Read Port Profile Counter Value Register, offset: 0x8C */
  __I  uint32_t PROFILE_AXIWRCNTRVAL_4;            /**< HPSMI AXI Master 4 Write Port Profile Counter Value Register, offset: 0x90 */
  __I  uint32_t PROFILE_AXIRDCNTRVAL_5;            /**< HPSMI AXI Master 5 Read Port Profile Counter Value Register, offset: 0x94 */
  __I  uint32_t PROFILE_AXIWRCNTRVAL_5;            /**< HPSMI AXI Master 5 Write Port Profile Counter Value Register, offset: 0x98 */
  __I  uint32_t PROFILE_AXIRDCNTRVAL_6;            /**< HPSMI AXI Master 6 Read Port Profile Counter Value Register, offset: 0x9C */
  __I  uint32_t PROFILE_AXIWRCNTRVAL_6;            /**< HPSMI AXI Master 6 Write Port Profile Counter Value Register, offset: 0xA0 */
  __I  uint32_t PROFILE_LSB2AXIRDCNTRVAL_0;        /**< HPSMI LSB2AXI Master 0 Read Port Profile Counter Value Register, offset: 0xA4 */
  __I  uint32_t PROFILE_LSB2AXIWRCNTRVAL_0;        /**< HPSMI LSB2AXI Master 0 Write Port Profile Counter Value Register, offset: 0xA8 */
  __I  uint32_t PROFILE_LSB2AXIRDCNTRVAL_1;        /**< HPSMI LSB2AXI Master 1 Read Port Profile Counter Value Register, offset: 0xAC */
  __I  uint32_t PROFILE_LSB2AXIWRCNTRVAL_1;        /**< HPSMI LSB2AXI Master 1 Write Port Profile Counter Value Register, offset: 0xB0 */
  __I  uint32_t PROFILE_AXICYCCNTRVAL_0;           /**< HPSMI AXI Master 0 Cycle Counter Value Register, offset: 0xB4 */
  __I  uint32_t PROFILE_AXICYCCNTRVAL_1;           /**< HPSMI AXI Master 1 Cycle Counter Value Register, offset: 0xB8 */
  __I  uint32_t PROFILE_AXICYCCNTRVAL_2;           /**< HPSMI AXI Master 2 Cycle Counter Value Register, offset: 0xBC */
  __I  uint32_t PROFILE_AXICYCCNTRVAL_3;           /**< HPSMI AXI Master 3 Cycle Counter Value Register, offset: 0xC0 */
  __I  uint32_t PROFILE_AXICYCCNTRVAL_4;           /**< HPSMI AXI Master 4 Cycle Counter Value Register, offset: 0xC4 */
  __I  uint32_t PROFILE_AXICYCCNTRVAL_5;           /**< HPSMI AXI Master 5 Cycle Counter Value Register, offset: 0xC8 */
  __I  uint32_t PROFILE_AXICYCCNTRVAL_6;           /**< HPSMI AXI Master 6 Cycle Counter Value Register, offset: 0xCC */
  __I  uint32_t RD_PL_LVL_31_0;                    /**< HPSMI Stream DMA Read Pipeline Level 31_0 Register, offset: 0xD0 */
  __I  uint32_t RD_PL_LVL_63_32;                   /**< HPSMI Stream DMA Read Pipeline Level 63_32 Register, offset: 0xD4 */
       uint8_t RESERVED_2[8];
  __I  uint32_t WR_PL_LVL_31_0;                    /**< HPSMI Stream DMA Write Pipeline Level 31_0 Register, offset: 0xE0 */
  __I  uint32_t WR_PL_LVL_63_32;                   /**< HPSMI Stream DMA Write Pipeline Level 63_32 Register, offset: 0xE4 */
       uint8_t RESERVED_3[24];
  __IO uint32_t DSRD_ADDRESS_ERROR0;               /**< HPSMI Stream DMA Read Address Error 0 Register, offset: 0x100 */
  __IO uint32_t DSRD_ADDRESS_ERROR1;               /**< HPSMI Stream DMA Read Address Error 1 Register, offset: 0x104 */
  __IO uint32_t DSWR_ADDRESS_ERROR0;               /**< HPSMI Stream DMA Write Address Error 0 Register, offset: 0x108 */
  __IO uint32_t DSWR_ADDRESS_ERROR1;               /**< HPSMI Stream DMA Write Address Error 1 Register, offset: 0x10C */
       uint8_t RESERVED_4[12];
  __I  uint32_t ADDR_ERROR_ADDRVAL;                /**< HPSMI Addressing Error Address Register, offset: 0x11C */
  __IO uint32_t ADDR_ERROR_INFO;                   /**< HPSMI Addressing Error Information Register, offset: 0x120 */
       uint8_t RESERVED_5[16];
  __IO uint32_t WRMPU_ERROR0;                      /**< HPSMI Stream DMA Write MPU Error 0 Register, offset: 0x134 */
  __IO uint32_t WRMPU_ERROR1;                      /**< HPSMI Stream DMA Write MPU Error 1 Register, offset: 0x138 */
  __I  uint32_t WRMPU_ERROR_ADDR;                  /**< HPSMI MPU Error Address Register, offset: 0x13C */
  __IO uint32_t WRMPU_ERROR_INFO;                  /**< HPSMI MPU Error Information Register, offset: 0x140 */
       uint8_t RESERVED_6[12];
  __IO uint32_t PM_ERROR_1_2;                      /**< , offset: 0x150 */
  __I  uint32_t PM_ERROR_ADDR_1_2;                 /**< , offset: 0x154 */
  __IO uint32_t PM_ERROR_INFO_1_2;                 /**< , offset: 0x158 */
       uint8_t RESERVED_7[36];
  __IO uint32_t ECC_ERR_INJECT;                    /**< HPSMI Error Inject Register, offset: 0x180 */
  __IO uint32_t ECC_ERRCNTR_MASK;                  /**< HPSMI ECC Error Counter Reset Mask, offset: 0x184 */
  __I  uint32_t ECC_SINGLE_ERRCNTR_1_2;            /**< Single Error counter for SEG1_2 Register, offset: 0x188 */
  __I  uint32_t ECC_UNCORR_ERRCNTR_1_2;            /**< Uncorrectable Error Counter for Seg1_2 Register, offset: 0x18C */
       uint8_t RESERVED_8[8];
  __I  uint32_t ECC_ERRADDR_SEG1;                  /**< HPSMI ECC Error Address Segment 1 Register, offset: 0x198 */
  __I  uint32_t ECC_ERRDATA0_SEG1;                 /**< HPSMI ECC Error Data 31:0 for Segment 1 Register, offset: 0x19C */
  __I  uint32_t ECC_ERRDATA1_SEG1;                 /**< HPSMI ECC Error Data 63:32 for Segment 1 Register, offset: 0x1A0 */
  __IO uint32_t ECC_ERRINFO_SEG1;                  /**< HPSMI ECC Error Information Segment 1 Register, offset: 0x1A4 */
  __I  uint32_t ECC_ERRADDR_SEG2;                  /**< HPSMI ECC Error Address Segment 2 Register, offset: 0x1A8 */
  __I  uint32_t ECC_ERRDATA0_SEG2;                 /**< HPSMI ECC Error Data 31:0 for Segment 2 Register, offset: 0x1AC */
  __I  uint32_t ECC_ERRDATA1_SEG2;                 /**< HPSMI ECC Error Data 63:32 for Segment 2 Register, offset: 0x1B0 */
  __IO uint32_t ECC_ERRINFO_SEG2;                  /**< HPSMI ECC Error Information Segment 2 Register, offset: 0x1B4 */
  __IO uint32_t ECC_ERROCCURRED_SEG1_2;            /**< , offset: 0x1B8 */
  __I  uint32_t ECC_ERRADDR_SEG0_EVEN;             /**< HPSMI ECC Error Address for Even Banks of Segment0 Register, offset: 0x1BC */
  __I  uint32_t ECC_ERRADDR_SEG0_ODD;              /**< HPSMI ECC Error Address for Odd Banks of Segment0 Register, offset: 0x1C0 */
  __I  uint32_t ECC_ERRDATA0_SEG0_EVEN;            /**< HPSMI ECC Error Data 31:0 for even banks of Segment0 Register, offset: 0x1C4 */
  __I  uint32_t ECC_ERRDATA1_SEG0_EVEN;            /**< HPSMI ECC Error Data 63:32 for even banks of Segment0 Register, offset: 0x1C8 */
  __I  uint32_t ECC_ERRDATA0_SEG0_ODD;             /**< HPSMI ECC Error Data 31:0 for odd banks of Segment0 Register, offset: 0x1CC */
  __I  uint32_t ECC_ERRDATA1_SEG0_ODD;             /**< HPSMI ECC Error Data 63:32 for odd banks of Segment0 Register, offset: 0x1D0 */
  __IO uint32_t ECC_ERRINFO_SEG0;                  /**< HPSMI ECC Error Information Segment 0 Register, offset: 0x1D4 */
  __IO uint32_t ECC_ERROCCURED_SEG0;               /**< , offset: 0x1D8 */
  __I  uint32_t ECC_SINGLE_ERRCNTR_ODD_EVEN_0;     /**< ECC Single Error Counter for Odd and Even Bank Register, offset: 0x1DC */
  __I  uint32_t ECC_UNCORR_ERRCNTR_ODD_EVEN_0;     /**< ECC Uncorectable Error Conter for Odd and Even Bank Register, offset: 0x1E0 */
  __IO uint32_t PM_ERROR_0;                        /**< , offset: 0x1E4 */
  __I  uint32_t PM_ERROR_ADDR_0;                   /**< , offset: 0x1E8 */
  __IO uint32_t PM_ERROR_INFO_0;                   /**< , offset: 0x1EC */
  __IO uint32_t POWMOD_CTRL_0;                     /**< , offset: 0x1F0 */
  __IO uint32_t POWMOD_CTRL_0_8_11;                /**< , offset: 0x1F4 */
       uint8_t RESERVED_9[8];
  __IO uint32_t POWMOD_CTRL_1_2;                   /**< , offset: 0x200 */
       uint8_t RESERVED_10[8];
  __IO uint32_t QOS_PRIORITY;                      /**< , offset: 0x20C */
       uint8_t RESERVED_11[20];
  __IO uint32_t IPCGE;                             /**< Interconnect Parity Checking Global Enable Register, offset: 0x224 */
  __IO uint32_t IPRCE;                             /**< Interconnect Parity Read Checking Enable Register, offset: 0x228 */
  __IO uint32_t IPWCE;                             /**< , offset: 0x22C */
  __IO uint32_t IPWACE;                            /**< Interconnect Parity Write Address Checking Enable Register, offset: 0x230 */
  __IO uint32_t IPRACE;                            /**< Interconnect Parity Read Address Checking Enable Register, offset: 0x234 */
  __IO uint32_t IPCGIE;                            /**< Interconnect Parity Checking Global Injection Enable Register, offset: 0x238 */
} HPSMI_Type, *HPSMI_MemMapPtr;

 /** Number of instances of the HPSMI module. */
#define HPSMI_INSTANCE_COUNT                     (1u)


/* HPSMI - Peripheral instance base addresses */
/** Peripheral HPSMI base address */
#define HPSMI_BASE                               (0x40068000u)
/** Peripheral HPSMI base pointer */
#define HPSMI                                    ((HPSMI_Type *)HPSMI_BASE)
/** Array initializer of HPSMI peripheral base addresses */
#define HPSMI_BASE_ADDRS                         { HPSMI_BASE }
/** Array initializer of HPSMI peripheral base pointers */
#define HPSMI_BASE_PTRS                          { HPSMI }

/* ----------------------------------------------------------------------------
   -- HPSMI Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup HPSMI_Register_Masks HPSMI Register Masks
 * @{
 */

/* GBL_CTRL Bit Fields */
#define HPSMI_GBL_CTRL_WR_ENABLE_MASK            0x2u
#define HPSMI_GBL_CTRL_WR_ENABLE_SHIFT           1u
#define HPSMI_GBL_CTRL_WR_ENABLE_WIDTH           1u
#define HPSMI_GBL_CTRL_WR_ENABLE(x)              (((uint32_t)(((uint32_t)(x))<<HPSMI_GBL_CTRL_WR_ENABLE_SHIFT))&HPSMI_GBL_CTRL_WR_ENABLE_MASK)
/* MPU_SEL0 Bit Fields */
#define HPSMI_MPU_SEL0_SELVAL_MASK               0xFFFFFFFFu
#define HPSMI_MPU_SEL0_SELVAL_SHIFT              0u
#define HPSMI_MPU_SEL0_SELVAL_WIDTH              32u
#define HPSMI_MPU_SEL0_SELVAL(x)                 (((uint32_t)(((uint32_t)(x))<<HPSMI_MPU_SEL0_SELVAL_SHIFT))&HPSMI_MPU_SEL0_SELVAL_MASK)
/* MPU_SEL1 Bit Fields */
#define HPSMI_MPU_SEL1_SELVAL_MASK               0xFFFFFFFFu
#define HPSMI_MPU_SEL1_SELVAL_SHIFT              0u
#define HPSMI_MPU_SEL1_SELVAL_WIDTH              32u
#define HPSMI_MPU_SEL1_SELVAL(x)                 (((uint32_t)(((uint32_t)(x))<<HPSMI_MPU_SEL1_SELVAL_SHIFT))&HPSMI_MPU_SEL1_SELVAL_MASK)
/* MPU_SEL2 Bit Fields */
#define HPSMI_MPU_SEL2_SELVAL_MASK               0xFFFFFFFFu
#define HPSMI_MPU_SEL2_SELVAL_SHIFT              0u
#define HPSMI_MPU_SEL2_SELVAL_WIDTH              32u
#define HPSMI_MPU_SEL2_SELVAL(x)                 (((uint32_t)(((uint32_t)(x))<<HPSMI_MPU_SEL2_SELVAL_SHIFT))&HPSMI_MPU_SEL2_SELVAL_MASK)
/* MPU_SEL3 Bit Fields */
#define HPSMI_MPU_SEL3_SELVAL_MASK               0xFFFFFFFFu
#define HPSMI_MPU_SEL3_SELVAL_SHIFT              0u
#define HPSMI_MPU_SEL3_SELVAL_WIDTH              32u
#define HPSMI_MPU_SEL3_SELVAL(x)                 (((uint32_t)(((uint32_t)(x))<<HPSMI_MPU_SEL3_SELVAL_SHIFT))&HPSMI_MPU_SEL3_SELVAL_MASK)
/* PRIO_CHANGE Bit Fields */
#define HPSMI_PRIO_CHANGE_SEL_PRIO_MASK          0x7Fu
#define HPSMI_PRIO_CHANGE_SEL_PRIO_SHIFT         0u
#define HPSMI_PRIO_CHANGE_SEL_PRIO_WIDTH         7u
#define HPSMI_PRIO_CHANGE_SEL_PRIO(x)            (((uint32_t)(((uint32_t)(x))<<HPSMI_PRIO_CHANGE_SEL_PRIO_SHIFT))&HPSMI_PRIO_CHANGE_SEL_PRIO_MASK)
#define HPSMI_PRIO_CHANGE_PRIO_MASK_MASK         0x7F00u
#define HPSMI_PRIO_CHANGE_PRIO_MASK_SHIFT        8u
#define HPSMI_PRIO_CHANGE_PRIO_MASK_WIDTH        7u
#define HPSMI_PRIO_CHANGE_PRIO_MASK(x)           (((uint32_t)(((uint32_t)(x))<<HPSMI_PRIO_CHANGE_PRIO_MASK_SHIFT))&HPSMI_PRIO_CHANGE_PRIO_MASK_MASK)
#define HPSMI_PRIO_CHANGE_CNTR_EN_MASK           0x8000u
#define HPSMI_PRIO_CHANGE_CNTR_EN_SHIFT          15u
#define HPSMI_PRIO_CHANGE_CNTR_EN_WIDTH          1u
#define HPSMI_PRIO_CHANGE_CNTR_EN(x)             (((uint32_t)(((uint32_t)(x))<<HPSMI_PRIO_CHANGE_CNTR_EN_SHIFT))&HPSMI_PRIO_CHANGE_CNTR_EN_MASK)
#define HPSMI_PRIO_CHANGE_CNTR_MODE_MASK         0x10000u
#define HPSMI_PRIO_CHANGE_CNTR_MODE_SHIFT        16u
#define HPSMI_PRIO_CHANGE_CNTR_MODE_WIDTH        1u
#define HPSMI_PRIO_CHANGE_CNTR_MODE(x)           (((uint32_t)(((uint32_t)(x))<<HPSMI_PRIO_CHANGE_CNTR_MODE_SHIFT))&HPSMI_PRIO_CHANGE_CNTR_MODE_MASK)
/* PROFILE_LSB2AXICYCCNTR_0 Bit Fields */
#define HPSMI_PROFILE_LSB2AXICYCCNTR_0_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_LSB2AXICYCCNTR_0_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_LSB2AXICYCCNTR_0_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_LSB2AXICYCCNTR_0_CNTRVAL(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_LSB2AXICYCCNTR_0_CNTRVAL_SHIFT))&HPSMI_PROFILE_LSB2AXICYCCNTR_0_CNTRVAL_MASK)
/* PROFILE_LSB2AXICYCCNTR_1 Bit Fields */
#define HPSMI_PROFILE_LSB2AXICYCCNTR_1_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_LSB2AXICYCCNTR_1_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_LSB2AXICYCCNTR_1_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_LSB2AXICYCCNTR_1_CNTRVAL(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_LSB2AXICYCCNTR_1_CNTRVAL_SHIFT))&HPSMI_PROFILE_LSB2AXICYCCNTR_1_CNTRVAL_MASK)
/* PROFILE_CYCCNTR_OVF Bit Fields */
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF0_MASK 0x1u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF0_SHIFT 0u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF0_WIDTH 1u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF0(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF0_SHIFT))&HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF0_MASK)
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF1_MASK 0x2u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF1_SHIFT 1u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF1_WIDTH 1u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF1(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF1_SHIFT))&HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF1_MASK)
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF2_MASK 0x4u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF2_SHIFT 2u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF2_WIDTH 1u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF2(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF2_SHIFT))&HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF2_MASK)
#define HPSMI_PROFILE_CYCCNTR_OVF_AXi_CYCCNTR_OVF3_MASK 0x8u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXi_CYCCNTR_OVF3_SHIFT 3u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXi_CYCCNTR_OVF3_WIDTH 1u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXi_CYCCNTR_OVF3(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_CYCCNTR_OVF_AXi_CYCCNTR_OVF3_SHIFT))&HPSMI_PROFILE_CYCCNTR_OVF_AXi_CYCCNTR_OVF3_MASK)
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF4_MASK 0x10u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF4_SHIFT 4u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF4_WIDTH 1u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF4(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF4_SHIFT))&HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF4_MASK)
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF5_MASK 0x20u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF5_SHIFT 5u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF5_WIDTH 1u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF5(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF5_SHIFT))&HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF5_MASK)
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF6_MASK 0x40u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF6_SHIFT 6u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF6_WIDTH 1u
#define HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF6(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF6_SHIFT))&HPSMI_PROFILE_CYCCNTR_OVF_AXI_CYCCNTR_OVF6_MASK)
#define HPSMI_PROFILE_CYCCNTR_OVF_LSB2AXI_CYCCNTR_OVF0_MASK 0x80u
#define HPSMI_PROFILE_CYCCNTR_OVF_LSB2AXI_CYCCNTR_OVF0_SHIFT 7u
#define HPSMI_PROFILE_CYCCNTR_OVF_LSB2AXI_CYCCNTR_OVF0_WIDTH 1u
#define HPSMI_PROFILE_CYCCNTR_OVF_LSB2AXI_CYCCNTR_OVF0(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_CYCCNTR_OVF_LSB2AXI_CYCCNTR_OVF0_SHIFT))&HPSMI_PROFILE_CYCCNTR_OVF_LSB2AXI_CYCCNTR_OVF0_MASK)
#define HPSMI_PROFILE_CYCCNTR_OVF_LSB2AXI_CYCCNTR_OVF1_MASK 0x100u
#define HPSMI_PROFILE_CYCCNTR_OVF_LSB2AXI_CYCCNTR_OVF1_SHIFT 8u
#define HPSMI_PROFILE_CYCCNTR_OVF_LSB2AXI_CYCCNTR_OVF1_WIDTH 1u
#define HPSMI_PROFILE_CYCCNTR_OVF_LSB2AXI_CYCCNTR_OVF1(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_CYCCNTR_OVF_LSB2AXI_CYCCNTR_OVF1_SHIFT))&HPSMI_PROFILE_CYCCNTR_OVF_LSB2AXI_CYCCNTR_OVF1_MASK)
/* PROFILE_STOP Bit Fields */
#define HPSMI_PROFILE_STOP_WEN_MASK              0x1u
#define HPSMI_PROFILE_STOP_WEN_SHIFT             0u
#define HPSMI_PROFILE_STOP_WEN_WIDTH             1u
#define HPSMI_PROFILE_STOP_WEN(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_STOP_WEN_SHIFT))&HPSMI_PROFILE_STOP_WEN_MASK)
#define HPSMI_PROFILE_STOP_STOP_TOTAL_MASK       0x2u
#define HPSMI_PROFILE_STOP_STOP_TOTAL_SHIFT      1u
#define HPSMI_PROFILE_STOP_STOP_TOTAL_WIDTH      1u
#define HPSMI_PROFILE_STOP_STOP_TOTAL(x)         (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_STOP_STOP_TOTAL_SHIFT))&HPSMI_PROFILE_STOP_STOP_TOTAL_MASK)
#define HPSMI_PROFILE_STOP_STOP_AXIMASTER_MASK   0x7F00u
#define HPSMI_PROFILE_STOP_STOP_AXIMASTER_SHIFT  8u
#define HPSMI_PROFILE_STOP_STOP_AXIMASTER_WIDTH  7u
#define HPSMI_PROFILE_STOP_STOP_AXIMASTER(x)     (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_STOP_STOP_AXIMASTER_SHIFT))&HPSMI_PROFILE_STOP_STOP_AXIMASTER_MASK)
#define HPSMI_PROFILE_STOP_STOP_LSB2AXIMASTER_MASK 0x30000u
#define HPSMI_PROFILE_STOP_STOP_LSB2AXIMASTER_SHIFT 16u
#define HPSMI_PROFILE_STOP_STOP_LSB2AXIMASTER_WIDTH 2u
#define HPSMI_PROFILE_STOP_STOP_LSB2AXIMASTER(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_STOP_STOP_LSB2AXIMASTER_SHIFT))&HPSMI_PROFILE_STOP_STOP_LSB2AXIMASTER_MASK)
/* INTR_EN0 Bit Fields */
#define HPSMI_INTR_EN0_ADDR_ERR_MASK             0x1u
#define HPSMI_INTR_EN0_ADDR_ERR_SHIFT            0u
#define HPSMI_INTR_EN0_ADDR_ERR_WIDTH            1u
#define HPSMI_INTR_EN0_ADDR_ERR(x)               (((uint32_t)(((uint32_t)(x))<<HPSMI_INTR_EN0_ADDR_ERR_SHIFT))&HPSMI_INTR_EN0_ADDR_ERR_MASK)
#define HPSMI_INTR_EN0_MPU_ERR_MASK              0x2u
#define HPSMI_INTR_EN0_MPU_ERR_SHIFT             1u
#define HPSMI_INTR_EN0_MPU_ERR_WIDTH             1u
#define HPSMI_INTR_EN0_MPU_ERR(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_INTR_EN0_MPU_ERR_SHIFT))&HPSMI_INTR_EN0_MPU_ERR_MASK)
#define HPSMI_INTR_EN0_PM_ERR_MASK               0x8u
#define HPSMI_INTR_EN0_PM_ERR_SHIFT              3u
#define HPSMI_INTR_EN0_PM_ERR_WIDTH              1u
#define HPSMI_INTR_EN0_PM_ERR(x)                 (((uint32_t)(((uint32_t)(x))<<HPSMI_INTR_EN0_PM_ERR_SHIFT))&HPSMI_INTR_EN0_PM_ERR_MASK)
#define HPSMI_INTR_EN0_ECC_ERR_SEG0_MASK         0x10u
#define HPSMI_INTR_EN0_ECC_ERR_SEG0_SHIFT        4u
#define HPSMI_INTR_EN0_ECC_ERR_SEG0_WIDTH        1u
#define HPSMI_INTR_EN0_ECC_ERR_SEG0(x)           (((uint32_t)(((uint32_t)(x))<<HPSMI_INTR_EN0_ECC_ERR_SEG0_SHIFT))&HPSMI_INTR_EN0_ECC_ERR_SEG0_MASK)
#define HPSMI_INTR_EN0_ECC_ERR_SEG1_MASK         0x20u
#define HPSMI_INTR_EN0_ECC_ERR_SEG1_SHIFT        5u
#define HPSMI_INTR_EN0_ECC_ERR_SEG1_WIDTH        1u
#define HPSMI_INTR_EN0_ECC_ERR_SEG1(x)           (((uint32_t)(((uint32_t)(x))<<HPSMI_INTR_EN0_ECC_ERR_SEG1_SHIFT))&HPSMI_INTR_EN0_ECC_ERR_SEG1_MASK)
#define HPSMI_INTR_EN0_ECC_ERR_SEG2_MASK         0x40u
#define HPSMI_INTR_EN0_ECC_ERR_SEG2_SHIFT        6u
#define HPSMI_INTR_EN0_ECC_ERR_SEG2_WIDTH        1u
#define HPSMI_INTR_EN0_ECC_ERR_SEG2(x)           (((uint32_t)(((uint32_t)(x))<<HPSMI_INTR_EN0_ECC_ERR_SEG2_SHIFT))&HPSMI_INTR_EN0_ECC_ERR_SEG2_MASK)
/* INTR_EN1 Bit Fields */
#define HPSMI_INTR_EN1_ECC_SINGLEERR_SEG0_OVF_MASK 0x1u
#define HPSMI_INTR_EN1_ECC_SINGLEERR_SEG0_OVF_SHIFT 0u
#define HPSMI_INTR_EN1_ECC_SINGLEERR_SEG0_OVF_WIDTH 1u
#define HPSMI_INTR_EN1_ECC_SINGLEERR_SEG0_OVF(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_INTR_EN1_ECC_SINGLEERR_SEG0_OVF_SHIFT))&HPSMI_INTR_EN1_ECC_SINGLEERR_SEG0_OVF_MASK)
#define HPSMI_INTR_EN1_ECC_SINGLEERR_SEG1_OVF_MASK 0x2u
#define HPSMI_INTR_EN1_ECC_SINGLEERR_SEG1_OVF_SHIFT 1u
#define HPSMI_INTR_EN1_ECC_SINGLEERR_SEG1_OVF_WIDTH 1u
#define HPSMI_INTR_EN1_ECC_SINGLEERR_SEG1_OVF(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_INTR_EN1_ECC_SINGLEERR_SEG1_OVF_SHIFT))&HPSMI_INTR_EN1_ECC_SINGLEERR_SEG1_OVF_MASK)
#define HPSMI_INTR_EN1_ECC_SINGLEERR_SEG2_OVF_MASK 0x4u
#define HPSMI_INTR_EN1_ECC_SINGLEERR_SEG2_OVF_SHIFT 2u
#define HPSMI_INTR_EN1_ECC_SINGLEERR_SEG2_OVF_WIDTH 1u
#define HPSMI_INTR_EN1_ECC_SINGLEERR_SEG2_OVF(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_INTR_EN1_ECC_SINGLEERR_SEG2_OVF_SHIFT))&HPSMI_INTR_EN1_ECC_SINGLEERR_SEG2_OVF_MASK)
#define HPSMI_INTR_EN1_ECC_UNCORRERR_SEG0_OVF_MASK 0x8u
#define HPSMI_INTR_EN1_ECC_UNCORRERR_SEG0_OVF_SHIFT 3u
#define HPSMI_INTR_EN1_ECC_UNCORRERR_SEG0_OVF_WIDTH 1u
#define HPSMI_INTR_EN1_ECC_UNCORRERR_SEG0_OVF(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_INTR_EN1_ECC_UNCORRERR_SEG0_OVF_SHIFT))&HPSMI_INTR_EN1_ECC_UNCORRERR_SEG0_OVF_MASK)
#define HPSMI_INTR_EN1_ECC_UNCORRERR_SEG1_OVF_MASK 0x10u
#define HPSMI_INTR_EN1_ECC_UNCORRERR_SEG1_OVF_SHIFT 4u
#define HPSMI_INTR_EN1_ECC_UNCORRERR_SEG1_OVF_WIDTH 1u
#define HPSMI_INTR_EN1_ECC_UNCORRERR_SEG1_OVF(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_INTR_EN1_ECC_UNCORRERR_SEG1_OVF_SHIFT))&HPSMI_INTR_EN1_ECC_UNCORRERR_SEG1_OVF_MASK)
#define HPSMI_INTR_EN1_ECC_UNCORERR_SEG2_OVF_MASK 0x20u
#define HPSMI_INTR_EN1_ECC_UNCORERR_SEG2_OVF_SHIFT 5u
#define HPSMI_INTR_EN1_ECC_UNCORERR_SEG2_OVF_WIDTH 1u
#define HPSMI_INTR_EN1_ECC_UNCORERR_SEG2_OVF(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_INTR_EN1_ECC_UNCORERR_SEG2_OVF_SHIFT))&HPSMI_INTR_EN1_ECC_UNCORERR_SEG2_OVF_MASK)
/* PROFILE_START Bit Fields */
#define HPSMI_PROFILE_START_WEN_MASK             0x1u
#define HPSMI_PROFILE_START_WEN_SHIFT            0u
#define HPSMI_PROFILE_START_WEN_WIDTH            1u
#define HPSMI_PROFILE_START_WEN(x)               (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_START_WEN_SHIFT))&HPSMI_PROFILE_START_WEN_MASK)
#define HPSMI_PROFILE_START_START_TOTAL_MASK     0x2u
#define HPSMI_PROFILE_START_START_TOTAL_SHIFT    1u
#define HPSMI_PROFILE_START_START_TOTAL_WIDTH    1u
#define HPSMI_PROFILE_START_START_TOTAL(x)       (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_START_START_TOTAL_SHIFT))&HPSMI_PROFILE_START_START_TOTAL_MASK)
#define HPSMI_PROFILE_START_START_AXIMASTER_MASK 0x7F00u
#define HPSMI_PROFILE_START_START_AXIMASTER_SHIFT 8u
#define HPSMI_PROFILE_START_START_AXIMASTER_WIDTH 7u
#define HPSMI_PROFILE_START_START_AXIMASTER(x)   (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_START_START_AXIMASTER_SHIFT))&HPSMI_PROFILE_START_START_AXIMASTER_MASK)
#define HPSMI_PROFILE_START_START_LSB2AXIMASTER_MASK 0x30000u
#define HPSMI_PROFILE_START_START_LSB2AXIMASTER_SHIFT 16u
#define HPSMI_PROFILE_START_START_LSB2AXIMASTER_WIDTH 2u
#define HPSMI_PROFILE_START_START_LSB2AXIMASTER(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_START_START_LSB2AXIMASTER_SHIFT))&HPSMI_PROFILE_START_START_LSB2AXIMASTER_MASK)
/* PROFILE_RESET Bit Fields */
#define HPSMI_PROFILE_RESET_WEN_MASK             0x1u
#define HPSMI_PROFILE_RESET_WEN_SHIFT            0u
#define HPSMI_PROFILE_RESET_WEN_WIDTH            1u
#define HPSMI_PROFILE_RESET_WEN(x)               (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_RESET_WEN_SHIFT))&HPSMI_PROFILE_RESET_WEN_MASK)
#define HPSMI_PROFILE_RESET_RESET_TOTAL_MASK     0x2u
#define HPSMI_PROFILE_RESET_RESET_TOTAL_SHIFT    1u
#define HPSMI_PROFILE_RESET_RESET_TOTAL_WIDTH    1u
#define HPSMI_PROFILE_RESET_RESET_TOTAL(x)       (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_RESET_RESET_TOTAL_SHIFT))&HPSMI_PROFILE_RESET_RESET_TOTAL_MASK)
#define HPSMI_PROFILE_RESET_RESET_AXIMASTER_MASK 0x7F00u
#define HPSMI_PROFILE_RESET_RESET_AXIMASTER_SHIFT 8u
#define HPSMI_PROFILE_RESET_RESET_AXIMASTER_WIDTH 7u
#define HPSMI_PROFILE_RESET_RESET_AXIMASTER(x)   (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_RESET_RESET_AXIMASTER_SHIFT))&HPSMI_PROFILE_RESET_RESET_AXIMASTER_MASK)
#define HPSMI_PROFILE_RESET_RESET_LSB2AXIMASTER_MASK 0x30000u
#define HPSMI_PROFILE_RESET_RESET_LSB2AXIMASTER_SHIFT 16u
#define HPSMI_PROFILE_RESET_RESET_LSB2AXIMASTER_WIDTH 2u
#define HPSMI_PROFILE_RESET_RESET_LSB2AXIMASTER(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_RESET_RESET_LSB2AXIMASTER_SHIFT))&HPSMI_PROFILE_RESET_RESET_LSB2AXIMASTER_MASK)
/* PROFILE_FREEZE Bit Fields */
#define HPSMI_PROFILE_FREEZE_WEN_MASK            0x1u
#define HPSMI_PROFILE_FREEZE_WEN_SHIFT           0u
#define HPSMI_PROFILE_FREEZE_WEN_WIDTH           1u
#define HPSMI_PROFILE_FREEZE_WEN(x)              (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_FREEZE_WEN_SHIFT))&HPSMI_PROFILE_FREEZE_WEN_MASK)
#define HPSMI_PROFILE_FREEZE_FREEZE_TOTAL_MASK   0x2u
#define HPSMI_PROFILE_FREEZE_FREEZE_TOTAL_SHIFT  1u
#define HPSMI_PROFILE_FREEZE_FREEZE_TOTAL_WIDTH  1u
#define HPSMI_PROFILE_FREEZE_FREEZE_TOTAL(x)     (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_FREEZE_FREEZE_TOTAL_SHIFT))&HPSMI_PROFILE_FREEZE_FREEZE_TOTAL_MASK)
#define HPSMI_PROFILE_FREEZE_FREEZE_AXIMASTER_MASK 0x7F00u
#define HPSMI_PROFILE_FREEZE_FREEZE_AXIMASTER_SHIFT 8u
#define HPSMI_PROFILE_FREEZE_FREEZE_AXIMASTER_WIDTH 7u
#define HPSMI_PROFILE_FREEZE_FREEZE_AXIMASTER(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_FREEZE_FREEZE_AXIMASTER_SHIFT))&HPSMI_PROFILE_FREEZE_FREEZE_AXIMASTER_MASK)
#define HPSMI_PROFILE_FREEZE_FREEZE_LSB2AXIMASTER_MASK 0x30000u
#define HPSMI_PROFILE_FREEZE_FREEZE_LSB2AXIMASTER_SHIFT 16u
#define HPSMI_PROFILE_FREEZE_FREEZE_LSB2AXIMASTER_WIDTH 2u
#define HPSMI_PROFILE_FREEZE_FREEZE_LSB2AXIMASTER(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_FREEZE_FREEZE_LSB2AXIMASTER_SHIFT))&HPSMI_PROFILE_FREEZE_FREEZE_LSB2AXIMASTER_MASK)
/* PROFILE_DECVAL Bit Fields */
#define HPSMI_PROFILE_DECVAL_DECVAL_MASK         0xFFFFFFFFu
#define HPSMI_PROFILE_DECVAL_DECVAL_SHIFT        0u
#define HPSMI_PROFILE_DECVAL_DECVAL_WIDTH        32u
#define HPSMI_PROFILE_DECVAL_DECVAL(x)           (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_DECVAL_DECVAL_SHIFT))&HPSMI_PROFILE_DECVAL_DECVAL_MASK)
/* PROFILE_DECVAL_CNTRNUM Bit Fields */
#define HPSMI_PROFILE_DECVAL_CNTRNUM_CNTRNUM_MASK 0xFu
#define HPSMI_PROFILE_DECVAL_CNTRNUM_CNTRNUM_SHIFT 0u
#define HPSMI_PROFILE_DECVAL_CNTRNUM_CNTRNUM_WIDTH 4u
#define HPSMI_PROFILE_DECVAL_CNTRNUM_CNTRNUM(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_DECVAL_CNTRNUM_CNTRNUM_SHIFT))&HPSMI_PROFILE_DECVAL_CNTRNUM_CNTRNUM_MASK)
#define HPSMI_PROFILE_DECVAL_CNTRNUM_DEC_MASK    0x10u
#define HPSMI_PROFILE_DECVAL_CNTRNUM_DEC_SHIFT   4u
#define HPSMI_PROFILE_DECVAL_CNTRNUM_DEC_WIDTH   1u
#define HPSMI_PROFILE_DECVAL_CNTRNUM_DEC(x)      (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_DECVAL_CNTRNUM_DEC_SHIFT))&HPSMI_PROFILE_DECVAL_CNTRNUM_DEC_MASK)
/* PROFILE_AXIRDCNTRVAL_0 Bit Fields */
#define HPSMI_PROFILE_AXIRDCNTRVAL_0_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXIRDCNTRVAL_0_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXIRDCNTRVAL_0_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXIRDCNTRVAL_0_CNTRVAL(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXIRDCNTRVAL_0_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXIRDCNTRVAL_0_CNTRVAL_MASK)
/* PROFILE_AXIWRCNTRVAL_0 Bit Fields */
#define HPSMI_PROFILE_AXIWRCNTRVAL_0_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXIWRCNTRVAL_0_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXIWRCNTRVAL_0_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXIWRCNTRVAL_0_CNTRVAL(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXIWRCNTRVAL_0_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXIWRCNTRVAL_0_CNTRVAL_MASK)
/* PROFILE_AXIRDCNTRVAL_1 Bit Fields */
#define HPSMI_PROFILE_AXIRDCNTRVAL_1_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXIRDCNTRVAL_1_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXIRDCNTRVAL_1_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXIRDCNTRVAL_1_CNTRVAL(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXIRDCNTRVAL_1_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXIRDCNTRVAL_1_CNTRVAL_MASK)
/* PROFILE_AXIWRCNTRVAL_1 Bit Fields */
#define HPSMI_PROFILE_AXIWRCNTRVAL_1_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXIWRCNTRVAL_1_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXIWRCNTRVAL_1_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXIWRCNTRVAL_1_CNTRVAL(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXIWRCNTRVAL_1_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXIWRCNTRVAL_1_CNTRVAL_MASK)
/* PROFILE_AXIRDCNTRVAL_2 Bit Fields */
#define HPSMI_PROFILE_AXIRDCNTRVAL_2_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXIRDCNTRVAL_2_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXIRDCNTRVAL_2_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXIRDCNTRVAL_2_CNTRVAL(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXIRDCNTRVAL_2_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXIRDCNTRVAL_2_CNTRVAL_MASK)
/* PROFILE_AXIWRCNTRVAL_2 Bit Fields */
#define HPSMI_PROFILE_AXIWRCNTRVAL_2_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXIWRCNTRVAL_2_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXIWRCNTRVAL_2_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXIWRCNTRVAL_2_CNTRVAL(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXIWRCNTRVAL_2_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXIWRCNTRVAL_2_CNTRVAL_MASK)
/* PROFILE_AXIRDCNTRVAL_3 Bit Fields */
#define HPSMI_PROFILE_AXIRDCNTRVAL_3_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXIRDCNTRVAL_3_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXIRDCNTRVAL_3_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXIRDCNTRVAL_3_CNTRVAL(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXIRDCNTRVAL_3_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXIRDCNTRVAL_3_CNTRVAL_MASK)
/* PROFILE_AXIWRCNTRVAL_3 Bit Fields */
#define HPSMI_PROFILE_AXIWRCNTRVAL_3_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXIWRCNTRVAL_3_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXIWRCNTRVAL_3_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXIWRCNTRVAL_3_CNTRVAL(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXIWRCNTRVAL_3_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXIWRCNTRVAL_3_CNTRVAL_MASK)
/* PROFILE_AXIRDCNTRVAL_4 Bit Fields */
#define HPSMI_PROFILE_AXIRDCNTRVAL_4_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXIRDCNTRVAL_4_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXIRDCNTRVAL_4_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXIRDCNTRVAL_4_CNTRVAL(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXIRDCNTRVAL_4_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXIRDCNTRVAL_4_CNTRVAL_MASK)
/* PROFILE_AXIWRCNTRVAL_4 Bit Fields */
#define HPSMI_PROFILE_AXIWRCNTRVAL_4_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXIWRCNTRVAL_4_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXIWRCNTRVAL_4_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXIWRCNTRVAL_4_CNTRVAL(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXIWRCNTRVAL_4_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXIWRCNTRVAL_4_CNTRVAL_MASK)
/* PROFILE_AXIRDCNTRVAL_5 Bit Fields */
#define HPSMI_PROFILE_AXIRDCNTRVAL_5_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXIRDCNTRVAL_5_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXIRDCNTRVAL_5_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXIRDCNTRVAL_5_CNTRVAL(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXIRDCNTRVAL_5_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXIRDCNTRVAL_5_CNTRVAL_MASK)
/* PROFILE_AXIWRCNTRVAL_5 Bit Fields */
#define HPSMI_PROFILE_AXIWRCNTRVAL_5_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXIWRCNTRVAL_5_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXIWRCNTRVAL_5_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXIWRCNTRVAL_5_CNTRVAL(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXIWRCNTRVAL_5_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXIWRCNTRVAL_5_CNTRVAL_MASK)
/* PROFILE_AXIRDCNTRVAL_6 Bit Fields */
#define HPSMI_PROFILE_AXIRDCNTRVAL_6_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXIRDCNTRVAL_6_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXIRDCNTRVAL_6_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXIRDCNTRVAL_6_CNTRVAL(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXIRDCNTRVAL_6_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXIRDCNTRVAL_6_CNTRVAL_MASK)
/* PROFILE_AXIWRCNTRVAL_6 Bit Fields */
#define HPSMI_PROFILE_AXIWRCNTRVAL_6_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXIWRCNTRVAL_6_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXIWRCNTRVAL_6_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXIWRCNTRVAL_6_CNTRVAL(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXIWRCNTRVAL_6_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXIWRCNTRVAL_6_CNTRVAL_MASK)
/* PROFILE_LSB2AXIRDCNTRVAL_0 Bit Fields */
#define HPSMI_PROFILE_LSB2AXIRDCNTRVAL_0_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_LSB2AXIRDCNTRVAL_0_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_LSB2AXIRDCNTRVAL_0_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_LSB2AXIRDCNTRVAL_0_CNTRVAL(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_LSB2AXIRDCNTRVAL_0_CNTRVAL_SHIFT))&HPSMI_PROFILE_LSB2AXIRDCNTRVAL_0_CNTRVAL_MASK)
/* PROFILE_LSB2AXIWRCNTRVAL_0 Bit Fields */
#define HPSMI_PROFILE_LSB2AXIWRCNTRVAL_0_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_LSB2AXIWRCNTRVAL_0_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_LSB2AXIWRCNTRVAL_0_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_LSB2AXIWRCNTRVAL_0_CNTRVAL(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_LSB2AXIWRCNTRVAL_0_CNTRVAL_SHIFT))&HPSMI_PROFILE_LSB2AXIWRCNTRVAL_0_CNTRVAL_MASK)
/* PROFILE_LSB2AXIRDCNTRVAL_1 Bit Fields */
#define HPSMI_PROFILE_LSB2AXIRDCNTRVAL_1_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_LSB2AXIRDCNTRVAL_1_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_LSB2AXIRDCNTRVAL_1_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_LSB2AXIRDCNTRVAL_1_CNTRVAL(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_LSB2AXIRDCNTRVAL_1_CNTRVAL_SHIFT))&HPSMI_PROFILE_LSB2AXIRDCNTRVAL_1_CNTRVAL_MASK)
/* PROFILE_LSB2AXIWRCNTRVAL_1 Bit Fields */
#define HPSMI_PROFILE_LSB2AXIWRCNTRVAL_1_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_LSB2AXIWRCNTRVAL_1_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_LSB2AXIWRCNTRVAL_1_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_LSB2AXIWRCNTRVAL_1_CNTRVAL(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_LSB2AXIWRCNTRVAL_1_CNTRVAL_SHIFT))&HPSMI_PROFILE_LSB2AXIWRCNTRVAL_1_CNTRVAL_MASK)
/* PROFILE_AXICYCCNTRVAL_0 Bit Fields */
#define HPSMI_PROFILE_AXICYCCNTRVAL_0_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXICYCCNTRVAL_0_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXICYCCNTRVAL_0_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXICYCCNTRVAL_0_CNTRVAL(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXICYCCNTRVAL_0_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXICYCCNTRVAL_0_CNTRVAL_MASK)
/* PROFILE_AXICYCCNTRVAL_1 Bit Fields */
#define HPSMI_PROFILE_AXICYCCNTRVAL_1_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXICYCCNTRVAL_1_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXICYCCNTRVAL_1_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXICYCCNTRVAL_1_CNTRVAL(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXICYCCNTRVAL_1_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXICYCCNTRVAL_1_CNTRVAL_MASK)
/* PROFILE_AXICYCCNTRVAL_2 Bit Fields */
#define HPSMI_PROFILE_AXICYCCNTRVAL_2_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXICYCCNTRVAL_2_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXICYCCNTRVAL_2_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXICYCCNTRVAL_2_CNTRVAL(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXICYCCNTRVAL_2_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXICYCCNTRVAL_2_CNTRVAL_MASK)
/* PROFILE_AXICYCCNTRVAL_3 Bit Fields */
#define HPSMI_PROFILE_AXICYCCNTRVAL_3_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXICYCCNTRVAL_3_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXICYCCNTRVAL_3_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXICYCCNTRVAL_3_CNTRVAL(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXICYCCNTRVAL_3_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXICYCCNTRVAL_3_CNTRVAL_MASK)
/* PROFILE_AXICYCCNTRVAL_4 Bit Fields */
#define HPSMI_PROFILE_AXICYCCNTRVAL_4_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXICYCCNTRVAL_4_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXICYCCNTRVAL_4_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXICYCCNTRVAL_4_CNTRVAL(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXICYCCNTRVAL_4_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXICYCCNTRVAL_4_CNTRVAL_MASK)
/* PROFILE_AXICYCCNTRVAL_5 Bit Fields */
#define HPSMI_PROFILE_AXICYCCNTRVAL_5_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXICYCCNTRVAL_5_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXICYCCNTRVAL_5_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXICYCCNTRVAL_5_CNTRVAL(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXICYCCNTRVAL_5_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXICYCCNTRVAL_5_CNTRVAL_MASK)
/* PROFILE_AXICYCCNTRVAL_6 Bit Fields */
#define HPSMI_PROFILE_AXICYCCNTRVAL_6_CNTRVAL_MASK 0xFFFFFFFFu
#define HPSMI_PROFILE_AXICYCCNTRVAL_6_CNTRVAL_SHIFT 0u
#define HPSMI_PROFILE_AXICYCCNTRVAL_6_CNTRVAL_WIDTH 32u
#define HPSMI_PROFILE_AXICYCCNTRVAL_6_CNTRVAL(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_PROFILE_AXICYCCNTRVAL_6_CNTRVAL_SHIFT))&HPSMI_PROFILE_AXICYCCNTRVAL_6_CNTRVAL_MASK)
/* RD_PL_LVL_31_0 Bit Fields */
#define HPSMI_RD_PL_LVL_31_0_MASTERNUM_MASK      0xFFFFFFFFu
#define HPSMI_RD_PL_LVL_31_0_MASTERNUM_SHIFT     0u
#define HPSMI_RD_PL_LVL_31_0_MASTERNUM_WIDTH     32u
#define HPSMI_RD_PL_LVL_31_0_MASTERNUM(x)        (((uint32_t)(((uint32_t)(x))<<HPSMI_RD_PL_LVL_31_0_MASTERNUM_SHIFT))&HPSMI_RD_PL_LVL_31_0_MASTERNUM_MASK)
/* RD_PL_LVL_63_32 Bit Fields */
#define HPSMI_RD_PL_LVL_63_32_MASTERNUM_MASK     0xFFFFFFFFu
#define HPSMI_RD_PL_LVL_63_32_MASTERNUM_SHIFT    0u
#define HPSMI_RD_PL_LVL_63_32_MASTERNUM_WIDTH    32u
#define HPSMI_RD_PL_LVL_63_32_MASTERNUM(x)       (((uint32_t)(((uint32_t)(x))<<HPSMI_RD_PL_LVL_63_32_MASTERNUM_SHIFT))&HPSMI_RD_PL_LVL_63_32_MASTERNUM_MASK)
/* WR_PL_LVL_31_0 Bit Fields */
#define HPSMI_WR_PL_LVL_31_0_MASTERNUM_MASK      0xFFFFFFFFu
#define HPSMI_WR_PL_LVL_31_0_MASTERNUM_SHIFT     0u
#define HPSMI_WR_PL_LVL_31_0_MASTERNUM_WIDTH     32u
#define HPSMI_WR_PL_LVL_31_0_MASTERNUM(x)        (((uint32_t)(((uint32_t)(x))<<HPSMI_WR_PL_LVL_31_0_MASTERNUM_SHIFT))&HPSMI_WR_PL_LVL_31_0_MASTERNUM_MASK)
/* WR_PL_LVL_63_32 Bit Fields */
#define HPSMI_WR_PL_LVL_63_32_MASTERNUM_MASK     0xFFFFFFFFu
#define HPSMI_WR_PL_LVL_63_32_MASTERNUM_SHIFT    0u
#define HPSMI_WR_PL_LVL_63_32_MASTERNUM_WIDTH    32u
#define HPSMI_WR_PL_LVL_63_32_MASTERNUM(x)       (((uint32_t)(((uint32_t)(x))<<HPSMI_WR_PL_LVL_63_32_MASTERNUM_SHIFT))&HPSMI_WR_PL_LVL_63_32_MASTERNUM_MASK)
/* DSRD_ADDRESS_ERROR0 Bit Fields */
#define HPSMI_DSRD_ADDRESS_ERROR0_DMA_RD_ADDR_ERROR_MASK 0xFFFFFFFFu
#define HPSMI_DSRD_ADDRESS_ERROR0_DMA_RD_ADDR_ERROR_SHIFT 0u
#define HPSMI_DSRD_ADDRESS_ERROR0_DMA_RD_ADDR_ERROR_WIDTH 32u
#define HPSMI_DSRD_ADDRESS_ERROR0_DMA_RD_ADDR_ERROR(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_DSRD_ADDRESS_ERROR0_DMA_RD_ADDR_ERROR_SHIFT))&HPSMI_DSRD_ADDRESS_ERROR0_DMA_RD_ADDR_ERROR_MASK)
/* DSRD_ADDRESS_ERROR1 Bit Fields */
#define HPSMI_DSRD_ADDRESS_ERROR1_DMA_RD_ADDR_ERROR_MASK 0xFFFFFFFFu
#define HPSMI_DSRD_ADDRESS_ERROR1_DMA_RD_ADDR_ERROR_SHIFT 0u
#define HPSMI_DSRD_ADDRESS_ERROR1_DMA_RD_ADDR_ERROR_WIDTH 32u
#define HPSMI_DSRD_ADDRESS_ERROR1_DMA_RD_ADDR_ERROR(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_DSRD_ADDRESS_ERROR1_DMA_RD_ADDR_ERROR_SHIFT))&HPSMI_DSRD_ADDRESS_ERROR1_DMA_RD_ADDR_ERROR_MASK)
/* DSWR_ADDRESS_ERROR0 Bit Fields */
#define HPSMI_DSWR_ADDRESS_ERROR0_DMA_WR_ADDR_ERROR_MASK 0xFFFFFFFFu
#define HPSMI_DSWR_ADDRESS_ERROR0_DMA_WR_ADDR_ERROR_SHIFT 0u
#define HPSMI_DSWR_ADDRESS_ERROR0_DMA_WR_ADDR_ERROR_WIDTH 32u
#define HPSMI_DSWR_ADDRESS_ERROR0_DMA_WR_ADDR_ERROR(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_DSWR_ADDRESS_ERROR0_DMA_WR_ADDR_ERROR_SHIFT))&HPSMI_DSWR_ADDRESS_ERROR0_DMA_WR_ADDR_ERROR_MASK)
/* DSWR_ADDRESS_ERROR1 Bit Fields */
#define HPSMI_DSWR_ADDRESS_ERROR1_DMA_WR_ADDR_ERROR_MASK 0xFFFFFFFFu
#define HPSMI_DSWR_ADDRESS_ERROR1_DMA_WR_ADDR_ERROR_SHIFT 0u
#define HPSMI_DSWR_ADDRESS_ERROR1_DMA_WR_ADDR_ERROR_WIDTH 32u
#define HPSMI_DSWR_ADDRESS_ERROR1_DMA_WR_ADDR_ERROR(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_DSWR_ADDRESS_ERROR1_DMA_WR_ADDR_ERROR_SHIFT))&HPSMI_DSWR_ADDRESS_ERROR1_DMA_WR_ADDR_ERROR_MASK)
/* ADDR_ERROR_ADDRVAL Bit Fields */
#define HPSMI_ADDR_ERROR_ADDRVAL_ERROR_ADDRESS_MASK 0xFFFFFFFFu
#define HPSMI_ADDR_ERROR_ADDRVAL_ERROR_ADDRESS_SHIFT 0u
#define HPSMI_ADDR_ERROR_ADDRVAL_ERROR_ADDRESS_WIDTH 32u
#define HPSMI_ADDR_ERROR_ADDRVAL_ERROR_ADDRESS(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ADDR_ERROR_ADDRVAL_ERROR_ADDRESS_SHIFT))&HPSMI_ADDR_ERROR_ADDRVAL_ERROR_ADDRESS_MASK)
/* ADDR_ERROR_INFO Bit Fields */
#define HPSMI_ADDR_ERROR_INFO_MASTERNUM_MASK     0x3Fu
#define HPSMI_ADDR_ERROR_INFO_MASTERNUM_SHIFT    0u
#define HPSMI_ADDR_ERROR_INFO_MASTERNUM_WIDTH    6u
#define HPSMI_ADDR_ERROR_INFO_MASTERNUM(x)       (((uint32_t)(((uint32_t)(x))<<HPSMI_ADDR_ERROR_INFO_MASTERNUM_SHIFT))&HPSMI_ADDR_ERROR_INFO_MASTERNUM_MASK)
#define HPSMI_ADDR_ERROR_INFO_MASTER_TYPE_MASK   0x40u
#define HPSMI_ADDR_ERROR_INFO_MASTER_TYPE_SHIFT  6u
#define HPSMI_ADDR_ERROR_INFO_MASTER_TYPE_WIDTH  1u
#define HPSMI_ADDR_ERROR_INFO_MASTER_TYPE(x)     (((uint32_t)(((uint32_t)(x))<<HPSMI_ADDR_ERROR_INFO_MASTER_TYPE_SHIFT))&HPSMI_ADDR_ERROR_INFO_MASTER_TYPE_MASK)
#define HPSMI_ADDR_ERROR_INFO_ADDR_ERR_MASK      0x200u
#define HPSMI_ADDR_ERROR_INFO_ADDR_ERR_SHIFT     9u
#define HPSMI_ADDR_ERROR_INFO_ADDR_ERR_WIDTH     1u
#define HPSMI_ADDR_ERROR_INFO_ADDR_ERR(x)        (((uint32_t)(((uint32_t)(x))<<HPSMI_ADDR_ERROR_INFO_ADDR_ERR_SHIFT))&HPSMI_ADDR_ERROR_INFO_ADDR_ERR_MASK)
/* WRMPU_ERROR0 Bit Fields */
#define HPSMI_WRMPU_ERROR0_DMA_WR_MPU_ERR_MASK   0xFFFFFFFFu
#define HPSMI_WRMPU_ERROR0_DMA_WR_MPU_ERR_SHIFT  0u
#define HPSMI_WRMPU_ERROR0_DMA_WR_MPU_ERR_WIDTH  32u
#define HPSMI_WRMPU_ERROR0_DMA_WR_MPU_ERR(x)     (((uint32_t)(((uint32_t)(x))<<HPSMI_WRMPU_ERROR0_DMA_WR_MPU_ERR_SHIFT))&HPSMI_WRMPU_ERROR0_DMA_WR_MPU_ERR_MASK)
/* WRMPU_ERROR1 Bit Fields */
#define HPSMI_WRMPU_ERROR1_DMA_WR_MPU_ERR_MASK   0xFFFFFFFFu
#define HPSMI_WRMPU_ERROR1_DMA_WR_MPU_ERR_SHIFT  0u
#define HPSMI_WRMPU_ERROR1_DMA_WR_MPU_ERR_WIDTH  32u
#define HPSMI_WRMPU_ERROR1_DMA_WR_MPU_ERR(x)     (((uint32_t)(((uint32_t)(x))<<HPSMI_WRMPU_ERROR1_DMA_WR_MPU_ERR_SHIFT))&HPSMI_WRMPU_ERROR1_DMA_WR_MPU_ERR_MASK)
/* WRMPU_ERROR_ADDR Bit Fields */
#define HPSMI_WRMPU_ERROR_ADDR_ERROR_ADDRESS_MASK 0xFFFFFFFFu
#define HPSMI_WRMPU_ERROR_ADDR_ERROR_ADDRESS_SHIFT 0u
#define HPSMI_WRMPU_ERROR_ADDR_ERROR_ADDRESS_WIDTH 32u
#define HPSMI_WRMPU_ERROR_ADDR_ERROR_ADDRESS(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_WRMPU_ERROR_ADDR_ERROR_ADDRESS_SHIFT))&HPSMI_WRMPU_ERROR_ADDR_ERROR_ADDRESS_MASK)
/* WRMPU_ERROR_INFO Bit Fields */
#define HPSMI_WRMPU_ERROR_INFO_MASTERNUM_MASK    0x3Fu
#define HPSMI_WRMPU_ERROR_INFO_MASTERNUM_SHIFT   0u
#define HPSMI_WRMPU_ERROR_INFO_MASTERNUM_WIDTH   6u
#define HPSMI_WRMPU_ERROR_INFO_MASTERNUM(x)      (((uint32_t)(((uint32_t)(x))<<HPSMI_WRMPU_ERROR_INFO_MASTERNUM_SHIFT))&HPSMI_WRMPU_ERROR_INFO_MASTERNUM_MASK)
#define HPSMI_WRMPU_ERROR_INFO_MPU_ERR_MASK      0x200u
#define HPSMI_WRMPU_ERROR_INFO_MPU_ERR_SHIFT     9u
#define HPSMI_WRMPU_ERROR_INFO_MPU_ERR_WIDTH     1u
#define HPSMI_WRMPU_ERROR_INFO_MPU_ERR(x)        (((uint32_t)(((uint32_t)(x))<<HPSMI_WRMPU_ERROR_INFO_MPU_ERR_SHIFT))&HPSMI_WRMPU_ERROR_INFO_MPU_ERR_MASK)
/* PM_ERROR_1_2 Bit Fields */
#define HPSMI_PM_ERROR_1_2_ERROR_BANK_MASK       0xFFFFu
#define HPSMI_PM_ERROR_1_2_ERROR_BANK_SHIFT      0u
#define HPSMI_PM_ERROR_1_2_ERROR_BANK_WIDTH      16u
#define HPSMI_PM_ERROR_1_2_ERROR_BANK(x)         (((uint32_t)(((uint32_t)(x))<<HPSMI_PM_ERROR_1_2_ERROR_BANK_SHIFT))&HPSMI_PM_ERROR_1_2_ERROR_BANK_MASK)
/* PM_ERROR_ADDR_1_2 Bit Fields */
#define HPSMI_PM_ERROR_ADDR_1_2_ERROR_ADDR_MASK  0xFFFFFFFFu
#define HPSMI_PM_ERROR_ADDR_1_2_ERROR_ADDR_SHIFT 0u
#define HPSMI_PM_ERROR_ADDR_1_2_ERROR_ADDR_WIDTH 32u
#define HPSMI_PM_ERROR_ADDR_1_2_ERROR_ADDR(x)    (((uint32_t)(((uint32_t)(x))<<HPSMI_PM_ERROR_ADDR_1_2_ERROR_ADDR_SHIFT))&HPSMI_PM_ERROR_ADDR_1_2_ERROR_ADDR_MASK)
/* PM_ERROR_INFO_1_2 Bit Fields */
#define HPSMI_PM_ERROR_INFO_1_2_MASTERNUM_MASK   0x3Fu
#define HPSMI_PM_ERROR_INFO_1_2_MASTERNUM_SHIFT  0u
#define HPSMI_PM_ERROR_INFO_1_2_MASTERNUM_WIDTH  6u
#define HPSMI_PM_ERROR_INFO_1_2_MASTERNUM(x)     (((uint32_t)(((uint32_t)(x))<<HPSMI_PM_ERROR_INFO_1_2_MASTERNUM_SHIFT))&HPSMI_PM_ERROR_INFO_1_2_MASTERNUM_MASK)
#define HPSMI_PM_ERROR_INFO_1_2_MASTER_TYPE_MASK 0xC0u
#define HPSMI_PM_ERROR_INFO_1_2_MASTER_TYPE_SHIFT 6u
#define HPSMI_PM_ERROR_INFO_1_2_MASTER_TYPE_WIDTH 2u
#define HPSMI_PM_ERROR_INFO_1_2_MASTER_TYPE(x)   (((uint32_t)(((uint32_t)(x))<<HPSMI_PM_ERROR_INFO_1_2_MASTER_TYPE_SHIFT))&HPSMI_PM_ERROR_INFO_1_2_MASTER_TYPE_MASK)
#define HPSMI_PM_ERROR_INFO_1_2_BANKNUM_MASK     0xF00u
#define HPSMI_PM_ERROR_INFO_1_2_BANKNUM_SHIFT    8u
#define HPSMI_PM_ERROR_INFO_1_2_BANKNUM_WIDTH    4u
#define HPSMI_PM_ERROR_INFO_1_2_BANKNUM(x)       (((uint32_t)(((uint32_t)(x))<<HPSMI_PM_ERROR_INFO_1_2_BANKNUM_SHIFT))&HPSMI_PM_ERROR_INFO_1_2_BANKNUM_MASK)
#define HPSMI_PM_ERROR_INFO_1_2_PM_ERR_MASK      0x2000u
#define HPSMI_PM_ERROR_INFO_1_2_PM_ERR_SHIFT     13u
#define HPSMI_PM_ERROR_INFO_1_2_PM_ERR_WIDTH     1u
#define HPSMI_PM_ERROR_INFO_1_2_PM_ERR(x)        (((uint32_t)(((uint32_t)(x))<<HPSMI_PM_ERROR_INFO_1_2_PM_ERR_SHIFT))&HPSMI_PM_ERROR_INFO_1_2_PM_ERR_MASK)
/* ECC_ERR_INJECT Bit Fields */
#define HPSMI_ECC_ERR_INJECT_ERR_INJECT_MASK     0xFFu
#define HPSMI_ECC_ERR_INJECT_ERR_INJECT_SHIFT    0u
#define HPSMI_ECC_ERR_INJECT_ERR_INJECT_WIDTH    8u
#define HPSMI_ECC_ERR_INJECT_ERR_INJECT(x)       (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERR_INJECT_ERR_INJECT_SHIFT))&HPSMI_ECC_ERR_INJECT_ERR_INJECT_MASK)
/* ECC_ERRCNTR_MASK Bit Fields */
#define HPSMI_ECC_ERRCNTR_MASK_SEG1_MASK_MASK    0x100u
#define HPSMI_ECC_ERRCNTR_MASK_SEG1_MASK_SHIFT   8u
#define HPSMI_ECC_ERRCNTR_MASK_SEG1_MASK_WIDTH   1u
#define HPSMI_ECC_ERRCNTR_MASK_SEG1_MASK(x)      (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRCNTR_MASK_SEG1_MASK_SHIFT))&HPSMI_ECC_ERRCNTR_MASK_SEG1_MASK_MASK)
#define HPSMI_ECC_ERRCNTR_MASK_SEG2_MASK_MASK    0x200u
#define HPSMI_ECC_ERRCNTR_MASK_SEG2_MASK_SHIFT   9u
#define HPSMI_ECC_ERRCNTR_MASK_SEG2_MASK_WIDTH   1u
#define HPSMI_ECC_ERRCNTR_MASK_SEG2_MASK(x)      (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRCNTR_MASK_SEG2_MASK_SHIFT))&HPSMI_ECC_ERRCNTR_MASK_SEG2_MASK_MASK)
#define HPSMI_ECC_ERRCNTR_MASK_SEG0_ODD_MASK_MASK 0x400u
#define HPSMI_ECC_ERRCNTR_MASK_SEG0_ODD_MASK_SHIFT 10u
#define HPSMI_ECC_ERRCNTR_MASK_SEG0_ODD_MASK_WIDTH 1u
#define HPSMI_ECC_ERRCNTR_MASK_SEG0_ODD_MASK(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRCNTR_MASK_SEG0_ODD_MASK_SHIFT))&HPSMI_ECC_ERRCNTR_MASK_SEG0_ODD_MASK_MASK)
#define HPSMI_ECC_ERRCNTR_MASK_SEG0_EVEN_MASK_MASK 0x800u
#define HPSMI_ECC_ERRCNTR_MASK_SEG0_EVEN_MASK_SHIFT 11u
#define HPSMI_ECC_ERRCNTR_MASK_SEG0_EVEN_MASK_WIDTH 1u
#define HPSMI_ECC_ERRCNTR_MASK_SEG0_EVEN_MASK(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRCNTR_MASK_SEG0_EVEN_MASK_SHIFT))&HPSMI_ECC_ERRCNTR_MASK_SEG0_EVEN_MASK_MASK)
/* ECC_SINGLE_ERRCNTR_1_2 Bit Fields */
#define HPSMI_ECC_SINGLE_ERRCNTR_1_2_ERR_CNTR_SEG1_MASK 0x3Fu
#define HPSMI_ECC_SINGLE_ERRCNTR_1_2_ERR_CNTR_SEG1_SHIFT 0u
#define HPSMI_ECC_SINGLE_ERRCNTR_1_2_ERR_CNTR_SEG1_WIDTH 6u
#define HPSMI_ECC_SINGLE_ERRCNTR_1_2_ERR_CNTR_SEG1(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_SINGLE_ERRCNTR_1_2_ERR_CNTR_SEG1_SHIFT))&HPSMI_ECC_SINGLE_ERRCNTR_1_2_ERR_CNTR_SEG1_MASK)
#define HPSMI_ECC_SINGLE_ERRCNTR_1_2_ECCERR_OVF_SEG1_MASK 0x40u
#define HPSMI_ECC_SINGLE_ERRCNTR_1_2_ECCERR_OVF_SEG1_SHIFT 6u
#define HPSMI_ECC_SINGLE_ERRCNTR_1_2_ECCERR_OVF_SEG1_WIDTH 1u
#define HPSMI_ECC_SINGLE_ERRCNTR_1_2_ECCERR_OVF_SEG1(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_SINGLE_ERRCNTR_1_2_ECCERR_OVF_SEG1_SHIFT))&HPSMI_ECC_SINGLE_ERRCNTR_1_2_ECCERR_OVF_SEG1_MASK)
#define HPSMI_ECC_SINGLE_ERRCNTR_1_2_ERR_CNTR_SEG2_MASK 0x3F00u
#define HPSMI_ECC_SINGLE_ERRCNTR_1_2_ERR_CNTR_SEG2_SHIFT 8u
#define HPSMI_ECC_SINGLE_ERRCNTR_1_2_ERR_CNTR_SEG2_WIDTH 6u
#define HPSMI_ECC_SINGLE_ERRCNTR_1_2_ERR_CNTR_SEG2(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_SINGLE_ERRCNTR_1_2_ERR_CNTR_SEG2_SHIFT))&HPSMI_ECC_SINGLE_ERRCNTR_1_2_ERR_CNTR_SEG2_MASK)
#define HPSMI_ECC_SINGLE_ERRCNTR_1_2_ECCERR_OVF_SEG2_MASK 0x4000u
#define HPSMI_ECC_SINGLE_ERRCNTR_1_2_ECCERR_OVF_SEG2_SHIFT 14u
#define HPSMI_ECC_SINGLE_ERRCNTR_1_2_ECCERR_OVF_SEG2_WIDTH 1u
#define HPSMI_ECC_SINGLE_ERRCNTR_1_2_ECCERR_OVF_SEG2(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_SINGLE_ERRCNTR_1_2_ECCERR_OVF_SEG2_SHIFT))&HPSMI_ECC_SINGLE_ERRCNTR_1_2_ECCERR_OVF_SEG2_MASK)
/* ECC_UNCORR_ERRCNTR_1_2 Bit Fields */
#define HPSMI_ECC_UNCORR_ERRCNTR_1_2_ERR_CNTR_SEG1_MASK 0x3Fu
#define HPSMI_ECC_UNCORR_ERRCNTR_1_2_ERR_CNTR_SEG1_SHIFT 0u
#define HPSMI_ECC_UNCORR_ERRCNTR_1_2_ERR_CNTR_SEG1_WIDTH 6u
#define HPSMI_ECC_UNCORR_ERRCNTR_1_2_ERR_CNTR_SEG1(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_UNCORR_ERRCNTR_1_2_ERR_CNTR_SEG1_SHIFT))&HPSMI_ECC_UNCORR_ERRCNTR_1_2_ERR_CNTR_SEG1_MASK)
#define HPSMI_ECC_UNCORR_ERRCNTR_1_2_ECCERR_OVF_SEG1_MASK 0x40u
#define HPSMI_ECC_UNCORR_ERRCNTR_1_2_ECCERR_OVF_SEG1_SHIFT 6u
#define HPSMI_ECC_UNCORR_ERRCNTR_1_2_ECCERR_OVF_SEG1_WIDTH 1u
#define HPSMI_ECC_UNCORR_ERRCNTR_1_2_ECCERR_OVF_SEG1(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_UNCORR_ERRCNTR_1_2_ECCERR_OVF_SEG1_SHIFT))&HPSMI_ECC_UNCORR_ERRCNTR_1_2_ECCERR_OVF_SEG1_MASK)
#define HPSMI_ECC_UNCORR_ERRCNTR_1_2_ERR_CNTR_SEG2_MASK 0x3F00u
#define HPSMI_ECC_UNCORR_ERRCNTR_1_2_ERR_CNTR_SEG2_SHIFT 8u
#define HPSMI_ECC_UNCORR_ERRCNTR_1_2_ERR_CNTR_SEG2_WIDTH 6u
#define HPSMI_ECC_UNCORR_ERRCNTR_1_2_ERR_CNTR_SEG2(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_UNCORR_ERRCNTR_1_2_ERR_CNTR_SEG2_SHIFT))&HPSMI_ECC_UNCORR_ERRCNTR_1_2_ERR_CNTR_SEG2_MASK)
#define HPSMI_ECC_UNCORR_ERRCNTR_1_2_ECCERR_OVF_SEG2_MASK 0x4000u
#define HPSMI_ECC_UNCORR_ERRCNTR_1_2_ECCERR_OVF_SEG2_SHIFT 14u
#define HPSMI_ECC_UNCORR_ERRCNTR_1_2_ECCERR_OVF_SEG2_WIDTH 1u
#define HPSMI_ECC_UNCORR_ERRCNTR_1_2_ECCERR_OVF_SEG2(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_UNCORR_ERRCNTR_1_2_ECCERR_OVF_SEG2_SHIFT))&HPSMI_ECC_UNCORR_ERRCNTR_1_2_ECCERR_OVF_SEG2_MASK)
/* ECC_ERRADDR_SEG1 Bit Fields */
#define HPSMI_ECC_ERRADDR_SEG1_ERROR_ADDR_MASK   0xFFFFFFFFu
#define HPSMI_ECC_ERRADDR_SEG1_ERROR_ADDR_SHIFT  0u
#define HPSMI_ECC_ERRADDR_SEG1_ERROR_ADDR_WIDTH  32u
#define HPSMI_ECC_ERRADDR_SEG1_ERROR_ADDR(x)     (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRADDR_SEG1_ERROR_ADDR_SHIFT))&HPSMI_ECC_ERRADDR_SEG1_ERROR_ADDR_MASK)
/* ECC_ERRDATA0_SEG1 Bit Fields */
#define HPSMI_ECC_ERRDATA0_SEG1_ERROR_DATA0_MASK 0xFFFFFFFFu
#define HPSMI_ECC_ERRDATA0_SEG1_ERROR_DATA0_SHIFT 0u
#define HPSMI_ECC_ERRDATA0_SEG1_ERROR_DATA0_WIDTH 32u
#define HPSMI_ECC_ERRDATA0_SEG1_ERROR_DATA0(x)   (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRDATA0_SEG1_ERROR_DATA0_SHIFT))&HPSMI_ECC_ERRDATA0_SEG1_ERROR_DATA0_MASK)
/* ECC_ERRDATA1_SEG1 Bit Fields */
#define HPSMI_ECC_ERRDATA1_SEG1_ERROR_DATA1_MASK 0xFFFFFFFFu
#define HPSMI_ECC_ERRDATA1_SEG1_ERROR_DATA1_SHIFT 0u
#define HPSMI_ECC_ERRDATA1_SEG1_ERROR_DATA1_WIDTH 32u
#define HPSMI_ECC_ERRDATA1_SEG1_ERROR_DATA1(x)   (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRDATA1_SEG1_ERROR_DATA1_SHIFT))&HPSMI_ECC_ERRDATA1_SEG1_ERROR_DATA1_MASK)
/* ECC_ERRINFO_SEG1 Bit Fields */
#define HPSMI_ECC_ERRINFO_SEG1_SYNDROME_MASK     0xFFu
#define HPSMI_ECC_ERRINFO_SEG1_SYNDROME_SHIFT    0u
#define HPSMI_ECC_ERRINFO_SEG1_SYNDROME_WIDTH    8u
#define HPSMI_ECC_ERRINFO_SEG1_SYNDROME(x)       (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRINFO_SEG1_SYNDROME_SHIFT))&HPSMI_ECC_ERRINFO_SEG1_SYNDROME_MASK)
#define HPSMI_ECC_ERRINFO_SEG1_ERR_TYPE_MASK     0x100u
#define HPSMI_ECC_ERRINFO_SEG1_ERR_TYPE_SHIFT    8u
#define HPSMI_ECC_ERRINFO_SEG1_ERR_TYPE_WIDTH    1u
#define HPSMI_ECC_ERRINFO_SEG1_ERR_TYPE(x)       (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRINFO_SEG1_ERR_TYPE_SHIFT))&HPSMI_ECC_ERRINFO_SEG1_ERR_TYPE_MASK)
#define HPSMI_ECC_ERRINFO_SEG1_ERR_BANK_MASK     0xE00u
#define HPSMI_ECC_ERRINFO_SEG1_ERR_BANK_SHIFT    9u
#define HPSMI_ECC_ERRINFO_SEG1_ERR_BANK_WIDTH    3u
#define HPSMI_ECC_ERRINFO_SEG1_ERR_BANK(x)       (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRINFO_SEG1_ERR_BANK_SHIFT))&HPSMI_ECC_ERRINFO_SEG1_ERR_BANK_MASK)
#define HPSMI_ECC_ERRINFO_SEG1_ECC_ERR_MASK      0x1000u
#define HPSMI_ECC_ERRINFO_SEG1_ECC_ERR_SHIFT     12u
#define HPSMI_ECC_ERRINFO_SEG1_ECC_ERR_WIDTH     1u
#define HPSMI_ECC_ERRINFO_SEG1_ECC_ERR(x)        (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRINFO_SEG1_ECC_ERR_SHIFT))&HPSMI_ECC_ERRINFO_SEG1_ECC_ERR_MASK)
/* ECC_ERRADDR_SEG2 Bit Fields */
#define HPSMI_ECC_ERRADDR_SEG2_ERROR_ADDR_MASK   0xFFFFFFFFu
#define HPSMI_ECC_ERRADDR_SEG2_ERROR_ADDR_SHIFT  0u
#define HPSMI_ECC_ERRADDR_SEG2_ERROR_ADDR_WIDTH  32u
#define HPSMI_ECC_ERRADDR_SEG2_ERROR_ADDR(x)     (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRADDR_SEG2_ERROR_ADDR_SHIFT))&HPSMI_ECC_ERRADDR_SEG2_ERROR_ADDR_MASK)
/* ECC_ERRDATA0_SEG2 Bit Fields */
#define HPSMI_ECC_ERRDATA0_SEG2_ERROR_DATA0_MASK 0xFFFFFFFFu
#define HPSMI_ECC_ERRDATA0_SEG2_ERROR_DATA0_SHIFT 0u
#define HPSMI_ECC_ERRDATA0_SEG2_ERROR_DATA0_WIDTH 32u
#define HPSMI_ECC_ERRDATA0_SEG2_ERROR_DATA0(x)   (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRDATA0_SEG2_ERROR_DATA0_SHIFT))&HPSMI_ECC_ERRDATA0_SEG2_ERROR_DATA0_MASK)
/* ECC_ERRDATA1_SEG2 Bit Fields */
#define HPSMI_ECC_ERRDATA1_SEG2_ERROR_DATA1_MASK 0xFFFFFFFFu
#define HPSMI_ECC_ERRDATA1_SEG2_ERROR_DATA1_SHIFT 0u
#define HPSMI_ECC_ERRDATA1_SEG2_ERROR_DATA1_WIDTH 32u
#define HPSMI_ECC_ERRDATA1_SEG2_ERROR_DATA1(x)   (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRDATA1_SEG2_ERROR_DATA1_SHIFT))&HPSMI_ECC_ERRDATA1_SEG2_ERROR_DATA1_MASK)
/* ECC_ERRINFO_SEG2 Bit Fields */
#define HPSMI_ECC_ERRINFO_SEG2_SYNDROME_MASK     0xFFu
#define HPSMI_ECC_ERRINFO_SEG2_SYNDROME_SHIFT    0u
#define HPSMI_ECC_ERRINFO_SEG2_SYNDROME_WIDTH    8u
#define HPSMI_ECC_ERRINFO_SEG2_SYNDROME(x)       (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRINFO_SEG2_SYNDROME_SHIFT))&HPSMI_ECC_ERRINFO_SEG2_SYNDROME_MASK)
#define HPSMI_ECC_ERRINFO_SEG2_ERR_TYPE_MASK     0x100u
#define HPSMI_ECC_ERRINFO_SEG2_ERR_TYPE_SHIFT    8u
#define HPSMI_ECC_ERRINFO_SEG2_ERR_TYPE_WIDTH    1u
#define HPSMI_ECC_ERRINFO_SEG2_ERR_TYPE(x)       (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRINFO_SEG2_ERR_TYPE_SHIFT))&HPSMI_ECC_ERRINFO_SEG2_ERR_TYPE_MASK)
#define HPSMI_ECC_ERRINFO_SEG2_ERR_BANK_MASK     0xE00u
#define HPSMI_ECC_ERRINFO_SEG2_ERR_BANK_SHIFT    9u
#define HPSMI_ECC_ERRINFO_SEG2_ERR_BANK_WIDTH    3u
#define HPSMI_ECC_ERRINFO_SEG2_ERR_BANK(x)       (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRINFO_SEG2_ERR_BANK_SHIFT))&HPSMI_ECC_ERRINFO_SEG2_ERR_BANK_MASK)
#define HPSMI_ECC_ERRINFO_SEG2_ECC_ERR_MASK      0x1000u
#define HPSMI_ECC_ERRINFO_SEG2_ECC_ERR_SHIFT     12u
#define HPSMI_ECC_ERRINFO_SEG2_ECC_ERR_WIDTH     1u
#define HPSMI_ECC_ERRINFO_SEG2_ECC_ERR(x)        (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRINFO_SEG2_ECC_ERR_SHIFT))&HPSMI_ECC_ERRINFO_SEG2_ECC_ERR_MASK)
/* ECC_ERROCCURRED_SEG1_2 Bit Fields */
#define HPSMI_ECC_ERROCCURRED_SEG1_2_ERR_MASK    0xFFFFu
#define HPSMI_ECC_ERROCCURRED_SEG1_2_ERR_SHIFT   0u
#define HPSMI_ECC_ERROCCURRED_SEG1_2_ERR_WIDTH   16u
#define HPSMI_ECC_ERROCCURRED_SEG1_2_ERR(x)      (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERROCCURRED_SEG1_2_ERR_SHIFT))&HPSMI_ECC_ERROCCURRED_SEG1_2_ERR_MASK)
/* ECC_ERRADDR_SEG0_EVEN Bit Fields */
#define HPSMI_ECC_ERRADDR_SEG0_EVEN_ERR_ADDR_EVEN_MASK 0xFFFFFFFFu
#define HPSMI_ECC_ERRADDR_SEG0_EVEN_ERR_ADDR_EVEN_SHIFT 0u
#define HPSMI_ECC_ERRADDR_SEG0_EVEN_ERR_ADDR_EVEN_WIDTH 32u
#define HPSMI_ECC_ERRADDR_SEG0_EVEN_ERR_ADDR_EVEN(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRADDR_SEG0_EVEN_ERR_ADDR_EVEN_SHIFT))&HPSMI_ECC_ERRADDR_SEG0_EVEN_ERR_ADDR_EVEN_MASK)
/* ECC_ERRADDR_SEG0_ODD Bit Fields */
#define HPSMI_ECC_ERRADDR_SEG0_ODD_ERR_ADDR_ODD_MASK 0xFFFFFFFFu
#define HPSMI_ECC_ERRADDR_SEG0_ODD_ERR_ADDR_ODD_SHIFT 0u
#define HPSMI_ECC_ERRADDR_SEG0_ODD_ERR_ADDR_ODD_WIDTH 32u
#define HPSMI_ECC_ERRADDR_SEG0_ODD_ERR_ADDR_ODD(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRADDR_SEG0_ODD_ERR_ADDR_ODD_SHIFT))&HPSMI_ECC_ERRADDR_SEG0_ODD_ERR_ADDR_ODD_MASK)
/* ECC_ERRDATA0_SEG0_EVEN Bit Fields */
#define HPSMI_ECC_ERRDATA0_SEG0_EVEN_ERR_DATA0_EVEN_MASK 0xFFFFFFFFu
#define HPSMI_ECC_ERRDATA0_SEG0_EVEN_ERR_DATA0_EVEN_SHIFT 0u
#define HPSMI_ECC_ERRDATA0_SEG0_EVEN_ERR_DATA0_EVEN_WIDTH 32u
#define HPSMI_ECC_ERRDATA0_SEG0_EVEN_ERR_DATA0_EVEN(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRDATA0_SEG0_EVEN_ERR_DATA0_EVEN_SHIFT))&HPSMI_ECC_ERRDATA0_SEG0_EVEN_ERR_DATA0_EVEN_MASK)
/* ECC_ERRDATA1_SEG0_EVEN Bit Fields */
#define HPSMI_ECC_ERRDATA1_SEG0_EVEN_ERR_DATA1_EVEN_MASK 0xFFFFFFFFu
#define HPSMI_ECC_ERRDATA1_SEG0_EVEN_ERR_DATA1_EVEN_SHIFT 0u
#define HPSMI_ECC_ERRDATA1_SEG0_EVEN_ERR_DATA1_EVEN_WIDTH 32u
#define HPSMI_ECC_ERRDATA1_SEG0_EVEN_ERR_DATA1_EVEN(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRDATA1_SEG0_EVEN_ERR_DATA1_EVEN_SHIFT))&HPSMI_ECC_ERRDATA1_SEG0_EVEN_ERR_DATA1_EVEN_MASK)
/* ECC_ERRDATA0_SEG0_ODD Bit Fields */
#define HPSMI_ECC_ERRDATA0_SEG0_ODD_ERR_DATA0_ODD_MASK 0xFFFFFFFFu
#define HPSMI_ECC_ERRDATA0_SEG0_ODD_ERR_DATA0_ODD_SHIFT 0u
#define HPSMI_ECC_ERRDATA0_SEG0_ODD_ERR_DATA0_ODD_WIDTH 32u
#define HPSMI_ECC_ERRDATA0_SEG0_ODD_ERR_DATA0_ODD(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRDATA0_SEG0_ODD_ERR_DATA0_ODD_SHIFT))&HPSMI_ECC_ERRDATA0_SEG0_ODD_ERR_DATA0_ODD_MASK)
/* ECC_ERRDATA1_SEG0_ODD Bit Fields */
#define HPSMI_ECC_ERRDATA1_SEG0_ODD_ERR_DATA1_ODD_MASK 0xFFFFFFFFu
#define HPSMI_ECC_ERRDATA1_SEG0_ODD_ERR_DATA1_ODD_SHIFT 0u
#define HPSMI_ECC_ERRDATA1_SEG0_ODD_ERR_DATA1_ODD_WIDTH 32u
#define HPSMI_ECC_ERRDATA1_SEG0_ODD_ERR_DATA1_ODD(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRDATA1_SEG0_ODD_ERR_DATA1_ODD_SHIFT))&HPSMI_ECC_ERRDATA1_SEG0_ODD_ERR_DATA1_ODD_MASK)
/* ECC_ERRINFO_SEG0 Bit Fields */
#define HPSMI_ECC_ERRINFO_SEG0_SYND_EVEN_MASK    0xFFu
#define HPSMI_ECC_ERRINFO_SEG0_SYND_EVEN_SHIFT   0u
#define HPSMI_ECC_ERRINFO_SEG0_SYND_EVEN_WIDTH   8u
#define HPSMI_ECC_ERRINFO_SEG0_SYND_EVEN(x)      (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRINFO_SEG0_SYND_EVEN_SHIFT))&HPSMI_ECC_ERRINFO_SEG0_SYND_EVEN_MASK)
#define HPSMI_ECC_ERRINFO_SEG0_ERR_TYPE_EVEN_MASK 0x100u
#define HPSMI_ECC_ERRINFO_SEG0_ERR_TYPE_EVEN_SHIFT 8u
#define HPSMI_ECC_ERRINFO_SEG0_ERR_TYPE_EVEN_WIDTH 1u
#define HPSMI_ECC_ERRINFO_SEG0_ERR_TYPE_EVEN(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRINFO_SEG0_ERR_TYPE_EVEN_SHIFT))&HPSMI_ECC_ERRINFO_SEG0_ERR_TYPE_EVEN_MASK)
#define HPSMI_ECC_ERRINFO_SEG0_ERR_EVEN_BANK_MASK 0x1E00u
#define HPSMI_ECC_ERRINFO_SEG0_ERR_EVEN_BANK_SHIFT 9u
#define HPSMI_ECC_ERRINFO_SEG0_ERR_EVEN_BANK_WIDTH 4u
#define HPSMI_ECC_ERRINFO_SEG0_ERR_EVEN_BANK(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRINFO_SEG0_ERR_EVEN_BANK_SHIFT))&HPSMI_ECC_ERRINFO_SEG0_ERR_EVEN_BANK_MASK)
#define HPSMI_ECC_ERRINFO_SEG0_ECC_ERR_EVEN_MASK 0x2000u
#define HPSMI_ECC_ERRINFO_SEG0_ECC_ERR_EVEN_SHIFT 13u
#define HPSMI_ECC_ERRINFO_SEG0_ECC_ERR_EVEN_WIDTH 1u
#define HPSMI_ECC_ERRINFO_SEG0_ECC_ERR_EVEN(x)   (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRINFO_SEG0_ECC_ERR_EVEN_SHIFT))&HPSMI_ECC_ERRINFO_SEG0_ECC_ERR_EVEN_MASK)
#define HPSMI_ECC_ERRINFO_SEG0_SYND_ODD_MASK     0xFF0000u
#define HPSMI_ECC_ERRINFO_SEG0_SYND_ODD_SHIFT    16u
#define HPSMI_ECC_ERRINFO_SEG0_SYND_ODD_WIDTH    8u
#define HPSMI_ECC_ERRINFO_SEG0_SYND_ODD(x)       (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRINFO_SEG0_SYND_ODD_SHIFT))&HPSMI_ECC_ERRINFO_SEG0_SYND_ODD_MASK)
#define HPSMI_ECC_ERRINFO_SEG0_ERR_TYPE_ODD_MASK 0x1000000u
#define HPSMI_ECC_ERRINFO_SEG0_ERR_TYPE_ODD_SHIFT 24u
#define HPSMI_ECC_ERRINFO_SEG0_ERR_TYPE_ODD_WIDTH 1u
#define HPSMI_ECC_ERRINFO_SEG0_ERR_TYPE_ODD(x)   (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRINFO_SEG0_ERR_TYPE_ODD_SHIFT))&HPSMI_ECC_ERRINFO_SEG0_ERR_TYPE_ODD_MASK)
#define HPSMI_ECC_ERRINFO_SEG0_ERR_ODD_BANK_MASK 0x1E000000u
#define HPSMI_ECC_ERRINFO_SEG0_ERR_ODD_BANK_SHIFT 25u
#define HPSMI_ECC_ERRINFO_SEG0_ERR_ODD_BANK_WIDTH 4u
#define HPSMI_ECC_ERRINFO_SEG0_ERR_ODD_BANK(x)   (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRINFO_SEG0_ERR_ODD_BANK_SHIFT))&HPSMI_ECC_ERRINFO_SEG0_ERR_ODD_BANK_MASK)
#define HPSMI_ECC_ERRINFO_SEG0_ECC_ERR_ODD_MASK  0x20000000u
#define HPSMI_ECC_ERRINFO_SEG0_ECC_ERR_ODD_SHIFT 29u
#define HPSMI_ECC_ERRINFO_SEG0_ECC_ERR_ODD_WIDTH 1u
#define HPSMI_ECC_ERRINFO_SEG0_ECC_ERR_ODD(x)    (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERRINFO_SEG0_ECC_ERR_ODD_SHIFT))&HPSMI_ECC_ERRINFO_SEG0_ECC_ERR_ODD_MASK)
/* ECC_ERROCCURED_SEG0 Bit Fields */
#define HPSMI_ECC_ERROCCURED_SEG0_ERR_MASK       0xFFFFFFu
#define HPSMI_ECC_ERROCCURED_SEG0_ERR_SHIFT      0u
#define HPSMI_ECC_ERROCCURED_SEG0_ERR_WIDTH      24u
#define HPSMI_ECC_ERROCCURED_SEG0_ERR(x)         (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_ERROCCURED_SEG0_ERR_SHIFT))&HPSMI_ECC_ERROCCURED_SEG0_ERR_MASK)
/* ECC_SINGLE_ERRCNTR_ODD_EVEN_0 Bit Fields */
#define HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_ODD_MASK 0x3Fu
#define HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_ODD_SHIFT 0u
#define HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_ODD_WIDTH 6u
#define HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_ODD(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_ODD_SHIFT))&HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_ODD_MASK)
#define HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_ODD_MASK 0x40u
#define HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_ODD_SHIFT 6u
#define HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_ODD_WIDTH 1u
#define HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_ODD(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_ODD_SHIFT))&HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_ODD_MASK)
#define HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_EVEN_MASK 0x3F00u
#define HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_EVEN_SHIFT 8u
#define HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_EVEN_WIDTH 6u
#define HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_EVEN(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_EVEN_SHIFT))&HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_EVEN_MASK)
#define HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_EVEN_MASK 0x4000u
#define HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_EVEN_SHIFT 14u
#define HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_EVEN_WIDTH 1u
#define HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_EVEN(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_EVEN_SHIFT))&HPSMI_ECC_SINGLE_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_EVEN_MASK)
/* ECC_UNCORR_ERRCNTR_ODD_EVEN_0 Bit Fields */
#define HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_ODD_MASK 0x3Fu
#define HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_ODD_SHIFT 0u
#define HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_ODD_WIDTH 6u
#define HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_ODD(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_ODD_SHIFT))&HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_ODD_MASK)
#define HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_ODD_MASK 0x40u
#define HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_ODD_SHIFT 6u
#define HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_ODD_WIDTH 1u
#define HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_ODD(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_ODD_SHIFT))&HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_ODD_MASK)
#define HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_EVEN_MASK 0x3F00u
#define HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_EVEN_SHIFT 8u
#define HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_EVEN_WIDTH 6u
#define HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_EVEN(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_EVEN_SHIFT))&HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_CNTR_EVEN_MASK)
#define HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_EVEN_MASK 0x4000u
#define HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_EVEN_SHIFT 14u
#define HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_EVEN_WIDTH 1u
#define HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_EVEN(x) (((uint32_t)(((uint32_t)(x))<<HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_EVEN_SHIFT))&HPSMI_ECC_UNCORR_ERRCNTR_ODD_EVEN_0_ECC_ERR_OVF_EVEN_MASK)
/* PM_ERROR_0 Bit Fields */
#define HPSMI_PM_ERROR_0_ERROR_BANK_MASK         0xFFFu
#define HPSMI_PM_ERROR_0_ERROR_BANK_SHIFT        0u
#define HPSMI_PM_ERROR_0_ERROR_BANK_WIDTH        12u
#define HPSMI_PM_ERROR_0_ERROR_BANK(x)           (((uint32_t)(((uint32_t)(x))<<HPSMI_PM_ERROR_0_ERROR_BANK_SHIFT))&HPSMI_PM_ERROR_0_ERROR_BANK_MASK)
/* PM_ERROR_ADDR_0 Bit Fields */
#define HPSMI_PM_ERROR_ADDR_0_ERROR_ADDR_MASK    0xFFFFFFFFu
#define HPSMI_PM_ERROR_ADDR_0_ERROR_ADDR_SHIFT   0u
#define HPSMI_PM_ERROR_ADDR_0_ERROR_ADDR_WIDTH   32u
#define HPSMI_PM_ERROR_ADDR_0_ERROR_ADDR(x)      (((uint32_t)(((uint32_t)(x))<<HPSMI_PM_ERROR_ADDR_0_ERROR_ADDR_SHIFT))&HPSMI_PM_ERROR_ADDR_0_ERROR_ADDR_MASK)
/* PM_ERROR_INFO_0 Bit Fields */
#define HPSMI_PM_ERROR_INFO_0_BANKNUM_MASK       0xFu
#define HPSMI_PM_ERROR_INFO_0_BANKNUM_SHIFT      0u
#define HPSMI_PM_ERROR_INFO_0_BANKNUM_WIDTH      4u
#define HPSMI_PM_ERROR_INFO_0_BANKNUM(x)         (((uint32_t)(((uint32_t)(x))<<HPSMI_PM_ERROR_INFO_0_BANKNUM_SHIFT))&HPSMI_PM_ERROR_INFO_0_BANKNUM_MASK)
#define HPSMI_PM_ERROR_INFO_0_PM_ERR_MASK        0x10u
#define HPSMI_PM_ERROR_INFO_0_PM_ERR_SHIFT       4u
#define HPSMI_PM_ERROR_INFO_0_PM_ERR_WIDTH       1u
#define HPSMI_PM_ERROR_INFO_0_PM_ERR(x)          (((uint32_t)(((uint32_t)(x))<<HPSMI_PM_ERROR_INFO_0_PM_ERR_SHIFT))&HPSMI_PM_ERROR_INFO_0_PM_ERR_MASK)
#define HPSMI_PM_ERROR_INFO_0_TRANSACTION_ID_MASK 0xFFFFF800u
#define HPSMI_PM_ERROR_INFO_0_TRANSACTION_ID_SHIFT 11u
#define HPSMI_PM_ERROR_INFO_0_TRANSACTION_ID_WIDTH 21u
#define HPSMI_PM_ERROR_INFO_0_TRANSACTION_ID(x)  (((uint32_t)(((uint32_t)(x))<<HPSMI_PM_ERROR_INFO_0_TRANSACTION_ID_SHIFT))&HPSMI_PM_ERROR_INFO_0_TRANSACTION_ID_MASK)
/* POWMOD_CTRL_0 Bit Fields */
#define HPSMI_POWMOD_CTRL_0_PM_BANK0_MASK        0x7u
#define HPSMI_POWMOD_CTRL_0_PM_BANK0_SHIFT       0u
#define HPSMI_POWMOD_CTRL_0_PM_BANK0_WIDTH       3u
#define HPSMI_POWMOD_CTRL_0_PM_BANK0(x)          (((uint32_t)(((uint32_t)(x))<<HPSMI_POWMOD_CTRL_0_PM_BANK0_SHIFT))&HPSMI_POWMOD_CTRL_0_PM_BANK0_MASK)
#define HPSMI_POWMOD_CTRL_0_PM_BANK1_MASK        0x38u
#define HPSMI_POWMOD_CTRL_0_PM_BANK1_SHIFT       3u
#define HPSMI_POWMOD_CTRL_0_PM_BANK1_WIDTH       3u
#define HPSMI_POWMOD_CTRL_0_PM_BANK1(x)          (((uint32_t)(((uint32_t)(x))<<HPSMI_POWMOD_CTRL_0_PM_BANK1_SHIFT))&HPSMI_POWMOD_CTRL_0_PM_BANK1_MASK)
#define HPSMI_POWMOD_CTRL_0_PM_BANK2_MASK        0x700u
#define HPSMI_POWMOD_CTRL_0_PM_BANK2_SHIFT       8u
#define HPSMI_POWMOD_CTRL_0_PM_BANK2_WIDTH       3u
#define HPSMI_POWMOD_CTRL_0_PM_BANK2(x)          (((uint32_t)(((uint32_t)(x))<<HPSMI_POWMOD_CTRL_0_PM_BANK2_SHIFT))&HPSMI_POWMOD_CTRL_0_PM_BANK2_MASK)
#define HPSMI_POWMOD_CTRL_0_PM_BANK3_MASK        0x3800u
#define HPSMI_POWMOD_CTRL_0_PM_BANK3_SHIFT       11u
#define HPSMI_POWMOD_CTRL_0_PM_BANK3_WIDTH       3u
#define HPSMI_POWMOD_CTRL_0_PM_BANK3(x)          (((uint32_t)(((uint32_t)(x))<<HPSMI_POWMOD_CTRL_0_PM_BANK3_SHIFT))&HPSMI_POWMOD_CTRL_0_PM_BANK3_MASK)
#define HPSMI_POWMOD_CTRL_0_PM_BANK4_MASK        0x70000u
#define HPSMI_POWMOD_CTRL_0_PM_BANK4_SHIFT       16u
#define HPSMI_POWMOD_CTRL_0_PM_BANK4_WIDTH       3u
#define HPSMI_POWMOD_CTRL_0_PM_BANK4(x)          (((uint32_t)(((uint32_t)(x))<<HPSMI_POWMOD_CTRL_0_PM_BANK4_SHIFT))&HPSMI_POWMOD_CTRL_0_PM_BANK4_MASK)
#define HPSMI_POWMOD_CTRL_0_PM_BANK5_MASK        0x380000u
#define HPSMI_POWMOD_CTRL_0_PM_BANK5_SHIFT       19u
#define HPSMI_POWMOD_CTRL_0_PM_BANK5_WIDTH       3u
#define HPSMI_POWMOD_CTRL_0_PM_BANK5(x)          (((uint32_t)(((uint32_t)(x))<<HPSMI_POWMOD_CTRL_0_PM_BANK5_SHIFT))&HPSMI_POWMOD_CTRL_0_PM_BANK5_MASK)
#define HPSMI_POWMOD_CTRL_0_PM_BANK6_MASK        0x7000000u
#define HPSMI_POWMOD_CTRL_0_PM_BANK6_SHIFT       24u
#define HPSMI_POWMOD_CTRL_0_PM_BANK6_WIDTH       3u
#define HPSMI_POWMOD_CTRL_0_PM_BANK6(x)          (((uint32_t)(((uint32_t)(x))<<HPSMI_POWMOD_CTRL_0_PM_BANK6_SHIFT))&HPSMI_POWMOD_CTRL_0_PM_BANK6_MASK)
#define HPSMI_POWMOD_CTRL_0_PM_BANK7_MASK        0x38000000u
#define HPSMI_POWMOD_CTRL_0_PM_BANK7_SHIFT       27u
#define HPSMI_POWMOD_CTRL_0_PM_BANK7_WIDTH       3u
#define HPSMI_POWMOD_CTRL_0_PM_BANK7(x)          (((uint32_t)(((uint32_t)(x))<<HPSMI_POWMOD_CTRL_0_PM_BANK7_SHIFT))&HPSMI_POWMOD_CTRL_0_PM_BANK7_MASK)
/* POWMOD_CTRL_0_8_11 Bit Fields */
#define HPSMI_POWMOD_CTRL_0_8_11_PM_BANK8_MASK   0x7u
#define HPSMI_POWMOD_CTRL_0_8_11_PM_BANK8_SHIFT  0u
#define HPSMI_POWMOD_CTRL_0_8_11_PM_BANK8_WIDTH  3u
#define HPSMI_POWMOD_CTRL_0_8_11_PM_BANK8(x)     (((uint32_t)(((uint32_t)(x))<<HPSMI_POWMOD_CTRL_0_8_11_PM_BANK8_SHIFT))&HPSMI_POWMOD_CTRL_0_8_11_PM_BANK8_MASK)
#define HPSMI_POWMOD_CTRL_0_8_11_PM_BANK9_MASK   0x38u
#define HPSMI_POWMOD_CTRL_0_8_11_PM_BANK9_SHIFT  3u
#define HPSMI_POWMOD_CTRL_0_8_11_PM_BANK9_WIDTH  3u
#define HPSMI_POWMOD_CTRL_0_8_11_PM_BANK9(x)     (((uint32_t)(((uint32_t)(x))<<HPSMI_POWMOD_CTRL_0_8_11_PM_BANK9_SHIFT))&HPSMI_POWMOD_CTRL_0_8_11_PM_BANK9_MASK)
#define HPSMI_POWMOD_CTRL_0_8_11_PM_BANK10_MASK  0x700u
#define HPSMI_POWMOD_CTRL_0_8_11_PM_BANK10_SHIFT 8u
#define HPSMI_POWMOD_CTRL_0_8_11_PM_BANK10_WIDTH 3u
#define HPSMI_POWMOD_CTRL_0_8_11_PM_BANK10(x)    (((uint32_t)(((uint32_t)(x))<<HPSMI_POWMOD_CTRL_0_8_11_PM_BANK10_SHIFT))&HPSMI_POWMOD_CTRL_0_8_11_PM_BANK10_MASK)
#define HPSMI_POWMOD_CTRL_0_8_11_PM_BANK11_MASK  0x3800u
#define HPSMI_POWMOD_CTRL_0_8_11_PM_BANK11_SHIFT 11u
#define HPSMI_POWMOD_CTRL_0_8_11_PM_BANK11_WIDTH 3u
#define HPSMI_POWMOD_CTRL_0_8_11_PM_BANK11(x)    (((uint32_t)(((uint32_t)(x))<<HPSMI_POWMOD_CTRL_0_8_11_PM_BANK11_SHIFT))&HPSMI_POWMOD_CTRL_0_8_11_PM_BANK11_MASK)
/* POWMOD_CTRL_1_2 Bit Fields */
#define HPSMI_POWMOD_CTRL_1_2_PM_SEG1_MASK       0x7u
#define HPSMI_POWMOD_CTRL_1_2_PM_SEG1_SHIFT      0u
#define HPSMI_POWMOD_CTRL_1_2_PM_SEG1_WIDTH      3u
#define HPSMI_POWMOD_CTRL_1_2_PM_SEG1(x)         (((uint32_t)(((uint32_t)(x))<<HPSMI_POWMOD_CTRL_1_2_PM_SEG1_SHIFT))&HPSMI_POWMOD_CTRL_1_2_PM_SEG1_MASK)
#define HPSMI_POWMOD_CTRL_1_2_PM_SEG2_MASK       0x38u
#define HPSMI_POWMOD_CTRL_1_2_PM_SEG2_SHIFT      3u
#define HPSMI_POWMOD_CTRL_1_2_PM_SEG2_WIDTH      3u
#define HPSMI_POWMOD_CTRL_1_2_PM_SEG2(x)         (((uint32_t)(((uint32_t)(x))<<HPSMI_POWMOD_CTRL_1_2_PM_SEG2_SHIFT))&HPSMI_POWMOD_CTRL_1_2_PM_SEG2_MASK)
/* QOS_PRIORITY Bit Fields */
#define HPSMI_QOS_PRIORITY_M_CORES_AXI4_MASK     0xFu
#define HPSMI_QOS_PRIORITY_M_CORES_AXI4_SHIFT    0u
#define HPSMI_QOS_PRIORITY_M_CORES_AXI4_WIDTH    4u
#define HPSMI_QOS_PRIORITY_M_CORES_AXI4(x)       (((uint32_t)(((uint32_t)(x))<<HPSMI_QOS_PRIORITY_M_CORES_AXI4_SHIFT))&HPSMI_QOS_PRIORITY_M_CORES_AXI4_MASK)
#define HPSMI_QOS_PRIORITY_M_FASTDMA_AXI6_MASK   0xF00u
#define HPSMI_QOS_PRIORITY_M_FASTDMA_AXI6_SHIFT  8u
#define HPSMI_QOS_PRIORITY_M_FASTDMA_AXI6_WIDTH  4u
#define HPSMI_QOS_PRIORITY_M_FASTDMA_AXI6(x)     (((uint32_t)(((uint32_t)(x))<<HPSMI_QOS_PRIORITY_M_FASTDMA_AXI6_SHIFT))&HPSMI_QOS_PRIORITY_M_FASTDMA_AXI6_MASK)
#define HPSMI_QOS_PRIORITY_M_LSB0_AXI7_MASK      0xF000u
#define HPSMI_QOS_PRIORITY_M_LSB0_AXI7_SHIFT     12u
#define HPSMI_QOS_PRIORITY_M_LSB0_AXI7_WIDTH     4u
#define HPSMI_QOS_PRIORITY_M_LSB0_AXI7(x)        (((uint32_t)(((uint32_t)(x))<<HPSMI_QOS_PRIORITY_M_LSB0_AXI7_SHIFT))&HPSMI_QOS_PRIORITY_M_LSB0_AXI7_MASK)
#define HPSMI_QOS_PRIORITY_M_LSB1_AXI8_MASK      0xF0000u
#define HPSMI_QOS_PRIORITY_M_LSB1_AXI8_SHIFT     16u
#define HPSMI_QOS_PRIORITY_M_LSB1_AXI8_WIDTH     4u
#define HPSMI_QOS_PRIORITY_M_LSB1_AXI8(x)        (((uint32_t)(((uint32_t)(x))<<HPSMI_QOS_PRIORITY_M_LSB1_AXI8_SHIFT))&HPSMI_QOS_PRIORITY_M_LSB1_AXI8_MASK)
#define HPSMI_QOS_PRIORITY_WEN_MASK              0x80000000u
#define HPSMI_QOS_PRIORITY_WEN_SHIFT             31u
#define HPSMI_QOS_PRIORITY_WEN_WIDTH             1u
#define HPSMI_QOS_PRIORITY_WEN(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_QOS_PRIORITY_WEN_SHIFT))&HPSMI_QOS_PRIORITY_WEN_MASK)
/* IPCGE Bit Fields */
#define HPSMI_IPCGE_GLBL_EN_MASK                 0x1u
#define HPSMI_IPCGE_GLBL_EN_SHIFT                0u
#define HPSMI_IPCGE_GLBL_EN_WIDTH                1u
#define HPSMI_IPCGE_GLBL_EN(x)                   (((uint32_t)(((uint32_t)(x))<<HPSMI_IPCGE_GLBL_EN_SHIFT))&HPSMI_IPCGE_GLBL_EN_MASK)
/* IPRCE Bit Fields */
#define HPSMI_IPRCE_IPRE_AXI0_MASK               0x1u
#define HPSMI_IPRCE_IPRE_AXI0_SHIFT              0u
#define HPSMI_IPRCE_IPRE_AXI0_WIDTH              1u
#define HPSMI_IPRCE_IPRE_AXI0(x)                 (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRCE_IPRE_AXI0_SHIFT))&HPSMI_IPRCE_IPRE_AXI0_MASK)
#define HPSMI_IPRCE_IPRE_AXI1_MASK               0x2u
#define HPSMI_IPRCE_IPRE_AXI1_SHIFT              1u
#define HPSMI_IPRCE_IPRE_AXI1_WIDTH              1u
#define HPSMI_IPRCE_IPRE_AXI1(x)                 (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRCE_IPRE_AXI1_SHIFT))&HPSMI_IPRCE_IPRE_AXI1_MASK)
#define HPSMI_IPRCE_IPRE_AXI2_MASK               0x4u
#define HPSMI_IPRCE_IPRE_AXI2_SHIFT              2u
#define HPSMI_IPRCE_IPRE_AXI2_WIDTH              1u
#define HPSMI_IPRCE_IPRE_AXI2(x)                 (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRCE_IPRE_AXI2_SHIFT))&HPSMI_IPRCE_IPRE_AXI2_MASK)
#define HPSMI_IPRCE_IPRE_AXI3_MASK               0x8u
#define HPSMI_IPRCE_IPRE_AXI3_SHIFT              3u
#define HPSMI_IPRCE_IPRE_AXI3_WIDTH              1u
#define HPSMI_IPRCE_IPRE_AXI3(x)                 (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRCE_IPRE_AXI3_SHIFT))&HPSMI_IPRCE_IPRE_AXI3_MASK)
#define HPSMI_IPRCE_IPRE_AXI4_MASK               0x100u
#define HPSMI_IPRCE_IPRE_AXI4_SHIFT              8u
#define HPSMI_IPRCE_IPRE_AXI4_WIDTH              1u
#define HPSMI_IPRCE_IPRE_AXI4(x)                 (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRCE_IPRE_AXI4_SHIFT))&HPSMI_IPRCE_IPRE_AXI4_MASK)
#define HPSMI_IPRCE_IPRE_AXI5_MASK               0x200u
#define HPSMI_IPRCE_IPRE_AXI5_SHIFT              9u
#define HPSMI_IPRCE_IPRE_AXI5_WIDTH              1u
#define HPSMI_IPRCE_IPRE_AXI5(x)                 (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRCE_IPRE_AXI5_SHIFT))&HPSMI_IPRCE_IPRE_AXI5_MASK)
#define HPSMI_IPRCE_IPRE_AXI6_MASK               0x400u
#define HPSMI_IPRCE_IPRE_AXI6_SHIFT              10u
#define HPSMI_IPRCE_IPRE_AXI6_WIDTH              1u
#define HPSMI_IPRCE_IPRE_AXI6(x)                 (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRCE_IPRE_AXI6_SHIFT))&HPSMI_IPRCE_IPRE_AXI6_MASK)
#define HPSMI_IPRCE_IPRE_LSB2AXI0_MASK           0x10000u
#define HPSMI_IPRCE_IPRE_LSB2AXI0_SHIFT          16u
#define HPSMI_IPRCE_IPRE_LSB2AXI0_WIDTH          1u
#define HPSMI_IPRCE_IPRE_LSB2AXI0(x)             (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRCE_IPRE_LSB2AXI0_SHIFT))&HPSMI_IPRCE_IPRE_LSB2AXI0_MASK)
#define HPSMI_IPRCE_IPRE_LSB2AXI1_MASK           0x20000u
#define HPSMI_IPRCE_IPRE_LSB2AXI1_SHIFT          17u
#define HPSMI_IPRCE_IPRE_LSB2AXI1_WIDTH          1u
#define HPSMI_IPRCE_IPRE_LSB2AXI1(x)             (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRCE_IPRE_LSB2AXI1_SHIFT))&HPSMI_IPRCE_IPRE_LSB2AXI1_MASK)
/* IPWCE Bit Fields */
#define HPSMI_IPWCE_IPWCE_S0_MASK                0x1u
#define HPSMI_IPWCE_IPWCE_S0_SHIFT               0u
#define HPSMI_IPWCE_IPWCE_S0_WIDTH               1u
#define HPSMI_IPWCE_IPWCE_S0(x)                  (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWCE_IPWCE_S0_SHIFT))&HPSMI_IPWCE_IPWCE_S0_MASK)
#define HPSMI_IPWCE_IPWCE_S1_MASK                0x2u
#define HPSMI_IPWCE_IPWCE_S1_SHIFT               1u
#define HPSMI_IPWCE_IPWCE_S1_WIDTH               1u
#define HPSMI_IPWCE_IPWCE_S1(x)                  (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWCE_IPWCE_S1_SHIFT))&HPSMI_IPWCE_IPWCE_S1_MASK)
#define HPSMI_IPWCE_IPWCE_S2_MASK                0x4u
#define HPSMI_IPWCE_IPWCE_S2_SHIFT               2u
#define HPSMI_IPWCE_IPWCE_S2_WIDTH               1u
#define HPSMI_IPWCE_IPWCE_S2(x)                  (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWCE_IPWCE_S2_SHIFT))&HPSMI_IPWCE_IPWCE_S2_MASK)
#define HPSMI_IPWCE_IPWCE_S3_MASK                0x8u
#define HPSMI_IPWCE_IPWCE_S3_SHIFT               3u
#define HPSMI_IPWCE_IPWCE_S3_WIDTH               1u
#define HPSMI_IPWCE_IPWCE_S3(x)                  (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWCE_IPWCE_S3_SHIFT))&HPSMI_IPWCE_IPWCE_S3_MASK)
#define HPSMI_IPWCE_IPWCE_S4_MASK                0x10u
#define HPSMI_IPWCE_IPWCE_S4_SHIFT               4u
#define HPSMI_IPWCE_IPWCE_S4_WIDTH               1u
#define HPSMI_IPWCE_IPWCE_S4(x)                  (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWCE_IPWCE_S4_SHIFT))&HPSMI_IPWCE_IPWCE_S4_MASK)
#define HPSMI_IPWCE_IPWCE_S5_MASK                0x20u
#define HPSMI_IPWCE_IPWCE_S5_SHIFT               5u
#define HPSMI_IPWCE_IPWCE_S5_WIDTH               1u
#define HPSMI_IPWCE_IPWCE_S5(x)                  (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWCE_IPWCE_S5_SHIFT))&HPSMI_IPWCE_IPWCE_S5_MASK)
#define HPSMI_IPWCE_IPWCE_S6_MASK                0x40u
#define HPSMI_IPWCE_IPWCE_S6_SHIFT               6u
#define HPSMI_IPWCE_IPWCE_S6_WIDTH               1u
#define HPSMI_IPWCE_IPWCE_S6(x)                  (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWCE_IPWCE_S6_SHIFT))&HPSMI_IPWCE_IPWCE_S6_MASK)
#define HPSMI_IPWCE_IPWCE_S7_MASK                0x80u
#define HPSMI_IPWCE_IPWCE_S7_SHIFT               7u
#define HPSMI_IPWCE_IPWCE_S7_WIDTH               1u
#define HPSMI_IPWCE_IPWCE_S7(x)                  (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWCE_IPWCE_S7_SHIFT))&HPSMI_IPWCE_IPWCE_S7_MASK)
#define HPSMI_IPWCE_IPWCE_S8_MASK                0x100u
#define HPSMI_IPWCE_IPWCE_S8_SHIFT               8u
#define HPSMI_IPWCE_IPWCE_S8_WIDTH               1u
#define HPSMI_IPWCE_IPWCE_S8(x)                  (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWCE_IPWCE_S8_SHIFT))&HPSMI_IPWCE_IPWCE_S8_MASK)
#define HPSMI_IPWCE_IPWCE_S9_MASK                0x200u
#define HPSMI_IPWCE_IPWCE_S9_SHIFT               9u
#define HPSMI_IPWCE_IPWCE_S9_WIDTH               1u
#define HPSMI_IPWCE_IPWCE_S9(x)                  (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWCE_IPWCE_S9_SHIFT))&HPSMI_IPWCE_IPWCE_S9_MASK)
#define HPSMI_IPWCE_IPWCE_S10_MASK               0x400u
#define HPSMI_IPWCE_IPWCE_S10_SHIFT              10u
#define HPSMI_IPWCE_IPWCE_S10_WIDTH              1u
#define HPSMI_IPWCE_IPWCE_S10(x)                 (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWCE_IPWCE_S10_SHIFT))&HPSMI_IPWCE_IPWCE_S10_MASK)
#define HPSMI_IPWCE_IPWCE_S11_MASK               0x800u
#define HPSMI_IPWCE_IPWCE_S11_SHIFT              11u
#define HPSMI_IPWCE_IPWCE_S11_WIDTH              1u
#define HPSMI_IPWCE_IPWCE_S11(x)                 (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWCE_IPWCE_S11_SHIFT))&HPSMI_IPWCE_IPWCE_S11_MASK)
#define HPSMI_IPWCE_IPWCE_S12_MASK               0x10000u
#define HPSMI_IPWCE_IPWCE_S12_SHIFT              16u
#define HPSMI_IPWCE_IPWCE_S12_WIDTH              1u
#define HPSMI_IPWCE_IPWCE_S12(x)                 (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWCE_IPWCE_S12_SHIFT))&HPSMI_IPWCE_IPWCE_S12_MASK)
#define HPSMI_IPWCE_IPWCE_S13_MASK               0x20000u
#define HPSMI_IPWCE_IPWCE_S13_SHIFT              17u
#define HPSMI_IPWCE_IPWCE_S13_WIDTH              1u
#define HPSMI_IPWCE_IPWCE_S13(x)                 (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWCE_IPWCE_S13_SHIFT))&HPSMI_IPWCE_IPWCE_S13_MASK)
/* IPWACE Bit Fields */
#define HPSMI_IPWACE_IPWACE_S0_MASK              0x1u
#define HPSMI_IPWACE_IPWACE_S0_SHIFT             0u
#define HPSMI_IPWACE_IPWACE_S0_WIDTH             1u
#define HPSMI_IPWACE_IPWACE_S0(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWACE_IPWACE_S0_SHIFT))&HPSMI_IPWACE_IPWACE_S0_MASK)
#define HPSMI_IPWACE_IPWACE_S1_MASK              0x2u
#define HPSMI_IPWACE_IPWACE_S1_SHIFT             1u
#define HPSMI_IPWACE_IPWACE_S1_WIDTH             1u
#define HPSMI_IPWACE_IPWACE_S1(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWACE_IPWACE_S1_SHIFT))&HPSMI_IPWACE_IPWACE_S1_MASK)
#define HPSMI_IPWACE_IPWACE_S2_MASK              0x4u
#define HPSMI_IPWACE_IPWACE_S2_SHIFT             2u
#define HPSMI_IPWACE_IPWACE_S2_WIDTH             1u
#define HPSMI_IPWACE_IPWACE_S2(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWACE_IPWACE_S2_SHIFT))&HPSMI_IPWACE_IPWACE_S2_MASK)
#define HPSMI_IPWACE_IPWACE_S3_MASK              0x8u
#define HPSMI_IPWACE_IPWACE_S3_SHIFT             3u
#define HPSMI_IPWACE_IPWACE_S3_WIDTH             1u
#define HPSMI_IPWACE_IPWACE_S3(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWACE_IPWACE_S3_SHIFT))&HPSMI_IPWACE_IPWACE_S3_MASK)
#define HPSMI_IPWACE_IPWACE_S4_MASK              0x10u
#define HPSMI_IPWACE_IPWACE_S4_SHIFT             4u
#define HPSMI_IPWACE_IPWACE_S4_WIDTH             1u
#define HPSMI_IPWACE_IPWACE_S4(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWACE_IPWACE_S4_SHIFT))&HPSMI_IPWACE_IPWACE_S4_MASK)
#define HPSMI_IPWACE_IPWACE_S5_MASK              0x20u
#define HPSMI_IPWACE_IPWACE_S5_SHIFT             5u
#define HPSMI_IPWACE_IPWACE_S5_WIDTH             1u
#define HPSMI_IPWACE_IPWACE_S5(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWACE_IPWACE_S5_SHIFT))&HPSMI_IPWACE_IPWACE_S5_MASK)
#define HPSMI_IPWACE_IPWACE_S6_MASK              0x40u
#define HPSMI_IPWACE_IPWACE_S6_SHIFT             6u
#define HPSMI_IPWACE_IPWACE_S6_WIDTH             1u
#define HPSMI_IPWACE_IPWACE_S6(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWACE_IPWACE_S6_SHIFT))&HPSMI_IPWACE_IPWACE_S6_MASK)
#define HPSMI_IPWACE_IPWACE_S7_MASK              0x80u
#define HPSMI_IPWACE_IPWACE_S7_SHIFT             7u
#define HPSMI_IPWACE_IPWACE_S7_WIDTH             1u
#define HPSMI_IPWACE_IPWACE_S7(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWACE_IPWACE_S7_SHIFT))&HPSMI_IPWACE_IPWACE_S7_MASK)
#define HPSMI_IPWACE_IPWACE_S8_MASK              0x100u
#define HPSMI_IPWACE_IPWACE_S8_SHIFT             8u
#define HPSMI_IPWACE_IPWACE_S8_WIDTH             1u
#define HPSMI_IPWACE_IPWACE_S8(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWACE_IPWACE_S8_SHIFT))&HPSMI_IPWACE_IPWACE_S8_MASK)
#define HPSMI_IPWACE_IPWACE_S9_MASK              0x200u
#define HPSMI_IPWACE_IPWACE_S9_SHIFT             9u
#define HPSMI_IPWACE_IPWACE_S9_WIDTH             1u
#define HPSMI_IPWACE_IPWACE_S9(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWACE_IPWACE_S9_SHIFT))&HPSMI_IPWACE_IPWACE_S9_MASK)
#define HPSMI_IPWACE_IPWACE_S10_MASK             0x400u
#define HPSMI_IPWACE_IPWACE_S10_SHIFT            10u
#define HPSMI_IPWACE_IPWACE_S10_WIDTH            1u
#define HPSMI_IPWACE_IPWACE_S10(x)               (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWACE_IPWACE_S10_SHIFT))&HPSMI_IPWACE_IPWACE_S10_MASK)
#define HPSMI_IPWACE_IPWACE_S11_MASK             0x800u
#define HPSMI_IPWACE_IPWACE_S11_SHIFT            11u
#define HPSMI_IPWACE_IPWACE_S11_WIDTH            1u
#define HPSMI_IPWACE_IPWACE_S11(x)               (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWACE_IPWACE_S11_SHIFT))&HPSMI_IPWACE_IPWACE_S11_MASK)
#define HPSMI_IPWACE_IPWACE_S12_MASK             0x10000u
#define HPSMI_IPWACE_IPWACE_S12_SHIFT            16u
#define HPSMI_IPWACE_IPWACE_S12_WIDTH            1u
#define HPSMI_IPWACE_IPWACE_S12(x)               (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWACE_IPWACE_S12_SHIFT))&HPSMI_IPWACE_IPWACE_S12_MASK)
#define HPSMI_IPWACE_IPWACE_S13_MASK             0x20000u
#define HPSMI_IPWACE_IPWACE_S13_SHIFT            17u
#define HPSMI_IPWACE_IPWACE_S13_WIDTH            1u
#define HPSMI_IPWACE_IPWACE_S13(x)               (((uint32_t)(((uint32_t)(x))<<HPSMI_IPWACE_IPWACE_S13_SHIFT))&HPSMI_IPWACE_IPWACE_S13_MASK)
/* IPRACE Bit Fields */
#define HPSMI_IPRACE_IPRACE_S0_MASK              0x1u
#define HPSMI_IPRACE_IPRACE_S0_SHIFT             0u
#define HPSMI_IPRACE_IPRACE_S0_WIDTH             1u
#define HPSMI_IPRACE_IPRACE_S0(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRACE_IPRACE_S0_SHIFT))&HPSMI_IPRACE_IPRACE_S0_MASK)
#define HPSMI_IPRACE_IPRACE_S1_MASK              0x2u
#define HPSMI_IPRACE_IPRACE_S1_SHIFT             1u
#define HPSMI_IPRACE_IPRACE_S1_WIDTH             1u
#define HPSMI_IPRACE_IPRACE_S1(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRACE_IPRACE_S1_SHIFT))&HPSMI_IPRACE_IPRACE_S1_MASK)
#define HPSMI_IPRACE_IPRACE_S2_MASK              0x4u
#define HPSMI_IPRACE_IPRACE_S2_SHIFT             2u
#define HPSMI_IPRACE_IPRACE_S2_WIDTH             1u
#define HPSMI_IPRACE_IPRACE_S2(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRACE_IPRACE_S2_SHIFT))&HPSMI_IPRACE_IPRACE_S2_MASK)
#define HPSMI_IPRACE_IPRACE_S3_MASK              0x8u
#define HPSMI_IPRACE_IPRACE_S3_SHIFT             3u
#define HPSMI_IPRACE_IPRACE_S3_WIDTH             1u
#define HPSMI_IPRACE_IPRACE_S3(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRACE_IPRACE_S3_SHIFT))&HPSMI_IPRACE_IPRACE_S3_MASK)
#define HPSMI_IPRACE_IPRACE_S4_MASK              0x10u
#define HPSMI_IPRACE_IPRACE_S4_SHIFT             4u
#define HPSMI_IPRACE_IPRACE_S4_WIDTH             1u
#define HPSMI_IPRACE_IPRACE_S4(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRACE_IPRACE_S4_SHIFT))&HPSMI_IPRACE_IPRACE_S4_MASK)
#define HPSMI_IPRACE_IPRACE_S5_MASK              0x20u
#define HPSMI_IPRACE_IPRACE_S5_SHIFT             5u
#define HPSMI_IPRACE_IPRACE_S5_WIDTH             1u
#define HPSMI_IPRACE_IPRACE_S5(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRACE_IPRACE_S5_SHIFT))&HPSMI_IPRACE_IPRACE_S5_MASK)
#define HPSMI_IPRACE_IPRACE_S6_MASK              0x40u
#define HPSMI_IPRACE_IPRACE_S6_SHIFT             6u
#define HPSMI_IPRACE_IPRACE_S6_WIDTH             1u
#define HPSMI_IPRACE_IPRACE_S6(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRACE_IPRACE_S6_SHIFT))&HPSMI_IPRACE_IPRACE_S6_MASK)
#define HPSMI_IPRACE_IPRACE_S7_MASK              0x80u
#define HPSMI_IPRACE_IPRACE_S7_SHIFT             7u
#define HPSMI_IPRACE_IPRACE_S7_WIDTH             1u
#define HPSMI_IPRACE_IPRACE_S7(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRACE_IPRACE_S7_SHIFT))&HPSMI_IPRACE_IPRACE_S7_MASK)
#define HPSMI_IPRACE_IPRACE_S8_MASK              0x100u
#define HPSMI_IPRACE_IPRACE_S8_SHIFT             8u
#define HPSMI_IPRACE_IPRACE_S8_WIDTH             1u
#define HPSMI_IPRACE_IPRACE_S8(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRACE_IPRACE_S8_SHIFT))&HPSMI_IPRACE_IPRACE_S8_MASK)
#define HPSMI_IPRACE_IPRACE_S9_MASK              0x200u
#define HPSMI_IPRACE_IPRACE_S9_SHIFT             9u
#define HPSMI_IPRACE_IPRACE_S9_WIDTH             1u
#define HPSMI_IPRACE_IPRACE_S9(x)                (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRACE_IPRACE_S9_SHIFT))&HPSMI_IPRACE_IPRACE_S9_MASK)
#define HPSMI_IPRACE_IPRACE_S10_MASK             0x400u
#define HPSMI_IPRACE_IPRACE_S10_SHIFT            10u
#define HPSMI_IPRACE_IPRACE_S10_WIDTH            1u
#define HPSMI_IPRACE_IPRACE_S10(x)               (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRACE_IPRACE_S10_SHIFT))&HPSMI_IPRACE_IPRACE_S10_MASK)
#define HPSMI_IPRACE_IPRACE_S11_MASK             0x800u
#define HPSMI_IPRACE_IPRACE_S11_SHIFT            11u
#define HPSMI_IPRACE_IPRACE_S11_WIDTH            1u
#define HPSMI_IPRACE_IPRACE_S11(x)               (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRACE_IPRACE_S11_SHIFT))&HPSMI_IPRACE_IPRACE_S11_MASK)
#define HPSMI_IPRACE_IPRACE_S12_MASK             0x10000u
#define HPSMI_IPRACE_IPRACE_S12_SHIFT            16u
#define HPSMI_IPRACE_IPRACE_S12_WIDTH            1u
#define HPSMI_IPRACE_IPRACE_S12(x)               (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRACE_IPRACE_S12_SHIFT))&HPSMI_IPRACE_IPRACE_S12_MASK)
#define HPSMI_IPRACE_IPRACE_S13_MASK             0x20000u
#define HPSMI_IPRACE_IPRACE_S13_SHIFT            17u
#define HPSMI_IPRACE_IPRACE_S13_WIDTH            1u
#define HPSMI_IPRACE_IPRACE_S13(x)               (((uint32_t)(((uint32_t)(x))<<HPSMI_IPRACE_IPRACE_S13_SHIFT))&HPSMI_IPRACE_IPRACE_S13_MASK)
/* IPCGIE Bit Fields */
#define HPSMI_IPCGIE_GLBL_IEN_MASK               0x1u
#define HPSMI_IPCGIE_GLBL_IEN_SHIFT              0u
#define HPSMI_IPCGIE_GLBL_IEN_WIDTH              1u
#define HPSMI_IPCGIE_GLBL_IEN(x)                 (((uint32_t)(((uint32_t)(x))<<HPSMI_IPCGIE_GLBL_IEN_SHIFT))&HPSMI_IPCGIE_GLBL_IEN_MASK)

/*!
 * @}
 */ /* end of group HPSMI_Register_Masks */


/*!
 * @}
 */ /* end of group HPSMI_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- I2C Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup I2C_Peripheral_Access_Layer I2C Peripheral Access Layer
 * @{
 */


/** I2C - Size of Registers Arrays */

/** I2C - Register Layout Typedef */
typedef struct {
  __IO uint8_t IBAD;                               /**< I2C Bus Address Register, offset: 0x0 */
  __IO uint8_t IBFD;                               /**< I2C Bus Frequency Divider Register, offset: 0x1 */
  __IO uint8_t IBCR;                               /**< I2C Bus Control Register, offset: 0x2 */
  __IO uint8_t IBSR;                               /**< I2C Bus Status Register, offset: 0x3 */
  __IO uint8_t IBDR;                               /**< I2C Bus Data I/O Register, offset: 0x4 */
  __IO uint8_t IBIC;                               /**< I2C Bus Interrupt Config Register, offset: 0x5 */
  __IO uint8_t IBDBG;                              /**< I2C Bus Debug Register, offset: 0x6 */
} I2C_Type, *I2C_MemMapPtr;

 /** Number of instances of the I2C module. */
#define I2C_INSTANCE_COUNT                       (3u)


/* I2C - Peripheral instance base addresses */
/** Peripheral I2C_0 base address */
#define I2C_0_BASE                               (0x40051000u)
/** Peripheral I2C_0 base pointer */
#define I2C_0                                    ((I2C_Type *)I2C_0_BASE)
/** Peripheral I2C_1 base address */
#define I2C_1_BASE                               (0x400B8000u)
/** Peripheral I2C_1 base pointer */
#define I2C_1                                    ((I2C_Type *)I2C_1_BASE)
/** Peripheral I2C_2 base address */
#define I2C_2_BASE                               (0x400BA000u)
/** Peripheral I2C_2 base pointer */
#define I2C_2                                    ((I2C_Type *)I2C_2_BASE)
/** Array initializer of I2C peripheral base addresses */
#define I2C_BASE_ADDRS                           { I2C_0_BASE, I2C_1_BASE, I2C_2_BASE }
/** Array initializer of I2C peripheral base pointers */
#define I2C_BASE_PTRS                            { I2C_0, I2C_1, I2C_2 }

/* ----------------------------------------------------------------------------
   -- I2C Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup I2C_Register_Masks I2C Register Masks
 * @{
 */

/* IBAD Bit Fields */
#define I2C_IBAD_ADR_MASK                        0xFEu
#define I2C_IBAD_ADR_SHIFT                       1u
#define I2C_IBAD_ADR_WIDTH                       7u
#define I2C_IBAD_ADR(x)                          (((uint8_t)(((uint8_t)(x))<<I2C_IBAD_ADR_SHIFT))&I2C_IBAD_ADR_MASK)
/* IBFD Bit Fields */
#define I2C_IBFD_IBC_MASK                        0xFFu
#define I2C_IBFD_IBC_SHIFT                       0u
#define I2C_IBFD_IBC_WIDTH                       8u
#define I2C_IBFD_IBC(x)                          (((uint8_t)(((uint8_t)(x))<<I2C_IBFD_IBC_SHIFT))&I2C_IBFD_IBC_MASK)
/* IBCR Bit Fields */
#define I2C_IBCR_DMAEN_MASK                      0x2u
#define I2C_IBCR_DMAEN_SHIFT                     1u
#define I2C_IBCR_DMAEN_WIDTH                     1u
#define I2C_IBCR_DMAEN(x)                        (((uint8_t)(((uint8_t)(x))<<I2C_IBCR_DMAEN_SHIFT))&I2C_IBCR_DMAEN_MASK)
#define I2C_IBCR_RSTA_MASK                       0x4u
#define I2C_IBCR_RSTA_SHIFT                      2u
#define I2C_IBCR_RSTA_WIDTH                      1u
#define I2C_IBCR_RSTA(x)                         (((uint8_t)(((uint8_t)(x))<<I2C_IBCR_RSTA_SHIFT))&I2C_IBCR_RSTA_MASK)
#define I2C_IBCR_NOACK_MASK                      0x8u
#define I2C_IBCR_NOACK_SHIFT                     3u
#define I2C_IBCR_NOACK_WIDTH                     1u
#define I2C_IBCR_NOACK(x)                        (((uint8_t)(((uint8_t)(x))<<I2C_IBCR_NOACK_SHIFT))&I2C_IBCR_NOACK_MASK)
#define I2C_IBCR_TXRX_MASK                       0x10u
#define I2C_IBCR_TXRX_SHIFT                      4u
#define I2C_IBCR_TXRX_WIDTH                      1u
#define I2C_IBCR_TXRX(x)                         (((uint8_t)(((uint8_t)(x))<<I2C_IBCR_TXRX_SHIFT))&I2C_IBCR_TXRX_MASK)
#define I2C_IBCR_MSSL_MASK                       0x20u
#define I2C_IBCR_MSSL_SHIFT                      5u
#define I2C_IBCR_MSSL_WIDTH                      1u
#define I2C_IBCR_MSSL(x)                         (((uint8_t)(((uint8_t)(x))<<I2C_IBCR_MSSL_SHIFT))&I2C_IBCR_MSSL_MASK)
#define I2C_IBCR_IBIE_MASK                       0x40u
#define I2C_IBCR_IBIE_SHIFT                      6u
#define I2C_IBCR_IBIE_WIDTH                      1u
#define I2C_IBCR_IBIE(x)                         (((uint8_t)(((uint8_t)(x))<<I2C_IBCR_IBIE_SHIFT))&I2C_IBCR_IBIE_MASK)
#define I2C_IBCR_MDIS_MASK                       0x80u
#define I2C_IBCR_MDIS_SHIFT                      7u
#define I2C_IBCR_MDIS_WIDTH                      1u
#define I2C_IBCR_MDIS(x)                         (((uint8_t)(((uint8_t)(x))<<I2C_IBCR_MDIS_SHIFT))&I2C_IBCR_MDIS_MASK)
/* IBSR Bit Fields */
#define I2C_IBSR_RXAK_MASK                       0x1u
#define I2C_IBSR_RXAK_SHIFT                      0u
#define I2C_IBSR_RXAK_WIDTH                      1u
#define I2C_IBSR_RXAK(x)                         (((uint8_t)(((uint8_t)(x))<<I2C_IBSR_RXAK_SHIFT))&I2C_IBSR_RXAK_MASK)
#define I2C_IBSR_IBIF_MASK                       0x2u
#define I2C_IBSR_IBIF_SHIFT                      1u
#define I2C_IBSR_IBIF_WIDTH                      1u
#define I2C_IBSR_IBIF(x)                         (((uint8_t)(((uint8_t)(x))<<I2C_IBSR_IBIF_SHIFT))&I2C_IBSR_IBIF_MASK)
#define I2C_IBSR_SRW_MASK                        0x4u
#define I2C_IBSR_SRW_SHIFT                       2u
#define I2C_IBSR_SRW_WIDTH                       1u
#define I2C_IBSR_SRW(x)                          (((uint8_t)(((uint8_t)(x))<<I2C_IBSR_SRW_SHIFT))&I2C_IBSR_SRW_MASK)
#define I2C_IBSR_IBAL_MASK                       0x10u
#define I2C_IBSR_IBAL_SHIFT                      4u
#define I2C_IBSR_IBAL_WIDTH                      1u
#define I2C_IBSR_IBAL(x)                         (((uint8_t)(((uint8_t)(x))<<I2C_IBSR_IBAL_SHIFT))&I2C_IBSR_IBAL_MASK)
#define I2C_IBSR_IBB_MASK                        0x20u
#define I2C_IBSR_IBB_SHIFT                       5u
#define I2C_IBSR_IBB_WIDTH                       1u
#define I2C_IBSR_IBB(x)                          (((uint8_t)(((uint8_t)(x))<<I2C_IBSR_IBB_SHIFT))&I2C_IBSR_IBB_MASK)
#define I2C_IBSR_IAAS_MASK                       0x40u
#define I2C_IBSR_IAAS_SHIFT                      6u
#define I2C_IBSR_IAAS_WIDTH                      1u
#define I2C_IBSR_IAAS(x)                         (((uint8_t)(((uint8_t)(x))<<I2C_IBSR_IAAS_SHIFT))&I2C_IBSR_IAAS_MASK)
#define I2C_IBSR_TCF_MASK                        0x80u
#define I2C_IBSR_TCF_SHIFT                       7u
#define I2C_IBSR_TCF_WIDTH                       1u
#define I2C_IBSR_TCF(x)                          (((uint8_t)(((uint8_t)(x))<<I2C_IBSR_TCF_SHIFT))&I2C_IBSR_TCF_MASK)
/* IBDR Bit Fields */
#define I2C_IBDR_DATA_MASK                       0xFFu
#define I2C_IBDR_DATA_SHIFT                      0u
#define I2C_IBDR_DATA_WIDTH                      8u
#define I2C_IBDR_DATA(x)                         (((uint8_t)(((uint8_t)(x))<<I2C_IBDR_DATA_SHIFT))&I2C_IBDR_DATA_MASK)
/* IBIC Bit Fields */
#define I2C_IBIC_BYTERXIE_MASK                   0x40u
#define I2C_IBIC_BYTERXIE_SHIFT                  6u
#define I2C_IBIC_BYTERXIE_WIDTH                  1u
#define I2C_IBIC_BYTERXIE(x)                     (((uint8_t)(((uint8_t)(x))<<I2C_IBIC_BYTERXIE_SHIFT))&I2C_IBIC_BYTERXIE_MASK)
#define I2C_IBIC_BIIE_MASK                       0x80u
#define I2C_IBIC_BIIE_SHIFT                      7u
#define I2C_IBIC_BIIE_WIDTH                      1u
#define I2C_IBIC_BIIE(x)                         (((uint8_t)(((uint8_t)(x))<<I2C_IBIC_BIIE_SHIFT))&I2C_IBIC_BIIE_MASK)
/* IBDBG Bit Fields */
#define I2C_IBDBG_IPG_DEBUG_EN_MASK              0x1u
#define I2C_IBDBG_IPG_DEBUG_EN_SHIFT             0u
#define I2C_IBDBG_IPG_DEBUG_EN_WIDTH             1u
#define I2C_IBDBG_IPG_DEBUG_EN(x)                (((uint8_t)(((uint8_t)(x))<<I2C_IBDBG_IPG_DEBUG_EN_SHIFT))&I2C_IBDBG_IPG_DEBUG_EN_MASK)
#define I2C_IBDBG_IPG_DEBUG_HALTED_MASK          0x2u
#define I2C_IBDBG_IPG_DEBUG_HALTED_SHIFT         1u
#define I2C_IBDBG_IPG_DEBUG_HALTED_WIDTH         1u
#define I2C_IBDBG_IPG_DEBUG_HALTED(x)            (((uint8_t)(((uint8_t)(x))<<I2C_IBDBG_IPG_DEBUG_HALTED_SHIFT))&I2C_IBDBG_IPG_DEBUG_HALTED_MASK)
#define I2C_IBDBG_BYTE_RX_MASK                   0x4u
#define I2C_IBDBG_BYTE_RX_SHIFT                  2u
#define I2C_IBDBG_BYTE_RX_WIDTH                  1u
#define I2C_IBDBG_BYTE_RX(x)                     (((uint8_t)(((uint8_t)(x))<<I2C_IBDBG_BYTE_RX_SHIFT))&I2C_IBDBG_BYTE_RX_MASK)
#define I2C_IBDBG_GLFLT_EN_MASK                  0x8u
#define I2C_IBDBG_GLFLT_EN_SHIFT                 3u
#define I2C_IBDBG_GLFLT_EN_WIDTH                 1u
#define I2C_IBDBG_GLFLT_EN(x)                    (((uint8_t)(((uint8_t)(x))<<I2C_IBDBG_GLFLT_EN_SHIFT))&I2C_IBDBG_GLFLT_EN_MASK)

/*!
 * @}
 */ /* end of group I2C_Register_Masks */


/*!
 * @}
 */ /* end of group I2C_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- IPL Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup IPL_Peripheral_Access_Layer IPL Peripheral Access Layer
 * @{
 */


/** IPL - Size of Registers Arrays */
#define IPL_TXDR_COUNT                           8u
#define IPL_RXDR_COUNT                           8u

/** IPL - Register Layout Typedef */
typedef struct {
  __IO uint32_t CFG;                               /**< Configuration Register, offset: 0x0 */
  __IO uint32_t SCOR;                              /**< Sequencer Configuration Register, offset: 0x4 */
  __IO uint32_t FFCR;                              /**< FIFO Configuration and Control Register, offset: 0x8 */
  __IO uint32_t CR;                                /**< Control Register, offset: 0xC */
  __IO uint32_t SR;                                /**< Status Register, offset: 0x10 */
  __IO uint32_t TXWCNTR;                           /**< Transmit Word Counter Register, offset: 0x14 */
  __IO uint32_t RXWCNTR;                           /**< Receive Word Counter Register, offset: 0x18 */
  __IO uint32_t FFDB;                              /**< FIFO Debugging Register, offset: 0x1C */
  __IO uint32_t RQER;                              /**< Request Enable Register, offset: 0x20 */
  __IO uint32_t RQSR;                              /**< Request Status Register, offset: 0x24 */
       uint8_t RESERVED_0[8];
  __IO uint32_t TXCIR;                             /**< Transmit CRC Initial Value Register, offset: 0x30 */
  __IO uint32_t TXCRC;                             /**< Transmit CRC Signature Register, offset: 0x34 */
  __IO uint32_t RXCIR;                             /**< Receive CRC Initial Value Register, offset: 0x38 */
  __I  uint32_t RXCRC;                             /**< Receive CRC Signature Register, offset: 0x3C */
  __IO uint32_t TXDR[IPL_TXDR_COUNT];              /**< Transmit Data Register, array offset: 0x40, array step: 0x4 */
  __I  uint32_t RXDR[IPL_RXDR_COUNT];              /**< Receive Data Register, array offset: 0x60, array step: 0x4 */
} IPL_Type, *IPL_MemMapPtr;

 /** Number of instances of the IPL module. */
#define IPL_INSTANCE_COUNT                       (1u)


/* IPL - Peripheral instance base addresses */
/** Peripheral IPL base address */
#define IPL_BASE                                 (0x400C3000u)
/** Peripheral IPL base pointer */
#define IPL                                      ((IPL_Type *)IPL_BASE)
/** Array initializer of IPL peripheral base addresses */
#define IPL_BASE_ADDRS                           { IPL_BASE }
/** Array initializer of IPL peripheral base pointers */
#define IPL_BASE_PTRS                            { IPL }

/* ----------------------------------------------------------------------------
   -- IPL Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup IPL_Register_Masks IPL Register Masks
 * @{
 */

/* CFG Bit Fields */
#define IPL_CFG_PORT_SIZE_MASK                   0x3u
#define IPL_CFG_PORT_SIZE_SHIFT                  0u
#define IPL_CFG_PORT_SIZE_WIDTH                  2u
#define IPL_CFG_PORT_SIZE(x)                     (((uint32_t)(((uint32_t)(x))<<IPL_CFG_PORT_SIZE_SHIFT))&IPL_CFG_PORT_SIZE_MASK)
#define IPL_CFG_ACP_MASK                         0xCu
#define IPL_CFG_ACP_SHIFT                        2u
#define IPL_CFG_ACP_WIDTH                        2u
#define IPL_CFG_ACP(x)                           (((uint32_t)(((uint32_t)(x))<<IPL_CFG_ACP_SHIFT))&IPL_CFG_ACP_MASK)
#define IPL_CFG_EOTFE_MASK                       0x10u
#define IPL_CFG_EOTFE_SHIFT                      4u
#define IPL_CFG_EOTFE_WIDTH                      1u
#define IPL_CFG_EOTFE(x)                         (((uint32_t)(((uint32_t)(x))<<IPL_CFG_EOTFE_SHIFT))&IPL_CFG_EOTFE_MASK)
#define IPL_CFG_TEN_MASK                         0x1000000u
#define IPL_CFG_TEN_SHIFT                        24u
#define IPL_CFG_TEN_WIDTH                        1u
#define IPL_CFG_TEN(x)                           (((uint32_t)(((uint32_t)(x))<<IPL_CFG_TEN_SHIFT))&IPL_CFG_TEN_MASK)
#define IPL_CFG_TXRXLB_MASK                      0x2000000u
#define IPL_CFG_TXRXLB_SHIFT                     25u
#define IPL_CFG_TXRXLB_WIDTH                     1u
#define IPL_CFG_TXRXLB(x)                        (((uint32_t)(((uint32_t)(x))<<IPL_CFG_TXRXLB_SHIFT))&IPL_CFG_TXRXLB_MASK)
/* SCOR Bit Fields */
#define IPL_SCOR_SSD_MASK                        0x7Fu
#define IPL_SCOR_SSD_SHIFT                       0u
#define IPL_SCOR_SSD_WIDTH                       7u
#define IPL_SCOR_SSD(x)                          (((uint32_t)(((uint32_t)(x))<<IPL_SCOR_SSD_SHIFT))&IPL_SCOR_SSD_MASK)
#define IPL_SCOR_POL_MASK                        0x80u
#define IPL_SCOR_POL_SHIFT                       7u
#define IPL_SCOR_POL_WIDTH                       1u
#define IPL_SCOR_POL(x)                          (((uint32_t)(((uint32_t)(x))<<IPL_SCOR_POL_SHIFT))&IPL_SCOR_POL_MASK)
#define IPL_SCOR_BCKO_MASK                       0xFF00u
#define IPL_SCOR_BCKO_SHIFT                      8u
#define IPL_SCOR_BCKO_WIDTH                      8u
#define IPL_SCOR_BCKO(x)                         (((uint32_t)(((uint32_t)(x))<<IPL_SCOR_BCKO_SHIFT))&IPL_SCOR_BCKO_MASK)
#define IPL_SCOR_COLTIM_MASK                     0xFF0000u
#define IPL_SCOR_COLTIM_SHIFT                    16u
#define IPL_SCOR_COLTIM_WIDTH                    8u
#define IPL_SCOR_COLTIM(x)                       (((uint32_t)(((uint32_t)(x))<<IPL_SCOR_COLTIM_SHIFT))&IPL_SCOR_COLTIM_MASK)
#define IPL_SCOR_IFTIM_MASK                      0xFF000000u
#define IPL_SCOR_IFTIM_SHIFT                     24u
#define IPL_SCOR_IFTIM_WIDTH                     8u
#define IPL_SCOR_IFTIM(x)                        (((uint32_t)(((uint32_t)(x))<<IPL_SCOR_IFTIM_SHIFT))&IPL_SCOR_IFTIM_MASK)
/* FFCR Bit Fields */
#define IPL_FFCR_TFWL_MASK                       0xFu
#define IPL_FFCR_TFWL_SHIFT                      0u
#define IPL_FFCR_TFWL_WIDTH                      4u
#define IPL_FFCR_TFWL(x)                         (((uint32_t)(((uint32_t)(x))<<IPL_FFCR_TFWL_SHIFT))&IPL_FFCR_TFWL_MASK)
#define IPL_FFCR_RFWL_MASK                       0xF00u
#define IPL_FFCR_RFWL_SHIFT                      8u
#define IPL_FFCR_RFWL_WIDTH                      4u
#define IPL_FFCR_RFWL(x)                         (((uint32_t)(((uint32_t)(x))<<IPL_FFCR_RFWL_SHIFT))&IPL_FFCR_RFWL_MASK)
#define IPL_FFCR_RFAFL_MASK                      0xF000u
#define IPL_FFCR_RFAFL_SHIFT                     12u
#define IPL_FFCR_RFAFL_WIDTH                     4u
#define IPL_FFCR_RFAFL(x)                        (((uint32_t)(((uint32_t)(x))<<IPL_FFCR_RFAFL_SHIFT))&IPL_FFCR_RFAFL_MASK)
#define IPL_FFCR_TFCLR_MASK                      0x10000u
#define IPL_FFCR_TFCLR_SHIFT                     16u
#define IPL_FFCR_TFCLR_WIDTH                     1u
#define IPL_FFCR_TFCLR(x)                        (((uint32_t)(((uint32_t)(x))<<IPL_FFCR_TFCLR_SHIFT))&IPL_FFCR_TFCLR_MASK)
#define IPL_FFCR_RFCLR_MASK                      0x1000000u
#define IPL_FFCR_RFCLR_SHIFT                     24u
#define IPL_FFCR_RFCLR_WIDTH                     1u
#define IPL_FFCR_RFCLR(x)                        (((uint32_t)(((uint32_t)(x))<<IPL_FFCR_RFCLR_SHIFT))&IPL_FFCR_RFCLR_MASK)
/* CR Bit Fields */
#define IPL_CR_MEN_MASK                          0x1u
#define IPL_CR_MEN_SHIFT                         0u
#define IPL_CR_MEN_WIDTH                         1u
#define IPL_CR_MEN(x)                            (((uint32_t)(((uint32_t)(x))<<IPL_CR_MEN_SHIFT))&IPL_CR_MEN_MASK)
#define IPL_CR_RXDRWL_MASK                       0xF000000u
#define IPL_CR_RXDRWL_SHIFT                      24u
#define IPL_CR_RXDRWL_WIDTH                      4u
#define IPL_CR_RXDRWL(x)                         (((uint32_t)(((uint32_t)(x))<<IPL_CR_RXDRWL_SHIFT))&IPL_CR_RXDRWL_MASK)
#define IPL_CR_RXDOE_MASK                        0x10000000u
#define IPL_CR_RXDOE_SHIFT                       28u
#define IPL_CR_RXDOE_WIDTH                       1u
#define IPL_CR_RXDOE(x)                          (((uint32_t)(((uint32_t)(x))<<IPL_CR_RXDOE_SHIFT))&IPL_CR_RXDOE_MASK)
#define IPL_CR_RXDUE_MASK                        0x20000000u
#define IPL_CR_RXDUE_SHIFT                       29u
#define IPL_CR_RXDUE_WIDTH                       1u
#define IPL_CR_RXDUE(x)                          (((uint32_t)(((uint32_t)(x))<<IPL_CR_RXDUE_SHIFT))&IPL_CR_RXDUE_MASK)
#define IPL_CR_TXDOE_MASK                        0x40000000u
#define IPL_CR_TXDOE_SHIFT                       30u
#define IPL_CR_TXDOE_WIDTH                       1u
#define IPL_CR_TXDOE(x)                          (((uint32_t)(((uint32_t)(x))<<IPL_CR_TXDOE_SHIFT))&IPL_CR_TXDOE_MASK)
/* SR Bit Fields */
#define IPL_SR_IPLRDY_MASK                       0x1u
#define IPL_SR_IPLRDY_SHIFT                      0u
#define IPL_SR_IPLRDY_WIDTH                      1u
#define IPL_SR_IPLRDY(x)                         (((uint32_t)(((uint32_t)(x))<<IPL_SR_IPLRDY_SHIFT))&IPL_SR_IPLRDY_MASK)
#define IPL_SR_TRS_MASK                          0x6u
#define IPL_SR_TRS_SHIFT                         1u
#define IPL_SR_TRS_WIDTH                         2u
#define IPL_SR_TRS(x)                            (((uint32_t)(((uint32_t)(x))<<IPL_SR_TRS_SHIFT))&IPL_SR_TRS_MASK)
#define IPL_SR_TFFL_MASK                         0xF8u
#define IPL_SR_TFFL_SHIFT                        3u
#define IPL_SR_TFFL_WIDTH                        5u
#define IPL_SR_TFFL(x)                           (((uint32_t)(((uint32_t)(x))<<IPL_SR_TFFL_SHIFT))&IPL_SR_TFFL_MASK)
#define IPL_SR_IPLLST_MASK                       0x100u
#define IPL_SR_IPLLST_SHIFT                      8u
#define IPL_SR_IPLLST_WIDTH                      1u
#define IPL_SR_IPLLST(x)                         (((uint32_t)(((uint32_t)(x))<<IPL_SR_IPLLST_SHIFT))&IPL_SR_IPLLST_MASK)
#define IPL_SR_ODT0_MASK                         0x200u
#define IPL_SR_ODT0_SHIFT                        9u
#define IPL_SR_ODT0_WIDTH                        1u
#define IPL_SR_ODT0(x)                           (((uint32_t)(((uint32_t)(x))<<IPL_SR_ODT0_SHIFT))&IPL_SR_ODT0_MASK)
#define IPL_SR_ODT1_MASK                         0x400u
#define IPL_SR_ODT1_SHIFT                        10u
#define IPL_SR_ODT1_WIDTH                        1u
#define IPL_SR_ODT1(x)                           (((uint32_t)(((uint32_t)(x))<<IPL_SR_ODT1_SHIFT))&IPL_SR_ODT1_MASK)
#define IPL_SR_RFFL_MASK                         0xF800u
#define IPL_SR_RFFL_SHIFT                        11u
#define IPL_SR_RFFL_WIDTH                        5u
#define IPL_SR_RFFL(x)                           (((uint32_t)(((uint32_t)(x))<<IPL_SR_RFFL_SHIFT))&IPL_SR_RFFL_MASK)
#define IPL_SR_EOTX_MASK                         0x10000u
#define IPL_SR_EOTX_SHIFT                        16u
#define IPL_SR_EOTX_WIDTH                        1u
#define IPL_SR_EOTX(x)                           (((uint32_t)(((uint32_t)(x))<<IPL_SR_EOTX_SHIFT))&IPL_SR_EOTX_MASK)
#define IPL_SR_SOTX_MASK                         0x20000u
#define IPL_SR_SOTX_SHIFT                        17u
#define IPL_SR_SOTX_WIDTH                        1u
#define IPL_SR_SOTX(x)                           (((uint32_t)(((uint32_t)(x))<<IPL_SR_SOTX_SHIFT))&IPL_SR_SOTX_MASK)
#define IPL_SR_EORX_MASK                         0x40000u
#define IPL_SR_EORX_SHIFT                        18u
#define IPL_SR_EORX_WIDTH                        1u
#define IPL_SR_EORX(x)                           (((uint32_t)(((uint32_t)(x))<<IPL_SR_EORX_SHIFT))&IPL_SR_EORX_MASK)
#define IPL_SR_SORX_MASK                         0x80000u
#define IPL_SR_SORX_SHIFT                        19u
#define IPL_SR_SORX_WIDTH                        1u
#define IPL_SR_SORX(x)                           (((uint32_t)(((uint32_t)(x))<<IPL_SR_SORX_SHIFT))&IPL_SR_SORX_MASK)
#define IPL_SR_TFO_MASK                          0x100000u
#define IPL_SR_TFO_SHIFT                         20u
#define IPL_SR_TFO_WIDTH                         1u
#define IPL_SR_TFO(x)                            (((uint32_t)(((uint32_t)(x))<<IPL_SR_TFO_SHIFT))&IPL_SR_TFO_MASK)
#define IPL_SR_RFO_MASK                          0x200000u
#define IPL_SR_RFO_SHIFT                         21u
#define IPL_SR_RFO_WIDTH                         1u
#define IPL_SR_RFO(x)                            (((uint32_t)(((uint32_t)(x))<<IPL_SR_RFO_SHIFT))&IPL_SR_RFO_MASK)
#define IPL_SR_RFU_MASK                          0x400000u
#define IPL_SR_RFU_SHIFT                         22u
#define IPL_SR_RFU_WIDTH                         1u
#define IPL_SR_RFU(x)                            (((uint32_t)(((uint32_t)(x))<<IPL_SR_RFU_SHIFT))&IPL_SR_RFU_MASK)
#define IPL_SR_THER_MASK                         0x800000u
#define IPL_SR_THER_SHIFT                        23u
#define IPL_SR_THER_WIDTH                        1u
#define IPL_SR_THER(x)                           (((uint32_t)(((uint32_t)(x))<<IPL_SR_THER_SHIFT))&IPL_SR_THER_MASK)
#define IPL_SR_RRFAF_MASK                        0x1000000u
#define IPL_SR_RRFAF_SHIFT                       24u
#define IPL_SR_RRFAF_WIDTH                       1u
#define IPL_SR_RRFAF(x)                          (((uint32_t)(((uint32_t)(x))<<IPL_SR_RRFAF_SHIFT))&IPL_SR_RRFAF_MASK)
#define IPL_SR_LRFAF_MASK                        0x2000000u
#define IPL_SR_LRFAF_SHIFT                       25u
#define IPL_SR_LRFAF_WIDTH                       1u
#define IPL_SR_LRFAF(x)                          (((uint32_t)(((uint32_t)(x))<<IPL_SR_LRFAF_SHIFT))&IPL_SR_LRFAF_MASK)
#define IPL_SR_TFWU_MASK                         0x4000000u
#define IPL_SR_TFWU_SHIFT                        26u
#define IPL_SR_TFWU_WIDTH                        1u
#define IPL_SR_TFWU(x)                           (((uint32_t)(((uint32_t)(x))<<IPL_SR_TFWU_SHIFT))&IPL_SR_TFWU_MASK)
#define IPL_SR_RFWO_MASK                         0x8000000u
#define IPL_SR_RFWO_SHIFT                        27u
#define IPL_SR_RFWO_WIDTH                        1u
#define IPL_SR_RFWO(x)                           (((uint32_t)(((uint32_t)(x))<<IPL_SR_RFWO_SHIFT))&IPL_SR_RFWO_MASK)
#define IPL_SR_COL_MASK                          0x10000000u
#define IPL_SR_COL_SHIFT                         28u
#define IPL_SR_COL_WIDTH                         1u
#define IPL_SR_COL(x)                            (((uint32_t)(((uint32_t)(x))<<IPL_SR_COL_SHIFT))&IPL_SR_COL_MASK)
#define IPL_SR_RXCE_MASK                         0x20000000u
#define IPL_SR_RXCE_SHIFT                        29u
#define IPL_SR_RXCE_WIDTH                        1u
#define IPL_SR_RXCE(x)                           (((uint32_t)(((uint32_t)(x))<<IPL_SR_RXCE_SHIFT))&IPL_SR_RXCE_MASK)
#define IPL_SR_RXCEP_MASK                        0x40000000u
#define IPL_SR_RXCEP_SHIFT                       30u
#define IPL_SR_RXCEP_WIDTH                       1u
#define IPL_SR_RXCEP(x)                          (((uint32_t)(((uint32_t)(x))<<IPL_SR_RXCEP_SHIFT))&IPL_SR_RXCEP_MASK)
#define IPL_SR_EOT_MASK                          0x80000000u
#define IPL_SR_EOT_SHIFT                         31u
#define IPL_SR_EOT_WIDTH                         1u
#define IPL_SR_EOT(x)                            (((uint32_t)(((uint32_t)(x))<<IPL_SR_EOT_SHIFT))&IPL_SR_EOT_MASK)
/* TXWCNTR Bit Fields */
#define IPL_TXWCNTR_TXWCNT_MASK                  0xFFFFFFFFu
#define IPL_TXWCNTR_TXWCNT_SHIFT                 0u
#define IPL_TXWCNTR_TXWCNT_WIDTH                 32u
#define IPL_TXWCNTR_TXWCNT(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_TXWCNTR_TXWCNT_SHIFT))&IPL_TXWCNTR_TXWCNT_MASK)
/* RXWCNTR Bit Fields */
#define IPL_RXWCNTR_RXWCNT_MASK                  0xFFFFFFFFu
#define IPL_RXWCNTR_RXWCNT_SHIFT                 0u
#define IPL_RXWCNTR_RXWCNT_WIDTH                 32u
#define IPL_RXWCNTR_RXWCNT(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RXWCNTR_RXWCNT_SHIFT))&IPL_RXWCNTR_RXWCNT_MASK)
/* FFDB Bit Fields */
#define IPL_FFDB_TFRDPTR_MASK                    0x1Fu
#define IPL_FFDB_TFRDPTR_SHIFT                   0u
#define IPL_FFDB_TFRDPTR_WIDTH                   5u
#define IPL_FFDB_TFRDPTR(x)                      (((uint32_t)(((uint32_t)(x))<<IPL_FFDB_TFRDPTR_SHIFT))&IPL_FFDB_TFRDPTR_MASK)
#define IPL_FFDB_TFE_MASK                        0x20u
#define IPL_FFDB_TFE_SHIFT                       5u
#define IPL_FFDB_TFE_WIDTH                       1u
#define IPL_FFDB_TFE(x)                          (((uint32_t)(((uint32_t)(x))<<IPL_FFDB_TFE_SHIFT))&IPL_FFDB_TFE_MASK)
#define IPL_FFDB_TFU_MASK                        0x40u
#define IPL_FFDB_TFU_SHIFT                       6u
#define IPL_FFDB_TFU_WIDTH                       1u
#define IPL_FFDB_TFU(x)                          (((uint32_t)(((uint32_t)(x))<<IPL_FFDB_TFU_SHIFT))&IPL_FFDB_TFU_MASK)
#define IPL_FFDB_TFWRPTR_MASK                    0x1F00u
#define IPL_FFDB_TFWRPTR_SHIFT                   8u
#define IPL_FFDB_TFWRPTR_WIDTH                   5u
#define IPL_FFDB_TFWRPTR(x)                      (((uint32_t)(((uint32_t)(x))<<IPL_FFDB_TFWRPTR_SHIFT))&IPL_FFDB_TFWRPTR_MASK)
#define IPL_FFDB_TFF_MASK                        0x2000u
#define IPL_FFDB_TFF_SHIFT                       13u
#define IPL_FFDB_TFF_WIDTH                       1u
#define IPL_FFDB_TFF(x)                          (((uint32_t)(((uint32_t)(x))<<IPL_FFDB_TFF_SHIFT))&IPL_FFDB_TFF_MASK)
#define IPL_FFDB_TFO_MASK                        0x4000u
#define IPL_FFDB_TFO_SHIFT                       14u
#define IPL_FFDB_TFO_WIDTH                       1u
#define IPL_FFDB_TFO(x)                          (((uint32_t)(((uint32_t)(x))<<IPL_FFDB_TFO_SHIFT))&IPL_FFDB_TFO_MASK)
#define IPL_FFDB_RFRDPTR_MASK                    0x1F0000u
#define IPL_FFDB_RFRDPTR_SHIFT                   16u
#define IPL_FFDB_RFRDPTR_WIDTH                   5u
#define IPL_FFDB_RFRDPTR(x)                      (((uint32_t)(((uint32_t)(x))<<IPL_FFDB_RFRDPTR_SHIFT))&IPL_FFDB_RFRDPTR_MASK)
#define IPL_FFDB_RFE_MASK                        0x200000u
#define IPL_FFDB_RFE_SHIFT                       21u
#define IPL_FFDB_RFE_WIDTH                       1u
#define IPL_FFDB_RFE(x)                          (((uint32_t)(((uint32_t)(x))<<IPL_FFDB_RFE_SHIFT))&IPL_FFDB_RFE_MASK)
#define IPL_FFDB_RFU_MASK                        0x400000u
#define IPL_FFDB_RFU_SHIFT                       22u
#define IPL_FFDB_RFU_WIDTH                       1u
#define IPL_FFDB_RFU(x)                          (((uint32_t)(((uint32_t)(x))<<IPL_FFDB_RFU_SHIFT))&IPL_FFDB_RFU_MASK)
#define IPL_FFDB_RFWRPTR_MASK                    0x1F000000u
#define IPL_FFDB_RFWRPTR_SHIFT                   24u
#define IPL_FFDB_RFWRPTR_WIDTH                   5u
#define IPL_FFDB_RFWRPTR(x)                      (((uint32_t)(((uint32_t)(x))<<IPL_FFDB_RFWRPTR_SHIFT))&IPL_FFDB_RFWRPTR_MASK)
#define IPL_FFDB_RFF_MASK                        0x20000000u
#define IPL_FFDB_RFF_SHIFT                       29u
#define IPL_FFDB_RFF_WIDTH                       1u
#define IPL_FFDB_RFF(x)                          (((uint32_t)(((uint32_t)(x))<<IPL_FFDB_RFF_SHIFT))&IPL_FFDB_RFF_MASK)
#define IPL_FFDB_RFO_MASK                        0x40000000u
#define IPL_FFDB_RFO_SHIFT                       30u
#define IPL_FFDB_RFO_WIDTH                       1u
#define IPL_FFDB_RFO(x)                          (((uint32_t)(((uint32_t)(x))<<IPL_FFDB_RFO_SHIFT))&IPL_FFDB_RFO_MASK)
#define IPL_FFDB_RFAF_MASK                       0x80000000u
#define IPL_FFDB_RFAF_SHIFT                      31u
#define IPL_FFDB_RFAF_WIDTH                      1u
#define IPL_FFDB_RFAF(x)                         (((uint32_t)(((uint32_t)(x))<<IPL_FFDB_RFAF_SHIFT))&IPL_FFDB_RFAF_MASK)
/* RQER Bit Fields */
#define IPL_RQER_EOTX_IRQ0_MASK                  0x1u
#define IPL_RQER_EOTX_IRQ0_SHIFT                 0u
#define IPL_RQER_EOTX_IRQ0_WIDTH                 1u
#define IPL_RQER_EOTX_IRQ0(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQER_EOTX_IRQ0_SHIFT))&IPL_RQER_EOTX_IRQ0_MASK)
#define IPL_RQER_EOTX_IRQ1_MASK                  0x2u
#define IPL_RQER_EOTX_IRQ1_SHIFT                 1u
#define IPL_RQER_EOTX_IRQ1_WIDTH                 1u
#define IPL_RQER_EOTX_IRQ1(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQER_EOTX_IRQ1_SHIFT))&IPL_RQER_EOTX_IRQ1_MASK)
#define IPL_RQER_SOTX_IRQ0_MASK                  0x4u
#define IPL_RQER_SOTX_IRQ0_SHIFT                 2u
#define IPL_RQER_SOTX_IRQ0_WIDTH                 1u
#define IPL_RQER_SOTX_IRQ0(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQER_SOTX_IRQ0_SHIFT))&IPL_RQER_SOTX_IRQ0_MASK)
#define IPL_RQER_SOTX_IRQ1_MASK                  0x8u
#define IPL_RQER_SOTX_IRQ1_SHIFT                 3u
#define IPL_RQER_SOTX_IRQ1_WIDTH                 1u
#define IPL_RQER_SOTX_IRQ1(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQER_SOTX_IRQ1_SHIFT))&IPL_RQER_SOTX_IRQ1_MASK)
#define IPL_RQER_EORX_IRQ0_MASK                  0x10u
#define IPL_RQER_EORX_IRQ0_SHIFT                 4u
#define IPL_RQER_EORX_IRQ0_WIDTH                 1u
#define IPL_RQER_EORX_IRQ0(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQER_EORX_IRQ0_SHIFT))&IPL_RQER_EORX_IRQ0_MASK)
#define IPL_RQER_EORX_IRQ1_MASK                  0x20u
#define IPL_RQER_EORX_IRQ1_SHIFT                 5u
#define IPL_RQER_EORX_IRQ1_WIDTH                 1u
#define IPL_RQER_EORX_IRQ1(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQER_EORX_IRQ1_SHIFT))&IPL_RQER_EORX_IRQ1_MASK)
#define IPL_RQER_SORX_IRQ0_MASK                  0x40u
#define IPL_RQER_SORX_IRQ0_SHIFT                 6u
#define IPL_RQER_SORX_IRQ0_WIDTH                 1u
#define IPL_RQER_SORX_IRQ0(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQER_SORX_IRQ0_SHIFT))&IPL_RQER_SORX_IRQ0_MASK)
#define IPL_RQER_SORX_IRQ1_MASK                  0x80u
#define IPL_RQER_SORX_IRQ1_SHIFT                 7u
#define IPL_RQER_SORX_IRQ1_WIDTH                 1u
#define IPL_RQER_SORX_IRQ1(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQER_SORX_IRQ1_SHIFT))&IPL_RQER_SORX_IRQ1_MASK)
#define IPL_RQER_TFO_IRQ0_MASK                   0x100u
#define IPL_RQER_TFO_IRQ0_SHIFT                  8u
#define IPL_RQER_TFO_IRQ0_WIDTH                  1u
#define IPL_RQER_TFO_IRQ0(x)                     (((uint32_t)(((uint32_t)(x))<<IPL_RQER_TFO_IRQ0_SHIFT))&IPL_RQER_TFO_IRQ0_MASK)
#define IPL_RQER_TFO_IRQ1_MASK                   0x200u
#define IPL_RQER_TFO_IRQ1_SHIFT                  9u
#define IPL_RQER_TFO_IRQ1_WIDTH                  1u
#define IPL_RQER_TFO_IRQ1(x)                     (((uint32_t)(((uint32_t)(x))<<IPL_RQER_TFO_IRQ1_SHIFT))&IPL_RQER_TFO_IRQ1_MASK)
#define IPL_RQER_RFO_IRQ0_MASK                   0x400u
#define IPL_RQER_RFO_IRQ0_SHIFT                  10u
#define IPL_RQER_RFO_IRQ0_WIDTH                  1u
#define IPL_RQER_RFO_IRQ0(x)                     (((uint32_t)(((uint32_t)(x))<<IPL_RQER_RFO_IRQ0_SHIFT))&IPL_RQER_RFO_IRQ0_MASK)
#define IPL_RQER_RFO_IRQ1_MASK                   0x800u
#define IPL_RQER_RFO_IRQ1_SHIFT                  11u
#define IPL_RQER_RFO_IRQ1_WIDTH                  1u
#define IPL_RQER_RFO_IRQ1(x)                     (((uint32_t)(((uint32_t)(x))<<IPL_RQER_RFO_IRQ1_SHIFT))&IPL_RQER_RFO_IRQ1_MASK)
#define IPL_RQER_RFU_IRQ0_MASK                   0x1000u
#define IPL_RQER_RFU_IRQ0_SHIFT                  12u
#define IPL_RQER_RFU_IRQ0_WIDTH                  1u
#define IPL_RQER_RFU_IRQ0(x)                     (((uint32_t)(((uint32_t)(x))<<IPL_RQER_RFU_IRQ0_SHIFT))&IPL_RQER_RFU_IRQ0_MASK)
#define IPL_RQER_RFU_IRQ1_MASK                   0x2000u
#define IPL_RQER_RFU_IRQ1_SHIFT                  13u
#define IPL_RQER_RFU_IRQ1_WIDTH                  1u
#define IPL_RQER_RFU_IRQ1(x)                     (((uint32_t)(((uint32_t)(x))<<IPL_RQER_RFU_IRQ1_SHIFT))&IPL_RQER_RFU_IRQ1_MASK)
#define IPL_RQER_THER_IRQ0_MASK                  0x4000u
#define IPL_RQER_THER_IRQ0_SHIFT                 14u
#define IPL_RQER_THER_IRQ0_WIDTH                 1u
#define IPL_RQER_THER_IRQ0(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQER_THER_IRQ0_SHIFT))&IPL_RQER_THER_IRQ0_MASK)
#define IPL_RQER_THER_IRQ1_MASK                  0x8000u
#define IPL_RQER_THER_IRQ1_SHIFT                 15u
#define IPL_RQER_THER_IRQ1_WIDTH                 1u
#define IPL_RQER_THER_IRQ1(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQER_THER_IRQ1_SHIFT))&IPL_RQER_THER_IRQ1_MASK)
#define IPL_RQER_RRFAF_IRQ0_MASK                 0x10000u
#define IPL_RQER_RRFAF_IRQ0_SHIFT                16u
#define IPL_RQER_RRFAF_IRQ0_WIDTH                1u
#define IPL_RQER_RRFAF_IRQ0(x)                   (((uint32_t)(((uint32_t)(x))<<IPL_RQER_RRFAF_IRQ0_SHIFT))&IPL_RQER_RRFAF_IRQ0_MASK)
#define IPL_RQER_RRFAF_IRQ1_MASK                 0x20000u
#define IPL_RQER_RRFAF_IRQ1_SHIFT                17u
#define IPL_RQER_RRFAF_IRQ1_WIDTH                1u
#define IPL_RQER_RRFAF_IRQ1(x)                   (((uint32_t)(((uint32_t)(x))<<IPL_RQER_RRFAF_IRQ1_SHIFT))&IPL_RQER_RRFAF_IRQ1_MASK)
#define IPL_RQER_LRFAF_IRQ0_MASK                 0x40000u
#define IPL_RQER_LRFAF_IRQ0_SHIFT                18u
#define IPL_RQER_LRFAF_IRQ0_WIDTH                1u
#define IPL_RQER_LRFAF_IRQ0(x)                   (((uint32_t)(((uint32_t)(x))<<IPL_RQER_LRFAF_IRQ0_SHIFT))&IPL_RQER_LRFAF_IRQ0_MASK)
#define IPL_RQER_LRFAF_IRQ1_MASK                 0x80000u
#define IPL_RQER_LRFAF_IRQ1_SHIFT                19u
#define IPL_RQER_LRFAF_IRQ1_WIDTH                1u
#define IPL_RQER_LRFAF_IRQ1(x)                   (((uint32_t)(((uint32_t)(x))<<IPL_RQER_LRFAF_IRQ1_SHIFT))&IPL_RQER_LRFAF_IRQ1_MASK)
#define IPL_RQER_TFWU_IRQ0_MASK                  0x100000u
#define IPL_RQER_TFWU_IRQ0_SHIFT                 20u
#define IPL_RQER_TFWU_IRQ0_WIDTH                 1u
#define IPL_RQER_TFWU_IRQ0(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQER_TFWU_IRQ0_SHIFT))&IPL_RQER_TFWU_IRQ0_MASK)
#define IPL_RQER_TFWU_IRQ1_MASK                  0x200000u
#define IPL_RQER_TFWU_IRQ1_SHIFT                 21u
#define IPL_RQER_TFWU_IRQ1_WIDTH                 1u
#define IPL_RQER_TFWU_IRQ1(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQER_TFWU_IRQ1_SHIFT))&IPL_RQER_TFWU_IRQ1_MASK)
#define IPL_RQER_RFWO_IRQ0_MASK                  0x400000u
#define IPL_RQER_RFWO_IRQ0_SHIFT                 22u
#define IPL_RQER_RFWO_IRQ0_WIDTH                 1u
#define IPL_RQER_RFWO_IRQ0(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQER_RFWO_IRQ0_SHIFT))&IPL_RQER_RFWO_IRQ0_MASK)
#define IPL_RQER_RFWO_IRQ1_MASK                  0x800000u
#define IPL_RQER_RFWO_IRQ1_SHIFT                 23u
#define IPL_RQER_RFWO_IRQ1_WIDTH                 1u
#define IPL_RQER_RFWO_IRQ1(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQER_RFWO_IRQ1_SHIFT))&IPL_RQER_RFWO_IRQ1_MASK)
#define IPL_RQER_COL_IRQ0_MASK                   0x1000000u
#define IPL_RQER_COL_IRQ0_SHIFT                  24u
#define IPL_RQER_COL_IRQ0_WIDTH                  1u
#define IPL_RQER_COL_IRQ0(x)                     (((uint32_t)(((uint32_t)(x))<<IPL_RQER_COL_IRQ0_SHIFT))&IPL_RQER_COL_IRQ0_MASK)
#define IPL_RQER_COL_IRQ1_MASK                   0x2000000u
#define IPL_RQER_COL_IRQ1_SHIFT                  25u
#define IPL_RQER_COL_IRQ1_WIDTH                  1u
#define IPL_RQER_COL_IRQ1(x)                     (((uint32_t)(((uint32_t)(x))<<IPL_RQER_COL_IRQ1_SHIFT))&IPL_RQER_COL_IRQ1_MASK)
#define IPL_RQER_RXCE_IRQ0_MASK                  0x4000000u
#define IPL_RQER_RXCE_IRQ0_SHIFT                 26u
#define IPL_RQER_RXCE_IRQ0_WIDTH                 1u
#define IPL_RQER_RXCE_IRQ0(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQER_RXCE_IRQ0_SHIFT))&IPL_RQER_RXCE_IRQ0_MASK)
#define IPL_RQER_RXCE_IRQ1_MASK                  0x8000000u
#define IPL_RQER_RXCE_IRQ1_SHIFT                 27u
#define IPL_RQER_RXCE_IRQ1_WIDTH                 1u
#define IPL_RQER_RXCE_IRQ1(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQER_RXCE_IRQ1_SHIFT))&IPL_RQER_RXCE_IRQ1_MASK)
#define IPL_RQER_TFWU_DMA0_MASK                  0x10000000u
#define IPL_RQER_TFWU_DMA0_SHIFT                 28u
#define IPL_RQER_TFWU_DMA0_WIDTH                 1u
#define IPL_RQER_TFWU_DMA0(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQER_TFWU_DMA0_SHIFT))&IPL_RQER_TFWU_DMA0_MASK)
#define IPL_RQER_TFWU_DMA1_MASK                  0x20000000u
#define IPL_RQER_TFWU_DMA1_SHIFT                 29u
#define IPL_RQER_TFWU_DMA1_WIDTH                 1u
#define IPL_RQER_TFWU_DMA1(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQER_TFWU_DMA1_SHIFT))&IPL_RQER_TFWU_DMA1_MASK)
#define IPL_RQER_RFWO_DMA0_MASK                  0x40000000u
#define IPL_RQER_RFWO_DMA0_SHIFT                 30u
#define IPL_RQER_RFWO_DMA0_WIDTH                 1u
#define IPL_RQER_RFWO_DMA0(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQER_RFWO_DMA0_SHIFT))&IPL_RQER_RFWO_DMA0_MASK)
#define IPL_RQER_RFWO_DMA1_MASK                  0x80000000u
#define IPL_RQER_RFWO_DMA1_SHIFT                 31u
#define IPL_RQER_RFWO_DMA1_WIDTH                 1u
#define IPL_RQER_RFWO_DMA1(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQER_RFWO_DMA1_SHIFT))&IPL_RQER_RFWO_DMA1_MASK)
/* RQSR Bit Fields */
#define IPL_RQSR_EOTX_IRQ0_MASK                  0x1u
#define IPL_RQSR_EOTX_IRQ0_SHIFT                 0u
#define IPL_RQSR_EOTX_IRQ0_WIDTH                 1u
#define IPL_RQSR_EOTX_IRQ0(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_EOTX_IRQ0_SHIFT))&IPL_RQSR_EOTX_IRQ0_MASK)
#define IPL_RQSR_EOTX_IRQ1_MASK                  0x2u
#define IPL_RQSR_EOTX_IRQ1_SHIFT                 1u
#define IPL_RQSR_EOTX_IRQ1_WIDTH                 1u
#define IPL_RQSR_EOTX_IRQ1(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_EOTX_IRQ1_SHIFT))&IPL_RQSR_EOTX_IRQ1_MASK)
#define IPL_RQSR_SOTX_IRQ0_MASK                  0x4u
#define IPL_RQSR_SOTX_IRQ0_SHIFT                 2u
#define IPL_RQSR_SOTX_IRQ0_WIDTH                 1u
#define IPL_RQSR_SOTX_IRQ0(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_SOTX_IRQ0_SHIFT))&IPL_RQSR_SOTX_IRQ0_MASK)
#define IPL_RQSR_SOTX_IRQ1_MASK                  0x8u
#define IPL_RQSR_SOTX_IRQ1_SHIFT                 3u
#define IPL_RQSR_SOTX_IRQ1_WIDTH                 1u
#define IPL_RQSR_SOTX_IRQ1(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_SOTX_IRQ1_SHIFT))&IPL_RQSR_SOTX_IRQ1_MASK)
#define IPL_RQSR_EORX_IRQ0_MASK                  0x10u
#define IPL_RQSR_EORX_IRQ0_SHIFT                 4u
#define IPL_RQSR_EORX_IRQ0_WIDTH                 1u
#define IPL_RQSR_EORX_IRQ0(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_EORX_IRQ0_SHIFT))&IPL_RQSR_EORX_IRQ0_MASK)
#define IPL_RQSR_EORX_IRQ1_MASK                  0x20u
#define IPL_RQSR_EORX_IRQ1_SHIFT                 5u
#define IPL_RQSR_EORX_IRQ1_WIDTH                 1u
#define IPL_RQSR_EORX_IRQ1(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_EORX_IRQ1_SHIFT))&IPL_RQSR_EORX_IRQ1_MASK)
#define IPL_RQSR_SORX_IRQ0_MASK                  0x40u
#define IPL_RQSR_SORX_IRQ0_SHIFT                 6u
#define IPL_RQSR_SORX_IRQ0_WIDTH                 1u
#define IPL_RQSR_SORX_IRQ0(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_SORX_IRQ0_SHIFT))&IPL_RQSR_SORX_IRQ0_MASK)
#define IPL_RQSR_SORX_IRQ1_MASK                  0x80u
#define IPL_RQSR_SORX_IRQ1_SHIFT                 7u
#define IPL_RQSR_SORX_IRQ1_WIDTH                 1u
#define IPL_RQSR_SORX_IRQ1(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_SORX_IRQ1_SHIFT))&IPL_RQSR_SORX_IRQ1_MASK)
#define IPL_RQSR_TFO_IRQ0_MASK                   0x100u
#define IPL_RQSR_TFO_IRQ0_SHIFT                  8u
#define IPL_RQSR_TFO_IRQ0_WIDTH                  1u
#define IPL_RQSR_TFO_IRQ0(x)                     (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_TFO_IRQ0_SHIFT))&IPL_RQSR_TFO_IRQ0_MASK)
#define IPL_RQSR_TFO_IRQ1_MASK                   0x200u
#define IPL_RQSR_TFO_IRQ1_SHIFT                  9u
#define IPL_RQSR_TFO_IRQ1_WIDTH                  1u
#define IPL_RQSR_TFO_IRQ1(x)                     (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_TFO_IRQ1_SHIFT))&IPL_RQSR_TFO_IRQ1_MASK)
#define IPL_RQSR_RFO_IRQ0_MASK                   0x400u
#define IPL_RQSR_RFO_IRQ0_SHIFT                  10u
#define IPL_RQSR_RFO_IRQ0_WIDTH                  1u
#define IPL_RQSR_RFO_IRQ0(x)                     (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_RFO_IRQ0_SHIFT))&IPL_RQSR_RFO_IRQ0_MASK)
#define IPL_RQSR_RFO_IRQ1_MASK                   0x800u
#define IPL_RQSR_RFO_IRQ1_SHIFT                  11u
#define IPL_RQSR_RFO_IRQ1_WIDTH                  1u
#define IPL_RQSR_RFO_IRQ1(x)                     (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_RFO_IRQ1_SHIFT))&IPL_RQSR_RFO_IRQ1_MASK)
#define IPL_RQSR_RFU_IRQ0_MASK                   0x1000u
#define IPL_RQSR_RFU_IRQ0_SHIFT                  12u
#define IPL_RQSR_RFU_IRQ0_WIDTH                  1u
#define IPL_RQSR_RFU_IRQ0(x)                     (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_RFU_IRQ0_SHIFT))&IPL_RQSR_RFU_IRQ0_MASK)
#define IPL_RQSR_RFU_IRQ1_MASK                   0x2000u
#define IPL_RQSR_RFU_IRQ1_SHIFT                  13u
#define IPL_RQSR_RFU_IRQ1_WIDTH                  1u
#define IPL_RQSR_RFU_IRQ1(x)                     (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_RFU_IRQ1_SHIFT))&IPL_RQSR_RFU_IRQ1_MASK)
#define IPL_RQSR_THER_IRQ0_MASK                  0x4000u
#define IPL_RQSR_THER_IRQ0_SHIFT                 14u
#define IPL_RQSR_THER_IRQ0_WIDTH                 1u
#define IPL_RQSR_THER_IRQ0(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_THER_IRQ0_SHIFT))&IPL_RQSR_THER_IRQ0_MASK)
#define IPL_RQSR_THER_IRQ1_MASK                  0x8000u
#define IPL_RQSR_THER_IRQ1_SHIFT                 15u
#define IPL_RQSR_THER_IRQ1_WIDTH                 1u
#define IPL_RQSR_THER_IRQ1(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_THER_IRQ1_SHIFT))&IPL_RQSR_THER_IRQ1_MASK)
#define IPL_RQSR_RRFAF_IRQ0_MASK                 0x10000u
#define IPL_RQSR_RRFAF_IRQ0_SHIFT                16u
#define IPL_RQSR_RRFAF_IRQ0_WIDTH                1u
#define IPL_RQSR_RRFAF_IRQ0(x)                   (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_RRFAF_IRQ0_SHIFT))&IPL_RQSR_RRFAF_IRQ0_MASK)
#define IPL_RQSR_RRFAF_IRQ1_MASK                 0x20000u
#define IPL_RQSR_RRFAF_IRQ1_SHIFT                17u
#define IPL_RQSR_RRFAF_IRQ1_WIDTH                1u
#define IPL_RQSR_RRFAF_IRQ1(x)                   (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_RRFAF_IRQ1_SHIFT))&IPL_RQSR_RRFAF_IRQ1_MASK)
#define IPL_RQSR_LRFAF_IRQ0_MASK                 0x40000u
#define IPL_RQSR_LRFAF_IRQ0_SHIFT                18u
#define IPL_RQSR_LRFAF_IRQ0_WIDTH                1u
#define IPL_RQSR_LRFAF_IRQ0(x)                   (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_LRFAF_IRQ0_SHIFT))&IPL_RQSR_LRFAF_IRQ0_MASK)
#define IPL_RQSR_LRFAF_IRQ1_MASK                 0x80000u
#define IPL_RQSR_LRFAF_IRQ1_SHIFT                19u
#define IPL_RQSR_LRFAF_IRQ1_WIDTH                1u
#define IPL_RQSR_LRFAF_IRQ1(x)                   (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_LRFAF_IRQ1_SHIFT))&IPL_RQSR_LRFAF_IRQ1_MASK)
#define IPL_RQSR_TFWU_IRQ0_MASK                  0x100000u
#define IPL_RQSR_TFWU_IRQ0_SHIFT                 20u
#define IPL_RQSR_TFWU_IRQ0_WIDTH                 1u
#define IPL_RQSR_TFWU_IRQ0(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_TFWU_IRQ0_SHIFT))&IPL_RQSR_TFWU_IRQ0_MASK)
#define IPL_RQSR_TFWU_IRQ1_MASK                  0x200000u
#define IPL_RQSR_TFWU_IRQ1_SHIFT                 21u
#define IPL_RQSR_TFWU_IRQ1_WIDTH                 1u
#define IPL_RQSR_TFWU_IRQ1(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_TFWU_IRQ1_SHIFT))&IPL_RQSR_TFWU_IRQ1_MASK)
#define IPL_RQSR_RFWO_IRQ0_MASK                  0x400000u
#define IPL_RQSR_RFWO_IRQ0_SHIFT                 22u
#define IPL_RQSR_RFWO_IRQ0_WIDTH                 1u
#define IPL_RQSR_RFWO_IRQ0(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_RFWO_IRQ0_SHIFT))&IPL_RQSR_RFWO_IRQ0_MASK)
#define IPL_RQSR_RFWO_IRQ1_MASK                  0x800000u
#define IPL_RQSR_RFWO_IRQ1_SHIFT                 23u
#define IPL_RQSR_RFWO_IRQ1_WIDTH                 1u
#define IPL_RQSR_RFWO_IRQ1(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_RFWO_IRQ1_SHIFT))&IPL_RQSR_RFWO_IRQ1_MASK)
#define IPL_RQSR_COL_IRQ0_MASK                   0x1000000u
#define IPL_RQSR_COL_IRQ0_SHIFT                  24u
#define IPL_RQSR_COL_IRQ0_WIDTH                  1u
#define IPL_RQSR_COL_IRQ0(x)                     (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_COL_IRQ0_SHIFT))&IPL_RQSR_COL_IRQ0_MASK)
#define IPL_RQSR_COL_IRQ1_MASK                   0x2000000u
#define IPL_RQSR_COL_IRQ1_SHIFT                  25u
#define IPL_RQSR_COL_IRQ1_WIDTH                  1u
#define IPL_RQSR_COL_IRQ1(x)                     (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_COL_IRQ1_SHIFT))&IPL_RQSR_COL_IRQ1_MASK)
#define IPL_RQSR_RXCE_IRQ0_MASK                  0x4000000u
#define IPL_RQSR_RXCE_IRQ0_SHIFT                 26u
#define IPL_RQSR_RXCE_IRQ0_WIDTH                 1u
#define IPL_RQSR_RXCE_IRQ0(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_RXCE_IRQ0_SHIFT))&IPL_RQSR_RXCE_IRQ0_MASK)
#define IPL_RQSR_RXCE_IRQ1_MASK                  0x8000000u
#define IPL_RQSR_RXCE_IRQ1_SHIFT                 27u
#define IPL_RQSR_RXCE_IRQ1_WIDTH                 1u
#define IPL_RQSR_RXCE_IRQ1(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_RXCE_IRQ1_SHIFT))&IPL_RQSR_RXCE_IRQ1_MASK)
#define IPL_RQSR_TFWU_DMA0_MASK                  0x10000000u
#define IPL_RQSR_TFWU_DMA0_SHIFT                 28u
#define IPL_RQSR_TFWU_DMA0_WIDTH                 1u
#define IPL_RQSR_TFWU_DMA0(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_TFWU_DMA0_SHIFT))&IPL_RQSR_TFWU_DMA0_MASK)
#define IPL_RQSR_TFWU_DMA1_MASK                  0x20000000u
#define IPL_RQSR_TFWU_DMA1_SHIFT                 29u
#define IPL_RQSR_TFWU_DMA1_WIDTH                 1u
#define IPL_RQSR_TFWU_DMA1(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_TFWU_DMA1_SHIFT))&IPL_RQSR_TFWU_DMA1_MASK)
#define IPL_RQSR_RFWO_DMA0_MASK                  0x40000000u
#define IPL_RQSR_RFWO_DMA0_SHIFT                 30u
#define IPL_RQSR_RFWO_DMA0_WIDTH                 1u
#define IPL_RQSR_RFWO_DMA0(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_RFWO_DMA0_SHIFT))&IPL_RQSR_RFWO_DMA0_MASK)
#define IPL_RQSR_RFWO_DMA1_MASK                  0x80000000u
#define IPL_RQSR_RFWO_DMA1_SHIFT                 31u
#define IPL_RQSR_RFWO_DMA1_WIDTH                 1u
#define IPL_RQSR_RFWO_DMA1(x)                    (((uint32_t)(((uint32_t)(x))<<IPL_RQSR_RFWO_DMA1_SHIFT))&IPL_RQSR_RFWO_DMA1_MASK)
/* TXCIR Bit Fields */
#define IPL_TXCIR_TXCIR_MASK                     0xFFFFFFFFu
#define IPL_TXCIR_TXCIR_SHIFT                    0u
#define IPL_TXCIR_TXCIR_WIDTH                    32u
#define IPL_TXCIR_TXCIR(x)                       (((uint32_t)(((uint32_t)(x))<<IPL_TXCIR_TXCIR_SHIFT))&IPL_TXCIR_TXCIR_MASK)
/* TXCRC Bit Fields */
#define IPL_TXCRC_TXCRC_MASK                     0xFFFFFFFFu
#define IPL_TXCRC_TXCRC_SHIFT                    0u
#define IPL_TXCRC_TXCRC_WIDTH                    32u
#define IPL_TXCRC_TXCRC(x)                       (((uint32_t)(((uint32_t)(x))<<IPL_TXCRC_TXCRC_SHIFT))&IPL_TXCRC_TXCRC_MASK)
/* RXCIR Bit Fields */
#define IPL_RXCIR_RXCIR_MASK                     0xFFFFFFFFu
#define IPL_RXCIR_RXCIR_SHIFT                    0u
#define IPL_RXCIR_RXCIR_WIDTH                    32u
#define IPL_RXCIR_RXCIR(x)                       (((uint32_t)(((uint32_t)(x))<<IPL_RXCIR_RXCIR_SHIFT))&IPL_RXCIR_RXCIR_MASK)
/* RXCRC Bit Fields */
#define IPL_RXCRC_RXCRC_MASK                     0xFFFFFFFFu
#define IPL_RXCRC_RXCRC_SHIFT                    0u
#define IPL_RXCRC_RXCRC_WIDTH                    32u
#define IPL_RXCRC_RXCRC(x)                       (((uint32_t)(((uint32_t)(x))<<IPL_RXCRC_RXCRC_SHIFT))&IPL_RXCRC_RXCRC_MASK)
/* TXDR Bit Fields */
#define IPL_TXDR_TXD_MASK                        0xFFFFFFFFu
#define IPL_TXDR_TXD_SHIFT                       0u
#define IPL_TXDR_TXD_WIDTH                       32u
#define IPL_TXDR_TXD(x)                          (((uint32_t)(((uint32_t)(x))<<IPL_TXDR_TXD_SHIFT))&IPL_TXDR_TXD_MASK)
/* RXDR Bit Fields */
#define IPL_RXDR_RXD_MASK                        0xFFFFFFFFu
#define IPL_RXDR_RXD_SHIFT                       0u
#define IPL_RXDR_RXD_WIDTH                       32u
#define IPL_RXDR_RXD(x)                          (((uint32_t)(((uint32_t)(x))<<IPL_RXDR_RXD_SHIFT))&IPL_RXDR_RXD_MASK)

/*!
 * @}
 */ /* end of group IPL_Register_Masks */


/*!
 * @}
 */ /* end of group IPL_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- IPUS Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup IPUS_Peripheral_Access_Layer IPUS Peripheral Access Layer
 * @{
 */


/** IPUS - Size of Registers Arrays */

/** IPUS - Register Layout Typedef */
typedef struct {
  __IO uint32_t HOST_START;                        /**< Command and Start Register, offset: 0x0 */
  __IO uint32_t HOST_STATUS;                       /**< Core Status Register, offset: 0x4 */
  __IO uint32_t HOST_POS;                          /**< Position Configuration Register, offset: 0x8 */
  __I  uint32_t HOST_CURRPOS;                      /**< Current Position Status Register, offset: 0xC */
  __IO uint32_t HOST_XCFG;                         /**< Line Width and Step Configuration Register, offset: 0x10 */
  __I  uint32_t HOST_CURRXCFG;                     /**< Current Line Width and Step Status Register, offset: 0x14 */
  __IO uint32_t HOST_INACFG;                       /**< INA Matrix Configuration Register, offset: 0x18 */
  __IO uint32_t HOST_INBCFG;                       /**< INB Matrix Configuration Register, offset: 0x1C */
  __IO uint32_t HOST_INALPHACFG;                   /**< INALPHA Configuration Register, offset: 0x20 */
  __IO uint32_t HOST_OUTCFG;                       /**< OUT Matrix Configuration Register, offset: 0x24 */
       uint8_t RESERVED_0[24];
  __IO uint32_t HOST_IMEMA;                        /**< Instruction Memory Address Register, offset: 0x40 */
  __IO uint32_t HOST_IMEMD;                        /**< Instruction Memory Data Register, offset: 0x44 */
  __IO uint32_t HOST_DREGA;                        /**< Data Register-Bank Address Register, offset: 0x48 */
  __IO uint32_t HOST_DREGD;                        /**< Data Register-Bank Datum Register, offset: 0x4C */
  __IO uint32_t HOST_HISTA;                        /**< Histogram Memory Address Register, offset: 0x50 */
  __IO uint32_t HOST_HISTD;                        /**< Histogram Memory Data Register, offset: 0x54 */
  __I  uint32_t HOST_HISTPEAK;                     /**< Histogram Peak Value Register, offset: 0x58 */
  __I  uint32_t HOST_HISTAMAX;                     /**< Histogram Maximum Value Address Register, offset: 0x5C */
  __IO uint32_t HOST_HISTCLEAR;                    /**< Histogram Clear Register, offset: 0x60 */
  __IO uint32_t HOST_STATA;                        /**< Statistics Memory Address Register, offset: 0x64 */
  __IO uint32_t HOST_STATD;                        /**< Statistics Memory Data Register, offset: 0x68 */
  __IO uint32_t HOST_STATCLEAR;                    /**< Statistics Memory Clear Register, offset: 0x6C */
       uint8_t RESERVED_1[8];
  __IO uint32_t HOST_LUTA;                         /**< LUT Memory Address Register, offset: 0x78 */
  __IO uint32_t HOST_LUTD;                         /**< LUT Memory Data Register, offset: 0x7C */
       uint8_t RESERVED_2[132];
  __IO uint32_t S_LINE0_LEN_INA;                   /**< INA Stream Line 0 Length Register, offset: 0x104 */
  __IO uint32_t S_CH0_CFG_INA;                     /**< INA Stream Line 0 Configuration Register, offset: 0x108 */
  __IO uint32_t S_LINE0_ADDR_INA;                  /**< INA Stream Line 0 Address Register, offset: 0x10C */
  __IO uint32_t S_LINE1_LEN_INA;                   /**< INA Stream Line 1 Length Register, offset: 0x110 */
  __IO uint32_t S_CH1_CFG_INA;                     /**< INA Stream Line 1 Configuration Register, offset: 0x114 */
  __IO uint32_t S_LINE1_ADDR_INA;                  /**< INA Stream Line 1 Address Register, offset: 0x118 */
  __IO uint32_t S_LINE2_LEN_INA;                   /**< INA Stream Line 2 Length Register, offset: 0x11C */
  __IO uint32_t S_CH2_CFG_INA;                     /**< INA Stream Line 2 Configuration Register, offset: 0x120 */
  __IO uint32_t S_LINE2_ADDR_INA;                  /**< INA Stream Line 2 Address Register, offset: 0x124 */
       uint8_t RESERVED_3[28];
  __IO uint32_t S_LINE0_LEN_INB;                   /**< INB Stream Line 0 Length Register, offset: 0x144 */
  __IO uint32_t S_CH0_CFG_INB;                     /**< INB Stream Line 0 Configuration Register, offset: 0x148 */
  __IO uint32_t S_LINE0_ADDR_INB;                  /**< INB Stream Line 0 Address Register, offset: 0x14C */
  __IO uint32_t S_LINE1_LEN_INB;                   /**< INB Stream Line 1 Length Register, offset: 0x150 */
  __IO uint32_t S_CH1_CFG_INB;                     /**< INB Stream Line 1 Configuration Register, offset: 0x154 */
  __IO uint32_t S_LINE1_ADDR_INB;                  /**< INB Stream Line 1 Address Register, offset: 0x158 */
  __IO uint32_t S_LINE2_LEN_INB;                   /**< INB Stream Line 2 Length Register, offset: 0x15C */
  __IO uint32_t S_CH2_CFG_INB;                     /**< INB Stream Line 2 Configuration Register, offset: 0x160 */
  __IO uint32_t S_LINE2_ADDR_INB;                  /**< INB Stream Line 2 Address Register, offset: 0x164 */
       uint8_t RESERVED_4[28];
  __IO uint32_t S_LINE0_LEN_INALPHA;               /**< INALPHA Stream Line 0 Length Register, offset: 0x184 */
  __IO uint32_t S_CH0_CFG_INALPHA;                 /**< INALPHA Stream 0 Configuration Register, offset: 0x188 */
  __IO uint32_t S_LINE0_ADDR_INALPHA;              /**< INALPHA Stream Line 0 Address Register, offset: 0x18C */
  __IO uint32_t S_LINE1_LEN_INALPHA;               /**< INALPHA Stream Line 1 Length Register, offset: 0x190 */
  __IO uint32_t S_CH1_CFG_INALPHA;                 /**< INALPHA Stream 1 Configuration Register, offset: 0x194 */
  __IO uint32_t S_LINE1_ADDR_INALPHA;              /**< INALPHA Stream Line 1 Address Register, offset: 0x198 */
  __IO uint32_t S_LINE2_LEN_INALPHA;               /**< INALPHA Stream Line 2 Length Register, offset: 0x19C */
  __IO uint32_t S_CH2_CFG_INALPHA;                 /**< INALPHA Stream 2 Configuration Register, offset: 0x1A0 */
  __IO uint32_t S_LINE2_ADDR_INALPHA;              /**< INALPHA Stream Line 2 Address Register, offset: 0x1A4 */
       uint8_t RESERVED_5[28];
  __IO uint32_t S_LINE0_LEN_OUT;                   /**< Stream OUT 0 Length Register, offset: 0x1C4 */
  __IO uint32_t S_CH0_CFG_OUT;                     /**< Stream OUT 0 Configuration Register, offset: 0x1C8 */
  __IO uint32_t S_LINE0_ADDR_OUT;                  /**< Stream OUT 0 Address Register, offset: 0x1CC */
  __IO uint32_t S_LINE1_LEN_OUT;                   /**< Stream OUT 1 Length Register, offset: 0x1D0 */
  __IO uint32_t S_CH1_CFG_OUT;                     /**< Stream OUT 1 Configuration Register, offset: 0x1D4 */
  __IO uint32_t S_LINE1_ADDR_OUT;                  /**< Stream OUT 1 Address Register, offset: 0x1D8 */
  __IO uint32_t S_LINE2_LEN_OUT;                   /**< Stream OUT 2 Length Register, offset: 0x1DC */
  __IO uint32_t S_CH2_CFG_OUT;                     /**< Stream OUT 2 Configuration Register, offset: 0x1E0 */
  __IO uint32_t S_LINE2_ADDR_OUT;                  /**< Stream OUT 2 Address Register, offset: 0x1E4 */
  __IO uint32_t S_LINE3_LEN_OUT;                   /**< Stream OUT 3 Length Register, offset: 0x1E8 */
  __IO uint32_t S_CH3_CFG_OUT;                     /**< Stream OUT 3 Configuration Register, offset: 0x1EC */
  __IO uint32_t S_LINE3_ADDR_OUT;                  /**< Stream OUT 3 Address Register, offset: 0x1F0 */
       uint8_t RESERVED_6[16];
  __I  uint32_t S_CURR_LINE0_LEN_INA;              /**< Current INA Stream Line0 Length Register, offset: 0x204 */
  __I  uint32_t S_CURR_CH0_CFG_INA;                /**< Current INA Stream Channel0 Configuration Register, offset: 0x208 */
  __I  uint32_t S_CURR_LINE0_ADDR_INA;             /**< Current Stream Line0 Address Register, offset: 0x20C */
  __I  uint32_t S_CURR_LINE1_LEN_INA;              /**< Current INA Stream Line1 Length Register, offset: 0x2