/*****************************************************************************
 *
 * Copyright (c) 2018 NXP
 * All Rights Reserved
 *
 *****************************************************************************
 *
 * THIS SOFTWARE IS PROVIDED BY NXP "AS IS" AND ANY EXPRESSED OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL NXP OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

#include "S32V234.h" /* include peripheral declarations S32V234 */
#include "APU_LOAD.h"
#include <string.h>

void setup_apex_memory (uint8_t);
void setup_apex_apu (uint8_t, uint8_t);
void enable_apex_apu(uint8_t apex_id, uint8_t apu_id);
int load_apex_code(uint8_t apex_id, uint8_t apu_id, const SEG_HOST_TYPE (*apex_binary)[4]);


void stall(void)
{
    while (1);
}

int main(void)
{
   // setup APEX APU0
	setup_apex_memory (0);
	setup_apex_apu(0,0);
	if (load_apex_code(0, 0, apx_LOAD_SEGMENTS) < 0) {
		stall();
	}
	enable_apex_apu(0, 0);
    stall();
    return 0;
}


// APEX base address
#define APEX_0_BASE    0x74000000
#define APEX_1_BASE    0x78000000

// APEX memory setup
#define APU_0_DM_START  0x00000000	// default 0x00000000
#define APU_0_PM_START  0x00008000	// default 0x00008000
#define APU_1_DM_START  0x00018000	// default 0x00010000
#define APU_1_PM_START  0x0001D000	// default 0x00018000

// APEX APU registers
#define CMEM_IF_BASE            0x00180000
#define CMEM_IF_APU0_PM_START   0x00000000
#define CMEM_IF_APU1_PM_START   0x00000004
#define CMEM_IF_APU0_DM_START   0x00000008
#define CMEM_IF_APU1_DM_START   0x0000000C
#define GLOBAL_BASE             0x00140000
#define GLOBAL_RST_SET          0x00000000
#define GLOBAL_RST_CLEAR        0x00000004
#define GLOBAL_MISC             0x0000001C

const uint32_t apex_base[] = { APEX_0_BASE, APEX_1_BASE };
const uint32_t pm_start_reg[]  = { CMEM_IF_APU0_PM_START, CMEM_IF_APU1_PM_START };
const uint32_t dm_start_reg[]  = { CMEM_IF_APU0_DM_START, CMEM_IF_APU1_DM_START };
const uint32_t pm_start[]  = { APU_0_PM_START, APU_1_PM_START };
const uint32_t dm_start[]  = { APU_0_DM_START, APU_1_DM_START };

//--APEX registers operations
// Sets bits in given APEX register by mask.
void reg_set(uint8_t apex_id, uint32_t reg, uint32_t bits) {
	uintptr_t addr;
	addr = apex_base[apex_id] + GLOBAL_BASE + reg;
	*(uint32_t*)(addr) |= (bits);
}

// Clears bits in given APEX register by mask.
void reg_clr(uint8_t apex_id, uint32_t reg, uint32_t bits){
	uintptr_t addr;
	addr = apex_base[apex_id] + GLOBAL_BASE + reg;
	*(uint32_t*)(addr) &= ~(bits);

}

// Overwrite the value in given APEX register
void reg_write(uint8_t apex_id, uint32_t reg, uint32_t bits){
	uintptr_t addr;
	addr = apex_base[apex_id] + GLOBAL_BASE + reg;
	*(uint32_t*)(addr) = (bits);
}

//APEX's APU memory prepare by APEX's index
void setup_apex_memory (uint8_t apex_id)
{
	uint32_t apu_id;
	for (apu_id=0; apu_id<2; apu_id++)
	{
		// set APUx_PM_START
		uintptr_t addr = apex_base[apex_id] + CMEM_IF_BASE + pm_start_reg[apu_id];
		*(uint32_t*)(addr) = pm_start[apu_id];
		// set APUx_DM_START
		addr = apex_base[apex_id] + CMEM_IF_BASE + dm_start_reg[apu_id];
		*(uint32_t*)(addr) = dm_start[apu_id];
	}
}


#define IDX 0 //defines index of index in array
#define SRC_ADDR 1 //defines index of source address in array
#define DST_ADDR 2 //defines index of destination address in array
#define SIZE 3 // defines index of size in array
int load_apex_code(uint8_t apex_id, uint8_t apu_id, const SEG_HOST_TYPE (*apex_binary)[4])
{
	uint16_t idx = 0;
	uint16_t size = 0;
	SEG_HOST_TYPE *src, *dst;

	while (true){
		int t_code = apex_binary[idx][IDX];
		if (t_code == 0) {
			dst = (SEG_HOST_TYPE *)(apex_binary[idx][DST_ADDR] + APEX_0_BASE + pm_start[apu_id]);
		}else if (t_code == 1){
			dst = (SEG_HOST_TYPE *)(apex_binary[idx][DST_ADDR] + APEX_0_BASE + dm_start[apu_id]);
		}else if (t_code == -1) {
			break;
		}else {
			return -1; 
		}

		src = (SEG_HOST_TYPE *)apex_binary[idx][SRC_ADDR];
		size =  apx_LOAD_SEGMENTS[idx][SIZE];

		memcpy(dst, src, size);
		idx++;
	}
	return 0;
}

// Apply memory settings for given APEX/APU by index 
void setup_apex_apu (uint8_t apex_id, uint8_t apu_id)
{
	// disable APU
	reg_clr(apex_id, GLOBAL_MISC, 1<<apu_id);
	// reset APU
	reg_write(apex_id, GLOBAL_RST_SET, 1<<(apu_id+7));
	reg_write(apex_id, GLOBAL_RST_CLEAR, 1<<(apu_id+7));
}

// Enable given APEX/APU by index
void enable_apex_apu(uint8_t apex_id, uint8_t apu_id){
	reg_set(apex_id, GLOBAL_MISC, 1<<apu_id);
}






