/*******************************************************************************
 * Copyright (c) 2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution.
 *******************************************************************************/
package com.freescale.sa.configurator.ui.dialogs;

import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.freescale.sa.configurator.ui.ICustomizablePlatformConfig;
import com.freescale.sa.configurator.ui.Messages;

/**
 * Dialog for setting a new name for the platform configuration file.
 */
public class RenamePlatformConfigDialog extends TitleAreaDialog {

	ICustomizablePlatformConfig m_Owner;
	Text m_NewNameText;
	String m_PlatformToRename;
	/**
	 * Constructor.
	 * 
	 * @param parentShell
	 *            the parent shell
	 * @param owner
	 *            The object that created this dialog. It must implement the
	 *            ICustomizablePlatformConfig interface to allow making updates
	 *            in the objects that use the platform configuration file.
	 */
	public RenamePlatformConfigDialog(Shell parentShell,
			ICustomizablePlatformConfig owner, String platformToRename) {
		super(parentShell);
		m_Owner = owner;
		m_PlatformToRename=platformToRename;
	}

	@Override
	public void create() {
		super.create();
		setTitle(Messages.PlatformCfg_RenameDialog_Title);
		setMessage(Messages.PlatformCfg_RenameDialog_Default_Message);
		// by default the OK button is disable so the user can't rename new file with empty string
		getButton(OK).setEnabled(false);
	}

	@Override
	protected Control createDialogArea(Composite parent) {
		GridLayout gridLayout = new GridLayout(2, false);
		parent.setLayout(gridLayout);
		
		//Remove extension from config file name
		m_PlatformToRename = m_PlatformToRename.substring(0, m_PlatformToRename.lastIndexOf("."));
		//Set the replace textbox to the name of the selected platform config
		m_NewNameText = new Text(parent, SWT.BORDER);
		m_NewNameText.setText(m_PlatformToRename);
		
		GridData gridData = new GridData(SWT.FILL, SWT.CENTER, true, false, 1, 1);
		m_NewNameText.setLayoutData(gridData);
		m_NewNameText.addKeyListener(new KeyAdapter() {
			public void keyPressed(KeyEvent event) {
				switch (event.keyCode) {
				case (SWT.CR):
				case (SWT.KEYPAD_CR):
					boolean valid = PlatformConfigDialogUtils.checkPlatformCfgName(
							RenamePlatformConfigDialog.this,
							m_NewNameText.getText(),
							m_Owner.getPlatformConfigStorageFolder());
					if (valid) {
						updateName(m_NewNameText.getText());
						m_NewNameText.dispose();
					}
					break;
				case SWT.ESC:
					m_NewNameText.dispose();
					break;
				default:
					break;
				}
			}

		});

		m_NewNameText.addModifyListener(new ModifyListener() {
			@Override
			public void modifyText(ModifyEvent e) {
				boolean valid = PlatformConfigDialogUtils.checkPlatformCfgName(
						RenamePlatformConfigDialog.this,
						m_NewNameText.getText(),
						m_Owner.getPlatformConfigStorageFolder());

				if (valid) {
					getButton(OK).setEnabled(true);
				} else {
					getButton(OK).setEnabled(false);
				}
			}
		});

		return parent;
	}

	@Override
	public boolean isHelpAvailable() {
		// Don't want help for this window.
		return false;
	}

	@Override
	protected void okPressed() {
		if (!m_NewNameText.isDisposed()) {
			updateName(m_NewNameText.getText());
		}
		super.okPressed();
	}

	/**
	 * Make any updates required by the file name change. The method assumes the
	 * new file name is valid.
	 * 
	 * @param newName
	 *            the new platform config name
	 */
	private void updateName(String newName) {
		m_Owner.updatePlatformConfigName(newName);
	}

	@Override
	protected void configureShell(Shell theShell) {
		super.configureShell(theShell);
		theShell.setText(Messages.PlatformCfg_RenameCfg_Shell_Title); 
	}

}
