/*******************************************************************************
 * Copyright 2016-2018 NXP

 * Copyright (c) 2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution.
 *******************************************************************************/
package com.freescale.sa.configurator.ui.dialogs;

import java.io.File;
import java.io.FileFilter;

import org.eclipse.core.runtime.IPath;
import org.eclipse.jface.dialogs.TitleAreaDialog;

import com.freescale.sa.SaConstants;
import com.freescale.sa.configurator.ui.Messages;
import com.freescale.sa.util.CommonConstants;

/**
 * Utility methods for the platform config dialog classes
 * (RenamePlatformConfigDialog, NewPlatformConfigDialog).
 */
public class PlatformConfigDialogUtils {
	

	/**
	 * Checks if the platform configuration name is valid.
	 * 
	 * @param dialog
	 *			the title area dialog used to set the platform configuration
	 *			name
	 * @param newPlatCfgName
	 *			the new platform configuration name
	 * @param platCfgFolder
	 *			the parent folder of the platform configuration
	 * @return true if the name is correct, false otherwise
	 */
	public static boolean checkPlatformCfgName(TitleAreaDialog dialog,
			String newPlatCfgName, IPath platCfgFolder) {
		if (platCfgFolder == null) {
			return false;
		}
	
		// \w - A word character, short for [a-zA-Z_0-9].
		// \S - A non-whitespace character.
		if (newPlatCfgName.matches("[A-Za-z0-9\\_\\-]+")) { //$NON-NLS-1$
			// Check if there is a set of results with the same name (not
			// allowed).
			if (configFileExists(newPlatCfgName, platCfgFolder)) {
				dialog.setErrorMessage(Messages.PlatformCfg_ErrMsg_AlreadyExists);
				return false;
			}

		} else {
			dialog.setErrorMessage(Messages.PlatformCfg_ErrMsg_ValidName);
			return false;
		}

		if ((platCfgFolder.toString().length() + newPlatCfgName.length()) > CommonConstants.MAX_LENGTH) {
			dialog.setErrorMessage(Messages.PlatformCfg_ErrMsg_PathTooLong);
			return false;
		}

		dialog.setErrorMessage(null);
		return true;

	}

	/**
	 * Checks if the platform configuration file with the given name exists in
	 * the specified folder.
	 * 
	 * @param platfCfgFileName
	 *			the name of the platform configuration file.
	 * @param parentFolder
	 *			the folder where to search
	 * @return true if the file was found, false otherwise
	 */
	public static boolean configFileExists(String platfCfgFileName, IPath parentFolder) {
		if (parentFolder == null) {
			return false;
		}
		File dir = parentFolder.toFile();
		if (!dir.exists()) {
			return true;
		}

		File[] platConfigs = dir.listFiles(new FileFilter() {

			@Override
			public boolean accept(File file) {
				String fileName = file.getName();
				int extensionPos = fileName.lastIndexOf('.');
				String extension = fileName.substring(extensionPos + 1);
				return (extension.equalsIgnoreCase(SaConstants.PLATFORM_CONFIG_EXTENSION));
			}
			
		});

		IPath newPath = parentFolder.append(platfCfgFileName + "." //$NON-NLS-1$
				+ SaConstants.PLATFORM_CONFIG_EXTENSION);

		if (platConfigs == null) {
			return false;
		} else {
			for (int i = 0; i < platConfigs.length; i++) {
				if (platConfigs[i].getAbsolutePath().equalsIgnoreCase(newPath.toFile().getAbsolutePath())) {
					return true;
				}
			}
		}

		return false;
	}
}
