/*******************************************************************************
 * Copyright (c) 2014, 2015 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution.
 *******************************************************************************/
package com.freescale.sa.configurator.ui.dialogs;

import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;

import com.freescale.sa.configurator.ui.Messages;
import com.freescale.sa.util.CommonConstants;

/**
 * This class holds the controls for choosing a custom folder for trace results/output
 */
public class OutputFolderControls {
	/* Miscellaneous tab configuration options */
	protected Text m_outputFolderText;
	protected boolean m_outputFolderTextValid = true;
	protected Label m_outputFolderErrorLabel;
	protected Label m_outputFolderErrorImage;
	protected boolean m_outputFolderTextRestoring = false;
	protected Button m_browseOutputFolderButton;
	
	private boolean m_updatePlatformConfig;
	private Composite m_parent;
	
	protected Label m_outputFolderLabel;
	protected Composite m_outputFolderComposite;
	
	// Results folder cached input.
    private String outputFolderCache = null;
	
	public OutputFolderControls(boolean updatePlatformConfig, Composite parent) {
		m_updatePlatformConfig = updatePlatformConfig;
		m_parent = parent;
	}
	
	/**
	 * Create a section of the Analysis config with the controls for modifying the
	 * location of the SA output folder.
	 */
	public void createOutputFolderControls() {
		m_outputFolderComposite = new Composite(m_parent, SWT.NONE);
		m_outputFolderComposite.setLayout(new GridLayout(3, false));
		m_outputFolderComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL,
				true, true, 1, 1));

		m_outputFolderLabel = new Label(m_outputFolderComposite, SWT.NONE);
		m_outputFolderLabel.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false, 1, 1));
		setupOutputFolderLabel();

		m_outputFolderText = new Text(m_outputFolderComposite, SWT.BORDER);
		
		GridData outputFolderTextGridData = new GridData(SWT.FILL, SWT.CENTER, true, false, 1, 1);
		
		if (m_updatePlatformConfig) {
			outputFolderTextGridData.widthHint = 140;
		} else {
			outputFolderTextGridData.widthHint = 300;
		}
				
		m_outputFolderText.setLayoutData(outputFolderTextGridData);
		
		m_outputFolderText.addFocusListener(new FocusListener(){

			@Override
			public void focusGained(FocusEvent e) {
				// Nothing to do in this case
			}

			@Override
			public void focusLost(FocusEvent e) {
				IPath path = new Path(m_outputFolderText.getText());
				if (!m_outputFolderTextValid) {
					outputFolderCache = m_outputFolderText.getText();
					return;
				}
				
				String outFolder = m_outputFolderText.getText();
				outputFolderCache = m_outputFolderText.getText();
				
				IPath checkOutputPath = new Path(outFolder);
				
				if (checkOutputPath.lastSegment() == null) {
					return;
				}
								
				if (!path.isEmpty()){
					m_outputFolderText.setText(path.toString());
					
					setOutputResultsPath(path.toString());
				}
			}
			
		});
		
		m_browseOutputFolderButton = new Button(m_outputFolderComposite, SWT.PUSH);
		m_browseOutputFolderButton.setText(Messages.Browse_Output_Folder);
		setupBrowseButton(m_outputFolderComposite);
		
		Composite errorComp = new Composite(m_parent, SWT.NONE);
		errorComp.setLayout(new GridLayout(2, false));
		errorComp.setLayoutData(new GridData(SWT.LEFT_TO_RIGHT, SWT.TOP,
				true, true, 1, 1));
		m_outputFolderErrorImage = new Label(errorComp, SWT.RIGHT);
		m_outputFolderErrorImage.setImage(PlatformUI.getWorkbench().getSharedImages().getImage(ISharedImages.IMG_OBJS_ERROR_TSK));
		m_outputFolderErrorImage.setVisible(false);
		m_outputFolderErrorLabel = new Label(errorComp, SWT.NONE);
		GridData saOutputFolderErrorGridData = new GridData();
		
		saOutputFolderErrorGridData.widthHint = 350;
		m_outputFolderErrorLabel.setLayoutData(saOutputFolderErrorGridData);
	}

	public boolean modifyText(ModifyEvent e) {
		String errorText = null;
		if (!isPathValid()) {
			errorText = Messages.Output_Folder_Full_Path;
		} else {
			errorText = CommonConstants.EMPTY_STRING;
		}

		showErrorLabel(errorText);

		// If output folder restoring was needed, now is completed.
		m_outputFolderTextRestoring = m_outputFolderTextRestoring ? false : m_outputFolderTextRestoring;

		return m_outputFolderTextValid;
	}

	public Text getOutputResultsText() {
		return m_outputFolderText;
	}

	protected void setupOutputFolderLabel() {
		m_outputFolderLabel.setText(Messages.Output_Folder_Text_Label);
		m_outputFolderLabel.setToolTipText(Messages.Output_Folder_Text_Tooltip);
	}
	
	protected void setupBrowseButton(Composite outputFolderComposite) {
		m_browseOutputFolderButton
		.addSelectionListener(new BrowseButtonSelectionAdapter(
				outputFolderComposite));
	}
	
	public String getOutputResultsPath() {
		if (m_outputFolderTextValid) {
			return m_outputFolderText.getText();
		}
		
		return m_outputFolderText.getText();
	}
	
	public void setOutputResultsPath(String path) {
		if (path != null ) {
			m_outputFolderText.setText(path);
		}
	}
	
	public void showErrorLabel(String message){
		m_outputFolderErrorLabel.setText(message);
		m_outputFolderErrorImage.setVisible(!message.isEmpty());
		m_outputFolderErrorImage.setToolTipText(message);
	}
	
	public boolean isPathValid() {
		if (m_outputFolderText.getText() != null) {
			IPath outputPath = new Path(m_outputFolderText.getText().trim());
			m_outputFolderTextValid = true;
			if (!outputPath.isEmpty()) {
				if (!outputPath.isAbsolute()){
					m_outputFolderTextValid = false;
				} else if (outputPath.toString().length() > CommonConstants.MAX_LENGTH) {
					m_outputFolderTextValid = false;
				} else if (!outputPath.toFile().exists()) {
					m_outputFolderTextValid = false;
				}
			} else {
				m_outputFolderTextValid = false;
			}
		} else {
			m_outputFolderTextValid = false; 
		}
		return m_outputFolderTextValid;
	}
	
	/**
	 * Delete results from resultsPath if they exist. 
	 */
	public void setNewPath(IPath resultsPath, Composite parent){
		m_outputFolderText.setText(resultsPath.toOSString());
		setOutputFolderCache(resultsPath.toOSString());
	}
	
	public boolean getUpdatePlatformConfig() {
		return m_updatePlatformConfig;
	}

	public void setUpdatePlatformConfig(boolean updatePlatformConfig) {
		m_updatePlatformConfig = updatePlatformConfig;
	}

	/**
	 * 
	 * Selection adapter for the button used to browse for the SA output folder.
	 */
	private class BrowseButtonSelectionAdapter extends SelectionAdapter {

		private Composite parentComposite;

		public BrowseButtonSelectionAdapter(Composite parentComposite) {
			this.parentComposite = parentComposite;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see
		 * org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse
		 * .swt.events.SelectionEvent)
		 */
		@Override
		public void widgetSelected(SelectionEvent e) {
			super.widgetSelected(e);

			DirectoryDialog dialog = new DirectoryDialog(
					parentComposite.getShell());
			dialog.setFilterPath(m_outputFolderText.getText());
			dialog.setText(Messages.Output_Folder_Dialog_Title);
			dialog.setMessage(Messages.Output_Folder_Dialog_Message);

			String oldPath = m_outputFolderText.getText();
			String newPath = dialog.open();

			if ((newPath != null) && (!newPath.equalsIgnoreCase(oldPath))) {
				IPath saOutputDirPath = new Path(newPath);
				setNewPath(saOutputDirPath, parentComposite);
			}
		}
	}

	public String getOutputFolderCache() {
		return outputFolderCache;
	}

	public void setOutputFolderCache(String resultsFolderCache) {
		this.outputFolderCache = resultsFolderCache;
	}
}
