/*******************************************************************************
 * Copyright (c) 2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution.
 *******************************************************************************/
package com.freescale.sa.configurator.ui.dialogs;

import java.util.List;

import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.freescale.sa.configurator.ui.ICustomizablePlatformConfig;
import com.freescale.sa.configurator.ui.Messages;

/**
 * Dialog for creating a new platform configuration file.
 */
public class NewPlatformConfigDialog extends TitleAreaDialog {

	ICustomizablePlatformConfig m_Owner;
	Text m_FileNameText;
	Combo m_boards;

	/**
	 * Constructor.
	 * 
	 * @param parentShell
	 *            the parent shell
	 * @param owner
	 *            The object that created this dialog. It must implement the
	 *            ICustomizablePlatformConfig interface to allow making updates
	 *            in the objects that use the platform configuration file.
	 */
	public NewPlatformConfigDialog(Shell parentShell,
			ICustomizablePlatformConfig owner) {
		super(parentShell);
		m_Owner = owner;
	}

	@Override
	protected Control createDialogArea(Composite parent) {
		GridLayout gridLayout = new GridLayout(2, false);
		parent.setLayout(gridLayout);
	
		m_FileNameText = new Text(parent, SWT.BORDER);
		GridData gridData = new GridData(SWT.FILL, SWT.TOP, true, false, 1, 1);
		m_FileNameText.setLayoutData(gridData);
		m_FileNameText.addKeyListener(new KeyAdapter() {
			public void keyPressed(KeyEvent event) {
				switch (event.keyCode) {
				case (SWT.CR):
				case (SWT.KEYPAD_CR):
					boolean valid = PlatformConfigDialogUtils.checkPlatformCfgName(
							NewPlatformConfigDialog.this,
							m_FileNameText.getText(),
							m_Owner.getPlatformConfigStorageFolder());
					if (valid) {
						updateName(m_FileNameText.getText());
						m_FileNameText.dispose();
					}
					break;
				case SWT.ESC:
					m_FileNameText.dispose();
					break;
				default:
					break;
				}
			}

		});
		
		m_FileNameText.addModifyListener(new ModifyListener() {
			@Override
			public void modifyText(ModifyEvent e) {
				boolean valid = PlatformConfigDialogUtils.checkPlatformCfgName(
						NewPlatformConfigDialog.this, m_FileNameText.getText(),
						m_Owner.getPlatformConfigStorageFolder());
				if (valid) {
					getButton(OK).setEnabled(true);
				} else {
					getButton(OK).setEnabled(false);
				}
			}
		});
		
		new Label(parent, SWT.NONE);
		m_boards = new Combo(parent, SWT.READ_ONLY);
		new Label(parent, SWT.NONE);

		List<String> procs = m_Owner.getSupportedBoards();
		
		for(String p : procs)
		{
			m_boards.add(p);
		}
		
		m_boards.select(0);
		
		return parent;
	}

	@Override
	public boolean isHelpAvailable() {
		// Don't want help for this window.
		return false;
	}

	@Override
	protected void okPressed() {
		if (!m_FileNameText.isDisposed()) {
			updateName(m_FileNameText.getText());
		}
		super.okPressed();
	}

	@Override
	public void create() {
		super.create();
		setTitle(Messages.PlatformCfg_NewDialog_Title);
		setMessage(Messages.PlatformCfg_NewDialog_Default_Message);
		// by default the OK button is disable so the user can't create a new file with empty name
		getButton(OK).setEnabled(false);
	}

	@Override
	protected void configureShell(Shell theShell) {
		super.configureShell(theShell);
		theShell.setText(Messages.PlatformCfg_NewCfg_Shell_Title); 
	}

	/**
	 * Make any updates required by the file name change. The method assumes the
	 * new file name is valid.
	 * 
	 * @param newFileName
	 *            the new platform config name
	 */
	private void updateName(String newFileName) {
		m_Owner.addNewPlatformConfig(newFileName,m_boards.getItem(m_boards.getSelectionIndex()));
	}

}
