/*******************************************************************************
 * Copyright (c) 2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.configurator.ui;

import org.apache.log4j.Logger;
import org.eclipse.cdt.debug.core.ICDTLaunchConfigurationConstants;
import org.eclipse.cdt.managedbuilder.core.IManagedBuildInfo;
import org.eclipse.cdt.managedbuilder.core.ManagedBuildManager;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Display;

import com.freescale.sa.SaConstants;
import com.freescale.sa.configurator.config.IRseUpdateTrigger;
import com.freescale.sa.configurator.config.IRseUpdateTrigger.LaunchPropType;
import com.freescale.sa.util.CommonConstants;

public final class LaunchUtils {

	static Logger LOGGER = Logger.getLogger(LaunchUtils.class);

	/**
	 * Get simulator type.
	 * 
	 * @param config
	 *            Launch configuration.
	 * @return Simulator type as string. Empty string if current configuration
	 *         does not use simulator.
	 */
	public static String getSimType(ILaunchConfiguration config) {

		try {
			String simType = (config.getAttribute(
					Constants.LAUNCH_SIM_TYPE_ATTR,
					CommonConstants.EMPTY_STRING)).trim();
			return simType;
		} catch (Exception e) {
			LOGGER.error("Could not determine current simulator type\n"); //$NON-NLS-1$
			return CommonConstants.EMPTY_STRING;
		}

	}

	/**
	 * Get the project associated with the launch config given as a parameter.
	 * 
	 * @param launchConfig
	 *            The launch config for which you want to obtain the project.
	 * @return The project associated with the launch config.
	 */
	public static IProject getProjectFromLaunchConfig(
			ILaunchConfiguration launchConfig) {
		String projName = CommonConstants.EMPTY_STRING;
		IProject project = null;

		try {
			projName = launchConfig.getAttribute(
					ICDTLaunchConfigurationConstants.ATTR_PROJECT_NAME,
					(String) null);
		} catch (CoreException e) {
			LOGGER.error(
					"Unable to get project name from config " + launchConfig.getName(), e); //$NON-NLS-1$
		}

		if (projName != null && !projName.equals(CommonConstants.EMPTY_STRING)) {
			project = ResourcesPlugin.getWorkspace().getRoot()
					.getProject(projName);
		}

		return project;
	}

	/**
	 * Get project name from launch configuration.
	 * 
	 * @param launchConfig
	 *            Launch configuration from where to get processor name.
	 * @return Project name as string.
	 */
	public static synchronized String getProjectNameFromLaunchConfig(
			ILaunchConfiguration launchConfig) {
		String procName = CommonConstants.EMPTY_STRING;

		if (launchConfig != null) {

			String procNameAttr = CommonConstants.LAUNCH_PROJECT_ATTR;
			try {
				procName = launchConfig.getAttribute(procNameAttr,
						CommonConstants.EMPTY_STRING);
			} catch (CoreException e) {
				LOGGER.error(
						"Unable to get project name from config " + launchConfig.getName(), e); //$NON-NLS-1$
			}
		}

		return procName;

	}

	/**
	 * Get elf name from launch configuration.
	 * 
	 * @param launchConfig
	 *            Launch configuration from where to get processor name.
	 * @return Elf name as string.
	 */
	public static synchronized String getElfNameFromLaunchConfig(ILaunchConfiguration launchConfig){
		String procName = CommonConstants.EMPTY_STRING;
			
		if(launchConfig != null) {
			
			String procNameAttr = CommonConstants.LAUNCH_ELF_FILE_ATTR;
			try {
				procName = launchConfig.getAttribute(procNameAttr, CommonConstants.EMPTY_STRING);
			} catch(CoreException e) {
				LOGGER.error("Unable to get elf name from config " + launchConfig.getName(), e); //$NON-NLS-1$
			}

			String projName = getProjectNameFromLaunchConfig(launchConfig);
			IProject proj = ResourcesPlugin.getWorkspace().getRoot().getProject(projName);
			IManagedBuildInfo buildInfo = ManagedBuildManager.getBuildInfo(proj);
			String config = buildInfo.getConfigurationName();

			procName = procName.replace("${BuildLocation}", config);  //$NON-NLS-1$
			
		}
		
		return procName;
	}

	/**
	 * Open warning dialog using as parent the launch plugin's shell.
	 * 
	 * @param title
	 *            Title of dialog.
	 * @param message
	 *            Message of dialog.
	 * @param sync
	 *            Display mode. True for message dialog to be displayed
	 *            synchronously. False for message dialog to be displayed
	 *            asynchronously
	 */
	public static void openWarningDialog(final String title,
			final String message, boolean sync) {

		Runnable run = new Runnable() {
			public void run() {

				MessageDialog.openWarning(
						Display.getDefault().getActiveShell(), title, message);

			}
		};
		if (sync) {
			Display.getDefault().syncExec(run);
		} else {
			Display.getDefault().asyncExec(run);
		}
	}

	/**
	 * The name of the launch config attributes change between SIM and HW. This
	 * methods return the correct name based on target type.
	 * 
	 * @param config
	 *            The current launch config.
	 * @param propType
	 *            An identifier for the launch config attribute for which you
	 *            want to obtain the name.
	 * @param rseUpdateTrigger
	 *            an object that triggers a RSE update
	 * @return The name of the launch config attribute, or an empty string if it
	 *         was not found.
	 */
	public static String getLaunchAttrName(
			ILaunchConfigurationWorkingCopy config, LaunchPropType propType,
			IRseUpdateTrigger rseUpdateTrigger) {
		try {
			if (propType == LaunchPropType.CCS_EXE_NAME) {
				if (config.hasAttribute(Constants.LAUNCH_CCS_EXE_NAME_SIM)) {
					return Constants.LAUNCH_CCS_EXE_NAME_SIM;
				}
				if (config.hasAttribute(Constants.LAUNCH_CCS_EXE_NAME_HW)) {
					return Constants.LAUNCH_CCS_EXE_NAME_HW;
				}
			} else if (propType == LaunchPropType.SPECIFY_CCS_EXE_NAME) {
				if (config
						.hasAttribute(Constants.LAUNCH_SPECIFY_CCS_EXE_NAME_SIM)) {
					return Constants.LAUNCH_SPECIFY_CCS_EXE_NAME_SIM;
				}
				if (config
						.hasAttribute(Constants.LAUNCH_SPECIFY_CCS_EXE_NAME_HW)) {
					return Constants.LAUNCH_SPECIFY_CCS_EXE_NAME_HW;
				}
			}
		} catch (CoreException e) {
			LOGGER.error("[getLaunchProperyName]: error obtaining launch property type: " //$NON-NLS-1$
					+ e.getLocalizedMessage());
			return CommonConstants.EMPTY_STRING;
		}

		LOGGER.error("[getLaunchProperyName]: unrecognized launch property type: " //$NON-NLS-1$
				+ propType);
		return CommonConstants.EMPTY_STRING;

	}

	public static boolean isLaunchConfigInDebug(String name) {
		return false;
	}

	public static void setStartTraceFromLaunch(ILaunch launch, boolean b) {
	}

	public static void setEnableUserCode(
			ILaunchConfigurationWorkingCopy launchConfiguration, boolean b) {
		launchConfiguration.setAttribute(SaConstants.ATTR_ENABLE_USER_CODE, b);
	}

	public static boolean getEnableUserCode(
			ILaunchConfiguration launchConfiguration) {
		if (launchConfiguration == null) {
			return false;
		}

		try {
			return launchConfiguration.getAttribute(
					SaConstants.ATTR_ENABLE_USER_CODE, false);
		} catch (CoreException e) {
			LOGGER.error(e.getLocalizedMessage());
		}

		return false;
	}

	public static void setEnableCollectAutomatically(
			ILaunchConfigurationWorkingCopy launchConfiguration, boolean b) {
		launchConfiguration.setAttribute(
				SaConstants.ATTR_ENABLE_COLLECT_AUTOMATICALLY, b);
	}

	public static boolean getEnableCollectAutomatically(
			ILaunchConfiguration launchConfiguration) {
		if (launchConfiguration == null) {
			return false;
		}

		try {
			return launchConfiguration.getAttribute(
					SaConstants.ATTR_ENABLE_COLLECT_AUTOMATICALLY, true);
		} catch (CoreException e) {
			LOGGER.error(e.getLocalizedMessage());
		}

		return false;
	}
}
