/*******************************************************************************
 * Copyright (c) 2014, 2015 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.configurator.ui;

import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.Logger;
import org.eclipse.cdt.launch.ui.CLaunchConfigurationTab;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.custom.StackLayout;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;

import com.freescale.sa.configurator.config.AnalysisConfigurationContents;
import com.freescale.sa.configurator.config.IAnalysisConfigurationContents;
import com.freescale.sa.model.PlatformType;
import com.freescale.sa.util.CommonConstants;

/**
 * Analysis launch configuration tab class.
 */
public abstract class AnalysisLaunchConfigurationTab extends
		CLaunchConfigurationTab {

	static final Logger LOGGER = Logger
			.getLogger(AnalysisLaunchConfigurationTab.class);

	protected IAnalysisConfigurationContents m_TabContents = null;

	protected Composite m_Parent = null;
	protected ScrolledComposite m_TabScrollComposite = null;
	protected Composite m_TabComposite = null;

	protected String m_ProcessorName = CommonConstants.EMPTY_STRING;

	protected boolean m_Updating = false;
	protected boolean m_Initialized = false;
	protected boolean m_Deactivating = false;
	protected boolean m_Activating = false;

	protected String m_PrevConfigName = CommonConstants.EMPTY_STRING;

	public static class ConfigTabContentsKey {
		private String launchConfigType;
		private String processor;
		private String core;

		public ConfigTabContentsKey(String launchConfig, String processor,
				String core) {
			this.launchConfigType = launchConfig;
			this.processor = processor;
			this.core = core;
		}

		@Override
		public boolean equals(Object key) {
			if (this == key)
				return true;

			if (!(key instanceof ConfigTabContentsKey))
				return false;

			ConfigTabContentsKey cfgKey = (ConfigTabContentsKey) key;
			return (this.launchConfigType.compareTo(cfgKey.launchConfigType) == 0
					&& this.processor.compareTo(cfgKey.processor) == 0 && this.core
						.compareTo(cfgKey.core) == 0);
		}

		@Override
		public int hashCode() {
			return String
					.format("%s-%s-%s", this.launchConfigType, this.processor, this.core).hashCode(); //$NON-NLS-1$
		}
	}

	protected Map<ConfigTabContentsKey, IAnalysisConfigurationContents> m_Tabs = new HashMap<ConfigTabContentsKey, IAnalysisConfigurationContents>();

	@Override
	public void createControl(Composite parent) {
		this.m_Parent = parent;
		this.setTabScrollComposite(new ScrolledComposite(this.m_Parent,
				SWT.V_SCROLL | SWT.H_SCROLL));
		this.setTabComposite(new Composite(this.getTabScrollComposite(),
				SWT.NULL));

		final StackLayout stackLayout = new StackLayout();
		getTabComposite().setLayout(stackLayout);
		getTabComposite().setLayoutData(
				new GridData(SWT.FILL, SWT.TOP, false, true, 1, 1));

		m_TabContents = new AnalysisConfigurationContents();
		m_TabContents.setParentTab(this);

		stackLayout.topControl = m_TabContents
				.createControls(getTabComposite());
		getTabComposite().layout();
		getTabComposite().pack();
		getTabComposite().redraw();

		getTabScrollComposite().setContent(getTabComposite());
		getTabScrollComposite().setExpandHorizontal(true);
		getTabScrollComposite().setExpandVertical(true);

		setControl(getTabScrollComposite());
	}

	@Override
	public void setDefaults(ILaunchConfigurationWorkingCopy config) {
		if (m_TabContents != null) {
			m_TabContents.setDefaults(config);
		}
	}

	protected abstract void updateContents(String processorName,
			ILaunchConfiguration config);

	@Override
	public void performApply(ILaunchConfigurationWorkingCopy config) {
		StackTraceElement[] elements = Thread.currentThread().getStackTrace();
		// Test if performApply was called from refresh method or
		// updating/activating/deactivating.
		if (elements[2].getMethodName().equals("refresh") || m_Updating || m_Activating || m_Deactivating) { //$NON-NLS-1$
			m_TabContents.updateConfig(config);
		} else {
			if (m_TabContents != null)
				m_TabContents.performApply(config);
		}
	}

	@Override
	public void activated(ILaunchConfigurationWorkingCopy workingCopy) {
		m_Activating = true;
		super.activated(workingCopy);
		m_Activating = false;
	}

	@Override
	public void deactivated(ILaunchConfigurationWorkingCopy workingCopy) {
		m_Deactivating = true;
		super.deactivated(workingCopy);
		m_Deactivating = false;
	}

	@Override
	public void initializeFrom(ILaunchConfiguration config) {

		if (!m_PrevConfigName.equals(config.getName())) {
			initFieldsForConfig(config);
		}

		if (!m_Initialized) {
			m_Initialized = true;
		}

		String newProcessorName = PlatformType.getProcessorName(config);
		/*
		 * Why not just sending the config as a parameter. We could squeeze the
		 * platform type out of the config there as well ...
		 */

		updateContents(newProcessorName, config);

		m_TabContents.initializeFrom(config);

	}

	@Override
	public void updateLaunchConfigurationDialog() {
		m_Updating = true;
		super.updateLaunchConfigurationDialog();
		m_Updating = false;
	}

	@Override
	public void createVerticalSpacer(Composite comp, int colSpan) {
		super.createVerticalSpacer(comp, colSpan);
	}

	@Override
	public String getName() {
		return Messages.ANALYSIS_TAB_NAME;
	}

	@Override
	public Image getImage() {
		ImageDescriptor descriptor = Activator
				.getImageDescriptor("icons/trace.gif");//$NON-NLS-1$

		Image image = descriptor.createImage();
		return image;
	}

	@Override
	public boolean isValid(ILaunchConfiguration config) {
		boolean isValidConfig = m_TabContents.isValid(config);
		if (!isValidConfig) {
			setErrorMessage(Messages.SAConfigurationTargetInvalidConfiguration);
		}
		return isValidConfig;
	}

	public void setControl(Control control) {
		super.setControl(control);
	}

	public Button createCheckButton(Composite parent, String label) {
		return super.createCheckButton(parent, label);
	}

	public Button createPushButton(Composite parent, String label, Image image) {
		return super.createPushButton(parent, label, image);
	}

	public Shell getShell() {
		return super.getShell();
	}

	public Button createRadioButton(Composite parent, String label) {
		return super.createRadioButton(parent, label);
	}

	public void setErrorMessage(String errorMessage) {
		super.setErrorMessage(errorMessage);
	}

	public void setMessage(String message) {
		super.setMessage(message);
	}

	public String getProcessorName() {
		return m_ProcessorName;
	}

	/**
	 * Initializes fields to their default values.
	 * 
	 * @param config
	 *            the launch configuration.
	 */
	protected void initFieldsForConfig(ILaunchConfiguration config) {
		m_Updating = false;
		m_Initialized = false;

		if (config != null) {
			m_PrevConfigName = config.getName();
		} else {
			m_PrevConfigName = CommonConstants.EMPTY_STRING;
		}

		m_ProcessorName = CommonConstants.EMPTY_STRING;
		m_TabContents = null;
	}

	public ScrolledComposite getTabScrollComposite() {
		return m_TabScrollComposite;
	}

	public void setTabScrollComposite(ScrolledComposite m_TabScrollComposite) {
		this.m_TabScrollComposite = m_TabScrollComposite;
	}

	public Composite getTabComposite() {
		return m_TabComposite;
	}

	public void setTabComposite(Composite m_TabComposite) {
		this.m_TabComposite = m_TabComposite;
	}
}
