/*******************************************************************************
 * Copyright (c) 2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution.
 *******************************************************************************/
package com.freescale.sa.configurator.ui;

import java.io.File;

import org.apache.log4j.Logger;
import org.eclipse.cdt.core.model.ICProject;
import org.eclipse.cdt.debug.core.CDebugUtils;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.debug.core.ILaunchConfiguration;

import com.freescale.sa.util.LaunchConfigUtils;
import com.freescale.sa.SaConstants;
import com.freescale.sa.model.PlatformType;
import com.freescale.sa.util.CommonConstants;

/**
 * /** Keeps the context of a launch session. E.g.: project, executable, launch
 * name.
 * 
 * Provides basic functionality. Its meant to be derived by the product specific
 * LaunchContext classes.
 */
public abstract class AbstractLaunchContext {
	protected ILaunchConfiguration m_LaunchConfig;
	protected String m_LaunchConfigName;
	protected PlatformType m_PlatformType;
	protected String m_TracePortType;
	protected ICProject m_icProject;
	protected String m_ProcessorName;
	protected IPath m_saOutputFolder;
	// Path of the executable.
	protected IPath m_exePath;
	// Path of the platform configuration file.
	protected File m_PlatformConfigFile;

	private Logger LOGGER = Logger.getLogger(AbstractLaunchContext.class);

	public AbstractLaunchContext(ILaunchConfiguration launchConfig)
			throws CoreException {
		m_LaunchConfig = launchConfig;

		m_LaunchConfigName = launchConfig.getName();

		m_PlatformType = PlatformType.getPlatformType(launchConfig);

		m_TracePortType = getTracePortType(launchConfig);

		m_icProject = CDebugUtils.verifyCProject(launchConfig);

		m_exePath = CDebugUtils.verifyProgramPath(launchConfig);
		
		loadProcName(launchConfig);
		// Load platform configuration before the producers to know which
		// mechanism is used. The goal is to access the producers using the
		// managers (DpCollectManager and DpConfigManager). So in the future the
		// data producers should't be stored in the LaunchContext.
		loadPlatformConfigFile(launchConfig);
		loadSaOutputFolder(launchConfig);
	
	}

	/**
	 * Loads the platform configuration file from the launch configuration, if
	 * the associated attribute exists.
	 * 
	 * @param launchConfig
	 *            The current launch configuration.
	 */
	private void loadPlatformConfigFile(ILaunchConfiguration launchConfig) {
		String path = null;
		try {
			path = launchConfig.getAttribute(SaConstants.LAUNCH_CFG_ATTR_PLATFORM_CONFIG,
					CommonConstants.EMPTY_STRING);
		} catch (CoreException e) {
			path = CommonConstants.EMPTY_STRING;
		}
		IPath platCfgPath = new Path(path);
		if (platCfgPath.isValidPath(path)) {
			m_PlatformConfigFile = platCfgPath.toFile();
			if (!m_PlatformConfigFile.exists()) {
				m_PlatformConfigFile = null;
			}
		}
	}

	protected void loadSaOutputFolder(ILaunchConfiguration launchConfig)
			throws CoreException {
		String outputFolderStr = LaunchConfigUtils.getSaOutputFolderFromLaunch(launchConfig);
		if (outputFolderStr == null) {
			outputFolderStr = CommonConstants.EMPTY_STRING;
		}
		m_saOutputFolder = new Path(outputFolderStr);
	}

	public IProject getIProject() {
		return m_icProject.getProject();
	}

	public void setICProject(ICProject icProj) {
		m_icProject = icProj;
	}

	public ICProject getICProject() {
		return m_icProject;
	}

	public IPath getExePath() {
		return m_exePath;
	}

	public void setExePath(IPath exePath) {
		m_exePath = exePath;
	}

	public void setLaunchConfigName(String launchName) {
		m_LaunchConfigName = launchName;
	}

	public String getLaunchConfigName() {
		return m_LaunchConfigName;
	}

	public ILaunchConfiguration getLaunchConfig() {
		return m_LaunchConfig;
	}

	public void setLaunchConfig(ILaunchConfiguration launchConfig) {
		m_LaunchConfig = launchConfig;
	}

	public IPath getProjectLocation() {
		if (m_icProject != null) {
			return m_icProject.getProject().getLocation();
		}
		LOGGER.error("[getProjectLocation]: null icProj"); //$NON-NLS-1$
		return null;
	}

	/**
	 * @return the path where the raw trace configuration is stored
	 *         (*.dpconfig).
	 */
	public IPath getRawTraceConfigPath() {
		return m_saOutputFolder.append(m_LaunchConfigName + "." //$NON-NLS-1$
				+ SaConstants.raw_trace_config_extension);
	}

	/**
	 * @return Path to the analysis results folder.
	 */
	public IPath getSaOutputFolder() {
		return m_saOutputFolder;
	}

	public IPath getTimelinePath() {
		return m_saOutputFolder.append(m_LaunchConfigName + "." //$NON-NLS-1$
				+ SaConstants.storage_timeline_file_extension);
	}

	public IPath getExcludedSymbolsFilePath() {

		return m_saOutputFolder.append(m_LaunchConfigName + "." //$NON-NLS-1$
				+ SaConstants.storage_excluded_symbols_file_extension);
	}

	public IPath getMetricsDescriptionPath() {
		return m_saOutputFolder.append(m_LaunchConfigName + "." //$NON-NLS-1$
				+ SaConstants.metrics_description_extension);
	}

	public void setProcessorName(String processorName) {
		m_ProcessorName = processorName;
	}

	public String getProcessorName() {
		return m_ProcessorName;
	}

	public void setTracePortType(String portType) {
		m_TracePortType = portType;
	}

	public String getTracePortType() {
		return m_TracePortType;
	}

	public PlatformType getPlatformType() {
		return m_PlatformType;
	}

	/**
	 * @return the platform config file. Null if no such file exists.
	 */
	public File getPlatformConfigFile() {
		return m_PlatformConfigFile;
	}

	/**
	 * The method getCoreIndex returns the core index selected within a launch
	 * configuration. In the case of an SMP where more than one core is
	 * selected, this only returns the very first core in the list.
	 * 
	 * @return Core index that is selected or -1 in case an error occurred.
	 */
	public int getCoreIndex() {		
		return LaunchConfigUtils.getCoreIdxFromLaunchConfig(m_LaunchConfig);
	}

	protected abstract String getTracePortType(ILaunchConfiguration launchConfig);

	protected abstract void loadProcName(ILaunchConfiguration launchConfig);

}
