/*******************************************************************************
 * Copyright (c) 2015 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/

package com.freescale.sa.configurator.export;

import java.io.File;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.Vector;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.rse.core.IRSECoreRegistry;
import org.eclipse.rse.core.IRSESystemType;
import org.eclipse.rse.core.RSECorePlugin;
import org.eclipse.rse.core.model.IHost;
import org.eclipse.rse.core.model.ISystemViewInputProvider;
import org.eclipse.rse.core.subsystems.ISubSystem;
import org.eclipse.rse.internal.ui.view.SystemView;
import org.eclipse.rse.services.clientserver.messages.SystemMessage;
import org.eclipse.rse.services.clientserver.messages.SystemMessageException;
import org.eclipse.rse.subsystems.files.core.model.RemoteFileUtility;
import org.eclipse.rse.subsystems.files.core.subsystems.IRemoteFile;
import org.eclipse.rse.subsystems.files.core.subsystems.IRemoteFileSubSystem;
import org.eclipse.rse.subsystems.files.core.subsystems.RemoteFileSubSystem;
import org.eclipse.rse.ui.messages.ISystemMessageLine;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.custom.StackLayout;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.internal.SharedImages;

import com.freescale.sa.configurator.ui.Activator;
import com.freescale.sa.configurator.ui.tree.ITreeDataNodeContainer;
import com.freescale.sa.configurator.ui.tree.TreeDataNode;
import com.freescale.sa.configurator.ui.tree.TreeViewContentProvider;
import com.freescale.sa.ui.common.utils.SaUiLsUtils;
import com.freescale.sa.util.CommonConstants;
import com.freescale.sa.util.Utils;

@SuppressWarnings("restriction")
public class ExportConfigDialog extends TitleAreaDialog implements ITreeDataNodeContainer {

    protected final static Logger LOGGER = Logger.getLogger(ExportConfigDialog.class);
    protected Composite m_exportLocalFolderComposite;
    protected Composite m_exportRSEFolderComposite;
    protected SashForm m_sashForm;
    private File m_platformConfigFile;
    protected ExportLocalFolderControls m_exportLocalFolderControls;
    private ExportConfigDialogControlManager m_exportControlManager = new ExportConfigDialogControlManager();

    protected Button c_btnLocalButton;
    protected Button c_btnRSEButton;
    protected Text c_destFileText;

    private TreeViewContentProvider m_contentProvider;
    private Vector<TreeDataNode> m_allElements;
    private TreeViewer c_destTree;
    private SystemView c_sysView;

    private static final String LABEL_NEW_LOCATION = com.freescale.sa.configurator.export.Messages.ExportConfigDialog_0;
    private static final String LABEL_HISTORY = Messages.ExportConfigDialog_1;
    private static final String ENCODING_UTF_8 = "UTF-8"; //$NON-NLS-1$
    private static final String LABEL_LOCAL_HOST_NAME = "localhost"; //$NON-NLS-1$

    IRemoteFile m_remoteDestination;
    String m_destFileName;
    String m_oldDestFileName;

    /**
     * Create the dialog.
     * 
     * @param parentShell
     */
    public ExportConfigDialog(Shell parentShell, File platformConfigFile) {
        super(parentShell);
        setShellStyle(SWT.BORDER | SWT.TITLE | SWT.APPLICATION_MODAL);
        m_platformConfigFile = platformConfigFile;
        m_destFileName = m_oldDestFileName = platformConfigFile.getName();
    }

    @Override
    public Shell getShell() {
        Shell crtShell = super.getShell();
        if (crtShell == null) {
            crtShell = Display.getDefault().getActiveShell();
        }

        return crtShell;
    }

    // Make Export Dialog resizable
    protected void setShellStyle(int newShellStyle) {
        super.setShellStyle(newShellStyle | SWT.RESIZE | SWT.MAX);
    }

    private void showLocalBrowser(boolean show) {
        c_btnLocalButton.setSelection(show);
        c_btnRSEButton.setSelection(!show);
        m_exportLocalFolderComposite.setVisible(show);
        m_sashForm.setVisible(!show);
    }

    /**
     * Create contents of the dialog.
     * 
     * @param parent
     */
    @Override
    protected Control createDialogArea(Composite parent) {
        Shell crtShell = getShell();

        if (crtShell != null) {
            crtShell.setText(Messages.ExportConfigDialog_3);
        }
        setTitle(Messages.ExportConfigDialog_4);
        setMessage(Messages.ExportConfigDialog_5, IMessageProvider.INFORMATION);

        Composite container = (Composite) super.createDialogArea(parent);
        container.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));
        container.setLayout(new GridLayout(1, true));

        Group grpExportTo = new Group(container, SWT.NONE);
        grpExportTo.setToolTipText(Messages.ExportConfigDialog_6);
        grpExportTo.setText(Messages.ExportConfigDialog_7);
        grpExportTo.setLayout(new GridLayout(3, true));
        GridData gd_grpExportTo = new GridData(SWT.FILL, SWT.CENTER, true, false, 1, 1);
        grpExportTo.setLayoutData(gd_grpExportTo);

        c_btnLocalButton = new Button(grpExportTo, SWT.RADIO);
        c_btnLocalButton.setSelection(true);
        c_btnLocalButton.setToolTipText(Messages.ExportConfigDialog_8);
        c_btnLocalButton.setText(Messages.ExportConfigDialog_9);

        c_btnRSEButton = new Button(grpExportTo, SWT.RADIO);
        c_btnRSEButton.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));
        c_btnRSEButton.setSelection(false);
        c_btnRSEButton.setToolTipText(Messages.ExportConfigDialog_10);
        c_btnRSEButton.setText(Messages.ExportConfigDialog_11);

        new Label(grpExportTo, SWT.NONE);

        Label destFileLabel = new Label(grpExportTo, SWT.NULL);
        destFileLabel.setText(Messages.ExportConfigDialog_13);

        c_destFileText = new Text(grpExportTo, SWT.SINGLE | SWT.BORDER);
        c_destFileText.setLayoutData(new GridData(SWT.LEFT, SWT.FILL, true, true, 2, 1));

        c_destFileText.setText(m_destFileName);

        final ExportConfigDialogControlClient client = new ExportConfigDialogControlClient(true);
        m_exportControlManager.registerClient(client);

        c_destFileText.addModifyListener(new ModifyListener() {

            @Override
            public void modifyText(ModifyEvent e) {
                m_destFileName = c_destFileText.getText();
                if (c_destFileText.getEnabled()) {
                    m_oldDestFileName = m_destFileName;
                }
                if (m_destFileName.isEmpty()) {
                	m_exportLocalFolderControls.showErrorLabel(Messages.ExportConfigDialog_21);
                    client.setReady(false);
                } else if (!Utils.isValidFileName(m_destFileName)){
                	m_exportLocalFolderControls.showErrorLabel(Messages.ExportConfigDialog_21);
                	client.setReady(false);
                } else {
                	m_exportLocalFolderControls.showErrorLabel(CommonConstants.EMPTY_STRING);
                    client.setReady(true);
                }
                enableExport();

            }
        });

        final Composite commonComposite = new Composite(container, SWT.NONE);
        commonComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));
        final StackLayout stackLayout = new StackLayout();
        commonComposite.setLayout(stackLayout);

        m_exportLocalFolderComposite = new Composite(commonComposite, SWT.NONE);
        m_exportLocalFolderComposite.setLayout(new GridLayout(1, false));
        m_exportLocalFolderComposite.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, true, 1,
                                                                1));

        createOutputFolderSection(m_exportLocalFolderComposite);
        m_exportLocalFolderComposite.setVisible(true);
        stackLayout.topControl = m_exportLocalFolderComposite;

        // Create the SashForm with HORIZONTAL
        m_sashForm = new SashForm(commonComposite, SWT.HORIZONTAL);
        m_sashForm.setLayout(new GridLayout());
        m_sashForm.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));
        createOptionsSection(m_sashForm);

        m_exportRSEFolderComposite = new Composite(m_sashForm, SWT.NONE);
        m_exportRSEFolderComposite.setLayout(new GridLayout(1, false));
        m_exportRSEFolderComposite.setLayoutData(new GridData(SWT.FILL, SWT.TOP | SWT.FILL, true,
                                                              true, 1, 1));

        createRSEOutputFolderSection(m_exportRSEFolderComposite);

        m_sashForm.setWeights(new int[] { 1, 2 });

        c_btnLocalButton.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                showLocalBrowser(true);
                stackLayout.topControl = m_exportLocalFolderComposite;
                commonComposite.layout();
            }
        });

        c_btnRSEButton.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                showLocalBrowser(false);
                stackLayout.topControl = m_sashForm;
                commonComposite.layout();
            }
        });
        container.update();

        return container;
    }

    private void createOptionsSection(Composite parentComposite) {

        c_destTree = new TreeViewer(parentComposite, SWT.BORDER);

        GridLayout viewerLayout = new GridLayout();

        viewerLayout.marginLeft = 1;
        viewerLayout.marginRight = 1;
        c_destTree.getTree().setLayout(viewerLayout);
        c_destTree.getTree().setToolTipText(Messages.ExportConfigDialog_16);

        GridData viewerLData = new GridData();

        viewerLData.grabExcessVerticalSpace = true;
        viewerLData.grabExcessHorizontalSpace = true;
        viewerLData.horizontalAlignment = GridData.FILL;
        viewerLData.verticalAlignment = GridData.FILL;
        c_destTree.getTree().setLayoutData(viewerLData);

        m_contentProvider = new TreeViewContentProvider();
        c_destTree.setContentProvider(m_contentProvider);
        c_destTree.setLabelProvider(new ExportTreeViewLabelProvider());

        m_allElements = getData();
        c_destTree.setInput(getInitialInput(0));

        // Register for dispose events
        c_destTree.getTree().addDisposeListener(new DisposeListener() {
            public void widgetDisposed(DisposeEvent e) {
            }
        });

        // by default the tree is expanded
        c_destTree.expandAll();

        ISelectionChangedListener selListener = new ISelectionChangedListener() {
            public void selectionChanged(SelectionChangedEvent event) {
                if (event.getSelection() instanceof IStructuredSelection) {
                    IStructuredSelection selection = (IStructuredSelection) event.getSelection();
                    Object objSelected = selection.getFirstElement();
                    if (objSelected == null || !(objSelected instanceof TreeDataNode)) {
                        return;
                    }

                    TreeDataNode node = (TreeDataNode) objSelected;
                    if ((node.getParentNode() == null)
                        && (node.getChildren(node.getId()).size() == 0)) {
                        // new location
                        m_exportRSEFolderComposite.setEnabled(true);
                        c_destFileText.setEnabled(true);
                        m_destFileName = m_oldDestFileName;
                        c_destFileText.setText(m_destFileName);
                    } else {
                        // history location
                        m_exportRSEFolderComposite.setEnabled(false);
                        c_destFileText.setEnabled(false);

                        if (node.getParentNode() != null) {

                            m_destFileName = node.getConfigFile();
                            c_destFileText.setText(m_destFileName);

                            String fullDesc = node.getName();
                            int sepIdx = fullDesc.indexOf(TreeDataNode.LABEL_SEPARATOR);
                            String conStr = fullDesc.substring(0, sepIdx);
                            String pathStr = fullDesc.substring(sepIdx
                                                                + TreeDataNode.LABEL_SEPARATOR.length());

                            for (TreeItem item : c_sysView.getTree().getItems()) {
                                Object data = item.getData();
                                if (data instanceof IHost) {
                                    IHost host = (IHost) data;
                                    if (host.getAliasName().equals(conStr)) {
                                        c_sysView.select(host, false);
                                    }
                                }
                            }

                            m_remoteDestination = validateAndExpandPath(conStr, pathStr, false);
                            if ((m_remoteDestination == null) || (!m_remoteDestination.exists())) {
                                exportError(Messages.ExportConfigDialog_12);
                                // remove the entry from the tree and refresh
                                m_allElements.remove(node);
                                // CopySATraceComponent.removeFromLocationHistory(conStr, pathStr);
                                getShell().getDisplay().asyncExec(new Runnable() {

                                    @Override
                                    public void run() {
                                        c_destTree.setInput(getInitialInput(0));
                                        c_destTree.refresh();
                                        c_destTree.expandAll();
                                        Tree thisTree = c_destTree.getTree();
                                        thisTree.select(thisTree.getItem(0));
                                        m_exportRSEFolderComposite.setEnabled(true);
                                        c_destFileText.setEnabled(true);
                                        m_destFileName = m_oldDestFileName;
                                        c_destFileText.setText(m_destFileName);
                                    }
                                });
                            }
                        }
                    }
                }
            }
        };

        c_destTree.addSelectionChangedListener(selListener);
        c_destTree.addPostSelectionChangedListener(selListener);

        Tree thisTree = c_destTree.getTree();
        thisTree.select(thisTree.getItem(0));
        
        // We do not support RSE anymore. This can be removed at cleanup.
        c_btnRSEButton.setVisible(false);
    }

    public void select(TreeItem item) {
        if (c_destTree != null) {
            c_destTree.setSelection(new StructuredSelection(item.getData()), true);
        }
    }

    private IRemoteFile validateAndExpandPath(String connection, String pathStr, boolean expand) {

        IPath path = new Path(pathStr);
        LinkedList<String> pathElements = new LinkedList<String>();

        if (path.getDevice() != null) {
            pathElements.add(path.getDevice());
        } else {
            pathElements.add(Messages.ExportConfigDialog_2);
            pathElements.add("/"); //$NON-NLS-1$
        }
        for (String elem : path.segments()) {
            pathElements.add(elem);
        }

        boolean valid = false;
        Tree tree = c_sysView.getTree();

        LinkedList<TreeItem> hosts = new LinkedList<TreeItem>();
        for (TreeItem item : tree.getItems()) {
            Object data = item.getData();
            if (data instanceof IHost) {
                hosts.add(item);
            }
        }

        TreeItem treeNode = null;

        // expand hosts
        for (TreeItem hostItem : hosts) {
            IHost host = (IHost) hostItem.getData();
            if (host.getAliasName().equals(connection)) {
                treeNode = hostItem;

                if (expand) {
                    c_sysView.select(host, true);
                } else {
                    c_sysView.select(host, false);
                }

                LinkedList<Object> ssNodes = new LinkedList<Object>();
                if (expand) {
                    TreeItem[] ssItems = treeNode.getItems();

                    for (TreeItem ssItem : ssItems) {
                        Object data = ssItem.getData();
                        if (data instanceof RemoteFileSubSystem) {
                            ssNodes.add(ssItem);
                        }
                    }
                } else {
                    ISubSystem[] sss = host.getSubSystems();
                    for (ISubSystem ss : sss) {
                        if (ss instanceof RemoteFileSubSystem) {
                            ssNodes.add(ss);
                        }
                    }
                }

                // expand subsystems
                for (Object ssItem : ssNodes) {
                    RemoteFileSubSystem fsItem = null;

                    if (ssItem instanceof TreeItem) {
                        TreeItem ti = (TreeItem) ssItem;
                        fsItem = (RemoteFileSubSystem) ti.getData();
                    } else if (ssItem instanceof RemoteFileSubSystem) {
                        fsItem = (RemoteFileSubSystem) ssItem;
                    }

                    if (fsItem != null) {
                        final RemoteFileSubSystem fs = fsItem;
                        getShell().getDisplay().syncExec(new Runnable() {
                            @Override
                            public void run() {
                                try {
                                    fs.connect(new NullProgressMonitor(), false);
                                } catch (Exception e) {
                                    exportError(e);
                                }
                            }
                        });

                        IRemoteFile file = null;
                        try {
                            file = fs.getRemoteFileObject(pathStr, new NullProgressMonitor());
                        } catch (SystemMessageException e) {
                            exportError(e);
                        }
                        if ((file != null) && (file.exists())) {
                            if (!expand) {
                                return file;
                            }

                            // expand the tree
                            if (ssItem instanceof TreeItem) {
                                treeNode = (TreeItem) ssItem;
                            }
                            expand(treeNode);
                            TreeItem[] treeNodes = treeNode.getItems();

                            for (String pathElement : pathElements) {
                                valid = false;
                                for (TreeItem crtNode : treeNodes) {
                                    if (crtNode.getText().equals(pathElement)) {
                                        valid = true;
                                        treeNode = crtNode;
                                        expand(treeNode);
                                    }
                                }
                                if (!valid) {
                                    break;
                                }
                                treeNodes = treeNode.getItems();
                            }

                            c_sysView.select(treeNode, false);
                            return file;

                        }
                    }
                }
            }
        }

        return null;

    }

    /*
     * TODO: currently not working. Children are created deferred and there is
     * no way to get notified or get a refreshed instance of the TreeItem
     */
    private void expand(TreeItem item) {
        item.setExpanded(true);
        c_sysView.refresh();
    }

    private void exportError(Exception e) {
        LOGGER.error(e.getLocalizedMessage());
        exportError(e.getLocalizedMessage());
    }

    private void exportError(String s) {
        MessageBox msg = new MessageBox(getShell(), SWT.ICON_ERROR);
        msg.setText(Messages.ExportConfigDialog_15);
        msg.setMessage(s);
        msg.open();
    }

    private ArrayList<TreeDataNode> getInitialInput(int id) {

        ArrayList<TreeDataNode> roots = new ArrayList<TreeDataNode>();
        if (!m_allElements.isEmpty()) {
            for (int i = 0; i < m_allElements.size(); i++) {
                TreeDataNode elem = m_allElements.elementAt(i);
                if (elem.getLevel() == 0) {
                    TreeDataNode root = new TreeDataNode(elem.getHost(), elem.getUiName(),
                                                         elem.getConfigFile(), elem.getId(), null,
                                                         this, 0, elem.isExcludedSymbol());
                    roots.add(root);
                }
            }
        }

        return roots;
    }

    private Vector<TreeDataNode> getData() {

        Vector<TreeDataNode> tempElements = new Vector<TreeDataNode>();

        TreeDataNode nodeRoot = new TreeDataNode(LABEL_NEW_LOCATION, LABEL_NEW_LOCATION, null, 0,
                                                 null, this, 0, true);
        tempElements.add(nodeRoot);
        TreeDataNode histRoot = new TreeDataNode(LABEL_HISTORY, LABEL_HISTORY, null, 1, null, this,
                                                 0, true);
        tempElements.add(histRoot);
        
        //TODO: remove this and all things related to RSE when performing general cleanup.
//        int idTraceGen = 1;
//        MultiValueMap<String, String> history = CopySATraceComponent.getLocationHistory();
//        for (String key : history.keySet()) {
//            Collection<String> values = history.getCollection(key);
//            for (String value : values) {
//                TreeDataNode histEntry = new TreeDataNode(key, key, value,
//                                                          histRoot.getId() + idTraceGen++, histRoot,
//                                                          this, histRoot.getLevel() + 1, true);
//                tempElements.add(histEntry);
//            }
//        }

        return tempElements;
    }

    private void createOutputFolderSection(Composite parent) {
        m_exportLocalFolderControls = new ExportLocalFolderControls(parent);
        m_exportLocalFolderControls.createOutputFolderControls();

        final ExportConfigDialogControlClient client = new ExportConfigDialogControlClient(false);
        m_exportControlManager.registerClient(client);

        Text inputText = m_exportLocalFolderControls.getOutputResultsText();
        inputText.addModifyListener((new ModifyListener() {

            @Override
            public void modifyText(ModifyEvent e) {

                if (m_exportLocalFolderControls.modifyText(e)) {
                    client.setReady(true);
                } else {
                    client.setReady(false);
                }
                enableExport();
            }
        }));
    }

    private void createRSEOutputFolderSection(Composite parent) {

        IRSECoreRegistry cr = RSECorePlugin.getTheCoreRegistry();
        IRSESystemType linuxSystemType = cr.getSystemTypeById(IRSESystemType.SYSTEMTYPE_LINUX_ID);
        final IRSESystemType localSystemType = cr.getSystemTypeById(IRSESystemType.SYSTEMTYPE_LOCAL_ID);
        final IRSESystemType[] restrictions = new IRSESystemType[] { linuxSystemType };

        ISystemViewInputProvider inputProvider = RSECorePlugin.getTheSystemRegistry();
        c_sysView = new SystemView(getShell(), parent,
                                   SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL | SWT.RESIZE,
                                   inputProvider, new SystemMessageLine());

        GridData treeData = new GridData();
        treeData.horizontalAlignment = GridData.FILL;
        treeData.verticalAlignment = GridData.FILL;
        treeData.grabExcessHorizontalSpace = true;
        treeData.grabExcessVerticalSpace = true;
        treeData.widthHint = 200;
        treeData.heightHint = 200;

        c_sysView.getTree().setLayout(new GridLayout());
        c_sysView.getTree().setLayoutData(treeData);
        c_sysView.getTree().setToolTipText(Messages.ExportConfigDialog_17);
        if (restrictions != null) {
            c_sysView.addFilter(new ViewerFilter() {
                @Override
                public boolean select(Viewer viewer, Object parentElement, Object element) {
                    if (restrictions.length > 0) {
                        if (element instanceof IHost) {
                            IHost hostElement = (IHost) element;
                            for (IRSESystemType restriction : restrictions) {
                                if (hostElement.getSystemType().equals(restriction)) {
                                    return true;
                                }
                            }
                            if (SaUiLsUtils.isInTestMode()
                                && hostElement.getSystemType().equals(localSystemType)
                                && hostElement.getAliasName().equals(LABEL_LOCAL_HOST_NAME)) {
                                return true;
                            }
                            return false;
                        }
                    }
                    return true;
                }
            });
        }

        c_sysView.addSelectionChangedListener(new ISelectionChangedListener() {

            @Override
            public void selectionChanged(SelectionChangedEvent event) {
                if (event.getSelection() instanceof IStructuredSelection) {
                    IStructuredSelection selection = (IStructuredSelection) event.getSelection();
                    Object objSelected = selection.getFirstElement();
                    if (objSelected instanceof IRemoteFile) {
                        m_remoteDestination = (IRemoteFile) objSelected;
                    }
                }
            }

        });

    }

    void exportToLocalFolder() {
        if (m_platformConfigFile.exists() && m_exportLocalFolderControls.isPathValid()) {
            StringBuffer destination = new StringBuffer(m_exportLocalFolderControls.getOutputResultsPath());

            destination.append('/').append(m_destFileName);
            if(!Utils.copyFolder(m_platformConfigFile, new File(destination.toString()))){
            	// there has been an error exporting the file. maybe a reserved filename was used.
            	MessageDialog.openError(getShell(), Messages.ExportConfigDialog_19, Messages.ExportConfigDialog_20);
            } 
        }
    }

    void exportToRSE() {
        if ((m_remoteDestination != null) && m_platformConfigFile.exists()) {
            IRemoteFileSubSystem rfs = RemoteFileUtility.getFileSubSystem(m_remoteDestination.getHost());

            String dest = m_remoteDestination.getAbsolutePath();
            if (m_remoteDestination.isDirectory()) {
                dest += rfs.getSeparator() + m_destFileName;
            }

            try {
                rfs.upload(m_platformConfigFile.getAbsolutePath(), ENCODING_UTF_8, dest,
                           ENCODING_UTF_8, new NullProgressMonitor());
            } catch (SystemMessageException e) {
                exportError(e);
            }
        }

    }

    /**
     * Create contents of the button bar.
     * 
     * @param parent
     */
    @Override
    protected void createButtonsForButtonBar(Composite parent) {
        createButton(parent, IDialogConstants.OK_ID, Messages.ExportConfigDialog_18, true);
        getButton(IDialogConstants.OK_ID).setEnabled(false);
        createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CANCEL_LABEL, false);
    }

    private void cleanup() {
        m_exportControlManager.unregisterAllClients();
    }

    protected void okPressed() {
        if (c_btnLocalButton.getSelection()) {
            exportToLocalFolder();
        } else {
            exportToRSE();
        }
        cleanup();

        super.okPressed();
    }

    protected void cancelPressed() {
        cleanup();

        super.cancelPressed();
    }

    public static class SystemMessageLine implements ISystemMessageLine {

        public void clearErrorMessage() {
        }

        public void clearMessage() {
        }

        public String getErrorMessage() {
            return null;
        }

        public SystemMessage getSystemErrorMessage() {
            return null;
        }

        public String getMessage() {
            return null;
        }

        public void setErrorMessage(String message) {
        }

        public void setErrorMessage(SystemMessage message) {
        }

        public void setErrorMessage(Throwable exc) {
        }

        public void setMessage(String message) {
        }

        public void setMessage(SystemMessage message) {
        }

    }

    public static class ExportTreeViewLabelProvider implements ILabelProvider {

        private Image m_iconLinux = Activator.getImageDescriptor("icons/systemlinux.gif") //$NON-NLS-1$
                                             .createImage();
        private Image m_iconConnection = Activator.getImageDescriptor("icons/systemconnection.gif") //$NON-NLS-1$
                                                  .createImage();

        @SuppressWarnings("deprecation")
        private Image m_iconHistory = PlatformUI.getWorkbench().getSharedImages()
                                                .getImage(SharedImages.IMG_OPEN_MARKER);
        private Image m_iconLocal = PlatformUI.getWorkbench().getSharedImages()
                                              .getImage(SharedImages.IMG_OBJ_FOLDER);

        public String getText(Object element) {
            return ((TreeDataNode) element).getName();
        }

        public Image getImage(Object arg0) {
            if (arg0 instanceof TreeDataNode) {
                TreeDataNode dataNode = (TreeDataNode) arg0;
                if (dataNode.getParentNode() == null) {
                    if (dataNode.getName().equals(LABEL_NEW_LOCATION)) {
                        // new remote location
                        return m_iconConnection;
                    } else {
                        return m_iconHistory;
                    }
                }
                if (dataNode.getParentNode() != null) {
                    if (SaUiLsUtils.isInTestMode()) {
                        String hostname = dataNode.getName()
                                                  .substring(0,
                                                             dataNode.getName()
                                                                     .indexOf(TreeDataNode.LABEL_SEPARATOR));
                        if (hostname.equals(LABEL_LOCAL_HOST_NAME)) {
                            return m_iconLocal;
                        }
                    }

                    // entry under the history root
                    return m_iconLinux;
                }
            }

            return null;
        }

        public void addListener(ILabelProviderListener arg0) {
        }

        public void dispose() {
            if (!m_iconLinux.isDisposed()) {
                m_iconLinux.dispose();
            }
            if (!m_iconConnection.isDisposed()) {
                m_iconConnection.dispose();
            }
        }

        public boolean isLabelProperty(Object arg0, String arg1) {
            return false;
        }

        public void removeListener(ILabelProviderListener arg0) {
        }
    }

    @Override
    public Vector<TreeDataNode> getAllElements() {
        return m_allElements;
    }

    void enableExport() {
        Boolean enabled = m_exportControlManager.enableExportButton();
        getButton(IDialogConstants.OK_ID).setEnabled(enabled);
    }

}
