/*******************************************************************************
 * Copyright (c) 2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.configurator.config;

import java.io.File;
import java.io.FileFilter;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;

import com.freescale.sa.SAPlugin;
import com.freescale.sa.SaConstants;
import com.freescale.sa.model.PlatformType;
import com.freescale.sa.util.CommonConstants;
import com.freescale.sa.util.Utils;

import javafx.beans.property.SimpleBooleanProperty;
import javafx.beans.property.SimpleStringProperty;

/**
 * Helper class for accessing a Configuration's elements more easily.
 * 
 */
public class ConfigurationUtils {

	private static Logger LOGGER = Logger.getLogger(ConfigurationUtils.class);
	private static SimpleStringProperty globalConfig = null;
	private static SimpleBooleanProperty refreshTraceCommanderUI = null;
	
	/**
	 * Show the error message given as a parameter.
	 * 
	 * @param errDescription
	 *            The description of the error. Will be set as the dialog title.
	 * @param errCause
	 *            The message to display. It's recommended to send a NLS string.
	 * 
	 */
	public static void showErrorMessage(String errDescription, String errCause) {
		Shell shell = Display.getCurrent().getActiveShell();
		if (shell != null) {
			MessageDialog.openError(shell, errDescription, errCause);
		} else {
			LOGGER.error(errDescription + ":" + errCause); //$NON-NLS-1$
		}
	}
	
	public static synchronized SimpleStringProperty getGlobalConfigProperty(){
		if(globalConfig == null)
			globalConfig = new SimpleStringProperty(null);
		return globalConfig;
	}
	
	public static synchronized SimpleBooleanProperty getRefreshTraceCommanderUIProperty(){
		if(refreshTraceCommanderUI == null)
			refreshTraceCommanderUI = new SimpleBooleanProperty(false);
		return refreshTraceCommanderUI;
	}

	/**
	 * Show the warning message given as a parameter.
	 * 
	 * @param warnDescription
	 *            The description of the warning. Will be set as the dialog
	 *            title.
	 * @param warnCause
	 *            The message to display. It's recommended to send a NLS string.
	 * 
	 */
	public static void showWarningMessage(String warnDescription,
			String warnCause) {
		Shell shell = Display.getCurrent().getActiveShell();
		if (shell != null) {
			MessageDialog.openWarning(shell, warnDescription, warnCause);
		} else {
			LOGGER.error(warnDescription + ":" + warnCause); //$NON-NLS-1$
		}
	}

	/**
	 * Get the name of the executable from its file path.
	 * 
	 * @param exePath
	 *            The path to the executable file.
	 * @return The name of the executable.
	 */
	public static String getExecutableNameFromPath(IPath exePath) {
		String exeName = exePath
				.removeFirstSegments(exePath.segmentCount() - 1)
				.removeFileExtension().lastSegment();
		return exeName;
	}

	/**
	 * Use this method to workaround Eclipse bug 377593
	 * 
	 * @see https://bugs.eclipse.org/bugs/show_bug.cgi?id=377593
	 */
	public static void refreshProject(IProject project, int depth,
			IProgressMonitor progressMonitor) throws CoreException {

		if (project != null && project.isOpen()) {
			try {
				project.refreshLocal(depth, progressMonitor);
			} catch (IllegalArgumentException e) {
				LOGGER.error(
						"[refreshProject] Error refreshing workspace. Error is related to eclipse bug 377593", e); //$NON-NLS-1$
			}
		}
	}

	/**
	 * Get all the target configurations (*.traceConfig; i.e., archived producer
	 * configs) from the project's list of resources.
	 * 
	 * @param project
	 *            The current project.
	 * @return The vector of files associates with the target configs.
	 */
	public static Vector<IResource> getTargetConfigsInProject(IProject project) {

		Vector<IResource> traceCfgFiles = new Vector<IResource>();

		if (project.isOpen()) {
			try {

				IResource[] projMembers = project.members();
				for (IResource resource : projMembers) {
					if (resource.getType() == IResource.FILE
							&& resource.getParent() != null
							&& resource.getParent().getType() == IResource.PROJECT) {
						String ext = resource.getFileExtension();
						if ((ext != null)
								&& (ext.compareToIgnoreCase(CommonConstants.DP_ARCHIVE_EXT) == 0)) {
							traceCfgFiles.add(resource);
						}
					}
				}
			} catch (CoreException e) {
				e.printStackTrace();
			}
		}
		return traceCfgFiles;
	}

	/**
	 * Get the archived producer config from the project's root corresponding to
	 * a launch config.
	 * 
	 * @param project
	 *            The current project.
	 * @param launchConfigName
	 *            The name of the current launch config.
	 * @return The file associates with the archived config if it exists, or
	 *         null otherwise.
	 */
	public static IFile getTargetConfigInProject(IProject project,
			String launchConfigName) {

		Vector<IResource> traceCfgFiles = getTargetConfigsInProject(project);

		for (IResource resource : traceCfgFiles) {
			IFile configFile = (IFile) resource;
			if (configFile.getName().compareToIgnoreCase(launchConfigName + "." //$NON-NLS-1$
					+ CommonConstants.DP_ARCHIVE_EXT) == 0) {
				return configFile;
			}
		}
		return null;
	}


	/**
	 * Get default storage path to the platform configuration files
	 * ('.metadata\.plugins\com.freescale.sa\platformConfig\').
	 * 
	 * @return the default storage path for the platform configuration file
	 */
	static public IPath getDefaultPlatformConfigStoragePath() {

		IPath path = SAPlugin.getDefault().getStateLocation();
		path = path.addTrailingSeparator()
				.append(SaConstants.PLATFORM_CONFIG_FOLDER)
				.addTrailingSeparator();
		return path;

	}

	/**
	 * Get the platform configuration files from the specified location.
	 * 
	 * @param path
	 *            the path where to search for platform configuration files
	 * @return the platform configuration files found in the specified location
	 */
	static public File[] getStoredPlatformConfigs(IPath path) {
		File dir = path.toFile();

		if (!dir.exists()) {
			if (!dir.mkdir()) {
				SAPlugin.logMessage("Could not create platformConfig folder in .metadata"); //$NON-NLS-1$
				return null;
			}
		}

		File[] platConfigs = dir.listFiles(new FileFilter() {
			@Override
			public boolean accept(File pathname) {
				return pathname.toString().endsWith(
						"." + SaConstants.PLATFORM_CONFIG_EXTENSION); //$NON-NLS-1$
			}
		});

		return platConfigs;
	}

	/**	
	 * Loads an existing platform configuration. If the path to the
	 * platform config file is set in the launch config, that one is loaded. If
	 * not, one of the stored platform configs is found (the one that is loaded
	 * first from the disk).
	 * 
	 * @param launchConfig
	 *            the launch configuration
	 * @return the platform configuration file, or null if one was not found.
	 * @throws CoreException
	 */
	public static File loadExistingPlatformConfig(
			ILaunchConfiguration launchConfig) throws CoreException {

		File platCfgFile = null;
		ILaunchConfigurationWorkingCopy launchWorkingCopy = launchConfig
				.getWorkingCopy();
		IPath platCfgPath = ConfigurationUtils
				.getDefaultPlatformConfigStoragePath();
		File[] platCfgsFiles = ConfigurationUtils
				.getStoredPlatformConfigs(platCfgPath);
		//platCfgsFiles = filterPlatCfgFilesByPlatform(platCfgsFiles,
		//		launchWorkingCopy);

		if ((platCfgsFiles != null) && (platCfgsFiles.length != 0)) {
			String[] platCfgNames = Utils.getFileNames(platCfgsFiles);
			File platCfgInLaunch = null;
			platCfgInLaunch = com.freescale.sa.util.LaunchConfigUtils
					.loadPlatformConfigFile(launchWorkingCopy);
			if (platCfgInLaunch != null) {
				String platCfgName = platCfgInLaunch.getName().replace(
						"." + SaConstants.PLATFORM_CONFIG_FOLDER, //$NON-NLS-1$
						CommonConstants.EMPTY_STRING);
				int index = Utils.foundStringInArray(platCfgName, platCfgNames);
				if (index != -1) {
					platCfgFile = platCfgInLaunch;
				} else {
					platCfgFile = platCfgsFiles[0];
					launchWorkingCopy.setAttribute(
							SaConstants.LAUNCH_CFG_ATTR_PLATFORM_CONFIG,
							platCfgFile.getAbsolutePath());
				}
			} else {
				platCfgFile = platCfgsFiles[0];
				launchWorkingCopy.setAttribute(
						SaConstants.LAUNCH_CFG_ATTR_PLATFORM_CONFIG,
						platCfgFile.getAbsolutePath());
			}
			return platCfgFile;
		}
		return null;
	}

	/**
	 * Filters the platform configuration files corresponding to the current
	 * platform type.
	 * 
	 * @param allPlatCfgFiles
	 *            the list of platform configuration files to filter
	 * @param launchWorkingCopy
	 *            Working copy of a launch configuration
	 * @return the list of platform configuration files corresponding to the
	 *         current platform type
	 */
	public static File[] filterPlatCfgFilesByPlatform(File[] allPlatCfgFiles,
			ILaunchConfigurationWorkingCopy launchWorkingCopy) {
		List<File> filteredFilesList = new ArrayList<File>();

		if (launchWorkingCopy == null) {
			return filteredFilesList
					.toArray(new File[filteredFilesList.size()]);
		}
		String procName = PlatformType.getProcessorName(launchWorkingCopy);

		for (int i = 0; i < allPlatCfgFiles.length; i++) {
			if (Pattern
					.compile(Pattern.quote(procName), Pattern.CASE_INSENSITIVE)
					.matcher(allPlatCfgFiles[i].getName()).find()) {
				filteredFilesList.add(allPlatCfgFiles[i]);
			}
		}
		return filteredFilesList.toArray(new File[filteredFilesList.size()]);
	}

}
