/*******************************************************************************
 * Copyright (c) 2014, 2015 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.configurator.config;

import java.io.File;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.custom.CTabItem;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;

import com.freescale.sa.configurator.ui.AnalysisLaunchConfigurationTab;
import com.freescale.sa.configurator.ui.ITargetSettingsManager;
import com.freescale.sa.configurator.ui.Messages;

/**
 * Generic analysis config contents.
 */
public class AnalysisConfigurationContents implements
		IAnalysisConfigurationContents {

	protected final static Logger LOGGER = Logger
			.getLogger(AnalysisConfigurationContents.class);

	protected Composite m_ParentComposite = null;
	protected CTabFolder folder = null;
	protected AnalysisLaunchConfigurationTab m_ParentTab = null;
	protected CTabItem item_overview = null;
	protected CTabItem item_basic = null;
	protected CTabItem item_intermediate = null;
	protected CTabItem item_advanced = null;

	// protected Group m_traceEnableGroup = null;
	protected Group m_analysisTypeGroup = null;
	protected Group m_traceConfigGroup = null;
	protected Group m_projectConfigGroup = null;
	protected Group m_controlSettingsGroup = null;

	public boolean m_FormInitialised = false;

	protected ILaunchConfiguration m_SavedConfig = null;
	protected ILaunchConfigurationWorkingCopy m_ConfigWorkingCopy = null;

	protected ITargetSettingsManager m_TargetSettingsMan = null;

	// The extending classes might want to customize the enable analysis
	// section.
	protected Composite m_OverviewComposite = null;
	protected Composite m_BasicComposite = null;
	protected Composite m_IntermediateComposite = null;
	protected Composite m_AdvancedComposite = null;
	
	private static int BASIC_TAB_INDEX = 1;

	public AnalysisConfigurationContents(
			AnalysisLaunchConfigurationTab parentTab) {
		this.m_ParentTab = parentTab;
	}

	public AnalysisConfigurationContents() {
	}

	public void setParentTab(AnalysisLaunchConfigurationTab parentTab) {
		m_ParentTab = parentTab;
	}

	public AnalysisLaunchConfigurationTab getParentTab() {
		return m_ParentTab;
	}

	protected void createCommonSAConfigControls(Composite parent) {
		m_ParentComposite = new Composite(parent, SWT.NONE);

		m_ParentComposite.setLayout(new GridLayout());
		m_ParentComposite.setLayoutData(new GridData(SWT.LEFT, SWT.TOP
				| SWT.FILL, true, true, 1, 1));

		folder = new CTabFolder(m_ParentComposite, SWT.NONE);
		folder.setSimple(false);

		item_overview = new CTabItem(folder, SWT.NONE);
		item_overview.setText(Messages.SA_CONFIG_OVERVIEW_TAB_NAME);

		item_basic = new CTabItem(folder, SWT.NONE);
		item_basic.setText(Messages.SA_CONFIG_BASIC_TAB_NAME);

		m_OverviewComposite = new Composite(folder, SWT.NONE);
		m_OverviewComposite.setLayout(new GridLayout(2, false));
		m_OverviewComposite.setLayoutData(new GridData(SWT.LEFT, SWT.TOP
				| SWT.FILL, true, true, 1, 1));

		m_BasicComposite = new Composite(folder, SWT.NONE);
		m_BasicComposite.setLayout(new GridLayout(1, false));
		m_BasicComposite.setLayoutData(new GridData(SWT.LEFT, SWT.TOP
				| SWT.FILL, true, true, 1, 1));

		createOverviewSection();
		createAnalysisSection();
		initComponentsBeforeSettingsSection();
		createGeneralSettingsSection();

		// The Intermediate and Advanced tabs are disabled for now
		// createControlSettingsSection();
		/*
		 * TBD: The SA EDD does specify a strict guideline on the advanced trace
		 * configuration tab layout. Therefore the core will only export a
		 * single section for now to be populated by the product launch UI
		 * routines.
		 */
		// createAdvancedSettingsSection();

		// default, select Basic configuration tab
		folder.setSelection(BASIC_TAB_INDEX);
	}

	protected void initComponentsBeforeSettingsSection() {
	}

	public void createTraceCommanderControls(Composite parent) {
		m_ParentComposite = new Composite(parent, SWT.NONE);

		m_ParentComposite.setLayout(new GridLayout());
		m_ParentComposite.setLayoutData(new GridData(SWT.LEFT, SWT.TOP
				| SWT.FILL, true, true, 1, 1));

		folder = new CTabFolder(m_ParentComposite, SWT.NONE);
		folder.setSimple(false);

		item_overview = new CTabItem(folder, SWT.NONE);
		item_overview.setText(Messages.SA_CONFIG_OVERVIEW_TAB_NAME);
		// don't show it from Trace Commander Config button
		item_overview.dispose();

		item_basic = new CTabItem(folder, SWT.NONE);
		item_basic.setText("Configure Trace IP"); //$NON-NLS-1$

		m_OverviewComposite = new Composite(folder, SWT.NONE);
		m_OverviewComposite.setLayout(new GridLayout(2, false));
		m_OverviewComposite.setLayoutData(new GridData(SWT.LEFT, SWT.TOP
				| SWT.FILL, true, true, 1, 1));

		m_BasicComposite = new Composite(folder, SWT.NONE);
		m_BasicComposite.setLayout(new GridLayout(1, false));
		m_BasicComposite.setLayoutData(new GridData(SWT.LEFT, SWT.TOP
				| SWT.FILL, true, true, 1, 1));

		if (m_ParentTab != null) {
			m_ParentTab.createVerticalSpacer(folder, 1);
		}

		item_basic.setControl(m_BasicComposite);
		m_analysisTypeGroup = new Group(m_BasicComposite,
				GridData.HORIZONTAL_ALIGN_END);
		m_analysisTypeGroup.setLayout(new GridLayout(1, false));

		m_projectConfigGroup = new Group(m_BasicComposite,
				GridData.HORIZONTAL_ALIGN_END);
		m_projectConfigGroup.setText(Messages.SAConfigurationTargetOSGroupName);
		m_projectConfigGroup.setLayout(new GridLayout(1, false));
		m_projectConfigGroup.setLayoutData(new GridData(SWT.LEFT, SWT.TOP
				| SWT.FILL, false, false, 1, 1));

		m_traceConfigGroup = new Group(m_BasicComposite,
				GridData.HORIZONTAL_ALIGN_END);
		m_traceConfigGroup
				.setText(Messages.SAConfigurationTraceConfigGroupName);
		m_traceConfigGroup.setLayout(new GridLayout(1, false));
		m_traceConfigGroup.setLayoutData(new GridData(SWT.LEFT, SWT.TOP
				| SWT.FILL, false, false, 1, 1));

		folder.setSelection(BASIC_TAB_INDEX);
	}

	/**
	 * Create a composite and the controls for the Analysis config
	 * 
	 * @param parent
	 *            The parent of the composite
	 * @return the new created composite
	 */
	public Composite createControls(Composite parent) {
		createCommonSAConfigControls(parent);

		setUIComponentState();

		Dialog.applyDialogFont(m_ParentComposite);

		return m_ParentComposite;
	}

	/**
	 * This sections is part of Overview tab Create a section of the Analysis
	 * config for Overview tab
	 */
	protected void createOverviewSection() {
		// For now, nothing from Core; everything is coming from Product
	}

	/**
	 * This sections is part of Basic tab Create a section of the Analysis
	 * config with controls for general settings
	 */
	protected void createGeneralSettingsSection() {

		m_analysisTypeGroup = new Group(m_BasicComposite,
				GridData.HORIZONTAL_ALIGN_END);
		m_analysisTypeGroup.setLayout(new GridLayout(1, false));
		m_analysisTypeGroup
				.setText(Messages.SAConfigurationAnalysisTypeGroupName);

		m_projectConfigGroup = new Group(m_BasicComposite,
				GridData.HORIZONTAL_ALIGN_END);
		m_projectConfigGroup.setText(Messages.SAConfigurationTargetOSGroupName);
		m_projectConfigGroup.setLayout(new GridLayout(1, false));
		m_projectConfigGroup.setLayoutData(new GridData(SWT.LEFT, SWT.TOP
				| SWT.FILL, false, false, 1, 1));

		m_traceConfigGroup = new Group(m_BasicComposite,
				GridData.HORIZONTAL_ALIGN_END);
		m_traceConfigGroup
				.setText(Messages.SAConfigurationTraceConfigGroupName);
		m_traceConfigGroup.setLayout(new GridLayout(1, false));
		m_traceConfigGroup.setLayoutData(new GridData(SWT.LEFT, SWT.TOP
				| SWT.FILL, false, false, 1, 1));
	}

	/**
	 * The advanced configuration section of the Trace and Profile tab. The
	 * settings found in here are almost entirely highly architecture-specific,
	 * so they will be set up by the product plugins.
	 */
	protected void createAdvancedSettingsSection() {
		item_advanced.setControl(m_AdvancedComposite);
	}

	/**
	 * This sections is part of Basic tab Create a section of the Analysis
	 * config with the controls for analysis type settings
	 */
	protected void createAnalysisSection() {
		if (m_ParentTab != null) {
			m_ParentTab.createVerticalSpacer(folder, 1);
		}

		item_basic.setControl(m_BasicComposite);
	}

	/**
	 * Sets some default values to the attributes specific to Software Analysis
	 * in the working launch config object
	 * 
	 * @param config
	 *            The working launch config object
	 */
	public void setDefaults(ILaunchConfigurationWorkingCopy config) {
		config.setContainer(null);

		m_SavedConfig = config;
	}

	/**
	 * Applies any changes to the form controls to the datasource file and to
	 * the working launch config object
	 * 
	 * @param config
	 *            The working launch config copy object
	 */
	public void performApply(ILaunchConfigurationWorkingCopy config) {
		updateConfig(config);
	}

	/**
	 * Update the working launch config copy based on the status of some of the
	 * form controls (e.g. analysis enabled)
	 * 
	 * @param config
	 *            The working launch config copy object
	 */
	public void updateConfig(ILaunchConfigurationWorkingCopy config) {

		m_SavedConfig = config;

		if (m_TargetSettingsMan != null) {
			m_TargetSettingsMan.loadConfigurationFromEngine(config);

		}
	}

	/**
	 * Method that initializes the form using data from the launch config
	 * 
	 * @param config
	 *            The launch config object
	 */
	public void initializeFrom(ILaunchConfiguration config) {
		if (m_FormInitialised) {
			return;
		} else {
			m_FormInitialised = true;
		}

		try {
			m_SavedConfig = config;
			m_ConfigWorkingCopy = config.getWorkingCopy();

			setUIComponentState();

		} catch (CoreException e) {
			LOGGER.error(
					"[initializeFrom]Error initializing from " + config.getName(), e); //$NON-NLS-1$
		}
	}

	/**
	 * Refresh the SA UI config content with the values from config.
	 * 
	 * @param config
	 *            The launch configuration containing the data to update UI
	 *            controls.
	 * @note Called during org.eclipse.debug.internal.ui.launchConfigurations.
	 *       LaunchConfigurationTabGroupViewer.handleRevertPressed()
	 */
	public void refreshFromConfig(ILaunchConfiguration config) {
		try {
			m_SavedConfig = config;
			// Set the current working copy.
			m_ConfigWorkingCopy = config.getWorkingCopy();

			// Saves this working copy to engine.
			saveSettingsFromConfig();

			// Refresh UI controls with values from config.
			dataBind();

			setUIComponentState();
		} catch (CoreException e) {
			LOGGER.error("Error at refresh from config " + config.getName(), e); //$NON-NLS-1$
		}
	}

	/**
	 * Updates the engine's trace settings with values from UI. Also saves trace
	 * target configuration to disk.
	 * 
	 * @param config
	 *            The launch configuration with settings that must be saved.
	 */
	protected void saveSettingsFromUi(ILaunchConfigurationWorkingCopy config) {
		if (m_TargetSettingsMan == null) {
			return;
		}
		// Saves it to disk.
		m_TargetSettingsMan.writeConfigurationAttributes();
		m_TargetSettingsMan.saveConfigurationToEngine();

	}

	/**
	 * Sets the config values to engine and also saves them to disk.
	 * 
	 * @see AnalysisConfigurationContents.refreshFromConfig()
	 */
	protected void saveSettingsFromConfig() {
		if (m_TargetSettingsMan == null) {
			return;
		}

		// Saves it to disk.
		m_TargetSettingsMan.writeConfigurationAttributes();
		m_TargetSettingsMan.saveConfigurationToEngine();

	}

	/**
	 * DataBind UI controls to the launch configuration. No need to bind any
	 * data as this tab does not contain any specific information.
	 */
	protected void dataBind() {
		// Empty implementation.
	}

	/**
	 * Enable all UI widgets according with enable/disable of 'Enable trace and
	 * profile' button state.
	 */
	public void setUIComponentState() {
		if (!m_FormInitialised) {
			return;
		}
	}

	public boolean isValid(ILaunchConfiguration config) {
		return true;
	}

	public void updateLaunchConfigurationDialog() {
	}

	@Override
	public void setParent(Composite parent) {
	}

	@Override
	public void initializeFrom(File platformConfig) {
	}

	@Override
	public void createConfigControls(Composite comp) {
	}

	@Override
	public void createConfigControls(Composite comp, File platformConfig) {
	}

    @Override
    public boolean isValid(String processorName) {
        return true;
    }
}
