/*******************************************************************************
 * Copyright 2016-2018 NXP
 *
 * Copyright (c) 2014-2015 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.criticalcode.editors;

import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.ParseException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.TreeMap;
import java.util.Vector;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.FontMetrics;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.progress.UIJob;

import com.freescale.sa.SaConstants;
import com.freescale.sa.model.AnalysisManager;
import com.freescale.sa.model.IAnalyzer;
import com.freescale.sa.ui.common.IHierarchicalItem;
import com.freescale.sa.ui.common.NLSUtils;
import com.freescale.sa.ui.common.SAUIUtils;
import com.freescale.sa.ui.common.UIConstants;
import com.freescale.sa.ui.common.UIMessages;
import com.freescale.sa.ui.common.editor.FileChooserDialog;
import com.freescale.sa.ui.common.editor.flatprofiler.FlatProfilerColumn;
import com.freescale.sa.ui.common.editor.flatprofiler.FlatProfilerConfigureTable;
import com.freescale.sa.ui.common.editor.flatprofiler.FlatProfilerFileInput;
import com.freescale.sa.ui.common.editor.flatprofiler.FlatProfilerTab;
import com.freescale.sa.ui.common.editor.flatprofiler.FlatProfilerTableCell;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.ParagraphData;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.SectionData;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.SubsectionData;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableCellConfiguration;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableCellData;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableData;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableRecordConfiguration;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableRecordData;
import com.freescale.sa.ui.common.sourcebrowser.TraceSourceBrowser;
import com.freescale.sa.ui.common.utils.ResourceManager;
import com.freescale.sa.ui.common.utils.SaUiLsUtils;
import com.freescale.sa.ui.criticalcode.editors.CriticalCodeProfilerInput.SHOW_CODE_TYPE;
import com.freescale.sa.util.Utils;

public class CriticalCodeProfilerTab extends FlatProfilerTab {
	
	private static Logger LOGGER = Logger.getLogger(CriticalCodeProfilerTab.class);
	
	private static final int IS_NOT_CONDITIONAL_BRANCH = 0;
	private static final int BRANCH_ALWAYS_TAKEN = 1;
	private static final int BRANCH_NEVER_TAKEN = 2;
	private static final int BRANCH_TAKEN_AND_NOT_TAKEN = 3;
	private static final int DETAILS_TABLE_NUMBER_OF_EXPORTED_COLS = 4;
	private static final int INVALID_LINE_NUMBER = -1;
	private static final String FILE = "File"; //$NON-NLS-1$
	private static final String COVERED_SOURCE_COLUMN = "Covered Source %"; //$NON-NLS-1$
	private static final String BIG_EXECUTABLES_OPTIMIZATION = "BIG_EXECUTABLES_OPTIMIZATION"; //$NON-NLS-1$
	
	private static int[] colors = new int[] { SWT.COLOR_CYAN, SWT.COLOR_GREEN, SWT.COLOR_YELLOW, SWT.COLOR_GRAY,
		SWT.COLOR_DARK_YELLOW, SWT.COLOR_BLUE, SWT.COLOR_RED };
	private int m_nextGraphicsColorIdx = 0;
	
	public class Graphics {
		public boolean show;
		Color color;

		public double maximumValue = 0;

		public Graphics(String name) {
			this.show = false;
			this.color = Display.getCurrent().getSystemColor(colors[m_nextGraphicsColorIdx]);
			m_nextGraphicsColorIdx++;
			if (m_nextGraphicsColorIdx == colors.length) {
				m_nextGraphicsColorIdx = 0;
			}
		}
	}
	
	public TreeMap<String, Graphics> m_graphics;
	
	public static class ShowCodeSelectionAdapter extends SelectionAdapter {

		// The code type this selection is associated with,
		private SHOW_CODE_TYPE codeType;
		private CriticalCodeProfilerTab ccTab;
		
		public ShowCodeSelectionAdapter(CriticalCodeProfilerTab ccTab, SHOW_CODE_TYPE codeType) {
			this.ccTab = ccTab;
			this.codeType = codeType;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
		 */
		@Override
		public void widgetSelected(SelectionEvent e) {
			super.widgetSelected(e);
			ccTab.setShowCode(codeType);
			ccTab.updateDetailsTableForCurrentDetailsHistoryPos();
		}
	}

	private int dataIdx;
	private int addressIdx;
	private Boolean m_hasSourceCoverage = null;
	// Stores user preferences
	private int[] m_columnsOrderAsm;
	private boolean[] m_showColumnsAsm;
	private int[] m_columnsOrderSource;
	private boolean[] m_showColumnsSource;
	private ToolItem m_filterButton;
	private HashSet<String> m_filteredFiles;
	private ToolItem m_switchButton;
	private SHOW_CODE_TYPE m_showCode;
	
	private boolean treeHasChildren = true;
	private boolean hasExportToHtml = true;
	private boolean bigElfOptimization = false;

	private TreeMap<String, TreeMap<String, Vector<Vector<String>>>> fileDetails = null;
	private TreeMap<String, Double> specialConditionalInstr = new TreeMap<String, Double>();

	public CriticalCodeProfilerTab(FlatProfilerFileInput input, String section,
			Composite parent, IAnalyzer analyzer) {
		super(input, section, parent, analyzer);

		dataIdx = 0;
		m_showCode = SHOW_CODE_TYPE.MIXED;
		m_graphics = new TreeMap<String, Graphics>();
		
		for (String lineName : new String[] { UIConstants.CC_SOURCE_LINE,
				UIConstants.CC_ASSEMBLY_LINE }) {
			TableRecordConfiguration config = fileInput.tableRecordsConfigurations
					.get(lineName);

			if (config == null) {
				return;
			}
			
			for (TableCellConfiguration cell : config.cells) {
				if (cell.allowGraphics) {
					String name = NLSUtils.getTranslatedString(config.name) + " > " + NLSUtils.getTranslatedString(cell.name); //$NON-NLS-1$
					if (!m_graphics.containsKey(name)) {
						m_graphics.put(name, new Graphics(name));
					}
				}
			}
		}

		String lineName = UIConstants.CC_FUNCTION_LINE;
		TableRecordConfiguration config = fileInput.tableRecordsConfigurations.get(lineName);

		if (config == null) {
		    return;
		}

		for (dataIdx = 0; dataIdx < config.cells.size(); dataIdx++) {
		    String columnName = config.cells.get(dataIdx).name;
		    if (columnName.equals(UIMessages.CriticalCode_StartAddress_Column)) {
		        addressIdx = dataIdx;
		    }
		}

		try {
			this.analyzer = AnalysisManager.getDefault().getAnalyzer(m_archId);
		} catch (CoreException e) {
			LOGGER.debug("[FlatProfilerInput] unable to get analyzer. ", e); //$NON-NLS-1$
		}

		if (input.getTraceType() != null
				&& input.getTraceType().toLowerCase().contains("kernel")
				|| input.getOptions().contains(BIG_EXECUTABLES_OPTIMIZATION)) { //$NON-NLS-1$
			treeSummaryView = false;
			bigElfOptimization = true;
		}
	}
	
	@Override
	protected void drawSummary(Composite parent) {
		super.drawSummary(parent);
	}
	
	protected void addDoubleClick(MouseEvent e, Tree m_details) {
	    if (m_details == null) {
			return;
		}

	    TableData data = null;
	    String instructionAddress = null;
	    String fileName = null;
	    int lineNumber = INVALID_LINE_NUMBER;
		try {
		    Point selectionPoint = new Point(e.x, e.y);

		    TreeItem item = m_summary.getItem(selectionPoint);
		    if (item == null || !(item.getData() instanceof TableRecordData)) {
		        return;
		    }

		    ItemPosition selectedItem = ItemPosition.getItemPosition(item);

		    Vector<TableCellData> cells = ((TableRecordData) item.getData()).cells;
		    if (cells == null) {
                return;
            }

		    TableRecordData summaryRecord = (TableRecordData) item.getData();
		    if (summaryRecord == null) {
		        return;
		    }
            TableRecordConfiguration summaryColumns = fileInput.tableRecordsConfigurations.get(summaryRecord.recordConfigurationName);
            if (summaryColumns == null) {
                return;
		    }
            for (dataIdx = 0; dataIdx < summaryRecord.cells.size(); dataIdx++) {
                if (dataIdx == addressIdx){
                    instructionAddress = cells.get(dataIdx).text;
                    break;
                }
            }

            Vector<Long> summaryLines = new Vector<Long>();
		    for (TableCellData cell : cells) {
		        if (cell.linksToMultipleTables.size() > 1) {
		            summaryLines.add(cell.linksToMultipleTables.firstElement());
		        } else if (cell.linkToAnotherTable >= 0) {
		            summaryLines.add(cell.linkToAnotherTable);
		        }
		    }

		    if (selectedItem == null) {
		        return;
		    }

		    data = fileInput.getDetailsTableData(selectedItem.getSection(),
                                                 selectedItem.getSubsection(),
                                                 selectedItem.getParagraph(),
                                                 summaryLines);

		    if (data.getRecords() == null || data.getRecords().isEmpty()) {
		        return;
		    }

		    Vector<TableCellData> detailsCells = data.getRecords().firstElement().cells;
		    if (detailsCells == null) {
		        return;
		    }

		    TableRecordData detailsRecord = data.getRecords().firstElement();
		    if (detailsRecord == null) {
		        return;
		    }
		    TableRecordConfiguration detailsColumns = fileInput.tableRecordsConfigurations.get(detailsRecord.recordConfigurationName);
		    if (detailsColumns == null) {
		        return;
		    }
            for (dataIdx = 0; dataIdx < detailsRecord.cells.size(); dataIdx++) {
                String columnName = detailsColumns.cells.get(dataIdx).name;
                if (columnName.equals(UIMessages.CriticalCode_Line_Column)) {
                    lineNumber = Integer.parseInt(detailsCells.get(dataIdx).text);
                }
                if (columnName.equals(UIMessages.CriticalCode_Instruction_Column)) {
                    fileName = detailsCells.get(dataIdx).text;
                }
            }

			if (fileName != null && !fileName.isEmpty() && lineNumber >= 0) {
			    if (instructionAddress.equals("N/A")) { //$NON-NLS-1$
			        lineNumber = 2;
			        TraceSourceBrowser browser = SAUIUtils.getSourceBrowser(fileInput.getFullFileName(), fileInput.getBaseName());
			        if (browser != null) {
			            browser.showSource(fileName, lineNumber);
			        }
			    } else {
			        TraceSourceBrowser browser = SAUIUtils.getSourceBrowser(fileInput.getFullFileName(), fileInput.getBaseName());
			        if (browser != null) {
			            browser.showSource(fileName, lineNumber);
			        }
			    }
			}
		} catch (Exception exception) {
			LOGGER.error(exception);
		}
	}

	protected void addRightClickMenu(MouseEvent e, final Tree tree) {

		// Only for right clock button pressed.
		if (e.button == SaConstants.RIGHT_CLICK_BUTTON) {
			Menu popupMenu = new Menu(tree);

			StringBuffer line = new StringBuffer();
			Vector<String> cellsInLine = new Vector<String>();
			String cell = null;

			// Get the number of selected column.
			Point selectionPoint = new Point(e.x, e.y);
			TreeItem treeItem = tree.getItem(selectionPoint);
			int columnSelected = 0;
			for (int i = 0; i < tree.getColumnCount(); i++) {
				if (treeItem != null) {
					Rectangle rect = treeItem.getBounds(i);
					if (rect.contains(selectionPoint)) {
						columnSelected = i;
						break;
					}
				}
			}

			// Get the selected cell and selected line.
			if (tree.getSelectionCount() > 0) {
				TreeItem item = tree.getSelection()[0];
				Object data = item.getData();
				if (data instanceof TableRecordData) {
					TableRecordData test = (TableRecordData) tree.getSelection()[0].getData();
					Vector<TableCellData> cells = test.cells;
					for (int i = 0; i < cells.size(); i++) {
						line.append(cells.elementAt(i).value + " "); //$NON-NLS-1$
						cellsInLine.add(cells.elementAt(i).value + " "); //$NON-NLS-1$
						if (columnSelected == i) {
							cell = cells.elementAt(i).value;
						}
					}
				}
				
				if (data instanceof FlatProfilerTableCell[]) {
					FlatProfilerTableCell[] cells = (FlatProfilerTableCell[])data;
					for (int i = 0; i < cells.length; i++) {
						line.append(cells[i].text + " "); //$NON-NLS-1$
						cellsInLine.add(cells[i].text + " "); //$NON-NLS-1$
						if (columnSelected == i) {
							cell = cells[i].text;
						}
					}
				}
			}

			final Vector<String> cellsInFullLine = cellsInLine;
			final String singleCell = cell;
			final String fullLine = line.toString();

			// Copy Cell menu.
			MenuItem copyCellItem = new MenuItem(popupMenu, SWT.NONE);
			copyCellItem.setText(UIMessages.CriticalCode_Copy_Cell_Menu);
			copyCellItem.addSelectionListener(new SelectionListener() {
				@Override
				public void widgetSelected(SelectionEvent sel) {
					if (singleCell == null) {
						return;
					}
					StringSelection clipString = new StringSelection(singleCell);
					Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
					clipboard.setContents(clipString, clipString);
				}

				@Override
				public void widgetDefaultSelected(SelectionEvent e) {
					// Nothing to do for this method.
				}
			});

			// Copy Line menu.
			MenuItem copyLineItem = new MenuItem(popupMenu, SWT.NONE);
			copyLineItem.setText(UIMessages.CriticalCode_Copy_Line_Menu);
			copyLineItem.addSelectionListener(new SelectionListener() {
				@Override
				public void widgetSelected(SelectionEvent sel) {
					if (fullLine == null) {
						return;
					}
					StringSelection clipString;
					StringBuffer createLine = new StringBuffer();
					FlatProfilerConfigureTable configuredTable = null;
					m_configuredSummaryTable = new FlatProfilerConfigureTable(m_summary, getSummaryConfiguration(),
							m_timeUnitGroup.getTimeUnitSelection(), m_timeUnitGroup.getFreqSelection(), m_coverageType
									.getExcludedColumns());
					// Check if summary table was reconfigured
					configuredTable = tableWasReconfigured(tree, m_configuredSummaryTable);
					if (m_details != null) {
						m_configuredDetailsTable = new FlatProfilerConfigureTable(m_details, getDetailsConfiguration(),
								m_timeUnitGroup.getTimeUnitSelection(), m_timeUnitGroup.getFreqSelection());
					}
					// Check if details table was reconfigured
					if (configuredTable == null)
						configuredTable = tableWasReconfigured(tree, m_configuredDetailsTable);
					if (configuredTable != null) {
						// If table was reconfigured, copy only the selected
						// columns.
						for (int i = 0; i < configuredTable.getColumnOrder().length; i++) {
							int columnIndex = configuredTable.getColumnOrder()[i];
							if (configuredTable.isColumnShown(columnIndex) && (cellsInFullLine.size() > columnIndex)) {
								String text = cellsInFullLine.elementAt(columnIndex);
								createLine.append(text);
							}
						}
						clipString = new StringSelection(createLine.toString());
					} else {
						// If table wasn't reconfigured, copy full line.
						clipString = new StringSelection(fullLine);
					}
					Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
					clipboard.setContents(clipString, clipString);
				}

				@Override
				public void widgetDefaultSelected(SelectionEvent e) {
					// Nothing to do for this method.
				}
			});
			
			if (!bigElfOptimization) {
				MenuItem switchItem = new MenuItem(popupMenu, SWT.NONE);
				if (treeSummaryView) {
					switchItem.setText(UIMessages.CriticalCode_FlatView);
				} else {
					switchItem.setText(UIMessages.CriticalCode_TreeView);
					if (!treeHasChildren) {
						switchItem.setEnabled(false);
					}
				}
				switchItem.addSelectionListener(new SelectionListener() {
					@Override
					public void widgetSelected(SelectionEvent sel) {
						treeSummaryView = !treeSummaryView;
						// Expand/Collapse and Filter buttons are only available on tree view.
						m_expandButton.setEnabled(treeSummaryView);
						updateSummaryTableData();
					}
	
					@Override
					public void widgetDefaultSelected(SelectionEvent e) {
						// Nothing to do for this method.
					}
				});
			}

			tree.setMenu(popupMenu);
			tree.getShell().open();
		}
	}
	
	/**
	 * Save all file that are currently filtered
	 */
	private void saveFilteredFiles() {
		m_filteredFiles = new HashSet<String>();
		SectionData section = (SectionData) m_summaryViewer.getTree().getData();
		
		for (Entry<String, SubsectionData> child : section.entrySet()) {
			for (Entry<String, ParagraphData> subEntry : child.getValue().entrySet()) {
				for (Entry<Long, TableData> pEntry : subEntry.getValue().entrySet()) {
					for (TableRecordData record : pEntry.getValue().records) {
						if (record.recordConfigurationName.equals(FILE)
								&& !record.isVisible()) {
							m_filteredFiles.add(record.fileName);
						}
					}
				}
			}
		}
	}

	public void drawFilterButton(ToolBar _toolBar) {

		m_filterButton = new ToolItem(_toolBar, SWT.PUSH);
		final URL filterUrl = FileLocator.find(Platform.getBundle(com.freescale.sa.ui.criticalcode.utils.UIConstants.PLUGIN_ID), //$NON-NLS-1$
				new Path("icons/filter.png"), null); //$NON-NLS-1$

		m_filterButton.setImage(ImageDescriptor.createFromURL(filterUrl).createImage());
		m_filterButton.setToolTipText(UIMessages.FlatProfilerInput_17);
		m_filterButton.setEnabled(true);
		m_filterButton.addSelectionListener(
				new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {

				IWorkbench workBench = PlatformUI.getWorkbench();
				IWorkbenchWindow wbWindow = workBench.getActiveWorkbenchWindow();
				
				FileChooserDialog chooserDialog = new FileChooserDialog(wbWindow.getShell(), m_summaryViewer, true);
				int retCode = chooserDialog.open();

				reloadTree(m_summaryViewer);

				if (retCode == IDialogConstants.OK_ID) {
					// Clear all history.
					clearHistory();
					saveFilteredFiles();
				}
			}
		});
	}

	private void updateSummaryTableData() {

		hasExportToHtml = true;
		if (treeSummaryView) {
			m_summaryData = getTreeSummary();
		} else {
			m_summaryData = getFlatSummary();
		}

		m_summaryViewer.setInput(m_summaryData);
		reapplyFilteredFiles();
		m_summaryViewer.expandAll();

		reloadTree(m_summaryViewer);
	}
	
	/**
	 * Applies the filter that was previously saved
	 */
	private void reapplyFilteredFiles() {
		if (m_filteredFiles == null) {
			return;
		}

		Object data = m_summaryViewer.getTree().getData();

		if (data instanceof SectionData) {
			SectionData section = (SectionData) data;
			for (Entry<String, SubsectionData> child : section.entrySet()) {
				for (Entry<String, ParagraphData> subEntry : child.getValue().entrySet()) {
					for (Entry<Long, TableData> pEntry : subEntry.getValue().entrySet()) {
						for (TableRecordData record : pEntry.getValue().records) {
							IHierarchicalItem parent = record.getParent();
							TableRecordData rec = null;
							
							if (record.recordConfigurationName.equals(FILE)) {
								/* "File" node */
								rec = record;
							} else {
								/* "Function" node */
								if (record.fileName == null || record.fileName.isEmpty()) {
									/* Tree view */
									if (parent instanceof TableData) {
										TableData tData = (TableData) parent;
										if (!tData.getRecords().isEmpty()) {
											rec = tData.getRecord(0);
										}
									} else if (parent instanceof TableRecordData) {
										rec = (TableRecordData)  parent;
									}
								} else {
									/* Flat view */
									rec = record;
								}
							}
							
							if (rec != null) {
								if (m_filteredFiles.contains(rec.fileName)) {
									record.setVisible(false);
								} else {
									record.setVisible(true);									
								}
							}
						}
					}
				}
			}
		}

		m_summaryViewer.refresh();
	}
	
	public void drawExportButton(ToolBar _toolBar) {
		final ToolBar toolBar = _toolBar;
		final ToolItem export = new ToolItem(toolBar, SWT.DROP_DOWN);
		URL exportUrl = FileLocator.find(Platform.getBundle(com.freescale.sa.ui.criticalcode.utils.UIConstants.PLUGIN_ID), //$NON-NLS-1$
				new Path("icons/export.gif"), null); //$NON-NLS-1$
		
		export.setImage(ImageDescriptor.createFromURL(exportUrl).createImage());
		export.setToolTipText(UIMessages.FlatProfilerInput_export);
		export.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				Shell shell = toolBar.getShell();
				final Menu menu = new Menu(shell, SWT.POP_UP);

				final MenuItem exportToCsv = new MenuItem(menu, SWT.CASCADE);
				final Menu exportToCsvMenu = new Menu(shell, SWT.DROP_DOWN);
				exportToCsv.setText(UIMessages.FlatProfilerInput_export_csv);
				exportToCsv.setEnabled(true);
				exportToCsv.setMenu(exportToCsvMenu);

				MenuItem exportSummary = new MenuItem(exportToCsvMenu, SWT.PUSH);
				exportSummary.setText(UIMessages.FlatProfilerInput_export_summary);
				URL exportSummaryUrl = FileLocator.find(Platform.getBundle(com.freescale.sa.ui.criticalcode.utils.UIConstants.PLUGIN_ID), //$NON-NLS-1$
						new Path("icons/bt-up-icon.png"), null); //$NON-NLS-1$

				exportSummary.setImage(ImageDescriptor.createFromURL(exportSummaryUrl).createImage());
				exportSummary.setEnabled(m_summary.getItemCount() > 0);
				exportSummary.addSelectionListener(new SelectionAdapter() {
					public void widgetSelected(SelectionEvent e) {
						exportToCSV(m_summary);
					}
				});
				m_exportDetails = new MenuItem(exportToCsvMenu, SWT.PUSH);
				m_exportDetails.setText(UIMessages.FlatProfilerInput_export_details);
				URL exportDetailsUrl = FileLocator.find(Platform.getBundle(UIConstants.PLUGIN_ID), new Path(
						"icons/bt-down-icon.png"), //$NON-NLS-1$ //$NON-NLS-2$   
						null);
				m_exportDetails.setImage(ImageDescriptor.createFromURL(exportDetailsUrl).createImage());
				m_exportDetails.setEnabled(m_details.getItemCount() > 0);
				m_exportDetails.addSelectionListener(new SelectionAdapter() {
					public void widgetSelected(SelectionEvent e) {
						exportToCSV(m_details);
					}
				});
				
				final MenuItem exportToHtml = new MenuItem(menu, SWT.CASCADE);
				exportToHtml.setText(UIMessages.FlatProfilerInput_33);
				exportToHtml.setEnabled(hasExportToHtml);
				final Menu exportToHtmlMenu = new Menu(shell, SWT.DROP_DOWN);
				exportToHtml.setMenu(exportToHtmlMenu);

				if (!bigElfOptimization) {
					final MenuItem exportAsSource = new MenuItem(exportToHtmlMenu, SWT.CASCADE);
					exportAsSource.setText(UIMessages.FlatProfilerInput_34);
					exportAsSource.setEnabled(hasExportToHtml);
					exportAsSource.addSelectionListener(new SelectionAdapter() {
						public void widgetSelected(SelectionEvent e) {
							// export to html as source														
							exportToHtml(SHOW_CODE_TYPE.SOURCE);
							
						}
					});
				}
				final MenuItem exportAsAsm = new MenuItem(exportToHtmlMenu, SWT.CASCADE);
				exportAsAsm.setText(UIMessages.FlatProfilerInput_35);
				exportAsAsm.setEnabled(hasExportToHtml);
				exportAsAsm.addSelectionListener(new SelectionAdapter() {
					public void widgetSelected(SelectionEvent e) {
						// export to html as assembly							
						exportToHtml(SHOW_CODE_TYPE.ASSEMBLY);						
					}
				});
				if (!bigElfOptimization) {
					final MenuItem exportAsMixed = new MenuItem(exportToHtmlMenu, SWT.CASCADE);
					exportAsMixed.setText(UIMessages.FlatProfilerInput_36);
					exportAsMixed.setEnabled(hasExportToHtml);
					exportAsMixed.addSelectionListener(new SelectionAdapter() {
						public void widgetSelected(SelectionEvent e) {
							// export to html as mixed
							exportToHtml(SHOW_CODE_TYPE.MIXED);						
						}
					});
				}

				menu.setVisible(true);
			}
		});
	}
	
	/**
	 * It exports critical code statistics into a html page.
	 */
	public void exportToHtml(SHOW_CODE_TYPE showCode){
		int response = SWT.NO;
		int filterIndex = 0;
		String filePath = null;

		while (response == SWT.NO) {
			response = SWT.YES;

			filePath = SaUiLsUtils.getExportFile(UIMessages.FlatProfilerInput_33, getShell(), new String[] { "*.html", "*.*" }, filterIndex); //$NON-NLS-1$ //$NON-NLS-2$
			if (filePath == null) {
				return;
			}
			File file = new File(filePath);

			if (file.exists()) {
				MessageBox messageBox = new MessageBox(getShell(), 
						SWT.ICON_QUESTION | SWT.YES | SWT.NO | SWT.CANCEL);

				messageBox.setMessage(UIMessages.FlatProfilerInput_54);
				messageBox.setText(UIMessages.FlatProfilerInput_55);
				response = messageBox.open();
				if (response == SWT.CANCEL) {
					return;
				}
			}
		}
		
		exportToHtml(filePath, showCode);
	}
	
	/**
	 * It exports critical code statistics into a html page.
	 * @param filePath When UI is not used, then the html page is saved to filePath path.
	 */
	public void exportToHtml(final String filePath, final SHOW_CODE_TYPE showCode) {

		try {

			final Job job = new Job(UIMessages.FlatProfilerInput_33) {

				@Override
				protected IStatus run(IProgressMonitor monitor) {

					try {

						BufferedWriter out = null;

						try {
							out = new BufferedWriter(new FileWriter(filePath));
							if (!m_summaryData.hasChildren()) {
								return Status.CANCEL_STATUS;
							}

							StringBuilder line = new StringBuilder();
							m_doubleFormatter = new DecimalFormat("##.##"); //$NON-NLS-1$

							String currentFileName = "", currentFunctionName = ""; //$NON-NLS-1$//$NON-NLS-2$							

							// Structure containing the association between a function name and corresponding details table.
							TreeMap<String, Vector<Vector<String>>> functionDetails = new TreeMap<String, Vector<Vector<String>>>();

							// Structure containing a table with details(source, assembly or mixed) as a matrix.
							Vector<Vector<String>> detailsTable = new Vector<Vector<String>>();

							// Structure containing the association between a function name and a map with values
							// from : "Covered %", "Not Covered", "Total ASM instructions" and "ASM Decision Coverage %".
							TreeMap<String, HashMap<String, Double>> childData = new TreeMap<String, HashMap<String, Double>>();
			
							// Write in html page the title(e.g. "Software Analysis Code Coverage"), 
							// the list of files contained by program and the description of
							// file analysis content.
							writeHtmlPageHeaders(line);
										
							int monitorTaskSize = 0;//(treeSummaryView == true) ? fileNames.size() : functionNames.size();
							
							for (String context : m_summaryData.getKeyset()) {
								monitorTaskSize += (treeSummaryView == true) ? getFileNames(context).size() : getFunctionNames(context).size();
							}
							monitor.beginTask(UIMessages.FlatProfilerInput_33, monitorTaskSize);
							
							for (String context : m_summaryData.getKeyset()) {
								Vector<String> fileNames = getFileNames(context);
								Vector<String> functionNames = getFunctionNames(context);							
				
								// Structure containing the association between a function name and a vector with values
								// for ASM decision coverage(e.g. "taken", "not taken", "both taken").
								TreeMap<String, Vector<Double>> asmDecisionMetrics;		
				
								for (int fileIndex = 0; fileIndex < fileNames.size(); fileIndex++) {

									if (monitor.isCanceled()) {
										break;
									}

									currentFileName = fileNames.get(fileIndex);

									asmDecisionMetrics = new TreeMap<String, Vector<Double>>();

									if (treeSummaryView) {
										// Write in html page the table containing coverage(percentage of executed ASM instructions from 
										// total ASM instructions) for functions from current analyzed file.											
										writeFuncCovTable(context, line, currentFileName,fileIndex, functionNames, childData);									
									} else {
										// In case flat view is enabled, there is only one function list containing
										// all the functions.
										fileIndex = fileNames.size();					
										setFunctionInformation(context, childData);
									}								
					
									// Analysis for functions from current file.
									if (functionNames.size() > 0) {
										
										for (int functionIndex = 0; functionIndex < functionNames.size(); functionIndex++) {
											currentFunctionName = functionNames.get(functionIndex);				
				
											// Load details for current function from current analyzed file.
											loadFunctionDetails(context, currentFunctionName, currentFileName, showCode);
											functionDetails = new TreeMap<String, Vector<Vector<String>>>();
											detailsTable = new Vector<Vector<String>>();
											functionDetails = fileDetails.get(currentFileName);
											
											if (functionDetails == null) {
												LOGGER.error("[exportToHtml] functionDetails is null!"); //$NON-NLS-1$
												return null;
											}
											
											detailsTable = functionDetails.get(currentFunctionName);								
				
											if (detailsTable == null) {
												LOGGER.error("[exportToHtml] detailsTable is null!"); //$NON-NLS-1$
												return null;
											}
											
											// Write in html page the details table(assembly, source or mixed) for each function.	
											writeDetailsTable(context, line, currentFunctionName, currentFileName, functionNames, functionIndex, detailsTable, showCode);						
				
											// Write in html page code coverage metrics table for current function.																				
											writeCodeCoverageTable(context, line, currentFunctionName, childData);										
				
											// If export as source is needed then ASM metrics have to be loaded
											// into fileDetails structure.										
											if (showCode == SHOW_CODE_TYPE.SOURCE) {
												loadFunctionDetails(context, currentFunctionName, currentFileName, SHOW_CODE_TYPE.ASSEMBLY);											
												functionDetails = fileDetails.get(currentFileName);
												
												if (functionDetails == null) {
													LOGGER.error("[exportToHtml] functionDetails is null!"); //$NON-NLS-1$
													return null;
												}
												
												detailsTable = functionDetails.get(currentFunctionName);
												
												if (detailsTable == null) {
													LOGGER.error("[exportToHtml] detailsTable is null!"); //$NON-NLS-1$
													return null;
												}
											}
											
											// Write in html page ASM decision metrics table for current function.
											writeAsmTable(context, line, "function asm", asmDecisionMetrics, currentFileName, //$NON-NLS-1$
													currentFunctionName, functionNames, detailsTable, childData);
											// Progress for flat view export monitor
											if (!treeSummaryView) {											
												monitor.worked(1);
											}
										}
									}

									if (treeSummaryView) {
										// Write in html page summary table for current file.
										writeSummaryTable(context, line, currentFileName, functionNames, childData);

										// Write in html page ASM decision coverage table for current file.
										writeAsmTable(context, line, "file asm", asmDecisionMetrics, currentFileName, //$NON-NLS-1$
												currentFunctionName, functionNames, detailsTable, childData);

										// Write in html page the end of analysis footer(e.g. "End of analysis for 'name of file'").
										writeFileAnalysisEnd(context, line, currentFileName);
										functionNames.removeAllElements();									
										// Progress for tree view export monitor
										monitor.worked(1);									
									}																							
								}	
							}
									
							childData.clear();
							line.append("</html>"); //$NON-NLS-1$
							String temp = line.toString();
							temp = temp.replaceAll("><", ">\n<"); //$NON-NLS-1$ //$NON-NLS-2$
							temp = temp.replaceAll("</", "\n</"); //$NON-NLS-1$//$NON-NLS-2$
							temp = temp.replaceAll("\n\n", "\n"); //$NON-NLS-1$ //$NON-NLS-2$
							out.write(temp);	
						}catch(IOException e){ 
							if(e instanceof FileNotFoundException)
							{
								new UIJob("") { //$NON-NLS-1$
									
									@Override
									public IStatus runInUIThread(IProgressMonitor monitor) {
										MessageBox messageBox = new MessageBox(getShell(), 
												SWT.ICON_ERROR | SWT.OK);

										messageBox.setMessage(UIMessages.FlatProfilerInput_57);
										messageBox.setText(UIMessages.FlatProfilerInput_58);
										messageBox.open();
										return null;
									}
								}.schedule();
								return Status.CANCEL_STATUS;
							}
						} finally {
							monitor.done();
							if(out!=null){
								out.close();
							}
						}
					} catch (IOException e) {

						LOGGER.error(e);
					} 
					return Status.OK_STATUS;
				}
			};
			job.setUser(true);
			job.addJobChangeListener(new JobChangeAdapter() {

				@Override
				public void done(IJobChangeEvent event) {
					job.removeJobChangeListener(this);
					final IStatus result = event.getResult();
					if (result.getSeverity() == IStatus.ERROR) {
						Display.getDefault().asyncExec(new Runnable() {

							@Override
							public void run() {
								ErrorDialog.openError(getShell(), null, null,
										result);
							}
						});
					}
				}

			});
			job.schedule();
		} catch (Exception e) {
			LOGGER.error(e);			
			return;
		}
	  		                 
	}
	
	public static String encodeHTML(String htmlText) {
		StringBuffer out = new StringBuffer();
		for (int i = 0; i < htmlText.length(); i++) {
			char c = htmlText.charAt(i);
			if (!Character.isLetterOrDigit(c)) {
				out.append("&#" + (int) c + ";"); //$NON-NLS-1$//$NON-NLS-2$                                                    
			} else {
				out.append(c);
			}
		}
		return out.toString();
	}
	
	public Vector<String> getFileNames(String subsection) {
		Vector<String> names = new Vector<String>();
		String name = null;
		
		for (TableRecordData record : m_summaryData.getSummaryRecords(subsection)) {
			if ((record.recordConfigurationName.equals(UIMessages.CriticalCode_File_Column)) 
					&& (record.isVisible()) ){
				name = record.fileName.substring(record.fileName.lastIndexOf("\\") + 1); //$NON-NLS-1$ 
				names.add(name);														
			}			
		}
		
		return names;
	}
	
	public Vector<String> getFunctionNames(String subsection) {
		Vector<String> names = new Vector<String>();
		String name = null;
		
		for (TableRecordData record : m_summaryData.getSummaryRecords(subsection)) {
			if ((record.recordConfigurationName.equals(UIMessages.CriticalCode_Function_Column)) 
					&& (record.isVisible()) && ! treeSummaryView){
				name = record.cells.get(0).value;
				names.add(name);
			}
		}
		
		return names;
	}
	
	private void writeFuncCovTable(String context, StringBuilder line, String currentFileName,
			int fileIndex, Vector<String> functionNames, TreeMap<String, HashMap<String, Double>> childData) {

		String childName, tempStringValue, tempFileName = null;
		Double tempDoubleVal = 0.00;
		HashMap<String, Double> tempMetrics = null;

		// Current file name.
		line.append("<table style='width:100%; height:8%;' align='center' border='1'>" + //$NON-NLS-1$
				"<tr>" + //$NON-NLS-1$
				"<th bgcolor='#8DB87C'>" + //$NON-NLS-1$
				"<a style='font-size: 150%; text-align: center;' id='" + context + " " + encodeHTML(UIMessages.CriticalCode_AnalysisFor + " ") //$NON-NLS-1$ //$NON-NLS-2$
				+ currentFileName + "'>" + UIMessages.CriticalCode_AnalysisFor + " " + currentFileName + "</a>" + //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				"</th>" + //$NON-NLS-1$
				"</tr>" + //$NON-NLS-1$
				"</table>"); //$NON-NLS-1$

		// Hyperlinks to list of files, to summary table and to ASM decision coverage table for current file.
		line.append("<p style='text-align: center;'>" + //$NON-NLS-1$
				"<a href='#Files:' style='text-decoration: none'>[ " + UIMessages.CriticalCode_Files + " | </a>" + //$NON-NLS-1$ //$NON-NLS-2$
				"<a style='text-decoration: none' href='#" + context + " " + encodeHTML(UIMessages.CriticalCode_SummaryTable_For + " ") + currentFileName + "'>" //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				+ UIMessages.CriticalCode_Summary_Table + " | " + "</a>" + //$NON-NLS-1$ //$NON-NLS-2$
				"<a style='text-decoration: none' href='#" + context + " " + encodeHTML(UIMessages.CriticalCode_AsmDecisionCoverage_For + " ") + currentFileName + "'>" //$NON-NLS-1$//$NON-NLS-2$ //$NON-NLS-3$
				+ UIMessages.CriticalCode_AsmDecisionCoverage_Table + " ]" + "</a>" + //$NON-NLS-1$//$NON-NLS-2$
				"</p>" + //$NON-NLS-1$
				"<br>"); //$NON-NLS-1$

		// Table containing function coverage for current analyzed file.
		line.append("<p style='text-align: center;'><b>" + UIMessages.CriticalCode_Function_Coverage_For + " " + currentFileName + "</b></p>"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		line.append("<table style='width:60%' border='1' align='center'>" + //$NON-NLS-1$
				"<tr>"); //$NON-NLS-1$

		// Function coverage table header.
		line.append("<th bgcolor='#F5D769'>" + UIMessages.CriticalCode_Function_Column + "</th>"); //$NON-NLS-1$ //$NON-NLS-2$
		if (m_coverageType.equals(SourceCoverageType.ASM_COVERAGE)) {
			line.append("<th bgcolor='#F5D769'>" + UIMessages.CriticalCode_Summary_Covered_ASM_Column + "</th>"); //$NON-NLS-1$ //$NON-NLS-2$
		} else {
			line.append("<th bgcolor='#F5D769'>" + UIMessages.CriticalCode_Summary_Covered_Source_Column + "</th>"); //$NON-NLS-1$ //$NON-NLS-2$
		}

		line.append("</tr>"); //$NON-NLS-1$

		// Function coverage table cell values.
		for (TableRecordData record : m_summaryData.getSummaryRecords(context)) {
			if (record.recordConfigurationName.equals(UIMessages.CriticalCode_File_Column)) {
				tempFileName = record.fileName.substring(record.fileName.lastIndexOf("\\") + 1); //$NON-NLS-1$
			}
			// Search for childs for current analyzed file
			if (tempFileName.equals(currentFileName)) {
				if ((record.recordConfigurationName.equals(UIMessages.CriticalCode_Function_Column)) && (record.isVisible())) {
					String summaryCellValue = ""; //$NON-NLS-1$
					childName = record.cells.get(0).value;
					functionNames.add(childName);
					tempMetrics = new HashMap<String, Double>();
					line.append("<tr>"); //$NON-NLS-1$

					TableRecordConfiguration columns = fileInput.tableRecordsConfigurations.get(record.recordConfigurationName);
					for (int dataIdx = 0; dataIdx < record.cells.size(); dataIdx++) {
						String columnName = columns.cells.get(dataIdx).name;
						summaryCellValue = record.cells.get(dataIdx).value;
						if (columnName.equals(UIMessages.CriticalCode_Summary_Covered_Column) // Backward compatibility
								|| columnName.equals(UIMessages.CriticalCode_Summary_NotCovered_Column) // Backward compatibility
								|| columnName.equals(UIMessages.CriticalCode_Summary_Covered_ASM_Column)
								|| columnName.equals(UIMessages.CriticalCode_Summary_NotCovered_ASM_Column)
								|| columnName.equals(UIMessages.CriticalCode_Asm_Decision_Coverage)
								|| columnName.equals(UIMessages.CriticalCode_Total_ASM_Instructions)
								// Source line coverage columns
								|| columnName.equals(UIMessages.CriticalCode_Summary_Covered_Source_Column)
								|| columnName.equals(UIMessages.CriticalCode_Summary_PartiallyCovered_Source_Column)
								|| columnName.equals(UIMessages.CriticalCode_Summary_NotCovered_Source_Column)
								|| columnName.equals(UIMessages.CriticalCode_Total_Number_Source_Lines)) {
							tempStringValue = record.cells.get(dataIdx).value.replace(" %", ""); //$NON-NLS-1$//$NON-NLS-2$							
							try {
								tempDoubleVal = Double.parseDouble(tempStringValue);
							} catch (NumberFormatException e) {
								LOGGER.error("[writeFuncCovTable]: number format exception: " + e.getLocalizedMessage()); //$NON-NLS-1$
							}
							tempMetrics.put(columnName, tempDoubleVal);
						}

						if (columnName.equals(UIMessages.CriticalCode_Summary_Covered_Column) // Backward compatibility
								|| columnName.equals(UIMessages.CriticalCode_FileFunction_Column)
								// ASM coverage
								|| (columnName.equals(UIMessages.CriticalCode_Summary_Covered_ASM_Column) 
										&& (m_coverageType == SourceCoverageType.ASM_COVERAGE))
								// Source coverage
								|| (columnName.equals(UIMessages.CriticalCode_Summary_Covered_Source_Column) 
										&& (m_coverageType == SourceCoverageType.SOURCE_COVERAGE))) {

							if (columnName.equals(UIMessages.CriticalCode_FileFunction_Column)) {
								summaryCellValue = "<a style='text-decoration: none' href='#" + context + " " + encodeHTML(UIMessages.CriticalCode_AnalysisFor + " ") + summaryCellValue + "'>" + summaryCellValue + "</a>"; //$NON-NLS-1$//$NON-NLS-2$ //$NON-NLS-3$
							}

							if (columnName.equals(UIMessages.CriticalCode_Summary_Covered_ASM_Column)
									|| columnName.equals(UIMessages.CriticalCode_Summary_Covered_Source_Column)
									|| columnName.equals(UIMessages.CriticalCode_Summary_Covered_Column)) { // Backward compatibility
								try {
									tempDoubleVal = Double.parseDouble(summaryCellValue);
								} catch (NumberFormatException e) {
									LOGGER.error("[writeFuncCovTable]: number format exception: " + e.getLocalizedMessage()); //$NON-NLS-1$
								}
								summaryCellValue = m_doubleFormatter.format(tempDoubleVal);
								summaryCellValue += " %"; //$NON-NLS-1$
							}
							line.append("<th bgcolor='#F5D769'>" + summaryCellValue + "</th>"); //$NON-NLS-1$ //$NON-NLS-2$
						}
					}

					childData.put(childName, tempMetrics);
					line.append("</tr>"); //$NON-NLS-1$
				}
			}
		}
		line.append("</table>" + //$NON-NLS-1$
				"<br>" + //$NON-NLS-1$
				"<br>"); //$NON-NLS-1$

	}
	
	private void loadFunctionDetails(String subsection, String functionName, String fName, SHOW_CODE_TYPE showCode) {

		if (m_summaryData == null ) {
			return;
		}
				

		fileDetails = new TreeMap<String, TreeMap<String, Vector<Vector<String>>>>();
		Vector<Vector<String>> asm = null;
		TreeMap<String, Vector<Vector<String>>> functionDetails;
		Vector<String> temp = null;

		for (int summaryIndex = 0; summaryIndex < m_summaryData.getSummaryRecords(subsection).size(); summaryIndex++) {
			TableRecordData record = m_summaryData.getSummaryRecords(subsection).get(summaryIndex);

			if (record.cells.get(functionNameIndex).value.equals(functionName)) {
				Vector<Long> summaryLines = record.cells.get(functionNameIndex).linksToMultipleTables;

				m_detailsData = fileInput.getDetailsTableData(m_section, subsection, FlatProfilerFileInput.SUMMARY_PARAGRAPH,
						summaryLines.get(0));
			
				String lastFile = ""; //$NON-NLS-1$					
				int lastLine = -1;
				// FileName (str) - FileLine (int) - Contents (str):
				m_sources = new TreeMap<String, TreeMap<Integer, String>>();
				for (int i = 0; i < m_detailsData.records.size(); i++) {

					String lineName = m_detailsData.records.elementAt(i).recordConfigurationName;
					TableRecordConfiguration config = fileInput.tableRecordsConfigurations.get(lineName);
					
					int lineIdx = -1, fileIdx = -1;
					for (int j = 0; j < config.cells.size(); j++) {
						if (config.cells.elementAt(j).sourceLine) {
							lineIdx = j;
						} else if (config.cells.elementAt(j).sourceFile 
								&& !m_detailsData.records.elementAt(i).cells.elementAt(j).value.isEmpty()) {
							fileIdx = j;
						}
					}

					if (fileIdx >= 0) {
						lastFile = m_detailsData.records.elementAt(i).cells.elementAt(fileIdx).value;
					}
					if (lineIdx >= 0) {
						try {
							String detailsData = m_detailsData.records.elementAt(i).cells.elementAt(lineIdx).value;
							lastLine = Integer.parseInt(detailsData);
						} catch (NumberFormatException e1) {
							LOGGER.debug("[loadFunctionDetails]: invalid details data - " + e1.getLocalizedMessage()); //$NON-NLS-1$
						} catch (Exception e) {
							LOGGER.error(e);
						}

						if (lastFile != null && !lastFile.isEmpty()) {
							if (!m_sources.containsKey(lastFile)) {
								m_sources.put(lastFile, new TreeMap<Integer, String>());
							}
							TreeMap<Integer, String> crtSourceFile = m_sources.get(lastFile);
							if (!crtSourceFile.containsKey(lastLine)) {
								crtSourceFile.put(lastLine, lastFile);
							}
						}
					}

					m_detailsData.records.elementAt(i).fileLine = lastLine;
					m_detailsData.records.elementAt(i).fileName = lastFile;
				}
				// Read the source files
				for (String sourceFile : m_sources.keySet()) {
					TreeMap<Integer, String> source = m_sources.get(sourceFile);
					// Disabled: sourceFile = getArchivedFile(sourceFile);
					String mappedSourceFile;
					mappedSourceFile = sourceFile; //We use directly the file here
					File file = new File(mappedSourceFile);
					if (file == null || !file.isFile()) {
						Integer line = source.ceilingKey(0);
						while (line != null) {
							source.remove(line);
							source.put(line, Utils.simplifiedFileName(mappedSourceFile, true));
							line = source.ceilingKey(line + 1);
						}
						continue;
					}

					Integer line = source.ceilingKey(0);
					FileInputStream fstream = null;
					DataInputStream in = null;
					BufferedReader br = null;
					try {
						fstream = new FileInputStream(mappedSourceFile);
						in = new DataInputStream(fstream);
						br = new BufferedReader(new InputStreamReader(in));
						String strLine = mappedSourceFile;
						int index = 1;

						while (line != null) {
							while ((strLine = br.readLine()) != null && index < line) {
								index++;
							}

							source.remove(line);
							if (strLine != null) {
								strLine = strLine.replaceAll("\t", "    "); //$NON-NLS-1$ //$NON-NLS-2$
								source.put(line, strLine);
							}

							line = source.ceilingKey(line + 1);
							index++;
						}
					} catch (Exception e) {
						LOGGER.error(e);
                    } finally {
                        if (fstream != null) {
                            try {
                                fstream.close();
                            } catch (IOException e) {
                                LOGGER.error(e);
                            }
                        }
                        if (br != null) {
                            try {
                                br.close();
                            } catch (IOException e) {
                                LOGGER.error(e);
                            }
                        }
                        if (in != null) {
                            try {
                                in.close();
                            } catch (IOException e) {
                                LOGGER.error(e);
                            }
                        }
                    }
                }

				// If the table contains references to source files, resolve them
				// Read the source file contents
				if (m_sources != null) {
					for (int i = 0; i < m_detailsData.records.size(); i++) {

						String lineName = m_detailsData.records.elementAt(i).recordConfigurationName;
						TableRecordConfiguration config = fileInput.tableRecordsConfigurations.get(lineName);

						int fileIdx = -1;
						for (int j = 0; j < config.cells.size(); j++) {
							if (config.cells.elementAt(j).sourceFile) {
								fileIdx = j;
								break;
							}
						}

						if (fileIdx >= 0) {
							int line = m_detailsData.records.elementAt(i).fileLine;
							String file = m_detailsData.records.elementAt(i).fileName;

							if (line >= 0 && file != null && !file.isEmpty()) {
								m_detailsData.records.elementAt(i).cells.elementAt(fileIdx).readSourceFile = m_sources.get(file).get(line);
							}
						}
					}
				}

				int nrColumns = 0;
				if (!m_detailsData.records.isEmpty()) {
					nrColumns = m_detailsData.records.get(0).cells.size();
				}

				// Depending on export demand, load into fileDetails structure, 
				// details table for each function(as a matrix).
				FlatProfilerTableCell[][] cells;
				cells = new FlatProfilerTableCell[getDetailsTableItemsCount()][];
				for (int lineIdx = 0; lineIdx < getDetailsTableItemsCount(); lineIdx++) {
					cells[lineIdx] = new FlatProfilerTableCell[nrColumns];
					temp = new Vector<String>();
					for (int columnIdx = 0; columnIdx < nrColumns; columnIdx++) {
						cells[lineIdx][columnIdx] = getDetailsTableItem(lineIdx, columnIdx);
					}
				}

				if (showCode == SHOW_CODE_TYPE.MIXED) {
					asm = new Vector<Vector<String>>();
					for (int lineIdx = 0; lineIdx < getDetailsTableItemsCount(); lineIdx++) {
						temp = new Vector<String>();
						for (int columnIdx = 0; columnIdx < nrColumns; columnIdx++) {
							temp.add(cells[lineIdx][columnIdx].text);
						}
						asm.add(temp);
					}
				} else if (showCode == SHOW_CODE_TYPE.ASSEMBLY) {
					Vector<FlatProfilerTableCell[]> assemblyCells = new Vector<FlatProfilerTableCell[]>();

					for (int lineIdx = 0; lineIdx < getDetailsTableItemsCount(); lineIdx++) {
						if (m_detailsData.records.elementAt(lineIdx).recordConfigurationName.compareTo(UIConstants.CC_ASSEMBLY_LINE) == 0) {
							assemblyCells.add(cells[lineIdx]);
						}
					}

					cells = new FlatProfilerTableCell[assemblyCells.size()][];
					asm = new Vector<Vector<String>>();
					for (int lineIdx = 0; lineIdx < assemblyCells.size(); lineIdx++) {
						cells[lineIdx] = new FlatProfilerTableCell[nrColumns];
						temp = new Vector<String>();
						for (int columnIdx = 0; columnIdx < nrColumns; columnIdx++) {
							cells[lineIdx][columnIdx] = assemblyCells.elementAt(lineIdx)[columnIdx];
							temp.add(cells[lineIdx][columnIdx].text);
						}
						asm.add(temp);
					}
				} else if (showCode == SHOW_CODE_TYPE.SOURCE) {
					asm = new Vector<Vector<String>>();
					for (String fileName : m_sources.keySet()) {
						for (Integer fileLine : m_sources.get(fileName).keySet()) {
							temp = new Vector<String>();
							for (int i = 0; i < m_detailsData.records.size(); i++) {
								if (m_detailsData.records.elementAt(i).fileName.equals(fileName) 
										&& m_detailsData.records.elementAt(i).fileLine == fileLine) {
									for (int j = 0; j < cells[i].length; j++) {
										temp.add(cells[i][j].text);
									}
									asm.add(temp);
									break;
								}
							}
						}
					}
				}
				functionDetails = new TreeMap<String, Vector<Vector<String>>>();
				functionDetails.put(functionName, asm);
				fileDetails.put(fName, functionDetails);
			}
		}

	}
	
	public void writeDetailsTable(String context, StringBuilder line, String currentFunctionName, String currentFileName,
			Vector<String> functionNames, int functionIndex, Vector<Vector<String>> detailsTable, SHOW_CODE_TYPE showCode) {
		
		String detailsTableName = null, firstColName = null, detailsTableValue = null;
		
		switch (showCode) {
		case ASSEMBLY:
			detailsTableName = UIMessages.CriticalCode_Assembly_TableName;
			firstColName = UIMessages.CriticalCode_StartAddress_Column;
			break;
		case SOURCE:
			detailsTableName = UIMessages.CriticalCode_Source_TableName;
			firstColName = UIMessages.CriticalCode_Line_Column;
			break;
		case MIXED:
			detailsTableName = UIMessages.CriticalCode_Mixed_TableName;
			firstColName = UIMessages.CriticalCode_AddressLine_Column;
			break;
		}

		// Current function name.
		line.append("<br>" + //$NON-NLS-1$
				"<table style='width:100%' align='center' border='1'>" + //$NON-NLS-1$
				"<tr>" + //$NON-NLS-1$
				"<th bgcolor='#ED8A3F'>" + //$NON-NLS-1$
				"<a style='text-align: center;' id='" + context + " " + encodeHTML(UIMessages.CriticalCode_AnalysisFor + " ") + currentFunctionName + "'>" + context + " - " + currentFunctionName + "</a>" + //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
				"</th>" + //$NON-NLS-1$
				"</tr>" + //$NON-NLS-1$
				"</table>"); //$NON-NLS-1$
		String backLink = "["; //$NON-NLS-1$
		if (treeSummaryView) {
			backLink = "<a href='#" + context + " " + encodeHTML(UIMessages.CriticalCode_AnalysisFor + " ") + currentFileName + "' style='text-decoration: none'>[ " + UIMessages.CriticalCode_BackToTop + " | </a>"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		} else {
			backLink = "<a href='#" + UIMessages.CriticalCode_Functions + ":' style='text-decoration: none'>[ " + UIMessages.CriticalCode_Functions + " | </a>"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		}
		line.append("<p style='text-align: center;'>" + backLink + //$NON-NLS-1$
				"<a style='text-decoration: none' href='#" + context + " " + encodeHTML(UIMessages.CriticalCode_CodeCoverageMetrics_For + " ") + currentFunctionName + "'>" + //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
				UIMessages.CriticalCode_CodeCoverageMetrics + " | " + "</a>" + //$NON-NLS-1$ //$NON-NLS-2$
				"<a style='text-decoration: none' href='#" + context + " " + encodeHTML(UIMessages.CriticalCode_AsmDecisionMetrics_For  + " ") + currentFunctionName + "'>" + //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$ 
				UIMessages.CriticalCode_AsmDecisionMetrics + " ]" + "</a>" + //$NON-NLS-1$ //$NON-NLS-2$
				"</p>"); //$NON-NLS-1$
		line.append("<p style='text-align: center;'>" + //$NON-NLS-1$
				"<b>" + detailsTableName + " " + currentFunctionName + "</b>" + //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
				"</p>"); //$NON-NLS-1$
		line.append("<table style='width:60%; border-spacing: 0; border-collapse: collapse;' align='center' border='1'>" + //$NON-NLS-1$
				"<tr>"); //$NON-NLS-1$

		// Details table header.
		line.append("<th bgcolor='#D4E8C1'>" + firstColName + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
				"<th bgcolor='#D4E8C1'>" + UIMessages.CriticalCode_Instruction_Column + "</th>" + //$NON-NLS-1$ //$NON-NLS-2$
				"<th bgcolor='#D4E8C1'>" + UIMessages.CriticalCode_Details_Coverage_Column + "</th>" + "<th bgcolor='#D4E8C1'>" //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				+ UIMessages.CriticalCode_Asm_Decision_Coverage.replace("%", "") + "</th>"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		line.append("</tr>"); //$NON-NLS-1$

		String coverageColumnValue = null, cellColour = null, cellStyle = null;
		// Details table cell values.
		for (int rowIndex = 0; rowIndex < detailsTable.size(); rowIndex++) {
			line.append("<tr>"); //$NON-NLS-1$
			for (int columnIndex = 0; columnIndex < DETAILS_TABLE_NUMBER_OF_EXPORTED_COLS; columnIndex++) {
				detailsTableValue = detailsTable.get(rowIndex).get(columnIndex);
				
				if (detailsTableValue == null) {
					LOGGER.error("[writeDetailsTable] detailsTableValue is null!"); //$NON-NLS-1$]
					return;
				} else {
					detailsTableValue = encodeHTML(detailsTableValue);
				}
								
				if (columnIndex == 1) {
					cellStyle = "style='padding-left: 30px;' align='left'"; //$NON-NLS-1$ 
					coverageColumnValue = detailsTable.get(rowIndex).get(2);
					
					if (coverageColumnValue == null) {
						LOGGER.error("[writeDetailsTable] coverageColumnValue is null!"); //$NON-NLS-1$
						return;
					}
					
					if (coverageColumnValue.equals(UIMessages.CriticalCode_NotCovered_Text)) {
						cellColour = "#FF0000"; //$NON-NLS-1$ 
					}
				} else {
					cellColour = "#D4E8C1"; //$NON-NLS-1$
					cellStyle = ""; //$NON-NLS-1$
				}
				line.append("<th " + cellStyle + " bgcolor='" + cellColour + "'>" + detailsTableValue + "</th>"); //$NON-NLS-1$//$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
			}
			line.append("</tr>"); //$NON-NLS-1$
		}
		line.append("</table>"); //$NON-NLS-1$
	}
	
	public void writeSummaryTable(String context, StringBuilder line, String currentFileName, 
			Vector<String> functionNames, TreeMap<String, HashMap<String, Double>> childData) {
		
		String currentFunctionName = null, strCoverage = null;
		Double covInsPercents, totalAsmIns, fileCovIns = 0.00, fileTotalIns = 0.00, totalFileCov = 0.00;
		Long functionCovIns;

		HashMap<String, Double> coverageMetrics;

		// Current file name.
		line.append("<p style='text-align: center;'>" + //$NON-NLS-1$
				"<b>" + //$NON-NLS-1$
				"<a id='" + context + " " + encodeHTML(UIMessages.CriticalCode_SummaryTable_For + " ") + currentFileName + "'>" + //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				UIMessages.CriticalCode_SummaryTable_For + " " + "<a style='text-decoration: none' href='#" + context + " " //$NON-NLS-1$ //$NON-NLS-2$
				+ encodeHTML(UIMessages.CriticalCode_AnalysisFor + " ") + //$NON-NLS-1$
				currentFileName + "'>" + currentFileName + //$NON-NLS-1$
				"</a>" + //$NON-NLS-1$
				"</a>" + //$NON-NLS-1$
				"</b>" + //$NON-NLS-1$
				"</p>"); //$NON-NLS-1$

		// Summary table header.
		line.append("<table align='center' style='width:60%; border-spacing: 0; border-collapse: collapse;' border='1'>" + //$NON-NLS-1$
				"<tr>" + //$NON-NLS-1$
				"<th bgcolor='#F5D769'>" + UIMessages.CriticalCode_Function_Column + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
				"<th bgcolor='#F5D769'>" + UIMessages.CriticalCode_Total_ASM_Instructions + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
				"<th bgcolor='#F5D769'>" + UIMessages.CriticalCode_Covered_ASM_Instructions + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
				"<th bgcolor='#F5D769'>" + UIMessages.CriticalCode_Details_Coverage_Column + " %" + "</th>" + //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
				"</tr>"); //$NON-NLS-1$

		// Summary table cell values for functions.
		for (int index = 0; index < functionNames.size(); index++) {
			currentFunctionName = functionNames.get(index);
			coverageMetrics = childData.get(currentFunctionName);											
			
			if (coverageMetrics == null) {
				LOGGER.error("[writeSummaryTable] coverageMetrics is null!"); //$NON-NLS-1$
				return;
			}
			
			// Backward compatibility
			if (coverageMetrics.containsKey(UIMessages.CriticalCode_Summary_Covered_Column)) {
				covInsPercents = coverageMetrics.get(UIMessages.CriticalCode_Summary_Covered_Column);
			} else {
				covInsPercents = coverageMetrics.get(UIMessages.CriticalCode_Summary_Covered_ASM_Column);
			}

			totalAsmIns = coverageMetrics.get(UIMessages.CriticalCode_Total_ASM_Instructions);	
			
			if (covInsPercents == null || totalAsmIns == null) {
				LOGGER.error("[writeSummaryTable] covInsPercents or totalAsmIns is null!"); //$NON-NLS-1$
				return;
			}
			
			functionCovIns = Math.round(( covInsPercents * totalAsmIns ) / 100);
			strCoverage = m_doubleFormatter.format(covInsPercents) + " %"; //$NON-NLS-1$
			line.append("<tr>" + //$NON-NLS-1$
					"<th bgcolor='#F5D769'>" + currentFunctionName + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
					"<th bgcolor='#F5D769'>" + totalAsmIns.intValue() + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
					"<th bgcolor='#F5D769'>" + functionCovIns + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
					"<th bgcolor='#F5D769'>" + strCoverage + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
					"</tr>"); //$NON-NLS-1$
			fileCovIns += functionCovIns;
			fileTotalIns += totalAsmIns;
		}

		// Summary table cell values for "Whole program".
		totalFileCov = (fileCovIns * 100) / fileTotalIns;
		strCoverage = m_doubleFormatter.format(totalFileCov) + " %"; //$NON-NLS-1$
		line.append("<tr>" + //$NON-NLS-1$
				"<th bgcolor='#F5D769'>" + //$NON-NLS-1$
				"<b>" + UIMessages.CriticalCode_WholeFile + "</b>" + //$NON-NLS-1$ //$NON-NLS-2$
				"</th>" + //$NON-NLS-1$
				"<th bgcolor='#F5D769'>" + fileTotalIns.intValue() + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
				"<th bgcolor='#F5D769'>" + fileCovIns.intValue() + "</th>" + //$NON-NLS-1$ //$NON-NLS-2$
				"<th bgcolor='#F5D769'>" + strCoverage + "</th>" + //$NON-NLS-1$ //$NON-NLS-2$
				"</tr>" + //$NON-NLS-1$
				"</table>" + //$NON-NLS-1$
				"<br>" + //$NON-NLS-1$
				"<br>"); //$NON-NLS-1$
	}
	
	public void setFunctionInformation(String subsection, TreeMap<String, HashMap<String, Double>> childData) {

		String childName = null, tempStringValue = null;
		Double tempDoubleVal = 0.00;
		HashMap<String, Double> tempMetrics = null;
		
		// Function coverage table cell values.
		for (TableRecordData record : m_summaryData.getSummaryRecords(subsection)) {
			
			if (record.recordConfigurationName.equals(UIMessages.CriticalCode_Function_Column)) {					
				childName = record.cells.get(0).value;						
				tempMetrics = new HashMap<String, Double>();

				TableRecordConfiguration columns = fileInput.tableRecordsConfigurations.get(record.recordConfigurationName);
				for (int dataIdx = 0; dataIdx < record.cells.size(); dataIdx++) {
					String columnName = columns.cells.get(dataIdx).name;
					if (columnName.equals(UIMessages.CriticalCode_Summary_Covered_Column) // Backward compatibility
							|| columnName.equals(UIMessages.CriticalCode_Summary_NotCovered_Column) // Backward compatibility
							|| columnName.equals(UIMessages.CriticalCode_Summary_Covered_ASM_Column)
							|| columnName.equals(UIMessages.CriticalCode_Summary_NotCovered_ASM_Column)
							|| columnName.equals(UIMessages.CriticalCode_Asm_Decision_Coverage)
							|| columnName.equals(UIMessages.CriticalCode_Total_ASM_Instructions)
							|| columnName.equals(UIMessages.CriticalCode_Summary_Covered_Source_Column)
							|| columnName.equals(UIMessages.CriticalCode_Total_Number_Source_Lines)) {						
						tempStringValue = record.cells.get(dataIdx).value.replace(" %", ""); //$NON-NLS-1$//$NON-NLS-2$							
						try {
							tempDoubleVal = Double.parseDouble(tempStringValue);
						} catch (NumberFormatException e) {
							LOGGER.error("[writeFuncCovTable]: number format exception: " + e.getLocalizedMessage()); //$NON-NLS-1$
						}
						tempMetrics.put(columnName, tempDoubleVal);
					}
					childData.put(childName, tempMetrics);
				}
			}
		}
	}
	
	public void writeFileAnalysisEnd(String context, StringBuilder line, String currentFileName) {

		line.append("<br>" + //$NON-NLS-1$
				"<p style='text-align: center; font-size: 150%'>" + //$NON-NLS-1$
				"<b>End of analysis for " + currentFileName + "</b>" + //$NON-NLS-1$ //$NON-NLS-2$
				"</p>" + //$NON-NLS-1$
				"<p style='text-align: center;'>" + //$NON-NLS-1$
				"<a style='text-decoration: none' href='#" + " " + encodeHTML(UIMessages.CriticalCode_AnalysisFor + " ") //$NON-NLS-1$//$NON-NLS-2$
				+ currentFileName + "'>[ " + UIMessages.CriticalCode_BackToTop + " ]</a>" + //$NON-NLS-1$ //$NON-NLS-2$
				"</p>" + //$NON-NLS-1$
				"<hr>" + //$NON-NLS-1$
				"<br>"); //$NON-NLS-1$
	}
	
	private void writeCodeCoverageTable(String context, StringBuilder line, String currentFunctionName, TreeMap<String, HashMap<String, Double>> childData) {

		Integer covIns, notCovIns;
		Double covInsPercent, totalIns;
		HashMap<String, Double> coverageMetrics;

		// Current function name.
		line.append("<p style='text-align: center;'>" + //$NON-NLS-1$
				"<b><a id='" + context + " " + encodeHTML(UIMessages.CriticalCode_CodeCoverageMetrics_For + " ") + currentFunctionName + "'>" + UIMessages.CriticalCode_CodeCoverageMetrics_For + " " + //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$ //$NON-NLS-4$
				"<a style='text-decoration: none' href='#" + context + " " + currentFunctionName + "'>" + currentFunctionName + //$NON-NLS-1$ //$NON-NLS-2$
				"</a>" + //$NON-NLS-1$
				"</a>" + //$NON-NLS-1$
				"</b>" + //$NON-NLS-1$
				"</p>"); //$NON-NLS-1$

		// Code coverage table header.
		if (m_coverageType.equals(SourceCoverageType.ASM_COVERAGE)) {
			line.append("<table style='width:60%' align='center' border='1'>" + //$NON-NLS-1$
					"<tr>" + //$NON-NLS-1$
					"<th bgcolor='#828277'>" + UIMessages.CriticalCode_Total_ASM_Instructions + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
					"<th bgcolor='#828277'>" + UIMessages.CriticalCode_Covered_ASM_Instructions + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
					"<th bgcolor='#828277'>" + UIMessages.CriticalCode_NotCovered_ASM_Instructions + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
					"</tr>"); //$NON-NLS-1$
		} else {
			line.append("<table style='width:60%' align='center' border='1'>" + //$NON-NLS-1$
					"<tr>" + //$NON-NLS-1$
					"<th bgcolor='#828277'>" + UIMessages.CriticalCode_Total_Number_Source_Lines + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
					"<th bgcolor='#828277'>" + UIMessages.CriticalCode_Covered_Number_Source_Lines + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
					"<th bgcolor='#828277'>" + UIMessages.CriticalCode_NotCovered_Number_Source_Lines + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
					"</tr>"); //$NON-NLS-1$
		}

		// Code coverage table cell values.
		coverageMetrics = childData.get(currentFunctionName);

		if (coverageMetrics == null) {
			LOGGER.error("[writeCodeCoverageTable] coverageMetrics is null!"); //$NON-NLS-1$
			return;
		}
		
		// Backward compatibility
		if (coverageMetrics.containsKey(UIMessages.CriticalCode_Summary_Covered_Column)) {
			covInsPercent = coverageMetrics.get(UIMessages.CriticalCode_Summary_Covered_Column);
		} else if (m_coverageType.equals(SourceCoverageType.ASM_COVERAGE)) {
			covInsPercent = coverageMetrics.get(UIMessages.CriticalCode_Summary_Covered_ASM_Column);
		} else {
			covInsPercent = coverageMetrics.get(UIMessages.CriticalCode_Summary_Covered_Source_Column);
		}

		if (m_coverageType.equals(SourceCoverageType.ASM_COVERAGE)) {
			totalIns = coverageMetrics.get(UIMessages.CriticalCode_Total_ASM_Instructions);
		} else {
			totalIns = coverageMetrics.get(UIMessages.CriticalCode_Total_Number_Source_Lines);
		}
		
		if (covInsPercent == null || totalIns == null) {
			LOGGER.error("[writeCodeCoverageTable] covInsPercent or totalIns is null!"); //$NON-NLS-1$
			return;
		}
		
		covIns = (int)((covInsPercent * totalIns) / 100);
		notCovIns = totalIns.intValue() - covIns;	
		
		line.append("<tr>" + //$NON-NLS-1$
				"<th bgcolor='#828277'>" + totalIns.intValue() + "</th>" + //$NON-NLS-1$ //$NON-NLS-2$
				"<th bgcolor='#828277'>" + covIns + "</th>" + //$NON-NLS-1$ //$NON-NLS-2$
				"<th bgcolor='#828277'>" + notCovIns + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
				"</tr>" + //$NON-NLS-1$
				"</table>"); //$NON-NLS-1$
	}
	
	public void writeAsmTable(String context, StringBuilder line, String type, TreeMap<String, Vector<Double>> asmDecisionMetrics,
		 	String currentFileName, String currentFunctionName, Vector<String> functionNames, Vector<Vector<String>> detailsTable,
		 	TreeMap<String, HashMap<String, Double>> childData) {
				
		String decisionCoverage = null, functionDecisionCoverage = null, fileDecisionCoverage = null;
		Double asmTaken, asmNotTaken, asmBoth, totalAsmDec, totalExecCondBranches, tempDoubleValue, 
			tempDecCoverage = 0.00, currentTotalAsmDec = 0.00, execConditionalBranches = 0.00, asmNotCovered = 0.00;
		Vector<Double> tempDecisions;
		HashMap<String, Double> coverageMetrics;

		// ASM decision metrics table for function.
		if (type.equals("function asm")) { //$NON-NLS-1$

			// Current function name.
			line.append("<p style='text-align: center;'>" + //$NON-NLS-1$
					"<b><a id='" + context + " " + encodeHTML(UIMessages.CriticalCode_AsmDecisionMetrics_For + " ") + currentFunctionName + "'>" + //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					UIMessages.CriticalCode_AsmDecisionMetrics_For + " " + //$NON-NLS-1$
					"<a style='text-decoration: none' href='#" + context + " " //$NON-NLS-1$
					+ currentFunctionName + "'>" + currentFunctionName + //$NON-NLS-1$
					"</a>" + //$NON-NLS-1$
					"</a>" + //$NON-NLS-1$
					"</b>" + //$NON-NLS-1$
					"</p>"); //$NON-NLS-1$

			// ASM decision metrics table header(for function).
			line.append("<table style='width:60%' align='center' border='1'>" + //$NON-NLS-1$
						"<tr>" + //$NON-NLS-1$
						"<th bgcolor='#F5D769'>" + UIMessages.CriticalCode_Total_ASM_Decisions + "</th>" + //$NON-NLS-1$ //$NON-NLS-2$
						"<th bgcolor='#F5D769'>" + UIMessages.CriticalCode_Taken_ASM_Decisions + "</th>" +	//$NON-NLS-1$//$NON-NLS-2$
						"<th bgcolor='#F5D769'>" + UIMessages.CriticalCode_NotTaken_ASM_Decisions + "</th>" + //$NON-NLS-1$ //$NON-NLS-2$
						"<th bgcolor='#F5D769'>" + UIMessages.CriticalCode_BothTaken_ASM_Decisions + "</th>" + //$NON-NLS-1$ //$NON-NLS-2$
						"<th bgcolor='#F5D769'>" + UIMessages.CriticalCode_NotCovered_ASM_Decisions + "</th>" + //$NON-NLS-1$ //$NON-NLS-2$
						"</tr>"); //$NON-NLS-1$
		
			String asmDecCoverage = null;
			totalAsmDec = asmTaken = asmNotTaken = asmBoth = 0.00;								
			
			int asmDecisionCoverageColumnIndex = -1;
			FlatProfilerColumn[] detailsTableColumns = getDetailsConfiguration();
			for (int i = 0; i < detailsTableColumns.length; i++) {
				if (detailsTableColumns[i].getName().equals(
						UIMessages.CriticalCode_Asm_Decision_Coverage.replace(
								" %", ""))) { //$NON-NLS-1$ //$NON-NLS-2$
					asmDecisionCoverageColumnIndex = i;
					break;
				}
			}
			
			if (asmDecisionCoverageColumnIndex == -1) {
				LOGGER.error("[writeAsmTable] Wrong asm decision coverage column index!"); //$NON-NLS-1$
				return;
			}
			
			for (int rowIndex=0; rowIndex < detailsTable.size(); rowIndex++) {
				
				asmDecCoverage = detailsTable.get(rowIndex).get(asmDecisionCoverageColumnIndex);									 					
				
				if (asmDecCoverage != null) {
					if (asmDecCoverage.equals(UIMessages.CriticalCode_Branch_Taken)) { // If branch is taken.
						asmTaken++;
					} else if (asmDecCoverage.equals(UIMessages.CriticalCode_Branch_Not_Taken)) { // If branch is not taken.														
						asmNotTaken++;				
					} else if (asmDecCoverage.equals(UIMessages.CriticalCode_Branch_Taken_And_Not_Taken)) { // If branch is both taken and not taken.													
						asmBoth++;							
					}
				} else {
					LOGGER.error("[writeAsmTable] asmDecCoverage is null!"); //$NON-NLS-1$
					return;
				}
			}

			tempDecisions = new Vector<Double>();
		
			coverageMetrics = childData.get(currentFunctionName);								
		
			if (coverageMetrics == null) {
				LOGGER.error("[writeAsmTable] coverageMetrics is null!"); //$NON-NLS-1$
				return;
			}
			
			if (specialConditionalInstr.containsKey(currentFunctionName)) {
				totalAsmDec = specialConditionalInstr.get(currentFunctionName);
			} else {
				tempDoubleValue = coverageMetrics.get(UIMessages.CriticalCode_Asm_Decision_Coverage);
				
				if (tempDoubleValue == null) {
					LOGGER.error("[writeAsmTable] tempDoubleValue is null!"); //$NON-NLS-1$
					return;
				}
				
				totalAsmDec = (double) Math.round((100 * (asmTaken/2 + asmNotTaken/2 + asmBoth)) / tempDoubleValue);		
			}											
		
			if (totalAsmDec == null) {
				LOGGER.error("[writeAsmTable] totalAsmDec is null!"); //$NON-NLS-1$
				return;
			}
			
			tempDecisions.add(totalAsmDec);
			tempDecisions.add(asmTaken);
			tempDecisions.add(asmNotTaken);
			tempDecisions.add(asmBoth);
			asmDecisionMetrics.put(currentFunctionName, tempDecisions);

			asmNotCovered = totalAsmDec - (asmTaken + asmNotTaken + asmBoth);

			// ASM decision metrics table cell values(for function).
			line.append("<tr>" + //$NON-NLS-1$
					"<th bgcolor='#F5D769'>" + totalAsmDec.intValue() + "</th>" + //$NON-NLS-1$ //$NON-NLS-2$
					"<th bgcolor='#F5D769'>" + asmTaken.intValue() + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
					"<th bgcolor='#F5D769'>" + asmNotTaken.intValue() + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
					"<th bgcolor='#F5D769'>" + asmBoth.intValue() + "</th>" + //$NON-NLS-1$ //$NON-NLS-2$
					"<th bgcolor='#F5D769'>" + asmNotCovered.intValue() + "</th>" + //$NON-NLS-1$ //$NON-NLS-2$
					"</tr>" + //$NON-NLS-1$
					"</table>" + //$NON-NLS-1$
					"<br>" + //$NON-NLS-1$
					"<br>"); //$NON-NLS-1$
		}

		// ASM decision metrics table for file.
		if (type.equals("file asm")) { //$NON-NLS-1$

			// Current file name.
			line.append("<p style='text-align: center;'>" + //$NON-NLS-1$
					"<b>" + //$NON-NLS-1$
					"<a id='" + context + " " + encodeHTML(UIMessages.CriticalCode_AsmDecisionCoverage_For + " ") + currentFileName + "'>" + //$NON-NLS-1$//$NON-NLS-2$ //$NON-NLS-3$
					UIMessages.CriticalCode_AsmDecisionCoverage_For + " " + //$NON-NLS-1$
					"<a style='text-decoration: none' href='#" + context + " " + encodeHTML(UIMessages.CriticalCode_AnalysisFor + " ") + currentFileName + "'>" + currentFileName + //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					"</a>" + //$NON-NLS-1$
					"</a>" + //$NON-NLS-1$
					"</a>" + //$NON-NLS-1$
					"</b>" + //$NON-NLS-1$
					"</p>"); //$NON-NLS-1$

			// ASM decision metrics table header(for file).
			line.append("<table align='center' style='width:60%; border-spacing: 0; border-collapse: collapse;' border='1'>" + //$NON-NLS-1$
					"<tr>" + //$NON-NLS-1$
					"<th bgcolor='#ED8A3F'>" + UIMessages.CriticalCode_Function_Column + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
					"<th bgcolor='#ED8A3F'>" + UIMessages.CriticalCode_Total_ASM_Decisions + "</th>" + //$NON-NLS-1$ //$NON-NLS-2$
					"<th bgcolor='#ED8A3F'>" + UIMessages.CriticalCode_Asm_Decision_Coverage + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
					"</tr>"); //$NON-NLS-1$
			totalAsmDec = totalExecCondBranches = 0.00;

			for (int index = 0; index < functionNames.size(); index++) {
				
				currentFunctionName = functionNames.get(index);										
				tempDecisions = asmDecisionMetrics.get(currentFunctionName);						
				
				if (tempDecisions == null) {
					LOGGER.error("[writeAsmTable] tempDecisions is null!"); //$NON-NLS-1$
					return;
				}
				
				currentTotalAsmDec = tempDecisions.get(0);
				
				if (currentTotalAsmDec == null) {
					LOGGER.error("[writeAsmTable] currentTotalAsmDec is null!"); //$NON-NLS-1$
					return;
				}
				
				totalAsmDec += currentTotalAsmDec; 
				execConditionalBranches = tempDecisions.get(1)/2 + tempDecisions.get(2)/2 + tempDecisions.get(3);
				totalExecCondBranches += execConditionalBranches; 						
				
				if (currentTotalAsmDec != 0) {
					tempDecCoverage = (execConditionalBranches / currentTotalAsmDec) * 100;
				} else {
					tempDecCoverage = 0.00;
				}

				decisionCoverage = m_doubleFormatter.format(tempDecCoverage) + " %"; //$NON-NLS-1$

				if (decisionCoverage.equals("0 %")) { //$NON-NLS-1$
					functionDecisionCoverage = "<a style='text-decoration: none' href='#" + context + " "; //$NON-NLS-1$
					if (specialConditionalInstr.containsKey(currentFunctionName)) {
						// This means that there were no executed conditional
						// branches.
						functionDecisionCoverage += UIMessages.CriticalCode_No_Executed_Conditional_Branches
								+ "'>N/A"; //$NON-NLS-1$
					} else {
						// This means that there were no conditional branches at
						// all.
						functionDecisionCoverage += UIMessages.CriticalCode_No_Conditional_Branches
								+ "'>0 %"; //$NON-NLS-1$
					}
					functionDecisionCoverage += "</a>"; //$NON-NLS-1$
				} else {
					functionDecisionCoverage = decisionCoverage;
				}

				// ASM decision metrics table cell values(for file).
				line.append("<tr>" + //$NON-NLS-1$
						"<th bgcolor='#ED8A3F'>" + currentFunctionName + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
						"<th bgcolor='#ED8A3F'>" + currentTotalAsmDec.intValue() + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
						"<th bgcolor='#ED8A3F'>" + functionDecisionCoverage + "</th>" + //$NON-NLS-1$ //$NON-NLS-2$
						"</tr>"); //$NON-NLS-1$
			}

			// ASM decision metrics table cell values for "Whole program".
			if (totalAsmDec != 0) {
				tempDecCoverage = (totalExecCondBranches / totalAsmDec) * 100;
			} else {
				tempDecCoverage = 0.00;
			}

			decisionCoverage = m_doubleFormatter.format(tempDecCoverage) + " %"; //$NON-NLS-1$

			if (decisionCoverage.equals("0 %")) { //$NON-NLS-1$
				fileDecisionCoverage = "<a style='text-decoration: none' href='#" + context + " "; //$NON-NLS-1$
				if (specialConditionalInstr.containsKey(currentFunctionName)) {
					// This means that there were no executed conditional branches.
					fileDecisionCoverage += UIMessages.CriticalCode_No_Executed_Conditional_Branches + "'>N/A"; //$NON-NLS-1$
				} else {
					// This means that there were no conditional branches at all.
					fileDecisionCoverage += UIMessages.CriticalCode_No_Conditional_Branches + "'>0 %"; //$NON-NLS-1$
				}
				fileDecisionCoverage += "</a>"; //$NON-NLS-1$
			} else {
				fileDecisionCoverage = decisionCoverage;
			}

			line.append("<tr>" + //$NON-NLS-1$
					"<th bgcolor='#ED8A3F'>" + UIMessages.CriticalCode_WholeFile + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
					"<th bgcolor='#ED8A3F'>" + totalAsmDec.intValue() + "</th>" + //$NON-NLS-1$ //$NON-NLS-2$
					"<th bgcolor='#ED8A3F'>" + fileDecisionCoverage + "</th>" + //$NON-NLS-1$//$NON-NLS-2$
					"</tr>"); //$NON-NLS-1$
			line.append("</table>" + //$NON-NLS-1$
					"<br >" + //$NON-NLS-1$
					"<br>"); //$NON-NLS-1$
		}
	}
	
	public void writeHtmlPageHeaders(StringBuilder line) {
		
		line.append("<!DOCTYPE html>"); //$NON-NLS-1$
		line.append("<html>" + //$NON-NLS-1$
				"<head><title>" + UIMessages.CriticalCode_HtmlPageTitle + "</title></head>" + //$NON-NLS-1$ //$NON-NLS-2$
				"<br>"); //$NON-NLS-1$
		line.append("<h1 style='text-align: center; font-size: x-large'>" + UIMessages.CriticalCode_HtmlPageTitle + "</h1>" + //$NON-NLS-1$//$NON-NLS-2$
				"<br>" + //$NON-NLS-1$
				"<br>"); //$NON-NLS-1$

		// List of files/functions selected by user.
		String link;
		if (treeSummaryView) {
			line.append("<h2><a id='" + UIMessages.CriticalCode_Files + ":'>" + UIMessages.CriticalCode_Files + ":</a></h2>"); //$NON-NLS-1$//$NON-NLS-2$ //$NON-NLS-3$
			link = "<a style='text-decoration: none' href='#" + encodeHTML(UIMessages.CriticalCode_AnalysisFor + " "); //$NON-NLS-1$ //$NON-NLS-2$;
		} else {
			line.append("<h2><a id='" + UIMessages.CriticalCode_Functions + ":'>" + UIMessages.CriticalCode_Functions + ":</a></h2>"); //$NON-NLS-1$//$NON-NLS-2$ //$NON-NLS-3$
			link = "<a style='text-decoration: none' href='#"; //$NON-NLS-1$
		}
		line.append("<ul>"); //NON-NLS-1$
		
		for (String context : m_summaryData.getKeyset()) {
			line.append("<b><a id='" + context + ":'>" + context + ":</a></b>"); //$NON-NLS-1$//$NON-NLS-2$ //$NON-NLS-3$
			link = "<a style='text-decoration: none' href='#" + context + " " +encodeHTML(UIMessages.CriticalCode_AnalysisFor + " "); //$NON-NLS-1$ //$NON-NLS-2$;
			Vector<String> names = (treeSummaryView == true) ? getFileNames(context) : getFunctionNames(context);
			for (int i = 0; i < names.size(); i++) {
				line.append("<li>" + link //$NON-NLS-1$
						+ names.get(i) + "'>" + names.get(i) + "</a>" + //$NON-NLS-1$ //$NON-NLS-2$
						"</li>"); //$NON-NLS-1$
			}
		}
		
						
		line.append("</ul>" + //$NON-NLS-1$
				"<br>"); //$NON-NLS-1$
		if (treeSummaryView) {
			// Description of file analysis content.
			line.append("<b>" + UIMessages.CriticalCode_File_Analysis_Content + "</b>" + //$NON-NLS-1$ //$NON-NLS-2$
					"<br>" + //$NON-NLS-1$
					"<ul>"); //$NON-NLS-1$
			line.append("<li><p>" + UIMessages.CriticalCode_Function_Coverage_Table + "</p></li>" + //$NON-NLS-1$ //$NON-NLS-2$
					"<li><p>" + UIMessages.CriticalCode_For_Each_Function + "</p></li>" + //$NON-NLS-1$//$NON-NLS-2$
					"<ul>" + //$NON-NLS-1$ 
					"<li>" + UIMessages.CriticalCode_Details_Table + "</li>" + //$NON-NLS-1$//$NON-NLS-2$
					"<li>" + UIMessages.CriticalCode_CodeCoverageMetrics_Table + "</li>" + //$NON-NLS-1$//$NON-NLS-2$
					"<li>" + UIMessages.CriticalCode_AsmDecisionMetrics_Table + "</li>" + //$NON-NLS-1$ //$NON-NLS-2$
					"</ul>" + //$NON-NLS-1$
					"<li><p>" + UIMessages.CriticalCode_Summary_Table + "</p></li>" + //$NON-NLS-1$ //$NON-NLS-2$
					"<li><p>" + UIMessages.CriticalCode_AsmDecisionCoverage_Table + ":" + "</p></li>" + //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					"<ul>" + //$NON-NLS-1$
					"<li><a id='" + UIMessages.CriticalCode_No_Executed_Conditional_Branches + "'>" + UIMessages.CriticalCode_No_Executed_Conditional_Branches + "</a></li>" + //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					"<li><a id='" + UIMessages.CriticalCode_No_Conditional_Branches + "'>" + UIMessages.CriticalCode_No_Conditional_Branches + "</a></li>" + //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					"</ul>" + //$NON-NLS-1$
					"</ul>" + //$NON-NLS-1$
					"<br>" + //$NON-NLS-1$
					"<hr>" + //$NON-NLS-1$
					"<br>"); //$NON-NLS-1$
		} else {
			// Description of function analysis content.
			line.append("<b>" + UIMessages.CriticalCode_Function_Analysis_Content + "</b>" + //$NON-NLS-1$ //$NON-NLS-2$
					"<br>" + //$NON-NLS-1$
					"<ul>"); //$NON-NLS-1$
			line.append("<li>" + UIMessages.CriticalCode_Details_Table + "</li>" + //$NON-NLS-1$//$NON-NLS-2$
					"<li>" + UIMessages.CriticalCode_CodeCoverageMetrics_Table + "</li>" + //$NON-NLS-1$//$NON-NLS-2$
					"<li>" + UIMessages.CriticalCode_AsmDecisionMetrics_Table + "</li>" + //$NON-NLS-1$ //$NON-NLS-2$
					"</ul>" + //$NON-NLS-1$
					"<br>" + //$NON-NLS-1$
					"<hr>" + //$NON-NLS-1$
					"<br>"); //$NON-NLS-1$
		}
	}
	
	private void drawSwitchButton(ToolBar toolBar) {
		m_switchButton = new ToolItem(toolBar, SWT.PUSH);
		final URL filterUrl = FileLocator.find(Platform.getBundle(com.freescale.sa.ui.criticalcode.utils.UIConstants.PLUGIN_ID), //$NON-NLS-1$
				new Path("icons/switch.png"), null); //$NON-NLS-1$

		if (m_coverageType == SourceCoverageType.ASM_COVERAGE) {
			m_switchButton.setToolTipText(UIMessages.FlatProfilerInput_19);
		} else {
			m_switchButton.setToolTipText(UIMessages.FlatProfilerInput_18);
		}

		m_switchButton.setImage(ImageDescriptor.createFromURL(filterUrl).createImage());

		if (hasSourceCoverage()) { // Backward compatibility.
			m_switchButton.setEnabled(true);
		} else {
			m_switchButton.setEnabled(false);
		}

		m_switchButton.addListener(SWT.Selection, 
				new Listener() {
			public void handleEvent(Event event) {
				int columnsOrder[] = null;
				boolean showColumns[] = null;

				// Save the order and visibility of the columns before changing view.
				if (m_coverageType == SourceCoverageType.ASM_COVERAGE) {

					for (int i = 0; i < m_summary.getColumnCount(); i++) {
						m_columnsOrderAsm[i] = m_summary.getColumnOrder()[i];
						m_showColumnsAsm[i] = m_summary.getColumn(i).getWidth() > 0;
					}

					m_switchButton.setToolTipText(UIMessages.FlatProfilerInput_18);

					columnsOrder = m_columnsOrderSource.clone();
					showColumns = m_showColumnsSource.clone();
					m_coverageType = SourceCoverageType.SOURCE_COVERAGE;
				} else {
					for (int i = 0; i < m_summary.getColumnCount(); i++) {
						m_columnsOrderSource[i] = m_summary.getColumnOrder()[i];
						m_showColumnsSource[i] = m_summary.getColumn(i).getWidth() > 0;
					}

					m_switchButton.setToolTipText(UIMessages.FlatProfilerInput_19);

					columnsOrder = m_columnsOrderAsm.clone();
					showColumns = m_showColumnsAsm.clone();
					m_coverageType = SourceCoverageType.ASM_COVERAGE;
				}

				m_summary.setColumnOrder(columnsOrder);

				// Change coverage view by hiding columns
				// Note : Each setWidth call notifies all SWT.Move and SWT.Resize listeners
				FlatProfilerColumn[] summaryTableColumns = getSummaryConfiguration();
				for (int j = 0; j < columnsOrder.length; j++) {
					TreeColumn column = m_summary.getColumn(j);

					if (showColumns[j]) {
						column.setWidth(summaryTableColumns[j].getWidth());
					} else {
						column.setWidth(0);
					}
				}

				reloadTree(m_summaryViewer);
			}
		});
	}
		
	@Override
	protected ToolBar createSummaryToolBar(Composite parent) {
		ToolBar toolBar = super.createSummaryToolBar(parent);
		
		drawExpandButton(toolBar);
		drawFilterButton(toolBar);
		drawSwitchButton(toolBar);
		
		toolBar.pack();

		return toolBar;
	}
	
	
	
	private void setGraphics(String name) {
		m_graphics.get(name).show = !m_graphics.get(name).show;
		m_details.layout();
	}
	
	public void drawGraphicsButton(ToolBar _toolBar) {
		final ToolBar toolBar = _toolBar;
		final ToolItem graphics = new ToolItem(toolBar, SWT.DROP_DOWN);
		URL graphicsUrl = FileLocator.find(Platform.getBundle(com.freescale.sa.ui.criticalcode.utils.UIConstants.PLUGIN_ID), //$NON-NLS-1$
				new Path("icons/bar-chart-icon.png"), null); //$NON-NLS-1$

		graphics.setImage(ImageDescriptor.createFromURL(graphicsUrl).createImage());
		graphics.setToolTipText(UIMessages.FlatProfilerInput_3);
		graphics.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				Rectangle bounds = graphics.getBounds();
				Point point = toolBar.toDisplay(bounds.x, bounds.y + bounds.height);
				final Menu menu = new Menu(toolBar.getShell(), SWT.POP_UP);

				for (String name : m_graphics.keySet()) {
					final MenuItem item = new MenuItem(menu, SWT.CHECK);
					final String itemName = name;
					item.setText(NLSUtils.getTranslatedString(itemName));
					item.setSelection(m_graphics.get(name).show);
					item.addSelectionListener(new SelectionAdapter() {
						public void widgetSelected(SelectionEvent e) {
							setGraphics(itemName);
							m_details.redraw();
						}
					});
				}

				menu.setLocation(point);
				menu.setVisible(true);
			}
		});
	}
	
	public void drawShowCodeButton(ToolBar _toolBar) {
		final ToolBar toolBar = _toolBar;
		final ToolItem showCode = new ToolItem(toolBar, SWT.DROP_DOWN);
		final CriticalCodeProfilerTab tab = this;
		URL showCodeUrl = FileLocator.find(Platform.getBundle(com.freescale.sa.ui.criticalcode.utils.UIConstants.PLUGIN_ID), //$NON-NLS-1$
				new Path("icons/source-c-icon.png"), null); //$NON-NLS-1$

		showCode.setImage(ImageDescriptor.createFromURL(showCodeUrl).createImage());
		showCode.setToolTipText(UIMessages.FlatProfilerInput_show_code);
		
		if (hasSourceCoverage()) { // Backward compatibility.
			showCode.setEnabled(true);
		} else {
			showCode.setEnabled(false);
		}
		
		showCode.addListener(SWT.Selection, 
				new Listener() {
			public void handleEvent(Event event) {
				Rectangle bounds = showCode.getBounds();
				Point point = toolBar.toDisplay(bounds.x, bounds.y + bounds.height);
				final Menu menu = new Menu(toolBar.getShell(), SWT.POP_UP);

				final MenuItem assembly = new MenuItem(menu, SWT.RADIO);
				

				assembly.setText(UIMessages.FlatProfilerInput_2);
				URL assemblyUrl = FileLocator.find(Platform.getBundle(com.freescale.sa.ui.criticalcode.utils.UIConstants.PLUGIN_ID), //$NON-NLS-1$
						new Path("icons/Letter-A-icon.png"), null); //$NON-NLS-1$

				assembly.setImage(ImageDescriptor.createFromURL(assemblyUrl).createImage());
				assembly.addSelectionListener(new ShowCodeSelectionAdapter(
						tab, SHOW_CODE_TYPE.ASSEMBLY));
				
				if (m_showCode == SHOW_CODE_TYPE.ASSEMBLY) {
					assembly.setSelection(true);
				} 

				if (!bigElfOptimization) {
					//Add source
					final MenuItem source = new MenuItem(menu, SWT.RADIO);
										
					source.setText(UIMessages.FlatProfilerInput_5);
					URL sourceUrl = FileLocator.find(Platform.getBundle(com.freescale.sa.ui.criticalcode.utils.UIConstants.PLUGIN_ID),
							new Path("icons/Letter-S-icon.png"), null); //$NON-NLS-1$
	
					source.setImage(ImageDescriptor.createFromURL(sourceUrl).createImage());
					source.addSelectionListener(new ShowCodeSelectionAdapter(
							tab, SHOW_CODE_TYPE.SOURCE));
	
					//Add mixed
					final MenuItem mixed = new MenuItem(menu, SWT.RADIO);
					
					mixed.setText(UIMessages.FlatProfilerInput_6);
					URL mixedUrl = FileLocator.find(Platform.getBundle(com.freescale.sa.ui.criticalcode.utils.UIConstants.PLUGIN_ID), //$NON-NLS-1$
							new Path("icons/Letter-M-icon.png"), null); //$NON-NLS-1$
	
					mixed.setImage(ImageDescriptor.createFromURL(mixedUrl).createImage());
					mixed.addSelectionListener(new ShowCodeSelectionAdapter(
							tab, SHOW_CODE_TYPE.MIXED));
					
					if (m_showCode == SHOW_CODE_TYPE.SOURCE) {
						source.setSelection(true);
					} else if (m_showCode == SHOW_CODE_TYPE.MIXED) {
						mixed.setSelection(true);
					}
				}

				menu.setLocation(point);
				menu.setVisible(true);
			}
		});
	}
	
	@Override
	protected ToolBar createDetailsToolBar(Composite parent) {
		ToolBar toolBar = super.createDetailsToolBar(parent);
		
		drawGraphicsButton(toolBar);
		drawShowCodeButton(toolBar);

		toolBar.pack();

		return toolBar;
	}
		
	private boolean hasSourceCoverage() {
		if (m_hasSourceCoverage == null) {
			FlatProfilerColumn[] summaryTableColumns = getSummaryConfiguration();
			m_hasSourceCoverage = false;

			for (FlatProfilerColumn column : summaryTableColumns) {
				if (column.getName().equals(COVERED_SOURCE_COLUMN)) {
					m_hasSourceCoverage = true;
					break;
				}
			}
		}

		return m_hasSourceCoverage;
	}
	
	@Override
	public Composite draw() {
		super.draw();
		
		FlatProfilerColumn[] summaryTableColumns = getSummaryConfiguration();
		
		if (m_columnsOrderAsm == null && hasSourceCoverage()) { // Backward compatibility.
			int columnsCount = summaryTableColumns.length;

			m_columnsOrderAsm = new int[columnsCount];
			m_columnsOrderSource = new int[columnsCount];
			m_showColumnsAsm = new boolean[columnsCount];
			m_showColumnsSource = new boolean[columnsCount];

			for (int i = 0; i < columnsCount; i++) {
				String columnName = summaryTableColumns[i].getName();

				m_columnsOrderAsm[i] = i;
				m_columnsOrderSource[i] = i;
				m_showColumnsAsm[m_columnsOrderAsm[i]] = true;
				m_showColumnsSource[m_columnsOrderSource[i]] = true;

				if (SourceCoverageType.ASM_COVERAGE.getExcludedColumns().contains(columnName)) {
					m_showColumnsAsm[m_columnsOrderAsm[i]] = false;
				}

				if (SourceCoverageType.SOURCE_COVERAGE.getExcludedColumns().contains(columnName)) {
					m_showColumnsSource[m_columnsOrderSource[i]] = false;
				}
			}
		}
		
		// Update table preferences from properties file
		String propertiesFile = getSummaryPropertiesFile();
		File file = new File(propertiesFile);
		int columnsOrder[] = null;
		boolean showColumns[] = null;
		int columnsCount = m_summary.getColumnCount();
		if (file.exists()) {
			boolean loadedProperties = false;

			if (hasSourceCoverage()) { // Backward compatibility.
				loadedProperties = loadPropertiesTable(propertiesFile, m_columnsOrderAsm, m_showColumnsAsm, 
						m_columnsOrderSource, m_showColumnsSource);
				if (loadedProperties) {
					if (m_coverageType == SourceCoverageType.ASM_COVERAGE) {
						columnsOrder = m_columnsOrderAsm.clone();
						showColumns = m_showColumnsAsm.clone();
					} else {
						columnsOrder = m_columnsOrderSource.clone();
						showColumns = m_showColumnsSource.clone();
					}
				}
			} else {
				columnsOrder = new int[columnsCount];
				showColumns = new boolean[columnsCount];
				loadedProperties = loadPropertiesTable(propertiesFile, columnsOrder, showColumns);
			}
		} else { // Set to default ASM coverage
			if (m_columnsOrderAsm != null && m_showColumnsAsm != null) {
				columnsOrder = m_columnsOrderAsm.clone();
				showColumns = m_showColumnsAsm.clone();
			}
		}
		
		if (columnsOrder != null) {
			m_summary.setColumnOrder(columnsOrder);
		}
		for (int i = 0; i < columnsCount; i++) {
			// by default show column
			if ((showColumns != null) && (showColumns[i] == false)) {
				m_summary.getColumn(i).setWidth(0);
			} else {
				m_summary.getColumn(i).setWidth(summaryTableColumns[i].getWidth());
			}
		}

		return m_composite;
	}
	
	@Override
	protected ConfigureSummaryAction initConfigureSummaryAction() {
		return new ConfigureSummaryAction(null, null);
	}
	
	public void selectFunction(ItemPosition item) {
		if (m_summaryData == null)
			return;

		Vector<Long> indexes = new Vector<Long>();
		indexes.add(item.getTable());

		item.setTable(0L); /* Select item from main table */
		setSummarySelection(item);

		updateDetailsTable(indexes, true);
	}
	
	@Override
	protected void initDetailsViewer(Composite parent) {
		m_details = new Tree(parent, SWT.SINGLE | SWT.FULL_SELECTION | SWT.BORDER);
		
		m_details.setLayout(new GridLayout());
		m_details.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
	}
	
	/**
	 * @return the current state of the m_showCode option
	 * @NOTE this method should be kept package-private.
	 */
	SHOW_CODE_TYPE getShowCode() {
		return m_showCode;
	}

	/**
	 * @param m_showCode
	 *            the new state to set for m_showCode
	 * @NOTE this method should be kept package-private.
	 */
	public void setShowCode(SHOW_CODE_TYPE m_showCode) {
		this.m_showCode = m_showCode;
	}
	
	@Override
	protected FlatProfilerColumn[] getDetailsConfiguration() {
		if (m_showCode == SHOW_CODE_TYPE.MIXED) {
			return getTableConfiguration(new String[] {
					UIConstants.CC_SOURCE_LINE,
					UIConstants.CC_ASSEMBLY_LINE });
		} else if (m_showCode == SHOW_CODE_TYPE.SOURCE) {
			return getTableConfiguration(new String[] { UIConstants.CC_SOURCE_LINE });
		} else if (m_showCode == SHOW_CODE_TYPE.ASSEMBLY) {
			return getTableConfiguration(new String[] { UIConstants.CC_ASSEMBLY_LINE });
		}
		
		return null;
	}
	
	@Override
	public void createDetailsTable(Composite parent) {
		super.createDetailsTable(parent);
		
		if (m_details == null) {
			return;
		}
		// Details table columns
		FlatProfilerColumn[] detailsTableColumns = getDetailsConfiguration();
		for (int i = 0; i < detailsTableColumns.length; i++) {
			TreeColumn column = m_details.getColumn(i);
			String columnName = detailsTableColumns[i].getName();
			detailsColsWidth[i] = detailsTableColumns[i].getWidth();
			
			// Columns that are not viewed in details table.
			detailsColsToShow[i] = false;
			// Which columns don't appear in configure table for details tree.
			detailsExcludedColumns.add(columnName);
			
			column.setData(detailsTableColumns[i]);
			column.setText(detailsTableColumns[i].getName());
			if (!detailsTableColumns[i].getToolTip().isEmpty()) {
				column.setToolTipText(detailsTableColumns[i].getToolTip());
			}
			
			column.setWidth(detailsTableColumns[i].getWidth());
			column.setResizable(true);

			column.setMoveable(true);
		}
		
		privateRegisterDetailsColumnListeners();
	}
	
	@Override
	protected void registerDetailsColumnsListeners() {
	}
	
	private void privateRegisterDetailsColumnListeners() {
		// Details table columns
		FlatProfilerColumn[] detailsTableColumns = getDetailsConfiguration();
		
		Listener columnListener = new Listener() {
			@Override
			public void handleEvent(Event event) {
				int[] m_columnOrder = new int[m_details.getColumnCount()];
				boolean[] m_showColumns = new boolean[m_details.getColumnCount()];

				for (int i = 0; i < m_columnOrder.length; i++) {
					m_columnOrder[i] = m_details.getColumnOrder()[i];
					m_showColumns[i] = m_details.getColumn(i).getWidth() > 0;
				}

				StringBuilder columnOrder = new StringBuilder();
				StringBuilder showColumns = new StringBuilder();

				columnOrder.append(m_columnOrder[0]);
				showColumns.append(m_showColumns[0]);

				for (int i = 1; i < m_details.getColumnCount(); i++) {
					columnOrder.append(UIConstants.PROPERTY_ARRAY_SEPARATOR + m_columnOrder[i]);
					showColumns.append(UIConstants.PROPERTY_ARRAY_SEPARATOR + m_showColumns[i]);
				}

				Properties properties = new Properties();
				FileOutputStream file = null;
				try {
					properties.setProperty(UIConstants.COLUMNS_ORDER_PROPERTY, columnOrder.toString());
					properties.setProperty(UIConstants.SHOW_COLUMNS_PROPERTY, showColumns.toString());

					file = new FileOutputStream(getDetailsPropertiesFile());
					properties.store(file, null);
				} catch (IOException e) {
				    LOGGER.error(e);
                } finally {
                    if (file != null) {
                        try {
                            file.close();
                        } catch (IOException e) {
                            LOGGER.error(e);
                        }
                    }
                }
			}
		};
		
		for (int i = 0; i < detailsTableColumns.length; i++) {
			TreeColumn column = m_details.getColumn(i);
		
			column.addListener(SWT.Move, columnListener);
			column.addListener(SWT.Resize, columnListener);	
		}
	}
	
	/**
	 * Selects an element from sumaryTable indicated by the given position. It
	 * will select the nearest parent (Section. subsection, table) if the
	 * element is not found.
	 * 
	 * @param position
	 *            The element's position
	 */
	private void setSummarySelection(ItemPosition position) {
		for (TreeItem subsectionItem : m_summary.getItems()) {
			if (!ItemPosition.getItemPosition(subsectionItem).getSubsection().equals(position.getSubsection()))
				continue;

			for (TreeItem rowItem : subsectionItem.getItems()) {

				if (!ItemPosition.getItemPosition(rowItem).equals(position))
					continue;

				m_summary.setSelection(rowItem);
				return;
			}

			m_summary.setSelection(subsectionItem);
		}
	}

	@Override
	public String getDisplayId() {
		return "Code Coverage Tab";
	}

	

	@Override
	protected void loadPropFile(File file) {

	}

	@Override
	protected String getDetailsPropertiesFile() {
		IPath fileLocation = new Path(getFileName()).removeLastSegments(1);
		return (fileLocation.append(UIConstants.DETAILES_CRITICAL_CODE_PROPERTIES_NAME)).toString();
	}

	@Override
	public String getSummaryPropertiesFile() {
		IPath fileLocation = new Path(getFileName()).removeLastSegments(1);
		
		return (fileLocation.append(UIConstants.SUMMARY_CRITICAL_CODE_PROPERTIES_NAME)).toString();

	}
	
	protected void updateDetailsTable(Vector<Long> summaryLines, boolean updateHistory) {
		super.updateDetailsTable(summaryLines, updateHistory);	
	}
	
	@Override
	protected void privateUpdateDetailsTable(Vector<Long> summaryLines, boolean updateHistory) {
		if (m_sources == null) {
			return;
		}

		// Read the source files
		for (String sourceFile : m_sources.keySet()) {
			TreeMap<Integer, String> source = m_sources.get(sourceFile);
			//Disabled for layerscape: sourceFile = getArchivedFile(sourceFile);
			String mappedSourceFile;
			mappedSourceFile = sourceFile; // When supported find the mapped source file here
			File file = new File(mappedSourceFile);
			if (file == null || !file.isFile()) {
				Integer line = source.ceilingKey(0);
				while (line != null) {
					source.remove(line);
					source.put(line, Utils.simplifiedFileName(mappedSourceFile, true));
					line = source.ceilingKey(line + 1);
				}
				continue;
			}
			
			Integer line = source.ceilingKey(0);
			DataInputStream in = null;
			BufferedReader br = null;
			FileInputStream fstream = null;
			try {
				fstream = new FileInputStream(mappedSourceFile);
				in = new DataInputStream(fstream);
				InputStreamReader isr = new InputStreamReader(in);
				
				br = new BufferedReader(isr);
				String strLine = mappedSourceFile;
				int index = 1;

				while (line != null) {
					while ((strLine = br.readLine()) != null && index < line) {
						index++;
					}

					source.remove(line);
					if (strLine != null) {
						strLine = strLine.replaceAll("\t", "    "); //$NON-NLS-1$ //$NON-NLS-2$
						source.put(line, strLine);
					}

					line = source.ceilingKey(line + 1);
					index++;
				}
			} catch (Exception e) {
				LOGGER.error(e);
			} finally {
				if (fstream != null) {
					try {
						fstream.close();
					} catch (IOException e) {
						LOGGER.error(e);
					}
				}
				if (br != null) {
					try {
						br.close();
					} catch (IOException e) {
						LOGGER.error(e);
					}
				}
				if (in != null) {
					try {
						in.close();
					} catch (IOException e) {
						LOGGER.error(e);
					}
				}
			}
		}

		// If the table contains references to source files, resolve them
		// Read the source file contents
		for (int i = 0; i < m_detailsData.records.size(); i++) {

				String lineName = m_detailsData.records.elementAt(i).recordConfigurationName;
				TableRecordConfiguration config = fileInput.tableRecordsConfigurations.get(lineName);

				int fileIdx = -1;
				for (int j = 0; j < config.cells.size(); j++) {
					if (config.cells.elementAt(j).sourceFile) {
						fileIdx = j;
						break;
					}
				}

				if (fileIdx >= 0) {
					int line = m_detailsData.records.elementAt(i).fileLine;
					String file = m_detailsData.records.elementAt(i).fileName;

					if (line >= 0 && file != null && !file.isEmpty()) {
						m_detailsData.records.elementAt(i).cells.elementAt(fileIdx).readSourceFile = m_sources.get(file).get(line);
					}

			}
		}
		
		FlatProfilerTableCell[][] cells = null;
		
		cells = new FlatProfilerTableCell[getDetailsTableItemsCount()][];
		for (int lineIdx = 0; lineIdx < getDetailsTableItemsCount(); lineIdx++) {
			cells[lineIdx] = new FlatProfilerTableCell[m_details.getColumnCount()];
			for (int columnIdx = 0; columnIdx < m_details.getColumnCount(); columnIdx++) {
				cells[lineIdx][columnIdx] = getDetailsTableItem(lineIdx, columnIdx);
				if (cells[lineIdx][columnIdx].hasGraphics) {
					String name = NLSUtils.getTranslatedString(cells[lineIdx][columnIdx].lineType) + " > " //$NON-NLS-1$
							+ NLSUtils.getTranslatedString(cells[lineIdx][columnIdx].columnType);
					if (m_graphics.containsKey(name)) {
						m_graphics.get(name).maximumValue = Math.max(m_graphics.get(name).maximumValue, cells[lineIdx][columnIdx].graphicsValue);
					}
				}
			}
		}
		
		if (m_showCode == SHOW_CODE_TYPE.MIXED) { }
		else if (m_showCode == SHOW_CODE_TYPE.ASSEMBLY) {
			Vector<FlatProfilerTableCell[]> assemblyCells = new Vector<FlatProfilerTableCell[]>();

			for (int lineIdx = 0; lineIdx < getDetailsTableItemsCount(); lineIdx++) {
				if (m_detailsData.records.elementAt(lineIdx).recordConfigurationName.compareTo(UIConstants.CC_ASSEMBLY_LINE) == 0) {
					assemblyCells.add(cells[lineIdx]);
				}
			}
			cells = new FlatProfilerTableCell[assemblyCells.size()][];
			for (int lineIdx = 0; lineIdx < assemblyCells.size(); lineIdx++) {
				cells[lineIdx] = new FlatProfilerTableCell[m_details.getColumnCount()];
				for (int columnIdx = 0; columnIdx < m_details.getColumnCount(); columnIdx++) {
					cells[lineIdx][columnIdx] = assemblyCells.elementAt(lineIdx)[columnIdx];
				}
			}
		} else if (m_showCode == SHOW_CODE_TYPE.SOURCE) {
			Vector<FlatProfilerTableCell[]> sourceCells = new Vector<FlatProfilerTableCell[]>();

			for (String fileName : m_sources.keySet()) {
				for (Integer fileLine : m_sources.get(fileName).keySet()) {
					for (int i = 0; i < m_detailsData.records.size(); i++) {
						if (m_detailsData.records.elementAt(i).fileName.equals(fileName) 
								&& m_detailsData.records.elementAt(i).fileLine == fileLine) {
							sourceCells.add(cells[i]);
							break;
						}
					}
				}
			}

			cells = new FlatProfilerTableCell[sourceCells.size()][];
			for (int lineIdx = 0; lineIdx < sourceCells.size(); lineIdx++) {
				cells[lineIdx] = new FlatProfilerTableCell[m_details.getColumnCount()];
				for (int columnIdx = 0; columnIdx < m_details.getColumnCount(); columnIdx++) {
					cells[lineIdx][columnIdx] = sourceCells.elementAt(lineIdx)[columnIdx];
				}
			}

		}

		m_detailsViewer.setInput(cells);
	}
	
	
	@Override
	protected void setSummaryTableItem(TableRecordData record) {
		super.setSummaryTableItem(record);
	}
	
	/**
	 * Set up the fields for the given coverage table cell.
	 * 
	 * @param covCell 
	 * 					A table cell that contains coverage information.
	 * @param lineNum 
	 * 					The line number in the table.
	 * @return The coverage cell with updated fields.
	 */
	FlatProfilerTableCell getDetailsCoverageCell(FlatProfilerTableCell covCell) {
		Image imgCovered = PlatformUI.getWorkbench().getSharedImages().getImage(
				org.eclipse.ui.ide.IDE.SharedImages.IMG_OBJS_TASK_TSK);
		Image imgNotCovered = PlatformUI.getWorkbench().getSharedImages().getImage(
				org.eclipse.ui.ISharedImages.IMG_OBJS_ERROR_TSK);
		Image imgPartiallyCovered = PlatformUI.getWorkbench().getSharedImages().getImage(
				org.eclipse.ui.ISharedImages.IMG_OBJS_WARN_TSK);

		if (covCell.text.isEmpty()) {
			return covCell;
		}

		// Coverage is expressed in strings: covered/not covered/partially covered.
		if (covCell.text.equals(UIMessages.CriticalCode_Covered_Text)) {
			covCell.image = imgCovered;
		} else if (covCell.text.equals(UIMessages.CriticalCode_PartiallyCovered_Text)) {
			covCell.image = imgPartiallyCovered;
		} else {
			covCell.image = imgNotCovered;
		}

		return covCell;
	}

	/**
	 * Set up the fields for the given asm decision coverage table cell.
	 * 
	 * @param covCell
	 *            A table cell that contains asm decision 
	 *            coverage information.
	 * @return The coverage cell with updated fields.
	 */
	FlatProfilerTableCell getAsmDecisionCoverageCell(
			FlatProfilerTableCell covCell) {

		if (covCell.text.isEmpty()) {
			return covCell;
		}

		int asmDecisionCovType = Integer.parseInt(covCell.text);

		Image imgTaken = PlatformUI.getWorkbench().getSharedImages().getImage(
				org.eclipse.ui.ISharedImages.IMG_OBJS_WARN_TSK);
		Image imgNotTaken = PlatformUI.getWorkbench().getSharedImages().getImage(
				org.eclipse.ui.ISharedImages.IMG_OBJS_WARN_TSK);
		Image imgTakenAndNotTaken = PlatformUI.getWorkbench().getSharedImages().getImage(
				org.eclipse.ui.ide.IDE.SharedImages.IMG_OBJS_TASK_TSK);

		switch (asmDecisionCovType) {
		case BRANCH_ALWAYS_TAKEN:
			covCell.text = UIMessages.CriticalCode_Branch_Taken;
			covCell.image = imgTaken;
			break;

		case BRANCH_NEVER_TAKEN:
			covCell.text = UIMessages.CriticalCode_Branch_Not_Taken;
			covCell.image = imgNotTaken;
			break;

		case BRANCH_TAKEN_AND_NOT_TAKEN:
			covCell.text = UIMessages.CriticalCode_Branch_Taken_And_Not_Taken;
			covCell.image = imgTakenAndNotTaken;
			break;

		case IS_NOT_CONDITIONAL_BRANCH:
			covCell.text = ""; //$NON-NLS-1$
			break;

		default:
			break;
		}
		return covCell;
	}

	@Override
	protected FlatProfilerTableCell getTableItem(Tree tree,
			TableData tableData, int lineNum, int columnNum) {
		FlatProfilerTableCell ret = super.getTableItem(tree, tableData, lineNum, columnNum);
		
		TableRecordData recordData = tableData.records.elementAt(lineNum);
		String lineName = recordData.recordConfigurationName;
		TableRecordConfiguration config = fileInput.tableRecordsConfigurations.get(lineName);
		TableCellConfiguration cell = config.cells.get(columnNum);
		
		if (cell.coverage) {
			ret = getDetailsCoverageCell(ret);
		} else if (cell.decisionCoverage) {
			ret = getAsmDecisionCoverageCell(ret);
		}
		
		return ret;
	}
	
	protected boolean isTimeColumn(String columnName) {
		return columnName.equals(UIMessages.CriticalCode_Time_Column);
	}
	
	protected void paintItem(Tree tree, Event event) {
		FlatProfilerTableCell cell = ((FlatProfilerTableCell[]) ((TreeItem) event.item).getData())[event.index];
		// dark_blue is send from CC to mark inline cell
		if (cell.background.equals("dark_blue")) { //$NON-NLS-1$
			TreeItem treeItm = ((TreeItem) (event.item));
			treeItm.setBackground(ResourceManager.getColor(200, 240, 255));
		}
		
		if (!cell.hasGraphics) {
			return;
		}
		String name = cell.lineType + " > " + cell.columnType; //$NON-NLS-1$
		if (m_graphics.containsKey(name) && m_graphics.get(name).show == true) {

			GC gc = event.gc;

			Color oldBackground = gc.getBackground();
			Color oldForeground = gc.getForeground();

			gc.setBackground(m_graphics.get(name).color);
			gc.fillRectangle(event.x, event.y + 2,
					(int) ((cell.graphicsValue / m_graphics.get(name).maximumValue) * (double) (event.width)),
					event.height - 4);
			
			gc.setBackground(oldBackground);
			gc.setForeground(oldForeground);

			FontMetrics fontMetrics = gc.getFontMetrics();
			int stringHeight = fontMetrics.getHeight();
			int x = ((TreeItem) (event.item)).getBounds(event.index).x + 5;
			int y = ((TreeItem) (event.item)).getBounds(event.index).y
					+ ((TreeItem) (event.item)).getBounds(event.index).height / 2;

			NumberFormat nf = NumberFormat.getInstance();
			DecimalFormat decimalFormat = new DecimalFormat("##.##");
			decimalFormat.setGroupingUsed(true);
			decimalFormat.setGroupingSize(3);

			try {
                gc.drawText(decimalFormat.format(nf.parse(cell.text).doubleValue()), x, y - stringHeight / 2 - 1, true);
            } catch (ParseException e) {}

			gc.setBackground(oldBackground);
			gc.setForeground(oldForeground);
			event.doit = false;
		}
	}

	@Override
	protected String getInstructionAddress(TableRecordData recordData) {
	    return recordData.cells.firstElement().text;
	}
}
