/*******************************************************************************
 * Copyright (c) 2015 - 2016 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.traceresults.views;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.IFontProvider;
import org.eclipse.jface.viewers.ITableColorProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Display;

import com.freescale.sa.ui.common.SAUILSPlugin;
import com.freescale.sa.ui.common.UIConstants;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableRecordConfiguration;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableRecordData;

/**
 * Decoration class for columns from Analysis Results view table 
 * @author B46903
 *
 */
public class TraceResultsViewColumnLabelProvider extends CellLabelProvider implements ITableLabelProvider, ITableColorProvider, IFontProvider {
	private FlatDataProvider dataProvider = null;

	private static final Color linkForeground = new Color(null, 0, 51, 153);
	private static Font m_boldFont = null;
	private static Font m_italicFont = null;
	private static Color m_lightBlueColor = null;
	private Integer columnIndex;
	
	static {
		Display display = Display.getCurrent();
		Font defaultFont = display.getSystemFont();
		m_lightBlueColor = new Color(Display.getCurrent(), 230, 230, 255);

		FontData boldFontData = defaultFont.getFontData().clone()[0];
		
		boldFontData.setStyle(SWT.BOLD);
		m_boldFont = new Font(Display.getCurrent(), boldFontData);
		
		FontData italicFontData = defaultFont.getFontData().clone()[0];
		italicFontData.setStyle(SWT.ITALIC);
		m_italicFont = new Font(Display.getCurrent(), italicFontData);
	}

	// icon cache
	private static Map<ImageDescriptor, Image> m_imageCache = new HashMap<ImageDescriptor, Image>(11);
	
	public TraceResultsViewColumnLabelProvider (FlatDataProvider dataProvider) {
		this.dataProvider = dataProvider;
	}
	
	@Override
	public void dispose() {
		super.dispose();
		for (Image image : m_imageCache.values()) {
			image.dispose();
		}
		m_imageCache.clear();
	}

	@Override
	public Color getForeground(Object element, int columnIndex) {
		if (!(element instanceof TableRecordData)) {
			return null;
		}

		TableRecordData recordData = (TableRecordData)element;
		if (TraceResultsFormDrawer.getCellType(recordData, columnIndex) == FlatDataProvider.LINK_CELL) {
			return linkForeground;			
		}
		
		return null;
	}

	@Override
	public Color getBackground(Object element, int columnIndex) {
		if (!(element instanceof TableRecordData)) {
			return null;
		}

		TableRecordData line = (TableRecordData) element;

		if (line.background.equals("light_blue")) { //$NON-NLS-1$
			return m_lightBlueColor;
		} else {
			return null;
		}
	}

	@Override
	public Image getColumnImage(Object element, int columnIndex) {
		if (!(element instanceof TableRecordData)) {
			return null;
		}

		TableRecordData data = (TableRecordData) element;
		if (columnIndex >= data.cells.size()) {
			return null;
		}
		return getColumnImage(data, columnIndex);
	}
	
	private Image getColumnImage(TableRecordData line, int index) {
		ImageDescriptor descriptor = null;
		String iconPath = null;

		if (line.cells.size() <= index) {
			return null;
		}
		if (line.cells.elementAt(index).value.isEmpty()) {
			return null;
		}

		// add 'datafile' icon for lines which are not roots
		if (line.parentIndex != -1) {
			String cellName = line.cells.elementAt(index).value;
			boolean latest = line.cells.elementAt(index).isLastModified();
			boolean projectLatest = line.cells.elementAt(index).isProjectLastModified();

			if (!line.isTraceConfig) {
				if (index == 0) {
					if (latest) {
						iconPath = UIConstants.analysis_latest_datafile_iconfile;
					} else {
						if (projectLatest) {
							iconPath = UIConstants.analysis_project_latest_datafile_iconfile;
						} else {
							iconPath = UIConstants.analysis_datafile_iconfile;
						}
					}
				} else if (cellName.equals(Messages.ProfileResultsInput_Trace)) {
					iconPath = UIConstants.analysis_trace_iconfile;
				} else if (cellName
						.equals(Messages.ProfileResultsInput_Timeline)) {
					iconPath = UIConstants.results_timeline_iconfile;
				} else if (cellName
						.equals(Messages.ProfileResultsInput_CriticalCode)) {
					iconPath = UIConstants.results_flatdata_iconfile;
				} else if (cellName
						.equals(Messages.ProfileResultsInput_Statistical)) {
					iconPath = UIConstants.results_flatdata_iconfile;
				} else if (cellName
						.equals(Messages.ProfileResultsInput_Performance)) {
					iconPath = UIConstants.analysis_performance_iconfile;
				} else if (cellName
						.equals(Messages.ProfileResultsInput_DataVis)) {
					iconPath = UIConstants.results_datavisualization_iconfile;
				} else if (cellName
						.equals(Messages.ProfileResultsInput_CallTree)) {
					iconPath = UIConstants.analysis_calltree_iconfile;
				} else if (cellName
						.equals(Messages.ProfileResultsInput_Counterpoints)) {
					iconPath = UIConstants.results_counterpoints_iconfile;
				} else if (cellName
						.equals(Messages.ProfileResultsInput_ConfigureResults)) {
					iconPath = UIConstants.results_configresults_iconfile;
				} 
				
			} else {
				if (index == 0) {
					iconPath = UIConstants.results_traceconfig_iconfile;
				}
			}
		} else {
			if (index == 0) {
				iconPath = UIConstants.analysis_project_iconfile;
			}
		}

		if (iconPath != null) {
			descriptor = SAUILSPlugin.getImageDescriptor(iconPath);
		}

		// obtain the cached image corresponding to the descriptor
		if (descriptor != null) {
			Image image = m_imageCache.get(descriptor);

			if (image == null) {
				image = descriptor.createImage();
				m_imageCache.put(descriptor, image);
			}
			return image;
		} else {
			return null;
		}
	}

	@Override
	public String getColumnText(Object element, int columnIndex) {
		if (!(element instanceof TableRecordData)) {
			return null;
		}
		TableRecordData line = (TableRecordData) element;

		if (line.cells.size() > columnIndex) {
			StringBuilder ident = new StringBuilder();

			TableRecordConfiguration trc = dataProvider.tableRecordsConfigurations.get(line.recordConfigurationName);

			if (trc == null)
				return null;

			for (int i = 0; i < dataProvider.tableRecordsConfigurations
					.get(line.recordConfigurationName).cells
					.elementAt(columnIndex).ident; i++) {
				ident.append(" "); //$NON-NLS-1$
			}

			return ident + line.cells.elementAt(columnIndex).value;
		} else {
			return null;
		}
	}

	@Override
	public void update(ViewerCell cell) {
		Integer index = cell.getColumnIndex();
		Object obj = cell.getItem().getData();
		columnIndex = index;

		cell.setText(getColumnText(obj, index));
		cell.setBackground(getBackground(obj, index));
		cell.setForeground(getForeground(obj, index));
		cell.setImage(getColumnImage(obj, index));
		cell.setFont(getFont(obj));	
	}

	@Override
	public Font getFont(Object element) {
		if (!(element instanceof TableRecordData)) {
			return null;
		}

		TableRecordData line = (TableRecordData) element;
		final TableRecordConfiguration tableRecordConfiguration = dataProvider.tableRecordsConfigurations
				.get(line.recordConfigurationName);
		
		if (tableRecordConfiguration == null)
			return null;

		if (tableRecordConfiguration.boldFont) {
			return m_boldFont;
		}
		if (tableRecordConfiguration.italicFont) {
			return m_italicFont;
		}
		return null;
	}

	@Override
	public String getToolTipText(Object element) {
		if (!(element instanceof TableRecordData)) {
			return null;
		}

		TableRecordData line = (TableRecordData) element;

		if (line.storageCfgFile == null)
			return null;

		if (TraceResultsFormDrawer.getCellType(line, columnIndex) != FlatDataProvider.LINK_CELL) {
			if (line.storageCfgFile != null) {
				return line.storageCfgFile.getAbsolutePath();
			}		
		}

		return null;
	}

}
