/*******************************************************************************
 * Copyright (c) 2014 - 2016 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.traceresults.views;


import java.io.File;
import java.io.FileFilter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.Vector;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.forms.widgets.Form;
import org.eclipse.ui.progress.UIJob;

import com.freescale.sa.SaConstants;
import com.freescale.sa.configurator.config.ConfigurationUtils;
import com.freescale.sa.configurator.ls.jni.AttributeType;
import com.freescale.sa.configurator.ls.jni.LSConfigLoader_JNI;
import com.freescale.sa.model.AnalysisManager;
import com.freescale.sa.model.IAnalyzer;
import com.freescale.sa.model.IAnalyzer.Feature;
import com.freescale.sa.ui.common.SAUIUtils;
import com.freescale.sa.ui.common.UIConstants;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider;
import com.freescale.sa.ui.common.utils.SaUiLsUtils;
import com.freescale.sa.ui.common.utils.StorageUtils;
import com.freescale.sa.ui.common.utils.StorageUtils.ResourceFolder;
import com.freescale.sa.ui.common.utils.StorageUtils.ResourceFolders;
import com.freescale.sa.ui.traceresults.utils.ResultsUtils;
import com.freescale.sa.util.CommonConstants;
import com.freescale.sa.util.CustomFileFilter;
import com.freescale.sa.util.Utils;


public class TraceResultsInput extends FlatDataProvider {
	
	static private class PlatformConfigEntry {
		String name;
		Map<String, DataStreamEntry> streams;
		
		public PlatformConfigEntry(String name, Map<String, DataStreamEntry> streams) {
			this.name = name;
			this.streams = streams;
		}
	}
	
	static private class DataStreamEntry {
		String name;
		Map<String, TraceDataEntry> traces;
		
		public DataStreamEntry(String name, Map<String, TraceDataEntry> traces) {
			this.name = name;
			this.traces = traces;
		}
	}
	
	static private class TraceDataEntry implements Comparable<TraceDataEntry>{
		public File resultsConfig;
		boolean raw;
		boolean latest;
		boolean projectLatest;
		String project = null;
			
		public TraceDataEntry(File resultsConfig, String arch, boolean raw) {
			//this.platformConfig = platformConfig;
			this.resultsConfig = resultsConfig;
			this.raw = raw;
			this.latest = false;
			this.projectLatest = false;
		}
		
		public Date getLastModified() {
			return new Date(resultsConfig.lastModified());
		}

		@Override
		public int compareTo(TraceDataEntry entry) {
			if (resultsConfig == null) {
				return -1;
			}
			
			if (entry == null) {
				return -1;
			}
			
			if (entry.resultsConfig == null) {
				return -1;
			}
			
			if (resultsConfig == null) {
				return -1;
			}
			
            if (entry.resultsConfig.getAbsolutePath() == null
                || resultsConfig.getAbsolutePath() == null) {
				return -1;
			}

			return entry.resultsConfig.getAbsolutePath().compareTo(resultsConfig.getAbsolutePath());
		}
	}
	
	private final static String emptyString = CommonConstants.EMPTY_STRING;
	
	private final static String openTrace = Messages.ProfileResultsInput_Trace;
	
	private final static String openTimeline = Messages.ProfileResultsInput_Timeline;
	private final static String openPerformance = Messages.ProfileResultsInput_Performance;
	private final static String openCallTree = Messages.ProfileResultsInput_CallTree;
	private final static String openFlatProfile = Messages.ProfileResultsInput_CriticalCode;
	private final static String openStatistical = Messages.ProfileResultsInput_Statistical;
	
	private final static String openCounterpoints = Messages.ProfileResultsInput_Counterpoints;
	private final static String openConfigureResults = Messages.ProfileResultsInput_ConfigureResults;
	
	
	private final static String linuxTag = "linux"; //$NON-NLS-1$

	public static final String importDataFromTxtFileAction = TraceResultsConstants.IMPORT_DATA_FROM_TXT_FILE_ACTION_NAME;
	
	private Action m_importDataFromTxtFileAction;
	private Action m_selectOutputFoler;
	
	private TableData m_profileResults;
	private boolean m_editorClosed;
	
	private int currentDisplayIndex;
		
	private final static Logger LOGGER = Logger.getLogger(TraceResultsInput.class);
	
	private boolean showTraceColumn;
	private boolean showTimelineColumn;
	private boolean showCriticalCodeColumn;
	private boolean showPerformanceColumn;
	private boolean showCallTreeColumn;
	private boolean showDataTraceImport;
	private boolean showCounterpointsColumn;
	private boolean showCommentsColumn;
	private boolean showConfigureResultsColumn;
	public String projectName;
	
	Set<TraceDataEntry> traceDataEntries = new TreeSet<TraceDataEntry>();
	
	public TraceResultsInput() {
		super();
        // Data trace import action is available if at least one architecture
        // from analyzers
		// supports data trace import feature.
		List<IAnalyzer> analyzers = AnalysisManager.getDefault().getAllAnalyzers();
		if (analyzers.size() > 0) {
			for (IAnalyzer analyzer : analyzers) {
                showDataTraceImport = hasFeature(analyzer, showDataTraceImport,
                                                 Feature.DATA_TRACE_IMPORT);
			}
		}

		// Start a thread to reload the view when results appear or are deleted
		listenerList = new javax.swing.event.EventListenerList();
		
		m_profileResults = new TableData(0);
	}
	
	@Override
	public String getName() {
		return Messages.ProfileResultsInput_3;
	}

	@Override
	public void setupCustomMenuActions(Form form, MenuManager menuManager) {
		if (showDataTraceImport) {
			form.getToolBarManager().add(m_importDataFromTxtFileAction);
			menuManager.add(m_importDataFromTxtFileAction);
		}
		
		m_selectOutputFoler = new Action() {
			public void run() {
                SelectOutputFolderDialog dl = new SelectOutputFolderDialog(Display.getDefault()
                                                                                  .getActiveShell());
				dl.open();
				getDrawer().refresh();
			}
		};
		m_selectOutputFoler.setText(Messages.FlatDataFormDrawer_57);
		m_selectOutputFoler.setToolTipText(Messages.FlatDataFormDrawer_58);
		
        m_selectOutputFoler.setImageDescriptor(ImageDescriptor.createFromImage(PlatformUI.getWorkbench()
                                                                                         .getSharedImages()
                                                                                         .getImage(ISharedImages.IMG_OBJ_FOLDER)));
		
		form.getToolBarManager().add(m_selectOutputFoler);
		menuManager.add(m_selectOutputFoler);
		
	}
		
	@Override
	public void executeAction(String cellValue, String action, String projectName) {
        SaUiLsUtils.openEditor(cellValue, action, projectName);
								for(ChangeListener listener : listenerList.getListeners(ChangeListener.class)) {
									ChangeEvent e = new ChangeEvent(this);
									listener.stateChanged(e);
								}
							}
		
	/**
     * Set the information for a table row corresponding to a platform
     * configuration.
	 * 
	 * @param platformRecord
	 *            The table record data (this object corresponds to a row in the
	 *            table).
	 * @param name
	 * 			  The name of the platform configuration.
	 */
	private void putPlatformConfigLine(TableRecordData platformRecord, String name) {
		platformRecord.cells.add(new TableCellData(name, -1)); // / project
		// name
		if (showTraceColumn) {
            platformRecord.cells.add(new TableCellData("", -1)); // / //$NON-NLS-1$
                                                                 // no trace for platform entries
		}
		if (showTimelineColumn) {
            platformRecord.cells.add(new TableCellData("", -1)); // / //$NON-NLS-1$
                                                                 // no timeline for platform entries
		}
		if (showCriticalCodeColumn) {
            platformRecord.cells.add(new TableCellData("", -1)); // / //$NON-NLS-1$
                                                                 // no critical code for platform entries
		}
		if (showPerformanceColumn) {
            platformRecord.cells.add(new TableCellData("", -1)); // / //$NON-NLS-1$
                                                                 // no performance for platform entries
		}
		if (showCallTreeColumn) {
            platformRecord.cells.add(new TableCellData("", -1)); // / //$NON-NLS-1$
                                                                 // no call tree for platform entries
		}	
		
		
		if (showCounterpointsColumn) {
            platformRecord.cells.add(new TableCellData("", name, -1)); // / //$NON-NLS-1$
                                                                       // no CP results for platform entries
		}
		if (showConfigureResultsColumn) {
			platformRecord.cells.add(new TableCellData("", name, -1)); //$NON-NLS-1$
		}
	}
	
	@Override
	public synchronized TableData getTableData(long id) {
		m_profileResults = new TableData(0, 2);
		
		try {
			SAUIUtils.refreshWorkspace(IResource.DEPTH_INFINITE, new NullProgressMonitor());
			
			traceDataEntries.clear();
			currentDisplayIndex = 0;
			
			Map<String, PlatformConfigEntry> platforms = new HashMap<String, PlatformConfigEntry>();
			File[] platformCfgs = loadDefaultPlatformConfigFiles();
			String resultsPath = null;
			
			if (platformCfgs != null) {
				/*
                 * Go through all the platform configuration files found into
                 * the default location and get their results output path. Then
                 * go through each of those paths and load whatever trace data
                 * lies in there.
				 */
				for (File platformCfg : platformCfgs) {
					LSConfigLoader_JNI platformLoader = new LSConfigLoader_JNI(platformCfg.getAbsolutePath());
                    resultsPath = platformLoader.getValue(UIConstants.PLATFORM_RESULTS + ":" //$NON-NLS-1$
                                                          + UIConstants.PLATFORM_OUTPUT_FOLDER,
                                                          AttributeType.STRING_ATTR.value());
					
					if (resultsPath == null || resultsPath.isEmpty()) {
                        resultsPath = ResourcesPlugin.getWorkspace().getRoot().getLocation()
                                                     .toOSString()
                                      + new Path(SaConstants.analysis_results_directory).toOSString();
					}
	
					loadStorageConfigFiles(resultsPath);
				}
			} else {
				//If there's is no platform config file, add the default folder
                resultsPath = ResourcesPlugin.getWorkspace().getRoot().getLocation().toString()
                              + SaConstants.analysis_results_directory;
				loadStorageConfigFiles(resultsPath);
			}

			// Read custom folders
			ResourceFolders folders = StorageUtils.getCustomFolders();
			for (ResourceFolder folder : folders.getFolders()) {
                if (resultsPath != null) {
					IPath path = new Path(resultsPath);
                    if (folder.getPath() == null
                        || (!path.isEmpty() && new Path(folder.getPath()).equals(path))) {
						continue;
					}
				}
				File outputFile = new File(folder.getPath());
				if (folder.getEnabled() && outputFile.exists()) {
					loadStorageConfigFiles(outputFile.getAbsolutePath());
				}
			}
			 						
			/*
             * Go through each trace configuration data entry and check their
             * registered platform configuration and collection data stream.
             * Link every one of them into their appropriate structure.
			 */
			Date latestDate = null;
			for (TraceDataEntry entry : traceDataEntries) {
				if (latestDate == null) {
					latestDate = entry.getLastModified();
				} else {
					if (latestDate.before(entry.getLastModified())) {
						latestDate = entry.getLastModified();
					}
				}
			}
			
			Map<String, Date> latestForProject = new HashMap<String, Date>();
			
			for (TraceDataEntry entry : traceDataEntries) {
				if (latestDate != null) {
					if (latestDate.equals(entry.getLastModified())) {
						entry.latest = true;
					}
				}
				
				String platformConfigName = CommonConstants.EMPTY_STRING;
				String dataStream = CommonConstants.EMPTY_STRING;

				if (entry.resultsConfig == null) {
					continue;
				}
				
				LSConfigLoader_JNI resultsConfig = new LSConfigLoader_JNI(entry.resultsConfig.getAbsolutePath()); 
				
                platformConfigName = resultsConfig.getValue(TraceResultsConstants.RESULTS_CW_BLOCK
                                                            + ":" //$NON-NLS-1$
                                                            + TraceResultsConstants.RESULTS_PLATFORM_NAME,
                                                            AttributeType.STRING_ATTR.value());
                dataStream = resultsConfig.getValue(TraceResultsConstants.RESULTS_CW_BLOCK + ":" //$NON-NLS-1$
                                                    + TraceResultsConstants.RESULTS_DATA_STREAM,
                                                    AttributeType.ENUM_ATTR.value());
                entry.project = resultsConfig.getValue(TraceResultsConstants.RESULTS_CW_BLOCK + ":" //$NON-NLS-1$
                                                       + TraceResultsConstants.RESULTS_PROJECT_NAME,
                                                       AttributeType.STRING_ATTR.value());
				
				if (platformConfigName.toLowerCase().trim().startsWith(linuxTag)) {
					/* Generate unique names for all duplicated entries */
					Integer genIndex = 0;
					String newPlatform = platformConfigName;
					while (platforms.containsKey(newPlatform)) {
						newPlatform = platformConfigName + "-" + genIndex;//$NON-NLS-1$
						genIndex++;
					}
					platformConfigName = newPlatform;
				}
				
				Date latestProjectDate = latestForProject.get(entry.project);
				if (latestProjectDate == null) {
					latestForProject.put(entry.project, entry.getLastModified());
				} else {
					if (latestProjectDate.before(entry.getLastModified())) {
						latestForProject.put(entry.project, entry.getLastModified());
					}
				}
				
				PlatformConfigEntry platform =  platforms.get(platformConfigName);
				if (platform == null) {					
                    platform = new PlatformConfigEntry(platformConfigName,
                                                       new HashMap<String, DataStreamEntry>());
					platforms.put(platformConfigName, platform);
				}
				
				DataStreamEntry dataStreamEntry = platform.streams.get(dataStream);
				if (dataStreamEntry == null) {
                    dataStreamEntry = new DataStreamEntry(dataStream,
                                                          new HashMap<String, TraceDataEntry>());
					platform.streams.put(dataStream, dataStreamEntry);
				}
				
				dataStreamEntry.traces.put(parseNameFromEntry(entry), entry);
			}
			
			//Mark latest results for each platform
			for (PlatformConfigEntry platform : platforms.values()) {
				for (DataStreamEntry dataStream : platform.streams.values()) {
					for (TraceDataEntry entry : dataStream.traces.values()) {
						Date latestProjectDate = latestForProject.get(entry.project);
						if (entry.getLastModified().equals(latestProjectDate)) {
							entry.projectLatest = true;
						}
					}
				}
			}

			/*
			 * Now go through all of it and dump it in the SA results view.
			 */
			for (PlatformConfigEntry platform : platforms.values()) {
				TableRecordData platformRecord = new TableRecordData(currentDisplayIndex, -1,
                                                                     TraceResultsConstants.TABLE_PLATFORM_COLUMN_NAME,
                                                                     "test", false, null); //$NON-NLS-1$
				int index = currentDisplayIndex;
				putPlatformConfigLine(platformRecord, platform.name);
				currentDisplayIndex++;
				for (DataStreamEntry dataStream : platform.streams.values()) {
					addResultsForDataStream(dataStream, index);
				}
				m_profileResults.records.add(platformRecord);
			}
		} catch (CoreException e) {
			LOGGER.error(e.getMessage());
		}
		
		return m_profileResults;
	}
	
    private void filterRecursive(File rootFolder, FileFilter filter,
                                 ArrayList<File> filteredFiles) {
		if (rootFolder == null) {
			return;
		}
		
		if (!rootFolder.exists()) {
			return;
		}
		
		if (rootFolder.isFile()) {
			return;
		}
		
		//Do this only for directories, null is return when listFiles() is called on a file.
		File [] list = rootFolder.listFiles();
		if(list != null){
			for (File file : list) {
				filterRecursive(file, filter, filteredFiles);
			}
			File[] storeCfgFiles = rootFolder.listFiles(new CustomFileFilter(SaConstants.raw_trace_config_extension));
			filteredFiles.addAll(Arrays.asList(storeCfgFiles));
		}
	}
	
	/**
     * Loads the storage configuration files associated with a launch
     * configuration.
	 * 
     * The storage configuration files (*.resultsConfig) cans be stored anywhere
     * in the file system.
	 * 
	 * @param launchCfg
	 *            The launch configuration.
	 * @param projLocation 
	 * @throws CoreException
	 */
	private void loadStorageConfigFiles(String outputFolderPath) throws CoreException {
		if (outputFolderPath == null) {
			// This shouldn't happen.
			return;
		}
		
		if (outputFolderPath.isEmpty()) {
			// This shouldn't happen.
			return;
		}
		
		File outputFolder = new File(outputFolderPath);
		
		if (!outputFolder.exists()) {
			return;
		}
		
		ArrayList<File> storeCfgFiles = new ArrayList<File>();
        filterRecursive(outputFolder, new CustomFileFilter(SaConstants.raw_trace_config_extension),
                        storeCfgFiles);
		
		for (File storeCfg : storeCfgFiles) {
			String arch = CommonConstants.STORAGE_ANY_TARGET;
			String resultsPath = storeCfg.getAbsolutePath();

			File resultsConfig = new File(resultsPath);
			String[] launchAndArchId = ResultsUtils.getLaunchNameAndArchIdFromRawTraceConfig(resultsConfig.getAbsolutePath());
			if (launchAndArchId != null) {
				arch = launchAndArchId[1];
			}
			
			TraceDataEntry entry = new TraceDataEntry(resultsConfig, arch, false);
			traceDataEntries.add(entry);
		}
	}
	
	private String parseNameFromEntry(TraceDataEntry entry) {
		String fileName = entry.resultsConfig.getName();
		int indexOfDot = fileName.lastIndexOf("."); //$NON-NLS-1$
		if (indexOfDot > 0) {
			fileName = fileName.substring(0, indexOfDot);
		}
		return fileName;
	}
		
	private void addResultsForDataStream(DataStreamEntry stream, int parentIndex) {
		SimpleDateFormat dateFormat = new SimpleDateFormat(TraceResultsConstants.DEFAULT_DATE_FORMAT);
		
        TableRecordData dataStreamRecord = new TableRecordData(currentDisplayIndex, parentIndex,
                                                               true,
                                                               TraceResultsConstants.TABLE_PLATFORM_COLUMN_NAME,
                                                               emptyString, true);
		
		dataStreamRecord.cells.add(new TableCellData(stream.name, -1));
				
		m_profileResults.records.add(dataStreamRecord);
		
		for (TraceDataEntry entry : stream.traces.values()) {
			String traceConfigName = parseNameFromEntry(entry);
			addEntry(entry, traceConfigName, dataStreamRecord.index, dateFormat, null);
			currentDisplayIndex++;
		}
		
	}
	
	// Add an individual entry (trace results)
	private void addEntry(TraceDataEntry entry, String traceConfigName, int parentIndex,
			SimpleDateFormat dateFormat, ILaunchConfiguration launchCfg) {
		
		File storageCfgFile = entry.resultsConfig;	
		LSConfigLoader_JNI resultsConfig = new LSConfigLoader_JNI(entry.resultsConfig.getAbsolutePath()); 
        String projName = resultsConfig.getValue(TraceResultsConstants.RESULTS_CW_BLOCK + ":" //$NON-NLS-1$
                                                 + TraceResultsConstants.RESULTS_PROJECT_NAME,
                                                 AttributeType.STRING_ATTR.value());
		String resultName = Utils.simplifiedFileName(storageCfgFile.getName(), false);
        String comment = resultsConfig.getValue(TraceResultsConstants.RESULTS_CW_BLOCK + ":" //$NON-NLS-1$
                                                + TraceResultsConstants.RESULTS_COMMENT,
                                                AttributeType.STRING_ATTR.value());
        String dataStream = resultsConfig.getValue(TraceResultsConstants.RESULTS_CW_BLOCK + ":" //$NON-NLS-1$
                                                   + TraceResultsConstants.RESULTS_DATA_STREAM,
                                                   AttributeType.ENUM_ATTR.value());
		
		// Default all viewers are enabled.
		boolean traceEnabled = true;
        boolean criticalCodeEnabled = true;
        boolean statisticalEnabled = false;
        boolean performaceEnabled = true;
        boolean timelineEnabled = true;
        boolean callTreeEnabled = true;
        boolean taskSummaryEnabled = true;
        boolean counterpointsEnabled = true;
		
		if (dataStream.equals(TraceResultsConstants.RESULTS_DATA_STREAM_STATISTICAL)) {
		    // Only Code Coverage for Statistical profiler.
		    traceEnabled = false;
	        criticalCodeEnabled = false;
	        statisticalEnabled = true;
	        performaceEnabled = false;
	        timelineEnabled = false;
	        callTreeEnabled = false;
	        taskSummaryEnabled = false;
	        counterpointsEnabled = false;
		}
		
		TableRecordData resultRecord = new TableRecordData(-1, parentIndex,
                                                           TraceResultsConstants.TABLE_RESULT_COLUMN_NAME,
                                                           CommonConstants.EMPTY_STRING, entry.raw,
                                                           storageCfgFile);
		
		String launchCfgName = CommonConstants.EMPTY_STRING;
		if (launchCfg != null) {
			launchCfgName = launchCfg.getName();
		}
		
        addResultsCell(resultRecord, resultName, projName, launchCfgName, entry.latest,
                       entry.projectLatest);
		
		
		if (showTraceColumn) {
			TableCellData tableCellData = null;
			if (traceEnabled) {
                tableCellData = new TableCellData(openTrace, -1, storageCfgFile.getAbsolutePath(),
                                                  projectName);
			} else {
                tableCellData = new TableCellData(CommonConstants.EMPTY_STRING, -1, null,
                                                  projectName);
			}
			resultRecord.cells.add(tableCellData);
		}
		
		if (showTimelineColumn) {
		    TableCellData tableCellData = null;
		    if (timelineEnabled) {
                tableCellData = new TableCellData(openTimeline, -1,
                                                  storageCfgFile.getAbsolutePath(), projectName);
		    } else {
                tableCellData = new TableCellData(CommonConstants.EMPTY_STRING, -1, null,
                                                  projectName);
		    }
			resultRecord.cells.add(tableCellData);
		}
		
		String profilersFileDir = storageCfgFile.getParent();
		if (showCriticalCodeColumn) {
		    TableCellData tableCellData = null;
            String flatProfilerFileName = SaUiLsUtils.getBaseNameFromConfig(storageCfgFile.getName()
                                                                                          .toString())
							+ "." //$NON-NLS-1$
							+ SaConstants.storage_flat_profiler_file_extension; //$NON-NLS-1$

			if (criticalCodeEnabled) {
                tableCellData = new TableCellData(openFlatProfile, -1,
                                                  profilersFileDir + Path.SEPARATOR
                                                                       + flatProfilerFileName,
                                                  projectName);
			} else if (statisticalEnabled) {
                tableCellData = new TableCellData(openStatistical, -1,
                                                  profilersFileDir + Path.SEPARATOR
                                                                       + flatProfilerFileName,
                                                  projectName);
			} else {
                tableCellData = new TableCellData(CommonConstants.EMPTY_STRING, -1, null,
                                                  projectName);
			}
   			resultRecord.cells.add(tableCellData);
		}
		
		if (showPerformanceColumn) {
		    TableCellData tableCellData = null;
            String performanceFileName = SaUiLsUtils.getBaseNameFromConfig(storageCfgFile.getName()
                                                                                         .toString())
							+ "." //$NON-NLS-1$
							+ SaConstants.storage_performance_file_extension; //$NON-NLS-1$

			if (performaceEnabled) {
                tableCellData = new TableCellData(openPerformance, -1,
                                                  profilersFileDir + Path.SEPARATOR
                                                                       + performanceFileName,
                                                  projectName);
			} else {
                tableCellData = new TableCellData(CommonConstants.EMPTY_STRING, -1, null,
                                                  projectName);
			}
			resultRecord.cells.add(tableCellData);
		}
		

		if (showCallTreeColumn) {
		    TableCellData tableCellData = null;
            String callTreeFileName = SaUiLsUtils.getBaseNameFromConfig(storageCfgFile.getName()
                                                                                      .toString())
							+ "." //$NON-NLS-1$
							+ SaConstants.storage_call_tree_file_extension;

			if (callTreeEnabled) {
                tableCellData = new TableCellData(openCallTree, -1,
                                                  profilersFileDir + Path.SEPARATOR
                                                                    + callTreeFileName,
                                                  projectName);
			} else {
                tableCellData = new TableCellData(CommonConstants.EMPTY_STRING, -1, null,
                                                  projectName);
			}
			resultRecord.cells.add(tableCellData);
		}
		
		

		if (showCounterpointsColumn) {
            TableCellData tableCellData = null;
            if (counterpointsEnabled) { 
                tableCellData = new TableCellData(openCounterpoints, -1,
                                                  storageCfgFile.getAbsolutePath(), projectName);
            } else {
                tableCellData = new TableCellData(CommonConstants.EMPTY_STRING, -1, null,
                                                  projectName);
            }
            resultRecord.cells.add(tableCellData);
        }
	      
		if (showConfigureResultsColumn) {
            TableCellData tableCellData = new TableCellData(openConfigureResults, -1,
                                                            storageCfgFile.getAbsolutePath(),
                                                            projectName);
			resultRecord.cells.add(tableCellData);
		}
							
        resultRecord.cells.add(new TableCellData(dateFormat.format(new Date(storageCfgFile.lastModified())),
                                                 projectName, -1));
		
		if (showCommentsColumn) {
			TableCellData tableCellData = new TableCellData(comment, projectName, -1);
			tableCellData.setCommentsCell(true);
			resultRecord.cells.add(tableCellData);
		}
		
		currentDisplayIndex++;
		m_profileResults.records.add(resultRecord);
	}
		
	/**
	 * Add cell that contains the name of profiling results.
	 * 
	 * @param resultRecord
	 *            the table record data
	 * @param resultName
	 *            the name of the results
	 * @param launchName
	 *            the launch name of the project
	 * @param latest
	 *            if this is the last modified entry in the table
	 */
    private void addResultsCell(TableRecordData resultRecord, String resultName, String projectName,
                                String launchName, boolean latest, boolean projectLatest) {
		TableCellData cellData = new TableCellData(resultName, projectName, -1);
		cellData.setResultsCell(true);
		cellData.setLaunchName(launchName);
		cellData.setLastModified(latest);
		cellData.setProjectLastModified(projectLatest);
		resultRecord.cells.add(cellData);
	}
	
	@Override
	public void initializeConfiguration() {	
		
		List<IAnalyzer> analyzers = AnalysisManager.getDefault().getAllAnalyzers();
		
		// If there are no extension points, show all columns.
		// This is to keep compatibility with platforms that
		// used the view before the extension point was implemented
		// (e.g.: MCU).
		if (analyzers.size() > 0) {
			for (IAnalyzer analyzer : analyzers) {
				showTraceColumn = hasFeature(analyzer, showTraceColumn, Feature.TRACE);
				showTimelineColumn = hasFeature(analyzer, showTimelineColumn, Feature.TIMELINE) 
						&& Utils.isExtension(TraceResultsConstants.TIMELINE_EXTENSION);
                showCriticalCodeColumn = hasFeature(analyzer, showCriticalCodeColumn,
                                                    Feature.CRITICAL_CODE)
						&& Utils.isExtension(TraceResultsConstants.CRITICALCODE_EXTENSION);
                showPerformanceColumn = hasFeature(analyzer, showPerformanceColumn,
                                                   Feature.PERFORMANCE)
						&& Utils.isExtension(TraceResultsConstants.PERFORMANCE_EXTENSION);
				showCallTreeColumn = hasFeature(analyzer, showCallTreeColumn, Feature.CALL_TREE)
						&& Utils.isExtension(TraceResultsConstants.CALL_TREE_EXTENSION);
                showCounterpointsColumn = hasFeature(analyzer, showCounterpointsColumn,
                                                     Feature.COUNTERPOINTS)
						&& Utils.isExtension(TraceResultsConstants.COUNTERPOINTS_EXTENSION);
				
                // TODO create Rec Engine extension point and update the
                // condition below.
				
                showConfigureResultsColumn = false; // Enable here configure
                                                    // column
				showCommentsColumn = true; //Enable by default comments
			}			
		} else {
			showTraceColumn = true;
			showTimelineColumn = true;
			showCriticalCodeColumn = true;
			showPerformanceColumn = true;
			showCallTreeColumn = true;
			showCounterpointsColumn = true;
			showConfigureResultsColumn = true;
			showCommentsColumn = true;
		}
		
		tableRecordsConfigurations = new HashMap<String, TableRecordConfiguration>();
		String resultsName = Messages.results_name; 
		String resultsTrace = Messages.results_trace; 
		String resultsTimeLine = Messages.results_time_line;
		String resultsFlatProfile = Messages.results_flat_profile;
		String resultsPerformance = Messages.results_performance;
		String resultsTime = Messages.results_time;
		
		String resultsCallTree = Messages.results_call_tree;
		String resultsCounterpoints = Messages.results_counterpoints;
		String resultsConfiguration = Messages.results_configuration;
		
		String resultsComment = Messages.results_comment;

		String traceCfgTableRecordConfigName = TraceResultsConstants.TABLE_TRACE_CFG_COLUMN_NAME;
		
        TableRecordConfiguration project = new TableRecordConfiguration(TraceResultsConstants.TABLE_PLATFORM_COLUMN_NAME,
                                                                        false, false);
        TableRecordConfiguration traceConfig = new TableRecordConfiguration(traceCfgTableRecordConfigName,
                                                                            false, false);
        TableRecordConfiguration result = new TableRecordConfiguration(TraceResultsConstants.TABLE_RESULT_COLUMN_NAME,
                                                                       false, false);
		
		project.cells.add(new TableCellConfiguration(resultsName, false, false, 150, 0, true));
		traceConfig.cells.add(new TableCellConfiguration(resultsName, false, false, 150, 0, true));
		result.cells.add(new TableCellConfiguration(resultsName, false, false, 130, 0, true));
		
		if (showTraceColumn) {
			project.cells.add(new TableCellConfiguration(resultsTrace, false, false, 0, 0, false));
            traceConfig.cells.add(new TableCellConfiguration(resultsTrace, false, false, 150, 0,
                                                             false, false, true, "button")); //$NON-NLS-1$
			result.cells.add(new TableCellConfiguration(resultsTrace, false, false, 65, 0, false));
		}
		if (showTimelineColumn) {
            project.cells.add(new TableCellConfiguration(resultsTimeLine, false, false, 0, 0,
                                                         false));
            traceConfig.cells.add(new TableCellConfiguration(resultsTimeLine, false, false, 0, 0,
                                                             false));
            result.cells.add(new TableCellConfiguration(resultsTimeLine, false, false, 75, 0,
                                                        false));
		}
		if (showCriticalCodeColumn) {
            project.cells.add(new TableCellConfiguration(resultsFlatProfile, false, false, 0, 0,
                                                         false));
            traceConfig.cells.add(new TableCellConfiguration(resultsFlatProfile, false, false, 0, 0,
                                                             false));
            result.cells.add(new TableCellConfiguration(resultsFlatProfile, false, false, 95, 0,
                                                        false));
		}
		if (showPerformanceColumn) {
            project.cells.add(new TableCellConfiguration(resultsPerformance, false, false, 0, 0,
                                                         false));
            traceConfig.cells.add(new TableCellConfiguration(resultsPerformance, false, false, 0, 0,
                                                             false));
            result.cells.add(new TableCellConfiguration(resultsPerformance, false, false, 95, 0,
                                                        false));
		}
		if (showCallTreeColumn) {
            project.cells.add(new TableCellConfiguration(resultsCallTree, false, false, 0, 0,
                                                         false));
            traceConfig.cells.add(new TableCellConfiguration(resultsCallTree, false, false, 0, 0,
                                                             false));
            result.cells.add(new TableCellConfiguration(resultsCallTree, false, false, 82, 0,
                                                        false));
		}
		
				
		if (showConfigureResultsColumn) {
            project.cells.add(new TableCellConfiguration(resultsConfiguration, false, false, 45, 0,
                                                         false));
            traceConfig.cells.add(new TableCellConfiguration(resultsConfiguration, false, false, 45,
                                                             0, false));
            result.cells.add(new TableCellConfiguration(resultsConfiguration, false, false, 45, 0,
                                                        false));
		}
				
		if (showCounterpointsColumn) {
            project.cells.add(new TableCellConfiguration(resultsCounterpoints, false, false, 45, 0,
                                                         false));
            traceConfig.cells.add(new TableCellConfiguration(resultsCounterpoints, false, false, 45,
                                                             0, false));
            result.cells.add(new TableCellConfiguration(resultsCounterpoints, false, false, 45, 0,
                                                        false));
		}
	
		project.cells.add(new TableCellConfiguration(resultsTime, true, true, 150, 0, true));
		traceConfig.cells.add(new TableCellConfiguration(resultsTime, false, false, 150, 0, true));
		result.cells.add(new TableCellConfiguration(resultsTime, true, true, 150, 0, true));
		
		project.cells.add(new TableCellConfiguration(resultsComment, true, true, 190, 0, true));
        traceConfig.cells.add(new TableCellConfiguration(resultsComment, false, false, 190, 0,
                                                         true));
		result.cells.add(new TableCellConfiguration(resultsComment, true, true, 190, 0, true));
		
		tableRecordsConfigurations.put(project.name, project);
		tableRecordsConfigurations.put(traceConfig.name, traceConfig);
		tableRecordsConfigurations.put(result.name, result);
	}
	
	/**
     * Loads the platform configuration files from their default location
     * (.metadata\.plugins\com.freescale.sa\platformConfig\).
	 */
	private File[] loadDefaultPlatformConfigFiles() {
		File outputFolder = ConfigurationUtils.getDefaultPlatformConfigStoragePath().toFile();
		return outputFolder.listFiles(new CustomFileFilter(SaConstants.PLATFORM_CONFIG_EXTENSION));
	}

    private boolean hasFeature(IAnalyzer analyzer, boolean flag, Feature feature) {
		return flag || analyzer.supports(feature);
	}

	@Override
	public void closeProvider() {
		m_editorClosed = true;
	}
	
	@Override
	public Vector<String> getMainMenuActions() {
		Vector<String> actions = new Vector<String>();

		actions.add(refreshAction);
		actions.add(expandAllAction);
		actions.add(collapseAllAction);
		actions.add(customFolderAction);
		
		return actions;
	}
	
	/*
     * When changing this function, make sure to update the unit test, because
     * it is overriding it
	 */
	protected void startAutomaticRefreshJob() {
		UIJob job = new UIJob(Messages.ProfileResultsInput_21) {
			@Override
			public IStatus runInUIThread(IProgressMonitor monitor) {
                try {
                } finally {
					if (!m_editorClosed) {
						schedule(500);
					}
				}
				return Status.OK_STATUS;
			}
		};

		job.schedule(); // start as soon as possible
	}
		
}
