/*******************************************************************************
 * Copyright (c) 2014, 2015, 2016 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution.
 *******************************************************************************/
package com.freescale.sa.ui.traceresults.views;

import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.Transferable;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.math.BigInteger;
import java.net.URL;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.core.model.IDebugTarget;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ColumnViewerToolTipSupport;
import org.eclipse.jface.viewers.FocusCellHighlighter;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.jface.viewers.TreeViewerFocusCellManager;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TreeEditor;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.MenuEvent;
import org.eclipse.swt.events.MenuListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.TreeEvent;
import org.eclipse.swt.events.TreeListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.forms.widgets.Form;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.progress.UIJob;

import com.freescale.sa.SaConstants;
import com.freescale.sa.configurator.config.ConfigurationUtils;
import com.freescale.sa.configurator.ls.jni.LSConfigLoader_JNI;
import com.freescale.sa.ui.common.AttributeType;
import com.freescale.sa.ui.common.UIConstants;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableCellConfiguration;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableCellData;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableData;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableRecordConfiguration;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableRecordData;
import com.freescale.sa.ui.common.flatdata.IDataFormDrawer;
import com.freescale.sa.ui.common.utils.SaUiLsUtils;
import com.freescale.sa.ui.common.utils.StorageUtils;
import com.freescale.sa.ui.traceresults.Activator;
import com.freescale.sa.ui.traceresults.utils.TraceResultsLock;
import com.freescale.sa.util.CommonConstants;
import com.freescale.sa.util.CustomFileFilter;
import com.freescale.sa.util.Utils;

public class TraceResultsFormDrawer implements IDataFormDrawer, ChangeListener,
		IResourceChangeListener {

	private Logger LOGGER = Logger.getLogger(TraceResultsFormDrawer.class);

	private FormToolkit m_toolkit;
	private Form m_form;

	private TreeViewer m_dataTableViewer;
	private Tree m_dataTable;

	protected TreeViewerFocusCellManager m_focusCellManager;

	MenuManager m_tableMenuManager = null;

	private Action m_homeAction;
	private Action m_refreshAction;
	private Action m_expandAllAction;
	private Action m_collapseAllAction;
	private Action m_exportAction;

	private Action m_showGridAction;

	private boolean m_showTableLines = true;

	private TableData m_tableData;

	static private class GraphicsDescription {
		String recordType;
		String cellName;
		int cellIndex;
		boolean show;

		public GraphicsDescription(String recordType, String cellName,
				int cellIndex, boolean show) {
			this.recordType = recordType;
			this.cellName = cellName;
			this.cellIndex = cellIndex;
			this.show = show;
		}
	}

	Vector<GraphicsDescription> m_graphicsOn;

	private static final String startStopTraceButtonType = "startStopBtn"; //$NON-NLS-1$
	private static final String resetTraceButtonType = "resetBtn"; //$NON-NLS-1$
	private static final String nameColumn = "Name"; //$NON-NLS-1$

	private static final long rightArrowCode = 16777220;
	private static final int standardMarginWidth = 5;
	private static final int standardIconWidth = 16;

	static public class ButtonsProperties {
		public String buttonType;
		public String attachedLaunchName;

		public ButtonsProperties() {
			buttonType = CommonConstants.EMPTY_STRING;
			attachedLaunchName = CommonConstants.EMPTY_STRING;
		}

		public void setAttributes(String btnType, String launchName) {
			buttonType = btnType;
			attachedLaunchName = launchName;
		}

		@Override
		public boolean equals(Object o) {
			if (!(o instanceof ButtonsProperties)) {
				return false;
			}
			ButtonsProperties bp = (ButtonsProperties) o;
			return bp.buttonType.equals(buttonType)
					&& bp.attachedLaunchName.equals(attachedLaunchName);
		}

		@Override
		public int hashCode() {
			return attachedLaunchName.hashCode();
		}
	}

	private FlatDataProvider m_dataProvider = null;

	private Vector<TreeEditor> m_treeEditors;

	private static Map<ButtonsProperties, Button> m_buttons = new HashMap<ButtonsProperties, Button>();

	Map<String, TableRecordConfiguration> m_currentTableRecordsConfigurations = null;
	private TableData m_currentTableData = null;

	// Keeps the user preferences for each table
	static private class TableSettings {
		int sortColumnIndex;
		int sortDirection;
	}

	protected Map<Long, TableSettings> m_tablesSettings = null;

	private void selectItem(TableRecordData selectData, TreeItem root) {
		TableRecordData rootData = (TableRecordData) root.getData();
		if (rootData == null)
			return;
		if (selectData.parentIndex == rootData.parentIndex
				&& selectData.index == rootData.index)
			m_dataTableViewer.getTree().setSelection(root);
		TreeItem[] items = root.getItems();
		for (TreeItem treeItem : items) {
			selectItem(selectData, treeItem);
		}
	}

	private UIJob refreshJob = new UIJob(Messages.FlatDataFormDrawer_9) {
		public IStatus runInUIThread(IProgressMonitor monitor) {
			// Get table data performs refresh on workspace.
			// Don't introduce get table data into critical section, as race
			// condition appears with all other threads calling refresh()
			// and needing to get lock on workspace first.

			TableRecordData[] records = null;
			TableData tableData = null;
			if (m_dataTableViewer != null) {
				Tree tree = m_dataTableViewer.getTree();
				if (tree != null && !tree.isDisposed()) {
					TreeItem[] selection = tree.getSelection();
					records = new TableRecordData[selection.length];
					for (int i = 0; i < selection.length; i++) {
						records[i] = (TableRecordData) selection[i].getData();
					}
	
					tableData = m_dataProvider.getTableData(m_currentTableData.id);
				}
			}

			try {
				if (m_dataTable == null || !m_dataTable.isDisposed()) {
					setTableViewerData(tableData);
				}
				if (records != null) {
					for (TableRecordData record : records) {
						TreeItem[] items = m_dataTableViewer.getTree()
								.getItems();
						for (TreeItem treeItem : items) {
							selectItem(record, treeItem);
						}
					}
				}
				return Status.OK_STATUS;
			} catch (Exception e) {
				e.printStackTrace();
				return Status.CANCEL_STATUS;
			}
		}
	};

	/**
	 * Creates a flat data editor example.
	 */
	public TraceResultsFormDrawer(FlatDataProvider dataProvider) {
		super();

		m_dataProvider = dataProvider;
		dataProvider.setDrawer(this);

		makeActions();

		m_tablesSettings = new HashMap<Long, TableSettings>();
	}

	public Form getForm() {
		return m_form;
	}

	/**
	 * Convenience method to get the workspace
	 */
	private IWorkspace getWorkspace() {
		return ResourcesPlugin.getWorkspace();
	}

	public void stateChanged(ChangeEvent e) {
		m_dataProvider.initializeConfiguration();
		refresh();
	}

	public void resourceChanged(IResourceChangeEvent event) {
		switch (event.getType()) {
		case IResourceChangeEvent.PRE_CLOSE:
			break;

		case IResourceChangeEvent.PRE_DELETE:
			break;

		case IResourceChangeEvent.POST_CHANGE:
			final ArrayList<String> changed = new ArrayList<String>();
			IResourceDeltaVisitor visitor = new IResourceDeltaVisitor() {
				public boolean visit(IResourceDelta delta) {
					IResource resource = delta.getResource();
					String resourceName = resource.getName();

					switch (delta.getKind()) {
					case IResourceDelta.ADDED:
						switch (resource.getType()) {
						case IResource.PROJECT: {
							// nothing to be done for a new project
							break;
						}

						case IResource.FOLDER: {
							// nothing to be done for a new folder
							break;
						}

						case IResource.FILE: {
							if (resourceName
									.endsWith(SaConstants.decoded_trace_config_extension)
									|| resourceName
											.endsWith(SaConstants.raw_trace_config_extension)) {
								changed.add(resource.getName());
							}
							break;
						}
						}
						break;

					case IResourceDelta.REMOVED:
						switch (resource.getType()) {
						case IResource.PROJECT: {
							changed.add(resource.getName());
							break;
						}

						case IResource.FOLDER: {
							// nothing to be done for a folder
							break;
						}

						case IResource.FILE: {
							if (resourceName
									.endsWith(SaConstants.decoded_trace_config_extension)
									|| resourceName
											.endsWith(SaConstants.raw_trace_config_extension)) {
								changed.add(resource.getName());
							}
							break;
						}
						}
						break;

					case IResourceDelta.CHANGED:
						switch (resource.getType()) {
						case IResource.PROJECT:
							break;

						case IResource.FOLDER:
							break;

						case IResource.FILE:
							if (resourceName
									.endsWith(SaConstants.decoded_trace_config_extension)
									|| resourceName
											.endsWith(SaConstants.raw_trace_config_extension)) {
								changed.add(resource.getName());
							}
							break;
						}
						break;
					}
					return true; // visit the children
				}
			};

			try {
				event.getDelta().accept(visitor);
			} catch (CoreException e) {
				e.printStackTrace();
				LOGGER.error("[resourceChanged]" + e.getMessage()); //$NON-NLS-1$
			}
			if (changed.size() > 0) {
				if (!m_form.isDisposed()) {
					Display display = m_form.getDisplay();

					if (!display.isDisposed()) {
						display.asyncExec(new Runnable() {
							public void run() {
								// refresh the form
								refresh();
							}
						});
					}
				}
			}
			break;
		}
	}

	public void drawForm(Composite parent) {
		m_toolkit = new FormToolkit(parent.getDisplay());
		m_form = m_toolkit.createForm(parent);

		m_toolkit.decorateFormHeading(m_form);

		m_tableMenuManager = new MenuManager();

		Vector<String> actions = m_dataProvider.getMainMenuActions();

		if (actions != null) {

			if (actions.contains(FlatDataProvider.homeAction)) {
				m_form.getToolBarManager().add(m_homeAction);
				m_tableMenuManager.add(m_homeAction);
			}
			if (actions.contains(FlatDataProvider.refreshAction)) {
				m_form.getToolBarManager().add(m_refreshAction);
				m_tableMenuManager.add(m_refreshAction);
			}
			if (actions.contains(FlatDataProvider.expandAllAction)) {
				m_form.getToolBarManager().add(m_expandAllAction);
				m_tableMenuManager.add(m_expandAllAction);
			}
			if (actions.contains(FlatDataProvider.collapseAllAction)) {
				m_form.getToolBarManager().add(m_collapseAllAction);
				m_tableMenuManager.add(m_collapseAllAction);
			}
			if (actions.contains(FlatDataProvider.exportAction)) {
				m_form.getToolBarManager().add(m_exportAction);
				m_tableMenuManager.add(m_exportAction);
			}
			if (actions.contains(FlatDataProvider.showGridAction)) {
				m_form.getToolBarManager().add(m_showGridAction);
				m_tableMenuManager.add(m_showGridAction);
				m_showTableLines = false;
			}
		}

		m_dataProvider.setupCustomMenuActions(m_form, m_tableMenuManager);

		m_form.getToolBarManager().update(true);

		GridLayout layout = new GridLayout();

		m_form.getBody().setLayout(layout);

		// Add listeners, removed by dispose() below
		m_dataProvider.addListener(this);
		getWorkspace().addResourceChangeListener(this,
				IResourceChangeEvent.POST_CHANGE);

		m_dataProvider.initializeConfiguration();
		setTableViewerData(m_dataProvider.getTableData(0));
		autoResizeAllColumns();
	}

	private void autoResizeAllColumns() {
		if (m_dataTableViewer == null || m_dataTableViewer.getTree() == null)
			return;

		/* automatically resize all columns */
		for (TreeColumn tc : m_dataTableViewer.getTree().getColumns()) {
			tc.pack();
			// The pack calculation includes only the text and not the icon
			// included in the editor,
			// so increase the width by one standard margin width and one
			// standard icon width.
			tc.setWidth(tc.getWidth() + standardMarginWidth + standardIconWidth);
		}

		// Give the first column a bit more - first check if column 0 exists
		if (m_dataTableViewer.getTree().getColumns().length != 0) {
			TreeColumn tc = m_dataTableViewer.getTree().getColumn(0);
			tc.setWidth(Math.min(300, Math.max(150, tc.getWidth())));
		}

	}

	private void makeActions() {
		m_homeAction = new Action() {
			public void run() {
				setTableViewerData(m_dataProvider.getTableData(0));
			}
		};
		m_homeAction.setText(Messages.FlatDataFormDrawer_11);
		m_homeAction.setToolTipText(Messages.FlatDataFormDrawer_12);

		m_homeAction.setImageDescriptor(PlatformUI.getWorkbench()
				.getSharedImages()
				.getImageDescriptor(ISharedImages.IMG_ETOOL_HOME_NAV));

		m_showGridAction = new Action() {
			public void run() {
				m_showTableLines = !m_showTableLines;
				m_dataTable.setLinesVisible(m_showTableLines);
			}
		};
		m_showGridAction.setText(Messages.FlatDataFormDrawer_13);
		m_showGridAction.setToolTipText(Messages.FlatDataFormDrawer_14);
		URL gridUrl = FileLocator.find(Platform.getBundle(Activator.PLUGIN_ID),
		// new Path(UIConstants.results_disabled_iconfile), null);
				new Path(UIConstants.analysis_trace_iconfile), null);

		m_showGridAction.setImageDescriptor(ImageDescriptor
				.createFromURL(gridUrl));

		m_refreshAction = new Action() {
			public void run() {
				refresh();
			}
		};
		m_refreshAction.setText(Messages.FlatDataFormDrawer_17);
		m_refreshAction.setToolTipText(Messages.FlatDataFormDrawer_18);
		URL refreshUrl = FileLocator.find(Platform
				.getBundle(Activator.PLUGIN_ID), new Path(
				UIConstants.results_refresh_iconfile), null);

		m_refreshAction.setImageDescriptor(ImageDescriptor
				.createFromURL(refreshUrl));

		m_dataProvider.makeCustomActions();

		m_expandAllAction = new Action() {
			public void run() {
				m_dataTableViewer.expandAll();
				treeChanged(null);
			}
		};

		m_expandAllAction.setText(Messages.FlatDataFormDrawer_21);
		m_expandAllAction.setToolTipText(Messages.FlatDataFormDrawer_22);
		URL expandAllUrl = FileLocator.find(Platform
				.getBundle(Activator.PLUGIN_ID),
				new Path("icons/expandall.gif"), null); //$NON-NLS-1$

		m_expandAllAction.setImageDescriptor(ImageDescriptor
				.createFromURL(expandAllUrl));

		m_collapseAllAction = new Action() {
			public void run() {
				m_dataTableViewer.collapseAll();
				treeChanged(null);
			}
		};
		m_collapseAllAction.setText(Messages.FlatDataFormDrawer_25);
		m_collapseAllAction.setToolTipText(Messages.FlatDataFormDrawer_26);
		URL collapseAllUrl = FileLocator.find(Platform
				.getBundle(Activator.PLUGIN_ID), new Path(
				"icons/collapseall.gif"), null); //$NON-NLS-1$

		m_collapseAllAction.setImageDescriptor(ImageDescriptor
				.createFromURL(collapseAllUrl));

		m_exportAction = new Action() {
			public void run() {

				int response = SWT.NO;
				int filterIndex = 0;
				String selected = null;

				while (response == SWT.NO) {
					response = SWT.YES;
					FileDialog fileDialog = new FileDialog(
							m_dataTable.getShell(), SWT.SAVE);

					fileDialog.setText(Messages.FlatDataFormDrawer_29);
					fileDialog.setFilterPath(CommonConstants.EMPTY_STRING);
					fileDialog.setFilterIndex(filterIndex);
					String[] filterExt = { "*.csv", "*.*" }; //$NON-NLS-1$ //$NON-NLS-2$

					fileDialog.setFilterExtensions(filterExt);
					selected = fileDialog.open();
					filterIndex = fileDialog.getFilterIndex();
					if (selected == null) {
						return;
					}

					File file = new File(selected);

					if (file.exists()) {
						MessageBox messageBox = new MessageBox(
								m_dataTable.getShell(), SWT.ICON_QUESTION
										| SWT.YES | SWT.NO | SWT.CANCEL);

						messageBox.setMessage(Messages.FlatDataFormDrawer_33);
						messageBox.setText(Messages.FlatDataFormDrawer_34);
						response = messageBox.open();
						if (response == SWT.CANCEL) {
							return;
						}
					}

				}

				try {
					BufferedWriter out = new BufferedWriter(new FileWriter(
							selected));

					try {

						if (m_dataTable.getColumnCount() == 0) {
							return;
						}

						StringBuilder line = new StringBuilder();

						for (int index : m_dataTable.getColumnOrder()) {
							if (line.length() > 0) {
								line.append(',');
							}

							line.append(SaUiLsUtils.formatForCSV(m_dataTable
									.getColumn(index).getText()));
						}

						line.append('\n');
						out.write(line.toString());

						writeItemsToFile(out, m_dataTable.getItems());
					} finally {
						out.close();
					}
				} catch (IOException e) {
					e.printStackTrace();
					return;
				}
			}
		};
		m_exportAction.setText(Messages.FlatDataFormDrawer_38);
		m_exportAction.setToolTipText(Messages.FlatDataFormDrawer_39);
		m_exportAction.setImageDescriptor(PlatformUI.getWorkbench()
				.getSharedImages()
				.getImageDescriptor(ISharedImages.IMG_ETOOL_SAVE_EDIT));
	}

	public void refresh() {
		if (m_dataTableViewer == null) {
			m_dataProvider.initializeConfiguration();
		}

		refreshJob.schedule();
	}

	private void writeItemsToFile(BufferedWriter out, TreeItem[] items) {
		try {
			for (TreeItem item : items) {
				StringBuilder line = new StringBuilder();

				for (int index : m_dataTable.getColumnOrder()) {
					if (line.length() > 0) {
						line.append(',');
					}
					line.append(SaUiLsUtils.formatForCSV(item.getText(index)));
				}

				line.append('\n');
				out.write(line.toString());

				if (item.getExpanded() && item.getItemCount() > 0) {
					writeItemsToFile(out, item.getItems());
				}
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	private void setTableViewerData(TableData tableData) {
		if (tableData == null) {
			return;
		}

		Vector<String> expandedProjects = new Vector<String>();
		Vector<Vector<String>> expandedLaunches = new Vector<Vector<String>>();

		// check which is the top visible item in the view (the expanded tree
		// may be
		// larger than the view's window and the scroll somewhere in the middle
		// so
		// the top visible item may not be the tree root)
		String topItemProject = null, topItemLaunch = null, topItemResult = null;
		if (m_dataTable != null && m_dataTable.getTopItem() != null) {
			TreeItem topItem = m_dataTable.getTopItem();
			if (topItem.getParentItem() == null) {
				topItemProject = topItem.getText();
			} else {
				if (topItem.getParent().getParentItem() == null) {
					topItemProject = topItem.getParentItem().getText(0);
					topItemLaunch = topItem.getText(0);
				} else {
					topItemProject = topItem.getParentItem().getParentItem()
							.getText(0);
					topItemLaunch = topItem.getParentItem().getText(0);
					topItemResult = topItem.getText(0);
				}
			}

			for (TreeItem project : m_dataTable.getItems()) {
				if (project.getExpanded()) {
					expandedProjects.add(project.getText(0));
					expandedLaunches.add(new Vector<String>());
					for (TreeItem launch : project.getItems()) {
						if (launch.getExpanded()) {
							expandedLaunches.lastElement().add(
									launch.getText(0));
						}
					}
				}
			}
		}

		m_currentTableData = tableData;

		if (m_dataTableViewer != null) {
			try {
				m_dataTableViewer.getTree().dispose();
			} catch (Exception e) {
				e.printStackTrace();
			}
			m_dataTableViewer = null;
		}

		m_dataTable = m_toolkit
				.createTree(m_form.getBody(), SWT.MULTI | SWT.FULL_SELECTION);
		m_toolkit.paintBordersFor(m_dataTable);
		m_dataTableViewer = new TreeViewer(m_dataTable);

		m_focusCellManager = new TreeViewerFocusCellManager(m_dataTableViewer,
				new FocusCellHighlighter(m_dataTableViewer) {
				});

		m_dataTable.setMenu(m_tableMenuManager.createContextMenu(m_dataTable));
		m_tableMenuManager.update(true);

		m_dataTable.setLayoutData(new GridData(GridData.FILL_BOTH));
		m_dataTable.setHeaderVisible(true);
		m_dataTable.setLinesVisible(m_showTableLines);
		m_dataTableViewer.setContentProvider(new ViewContentProvider());
		ColumnViewerToolTipSupport.enableFor(m_dataTableViewer);
		m_dataTableViewer.setSorter(new FlatDataSorter());
		m_dataTable.addMouseListener(new TraceResultsFormMouseAdapter(this));

		m_currentTableRecordsConfigurations = new HashMap<String, TableRecordConfiguration>();

		Vector<String> columnNames = new Vector<String>();
		Vector<Integer> columnWidths = new Vector<Integer>();
		Vector<Integer> allowSort = new Vector<Integer>();

		m_graphicsOn = new Vector<GraphicsDescription>();

		for (TableRecordData record : tableData.records) {
			String recordConfigName = record.recordConfigurationName;

			m_currentTableRecordsConfigurations.put(recordConfigName,
					m_dataProvider.tableRecordsConfigurations
							.get(recordConfigName));
		}

		for (TableRecordConfiguration recordConfig : m_currentTableRecordsConfigurations
				.values()) {

			int cellIndex = -1;

			if (recordConfig == null)
				continue;

			for (TableCellConfiguration cell : recordConfig.cells) {
				cellIndex++;
				if (cell.allowGraphics) {
					m_graphicsOn.add(new GraphicsDescription(recordConfig.name,
							cell.name, recordConfig.cells.indexOf(cell),
							cell.showGraphics));
				}
				if (allowSort.size() <= cellIndex) {
					allowSort.add(cell.allowSort ? 1 : 0);
				} else {
					allowSort.set(cellIndex, Math.max(allowSort.get(cellIndex),
							cell.allowSort ? 1 : 0));
				}
			}

			if (columnNames.size() <= recordConfig.cells.size()) {
				for (int j = 0; j < Math.min(columnNames.size(),
						recordConfig.cells.size()); j++) {
					columnWidths.set(j, Math.max(columnWidths.elementAt(j),
							recordConfig.cells.elementAt(j).actualWidth));
					if (!columnNames.elementAt(j).equals(
							recordConfig.cells.elementAt(j).name)) {
						columnNames.set(j, columnNames.elementAt(j)
								+ " / " + recordConfig.cells.elementAt(j).name); //$NON-NLS-1$
					}
				}
				for (int j = columnNames.size(); j < recordConfig.cells.size(); j++) {
					columnNames.add(recordConfig.cells.elementAt(j).name);
					columnWidths.add(Integer.valueOf(recordConfig.cells
							.elementAt(j).actualWidth));
				}
			}
		}

		Menu tableMenu = m_dataTable.getMenu();

		if (tableMenu.getItemCount() > 0) {
			new MenuItem(tableMenu, SWT.SEPARATOR);
		}

        // Create right click menu.
        createMenuItemsForCopy(tableMenu);

		if (!m_graphicsOn.isEmpty()) {

			if (tableMenu.getItemCount() > 0) {
				new MenuItem(tableMenu, SWT.SEPARATOR);
			}

			MenuItem graphicsOnMenuItem = new MenuItem(tableMenu, SWT.CASCADE);

			graphicsOnMenuItem.setText(Messages.FlatDataFormDrawer_46);
			final Menu graphicsOnMenu = new Menu(tableMenu);

			graphicsOnMenuItem.setMenu(graphicsOnMenu);

			boolean onlyOneRecordType = true;

			for (GraphicsDescription graphic : m_graphicsOn) {
				if (!graphic.recordType
						.equals(m_graphicsOn.firstElement().recordType)) {
					onlyOneRecordType = false;
					break;
				}
			}

			for (GraphicsDescription graphic : m_graphicsOn) {
				MenuItem item = new MenuItem(graphicsOnMenu, SWT.CHECK);

				if (onlyOneRecordType) {
					item.setText(graphic.cellName);
				} else {
					item.setText(graphic.recordType + " > " + graphic.cellName); //$NON-NLS-1$
				} //$NON-NLS-1$
				item.setSelection(graphic.show);
				item.setData(graphic);

				item.addSelectionListener(new SelectionAdapter() {
					public void widgetSelected(SelectionEvent e) {
						GraphicsDescription graphic = (GraphicsDescription) ((MenuItem) e.widget)
								.getData();

						graphic.show = !graphic.show;
						m_currentTableRecordsConfigurations
								.get(graphic.recordType).cells
								.elementAt(graphic.cellIndex).showGraphics = graphic.show;
						m_dataTable.redraw();
					}
				});
			}
		}

		TreeColumn column;

		for (int i = 0; i < columnNames.size(); i++) {
			column = new TreeColumn(m_dataTable, SWT.LEFT);
			column.setText(columnNames.elementAt(i));
			column.setWidth(columnWidths.elementAt(i).intValue());
			column.setResizable(true);
			TreeViewerColumn columnView = new TreeViewerColumn(m_dataTableViewer, column);
			columnView.setLabelProvider(new TraceResultsViewColumnLabelProvider(m_dataProvider));

			if (!column.getText().equals(nameColumn)) { // Name column is fixed
														// while the others can
														// be movable
				column.setMoveable(true);
			} else {
				column.addControlListener(new ControlAdapter() {
					public void controlMoved(ControlEvent e) {
						List<Integer> newOrder = new ArrayList<Integer>();
						// add the first
						newOrder.add(0);
						// loop around all columns
						for (int col : m_dataTable.getColumnOrder()) {
							// if not the first that was added, add to list
							if (col != 0) {
								newOrder.add(col);
							}
						}
						// convert list to array
						int[] newOrderArray = new int[newOrder.size()];
						for (int j = 0; j < newOrder.size(); j++) {
							newOrderArray[j] = newOrder.get(j);
						}
						// update tree order
						m_dataTable.setColumnOrder(newOrderArray);
					}
				});
			}

			if (allowSort.get(i).intValue() == 1) {
				column.addSelectionListener(new SelectionAdapter() {

					@Override
					public void widgetSelected(SelectionEvent e) {
						if (m_dataTable.getSortColumn() != (TreeColumn) e.widget) {
							m_dataTable.setSortColumn((TreeColumn) e.widget);
							m_dataTable.setSortDirection(SWT.UP);
						} else {
							if (m_dataTable.getSortDirection() == SWT.NONE) {
								m_dataTable.setSortDirection(SWT.UP);
							} else if (m_dataTable.getSortDirection() == SWT.UP) {
								m_dataTable.setSortDirection(SWT.DOWN);
							} else {
								m_dataTable.setSortDirection(SWT.NONE);
							}
						}
						m_dataTableViewer.refresh();
						treeChanged(null);

						// / Keeps the sort column and direction
						if (!m_tablesSettings
								.containsKey(m_currentTableData.id)) {
							m_tablesSettings.put(m_currentTableData.id,
									new TableSettings());
						}
						TableSettings settings = m_tablesSettings
								.get(m_currentTableData.id);
						TreeColumn sortColumn = m_dataTable.getSortColumn();

						for (int i = 0; i < m_dataTable.getColumnCount(); i++) {
							if (m_dataTable.getColumn(i) == sortColumn) {
								settings.sortColumnIndex = i;
							}
						}
						settings.sortDirection = m_dataTable.getSortDirection();

					}
				});
			}
			// / Keep the user preferences for column size and order
			column.addControlListener(new ControlAdapter() {
				@Override
				public void controlMoved(ControlEvent e) {
					updateUserPreferences();
				}

				@Override
				public void controlResized(ControlEvent e) {
					updateUserPreferences();
				}

				private void updateUserPreferences() {
					for (TableRecordConfiguration recordConfig : m_currentTableRecordsConfigurations
							.values()) {
						if (recordConfig == null || m_dataTable == null)
							continue;

						recordConfig.cellsOrder = m_dataTable.getColumnOrder();
						for (int i = 0; i < recordConfig.cells.size(); i++) {
							recordConfig.cells.elementAt(i).actualWidth = m_dataTable
									.getColumn(i).getWidth();
						}
					}
				}
			});
		}

		m_dataTable.addListener(SWT.MouseMove, new Listener() {
			public void handleEvent(Event event) {
				Point point = new Point(event.x, event.y);
				TreeItem item = m_dataTable.getItem(point);
				m_dataTable.setCursor(null);

				if (item == null) {
					return;
				}

				if (!(item.getData() instanceof TableRecordData)) {
					return;
				}

				TableRecordData recordData = (TableRecordData)item.getData();
				Integer columnIndex = getColumnIndex(item, point);
				
				if (getCellType(recordData, columnIndex) == FlatDataProvider.LINK_CELL) {
					m_dataTable.setCursor(m_dataTable.getDisplay().getSystemCursor(SWT.CURSOR_HAND));
				}

			}
		});
		
		m_dataTable.addListener(SWT.MouseDown, new Listener() {
			@Override
			public void handleEvent(Event event) {
				Point pt = new Point(event.x, event.y);
				TreeItem selectedItem = m_dataTable.getItem(pt);
				if (selectedItem == null || !(selectedItem.getData() instanceof TableRecordData)) {
					return;
				}
				m_dataTable.select(selectedItem);

				/* Find column index */
				Integer columnIndex = getColumnIndex(selectedItem, pt);
				TableRecordData recordData = (TableRecordData)selectedItem.getData();
				
				if (getCellType(recordData, columnIndex) != FlatDataProvider.LINK_CELL) {
					return;
				}

				TableCellData cell = recordData.cells.get(columnIndex);
				
				if (cell.linkToAnotherTable >= 0) {
					setTableViewerData(m_dataProvider
							.getTableData(cell.linkToAnotherTable));
				}
				if (!cell.externalAction.isEmpty()) {
					m_dataProvider.executeAction(selectedItem.getText(columnIndex),
							cell.externalAction, cell.projectName);
				}
			}
		});


		// / Restore the column order from user preferences
		for (TableRecordConfiguration recordConfig : m_currentTableRecordsConfigurations
				.values()) {
			if (recordConfig == null)
				continue;

			if (recordConfig.cellsOrder != null) {
				m_dataTable.setColumnOrder(recordConfig.cellsOrder);
			}
		}
		// / Restore the sort column and direction from user preferences
		if (m_tablesSettings.containsKey(m_currentTableData.id) && m_dataTable.getColumnCount() > 0) {
			m_dataTable.setSortColumn(m_dataTable.getColumn(m_tablesSettings
					.get(m_currentTableData.id).sortColumnIndex));
			m_dataTable.setSortDirection(m_tablesSettings
					.get(m_currentTableData.id).sortDirection);
		}

		m_dataTableViewer.setInput(tableData);

		m_dataTableViewer.expandToLevel(tableData.autoExpandLevel);

		// Create right click menu.
		createMenuItemsForHandlingSaResults(tableMenu);

		m_dataTable.addKeyListener(new KeyListener() {
			public void keyPressed(KeyEvent e) {

				// Link the cells to editors only if right key is pressed (tree
				// expanded)
				// and if the selected line is trace config line.
				if (e.keyCode == rightArrowCode) {
					TreeItem[] selection = m_dataTableViewer.getTree()
							.getSelection();

					for (TreeItem item : selection) {
						TableRecordData line = (TableRecordData) item.getData();

						if (line.isTraceConfig) {
							treeChanged(null);
						}
					}
				}
			}

			public void keyReleased(KeyEvent e) {
			}
		});

		m_dataTable.addTreeListener(new TreeListener() {
			public void treeCollapsed(TreeEvent e) {
				treeChanged(null);
			}

			public void treeExpanded(TreeEvent e) {
				treeChanged(null);
			}
		});

		treeChanged(null);

		m_dataTable.addListener(SWT.PaintItem, new Listener() {

			public void handleEvent(Event event) {

				GC gc = event.gc;

				TreeItem item = (TreeItem) event.item;
				TableRecordData recordData = (TableRecordData) item.getData();

				if (event.index >= recordData.cells.size()) {
					return;
				}

				TableRecordConfiguration trc = m_currentTableRecordsConfigurations
						.get(recordData.recordConfigurationName);

				if (trc == null)
					return;

				TableCellConfiguration cellConfig = trc.cells
						.elementAt(event.index);

				if (cellConfig.showGraphics) {
					Color background = gc.getBackground();

					gc.setForeground(background);
					gc.fillRectangle(event.x, event.y, event.width,
							event.height);

					gc.setForeground(Display.getCurrent().getSystemColor(
							SWT.COLOR_RED));
					gc.setBackground(Display.getCurrent().getSystemColor(
							SWT.COLOR_YELLOW));

					int width = 0;
					int maxValueOfColumn = computeMaximumValueOfColumn(
							recordData.recordConfigurationName, event.index);

					if (maxValueOfColumn == 0) {
						width = 0;
					} else {
						width = (int) ((double) m_dataTable.getColumn(
								event.index).getWidth() * ((double) getInt(recordData.cells
								.elementAt(event.index).value) / (double) (maxValueOfColumn)));
					}
					if (width >= 1) {
						gc.fillGradientRectangle(event.x, event.y + 1, width,
								event.height - 2, true);
						gc.drawRectangle(event.x, event.y + 1, width - 1,
								event.height - 3);
					}

					String text = item.getText(event.index);

					if (!text.isEmpty()) {

						Point size = event.gc.textExtent(text);
						int offset = Math.max(0, (event.height - size.y) / 2);

						if ((event.detail & SWT.SELECTED) != 0) {
							gc.setForeground(Display.getCurrent()
									.getSystemColor(
											SWT.COLOR_LIST_SELECTION_TEXT));
						} else {
							gc.setForeground(Display.getCurrent()
									.getSystemColor(SWT.COLOR_LIST_FOREGROUND));
						}
						gc.drawText(text, event.x + 5, event.y + offset, true);
					}
				}
			}
		});

		// restore the state of the previous TreeViewer - expanded nodes, as
		// well as the top visible item in the view
		TreeItem topItem = null;

		for (TreeItem project : m_dataTable.getItems()) {

			if (topItemProject != null
					&& topItemProject.equals(project.getText(0))) {
				topItem = project;
			}
			for (int i = 0; i < expandedProjects.size(); i++) {
				if (expandedProjects.elementAt(i).equals(project.getText(0))) {
					project.setExpanded(true);
					m_dataTableViewer.refresh();
					if (project.getItemCount() > 0) {
						for (TreeItem launch : project.getItems()) {

							if (topItemProject != null
									&& topItemProject
											.equals(project.getText(0))
									&& topItemLaunch != null
									&& topItemLaunch.equals(launch.getText(0))) {
								topItem = launch;
							}
							for (int j = 0; j < expandedLaunches.elementAt(i)
									.size(); j++) {
								if (expandedLaunches.elementAt(i).elementAt(j)
										.equals(launch.getText(0))) {
									launch.setExpanded(true);
									m_dataTableViewer.refresh();

									if (topItemProject != null
											&& topItemProject.equals(project
													.getText(0))
											&& topItemLaunch != null
											&& topItemLaunch.equals(launch
													.getText(0))
											&& topItemResult != null
											&& launch.getItemCount() > 0) {
										for (TreeItem result : launch
												.getItems()) {
											if (result.getText(0).equals(
													topItemResult)) {
												topItem = result;
												break;
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}

		if (topItem != null) {
			m_dataTable.setTopItem(topItem);
		}

		treeChanged(null);
	}
	
	static int getCellType(TableRecordData recordData, Integer columnIndex) {
		if (recordData.cells.size() > columnIndex && columnIndex >= 0) {
			TableCellData cell = recordData.cells.get(columnIndex);
			return cell.type;
		}
		
		return FlatDataProvider.INVALID_CELL_TYPE;
	}
	
	static int getCellType(Object data, Integer columnIndex) {
		if (data instanceof TableRecordData) {
			return getCellType((TableRecordData) data, columnIndex);
		}

		return FlatDataProvider.INVALID_CELL_TYPE;
	}
	
	/**
	 * Returns the index of column pointed by <code>point</code>.
	 * 
	 * @param item
	 *            The tree item with all column
	 * @param point
	 *            The coordinates of a point
	 * @return The index of the column or a negative value if it were not found
	 */
	private Integer getColumnIndex(TreeItem item, Point point) {
		Integer columnIndex = -1;
		for (int i = 0; i < m_dataTable.getColumnCount(); i++) {
			Rectangle rect = item.getBounds(i);
			if (rect.contains(point)) {
				columnIndex = i;
			}
		}

		return columnIndex;
	}

	/**
	 * Creates the menu items from the right click menu that perform operations
	 * on the SA results (e.g., copy, delete, rename results).
	 * 
	 * @param tableMenu
	 *            the parent table menu.
	 */
	private void createMenuItemsForHandlingSaResults(Menu tableMenu) {

		new MenuItem(tableMenu, SWT.SEPARATOR);

		// Copy results.
		final MenuItem copyData = new MenuItem(tableMenu, SWT.PUSH);
		copyData.setText(Messages.FlatDataFormDrawer_48);
		copyData.setAccelerator(SWT.CONTROL | 'S');
		copyData.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				backupSelectedResults();
			}
		});

		// Delete results.
		final MenuItem deleteData = new MenuItem(tableMenu, SWT.PUSH);

		deleteData.setText(Messages.FlatDataFormDrawer_49);
		deleteData.setAccelerator(SWT.DEL);
		deleteData.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				deleteSelectedResults();
			}
		});

		// Rename results.
		final MenuItem renameData = new MenuItem(tableMenu, SWT.PUSH);
		renameData.setText(Messages.FlatDataFormDrawer_50);
		renameData.setAccelerator(SWT.CONTROL | 'R');
		renameData.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				renameSelectedResults();
			}
		});
		
		// Edit comments
		final MenuItem editComment = new MenuItem(tableMenu, SWT.PUSH);
		editComment.setText(Messages.FlatDataFormDrawer_59);
		editComment.setAccelerator(SWT.INSERT);
		editComment.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				editSelectedComment();
			}
		});
		
		// Edit comments


		tableMenu.addMenuListener(new MenuListener() {

			@Override
			public void menuHidden(MenuEvent e) {
			}

			@Override
			public void menuShown(MenuEvent e) {
				String extension = null; // stored results extension
				int lastDot = -1; // the last '.' in the name of the stored data
				String name = null; // name of the stored results

				TreeItem[] selection = m_dataTableViewer.getTree()
						.getSelection();

				int selectedLines = selection.length;

				boolean enableRename = false;
				boolean enableDelete = false;
				boolean enableSave = false;
				boolean enableComments = false;

				boolean backup = false;

				File storageCfgFile = null;

				for (TreeItem item : selection) {

					TableRecordData line = (TableRecordData) item.getData();
					boolean projectLineSel = false;
					boolean traceCfgLineSel = line.isTraceConfig;

					// Case for project line selected.
					if (item.getParentItem() == null) {
						projectLineSel = true;
					} else {
						if (item.getParentItem().getParentItem() == null)
							projectLineSel = true;
					}

					// Case for saved results selected.

					TableRecordData recordData = (TableRecordData) item
							.getData();

					name = recordData.cells.get(0).value;
					lastDot = name.lastIndexOf('.');
					if (lastDot >= 0) {
						extension = name.substring(lastDot + 1);
						if (extension.length() == 4) {
							try {
								Integer.parseInt(extension);
								backup = true;
							} catch (NumberFormatException ex) {
								LOGGER.error("[menuShown] Error conversion.",
										ex);
							}
						}
					}

					// If multiple selection then, disable 'save', 'delete' and
					// 'rename'
					// when trace config or project line is selected (or both).
					if (selectedLines > 1) {
						if (traceCfgLineSel || projectLineSel) {
							break;
						} else {
							// Or if multiple results are selected
							// enable only 'save' and 'delete'.
							enableSave = true;
							enableDelete = true;
						}
					} else {
						if (!projectLineSel) {
							storageCfgFile = line.storageCfgFile;
						}
					}
				}

				copyData.setEnabled(enableSave);
				deleteData.setEnabled(enableDelete);
				renameData.setEnabled(enableRename);
				editComment.setEnabled(enableComments);
				
		

				// The enabled actions will be determined based on the
				// current state of the application: running,
				// suspended or terminated and/or on IAnalyzer.hasFeature().
				try {
					if (storageCfgFile != null) {
						enableActions(name, storageCfgFile, copyData,
								deleteData, renameData, editComment);
					}
				} catch (DebugException e1) {
					e1.printStackTrace();
				} catch (FileNotFoundException e1) {
					LOGGER.error("[createMenuItemsForHandlingSaResults] File not found");
				}

				// If the menu item is a backup copy, only the
				// 'delete' and 'rename' operations should be enabled.
				if (backup) {
					copyData.setEnabled(false);
					deleteData.setEnabled(true);
				}
			}
		});

	}

    /**
     * Creates the menu items from the right click menu that perform operations
     * on the SA results (e.g., copy line and copy cell).
     * 
     * @param tableMenu
     *            the parent table menu.
     */
    private void createMenuItemsForCopy(Menu tableMenu) {

        final MenuItem copyCell = new MenuItem(tableMenu, SWT.PUSH);
        copyCell.setText(Messages.FlatDataFormDrawer_44);
        copyCell.addSelectionListener(new SelectionAdapter()
        {
            public void widgetSelected(SelectionEvent e) {
                copyCell();
            }
        });

        final MenuItem copyLine = new MenuItem(tableMenu, SWT.PUSH);
        copyLine.setText(Messages.FlatDataFormDrawer_45);
        copyLine.setAccelerator(SWT.CONTROL | 'L');
        copyLine.addSelectionListener(new SelectionAdapter()
        {
            public void widgetSelected(SelectionEvent e) {
                copyLine();
            }
        });

        tableMenu.addMenuListener(new MenuListener()
        {

            @Override
            public void menuHidden(MenuEvent e) {
            }

            @Override
            public void menuShown(MenuEvent e) {
                String extension = null; // stored results extension
                int lastDot = -1; // the last '.' in the name of the stored data
                String name = null; // name of the stored results

                TreeItem[] selection = m_dataTableViewer.getTree().getSelection();

                int selectedLines = selection.length;

                boolean enableCopy = false;

                for (TreeItem item : selection) {

                    TableRecordData line = (TableRecordData) item.getData();
                    boolean projectLineSel = false;
                    boolean traceCfgLineSel = line.isTraceConfig;

                    // Case for project line selected.
                    if (item.getParentItem() == null) {
                        projectLineSel = true;
                    } else {
                        if (item.getParentItem().getParentItem() == null)
                            projectLineSel = true;
                    }

                    // Case for saved results selected.
                    TableRecordData recordData = (TableRecordData) item.getData();
                    name = recordData.cells.get(0).value;
                    lastDot = name.lastIndexOf('.');
                    if (lastDot >= 0) {
                        extension = name.substring(lastDot + 1);
                        if (extension.length() == 4) {
                            try {
                                Integer.parseInt(extension);
                            } catch (NumberFormatException ex) {
                                LOGGER.error("[menuShown] Error conversion.", ex);
                            }
                        }
                    }

                    // If multiple selection then, disable 'copy line', 'copy cell'
                    // when trace config or project line is selected (or both).
                    if (selectedLines > 1) {
                        if (traceCfgLineSel || projectLineSel) {
                            enableCopy = true;
                        } else {
                            // Or if multiple results are selected
                            break;
                        }
                    } else {
                        enableCopy = true;
                    }
                }

                copyLine.setEnabled(enableCopy);
                copyCell.setEnabled(enableCopy);
            }
        });
    }
    
	/**
	 * Set the data producer config file selection.
	 * 
	 * @param selection
	 *            - data producer config file
	 */
	public void setSelection(IFile selection) {
		if (selection == null) {
			return;
		}
		for (TableRecordData data : m_tableData.records) {
			if (((File)selection).equals(data.storageCfgFile)) {
				m_dataTableViewer.expandToLevel(data, 0);
				m_dataTableViewer.setSelection(new StructuredSelection(data));
				m_dataTableViewer.refresh();
				treeChanged(null);
				return;
			}
		}
	}

	/**
	 * Sets the 'enable' state for the results related actions available on the
	 * right click menu (copy and delete results).
	 * 
	 * @param launchName
	 *            The name of the launch config associated with the tree item
	 *            that is currently selected.
	 * @param storageCfgFile
	 * @param copyData
	 *            The 'copyData' action.
	 * @param deleteData
	 *            The 'deleteData' action.
	 * @throws DebugException
	 * 
	 * @note [raluca] I guess the logic of determining the state of the launches
	 *       (suspended, running or terminated) would stay better in
	 *       'LaunchHook'. I could't implement it like this because a circular
	 *       dependency is created if I add the launch plug-in as a dependency.
	 *       Maybe we should consider solving this in the future (?).
	 */
	private void enableActions(String launchName, File storageCfgFile,
			MenuItem copyData, MenuItem deleteData, MenuItem renameData, MenuItem editComments)
			throws DebugException, FileNotFoundException {

		// tells if the current launch is suspended;
		// the launch is suspended if all it's threads
		// are suspended
		boolean suspended = false;
		// tells if the current launch is terminated;
		// the launch is suspended if all it's threads
		// are suspended
		boolean terminated = false;
		// the name of the launch config associated with the debug target
		String dbgTargetLaunchName = null;

		// enable both actions by default
		boolean enableCopy = true;
		boolean enableDelete = true;
		boolean enableRename = true;
		boolean enableComments = true;
		/* Get the list of launches */
		ILaunchManager launchMgr = DebugPlugin.getDefault().getLaunchManager();

		/* Get the list of registered launches */
		IDebugTarget[] registeredLaunches = launchMgr.getDebugTargets();

		/* Get the list of launches */
		for (IDebugTarget dbgTarget : registeredLaunches) {
			dbgTargetLaunchName = null;
			IDebugTarget target = dbgTarget.getDebugTarget();
			if (target != null) {
				ILaunch launch = target.getLaunch();
				if (launch == null) {
					continue;
				}
				ILaunchConfiguration config = launch
						.getLaunchConfiguration();
				if (config == null || config.getName() == null) {
					continue;
				}

				dbgTargetLaunchName = config.getName();
			}

			if (launchName.equalsIgnoreCase(dbgTargetLaunchName)) {
				suspended = dbgTarget.isSuspended();
				terminated = dbgTarget.isTerminated();

				if (suspended) {
					enableDelete = false;
					enableRename = false;
					enableComments = false;
				} else if (!terminated) {
					// running
					enableCopy = false;
					enableDelete = false;
					enableRename = false;
					enableComments = false;
				}
				break;
			}
		}

		if (enableRename) {
			String storageDir = StorageUtils.getStorageDir(storageCfgFile);
			String baseFileName = StorageUtils.getBaseFileName(storageCfgFile);
			enableRename = TraceResultsLock.tryLock(storageDir, baseFileName);
			if (enableRename) {
				TraceResultsLock.releaseLock(storageDir, baseFileName);
			}
		}

		copyData.setEnabled(enableCopy);
		deleteData.setEnabled(enableDelete);
		renameData.setEnabled(enableRename);
		editComments.setEnabled(enableComments);

	} // enableActions

	private void copyCell() {
		if (m_focusCellManager.getFocusCell() != null) {
			StringSelection cell = new StringSelection(m_focusCellManager
					.getFocusCell().getText());

			Toolkit.getDefaultToolkit().getSystemClipboard()
					.setContents(cell, new ClipboardOwner() {
						public void lostOwnership(Clipboard c, Transferable t) {
						}
					});
		}
	}

	private void copyLine() {
		StringBuilder lines = new StringBuilder(); //$NON-NLS-1$

		TreeItem[] selectedItems = m_dataTable.getSelection();

		for (TreeItem item : selectedItems) {

			// / Put a new-line character between lines
			if (lines.length() != 0) {
				lines.append('\n');
			}
			for (int i = 0; i < m_dataTable.getColumnCount(); i++) {
				if (i > 0) {
					lines.append('\t');
				}
				if (m_dataTable.getColumn(m_dataTable.getColumnOrder()[i])
						.getWidth() > 0) {
					lines.append(item.getText(m_dataTable.getColumnOrder()[i]));
				}
			}

			StringSelection text = new StringSelection(lines.toString());

			Toolkit.getDefaultToolkit().getSystemClipboard()
					.setContents(text, new ClipboardOwner() {
						public void lostOwnership(Clipboard c, Transferable t) {
						}
					});
		}
	}

	private void backupSelectedResults() {
		TreeItem[] selection = m_dataTable.getSelection();
		TableRecordData line = null;

		for (TreeItem item : selection) {
			TableRecordData recordData = (TableRecordData) item.getData();

			if (item.getParentItem() == null) {
				// this is a root. should never have gotten here, so return
				return;
			} else {
				line = (TableRecordData) item.getData();
				if (line.isTraceConfig) {
					// we should never land here because the menu item is
					// disabled for this case
					return;
				}
			}

			String baseFileName = CommonConstants.EMPTY_STRING;
			String storageDir = CommonConstants.EMPTY_STRING;
			// get the results files
			try {
				TreeItem traceConfigItem = item.getParentItem();
				TableRecordData projectItem = null;
				if (traceConfigItem != null) {
					projectItem = (TableRecordData) traceConfigItem
							.getParentItem().getData();
				}

				if (projectItem != null) {
				}

				baseFileName = recordData.cells.get(0).value;
			} catch (NullPointerException ex) {
				ex.printStackTrace();
				LOGGER.error("[backupSelectedResults] Error while getting results files"
						+ ex.getMessage());
			}

			storageDir = StorageUtils.getStorageDir(recordData.storageCfgFile);
			
			if (storageDir == null) {
				return;
			}

			/*
			 * Make a clone if it's a folder resulted after unpacking an *cwzsa, *.kcwzsa, *.scwzsa
			 */
			String nextFileIndex = CommonConstants.EMPTY_STRING;
			
			File storageDirFile = new File(storageDir);
			if (storageDirFile.exists() && storageDirFile.isDirectory()) {
				String parentDir = storageDirFile.getParent();
				String newStorageDir = null;
				if (storageDirFile.getName().matches("\\d{14}")) { //$NON-NLS-1$
					BigInteger cloneDirName = new BigInteger(
							storageDirFile.getName());
					/* Generate a new valid name for clone */
					while (new File(parentDir, cloneDirName.toString())
							.exists()) {
						cloneDirName = cloneDirName.add(BigInteger.ONE);
					}
					newStorageDir = parentDir + Path.SEPARATOR + cloneDirName;
				} else if (!storageDirFile.getName().equals(
						SaConstants.analysis_results_directory)) {
					nextFileIndex = StorageUtils.nextFileIndex(parentDir, baseFileName, CommonConstants.EMPTY_STRING, false);
					newStorageDir = parentDir
							+ Path.SEPARATOR
							+ baseFileName
							+ nextFileIndex;
				}

				Utils.copyFolder(storageDirFile, new File(newStorageDir));

				renameResults(
						baseFileName,
						baseFileName + nextFileIndex,
						new File(newStorageDir), false);
			} else {
				renameResults(
						baseFileName,
						baseFileName + nextFileIndex,
						new File(storageDir), true);
			}
			refresh();
		}
	}

	private void deleteSelectedResults() {
		TreeItem[] selection = m_dataTable.getSelection();
		TableRecordData line = null;

		for (TreeItem item : selection) {
			TableRecordData recordData = (TableRecordData) item.getData();

			if (item.getParentItem() == null) {
				// this is a root. should never have gotten here, so return
				return;
			} else {
				line = (TableRecordData) item.getData();
				if (line.isTraceConfig) {
					// we should never land here because the menu item is
					// disabled for this case
					return;
				}
			}

			String baseFileName = CommonConstants.EMPTY_STRING;
			String storageDir = CommonConstants.EMPTY_STRING;

			// get the results files
			try {
				baseFileName = recordData.cells.get(0).value;
			} catch (NullPointerException ex) {
				LOGGER.error(
						"[deleteSelectedResults] Error getting base filename.",
						ex);
			}

			storageDir = StorageUtils.getStorageDir(recordData.storageCfgFile);

			try {
				StorageUtils.deleteResults(storageDir, baseFileName);
			} catch (RuntimeException ex) {
				ex.printStackTrace();
				LOGGER.error("[deleteSelectedResults] Error deleting results.",
						ex);
			}

			refresh();
		}
	}

	/**
	 * Rename the results currently selected, if the new name is valid.
	 */
	synchronized void renameSelectedResults() {
		TreeItem[] selection = m_dataTable.getSelection();

		for (TreeItem item : selection) {
			TableRecordData recordData = (TableRecordData) item.getData();

			if (recordData == null) {
				continue;
			}
			
			if (recordData.storageCfgFile == null) {
				continue;
			}
			final String storageDir = StorageUtils
					.getStorageDir(recordData.storageCfgFile);
			final String baseFileName = StorageUtils
					.getBaseFileName(recordData.storageCfgFile);
			boolean canRename = TraceResultsLock.tryLock(storageDir,
					baseFileName);
			if (!canRename) {
				Display.getDefault().syncExec(new Runnable() {
					public void run() {
						MessageDialog
								.openWarning(
										Display.getDefault().getActiveShell(),
										Messages.TraceDataExporter_FailLockTitle,
										MessageFormat
												.format(Messages.TraceDataExporter_FailLockMessage,
														baseFileName,
														storageDir,
														Messages.Flatdata_RenameResults));
					}
				});
				continue;
			}
			for (TableCellData cell : recordData.cells) {
				if (cell.isResultsCell()) {
					String storageFolderPath = StorageUtils
							.getStorageDir(recordData.storageCfgFile);
					File storageFolder = new File(storageFolderPath);
					if (storageFolder.exists() && storageFolder.isDirectory()) {
						addEditorForResultsCell(item, cell, storageFolder);
					} else {
						LOGGER.error("[renameSelectedResults]: invalid results folder"); //$NON-NLS-1$
					}
					break; // should be only one cell that contains the results
							// name.
				}
			}
		}
	}
	
	/**
	 * Rename the results currently selected, if the new name is valid.
	 */
	void editSelectedComment() {
		TreeItem[] selection = m_dataTable.getSelection();

		for (TreeItem item : selection) {
			TableRecordData recordData = (TableRecordData) item.getData();

			if (recordData == null) {
				continue;
			}
			
			int cellIndex = -1;
			for (TableCellData cell : recordData.cells) {
				cellIndex++;
				if (cell.isCommentsCell()) {
					String storageFolderPath = StorageUtils
							.getStorageDir(recordData.storageCfgFile);
					File storageFolder = new File(storageFolderPath);
					if (storageFolder.exists() && storageFolder.isDirectory()) {
						addEditorForCommentsCell(item, cell, storageFolder, cellIndex);
					} else {
						LOGGER.error("[editComment]: invalid results folder"); //$NON-NLS-1$
					}
					break; // should be only one cell that contains the results
							// name.
				}
			}
		}
	}
	
	/**
	 * Rename the results currently selected, if the new name is valid.
	 */

	/**
	 * Flat Data implementation of viewer sorter
	 */
	private class FlatDataSorter extends ViewerSorter {

		@Override
		public int compare(Viewer viewer, Object e1, Object e2) {
			TableRecordData data1 = (TableRecordData) e1;
			TableRecordData data2 = (TableRecordData) e2;
			TreeColumn sortColumn = m_dataTable.getSortColumn();
			int sortDirection = m_dataTable.getSortDirection();
			int sortColumnIndex = 0;

			// / Find the index of the sort column
			if (sortColumn == null) {
				return 0;
			}
			for (TreeColumn column : m_dataTable.getColumns()) {
				if (column == sortColumn) {
					break;
				}
				sortColumnIndex++;
			}

			TableCellConfiguration cell1 = m_currentTableRecordsConfigurations
					.get(data1.recordConfigurationName).cells
					.get(sortColumnIndex);
			TableCellConfiguration cell2 = m_currentTableRecordsConfigurations
					.get(data2.recordConfigurationName).cells
					.get(sortColumnIndex);
			int compare = 0;

			if (sortDirection == SWT.NONE || !cell1.allowSort
					|| !cell2.allowSort) {
				return 0;
			}

			// / When the cells have different types, compare them as strings
			if (!cell1.type.equals(cell2.type) || cell1.type.equals("string")) { //$NON-NLS-1$
				compare = getString(data1.cells.get(sortColumnIndex).value)
						.compareTo(
								getString(data2.cells.get(sortColumnIndex).value));
			} else if (cell1.type.equals("integer")) { //$NON-NLS-1$
				compare = Integer.valueOf(
						getInt(data1.cells.get(sortColumnIndex).value))
						.compareTo(
								Integer.valueOf(getInt(data2.cells
										.get(sortColumnIndex).value)));
			} else if (cell1.type.equals("float")) { //$NON-NLS-1$
				compare = Double.valueOf(
						getDouble(data1.cells.get(sortColumnIndex).value))
						.compareTo(
								Double.valueOf(getDouble(data2.cells
										.get(sortColumnIndex).value)));
			}

			if (sortDirection == SWT.UP) {
				return compare;
			} else if (sortDirection == SWT.DOWN) {
				return -compare;
			} else {
				return 0;
			}
		}
	}

	private int computeMaximumValueOfColumn(String recordType, int column) {
		int max = 0;

		for (TableRecordData data : m_currentTableData.records) {
			if (data.recordConfigurationName.equals(recordType)
					&& data.cells.size() > column) {
				max = Math.max(max, getInt(data.cells.elementAt(column).value));
			}
		}
		return max;
	}

	private double getDouble(String text) {
		if (text == null || text.isEmpty()) {
			return 0;
		}

		text = text.trim();

		if (text == null || text.isEmpty()) {
			return 0;
		}

		try {
			double value = Double.parseDouble(text);

			return value;
		} catch (NumberFormatException e) {
			return 0;
		}
	}

	private int getInt(String text) {

		int value = 0;

		if (text == null || text.isEmpty()) {
			return value;
		}

		text = text.trim();

		if (text == null || text.isEmpty()) {
			return value;
		}

		try {
			value = Integer.parseInt(text, 10);
			return value;
		} catch (NumberFormatException e10) {
			try {
				if (text.length() > 2) {
					value = Integer.parseInt(text.substring(2), 16);
				}
				return value;
			} catch (NumberFormatException e16) {
				return value;
			}
		}
	}

	private String getString(String text) {

		if (text == null || text.isEmpty()) {
			return CommonConstants.EMPTY_STRING;
		}

		return text;
	}

	private void disposeTreeEditors() {

		Collection<Button> c = m_buttons.values();
		Iterator<Button> itr = c.iterator();

		while (itr.hasNext()) {
			Button b = (Button) itr.next();
			try {
				b.dispose();
			} catch (Exception e) {
				// Sometimes this dispose throws an error : Graphics is
				// disposed.
				LOGGER.error(e);
			}
		}
		m_buttons.clear();

		if (m_treeEditors != null) {

			for (TreeEditor tEditor : m_treeEditors) {
				if (tEditor == null) {
					continue;
				}
				if (tEditor.getEditor() != null) {
					tEditor.getEditor().dispose();
				}
				tEditor.dispose();
			}
			m_treeEditors.clear();
		}
	}

	private void treeChanged(final TreeItem[] items) {
		if (items == null) {
			disposeTreeEditors();

			m_treeEditors = new Vector<TreeEditor>();
			m_buttons = new HashMap<ButtonsProperties, Button>();
			treeChanged(m_dataTable.getItems());

			m_dataTable.addDisposeListener(new DisposeListener() {
				public void widgetDisposed(DisposeEvent e) {
					disposeTreeEditors();
				}
			});

			return;
		}

		m_form.getBody().layout();
	}

	/**
	 * Adds an editor for a cell that contains the profiling results name.
	 * 
	 * @param item
	 *            the current tree item.
	 * @param cell
	 *            the current cell.
	 * @param cellIndex
	 *            the current cell index.
	 * 
	 * @NOTE package access level should be kept.
	 */
	void addEditorForResultsCell(final TreeItem item, TableCellData cell,
			final File storageFolder) {

		boolean terminated = true;

		final String oldText = item.getText();

		TreeEditor editor = new TreeEditor(m_dataTable);
		editor.horizontalAlignment = SWT.LEFT;
		editor.grabHorizontal = true;
		m_treeEditors.add(editor);

		final Text text = new Text(m_dataTable, SWT.NONE);
		text.setText(item.getText());
		text.selectAll();
		text.setFocus();

		/* Get the list of launches */
		ILaunchManager launchMgr = DebugPlugin.getDefault().getLaunchManager();

		/* Get the list of registered launches */
		IDebugTarget[] registeredLaunches = launchMgr.getDebugTargets();

		/* Get the list of launches */
		for (IDebugTarget dbgTarget : registeredLaunches) {
			ILaunch launch = dbgTarget.getDebugTarget().getLaunch();
			if (launch == null) {
				continue;
			}
			
			ILaunchConfiguration config = dbgTarget.getDebugTarget().getLaunch()
					.getLaunchConfiguration();
			if (config == null || config.getName() == null) {
				continue;
			}
			String dbgTargetLaunchName = config.getName();
			if (cell.getLaunchName().equalsIgnoreCase(dbgTargetLaunchName)) {
				terminated = dbgTarget.isTerminated();
				break;
			}
		}

		// If the debug session is not terminated you cannot rename the results.
		final boolean canRename = terminated;

		Listener releaseLock = new Listener () {
            @Override
            public void handleEvent(Event event) {
                // Lock for old file name must be released.
                TraceResultsLock.releaseLock(storageFolder.getAbsolutePath(), oldText);
            }
        };
        text.addListener(SWT.Dispose, releaseLock);
        text.addListener(SWT.FocusOut, releaseLock);

		text.addKeyListener(new KeyAdapter() {
			public void keyPressed(KeyEvent event) {
				switch (event.keyCode) {
				case (SWT.CR):
					// Intentional fall through.
				case (SWT.KEYPAD_CR):
					if (canRename) {
						updateResultsName(item, text.getText(), storageFolder);
					} else {
						ConfigurationUtils.showErrorMessage(
								Messages.flatdata_cannot_rename_res_title,
								Messages.flatdata_cannot_rename_res_name);
					}
					// Intentional fall through.
				case SWT.ESC:
					text.dispose();
					// Lock for old file name must be released.
					// Expected keys to release lock are: Enter, Enter from
					// numeric keypad and Escape.
					TraceResultsLock.releaseLock(
							storageFolder.getAbsolutePath(), oldText);
					break;
				default:
					break;
				}

			}
		});
		editor.setEditor(text, item);
	}
	
	/**
	 * Adds an editor for a cell that contains the profiling results name.
	 * 
	 * @param item
	 *            the current tree item.
	 * @param cell
	 *            the current cell.
	 * @param cellIndex
	 *            the current cell index.
	 * 
	 * @NOTE package access level should be kept.
	 */
	void addEditorForCommentsCell(final TreeItem item, TableCellData cell,
			final File storageFolder, int cellIndex) {
		final String oldText = cell.text;

		TreeEditor editor = new TreeEditor(m_dataTable);
		editor.horizontalAlignment = SWT.LEFT;
		editor.grabHorizontal = true;
		m_treeEditors.add(editor);

		final Text text = new Text(m_dataTable, SWT.NONE);
		text.setText(cell.text);
		text.selectAll();
		text.setFocus();

        Listener releaseLock = new Listener () {
            @Override
            public void handleEvent(Event event) {
                // Lock for old file name must be released.
                TraceResultsLock.releaseLock(storageFolder.getAbsolutePath(), oldText);
            }
        };
        text.addListener(SWT.Dispose, releaseLock);
        text.addListener(SWT.FocusOut, releaseLock);

		text.addKeyListener(new KeyAdapter() {
			public void keyPressed(KeyEvent event) {
				switch (event.keyCode) {
				case (SWT.CR):
					// Intentional fall through.
				case (SWT.KEYPAD_CR):
					updateComment(item, text.getText(), storageFolder);
					// Intentional fall through.
				case SWT.ESC:
					text.dispose();
					// Lock for old file name must be released.
					// Expected keys to release lock are: Enter, Enter from
					// numeric keypad and Escape.
					TraceResultsLock.releaseLock(
							storageFolder.getAbsolutePath(), oldText);
					break;
				default:
					break;
				}

			}
		});
		editor.setEditor(text, item, cellIndex);
	}

	/**
	 * Rename the set of analysis results with the given base name.
	 * 
	 * @param oldName
	 *            the current name of the results set.
	 * @param newName
	 *            the new name for the results set.
	 * @param storageFolder
	 *            the folder that stores sa results.
	 */
	private void renameResults(String oldName, String newName,
			File storageFolder, boolean keepOldFiles) {

		// Load all the files that need to be renamed (all
		// the files belonging to the results set with the given name).
		File[] allStoredFiles = storageFolder.listFiles();
		List<File> filesToRename = new LinkedList<File>();

		for (File storedFile : allStoredFiles) {
			String extension = StorageUtils.getFileExtension(storedFile);
			
			if (extension == null) {
				if (storedFile.getName().equals(oldName)) {
					filesToRename.add(storedFile);
				}
				
				continue;
			}
			
			if (storedFile.getName().equals(oldName + "." + extension)) { //$NON-NLS-1$
				filesToRename.add(storedFile);
			}

			if (extension.equals(UIConstants.ELF_EXT)) {
				// Get the core index
				String baseName = StorageUtils.getBaseFileName(storedFile);
				String coreIndex = baseName.substring(baseName.length() - 1);
				if (storedFile.getName().equals(
						oldName + "_core" + coreIndex + "." + extension)) { //$NON-NLS-1$
					filesToRename.add(storedFile);

					// Update in decoder config xml
					String newFilePath = storageFolder.getAbsolutePath()
							+ IPath.SEPARATOR + newName
							+ "_core" + coreIndex + "." + extension; //$NON-NLS-1$ //$NON-NLS-2$

					LSConfigLoader_JNI confLoader;
					String decoderConfigFile = storageFolder.getAbsolutePath()
							+ IPath.SEPARATOR + oldName
							+ UIConstants.DECODER_CONF_EXT;
					File decoderConf = new File(decoderConfigFile);
					if (!decoderConf.exists()) {
						LOGGER.error("[renameResults] Decoder configuration xml does not exist: " + decoderConfigFile + "."); //$NON-NLS-1$ //$NON-NLS-2$
						return;
					}
					confLoader = new LSConfigLoader_JNI(decoderConfigFile);
					boolean status = confLoader.setValue( // try save it for older ARM
							"Trace Generators:Core " + coreIndex //$NON-NLS-1$
									+ ":" + "Target Images" + ":" + "Image 0", //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
							AttributeType.FILE_ATTR.value(), newFilePath);
					if (!status) { // if couldn't save it, try for ARMv8
						status = confLoader.setValue(
								"Trace Generators:CORE:Core " + coreIndex //$NON-NLS-1$
										+ ":" + "Target Images" + ":" + "Image 0", //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
								AttributeType.FILE_ATTR.value(), newFilePath);
					}
					status = status
							&& confLoader.saveAsConfigToXml(decoderConfigFile);
					if (!status) {
						LOGGER.error("[renameResults] Failed to save new decoder configuration version to " + decoderConfigFile + "."); //$NON-NLS-1$ //$NON-NLS-2$
					}
				}
			}
		}

		// Rename files.
		for (File fileToRename : filesToRename) {
			String extension = StorageUtils.getFileExtension(fileToRename);

			String newFilePath = null;
			if (extension == null) {
				newFilePath = storageFolder.getAbsolutePath()
						+ IPath.SEPARATOR + newName;
				extension = CommonConstants.EMPTY_STRING;
			} else {
				newFilePath = storageFolder.getAbsolutePath()
						+ IPath.SEPARATOR + newName + "." + extension; //$NON-NLS-1$				
			}

			if (extension.equals(UIConstants.ELF_EXT) && fileToRename.getName().contains("_core")) { //$NON-NLS-1$
				// Get the core index
				String baseName = StorageUtils.getBaseFileName(fileToRename);
				String coreIndex = baseName.substring(baseName.length() - 1);
				
				String oldElfName = oldName + "_core" + coreIndex; //$NON-NLS-1$
				if (baseName.equals(oldElfName)) {
					newFilePath = storageFolder.getAbsolutePath() + IPath.SEPARATOR
							+ newName + "_core" + coreIndex + "." + extension; //$NON-NLS-1$ //$NON-NLS-2$
				}
			}

			File newFile = new File(newFilePath);
			if (keepOldFiles) {
				try {
					StorageUtils.copyFile(fileToRename, newFile);
				} catch (Exception e) {
					LOGGER.error("[renameResults]: Copy operation of executable failed:" //$NON-NLS-1$
							+ fileToRename.getAbsoluteFile());
				}
				continue;
			} else {
				boolean success = fileToRename.renameTo(newFile);
				if (!success) {
					LOGGER.warn("[renameResults]: error renaming file: " //$NON-NLS-1$
							+ fileToRename.getAbsoluteFile()
							+ ", performing copy operation instead"); //$NON-NLS-1$
					try {
						StorageUtils.copyFile(fileToRename, newFile);
					} catch (Exception e) {
						LOGGER.error("[renameResults]: Copy operation of executable failed:" //$NON-NLS-1$
								+ fileToRename.getAbsoluteFile());
					}
					continue;
				} else {
					for (IEditorReference edref : PlatformUI.getWorkbench()
							.getActiveWorkbenchWindow().getActivePage()
							.getEditorReferences()) {
						try {
							String name = edref.getEditorInput().getName();
							if (name.equals(fileToRename.getName())) {
								PlatformUI
										.getWorkbench()
										.getActiveWorkbenchWindow()
										.getActivePage()
										.closeEditor(edref.getEditor(false),
												false);
							}
						} catch (PartInitException e) {
							LOGGER.error(
									"[renameResults] Error while closin editor windows",
									e);
						}
					}
				}
			}
		}
		
		//Rename the folder
		String parentFolderName = storageFolder.getParentFile().getAbsolutePath();
		File newfolderName = new File(parentFolderName + IPath.SEPARATOR + newName);
		
		boolean success = storageFolder.renameTo(newfolderName);
		
		if (!success) {
			LOGGER.warn("[renameResults]: error renaming folder: " //$NON-NLS-1$
					+ newfolderName.getAbsoluteFile());
		}

	}

	/**
	 * Updates the name of the profiling results if the newly set name is valid.
	 * 
	 * @param treeItem
	 *            the tree item corresponding to the name of the profiling
	 *            results.
	 * @param newName
	 *            the new name to set for the profiling results.
	 * @param storageFolderPath
	 *            The sa results folder.
	 */
	private void updateResultsName(TreeItem treeItem, String newName,
			File storageFolder) {
		boolean validText = checkResultsName(newName, storageFolder);
		if (validText) {
			String oldName = treeItem.getText();
			// Close the opened trace editor of this application
			StorageUtils.closeViews(oldName, storageFolder.getAbsolutePath());
			
			renameResults(oldName, newName, storageFolder, false);
			treeItem.setText(newName);
			refresh();
		}
	}
	
	/**
	 * Updates the name of the profiling results if the newly set name is valid.
	 * 
	 * @param treeItem
	 *            the tree item corresponding to the name of the profiling
	 *            results.
	 * @param newName
	 *            the new name to set for the profiling results.
	 * @param storageFolderPath
	 *            The sa results folder.
	 */
	private void updateComment(TreeItem treeItem, String newComment,
			File storageFolder) {
		LSConfigLoader_JNI confLoader;
		String resultsConfigXml = storageFolder+ "/" + treeItem.getText() + "." + SaConstants.raw_trace_config_extension;
		
		File resConfig = new File(resultsConfigXml);
		
		if (resConfig.exists()) {
			long collectedDate = resConfig.lastModified();
			confLoader = new LSConfigLoader_JNI(resultsConfigXml);
			boolean status = confLoader.setValue("S32DS" //$NON-NLS-1$
					+ ":" + "Comment", //$NON-NLS-1$ //$NON-NLS-2$
					AttributeType.STRING_ATTR.value(), newComment); //$NON-NLS-1$
			status = status
					&& confLoader.saveAsConfigToXml(resultsConfigXml);
			if (!status) {
				LOGGER.error("[editComments] Failed to save new decoder configuration version to " + resultsConfigXml + "."); //$NON-NLS-1$ //$NON-NLS-2$
			}
			
			resConfig.setLastModified(collectedDate);
			
			refresh();
		}
	}

	/**
	 * @param resName
	 *            The name of the SA results.
	 * @param storageFolder
	 *            The sa results folder.
	 * @return true if the name is valid, false otherwise.
	 */
	private boolean checkResultsName(String resName, File storageFolder) {
		// \w - A word character, short for [a-zA-Z_0-9].
		// \S - A non-whitespace character.
		if (resName.matches("[\\w\\S&&[^/:\"*?<>|\\\\]]+")) { //$NON-NLS-1$
			// Check if there is a set of results with the same name (not
			// allowed).
			File newfolderName = new File(storageFolder.getParentFile(), resName);			
			if (configFileExists(resName, storageFolder) || (newfolderName.exists() && newfolderName.isDirectory())) {
				ConfigurationUtils.showErrorMessage(
						Messages.flatdata_wrong_res_name_title,
						Messages.flatdata_res_name_exists_message);
				return false;
			}
		} else {
			ConfigurationUtils.showErrorMessage(
					Messages.flatdata_wrong_res_name_title,
					Messages.flatdata_res_name_invalid_message);
			return false;
		}

		if ((storageFolder.toString().length() + resName.length()) > CommonConstants.MAX_LENGTH) {
			ConfigurationUtils.showErrorMessage(
					Messages.flatdata_wrong_res_name_title,
					Messages.flatdata_res_name_too_long_message);
			return false;
		}

		return true;

	}

	/**
	 * Checks if a set of results with the given name is stored in the given
	 * location.
	 * 
	 * @param resName
	 *            The name of the sa results.
	 * @param storageFolder
	 *            the location of the results folder.
	 * @return true if a set of results with the given name exists, false
	 *         otherwise.
	 */
	private boolean configFileExists(String resName, File storageFolder) {

		File[] rawTraceCfgFiles = storageFolder.listFiles(new CustomFileFilter(
				SaConstants.raw_trace_config_extension));

		for (int i = 0; i < rawTraceCfgFiles.length; i++) {
			if (rawTraceCfgFiles[i].getName().compareTo(
					resName + "." + SaConstants.raw_trace_config_extension) == 0) { //$NON-NLS-1$
				ConfigurationUtils.showErrorMessage(
						Messages.flatdata_wrong_res_name_title,
						Messages.flatdata_res_name_already_exists_message);

				return true;
			}
		}

		return false;
	}

	/**
	 * The content provider class is responsible for providing objects to the
	 * view. It can wrap existing objects in adapters or simply return objects
	 * as-is. These objects may be sensitive to the current input of the view,
	 * or ignore it and always show the same content (like Task List, for
	 * example).
	 */

	private class ViewContentProvider implements ITreeContentProvider {

		public void inputChanged(Viewer v, Object oldInput, Object newInput) {
			m_tableData = (TableData) newInput;
		}

		public void dispose() {
		}

		public Object[] getElements(Object inputElement) {
			return getChildren(inputElement);
		}

		public Object[] getChildren(Object parent) {
			Vector<TableRecordData> children = new Vector<TableRecordData>();
			int parentIndex = -1;

			if (parent != m_tableData) {
				parentIndex = ((TableRecordData) parent).index;
			}

			for (TableRecordData data : m_tableData.records) {
				if (data != parent && data.parentIndex == parentIndex) {
					children.add(data);
				}
			}

			return children.toArray();
		}

		public Object getParent(Object element) {

			if (element == m_tableData) {
				return null;
			}

			int parentIndex = ((TableRecordData) element).parentIndex;
			for (TableRecordData data : m_tableData.records) {
				if (data != element && data.index == parentIndex) {
					return data;
				}
			}
			return null;
		}

		public boolean hasChildren(Object element) {
			int parentIndex = ((TableRecordData) element).index;

			if (parentIndex == -1) {
				return false;
			}

			for (TableRecordData data : m_tableData.records) {
				if (data != element && data.parentIndex == parentIndex) {
					return true;
				}
			}

			return false;
		}
	}

	public void dispose() {
		m_dataProvider.closeProvider();
		m_dataProvider.removeListener(TraceResultsFormDrawer.this);
		m_toolkit.dispose();

		getWorkspace().removeResourceChangeListener(this);

		disposeTreeEditors();
	}

	public Button getStartStopButton(String launchName) {
		ButtonsProperties bp = new ButtonsProperties();
		bp.setAttributes(startStopTraceButtonType, launchName);
		return m_buttons.get(bp);
	}

	public Button getResetButton(String launchName) {
		ButtonsProperties bp = new ButtonsProperties();
		bp.setAttributes(resetTraceButtonType, launchName);
		return m_buttons.get(bp);
	}


	/**
	 * @return the data tree table.
	 * 
	 * @NOTE package access level should be kept.
	 */
	Tree getTree() {
		return m_dataTable;
	}

}
