/*******************************************************************************
 * Copyright (c) 2015 - 2016 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.timeline;


import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;
import java.util.Vector;

import org.apache.log4j.Logger;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.MouseTrackAdapter;
import org.eclipse.swt.events.MouseWheelListener;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.FontMetrics;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Slider;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;



import com.freescale.sa.ui.common.IConfigurableDisplay;
import com.freescale.sa.ui.timeline.UIMessages;


/**
 * Plots an input vector onto a canvas
 * Answers to mouse and keyboard events
 * 
 * The plot is divided into these sections:
 *  - the left most part is for displaying functions/groups names
 *    It is divided by a sash from the body, in order for it to be resized
 *  - the upper part is dedicated to the ruler, which displays the current
 *    horizontal coordinates (time-stamps perhaps) and the coordinates of the
 *    cursors
 *  - the right-most part is dedicated to the plot's menu buttons
 *  - the body is surrounded by two canvases, one below and another to the right,
 *     that will be sued for scroll-bars. These canvases can be hidden when 
 *     the scroll bars are not visible
 * 
 * @author B11543
 *
 */ 

class DropdownSelectionListener extends SelectionAdapter {
	  private ToolItem dropdown;

	  private Menu menu;

	  public DropdownSelectionListener(ToolItem dropdown) {
	    this.dropdown = dropdown;
	    menu = new Menu(dropdown.getParent().getShell());
	  }

	  public void add(String item) {
	    MenuItem menuItem = new MenuItem(menu, SWT.NONE);
	    menuItem.setText(item);
	    menuItem.addSelectionListener(new SelectionAdapter() {
	      public void widgetSelected(SelectionEvent event) {
	        MenuItem selected = (MenuItem) event.widget;
	        dropdown.setText(selected.getText());
	      }
	    });
	  }

	  public void widgetSelected(SelectionEvent event) {
	    if (event.detail == SWT.ARROW) {
	      ToolItem item = (ToolItem) event.widget;
	      Rectangle rect = item.getBounds();
	      Point pt = item.getParent().toDisplay(new Point(rect.x, rect.y));
	      menu.setLocation(pt.x, pt.y + rect.height);
	      menu.setVisible(true);
	    }
	  }
	}

public class TraceTimelinePlot {
	
	private SashForm m_plotSashForm; 
	private Canvas m_mainCanvas;
	private Canvas m_canvas;
	private Canvas m_rulerCanvas;
	private Canvas m_yLabelsCanvas;
	private ToolBar m_toolBar;
	public boolean initialized = false;
	/**
	 * The ruler height in pixels:
	 * 20 pixels for displaying the general coordinates
	 * 10 pixels for displaying the ticks
	 * 20 pixels for the reference cursor's coordinate
	 * 20 pixels for displaying the cursor's coordinate
	 */
	private int m_rulerHeight = 70;
	
	private int m_sliderSize = 15;
	
	// / Zoom-in to no more than 100 pixels per clock cycle
	private static final float m_maximumZoom = 100;
	
	private LongSlider m_sliderX;
	private Slider m_sliderY;
	
	private boolean m_selectNotZoom = true;
	
	private int  m_pointHeight = 20;
	private boolean m_drawGrid = true, m_reducedPointHeight = true;
	private final static Integer COLOR_INLINE_INDEX = 0x100;

	
	/* System color id constants used by the timeline plot.
	 * When changing them or adding new constants, please also update 
	 * createColor(Display device, int colorIdx).   
	 */
	private static final int BACKGROUND_COLOR = SWT.COLOR_BLACK, 
			POINT_COLOR = SWT.COLOR_GREEN, 
			GRID_LINE_COLOR = SWT.COLOR_WHITE,
			RULER_COLOR = SWT.COLOR_GREEN,
			CURSOR_1_COLOR = SWT.COLOR_YELLOW,
			CURSOR_2_COLOR = SWT.COLOR_RED,
			DIFFERENCE_COLOR = SWT.COLOR_WHITE,
			Y_LABELS_BACKGROUND_COLOR = SWT.COLOR_DARK_GRAY,
			Y_LABELS_TEXT_COLOR = SWT.COLOR_WHITE,
			RESEVED_COLOR = SWT.COLOR_WIDGET_LIGHT_SHADOW,
			SPECIAL_GROUP_COLOR = SWT.COLOR_GRAY,
			IDLE_GROUP_COLOR = SWT.COLOR_CYAN,
			INLINE_GROUP_COLOR = COLOR_INLINE_INDEX,
			MAX_ADJUST_CURSOR = 7;
	
	/* 
	 * normally colors are in ascending order, but use a hashmap anyway so that 
	 * we won't depend on this particularity
	 */
	@SuppressWarnings("serial")
	private static final HashMap<Integer, RGB> m_RGBMap = new HashMap<Integer, RGB>() {{
		put(SWT.COLOR_BLACK, new RGB(0, 0, 0));
		put(SWT.COLOR_DARK_RED, new RGB(128, 0, 0));
		put(SWT.COLOR_DARK_GREEN, new RGB(0, 128, 0));
		put(SWT.COLOR_DARK_YELLOW, new RGB(128, 128, 0));
		put(SWT.COLOR_DARK_BLUE, new RGB(0, 0, 128));
		put(SWT.COLOR_DARK_MAGENTA, new RGB(128, 0, 128));
		put(SWT.COLOR_DARK_CYAN, new RGB(0, 128, 128));
		put(SWT.COLOR_GRAY,	new RGB(192, 192, 192));
		put(SWT.COLOR_DARK_GRAY, new RGB(128, 128, 128));
		put(SWT.COLOR_RED, new RGB(255, 0, 0));
		put(SWT.COLOR_GREEN, new RGB(0, 255, 0));
		put(SWT.COLOR_YELLOW, new RGB(255, 255, 0));
		put(SWT.COLOR_BLUE, new RGB(0, 0, 255));
		put(SWT.COLOR_MAGENTA, new RGB(255, 0, 255));
		put(SWT.COLOR_CYAN, new RGB(174, 182, 64)); // custom color
		put(SWT.COLOR_WHITE, new RGB(255, 255, 255));
		put(COLOR_INLINE_INDEX , new RGB(75, 75, 180));
	}};
	
	private long m_cursor1X = -1, m_cursor2X = -1;
	
	private static Logger LOGGER = Logger.getLogger(TraceTimelinePlot.class);
	
	@SuppressWarnings("serial")
	private static final HashMap<RGB, Color> m_colorMap = new HashMap<RGB, Color>() {{
		for (RGB rgb : m_RGBMap.values()) {
			Color color = createColor(Display.getCurrent(), rgb);
			put(rgb, color);
		}
	}};
	
	private static final Map<String, String> SHORT_TIME_NOTATION = new HashMap<String, String>() {
		private static final long serialVersionUID = -7041801848713112487L;

		{
			put(IConfigurableDisplay.TIME_UNIT_CYCLES,
					UIMessages.TraceTimeline_Short_Notation_Cycles);
			put(IConfigurableDisplay.TIME_UNIT_MICROSEC,
					UIMessages.TraceTimeline_Short_Notation_Microseconds);
			put(IConfigurableDisplay.TIME_UNIT_MILLISEC,
					UIMessages.TraceTimeline_Short_Notation_Milliseconds);
			put(IConfigurableDisplay.TIME_UNIT_NANOSEC,
					UIMessages.TraceTimeline_Short_Notation_Nanoseconds);
		}
	};

	public void forceRedraw() {
		if (m_mainCanvas != null) {
			if (!m_mainCanvas.isDisposed())
				m_mainCanvas.redraw();
		}		
		if (m_rulerCanvas != null) {
			if (!m_rulerCanvas.isDisposed())
				m_rulerCanvas.redraw();
		}
		if (m_canvas != null) {
			if (!m_canvas.isDisposed())
				m_canvas.redraw();
		}
		if (m_yLabelsCanvas != null) {
			if (!m_yLabelsCanvas.isDisposed())
				m_yLabelsCanvas.redraw();
		}
	}

	private static Color getColor(int colorIdx) {
		Color color = null;
		
		RGB rgb = m_RGBMap.get(colorIdx);
		if (rgb != null && m_colorMap.containsKey(rgb)) {
			color = m_colorMap.get(rgb);

			// this should never happen
			if (color.isDisposed() || (color.getDevice() == null)) {
				color = null;
				m_colorMap.remove(rgb);
			}
			
			if (color == null) {
				color = createColor(Display.getCurrent(), colorIdx);
				m_colorMap.put(rgb, color);
			}
		}
		
		return color;
	}
	
	private static Color createColor(Display device, int colorIdx){
		
		RGB rgb = m_RGBMap.get(colorIdx);
		if (rgb != null) {
			return createColor(device, rgb);
		}
		/* in case of unsupported id return a null object */
		return null;
	}
	
	private static Color getColor(RGB rgb) {
		Color color = null;
		
		if (rgb != null) {
			if (m_colorMap.containsKey(rgb)) {
				color = m_colorMap.get(rgb);
				if (color.isDisposed() || (color.getDevice() == null)) {
					color = null;
					m_colorMap.remove(rgb);
				}
			}
			if (color == null) {
				color = new Color(Display.getCurrent(), rgb); 
				m_colorMap.put(rgb, color);
			}
		}		

		return color;
	}
	
	private static Color createColor(Display device, RGB rgb){
		if (rgb != null) {
			return new Color(device, rgb);
		}
		return null;
	}
	
	static public Color getDefaultPointColor() {
		return getColor(POINT_COLOR);
	}
	
	static public Color getInlineColor() {
		return getColor(INLINE_GROUP_COLOR);
	}

	static public Color getDefaultBackgroundColor() {
		return getColor(BACKGROUND_COLOR);
	}

	static public Color getDefaultGridColor() {
		return getColor(GRID_LINE_COLOR);
	}
	
	static public Color getDefaultReservedColor() {
		return getColor(RESEVED_COLOR);
	}
	
	static public Color getDefaultSpecialGroupColor() {
		return getColor(SPECIAL_GROUP_COLOR);
	}
	
	static public Color getDefaultIdleGroupColor() {
		return getColor(IDLE_GROUP_COLOR);
	}

	static public class Group implements Comparable<Group>{
		// The index of the line where it will be displayed in the editor
		private int m_showIndex = -1;

		// The name of the group (e.g. "main")
		private String m_name = null;
		private float m_percent;
		
		/* last user input */
		private String m_addresses = ""; //$NON-NLS-1$
		
		/* list of start and end addresses corresponding to the ranges in the group */
		private BigInteger m_startAddress[] = null;
		private BigInteger m_endAddress[] = null;
		
		/*multicore info*/
		private String m_source = "";
		private String m_context = "";
		private boolean m_isContext = false;
		
		/* attribute stating if this is a group with special behavior */
		private boolean m_special = false;
		
		private RGB m_color = null;
		
		public Group(String name, int showIndex) {
			this(name, "", getDefaultPointColor().getRGB(), showIndex); //$NON-NLS-1$
		}
		
		public Group(String name, String addresses, RGB color, int showIndex) {
			m_name = name;
			m_addresses = addresses;
			m_showIndex = showIndex;
			
			m_color = color;
			
			if (m_addresses != null) {
				parseAddresses();
			} else { 
				m_addresses = ""; //$NON-NLS-1$
			}
		}
		
		public Group(String name, BigInteger startAddresses[], BigInteger endAddresses[], RGB color, int showIndex) {
			m_name = name;
			m_showIndex = showIndex;
			m_source = ""; //$NON-NLS-1$
			m_context = ""; //$NON-NLS-1$
			
			m_color = color;
			
			if ((startAddresses != null) && (endAddresses != null)) {
				if (startAddresses.length == endAddresses.length) {
					m_startAddress = startAddresses.clone();
					m_endAddress = endAddresses.clone();
				}
			}
			
			genAddressesString();
		}
		
		public Group(String name, BigInteger startAddresses[], BigInteger endAddresses[], RGB color, int showIndex, String source, String context) {
			m_name = name;
			m_showIndex = showIndex;
			m_source = source;
			m_context = context;
			
			m_color = color;
			
			if ((startAddresses != null) && (endAddresses != null)) {
				if (startAddresses.length == endAddresses.length) {
					m_startAddress = startAddresses.clone();
					m_endAddress = endAddresses.clone();
				}
			}
			
			genAddressesString();
		}

		/**
		 * Constructor
		 * 
		 * @param name
		 *            the group's name
		 * @param labelColor
		 *            label color for the group
		 * @param barColor
		 *            bar color for the group
		 */
		public Group(String name, RGB color, int showIndex) {
			this(name, showIndex);
			m_color = color;
		}

		public String getName() {
			return m_name;
		}
		
		public String getSource() {
			return m_source;
		}
		
		public String getContext() {
			return m_context;
		}
		
		public void setSource(String source) {
			this.m_source = source;
		}
		
		public void setContext(String context) {
			this.m_context = context;
		}
		
		public String getUserAddresses() {
			return m_addresses;
		}
		
		public int getShowIndex() {
			return m_showIndex;
		}
		
		public RGB getColor() {
			return m_color;
		}
		
		public boolean isContext() {
			return m_isContext;
		}

		public void setIsContext(boolean m_isContext) {
			this.m_isContext = m_isContext;
		}
		
		/**
		 * Add a new pair of start - end addresses to this group. 
		 * 
		 * @param startAddress The start address.
		 * @param endAddress The end address.
		 */
		public void updateStartAndEndAddresses(BigInteger startAddress, BigInteger endAddress) {
		    if (containsAddress(startAddress)) {
		        return;
		    }
		    
		    int currentLength = m_startAddress.length;		    
		    
		    BigInteger[] tempStartAddress = new BigInteger[currentLength + 1];
		    BigInteger[] tempEndAddress = new BigInteger[currentLength + 1];
		    
		    System.arraycopy(m_startAddress, 0, tempStartAddress, 0, currentLength);
		    System.arraycopy(m_endAddress, 0, tempEndAddress, 0, currentLength);
		    
		    tempStartAddress[currentLength] = startAddress;
		    tempEndAddress[currentLength] = endAddress;
		    
		    m_startAddress = tempStartAddress;
		    m_endAddress = tempEndAddress;
		    
		    genAddressesString();
		}
		
		public boolean containsAddress(BigInteger addr, Vector<Group> all_groups) {
			if (m_startAddress == null || m_endAddress == null) {
				return false;
			}
			

			
			Vector<TraceTimelinePlot.Group> groups = new Vector<TraceTimelinePlot.Group>();
			
			for (int y = 0; y < all_groups.size(); y++) {
				
				TraceTimelinePlot.Group gr = all_groups.elementAt(y);
				for (int i = 0; i < gr.m_startAddress.length; i++) {
					if (addr.compareTo(gr.m_startAddress[i]) >= 0 && addr.compareTo(gr.m_endAddress[i]) <= 0) {
						groups.add(gr);
					}
				}
				
			}

			TraceTimelinePlot.Group theGroup = null;
			BigInteger minSize = TimelineFunction.INVALID_ADDRESS;
			//use the group with min size (inline)
			for (TraceTimelinePlot.Group gr : groups) {
				for (int i = 0; i < gr.m_startAddress.length; i++) {
					BigInteger diff = gr.m_endAddress[i].subtract(gr.m_startAddress[i]);
					if((minSize.compareTo(BigInteger.ZERO) == 0) || (minSize.compareTo(diff) > 0)){
						minSize = diff;
						theGroup = gr;
					}	
				}
			}
			boolean ret = this.equals(theGroup);
			return ret;

		}
		
		public boolean containsAddress(BigInteger addr) {
			if (m_startAddress == null || m_endAddress == null) {
				return false;
			}
			
			for (int i = 0; i < m_startAddress.length; i++) {
				if (addr.compareTo(m_startAddress[i]) >= 0 && addr.compareTo(m_endAddress[i]) < 0) {
					return true;
				}
			}
			
			return false;
		}
		
		public boolean containsAddress(TimelineFunction func) {
			if (m_startAddress == null || m_endAddress == null) {
				return false;
			}
			
					
			for (int i = 0; i < m_startAddress.length; i++) {

				
				if(func.getStartAddress().compareTo(m_startAddress[i]) == 0){
					return true;
				}
			}
			
			return false;
		}
		
		public long getSize() {
			if ((m_startAddress == null) || (m_endAddress == null)) { 
				return 0;
			}
			
			long size = 0;
			
			for (int i = 0; i < m_startAddress.length; i++) {
				size += m_endAddress[i].subtract(m_startAddress[i]).longValue();
			}
			return size;
		}
		
		public void genAddressesString() {
			m_addresses = ""; //$NON-NLS-1$
			
			if ((m_startAddress == null) || (m_endAddress == null)) { 
				return;
			}
			
			Arrays.sort(m_startAddress);
			Arrays.sort(m_endAddress);
			
			for (int i = 0; i < m_startAddress.length; i++) {
				m_addresses += "0x" + m_startAddress[i].toString(16) + " - " + "0x" //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
						+ m_endAddress[i].toString(16);
				if (i < m_startAddress.length - 1) {
					m_addresses += ", "; //$NON-NLS-1$
				}
			}
		}
		
		private void parseAddresses() {
    		
			m_startAddress = null; 
			m_endAddress = null;
    		
			String[] ranges = m_addresses.replace(" ", "").replace(";", ",").split(","); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
			if (ranges.length == 0) { 
				return;
			}
    		
			m_startAddress = new BigInteger[ranges.length];
			m_endAddress = new BigInteger[ranges.length];
    	
			for (int i = 0; i < ranges.length; i++) {
				String parts[] = ranges[i].split("-"); //$NON-NLS-1$
				if ((parts.length == 0) || (parts.length > 2)) {
					m_startAddress = null;
					m_endAddress = null;
					
					/* invalid entry - skip */
					continue;
				}
    			
				int radix = 10;
				parts[0] = parts[0].toLowerCase();
				if (parts[0].startsWith("0x")) {//$NON-NLS-1$
					parts[0] = parts[0].substring(2);
					radix = 16;
				}
				try {
					m_startAddress[i] = new BigInteger(parts[0], radix);
					if (parts.length == 1) {
						m_endAddress[i] = m_startAddress[i];
					} else {
						radix = 10;
						parts[1] = parts[1].toLowerCase();
						if (parts[1].startsWith("0x")) {//$NON-NLS-1$
							parts[1] = parts[1].substring(2);
							radix = 16;
						}
						m_endAddress[i] = new BigInteger(parts[1], radix);
					}
				} catch (NumberFormatException e) {
					m_startAddress = null;
					m_endAddress = null;
					
					/* invalid entry - skip */
					continue;
				}
    			
				if (m_endAddress[i].compareTo(m_startAddress[i]) < 0) {
					m_startAddress = null;
					m_endAddress = null;
					
					/* invalid entry - skip */
					continue;
				}
				
			}
    		
			/* re-create the addresses string from what was successfully parsed */
			genAddressesString();
		}

		public float getPercent() {
			return m_percent;
		}
		
		public void setPercent(float percent) {
			m_percent = percent;
		}
		
		public void setSpecial(boolean special) {
			m_special = special;
		}
		
		public boolean isSpecial() {
			return m_special;
		}
		
		public BigInteger[] getStartAddresses() {
			if (m_startAddress != null)
				return m_startAddress.clone();
			return null;
		}
		
		public BigInteger[] getEndAddresses() {
			if (m_endAddress != null)
				return m_endAddress.clone();
			return null;
		}
		
		public void setStartAddresses(BigInteger[] newVal) {
			m_startAddress = newVal.clone();
		}
		
		public void setEndAddresses(BigInteger[] newVal) {
			m_endAddress = newVal.clone();
		}
		
		public String getAddressString() {
			return m_addresses;
		}


		@Override
		public int hashCode() {
			return super.hashCode();
		}
		
		@Override
		public boolean equals(Object o) {
			if(o instanceof Group){
				Group g = (Group)o;
				
				if(this.compareTo(g) == 0){
					return true;
				}
			}
			return false;
		}
		
		@Override
		public int compareTo(Group o) {

			BigInteger oMaxSize = o.getEndAddresses()[o.getEndAddresses().length - 1].subtract(o.getStartAddresses()[0]);
			BigInteger thisMaxSize = this.getEndAddresses()[this.getEndAddresses().length - 1].subtract(this.getStartAddresses()[0]);
			if (thisMaxSize.compareTo(oMaxSize) == 0) {
				if (o.getEndAddresses()[o.getEndAddresses().length - 1].compareTo(this
						.getEndAddresses()[this.getEndAddresses().length - 1]) == 0) {
					return 0;
				}
				return -1;	
			} else {
				return (thisMaxSize.compareTo(oMaxSize) > 0) ? 1 : -1;
			}
			
		}
	}
	
	Vector<Group> groups;
	
	/**
	 * The horizontal slider. 
	 * It has a special nature because it may represent data of more than MAX_INT length
	 */
	private class LongSlider extends Slider {
		
		double logicalPosition;
		double logicalSize;
		double logicalIncrement;
		double logicalPageIncrement;
		
		int realPosition;
		int realSize;
		
		double factor;
		
		@Override
		protected void checkSubclass() {}
		
		public LongSlider(Composite parent, int style) {
			super(parent, style);
			
			logicalPosition = 0;
			logicalSize = 0;
			
			this.addSelectionListener(
					new SelectionAdapter() {
				@Override
				public void widgetSelected(SelectionEvent e) {
					
					double newStartCycleX;
					
					// / Right arrow was pressed
					if (e.detail == SWT.ARROW_DOWN) {
						newStartCycleX = Math.min(screen.startCycleX + logicalIncrement,
								(logicalSize - logicalPageIncrement));
						if (newStartCycleX == screen.startCycleX) {
							return;
						}
						screen.startCycleX = newStartCycleX;
					} // / Left Arrow was pressed
					else if (e.detail == SWT.ARROW_UP) {
						newStartCycleX = Math.max(0, screen.startCycleX - logicalIncrement);
						if (newStartCycleX == screen.startCycleX) {
							return;
						}
						screen.startCycleX = newStartCycleX;
					} // / Next-page was requested (the scrollbar was pressed right of the thumb)
					else if (e.detail == SWT.PAGE_DOWN) {
						newStartCycleX = Math.min(screen.startCycleX + logicalPageIncrement,
								(logicalSize - logicalPageIncrement));
						if (newStartCycleX == screen.startCycleX) {
							return;
						}
						screen.startCycleX = newStartCycleX;
					} // / Previous-page was requested (the scrollbar was pressed left of the thumb)
					else if (e.detail == SWT.PAGE_UP) {
						newStartCycleX = Math.max(0, screen.startCycleX - logicalPageIncrement);
						if (newStartCycleX == screen.startCycleX) {
							return;
						}
						screen.startCycleX = newStartCycleX;
					} // / The thumb was dragged
					else {
					
						newStartCycleX = getSelection();
						if (newStartCycleX == realPosition) { 
							return;
						}
						realPosition = (int)newStartCycleX;
						
						logicalPosition = (long)(realPosition * factor);
						if (realPosition == (realSize - 1)) {
							logicalPosition = logicalSize - 1;
						}
						if (logicalPosition >= logicalSize) {
							logicalPosition = logicalSize - 1;
						}
						
						screen.startCycleX = logicalPosition;
					}
					
					// / Update the position of the thumb, based on all the computations above
					realPosition = (int)(logicalPosition / factor);
					setSelection(realPosition);
					
					// / Make sure the start cycle is not outside allowable limits
					screen.startCycleX = Math.min(screen.startCycleX,
							countCycles - screen.countCyclesPerPixel * screen.screenWidthInPixels);
					screen.startCycleX = Math.max(0, screen.startCycleX);
					
					// / Redraw the plot
					forceRedraw();
				}
			});
		}
		
		/**
		 * @param totalLength - the logical size of the data
		 * @param increment - the increment of the slider in logical space.
		 *  It depends on the zoom level of the data 
		 */
		public void changeSliderValues(double totalLength, double pageIncrement) {
			
			if (logicalSize == totalLength && pageIncrement == logicalPageIncrement) {
				return;
			}
			
			double increment = pageIncrement / 16;
			
			logicalSize = (long)totalLength;
			logicalPosition = Math.min(logicalPosition, logicalSize - 1);
			logicalIncrement = increment;
			logicalPageIncrement = pageIncrement;
			
			realSize = (int)Math.min(logicalSize, 1000000000);
			if (realSize == logicalSize) {
				factor = 1;
			} else {
				factor = (double)logicalSize / realSize;
			}
			
			logicalPosition = screen.startCycleX;
			realPosition = (int)(logicalPosition / factor);
			
			int realPageIncrement = (int)Math.max(logicalPageIncrement / factor, 1);
			
			setValues(realPosition, 0, realSize, realPageIncrement,
					(int)Math.max(logicalIncrement / factor, 1), realPageIncrement);
			
		}
	}
	

	/**
	 * A point in the graph to be plotted
	 * The comparison is done on x-axis values, then on y-axis values
	 * @author B11543
	 *
	 */
	static public class Bar {
		long xStart;
		long xEnd;
		
		int y;
		
		/**
		 * Initializes a simple bar of length and countCycles 1
		 * @param x - horizontal position
		 * @param y - vertical position
		 */
		public Bar(long x, int y, long countCycles) {
			this.xStart = x;
			if (countCycles > 0) {
				this.xEnd = x + countCycles - 1;
			} else {
				this.xEnd = x;
			}
			this.y = y;
			
		}
		
		public long getX() {
			return this.xStart;
		}
		
		public long getY() {
			return this.y;
		}
		
		public long getCycles() {
			
			if (this.xStart == this.xEnd)
				return 0;
			
			return this.xEnd - this.xStart + 1;
		}
		
		public boolean intersects(long x1, long x2) {
			if (xStart >= x1 && xStart <= x2) {
				return true;
			}
			if (x1 >= xStart && x1 <= xEnd) {
				return true;
			}
			return false;
		}
		
	}
	TreeMap<Integer, TreeMap<Long, Bar>> graph;
	long xMin, xMax;

	/**
	 * The total number of cycles that passes so far (keeps the maximum
	 * value counted so far on the Ox axis). 
	 */
	double countCycles;
	int height;
	
	private class Screen {

		/**
		 *  The number of pixels
		 */
		private int screenWidthInPixels, screenHeightInPixels;
		
		/**
		 *  The start point (the point in the upper-left corner)
		 */
		private double startCycleX;
		private int startGraphPointY; 
		
		/**
		 *  The logical connection between points from the logical graph and points drawn on screen
		 */
		private double countCyclesPerPixel;
		
		/**
		 *  The constructor
		 *  By default, display the whole graph
		 */
		private Screen() {
			startCycleX = 0;
			startGraphPointY = 0;
			screenWidthInPixels = 1;
			screenHeightInPixels = 1;
			countCyclesPerPixel = 0;
		}
		
		/**
		 *  The window is resized. This means that the coordinates need be recalculated,
		 *   and the window be redrawn.
		 *  The fact that the window is resized does not mean that some points in the lower-right corner
		 *  will appear or disappear. There is a case, when the window is resized to contain all the
		 *  points, when points may also appear in the upper-left corner
		 *  
		 * @param countScreenPixelsWidth - the number of x pixels
		 * @param countScreenPixelsHeight - the number of y pixels
		 */
		private void resize() {
			
			boolean wasFullScreen = Math.round(countCycles / screenWidthInPixels) == Math.round(countCyclesPerPixel);
			
			this.screenWidthInPixels = m_canvas.getSize().x;
			this.screenHeightInPixels = m_canvas.getSize().y;
			
			if (!isSliderYVisible()) {
				startGraphPointY = 0;
			}
			
			if (wasFullScreen || countCyclesPerPixel == 0) {
				countCyclesPerPixel = countCycles / screenWidthInPixels;
			}
		}
		
		private Bar searchBar(long xStart, long xEnd, int y) {
			return searchLeftmostBar(xStart, xEnd, y);
		}
		
		private Bar searchLeftmostBar(long xStart, long xEnd, int y) {
			try {
				TreeMap<Long, Bar> horizontal = graph.get(Integer.valueOf(y));

				if (horizontal == null) {
					return null;
				}
				
				Map.Entry<Long, Bar> prevPoint = horizontal.floorEntry(Long.valueOf(xStart));
				
				if (prevPoint != null && prevPoint.getValue().intersects(xStart, xEnd)) {
					return prevPoint.getValue();
				}
				
				NavigableMap<Long, Bar> subMap = graph.get(Integer.valueOf(y)).subMap(
						Long.valueOf(xStart), true, Long.valueOf(xEnd), true);
				Collection<Bar> collection = subMap.values();
				Iterator<Bar> itr = collection.iterator();
				Bar leftmostBar = null;

				while (itr.hasNext()) {
					Bar bar = itr.next();

					if (bar.intersects(xStart, xEnd)) {
						if (leftmostBar == null) {
							leftmostBar = bar;
						} else if (leftmostBar.xStart > bar.xStart) {
							leftmostBar = bar;
						}
					}
				}
				if (leftmostBar != null) {
					return leftmostBar;
				}
			} catch (Exception e) {
				/* ignore the error */
				return null;
			}
			
			return null;
		}
		
		private Bar searchRightmostBar(long xStart, long xEnd, int y) {
			try {
				TreeMap<Long, Bar> horizontal = graph.get(Integer.valueOf(y));

				if (horizontal == null) {
					return null;
				}
				
				Map.Entry<Long, Bar> prevPoint = horizontal.floorEntry(Long.valueOf(xStart));
				
				NavigableMap<Long, Bar> subMap = graph.get(Integer.valueOf(y)).subMap(
						Long.valueOf(xStart), true, Long.valueOf(xEnd), true);
				Collection<Bar> collection = subMap.values();
				Iterator<Bar> itr = collection.iterator();
				Bar rightmostBar = null;

				while (itr.hasNext()) {
					Bar bar = itr.next();

					if (bar.intersects(xStart, xEnd)) {
						if (rightmostBar == null) {
							rightmostBar = bar;
						} else if (rightmostBar.xStart < bar.xStart) {
							rightmostBar = bar;
						}
					}
				}
				if (rightmostBar != null) {
					return rightmostBar;
				}
				
				if (prevPoint != null && prevPoint.getValue().intersects(xStart, xEnd)) {
					return prevPoint.getValue();
				}
			} catch (Exception e) {
				/* ignore the error */
				return null;
			}
			
			return null;
		}
		
		private long getValueX(long xPixel) {
			double ret = startCycleX + countCyclesPerPixel * xPixel;

			if (ret >= countCycles) {
				ret = countCycles - 1;
			}
			
			return (long)ret;
			
		}
		
		private int getPixelX(long xValue) {
			double ret = (xValue - startCycleX) / countCyclesPerPixel;

			return (int)ret;
			
		}
		
		private Bar searchBar(Point pixelLeft, Point pixelRight) {
			long xStart = (long)getValueX(pixelLeft.x);
			long xEnd = (long)getValueX(pixelRight.x + 1);
			int y = startGraphPointY + pixelLeft.y;

			return searchBar(xStart, xEnd, y);
		}
		
		@SuppressWarnings("unused")
		private double getPointWidth() {
			return Math.max(1.0, Math.round(1.0 / screen.countCyclesPerPixel));
		}
		
		private void drawBinary(GC gc, int startX, int endX, int y) {
			if (searchBar(new Point(startX, y), new Point(endX, y)) == null) {
				return;
			} else if (startX == endX) {
				if (groups.elementAt(startGraphPointY + y).m_isContext)
					return;
				
				gc.setBackground(getColor(groups.elementAt(startGraphPointY + y).getColor()));
				int a = startX; /* (int) (getPixelX(getValueX(startX)) * getPointWidth());*/
				int b = y * m_pointHeight;
				int width = 1; /* (int) getPointWidth();*/
				int height = m_pointHeight;

				if (m_reducedPointHeight) {
					b += 2;
					height -= 4;
				}
				
				gc.fillRectangle(a, b, width, height);
			} else {
				int xMiddle = startX + (endX - startX) / 2;

				drawBinary(gc, startX, xMiddle, y);
				drawBinary(gc, xMiddle + 1, endX, y);
			}
		}
		
		private void drawYLabels(GC gc) {
			Color prevColor = gc.getForeground();

			Color color = Display.getCurrent().getSystemColor(Y_LABELS_TEXT_COLOR);
			if (color.isDisposed())
				color = createColor(Display.getCurrent(), Y_LABELS_TEXT_COLOR);
			gc.setForeground(color);
			
			int i;

			// find the aproximate dimensions of the percent text
			FontMetrics metrics = gc.getFontMetrics();
			// Compute the number of letters that fit the window
			// for the group name. 
			int availableLetters = (m_yLabelsCanvas.getSize().x - 10)
					/ metrics.getAverageCharWidth()
							- 6;
		    
			// Put percent column header
			gc.drawText("%", m_yLabelsCanvas.getSize().x - metrics.getAverageCharWidth() * 3, //$NON-NLS-1$
					m_rulerHeight + (-startGraphPointY) * m_pointHeight + 4 - 20);
			
			for (i = startGraphPointY; i
					< Math.min(startGraphPointY + screenHeightInPixels / m_pointHeight, groups.size()); i++) {
				Group group = groups.elementAt(i);
				
				gc.setForeground(getColor(group.getColor()));
				
				int x = 10;
				if (group.isContext()) {
					x = 3;
				}
				
				// Print function name, and if too big put "..."
				if (availableLetters > group.m_name.length()) {
					gc.drawText(group.m_name, x,
							m_rulerHeight + (i - startGraphPointY) * m_pointHeight + 4);
				} else if (availableLetters > 4) {
					//availableLetters-3 should have been correct but is wrong visually
					gc.drawText(
							group.m_name.substring(0, availableLetters - 4) + "...", x, //$NON-NLS-1$
							m_rulerHeight + (i - startGraphPointY) * m_pointHeight + 4);
				}
				
				//Add percent
				if (!group.isContext()) {
					String percent;
					percent = String.format(" %05.2f", group.m_percent); //$NON-NLS-1$
					gc.drawText(percent, m_yLabelsCanvas.getSize().x - metrics.getAverageCharWidth() * 6,
							m_rulerHeight + (i - startGraphPointY) * m_pointHeight + 4);
				}

				color = Display.getCurrent().getSystemColor(Y_LABELS_TEXT_COLOR);
				if (color.isDisposed())
					color = createColor(Display.getCurrent(), Y_LABELS_TEXT_COLOR);
				gc.setForeground(color);
				
				gc.drawLine(0, m_rulerHeight + (i - startGraphPointY) * m_pointHeight,
						m_yLabelsCanvas.getSize().x,
						m_rulerHeight + (i - startGraphPointY) * m_pointHeight);
			}
			gc.drawLine(0, m_rulerHeight + (i - startGraphPointY) * m_pointHeight,
					m_yLabelsCanvas.getSize().x, m_rulerHeight + (i - startGraphPointY) * m_pointHeight);
			
			gc.setForeground(prevColor);
		}
		
		private void drawGridLine(GC gc, int y) {
			gc.drawLine(0, y * m_pointHeight, screenWidthInPixels, y * m_pointHeight);
		}
		
		private void drawCursor(GC gc, boolean body) {
			
			if (m_cursor1X > -1) {
			
				Color color = Display.getCurrent().getSystemColor(CURSOR_1_COLOR);
				if (color.isDisposed())
					color = createColor(Display.getCurrent(), CURSOR_1_COLOR);
				gc.setForeground(color);
				
				if (body) {
					if (m_cursor1X >= startCycleX
							&& m_cursor1X <= (startCycleX + screenWidthInPixels * countCyclesPerPixel)) {
						gc.drawLine(screen.getPixelX(m_cursor1X), 0, screen.getPixelX(m_cursor1X),
								m_canvas.getSize().y);
					}
				} else { // ruler
					if (m_cursor1X >= startCycleX
							&& m_cursor1X <= (startCycleX + screenWidthInPixels * countCyclesPerPixel)) {
						gc.drawLine(screen.getPixelX(m_cursor1X), 35, screen.getPixelX(m_cursor1X),
								m_rulerCanvas.getSize().y);
						
						// Make the time unit conversion here.
						double convertedCursor1X = TimeUnitUtils.convertCycles(
								m_cursor1X, m_timeUnit, m_cpuFreq);
						String text = TimeUnitUtils.formatTimeValue(convertedCursor1X, m_timeUnit);
						
						gc.drawText(text, screen.getPixelX(m_cursor1X) - gc.textExtent(text).x / 2,
								25);
					}
				}
			}
			
			if (m_cursor2X > -1) {
				
				Color color = Display.getCurrent().getSystemColor(CURSOR_2_COLOR);
				if (color.isDisposed()) 
					color = createColor(Display.getCurrent(), CURSOR_2_COLOR);
				gc.setForeground(color);
				
				if (body) {
					if (m_cursor2X >= startCycleX
							&& m_cursor2X <= (startCycleX + screenWidthInPixels * countCyclesPerPixel)) {
						gc.drawLine(screen.getPixelX(m_cursor2X), 0, screen.getPixelX(m_cursor2X),
								m_canvas.getSize().y);
					}
				} else { // ruler
					if (m_cursor2X >= startCycleX
							&& m_cursor2X <= (startCycleX + screenWidthInPixels * countCyclesPerPixel)) {
						gc.drawLine(screen.getPixelX(m_cursor2X), 50, screen.getPixelX(m_cursor2X),
								m_rulerCanvas.getSize().y);
												
						// Make the time unit conversion here.
						double convertedCursor2X = TimeUnitUtils.convertCycles(
								m_cursor2X, m_timeUnit, m_cpuFreq);
						String text = TimeUnitUtils.formatTimeValue(convertedCursor2X, m_timeUnit);

						gc.drawText(text, screen.getPixelX(m_cursor2X) - gc.textExtent(text).x / 2,
								40);
					}
				}
			}
			
			if (!body && m_cursor1X > -1 && m_cursor2X > -1) {
				long difference = Math.abs(m_cursor1X - m_cursor2X);
				// Make the time unit conversion here.
				double convertedDifference = TimeUnitUtils.convertCycles(
						difference, m_timeUnit, m_cpuFreq);
				String differenceText = TimeUnitUtils.formatTimeValue(convertedDifference, m_timeUnit);

				int differenceSize = gc.textExtent(differenceText).x + 3;
				int pixel1X, pixel2X;
				int pixelMinX, pixelMaxX;
				
				if (m_cursor1X >= startCycleX
						&& m_cursor1X <= (startCycleX + screenWidthInPixels * countCyclesPerPixel)) {
					pixel1X = screen.getPixelX(m_cursor1X);
				} else if (m_cursor1X < startCycleX) {
					pixel1X = 0;
				} else {
					pixel1X = screen.screenWidthInPixels;
				}
				
				if (m_cursor2X >= startCycleX
						&& m_cursor2X <= (startCycleX + screenWidthInPixels * countCyclesPerPixel)) {
					pixel2X = screen.getPixelX(m_cursor2X);
				} else if (m_cursor2X < startCycleX) {
					pixel2X = 0;
				} else {
					pixel2X = screen.screenWidthInPixels;
				}
				
				pixelMinX = Math.min(pixel1X, pixel2X);
				pixelMaxX = Math.max(pixel1X, pixel2X);
				
				Color color = Display.getCurrent().getSystemColor(DIFFERENCE_COLOR);
				if (color.isDisposed())
					color = createColor(Display.getCurrent(), DIFFERENCE_COLOR);
				gc.setForeground(color);
				
				if (differenceSize > pixelMaxX - pixelMinX) {
					if (pixelMinX > screen.screenWidthInPixels - pixelMaxX) {
						if (pixelMinX > differenceSize) {
							pixelMaxX = pixelMinX;
							pixelMinX = Math.max(pixelMaxX - differenceSize - 16, 0);
						} else {
							return;
						}
					} else if (screen.screenWidthInPixels - pixelMaxX > differenceSize) {
						pixelMinX = pixelMaxX;
						pixelMaxX = Math.min(pixelMinX + differenceSize + 16,
								screen.screenWidthInPixels);
					} else {
						return;
					}
							
				}
				
				int smallLineWidth = Math.max((pixelMaxX - pixelMinX - differenceSize - 4) / 2, 0);

				if (smallLineWidth > 0) {
					gc.drawLine(pixelMinX, 62, pixelMinX + smallLineWidth, 62);
					gc.drawLine(pixelMaxX, 62, pixelMaxX - smallLineWidth, 62);
					gc.drawText(differenceText, pixelMinX + smallLineWidth + 4, 55);
				} else {
					gc.drawText(differenceText, pixelMinX + 3, 55);
				}
			}
		}
		
		/**
		 * Draws the ruler for the Ox axis (displays the time range).
		 *  
		 * @param gc
		 */
		private void drawRuler(GC gc) {
			
			if (countCycles == 0) {
				// Nothing to draw.
				LOGGER.debug("[drawRuler] cycleCount is zero; nothing to draw"); //$NON-NLS-1$
				return;
			}
			
			Color color = Display.getCurrent().getSystemColor(RULER_COLOR);
			if (color.isDisposed())
				color = createColor(Display.getCurrent(), RULER_COLOR);
			gc.setForeground(color);
			
			long minimum = (long)startCycleX;
			long maximum = (long)(startCycleX + screenWidthInPixels * countCyclesPerPixel);
			
			long approximateDistance = (long)(300 * countCyclesPerPixel);
			boolean multiplyBy2 = true;
			long difference = Long.MAX_VALUE;
			long exactDistance = 1;

			for (long pow = exactDistance; pow < countCycles; multiplyBy2 = !multiplyBy2, pow *= multiplyBy2
					? 2
					: 5) {
				if (Math.abs(approximateDistance - pow) < difference) {
					difference = Math.abs(approximateDistance - pow); 
					exactDistance = pow;
				} else {
					break;
				} 
			}
			
			long labelsMin = (minimum / exactDistance) * exactDistance;
			long secondLevelDistance = exactDistance / 10;
			List<Long> labels = new ArrayList<Long>();

			for (long l = labelsMin; l <= maximum; l += exactDistance) { 
				if (l >= minimum && l <= maximum) {
					labels.add(Long.valueOf(l));
				}
			}
			
			for (Long labelValue: labels) {
				int pixelX = getPixelX(labelValue);
				
				gc.drawLine(pixelX, 0, pixelX, 10);
				
				// Make the time unit conversion here.
				double convertedLabelValue = TimeUnitUtils.convertCycles(
						labelValue, m_timeUnit, m_cpuFreq);
				String timeUnit = ""; //$NON-NLS-1$
				if (SHORT_TIME_NOTATION.containsKey(m_timeUnit)) {
					timeUnit = SHORT_TIME_NOTATION.get(m_timeUnit);
				}
				String text = TimeUnitUtils.formatTimeValue(
						convertedLabelValue, m_timeUnit) + timeUnit;

				gc.drawText(text, pixelX - gc.textExtent(text).x / 2, 10);
				
				for (int i = 1; i < 10; i++) {
					if (labelValue + i * secondLevelDistance < countCycles) {
						gc.drawLine(getPixelX(labelValue + i * secondLevelDistance), 0,
								getPixelX(labelValue + i * secondLevelDistance), 5);
					}
				}
				if (labelValue.equals(labels.get(0))) { 
					for (int i = 1; i < 10; i++) {
						if (labelValue - i * secondLevelDistance > 0) {
							gc.drawLine(getPixelX(labelValue - i * secondLevelDistance), 0,
									getPixelX(labelValue - i * secondLevelDistance), 5);
						}
					}
				}
				
			}
				
		}
		
		private void zoom(boolean inNotOut, int x) {
			
			// / Perform the zoom operation: change the countCyclesPerPixel parameter
			double newCountCyclesPerPixel = countCyclesPerPixel;

			if (inNotOut) {
				newCountCyclesPerPixel /= 2;
			} else {
				newCountCyclesPerPixel *= 2;
			}
			
			// / Make sure we don't zoom-out more than the width of the visible screen
			// / And make usre we don't zoom-in more that a predefined number of pixels per clock cycles
			if (newCountCyclesPerPixel >= countCycles / screen.screenWidthInPixels) {
				newCountCyclesPerPixel = countCycles / screen.screenWidthInPixels;
				screen.startCycleX = 0;
			}
			if (newCountCyclesPerPixel < 1 / m_maximumZoom) {
				newCountCyclesPerPixel = 1 / m_maximumZoom;
			}
			
			// / Continue only if the zoom ratio changed
			if (Math.round(newCountCyclesPerPixel) != Math.round(countCyclesPerPixel)) {
				
				// / Make sure the timeline is centered around the point that was zoomed.
				long xValueBefore = screen.getValueX(x);

				countCyclesPerPixel = newCountCyclesPerPixel;
				long xValueAfter = screen.getValueX(x);

				screen.startCycleX += xValueBefore - xValueAfter;

				// / Make sure the start point of the timeline didn't get out of range
				screen.startCycleX = Math.min(screen.startCycleX,
						countCycles - screen.countCyclesPerPixel * screen.screenWidthInPixels);
				screen.startCycleX = Math.max(0, screen.startCycleX);

				// / Recompute the parameters and redraw
				screen.resize();
				forceRedraw();
			}
		}
		
	}
	private Screen screen;
	private double m_cpuFreq = -1;
	private String m_timeUnit = IConfigurableDisplay.TIME_UNIT_CYCLES;
	
	public void setFocus() {
		m_mainCanvas.setFocus();
	}
	
	private boolean isSliderXVisible() {
		return screen.countCyclesPerPixel < countCycles / screen.screenWidthInPixels;
	}
	
	private boolean isSliderYVisible() {
		return height * m_pointHeight > m_mainCanvas.getSize().y - m_rulerHeight;
	}
	
	private void reInitComponents() {
		if (m_rulerCanvas == null) {
			m_rulerCanvas = new Canvas(m_mainCanvas, SWT.NONE);
			m_rulerCanvas.setBackground(Display.getCurrent().getSystemColor(BACKGROUND_COLOR));
			m_rulerCanvas.addPaintListener(new PaintListener() {
				public void paintControl(PaintEvent e) {
					m_rulerCanvas.setBackground(Display.getCurrent().getSystemColor(BACKGROUND_COLOR));
					screen.resize();
					screen.drawRuler(e.gc);
					screen.drawCursor(e.gc, false);
				}
			});
		}
		
		if (m_canvas == null) {
			m_canvas = new Canvas(m_mainCanvas, SWT.NONE);
			m_canvas.setBackground(Display.getCurrent().getSystemColor(BACKGROUND_COLOR));
			m_canvas.addMouseListener(new MouseListener() {
				public void mouseDown(MouseEvent e) {
					
					if (m_selectNotZoom) {
						if (e.button == 1) { 
							m_cursor1X = getAdjustedCursorPosition(e.x, e.y);
						} else if (e.button == 3) {
							m_cursor2X = getAdjustedCursorPosition(e.x, e.y);
						}
						
						forceRedraw();
					} else {
						if (e.button == 1) {
							screen.zoom(true, e.x);
						} else if (e.button == 3) {
							screen.zoom(false, e.x);
						}
					}
				}

				public void mouseUp(MouseEvent e) {}

				public void mouseDoubleClick(MouseEvent e) {}
			});
			m_canvas.addMouseTrackListener(new MouseTrackAdapter() {
				@Override
				public void mouseEnter(MouseEvent e) {
					if (m_selectNotZoom) {
						m_canvas.setCursor(new Cursor(Display.getCurrent(), SWT.CURSOR_ARROW));
					} else {
						m_canvas.setCursor(new Cursor(Display.getCurrent(), SWT.CURSOR_CROSS));
					}
				}
			});
			m_canvas.addPaintListener(
					new PaintListener() {
				public void paintControl(PaintEvent e) {
					m_canvas.setBackground(Display.getCurrent().getSystemColor(BACKGROUND_COLOR));
					if (groups == null) {
						return;
					}
					screen.resize();
					m_yLabelsCanvas.redraw();
					
					GC gc = e.gc;
					
					if (isSliderXVisible()) {
						m_sliderX.setBackground(Display.getCurrent().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
						m_sliderX.changeSliderValues(countCycles,
								screen.screenWidthInPixels * screen.countCyclesPerPixel);
						m_sliderX.setSize(m_canvas.getSize().x, m_sliderSize);
						m_sliderX.setLocation(0, m_canvas.getSize().y - m_sliderSize);
						m_sliderX.setVisible(true);
						m_sliderX.update();
					} else {
						m_sliderX.setVisible(false);
					}
					
					if (isSliderYVisible()) {
						m_sliderY.setValues(screen.startGraphPointY, 0,
								height - screen.screenHeightInPixels / m_pointHeight + 1, 
								Math.max(1,
								(screen.screenHeightInPixels / m_pointHeight / height)
								* m_canvas.getSize().y),
								1, 
								screen.screenHeightInPixels / m_pointHeight);
						m_sliderY.setSize(m_sliderSize,
								m_canvas.getSize().y - (isSliderXVisible() ? m_sliderSize : 0));
						m_sliderY.setLocation(m_mainCanvas.getSize().x - m_sliderSize, m_rulerHeight);
						m_sliderY.setVisible(true);
						m_sliderY.update();
					} else {
						m_sliderY.setVisible(false);
					}

					Color color = Display.getCurrent().getSystemColor(POINT_COLOR);
					if (color.isDisposed()) 
						color = createColor(Display.getCurrent(), POINT_COLOR);
					gc.setBackground(color);
					 
					for (int y = 0; y
							< Math.min(screen.screenHeightInPixels / m_pointHeight, groups.size()); y++) {
						screen.drawBinary(gc, 0, screen.screenWidthInPixels, y);
					}
					
					color = Display.getCurrent().getSystemColor(GRID_LINE_COLOR);
					if (color.isDisposed()) 
						color = createColor(Display.getCurrent(), GRID_LINE_COLOR);
					gc.setForeground(color);
					
					if (m_drawGrid) {
						int y;

						for (y = 0; y
								< Math.min(screen.screenHeightInPixels / m_pointHeight,
								height - screen.startGraphPointY); y++) {
							screen.drawGridLine(gc, y);
						}
						screen.drawGridLine(gc, y);
					}
					
					screen.drawCursor(gc, true);
				}
			});
		}
		
		if (m_sliderX == null) {
			m_sliderX = new LongSlider(m_canvas, SWT.HORIZONTAL);
		}
		if (m_sliderY == null) {
			m_sliderY = new Slider(m_mainCanvas, SWT.VERTICAL);
			m_sliderY.addSelectionListener(new SelectionAdapter() {
				@Override
				public void widgetSelected(SelectionEvent e) {
					screen.startGraphPointY = m_sliderY.getSelection();
					forceRedraw();
				}
			});
		}
		
		m_rulerCanvas.setLocation(0, 0);
		m_rulerCanvas.setSize(m_mainCanvas.getSize().x - (isSliderYVisible() ? m_sliderSize : 0),
				m_rulerHeight);
		
		m_canvas.setLocation(0, m_rulerHeight);
		m_canvas.setSize(m_mainCanvas.getSize().x - (isSliderYVisible() ? m_sliderSize : 0), 
				m_mainCanvas.getSize().y - m_rulerHeight);
	}
	
	private long getAdjustedCursorPosition(int pixelX, int pixelY) {
		int y = screen.startGraphPointY + pixelY / m_pointHeight;
		
		boolean hasPoint = screen.searchBar(screen.getValueX(pixelX), 
				Math.min((long)countCycles - 1, screen.getValueX(pixelX + 1) - 1), y)
				!= null;
		
		for (int i = 1; i <= MAX_ADJUST_CURSOR; i++) {
			
			// / Search to the left
			if (0 <= (pixelX - i)) {
				if (hasPoint
						!= (screen.searchBar(screen.getValueX(pixelX - i),
						Math.min((long)countCycles - 1, screen.getValueX(pixelX - i + 1) - 1), y)
								!= null)) {
					if (!hasPoint) {
						Bar leftPoint = screen.searchRightmostBar(screen.getValueX(pixelX - i), 
								Math.min((long)countCycles - 1, screen.getValueX(pixelX - i + 1) - 1), y);

						if (leftPoint != null) {
							return leftPoint.xEnd + 1;
						}
					} else {
						Bar leftPoint = screen.searchLeftmostBar(screen.getValueX(pixelX - i + 1), 
								Math.min((long)countCycles - 1, screen.getValueX(pixelX - i + 2) - 1), y);

						if (leftPoint != null) {
							return leftPoint.xStart;
						}
					}
				}
			}
			
			// / Search to the right
			if (hasPoint
					!= (screen.searchBar(screen.getValueX(pixelX + i),
					Math.min((long)countCycles - 1, screen.getValueX(pixelX + i + 1) - 1), y)
							!= null)) {
				if (!hasPoint) {
					Bar rightPoint = screen.searchLeftmostBar(screen.getValueX(pixelX + i), 
							Math.min((long)countCycles - 1, screen.getValueX(pixelX + i + 1) - 1), y);

					if (rightPoint != null) {
						return rightPoint.xStart;
					}
				} else {
					Bar rightPoint = screen.searchRightmostBar(screen.getValueX(pixelX + i - 1), 
							Math.min((long)countCycles - 1, screen.getValueX(pixelX + i) - 1), y);

					if (rightPoint != null) {
						return rightPoint.xEnd + 1;
					}
				}
			}
		}
		
		return screen.getValueX(pixelX);
	}
	
	/**
	 * Init the plot
	 * @param parent - the composite that will hold the canvas
	 */
	public TraceTimelinePlot(Composite parent) {		
		if (parent != null) {			
			Composite comp  = new Composite(parent, SWT.FILL);
			comp.setLayout(new GridLayout());
			comp.setLayoutData(new GridData(GridData.FILL_BOTH));
			
			m_toolBar = new ToolBar(comp, SWT.WRAP);
			GridLayout toolBarLayout = new GridLayout();

			toolBarLayout.makeColumnsEqualWidth = true;
			toolBarLayout.numColumns = 3;
			m_toolBar.setLayout(toolBarLayout);
			m_toolBar.setLayoutData(new GridData());
			
			makeActions();
			
			m_plotSashForm = new SashForm(comp, SWT.HORIZONTAL);
			m_plotSashForm.setLayout(new GridLayout());
			m_plotSashForm.setLayoutData(new GridData(GridData.FILL_BOTH));
			
			m_yLabelsCanvas = new Canvas(m_plotSashForm, SWT.NONE);
			m_yLabelsCanvas.setLayout(new GridLayout());
			m_yLabelsCanvas.setLayoutData(new GridData(GridData.FILL_BOTH));
			m_yLabelsCanvas.setBackground(
					Display.getCurrent().getSystemColor(Y_LABELS_BACKGROUND_COLOR));
			m_yLabelsCanvas.addPaintListener(new PaintListener() {
				public void paintControl(PaintEvent e) {
					m_yLabelsCanvas.setBackground(
							Display.getCurrent().getSystemColor(Y_LABELS_BACKGROUND_COLOR));
					if ((groups != null) && (!groups.isEmpty())){
						screen.drawYLabels(e.gc);
					}
				}
			});
			
			m_mainCanvas = new Canvas(m_plotSashForm, SWT.NONE);
			m_mainCanvas.setLayout(new GridLayout());
			m_mainCanvas.setLayoutData(new GridData(GridData.FILL_BOTH));
			m_mainCanvas.addPaintListener(new PaintListener() {
				public void paintControl(PaintEvent e) {
					reInitComponents();
				}
			});
			
			m_plotSashForm.setWeights(new int[] { 1, 5});
			
			screen = new Screen();
			
			comp.addMouseWheelListener(new MouseWheelListener() {
				public void mouseScrolled(MouseEvent e) {
					int x = e.x - m_yLabelsCanvas.getSize().x - 8;

					if (x > 0) { 
						screen.zoom(e.count > 0, x);
					}
				}
			});
		}
		
		graph = new TreeMap<Integer, TreeMap<Long, Bar>>();
		countCycles = 0;
		height = 0;
		xMin = 0;
		xMax = 0;

		initialized = true;
	}
	
	private void makeActions() {
		
		final ToolItem selectionModeItem = new ToolItem(m_toolBar, SWT.CHECK);
		final ToolItem zoomModeItem = new ToolItem(m_toolBar, SWT.CHECK);

		new ToolItem(m_toolBar, SWT.SEPARATOR);
		final ToolItem fullViewItem = new ToolItem(m_toolBar, SWT.PUSH);
		new ToolItem(m_toolBar, SWT.SEPARATOR);
				
		//final ToolItem multiCore = new ToolItem(m_toolBar, SWT.DROP_DOWN);
		
		selectionModeItem.setText(UIMessages.TraceTimelinePlot_0);
		selectionModeItem.setToolTipText(UIMessages.TraceTimelinePlot_1);
		selectionModeItem.setSelection(true);
		selectionModeItem.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				m_selectNotZoom = true;
				zoomModeItem.setSelection(false);
			}
		});
		
		zoomModeItem.setText(UIMessages.TraceTimelinePlot_2);
		zoomModeItem.setToolTipText(UIMessages.TraceTimelinePlot_3);
		zoomModeItem.setSelection(false);
		zoomModeItem.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				m_selectNotZoom = false;
				selectionModeItem.setSelection(false);
			}
		});
		
		fullViewItem.setText(UIMessages.TraceTimelinePlot_4);
		fullViewItem.setToolTipText(UIMessages.TraceTimelinePlot_5);
		fullViewItem.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				screen.startCycleX = 0;
				screen.countCyclesPerPixel = countCycles / screen.screenWidthInPixels;
				forceRedraw();
			}
		});
		
	}
	
	public void setGroups(Vector<Group> groups) {
		this.groups = groups;
	}
	
	public ToolBar getToolBar() {
		return m_toolBar;
	}
	
	/**
	 * Add a new set of points at the end of the already existent set of points
	 * @param points - the vector of points to be added
	 */
	public void addPoints(Vector<Bar> points) {
		// this.points.addAll(points);
		for (Bar point: points) {
			TreeMap<Long, Bar> horizontal;

			if (!graph.containsKey(Integer.valueOf(point.y))) { 
				graph.put(Integer.valueOf(point.y), new TreeMap<Long, Bar>());
			}
			
			horizontal = graph.get(Integer.valueOf(point.y));
			horizontal.put(Long.valueOf(point.xStart), point);
			
			if (xMin > point.xStart) {
				xMin = point.xStart;
			}
			if (xMax < point.xEnd) {
				xMax = point.xEnd;
			}
			if (height <= point.y) {
				height = point.y + 1;
			}
			
		}
		
		countCycles = xMax - xMin + 1;
		
		// UI components need a reinit to integrate the new points added.
		// (e.g., new points might need an OY slider to be accessed).
		reInitComponents();
		
		forceRedraw();
	}
	
	public void resize() {
		screen.startCycleX = 0;
		screen.countCyclesPerPixel = countCycles / screen.screenWidthInPixels;
		
		forceRedraw();
	}
	
	public void clear(boolean clearGroups) {
		countCycles = 0;
		height = 0;
		xMin = 0;
		xMax = 0;
		if ( graph != null) {
			graph.clear();
		}		
		/* the groups are kept as references in 
		 * TraceTimelineEditor, TraceTimelinePlot and also 
		 * EditTimelineGroupsDialog. Sometimes we may want to reuse
		 * the groups but clear the rest of the data (e.g. reuse
		 * the groups created by EditTimelineGroupsDialog)
		 */
		if (clearGroups && (groups != null)) {
			groups.clear();
		}
	}
	
	public void setFrequency(double freq) {
		m_cpuFreq = freq;
	}
	
	public void setTimeUnit(String timeUnit) {
		m_timeUnit = timeUnit;	
	}
	
	
}
