/*******************************************************************************
 * Copyright (c) 2014, 2015 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.timeline;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Scanner;
import java.util.Vector;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.custom.CTabItem;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Device;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.FileEditorInput;

import com.freescale.sa.SaConstants;
import com.freescale.sa.ui.common.AbstractSaEditor;
import com.freescale.sa.ui.common.IConfigurableDisplay;
import com.freescale.sa.ui.common.IRefreshableDisplay;
import com.freescale.sa.ui.common.ITraceEditorInput;
import com.freescale.sa.ui.common.editor.DragAndDropManager;

/**
 * @author Marius Cetateanu
 * @version 1.0
 */
public class TraceTimelineEditor extends AbstractSaEditor implements
		IConfigurableDisplay, IRefreshableDisplay {

	/**
	 * <p>
	 * A base name that identifies this editors launch config
	 * </p>
	 */
	String m_baseName;

	/**
	 * <p>
	 * The unique ID of this editor.
	 * </p>
	 */
	public static final String ID = "com.freescale.sa.ui.traceTimelineEditor"; //$NON-NLS-1$

	/**
	 * <p>
	 * The Timeline input.
	 * <p>
	 */
	protected TimelineInput m_timelineInput = null;
	
	private boolean m_isFilteredEditor = false;
	
	/**
	 * <p>
	 * The TimelineFunction model.
	 * </p>
	 */
	protected ITimelineFunctionModel m_functionModel = null;

	protected Composite m_parent;

	/**
	 * names for 'special' functions, which are used as 'place holders' in
	 * timeline. <no debug info> is a special function for all code without
	 * debug information \<other\> is a special function for the code with debug
	 * information but not associated to any of the functions in a particular
	 * list or group.
	 */
	public final static String NO_DEBUG_INFO_FUNC = "<no debug info>"; //$NON-NLS-1$
	public final static String OTHER_FUNC = "<other>"; //$NON-NLS-1$
	public final static String SOURCE = "source"; //$NON-NLS-1$

	
	private static boolean m_configurableAdded = false;

	private CTabFolder m_tabFolder = null;

	private boolean m_timeUnitMenu = false;

	/**
	 * The timestamp value might not be updated for each trace event. For this
	 * situation, we approximate the time by considering each assembly
	 * instruction lasted one cycle. This member keeps the count of assembly
	 * instructions for a group of consecutive events that have the same
	 * timestamp.
	 */

	

	/**
	 * Keeps the timestamp for the event that contains a program trace error.
	 */
	long m_errorEntryTs = -1;

	/**
	 * Path to the 'Analysis Results' folder (it keeps all the profiling
	 * results).
	 */
	private String m_analysisResultsPath;

	/**
	 * Class logger.
	 */
	private static Logger LOGGER = Logger.getLogger(TraceTimelineEditor.class);

	/**
	 * Menu for configuring the time unit.
	 */
	protected TimeUnitGroup m_timeUnitGroup;
	
	public TraceTimelineEditor() {
		super(ID, UIHelp.Timeline_Context_Id); 
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.part.EditorPart#doSave(org.eclipse.core.runtime.
	 * IProgressMonitor)
	 */
	@Override
	public void doSave(IProgressMonitor monitor) {// saving not supported by
													// this editor
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.part.EditorPart#doSaveAs()
	 */
	@Override
	public void doSaveAs() {// saving not supported by this editor
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.part.EditorPart#init(org.eclipse.ui.IEditorSite,
	 * org.eclipse.ui.IEditorInput)
	 */
	@Override
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException {

		TimelineInputFile timelineInputFile = new TimelineInputFile(input);
		m_timelineInput = new TimelineInput(timelineInputFile.getPath());
		setSite(site);
		setInput(timelineInputFile);
		setPartName(input.getName());

		if (input instanceof FileEditorInput) {
			FileEditorInput inputFile = (FileEditorInput) input;
			setProject(inputFile);
		}

		// open file from drag-and-drop
		m_baseName = new Path(timelineInputFile.getPath()).removeFileExtension().lastSegment();
		m_analysisResultsPath = timelineInputFile.getStorageDirectory();

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.part.EditorPart#isDirty()
	 */
	@Override
	public boolean isDirty() {
		// saving not supported by this editor
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.part.EditorPart#isSaveAsAllowed()
	 */
	@Override
	public boolean isSaveAsAllowed() {
		// saving not supported by this editor
		return false;
	}

	private void initSourceTabs(Composite parent) {
		setTabFolder(new CTabFolder(parent, SWT.NONE));
		getTabFolder().setLayoutData(new GridData(GridData.FILL_BOTH));

		getTabFolder().addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(
					org.eclipse.swt.events.SelectionEvent event) {
				setFocus();
			}
		});
		
		final DragAndDropManager dndm = DragAndDropManager.getInstance();
		String drSrc = dndm.getDraggedString();
		setIsFilteredEditor(drSrc != null && !drSrc.isEmpty());
		
		if (!getIsFilteredEditor()) {
			dndm.addListeners(getTabFolder(), m_timelineInput.getTimelineFile());
		} else {
			if (!drSrc.isEmpty()) {
				m_timelineInput.addFilter(drSrc);
				setPartName(drSrc);
				dndm.clearDraggedString();
			}
		}
	}

	private void createSourceTab(Composite parent, String source, Scanner scan) {
		CTabItem tab = new CTabItem(getTabFolder(), SWT.NONE);
		tab.setText(source);
		tab.setData(UIMessages.TraceTimelineEditor_11, source);
		tab.setShowClose(true);
		
		Device device = Display.getCurrent ();
		Color white = new Color (device, 255, 255, 255);
		parent.setBackground(white);
		parent.setBackgroundMode(SWT.INHERIT_FORCE);
		
		SashForm sash1 = new SashForm(parent, SWT.NONE);
		tab.setControl(sash1);
					
		Composite summary = new Composite(sash1, SWT.NONE);
		GridLayout summaryLayout = new GridLayout();

		summaryLayout.marginBottom = 0;
		summaryLayout.marginTop = 0;
		summaryLayout.marginHeight = 0;
		summary.setLayout(summaryLayout);
		summary.setLayoutData(new GridData(GridData.FILL_BOTH));

		TraceTimelinePlot traceTimelinePlot = new TraceTimelinePlot(summary);

		makeCustomActions(source, traceTimelinePlot, scan);

		if (m_timeUnitMenu && m_timeUnitGroup != null) {
			traceTimelinePlot.setTimeUnit(m_timeUnitGroup
					.getTimeUnitSelection());
			traceTimelinePlot.setFrequency(m_timeUnitGroup.getFreqSelection());
		}

		m_timelineInput.addPlot(source, traceTimelinePlot);
	}
	
	@Override
	protected void createPartControl0(Composite parent) {
		m_parent = parent;
		
		// b20491
		/*
		 * Flow: 
		 * - Read the timeline file check if it contains anything, 
		 * - Read number of sources 
		 * - Create a TabFolder 
		 * - For each source in the timeline file add a tab 
		 * - In each tab put a TraceTimelinePlot 
		 * - A groups/configure table must be in each plot (makeCustomActions) 
		 * - initTimelineData - will init all plots, if we make a initTimelineData
		 *                      for each plot the progress will be harder to get 
		 * - add a clearTimelineData for each plot
		 * 
		 * also: 
		 * - refresh could work on all tabs or just the active one and redraw when changing tabs
		 */

		boolean timeLineHasEvents = false;

		File timelineFile = new File(m_timelineInput.getTimelineFile());
		if (timelineFile.exists() && (timelineFile.length() != 0)) {					
			timeLineHasEvents = m_timelineInput.timeLineHasEvents(timelineFile);
			
			// Return if there are no events
			if (!timeLineHasEvents) {
				return;
			}

			initSourceTabs(parent);

			// Create the tabs
			Scanner scan;
			try {
				scan = new Scanner(timelineFile);
				// Old format starts with an int (the first event), new format
				// with a **[version id]
				m_timelineInput.setMulticore(!scan.hasNextInt());

				// if it's multicore read source
				if (m_timelineInput.getMulticore()) {
					/* Skip version */
					scan.nextLine();
					
					/* Skip trace type line */
					if (m_timelineInput.hasTraceType()) {
						scan.nextLine();
					}

					if (scan != null) {
						scan.close();
						scan = null;
					}
				}
			} catch (FileNotFoundException e1) {
				LOGGER.error(UIMessages.TimelineInput_not_found, e1);
				e1.printStackTrace();
			}

			m_timelineInput.clearTimelineData(true);

			if (getEditorInput() == null) {
				return;
			}
			
			m_timelineInput.initTimelineData((ITraceEditorInput) getEditorInput());
			startModel((ITraceEditorInput) getEditorInput());

			try {
				scan = new Scanner(timelineFile);
				// Old format starts with an int (the first event), new format
				// with a **[version id]
				m_timelineInput.setMulticore(!scan.hasNextInt());

				// if it's multicore read source
				if (m_timelineInput.getMulticore()) {
					/* Skip version */
					scan.nextLine();
					
					/* Skip trace type line */
					if (m_timelineInput.hasTraceType()) {
						scan.nextLine();
					}

					m_configurableAdded = false;
					while (scan.next().equals(UIConstants.TIMELINE_CONT_SOURCE)) {
						String src = scan.nextLine();
						
						if (getIsFilteredEditor()) {
							if (!m_timelineInput.getSourceIsDisplayed(src)) {
								//Must skip the context part if source is not displayed 
								while (scan.next().equals(UIConstants.TIMELINE_CONT_CONTEXT)) {
									scan.next();
								}
								continue;
							}
						}
												
						createSourceTab(getTabFolder(), src, scan);
					}

					if (scan != null) {
						scan.close();
						scan = null;
					}
					
					//Select first tab
					if (getTabFolder().getItemCount() > 0) {
						getTabFolder().setSelection(0);
					}
				}
			} catch (FileNotFoundException e1) {
				LOGGER.error(UIMessages.TimelineInput_not_found, e1);
				e1.printStackTrace();
			}
		}
	}

	private void makeCustomActions(String source,
			TraceTimelinePlot traceTimelinePlot, Scanner scan) {
		ToolBar toolBar = traceTimelinePlot.getToolBar();

		List<String> srcContexts = new ArrayList<String>();
		
		while (scan.next().equals(UIConstants.TIMELINE_CONT_CONTEXT)) {
			srcContexts.add(scan.next());
		}

		new ToolItem(toolBar, SWT.SEPARATOR);

		if (srcContexts.size() == 1) { // Single context
			final ToolItem editGroups = new ToolItem(toolBar, SWT.PUSH);

			editGroups.setText(UIMessages.TraceTimelinePlot_6);
			editGroups.setData(srcContexts.get(0));
			editGroups.setToolTipText(UIMessages.TraceTimelinePlot_7);
			editGroups.addSelectionListener(new SelectionAdapter() {
				@Override
				public void widgetSelected(SelectionEvent e) {
					try {

						editGroups((String) editGroups.getData());
					} catch (Exception ex) {
						ex.printStackTrace();
					}
				}
			});
		} else if (srcContexts.size() > 1) { // Multi-context
			final ToolItem editGroups = new ToolItem(toolBar, SWT.DROP_DOWN);

			editGroups.setText(UIMessages.TraceTimelinePlot_6);
			editGroups.setData(srcContexts);

			editGroups.setToolTipText(UIMessages.TraceTimelinePlot_7);
			editGroups.addListener(SWT.Selection, new Listener() {
				public void handleEvent(Event event) {
					Shell shell = new Shell(event.display);
					MenuManager menuMgr = new MenuManager();
					Menu menu = menuMgr.createContextMenu(shell);

					IMenuManager subMenu = new MenuManager(
							UIMessages.TraceTimelineEditor_12, "context"); //$NON-NLS-1$

					@SuppressWarnings("unchecked")
					List<String> contexts = (List<String>) editGroups.getData();

					Iterator<String> itr = contexts.iterator();
					while (itr.hasNext()) {
						subMenu.add(new Action(itr.next(), null) {
							public void run() {
								editGroups(getText());
							}
						});
					}

					menuMgr.add(subMenu);
					menuMgr.add(new GroupMarker("other-actions")); //$NON-NLS-1$

					menu.setVisible(true);
				}
			});
		} else {
			// No context specified
		}

		toolBar.redraw();

		if (m_timeUnitMenu) {
			// Create time unit settings menu.
			String decodedTraceConfigPath = m_analysisResultsPath
					+ IPath.SEPARATOR + m_baseName + "." //$NON-NLS-1$
					+ SaConstants.decoded_trace_config_extension;

			m_timeUnitGroup = TimeUnitGroup
					.getGroupForConfig(decodedTraceConfigPath);

			// Do this only once
			if (!m_configurableAdded) {
				m_timeUnitGroup.addParent((IConfigurableDisplay) this);
				m_timeUnitGroup.addParent((IRefreshableDisplay) this);
				m_configurableAdded = true;
			}

			final ToolItem configureColumnsToolItem = new ToolItem(toolBar,
					SWT.DROP_DOWN);

			configureColumnsToolItem
					.setText(UIMessages.TimeUnit_Configure_Table);
			configureColumnsToolItem
					.setToolTipText(UIMessages.TimeUnit_Configure_Table);

			configureColumnsToolItem.addListener(SWT.Selection, new Listener() {
				public void handleEvent(Event event) {
					Shell shell = new Shell(event.display);
					MenuManager menuMgr = new MenuManager();
					Menu menu = menuMgr.createContextMenu(shell);

					if (m_timeUnitGroup != null) {
						m_timeUnitGroup.fillContextMenu(menuMgr);
					}

					menu.setVisible(true);
				}
			});
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.part.WorkbenchPart#setFocus()
	 */
	@Override
	public void setFocus() {
		if (getTabFolder() != null) {
			CTabItem sel = getTabFolder().getSelection();
			
			if (sel != null) {
				String source = (String) sel.getData(SOURCE);
				if (source != null) {
					TraceTimelinePlot ts = m_timelineInput.getPlot(source);
					ts.setFocus();
				}
			}
		}
	}

	@Override
	public void dispose() {
		super.dispose();

		if (m_timeUnitGroup != null) {
			m_timeUnitGroup.removeParent((IConfigurableDisplay) this);
			m_timeUnitGroup.removeParent((IRefreshableDisplay) this);
		}
	}

	/**
	 * 
	 * @param theInput
	 *            the trace editor input used to initialize the specific trace
	 *            model
	 */
	protected synchronized void startModel(final ITraceEditorInput theInput) {

		m_timelineInput.setInput(theInput);

		m_functionModel = m_timelineInput;
	}

	

	/**
	 * Refresh editor input using new trace data.
	 */
	public synchronized void refresh() {

		// This must be called only 1 time when changing frequency

		m_timelineInput.clearTimelineData(false); // don't clear the groups

		m_timelineInput.initTimelineData((ITraceEditorInput) getEditorInput()); // assume we have custom groups

		m_timelineInput.resizePlots();

	}

	public synchronized void editGroups(String context) {
		CTabItem sel = getTabFolder().getSelection();

		if (sel != null) {
			String source = (String) sel.getData(SOURCE);
			TimelineSource ts = m_timelineInput.getSource(source);

			// use previous m_allGroups, if available. If not build a new one
			if (ts.getAllGroups(context) == null) {
				ts.setAllGroups(
						context,
						new Vector<TraceTimelinePlot.Group>(ts
								.getGroups(context)));

				// remove the "other" group - it is best to be left unseen when
				// editing groups
				if (ts.getOtherGroupId(context) >= 0) {
					ts.getAllGroups(context)
							.remove(ts.getOtherGroupId(context));
				}
			}

			EditTimelineGroupsDialog dialog = new EditTimelineGroupsDialog(
					m_parent.getShell(), m_functionModel,
					ts.getAllGroups(context), source, context);
			dialog.setBlockOnOpen(true);
			int res = dialog.open();

			if (res == Window.OK) {
				// update m_groups from m_allGroups, leaving only the visible
				// ones
				ts.getGroups(context).clear();
				for (TraceTimelinePlot.Group gr : ts.getAllGroups(context)) {
					if (gr.getShowIndex() >= 0) {
						ts.getGroups(context).add(gr);
					}
				}
				ts.setCustomGroups(true);
				m_timelineInput.clearTimelineData(false); // don't clear the groups

				m_timelineInput.initTimelineData((ITraceEditorInput) getEditorInput());
			}
		}
	}

	public String getBaseName() {
		return m_baseName;
	}

	@Override
	public String getDisplayId() {
		return "TimelineDisplay"; //$NON-NLS-1$
	}

	@Override
	public void refreshDisplay() {
		refresh();
	}

	@Override
	public void configure(CONFIG_ELEMENT elementType, String value) {
		// This method is called when one of the config elements (e.g.: time
		// unit, freq) is changed by the user form the UI, using the 'configure
		// table' menu actions.

		switch (elementType) {
		case TIME_UNIT_ELM:
			configureTimeUnit(value);
			break;

		case FREQ_ELM:
			configureFrequency(value);
			break;

		default:
			LOGGER.error("[configure]: unknown CONFIG_ELEMENT: " + elementType); //$NON-NLS-1$
		}
	}

	/**
	 * Update the time unit in the viewer.
	 * 
	 * @param timeUnit
	 *            the new time unit (see types defined in
	 *            {@link IConfigurableDisplay}).
	 */
	private void configureTimeUnit(String timeUnit) {
		// Sync with other viewers
		String decodedTraceConfigPath = m_analysisResultsPath + IPath.SEPARATOR
				+ m_baseName + "." //$NON-NLS-1$
				+ SaConstants.decoded_trace_config_extension;

		TimeUnitUtils.saveTimeUnit(decodedTraceConfigPath, timeUnit);

		m_timelineInput.setTimeUnit(timeUnit);
	}

	/**
	 * Convert the frequency value from string to double and update its value.
	 * 
	 * @param value
	 *            The frequency value, as a string.
	 */
	private void configureFrequency(String value) {
		double freq = -1;
		try {
			freq = Double.parseDouble(value);
		} catch (NumberFormatException e) {
			e.printStackTrace();
		}

		// Sync with other viewers
		String decodedTraceConfigPath = m_analysisResultsPath + IPath.SEPARATOR
				+ m_baseName + "." //$NON-NLS-1$
				+ SaConstants.decoded_trace_config_extension;

		TimeUnitUtils.saveFrequency(decodedTraceConfigPath, freq);

		m_timelineInput.setFrequency(freq);
	}

	@Override
	public String getEditorType() {
		if (m_timelineInput == null || m_timelineInput.getTraceType() == null || m_timelineInput.getTraceType().isEmpty()) {
			return "Timeline"; //$NON-NLS-1$
		}
		
		return m_timelineInput.getTraceType() + " Timeline"; //$NON-NLS-1$
	}

	@Override
	public Image getHeaderImage() {
		return TimelineUIPlugin.getImageDescriptor(
				UIConstants.results_timeline_iconfile).createImage();
	}

	@Override
	public String getStoragePath() {
		return m_analysisResultsPath;
	}

	public CTabFolder getTabFolder() {
		return m_tabFolder;
	}

	public void setTabFolder(CTabFolder m_tabFolder) {
		this.m_tabFolder = m_tabFolder;
	}

	protected boolean getIsFilteredEditor() {
		return m_isFilteredEditor;
	}

	protected void setIsFilteredEditor(boolean m_isFilteredEditor) {
		this.m_isFilteredEditor = m_isFilteredEditor;
	}

}
