/*******************************************************************************
 * Copyright 2016-2018 NXP
 *
 * Copyright (c) 2014 - 2016 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.timeline;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;
import java.util.Vector;
import java.util.Map.Entry;

import org.apache.log4j.Logger;
import org.eclipse.swt.graphics.RGB;

import com.freescale.sa.ui.common.BaseTraceEventData;
import com.freescale.sa.util.CommonConstants;



public class TimelineSource {
	
	private class TimelineContext {
		public Vector<TraceTimelinePlot.Group> m_groups = new Vector<TraceTimelinePlot.Group>();
		public Vector<TraceTimelinePlot.Bar>   m_points = new Vector<TraceTimelinePlot.Bar>();
		
		private BigInteger m_timestampOfPreviousEvent = new BigInteger("-1"); //$NON-NLS-1$
		
		/**
		 * The length of the timeline bar of the function that is currently being
		 * processed.
		 */
		private long m_barLength = 0;
				
		private int m_otherGroupId = invalidGroupId;
		
		private int m_noDbgGroupId = invalidGroupId;
		private int m_errorGroupId = invalidGroupId;
		
		private Map<String, Integer> m_customEventGroupsMap = new HashMap<String, Integer>();

		private int m_prevGroupIdx = invalidGroupId;
		
		private TimelineFunction m_prevFunc = null;

		private TimelineFunction errorFunc = new TimelineFunction(ERROR_GROUP_NAME, BigInteger.ZERO, BigInteger.ZERO);
		private TimelineFunction noDbgFunc = new TimelineFunction(NO_DEBUG_INFO_FUNC, BigInteger.ZERO, BigInteger.ZERO);
		private TimelineFunction otherFunc = new TimelineFunction(OTHER_FUNC, BigInteger.ZERO, TimelineConstants.BIG_INTEGER_64_BIT_MAX_VALUE);
		
		private Map<String, TimelineFunction> m_customFunctions = new HashMap<String, TimelineFunction>(); 
		
		private boolean m_customGroups = false;
		
		/**
		 * The timeline groups, the complete set, including the hidden ones.
		 */
		private Vector<TraceTimelinePlot.Group> m_allGroups = null;
		
		
		private ITimelineFunctionModel m_timelineInput;
		
		/*------------------------------------------------------------------------------------*/
		
		/**
		 * Public constructor
		 * @param timelineInput an instance of the timeline input class
		 */
		public TimelineContext(ITimelineFunctionModel timelineInput) {
			m_timelineInput = timelineInput;
		}
		
		/**
		 * Updates the timeline graphic information for the given function (adds a
		 * new bar for the function). If the function doesn't exist in the timeline
		 * m_groups, it will be added.
		 * 
		 * @param groupId
		 *            the timeline group id
		 * @param horisontalPos
		 *            the horizontal position of the functions bar.
		 * @param barLength
		 *            the length of the functions bar
		 * @return the index of the group this function belongs to
		 */
		public void updateTimelineGraphic(int groupIdx, long horisontalPos, long barLength) {
			
			m_points.add(new TraceTimelinePlot.Bar(horisontalPos, groupIdx, barLength));

			if (groupIdx < 0) {
				LOGGER.debug("[updateTimelineGraphic]: invalid group (negative index)"); //$NON-NLS-1$
			}
			else {
				String msg = "[updateTimelineGraphic]: added m_points for [" + groupIdx + "]:" + " x = " //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
					+ horisontalPos + "; y = " + groupIdx + "; length = " + barLength; //$NON-NLS-1$ //$NON-NLS-2$
				LOGGER.debug(msg);
			}

		}
		
		/**
		 * Calculates percentage from total running time for each function group
		 * 
		 */
		public void processPercentages() {

			if (m_points.size() <= 0) {
				return;
			}
			
			// initialize a vector to store the running time for each group
			Vector<Float> groupsTotal = new Vector<Float>();
			
			for (int i = 0; i < m_groups.size(); i++) {
				groupsTotal.add(new Float(0.0f));
			}
			
			float total = 0;
			
			// compute running time for each group		
			for (TraceTimelinePlot.Bar bar: m_points) {
				if (bar.y >= m_groups.size()) {
					LOGGER.error("[processPercentages] There are more function bars than functions."); //$NON-NLS-1$
					return;
				}
				
				groupsTotal.set(bar.y, groupsTotal.get(bar.y) + bar.xEnd + 1 - bar.xStart);
				total += bar.xEnd + 1 - bar.xStart;
			}
			
			float percentFactor = total > 0.0f ? 100.0f / total : 0.0f;
			
			// compute and store percentages
			for (int i = 0; i < m_groups.size(); i++) {
				float percent = percentFactor * (float)groupsTotal.get(i);
				m_groups.elementAt(i).setPercent(percent);
			}
		}
		
		public void postProcess() {
		    
			if (m_barLength > 0) {
				// add the last bar to graphic
				if (m_customGroups) {
					updateTimelineGraphic(m_prevGroupIdx, m_prevLength, m_barLength);
					m_prevLength += m_barLength;
					m_barLength = 0;
				}
				else {
					int y = findGroup(m_prevFunc);
					updateTimelineGraphic(y, m_prevLength, m_barLength);
					m_prevGroupIdx = y;			

					m_prevLength += m_barLength;
					m_barLength = 0;
				}
			}
			processPercentages();
		}
		
		public void preProcess(){
			m_prevFunc = null;
			m_prevGroupIdx = invalidGroupId;
			m_timestampOfPreviousEvent = new BigInteger("-1"); //$NON-NLS-1$
			
			// init the 'special' groups
			if (m_groups.size() > 0) {
				m_noDbgGroupId = invalidGroupId;
				m_otherGroupId = invalidGroupId;
				m_errorGroupId = invalidGroupId;
				for (int i = 0; i < m_groups.size(); i++) {
					TraceTimelinePlot.Group group = m_groups.get(i);
					if (group.getName().equals(noDbgFunc.getName())) {
						m_noDbgGroupId = i;
						continue;
					}
					if (group.getName().equals(otherFunc.getName())) {
						m_otherGroupId = i;
						continue;
					}
					if (group.getName().equals(errorFunc.getName())) {
						m_errorGroupId = i;
						continue;
					}
				}
			}
		}
		
		public void processProgramTraceEvent(BaseTraceEventData eventData) {
			
			String funcName = eventData.getSrcFuncname();
			String funcSource = eventData.getSource();
			String funcContext = eventData.getContext();
									
			TimelineFunction crtFunc = null;
			
			if (funcName == null || funcName.length() == 0) {
				crtFunc = noDbgFunc;
			} // otherwise we have a valid function name
			else 
				if ((m_prevFunc == null) || (m_prevFunc.getName() != funcName)) {
				    String addr = eventData.getFuncAddress();
				    if (addr != null && !addr.isEmpty()) {
				        crtFunc = m_functionModel.getFunction(new BigInteger(addr, 16), funcSource, funcContext);
				    }
				} //otherwise crtFunc is the right one
				else {
					crtFunc = m_prevFunc;
				}

			if (m_prevFunc == null) {
				m_prevFunc = crtFunc;
			}
			
			if (m_customGroups) {
				//work by each address
				
				String addressStr = eventData.getSrcInstAddress();
				BigInteger address = TimelineFunction.INVALID_ADDRESS;
				if ((addressStr != null) && (addressStr != "")) { //$NON-NLS-1$
					address = getAddressLong(addressStr);
				}
				
				if (address != TimelineFunction.INVALID_ADDRESS) {
					// if there is no function name, then we have no debug info
					
					if (m_prevGroupIdx < 0){
						m_prevGroupIdx = findGroup(address, funcName);
						// nothing else to be done
					}
					else if (m_prevGroupIdx == m_otherGroupId) {
						int newGroupIdx = findGroup(address, funcName);
						if ((newGroupIdx != m_otherGroupId) && (m_barLength > 0)) {
							updateTimelineGraphic(m_prevGroupIdx, m_prevLength, m_barLength);
							m_prevLength += m_barLength;
							m_barLength = 0;
						}
						m_prevGroupIdx = newGroupIdx;
					} 
					else if (!m_groups.get(m_prevGroupIdx).containsAddress(address, m_groups)) {
						if (m_barLength > 0) {
							updateTimelineGraphic(m_prevGroupIdx, m_prevLength, m_barLength);
							m_prevLength += m_barLength;
							m_barLength = 0;
						}

						m_prevGroupIdx = findGroup(address, funcName);
					}
					
					// when there is more than one assembly instruction for the event,
					// check if all disassembly instructions in the eventData
					// belong to the same group. If not, split the event data in two.
					// TODO: critical code (TinelineFunction vector in the visitor) does not contain
					// any instructions from the symbols without debug info. We need to add them to 
					// a 'no debug info' function and display it in critical code, also use it here
					if ( (eventData.getNumAssembly() > 1) && (crtFunc != null) && (crtFunc.getNumInstructions() > 1)) {
						TraceTimelinePlot.Group crtGroup = m_groups.get(m_prevGroupIdx);
						
						if (crtGroup.getSize() > 0) {
							try {
								// identify the instructions in crt group
								int initialIdx = crtFunc.contains(address); // index of first inst in eventData
								
								if (eventData.getNumAssembly() > (crtFunc.getNumInstructions() - initialIdx)) {
									LOGGER.error("Mismatch between num instructions of crt program trace event and num of assemly of crt function ("  //$NON-NLS-1$
											+ crtFunc.getName() + "). Crt group (" + crtGroup.getName() + ") may be out of sync"); //$NON-NLS-1$ //$NON-NLS-2$
									assert (false);
								}
								if (initialIdx < 0) {
									LOGGER.error("Invalid number of instructions for crt program trace event"); //$NON-NLS-1$
									assert (false);
								}
								
								for (int idx = initialIdx; idx < eventData.getNumAssembly(); idx++) {
									BigInteger instAddress = crtFunc.getAddressForInstruction(idx);
									boolean needSplit = false;
									
									if (crtGroup.isSpecial()) {
										// find out if another group contains this instruction's address
										int y = findGroup(instAddress, crtFunc.getName());
										if ((y != invalidGroupId) && (y != m_prevGroupIdx)) {
											needSplit = true;
										}
									} else {
										if (!crtGroup.containsAddress(instAddress, m_groups)) {
											needSplit = true;
										}
									}
									if (needSplit) {
										// split event data
										BaseTraceEventData newEventData = new BaseTraceEventData();
										newEventData.init(eventData);
										// update the new event data
										newEventData.setSrcInstAddress(instAddress.toString(16));
										newEventData.setNumAssembly(eventData.getNumAssembly() - idx);
										
										// update assembly for current event data
										eventData.setNumAssembly(idx - initialIdx);
										
										// update the timestamp - by substracting the assembly instructions
										// to be assigned to newEventData. 
										BigInteger timestamp = eventData.getTimestamp();
										if (timestamp.compareTo(BigInteger.ZERO) > 0) {
											eventData.setTimestamp(timestamp.subtract(new BigInteger("" + newEventData.getNumAssembly()))); //$NON-NLS-1$
										}
			
										// finalize processing of crt event data
										updateTimelineData(eventData);
										m_prevFunc = crtFunc;
										
										// process the new event data
										processProgramTraceEvent(newEventData);
										
										// return, all done
										return;
									}
								}
							} catch (AssertionError er) {
								LOGGER.error("[processProgramTraceEvent]:", er); //$NON-NLS-1$
							}
						}
					}
				}

			} else {
				// work by function
				if (!m_prevFunc.getName().equals(funcName)) {
					
					if (m_barLength > 0) {
						int y = findGroup(m_prevFunc);
						updateTimelineGraphic(y, m_prevLength, m_barLength);
						m_prevGroupIdx = y;
						
						m_prevLength += m_barLength;
						m_barLength = 0;
					}
				}
			}
			
			updateTimelineData(eventData);
			m_prevFunc = crtFunc;
		}
		
		/**
		 * Process an program trace error event.
		 * 
		 * @param eventData
		 *            the event that contains the error entry
		 * @param funcName
		 *            the name of the function where error occured
		 */
		public void processErrorEntry(BaseTraceEventData eventData) {
			
			if (m_prevFunc == null) {
				m_prevFunc = errorFunc;
			}
			
			if (m_customGroups) {
				int newGroupIdx = findErrorGroup();
				if (m_barLength > 0) {
					// steal one cycle from the previous bar, in order to maintain 
					// overall cycle numbers in sync with other views
					m_barLength--;
					updateTimelineGraphic(m_prevGroupIdx, m_prevLength, m_barLength);
					m_prevLength += m_barLength;
					m_barLength = 1; // assume size 1 for the error event
				}
				m_prevGroupIdx = newGroupIdx;

			} else {
				if (m_barLength > 0) {
					// steal one cycle from the previous bar, in order to maintain 
					// overall cycle numbers in sync with other views
					m_barLength--;
					
					int y = findGroup(m_prevFunc);
					updateTimelineGraphic(y, m_prevLength, m_barLength);
					m_prevGroupIdx = y;

					m_prevLength += m_barLength;
					m_barLength = 1; // assume size 1 for the error event
				}
			}
			
			updateTimelineData(eventData);
			m_prevFunc = errorFunc; 
		}
		
		/**
		 * Process an program trace custom event - E.g. AIOP task terminates
		 * 
		 * @param eventData
		 *            the event that contains the error entry
		 * @param funcName
		 *            the name of the function where error occured
		 */
		public void processCustomEntry(BaseTraceEventData eventData) {
			
			// identify function
			TimelineFunction func;
			String funcName = eventData.getFuncname();
			if (!m_customFunctions.containsKey(funcName)) {
				func = new TimelineFunction(funcName, BigInteger.ZERO, BigInteger.ZERO);
				m_customFunctions.put(funcName, func);
			}
			else {
				func = m_customFunctions.get(funcName);
			}
					
			if (m_prevFunc == null) {
				m_prevFunc = func;
			}
			
			if (m_customGroups) {
				int newGroupIdx = findCustomEventGroup(eventData.getFuncname());
				if (m_barLength > 0) {
					// steal one cycle from the previous bar, in order to maintain 
					// overall cycle numbers in sync with other views
					m_barLength--;
					updateTimelineGraphic(m_prevGroupIdx, m_prevLength, m_barLength);
					m_prevLength += m_barLength;
					m_barLength = 1; // assume size 1 for the custom event
				}
				m_prevGroupIdx = newGroupIdx;

			} else {
				if (m_barLength > 0) {
					// steal one cycle from the previous bar, in order to maintain 
					// overall cycle numbers in sync with other views
					m_barLength--;
					
					int y = findGroup(m_prevFunc);
					updateTimelineGraphic(y, m_prevLength, m_barLength);
					m_prevGroupIdx = y;

					m_prevLength += m_barLength;
					m_barLength = 1; // assume size 1 for the custom event
				}
			}
			
			updateTimelineData(eventData);
			m_prevFunc = func; 
		}		
		
		/**
		 * Update the timeline data with the information for the current event. If
		 * the timestamp value is not updated for the current event, we will
		 * consider each assembly instruction lasted one cycle, and take this as the
		 * delta time between events.
		 * 
		 * @param eventData
		 *            the data for the current event.
		 */
		private void updateTimelineData(BaseTraceEventData eventData) {
			int numAssembly = 0;
			BigInteger crtTimestamp;

			if (eventData == null) {
				return;
			}
			numAssembly = eventData.getNumAssembly();
			crtTimestamp = eventData.getTimestamp();
			
			if (crtTimestamp.compareTo(BigInteger.ZERO) == 0 && m_prevTimestamp.compareTo(BigInteger.ZERO) == 0 && numAssembly != 0) {
				// no timestamp available
				m_barLength += numAssembly;				
			}
			else if (m_prevTimestamp.equals(BigInteger.ZERO)) {
				// first event
				m_prevTimestamp = crtTimestamp;
				m_barLength += numAssembly;
			}
			else if (crtTimestamp.subtract(m_prevTimestamp).compareTo(BigInteger.ZERO)  > 0) {
				// timestamps available
				m_barLength += crtTimestamp.subtract(m_prevTimestamp).longValue();
				m_prevTimestamp = crtTimestamp;	
			}
		}
		
		private int findGroup(TimelineFunction func) {

			if ((func == null) || func.getName().equals(errorFunc.getName())) {
				return findErrorGroup();
			}
			
			if (m_customFunctions.containsKey(func.getName())) {
				return findCustomEventGroup(func.getName());
			}
			
			if (func.getName().equals(noDbgFunc.getName())) {
				return findNoDebugGroup();
			}
			
			int y;				
			
			for (y = 0; y < m_groups.size(); y++) {
				TraceTimelinePlot.Group grp = m_groups.elementAt(y);
				if (grp.getSource().equals(func.getSource()) && (grp.getContext().equals(func.getContext()))) {
					if ( (!grp.getName().equals(otherFunc.getName()))
							&& ((grp.getName().equals(func.getName())
							)
							|| (grp.containsAddress(func))) ) {
						grp.updateStartAndEndAddresses(func.getStartAddress(), func.getEndAddress().subtract(BigInteger.ONE));
					    break;
					}
				}
			}
			
			if (y == m_groups.size()) {
				BigInteger[] startAddr = new BigInteger[1];
				BigInteger[] endAddr = new BigInteger[1];
				startAddr[0] = func.getStartAddress();
				endAddr[0] = func.getEndAddress().subtract(BigInteger.ONE);
				
				m_groups.add(new TraceTimelinePlot.Group(
						func.getName(), startAddr, endAddr,
						func.isInline() ? TraceTimelinePlot.getInlineColor().getRGB() : TraceTimelinePlot.getDefaultPointColor().getRGB(),
						y + 1, func.getSource(), func.getContext()));
			}
			
			return y;
		}
		
		private int findGroup(BigInteger address, String funcName) {
			if (address == TimelineFunction.INVALID_ADDRESS) {
				return invalidGroupId;
			}
			
			// see if 'no debug' group can be used
			if ((funcName == null) || (funcName.equals(""))) { //$NON-NLS-1$
				return findNoDebugGroup();
			}
			
			if (funcName.equals(errorFunc.getName())) {
				return findErrorGroup();
			}
			
			if (m_customFunctions.containsKey(funcName)) {
				return findCustomEventGroup(funcName);
			}
			
			// search 'normal' groups
			for (int y = 0; y < m_groups.size(); y++) {
				if (y != m_otherGroupId) {
					TraceTimelinePlot.Group gr = m_groups.elementAt(y);
					if (gr.containsAddress(address, m_groups)) {
						// if group is invisible, "don't find" it
						if (gr.getShowIndex() >= 0)
							return y;
					}
				}
			}
			
			// then use the 'other' group
			return findOtherGroup();
		}
				
		private int findOtherGroup() {
			if (m_otherGroupId < 0) {
				// not created yet
				BigInteger[] startAddr = new BigInteger[1];
				BigInteger[] endAddr = new BigInteger[1];
				startAddr[0] = otherFunc.getStartAddress();
				endAddr[0] = otherFunc.getEndAddress();
				TraceTimelinePlot.Group group = new TraceTimelinePlot.Group(
						otherFunc.getName(), startAddr, endAddr,
						TraceTimelinePlot.getDefaultSpecialGroupColor().getRGB(),
						Integer.MAX_VALUE); // always show last
				group.setSpecial(true);
				m_groups.add(group);
				
				m_otherGroupId = m_groups.size() - 1;
			} /* else we already have an 'other' group */
			
			return m_otherGroupId;
		}
						
		private int findNoDebugGroup() {
			if (m_noDbgGroupId < 0) {
				// not created yet
				BigInteger[] startAddr = new BigInteger[1];
				BigInteger[] endAddr = new BigInteger[1];
				startAddr[0] = noDbgFunc.getStartAddress();
				endAddr[0] = noDbgFunc.getEndAddress();
				TraceTimelinePlot.Group group = new TraceTimelinePlot.Group(
						noDbgFunc.getName(), startAddr, endAddr,
						TraceTimelinePlot.getDefaultSpecialGroupColor().getRGB(),
						Integer.MAX_VALUE - 2); // always show last, before 'other' and 'error'
				group.setSpecial(true);
				m_groups.add(group);
				m_noDbgGroupId = m_groups.size() - 1;
			} /* else we already have an 'other' group */
			
			return m_noDbgGroupId;
		}
					
		private int findErrorGroup() {
			if (m_errorGroupId < 0) {
				// not created yet
				BigInteger[] startAddr = new BigInteger[1];
				BigInteger[] endAddr = new BigInteger[1];
				startAddr[0] = errorFunc.getStartAddress();
				endAddr[0] = errorFunc.getEndAddress();
				TraceTimelinePlot.Group group = new TraceTimelinePlot.Group(
						errorFunc.getName(), startAddr, endAddr,
						TraceTimelinePlot.getDefaultSpecialGroupColor().getRGB(),
						Integer.MAX_VALUE - 1);
				group.setSpecial(true);
				m_groups.add(group);
				m_errorGroupId = m_groups.size() - 1;
			} /* else we already have an 'error' group */
			
			return m_errorGroupId;
		}
		
		private int createCustomEventGroup(String eventName) {
			
			// Get event color from input if it exists
			RGB color = TraceTimelinePlot.getDefaultIdleGroupColor().getRGB();
			if (m_timelineInput != null) {
				ArrayList<TimelineCustomEvent> customEventsList = 
						m_timelineInput.getCustomEvents();
				for (TimelineCustomEvent e: customEventsList) {
					if (e.getName().equals(eventName)) {
						color = e.getColor();
						break;
					}
				}
			}
			
			BigInteger[] startAddr = new BigInteger[1];
			BigInteger[] endAddr = new BigInteger[1];
			startAddr[0] = BigInteger.ZERO;
			endAddr[0] = BigInteger.ZERO;
			TraceTimelinePlot.Group group = new TraceTimelinePlot.Group(
					eventName, startAddr, endAddr,
					color,
					Integer.MAX_VALUE - 1);
			group.setSpecial(true);
			m_groups.add(group);
			m_customEventGroupsMap.put(eventName, m_groups.size() - 1);
			return m_groups.size() - 1;
		}
		
		private int findCustomEventGroup(String eventName) {
			if (!m_customEventGroupsMap.containsKey(eventName)) {
				// not created yet
				return createCustomEventGroup(eventName);
			} 
			else {
				return m_customEventGroupsMap.get(eventName);
			}
			
		}
		
		public void clearData(boolean clearGroups){
			
			m_points.clear();
						
			/* Sometimes we may want to reuse
			 * the groups but clear the rest of the data (e.g. reuse
			 * the groups created by EditTimelineGroupsDialog)
			 */
			if (clearGroups) {
				m_groups.clear();
				if (m_allGroups != null) {
					m_allGroups.clear();
					m_allGroups = null;
				}
			}
						
			m_prevTimestamp = BigInteger.ZERO;
			m_barLength = 0;
			m_prevLength = 0;
		}
		
		public void processContextInEvent(BaseTraceEventData eventData) {
			BigInteger timestampOfThisEvent = eventData.getTimestamp();

			if (timestampOfThisEvent.compareTo(BigInteger.ZERO) > 0) {
				if (m_timestampOfPreviousEvent.compareTo(timestampOfThisEvent) > 0) {
					m_prevTimestamp = m_prevTimestamp.subtract(m_timestampOfPreviousEvent.subtract(timestampOfThisEvent));
				}
				
				m_timestampOfPreviousEvent = timestampOfThisEvent;
				m_prevTimestamp = timestampOfThisEvent;
			}
		}
		
		public void processContextOutEvent(BaseTraceEventData eventData) {
						
			updateTimelineData(eventData);
			
			if (m_customGroups) {
				if (m_prevGroupIdx >= 0){
					if (m_barLength > 0) {
						updateTimelineGraphic(m_prevGroupIdx, m_prevLength, m_barLength);
						m_prevLength += m_barLength;
						m_barLength = 0;
					}
				}
			} else {
				if (m_barLength > 0) {
					int y = findGroup(m_prevFunc);
					updateTimelineGraphic(y, m_prevLength, m_barLength);
					m_prevGroupIdx = y;
					
					m_prevLength += m_barLength;
					m_barLength = 0;
				}
			}
			
		}
		
		public void processEventData(BaseTraceEventData eventData) {		
			if (isProgramTraceError(eventData)) {
				LOGGER.debug("[processEventData]: event with ts " + eventData.getTimestamp() + " is a program trace error"); //$NON-NLS-1$ //$NON-NLS-2$
				processErrorEntry(eventData);
				
				return;
			}
			
			if (isContextInEvent(eventData)) {
				processContextInEvent(eventData);
				
				return;
			}
			
			if (isContextOutEvent(eventData)) {
				processContextOutEvent(eventData);
				
				return;
			}
			
			if (isCustomEvent(eventData)) {
				processCustomEntry(eventData);
				
				return;
			}
			
			if (isProgramTraceEventData(eventData)) {
				BigInteger timestampOfThisEvent = eventData.getTimestamp();

				if (timestampOfThisEvent.compareTo(BigInteger.ZERO) > 0) {
					if (m_timestampOfPreviousEvent.compareTo(timestampOfThisEvent) > 0) {
						m_prevTimestamp = m_prevTimestamp.subtract(m_timestampOfPreviousEvent.subtract(timestampOfThisEvent));
					} 
					m_timestampOfPreviousEvent = timestampOfThisEvent;
				}
				
				processProgramTraceEvent(eventData);
				return;
			}		
			
			LOGGER.debug("[processEventData]: Unidentified event type!"); //$NON-NLS-1$
		}
		
		private boolean isProgramTraceError(BaseTraceEventData eventData) {			
			return eventData.getLabel().equals(TimelineConstants.ERROR_EVT_LABEL);
		}
		
		private boolean isCustomEvent(BaseTraceEventData eventData) {
			
			return eventData.getLabel().equals(TimelineConstants.CUSTOM_EVT_LABEL);
		}
		
		private boolean isProgramTraceEventData(BaseTraceEventData eventData) {
			return (eventData.getLabel().equals(TimelineConstants.BRANCH_EVT_LABEL)
					|| eventData.getLabel().equals(TimelineConstants.FUNCTION_CALL_EVT_LABEL)
					|| eventData.getLabel().equals(TimelineConstants.FUNCTION_RETURN_EVT_LABEL)
					|| eventData.getLabel().equals(TimelineConstants.INTERRUPT_EVT_LABEL)
					|| eventData.getLabel().equals(TimelineConstants.LINEAR_EVT_LABEL));
		}
		
		private boolean isContextInEvent(BaseTraceEventData eventData) {
			return eventData.getLabel().equals(TimelineConstants.CONTEXT_IN);
		}
		
		private boolean isContextOutEvent(BaseTraceEventData eventData) {
			return eventData.getLabel().equals(TimelineConstants.CONTEXT_OUT);
		}
		
		/**
		 * Get a long value out of the address string (hex value)
		 * @param address - the address string in hex as provided in event data
		 * @return the address as a long
		 */
		private BigInteger getAddressLong(String address) {
			try {
				return new BigInteger(address, 16);
			} catch (Exception e) {
				String msg = "[getAddressLong] Address (\"" + address + "\") is empty or non-numerical"; //$NON-NLS-1$ //$NON-NLS-2$ $NON-NLS-2$
				LOGGER.warn(msg);
				return TimelineFunction.INVALID_ADDRESS;
			}
		}
	}
	
	protected ITimelineFunctionModel m_functionModel = null;
		
	private Hashtable<String, TimelineContext> m_groups = new Hashtable<String, TimelineContext>(); 
	/**
	 * The timeline groups, the working set.
	 */
	//private Vector<TraceTimelinePlot.Group> m_groups = new Vector<TraceTimelinePlot.Group>();
		
	/**
	 * Class logger.
	 */
	private static Logger LOGGER = Logger.getLogger(TraceTimelineEditor.class);
	
	public final static String NO_DEBUG_INFO_FUNC = "<no debug info>"; //$NON-NLS-1$
	public final static String OTHER_FUNC = "<other>"; //$NON-NLS-1$
	public static final String ERROR_GROUP_NAME = "<error>"; //$NON-NLS-1$
	
	private static final int invalidGroupId = -1;
		
	/**
	 * This members keeps the current x position in the timeline graphic
	 * (excluding the time information for the event that is currently
	 * processed).
	 */
	private long m_prevLength = 0;
	
	/**
	 * The timestamp of the event previously processed. Used to compute the time
	 * difference between events.
	 */
	private BigInteger m_prevTimestamp = BigInteger.ZERO;
				
	/*-----------------------------------------------------------------*/
	
	public Vector<TraceTimelinePlot.Bar> getPoints() {
		
		Vector<TraceTimelinePlot.Bar> merge = new Vector<TraceTimelinePlot.Bar>();
		
		int absY = 0;
		
		for(String key: m_groups.keySet()) {
			
			if (m_groups.size() > 1) {
				TraceTimelinePlot.Bar contextBar = new TraceTimelinePlot.Bar(0, absY, 0);
				merge.add(contextBar);
				absY++;
			}
			
			for (int i = 0; i < m_groups.get(key).m_points.size(); i++) {
				TraceTimelinePlot.Bar plotBar = new TraceTimelinePlot.Bar(m_groups.get(key).m_points.elementAt(i).getX(),
						(int)m_groups.get(key).m_points.elementAt(i).getY() + absY,
						m_groups.get(key).m_points.elementAt(i).getCycles());
				merge.add(plotBar);
			}
			
			absY += m_groups.get(key).m_groups.size();
		}
		
		return merge;
	}
		
	public Vector<TraceTimelinePlot.Group> getGroups() {
		
		Vector<TraceTimelinePlot.Group> merge = new Vector<TraceTimelinePlot.Group>();
		
		for(String key: m_groups.keySet()) {
			if (m_groups.size() > 1) {
				RGB rgb = new RGB(100, 0, 100);
				TraceTimelinePlot.Group contextLine = new TraceTimelinePlot.Group("Context - " + key, rgb, 0);  //$NON-NLS-1$
				contextLine.setIsContext(true);
				merge.add(contextLine);
			}
			merge.addAll(m_groups.get(key).m_groups);
		}
		
		return merge;
	}
	
	public Vector<TraceTimelinePlot.Group> getGroups(String context) {
		return m_groups.get(context).m_groups;
	}
		
	public Vector<TraceTimelinePlot.Group> getAllGroups(String context) {
		return m_groups.get(context).m_allGroups;
	}
	
	public void setAllGroups(String context, Vector<TraceTimelinePlot.Group> m_allGroups) {
		this.m_groups.get(context).m_allGroups = m_allGroups;
	}
		
	public void processEventData(BaseTraceEventData eventData) {
		
		String funcContext = eventData.getContext();
		
		if (!m_groups.containsKey(funcContext)) {
			TimelineContext tc = new TimelineContext(m_functionModel);
			m_groups.put(funcContext, tc);
			tc.processProgramTraceEvent(eventData);
			
			return;
		}
		
		TimelineContext ctx = m_groups.get(funcContext);
		
		ctx.processEventData(eventData);									
	}
					
    public void preProcess() {
        for (Entry<String, TimelineContext> entry : m_groups.entrySet()) {
            entry.getValue().preProcess();
        }
    }

    public void postProcess() {
        for (Entry<String, TimelineContext> entry : m_groups.entrySet()) {
            entry.getValue().postProcess();
        }
    }
	
	public void setModel(TimelineInput input) {
		m_functionModel = input;
	}
	
    public void clearData(boolean clearGroups) {
        for (Entry<String, TimelineContext> entry : m_groups.entrySet()) {
            entry.getValue().clearData(clearGroups);
        }
    }
	
	public int getOtherGroupId(String context) {
		return m_groups.get(context).m_otherGroupId;
	}
	
	public boolean isSingleContext() {
		return m_groups.size() == 1;
	}
	
	public String getFirstContext() {
		for(String key: m_groups.keySet()) {
			return key;
		}
		
		return CommonConstants.EMPTY_STRING;
	}
		
	public boolean isCustomGroups(String context) {
		return m_groups.get(context).m_customGroups;
	}
	
	public void setCustomGroups(String context, boolean customGroups) {
		this.m_groups.get(context).m_customGroups = customGroups;
	}
	
    public void setCustomGroups(boolean customGroups) {
        for (Entry<String, TimelineContext> entry : m_groups.entrySet()) {
            entry.getValue().m_customGroups = customGroups;
        }
    }
}
