/*******************************************************************************
 * Copyright 2016-2018 NXP
 *
 * Copyright (c) 2014 - 2017 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.timeline;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.math.BigInteger;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Map.Entry;
import java.util.Queue;
import java.util.Scanner;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.StatusDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.progress.UIJob;

import com.freescale.sa.SaConstants;
import com.freescale.sa.ui.common.BaseTraceEventData;
import com.freescale.sa.ui.common.ITraceEditorInput;
import com.freescale.sa.ui.common.StorageEditorInput;
import com.freescale.sa.util.CommonConstants;

public class TimelineInput implements ITimelineFunctionModel {	
	private static Logger LOGGER = Logger.getLogger(TimelineInput.class);
	private TimelineFunctionInfoBase mFunctions;
	private Queue<BaseTraceEventData> mEvents;
	private String mTimelineFile = null;
	private Scanner mScanner = null; 
	private ArrayList<TimelineCustomEvent> mCustomEvents;
	
	private Integer mVersion = 0;
	
	private static final int MAX_TOTAL_EVENTS = 1000000;
	private static final int PROGRAM_EVENT_TYPE = 0;
	private static final int INFO_EVENT_TYPE = 1;
	private static final int ERROR_EVENT_TYPE = 2;
	private static final int TOTAL_COUNT_EVENT_TYPE = 3;
	private static final int CONTEXT_IN = 4;
	private static final int CONTEXT_OUT = 5;
	private static final int CONTEXT_SWITCH = 6;
	private static final int CUSTOM_EVENT_TYPE = 7;
	
	private static final String TRACE_TYPE_TAG = "***TraceType:"; //$NON-NLS-1$
    private static final String LINUX_KERNEL_SPACE_TRACE = "Linux Kernel Space"; //$NON-NLS-1$
	
	private static final int QUEUE_SIZE = 10;
	
	private int minIndex = 0;
	private int maxIndex = 0;
	private int crtIndex = 0;
	private BigInteger baseTimestamp = BigInteger.ZERO;
	private boolean multicore; //Refers to the new format
	private String currentContext;
	private String currentSource;
	private String traceType;
	
	protected Hashtable<String, TimelineSource> m_sources = new Hashtable<String, TimelineSource>();
	protected Hashtable<String, TraceTimelinePlot> m_plots = new Hashtable<String, TraceTimelinePlot>();
	protected Set<String> m_sourceFilter = new HashSet<String>();
	
	private Job m_job = null;
	private int m_totalNoOfEvents;
	
	public void addFilter(String source) {
		m_sourceFilter.add(source.trim());
	}
	
	public boolean getSourceIsDisplayed(String source) {
		return m_sourceFilter.contains(source.trim());
	}
	
	public String getFilteredSources() {
		if (m_sourceFilter.isEmpty()) {
			return null;
		}
		
		return m_sourceFilter.toString();
	}
	
	public void addPlot(String source, TraceTimelinePlot traceTimelinePlot) {
		m_plots.put(source, traceTimelinePlot);
	}
	
	public TraceTimelinePlot getPlot(String source) {
		return m_plots.get(source);
	}
	
	public TimelineSource getSource(String source) {
		return m_sources.get(source);
	}
	
    public void setTimeUnit(String timeUnit) {
        for (Entry<String, TraceTimelinePlot> entry : m_plots.entrySet()) {
            entry.getValue().setTimeUnit(timeUnit);
        }
    }

    public void setFrequency(double freq) {
        for (Entry<String, TraceTimelinePlot> entry : m_plots.entrySet()) {
            entry.getValue().setFrequency(freq);
        }
    }

	public void initTimelineData(final ITraceEditorInput input) {

		// refresh the number of events
		m_totalNoOfEvents = 0;

		File timelineFile = new File(getTimelineFile());

		if (timelineFile.exists() && (timelineFile.length() != 0)) {
			m_totalNoOfEvents = getTotalNumberOfEvents(timelineFile);
		}

		// Return if there are no events
		if (m_totalNoOfEvents == 0) {
			return;
		}

		if (m_job != null) {
			try {
				m_job.cancel();
				m_job.join();
			} catch (InterruptedException e) {
				LOGGER.error("", e); //$NON-NLS-1$
			}
		}
		
		setInput(input);

		m_job = new Job(UIMessages.PopulatingTimelineEditor) {

			@Override
			public IStatus run(IProgressMonitor monitor) {
				monitor.beginTask(UIMessages.PopulatingTimelineEditor,
						m_totalNoOfEvents);

				try {

					UIJob updateJob = new UIJob("") { //$NON-NLS-1$
						@Override
						public IStatus runInUIThread(IProgressMonitor monitor) {
							return Status.OK_STATUS;
						}
					};

					updateJob.schedule();
					updateJob.join();

					BaseTraceEventData eventData = null;

					// Pre processing before reading events
					for (String key : m_sources.keySet())
						m_sources.get(key).preProcess();

					readTimelineFile();
					while (hasMoreEvents()) {
						if (isCancelled()) {
							closeFile();
							return Status.CANCEL_STATUS;
						}
						eventData = getEventData();

						// Create the source if it does not exist
						TimelineSource ts = findSource(eventData);

						if (ts != null)
							ts.processEventData(eventData);

						monitor.worked(1);
					}
					closeFile();

					// post processing in all sources
					for (String key : m_sources.keySet()) {
						m_sources.get(key).postProcess();

						TraceTimelinePlot tp = m_plots.get(key);

						if (tp != null)
							tp.setGroups(m_sources.get(key).getGroups());

					}

					UIJob uijob = new UIJob("") { //$NON-NLS-1$
						@Override
						public IStatus runInUIThread(IProgressMonitor monitor) {
							for (String key : m_sources.keySet()) {
								TraceTimelinePlot tp = m_plots.get(key);

								if (tp != null) {
									tp.addPoints(m_sources.get(key).getPoints());
								}
							}

							return Status.OK_STATUS;
						}
					};
					uijob.schedule();
					uijob.join();

				} catch (InterruptedException e) {
					LOGGER.error("", e); //$NON-NLS-1$
					e.printStackTrace();
				} finally {
					monitor.done();
				}

				return Status.OK_STATUS;
			}

			private boolean cancelRequests = false;

			public void setCancelRequests(boolean _cancelRequests) {
				cancelRequests = _cancelRequests;
			}

			public boolean isCancelled() {
				return cancelRequests;
			}

			@Override
			protected void canceling() {
				try {
					setCancelRequests(true);
					m_job.join();
					IWorkbenchPage activePage = PlatformUI.getWorkbench()
							.getActiveWorkbenchWindow().getActivePage();
					IEditorPart editor = activePage.getActiveEditor();
					activePage.closeEditor(editor, false);
				} catch (InterruptedException e) {
					LOGGER.error("", e); //$NON-NLS-1$
				}
			}

		};
		m_job.schedule();
	}
	
	
	/**
	 * Clears all the data related to a timeline graphic. It should be called
	 * each time before drawing a new graphic.
	 */
	public synchronized void clearTimelineData(boolean clearGroups) {

        for (Entry<String, TimelineSource> entry : m_sources.entrySet()) {
            entry.getValue().clearData(clearGroups);
            if (m_plots.get(entry.getKey()) != null) {
                m_plots.get(entry.getKey()).clear(clearGroups);
            }
        }
	}
	
	public TimelineInput() {
		multicore = true;
		currentContext = "0"; //$NON-NLS-1$
		currentSource = ""; //$NON-NLS-1$
	}
	
	public TimelineInput(String inputFilePath) {
		this();
		mTimelineFile = inputFilePath;
	}
	
	@Override
	public int getFunctionCount() {
		return mFunctions.getSize();
	}
	
	@Override
	public int getFunctionCount(String source, String context) {
		return mFunctions.getFunctionCount(source, context);
	}
	
	@Override
	public int findInstruction(BigInteger address, TimelineFunction function) {
		
		for (int i = 0; i < function.getNumInstructions(); i++) {
			TimelineInstruction instruction = function.getInstruction(i);

			if (instruction.mAddress.compareTo(address) == 0) {
				return i;
			}
		}
		return -1;
	}
		
	public void setInput(ITraceEditorInput input) {
		if (input instanceof StorageEditorInput) {
			StorageEditorInput storageInput = (StorageEditorInput)input;
			
			setTimelineFile(storageInput.getStorageDirectory() + IPath.SEPARATOR + 
					storageInput.getBaseFileName() + "." + SaConstants.storage_timeline_file_extension); //$NON-NLS-1$
		} else if (input instanceof TimelineInputFile) {
			//Set file, we read functions while parsing events
			if (!mTimelineFile.endsWith(SaConstants.storage_timeline_file_extension)) {
				setTimelineFile(input.getStorageDirectory() + IPath.SEPARATOR + input.getName());
			}
		}
	}
	
	public void setTimelineFile(String file) {
		mTimelineFile = file;
	}
	
	public String getTimelineFile() {
		return mTimelineFile;
	}
	
	public boolean hasTraceType() {
		if (traceType == null || traceType.isEmpty()) {
			return false;
		}
		
		return true;
	}
	
	public String getTraceType() {
		
		/* Reads the trace type (Linux User Space, Linux Kernel Space or Linux System) from the input file */
		if (traceType == null) {
			traceType = CommonConstants.EMPTY_STRING;
			try {
				Scanner scanner = new Scanner(new File(mTimelineFile));
				String traceLine;

				/* Skip the version */
				scanner.nextLine();

				traceLine = scanner.nextLine();
				if (traceLine.startsWith(TRACE_TYPE_TAG)) {
					traceType = traceLine.replace(TRACE_TYPE_TAG, "").trim(); //$NON-NLS-1$
				}
				
				scanner.close();
			} catch (FileNotFoundException e) {
				LOGGER.debug(e.getLocalizedMessage());
			}
		}
		
		return traceType;
	}
	
	public boolean hasMoreEvents() {
		if (mScanner == null || mEvents == null || mEvents.isEmpty()) {
			return false;
		}
		// Has elements to read or process
		// crtIndex <= maxIndex -- Limits the number of trace events to maxIndex
		return (mScanner.hasNext() || !mEvents.isEmpty()) && (crtIndex <= maxIndex);
	}
	
	public BaseTraceEventData getEventData() {
		if (mEvents.isEmpty()) {
			return null;
		}
		
		BaseTraceEventData eventData = mEvents.remove();
		if (mScanner.hasNext() && mEvents.size() < QUEUE_SIZE - 1) {
			readEvent();
			crtIndex++;
		}
		
		if (eventData.getTimestamp() != null) {
			BigInteger timestamp = eventData.getTimestamp().subtract(new BigInteger("" + baseTimestamp));
			eventData.setTimestamp(timestamp);
		}
		return eventData;
	}
	
	public BaseTraceEventData[] getEventDataArray() {
		return mEvents.toArray(new BaseTraceEventData[mEvents.size()]);
	}
	
	public void closeFile() {
		if (mScanner != null) {
			mScanner.close();
			mScanner = null;
		}
	}
	
    public void readTimelineFile() {
        try {
            if (!getTraceType().equals(LINUX_KERNEL_SPACE_TRACE)) {
                // For old format we need the list of function populated from
                // the critical code file
                mFunctions = new TimelineFunctionInfo();
            } else {
                mFunctions = new KernelTimelineFunctionInfo();
            }

            mEvents = new ArrayBlockingQueue<BaseTraceEventData>(QUEUE_SIZE);
            mScanner = new Scanner(new File(mTimelineFile));
            while (mScanner.hasNext() && mEvents.size() < QUEUE_SIZE - 1) {
                readEvent();
            }
            crtIndex = 0;
            baseTimestamp = BigInteger.ZERO;
            if (minIndex > 0) {
                BigInteger tempTimestamp = BigInteger.ZERO;
                for (int i = 0; i < minIndex; i++) {
                    BaseTraceEventData eventData = getEventData();
                    if (minIndex - i < 5 && eventData.getTimestamp() != null) {
                        tempTimestamp = eventData.getTimestamp();
                    }
                }
                baseTimestamp = tempTimestamp;
            }
        } catch (IOException e) {
            LOGGER.error("[readTimelineFile]", e); //$NON-NLS-1$
        }
    }
	
	private void readTraceEvent() {
		
		int type = mScanner.nextInt();
		
		// If TOTAL_COUNT_EVENT_TYPE skip to next event.
		while (type == TOTAL_COUNT_EVENT_TYPE) {
			if (mScanner.hasNext())
				type = mScanner.nextInt();
			else return;
			if (mScanner.hasNext())
				type = mScanner.nextInt();
			else return;
		}		
		
		
		if (currentSource == null) {
			return;
		}
		
		BaseTraceEventData event = new BaseTraceEventData();
		
		event.setContext(currentContext);
		event.setSource(currentSource);
		
		if (type == PROGRAM_EVENT_TYPE) {
			event.setLabel(TimelineConstants.BRANCH_EVT_LABEL);
			BigInteger address = mScanner.nextBigInteger();
			event.setSrcInstAddress(address.toString(16));
			BigInteger bigTs = mScanner.nextBigInteger();
			event.setTimestamp(bigTs);
			int size = mScanner.nextInt();
			event.setNumAssembly(size);
			
			TimelineFunction function = null;
			if (mVersion == 2) {
			    mScanner.nextBigInteger();	
			}
			function = getFunction(address, currentSource, currentContext);
			
			if (function != null) {
				event.setSrcFuncname(function.getName());
				if (function.getStartAddress() != null) {
				    event.setFuncAddress(function.getStartAddress().toString(16));
				}
			}
			
		} else if (type == INFO_EVENT_TYPE) {	
			event.setLabel(TimelineConstants.INFO_EVT_LABEL);
		} else if (type == ERROR_EVENT_TYPE) {
			event.setLabel(TimelineConstants.ERROR_EVT_LABEL);
			event.setTimestamp(mScanner.nextBigInteger());
			event.setDesc(mScanner.nextLine());
		} else if (type == CUSTOM_EVENT_TYPE) {
			if (mCustomEvents != null) {
				int index = mScanner.nextInt();
				if (index < mCustomEvents.size()) {
					event.setLabel(TimelineConstants.CUSTOM_EVT_LABEL);
					event.setFuncname(mCustomEvents.get(index).getName());
					event.setTimestamp(mScanner.nextBigInteger());
				}
			}
		} else if (type == CONTEXT_SWITCH) {
			BigInteger timestamp = mScanner.nextBigInteger();	
			//Create a context out event if we had some events previously
			if (crtIndex > 0) {
				BaseTraceEventData outEvent = new BaseTraceEventData();
				outEvent.setContext(currentContext);
				outEvent.setSource(currentSource);
				outEvent.setLabel(TimelineConstants.CONTEXT_OUT);
				outEvent.setTimestamp(timestamp);
				mEvents.add(outEvent);
			}
			
			//Create context in event
			event.setLabel(TimelineConstants.CONTEXT_IN);
			event.setTimestamp(timestamp);
			currentContext = mScanner.next();
			event.setContext(currentContext);
		} else if (type == CONTEXT_IN) {
			event.setLabel(TimelineConstants.CONTEXT_IN);
			event.setTimestamp(mScanner.nextBigInteger());	
		} else if (type == CONTEXT_OUT) {
			event.setLabel(TimelineConstants.CONTEXT_OUT);
			event.setTimestamp(mScanner.nextBigInteger());	
		} else {
			LOGGER.error("Unknown type of events : " + type);
		}
		mEvents.add(event);
	}
	
	private String readTag() {
		String tag = CommonConstants.EMPTY_STRING;
		//We assume labels are 1 word
		if (mScanner.hasNext()) 
			tag = mScanner.next();
				
		return tag;
	}
	
	private enum TimelineFunctionLine {
		Name,
		StartAddress,
		Size,
		Inline
	}

	/**
	 * Builds a TimelineFunction based on a line from .timeline file
	 * @param line	A "Function" line
	 * @return The TimelineFunction built on line
	 */
	private TimelineFunction buildTimelineFunction(String line) {
		TimelineFunction function = new TimelineFunction();
		// CSV split regex
		String [] items = line.split(",(?=([^\"]*\"[^\"]*\")*[^\"]*$)"); //$NON-NLS-1
		
		if (items.length > TimelineFunctionLine.Name.ordinal()) {
			function.setName(items[TimelineFunctionLine.Name.ordinal()]);
		}
		if (items.length > TimelineFunctionLine.StartAddress.ordinal()) {
			function.setStartAddress(new BigInteger(items[TimelineFunctionLine.StartAddress.ordinal()], 10));
		}
		if (items.length > TimelineFunctionLine.Size.ordinal()) {
			function.setSize(new BigInteger(items[TimelineFunctionLine.Size.ordinal()], 10));
		}
		if (items.length > TimelineFunctionLine.Inline.ordinal()) {
			function.setInline(Boolean.parseBoolean(items[TimelineFunctionLine.Inline.ordinal()]));
		}

		return function;
	}
	
    private void readFunctions() {
        String line = mScanner.nextLine();
        boolean skipFunctionAdd = false;

        if (getTraceType().equals(LINUX_KERNEL_SPACE_TRACE) && (mFunctions.getSize() != 0)) {
            skipFunctionAdd = true;
        }

        do {
            // Line describing a function
            line = mScanner.nextLine();

            if (!skipFunctionAdd) {
                TimelineFunction newFunction = buildTimelineFunction(line);
                newFunction.setContext(currentContext);
                newFunction.setSource(currentSource);
                mFunctions.addFunction(newFunction);
            }
        } while (!line.isEmpty());
    }
	
	private void readEvent() {
		if (multicore) {
			//Read multicore event
			//We are positioned with the scanner at the start of a line at this moment			
			if (mScanner.hasNext()) {
				//We check if it's a tag or actual event
				if (mScanner.hasNextInt()) {
					//We have an event, we in
					if (currentSource == null) {
						mScanner.nextLine();
					} else {
						readTraceEvent();
					}
				} else {
					//We got a tag
					//We parse the tag section until we find an event
					
					while(!mScanner.hasNextInt()) {
						String tag = readTag();
						
						if (tag.equals(UIConstants.TIMELINE_VERSION)) {
							mVersion = mScanner.nextInt();
						} else if (tag.equals(UIConstants.TIMELINE_SOURCE)) {
							currentSource = mScanner.nextLine();
							if (!m_sourceFilter.isEmpty() && !m_sourceFilter.contains(currentSource.trim())) {
								currentSource = null;
							}
						} else if (tag.equals(UIConstants.TIMELINE_CONTEXT)) {
							currentContext = mScanner.next();
						} else if (tag.equals(UIConstants.TIMELINE_FUNCTIONS)) {
							readFunctions();
						} else if (tag.equals(UIConstants.TIMELINE_CONT_SOURCE) || tag.equals(UIConstants.TIMELINE_CONT_CONTEXT) ) {
							mScanner.nextLine(); //We got this already
						} else if (tag.equals(UIConstants.TIMELINE_CUSTOM_EVENTS)) {
							mScanner.nextLine();
							readCustomEvents();
						}
						
					}
					
					//then we read the event
					readTraceEvent();
				}
			}
		} else {
			readTraceEvent();
		}
	}

	private static String tail( File file ) {
	    try {
	        RandomAccessFile fileHandler = new RandomAccessFile( file, "r" ); //$NON-NLS-1$
	        long fileLength = file.length() - 1;
	        StringBuilder sb = new StringBuilder();

	        for( long filePointer = fileLength; filePointer != -1; filePointer-- ) {
	            fileHandler.seek( filePointer );
	            int readByte = fileHandler.readByte();

	            if( readByte == 0xA ) {
	                if( filePointer == fileLength ) {
	                    continue;
	                } else {
	                    break;
	                }
	            } else if( readByte == 0xD ) {
	                if( filePointer == fileLength - 1 ) {
	                    continue;
	                } else {
	                    break;
	                }
	            }

	            sb.append( ( char ) readByte );
	        }
	        
	        fileHandler.close();

	        String lastLine = sb.reverse().toString();
	        return lastLine;
	    } catch( java.io.FileNotFoundException e ) {
	        e.printStackTrace();
	        return null;
	    } catch( java.io.IOException e ) {
	        e.printStackTrace();
	        return null;
	    }
	}


	/**
	 *  Reads from timeline file the total number of events.
	 * @param timelineFile
	 * @return
	 */
	public int getTotalNumberOfEvents(File timelineFile) {
		String test = tail(timelineFile);
		String[] splitedTest = test.split(" "); //$NON-NLS-1$
		int totalNumberOfEvents = Integer.parseInt(splitedTest[1]);
		
		minIndex = 0;
		maxIndex = totalNumberOfEvents;
		
		return totalNumberOfEvents;
	}
	
	public boolean timeLineHasEvents(File timelineFile) {
		String test = tail(timelineFile);
		String[] splitedTest = test.split(" "); //$NON-NLS-1$
		try {
			if (new BigInteger(splitedTest[1]).compareTo(BigInteger.ZERO) == 0) {
				return false;
			}
		} catch (NumberFormatException e) {
			return false;
		}
		return true;
	}
	
	public boolean getMulticore() {
		return multicore;
	}

	public void setMulticore(boolean val) {
		this.multicore = val;
	}
	
	private void readCustomEvents() {
		
		mCustomEvents = new ArrayList<TimelineCustomEvent>();
		
		String line = mScanner.nextLine();
		try {	
			do {
				//Line describing a function
				String[] tokens = line.split(","); //$NON-NLS-1$
				if (tokens.length >= 4) {
					String name = tokens[0];
					int colorRed = Integer.parseInt(tokens[1]);
					int colorGreen = Integer.parseInt(tokens[2]);
					int colorBlue = Integer.parseInt(tokens[3]);
					
					TimelineCustomEvent evt = new TimelineCustomEvent(name, new RGB(colorRed, colorGreen, colorBlue));
					mCustomEvents.add(evt);
					line = mScanner.nextLine();
				}

			} while (!line.isEmpty());

		}
		catch (NumberFormatException e)	{
			LOGGER.error("[readTimelineFile]", e); //$NON-NLS-1$
		}
	}
	
	/*
	 * (non-Javadoc)
	 * @see com.freescale.sa.ui.timeline.ITimelineFunctionModel#getCustomEvents()
	 */
	public ArrayList<TimelineCustomEvent> getCustomEvents() {
		return mCustomEvents;
	}
	
	private TimelineSource findSource(BaseTraceEventData eventData) {
		if (m_sources == null || eventData == null || eventData.getSource() == null) {
			return null;
		}
		
		TimelineSource ts = m_sources.get(eventData.getSource());
		if (ts == null) {
			// Create the source
			ts = new TimelineSource();
			ts.setModel(this);
			m_sources.put(eventData.getSource(), ts);
			return ts;
		}

		return ts;
	}
	
	static class IndexChooserDialog extends StatusDialog {

		private Shell parentShell = null;
		private Label beginIndexLabel = null;
		private Label endIndexLabel = null;
		private Label noteLabel = null;
		private Text beginIndexValue = null;
		private Text endIndexValue = null;
		protected Label indexValueErrorLabel = null;
		protected Label indexValueErrorImage = null;
		private int minIndex = 0;
		private int maxIndex = 0;
		
		public IndexChooserDialog(Shell parent, int _maxIndex) {
			super(parent);
			parentShell = parent;
			maxIndex = _maxIndex;
			setTitle(UIMessages.TimelineTraceTooLargeTitle);
		}

		protected Control createDialogArea(Composite parent) {
			Composite composite = (Composite)super.createDialogArea(parent);
			SashForm sashForm = new SashForm(composite, SWT.VERTICAL);
			addIndexSelectionSection(sashForm);
			
			return composite;
		}

		protected void addIndexSelectionSection(SashForm sashComposite) {
			ScrolledComposite scrolledComposite = new ScrolledComposite(sashComposite,
					SWT.H_SCROLL | SWT.V_SCROLL);
			scrolledComposite.setLayout(new GridLayout());
			scrolledComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false));
			
			Composite contentsComposite = new Composite(scrolledComposite, SWT.FILL);
			contentsComposite.setLayout(new GridLayout());
			contentsComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false));
					
			Group configGoup = new Group(contentsComposite,
					SWT.SHADOW_ETCHED_IN | SWT.NO_RADIO_GROUP | SWT.FILL);
			configGoup.setLayout(new GridLayout(4, false)); {
				beginIndexLabel = new Label(configGoup, SWT.NONE);
				beginIndexLabel.setText(UIMessages.TraceDataExporter_5);
				beginIndexLabel.setEnabled(true);
				GridData gd1 = new GridData(SWT.BEGINNING);
				gd1.horizontalAlignment = SWT.LEFT;
				beginIndexLabel.setLayoutData(gd1);
				
				beginIndexValue = new Text(configGoup, SWT.SINGLE | SWT.BORDER);
				beginIndexValue.setEnabled(true);
				StringBuffer zeroString = new StringBuffer();
				int temp = maxIndex;
				while (temp > 0) {
					zeroString.append("0"); //$NON-NLS-1$
					temp /= 10;
				}
				beginIndexValue.setText(zeroString.toString());
				GridData gd2 = new GridData(GridData.FILL_HORIZONTAL);
				gd2.horizontalSpan = 1;
				beginIndexValue.setLayoutData(gd2);
				
				beginIndexValue.addModifyListener(new ModifyListener() {

					@Override
					public void modifyText(ModifyEvent e) {
						String firstIndexValue = beginIndexValue.getText();
						String lastIndexValue = endIndexValue.getText();
						
						String errorText = "";
						if (firstIndexValue.isEmpty()){
							errorText = UIMessages.Begin_Index_Value_Empty;
						} 

						try {
							if (Integer.parseInt(lastIndexValue) - Integer.parseInt(firstIndexValue)> MAX_TOTAL_EVENTS){
								errorText = UIMessages.Wrong_Difference;
							}
							if (Integer.parseInt(firstIndexValue) > Integer.parseInt(lastIndexValue)){
								errorText = UIMessages.Wrong_Index_Values;
							}
							if (Integer.parseInt(firstIndexValue) < 0){
								errorText = UIMessages.Wrong_Min_Index;
							}
						} catch (NumberFormatException numE) {
							errorText = UIMessages.Index_Value_Invalid;
						}
						
						indexValueErrorLabel.setText(errorText);
						indexValueErrorImage.setVisible(!errorText.isEmpty());
						indexValueErrorImage.setToolTipText(errorText);
						updateOkButton(errorText.isEmpty());
					}
				});
				
			} {
				endIndexLabel = new Label(configGoup, SWT.NONE);
				endIndexLabel.setText(UIMessages.TraceDataExporter_6);
				endIndexLabel.setEnabled(true);
				GridData gd1 = new GridData(SWT.BEGINNING);
				gd1.horizontalAlignment = SWT.LEFT;
				endIndexLabel.setLayoutData(gd1);
				
				endIndexValue = new Text(configGoup, SWT.SINGLE | SWT.BORDER);
				endIndexValue.setEnabled(true);
				endIndexValue.setText(String.valueOf(maxIndex));
				GridData gd2 = new GridData(GridData.FILL_HORIZONTAL);
				gd2.horizontalSpan = 1;
				endIndexValue.setLayoutData(gd2);
				endIndexValue.addModifyListener(new ModifyListener() {

					@Override
					public void modifyText(ModifyEvent e) {
						String firstIndexValue = beginIndexValue.getText();
						String lastIndexValue = endIndexValue.getText();
						
						String errorText = CommonConstants.EMPTY_STRING;
						if (lastIndexValue.isEmpty()){
							errorText = UIMessages.End_Index_Value_Empty;
						} 
						
						try {
							if (Integer.parseInt(lastIndexValue) - Integer.parseInt(firstIndexValue)> MAX_TOTAL_EVENTS){
								errorText = UIMessages.Wrong_Difference;
							}
							if (Integer.parseInt(lastIndexValue) < Integer.parseInt(firstIndexValue)){
								errorText = UIMessages.Wrong_Index_Values;
							}
							if ((Integer.parseInt(lastIndexValue) > maxIndex)||(Integer.parseInt(lastIndexValue) < 0)){
								errorText = UIMessages.Wrong_Max_Index;
							}
						} catch (NumberFormatException numE) {
							errorText = UIMessages.Index_Value_Invalid;
						}
						
						indexValueErrorLabel.setText(errorText);
						indexValueErrorImage.setVisible(!errorText.isEmpty());
						indexValueErrorImage.setToolTipText(errorText);
						updateOkButton(errorText.isEmpty());
					}
				});
			
			} {
			
				Composite errorComp = new Composite(contentsComposite, SWT.NONE);
				errorComp.setLayout(new GridLayout(2, false));
				errorComp.setLayoutData(new GridData(SWT.LEFT_TO_RIGHT, SWT.TOP,
						true, true, 1, 1));
				indexValueErrorImage = new Label(errorComp, SWT.RIGHT);
				indexValueErrorImage.setImage(PlatformUI.getWorkbench().getSharedImages().getImage(ISharedImages.IMG_OBJS_ERROR_TSK));
				indexValueErrorImage.setVisible(false);
				indexValueErrorLabel = new Label(errorComp, SWT.NONE);
				GridData saIndexErrorGridData = new GridData();
				saIndexErrorGridData.widthHint = 350;
				indexValueErrorLabel.setLayoutData(saIndexErrorGridData);
				
				indexValueErrorImage = new Label(contentsComposite, SWT.RIGHT);
				indexValueErrorImage.setImage(PlatformUI.getWorkbench().getSharedImages().getImage(ISharedImages.IMG_OBJS_ERROR_TSK));
				indexValueErrorImage.setVisible(false);
				indexValueErrorLabel = new Label(contentsComposite, SWT.NONE);
				GridData saOutputFolderErrorGridData = new GridData();
				saOutputFolderErrorGridData.widthHint = 350;
				indexValueErrorLabel.setLayoutData(saOutputFolderErrorGridData);
			
				noteLabel = new Label(contentsComposite, SWT.NONE);
				String msg = MessageFormat.format(UIMessages.TimelineTraceTooLargeMessage,
						String.valueOf(MAX_TOTAL_EVENTS), String.valueOf(maxIndex));
				noteLabel.setText(msg);
				noteLabel.setEnabled(true);
				GridData gd1 = new GridData(SWT.BEGINNING);
				gd1.horizontalAlignment = SWT.CENTER;
				noteLabel.setLayoutData(gd1);
			}
			
			scrolledComposite.setContent(contentsComposite);
			scrolledComposite.setSize(50, 50);
			scrolledComposite.setExpandHorizontal(true);
			scrolledComposite.setExpandVertical(true);
		}
		
		protected void okPressed() {
			if (handleOk()) {
				super.okPressed();
			} else {
				MessageDialog dialog = new MessageDialog(parentShell, UIMessages.TraceDataExporter_InvalidIndices_DialogTitle, null, //$NON-NLS-1$
						UIMessages.TraceDataExporter_8, MessageDialog.QUESTION, new String[] { //$NON-NLS-1$
					IDialogConstants.OK_LABEL, IDialogConstants.CANCEL_LABEL }, 0);
				dialog.open();
			}
		}

		protected void cancelPressed() {
			super.cancelPressed();
			minIndex = 0;
			maxIndex = 0;
		}
		
		private boolean handleOk() {
			int tempBeginIndex = Integer.parseInt(beginIndexValue.getText());
			int tempEndIndex = Integer.parseInt(endIndexValue.getText());

			if ((tempBeginIndex > tempEndIndex) 
					|| (tempBeginIndex < 0) 
					|| (tempEndIndex > maxIndex)
					|| (tempEndIndex - tempBeginIndex > MAX_TOTAL_EVENTS)) {
				return false;
			}
			
			minIndex = tempBeginIndex;
			maxIndex = tempEndIndex;
			return true;
		}

		private void updateOkButton(boolean isEnabled) {
			this.getButton(OK).setEnabled(isEnabled);
		}
		
		public int getMinIndex() {
			return minIndex;
		}

		public void setMinIndex(int minIndex) {
			this.minIndex = minIndex;
		}

		public int getMaxIndex() {
			return maxIndex;
		}

		public void setMaxIndex(int maxIndex) {
			this.maxIndex = maxIndex;
		}
	}

	public void resizePlots() {
        for (Entry<String, TraceTimelinePlot> entry : m_plots.entrySet()) {
            entry.getValue().resize();
        }
	}

	public void removeSource(String src) {
		m_sources.remove(src);
		m_plots.remove(src);
	}

    @Override
    public TimelineFunction getFunction(BigInteger startAddress, String source, String context) {
        return mFunctions.findFunction(startAddress, source, context);
    }

    @Override
    public List<TimelineFunction> getFunctions(String source, String context) {
        return mFunctions.getFunctions(source, context);
    }

    @Override
    public TimelineFunction getFunction(int index, String source, String context) {
        return mFunctions.getFunction(index, source, context);
    }
}
