/*******************************************************************************
 * Copyright (c) 2016 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.timeline;

import java.math.BigInteger;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Vector;

/**
 * Stub for functions manager of the Timeline Processor
 * @author B46903
 *
 */
public abstract class TimelineFunctionInfoBase {
    /**
     * Maps Timeline sources and Timeline contexts Key = Context name, Value =
     * context
     *
     * @author B46903
     *
     */
    static class TimelineSourceInfo extends HashMap<String, TimelineContextInfo> {

        private static final long serialVersionUID = 1L;
    }

    static class TimelineContextInfo {

        private List<TimelineFunction> functions;

        private Boolean needsSort;
        protected Integer cachedIndex;

        TimelineContextInfo() {
            cachedIndex = 0;
            functions = new Vector<TimelineFunction>();
            needsSort = true;
        }

        /**
         * Adds a function to context
         *
         * @param function
         *            The function to be added
         */
        void addFunction(TimelineFunction function) {
            functions.add(function);
            needsSort = true;
        }

        /**
         * Sort list of functions depending on their start address.
         */
        private void sortFunctions() {
            Collections.sort(functions, new Comparator<TimelineFunction>() {
                @Override
                public int compare(TimelineFunction f1, TimelineFunction f2) {
                    return f1.getStartAddress().compareTo(f2.getStartAddress());
                }
            });
            needsSort = false;
        }

        /**
         * Finds a function by its load address
         *
         * @param loadAddress
         *            The load address of the function
         * @return null if the load address isn't found, a valid function
         *         otherwise
         */
        public TimelineFunction getFunctionByLoadAddress(BigInteger loadAddress) {
            TimelineFunction result = null;
            Integer left = 0;
            Integer right = functions.size() - 1;
            Integer mid = 0;

            while (left <= right) {
                mid = left + (right - left) / 2;
                if (loadAddress.compareTo(functions.get(mid).getStartAddress()) < 0) {
                    right = mid - 1;
                } else if (loadAddress.compareTo(functions.get(mid).getStartAddress()) > 0) {
                    left = mid + 1;
                } else {
                    return functions.get(mid);
                }
            }

            return result;
        }

        /**
         * Returns the function contained by the context
         *
         * @return A list of functions
         */
        public List<TimelineFunction> getFunctions() {
            if (needsSort) {
                sortFunctions();
            }

            return functions;
        }

        public TimelineFunction getFunction(int index) {
            if (needsSort) {
                sortFunctions();
            }

            return functions.get(index);
        }

        /**
         * Returns number of functions contained by the context
         *
         * @return Number of functions
         */
        public Integer getSize() {
            return functions.size();
        }
    }

    /**
     * Adds a function to Timeline function management
     *
     * @param function
     *            The function
     */
    abstract public void addFunction(TimelineFunction function);

    /**
     * Finds the function associated with an address given as argument
     *
     * @param address
     *            The instruction address
     * @param source
     *            The trace source (Core 0, Core 1 etc)
     * @param context
     *            The software context
     * @return A valid function otherwise if an association is found, null
     *         otherwise
     */
    abstract public TimelineFunction findFunction(BigInteger address, String source,
                                                  String context);

    /**
     * Returns number of functions contained by all sources and contexts
     *
     * @return The number of functions
     */
    abstract public Integer getSize();

    /**
     * Returns the number of functions contained by a context
     *
     * @param source
     *            The trace source (Core 0, Core 1 etc)
     * @param context
     *            The software context
     * @return The number of functions contained by a software context
     */
    abstract public Integer getFunctionCount(String source, String context);

    /**
     * Returns the functions list ordered by their start address. It is possible
     * to have duplicate function names with different address ranges. It is
     * user responsibility to filter them. The API guarantees that there are no
     * address overlaps.
     *
     * @param source
     *            The trace source (Core 0, Core 1 etc).
     * @param context
     *            The software context.
     * @return The list of functions.
     */
    abstract public List<TimelineFunction> getFunctions(String source, String context);

    /**
     * Returns the function at index in functions list.
     *
     * @param index
     *            The index of the function.
     * @param source
     *            The trace source (Core 0, Core 1 etc).
     * @param context
     *            The software context.
     * @return The function at index.
     */
    abstract public TimelineFunction getFunction(int index, String source, String context);

    protected TimelineFunction getFunctionFromTimelineContext(TimelineContextInfo timelineContext,
                                                              BigInteger address) {
        if (timelineContext == null) {
            return null;
        }

        List<TimelineFunction> functions = timelineContext.getFunctions();
        Integer startIndex = 0;
        Integer endIndex = functions.size() - 1;
        Integer cachedIndex = timelineContext.cachedIndex;

        if (functions.isEmpty()) {
            return null;
        }

        // Out of intervals
        if (address.compareTo(functions.get(startIndex).getStartAddress()) < 0
            || address.compareTo(functions.get(endIndex).getEndAddress()) > 0) {
            return null;
        }

        // "address" represents the load address of a function (version 2)
        TimelineFunction result;
        result = timelineContext.getFunctionByLoadAddress(address);
        if (result != null) {
            return result;
        }

        if (functions.get(cachedIndex).getEndAddress().compareTo(address) < 0) {
            startIndex = cachedIndex;
        }

        Vector<TimelineFunction> funcs = new Vector<TimelineFunction>();
        for (int i = startIndex; i < functions.size(); i++) {
            TimelineFunction function = functions.get(i);
            if (function.getStartAddress().compareTo(address) > 0) {
                break;
            }
            if (address.compareTo(function.getStartAddress()) >= 0
                && address.compareTo(function.getEndAddress()) < 0) {
                timelineContext.cachedIndex = i;
                funcs.add(function);
            }
        }

        if (funcs.isEmpty()) {
            return null;
        } else {
            result = funcs.get(0);
            BigInteger minSize = result.getCodeSize();

            for (TimelineFunction function : funcs) {
                BigInteger size = function.getCodeSize();
                if (size.compareTo(minSize) < 0) {
                    minSize = size;
                    result = function;
                }
            }

            return result;
        }
    }
}
