/*******************************************************************************
 * Copyright (c) 2015 - 2016 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.timeline;

import java.math.BigInteger;
import java.util.HashMap;
import java.util.List;
import java.util.Map.Entry;

/**
 * This class organizes all functions needed by Timeline processor by Source and
 * Context ID. The scope of it is to significantly reduce the search time.
 * 
 * @author B46903
 * 
 */
public class TimelineFunctionInfo extends TimelineFunctionInfoBase {

	private HashMap<String, TimelineSourceInfo> timelineSources;

	public TimelineFunctionInfo() {
		timelineSources = new HashMap<String, TimelineSourceInfo>();
	}

	/**
	 * Adds a function to Timeline function management
	 * @param function    The function
	 */
	public void addFunction(TimelineFunction function) {
		String source = function.getSource();
		String context = function.getContext();

		HashMap<String, TimelineContextInfo> sourceMap = null;
		TimelineContextInfo timelineContext = null;
		if (!timelineSources.containsKey(source)) {
			timelineSources.put(source, new TimelineSourceInfo());
		}

		sourceMap = timelineSources.get(source);

		if (!sourceMap.containsKey(context)) {
			sourceMap.put(context, new TimelineContextInfo());
		}

		timelineContext = sourceMap.get(context);

	    timelineContext.addFunction(function);
	}

	/**
	 * Finds the function associated with an address given as argument
	 * @param address    The instruction address
	 * @param source     The trace source (Core 0, Core 1 etc)
	 * @param context    The software context
	 * @return           A valid function otherwise if an association is found, null otherwise
	 */
	public TimelineFunction findFunction(BigInteger address, String source,
			String context) {
		HashMap<String, TimelineContextInfo> sourceMap = null;
		TimelineContextInfo timelineContext = null;

		if (!timelineSources.containsKey(source)) {
			return null;
		}

		sourceMap = timelineSources.get(source);
		if (!sourceMap.containsKey(context)) {
			return null;
		}

		timelineContext = sourceMap.get(context);
		if (timelineContext == null) {
		    return null;
		}
		
		return getFunctionFromTimelineContext(timelineContext, address);
	}

	/**
	 * Returns number of functions contained by all sources and contexts
	 * @return         The number of functions
	 */
	public Integer getSize() {
		Integer size = 0;
		for (Entry<String, TimelineSourceInfo> sEntry : timelineSources
				.entrySet()) {
			for (Entry<String, TimelineContextInfo> cEntry : sEntry.getValue()
					.entrySet()) {
				size += cEntry.getValue().getSize();
			}
		}

		return size;
	}

	/**
	 * Returns the number of functions contained by a context
	 * @param source     The trace source (Core 0, Core 1 etc)
	 * @param context    The software context
	 * @return           The number of functions contained by a software context
	 */
	public Integer getFunctionCount(String source, String context) {
		HashMap<String, TimelineContextInfo> sourceMap = null;
		TimelineContextInfo timelineContext = null;

		if (!timelineSources.containsKey(source)) {
			return 0;
		}

		sourceMap = timelineSources.get(source);
		if (!sourceMap.containsKey(context)) {
			return 0;
		}

		timelineContext = sourceMap.get(context);
		return timelineContext.getSize();
	}
	
	/**
	 * Returns the functions list ordered by their start address. It is possible to
     * have duplicate function names with different address ranges. It is user
     * responsibility to filter them. The API guarantees that there are no address overlaps.
	 * 
	 * @param source     The trace source (Core 0, Core 1 etc).
     * @param context    The software context.
	 * @return           The list of functions.
	 */
	public List<TimelineFunction> getFunctions(String source, String context) {
	    HashMap<String, TimelineContextInfo> sourceMap = timelineSources.get(source);
	    if (sourceMap == null) {
	        return null;
	    }
	    
	    TimelineContextInfo timelineContext = sourceMap.get(context); 
	    if (timelineContext == null) {
	        return null;
	    }
	    
	    return timelineContext.getFunctions();
	}
	
	/**
	 * Returns the function at index in functions list.
	 * 
	 * @param index      The index of the function.
     * @param source     The trace source (Core 0, Core 1 etc).
     * @param context    The software context.
	 * @return           The function at index.
	 */
	public TimelineFunction getFunction(int index, String source, String context) {
	    HashMap<String, TimelineContextInfo> sourceMap = timelineSources.get(source);
        if (sourceMap == null) {
            return null;
        }
        
        TimelineContextInfo timelineContext = sourceMap.get(context); 
        if (timelineContext == null) {
            return null;
        }
        
	    return timelineContext.getFunction(index);
	}
}
