/*******************************************************************************
 * Copyright (c) 2014 - 2016 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.timeline;


import java.math.BigInteger;
import java.util.Vector;

import org.apache.log4j.Logger;


public class TimelineFunction {
	private Logger LOGGER = Logger.getLogger(TimelineFunction.class);
	
	public final static BigInteger INVALID_ADDRESS = new BigInteger("FFFFFFFFFFFFFFFFF", 16); //$NON-NLS-1$
	
	private String m_Name;
	private BigInteger m_StartAddress = INVALID_ADDRESS;
	private BigInteger m_Size;	
	private BigInteger m_EndAddress = INVALID_ADDRESS;
	private String m_Context;
	private String m_Source;
	private Vector<TimelineInstruction> m_instructions;
	private Boolean m_isInline = false;

	public TimelineFunction() {
		
	}
	
	public TimelineFunction(String name, String startAddress, String size) {
		m_instructions = new Vector<TimelineInstruction>();
		m_Name = name;
		if (startAddress.charAt(1) == 'x' || startAddress.charAt(1) == 'X') {
			startAddress = startAddress.substring(2);
		}
		m_StartAddress = new BigInteger(startAddress, 16);
		try {
			m_Size = new BigInteger(size);
		} catch (NumberFormatException e) {
			LOGGER.error("[TimelineFunction]Error when trying to get the size of a function!", e); //$NON-NLS-1$
			m_Size = BigInteger.ZERO;
		}
		m_EndAddress = m_StartAddress.add(m_Size);
		m_isInline = false;
	}

	public TimelineFunction(String name, BigInteger startAddress, BigInteger size) {
		m_instructions = new Vector<TimelineInstruction>();
		m_Name = name;

		m_StartAddress = startAddress;
		m_Size = size;	
		m_EndAddress = m_StartAddress.add(m_Size);
		
		m_Source = "";
		m_Context = "";
		m_isInline = false;
	}

	public String getName() {
		return m_Name;
	}

	public BigInteger getStartAddress() {
		return m_StartAddress;
	}
	
	public BigInteger getCodeSize() {
		return m_Size;
	}
	
	public BigInteger getEndAddress() {
		return m_EndAddress;
	}
	
	public String getSource() {
		return m_Source;
	}
	
	public String getContext() {
		return m_Context;
	}
	
	public Vector<TimelineInstruction> getInstructions() {
		return m_instructions;
	}
	
	public int getNumInstructions() {
		if (m_instructions != null) {
			return m_instructions.size();
		}
		
		return 0;
	}

	
	public void setName(String m_Name) {
		this.m_Name = m_Name;
	}

	public void setStartAddress(BigInteger m_StartAddress) {
		this.m_StartAddress = m_StartAddress;
	}

	public void setSize(BigInteger m_Size) {
		this.m_Size = m_Size;
		m_EndAddress = m_StartAddress.add(m_Size);
	}

	public void setContext(String m_Context) {
		this.m_Context = m_Context;
	}

	public void setSource(String m_Source) {
		this.m_Source = m_Source;
	}

	public void setInstructions(Vector<TimelineInstruction> m_instructions) {
		this.m_instructions = m_instructions;
	}

	public void setInline(Boolean m_isInline) {
		this.m_isInline = m_isInline;
	}
	
	public TimelineInstruction getInstruction(int index) {
		if (index < m_instructions.size()) 
			return m_instructions.get(index);
		
		return null;
	}
	
	public BigInteger getAddressForInstruction(int index) {
		TimelineInstruction inst = getInstruction(index);
		
		if (inst != null) {
			return inst.mAddress;
		}
		
		return INVALID_ADDRESS;
	}
	
	public void addInstruction(TimelineInstruction inst) {
		m_instructions.add(inst);
	}
	
	public int contains(BigInteger address) {
		for (int i = 0; i < m_instructions.size(); i++) {
			if (m_instructions.get(i).mAddress == address)
				return i;
		}
		
		return -1;
	}
	
	public boolean isInline(){
		return m_isInline;
	}
	
	public String toString() {
		return "(0x" + getStartAddress().toString(16) + ", 0x" + getEndAddress().toString(16) + ")"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	}
}
