/*******************************************************************************
 * Copyright (c) 2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.timeline;


import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.InvalidRegistryObjectException;
import org.eclipse.core.runtime.Platform;
import com.freescale.sa.ui.timeline.TimelineUIPlugin;

/**
 * <p>
 * Represents a timelineContributor extension. This class parses the extension
 * declaration (in a plugin.xml) and holds the configuration for further usage.
 * Through static fields and methods, it also acts as a registry of all such
 * contributor instances.
 * </p>
 * 
 */
public class TimelineContributor {

	// Extension point names, and the elements and attributes used in those
	// extension points
	private static final String EXTPT_TIMELINE_CONTRIBUTOR = "timelineContributor"; //$NON-NLS-1$
	public static final String ATT_TIMELINE_CONTRIBUTOR_ID = "contributorId"; //$NON-NLS-1$
	private static final String ATT_ARCH_NAME = "archName"; //$NON-NLS-1$

	/**
	 * The Timeline Contributor ID, as found in the XML
	 */
	protected String m_TimelineContributorId = null;

	/**
	 * The name of the architecture, as found in the xml.
	 */
	protected String m_ArchName = null;

	/**
	 * The XML element that describes the timelineContributor
	 */
	protected IConfigurationElement m_ContributorConfigurationElement = null;

	/**
	 * The list of <i>instantiated</i> Timeline Contributors. There may be
	 * numerous in the Eclipse extension registry, but we only instantiate the
	 * ones that are requested at runtime.
	 */
	private static List<TimelineContributor> m_timelineContributors = new ArrayList<TimelineContributor>(5);

	/**
	 * Logger
	 */
	private static Logger LOGGER = Logger.getLogger(TimelineContributor.class);

	/**
	 * Constructor. Protected as an instance is not supposed to be instantiated
	 * directly but through a <code>TabbedRegistryFactory</code>.
	 * 
	 * There is one singleton instance for each registered Timeline Contributor.
	 * Parses the extension points and gets all the attributes.
	 * 
	 * @param configurationElement
	 *            the timelineContributor configuration element
	 */
	protected TimelineContributor(IConfigurationElement configurationElement) {

		m_ContributorConfigurationElement = configurationElement;

		m_TimelineContributorId = configurationElement.getAttribute(ATT_TIMELINE_CONTRIBUTOR_ID);

		m_ArchName = configurationElement.getAttribute(ATT_ARCH_NAME);
	}

	/**
	 * Accessor method. See m_TimelineContributorId
	 */
	public String getId() {
		return m_TimelineContributorId;
	}

	/**
	 * Find the first Timeline Contributor associated with the given
	 * architecture name. Really, there should be one or none, but not multiple.
	 * But since we can't enforce that, we just ignore all but the first.
	 * 
	 * @param archName
	 *            the name of the architecture
	 * @return the TimelineContributor instance or null if there's no registered
	 *         timelineContributor extension for that architecture
	 */
	public static TimelineContributor getTimelineContributorForArch(
			String archName) {
		// We look to see if we've already instantiated a Timeline Contributor
		// description that matches the input. If not, we look through all
		// Timeline Contributor extensions (XML) for a match, and instantiate
		// it if we find it.
		
		// The arch name saved in the config can be as it's found in the launch
		// config (e.g.: MC9S08QE128) or a mapping used to identify the target easier
		// (e.g.: HCS08).
		//String mappedArchId = PlatformType.getPlatformType(archName).toString();

		synchronized (m_timelineContributors) {
			IExtensionPoint extPoint = Platform.getExtensionRegistry().getExtensionPoint(TimelineUIPlugin.PLUGIN_ID,
					EXTPT_TIMELINE_CONTRIBUTOR);

			if (null != extPoint) {
				try {
					final IConfigurationElement[] contribs = extPoint.getConfigurationElements();

					for (IConfigurationElement contributor : contribs) {
						String thisArchName = contributor.getAttribute(ATT_ARCH_NAME);

						if (thisArchName != null
								&& (thisArchName.equalsIgnoreCase(archName))) {
							TimelineContributor traceContr = new TimelineContributor(contributor);

							m_timelineContributors.add(traceContr);
							return traceContr;
						}
					}
				} catch (InvalidRegistryObjectException iroe) {
					iroe.printStackTrace();
					LOGGER.error("[getTimelineContributorForArch]: Error getting contributor for arch " + archName, iroe); //$NON-NLS-1$
				}
			} else {
				LOGGER.error(
						"[getTimelineContributorForArch]: Error getting contributor for arch " + archName //$NON-NLS-1$
						+ ": null extension point"); //$NON-NLS-1$
			}
		}

		return null;
	}

	/**
	 * @return the architecture name, as found in the extension point's xml
	 */
	public String archName() {
		return m_ArchName;
	}
}
