/*******************************************************************************
 * Copyright (c) 2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.timeline;

import java.text.DecimalFormat;
import java.text.ParseException;

import org.apache.log4j.Logger;

import com.freescale.sa.ui.common.IConfigurableDisplay;

/**
 * Place holder for time unit related operations.
 * 
 */
public class TimeUnitUtils {
	private static Logger LOGGER = Logger.getLogger(TimeUnitUtils.class);

	private static final DecimalFormat FORMAT_INT = new DecimalFormat(",##0.###"); //$NON-NLS-1$
	private static final DecimalFormat FORMAT_DOUBLE = new DecimalFormat(",##0.0000"); //$NON-NLS-1$
	
	/**
	 * Converts the time from CPU cycles to a different time unit.
	 * Returns the value as a string. 
	 * 
	 * @param cycles
	 *            the current time value (in cycles).
	 * @param newTimeUnit
	 *            the time unit to convert into (see types defined in
	 *            {@link IConfigurableDisplay}).
	 * @param cpuFreq
	 *            the CPU frequency in MHz (needed to convert from CPU cycles to real
	 *            time).
	 * @return the time value converted to the selected unit
	 */
	public static String convertCycles(String cycles, String newTimeUnit,
			double cpuFreq) {
		Double tsVal = (double) 0;

		if (cycles != null && !cycles.equals("")) { //$NON-NLS-1$
			tsVal = getDouble(cycles);
		}

		double tsConverted = convertCycles(tsVal, newTimeUnit, cpuFreq);
		
		return formatTimeValue(tsConverted, newTimeUnit);
	}
	
	private static Double getDouble(String text) {
		if (text == null || text.length() == 0) {
			return new Double(0);
		}
		
		//First try to unformat the string.
		DecimalFormat comma1000sSeparator = new DecimalFormat(",##0"); //$NON-NLS-1$
		Number no = null;
		try {
			 no = comma1000sSeparator.parse(text);
			 return new Double(no.doubleValue());
		} catch (ParseException parseEx) {
		}
		
		try {
			return Double.parseDouble(text);		
		} catch (Exception e) {	
			return new Double(0);
		}
	}

	/**
	 * Converts the time from CPU cycles to a different time unit.
	 * 
	 * @param cycles
	 *            the current time value (in cycles).
	 * @param newTimeUnit
	 *            the time unit to convert into (see types defined in
	 *            {@link IConfigurableDisplay}).
	 * @param cpuFreq
	 *            the CPU frequency in MHz (needed to convert from CPU cycles to real
	 *            time).
	 * @return the time value converted to the selected unit
	 */
	public static double convertCycles(double cycles, String newTimeUnit,
			double cpuFreq) {
		Double tsConverted = cycles;

		if (cpuFreq == -1 || cpuFreq == 0) {
			LOGGER.debug("[convertTime]: invalid frequency"); //$NON-NLS-1$
			return tsConverted;
		}
		if (newTimeUnit.equals(IConfigurableDisplay.TIME_UNIT_CYCLES)) {
			tsConverted = cycles;
		} else if (newTimeUnit.equals(IConfigurableDisplay.TIME_UNIT_MILLISEC)) {
			tsConverted = (double) (cycles / cpuFreq / 1E3);
		} else if (newTimeUnit.equals(IConfigurableDisplay.TIME_UNIT_MICROSEC)) {
			tsConverted = (double) (cycles / cpuFreq);
		} else if (newTimeUnit.equals(IConfigurableDisplay.TIME_UNIT_NANOSEC)) {
			tsConverted = (cycles / cpuFreq) * 1E3;
		}

		return tsConverted;
	}
		
	/**
	 * Save frequency in the storage config file.
	 * 
	 * @param configPath
	 *            absolute path to the trace config file
	 * @param cpuClockFreq
	 *            the value of the CPU clock frequency
	 */
	public static void saveFrequency(String configPath,
			double cpuClockFreq) {

		//Add here save to file code, must be in sync with other ported viewers
	}
	
	/**
	 * Load frequency from the storage config file.
	 * 
	 * @param configPath
	 *            absolute path to the trace config file
	 * @return the value of the CPU clock frequency
	 */
	public static double loadFrequency(String configPath) {
		double freq = -1;

		//Add here load from file code, must be in sync with other ported viewers
		return freq;
	}
	
	
	
	/**
	 * Save time unit in the storage config file.
	 * 
	 * @param configPath
	 *            absolute path to the trace config file
	 * @param timeUnit
	 *            the value of the time unit
	 */
	public static void saveTimeUnit(String configPath,
			String timeUnit) {
		//Add here save to file code, must be in sync with other ported viewers
	}
	
	/**
	 * Load time unit from the storage config file.
	 * 
	 * @param configPath
	 *            absolute path to the trace config file
	 * @param the
	 *            value of the time unit
	 */
	public static String loadTimeUnit(String configPath) {
		String timeUnit = IConfigurableDisplay.TIME_UNIT_CYCLES;
		//Add here load from file code, must be in sync with other ported viewers
		return timeUnit;
	}
	
	/**
	 * Format time string.
	 * 
	 * @param value
	 *            the value to format
	 * @param timeUnit
	 *            the time unit
	 * @return the formatted time value
	 */
	public static String formatTimeValue(double value, String timeUnit) {
		String formattedTime;
		if (timeUnit.equals(IConfigurableDisplay.TIME_UNIT_CYCLES)) {
			formattedTime = FORMAT_INT.format(value);
		} else if (timeUnit.equals(IConfigurableDisplay.TIME_UNIT_MICROSEC)
				|| timeUnit.equals(IConfigurableDisplay.TIME_UNIT_MILLISEC)
				|| timeUnit.equals(IConfigurableDisplay.TIME_UNIT_NANOSEC)) {
			formattedTime = FORMAT_DOUBLE.format(value);
		} else {
			formattedTime = ""; //$NON-NLS-1$
		}
		
		return formattedTime;
	}
	
	/**
	 * Converts time according with all possible time elements of configuration.
	 * @param ts Timestamp to be converted.
	 * @param timeUnit Time unit.
	 * @param cpuClockFreq CPU clock frequency.
	 * @param timeRadix Time format radix.
	 * @param timeValue Time format value.
	 * @param tsPrev Previous timestamp. It must be provided only if
	 * timeValue is IConfigurableDisplay.TIME_VALUE_DELTA, otherwise 
	 * it should be set to null.
	 * @return Converted timestamp.
	 * If at least one time element of configuration is null timestamp 
	 * parameter is returned.
	 * @Note For timeUnit, timeRadix, timeValue, see types defined in
	 * {@link IConfigurableDisplay}. 
	 */
	public static String convertTime(String ts, String timeUnit, double cpuClockFreq, 
			String timeRadix, String timeValue, String tsPrev) {
	
		if (timeUnit == null || timeRadix == null  || timeValue == null) {
			return ts;
		}
		
		DecimalFormat formatInt = new DecimalFormat(",##0.###"); //$NON-NLS-1$
		DecimalFormat formatDouble = new DecimalFormat(",##0.0000"); //$NON-NLS-1$
		DecimalFormat formatAproxDouble = new DecimalFormat("##0.0000"); //$NON-NLS-1$
		Double tsVal = (double)0;
		
		if (ts != null && !ts.equals("")) { //$NON-NLS-1$
			tsVal = Double.parseDouble(ts);
		}

		String tsConverted = ""; //$NON-NLS-1$
		
		tsVal = Double.parseDouble(formatAproxDouble.format(TimeUnitUtils.convertCycles(tsVal, timeUnit,
						cpuClockFreq)));
		
		if (timeValue.equals(IConfigurableDisplay.TIME_VALUE_DELTA)) {
			Double tsPrevVal = (double)0;
			if (tsPrev != null && !tsPrev.equals("")) { //$NON-NLS-1$
				tsPrevVal = Double.parseDouble(tsPrev);
			}
			tsPrevVal = Double.parseDouble(formatAproxDouble
					.format(TimeUnitUtils.convertCycles(tsPrevVal, timeUnit,
							cpuClockFreq)));
			tsVal -= tsPrevVal;
			tsVal = tsVal > 0 ? tsVal : 0;
		}
		
		if (timeUnit.equals(IConfigurableDisplay.TIME_UNIT_CYCLES)) {
			tsConverted = formatInt.format(tsVal);
			if (timeRadix.equals(IConfigurableDisplay.TIME_RADIX_HEX)) {
				tsConverted = "0x" + Integer.toHexString(tsVal.intValue()); //$NON-NLS-1$
			}
		} else {
			tsConverted = formatDouble.format(tsVal);
		}

		return tsConverted;
	}
}
