/*******************************************************************************
 * Copyright (c) 2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.timeline;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.swt.widgets.Event;
import org.eclipse.ui.actions.ActionGroup;

import com.freescale.sa.ui.common.IConfigurableDisplay;
import com.freescale.sa.ui.common.IRefreshableDisplay;
import com.freescale.sa.ui.common.TraceAction;
import com.freescale.sa.ui.timeline.UIMessages;

/**
 * <p>
 * Action group for the time unit actions. These actions are mutually exclusive
 * in execution (radio button actions) and are supposed to always appear
 * together in a submenu. The actions this group holds are subclasses of
 * <code>TraceAction</code> and are very lightweight in nature.
 * 
 * The submenu also has an option that allows the user to set the CPU frequency.
 * It's needed to make the conversion from cycles to real time (e.g.: seconds).
 * 
 * </p>
 * 
 * @NOTE Each Time Unit Group is associate with a set of trace results. The
 *       config file for decoded trace is used as a key to map the trace results
 *       with the time unit group. To be more explicit - the trace, critical
 *       code, performance (and other results that involve time) share the same
 *       instance of a Time Unit Group. This means they share the same settings
 *       for time unit and CPU frequency.
 * 
 */
public class TimeUnitGroup extends ActionGroup {

	private static HashMap<String, TimeUnitGroup> m_groups = new HashMap<String, TimeUnitGroup>();

	/**
	 * The submenu ID.
	 */
	private final static String SUBMENU_ID = "com.freescale.sa.ui.editor.trace.actions.menu"; //$NON-NLS-1$

	/**
	 * List of supported time unit actions.
	 */
	private List<TraceAction> m_timeUnitActions = new ArrayList<TraceAction>();

	/**
	 * List of supported config action.
	 */
	private List<TraceAction> m_configActions = new ArrayList<TraceAction>();
	
	/**
	 * Flag that specifies whether time and config menu items are activated (true) or disabled (false)
	 */
	private boolean m_timeAvailable = true;

	private String m_timeUnitSelection = IConfigurableDisplay.TIME_UNIT_CYCLES;
	private double m_freqSelection = -1;
	
	/**
	 * Constructor.
	 * 
	 * @param decodedTraceConfigFilePath
	 *            Full path to the decoded trace configuration file.
	 */
	private TimeUnitGroup(String decodedTraceConfigFilePath) {
		m_timeUnitSelection = TimeUnitUtils.loadTimeUnit(decodedTraceConfigFilePath);
		m_freqSelection = TimeUnitUtils.loadFrequency(decodedTraceConfigFilePath);
		
		m_timeUnitActions.add(new CyclesAction(m_timeUnitSelection));
		m_timeUnitActions.add(new MillisecAction(m_timeUnitSelection));
		m_timeUnitActions.add(new MicrosecAction(m_timeUnitSelection));
		m_timeUnitActions.add(new NanosecAction(m_timeUnitSelection));

		m_configActions.add(new FrequencyAction(m_freqSelection, this));

	}

	/**
	 * Get group for a certain config file.
	 * 
	 * @param decodedTraceConfigFilePath
	 *            the full path to the decoded trace config file.
	 * @return the time unit group associated with the file. If a group doesn't
	 *         already exist, a new one will be created.
	 */
	public static TimeUnitGroup getGroupForConfig(
			String decodedTraceConfigFilePath) {
		if (m_groups.containsKey(decodedTraceConfigFilePath)) {
			return m_groups.get(decodedTraceConfigFilePath);
		}
		TimeUnitGroup newGroup = new TimeUnitGroup(decodedTraceConfigFilePath);
		m_groups.put(decodedTraceConfigFilePath, newGroup);
		return newGroup;
	}

	public String getTimeUnitSelection() {
		return m_timeUnitSelection;
	}
	
	public void setTimeUnitSelection(String timeUnit) {
		m_timeUnitSelection = timeUnit;
		
		// Update selection in menu.
		Iterator<TraceAction> timeUnitIterator = m_timeUnitActions.iterator();
		while (timeUnitIterator.hasNext()){
			TraceAction element = timeUnitIterator.next();
			if (element.getText().equals(timeUnit)) {
				element.setChecked(true);
			} else {
				element.setChecked(false);
			}
		}
		
	}
		
	public void setFreqSelection(double selectedFreq) {
		m_freqSelection = selectedFreq;
	}
	
	public double getFreqSelection() {
		return m_freqSelection;
	}
	
	/**
	 * Adds a configurable parent for the time unit group. The parent should be
	 * removed as soon as the group isn't used anymore (e.g.: the parent editor
	 * is closed).
	 * 
	 * @param configurableParent
	 */
	public void addParent(IConfigurableDisplay configurableParent) {
		for (TraceAction traceAction : m_timeUnitActions) {
			traceAction.addParent(configurableParent);
		}

		for (TraceAction configAction : m_configActions) {
			configAction.addParent(configurableParent);
		}
	}
	
	/**
	 * Removes the given configurable parent from the list.
	 * @param deadParent
	 */
	public void removeParent(IConfigurableDisplay deadParent) {
		for (TraceAction traceAction : m_timeUnitActions) {
			traceAction.removeParent(deadParent);
		}

		for (TraceAction configAction : m_configActions) {
			configAction.removeParent(deadParent);
		}	
	}

	/**
	 * Adds a refreshable parent for the time unit group. The parent should be
	 * removed as soon as the group isn't used anymore (e.g.: the parent editor
	 * is closed).
	 * @param refreshableParent
	 */
	public void addParent(
			IRefreshableDisplay refreshableParent) {
		for (TraceAction traceAction : m_timeUnitActions) {
			traceAction.addParent(refreshableParent);
		}

		for (TraceAction configAction : m_configActions) {
			configAction.addParent(refreshableParent);
		}
	}
	
	/**
	 * Removes the given refreshable parent from the list.
	 * @param deadParent
	 */
	public void removeParent(
			IRefreshableDisplay deadParent) {
		for (TraceAction traceAction : m_timeUnitActions) {
			traceAction.removeParent(deadParent);
		}

		for (TraceAction configAction : m_configActions) {
			configAction.removeParent(deadParent);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.actions.ActionGroup#dispose()
	 */
	public void dispose() {
		m_timeUnitActions = null;
		super.dispose();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.ui.actions.ActionGroup#fillContextMenu(org.eclipse.jface.
	 * action.IMenuManager)
	 */
	public void fillContextMenu(IMenuManager menu) {
		super.fillContextMenu(menu);

		IMenuManager subMenu = new MenuManager(UIMessages.ConfigTimeUnit,
				SUBMENU_ID);

		for (TraceAction action : m_timeUnitActions) {
			action.setEnabled(m_timeAvailable);
			subMenu.add(action);
		}

		subMenu.add(new Separator());

		for (TraceAction action : m_configActions) {
			action.setEnabled(m_timeAvailable);
			subMenu.add(action);
		}

		menu.add(subMenu);
		menu.add(new Separator());
	}

	/**
	 * Controls the state (active or disabled) of time and config menu items
	 * @param timeAvailable true if menu items shall be active or false to disable them
	 */
	public void setTimeAvailable(boolean timeAvailable) {
		m_timeAvailable = timeAvailable;
	}

	/**
	 * Action class that will switch the time representation in the
	 * corresponding trace table columns to processor cycles. Processor is the
	 * cycles is the default representation.
	 */
	private class CyclesAction extends TimeAction {

		public CyclesAction(String initialTimeUnit) {
			super(UIMessages.ConfigTimeUnit_Cycles, IAction.AS_RADIO_BUTTON,
					IConfigurableDisplay.TIME_UNIT_CYCLES, initialTimeUnit);
		}
	}

	/**
	 * Action class that will switch the time representation in the
	 * corresponding trace table columns to nanoseconds.
	 * 
	 */
	private class NanosecAction extends TimeAction {

		public NanosecAction(String initialTimeUnit) {
			super(UIMessages.ConfigTimeUnit_NanoSec, IAction.AS_RADIO_BUTTON,
					IConfigurableDisplay.TIME_UNIT_NANOSEC, initialTimeUnit);
		}
	}

	/**
	 * Action class that will switch the time representation in the
	 * corresponding trace table columns to microseconds.
	 */
	private class MicrosecAction extends TimeAction {

		public MicrosecAction(String initialTimeUnit) {
			super(UIMessages.ConfigTimeUnit_MicroSec, IAction.AS_RADIO_BUTTON,
					IConfigurableDisplay.TIME_UNIT_MICROSEC, initialTimeUnit);
		}		
	}

	/**
	 * Action class that will switch the time representation in the
	 * corresponding trace table columns to milliseconds.
	 */
	private class MillisecAction extends TimeAction {

		public MillisecAction(String initialTimeUnit) {
			super(UIMessages.ConfigTimeUnit_MilliSec, IAction.AS_RADIO_BUTTON,
					IConfigurableDisplay.TIME_UNIT_MILLISEC, initialTimeUnit);
		}		
	}

	/**
	 * Action class that should be used as a base for all the time unit related
	 * actions.
	 */
	private class TimeAction extends TraceAction {

		/** The time unit corresponding to this action */
		private final String m_timeUnit;

		protected TimeAction(String name, int style, String timeUnit, String initialTimeUnit) {
			super(name, style);
			m_timeUnit = timeUnit;
			setChecked(m_timeUnit.equals(initialTimeUnit));
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.action.Action#run()
		 */
		public void runWithEvent(Event e) {
			if (this.isChecked()) {
				for (IConfigurableDisplay configurableParent : m_configurableParents) {
					configurableParent.configure(
							IConfigurableDisplay.CONFIG_ELEMENT.TIME_UNIT_ELM,
							m_timeUnit);
				}
				m_timeUnitSelection = m_timeUnit;
				super.run();
			}
		}
	} // class TimeAction

} //class TimeUnitGroup
