/*******************************************************************************
 * Copyright (c) 2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.timeline;


import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.freescale.sa.ui.common.IConfigurableDisplay;
import com.freescale.sa.ui.timeline.UIMessages;
import com.freescale.sa.util.CommonConstants;
import com.freescale.sa.util.DoubleFieldModifyListener;
//import com.freescale.sa.util.CommonConstants;
//import com.freescale.sa.util.DoubleFieldModifyListener;


/**
 * Dialog that allows the user to set the CPU frequency.
 */
public class SetFrequencyDialog extends Dialog {

	/**
	 * The value of the frequency, measured in MHz.
	 */
	private Double m_Frequency = -1.0;

	private Text m_newFreqText;

	private List<IConfigurableDisplay> m_configurableParents = new ArrayList<IConfigurableDisplay>();
	
	private FrequencyAction m_frequencyAction;
	
	private static final double MAX_FREQUENCY = 100000;
	private static final double MIN_FREQUENCY = 0.1;
	
	private static final String FREQ_UNIT_MHZ = "MHz"; //$NON-NLS-1$

	/**
	 * @param parentShell
	 *            The shell to use as the dialog's parent.
	 * @param configurableParents
	 * @param m_freqInitialValue
	 *            The value that is currently used for the frequency.
	 * @param frequencyAction 
	 */
	public SetFrequencyDialog(Shell parentShell,
			List<IConfigurableDisplay> configurableParents, double m_freqInitialValue, FrequencyAction frequencyAction) {
		super(parentShell);
		m_Frequency = m_freqInitialValue;
		m_configurableParents = configurableParents;
		m_frequencyAction = frequencyAction;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets
	 * .Composite)
	 */
	@Override
	protected Control createDialogArea(Composite parent) {
		parent.getShell().setText(UIMessages.FrequencyDialog_Title);

		Composite parentComposite = new Composite(parent, SWT.NONE);

		parentComposite.setLayout(new GridLayout(3, false));

		// Current frequency.
		initCurrentFreqWidgets(parentComposite);
		initNewFreqWidgets(parentComposite);

		return parent;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.jface.dialogs.Dialog#createButtonsForButtonBar(org.eclipse
	 * .swt.widgets.Composite)
	 */
	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		super.createButtonsForButtonBar(parent);

		// Disable the OK button by default, since there is no frequency set.
		getButton(IDialogConstants.OK_ID).setEnabled(false);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.dialogs.Dialog#okPressed()
	 */
	@Override
	protected void okPressed() {
		super.okPressed();
		for (IConfigurableDisplay configurableParent:m_configurableParents) {
			configurableParent.configure(IConfigurableDisplay.CONFIG_ELEMENT.FREQ_ELM,
					m_Frequency.toString());
		}
		m_frequencyAction.setFreqSelection(m_Frequency);
	}

	private void initCurrentFreqWidgets(Composite parentComposite) {

		Label label = new Label(parentComposite, SWT.NULL);
		label.setText(UIMessages.FrequencyDialog_CrtFreq);

		Label freqLabel = new Label(parentComposite, SWT.NULL);
		freqLabel.setText(m_Frequency.toString());

		Label unitLabel = new Label(parentComposite, SWT.LEAD);
		unitLabel.setText(FREQ_UNIT_MHZ);

	}

	private void initNewFreqWidgets(Composite parentComposite) {
		Label label = new Label(parentComposite, SWT.NULL);
		label.setText(UIMessages.FrequencyDialog_NewFreq);

		m_newFreqText = new Text(parentComposite, SWT.SINGLE | SWT.BORDER);
		m_newFreqText.setText(CommonConstants.EMPTY_STRING);		

		Label unitLabel = new Label(parentComposite, SWT.LEAD);
		unitLabel.setText(FREQ_UNIT_MHZ);

		m_newFreqText.addModifyListener(new DoubleFieldModifyListener(
				m_newFreqText, CommonConstants.EMPTY_STRING, MIN_FREQUENCY, MAX_FREQUENCY) {

			@Override
			public void modifyText(ModifyEvent e) {
				boolean valid = validateText();
				if (valid) {
					m_Frequency = Double.parseDouble(m_newFreqText.getText());
					getButton(IDialogConstants.OK_ID).setEnabled(true);
				} else {
					getButton(IDialogConstants.OK_ID).setEnabled(false);
				}

			}
		});		
	}
}
