/*******************************************************************************
 * Copyright (c) 2014 - 2016 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.timeline;


import java.math.BigInteger;
import java.util.ArrayList;
import java.util.List;

import com.freescale.sa.ui.timeline.TimelineFunction;


public interface ITimelineFunctionModel {	
	
    /** 
     * Get the function from the internal function list for desired source and context by start address.
     *  
     * @param startAddress The start address of the desired function.
     * 
     * @param source The source of the desired function.
     * 
     * @param context The context of the desired function.
     */
    public TimelineFunction getFunction(BigInteger startAddress, String source, String context);
	
	/** 
	 * Get the number of functions. 
	 */
	public int getFunctionCount();
	
	/** 
	 * Get the number of functions for a given source and context
	 * 
	 * @param idx The source of the desired function.
	 * 
	 * @param idx The context of the desired function.
	 */
	public int getFunctionCount(String source, String context);
	
	/**
	 * Find an instruction by address inside a function
	 *  
	 * @param address	The address of teh instruction.
	 * @param function	The TimelineFunction object containing it.
	 * @return	The index of the instruction from the TimelineFunction instruction vector.
	 */
	public int findInstruction(BigInteger address, TimelineFunction function);
	
	/**
	 * Gets an array of custom events that may appear in the timeline
	 * @return an array of Custom Timeline Events
	 */
	public ArrayList<TimelineCustomEvent> getCustomEvents();
	
	/**
	 * Gets a list with functions ordered by startAddrress. It is possible to
     * have duplicate function names with different address ranges. It is user 
     * responsibility to filter them. The API guarantees that there are no address overlaps.
     * 
	 * @param source The source of the desired function.
     * @param context The context of the desired function.
	 * 
	 * @return  The list of functions.
	 */
	public List<TimelineFunction> getFunctions(String source, String context);
	
	/**
	 * Gets the timeline function at the index from the list of functions
	 * ordered by start address. 
	 * 
	 * @param index The index of the function.
	 * @param source The source of the desired function.
     * @param context The context of the desired function.
	 * 
	 * @return The function at index.
	 */
	public TimelineFunction getFunction(int index, String source, String context);
}
