/*******************************************************************************
* Copyright 2016-2018 NXP. (c) 2014 - 2016 Freescale Semiconductor, Inc.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*
 * Contributors:
 *     Stiftung Deutsches Elektronen-Synchrotron, Member of the Helmholtz
 *     Association, (DESY), HAMBURG, GERMANY.
 *
 * Used:
 *     mouseDoubleClick method from
 *     org.csstudio.alarm.table.preferences.ColumnTableEditorMouseListener
 *******************************************************************************/
package com.freescale.sa.ui.timeline;


import java.math.BigInteger;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableEditor;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DragSourceAdapter;
import org.eclipse.swt.dnd.DragSourceEvent;
import org.eclipse.swt.dnd.DropTargetAdapter;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.ColorDialog;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;

import com.freescale.sa.ui.timeline.TimelineUIPlugin;
import com.freescale.sa.ui.timeline.UIHelp;
import com.freescale.sa.ui.timeline.UIMessages;


/**
 * Dialog used for Grouping/Ungrouping functions to be displayed as a single group.
 */
public class EditTimelineGroupsDialog extends Dialog {
    	
	private class AddFunctionDialog extends Dialog {
		
		List<Integer> m_selectedFuncs = new ArrayList<Integer>();

		public void addSelectedFunctionIndex(int idx) {
			m_selectedFuncs.add(idx);
		}
		
		public int getSelectedFunctionIndex(int idx) {    			
			return m_selectedFuncs.get(idx);
		}
		    		    		
		public void setSelectedItems(int[] indices) {
			for (int loopcnt = 0; loopcnt < indices.length; loopcnt++) {
				m_selectedFuncs.add(indices[loopcnt]);
			}
		}
		
		public void clearListItems() {
			m_selectedFuncs.clear();
		}
		
		public AddFunctionDialog(Shell parent) {
			super(parent);
		}
		
		@Override
		protected void okPressed() {
			if (!m_selectedFuncs.isEmpty()) {
				super.okPressed();
			}
		}
    	
		@Override
		protected Control createDialogArea(Composite parent) {
    		
			Shell shell = getShell();
			if (shell != null) {	
				shell.setText(UIMessages.EditTimelineGroupsDialog_3);
			}
			
			parent.setLayout(new GridLayout());
			parent.setLayoutData(new GridData(GridData.FILL_BOTH));
    		
			final Table functions = new Table(parent, SWT.MULTI | SWT.FULL_SELECTION | SWT.VIRTUAL);
			functions.addListener(SWT.SetData, new Listener() {
			    public void handleEvent(Event event) {
			        TableItem item = (TableItem) event.item;
			        int index = event.index;
			        TimelineFunction func = m_functionModel.getFunction(index, m_source, m_context);
			        item.setText(0, func.getName());
			        item.setText(1,
			                     "0x" + func.getStartAddress().toString(16)); //$NON-NLS-1$
			        item.setText(
			                     2,
			                     "0x" //$NON-NLS-1$
			                     + func.getEndAddress().subtract(BigInteger.ONE).toString(16));
			    }
			});
			functions.setItemCount(m_functionModel.getFunctionCount(m_source, m_context));
			
			functions.setHeaderVisible(true);
			functions.setLayoutData(new GridData(GridData.FILL_BOTH));
    		
			final TableColumn name = new TableColumn(functions, SWT.LEFT);
			name.setText(UIMessages.EditTimelineGroupsDialog_0);
			name.setWidth(150);
			final TableColumn startAddress = new TableColumn(functions, SWT.LEFT);
			startAddress.setText(UIMessages.EditTimelineGroupsDialog_1);
			startAddress.setWidth(80);
			final TableColumn endAddress = new TableColumn(functions, SWT.LEFT);
			endAddress.setText(UIMessages.EditTimelineGroupsDialog_2);
			endAddress.setWidth(80);
			
			functions.addMouseListener(new MouseAdapter() {
				public void mouseDoubleClick(MouseEvent event) {

					// Determine where the mouse was clicked
					Point pt = new Point(event.x, event.y);

					// Determine which row was selected
					final TableItem item = functions.getItem(pt);
					if (item != null) {
						// clear the selection indices list
						clearListItems();
        				
						// add the double-clicked item index
						addSelectedFunctionIndex(functions.getSelectionIndex());
    	    			
						okPressed();
					}
				}
			});
    		
			functions.addSelectionListener(new SelectionAdapter() {
				public void widgetSelected(SelectionEvent e) {
    				
					// clear the selection indices list
					clearListItems();
    				
					// add the new selection indices 
					setSelectedItems(functions.getSelectionIndices());
					
					updateOkButton();
				}
			});
    		
 			return parent;
		}
		
		private void updateOkButton() {
			// enable ok button
			Button ok = getButton(IDialogConstants.OK_ID);
			if (ok != null) {
				// disable by default hwn no function is selected
				ok.setEnabled(!m_selectedFuncs.isEmpty());
			}
		}
		
		@Override
		protected void createButtonsForButtonBar(Composite parent) {
			// create OK and Cancel buttons by default
			createButton(parent, IDialogConstants.OK_ID, IDialogConstants.OK_LABEL,
					true);
			updateOkButton();
			createButton(parent, IDialogConstants.CANCEL_ID,
					IDialogConstants.CANCEL_LABEL, false);
		}
	}

	private void updateOkButton() {
		Button ok = getButton(IDialogConstants.OK_ID);
		if (ok != null) {
			// disable by default when nothing is in the dialog table
			int countChecked = 0;
			for (int i = 0; i < m_groupsTable.getItemCount(); i++) {
				if (m_groupsTable.getItem(i).getChecked() && 
						!m_specialGroups.contains(m_groupsTable.getItem(i).getText())) {
					countChecked++;
				}
			}
			getButton(IDialogConstants.OK_ID).setEnabled(countChecked > 0);
		}
	}
	
	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		// create OK and Cancel buttons by default
		createButton(parent, IDialogConstants.OK_ID, IDialogConstants.OK_LABEL,
				true);
		updateOkButton();
		createButton(parent, IDialogConstants.CANCEL_ID,
				IDialogConstants.CANCEL_LABEL, false);
	}
    	
	public EditTimelineGroupsDialog(Shell parent,
			ITimelineFunctionModel functionsModel,
			Vector<TraceTimelinePlot.Group> initialGroups,
			String source,
			String context) {
		super(parent);
		setShellStyle(getShellStyle() | SWT.RESIZE);
		
		m_functionModel = functionsModel;
		m_groups = initialGroups;
		m_context = context;
		m_source = source;
		
		for (TraceTimelinePlot.Group group : m_groups) {
			if (group.isSpecial()) {
				m_specialGroups.add(group.getName());
			}
		}
	}
	
	private TableViewer m_groupsTableViewer;
	private Table m_groupsTable;
	private Vector<TableEditor> m_colorEditors;
	private Vector<Button> m_colorButtons;
	private Vector<Image> m_images;
	private Vector<Color> m_colors;
	
	private Menu m_menu;
	private MenuItem m_deleteGroup;
	private MenuItem m_moveUp;
	private MenuItem m_moveDown;
	private MenuItem m_merge;
	
	private static String TEXT_FULL_RANGE = "0x0 - 0xFFFFFFFF"; //$NON-NLS-1$
	
	/**
	 * The timeline m_groups.
	 */
	private Vector<TraceTimelinePlot.Group> m_groups = null;
	
	/**
	 * The timeline backup groups.
	 */
	
	Vector<TraceTimelinePlot.Group> m_backup = null;
	
	/**
	 * The list of special groups. These are not modifiable and may not support
	 * the same operations as normal groups.
	 */
	private List<String> m_specialGroups = new ArrayList<String>();
	
	/**
	 * The TImelineFunction model used to access function information.
	 */
	ITimelineFunctionModel m_functionModel = null;
	
	private String m_context = "";
	private String m_source = "";
	
	private boolean Ctrl = false;
	
	private int dragItemIndex[] = null;
	
	private int addTableItem(String name, String addresses, boolean checked, Color color) {
		
		int i = m_groupsTable.getItemCount();
		
		TableItem item = new TableItem(m_groupsTable, SWT.NONE, i);
		item.setText(0, name);
		item.setText(1, addresses);
		m_groupsTable.getItem(i).setChecked(checked);
		
		if (m_specialGroups.contains(name)) {
			item.setBackground(TraceTimelinePlot.getDefaultReservedColor());
		}
		
		m_colorEditors.add(i, new TableEditor(m_groupsTable));
		m_colorButtons.add(i, new Button(m_groupsTable, SWT.PUSH));
		m_colors.add(i, color);

		// Set attributes of the button
		m_colorButtons.elementAt(i).computeSize(SWT.DEFAULT, m_groupsTable.getItemHeight());

		// Set attributes of the editor
		m_colorEditors.elementAt(i).grabHorizontal = true;
		m_colorEditors.elementAt(i).minimumHeight = m_colorButtons.elementAt(i).getSize().y;
		m_colorEditors.elementAt(i).minimumWidth = m_colorButtons.elementAt(i).getSize().x;

		// Set the editor for the first column in the row
		m_colorEditors.elementAt(i).setEditor(m_colorButtons.elementAt(i), m_groupsTable.getItem(i),
				2);
		
		//Default color - from groups
		try {
			m_images.add(i,
					new Image(getShell().getDisplay(), m_groupsTable.getColumn(2).getWidth() - 8,
					m_groupsTable.getItemHeight() - 8));
			GC gc = new GC(m_images.elementAt(i));
			gc.setBackground(m_colors.elementAt(i));
			gc.fillRectangle(m_images.elementAt(i).getBounds());
			gc.dispose();
			m_colorButtons.elementAt(i).setImage(m_images.elementAt(i));
		} catch (Exception e) {
			e.printStackTrace();
		}

		// Create a handler for the button
		final int index = i;
		m_colorButtons.elementAt(i).addSelectionListener(
				new SelectionAdapter() {
			public void widgetSelected(SelectionEvent event) {
				ColorDialog dialog = new ColorDialog(getShell());
				if (m_colors.elementAt(index) != null) {
					dialog.setRGB(m_colors.elementAt(index).getRGB());
				}
				RGB rgb = dialog.open();
				if (rgb != null) {
					
					m_colors.set(index, new Color(getShell().getDisplay(), rgb));
					if (m_images.elementAt(index) != null) { 
						m_images.elementAt(index).dispose();
					}
					m_images.set(index,
							new Image(getShell().getDisplay(), m_groupsTable.getColumn(2).getWidth() - 8,
							m_groupsTable.getItemHeight() - 8));
					GC gc = new GC(m_images.elementAt(index));
					gc.setBackground(m_colors.elementAt(index));
					gc.fillRectangle(m_images.elementAt(index).getBounds());
					gc.dispose();
					m_colorButtons.elementAt(index).setImage(m_images.elementAt(index));
				}
			}
		});
		
		m_groupsTable.redraw();
		
		return i;
	}
	
	private void deleteTableItem(int i) {
		
		int lastGroup = m_groupsTable.getItemCount() - 1;
		while (i < lastGroup) {
			swapHandler(i, i + 1);
			i = i + 1;
		}
		m_groupsTable.remove(lastGroup);
    	
		m_colorEditors.elementAt(m_colorEditors.size() - 1).dispose();
		m_colorButtons.elementAt(m_colorButtons.size() - 1).dispose();
    	
		int lastImage = m_images.size() - 1;
		m_images.elementAt(lastImage).dispose();
		
		// colors are reusable. dispose color if not used by another item
		boolean used = false;
		int lastColor = m_colors.size() - 1;
		for (int col = 0; col < lastColor; col++) {
			if (m_colors.elementAt(col).equals(m_colors.elementAt(lastImage))) {
				used = true;
				break;
			}
		}
		if (!used) {
			m_colors.elementAt(lastColor).dispose();
		}
		
		m_colorEditors.remove(m_colorEditors.size() - 1);
		m_colorButtons.remove(m_colorButtons.size() - 1);
    	
		m_images.remove(lastImage);
		m_colors.remove(lastColor);
    	
		m_groupsTable.redraw();
		
		updateOkButton();
	}
	
	private void insertFunctionHandler() {
	
		AddFunctionDialog addFunction = new AddFunctionDialog(getShell());
		if (addFunction.open() != Dialog.OK) {
			return;
		}
							
		int insertIndex = 0;
		if (m_groupsTable.getSelectionCount() == 0) {
			insertIndex = m_groupsTable.getItemCount();
		} else {
			insertIndex = m_groupsTable.getSelectionIndex();
		}
		
		/* Insert each selected function to groups table */
		
		for (int loopcnt = 0; loopcnt < addFunction.m_selectedFuncs.size(); loopcnt++) {
			int funcIdx = addFunction.getSelectedFunctionIndex(loopcnt);		
			
			if (funcIdx < 0) {
				return;
			}
							
			/* Check if the item is not already listed in the table */
			
			int tableItemsCnt = m_groupsTable.getItemCount();
			int cnt = 0;
			boolean boFound = false;
			
			while (cnt < tableItemsCnt) {						
				if (m_functionModel.getFunction(funcIdx, m_source, m_context).getName().equals(
						m_groupsTable.getItem(cnt).getText())) {	
					boFound = true;
					break;
				}
				cnt++;
			}
												
			if (!boFound) {
			    TimelineFunction func = m_functionModel.getFunction(funcIdx, m_source, m_context);
				int added = addTableItem(func.getName(), 
						"0x" + func.getStartAddress().toString(16) //$NON-NLS-1$
						+ " - " //$NON-NLS-1$
						+ "0x" //$NON-NLS-1$ //$NON-NLS-2$
						+ func.getEndAddress().subtract(BigInteger.ONE).toString(16),
								true,
								TraceTimelinePlot.getDefaultPointColor());
			
				for (int i = added; i > insertIndex; i--) {
					swapHandler(i, i - 1);
				}
			}
		}
		
		m_groupsTable.setSelection(insertIndex);
		
		updateOkButton();
	}
	
	private void insertGroupHandler() {
		int insertIndex = 0;
		if (m_groupsTable.getSelectionCount() == 0) {
			insertIndex = m_groupsTable.getItemCount();
		} else {
			insertIndex = m_groupsTable.getSelectionIndex();
		}
		
		int added = addTableItem(UIMessages.EditTimelineGroupsDialog_8, TEXT_FULL_RANGE, true,
				TraceTimelinePlot.getDefaultPointColor());
		for (int i = added; i > insertIndex; i--) {
			swapHandler(i, i - 1);
		}
		m_groupsTable.setSelection(insertIndex);
		
		updateOkButton();
	}
	
	private void deleteGroupsHandler() {
		int delete[] = m_groupsTable.getSelectionIndices().clone();
		int max = -1, position = 0;
		for (int i = 0; i < delete.length; i++) {
			// do not delete if special groups are selected
			if (m_specialGroups.contains(m_groupsTable.getItem(delete[i]).getText())) {
				return;
			}
		}
		for (int i = 0; i < delete.length; i++) {
			// find maximum
			max = -1;
			for (int j = 0; j < delete.length; j++) {
				if (delete[j] > max) {
					position = j;
					max = delete[j];
				}
			}
			deleteTableItem(delete[position]);
			delete[position] = -1;
		}
		m_groupsTable.setSelection(-1);
		
		updateOkButton();
	}
	
	private void mergeHandler(int sources[], int destination) {
		TableItem destinationItem = m_groupsTable.getItem(destination);
		mergeHandler(sources, destinationItem);
	}
	
	private void mergeHandler(int sources[], TableItem destinationItem) {
		// do not merge special groups
		if (m_specialGroups.contains(destinationItem.getText())) {
			return;
		}
		
		TableItem sourceItems[] = new TableItem[sources.length];
		
		for (int i = 0; i < sources.length; i++) { 
			sourceItems[i] = m_groupsTable.getItem(sources[i]);
			// do not merge special groups
			if (m_specialGroups.contains(sourceItems[i].getText())) {
				return;
			}
		}
		
		/* Merge into destination */
		for (int i = 0; i < sources.length; i++) {
			destinationItem.setText(1, destinationItem.getText(1) + ", " + sourceItems[i].getText(1)); //$NON-NLS-1$
			destinationItem.setChecked(
					destinationItem.getChecked() || sourceItems[i].getChecked());
		}
		
		/* Delete the merged lines. */
		m_groupsTable.setSelection(sources);
		deleteGroupsHandler();
		
		m_groupsTable.setSelection(destinationItem);
		m_groupsTable.redraw();
	}
	
	private void swapHandler(int item1, int item2) {
		
		TableItem startItem = m_groupsTable.getItem(item1);
		TableItem endItem = m_groupsTable.getItem(item2);
		boolean checked1 = startItem.getChecked(), checked2 = endItem.getChecked();
		String name1 = startItem.getText(0), name2 = endItem.getText(0);
		String addresses1 = startItem.getText(1), addresses2 = endItem.getText(1);
		startItem.setChecked(checked2);
		startItem.setText(0, name2);
		startItem.setText(1, addresses2);
		if (m_specialGroups.contains(startItem.getText(0))) {
			startItem.setBackground(TraceTimelinePlot.getDefaultReservedColor());
		} else {
			startItem.setBackground(startItem.getParent().getBackground());
		}
		endItem.setChecked(checked1);
		endItem.setText(0, name1);
		endItem.setText(1, addresses1);
		if (m_specialGroups.contains(endItem.getText(0))) {
			endItem.setBackground(TraceTimelinePlot.getDefaultReservedColor());
		} else {
			endItem.setBackground(endItem.getParent().getBackground());
		}
		
		Image img1 = m_images.elementAt(item1), img2 = m_images.elementAt(item2);
		m_images.remove(item1);
		m_images.add(item1, img2);
		m_images.remove(item2);
		m_images.add(item2, img1);
    	
		Color colors1 = m_colors.elementAt(item1), colors2 = m_colors.elementAt(item2);
		m_colors.remove(item1);
		m_colors.add(item1, colors2);
		m_colors.remove(item2);
		m_colors.add(item2, colors1);
		
		m_colorButtons.elementAt(item1).setImage(m_images.elementAt(item1));
		m_colorButtons.elementAt(item2).setImage(m_images.elementAt(item2));
		
		m_groupsTable.setSelection(item2);
	}
	
	protected Control createDialogArea(Composite parent) {
		
		parent.setLayout(new GridLayout());
		GridData gridData = new GridData(GridData.FILL_BOTH);
		gridData.minimumHeight = 200;
		parent.setLayoutData(gridData);
		
		getShell().setText(UIMessages.EditTimelineGroupsDialog_11);
		
		m_groupsTableViewer = new TableViewer(parent, SWT.MULTI | SWT.CHECK | SWT.FULL_SELECTION);
		m_groupsTable = m_groupsTableViewer.getTable();
		GridData data = new GridData(SWT.FILL, SWT.FILL, true, true);
		m_groupsTable.setLayoutData(data);
		m_groupsTable.setHeaderVisible(true);
		m_groupsTable.setLinesVisible(true);
		m_groupsTable.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent event) {
				updateOkButton();
			}
		});
		
		m_colorEditors = new Vector<TableEditor>();
		m_colorButtons = new Vector<Button>();
		m_images = new Vector<Image>();
		m_colors = new Vector<Color>();

		final TableColumn name = new TableColumn(m_groupsTable, SWT.LEFT);
		name.setText(UIMessages.EditTimelineGroupsDialog_12);
		name.setWidth(150);
		final TableColumn addresses = new TableColumn(m_groupsTable, SWT.LEFT);
		addresses.setText(UIMessages.EditTimelineGroupsDialog_13);
		addresses.setWidth(400);
		final TableColumn color = new TableColumn(m_groupsTable, SWT.LEFT);
		color.setText(UIMessages.EditTimelineGroupsDialog_14);
		color.setWidth(50);
		color.setResizable(false);
		
		// Create an editor object to use for text editing
		final TableEditor editor = new TableEditor(m_groupsTable);
		editor.horizontalAlignment = SWT.LEFT;
		editor.grabHorizontal = true;
	    
		for (int i = 0; i < m_groups.size(); i++) {
			addTableItem(m_groups.elementAt(i).getName(), m_groups.elementAt(i).getUserAddresses(),
					m_groups.elementAt(i).getShowIndex() >= 0,
					new Color(getShell().getDisplay(), m_groups.elementAt(i).getColor()));
		}
	    
		Transfer[] transferTypes = new Transfer[] { TextTransfer.getInstance() };
		m_groupsTableViewer.addDragSupport(DND.DROP_MOVE | DND.DROP_COPY, transferTypes,
				new DragSourceAdapter() {
			public void dragSetData(DragSourceEvent event) {
				if (m_groupsTable.getSelectionCount() > 0) {
					dragItemIndex = m_groupsTable.getSelectionIndices().clone();
					event.data = m_groupsTable.getSelection()[0].getText();
				}
			}
		});

		m_groupsTableViewer.addDropSupport(DND.DROP_MOVE | DND.DROP_COPY | DND.DROP_DEFAULT,
				transferTypes, new DropTargetAdapter() {
			public void dragOver(DropTargetEvent event) {
				event.feedback = DND.FEEDBACK_SELECT | DND.FEEDBACK_SCROLL;
			}

			public void drop(DropTargetEvent event) {
				if (dragItemIndex != null) {
					if (event.item != null) {
						for (int i = 0; i < dragItemIndex.length; i++) {
							if (m_groupsTable.getItem(dragItemIndex[i]) == event.item) {
								break;
							}
						}
						mergeHandler(dragItemIndex, (TableItem)event.item);
					}
					
					dragItemIndex = null;
				}
			}
		});

		m_groupsTable.addKeyListener(
				new KeyAdapter() {
			public void keyPressed(KeyEvent e) {
				char ch = 0;
				if (e.keyCode < 256) {
					ch = Character.toLowerCase((char)e.keyCode);
				}
				if (e.keyCode == SWT.DEL) { 
					deleteGroupsHandler();
				} else if (e.keyCode == SWT.CTRL) { 
					Ctrl = true;
				} else if (e.keyCode == SWT.ARROW_UP && Ctrl
						&& m_groupsTable.getSelectionCount() > 0 && m_groupsTable.getSelectionIndex() > 0) { 
					swapHandler(m_groupsTable.getSelectionIndex(),
							m_groupsTable.getSelectionIndex() - 1);
				} else if (e.keyCode == SWT.ARROW_DOWN && Ctrl
						&& m_groupsTable.getSelectionCount() > 0
						&& m_groupsTable.getSelectionIndex() < (m_groupsTable.getItemCount() - 1)) { 
					swapHandler(m_groupsTable.getSelectionIndex(),
							m_groupsTable.getSelectionIndex() + 1);
				} else if (ch == 'g' && Ctrl) {
					insertGroupHandler();	
				} else if (ch == 'f' && Ctrl) {
					insertFunctionHandler();	
				}
			}
	    	
			public void keyReleased(KeyEvent e) {
	    		
				if (e.keyCode == SWT.CTRL) { 
					Ctrl = false;
				}
			}
		});
		
		m_menu = new Menu(getShell());
		MenuItem insertGroup = new MenuItem(m_menu, SWT.PUSH);
        insertGroup.setText(UIMessages.EditTimelineGroupsDialog_15);
        insertGroup.addListener(SWT.Selection, new Listener() {
        	public void handleEvent(Event e) {
        		insertGroupHandler();
        	}
        });
        
		MenuItem insertFunction = new MenuItem(m_menu, SWT.PUSH);
		insertFunction.setText(UIMessages.EditTimelineGroupsDialog_16);
		insertFunction.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				insertFunctionHandler();
			}
		});
        
		new MenuItem(m_menu, SWT.SEPARATOR);
        
		m_deleteGroup = new MenuItem(m_menu, SWT.PUSH);
		m_deleteGroup.setText(UIMessages.EditTimelineGroupsDialog_17);
		m_deleteGroup.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				deleteGroupsHandler();
			}
		});
        
		new MenuItem(m_menu, SWT.SEPARATOR);
        
		m_moveUp = new MenuItem(m_menu, SWT.PUSH);
		m_moveUp.setText(UIMessages.EditTimelineGroupsDialog_18);		
		m_moveUp.addListener(SWT.Selection,
				new Listener() {
			public void handleEvent(Event e) {
				swapHandler(m_groupsTable.getSelectionIndex(),
						m_groupsTable.getSelectionIndex() - 1);
			}
		});
        
		m_moveDown = new MenuItem(m_menu, SWT.PUSH);
		m_moveDown.setText(UIMessages.EditTimelineGroupsDialog_19);		
		m_moveDown.addListener(SWT.Selection,
				new Listener() {
			public void handleEvent(Event e) {
				swapHandler(m_groupsTable.getSelectionIndex(),
						m_groupsTable.getSelectionIndex() + 1);
			}
		});
        
		new MenuItem(m_menu, SWT.SEPARATOR);
        
		m_merge = new MenuItem(m_menu, SWT.PUSH);
		m_merge.setText(UIMessages.EditTimelineGroupsDialog_20);		
		m_merge.addListener(SWT.Selection,
				new Listener() {
			public void handleEvent(Event e) {
				if (m_groupsTable.getSelectionCount() == 0) {
					return;
				}

				int[] sources = new int[m_groupsTable.getSelectionCount() - 1];
				int count = 0;
				for (int i = 0; i < m_groupsTable.getSelectionCount(); i++) {
					if (m_groupsTable.getSelectionIndices()[i]
							!= m_groupsTable.getSelectionIndex()) {
						sources[count] = m_groupsTable.getSelectionIndices()[i];
						count++;
					}
				}
				mergeHandler(sources, m_groupsTable.getSelectionIndex());
			}
		});

		m_groupsTable.setMenu(m_menu);		
	    
		// Use a mouse listener, not a selection listener, since we're interested
		// in the selected column as well as row
		m_groupsTable.addMouseListener(
				new MouseAdapter() {
			public void mouseDown(MouseEvent event) {
	    		
				// Dispose any existing editor
				Control old = editor.getEditor();
				if (old != null) {
					old.dispose();
				}
	    		
				if (event.button == 3) {
					// Determine where the mouse was clicked
					Point pt = new Point(event.x, event.y);

					// Determine which row was selected
					final TableItem item = m_groupsTable.getItem(pt);
					if (item == null) {
						m_groupsTable.setSelection(-1);
					}
					
					m_menu.setVisible(true);
					if (item != null && m_specialGroups.contains(item.getText(0))) {
						m_deleteGroup.setEnabled(false);
					} else {
						m_deleteGroup.setEnabled(m_groupsTable.getSelectionCount() > 0);
					}
					m_moveUp.setEnabled(
							m_groupsTable.getSelectionCount() > 0 && m_groupsTable.getSelectionIndex() > 0);
					m_moveDown.setEnabled(
							m_groupsTable.getSelectionCount() > 0
									&& m_groupsTable.getSelectionIndex() < (m_groupsTable.getItemCount() - 1));
					m_merge.setEnabled(m_groupsTable.getSelectionCount() > 1);
				}
			}

			public void mouseDoubleClick(MouseEvent event) {

				// Dispose any existing editor
				Control old = editor.getEditor();
				if (old != null) {
					old.dispose();
				}

				// Determine where the mouse was clicked
				Point pt = new Point(event.x, event.y);

				// Determine which row was selected
				final TableItem item = m_groupsTable.getItem(pt);
				if (item != null) {
					
					// special groups are not modifiable
					if (m_specialGroups.contains(item.getText(0))) {
						return;
					}
					
					// Determine which column was selected
					int column = -1;
					for (int i = 0, n = m_groupsTable.getColumnCount(); i < n; i++) {
						Rectangle rect = item.getBounds(i);
						if (rect.contains(pt)) {
							// This is the selected column
							column = i;
							break;
						}
					}
					
					// Create the Text object for our editor
					final Text text = new Text(m_groupsTable, SWT.NONE);
					text.setForeground(item.getForeground());

					// Transfer any text from the cell to the Text control,
					// set the color to match this row, select the text,
					// and set focus to the control
					text.setText(item.getText(column));
					text.setForeground(item.getForeground());
					text.selectAll();
					text.setFocus();

					// Recalculate the minimum width for the editor
					editor.minimumWidth = text.getBounds().width;

					// Set the control into the editor
					editor.setEditor(text, item, column);

					// Add a handler to transfer the text back to the cell
					// any time it's modified
					final int col = column;
					text.addModifyListener(new ModifyListener() {
						public void modifyText(ModifyEvent event) {
							// Set the text of the editor's control back into
							// the cell
							item.setText(col, text.getText());
						}
					});
				}
			}
		});
	    
		parent.addDisposeListener(new DisposeListener() {
			public void widgetDisposed(DisposeEvent e) {
				for (int i = 0; i < m_groupsTable.getItemCount(); i++) {
					if (m_images.elementAt(i) != null) {
						m_images.elementAt(i).dispose();
					}
				}
			}
		});
	    
		// Set ID for context help.
		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, UIHelp.Timeline_Context_Id);
		
		updateOkButton();
		return parent;
	}
	
	@Override
	protected void okPressed() {

		/* backup crt groups, to restore them in case of error */
		m_backup = new Vector<TraceTimelinePlot.Group>(m_groups);
		
		/* Delete previous groups and reinit groups. */
		m_groups.clear();
		
		int countShow = 0;
		int totalCount = m_groupsTable.getItemCount();
		for (int i = 0; i < totalCount; i++) {
			try {
				TraceTimelinePlot.Group group = new TraceTimelinePlot.Group(m_groupsTable.getItem(i).getText(0),
						m_groupsTable.getItem(i).getText(1),
						m_colors.get(i).getRGB(), /*TraceTimelinePlot.getDefaultPointColor(),*/
						m_groupsTable.getItem(i).getChecked()
								? countShow
								: -1);
				if (m_specialGroups.contains(group.getName())) {
					group.setSpecial(true);
				}
			
				boolean valid = true;
				// validate names, to find duplicate group names
				// only if both are checked
				for (int j = 0; j < i; j++) {
					TraceTimelinePlot.Group grj = m_groups.get(j);
					if (group.getName().equals(grj.getName()) 
							&& (group.getShowIndex() >= 0) && (grj.getShowIndex() >= 0) ) {
						showDuplicateNameError(group.getName());
						valid = false;
						break;
					}
				}
				if (!valid) {
					m_groups.clear();
					break;
				}
				// check that the address string was correct
				if ((group.getAddressString() == null) || (group.getAddressString().isEmpty())) {
					showInvalidError(group.getName());
					m_groups.clear();
					break;
				}
				m_groups.add(group);
				if (m_groupsTable.getItem(i).getChecked()) {
					countShow++;
				}
			}
			catch (Exception e) {
				// catch any unexpected exceptions, such as widget disposed etc.
				// do not let the validation functionality be altered
				e.printStackTrace();
			}
		}
		
		/* validate groups. If there are overlaps, show message, restore groups from backup 
		 	and keep the dialog open. */
		if (m_groups.size() > 0) {
			try {
				if (validateGroups()) {
					super.okPressed();
					m_backup.clear();
					return;
				}
			} catch (Exception ex) {
				/* catch everything here since we HAVE to restore the old groups in 
				 * case of error. ANY error.
				 */
				ex.printStackTrace();
			}
		}
		/* in case anything went wrong, restore groups from backup */
		m_groups.clear();
		m_groups.addAll(m_backup);
		m_backup.clear();
	}
	
	public Vector<TraceTimelinePlot.Group> getGroups() {
		return m_groups;
	}
	
	private void showOverlapError(String name1, String name2) {
		String text = MessageFormat.format(UIMessages.EditTimelineGroupsDialog_error_0,
				new Object[] { name1, name2});
		
		Status status = new Status(IStatus.ERROR, TimelineUIPlugin.PLUGIN_ID,
				text, null);
		
		ErrorDialog.openError(Display.getCurrent().getActiveShell(),
				UIMessages.EditTimelineGroupsDialog_error_1, null, status);
	}
	
	private void showInvalidError(String name) {
		String text = MessageFormat.format(UIMessages.EditTimelineGroupsDialog_error_2,
				new Object[] { name });
		
		Status status = new Status(IStatus.ERROR, TimelineUIPlugin.PLUGIN_ID,
				text, null);
		
		ErrorDialog.openError(Display.getCurrent().getActiveShell(),
				UIMessages.EditTimelineGroupsDialog_error_3, null, status);
	}
	
	private void showDuplicateNameError(String name) {
		String text = MessageFormat.format(UIMessages.EditTimelineGroupsDialog_error_4,
				new Object[] { name });
		
		Status status = new Status(IStatus.ERROR, TimelineUIPlugin.PLUGIN_ID,
				text, null);
		
		ErrorDialog.openError(Display.getCurrent().getActiveShell(),
				UIMessages.EditTimelineGroupsDialog_error_5, null, status);
	}
	
	private boolean validateGroups() {		
		// validate address ranges, to find overlapped groups
		for (int i = 0; i < m_groups.size() - 1; i++) {
			TraceTimelinePlot.Group gri = m_groups.get(i);
			if (gri.isSpecial()) 
				continue;
			if (gri.getShowIndex() < 0)
				continue;
			for (int j = i+1; j < m_groups.size(); j++) {
				TraceTimelinePlot.Group grj = m_groups.get(j);
				if (grj.isSpecial())
					continue;
				if (grj.getShowIndex() < 0)
					continue;

				// group i contains range ends from j
				BigInteger[] startAddrVec = grj.getStartAddresses();
				BigInteger[] endAddrVec = grj.getEndAddresses();
				for (BigInteger addr : startAddrVec) {
					if (gri.containsAddress(addr)) {
						showOverlapError(gri.getName(), grj.getName());
						return false;
					}
				}
				for (BigInteger addr : endAddrVec) {
					if (gri.containsAddress(addr)) {
						showOverlapError(gri.getName(), grj.getName());
						return false;
					}
				}
				// group j contains range ends from i
				startAddrVec = gri.getStartAddresses();
				endAddrVec = gri.getEndAddresses();
				for (BigInteger addr : startAddrVec) {
					if (grj.containsAddress(addr)) {
						showOverlapError(gri.getName(), grj.getName());
						return false;
					}
				}
				for (BigInteger addr : endAddrVec) {
					if (grj.containsAddress(addr)) {
						showOverlapError(gri.getName(), grj.getName());
						return false;
					}
				}
				
			}
		}
		
		// all ok if we got here
		return true;
	}

}
