/*******************************************************************************
 * Copyright (c) 2014-2016: Freescale Semiconductor, Inc. All Rights Reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.OutputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigInteger;
import java.net.URL;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.TreeSet;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.osgi.framework.Bundle;

import com.freescale.sa.Messages;
import com.freescale.sa.SAPlugin;
import com.freescale.sa.SaConstants;

public class Utils {

    private static Logger LOGGER = Logger.getLogger(Utils.class);
    private final static String LAUNCH_CONFIG_CORE_ATTR = "com.freescale.cdt.debug.cw.CoreNameList"; //$NON-NLS-1$
    private final static String LAUNCH_CONFIG_CORE_INDEX_ATTR = "com.freescale.cdt.debug.cw.CW_SHADOWED_PREF.Embedded Initialization.coreIndex"; //$NON-NLS-1$
    private final static int LAUNCH_CONFIG_CORE_INDEX_DEFAULT_VALUE = 0;

    /**
     * Check validity for a hex field.
     *
     * @param hexString
     *            The string that contains the hex value.
     * @param fieldSize
     *            The size of the field (in no of bits).
     * @param minHexVal
     *            The minimum hex value valid for the hex value. Null or empty
     *            string if no constraint on on minimum value.
     * @param maxHexVal
     *            The maximum hex value valid for the hex value. Null or empty
     *            string if no constraint on on maximum value.
     * @return An error message describing the problem in case the field is
     *         invalid.
     */
    public static String verifyHexField(String hexString, int fieldSize, String minHexVal,
                                        String maxHexVal) {
        BigInteger value = null;

        // must start with '0x'
        if (!hexString.toLowerCase().startsWith("0x")) { //$NON-NLS-1$
            return Messages.invalid_hex_field_0x;
        }

        // must have a valid hex value
        try {
            value = new BigInteger(hexString.substring(2), 16);
        } catch (Exception e) {
            return Messages.invalid_hex_field;
        }

        // must be in range
        if (value.bitLength() > fieldSize) {
            switch (fieldSize) {
                case 4:
                    return Messages.invalid_hex_field_invalid_size_4;

                case 8:
                    return Messages.invalid_hex_field_invalid_size_8;

                case 11:
                    return Messages.invalid_hex_field_invalid_size_11;

                case 16:
                    return Messages.invalid_hex_field_invalid_size_16;

                case 32:
                    return Messages.invalid_hex_field_invalid_size_32;

                case 64:
                    return Messages.invalid_hex_field_invalid_size_64;

                default:
                    return Messages.invalid_hex_field_invalid_size;
            }
        }

        // It must be inside range if range is provided.
        if ((minHexVal != null) && (maxHexVal != null)
            && (!minHexVal.equals(CommonConstants.EMPTY_STRING))
            && (!maxHexVal.equals(CommonConstants.EMPTY_STRING))) {
            try {
                BigInteger minVal = null, maxVal = null;
                if (minHexVal.startsWith("0x")) { //$NON-NLS-1$ ;
                    minVal = new BigInteger(minHexVal.substring(2), 16);
                } else {
                    minVal = new BigInteger(minHexVal, 16);
                }
                if (maxHexVal.startsWith("0x")) { //$NON-NLS-1$ ;
                    maxVal = new BigInteger(maxHexVal.substring(2), 16);
                } else {
                    maxVal = new BigInteger(maxHexVal, 16);
                }

                if (value.compareTo(maxVal) == 1) {
                    return (Messages.invalid_dec_field_overflow + " " + maxHexVal); //$NON-NLS-1$ ;
                }
                if (value.compareTo(minVal) == -1) {
                    return (Messages.invalid_dec_field_underflow + " " + minHexVal); //$NON-NLS-1$
                }
            } catch (Exception e) {
                // If any exception occurs while getting range boundaries, range
                // constraint
                // is no longer applied.
            }
        }

        return CommonConstants.EMPTY_STRING;

    }

    /**
     * Check validity for a decimal field
     *
     * @param decString
     *            The string that contains the decimal value
     * @param minVal
     *            The minimum value accepted for this field (inclusive)
     * @param maxVal
     *            The mmaximum value accepted for this field (inclusive)
     * @return An error message describing the problem in case the field is
     *         invalid.
     */
    public static String verifyDecField(String decString, long minVal, long maxVal) {
        long longVal = 0;

        // must have a valid decimal value
        try {
            longVal = Long.parseLong(decString, 10/* radix */);

        } catch (Exception e) {
            return Messages.invalid_dec_field;
        }

        // must be in range
        if (longVal > maxVal) {
            return (Messages.invalid_dec_field_overflow + " " + maxVal); //$NON-NLS-1$
        }
        if (longVal < minVal) {
            return (Messages.invalid_dec_field_underflow + " " + minVal); //$NON-NLS-1$
        }
        return CommonConstants.EMPTY_STRING;

    }

    /**
     * Check validity for a double field
     *
     * @param doubleString
     *            The string that contains the double value
     * @param minVal
     *            The minimum value accepted for this field (inclusive)
     * @param maxVal
     *            The mmaximum value accepted for this field (inclusive)
     * @return An error message describing the problem in case the field is
     *         invalid.
     */
    public static String verifyDoubleField(String doubleString, double minVal, double maxVal) {
        double doubleVal = 0;

        // must have a valid double value
        try {
            doubleVal = Double.parseDouble(doubleString);

        } catch (Exception e) {
            return Messages.invalid_double_field;
        }

        // must be in range
        if (doubleVal > maxVal) {
            return (MessageFormat.format(Messages.invalid_double_field_overflow, maxVal));
        }
        if (doubleVal < minVal) {
            return (MessageFormat.format(Messages.invalid_double_field_underflow, minVal));
        }
        return CommonConstants.EMPTY_STRING;
    }

    /**
     * Gets a file path and returns only its name, no other path segments. May
     * also strip the extension. Use this with strange path strings (mixed or
     * non-platform specific separators, trailing separators etc.)
     *
     * @param fullName
     *            the path
     * @param keepExtention
     *            true if the function should also strip the extension
     * @return - file name only, if everything ok - empty string if the fullName
     *         is empty, or fullName is ending in a path separator, or filename
     *         only is an extension only name (e.g. "path/.config") and
     *         keepExtension is false - null if error
     */
    public static String simplifiedFileName(String fullName, boolean keepExtension) {

        if (fullName == null) {
            return null;
        }

        // Both java.io.File and org.eclipse.core.runtime do either support host
        // paths or normalized paths, but none
        // is considering both forward and backward slashes as directory
        // separator.
        int lastFwSlash = fullName.lastIndexOf('\\');
        int lastBackSlash = fullName.lastIndexOf('/');
        int lastSlash = Math.max(lastFwSlash, lastBackSlash);

        String simpleName = fullName;

        if (lastSlash >= 0) {
            simpleName = fullName.substring(lastSlash + 1);
        }

        if (keepExtension) {
            return simpleName;
        }

        // remove any extension
        int last = simpleName.lastIndexOf('.');

        if (last >= 0) {
            simpleName = simpleName.substring(0, last);
        }
        return simpleName;
    }

    /**
     * Get core name from launch configuration.
     *
     * @Note List of cores from launch configuration may contain cores of
     *       different types if architecture is heterogeneous.
     * @param launchConfig
     *            Launch configuration from where to get core name.
     * @return Core name as string.
     */
    public static synchronized String getCoreNameFromLaunch(ILaunchConfiguration launchConfig) {
        String coreName = " ";//$NON-NLS-1$

        try {

            if (launchConfig != null) {

                int coreIndex = LAUNCH_CONFIG_CORE_INDEX_DEFAULT_VALUE;
                try {
                    coreIndex = launchConfig.getAttribute(LAUNCH_CONFIG_CORE_INDEX_ATTR,
                                                          LAUNCH_CONFIG_CORE_INDEX_DEFAULT_VALUE);
                } catch (CoreException e) {
                    LOGGER.error("Unable to get core index from config " + launchConfig.getName(), //$NON-NLS-1$
                                 e);
                } finally {
                    // If core index is invalid set it to its default value.
                    if (coreIndex < 0) {
                        coreIndex = LAUNCH_CONFIG_CORE_INDEX_DEFAULT_VALUE;
                    }
                }

                List<?> coreList = launchConfig.getAttribute(LAUNCH_CONFIG_CORE_ATTR,
                                                             new ArrayList<String>());
                // Get core name according to its index in list of cores.
                if ((coreList != null) && (coreList.size() > coreIndex)) {
                    String coreNameIndexed = (String) coreList.get(coreIndex);
                    int coreIndexStartPos = coreNameIndexed.indexOf("#"); //$NON-NLS-1$
                    // Remove core's index.
                    if (coreIndexStartPos > 0) {
                        coreName = coreNameIndexed.substring(0, coreIndexStartPos);
                    }
                }
            }

        } catch (CoreException e) {
            LOGGER.error("Unable to get core name from config " + launchConfig.getName(), e); //$NON-NLS-1$
        }
        return coreName.trim();

    }

    /**
     * Get processor name from launch configuration.
     *
     * @param launchConfig
     *            Launch configuration from where to get processor name.
     * @return Processor name as string.
     */
    public static synchronized String getProcessorNameFromLaunch(ILaunchConfiguration launchConfig) {
        String procName = CommonConstants.EMPTY_STRING;

        if (launchConfig != null) {

            try {
                procName = launchConfig.getAttribute(CommonConstants.LAUNCH_TARGET_ARM_PROCESSOR_ATTR, CommonConstants.EMPTY_STRING);
            } catch (CoreException e) {
                LOGGER.error("Unable to get processor name from config " + launchConfig.getName(), //$NON-NLS-1$
                             e);
            }

        }

        return procName;

    }

    public static void deleteFile(File file) {
        boolean stat = false;
        try {
            stat = file.delete();
        } catch (SecurityException e) {
            LOGGER.warn("File cannot be deleted", e); //$NON-NLS-1$
        }
        if (!stat) {
            LOGGER.warn("File cannot be deleted"); //$NON-NLS-1$
        }
    }

    public static void createDir(File file) {
        boolean stat = false;
        try {
            stat = file.mkdirs();
        } catch (SecurityException e) {
            LOGGER.warn("Unable to create directory!", e); //$NON-NLS-1$
        }
        if (!stat) {
            LOGGER.warn("Unable to create directory!"); //$NON-NLS-1$
        }
    }

    public static void createSaOutputFolder(IPath saOutputFolder) {
        File traceFolder = saOutputFolder.toFile();
        if (!traceFolder.exists()) {
            createDir(traceFolder);
        }
    }

    /**
     * Get default storage path to the output configuration files
     * ('.metadata\.plugins\com.freescale.sa\outputConfig\').
     *
     * @return the default storage path for the output configuration file
     */
    static public IPath getDefaultOutputConfigStoragePath() {

        IPath path = SAPlugin.getDefault().getStateLocation();
        path = path.addTrailingSeparator().append(SaConstants.OUTPUT_CONFIG_FOLDER)
                   .addTrailingSeparator();
        return path;

    }

    /**
     * Extracts the file names (just the names, no extension) from the given
     * list of File objects. Keeps the order in which the files were initially
     * found in the list.
     *
     * @param filesList
     *            the list of files
     * @return the list of file names
     */
    public static String[] getFileNames(File[] filesList) {
        String[] fileNames = new String[filesList.length];
        for (int i = 0; i < filesList.length; i++) {
            String platCfgName = filesList[i].getName()
                                             .replace("." + SaConstants.PLATFORM_CONFIG_FOLDER, //$NON-NLS-1$
                                                      CommonConstants.EMPTY_STRING);
            fileNames[i] = platCfgName;
        }
        return fileNames;
    }

    /**
     * Checks if the given string is in a list of strings.
     *
     * @param stringToFind
     *            the string to find
     * @param stringsList
     *            the list of strings
     * @return true if the strings was found in the list, false otherwise
     */
    public static int foundStringInArray(String stringToFind, String[] stringsList) {
        for (int i = 0; i < stringsList.length; i++) {
            if (stringsList[i].equals(stringToFind)) {
                return i;
            }
        }
        return -1;
    }

    /**
     * Checks if the given path is valid
     *
     * @param path
     *            the path to check
     * @return true if it's valid
     */
    public static boolean isPathValid(IPath path) {
        boolean valid = true;
        if (!path.isEmpty()) {
            if (path.getDevice() == null) {
                valid = false;
            } else if (path.toString().length() > CommonConstants.MAX_LENGTH) {
                valid = false;
            } else if ((!path.toFile().exists())) {
                valid = false;
            } else {
                // Make sure no leading and trailing spaces are added.
                String trimmedPath = path.toOSString().trim();
                if (path.toOSString().length() != trimmedPath.length()) {
                    valid = false;
                }
            }
        }
        return valid;
    }

    /**
     * Get the value of "CW_SA_HOME" environment variable
     *
     * @return
     */
    public static String getCWSAHome() {
        return System.getenv(SaConstants.SA_HOME_DIR_VAR);
    }

    /**
     * Get the path to Python folder from SA layout
     *
     * @return
     */
    public static String getSAPythonFolder() {
        return getCWSAHome() + "/bin/py"; //$NON-NLS-1$
    }

    /**
     * Get the value of "CW_SA_CONFIG" environment variable
     *
     * @return
     */
    public static String getCWSAConfig() {
        return System.getenv(SaConstants.SA_CONFIG_DIR_VAR);
    }

    /**
     * Checks if the given extensionPoint exists
     *
     * @param extensionPoint
     *            the extension point name
     * @return true if it exists
     */
    public static boolean isExtension(String extensionPoint) {
        IExtensionRegistry registry = Platform.getExtensionRegistry();
        IExtensionPoint point = registry.getExtensionPoint(extensionPoint);

        if (point != null)
            return true;
        return false;
    }

    /**
     * Checks if the given view ID exists in all 'org.eclipse.ui.views'
     * extension points
     *
     * @param viewID
     *            the view extension point ID
     * @return true if it exists
     */
    public static boolean isExtensionPointView(String viewID) {
        IExtensionRegistry registry = Platform.getExtensionRegistry();
        IExtensionPoint point = registry.getExtensionPoint("org.eclipse.ui.views");
        if (point == null)
            return false;
        IExtension[] extensions = point.getExtensions();
        for (IExtension ext : extensions) {
            if (ext.getContributor().getName().equals(viewID))
                return true;
        }
        return false;
    }

    /**
     * Determine class from bundle and call a method from that class with given
     * parameters
     *
     * @param Bundle
     * @param className
     * @param methodName
     * @param invokeObj
     * @param parameterTypes
     * @throws Exception
     */
    public static Object callClassMethod(org.osgi.framework.Bundle b, String className,
                                         String methodName, Object[] invokeObj,
                                         Class<?>... parameterTypes) throws Exception {
        Object tmpObj = null;
        Object retObj = null;
        Class<?> clazz;
        clazz = b.loadClass(className);
        tmpObj = clazz.newInstance();
        if (tmpObj != null) {
            retObj = callMethod(tmpObj, methodName, invokeObj, parameterTypes);
        }

        return retObj;
    }

    /**
     * Call method from given class
     *
     * @param Bundle
     * @param clazz
     * @param methodName
     * @param invokeObj
     * @param parameterTypes
     * @throws NoSuchMethodException
     * @throws SecurityException
     * @throws InvocationTargetException
     * @throws IllegalAccessException
     * @throws IllegalArgumentException
     */
    public static Object callMethod(Object clazz, String methodName, Object[] invokeObj,
                                    Class<?>... parameterTypes) throws SecurityException,
                                                                NoSuchMethodException,
                                                                IllegalArgumentException,
                                                                IllegalAccessException,
                                                                InvocationTargetException {

        Object retObj = null;
        if (clazz != null) {
            Method m = clazz.getClass().getMethod(methodName, parameterTypes);
            retObj = m.invoke(clazz, invokeObj);
        }
        return retObj;
    }

    /**
     * Copies recursively a folder
     *
     * @param source
     * @param destination
     */
    public static void copyFolder(String source, String destination) {
        copyFolder(new File(source), new File(destination));
    }

    /**
     * Copies recursively a folder
     *
     * @param source
     * @param destination
     */
    public static boolean copyFolder(File source, File destination) {
        if (source.isDirectory()) {
            boolean status = true;
            if (!destination.exists()) {
                createDir(destination);
            }

            for (String f : source.list()) {
                status |= copyFolder(new File(source, f), new File(destination, f));
            }

            return status;
        } else {
            File destDir = destination.getParentFile();
            if (!destDir.exists()) {
                createDir(destDir);
            }

            InputStream in = null;
            OutputStream out = null;
            try {
                in = new FileInputStream(source);
                out = new FileOutputStream(destination);

                byte[] buf = new byte[1024];
                int length;
                while ((length = in.read(buf)) > 0) {
                    out.write(buf, 0, length);
                }

                return true;

            } catch (FileNotFoundException e) {
                LOGGER.error(e.getLocalizedMessage());
            } catch (IOException e) {
                LOGGER.error(e.getLocalizedMessage());
            } catch (Exception e) {
                LOGGER.error(e.getLocalizedMessage());
            } finally {
                if (in != null) {
                    try {
                        in.close();
                    } catch (IOException e) {
                        LOGGER.error(e.getLocalizedMessage());
                    }
                }
                if (out != null) {
                    try {
                        out.close();
                    } catch (IOException e) {
                        LOGGER.error(e.getLocalizedMessage());
                    }
                }
            }
        }

        return false;
    }

    private static final class ThreadConsumer extends Thread {
        private final Process p;
        private final List<String> list;
        private final String newExtension;

        ThreadConsumer(Process p, List<String> files, String newExtension) {
            super();
            this.p = p;
            this.list = files;
            this.newExtension = newExtension;
        }

        @Override
        public void run() {
            try {
                populateGCNOFiles(p.getInputStream());
            } catch (Exception _) {
                LOGGER.error("[DpaaTraceStream][ThreadConsumer] Error while reading GCNO list"); //$NON-NLS-1$
            }
        }

        private void populateGCNOFiles(InputStream s) throws IOException {
            InputStreamReader isr = new InputStreamReader(s);
            LineNumberReader lnr = new LineNumberReader(isr);
            String line = null;
            while ((line = lnr.readLine()) != null) {
                if (line.endsWith(".gcda")) //$NON-NLS-1$
                {
                    while ((line.length() > 6) && !line.matches("^([A-Za-z]:)?[/\\\\].*")) { //$NON-NLS-1$
                        line = line.substring(1);
                    }
                    IPath p = new Path(line);
                    String filename = p.toString();

                    if (!newExtension.isEmpty()) {
                        filename = filename.replace(".gcda", newExtension); //$NON-NLS-1$
                    }

                    if (!list.contains(filename))
                        list.add(filename);
                }
            }
        }
    }

    public static List<String> getGCNOLocations(String binaryPath) throws InterruptedException {
        List<String> l = new LinkedList<String>();
        Process p;
        String stringsTool = CommonConstants.T2G_STRINGS_UTIL;
        p = getStringsProcess(stringsTool, binaryPath);
        if (p == null) {
            LOGGER.error("[Utils][getGCNOLocations] Cannot start strings binutil!"); //$NON-NLS-1$
            return l;
        }
        ThreadConsumer t = new ThreadConsumer(p, l, ".gcno"); //$NON-NLS-1$
        t.start();
        p.waitFor();
        t.join();
        return l;
    }

    public static List<String> getGCDALocations(String binaryPath) throws InterruptedException {
        List<String> l = new LinkedList<String>();
        Process p;
        String stringsTool = CommonConstants.T2G_STRINGS_UTIL;
        p = getStringsProcess(stringsTool, binaryPath);
        if (p == null) {
            LOGGER.error("[Utils][getGCNOLocations] Cannot start strings binutil!"); //$NON-NLS-1$
            return l;
        }
        ThreadConsumer t = new ThreadConsumer(p, l, "");
        t.start();
        p.waitFor();
        t.join();
        return l;
    }

    private static Process getStringsProcess(String stringsTool, String binaryPath) {
        try {
            Process p = Runtime.getRuntime().exec(new String[] { stringsTool, binaryPath });
            return p;
        } catch (Exception _) {
            return null;
        }
    }

    private static final Character[] ILLEGAL_CHARACTERS = { ' ', '/', '\t', '\0', '?', '*', '\\',
                                                            '<', '>', '|', '\"', ':', ';', '%', '^', '&', '.', '`'};
    private static final List<Character> ILLEGAL_CHARACTERS_LIST = Arrays.asList(ILLEGAL_CHARACTERS);
    private static final TreeSet<Character> ILLEGAL_CHARACTERS_SET = new TreeSet<Character>(ILLEGAL_CHARACTERS_LIST);

    public static boolean isValidFileName(String filename) {
        for (int i = 0; i < filename.length(); i++) {
            char ch = filename.charAt(i);
            if ((i == 0) && (ch >= '0') && (ch <= '9')) {
                return false;
            }
            if (ch <= ' ' || ch > '~' || ILLEGAL_CHARACTERS_SET.contains(ch)) {
                return false;
            }
        }
        return true;
	}
	
	public static String getPathForFile(String path, String pluginId) {

		// if the bundle is not ready then there is no resource path
		Bundle bundle = Platform.getBundle(pluginId);

		if (Bundle.ACTIVE != bundle.getState()) {
			return null;
		}

		URL pathURL = bundle.getEntry(path);

		try {
			pathURL = FileLocator.resolve(pathURL);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return null == pathURL ? null : pathURL.getPath();
    }
}
