/*******************************************************************************
 * Copyright (c) 2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.util;

import java.io.File;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Comparator;

import org.apache.log4j.Logger;
import org.eclipse.cdt.debug.core.ICDTLaunchConfigurationConstants;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.core.sourcelookup.ISourceLookupDirector;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

import com.freescale.sa.SaConstants;

public class LaunchConfigUtils {

	private static Logger LOGGER = Logger.getLogger(LaunchConfigUtils.class);

	public static Shell getDefaultShell() {
		Shell aShell = null;

		try {
			IWorkbenchWindow window = PlatformUI.getWorkbench()
					.getActiveWorkbenchWindow();

			if (window != null) {
				aShell = window.getShell();
			} else {
				Display display = PlatformUI.getWorkbench().getDisplay();

				if (display != null) {
					aShell = display.getActiveShell();
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		if (aShell == null) {
			// Still no shell? we must be doing something wrong.
			// get the shell of the first window in workbench
			IWorkbenchWindow[] wins = PlatformUI.getWorkbench()
					.getWorkbenchWindows();

			if (wins.length > 0) {
				aShell = wins[0].getShell();
			}
		}

		return aShell;
	}

	/**
	 * Returns the launch manager.
	 * 
	 * @return launch manager
	 */
	private static ILaunchManager getLaunchManager() {
		return DebugPlugin.getDefault().getLaunchManager();
	}

	/**
	 * Use this function only after the debug session has been started. Otherwise
	 * check for the launch config name inside *.resultsConfig file.
	 *  
	 * Given a launch configuration name, get the launch configuration using the
	 * Debugger Plugin launch manager.
	 * 
	 * @param launchConfigName
	 * @return ILaunchConfiguration object or null if error.
	 */
	public static ILaunchConfiguration getLaunchConfigByName(
			String launchConfigName) {
		if (launchConfigName == null) {
			LOGGER.warn("[getLaunchConfigByName] Null argument"); //$NON-NLS-1$
			return null;
		}
		// Get the launch manager.
		ILaunchManager launchManager = getLaunchManager();

		try {
			for (ILaunchConfiguration config : launchManager
					.getLaunchConfigurations()) {
				// Match the launch config to the name that was passed in.
				if (config.getName().equals(launchConfigName)) {
					return config;
				}
			}
		} catch (CoreException e) {
			LOGGER.error(e.getMessage());
		}

		return null;
	}

	/**
	 * Helper function which return an ILaunch object for a given launch config
	 * name.
	 * 
	 * @param launchConfigName
	 * @return The ILaunch object, null if not available.
	 */
	public static ILaunch getLaunch(String launchConfigName) {

		if (launchConfigName == null) {
			LOGGER.warn("[getLaunchByName] Null argument"); //$NON-NLS-1$
			return null;
		}
		// Get the launch manager.
		ILaunchManager manager = getLaunchManager();

		for (ILaunch launch : manager.getLaunches()) {
			ILaunchConfiguration config = launch.getLaunchConfiguration();
			if (config != null && config.getName().equals(launchConfigName)) {
				return launch;
			}
		}

		return null;

	}

	/**
	 * Helper function that returns the launch configuration memento.
	 * 
	 * @param config
	 *            the launch configuration object
	 * @return the memento string, or null if error.
	 */
	public static String getMemento(ILaunchConfiguration config) {
		String memento = null;

		if (config != null) {
			ILaunchConfiguration orig = config;

			try {
				while (orig.isWorkingCopy()) {
					ILaunchConfigurationWorkingCopy copy = (ILaunchConfigurationWorkingCopy) orig
							.getAdapter(ILaunchConfigurationWorkingCopy.class);

					orig = copy.getOriginal();
				}
				memento = orig.getMemento();
			} catch (CoreException e) {
				LOGGER.error(e.getMessage());
			}
		} else {
			LOGGER.warn("[getMemento] Null argument"); //$NON-NLS-1$
		}
		return memento;
	}

	/**
	 * Helper function that returns a working copy for a launch configuration
	 * object.
	 * 
	 * @param config
	 *            the launch configuration to be copied
	 * @return the launch configuration working copy object, or null if error.
	 */
	public static ILaunchConfigurationWorkingCopy getWorkingCopy(
			ILaunchConfiguration config) {
		ILaunchConfigurationWorkingCopy copy = null;

		try {
			copy = config.getWorkingCopy();
		} catch (CoreException e) {
			LOGGER.error(e.getMessage());
		}

		return copy;
	}

	/**
	 * Helper function that does the actual update of the launch config resource
	 * from a (presumingly) updated source lookup director.
	 * 
	 * @param config
	 *            the launch configuration object to be updated
	 * @param director
	 *            the source lookup director
	 * @throws CoreException
	 */
	public static void updateLaunchConfigurationFromDirector(
			ILaunchConfiguration config, ISourceLookupDirector director)
			throws CoreException {
		if ((config == null) || (director == null)) {
			LOGGER.warn("[updateLaunchConfigurationFromDirector] Null argument"); //$NON-NLS-1$
			return;
		}

		ILaunchConfigurationWorkingCopy wCopy = getWorkingCopy(config);

		wCopy.setAttribute(ILaunchConfiguration.ATTR_SOURCE_LOCATOR_MEMENTO,
				director.getMemento());
		wCopy.setAttribute(ILaunchConfiguration.ATTR_SOURCE_LOCATOR_ID,
				director.getId());
		wCopy.doSave();
	}

	/**
	 * Get the launch configuration based on the project's name. In the case of
	 * more launch configurations for the same project, the first one found is
	 * the one retrieved.
	 * 
	 * @Note For using different launch configurations for the same project, the
	 *       other method getLaunchConfigByName(String launchConfigName) should
	 *       be used instead.
	 * 
	 * @param projectName
	 *            The project name.
	 * 
	 * @return The ILaunchConfiguration instance.
	 */
	public static ILaunchConfiguration getLaunchConfig(String projectName) {
		IPath projectPath = null;
		String launchConfigName = null;
		IProject project = ResourcesPlugin.getWorkspace().getRoot()
				.getProject(projectName);
		IPath debugSettingsPath = null;
		if (project != null) {
			projectPath = project.getLocation();
			if (projectPath != null) {
				debugSettingsPath = projectPath
						.append("Debug_Settings" + IPath.SEPARATOR); //$NON-NLS-1$
			} else {
				return null;
			}
		} else {
			return null;
		}

		File debugSettingsDir = debugSettingsPath.toFile();
		File[] files = debugSettingsDir.listFiles();
		for (File file : files) {
			String fileName = file.getName();
			int extensionIndex = fileName.lastIndexOf('.');
			if (extensionIndex > 0) {
				String fileExtension = fileName.substring(extensionIndex + 1);
				if (fileExtension.equals("launch")) { //$NON-NLS-1$
					launchConfigName = fileName.substring(0, extensionIndex);
					break;
				}
			}
		}
		ILaunchConfiguration launchConfiguration = LaunchConfigUtils
				.getLaunchConfigByName(launchConfigName);

		return launchConfiguration;
	}

	/**
	 * Returns the processor specified by the launch configuration.
	 * 
	 * @return String processor or empty string if unknown
	 */
	public static String getProcessor(ILaunchConfiguration config) {
		return CommonConstants.EMPTY_STRING;
	}

	/**
	 * Returns the target name as processor + simulation type specified by the
	 * launch configuration. Spaces are replaced with "_" so names can be used
	 * e.g. in PlatformTypes as map entry
	 * 
	 * @return String processor + simulation type or empty string if unknown
	 */
	public static String getTargetType(ILaunchConfiguration config) {

		String targetType = CommonConstants.EMPTY_STRING;
		try {
			if (config != null) {
				String procName = Utils.getProcessorNameFromLaunch(config);
				if (procName != null) {
					procName = procName.trim();
					String simType = (config.getAttribute(
							CommonConstants.LAUNCH_SIM_TYPE_ATTR,
							CommonConstants.EMPTY_STRING)).trim();
					if (simType.equals(CommonConstants.EMPTY_STRING)
							|| simType
									.equalsIgnoreCase(CommonConstants.SIMULATOR_NONE)) {
						targetType = procName;
					} else {
						targetType = procName + "_" + simType; //$NON-NLS-1$
					}
				}
			}
		} catch (Exception e) {
			LOGGER.error("Could not determine current target type\n"); //$NON-NLS-1$
			return CommonConstants.EMPTY_STRING;
		}

		return targetType.replace(" ", "_"); //$NON-NLS-1$ //$NON-NLS-2$
	}

	/**
	 * @param launchConfig
	 *            The launch configuration.
	 * @return The core index set as active in the launch configuration. -1 in
	 *         case an error occurred.
	 */
	public static int getCoreIdxFromLaunchConfig(
			ILaunchConfiguration launchConfig) {
		int coreIdx = -1;
		try {
			coreIdx = launchConfig.getAttribute(
					CommonConstants.LAUNCH_CORE_IDX_ATTR, -1);
		} catch (CoreException e) {
			LOGGER.error("[getCoreIdxFromLaunchConfig]: " //$NON-NLS-1$
					+ e.getLocalizedMessage());
		}
		return coreIdx;
	}

	public static String getSaOutputFolderFromLaunch(
			ILaunchConfiguration launchConfig) {
		return null;
	}

	/**
	 * Loads the platform configuration file from the launch configuration, if
	 * the associated attribute exists.
	 * 
	 * @param launchConfig
	 *            The current launch configuration.
	 * @return the platform configuration file or null if an error occurred or
	 *         the attribute could not be found.
	 */
	public static File loadPlatformConfigFile(ILaunchConfiguration launchConfig) {
		File platformConfigFile = null;
		String path = null;
		try {
			path = launchConfig.getAttribute(
					SaConstants.LAUNCH_CFG_ATTR_PLATFORM_CONFIG,
					CommonConstants.EMPTY_STRING);
		} catch (CoreException e) {
			path = CommonConstants.EMPTY_STRING;
		}
		IPath platCfgPath = new Path(path);
		if (platCfgPath.isValidPath(path)) {
			platformConfigFile = platCfgPath.toFile();
			if (!platformConfigFile.exists()) {
				platformConfigFile = null;
			}
		}

		return platformConfigFile;
	}

	/**
	 * Gets the enable/disable state for SA for an active launch session
	 * 
	 * @param launch
	 *            the launch session for which to obtain the SA state
	 * @return the enable/disable state of SA
	 */
	public static boolean getEnableAnalysisStateFromLaunch(ILaunch launch) {
		if (launch == null) {
			return false;
		}
		return getEnableAnalysisStateFromLaunchConfig(launch
				.getLaunchConfiguration());
	}

	/**
	 * Gets the enable/disable state for SA from the given launch configuration
	 * 
	 * @param launch
	 *            the launch configuration
	 * @return the enable/disable state of SA
	 */
	public static boolean getEnableAnalysisStateFromLaunchConfig(
			ILaunchConfiguration launchConfig) {
		if (launchConfig == null) {
			return false;
		}
		try {
			ILaunchConfigurationWorkingCopy copy = launchConfig
					.getWorkingCopy();
			boolean enableAnalysisState = copy.getAttribute(
					SaConstants.ATTR_ENABLE_ANALYSIS, false);
			return enableAnalysisState;
		} catch (CoreException e) {// ignore error and return null
			LOGGER.error("Error getting ENABLE_ANALYSIS attribute from launch config"); //$NON-NLS-1$
		}
		return false;
	}


	/**
	 * Comparator for launch configuration that compares launch configuration
	 * name.
	 * 
	 */
	public static class LaunchConfigurationNameComparator implements
			Comparator<ILaunchConfiguration>, Serializable {

		/**
		 * Serialize version UID.
		 */
		private static final long serialVersionUID = 274999031833664435L;

		@Override
		public int compare(ILaunchConfiguration arg0, ILaunchConfiguration arg1) {

			return arg0.getName().compareTo(arg1.getName());

		}
	}

	/**
	 * Get all launch configurations of a project.
	 * 
	 * @param projectName
	 *            The project name.
	 * 
	 * @return List of all launch configurations of the project or null if error
	 *         occurs while retrieving them.
	 */
	public static ArrayList<ILaunchConfiguration> getAllLaunchConfigs(
			String projectName) {
		if (projectName == null) {
			return null;
		}
		// Get the launch manager.
		ILaunchManager manager = getLaunchManager();

		ArrayList<ILaunchConfiguration> launchCfgsList = new ArrayList<ILaunchConfiguration>();

		try {
			for (ILaunchConfiguration launchConfig : manager
					.getLaunchConfigurations()) {
				if (launchConfig.getAttribute(
						ICDTLaunchConfigurationConstants.ATTR_PROJECT_NAME,
						CommonConstants.EMPTY_STRING).equals(projectName)) {
					launchCfgsList.add(launchConfig);
				}
			}
		} catch (CoreException e) {
			LOGGER.error("[getLaunchConfig] Error getting project attribute", e);
			return null;
		}

		return launchCfgsList;
	}
}
