/*******************************************************************************
 * Copyright 2016-2018 NXP
 *
 * Copyright (c) 2014, 2015 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/

package com.freescale.sa.model;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Vector;

import org.eclipse.cdt.debug.core.ICDTLaunchConfigurationConstants;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchManager;

import com.freescale.sa.SAPlugin;
import com.freescale.sa.util.CommonConstants;
import com.freescale.sa.util.LaunchConfigUtils;
import com.freescale.sa.util.Utils;

/**
 * Util class to determine supported platforms. 
 * @author B20977
 *
 */

public class PlatformType {
	
	private int  __value;
	private static int __size = 22;
	private static com.freescale.sa.model.PlatformType[] __array = new com.freescale.sa.model.PlatformType[__size];

	public static final int _TYPE_INVALID = 0;
	public static final int _TYPE_LSv7_SIM = 1;
	public static final int _TYPE_LSv7_HW = 2;
	public static final int _TYPE_LSv8_HW = 3;
			
    private static final String[] ConfigTypeStrings = new String[] { "INVALID", "LSv7_SIM",
                                                                    "LSv7_HW", "LSv8_HW"};
    
	public static final PlatformType TYPE_INVALID = new PlatformType(_TYPE_INVALID);
	public static final PlatformType TYPE_LSv7_SIM = new PlatformType(_TYPE_LSv7_SIM);
	public static final PlatformType TYPE_LSv7_HW = new PlatformType(_TYPE_LSv7_HW);
	public static final PlatformType TYPE_LSv8_HW = new PlatformType(_TYPE_LSv8_HW);

	public final static Map<String, PlatformType> platformMap = new HashMap<String, PlatformType>();

	private static Properties platformTypeProperties;
	private static String SA_CONFIG_DIR_VAR = "CW_SA_CONFIG"; //$NON-NLS-1$
	private static final String BUNDLE_PLATFORMS_LOCATION = "platforms"; //$NON-NLS-1$
	private static final String DATA_PLATFORMS_LOCATION = "/data/fsl.configs.sa.platforms"; //$NON-NLS-1$
	private static final String PLATFORMS_EXTENSION = "properties"; //$NON-NLS-1$

	synchronized public static PlatformType getPlatformType(IProject project) {
		
		if (project == null) {
			return PlatformType.TYPE_INVALID;
		} else {
			return getPlatformType(getProcessorName(project));
		}
	}

	/**
	 * Gets processor name for a project.
	 * @param project
	 * @return Processor name for project or null if any error occurs when getting processor name
	 * of project.
	 */
	public static String getProcessorName(IProject project) {
		
		if (project == null) {
			return null;
		} 
		
		ILaunchManager manager = DebugPlugin.getDefault().getLaunchManager();
	
		try {
			String procName =  CommonConstants.EMPTY_STRING;
			for (ILaunchConfiguration crLa : manager.getLaunchConfigurations()) {
				if (crLa.getAttribute(ICDTLaunchConfigurationConstants.ATTR_PROJECT_NAME, CommonConstants.EMPTY_STRING).equals(
						project.getName())) {
					
					procName =  getProcessorName(crLa);
					PlatformType type = getPlatformType(procName);
					
					if(type == PlatformType.TYPE_INVALID){
						type = getPlatformType(procName, getCoreName(crLa));
						if(type != PlatformType.TYPE_INVALID){
							return procName + getCoreName(crLa);
						}
					}
					
					return procName;
				}	
			}
		} catch (CoreException e) {
			return null;
		}
		
		return null;
	}

	public static PlatformType getPlatformType(String processorName) {	
		PlatformType type = PlatformType.platformMap.get(processorName);
		
		if (type == null) {
			// Key does not match exactly any of the keys.
			// Try a more "permitting" search.
			type = matchPlatformType(processorName);
		}
		
		// result will not be null 
		return type;
	}
	
	
	
	public static PlatformType getPlatformType(String processorName, String coreName) {	
		PlatformType type = PlatformType.platformMap.get(processorName + coreName);
		
		//If there is a coreName, do not allow to check only by processor
		if (type == null && (coreName !=null && coreName.isEmpty())) {
			// Key does not match exactly any of the keys.
			// Try a more "permitting" search.
			type = matchPlatformType(processorName);
		}
		
		// result will not be null 
		return type;
	}
	
	
	/**
	 * Attempt to match a processor name with the keys in platformMap - by doing
	 * 	case insensitive search in the map. 
	 * 
	 * @param processorName	The processor name to be matched.
	 * @return	The processor name key in the map.
	 */
	public static PlatformType matchPlatformType(String processorName) {
		if (processorName != null) {
			for (Map.Entry <String, PlatformType> entry : PlatformType.platformMap.entrySet()) {
			    if (entry.getKey().equalsIgnoreCase(processorName)) {
			        return entry.getValue();
			    }
			}
		}
		return PlatformType.TYPE_INVALID;
	}
	
	public static PlatformType getPlatformType(ILaunchConfiguration config) {
		PlatformType type = PlatformType.TYPE_INVALID;
				
		// First get platform type for "cpu name + sim type" (this is the case for VSPA where for sim and hw we have same cpu and core name)
		String procWithSimType = LaunchConfigUtils.getTargetType(config);
		type = PlatformType.getPlatformType(procWithSimType);
		
		if(type == PlatformType.TYPE_INVALID) { 
			type = getPlatformType(getProcessorName(config));
		}
		
		if(type == PlatformType.TYPE_INVALID) {
			//now try the key with core name
			type = getPlatformType(Utils.getProcessorNameFromLaunch(config));
		}
		
		return type;
	}
	
	/**
	 * Convert string into platform type.
	 * @param platformTypeStr String to be converted into platform type.
	 * @return Conversion of string into platform type. Invalid platform
	 * type is returned if string could not be converted into platform type.
	 */
	public static PlatformType convertStrToPlatformType(String platformTypeStr) {
		PlatformType platformType = PlatformType.TYPE_INVALID;
		
		for (int index = 0; index < ConfigTypeStrings.length; index++) {
			if (ConfigTypeStrings[index].equals(platformTypeStr)) {
				platformType =  new PlatformType(index);
				break;
			}
		}
		
		return platformType;
	}

	public synchronized static String getProcessorName(ILaunchConfiguration config) {
		String newProcessorName = null;
	
		return newProcessorName;
	}
	
	/**
	 * Gets a processor name for a specific platform type.
	 * @param platformType Platform type for which to get processor name.
	 * @return A processor name for given platform type.
	 * @Note There could be more processor names identified by a platform type.
	 * In this case first processor name found will be returned.
	 */
	public synchronized static String getProcessorName(PlatformType platformType) {
		String processorName = null;
		Iterator<Entry<String, PlatformType>> it = platformMap.entrySet().iterator();
		while (it.hasNext()) {
			Entry<String, PlatformType> entry = it.next();
			if (entry.getValue().toString().equals(platformType.toString())) {
				processorName = entry.getKey();
				break;
			}
		}
		
		return processorName;
	}
	
	/**
	 * Gets a processor names for a specific platform type.
	 * @param platformType Platform type for which to get processor name.
	 * @return A processor name for given platform type.
	 * @Note There could be more processor names identified by a platform type.
	 * In this case first processor name found will be returned.
	 */
	public synchronized static Vector<String> getSupportedProcessorNames(PlatformType platformType) {
		Vector<String> processorNames = new Vector<String>();
		Iterator<Entry<String, PlatformType>> it = platformMap.entrySet().iterator();
		while (it.hasNext()) {
			Entry<String, PlatformType> entry = it.next();
			if (entry.getValue().toString().equals(platformType.toString())) {
				processorNames.add(entry.getKey());
			}
		}
		return  processorNames;
	}
	
	/**
	 * Return core name from launch - may be different than internal core name.
	 * 
	 * @param config
	 * @return core name from launch configuration
	 */
	public synchronized static String getCoreName(ILaunchConfiguration config) {
		String newCoreName = CommonConstants.EMPTY_STRING;
	
		try {
			newCoreName = Utils.getCoreNameFromLaunch(config);
		} catch (Exception e) {
			e.printStackTrace();
		}
	
		return newCoreName;
	}
	
	public static PlatformType checkConfig(ILaunchConfiguration config) {
		PlatformType platType = getPlatformType(getProcessorName(config));
		
		if(platType == PlatformType.TYPE_INVALID){
			getPlatformType(getProcessorName(config) + getCoreName(config));
		}
		
		return getPlatformType(getProcessorName(config));
	}
	
	public synchronized static void loadPlatformTypeProperties() {
		if (platformTypeProperties == null) {
			platformTypeProperties = new Properties();

			// search first in the bundle's "platforms" directory (backward compatibility)
			String bundleLocation = SAPlugin.getDefault().getBundle().getLocation();
			// remove "reference:file:/"
			bundleLocation = bundleLocation.substring(bundleLocation.indexOf(":/") + 2); //$NON-NLS-1$
			File dir = new File(bundleLocation, BUNDLE_PLATFORMS_LOCATION);
			loadPlatformTypePropertiesFromFiles(dir.listFiles());

			// now search in the data/*.platforms directory
			String platformsPathString = System.getenv(SA_CONFIG_DIR_VAR);
			dir = new File(platformsPathString, DATA_PLATFORMS_LOCATION);
			loadPlatformTypePropertiesFromFiles(dir.listFiles());
		}
	}
	
	private static void loadPlatformTypePropertiesFromFiles(File[] contents) {
		if (contents == null) {
			return;
		}
		for (File file : contents) {
			// only work with .properties files
			String path = file.getPath();
			int extensionIndex = path.lastIndexOf('.') + 1;
			String extension = path.substring(extensionIndex);
			if (extension.compareToIgnoreCase(PLATFORMS_EXTENSION) != 0)
				continue;
			
			if (!file.isFile()) {
				continue;
			}
			FileInputStream stream = null;
			try {
				stream = new FileInputStream(file);
				platformTypeProperties.load(stream);
			}
			catch (IOException e) {
				SAPlugin.log(e);
				e.printStackTrace();
			} finally {
				if (stream != null) {
					try {
						stream.close();
					} catch (IOException e) {
					}
				}
			}
		}
	}
	
	static {
		try {
			loadPlatformTypeProperties();
			Iterator<Entry<Object, Object> > iter = platformTypeProperties.entrySet().iterator();
			
			while (iter.hasNext()) {
				Entry<Object, Object> entry = iter.next();
				String procName = (String)entry.getKey();
				String type = (String)entry.getValue();
				boolean placed = false;
				for (int i = 0; i < __size; i++) {
					PlatformType platformType = from_int(i);
					if (platformType != null && platformType.toString().equals(type)) {
						placed = true;
						platformMap.put(procName, platformType);
						break;
					}
				}
				if (!placed) {
					SAPlugin.log(new Status(IStatus.ERROR, SAPlugin.getUniqueIdentifier(), "PlatformType error!")); //$NON-NLS-1$
				}				
			}
		} catch (Exception e) {
			SAPlugin.log(e);
			e.printStackTrace();
		}
	}
    
	protected PlatformType(int val) {
		__value = _TYPE_INVALID;
		if (val >= 0 && val < __size) {
			__value = val;
		}
		__array[__value] = this;
	}
    
	public static PlatformType from_int(int value) {
		if (value >= 0 && value < __size) {
			return __array[value];
		} else {
			throw new IndexOutOfBoundsException();
		}
	}

	public String toString() {
		if (__value >= 0 && __value < PlatformType.ConfigTypeStrings.length) { 
			return  PlatformType.ConfigTypeStrings[__value];
		}
		return super.toString();
	}
    
	public boolean equals(PlatformType ct) {
		if (ct == null) {
			return false;
		}
		return __value == ct.__value;
	}

	public boolean equals(int i) {
		return __value == i;
	}
    
	public boolean equals(String s) {
		if (s == null) {
			return false;
		}
		return this.toString().equals(s);
	}
 
	public boolean equals(Object ct) {
		if (ct instanceof PlatformType) {
			return equals((PlatformType)ct);
		}
		return false;
	}
	
	public int hashCode() {
		return 	__value;
	}
}
