/*******************************************************************************
 * Copyright 2016-2018 NXP
 *
 * Copyright (c) 2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution.
 *******************************************************************************/
package com.freescale.sa.model;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;

import org.eclipse.debug.core.ILaunchConfiguration;

import com.freescale.sa.SAPlugin;
import com.freescale.sa.util.Utils;

public class BoardType {
	
	private final static Map<String, String> boardMap = new HashMap<String, String>();
	private static Properties boardTypeProperties;
	
	private static String SA_CONFIG_DIR_VAR = "CW_SA_CONFIG"; //$NON-NLS-1$
	private static final String DATA_BOARDS_LOCATION = "/data/fsl.configs.sa.boards"; //$NON-NLS-1$
	private static final String PREFIX_SEPARATOR = "_"; //$NON-NLS-1$
	private static final String XML_EXTENSION = ".xml"; //$NON-NLS-1$
	
	private synchronized static void loadBoardTypeProperties() {
		if (boardTypeProperties == null) {
			boardTypeProperties = new Properties();

			// search in the data/fsl.configs.sa.boards directory
			String platformsPathString = System.getenv(SA_CONFIG_DIR_VAR);
			File dir = new File(platformsPathString, DATA_BOARDS_LOCATION);
			loadBoardTypePropertiesFromFiles(dir.listFiles());
        }
	}
	
	private static void loadBoardTypePropertiesFromFiles(File[] contents) {
		
		if (contents == null) {
			return;
		}
		for (File file : contents) {
			if (!file.isFile()) {
				continue;
			}
			FileInputStream stream = null;
			try {
				stream = new FileInputStream(file);
				boardTypeProperties.load(stream);
			}
			catch (IOException e) {
				SAPlugin.log(e);
			} finally {
				if (stream != null) {
					try {
						stream.close();
					} catch (IOException e) {
					}
				}
			}
		}
	}
	
	static {
		try {
			loadBoardTypeProperties();

			Iterator<Entry<Object, Object> > iter = boardTypeProperties.entrySet().iterator();
			
			while (iter.hasNext()) {
				Entry<Object, Object> entry = iter.next();
				String boardName = (String)entry.getKey();
				String platformTypeTemplate = (String)entry.getValue();
				
				boardMap.put(boardName, platformTypeTemplate);

			}
		} catch (Exception e) {
			SAPlugin.log(e);
		}
	}
	
	/**
	 * Get platformConfig template name for a certain board.
	 * @param boardName Name of board for which to get platformConfig template.
	 * @param optionalPrefix Used to identify same boards used in different CW (e.g., APP and ARMv8)
	 * @return platformConfig template name.
	 */
	public static String getPlatformConfigTemplate(String boardName, String optionalPrefix) {
		if (optionalPrefix == null) {
			return null;
		} else if (!optionalPrefix.endsWith(PREFIX_SEPARATOR)) {
			optionalPrefix += PREFIX_SEPARATOR;
		}

		String platformConfigTemplateName = boardMap.get(optionalPrefix + boardName);
		if(platformConfigTemplateName != null)
		{
			platformConfigTemplateName = platformConfigTemplateName + XML_EXTENSION;
		}
		return platformConfigTemplateName;
	}
	
	/**
	 * Get platformConfig template name for a certain board.
	 * @param config Launch configuration for which to get platformConfig template.
	 * @param optionalPrefix Used to identify same boards used in different CW (e.g., APP and ARMv8)
	 * @return Internal core name.
	 */
	public static String getPlatformConfigTemplate(ILaunchConfiguration config, String optionalPrefix) {
		String procName = Utils.getProcessorNameFromLaunch(config);
		String platformConfigTemplateName;

		if (optionalPrefix == null) {
			return null;
		} else if (!optionalPrefix.endsWith(PREFIX_SEPARATOR)) {
			optionalPrefix += PREFIX_SEPARATOR;
		}
		
		// when have a launch config to extract boardName, convention is that board_name = processor_name
		// this is used for example for platforms that works with RT23 (do not have TCC approach)
		platformConfigTemplateName = boardMap.get(optionalPrefix + procName);
		
		if(platformConfigTemplateName != null)
		{
			platformConfigTemplateName = platformConfigTemplateName + XML_EXTENSION;
		}
		
		return platformConfigTemplateName;
	}
	
	/**
	 * Get supported boards for a CW flavor.
	 * @param cwFlavor String that identifies a CW from SA perspective.
	 * @return A list with boards supported.
	 */
	public static List<String> getSupportedBoardsForCW(String cwFlavor) {
		List<String> boardsList = new ArrayList<String>();
		
		try {
			
			for (Object key : boardTypeProperties.keySet()) {
				if(((String)key).startsWith(cwFlavor)) {
					int index = ((String)key).indexOf(cwFlavor);
					index = index + cwFlavor.length() + 1; //skip over cwFlavor string size + "_" => have only the real board name
					boardsList.add(((String)key).substring(index));
				}
	        }
			

		} catch (Exception e) {
			SAPlugin.log(e);
		}
		
		return boardsList;
	}
	
}