/*******************************************************************************
 * Copyright (c) 2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.model;


import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;

import com.freescale.sa.util.CommonConstants;
import com.freescale.sa.util.LaunchConfigUtils;
import com.freescale.sa.Messages;
import com.freescale.sa.SAPlugin;
import com.freescale.sa.model.IAnalyzer.Feature;


public class AnalysisManager {
	
	static Logger LOGGER = Logger.getLogger(AnalysisManager.class);
	private static AnalysisManager instance = new AnalysisManager();
	private List<IAnalyzer> analyzers = new ArrayList<IAnalyzer>();
	public static final String PROC_NAME_DELIM = " "; //$NON-NLS-1$

	private AnalysisManager() {
		IExtensionPoint ep = Platform.getExtensionRegistry().getExtensionPoint(SAPlugin.PLUGIN_ID,
				"analyzer"); //$NON-NLS-1$
		if (ep == null) {
			LOGGER.error("No analyzers have been defined."); //$NON-NLS-1$
			return;
		}
		for (IConfigurationElement element : ep.getConfigurationElements()) {
			try {
				IAnalyzer analyzer = (IAnalyzer)element.createExecutableExtension("class"); //$NON-NLS-1$
				analyzers.add(analyzer);
			} catch (CoreException e) {
				LOGGER.error(e);
			}
		}
	}
	
	public static AnalysisManager getDefault() {
		return instance;
	}

	/**
	 * Returns an analyzer for the specified launch or throws CoreException.
	 * 
	 * @param launch
	 * @return
	 * @throws CoreException 
	 */
	public IAnalyzer getAnalyzer(ILaunch launch) throws CoreException {
		return getAnalyzer(launch.getLaunchConfiguration());
	}
	
	/**
	 * Returns an analyzer for the specified launch config or throws CoreException.
	 * 
	 * @param config
	 * @return
	 * @throws CoreException if trace is not supported for the processor
	 */
	public IAnalyzer getAnalyzer(ILaunchConfiguration config) throws CoreException {
		return getAnalyzer(LaunchConfigUtils.getProcessor(config), config);
	}
	
	
	/**
	 * Returns an analyzer for the specified processor or throws CoreException.
	 * 
	 * @param processor
	 * @return
	 * @throws CoreException if no analyzer is found for the processor
	 */
	public IAnalyzer getAnalyzer(String processor) throws CoreException {
		
		return getAnalyzer(processor, null);
	}
	
	/**
	 * Returns an analyzer for the specified processor or throws CoreException.
	 * 
	 * @param processor
	 * @param launchConfig
	 * @return
	 * @throws CoreException if no analyzer is found for the processor
	 */
	public IAnalyzer getAnalyzer(String processor, ILaunchConfiguration launchConfig) throws CoreException {
		PlatformType platformType = PlatformType.TYPE_INVALID;
		
		// This is a workaround for VSPA sim targets that have same processor with HW targets
		// First assume processor name from PlatformType.properties is "cpu name + sim type" (this is the case for VSPA where for sim and hw we have same cpu and core name)
		if (launchConfig != null) {
			String procWithSimType = LaunchConfigUtils.getTargetType(launchConfig);
			platformType = PlatformType.getPlatformType(procWithSimType);
		}

		// it's a real processor so use only processor name
		if (platformType == PlatformType.TYPE_INVALID) {
			platformType = PlatformType.getPlatformType(processor);
		}
		
		String coreName  = CommonConstants.EMPTY_STRING;
		// still an invalid platform, try a key with processor + core name
		if (platformType == PlatformType.TYPE_INVALID && launchConfig != null)
		{
			coreName  =  PlatformType.getCoreName(launchConfig);
			platformType = PlatformType.getPlatformType(processor + coreName);
		}
		
		String platformName = platformType.toString();
		// if after all this we still have an invalid platform, maybe processor is actual architecture name and we have the analyzer
		if (platformType == PlatformType.TYPE_INVALID) {
			platformName = processor;
		}
		
		for (IAnalyzer analyzer : getAllAnalyzers()) {
			// First check if we can use the arch name.
			if(analyzer.getArchName().equals(platformName)) {
				return analyzer;
			}
			
			for (String cur : analyzer.getProcessors()) {
				//First string is always the debugger known name - support for Qonverge processor name e.g B4860 e6500 cores
				String[] strVect = cur.split(PROC_NAME_DELIM);
				//Processor name + core name
				if(strVect.length >= 2) {
					cur = strVect[0] + strVect[1];
				}
				
				if (cur.equals(processor) ||
					cur.equals(processor + coreName) ||
					cur.equals(platformName)) {
					
					return analyzer;
				}
			}
		}

		String msg = MessageFormat.format(Messages.AnalysisManager_ArchitectureNotSupported,
										  processor != null ? processor : "null"); //$NON-NLS-1$
		IStatus status = new Status(IStatus.ERROR, SAPlugin.PLUGIN_ID, msg);
		throw new CoreException(status);
	}

	/**
	 * Return the list of known analyzers.
	 * @return
	 */
	public List<IAnalyzer> getAllAnalyzers() {
		return analyzers;
	}

	/**
	 * Returns a String array of supported processors.
	 * 
	 * @return String array of supported processors.
	 */
	public String[] getProcessors() {
		List<String> processors = new ArrayList<String>();
		for (IAnalyzer analyzer : analyzers) {
			for (String processor : analyzer.getProcessors()) {
				processors.add(processor);
			}
		}
		return processors.toArray(new String[processors.size()]);
	}
	
	/**
	 * Returns a String map of supported processors. 
	 * Key is processor name (architecture string for the corresponding trace contributor)
	 * Value is the description. null if not available.
	 * 
	 * @return String map of supported processors.
	 */
	public Map<String, String> getProcessorsWithDescription() {
		Map<String, String> processors = new HashMap<String, String>();
		for (IAnalyzer analyzer : analyzers) {
			boolean supports = analyzer.supports(Feature.IMPORT);
			if (supports) {
				String[] procs = analyzer.getProcessors();
				for (String proc : procs) {
					String description = null;
					if (analyzer instanceof IProcDetails) {
						description = ((IProcDetails)analyzer).getDescription(proc);
					}
					processors.put(proc, description);
				}

			}
		}
		return processors;
	}
	
}
