/*******************************************************************************
 * Copyright (c) 2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa;

import java.io.File;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;

import com.freescale.sa.model.PlatformType;
import com.freescale.sa.util.CommonConstants;

/**
 * LaunchConfigUtils provides utilities for simplifying the use of the xpcom
 * interface that provides access to the launch configuration attributes as well
 * as for finding launch configurations of CodeWarrior type in the platform.
 */
public class LaunchConfigUtils {

	private static Logger log = Logger.getLogger(LaunchConfigUtils.class);

	/**
	 * Returns the target name as processor + simulation type specified by the
	 * launch configuration. Spaces are replaced with "_" so names can be used
	 * e.g. in PlatformTypes as map entry
	 * 
	 * @return String processor + simulation type or empty string if unknown
	 */
	public static String getTargetType(ILaunchConfiguration config) {

		String targetType = CommonConstants.EMPTY_STRING;
		try {
			if (config != null) {
				String procName = PlatformType.getProcessorName(config).trim();
				String simType = (config.getAttribute(
						CommonConstants.LAUNCH_SIM_TYPE_ATTR,
						CommonConstants.EMPTY_STRING)).trim();
				if (simType.equals(CommonConstants.EMPTY_STRING)) {
					targetType = procName;
				} else {
					targetType = procName + " " + simType; //$NON-NLS-1$
				}
			}
		} catch (Exception e) {
			log.error("Could not determine current target type\n"); //$NON-NLS-1$
			return CommonConstants.EMPTY_STRING;
		}

		return targetType.replace(" ", "_"); //$NON-NLS-1$ //$NON-NLS-2$
	}

	/**
	 * @param launchConfig
	 *            The launch configuration.
	 * @return The core index set as active in the launch configuration. -1 in
	 *         case an error occurred.
	 */
	public static int getCoreIdxFromLaunchConfig(
			ILaunchConfiguration launchConfig) {
		int coreIdx = -1;
		try {
			coreIdx = launchConfig.getAttribute(
					CommonConstants.LAUNCH_CORE_IDX_ATTR, -1);
		} catch (CoreException e) {
			log.error("[getCoreIdxFromLaunchConfig]: " //$NON-NLS-1$
					+ e.getLocalizedMessage());
		}
		return coreIdx;
	}

	public static String getSaOutputFolderFromLaunch(
			ILaunchConfiguration launchConfig) {
		return null;
	}

	/**
	 * Loads the platform configuration file from the launch configuration, if
	 * the associated attribute exists.
	 * 
	 * @param launchConfig
	 *            The current launch configuration.
	 * @return the platform configuration file or null if an error occurred or
	 *         the attribute could not be found.
	 */
	public static File loadPlatformConfigFile(ILaunchConfiguration launchConfig) {
		File platformConfigFile = null;
		String path = null;
		try {
			path = launchConfig.getAttribute(
					SaConstants.LAUNCH_CFG_ATTR_PLATFORM_CONFIG,
					CommonConstants.EMPTY_STRING);
		} catch (CoreException e) {
			path = CommonConstants.EMPTY_STRING;
		}
		IPath platCfgPath = new Path(path);
		if (platCfgPath.isValidPath(path)) {
			platformConfigFile = platCfgPath.toFile();
			if (!platformConfigFile.exists()) {
				platformConfigFile = null;
			}
		}

		return platformConfigFile;
	}

	/**
	 * Sets the platform configuration file in the launch configuration, if the
	 * associated attribute exists.
	 * 
	 * @param launchConfig
	 *            The current launch configuration.
	 * @param platformConfigPath
	 *            The path for new platform configuration.
	 */
	public static void setPlatformConfigFile(ILaunchConfiguration launchConfig,
			String platformConfigPath) {
		try {
			ILaunchConfigurationWorkingCopy workingCopy = launchConfig
					.getWorkingCopy();
			workingCopy.setAttribute(
					SaConstants.LAUNCH_CFG_ATTR_PLATFORM_CONFIG,
					platformConfigPath);
			workingCopy.doSave();
		} catch (CoreException e) {
			log.error("[setPlatformConfigFile] Error setting platformconfig path in launch config: " //$NON-NLS-1$
					+ e.getLocalizedMessage());
		}
	}
}
