/*
 ****************************************************************************
 * Copyright(c) 2017 NXP Semiconductors                                     *
 * All rights are reserved.                                                 *
 *                                                                          *
 * Software that is described herein is for illustrative purposes only.     *
 * This software is supplied "AS IS" without any warranties of any kind,    *
 * and NXP Semiconductors disclaims any and all warranties, express or      *
 * implied, including all implied warranties of merchantability,            *
 * fitness for a particular purpose and non-infringement of intellectual    *
 * property rights.  NXP Semiconductors assumes no responsibility           *
 * or liability for the use of the software, conveys no license or          *
 * rights under any patent, copyright, mask work right, or any other        *
 * intellectual property rights in or to any products. NXP Semiconductors   *
 * reserves the right to make changes in the software without notification. *
 * NXP Semiconductors also makes no representation or warranty that such    *
 * application will be suitable for the specified use without further       *
 * testing or modification.                                                 *
 *                                                                          *
 * Permission to use, copy, modify, and distribute this software and its    *
 * documentation is hereby granted, under NXP Semiconductors' relevant      *
 * copyrights in the software, without fee, provided that it is used in     *
 * conjunction with NXP Semiconductor products(UCODE I2C, NTAG I2C, LPC8N04)*
 * This  copyright, permission, and disclaimer notice must appear in all    *
 * copies of this code.                                                     *
 ****************************************************************************
 */
package com.nxp.lpc8nxxnfcdemo.reader;

import java.io.IOException;

import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.util.Arrays;

import org.ndeftools.EmptyRecord;
import org.ndeftools.Message;
import org.ndeftools.MimeRecord;
import org.ndeftools.Record;
import org.ndeftools.wellknown.TextRecord;
import org.w3c.dom.Text;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.DialogInterface;
import android.content.Intent;
import android.nfc.FormatException;
import android.nfc.NdefMessage;
import android.nfc.NdefRecord;
import android.nfc.Tag;
import android.nfc.tech.Ndef;
import android.nfc.tech.NfcA;
import android.nfc.tech.NfcB;
import android.nfc.tech.NfcF;
import android.nfc.tech.NfcV;
import android.os.AsyncTask;
import android.widget.TextView;
import android.widget.Toast;

import com.nxp.lpc8nxxnfcdemo.activities.MainActivity;
import com.nxp.lpc8nxxnfcdemo.crypto.CRC32Calculator;
import com.nxp.lpc8nxxnfcdemo.exceptions.CommandNotSupportedException;
import com.nxp.lpc8nxxnfcdemo.fragments.LEDScrollingDisplayFragment;
import com.nxp.lpc8nxxnfcdemo.fragments.NdefFragment;
import com.nxp.lpc8nxxnfcdemo.fragments.OtaUpdateFragment;
import com.nxp.lpc8nxxnfcdemo.listeners.WriteEEPROMListener;
import com.nxp.lpc8nxxnfcdemo.reader.Nfc_Get_Version.Prod;
import com.nxp.lpc8nxxnfcdemo.utils.LEDUtil;
import com.nxp.lpc8nxxnfcdemo.R;

import static android.view.View.VISIBLE;
import static java.lang.Boolean.FALSE;
import static java.lang.Boolean.TRUE;
import static java.lang.Boolean.logicalAnd;

/**
 * Class for the different Demos.
 *
 * @author NXP67729
 *
 */

public class Nfc_lpc8nxx_Demo implements WriteEEPROMListener {

	private Lpc_Enabled_Commands reader;
	private Activity main;
	private Tag tag;

	/**
	 *
	 * Taskreferences.
	 *
	 */
	private WriteEmptyNdefTask emptyNdeftask;
	private WriteDefaultNdefTask defaultNdeftask;
	private NDEFReadTask ndefreadtask = null;
	private NDEFLedScrollDisplayReadTask ndefledscrolldisplayreadtask = null;
	private flashTask flashMemorytask = null;

	/**
	 *
	 * DEFINES.
	 *
	 */
	private static final int LAST_FOUR_BYTES 	= 4;
	private static final int DELAY_TIME 		= 100;
	private static final int TRAILS 			= 300;
	private static final int DATA_SEND_BYTE 	= 12;
	private static final int VERSION_BYTE 		= 63;
	private static final int GET_VERSION_NR 	= 12;
	private static final int GET_FW_NR 			= 28;
	private static final int THREE_BYTES 		= 3;
	private static final int PAGE_SIZE 			= 4096;
	private static final byte LED_EXTENDED_DISPLAY_IDENTIFIER = (byte) 0x52;
	private static final byte TUNE_MIME_RECORD_IDENTIFIER = (byte) 0x53;
	private static final int FLASHDOWNLOAD_CHUNK_SIZE = 260;
	private static final byte LPC_DEVICE_STATUS_ID = (byte) 0xB0;
	private static final byte FLASHPROG_CMD_ID = (byte) 0x31;
	private static final byte FLASHPROG_RESPONSE_ID = (byte) 0xB1;
	private static final byte FORCESBL_CMD_ID = (byte)0x34;
	private static final byte FORCESBLCMD_RESPONSE_ID = (byte)0xB4;
	private static final byte VERIFYPROG_CMD_ID = (byte) 0x32;
	private static final byte VERIFYPROG_RESPONSE_ID = (byte) 0xB2;
	private static final byte FINISHDOWNLOAD_CMD_ID = (byte) 0x33;
	private static final byte FINISHDOWNLOAD_RESPONSE_ID = (byte) 0xB3;
	private static final byte RESP_STATUS_SBL_ERR_OK = (byte) 0x00;
	private static final byte RESP_STATUS_SBL_ERR_WRONG_LEN = (byte) 0x80;
	private static final byte RESP_STATUS_SBL_ERR_CRC_FAIL = (byte) 0x81;
	private static final byte RESP_STATUS_SBL_ERR_INVALID_HEADER = (byte) 0x82;
	private static final byte RESP_STATUS_SBL_ERR_WRONG_CHECKSUM = (byte) 0x83;
	private static final byte RESP_STATUSSBL_ERR_IMG_TOO_BIG = (byte) 0x84;
	private static final byte RESP_STATUS_SBL_ERR_IMG_VERIFY_FAIL = (byte) 0x85;
	private static final byte RESP_STATUS_SBL_ERR_WRONG_NDEF = (byte) 0x86;
	private static final byte RESP_STATUS_SBL_ERR_LOWER_VERSION = (byte) 0x87;
	private static final byte RESP_STATUS_SBL_DOWNLOAD_NOT_POSSIBLE = (byte) 0xC0;
	private static final String PROGRESSBARFAIL = "2";
	private static final String PROGRESSBARPASS = "1";
	private static final String PROGRESSBARDEFAULT = "0";
	private static  int success_flag = 0;
	private static final String DISABLE_BUTTONS = "Disable buttons";
	private static final String ENABLE_BUTTONS = "Enable Buttons";

	/**
	 * Constructor.
	 *
	 * @param tag
	 *            Tag with which the Demos should be performed
	 * @param main
	 *            MainActivity
	 */
	public Nfc_lpc8nxx_Demo(Tag tag, final Activity main) {
		try {
			if (tag == null) {
				this.main = null;
				this.tag = null;
				return;
			}
			this.main = main;
			this.tag = tag;

			reader = Lpc_Enabled_Commands.get(tag);

			if (reader == null) {
				String message = "The Tag could not be identified or this NFC device does not "
						+ "support the NFC Forum commands needed to access this tag";
				String title = "Communication failed";
				showAlert(message, title);
			} else {
				reader.connect();
			}

			Nfc_Get_Version.Prod prod = reader.getProduct();

			if (!prod.equals(Nfc_Get_Version.Prod.Unknown)) {
				if (prod.equals(Nfc_Get_Version.Prod.NTAG_I2C_1k_Plus)
			     || prod.equals(Nfc_Get_Version.Prod.NTAG_I2C_2k_Plus)) {
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	private void showAlert(final String message, final String title) {
		main.runOnUiThread(new Runnable() {
			@Override
			public void run() {
				new AlertDialog.Builder(main)
						.setMessage(message)
						.setTitle(title)
						.setPositiveButton("OK",
								new DialogInterface.OnClickListener() {
									@Override
									public void onClick(DialogInterface dialog,
											int which) {

									}
								}).show();
			}
		});

	}

	public void debugMessage(final String message) {
		main.runOnUiThread(new Runnable() {
			@Override
			public void run() {
				OtaUpdateFragment.setOtaTextBox(message);
			}
		});

	}


	/**
	 * Checks if the tag is still connected based on the previously detected reader.
	 *
	 * @return Boolean indicating tag connection
	 *
	 */
	public boolean isConnected() {
		return reader.isConnected();
	}

	/**
	 * Checks if the tag is still connected based on the tag.
	 *
	 * @return Boolean indicating tag presence
	 *
	 */
	public static boolean isTagPresent(Tag tag) {
		final Ndef ndef = Ndef.get(tag);
		if (ndef != null && !ndef.getType().equals("android.ndef.unknown")) {
			try {
				ndef.connect();
				final boolean isConnected = ndef.isConnected();
				ndef.close();
				return isConnected;
			} catch (final IOException e) {
				e.printStackTrace();
				return false;
			}
		} else {
			final NfcA nfca = NfcA.get(tag);
			if (nfca != null) {
				try {
					nfca.connect();
					final boolean isConnected = nfca.isConnected();
					nfca.close();

					return isConnected;
				} catch (final IOException e) {
					e.printStackTrace();
					return false;
				}
			} else {
				final NfcB nfcb = NfcB.get(tag);
				if (nfcb != null) {
					try {
						nfcb.connect();
						final boolean isConnected = nfcb.isConnected();
						nfcb.close();
						return isConnected;
					} catch (final IOException e) {
						e.printStackTrace();
						return false;
					}
				} else {
					final NfcF nfcf = NfcF.get(tag);
					if (nfcf != null) {
						try {
							nfcf.connect();
							final boolean isConnected = nfcf.isConnected();
							nfcf.close();
							return isConnected;
						} catch (final IOException e) {
							e.printStackTrace();
							return false;
						}
					} else {
						final NfcV nfcv = NfcV.get(tag);
						if (nfcv != null) {
							try {
								nfcv.connect();
								final boolean isConnected = nfcv.isConnected();
								nfcv.close();
								return isConnected;
							} catch (final IOException e) {
								e.printStackTrace();
								return false;
							}
						} else {
							return false;
						}
					}
				}
			}
		}
	}

	/**
	 *
	 * Finish all tasks.
	 *
	 */
	public void finishAllTasks() {
		NDEFReadFinish();
		//OtaFinish();
	}

	/**
	 * Checks if the demo is ready to be executed.
	 *
	 * @return Boolean indicating demo readiness
	 *
	 */
	public boolean isReady() {
		if (tag != null && reader != null) {
			return true;
		}
		return false;
	}

	/**
	 * Get the product from reader.
	 *
	 * @return product
	 *
	 */
	public Prod getProduct() throws IOException {
		return reader.getProduct();
	}


	/**
	 * Reads the whole tag memory content.
	 * 
	 * @return Boolean indicating success or error
	 */
	public byte[] readTagContent() {
		byte[] bytes = null;
		try {
			// The user memory and the first four pages are displayed
			int memSize = reader.getProduct().getMemsize() + 16;
			// Read all the pages using the fast read method
			bytes = reader.readEEPROM(0, memSize / reader.getBlockSize());
		} catch (IOException e) {
			e.printStackTrace();
		} catch (FormatException e) {
			e.printStackTrace();
		} catch (CommandNotSupportedException e) {
			e.printStackTrace();
			showDemoNotSupportedAlert();
		} catch (Exception e) {
			e.printStackTrace();
		}
		return bytes;
	}
	
	private void showTagNotPlusAlert() {
		String message = main.getString(R.string.tag_not_supported);
		String title = main.getString(R.string.tag_not_supported_title);
		showAlert(message, title);
	}

	private void showDemoNotSupportedAlert() {
		String message = main.getString(R.string.demo_not_supported);
		String title = main.getString(R.string.demo_not_supported_title);
		showAlert(message, title);
	}

	/**
	 * Resets the whole tag memory content (Memory to 00000...)
	 * 
	 * @return Boolean indicating success or error
	 */
	public boolean resetTagContent() {
		boolean success = true;
		try {
			byte[] d = new byte[reader.getProduct().getMemsize()];
			reader.writeEEPROM(d,this);
		} catch (IOException e) {
			success = false;
			e.printStackTrace();
		} catch (FormatException e) {
			success = false;
			e.printStackTrace();
		} catch (CommandNotSupportedException e) {
			showDemoNotSupportedAlert();
			e.printStackTrace();
		}
		return success;
	}



	/**
	 * Stops the NDEFRead Demo
	 */
	public void NDEFReadFinish() {
		if (ndefreadtask != null && !ndefreadtask.isCancelled()) {
			ndefreadtask.exit = true;
			try {
				ndefreadtask.get();
			} catch (Exception e) {
				e.printStackTrace();
			}
			ndefreadtask = null;
			// Clean all the fields
			NdefFragment.resetNdefDemo();
		}
	}

	/* Concatenate two byte arrays*/
	public byte[] concatByteArrays(byte[] s1,byte[]s2)
	{
		byte[] dest = new byte[s1.length + s2.length];

		System.arraycopy(s1, 0, dest, 0, s1.length);
		System.arraycopy(s2, 0, dest, s1.length, s2.length);

		return dest;
	}


	/* Performs LED SCrolling Display Demo*/

	public void LEDSCROLLDISPLAY(){

		ndefledscrolldisplayreadtask = new NDEFLedScrollDisplayReadTask();
		ndefledscrolldisplayreadtask.execute();
	}

	public boolean LEDScrollDisplayTaskStatus()
	{
		if(ndefledscrolldisplayreadtask != null)
		{
			return true;
		}
		else
			return false;
	}

	public void LEDSCrollDisplayTaskCancel()
	{
		boolean[][] ClearBuffer = new boolean[25][7];
		LEDScrollingDisplayFragment.setLedMatrix(ClearBuffer);
		ndefledscrolldisplayreadtask.cancel(true);
	}
	/**
	 * Performs the NDEF Demo
	 */
	public void NDEF() throws IOException, FormatException, CommandNotSupportedException, InterruptedException {
		// Check if the operation is read or write
		if (NdefFragment.isWriteChosen() == true) {

			// NDEF Message to write in the tag
			NdefMessage msg = null;
			NdefRecord recordTune = null;
			NdefRecord recordText = null;
			NdefRecord recordTextScrollSpeed = null;

			// Get the selected NDEF type since the creation of the NDEF Msg
			// will vary depending on the type
			if (NdefFragment.getNdefType().equalsIgnoreCase(
					main.getResources().getString(R.string.radio_text))) {
				//msg = createNdefTextMessage(NdefFragment.getText());
				recordText = createNdefTextRecord(NdefFragment.getText());
			} else if (NdefFragment.getNdefType().equalsIgnoreCase(
					main.getResources().getString(R.string.radio_none))) {
				recordText = null;
			}


				byte scrollSpeed = NdefFragment.getLedScrollSpeed();
				byte[] sp = {0x51, scrollSpeed}; /* LED SCROLLING SPEED MIME IDENTIFIER = 0x51*/

				recordTextScrollSpeed = new NdefRecord(NdefRecord.TNF_MIME_MEDIA,"application/octet-stream".getBytes(), new byte[0],sp);

			/* Append MIME Record for Tunes here*/
			if(NdefFragment.getNdefTune().equalsIgnoreCase(main.getString(R.string.tune1))){
				/* Tune 1 - Jingle Bells*/
				int tempo1 = 200;
				byte songLength1 = 26;
				String song1 = "eeeeeeegcde fffffeeeeddedg";
				byte[] beats = {1, 1, 2, 1, 1, 2, 1, 1, 1, 1, 4, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2};

				byte[] b1 = ByteBuffer.allocate(4).putInt(tempo1).array();
				byte[] temp1 = {TUNE_MIME_RECORD_IDENTIFIER, b1[3],b1[2], (byte) songLength1}; /* TUNE MIME RECORD IDENTIFIER = 0x53*/
				byte[] s1 = song1.getBytes();

				byte[] ires1 = concatByteArrays(temp1,s1);
				byte[] tune1 = concatByteArrays(ires1,beats); /* Final Tune1 byte array*/

				recordTune = new NdefRecord(NdefRecord.TNF_MIME_MEDIA,"application/octet-stream".getBytes(), new byte[0],tune1);
			}else if(NdefFragment.getNdefTune().equalsIgnoreCase(main.getString(R.string.tune2))){
				/* Tune 2 - twinkle twinkle little star*/
				int tempo1 = 300;
				byte songLength1 = 15;
				String song1 = "ccggaag ffeeddc";
				byte[] beats =  { 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 2 };

				byte[] b1 = ByteBuffer.allocate(4).putInt(tempo1).array();
				byte[] temp1 = {TUNE_MIME_RECORD_IDENTIFIER, b1[3],b1[2], (byte) songLength1}; /* TUNE MIME RECORD IDENTIFIER = 0x53*/
				byte[] s1 = song1.getBytes();

				byte[] ires1 = concatByteArrays(temp1,s1);
				byte[] tune2 = concatByteArrays(ires1,beats); /* Final Tune2 byte array*/

				recordTune = new NdefRecord(NdefRecord.TNF_MIME_MEDIA,"application/octet-stream".getBytes(), new byte[0],tune2);
			}else if(NdefFragment.getNdefTune().equalsIgnoreCase(main.getString(R.string.tune3))){
				/* Tune 3  - Fur Elise*/
				int tempo1 = 175;
				byte songLength1 = (byte) 132;
				String song1 = "EDEDEbDCaceabegbCeEDEDEbDCaceabeCbaEDEDEbDCaceabegbCeEDEDEbDCaceabeCbabCDEgFEDfEDCeDCbeEeEDEDEDEDEDEDEbDCaceabegbCeEDEDEbDCaceabeCba";
				byte[] beats = { 1,1,1,1,1,1,1,1,3,1,1,1,3,1,1,1,3,1,1,1,1,1,1,1,1,1,3,1,1,1,3,1,1,1,3,1,1,1,1,1,1,1,1,3,1,1,1,3,1,1,1,3,1,1,1,1,1,1,1,1,1,3,1,1,
						1,3,1,1,1,3,1,1,1,2,1,1,1,2,1,1,1,2,1,1,1,3,1,5,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,3,1,1,1,3,1,1,1,3,1,1,1,1,1,1,1,1,1,3,1,1,1,3,1,1,1,6};


				byte[] b1 = ByteBuffer.allocate(4).putInt(tempo1).array();
				byte[] temp1 = {TUNE_MIME_RECORD_IDENTIFIER, b1[3],b1[2], (byte) songLength1}; /* TUNE MIME RECORD IDENTIFIER = 0x53*/
				byte[] s1 = song1.getBytes();

				byte[] ires1 = concatByteArrays(temp1,s1);
				byte[] tune4 = concatByteArrays(ires1,beats); /* Final Tune4 byte array*/

				recordTune = new NdefRecord(NdefRecord.TNF_MIME_MEDIA,"application/octet-stream".getBytes(), new byte[0],tune4);
			}else {
				recordTune = null;
			}

			if(recordText != null && recordTune != null){
				NdefRecord[] records = {recordText, recordTextScrollSpeed, recordTune};
				msg = new NdefMessage(records);
			}else if(recordTune != null){
				NdefRecord[] records = {recordTextScrollSpeed, recordTune};
				msg = new NdefMessage(records);
			}else if(recordText != null){
				NdefRecord[] records = {recordText, recordTextScrollSpeed};
				msg = new NdefMessage(records);
			}else if(recordText == null && recordTune == null)
			{
				NdefRecord[] records = {recordTextScrollSpeed};
				msg = new NdefMessage(records);
			}


			nfcReadWriteOperationCheck(); // Check state of nfctag before read/write operation

			try {
				long timeToWriteNdef = NDEFWrite(msg);


				Toast.makeText(main, "write tag successfully done", Toast.LENGTH_LONG).show();
				NdefFragment.setAnswer("Tag successfully written");

				/* Write Idle State once NFC write operation is completed*/
				writeNfcRow(5,false);

				int bytes = msg.toByteArray().length;
				String Message = "";
				
				// Transmission Results
				Message = Message.concat("Speed (" + bytes + " Byte / "
						+ timeToWriteNdef + " ms): "
						+ String.format("%.0f", bytes / (timeToWriteNdef / 1000.0))
						+ " Bytes/s");
				NdefFragment.setDatarate(Message);
			} catch (Exception e) {
				Toast.makeText(main, "write tag failed", Toast.LENGTH_LONG).show();
				NdefFragment.setDatarate("Error writing NDEF");
				e.printStackTrace();
			}
		} else {
			ndefreadtask = new NDEFReadTask();
			ndefreadtask.execute();
		}
	}

	public void nfcReadWriteOperationCheck() throws IOException, FormatException, InterruptedException, CommandNotSupportedException {
		int retryCount = 0;
		int MAX_RETRY_COUNT = 1000;
		byte[] tempread = new byte[20];
		for(retryCount = 0; retryCount < MAX_RETRY_COUNT; retryCount++)
		{
			tempread = reader.readEEPROM(4,7); // reads 4 pages always
			if((tempread[0] == (byte)0xFD) && (tempread[1] == (byte)0x02) && (tempread[2] == (byte)0x00) && (tempread[3] == (byte)0x00)) {
				writeNfcRow(5, true);
				tempread = reader.readEEPROM(4, 7); // reads 4 pages always

				/* If nfc tag on embedded side is idle*/
				if (((tempread[0] == (byte)0xFD) && (tempread[1] == (byte)0x02) && (tempread[2] == (byte)0x00) && (tempread[3] == (byte)0x00))
					&& ((tempread[4] == (byte) 0xFD) && (tempread[5] == (byte) 0x02) && (tempread[6] == (byte) 0x01) && (tempread[7] == (byte) 0x00)))
				{
					break;
				} else {	/* if nfc tag is not idle on embedded side*/
					writeNfcRow(5, false);
					Thread.sleep(5); // 5ms

				}
			}else {	/* if nfc tag is not idle on embedded side*/
				Thread.sleep(5);
			}
		}

	}

	public boolean nfcReadScrollOperationCheck() throws IOException, FormatException, InterruptedException, CommandNotSupportedException {

		byte[] tempread ={0};
		boolean stat = false;
		tempread = reader.readEEPROM(4, 7); // reads 4 pages always

		if (tempread[0] == (byte) 0xFD && tempread[1] == (byte) 0x02 && tempread[2] == (byte) 0x00 && tempread[3] == (byte) 0x00) {
			writeNfcRow(5, true);

			tempread[2] = (byte) 0x00;
			tempread = reader.readEEPROM(4, 7); // reads 4 pages always

			/* If nfc tag on embedded side is idle*/
			if ((tempread[0] == (byte) 0xFD && tempread[1] == (byte) 0x02 && tempread[2] == (byte) 0x00 && tempread[3] == (byte) 0x00)
				&& (tempread[4] == (byte) 0xFD && tempread[5] == (byte) 0x02 && tempread[6] == (byte) 0x01 && tempread[7] == (byte) 0x00))
			{
				return true;
			} else {	/* if nfc tag is not idle on embedded side*/
				writeNfcRow(5, false);
				return false;
			}
	} else {	/* if nfc tag is not idle on embedded side*/
			return false;
		}

	}

	/* Write the Second page of NFC memory*/
	public void writeNfcRow(int rowNum, boolean busy) throws IOException, FormatException, CommandNotSupportedException {

		byte[] nfcBusyWriteRow = new byte[4];

		nfcBusyWriteRow[0] = (byte) 0xFD; // proprietary record
		nfcBusyWriteRow[1] = (byte) 0x02; // length
		nfcBusyWriteRow[2] = (byte) 0x01; // Busy
		nfcBusyWriteRow[3] = (byte) 0x00; // reserved

		byte[] nfcIdleWriteRow = new byte[4];
		nfcIdleWriteRow[0] = (byte) 0xFD; // proprietary record
		nfcIdleWriteRow[1] = (byte) 0x02; // length
		nfcIdleWriteRow[2] = (byte) 0x00; // Idle
		nfcIdleWriteRow[3] = (byte) 0x00; // reserved

		if(busy){
			reader.writeEEPROMCustom(nfcBusyWriteRow,rowNum,null, false);
		}else {
			reader.writeEEPROMCustom(nfcIdleWriteRow,rowNum,null,false);
		}

	}
	private class NDEFReadTask extends AsyncTask<Void, String, Void> {

		private Boolean exit = false;

		@Override
		protected Void doInBackground(Void... params) {
			try {
				while (true) {

					// cancel task if requested
					if (exit) {
						cancel(true);
						return null;
					}

					nfcReadWriteOperationCheck(); // Check state of nfctag before read/write operation

					long RegTimeOutStart = System.currentTimeMillis();
					// Get the message Type
					NdefMessage msg = reader.readNDEF();

					// NDEF Reading time statistics
					long timeToReadNdef = System.currentTimeMillis() - RegTimeOutStart;
					/* Write Idle State once NFC read operation is completed*/
					writeNfcRow(5,false);

					Message highLevelMsg = new Message(msg);

					// Get the message type in order to deal with it
					// appropriately
					String type = "none";
					if (!highLevelMsg.isEmpty()) {
						type = highLevelMsg.get(0).getClass().getSimpleName();
					}

					byte[] temp = new byte[10];
					String message = "";
					int num_record = 1;
					byte[] ledScrollParameters = new byte[9];
					for (Record rec : highLevelMsg) {
						if (rec instanceof EmptyRecord) {
							message += "#" + num_record++ + ": "
									+ "EmptyRecord\n";
						} else if (rec instanceof TextRecord) {
							message += "#" + num_record++ + " "
									+ rec.getClass().getSimpleName() + ":\n"
									+ ((TextRecord) rec).getText() + "\n\n";
						}else if (rec instanceof MimeRecord) {
							temp = ((MimeRecord) rec).getData();
							if(temp[0] == LED_EXTENDED_DISPLAY_IDENTIFIER)
							{
								ledScrollParameters = ((MimeRecord) rec).getData();
							}
						}
					}

					int scrollSpeed = 25 - ( ledScrollParameters[7] & 0xFF);
					NdefFragment.setLedScrollSpeed(scrollSpeed);

					int bytes = msg.toByteArray().length;
					String readTimeMessage = "";

					// Transmission Results
					readTimeMessage = readTimeMessage.concat("Speed (" + bytes + " Byte / "
							+ timeToReadNdef + " ms): "
							+ String.format("%.0f", bytes / (timeToReadNdef / 1000.0))
							+ " Bytes/s");

					// Put the message content on the screen
					publishProgress(type, message, readTimeMessage);

					// sleep 500ms, but check if task was cancelled
					for (int i = 0; i < 1; i++) {
						// cancle task if requested
						if (exit || !NdefFragment.isNdefReadLoopSelected()) {
							cancel(true);
							return null;
						}
						Thread.sleep(10);
					}

				}
			} catch (CommandNotSupportedException e) {
				e.printStackTrace();
				showAlert(
						"The NDEF Message is to long to read with this Nfc Device",
						"Demo not supported");
			} catch (Exception e) {
				e.printStackTrace();
			}
			return null;
		}

		@Override
		protected void onProgressUpdate(String... progress) {
			NdefFragment.setNdefType(progress[0]);
			NdefFragment.setNdefMessage(progress[1]);
			NdefFragment.setDatarate(progress[2]);
		}

		@Override
		protected void onPostExecute(Void bla) {
			if (!exit) {
				Toast.makeText(main, main.getString(R.string.Tag_lost),
						Toast.LENGTH_LONG).show();
			}
			NdefFragment.resetNdefDemo();
		}
	}


	private class NDEFLedScrollDisplayReadTask extends AsyncTask<Void, String, Void> {

		private Boolean exit = false;
		private boolean[][] DisplayBuffer = null;
		private boolean[][] ClearBuffer = new boolean[25][7];
		private boolean ret = false;
		@Override
		protected Void doInBackground(Void... params) {
			try {
				while (true) {
					// cancle task if requested
					if (exit) {
						cancel(true);
						return null;
					}
					long RegTimeOutStart = System.currentTimeMillis();
					ret = false;
					ret = nfcReadScrollOperationCheck(); // Check state of nfctag before read/write operation

					if(ret == true) {
						// Get the message Type
						NdefMessage msg = reader.readNDEF();

					/* Write Idle State once NFC read operation is completed*/
						writeNfcRow(5, false);

						if (msg != null) {
							// NDEF Reading time statistics
							long timeToReadNdef = System.currentTimeMillis() - RegTimeOutStart;
							Message highLevelMsg = new Message(msg);

							// Get the message type in order to deal with it
							// appropriately
							String type = "none";
							if (!highLevelMsg.isEmpty()) {
								type = highLevelMsg.get(0).getClass().getSimpleName();
							}

							String ledScrollDisplayText = "";
							byte[] ledScrollParameters = new byte[9];
							String message = "";
							int num_record = 0;
							byte[] temp = new byte[10];
							for (Record rec : highLevelMsg) {
								num_record++;
								if (rec instanceof TextRecord) {
									ledScrollDisplayText = ((TextRecord) rec).getText();
								} else if (rec instanceof MimeRecord) {
									temp = ((MimeRecord) rec).getData();
									if(temp[0] == LED_EXTENDED_DISPLAY_IDENTIFIER)
									{
										ledScrollParameters = ((MimeRecord) rec).getData();
									}
								}
							}

							if (num_record != 0) {
					/* ledScrollParameters data format*/
								byte ledScrollMimeIdentifier = ledScrollParameters[0];

								int charlIdx = ledScrollParameters[1] & 0xFF;

								int scrollColumnNum = ledScrollParameters[2] & 0xFF;

								byte[] bcharScrollRemainingTime = {ledScrollParameters[3], ledScrollParameters[4]};
								int charScrollRemainingTime = ByteBuffer.wrap(bcharScrollRemainingTime).getShort();

								byte[] bscrollPeriod = {ledScrollParameters[5], ledScrollParameters[6]};

								int scrollPeriod = ByteBuffer.wrap(bscrollPeriod).getShort();

								int scrollSpeed = 25 - (ledScrollParameters[7] & 0xFF);
								byte ledFontIdentifier = ledScrollParameters[8];

								if (ledScrollMimeIdentifier == LED_EXTENDED_DISPLAY_IDENTIFIER) {  /*LED Extended display Identifier is 0x52*/
									DisplayBuffer = LEDUtil.getInstance().setScrollText(ledScrollDisplayText, charlIdx, scrollColumnNum, charScrollRemainingTime, scrollPeriod,ledFontIdentifier);
								}

								publishProgress("scroll");
							}
						}
					}
					Thread.sleep(4); // Run thread every n ms
				}
			} catch (CommandNotSupportedException e) {
				e.printStackTrace();
			} catch (Exception e) {
				e.printStackTrace();
			}
			return null;
		}

		@Override
		protected void onProgressUpdate(String... progress) {
			if(DisplayBuffer != null)
			{
				LEDScrollingDisplayFragment.setLedMatrix(DisplayBuffer);
			}

		}

		@Override
		protected void onPostExecute(Void bla) {
			if (!exit) {
				Toast.makeText(main, main.getString(R.string.Tag_lost),
						Toast.LENGTH_LONG).show();
				LEDScrollingDisplayFragment.setLedMatrix(ClearBuffer);
				ndefledscrolldisplayreadtask.cancel(true);
			}

		}
	}


	/* OTA Update Demo*/
	public void OtaUpdate(){
		// NDEF Message to write in the tag
		NdefMessage msg = null;
		NdefRecord recordTune = null;
	}

	public boolean OtaUpdateTaskStatus()
	{
		if(flashMemorytask != null)
		{
			return true;
		}
		else
			return false;
	}

	/* Stops OTA Demo */
	public void OtaFinish()
	{
        if(flashMemorytask != null && !flashMemorytask.isCancelled()){
            flashMemorytask.exit = true;

        }
        try {
            flashMemorytask.get();
        }catch (Exception e){
            e.printStackTrace();
        }
        flashMemorytask = null;
        //Clean all the fields
        OtaUpdateFragment.resetOtaDemo();
	}

	public void OtaUpdateTaskCancel()
	{
		OtaUpdateFragment.resetOtaDemo();
		flashMemorytask.cancel(true);

	}
	public byte[] readLPCDeviceStatus() throws InterruptedException, FormatException, CommandNotSupportedException, IOException {

		byte[] deviceStatus = new byte[10];

		// Check state of nfctag before read/write operation
		nfcReadWriteOperationCheck();

		// Get the message Type
		NdefMessage msg = reader.readNDEF();
		/* Write Idle State once NFC read operation is completed*/
		writeNfcRow(5,false);

		Message highLevelMsg = new Message(msg);

		// Get the message type in order to deal with it appropriately
		String type = "none";
		if (!highLevelMsg.isEmpty()) {
			type = highLevelMsg.get(0).getClass().getSimpleName();
		}
		int record_found = 0;
		byte[] temp = new byte[20];
		for (Record rec : highLevelMsg) {
			temp[0] = 0x00;
			if (rec instanceof MimeRecord) {
				temp = ((MimeRecord) rec).getData();
				if(temp[0] == LPC_DEVICE_STATUS_ID)
				{
					deviceStatus = ((MimeRecord) rec).getData();
					record_found = 1;
					return deviceStatus;
				}
			}
		}

		return null;
	}

	public boolean readIsLPCDeviceStatusSBL() throws InterruptedException, FormatException, CommandNotSupportedException, IOException {

		byte[] deviceStatus = new byte[10];
		int i =0;
		int MAX_RETRY_COUNT = 1000;
		int record_found = 0;

		for(i = 0; i <	MAX_RETRY_COUNT; i++)
		{
			// Check state of nfctag before read/write operation
			nfcReadWriteOperationCheck();

			// Get the message Type
			NdefMessage msg = reader.readNDEF();
			/* Write Idle State once NFC read operation is completed*/
			writeNfcRow(5,false);

			Message highLevelMsg = new Message(msg);

			// Get the message type in order to deal with it appropriately
			String type = "none";
			if (!highLevelMsg.isEmpty()) {
				type = highLevelMsg.get(0).getClass().getSimpleName();
			}
			byte[] temp = new byte[20];
			for (Record rec : highLevelMsg) {
				temp[0] = 0x00;
				if (rec instanceof MimeRecord) {
					temp = ((MimeRecord) rec).getData();
					if((temp[0] == LPC_DEVICE_STATUS_ID) && (temp[1] == 'S') && (temp[2] == 'B') && (temp[3] == 'L'))
					{
						flashMemorytask.publishmessage("","","");
						return true;
					}
				}
			}
		}

		return false;
	}

	public boolean readIsLPCDeviceStatusApp() throws InterruptedException, FormatException, CommandNotSupportedException, IOException {

		byte[] deviceStatus = new byte[10];
		int i =0;
		int MAX_RETRY_COUNT = 1000;

		for(i = 0; i <	MAX_RETRY_COUNT; i++)
		{
			// Check state of nfctag before read/write operation
			nfcReadWriteOperationCheck();

			// Get the message Type
			NdefMessage msg = reader.readNDEF();
			/* Write Idle State once NFC read operation is completed*/
			writeNfcRow(5,false);

			Message highLevelMsg = new Message(msg);

			// Get the message type in order to deal with it appropriately
			String type = "none";
			if (!highLevelMsg.isEmpty()) {
				type = highLevelMsg.get(0).getClass().getSimpleName();
			}
			int record_found = 0;
			byte[] temp = new byte[20];
			for (Record rec : highLevelMsg) {
				temp[0] = 0x00;
				if (rec instanceof MimeRecord) {
					temp = ((MimeRecord) rec).getData();
					if((temp[0] == LPC_DEVICE_STATUS_ID) && (temp[1] == 'A') && (temp[2] == 'P') && (temp[3] == 'P'))
					{
						return true;
					}
				}
			}
		}


		return false;
	}

	// Send command to force LPC8N04 to SBL Mode
	public  boolean forceLPCSBLMode() throws InterruptedException, FormatException, CommandNotSupportedException, IOException {
		// NDEF Message to write in the tag
		NdefMessage msg = null;
		NdefRecord recordSbl = null;
		byte[] cmdStatus = new byte[2];
		byte[] deviceStatus = new byte[10];
		int record_found = 0;
		int retry_count = 0;
		int MAX_RETRY_COUNT = 1000;

		byte[] sblCommand = {FORCESBL_CMD_ID}; // 0x34 is command to force LPC into SBL mode
		recordSbl = new NdefRecord(NdefRecord.TNF_MIME_MEDIA,"application/octet-stream".getBytes(),new byte[0],sblCommand);

		NdefRecord[] records = {recordSbl};
		msg = new NdefMessage(records);

		try{
			/* Prepare to Write force sbl command to LPC8N04 */
			// Check state of nfctag before read/write operation
			nfcReadWriteOperationCheck();
			long timeToWriteNdef = NDEFWrite(msg);
			/* Write Idle State once NFC write operation is completed*/
			writeNfcRow(5,false);

		}catch (Exception e) {
			flashMemorytask.publishmessage("Flash programming terminated.","","");
			e.printStackTrace();
            return false;
		}

		cmdStatus[0] = 0x00;
		cmdStatus[1] = 0x00;

		/* Prepare to Read Command Status from LPC8N04 */
		for(retry_count = 0;((retry_count < MAX_RETRY_COUNT) && (record_found == 0)); retry_count++)
		{
			// Check state of nfctag before read/write operation
			nfcReadWriteOperationCheck();
			// Get the message Type
			NdefMessage smsg = reader.readNDEF();
			/* Write Idle State once NFC read operation is completed*/
			writeNfcRow(5,false);
			Message highLevelMsg = new Message(smsg);

			// Get the message type in order to deal with it appropriately
			String type = "none";
			if (!highLevelMsg.isEmpty()) {
				type = highLevelMsg.get(0).getClass().getSimpleName();
			}

			byte[] temp = new byte[20];

			// Check record for command status or device status
			for (Record rec : highLevelMsg) {
				temp[0] = 0x00;
				if (rec instanceof MimeRecord) {
					temp = ((MimeRecord) rec).getData();
					if(temp[0] == FORCESBLCMD_RESPONSE_ID)
					{
						cmdStatus = ((MimeRecord) rec).getData();
						record_found = 1;
						break;
					}
					if((temp[0] == LPC_DEVICE_STATUS_ID) && (temp[1] == 'S') && (temp[2] == 'B')
							&& (temp[3] == 'L'))
					{
						deviceStatus = ((MimeRecord) rec).getData();
						record_found = 1;
						break;
					}
				}
			}

		}/* End retry for loop*/

		// No record found with valid response id or timed out
		if((record_found == 0) || (retry_count >= MAX_RETRY_COUNT))
		{
			flashMemorytask.publishmessage("Flash programming time out ! Process terminated.","","");
			return false;
		}else if((cmdStatus[0] == FORCESBLCMD_RESPONSE_ID) && (cmdStatus[1] == RESP_STATUS_SBL_ERR_OK))
		{
			flashMemorytask.publishmessage(" ","","");
			return true;
		} else if((deviceStatus[0] == LPC_DEVICE_STATUS_ID) && (deviceStatus[1] == 'S') && (deviceStatus[2] == 'B')
				&& (deviceStatus[3] == 'L'))
		{
			flashMemorytask.publishmessage(" ","","");
			return true;
		}
		else
		{
			flashMemorytask.publishmessage("Flash Programming not possible","","");
			return false;
		}

	}

	/**
	 * Performs the flashing of a new bin file from the NFC Device.
	 *
	 * @param bytesToFlash
	 *           Byte Array containing the new firmware to be flashed
	 * @return Boolean operation result
	 * @throws IOException
	 * @throws FormatException
	 */
	public boolean programFlash(byte[] bytestoFlash) throws InterruptedException, FormatException, CommandNotSupportedException, IOException {
		byte[] flashData = new byte[261];
        NdefRecord recordFlash = null;
        NdefMessage msg = null;
        int MAX_RETRY_COUNT = 1000;
        int record_found = 0;
        int retry_count = 0;
		byte[] temparray = {FLASHPROG_CMD_ID,(byte) 0x00};
        byte[] cmdStatus = new byte[2];
		int length = bytestoFlash.length;
		int total_num_download_chunks = (bytestoFlash.length/FLASHDOWNLOAD_CHUNK_SIZE);
        int pos = 0;
		int progressvalue = 0;

		flashMemorytask.publishmessage("OTA Flash Programming Started...","","");

		for(int i=1; i <= total_num_download_chunks; pos = pos + FLASHDOWNLOAD_CHUNK_SIZE, i++)
		{
			record_found = 0;
			Arrays.fill(flashData,(byte)0x00);
			System.arraycopy(temparray, 0, flashData, 0, 1);
			System.arraycopy(bytestoFlash, pos, flashData, 1, FLASHDOWNLOAD_CHUNK_SIZE);
            recordFlash = new NdefRecord(NdefRecord.TNF_MIME_MEDIA,"application/octet-stream".getBytes(),new byte[0],flashData);
            NdefRecord[] records = {recordFlash};
            msg = new NdefMessage(records);

            try {
                nfcReadWriteOperationCheck();
                long timeToWriteNdef = NDEFWrite(msg);
                /* Write Idle State once NFC write operation is completed*/
                writeNfcRow(5,false);
            } catch (Exception e) {
				flashMemorytask.publishmessage(ENABLE_BUTTONS,"","");
				flashMemorytask.publishmessage("Flash Programming terminated !!!", String.valueOf(progressvalue),PROGRESSBARFAIL);
                e.printStackTrace();
				return false;
            }

            /* Prepare to Read Command Status from LPC8N04 */
            for( retry_count = 0;((retry_count < MAX_RETRY_COUNT) && (record_found == 0)); retry_count++)
            {
                // Check state of nfctag before read/write operation
                nfcReadWriteOperationCheck();
                // Get the message Type
                NdefMessage smsg = reader.readNDEF();
			    /* Write Idle State once NFC read operation is completed*/
                writeNfcRow(5,false);
                Message highLevelMsg = new Message(smsg);

                // Get the message type in order to deal with it appropriately
                String type = "none";
                if (!highLevelMsg.isEmpty()) {
                    type = highLevelMsg.get(0).getClass().getSimpleName();
                }

                byte[] temp = new byte[20];

                // Check record for command status
                for (Record rec : highLevelMsg) {
                    temp[0] = 0x00;
                    if (rec instanceof MimeRecord) {
                        temp = ((MimeRecord) rec).getData();
                        if(temp[0] == FLASHPROG_RESPONSE_ID)
                        {
                            cmdStatus = ((MimeRecord) rec).getData();
                            record_found = 1;
                            break;
                        }
                    }
                }

            }/* End retry for loop*/

            // No record found with valid response id or timed out
            if((record_found == 0) || (retry_count >=MAX_RETRY_COUNT))
            {
                flashMemorytask.publishmessage("Flash programming time out ! Process terminated !!!",
									String.valueOf(progressvalue),PROGRESSBARFAIL);
                return false;
            }

            if(cmdStatus[1] == RESP_STATUS_SBL_ERR_OK)
            {
				progressvalue = ((i * FLASHDOWNLOAD_CHUNK_SIZE) * 100) /length;
				flashMemorytask.publishmessage("",String.valueOf(progressvalue),PROGRESSBARDEFAULT);
                continue;
            }
            else
            {   switch (cmdStatus[1])
                {
                    case RESP_STATUS_SBL_ERR_CRC_FAIL:  flashMemorytask.publishmessage("CRC failed ! Flash programming terminated.","","");
                                                        break;
                    case RESP_STATUS_SBL_ERR_WRONG_LEN: flashMemorytask.publishmessage("Wrong Length! Flash programming terminated.","","");
                                                        break;
                    case RESP_STATUS_SBL_ERR_INVALID_HEADER: flashMemorytask.publishmessage("Invalid Header! Flash programming terminated","","");
                                                             break;
                    case RESP_STATUS_SBL_ERR_WRONG_CHECKSUM: flashMemorytask.publishmessage("Wrong Checksum! Flash programming terminated","","");
                                                             break;
                    case RESP_STATUSSBL_ERR_IMG_TOO_BIG: flashMemorytask.publishmessage("Image too big! Flash programming terminated","","");
                                                         break;
                    case RESP_STATUS_SBL_ERR_WRONG_NDEF: flashMemorytask.publishmessage("Flash programming terminated","","");
                                                        break;
                    case RESP_STATUS_SBL_ERR_LOWER_VERSION: flashMemorytask.publishmessage("Image lower version! Flash programming terminated","","");
                                                            break;
					default:							flashMemorytask.publishmessage("Flash programming terminated !!!","","");
														break;

                }
				flashMemorytask.publishmessage("",String.valueOf(progressvalue),PROGRESSBARFAIL);
                return false;

            }

        }
		flashMemorytask.publishmessage("", String.valueOf(progressvalue),PROGRESSBARPASS);

		return true; // Flash Programming Successful
	}


	public boolean VerifyDownloadImage() throws InterruptedException, FormatException, CommandNotSupportedException, IOException {
		// NDEF Message to write in the tag
		NdefMessage msg = null;
		NdefRecord recordVerify = null;
		byte[] cmdStatus = new byte[2];
		int record_found = 0;
		int retry_count = 0;
		int MAX_RETRY_COUNT = 1000;

		byte[] sblCommand = {VERIFYPROG_CMD_ID};
		recordVerify = new NdefRecord(NdefRecord.TNF_MIME_MEDIA,"application/octet-stream".getBytes(),new byte[0],sblCommand);

		NdefRecord[] records = {recordVerify};
		msg = new NdefMessage(records);

		try{
			/* Prepare to Write force sbl command to LPC8N04 */
			// Check state of nfctag before read/write operation
			nfcReadWriteOperationCheck();
			long timeToWriteNdef = NDEFWrite(msg);
			/* Write Idle State once NFC write operation is completed*/
			writeNfcRow(5,false);

		}catch (Exception e) {
			flashMemorytask.publishmessage("Error ! Image Verification failed","",PROGRESSBARFAIL);
			e.printStackTrace();
			return false;
		}

		/* Prepare to Read Command Status from LPC8N04 */
		for(retry_count = 0;((retry_count < MAX_RETRY_COUNT) && (record_found == 0)); retry_count++)
		{
			// Check state of nfctag before read/write operation
			nfcReadWriteOperationCheck();
			// Get the message Type
			NdefMessage smsg = reader.readNDEF();
			/* Write Idle State once NFC read operation is completed*/
			writeNfcRow(5,false);
			Message highLevelMsg = new Message(smsg);

			// Get the message type in order to deal with it appropriately
			String type = "none";
			if (!highLevelMsg.isEmpty()) {
				type = highLevelMsg.get(0).getClass().getSimpleName();
			}

			byte[] temp = new byte[20];

			// Check record for command status
			for (Record rec : highLevelMsg) {
				temp[0] = 0x00;
				if (rec instanceof MimeRecord) {
					temp = ((MimeRecord) rec).getData();
					if(temp[0] == VERIFYPROG_RESPONSE_ID)
					{
						cmdStatus = ((MimeRecord) rec).getData();
						record_found = 1;
						break;
					}
				}
			}

		}/* End retry for loop*/

		// No record found with valid response id or timed out
		if((record_found == 0) || (retry_count >=MAX_RETRY_COUNT))
		{
			flashMemorytask.publishmessage("Image Verification time out ! Process terminated.","",PROGRESSBARFAIL);
			return false;
		}

		if(cmdStatus[1] == RESP_STATUS_SBL_ERR_OK)
		{
			return true;
		}
		else
		{
			flashMemorytask.publishmessage("Image Verification failed ! Process terminated.","",PROGRESSBARFAIL);
			return false;
		}
	}

	public boolean DownloadComplete() throws InterruptedException, FormatException, CommandNotSupportedException, IOException {
		// NDEF Message to write in the tag
		NdefMessage msg = null;
		NdefRecord recordFinish = null;
		byte[] cmdStatus = new byte[2];
		byte[] deviceStatus = new byte[10];
		int record_found = 0;
		int retry_count = 0;
		int MAX_RETRY_COUNT = 1000;

		byte[] sblCommand = {FINISHDOWNLOAD_CMD_ID};
		recordFinish = new NdefRecord(NdefRecord.TNF_MIME_MEDIA,"application/octet-stream".getBytes(),new byte[0],sblCommand);

		NdefRecord[] records = {recordFinish};
		msg = new NdefMessage(records);

		try{
			/* Prepare to Write force sbl command to LPC8N04 */
			// Check state of nfctag before read/write operation
			nfcReadWriteOperationCheck();
			long timeToWriteNdef = NDEFWrite(msg);
			/* Write Idle State once NFC write operation is completed*/
			writeNfcRow(5,false);

		}catch (Exception e) {
			flashMemorytask.publishmessage("Error ! Flash Programming terminated !!","",PROGRESSBARFAIL);
			e.printStackTrace();
			return false;
		}

		/* Prepare to Read Command Status from LPC8N04 */
		for(retry_count = 0;((retry_count < MAX_RETRY_COUNT) && (record_found == 0)); retry_count++)
		{
			// Check state of nfctag before read/write operation
			nfcReadWriteOperationCheck();
			// Get the message Type
			NdefMessage smsg = reader.readNDEF();
			/* Write Idle State once NFC read operation is completed*/
			writeNfcRow(5,false);
			Message highLevelMsg = new Message(smsg);

			// Get the message type in order to deal with it appropriately
			String type = "none";
			if (!highLevelMsg.isEmpty()) {
				type = highLevelMsg.get(0).getClass().getSimpleName();
			}

			byte[] temp = new byte[20];

			// Check record for command status or device status
			for (Record rec : highLevelMsg) {
				temp[0] = 0x00;
				if (rec instanceof MimeRecord) {
					temp = ((MimeRecord) rec).getData();
					if(temp[0] == FINISHDOWNLOAD_RESPONSE_ID)
					{
						cmdStatus = ((MimeRecord) rec).getData();
						record_found = 1;
						break;
					}
					if(temp[0] == LPC_DEVICE_STATUS_ID)
					{
						deviceStatus = ((MimeRecord)rec).getData();
						record_found = 1;
						break;
					}
				}
			}

		}/* End retry for loop*/

		// No record found with valid response id or timed out
		if((record_found == 0) || (retry_count >=MAX_RETRY_COUNT))
		{
			flashMemorytask.publishmessage("Flash Programming time out...! Process terminated.","",PROGRESSBARFAIL);
			return false;
		}

		if(cmdStatus[1] == RESP_STATUS_SBL_ERR_OK)
		{
			return true;
		} else if ((deviceStatus[0] == LPC_DEVICE_STATUS_ID) && (deviceStatus[1] == 'A') && (deviceStatus[2] == 'P')
				&& (deviceStatus[3] == 'P'))
		{
			return true;
		}
		else
		{
			flashMemorytask.publishmessage("Flash Programming failed...! Process terminated.","",PROGRESSBARFAIL);
			return false;
		}
	}

	public void readImageVersionfromDevice()
	{
		byte[] device_status = null;
		short major_ver_num = 0;
		short minor_ver_num = 0;
		short rev_num = 0;

		//Check LPC8N04 Device Status ( Application running mode or SBL mode)
		try {
			device_status = readLPCDeviceStatus();
		} catch (Exception e) {
			device_status = null;
			OtaUpdateFragment.setOtaTextBox("Cannot read LPC8N04 device status !");
			e.printStackTrace();
		}
		if (device_status == null) {
			OtaUpdateFragment.setOtaTextBox("Cannot read LPC8N04 device status !");
		}
		else if ((device_status[0] == LPC_DEVICE_STATUS_ID) && (device_status[1] == 'A') && (device_status[2] == 'P')
				&& (device_status[3] == 'P')){

			major_ver_num = (short)((device_status[5] << 8) | (device_status[4] & 0xFF));
			minor_ver_num = (short)((device_status[7] << 8) | (device_status[6] & 0xFF));
			rev_num = (short)((device_status[9] << 8) | (device_status[8] & 0xFF));

			OtaUpdateFragment.setOtaTextBox("Application Image Version : " + major_ver_num + "." + minor_ver_num + "." + rev_num);

		}else if ((device_status[0] == LPC_DEVICE_STATUS_ID) && (device_status[1] == 'S') && (device_status[2] == 'B')
				&& (device_status[3] == 'L'))
		{
			major_ver_num = (short)((device_status[5] << 8) | (device_status[4] & 0xFF));
			minor_ver_num = (short)((device_status[7] << 8) | (device_status[6] & 0xFF));
			rev_num = (short)((device_status[9] << 8) | (device_status[8] & 0xFF));

			OtaUpdateFragment.setOtaTextBox("Bootloader Version : " + major_ver_num + "." + minor_ver_num + "." + rev_num);
		}
		else
		{
			OtaUpdateFragment.setOtaTextBox("LPC8N04 Device Status incorrect !");
		}
	}

	public void flashMemoryOTA()
	{
		flashMemorytask = new flashTask();
		flashMemorytask.execute();

	}

	private class flashTask extends AsyncTask<Void, String, Void> {

		private Boolean exit = false;
		byte[] device_status = null;
		boolean ret = false;
		byte[] binaryFileBytes = OtaUpdateFragment.getBinaryFileBytes();

		@Override
		protected Void doInBackground(Void... params) {
			try {
				while (true) {
					// cancel task if requested
					if (exit || isCancelled()) {
						cancel(true);
						return null;
					}
					//Check LPC8N04 Device Status ( Application running mode or SBL mode)
					device_status = readLPCDeviceStatus();

					if (device_status == null) {
						publishProgress("Cannot read LPC8N04 device status. Flash programming terminated !!!","","");
						flashMemorytask.cancel(true);
					} else {    // Check if device status is APP or SBL
						if ((device_status[0] == LPC_DEVICE_STATUS_ID) && (device_status[1] == 'A') && (device_status[2] == 'P')
								&& (device_status[3] == 'P')) {
								ret = false;
								// Force LPC8N04 to SBL mode
								ret = forceLPCSBLMode();
								publishProgress(" ","","");
								if(ret)
								{
									publishProgress(" ","","");
									ret = readIsLPCDeviceStatusSBL(); // Confirm if LPC8N04 is in SBL mode
								}
								if(ret == false)
								{
									// Error in forcing LPC8N04 into SBL Mode. Exit task and terminate flash programming.
									publishProgress("Flash Programming terminated !","",PROGRESSBARFAIL);
									flashMemorytask.cancel(true);
								} else // LPC8N04 is in SBL mode. Proceed with Flash Programming.
								{
									long RegTimeOutStart = System.currentTimeMillis();
									ret = false;
									byte[] bytesToFlash = binaryFileBytes;

									/* Disable button touch  while flash programming in progress */
									flashMemorytask.publishmessage(DISABLE_BUTTONS,"","");
									ret = programFlash(bytesToFlash);
									flashMemorytask.publishmessage(ENABLE_BUTTONS,"","");

									if(ret == true)
									{
										publishProgress("Flash Programming Completed Successfully !","","");
									}else
									{
										flashMemorytask.cancel(true);
									}
								}
						}
						else if ((device_status[0] == LPC_DEVICE_STATUS_ID) && (device_status[1] == 'S') && (device_status[2] == 'B')
								&& (device_status[3] == 'L'))
						{
							long RegTimeOutStart = System.currentTimeMillis();
							ret = false;
							byte[] bytesToFlash = binaryFileBytes;
							/* Disable button touch  while flash programming in progress */
							flashMemorytask.publishmessage(DISABLE_BUTTONS,"","");
							ret = programFlash(bytesToFlash);
							flashMemorytask.publishmessage(ENABLE_BUTTONS,"","");
							if(ret == true)
							{
								publishProgress("Flash Programming Completed Successfully !","","");
							}
							else
							{
								flashMemorytask.cancel(true);
							}
						} else
						{
							publishProgress("Invalid LPC8N04 device status. Flash programming terminated !!!","","");
							flashMemorytask.cancel(true);
						}

						if(ret)
						{
							// Verify Image
							ret = false;
							ret = VerifyDownloadImage();
							if(!ret)
							{
								flashMemorytask.cancel(true);
							}else
							{
								ret = false;
								//Send Download complete
								ret = DownloadComplete();
								if(ret)
								{
									ret = readIsLPCDeviceStatusApp(); // Confirm LPC8N04 is in Application mode
								}
								if(!ret)
								{
									flashMemorytask.cancel(true);
								}
							}
							if(ret)
							{
								success_flag = 1;
								flashMemorytask.cancel(true);
							}

						}
					}
				}
			}catch (Exception e) {
				e.printStackTrace();
			}
			return null;
		}

		@Override
		protected void onPreExecute() {

		}

		@Override
		protected void onProgressUpdate(String... progress) {   // Parameters < DebugMessage, progressvalue,barcolor>
			if(progress[0] == DISABLE_BUTTONS)
			{
				OtaUpdateFragment.disableButtons();

			} else if(progress[0] == ENABLE_BUTTONS)
			{
				OtaUpdateFragment.enableButtons();
			}else
			{
				if(progress[0] != "")
				{
					OtaUpdateFragment.setOtaTextBox(progress[0]); // Set Debug message on OTA Text View
				}
				if(progress[1] != "")
				{
					OtaUpdateFragment.setProgressPercentage(Integer.parseInt(progress[1])); // Set Percentage Value
					OtaUpdateFragment.setProgressBar(Integer.parseInt(progress[1])); // Set progress value
				}
				if(progress[2] != "")
				{
					OtaUpdateFragment.setProgressBarColor(Integer.parseInt(progress[2])); // Set Progress bar color
				}
			}

		}

		@Override
		protected void onPostExecute(Void bla) {
			if(success_flag == 1) // Flash Programming is successful
			{
				flashMemorytask.cancel(true);
			}else if (!exit) {
				OtaUpdateFragment.enableButtons();
				OtaUpdateFragment.setProgressBarColor(Integer.parseInt(PROGRESSBARFAIL)); // Set Progress bar color
				OtaUpdateFragment.setOtaTextBox("Error ! Flash Programming terminated !!!");
				flashMemorytask.cancel(true);
			}

		}

		public void publishmessage(String s1,String s2, String s3)
		{
			publishProgress(s1,s2,s3);
		}

	}

	/***
	 * Helper method to adjusts the number of bytes to be sent during the last flashing sector
	 * 
	 * @param num to round up
	 * @return number roundep up
	 */
	int roundUp(int num) {
		if(num <= 256) {
			return 256;
		} else if(num > 256 && num <= 512) {
			return 512;
		} else if(num > 512 && num <= 1024) {
			return 1024;
		} else {
			return 4096;
		}
	}


	/**
	 * Appends the CRC32 to the message transmitted during the SRAM SpeedTest.
	 *
	 * @param bytes
	 *            The content to be transmitted to the NTAGI2C board
	 *
	 * @return Byte Array with the CRC32s embedded in it
	 */
	private byte[] appendCRC32(byte[] bytes) {
		byte[] temp = new byte[bytes.length - LAST_FOUR_BYTES];
		System.arraycopy(bytes, 0, temp, 0, temp.length);
		byte[] crc = CRC32Calculator.CRC32(temp);
		System.arraycopy(crc, 0, bytes, bytes.length - crc.length, crc.length);
		return bytes;
	}

	/**
	 * Checks if the CRC32 value has been appended in the message.
	 *
	 * @param bytes
	 *            The whole message received from the board
	 *
	 * @return boolean that indicates the presence of the CRC32
	 */
	private boolean isCRC32Appended(byte[] bytes) {
		for (int i = bytes.length - LAST_FOUR_BYTES; i < bytes.length; i++) {
			if (bytes[i] != 0x00)
				return true;
		}
		return false;
	}

	/**
	 * Checks the received CRC32 value in the message received from the NTAG I2C
	 * board during the SRAM SpeedTest.
	 *
	 * @param bytes
	 *            The whole message received from the board
	 *
	 * @return boolean with the result of the comparison between the CRC32
	 *         received and the CRC32 calculated
	 */
	private boolean isValidCRC32(byte[] bytes) {
		byte[] receivedCRC = { bytes[bytes.length - LAST_FOUR_BYTES],
				bytes[bytes.length - 3],
				bytes[bytes.length - 2],
				bytes[bytes.length - 1] };

		byte[] temp = new byte[bytes.length - LAST_FOUR_BYTES];
		System.arraycopy(bytes, 0, temp, 0, bytes.length - LAST_FOUR_BYTES);
		byte[] calculatedCRC = CRC32Calculator.CRC32(temp);
		return Arrays.equals(receivedCRC, calculatedCRC);
	}


	/**
	 * Write empty ndef task.
	 */
	private class WriteEmptyNdefTask extends AsyncTask<Void, Void, Void> {

		@SuppressWarnings("unused")
		private Boolean exit = false;

		@Override
		protected Void doInBackground(Void... params) {
			try {
				reader.writeEmptyNdef();
			} catch (Exception e) {
				e.printStackTrace();
				cancel(true);
				return null;
			}
			return null;
		}
	}

	/**
	 * Write empty ndef finish.
	 */
	public void WriteEmptyNdefFinish() {
		if (emptyNdeftask != null && !emptyNdeftask.isCancelled()) {
			emptyNdeftask.exit = true;
			try {
				emptyNdeftask.get();
			} catch (Exception e) {
				e.printStackTrace();
			}
			emptyNdeftask = null;
		}
	}

	/**
	 * Write default ndef task.
	 */
	private class WriteDefaultNdefTask extends AsyncTask<Void, Void, Void> {

		@SuppressWarnings("unused")
		private Boolean exit = false;

		@Override
		protected Void doInBackground(Void... params) {
			try {
				reader.writeDefaultNdef();
			} catch (Exception e) {
				e.printStackTrace();
				cancel(true);
				return null;
			}
			return null;
		}
	}

	/**
	 * Write default ndef finish.
	 */
	public void WriteDefaultNdefFinish() {
		if (defaultNdeftask != null && !defaultNdeftask.isCancelled()) {
			defaultNdeftask.exit = true;
			try {
				defaultNdeftask.get();
			} catch (Exception e) {
				e.printStackTrace();
			}
			defaultNdeftask = null;
		}
	}

	/**
	 * Creates a NDEF Text Message.
	 * 
	 * @param text
	 *            Text to write
	 * @return NDEF Message
	 * @throws UnsupportedEncodingException
	 */
	private NdefMessage createNdefTextMessage(String text)
			throws UnsupportedEncodingException {
		if(text.length() == 0) {
			return null;
		}
		String lang = "en";
		byte[] textBytes = text.getBytes();
		byte[] langBytes = lang.getBytes("US-ASCII");
		int langLength = langBytes.length;
		int textLength = textBytes.length;
		byte[] payload = new byte[1 + langLength + textLength];
		payload[0] = (byte) langLength;
		System.arraycopy(langBytes, 0, payload, 1, langLength);
		System.arraycopy(textBytes, 0, payload, 1 + langLength, textLength);
		NdefRecord record = new NdefRecord(NdefRecord.TNF_WELL_KNOWN,
				NdefRecord.RTD_TEXT, new byte[0], payload);
		NdefRecord[] records = { record };
		NdefMessage message = new NdefMessage(records);
		return message;
	}

	/**
	 * Creates a NDEF Text Record.
	 *
	 * @param text
	 *            Text to write
	 * @return NDEF Record
	 * @throws UnsupportedEncodingException
	 */
	private NdefRecord createNdefTextRecord(String text)
			throws UnsupportedEncodingException {
		if(text.length() == 0) {
			return null;
		}
		String lang = "en";
		byte[] textBytes = text.getBytes();
		byte[] langBytes = lang.getBytes("US-ASCII");
		int langLength = langBytes.length;
		int textLength = textBytes.length;
		byte[] payload = new byte[1 + langLength + textLength];
		payload[0] = (byte) langLength;
		System.arraycopy(langBytes, 0, payload, 1, langLength);
		System.arraycopy(textBytes, 0, payload, 1 + langLength, textLength);
		NdefRecord record = new NdefRecord(NdefRecord.TNF_WELL_KNOWN,
				NdefRecord.RTD_TEXT, new byte[0], payload);
		return record;
	}


	/**
	 * Creates a NDEF Uri Message.
	 * 
	 * @param uri
	 *            Uri to write
	 * @return NDEF Message
	 */
	private NdefMessage createNdefUriMessage(String uri) {
		if(uri.length() == 0
	    || uri.endsWith("//")
		|| uri.endsWith(":")
		|| uri.endsWith(".")) {
			return null;
		}
		NdefRecord record = NdefRecord.createUri(uri);
		NdefRecord[] records = { record };
		NdefMessage message = new NdefMessage(records);
		return message;
	}

	/**
	 * Creates a Bluetooth Secure Simple Pairing Message.
	 * 
	 * @param mac
	 *            Bluetooth MAC Address to Write
	 * @return NDEF Message
	 */
	private NdefMessage createNdefBSSPMessage() {
		byte[] payloadHs = { 0x12, (byte) 0xD1, 0x02, 0x04, 0x61, 0x63, 0x01, 0x01, 0x30, 0x00 };
		NdefRecord recordHs = new NdefRecord(NdefRecord.TNF_WELL_KNOWN,
				NdefRecord.RTD_HANDOVER_SELECT, new byte[0], payloadHs);

		byte[] deviceMAC = hexStringToByteArray(NdefFragment.getBtMac().replaceAll(":", ""));
		byte[] deviceName = NdefFragment.getBtName().getBytes();
		byte[] deviceClass = hexStringToByteArray(NdefFragment.getBtClass().replaceAll(":", ""));
		
		if(deviceMAC.length != 6
		|| deviceName.length == 0
		|| deviceClass.length != 3) {
			return null;
		}
		byte[] payloadBt = new byte[deviceMAC.length + deviceName.length
				 + deviceClass.length + 2 + 4];

		// Payload Size
		payloadBt[0] = (byte) payloadBt.length;

		System.arraycopy(deviceMAC, 0, payloadBt, 2, deviceMAC.length);
		payloadBt[8] = (byte) (deviceName.length + 1);
		payloadBt[9] = 0x09; // Device Name identifier
		System.arraycopy(deviceName, 0, payloadBt, 10, deviceName.length);
		payloadBt[8 + deviceName.length + 2] = (byte) (deviceClass.length + 1);
		payloadBt[8 + deviceName.length + 3] = 0x0D; // Service Name identifier
		System.arraycopy(deviceClass, 0, payloadBt, 8 + deviceName.length + 4,
				deviceClass.length);
		NdefRecord recordBt = new NdefRecord(NdefRecord.TNF_MIME_MEDIA,
				"application/vnd.bluetooth.ep.oob".getBytes(), new byte[0],
				payloadBt);
		NdefRecord[] records = { recordHs, recordBt };
		NdefMessage message = new NdefMessage(records);
		return message;
	}
	
	/**
	 * Creates a NDEF SmartPoster Message.
	 * 
	 * @param uri
	 *            Uri to write
	 * @param title
	 *            Text to write
	 * @return NDEF Message
	 * @throws UnsupportedEncodingException 
	 */
	private NdefMessage createNdefSpMessage(String title, String uri) throws UnsupportedEncodingException {
		if(title.length() == 0)
			return null;
		
		if(uri.length() == 0
		|| uri.endsWith("//")
		|| uri.endsWith(":")
		|| uri.endsWith(".")) {
			return null;
		}
		
		String lang = "en";
		byte[] textBytes = title.getBytes();
		byte[] langBytes = lang.getBytes("US-ASCII");
		int langLength = langBytes.length;
		int textLength = textBytes.length;
		byte[] payload = new byte[1 + langLength + textLength];
		payload[0] = (byte) langLength;
		System.arraycopy(langBytes, 0, payload, 1, langLength);
		System.arraycopy(textBytes, 0, payload, 1 + langLength, textLength);

		NdefRecord recordTitle = new NdefRecord(NdefRecord.TNF_WELL_KNOWN,
				NdefRecord.RTD_TEXT, new byte[0], payload);
		NdefRecord recordLink = NdefRecord.createUri(uri);

		NdefRecord[] records = new NdefRecord[] { recordTitle, recordLink };
		NdefMessage messageSp = new NdefMessage(records);
		
		byte[] bytes = messageSp.toByteArray();
		NdefRecord recordSp = new NdefRecord((short) 0x01,
				NdefRecord.RTD_SMART_POSTER, null, bytes);

		return new NdefMessage( new NdefRecord[] { recordSp });
	}

	public static byte[] hexStringToByteArray(String s) {
		int len = s.length();
		byte[] data = new byte[len / 2];
		for (int i = 0, j = (len - 2) / 2; i < len; i += 2, j--) {
			data[j] = (byte) ((Character.digit(s.charAt(i), 16) << 4) + Character
					.digit(s.charAt(i + 1), 16));
		}
		return data;
	}

	/**
	 * Closes the Reader, Writes the NDEF Message via Android function and
	 * reconnects the Reader.
	 * 
	 * @return NDEF Writing time
	 * @throws IOException
	 * @throws FormatException
	 * @throws CommandNotSupportedException 
	 */
	private long NDEFWrite(NdefMessage msg) throws IOException, FormatException, CommandNotSupportedException {
		// Time statistics to return
		long timeNdefWrite = 0;
		long RegTimeOutStart = System.currentTimeMillis();
		
		// Write the NDEF using NfcA commands to avoid problems when dealing with protected tags
		// Calling reader close / connect resets the authenticated status
		reader.writeNDEF(msg, null);

		timeNdefWrite = System.currentTimeMillis() - RegTimeOutStart;
		
		// Return the calculated time value
		return timeNdefWrite;
	}





	/**
	 * Rounds the voltage to one single decimal.
	 */
	public double round(double value) {
		return Math.rint(value * 10) / 10;
	}

	/**
	 * Helper function to show messages on the screen.
	 * 
	 * @param temp Message
	 */
	protected void showResultDialogMessage(String temp) {
		new AlertDialog.Builder(main).setMessage(temp)
				.setPositiveButton("OK", new DialogInterface.OnClickListener() {
					@Override
					public void onClick(DialogInterface dialog, int which) {
					}
				}).show();
	}

	/**
	 * Creates a NDEF Message
	 * 
	 * @param text
	 *            Text to write
	 * @return NDEF Message
	 * @throws UnsupportedEncodingException
	 */
	private NdefMessage createNdefMessage(String text)
			throws UnsupportedEncodingException {
		String lang = "en";
		byte[] textBytes = text.getBytes();
		byte[] langBytes = lang.getBytes("US-ASCII");
		int langLength = langBytes.length;
		int textLength = textBytes.length;
		byte[] payload = new byte[1 + langLength + textLength];
		payload[0] = (byte) langLength;
		System.arraycopy(langBytes, 0, payload, 1, langLength);
		System.arraycopy(textBytes, 0, payload, 1 + langLength, textLength);

		NdefRecord record = new NdefRecord(NdefRecord.TNF_WELL_KNOWN,
				NdefRecord.RTD_TEXT, new byte[0], payload);
		NdefRecord[] records = { record };
		NdefMessage message = new NdefMessage(records);
		return message;
	}

	protected byte[] concat(byte[] one, byte[] two) {
		if (one == null) {
			one = new byte[0];
		}
		if (two == null) {
			two = new byte[0];
		}
		byte[] combined = new byte[one.length + two.length];
		System.arraycopy(one, 0, combined, 0, one.length);
		System.arraycopy(two, 0, combined, one.length, two.length);
		return combined;
	}

	@Override
	public void onWriteEEPROM(final int bytes) {
		main.runOnUiThread(new Runnable() {
			@Override
			public void run() {
				//SpeedTestFragment.setDatarateCallback(String.valueOf(bytes) + " Bytes written");
			}
		});
	}
	

}
