/*
 * File:		servos.c
 * Purpose:		Servo Routines for Tower Mechatronic board 
 * 
 * License:     All software covered by license agreement in -
 *              docs/Freescale_Software_License.pdf
 */

#include "common.h"
#include "pit.h"
#include "pwm.h"


unsigned char Servo[8];

void InitServos( void )
{
	Servo[0] = 128;
	Servo[1] = 128;
	Servo[2] = 128;
	Servo[3] = 128;
	Servo[4] = 128;
	Servo[5] = 128;
	Servo[6] = 128;
	Servo[7] = 128;

	/* Init GPT as PWM for servo control */
	GPTInitPWM();
	GPTSetPWM( 0, Servo[0] );
	GPTSetPWM( 1, Servo[1] );
	GPTSetPWM( 2, Servo[2] );
	GPTSetPWM( 3, Servo[3] );
	
	PWMInitServo();
	PWMSetServo( 4, Servo[4] );
	PWMSetServo( 5, Servo[5] );
	PWMSetServo( 6, Servo[6] );
	PWMSetServo( 7, Servo[7] );
		
}


// rate = 1 to 255  ( 255 is the fastest )
// 255 still waits 20ms
// 1 = 1 count per 100ms
//
// rate < 80 -> step=1, delay=100ms-rate
// rate >= 80 -> delay=20ms, step=(rate-80)/32+1
void MoveServo( unsigned char channel, uint8 pos, unsigned char rate )
{
	unsigned char timeout, step, delay;

	if( channel > 7 ) return;
	
	if( rate < 80 )
	{
		step = 1;
		delay = 100-rate;
	}
	else
	{
		step = (rate-80)/32+1;
		delay = 20;
	}
	
	if( rate == 255 )
	{
		step = 200; // immediate
	}
	
	for( timeout=0; timeout != 0xff; ++timeout )
	{
		if( Servo[channel] > pos )
		{
			if( (Servo[channel] - pos) < step )
				step = Servo[channel] - pos;
			Servo[channel]-=step;
		}
		else if( Servo[channel] < pos )
		{
			if( (pos - Servo[channel]) < step )
				step = pos - Servo[channel];
			Servo[channel]+=step;
		}
		else
			break;

		PITDelay(delay, 0);
	
		if( channel < 4 )
			GPTSetPWM( channel, Servo[channel] );
		else
			PWMSetServo( channel, Servo[channel] );
	}
}

// rate = 1 to 255  ( 255 is the fastest )
// 255 still waits 20ms
// 1 = 1 count per 100ms
//
// rate < 80 -> step=1, delay=100ms-rate
// rate >= 80 -> delay=20ms, step=(rate-80)/32+1
void MoveServos( 
					uint8 pos1, // J31
					uint8 pos2, // J33
					uint8 pos3, // J35
					uint8 pos4, // J37
					uint8 pos5, 
					uint8 pos6, 
					uint8 pos7, 
					uint8 pos8, 
					unsigned char rate )
{
	unsigned char timeout, step, delay, i, done;
	unsigned char lpos[8];
	
	if( rate < 80 )
	{
		step = 1;
		delay = 100-rate;
	}
	else
	{
		step = (rate-80)/32+1;
		delay = 20;
	}
	
	if( rate == 255 )
	{
		step = 200; // immediate
	}
	
	lpos[0] = pos1; 
	lpos[1] = pos2; 
	lpos[2] = pos3; 
	lpos[3] = pos4; 
	lpos[4] = pos5; 
	lpos[5] = pos6; 
	lpos[6] = pos7; 
	lpos[7] = pos8; 
	
	for( timeout=0; timeout != 0xff; ++timeout )
	{
		done = 0;
		for( i=0; i<8; ++i )
		{
			if( Servo[i] > lpos[i] )
			{
				if( (Servo[i] - lpos[i]) < step )
					step = Servo[i] - lpos[i];
				Servo[i]-=step;
			}
			else if( Servo[i] < lpos[i] )
			{
				if( (lpos[i] - Servo[i]) < step )
					step = lpos[i] - Servo[i];
				Servo[i]+=step;
			}
			else
			{
				++done;
				continue;
			}
		} // for i

		PITDelay(delay, 0);
	
		GPTSetPWM(   0, Servo[0] );
		GPTSetPWM(   1, Servo[1] );
		GPTSetPWM(   2, Servo[2] );
		GPTSetPWM(   3, Servo[3] );
		PWMSetServo( 4, Servo[4] );
		PWMSetServo( 5, Servo[5] );
		PWMSetServo( 6, Servo[6] );
		PWMSetServo( 7, Servo[7] );
		
		if( done > 7 ) break;
	}
}