/**HEADER********************************************************************
*
* Copyright (c) 2012 Freescale Semiconductor;
* All Rights Reserved
*
***************************************************************************
*
* THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR
* IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
* IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
* SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
* STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
* IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
* THE POSSIBILITY OF SUCH DAMAGE.
*
**************************************************************************
*
* $FileName: main.c$
* $Version : 1.0.0.0$
* $Date    : Nov-19-2012$
*
* Comments:
*  TWR-MCF51JG Demo
*  This simple application demonstrates entry/exit from low power states.  
*  See the lowpower demo in the \mqx\examples\lowpower\ for more details.  
*  Initially, the orange LED blinks at a slow rate.  The MCU prints a short 
*  message to the terminal once a second.  Press the SW2 button.  The demo 
*  application goes into timeout mode.  The led blinks faster.  It will count
*  down from 20 seconds.  Pressing SW2 again will set the timeout back to 20 
*  seconds.   Once the timeout has elapsed, it will put the MCU into very low 
*  power stop (VLPS) mode to conserve power.  After 10 seconds the Real Time 
*  Clock (RTC) will wake up the MCU and start the demo over.  Also, you can 
*  wake up the MCU by pressing the SW2 button.    
*
*END************************************************************************/

#include <mqx.h>
#include <bsp.h>
#include <lwevent.h>

#if ! BSPCFG_ENABLE_IO_SUBSYSTEM
#error This application requires BSPCFG_ENABLE_IO_SUBSYSTEM defined non-zero in user_config.h. Please recompile BSP with this option.
#endif


#ifndef BSP_DEFAULT_IO_CHANNEL_DEFINED
#error This application requires BSP_DEFAULT_IO_CHANNEL to be not NULL. Please set corresponding BSPCFG_ENABLE_TTYx to non-zero in user_config.h and recompile BSP with this option.
#endif


#if ! MQX_ENABLE_LOW_POWER
#error This application requires MQX_ENABLE_LOW_POWER to be defined non-zero in user_config.h. Please recompile BSP with this option.
#endif


#if ! BSPCFG_ENABLE_RTCDEV
#error This application requires BSPCFG_ENABLE_RTCDEV to be defined non-zero in user_config.h. Please recompile BSP with this option.
#endif

/* idle time until timeout */
#define JGDEMO_TIMEOUT_START  21
/* time to trigger faster led blinking to warn of impending timeout */
#define JGDEMO_TIMEOUT_WARN    5
/* Normal LED blink rate */
#define JGDEMO_BLINKRATE_NORMAL  1
/* Fast LED blink rate */
#define JGDEMO_BLINKRATE_FAST    4
/* Very Fast LED blink rate */
#define JGDEMO_BLINKRATE_VERY_FAST  8

typedef struct {
   uint_32              BlinkRate;
   uint_32              TimeOut;
   boolean              TimeOutDemo;
} JG_DEMO_PARAMS;

/* Task IDs */
enum task_ids {
    MAIN_TASK = 1,
    LED_TASK,
    TIMEOUT_TASK
};

/* Task prototypes */
void main_task(uint_32);
void led_task(uint_32);
void timeout_task(uint_32);

const TASK_TEMPLATE_STRUCT  MQX_template_list[] =
{
    {
        /* Task Index       */  MAIN_TASK,
        /* Function         */  main_task,
        /* Stack Size       */  1000,
        /* Priority Level   */  10,
        /* Name             */  "Main Task",
        /* Attributes       */  MQX_AUTO_START_TASK,
        /* Creation Params  */  0,
        /* Time Slice       */  0,
    },
    {
        /* Task Index       */  LED_TASK,
        /* Function         */  led_task,
        /* Stack Size       */  1000,
        /* Priority Level   */  11,
        /* Name             */  "LED Task",
        /* Attributes       */  0,
        /* Creation Params  */  0,
        /* Time Slice       */  0,
    },
    {
        /* Task Index       */  TIMEOUT_TASK,
        /* Function         */  timeout_task,
        /* Stack Size       */  1000,
        /* Priority Level   */  12,
        /* Name             */  "Timeout Task",
        /* Attributes       */  0,
        /* Creation Params  */  0,
        /* Time Slice       */  0,
    },
    { 0 }
};

JG_DEMO_PARAMS JGDemo;
LWGPIO_STRUCT  led1;
LWGPIO_STRUCT led2;

/*FUNCTION*-----------------------------------------------------
*
* Task Name    : my_rtc_isr
* Comments     : RTC interrupt isr used to catch RTC alarm
*                and escape from "interrupts only" low power mode.
*
*END*-----------------------------------------------------------*/

static void my_rtc_isr
    (
        pointer rtc_registers_ptr
    )
{
    uint_32 state = _rtc_get_status ();

    _rtc_clear_requests (state);

}


/*FUNCTION*-----------------------------------------------------
*
* Task Name    : install_rtc_interrupt
* Comments     : Installs RTC interrupt and enables alarm.
*
*END*-----------------------------------------------------------*/

static void install_rtc_interrupt
    (
    )
{
    if (MQX_OK != _rtc_int_install ((pointer)my_rtc_isr))
    {
        printf ("\nError installing RTC interrupt!\n");
        _task_block();
    }

    _rtc_clear_requests (RTC_RTCISR_ALM);

    if (0 == _rtc_int_enable (TRUE, RTC_RTCIENR_ALM))
    {
        printf ("\nError enabling RTC interrupt!\n");
        _task_block();
    }
}


/*FUNCTION*-----------------------------------------------------
*
* Task Name    : set_rtc_alarm
* Comments     : Setup RTC alarm to given number of seconds from now.
*
*END*-----------------------------------------------------------*/

static void set_rtc_alarm
    (
        uint_32 seconds
    )
{
    DATE_STRUCT     time_rtc;
    TIME_STRUCT     time_mqx;

    _rtc_get_time_mqxd (&time_rtc);
    _time_from_date (&time_rtc, &time_mqx);
    time_mqx.SECONDS += seconds;
    _time_to_date (&time_mqx, &time_rtc);
    _rtc_set_alarm_mqxd (&time_rtc);
    printf ("\nRTC alarm set to +%d seconds.\n", seconds);
}


/*FUNCTION*-----------------------------------------------------
*
* Task Name    : button_isr
* Comments     : SW interrupt callback used to catch SW press
*                and trigger SW_EVENT
*
*END*-----------------------------------------------------------*/

static void button_isr
    (
         pointer pin
    )
{

    /* turn on LED1 - drive low to turn on*/
    lwgpio_set_value(&led1, LWGPIO_VALUE_LOW);
    /* start timeout demo if not already started */
    JGDemo.TimeOutDemo = TRUE;
    /* re-set timeout to initial value */
    JGDemo.TimeOut = JGDEMO_TIMEOUT_START;
    /* set LED2 blink rate to fast */
    JGDemo.BlinkRate = JGDEMO_BLINKRATE_FAST;

    lwgpio_int_clear_flag (pin);

}


/*FUNCTION*-----------------------------------------------------
*
* Task Name    : button_led_init
* Comments     : Setup the button to trigger interrupt on each button press.
*
*END*-----------------------------------------------------------*/

static void button_led_init
    (
        void
    )
{
    static LWGPIO_STRUCT                   sw;
    /* Set the pin to input */
    if (!lwgpio_init(&sw, BSP_SW2, LWGPIO_DIR_INPUT, LWGPIO_VALUE_NOCHANGE))
    {
        printf("\nSW initialization failed.\n");
        _task_block();
    }

    /* Set functionality to GPIO mode */
    lwgpio_set_functionality(&sw, BSP_SW2_MUX_GPIO);

    /* Enable pull up */
    lwgpio_set_attribute(&sw, LWGPIO_ATTR_PULL_UP, LWGPIO_AVAL_ENABLE);

    /* Setup the pin interrupt mode */
    if (!lwgpio_int_init(&sw, LWGPIO_INT_MODE_FALLING))
    {
        printf("Initializing SW for interrupt failed.\n");
        _task_block();
    }

    /* Install gpio interrupt service routine */
    _int_install_isr(lwgpio_int_get_vector(&sw), button_isr, (void *) &sw);

    /* Set interrupt priority and enable interrupt source in the interrupt controller */
    _bsp_int_init(lwgpio_int_get_vector(&sw), 3, 0, TRUE);

    /* Enable interrupt for pin */
    lwgpio_int_enable(&sw, TRUE);

    /* Initialize LED pin for output */
    if (!lwgpio_init(&led1, BSP_LED1, LWGPIO_DIR_OUTPUT, LWGPIO_VALUE_HIGH))
    {
        printf("\nLED1 initialization failed.\n");
        _task_block();
    }
    /* Set LED pin to GPIO functionality */
    lwgpio_set_functionality(&led1, BSP_LED1_MUX_GPIO);
}



/*TASK*-----------------------------------------------------
*
* Task Name    : main_task
* Comments     : Low power modes switching.
*
*END*-----------------------------------------------------*/

void main_task
    (
        uint_32 initial_data
    )
{
    DATE_STRUCT     time_rtc;
    TIME_STRUCT     time_mqx;

    /* Initialize switches */
    button_led_init();

    /* initialize time */
    time_rtc.YEAR     = 2012;
    time_rtc.MONTH    = 11;
    time_rtc.DAY      = 9;
    time_rtc.HOUR     = 10;
    time_rtc.MINUTE   = 8;
    time_rtc.SECOND   = 0;
    time_rtc.MILLISEC = 0;

    _time_from_date (&time_rtc, &time_mqx);

    _time_set( &time_mqx);
    if( _rtc_sync_with_mqx(FALSE) != MQX_OK )
    {
        printf("\nError synchronize time!\n");
        _task_block();
    }

    /* Install interrupt for RTC alarm */
    install_rtc_interrupt();

    printf("\nTWR-MCF51JG Demo\n");

    /* Turn time out off until button press */
    JGDemo.TimeOutDemo = FALSE;

    /* Blink LED at normal rate */
    JGDemo.BlinkRate = JGDEMO_BLINKRATE_NORMAL;

    /* start blinking LED */
    _task_create(0, LED_TASK, 0);

    while (1)
    {
        /* If button press set time out demo to start */
        if(JGDemo.TimeOutDemo == TRUE)
        {
        	/* Timeout demo has been started by a button press */
        	
            if(_task_get_id_from_name("Timeout Task") == MQX_NULL_TASK_ID)
            {
              /* If the timeout task hasn't been created yet, create it now */
                _task_create(0,TIMEOUT_TASK,0);
            }

            if(JGDemo.TimeOut <= JGDEMO_TIMEOUT_WARN)
            {
            	/* Time is in the warning period */
            	
                printf("Going to Stop Soon...Blinking LED Very Fast\n");
                JGDemo.BlinkRate = JGDEMO_BLINKRATE_VERY_FAST;
            }
            if(JGDemo.TimeOut <= 0)
            {
            	/* Time ran out */
            	
                printf("Stopping Now...\n");

                printf(
                "Info: STOP operation mode is mapped to VLPS power mode by default.\n"
                "      Core and most peripherals are inactive in this mode, reacting only to\n"
                "      specified wake up events. The events can be changed in BSP (init_lpm.c).\n"
                "      Serial line is turned off in this mode. The core will wake up from\n"
                "      RTC interrupt or SW2 button press.\n");

                /* Wake up in 10 seconds */
                set_rtc_alarm(10);

                /* Turn off LEDs - setting value to high will turn them off */
                lwgpio_set_value(&led1, LWGPIO_VALUE_HIGH);
                lwgpio_set_value(&led2, LWGPIO_VALUE_HIGH);

                /* Change the operation mode */
                _lpt_mask_int(0);
                _lpm_set_operation_mode (LPM_OPERATION_MODE_STOP);
                _lpt_unmask_int(0);
                /**************************************************************************************************/
                /* SCI HW MODULE IS DISABLED AT THIS POINT - SERIAL DRIVER MUST NOT BE USED UNTIL MODE IS CHANGED */
                /**************************************************************************************************/

                /* Return to RUN mode */
                _lpm_set_operation_mode (LPM_OPERATION_MODE_RUN);

                printf("\nCore is awake. \n");

                JGDemo.TimeOutDemo = FALSE;
                JGDemo.BlinkRate = JGDEMO_BLINKRATE_NORMAL;

            } 
        }
        else
        {   /* Timeout demo has not been started by button press */
            /* Print this message */
            printf("Hi from MCF51JG256...\n");            
        }

        /* loop every second */
        _time_delay(1000);

    } /*  while (1) */
}

/*TASK*-----------------------------------------------------
*
* Task Name    : led_task
* Comments     : LEDs setup and blinking
*
*END*-----------------------------------------------------*/

void led_task
    (
        uint_32 initial_data
    )
{

    /* Initialize LED pin for output */
    if (!lwgpio_init(&led2, BSP_LED2, LWGPIO_DIR_OUTPUT, LWGPIO_VALUE_HIGH))
    {
        printf("\nLED2 initialization failed.\n");
        _task_block();
    }
    /* Set LED pin to GPIO functionality */
    lwgpio_set_functionality(&led2, BSP_LED2_MUX_GPIO);

    while(1)
    {
        /* Toggle led */
        lwgpio_toggle_value(&led2);

        /* Delay */
        _time_delay (1000/JGDemo.BlinkRate);

    }
}

/*TASK*-----------------------------------------------------
*
* Task Name    : timeout_task
* Comments     : if timeout demo started, then this task will decrement the timeout value
*
*END*-----------------------------------------------------*/

void timeout_task
    (
        uint_32 initial_data
    )
{
    while(1)
    {
       if(JGDemo.TimeOutDemo == TRUE )
       {
           if(JGDemo.TimeOut >= 1)
           {
               /* decrement timeout */
               JGDemo.TimeOut--;
               printf("Timing out in %d seconds...\n",JGDemo.TimeOut);
           }
       }
       /* loop every one second */
        _time_delay (1000);

    }
}

/* EOF */