/*
*                    Copyright (c), NXP Semiconductors
*
*                       (C) NXP Semiconductors 2018
*
*         All rights are reserved. Reproduction in whole or in part is
*        prohibited without the written consent of the copyright owner.
*    NXP reserves the right to make changes without notice at any time.
*   NXP makes no warranty, expressed, implied or statutory, including but
*   not limited to any implied warranty of merchantability or fitness for any
*  particular purpose, or that the use will not infringe any third party patent,
*   copyright or trademark. NXP must not be liable for any loss or damage
*                            arising from its use.
*/


#ifndef NDEF_REC_URI_H_
#define NDEF_REC_URI_H_

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

/* ********************************************************************************************
* Includes
* ******************************************************************************************* */
#include <stdint.h>
#include "NDEF_Status.h"
#include "NDEF_Record.h"


/** \defgroup uri Uri record
*
* \brief
* @{
*/

/* ********************************************************************************************
* Types/Structure Declarations
* ******************************************************************************************* */
/**
* \brief Uri record structure
*/
typedef struct
{
	RecordInfo_t INT_sRecordInfo;			/**< Internal record information. */
	NDEF_Record_Header_t sHeader;			/**< Record header structure. */
	uint8_t bIdentifierCode;				/**< Identifier code. */
	uint8_t * pUriField;					/**< Pointer to UriField. */

}NDEF_Rec_Uri_t;

/* ********************************************************************************************
* Function Prototypes
* ******************************************************************************************* */
/**
 * \brief  This function will allocate Text record on heap(including all parameters of variable length)  and return a handle by which the record can be referenced.
		  Note: This API function is intended to be used only with ASCII subset of UTF-8 encoding. In case the full UTF-8 character support  for Uri field is desired,  NDEF_Rec_Uri_CreateByArray should be used.
 *
 * Identifier code table
 * ---------------------
 * Input Value | Protocol
 * ---------- | ---------
 * 0x00 | N/A. No prepending is done, and the URI field contains the unabridged URI.
 * 0x01 | http://www.
 * 0x02 | https://www.
 * 0x03 | http://
 * 0x04 | https://
 * 0x05 | tel:
 * 0x06 | mailto:
 * 0x07 | ftp://anonymous:anonymous@
 * 0x08 | ftp://ftp.
 * 0x09 | ftps://
 * 0x0A | sftp://
 * 0x0B | smb://
 * 0x0C | nfs://
 * 0x0D | ftp://
 * 0x0E | dav://
 * 0x0F | news:
 * 0x10 | telnet://
 * 0x11 | imap:
 * 0x12 | rtsp://
 * 0x13 | urn:
 * 0x14 | pop:
 * 0x15 | sip:
 * 0x16 | sips:
 * 0x17 | tftp:
 * 0x18 | btspp://
 * 0x19 | btl2cap://
 * 0x1A | btgoep://
 * 0x1B | tcpobex://
 * 0x1C | irdaobex://
 * 0x1D | file://
 * 0x1E | urn:epc:id:
 * 0x1F | urn:epc:tag:
 * 0x20 | urn:epc:pat:
 * 0x21 | urn:epc:raw:
 * 0x22 | urn:epc:
 * 0x23 | urn:nfc:
 *
 * @param[in] bIdCode   			Identifier code. 
 * @param[in] pUriField   			UriField string - NULL terminated.
 *
 * @return Record handle
 * @retval In case of error returns NULL
 */
extern NDEF_Rec_Uri_t * NDEF_Rec_Uri_CreateByStr(uint8_t  bIdCode, const char * pUriField);


/**
* \brief This function will allocate Uri record on heap(including all parameters of variable length)  and return a handle by which the record can be referenced.
*
* Identifier code table
* ---------------------
* Input Value  | Protocol
* ---------- | ---------
* 0x00 | N/A. No prepending is done, and the URI field contains the unabridged URI.
* 0x01 | http://www.
* 0x02 | https://www.
* 0x03 | http://
* 0x04 | https://
* 0x05 | tel:
* 0x06 | mailto:
* 0x07 | ftp://anonymous:anonymous@
* 0x08 | ftp://ftp.
* 0x09 | ftps://
* 0x0A | sftp://
* 0x0B | smb://
* 0x0C | nfs://
* 0x0D | ftp://
* 0x0E | dav://
* 0x0F | news:
* 0x10 | telnet://
* 0x11 | imap:
* 0x12 | rtsp://
* 0x13 | urn:
* 0x14 | pop:
* 0x15 | sip:
* 0x16 | sips:
* 0x17 | tftp:
* 0x18 | btspp://
* 0x19 | btl2cap://
* 0x1A | btgoep://
* 0x1B | tcpobex://
* 0x1C | irdaobex://
* 0x1D | file://
* 0x1E | urn:epc:id:
* 0x1F | urn:epc:tag:
* 0x20 | urn:epc:pat:
* 0x21 | urn:epc:raw:
* 0x22 | urn:epc:
* 0x23 | urn:nfc:
*
* @param[in] bIdCode   				Identifier code.
* @param[in] pUriField			Pointer to UriField array.
* @param[in] dwLength			Byte length of UriField.
*
* @return Record handle
* @retval In case of error returns NULL
*/
extern NDEF_Rec_Uri_t * NDEF_Rec_Uri_CreateByArray(uint8_t bIdCode, const uint8_t * pUriField, uint32_t dwLength);


/**
 * \brief This function is used to get the unabriged Uri from a record specified by handle. The content of Uri field inside the record is UTF-8 encoded, but the actuall URIs are defined in 7bit ASCII subset of UTF-8. Therfore the codepoints above 127 should be encoded by technique  described in [RFC 3492]. This transformation is not implemented in the library - only ASCII encoding of the Uri field is supported by this function.
 *
 * @param[in] 		pRecord	   				Record handle.
 * @param[out] 		pUri   					Pointer Uri buffer - output will be null terminated.
 * @param[in] 		dwLength   				Size of Uri output buffer.
 *
 * @return Status of the API
 * @retval NDEF_STATUS_SUCCESS				Function returned successfully.
 * @retval NDEF_STATUS_ERROR_INVALID_PARAM	Null/Invalid function parameter.
 * @retval NDEF_STATUS_ERROR_BUFF_OVF		Output won't fit into buffer.
 */
extern Status_t NDEF_Rec_Uri_GetUri(const NDEF_Rec_Uri_t * pRecord, char * pUri, uint32_t dwLength);




/**
 * \brief  This function is used to return the length of unabriged Uri  of a record specified by handle.
 *
 * @param[in] pRecord	   		Record handle.
 *
 * @return Length of the Uri
 */
extern uint32_t NDEF_Rec_Uri_GetUriLen(const NDEF_Rec_Uri_t * pRecord);





/** @}
* end of uri Uri record
*/


#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* NDEF_REC_URI_H_ */
