/*
 * NDEF_Handover.h
 *
 *  Created on: 2. 10. 2017
 *      Author: nxf19239
 */

#ifndef NDEF_REC_HANDOVER_H_
#define NDEF_REC_HANDOVER_H_

#include <stdint.h>
#include "NDEF_Status.h"
#include "NDEF_Record.h"
#include "NDEF_Message.h"

/** \defgroup handover Handover
*
* \brief
* @{
*/

/**
* \brief Power state of alternative carrier
*/
typedef enum
{
	InactiveState,
	ActiveState,
	ActivatingState,
	UnknownState
}CarrierPowerState_t;


/**
* \brief Carrier data reference structure
*/
typedef struct
{
	void * pCarrierDataRecord;					/**< Pointer to carrier data record */
	uint8_t bCarrierDataReferenceLength;		/**< Length of CarrierDataReference. */
	uint8_t * pCarrierDataReferenceChar;	    /**< CarrierDataReference character. */
}CarrierDataReference_t;



/**
* \brief Auxiliary data reference structure
*/
typedef struct
{
	void * pAuxDataRecord;						/**< Pointer to auxiliary data record. */
	uint8_t bAuxiliaryDataReferenceLength;		/**< Length of AuxiliaryDataReference. */
	uint8_t * pAuxiliaryDataReferenceChar;		/**< AuxiliaryDataReference character. */
}AuxiliaryDataReference_t;


/**
* \brief Alternative carrier record structure
*/
typedef struct
{
	RecordInfo_t INT_sRecordInfo;																/**< Internal record information. */
	NDEF_Record_Header_t sHeader;																				/**< Record header structure. */
	CarrierPowerState_t 		eCps;															/**< Power state of the carrier. */
	CarrierDataReference_t 		sCarrierDataReference;											/**< Carrier data reference. */
	uint8_t 					bAuxDataReferenceCount;											/**< Auxiliary data reference count. */
	AuxiliaryDataReference_t *	pAuxiliaryDataReference;					  				    /**< Array of Auxiliary data references. */
}NDEF_Rec_AltCarrier_t;



/**
* \brief Handover select record structure
*/
typedef struct
{
	RecordInfo_t INT_sRecordInfo;																				/**< Internal record information. */
	NDEF_Record_Header_t sHeader;																				/**< Record header structure. */
	uint8_t bVersion;																							/**< Major version number. */																					/**< Minor version number. */
	NDEF_Rec_AltCarrier_t * pAlternativeCarrier;																/**< Array of pointers to alternative carrier records. */

}NDEF_Rec_HandoverSelect_t;



/**
 * \brief This function will allocate HandoverSelect record on heap  and return a handle by which the record can be referenced. Content of all parameters is stored internally on heap.
 *
 * @param[in] 	  bAltCarrierCount  Number of alternative carriers.
 *
 * @return Record handle
 * @retval Pointer to NDEF_Rec_HandoverSelect_t allocated on heap
 * @retval NULL in case of an error
 */

extern NDEF_Rec_HandoverSelect_t * NDEF_Rec_HandoverSelect_Create(uint8_t bAltCarrierCount);


/**
 * \brief This function is used to append Alternative carrier  record into HandoverSelect record specified by handle.
 *
 * @param[in] pRecord	   		HandoverSelect record handle.
 * @param[in] pAltCarrier    Alternative carrier handle .
 *
 * @return Status of the API
 * @retval TODO
 */
extern Status_t NDEF_Rec_HandoverSelect_AppendAltCarrier(NDEF_Rec_HandoverSelect_t * pRecord, NDEF_Rec_AltCarrier_t * pAltCarrier);


/**
 * \brief This function will allocate AlternativeCarrier record on heap  and return a handle by which the record can be referenced. Content of all parameters is stored internally on heap.
 *
 * @param[in] pCarrierRecord	   		Pointer to Carrier record structure.
 * @param[in] eCps						Carrier power state.	
 * @param[in] bAuxDataCount    			Auxiliary data count.
 *
 * @return Record handle
 * @retval Pointer to NDEF_Rec_AltCarrier_t allocated on heap
 * @retval NULL in case of an error
 */
extern NDEF_Rec_AltCarrier_t * NDEF_Rec_AltCarrier_Create(void * pCarrierRecord, CarrierPowerState_t eCps, uint8_t bAuxDataCount);


/**
 * \brief This function is used to append Auxiliary data  record into AlternativeCarrier record specified by handle.
 *
 * @param[in] pAltCarrier	   		AlternativeCarrier handle.
 * @param[in] pRecToAppend			Handle of record to append.
 * @param[in] pRefChar				Character of the data.
 * @param[in] bRefCharLen			Length of character.
 *
 * @return Status of the API
 * @retval TODO
 */
extern Status_t NDEF_Rec_AltCarrier_AppendAuxData(NDEF_Rec_AltCarrier_t * pAltCarrier, void * pRecToAppend, uint8_t * pRefChar, uint8_t bRefCharLen);


/** @}
* end of handover Handover
*/



#endif /* NDEF_REC_HANDOVER_H_ */
