/*
*                    Copyright (c), NXP Semiconductors
*
*                       (C) NXP Semiconductors 2018
*
*         All rights are reserved. Reproduction in whole or in part is
*        prohibited without the written consent of the copyright owner.
*    NXP reserves the right to make changes without notice at any time.
*   NXP makes no warranty, expressed, implied or statutory, including but
*   not limited to any implied warranty of merchantability or fitness for any
*  particular purpose, or that the use will not infringe any third party patent,
*   copyright or trademark. NXP must not be liable for any loss or damage
*                            arising from its use.
*/


#ifndef NDEF_REC_GENERICPAYLOAD_H_
#define NDEF_REC_GENERICPAYLOAD_H_

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

/* ********************************************************************************************
* Includes
* ******************************************************************************************* */
#include <stdint.h>
#include "NDEF_Status.h"
#include "NDEF_Config.h"
#include "NDEF_Record.h"

/** \defgroup  generic Generic Payload record
*
* \brief
* @{
*/


/* ********************************************************************************************
* Types/Structure Declarations
* ******************************************************************************************* */

/**
* \brief Generic payload record structure
*/
typedef struct
{
	RecordInfo_t INT_sRecordInfo;		/**< Internal record information. */
	NDEF_Record_Header_t sHeader;		/**< Record header structure. */
	uint8_t * pPayload;					/**< Payload pointer holder. */
} NDEF_Rec_GenericPayload_t;


/* ********************************************************************************************
* Function Prototypes
* ******************************************************************************************* */
/**
 * \brief This function will allocate Generic record on heap  and return a handle by which the record can be referenced. Content of all parameters is stored internally on heap.
 *
 * @param[in] 	  pPayload  Pointer to Payload.
 * @param[in]	  wLength	Length of Payload in bytes.
 *
 * @return Record handle
 * @retval In case of error returns NULL.
 */
extern NDEF_Rec_GenericPayload_t *  NDEF_Rec_Generic_Create(const uint8_t * pPayload, uint32_t  wLength);



/**
 * \brief This function is same as NDEF_Rec_Generic_Create with exception of memory allocation for Payload. In this case  the pointer to user buffer containing Payload is only assigned into the record structure.
 * NOTE: Data integrity of this record depends on application because the payload within this record exist only as reference to application buffer.
 *
 * @param[in] 	  pPayload  Pointer to Payload.
 * @param[in]	  wLength	Length of Payload.
 *
 * @return Record handle
 * @retval In case of error returns NULL.
 */
extern NDEF_Rec_GenericPayload_t * NDEF_Rec_Generic_CreateByPtr(const uint8_t * pPayload, uint32_t  dwLength);




/**
 * \brief This function is used to get the Payload of a record specified by handle.
 *
 * @param[in] 	  pRecord   Generic record handle.
 * @param[in] 	  wLength	Size of output buffer.
 * @param[out] 	  pPayload  Pointer to Payload output buffer.
 *
 * @return Status of the API
 * @retval NDEF_STATUS_SUCCESS				Function returned successfully.
 * @retval NDEF_STATUS_ERROR_INVALID_PARAM	Null/Invalid function parameter.
 * @retval NDEF_STATUS_ERROR_BUFF_OVF		Output won't fit into buffer.
 */
extern Status_t NDEF_Rec_Generic_GetPayload(const NDEF_Rec_GenericPayload_t * pRecord, uint8_t * pPayload, uint32_t  dwLength);

/**
 * \brief This function is used to get the reference to Payload of a record specified by handle.
 * 		  Note: After the record/message is destroyed, this pointer is no longer valid.
 *
 * @param[in] 	  pRecord    Record handle.
 *
 * @return Pointer to payload.
 */
extern uint8_t * NDEF_Rec_Generic_GetPtrToPayload(const NDEF_Rec_GenericPayload_t * pRecord);

/**
* \brief  This function is a wrapper of NDEF_Rec_GetPayloadLen  from generic record interface, to allow easy access to payload length from generic payload interface.
*
*	@param[in] 	  pRecord    Record handle.
*
* @return Length of payload in bytes
*/
extern uint32_t  NDEF_Rec_Generic_GetPayloadLen(const NDEF_Rec_GenericPayload_t * pRecord);



#ifdef __cplusplus
} /* Extern C */
#endif

/** @}
* end of Rec Ndef Generic Record Type
*/

#endif /* NDEF_REC_GENERICPAYLOAD_H_ */
