/*
 * File:        pll_main.c
 * Purpose:     PLL Example
 *
 * License:     All software covered by license agreement in -
 *              docs/Freescale_Software_License.pdf
 */

#include "common.h"
#include "uif.h"
#include "clock/clock.h"
#include "uart/uart.h"

/********************************************************************/
/*
 * Test register definitions
 */
#define MCF_CLOCK_SYNTR          (*(vuint8 *)(&__IPSBAR[0x120003]))
#define MCF_CLOCK_OCR            (*(vuint16*)(&__IPSBAR[0x120004]))
#define MCF_CLOCK_OTR            (*(vuint8 *)(&__IPSBAR[0x120006]))

/*
 * Setup user interface
 */
const char PROMPT[] = "PLL> ";
    
void cmd_sysclk (int, char **);
void cmd_dump (int, char **);

UIF_CMD UIF_CMDTAB[] =
{
    UIF_CMDS_ALL
    {"clk",0,1,0,cmd_sysclk,"Set PLL output clock","<sys_clk (in KHz)>"},
    {"dump",0,0,0,cmd_dump,"Display the PLL registers",""},
};

void setshow_sysclk (int, char **);
void setshow_syncr (int, char **);
void setshow_mfd (int, char **);
void setshow_rfd (int, char **);
void setshow_lpd (int, char **);
void setshow_otr (int, char **);
void setshow_baud (int, char **);

UIF_SETCMD UIF_SETCMDTAB[] =
{
    {"sysclk", 0,1,setshow_sysclk,""},
    {"syncr", 0,1,setshow_syncr,""},
    {"mfd", 0,1,setshow_mfd,""},
    {"rfd", 0,1,setshow_rfd,""},
    {"lpd", 0,1,setshow_lpd,""},
    {"otr", 0,1,setshow_otr,""},
    {"baud", 0,1,setshow_baud,""},
};

const int UIF_NUM_CMD    = UIF_CMDTAB_SIZE;
const int UIF_NUM_SETCMD = UIF_SETCMDTAB_SIZE;

int baud = 19200;

/********************************************************************/
void 
main (void)
{
    mcf5xxx_irq_enable();
    
    printf("\n");
    printf("**************************************************\n");
    printf("*                                                *\n");
    printf("*                  PLL Utility                   *\n");
    printf("*                                                *\n");
    printf("**************************************************\n");
    printf(HELPMSG);
    printf("Enter all clock settings in KHz!\n");
    printf("\n");

    while (TRUE)
    {
        printf(PROMPT);
        run_cmd();
    }
}
/********************************************************************/
void
cmd_sysclk (int argc, char **argv)
{
    int success, sysclk, sys_clk_pll;
    
    if (argc == 1)
    {
        printf("System Clock: %d KHz\n", sys_clk_khz);
    }
    else
    {
        /* Wait for the UART to finish transmitting */
        while(!(MCF_UART_UCSR(TERMINAL_PORT) & MCF_UART_USR_TXEMP)) {}; 

        /* Get user input (in KHz) */
        sysclk = get_value(argv[1],&success,10);
        if (success == 0)
        {
            printf(INVALUE,argv[1]);
            return;
        }
        
        if (sysclk == 0)
        {
            /* Disable the PLL */
            sys_clk_khz = clock_pll(REF_CLK_KHZ, sysclk, PLL_DISABLE) / clock_lpd(0);
        }
        else
        {
            /* Set the PLL to the desired system clock */
            sys_clk_pll = clock_pll(REF_CLK_KHZ, sysclk, 0) / clock_lpd(0);
        }

        /* Re-init the UART with the new system clock setting */
        uart_init(TERMINAL_PORT, sys_clk_pll, baud);
        
        printf("System Clock: %d KHz\n", sys_clk_khz);
    }
}
/********************************************************************/
void
cmd_dump (int argc, char **argv)
{
    uint16 syncr = MCF_CLOCK_SYNCR;
    uint16 synsr = MCF_CLOCK_SYNSR;
    
    printf("\n");
    printf("SYNCR = %#04X\n",  syncr);
    printf("\tPLLEN   = %d\n",  (syncr & MCF_CLOCK_SYNCR_PLLEN));
    printf("\tPLLMODE = %d\n",  (syncr & MCF_CLOCK_SYNCR_PLLMODE));
    printf("\tCLKSRC  = %d\n",  (syncr & MCF_CLOCK_SYNCR_CLKSRC));
    printf("\tFWKUP   = %d\n",  (syncr & MCF_CLOCK_SYNCR_FWKUP));
    printf("\tDISCLK  = %d\n",  (syncr & MCF_CLOCK_SYNCR_DISCLK));
    printf("\tLOCEN   = %d\n",  (syncr & MCF_CLOCK_SYNCR_LOCEN));
    printf("\tRFD     = %d\n",  (syncr & 0x0700) >> 8);
    printf("\tLOCRE   = %d\n",  (syncr & MCF_CLOCK_SYNCR_LOCRE));
    printf("\tMFD     = %d\n",  (syncr & 0x7000) >> 12);
    printf("\tLOLRE   = %d\n",  (syncr & MCF_CLOCK_SYNCR_LOLRE));
    printf("\n");
    printf("SYNSR = %#02X\n",  synsr);
    printf("\tLOCS   = %d\n",   (synsr & MCF_CLOCK_SYNSR_LOCS));
    printf("\tLOCK   = %d\n",   (synsr & MCF_CLOCK_SYNSR_LOCK));
    printf("\tLOCKS  = %d\n",   (synsr & MCF_CLOCK_SYNSR_LOCKS));
    printf("\tCRYOSC = %d\n",   (synsr & MCF_CLOCK_SYNSR_CRYOSC));
    printf("\tOCOSC  = %d\n",   (synsr & MCF_CLOCK_SYNSR_OCOSC));
    printf("\tEXTOSC = %d\n",   (synsr & MCF_CLOCK_SYNSR_EXTOSC));
    printf("\n");
    printf("SYNTR    = %#02X\n", MCF_CLOCK_SYNTR);
    printf("OCR      = %#04X\n", MCF_CLOCK_OCR);
    printf("OTR      = %#02X\n", MCF_CLOCK_OTR);
    printf("LPCR     = %#02X\n", MCF_PMM_LPCR);
    printf("PPMRH    = %#08X\n", MCF_SCM_PPMRH);
    printf("PPMRL    = %#08X\n", MCF_SCM_PPMRL);
}
/********************************************************************/
void
setshow_sysclk (int argc, char **argv)
{
    int success, sysclk;
    
    /* Set */
    if (argv[2] != NULL)
    {
        /* Wait for the UART to finish transmitting */
        while(!(MCF_UART_UCSR(TERMINAL_PORT) & MCF_UART_USR_TXEMP)) {}; 

        /* Get user input (in KHz) */
        sysclk = get_value(argv[2],&success,10);
        if (success == 0)
        {
            printf(INVALUE,argv[2]);
            return;
        }
        
        if (sysclk == 0)
        {
            /* Disable the PLL */
            sys_clk_khz = clock_pll(REF_CLK_KHZ, sysclk, PLL_DISABLE) / clock_lpd(0);
        }
        else
        {
            /* Set the PLL to the desired system clock */
            sys_clk_khz = clock_pll(REF_CLK_KHZ, sysclk, 0) / clock_lpd(0);
        }

        /* Re-init the UART with the new system clock setting */
        uart_init(TERMINAL_PORT, sys_clk_khz, baud);
        
        printf("System Clock: %d KHz\n", sys_clk_khz);
    }
    
    /* Show */
    else
        printf("%d KHz", sys_clk_khz);
}
/********************************************************************/
void
setshow_syncr (int argc, char **argv)
{
    int success;
    uint16 syncr;

    /* Set */
    if (argv[2] != NULL)
    {
        /* Wait for the UART to finish transmitting */
        while(!(MCF_UART_UCSR(TERMINAL_PORT) & MCF_UART_USR_TXEMP)) {}; 

        syncr = (uint16) get_value(argv[2],&success,16);
        if (success == 0)
        {
            printf(INVALUE,argv[2]);
            return;
        }
        
        MCF_CLOCK_SYNCR = syncr;
        
        if ((MCF_CLOCK_SYNCR & MCF_CLOCK_SYNCR_CLKSRC) &&
            (MCF_CLOCK_SYNCR & MCF_CLOCK_SYNCR_PLLEN))
        {
            /* Wait for the PLL to lock */  
            while (!(MCF_CLOCK_SYNSR & MCF_CLOCK_SYNSR_LOCK)) {};
        }
        
        sys_clk_khz = clock_pll(REF_CLK_KHZ, 0, 0) / clock_lpd(0);

        /* Re-init the UART with the new system clock setting */
        uart_init(TERMINAL_PORT, sys_clk_khz, baud);
        
        printf("System Clock: %d KHz\n", sys_clk_khz);
    }
    
    /* Show */
    else
        printf("%#04X", MCF_CLOCK_SYNCR);
}
/********************************************************************/
void
setshow_otr (int argc, char **argv)
{
    int success;
    uint8 otr;

    /* Set */
    if (argv[2] != NULL)
    {
        otr = (uint8) get_value(argv[2],&success,16);
        if (success == 0)
        {
            printf(INVALUE,argv[2]);
            return;
        }
        
        MCF_CLOCK_OTR = otr;
    }
    
    /* Show */
    else
        printf("%#02X", MCF_CLOCK_OTR);
}
/********************************************************************/
void
setshow_lpd (int argc, char **argv)
{
    int success, lpd;

    /* Set */
    if (argv[2] != NULL)
    {
        /* Wait for the UART to finish transmitting */
        while(!(MCF_UART_UCSR(TERMINAL_PORT) & MCF_UART_USR_TXEMP)) {}; 

        lpd = (uint8) get_value(argv[2],&success,10);
        if (success == 0)
        {
            printf(INVALUE,argv[2]);
            return;
        }
        
        sys_clk_khz = clock_pll(REF_CLK_KHZ, 0, 0) / clock_lpd(lpd);

        /* Re-init the UART with the new system clock setting */
        uart_init(TERMINAL_PORT, sys_clk_khz, baud);
    }
    
    /* Show */
    else
        printf("%d", clock_lpd(0));
}
/********************************************************************/
void
setshow_mfd (int argc, char **argv)
{
    int success;
    uint8 mfd;

    /* Set */
    if (argv[2] != NULL)
    {
        /* Wait for the UART to finish transmitting */
        while(!(MCF_UART_UCSR(TERMINAL_PORT) & MCF_UART_USR_TXEMP)) {}; 

        mfd = (uint8) get_value(argv[2],&success,10);
        if (success == 0)
        {
            printf(INVALUE,argv[2]);
            return;
        }
        
        MCF_CLOCK_SYNCR = 0
            | (MCF_CLOCK_SYNCR & ~0x7000) 
            | MCF_CLOCK_SYNCR_MFD(mfd);

        /* Wait for the PLL to lock */  
        while (!(MCF_CLOCK_SYNSR & MCF_CLOCK_SYNSR_LOCK)) {};
        
        sys_clk_khz = clock_pll(REF_CLK_KHZ, 0, 0) / clock_lpd(0);

        /* Re-init the UART with the new system clock setting */
        uart_init(TERMINAL_PORT, sys_clk_khz, baud);
        
        printf("System Clock: %d KHz\n", sys_clk_khz);
    }
    
    /* Show */
    else
        printf("%d", (MCF_CLOCK_SYNCR & 0x7000) >> 12);
}
/********************************************************************/
void
setshow_rfd (int argc, char **argv)
{
    int success;
    uint8 rfd;

    /* Set */
    if (argv[2] != NULL)
    {
        /* Wait for the UART to finish transmitting */
        while(!(MCF_UART_UCSR(TERMINAL_PORT) & MCF_UART_USR_TXEMP)) {}; 

        rfd = (uint8) get_value(argv[2],&success,10);
        if (success == 0)
        {
            printf(INVALUE,argv[2]);
            return;
        }
        
        MCF_CLOCK_SYNCR = 0
            | (MCF_CLOCK_SYNCR & ~0x0700) 
            | MCF_CLOCK_SYNCR_RFD(rfd);
        
        /* 
         * Changing RFD doesn't create a loss of lock
         * so no need to wait for PLL to lock again
         */

        sys_clk_khz = clock_pll(REF_CLK_KHZ, 0, 0) / clock_lpd(0);

        /* Re-init the UART with the new system clock setting */
        uart_init(TERMINAL_PORT, sys_clk_khz, baud);
        
        printf("System Clock: %d KHz\n", sys_clk_khz);
    }
    
    /* Show */
    else
        printf("%d", (MCF_CLOCK_SYNCR & 0x0700) >> 8);
}
/********************************************************************/
void
setshow_baud (int argc, char **argv)
{
    int success, temp;

    /* Set */
    if (argv[2] != NULL)
    {
        temp = get_value(argv[2],&success,10);
        if (success == 0)
        {
            printf(INVALUE,argv[2]);
            return;
        }
        switch (temp)
        {
            case 110:  
            case 300:  
            case 600:  
            case 1200:  
            case 2400:  
            case 4800:  
            case 9600:  
            case 14400:
            case 19200:
            case 38400: 
            case 57600: 
            case 115200:
                baud = temp;
                break;    
            default:    
                printf(INVALUE,argv[2]);
                return;
        }
        /* Re-init the UART with the new system clock setting */
        uart_init(TERMINAL_PORT, sys_clk_khz, baud);
    }
    
    /* Show */
    else
        printf("%d", baud);
}
/********************************************************************/
